/*!
  * \file spm_CriticalVoltageManager.cpp
  *  \brief Base class implementation of critical voltage management.
  *
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * 1.0 |  05.01.11  | TMS Fischer       | initial version
  ******
  */

#include <boost/format.hpp>
#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_TYPES
#include "spm_fi_if.h"

// SPM  configuration
#include "spm_Config.h"
#include "spm_GlobDefs.h"

// my class header
#include "spm_CriticalVoltageManager.h"

// interfaces class definitions
#include "spm_ISystemPowerManager.h"
#include "spm_IOsalProxy.h"

#include "spm_IFactory.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM_LAM
#include "trcGenProj/Header/spm_CriticalVoltageManager.cpp.trc.h"
#endif

// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/
// #define SPM_TRACE_FILE_ID   SPM_FILE_CRITICALVOLTAGEMANAGER

spm_tclCriticalVoltageManager::spm_tclCriticalVoltageManager( const ISpmFactory& factory ) : ISpmCvmClient( factory )
   , _u32VoltageState( OSALCVM_LOW_VOLTAGE_END )
   , _hCriticalVoltageTimer(OSAL_C_INVALID_HANDLE)
   , _hVoltageTimer(OSAL_C_INVALID_HANDLE)
   , _u32CcaCvmEvent( spm_fi_tcl_SPM_e32_CVM_EVENT::FI_EN_SPM_U32_CVM_NO_EVENT )
   , _poclSystemPowerManager( NULL )
   , _poclWorkerServer( NULL ){
/*!
  * \fn
  *  \brief This constructor creates a timer for handling critical voltage events.
  *
  *  \param[in] factory: spm factory object.
  ******
  */
   dp_tclSpmDpConfigLowVoltageShutdownTime         oLowVoltShutdown;
   dp_tclSpmDpConfigCriticalLowVoltageShutdownTime oCrLowVoltShutdown;

   oLowVoltShutdown >> _u32LowVoltageTimeout;
   oCrLowVoltShutdown >> _u32CritLowVoltageTimeout;

   if ( OSAL_s32TimerCreate(  vCriticalVoltageTimerCallback, this, &_hCriticalVoltageTimer ) != OSAL_OK ){
      tU32 u32ErrorReason = OSAL_u32ErrorCode( );
      ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!! cannot create CriticalVoltageTimer: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
   }
   
   if ( OSAL_s32TimerCreate( vVoltageTimerCallback, this, &_hVoltageTimer ) != OSAL_OK ){
      tU32 u32ErrorReason = OSAL_u32ErrorCode( );
      ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!! cannot create VoltageTimer: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
   }
}

spm_tclCriticalVoltageManager::~spm_tclCriticalVoltageManager( ){
/*!
  * \fn
  *  \brief
  *    Destructor
  ******
  */
   SPM_NULL_POINTER_CHECK( _poclWorkerServer );
   _poclWorkerServer->vRemoveClient( this );

   // stop timer before deleting...
   if ( OSAL_s32TimerSetTime( _hCriticalVoltageTimer, 0, 0 ) != OSAL_OK ){
      tU32 u32ErrorReason = OSAL_u32ErrorCode( );
      ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!! cannot set CriticalVoltageTimer: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
   }
   if ( OSAL_s32TimerDelete( _hCriticalVoltageTimer ) != OSAL_OK ){
      tU32 u32ErrorReason = OSAL_u32ErrorCode( );
      ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!! cannot delete CriticalVoltageTimer: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
   }
   // stop timer before deleting...
   if ( OSAL_s32TimerSetTime( _hVoltageTimer, 0, 0 ) != OSAL_OK ){
      tU32 u32ErrorReason = OSAL_u32ErrorCode( );
      ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!! cannot set VoltageTimer: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
   }
   if ( OSAL_s32TimerDelete( _hVoltageTimer ) != OSAL_OK ){
      tU32 u32ErrorReason = OSAL_u32ErrorCode( );
      ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!! cannot delete VoltageTimer: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
   }
   
   _hCriticalVoltageTimer = OSAL_NULL;
   _hVoltageTimer = OSAL_NULL;
   _poclSystemPowerManager = NULL;
   _poclWorkerServer       = NULL;
}

tVoid spm_tclCriticalVoltageManager::vGetReferences( ){
/*!
  * \fn
  *  \brief
  *   SPM factory invokes this method to initialize the dependency of the spm_tclCriticalVoltageManager.
  ******
  */
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSystemPowerManager, ISpmSystemPowerManager );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclWorkerServer,       ISpmWorkerServer );
}

tVoid spm_tclCriticalVoltageManager::vStartCommunication( ){
   SPM_NULL_POINTER_CHECK( _poclWorkerServer );
   _poclWorkerServer->vAddClient( this );
}

tVoid spm_tclCriticalVoltageManager::vHandleCvmEvent( tU32 u32CvmEvent ){
/*!
  * \fn
  *  \brief This method receives the critical voltage events - LOW VOLTAGE START, LOW VOLTAGE END,
  *          CRITICAL LOW VOLTAGE START,CRITICAL LOW VOLTAGE END, HIGH VOLTAGE START, HIGH VOLTAGE END,
  *          CRITICAL HIGH VOLTAGE START,CRITICAL HIGH VOLTAGE END and handles the same in extemded classes of spm_tclCriticalVoltageManager.
  *
  *  \param[in] u32CvmEvent: critical voltage event type.
  ******
  */
   tU32 u32CcaCvmEvent = spm_fi_tcl_SPM_e32_CVM_EVENT::FI_EN_SPM_U32_CVM_NO_EVENT;

   ETG_TRACE_USR4( ( "spm_tclCriticalVoltageManager::vHandleCvmEvent(): New CVM event detected: %u (Current CVM state: %u)",
                     ETG_ENUM( SPM_CVM_EVENT, (tU32)u32CvmEvent ),
                     ETG_ENUM( SPM_CVM_EVENT, (tU32)_u32VoltageState )
                     ) );
   switch ( u32CvmEvent ){
      case OSALCVM_LOW_VOLTAGE_START:
      {
         if ( _u32LowVoltageTimeout > 0 ){
            if ( OSAL_s32TimerSetTime( _hVoltageTimer, _u32LowVoltageTimeout, 0 ) != OSAL_OK ){
               ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!! - OSALCVM_LOW_VOLTAGE_START cannot start timer" ) );
            }
         }
         _u32VoltageState = OSALCVM_LOW_VOLTAGE_START;

         u32CcaCvmEvent   = spm_fi_tcl_SPM_e32_CVM_EVENT::FI_EN_SPM_U32_CVM_LOW_VOLTAGE_START;

         vHandleOnLowVoltageStart( );
      }
      break;

      case OSALCVM_LOW_VOLTAGE_END:
      {
         if ( _u32LowVoltageTimeout > 0 ){
            vHandleOnLowVoltageEndTimeout( );

            if ( OSAL_s32TimerSetTime( _hVoltageTimer, 0, 0 ) != OSAL_OK ){
               ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!! - OSALCVM_LOW_VOLTAGE_END cannot start timer" ) );
            }
         }

         _u32VoltageState = OSALCVM_LOW_VOLTAGE_END;
         u32CcaCvmEvent   = spm_fi_tcl_SPM_e32_CVM_EVENT::FI_EN_SPM_U32_CVM_LOW_VOLTAGE_END;

         vHandleOnLowVoltageEnd( );
      }
      break;

      case OSALCVM_CRITICAL_LOW_VOLTAGE_START:
      {
         if ( _u32CritLowVoltageTimeout > 0 ){
            if ( OSAL_s32TimerSetTime( _hCriticalVoltageTimer, _u32CritLowVoltageTimeout, 0 ) != OSAL_OK ){
               ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!! - OSALCVM_CRITICAL_LOW_VOLTAGE_START cannot start timer" ) );
            }
         }
         _u32VoltageState = OSALCVM_CRITICAL_LOW_VOLTAGE_START;
         u32CcaCvmEvent   = spm_fi_tcl_SPM_e32_CVM_EVENT::FI_EN_SPM_U32_CVM_CRITICAL_LOW_VOLTAGE_START;

         vHandleOnCriticalLowVoltageStart( );
      }
      break;

      case OSALCVM_CRITICAL_LOW_VOLTAGE_END:
      {
         if ( _u32CritLowVoltageTimeout > 0 ){
            vHandleOnCriticalLowVoltageEndTimeout( );

            if ( OSAL_s32TimerSetTime( _hCriticalVoltageTimer, 0, 0 ) != OSAL_OK ){
               ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!! - OSALCVM_CRITICAL_LOW_VOLTAGE_END cannot start timer" ) );
            }
         }
         _u32VoltageState = OSALCVM_CRITICAL_LOW_VOLTAGE_END;
         u32CcaCvmEvent   = spm_fi_tcl_SPM_e32_CVM_EVENT::FI_EN_SPM_U32_CVM_CRITICAL_LOW_VOLTAGE_END;

         vHandleOnCriticalLowVoltageEnd( );
      }
      break;

      case OSALCVM_HIGH_VOLTAGE_START:
      {
         _u32VoltageState = OSALCVM_HIGH_VOLTAGE_START;
         u32CcaCvmEvent   = spm_fi_tcl_SPM_e32_CVM_EVENT::FI_EN_SPM_U32_CVM_HIGH_VOLTAGE_START;

         vHandleOnHighVoltageStart( );
      }
      break;

      case OSALCVM_HIGH_VOLTAGE_END:
      {
         _u32VoltageState = OSALCVM_HIGH_VOLTAGE_END;
         u32CcaCvmEvent   = spm_fi_tcl_SPM_e32_CVM_EVENT::FI_EN_SPM_U32_CVM_HIGH_VOLTAGE_END;

         vHandleOnHighVoltageEnd( );
      }
      break;

      case OSALCVM_CRITICAL_HIGH_VOLTAGE_START:
      {
         _u32VoltageState = OSALCVM_CRITICAL_HIGH_VOLTAGE_START;
         u32CcaCvmEvent   = spm_fi_tcl_SPM_e32_CVM_EVENT::FI_EN_SPM_U32_CVM_CRITICAL_HIGH_VOLTAGE_START;

         vHandleOnCriticalHighVoltageStart( );
      }
      break;

      case OSALCVM_CRITICAL_HIGH_VOLTAGE_END:
      {
         _u32VoltageState = OSALCVM_CRITICAL_HIGH_VOLTAGE_END;
         u32CcaCvmEvent   = spm_fi_tcl_SPM_e32_CVM_EVENT::FI_EN_SPM_U32_CVM_CRITICAL_HIGH_VOLTAGE_END;

         vHandleOnCriticalHighVoltageEnd( );
      }
      break;

      case OSALCVM_CRITICAL_LOW_VOLTAGE_AT_STARTUP:
      {
         vHandleOnLowVoltageAtStartup( );
      }
      break;

      default:
      {
         ETG_TRACE_ERRMEM( ( "spm_tclCriticalVoltageManager::vHandleCvmEvent: !!!!!! Unknown CVM event (%u) detected !!!!!!",
                             (tUInt)ETG_ENUM( SPM_CVM_EVENT, (tU32)u32CvmEvent )
                             ) );
      }

   } // switch

   if ( _u32CcaCvmEvent != u32CcaCvmEvent ){
      _u32CcaCvmEvent = u32CcaCvmEvent;
      vUpdateCvmEvent( u32CcaCvmEvent );
   }
} // vHandleCvmEvent

tVoid spm_tclCriticalVoltageManager::vCriticalVoltageTimerCallback( tVoid *pArg ){
/*!
  * \fn
  *  \brief Timer Callback for critical voltage management.
  *
  *  \param[in] pArg: instance of spm_tclCriticalVoltageManager.
  ******
  */

   spm_tclCriticalVoltageManager *poCriticalVoltageManager = (spm_tclCriticalVoltageManager*)pArg;

   SPM_NULL_POINTER_CHECK( poCriticalVoltageManager );

   poCriticalVoltageManager->_poclWorkerServer->bPostMessage( "spm_tclCriticalVoltageManagerIntern", SPM_U32_WORKER_CVM_CRITICAL_TIMEOUT );
}

tVoid spm_tclCriticalVoltageManager::vVoltageTimerCallback( tVoid *pArg ){
/*!
  * \fn
  *  \brief Timer Callback for critical voltage management.
  *
  *  \param[in] pArg: instance of spm_tclCriticalVoltageManager.
  ******
  */

   spm_tclCriticalVoltageManager *poCriticalVoltageManager = (spm_tclCriticalVoltageManager*)pArg;

   SPM_NULL_POINTER_CHECK( poCriticalVoltageManager );

   poCriticalVoltageManager->_poclWorkerServer->bPostMessage( "spm_tclCriticalVoltageManagerIntern", SPM_U32_WORKER_CVM_TIMEOUT );
}

tVoid spm_tclCriticalVoltageManager::vHandleMessage( tU32 u32Message,
                                                     tU32 u32Parm ){
/*!
  * \fn
  *  \brief
  *    Handles the Message from WorkerServer.
  *
  *  \param[in] u32Message: Message Identifier.
  *  \param[in] u32Parm:
  ******
  */
   (tVoid)u32Parm;
   SPM_NULL_POINTER_CHECK( _poclSystemPowerManager );
   SPM_GET_IF_REFERENCE_NEW_VAR( _poclOsalProxy, ISpmOsalProxy);
   std::string strBuffer;
   std::string strActive;
   tBool bUdrop90State = FALSE;
   tBool bUdrop60State = FALSE;
   tU16 u16Ubat        = 0;
   (tVoid)(_poclOsalProxy->bUbatSense(&u16Ubat));
   (tVoid)(_poclOsalProxy->bGetGpio( OSAL_EN_PWR_UDROP_60, &bUdrop60State ));
   (tVoid)(_poclOsalProxy->bGetGpio( OSAL_EN_PWR_UDROP_90, &bUdrop90State ));
   ETG_TRACE_USR4( ( "spm_tclCriticalVoltageManager::vHandleMessage(): _u32VoltageState %u, u32Message %u",
                     ETG_ENUM( SPM_CVM_EVENT, (tU32)_u32VoltageState ),
                     u32Message
                     ) );

   if ( u32Message == SPM_U32_WORKER_CVM_TIMEOUT ){

      if ( ( _u32VoltageState == OSALCVM_LOW_VOLTAGE_START )
           || ( _u32VoltageState == OSALCVM_CRITICAL_LOW_VOLTAGE_START )
           || ( _u32VoltageState == OSALCVM_CRITICAL_LOW_VOLTAGE_END )
           ){
         strActive = bUdrop90State ? "ACTIVE(1)":"INACTIVE(0)";
         strBuffer = std::string(getName( )) + ": Low voltage timeout (" + std::to_string(_u32LowVoltageTimeout) + " msec): shutdown (Current Battery ADC Value " + std::to_string(u16Ubat) + " mV, UDROP_90 Pin LogicalState: " + strActive + ")";
         _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strBuffer.c_str( ), (tU16)strBuffer.length( ) );

         vHandleOnLowVoltageTimeout( );

      } else if ( ( _u32VoltageState == OSALCVM_HIGH_VOLTAGE_START )
                  || ( _u32VoltageState == OSALCVM_CRITICAL_HIGH_VOLTAGE_START )
                  || ( _u32VoltageState == OSALCVM_CRITICAL_HIGH_VOLTAGE_END )
                  ){
         strBuffer = (std::string)getName( ) + ": Permanent High voltage : Shutdown";
         _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strBuffer.c_str( ), (tU16)strBuffer.length( ) );
         vHandleOnHighVoltageTimeout( );
      }
   } else if ( u32Message == SPM_U32_WORKER_CVM_CRITICAL_TIMEOUT ){

      if ( _u32VoltageState == OSALCVM_CRITICAL_LOW_VOLTAGE_START ){
         strActive = bUdrop60State? "ACTIVE(1)":"INACTIVE(0)" ;
         strBuffer = std::string(getName( )) + ": Critical Low voltage timeout (" + std::to_string(_u32CritLowVoltageTimeout) + " msec): shutdown (Current Battery ADC Value " + std::to_string(u16Ubat) + " mV, UDROP_60 Pin LogicalState: " + strActive + " )";
         _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strBuffer.c_str( ), (tU16)strBuffer.length( ) );
         vHandleOnCriticalVoltageTimeout( );
      } else if ( _u32VoltageState == OSALCVM_CRITICAL_HIGH_VOLTAGE_START ){
         strBuffer = (std::string)getName( ) + ": Permanent Critical High voltage : Shutdown";
         _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strBuffer.c_str( ), (tU16)strBuffer.length( ) );

         vHandleOnCriticalVoltageTimeout( );
      }
   }
} // vHandleMessage

tU32 spm_tclCriticalVoltageManager::u32GetRemainingCriticalVoltageTime( ){
/*!
  * \fn
  *  \brief This method returns the remaining critical voltage time.
  ******
  */
   tU32 u32RemainingTime = 0;
   tU32 u32IntervalTime  = 0;

   if ( OSAL_s32TimerGetTime( _hCriticalVoltageTimer, &u32RemainingTime, &u32IntervalTime ) != OSAL_OK ){
      // don't need to error trace -> on ADIT platform even if timer isn't running return value is != OK
      // ETG_TRACE_ERR(("SPM: !!!!!! Error detected !!!!!!"));
   }

   return( u32RemainingTime );
}

