/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_SystemStateMachineServiceHandler.cpp
  * @brief        Client Handler providing the interface to System State Machine Module of System Communication Controller(SCC).
  *               Implementation of System State Machine at iMX side to handle messages from System State Machine Module of System Communication Controller.
  * @copyright    (C) 2015 - 2016 Robert Bosch GmbH
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#include "spm_Config.h"

#include "inc.h"
#include "inc_ports.h"
#include "spm_SCCIncMessage.h"

#include "spm_SystemStateMachineServiceHandler.h"

#include "spm_IFactory.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
#include "trcGenProj/Header/spm_SystemStateMachineServiceHandler.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

#define SYSTEMSM_U8_HOST_APP_VERSION 0x01

spm_tclSystemStateMachineServiceHandler::spm_tclSystemStateMachineServiceHandler( const ISpmFactory& factory )
   : ISpmSystemStateMachineIncClientHandler( factory )
   , spm_tclSCCSocketInterface( AF_BOSCH_INC_AUTOSAR, SYSTEM_STATEMACHINE_PORT )
   , _eClientComponentStatus( SystemSm_INACTIVE )
   , _eServerComponentStatus( SystemSm_INACTIVE ){
/*!
  * \fn
  *  \brief
  *    Constructor
  *
  *  \param[in] factory: spm factory object.
  ******
  */
}

spm_tclSystemStateMachineServiceHandler::~spm_tclSystemStateMachineServiceHandler( ){
/*!
  * \fn
  *  \brief
  *    Destructor
  ******
  */
}

tVoid spm_tclSystemStateMachineServiceHandler::vGetReferences( ){
/*!
  * \fn
  *  \brief
  *   SPM factory invokes this method to initialize the dependency of the spm_tclSystemStateMachineServiceHandler.
  ******
  */
}

tVoid spm_tclSystemStateMachineServiceHandler::vStartCommunication( ){
/*!
  * \fn
  *  \brief
  *   SPM factory invokes this method after invoking the method - vGetReferences.
  *   This method sends SYSTEMSM_U8_INC_MSGID_C_COMPONENT_STATUS PDU to System Communication Controller.
  ******
  */
   //Set the client status to ACTIVE and request server to enter the ACTIVE state as well.
   vSetComponentStatus( SystemSm_ACTIVE );
}

tVoid spm_tclSystemStateMachineServiceHandler::vProcessSCCMessage( tU8 *pu8Message ){
/*!
  * \fn
  *  \brief
  *   This method processes the messages related to System State Machine, received from System Communication Controller.
  *  \param[in]
  *     pu8Message: Message received from System Communication Controller.
  ******
  */
   //Only accept the INC message when the communication is activated/
   //The R_COMPONENT_STATUS is always accepted
   if( bIsCommunicationActivated( ) || ( SYSTEMSM_U8_INC_MSGID_R_COMPONENT_STATUS == pu8Message[0] ) ){
      ETG_TRACE_USR4( ( "spm_tclSystemStateMachineServiceHandler::vProcessSCCMessage Received INC message %x",
                        ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_MSG_ID, pu8Message[0] ) ) );
      switch ( pu8Message[0] ){
         case SYSTEMSM_U8_INC_MSGID_R_REJECT:
         {
            vHandleIncMsgSystemSMRReject( pu8Message );

            break;
         }

         case SYSTEMSM_U8_INC_MSGID_R_COMPONENT_STATUS:
         {
            vHandleIncMsgSystemSMRComponentStatus( pu8Message );

            break;
         }

         case SYSTEMSM_U8_INC_MSGID_R_SET_STATEMACHINE_TYPE:
         {
            vHandleIncMsgSystemSMRSetStateMachineType( pu8Message );

            break;
         }

         case SYSTEMSM_U8_INC_MSGID_R_SET_SM_TRIGGER:
         {
            vHandleIncMsgSystemSMRSetSMTrigger( pu8Message );

            break;
         }

         case SYSTEMSM_U8_INC_MSGID_R_SET_SCC_SYSTEMSTATE:
         {
            vHandleIncMsgSystemSMRSetSccSystemState( pu8Message );

            break;
         }

         case SYSTEMSM_U8_INC_MSGID_R_SEND_APP_SYSTEMSTATE:
         {
            vHandleIncMsgSystemSMRSendAppSystemState( pu8Message );
            break;
         }

         case SYSTEMSM_U8_INC_MSGID_R_SCC_SYSTEMSTATE:
         {
            vHandleIncMsgSystemSMRSccSystemState( pu8Message );

            break;
         }

         case SYSTEMSM_U8_INC_MSGID_R_APP_SYSTEMSTATE:
         {
            vHandleIncMsgSystemSMRAppSystemState( pu8Message );

            break;
         }

         default:
         {
            ETG_TRACE_ERRMEM( ( "spm_tclSystemStateMachineServiceHandler::vProcessSCCMessage SCC Message(System State Machine) - %x Received Not Handled", pu8Message[0] ) );
            break;
         }
      } // end of switch pu8Message[0]
   } else {
      ETG_TRACE_FATAL( ( "spm_tclSystemStateMachineServiceHandler::vProcessSCCMessage: Receiving an INC message %x while communication is inactive. This message is ignored", \
                          ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_MSG_ID, pu8Message[0] ) ) );

      ETG_TRACE_ERRMEM( ( "spm_tclSystemStateMachineServiceHandler::vProcessSCCMessage: Receiving an INC message %x while communication is inactive. This message is ignored", \
                           ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_MSG_ID, pu8Message[0] ) ) );
   }

}    // vProcessSCCMessage

tBool spm_tclSystemStateMachineServiceHandler::bIsCommunicationActivated( ){
/*!
  * \fn
  *  \brief
  *   This method checks whether the communication between client and server is activated or not.
  *  \return
  *      TRUE  - if communication is ACTIVE
  *      FALSE - if communication is INACTIVE
  ******
  */
   tBool bReturnVal = false;

   if ( ( SystemSm_ACTIVE == _eClientComponentStatus ) && ( SystemSm_ACTIVE == _eServerComponentStatus ) ){
      ETG_TRACE_USR3( ( "spm_tclSystemStateMachineServiceHandler::bIsCommunicationActivated: Communication is activated" ) );
      bReturnVal = true;
   } else {
      ETG_TRACE_USR3( ( "spm_tclSystemStateMachineServiceHandler::bIsCommunicationActivated: Communication is not activated" ) );
   }

   return( bReturnVal );
} //bIsCommunicationActivated

tVoid spm_tclSystemStateMachineServiceHandler::vSetComponentStatus( eSystemSm_ApplicationStatus eClientComponentStatus ){
/*!
  * \fn
  *  \brief
  *   This method is called to send the C_COMPONENT_STATUS message to SCC to request to enter the given state
  *   We expect SCC will respond the same in the R_COMPONENT_STATUS. The INC communication is allowed only both client and server are in ACTIVE state.
  *
  *  \param[in]
  *     eHostAppStatus: Component status requested by client
  *  \details
  *     This method sends INC message C_COMPONENT_STATUS to SCC to perform the request.
  *     Expecting to receive the confirmation from SCC in the response R_COMPONENT_STATUS.
  ******
  */
   ETG_TRACE_USR1( ( "spm_tclSystemStateMachineServiceHandler::vSetComponentStatus The client component status is updated from %x to %x",
                     ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_APPLSTAT, _eClientComponentStatus ),
                     ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_APPLSTAT, eClientComponentStatus )) );
   _eClientComponentStatus = eClientComponentStatus;

   /*construct SCC message to send to INC socket*/
   spm_tclSCCMessage oSystemSMCComponentStatus( SYSTEMSM_U8_INC_MSGID_C_COMPONENT_STATUS, SYSTEMSM_U8_INC_MSGLEN_C_COMPONENT_STATUS );
   SYSTEMSM_lSerialize_SCC_SYSTEM_STATEMACHINE_C_COMPONENT_STATUS( oSystemSMCComponentStatus.pu8GetMsgData( ), oSystemSMCComponentStatus.u8GetMsgId( ), (uint8_t)_eClientComponentStatus, SYSTEMSM_U8_HOST_APP_VERSION );

   ETG_TRACE_USR1( ( "spm_tclSystemStateMachineServiceHandler::vSetComponentStatus: Send %x to SCC. HostAppStatus %x", \
                      ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_MSG_ID, oSystemSMCComponentStatus.u8GetMsgId( ) ), \
                      ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_APPLSTAT, _eClientComponentStatus ) ) );

   if(SCC_ERROR == s32SendToINCSocket( &oSystemSMCComponentStatus )) {
      ETG_TRACE_ERRMEM( ( "spm_tclSystemStateMachineServiceHandler::vSetComponentStatus: Could not send message ") );
   }
} // vSetComponentStatus


tBool spm_tclSystemStateMachineServiceHandler::bSetStateMachineType( eSystemSm_StateMachineType eStateMachineType ){
/*!
  * \fn
  *  \brief
  *      This interface method is used to tell the server SCC to change the state machine type.
  *  \param[in]
  *      eStateMachineType: state machine type
  *  \return
  *      TRUE : C_SET_STATEMACHINE_TYPE INC message has been sent to server to perform the request.
  *      FALSE: The communication is not activated. The request can not be fulfilled.
  *  \details
  *      The 'C_SET_STATEMACHINE_TYPE' INC message is sent to server SCC to perform the request.
  *      Server will confirm the state machine type after update in the R_SET_STATEMACHINE_TYPE.
  *      The confirmation is handled in the vOnSetStateMachineTypeRespond function.
  ******
  */
   tBool bReturnVal = false;
   ETG_TRACE_USR1( ( "spm_tclSystemStateMachineServiceHandler::bSetStateMachineType: Received the request to change the state machine type to %x", \
                      ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_STATEMACHINETYPE, eStateMachineType ) ) );

   /*construct SCC message to send to INC socket*/
   spm_tclSCCMessage oSystemSMCStateMachineType( SYSTEMSM_U8_INC_MSGID_C_SET_STATEMACHINE_TYPE, SYSTEMSM_U8_INC_MSGLEN_C_SET_STATEMACHINE_TYPE );

   SYSTEMSM_lSerialize_SCC_SYSTEM_STATEMACHINE_C_SET_STATEMACHINE_TYPE( oSystemSMCStateMachineType.pu8GetMsgData( ), oSystemSMCStateMachineType.u8GetMsgId( ), (uint8_t)eStateMachineType );

   if ( bIsCommunicationActivated( ) ){
      ETG_TRACE_USR1( ( "spm_tclSystemStateMachineServiceHandler::bSetStateMachineType: Send %x to SCC. StateMachineType %x", \
                         ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_MSG_ID, oSystemSMCStateMachineType.u8GetMsgId( ) ), \
                         ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_STATEMACHINETYPE, eStateMachineType ) ) );

      if(SCC_ERROR == s32SendToINCSocket( &oSystemSMCStateMachineType )) {
         ETG_TRACE_ERRMEM( ( "spm_tclSystemStateMachineServiceHandler::bSetStateMachineType: Could not send message ") );
         bReturnVal = false;
      } else {
         bReturnVal = true;
      }
   } else {
      ETG_TRACE_FATAL( ( "spm_tclSystemStateMachineServiceHandler::bSetStateMachineType: can not send %x to SCC to set StateMachineType as '%x' because SCCAppStatus is INACTIVE", \
                          ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_MSG_ID, oSystemSMCStateMachineType.u8GetMsgId( ) ), \
                          ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_STATEMACHINETYPE, eStateMachineType ) ) );

      ETG_TRACE_ERRMEM( ( "spm_tclSystemStateMachineServiceHandler::bSetStateMachineType: can not send %x to SCC to set StateMachineType as '%x' because SCCAppStatus is INACTIVE", \
                           ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_MSG_ID, oSystemSMCStateMachineType.u8GetMsgId( ) ), \
                           ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_STATEMACHINETYPE, eStateMachineType ) ) );
   }

   return ( bReturnVal );
} // bSetStateMachineType

tBool spm_tclSystemStateMachineServiceHandler::bSetStateMachineTrigger( eSystemSm_StateMachineTrigger eStateMachineTrigger,
                                                                        eSystemSm_TriggerState        eTriggerState ){
/*!
  * \fn
  *  \brief
  *      This interface method is used to tell the server SCC to set the trigger state of the given state machine trigger.
  *  \param[in]
  *      eStateMachineTrigger: state machine trigger
  *      eTriggerState       : trigger state
  *  \return
  *      TRUE : C_SET_SM_TRIGGER INC message has been sent to server to perform the request.
  *      FALSE: The communication is not activated. The request can not be fulfilled.
  *  \details
  *      The INC message 'C_SET_SM_TRIGGER' is sent to the server SCC to request update the state of the given state machine trigger.
  *      In return, the server will respond with the complete list of triggers and their states.
  ******
  */
   tBool bReturnVal = false;
   ETG_TRACE_USR1( ( "spm_tclSystemStateMachineServiceHandler::bSetStateMachineTrigger: Received the request to change the state of StateMachineTrigger %x to TriggerState %x ", \
                      ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_STATEMACHINETRIGGER, eStateMachineTrigger ), ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_TRIGGERSTATE, eTriggerState ) ) );

   /*construct SCC message to send to INC socket*/
   spm_tclSCCMessage oSystemSMCSetSmTrigger( SYSTEMSM_U8_INC_MSGID_C_SET_SM_TRIGGER, SYSTEMSM_U8_INC_MSGLEN_C_SET_SM_TRIGGER );

   SYSTEMSM_lSerialize_SCC_SYSTEM_STATEMACHINE_C_SET_SM_TRIGGER( oSystemSMCSetSmTrigger.pu8GetMsgData( ), oSystemSMCSetSmTrigger.u8GetMsgId( ), (uint8_t)eStateMachineTrigger, (uint8_t)eTriggerState );

   if ( bIsCommunicationActivated( ) ){

      ETG_TRACE_USR1( ( "spm_tclSystemStateMachineServiceHandler::bSetStateMachineTrigger: Send  %x to SCC. StateMachineTrigger %x and TriggerState %x ", \
                         ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_MSG_ID, oSystemSMCSetSmTrigger.u8GetMsgId( ) ), \
                         ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_STATEMACHINETRIGGER, eStateMachineTrigger ), ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_TRIGGERSTATE, eTriggerState ) ) );

      if(SCC_ERROR == s32SendToINCSocket( &oSystemSMCSetSmTrigger )) {
         ETG_TRACE_ERRMEM( ( "spm_tclSystemStateMachineServiceHandler::bSetStateMachineTrigger: Could not send message ") );
         bReturnVal = false;
      } else {
         bReturnVal = true;
      }
   } else {
      ETG_TRACE_FATAL( ( "spm_tclSystemStateMachineServiceHandler::bSetStateMachineTrigger: can not send %x to SCC to set StateMachineTrigger '%x' with TriggerState '%x' because SCCAppStatus is INACTIVE", \
                          ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_MSG_ID, oSystemSMCSetSmTrigger.u8GetMsgId( ) ), \
                          ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_STATEMACHINETRIGGER, eStateMachineTrigger ), ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_TRIGGERSTATE, eTriggerState ) ) );

      ETG_TRACE_ERRMEM( ( "spm_tclSystemStateMachineServiceHandler::bSetStateMachineTrigger: can not send %x to SCC to set StateMachineTrigger '%x' with TriggerState '%x' because SCCAppStatus is INACTIVE", \
                           ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_MSG_ID, oSystemSMCSetSmTrigger.u8GetMsgId( ) ), \
                           ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_STATEMACHINETRIGGER, eStateMachineTrigger ), ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_TRIGGERSTATE, eTriggerState ) ) );
   }
   return ( bReturnVal );
} // bSetStateMachineTrigger

tBool spm_tclSystemStateMachineServiceHandler::bSetSCCSystemState( eSystemSm_SccSystemState eSccSystemState ){
/*!
  * \fn
  *  \brief
  *      This interface method is used to request SCC state machine to enter a specific state.
  *  \param[in]
  *      eSccSystemState: SCC System State
  *  \return
  *      TRUE : C_SET_SCC_SYSTEMSTATE INC message has been sent to server to perform the request.
  *      FALSE: The communication is not activated. The request can not be fulfilled.
  *  \details
  *      'C_SET_SCC_SYSTEMSTATE' message is sent to server SCC to perform the request.
  *      In return, the server will respond the confirmation of new state in the R_SET_SCC_SYSTEMSTATE message.
  ******
  */
   tBool bReturnVal = false;
   ETG_TRACE_USR1( ( "spm_tclSystemStateMachineServiceHandler::bSetSCCSystemState: Received the request to change the state machine of SCC to enter the %x state ", \
                      ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_SCCSYSTEMSTATE, eSccSystemState ) ) );

   /*construct SCC message to send to INC socket*/
   spm_tclSCCMessage oSystemSMCSetSccSystemState( SYSTEMSM_U8_INC_MSGID_C_SET_SCC_SYSTEMSTATE, SYSTEMSM_U8_INC_MSGLEN_C_SET_SCC_SYSTEMSTATE );

   SYSTEMSM_lSerialize_SCC_SYSTEM_STATEMACHINE_C_SET_SCC_SYSTEMSTATE( oSystemSMCSetSccSystemState.pu8GetMsgData( ), oSystemSMCSetSccSystemState.u8GetMsgId( ), (uint8_t)eSccSystemState );

   if ( bIsCommunicationActivated( ) ){
      ETG_TRACE_USR1( ( "spm_tclSystemStateMachineServiceHandler::bSetSCCSystemState: Send  %x to SCC. SccSystemState %x ", \
                         ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_MSG_ID, oSystemSMCSetSccSystemState.u8GetMsgId( ) ), \
                         ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_SCCSYSTEMSTATE, eSccSystemState ) ) );
      if(SCC_ERROR == s32SendToINCSocket( &oSystemSMCSetSccSystemState )) {
         ETG_TRACE_ERRMEM( ( "spm_tclSystemStateMachineServiceHandler::bSetSCCSystemState: Could not send message ") );
         bReturnVal = false;
      } else {
         bReturnVal = true;
      }
   } else {
      ETG_TRACE_FATAL( ( "spm_tclSystemStateMachineServiceHandler::bSetSCCSystemState: can not send %x to SCC to set SccSystemState as '%x' because SCCAppStatus is INACTIVE", \
                          ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_MSG_ID, oSystemSMCSetSccSystemState.u8GetMsgId( ) ), \
                          ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_SCCSYSTEMSTATE, eSccSystemState ) ) );

      ETG_TRACE_ERRMEM( ( "spm_tclSystemStateMachineServiceHandler::bSetSCCSystemState: can not send %x to SCC to set SccSystemState as '%x' because SCCAppStatus is INACTIVE", \
                           ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_MSG_ID, oSystemSMCSetSccSystemState.u8GetMsgId( ) ), \
                           ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_SCCSYSTEMSTATE, eSccSystemState ) ) );
   }
   return ( bReturnVal );
} // bSetSCCSystemState

tBool spm_tclSystemStateMachineServiceHandler::bSetAppSystemState( eSystemSm_AppSystemState eAppSystemState ){
/*!
  * \fn
  *  \brief
  *      This interface method is used to update to the server SCC that the client system state has changed.
  *  \param[in]
  *      eAppSystemState: Application system state
  *  \details
  *      The message C_SEND_APP_SYSTEMSTATE is sent to the server to inform the update.
  *      In return, the server SCC will respond its new state due to the change of client's state in the R_SEND_APP_SYSTEMSTATE
  ******
  */
   tBool bReturnVal = false;
   ETG_TRACE_USR1( ( "spm_tclSystemStateMachineServiceHandler::bSetAppSystemState: The client app system state has been changed to %x and this will be inform the server SCC", \
                      ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_APPSYSTEMSTATE, eAppSystemState ) ) );

   /*construct SCC message to send to INC socket*/
   spm_tclSCCMessage oSystemSMCSendAppSystemState( SYSTEMSM_U8_INC_MSGID_C_SEND_APP_SYSTEMSTATE, SYSTEMSM_U8_INC_MSGLEN_C_SEND_APP_SYSTEMSTATE );

   SYSTEMSM_lSerialize_SCC_SYSTEM_STATEMACHINE_C_SEND_APP_SYSTEMSTATE( oSystemSMCSendAppSystemState.pu8GetMsgData( ), oSystemSMCSendAppSystemState.u8GetMsgId( ), (uint8_t)eAppSystemState );

   if ( bIsCommunicationActivated( ) ){
      ETG_TRACE_USR1( ( "spm_tclSystemStateMachineServiceHandler::bSetAppSystemState: Send  %x to SCC. AppSystemState %x ", \
                         ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_MSG_ID, oSystemSMCSendAppSystemState.u8GetMsgId( ) ), \
                         ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_APPSYSTEMSTATE, eAppSystemState ) ) );

      if(SCC_ERROR == s32SendToINCSocket( &oSystemSMCSendAppSystemState )) {
         ETG_TRACE_ERRMEM( ( "spm_tclSystemStateMachineServiceHandler::bSetAppSystemState: Could not send message ") );
         bReturnVal = false;
      } else {
         bReturnVal = true;
      }
   } else {
      ETG_TRACE_FATAL( ( "spm_tclSystemStateMachineServiceHandler::bSetAppSystemState: can not send %x to SCC to send AppSystemState as '%x' because SCCAppStatus is INACTIVE", \
                          ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_MSG_ID, oSystemSMCSendAppSystemState.u8GetMsgId( ) ), \
                          ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_APPSYSTEMSTATE, eAppSystemState ) ) );

      ETG_TRACE_ERRMEM( ( "spm_tclSystemStateMachineServiceHandler::bSetAppSystemState: can not send %x to SCC to send AppSystemState as '%x' because SCCAppStatus is INACTIVE", \
                           ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_MSG_ID, oSystemSMCSendAppSystemState.u8GetMsgId( ) ), \
                           ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_APPSYSTEMSTATE, eAppSystemState ) ) );
   }
   return ( bReturnVal );
} // bSetAppSystemState

tVoid spm_tclSystemStateMachineServiceHandler::vHandleIncMsgSystemSMRComponentStatus( tU8 *pu8Message ){
/*!
  * \fn
  *  \brief
  *   This method handles INC message 'R_COMPONENT_STATUS' received from System Communication Controller.
  *  \param[in]
  *     pu8Message: Message data received from System Communication Controller.
  *  \details
  *     This method calls 'vOnSCCAppStatusUpdate' to perform required action when SCC application status is updated.
  ******
  */
   eSystemSm_ApplicationStatus eSCCAppStatus = SystemSm_INACTIVE;
   tU8                         u8SCCAppVersion;
   tU8                         u8MsgId;

   spm_tclSCCMessage           oSystemSMRComponentStatus( pu8Message, SYSTEMSM_U8_INC_MSGLEN_R_COMPONENT_STATUS );

   SYSTEMSM_lDeSerialize_SCC_SYSTEM_STATEMACHINE_R_COMPONENT_STATUS( oSystemSMRComponentStatus.pu8GetMsgData( ), &u8MsgId, (tU8*)( &eSCCAppStatus ), &u8SCCAppVersion );
   ETG_TRACE_USR1( ( "spm_tclSystemStateMachineServiceHandler::vHandleIncMsgSystemSMRComponentStatus: MsgId = %x, SCCAppStatus = %x and SCCAppVersion = %x", \
                      ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_MSG_ID, u8MsgId ), \
                      ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_APPLSTAT, eSCCAppStatus ), u8SCCAppVersion ) );
   vOnComponentStatusRespond( eSCCAppStatus );
} //vHandleIncMsgSystemSMRComponentStatus

tVoid spm_tclSystemStateMachineServiceHandler::vHandleIncMsgSystemSMRReject( tU8 *pu8Message ){
/*!
  * \fn
  *  \brief
  *   This method handles INC message 'R_REJECT' received from System Communication Controller.
  *  \param[in]
  *     pu8Message: Message data received from System Communication Controller.
  *  \details
  *     This method prints the Rejected message ID and also the reject reason.
  *     The vOnMessageReject is called to handle the rejected message.
  ******
  */
   eSystemSm_RejReason eRejectReason = SystemSm_REJ_NO_REASON;
   tU8                 u8RejectedMsgID;
   tU8                 u8MsgId;

   spm_tclSCCMessage   oSystemSMRReject( pu8Message, SYSTEMSM_U8_INC_MSGLEN_R_REJECT );

   SYSTEMSM_lDeSerialize_SCC_SYSTEM_STATEMACHINE_R_REJECT( oSystemSMRReject.pu8GetMsgData( ), &u8MsgId, (tU8*)( &eRejectReason ), &u8RejectedMsgID );

   ETG_TRACE_FATAL( ( "spm_tclSystemStateMachineServiceHandler::vHandleIncMsgSystemSMRReject: MsgId = %x, RejectReason = %x and RejectedMsgID = 0x%x", \
                       ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_MSG_ID, u8MsgId ), \
                       ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_REJECTREASON, eRejectReason ), u8RejectedMsgID ) );

   vOnMessageReject( u8RejectedMsgID, eRejectReason );
} //vHandleIncMsgSystemSMRReject

tVoid spm_tclSystemStateMachineServiceHandler::vHandleIncMsgSystemSMRSetStateMachineType( tU8 *pu8Message ){
/*!
  * \fn
  *  \brief
  *   This method handles INC message 'R_SET_STATEMACHINE_TYPE' received from System Communication Controller.
  *  \param[in]
  *     pu8Message: Message data received from System Communication Controller.
  *  \detail
  *     vOnSetStateMachineTypeRespond is called to handle the response from SCC
  ******
  */
   eSystemSm_StateMachineType eStateMachineType = SYSTEMSM_SMTYPE_DEFAULT;
   tU8                        u8MsgId;

   spm_tclSCCMessage          oSystemSMRStateMachineType( pu8Message, SYSTEMSM_U8_INC_MSGLEN_R_SET_STATEMACHINE_TYPE );

   SYSTEMSM_lDeSerialize_SCC_SYSTEM_STATEMACHINE_R_SET_STATEMACHINE_TYPE( oSystemSMRStateMachineType.pu8GetMsgData( ), &u8MsgId, (tU8*)( &eStateMachineType ) );

   ETG_TRACE_USR1( ( "spm_tclSystemStateMachineServiceHandler::vHandleIncMsgSystemSMRSetStateMachineType: MsgId = %x and StateMachineType = %x", \
                      ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_MSG_ID, u8MsgId ), \
                      ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_STATEMACHINETYPE, eStateMachineType ) ) );

   vOnSetStateMachineTypeRespond( eStateMachineType );
} //vHandleIncMsgSystemSMRSetStateMachineType

tVoid spm_tclSystemStateMachineServiceHandler::vHandleIncMsgSystemSMRSetSccSystemState( tU8 *pu8Message ){
/*!
  * \fn
  *  \brief
  *   This method handles INC message 'R_SET_SCC_SYSTEMSTATE' received from System Communication Controller.
  *  \param[in]
  *     pu8Message: Message data received from System Communication Controller.
  *  \details
  *     vOnSCCSystemStateUpdate handler is called to handle this response from SCC.
  ******
  */
   eSystemSm_SccSystemState eSccSystemState = SystemSm_HMI_NOT_ALLOWED;
   tU8                      u8MsgId;

   spm_tclSCCMessage        oSystemSMRSetSccSystemState( pu8Message, SYSTEMSM_U8_INC_MSGLEN_R_SET_SCC_SYSTEMSTATE );

   SYSTEMSM_lDeSerialize_SCC_SYSTEM_STATEMACHINE_R_SET_SCC_SYSTEMSTATE( oSystemSMRSetSccSystemState.pu8GetMsgData( ), &u8MsgId, (tU8*)( &eSccSystemState ) );

   ETG_TRACE_USR1( ( "spm_tclSystemStateMachineServiceHandler::vHandleIncMsgSystemSMRSetSccSystemState: MsgId = %x and SccSystemState = %x", \
                     ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_MSG_ID, u8MsgId ), \
                     ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_SCCSYSTEMSTATE, eSccSystemState ) ) );
   vOnSCCSystemStateUpdate( eSccSystemState );
} //vHandleIncMsgSystemSMRSetSccSystemState

tVoid spm_tclSystemStateMachineServiceHandler::vHandleIncMsgSystemSMRSetSMTrigger( tU8 *pu8Message ){
/*!
  * \fn
  *  \brief
  *   This method handles INC message 'R_SET_SM_TRIGGER' received from System Communication Controller.
  *  \param[in]
  *     pu8Message: Message data received from System Communication Controller.
  *  \detail
  *     The method vOnSetStateMachineTriggerRespond is used to handle the response.
  ******
  */

   tU8      u8NumOfTriggers                   = 0;
   Length_t u16NumOfStateMachineTrigger_List1 = 0;
   Length_t u16NumOfStateMachineTrigger_List2 = 0;
   Length_t u16NumOfStateMachineTrigger_List3 = 0;
   tU8      u8StateMachineTrigger_List1[8];
   tU8      u8StateMachineTrigger_List2[8];
   tU8      u8StateMachineTrigger_List3[8];
   tU8      u8MsgId;

   memset( u8StateMachineTrigger_List1, 0, 8 );
   memset( u8StateMachineTrigger_List2, 0, 8 );
   memset( u8StateMachineTrigger_List3, 0, 8 );

   spm_tclSCCMessage oSystemSMRSetSmTrigger( pu8Message, SYSTEMSM_U8_INC_MSGLEN_R_SET_SM_TRIGGER );
   SYSTEMSM_lDeSerialize_SCC_SYSTEM_STATEMACHINE_R_SET_SM_TRIGGER( oSystemSMRSetSmTrigger.pu8GetMsgData( ), &u8MsgId, &u8NumOfTriggers,
                                                                   u8StateMachineTrigger_List1, &u16NumOfStateMachineTrigger_List1,
                                                                   u8StateMachineTrigger_List2, &u16NumOfStateMachineTrigger_List2,
                                                                   u8StateMachineTrigger_List3, &u16NumOfStateMachineTrigger_List3 );

   ETG_TRACE_USR1( ( "spm_tclSystemStateMachineServiceHandler::vHandleIncMsgSystemSMRSetSMTrigger: MsgId = %x and NumOfTriggers = %x", \
                     ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_MSG_ID, u8MsgId ), \
                     u8NumOfTriggers ) );

   vOnSetStateMachineTriggerRespond(u8NumOfTriggers, u8StateMachineTrigger_List1, (tU8)u16NumOfStateMachineTrigger_List1,
                                     u8StateMachineTrigger_List2, (tU8)u16NumOfStateMachineTrigger_List2,
                                     u8StateMachineTrigger_List3, (tU8)u16NumOfStateMachineTrigger_List3 );
} // vHandleIncMsgSystemSMRSetSMTrigger
tVoid spm_tclSystemStateMachineServiceHandler::vHandleIncMsgSystemSMRSendAppSystemState( tU8 *pu8Message ){
/*!
  * \fn
  *  \brief
  *   This method handles INC message 'R_SEND_APP_SYSTEMSTATE' received from System Communication Controller.
  *  \param[in]
  *     pu8Message: Message data received from System Communication Controller.
  *  \details
  *     Function vOnSCCSystemStateUpdate is used to handle this response.
  ******
  */
   eSystemSm_SccSystemState eSccSystemState = SystemSm_HMI_NOT_ALLOWED;
   tU8                      u8MsgId;

   spm_tclSCCMessage        oSystemSMRSendAppSystemState( pu8Message, SYSTEMSM_U8_INC_MSGLEN_R_SEND_APP_SYSTEMSTATE );

   SYSTEMSM_lDeSerialize_SCC_SYSTEM_STATEMACHINE_R_SEND_APP_SYSTEMSTATE( oSystemSMRSendAppSystemState.pu8GetMsgData( ), &u8MsgId, (tU8*)( &eSccSystemState ) );

   ETG_TRACE_USR1( ( "spm_tclSystemStateMachineServiceHandler::vHandleIncMsgSystemSMRSendAppSystemState: MsgId = %x and SccSystemState = %x", \
                     ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_MSG_ID, u8MsgId ), \
                     ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_SCCSYSTEMSTATE, eSccSystemState ) ) );

   vOnSCCSystemStateUpdate( eSccSystemState );
} //vHandleIncMsgSystemSMRSendAppSystemState

tVoid spm_tclSystemStateMachineServiceHandler::vHandleIncMsgSystemSMRSccSystemState( tU8 *pu8Message ){
/*!
  * \fn
  *  \brief
  *   This method handles INC message 'R_SCC_SYSTEMSTATE' received from System Communication Controller. This is a spontaneous message.
  *  \param[in]
  *     pu8Message: Message data received from System Communication Controller.
  *  \detail
  *     The method vOnSCCSystemStateUpdate is used to handle this state update
  ******
  */
   eSystemSm_SccSystemState eSccSystemState = SystemSm_HMI_NOT_ALLOWED;
   tU8                      u8MsgId;

   spm_tclSCCMessage        oSystemSMRSccSystemState( pu8Message, SYSTEMSM_U8_INC_MSGLEN_R_SCC_SYSTEMSTATE );

   SYSTEMSM_lDeSerialize_SCC_SYSTEM_STATEMACHINE_R_SCC_SYSTEMSTATE( oSystemSMRSccSystemState.pu8GetMsgData( ), &u8MsgId, (tU8*)( &eSccSystemState ) );

   ETG_TRACE_USR1( ( "spm_tclSystemStateMachineServiceHandler::vHandleIncMsgSystemSMRSccSystemState: MsgId = %x and SccSystemState = %x", \
                     ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_MSG_ID, u8MsgId ), \
                     ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_SCCSYSTEMSTATE, eSccSystemState ) ) );

   vOnSCCSystemStateUpdate( eSccSystemState );
} //vHandleIncMsgSystemSMRSccSystemState

tVoid spm_tclSystemStateMachineServiceHandler::vHandleIncMsgSystemSMRAppSystemState( tU8 *pu8Message ){
/*!
  * \fn
  *  \brief
  *   This method handles INC message 'R_APP_SYSTEMSTATE' received from System Communication Controller.
  *  \param[in]
  *     pu8Message: Message data received from System Communication Controller.
  *  \detail
  *     Method vOnSetAppSystemStateRequest is used to handle the request
  ******
  */
   eSystemSm_AppSystemState eAppSystemState = SystemSm_SPM_SYSTEM_SUSPEND;
   tU8                      u8MsgId;

   spm_tclSCCMessage        oSystemSMRAppSystemState( pu8Message, SYSTEMSM_U8_INC_MSGLEN_R_APP_SYSTEMSTATE );

   SYSTEMSM_lDeSerialize_SCC_SYSTEM_STATEMACHINE_R_APP_SYSTEMSTATE( oSystemSMRAppSystemState.pu8GetMsgData( ), &u8MsgId, (tU8*)( &eAppSystemState ) );

   ETG_TRACE_USR1( ( "spm_tclSystemStateMachineServiceHandler::vHandleIncMsgSystemSMRAppSystemState: MsgId = %x and AppSystemState = %x", \
                     ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_MSG_ID, u8MsgId ), \
                     ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_APPSYSTEMSTATE, eAppSystemState ) ) );

   vOnSetAppSystemStateRequest( eAppSystemState );
} //vHandleIncMsgSystemSMRAppSystemState


tVoid spm_tclSystemStateMachineServiceHandler::vOnComponentStatusRespond( eSystemSm_ApplicationStatus eServerComponentStatus ){
/*!
  * \fn
  *  \brief
  *      This method handles the server application status responded from SCC in R_COMPONENT_STATUS message.
  *  \param[in]
  *      eServerComponentStatus: The server component status received from System Communication Controller.
  *  \details
  *      The responded status will be stored locally and match with client component status.
  ******
  */
   ETG_TRACE_USR1( ( "spm_tclSystemStateMachineServiceHandler::vOnComponentStatusRespond: Server confirmed the component status is %x",
                     ETG_ENUM( FD_SYSTEM_SM_SYSTEM_SM_APPLSTAT, eServerComponentStatus )) );
   _eServerComponentStatus = eServerComponentStatus;
} //vOnComponentStatusRespond


