/*****************************************************************************
 * FILE:         spm_LateServiceHandler.cpp
 * PROJECT:      G3G
 * SW-COMPONENT:
 * ----------------------------------------------------------------------------
 *
 * DESCRIPTION:  Handles application interface for applications started by
 *               late LCM
 *
 * ----------------------------------------------------------------------------
 * COPYRIGHT:    (c) 2012 Robert Bosch Car Multimedia GmbH, Hildesheim
 * HISTORY:
 * Date      | Author            | Modification
 * 23.05.12  | CM-AI/VW32 kollai | Initial version for G3G
 *****************************************************************************/
#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#include "common_fi_if.h"

#define LCM_FI_S_IMPORT_INTERFACE_LCM_DBUSCTRLFI_STDVISITORS
#define LCM_FI_S_IMPORT_INTERFACE_LCM_DBUSCTRLFI_FUNCTIONIDS
#include "lcm_fi_if.h"

// SPM  configuration
#include "spm_Config.h"

// my class header
#include "spm_LateServiceHandler.h"

// spm class definitions

// interfaces class definitions
#include "spm_ICcaServiceServer.h"
#include "spm_ILocalAppManager.h"
#include "spm_IStartupCommon.h"
#include "spm_IApplicationDatabase.h"
#include "spm_IApplicationConfiguration.h"
#include "spm_IStartupSystem.h"
#include "IDbusProxy.h"
#include "spm_IApplicationErrorHandler.h"
#include "spm_IStartupInvestigationServer.h"

#include "spm_SoftwareBlock.h"
#include "spm_IFactory.h"

#include <sstream>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM_SSM
#include "trcGenProj/Header/spm_LateServiceHandler.cpp.trc.h"
#endif

// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

#define SPMSLV_FI_MAJOR_VERSION  1
#define SPMSLV_FI_MINOR_VERSION  0
#define SPMSLV_FI_PATCH_VERSION  0  //lint -e750 -> only for reference

/*
   --------------------------------------------------------------------------
   MSG MAP
   --------------------------------------------------------------------------
  */
BEGIN_MSG_MAP( spm_tclSpmLateServiceHandler, spm_tclClientHandlerBase )
// Property handler
ON_MESSAGE_SVCDATA( LCM_DBUSCTRLFI_C_U16_STARTPROCESS,       AMT_C_U8_CCAMSG_OPCODE_METHODRESULT, vHandleStartProcessMResult )
ON_MESSAGE_SVCDATA( LCM_DBUSCTRLFI_C_U16_GETPROCESSID,       AMT_C_U8_CCAMSG_OPCODE_METHODRESULT, vHandleGetProcessIdMResult )

// method handler
ON_MESSAGE_SVCDATA( LCM_DBUSCTRLFI_C_U16_APPSTARTREQUEST,           AMT_C_U8_CCAMSG_OPCODE_STATUS,       vHandleAppStartReqUpdate )
ON_MESSAGE_SVCDATA( LCM_DBUSCTRLFI_C_U16_APPINITIALIZED,            AMT_C_U8_CCAMSG_OPCODE_STATUS,       vHandleAppInitUpdate )
ON_MESSAGE_SVCDATA( LCM_DBUSCTRLFI_C_U16_APPSTATECHANGE,            AMT_C_U8_CCAMSG_OPCODE_STATUS,       vHandleAppStateChangeUpdate )
ON_MESSAGE_SVCDATA( LCM_DBUSCTRLFI_C_U16_NEWSERVICELOADED,          AMT_C_U8_CCAMSG_OPCODE_STATUS,       vHandleNewServiceLoadedUpdate )
ON_MESSAGE_SVCDATA( LCM_DBUSCTRLFI_C_U16_FAILEDSERVICESLIST,        AMT_C_U8_CCAMSG_OPCODE_STATUS,       vHandleFailedServicesListUpdate )
ON_MESSAGE_SVCDATA( LCM_DBUSCTRLFI_C_U16_NEWPROCESSID,              AMT_C_U8_CCAMSG_OPCODE_STATUS,       vHandleNewProcessIdUpdate )
ON_MESSAGE_SVCDATA( LCM_DBUSCTRLFI_C_U16_STARTUPFAILUREINDICATION,  AMT_C_U8_CCAMSG_OPCODE_STATUS,       vHandleStartupFailureIndUpdate )


END_MSG_MAP( ) spm_tclSpmLateServiceHandler::spm_tclSpmLateServiceHandler( const ISpmFactory& factory )
   : spm_ISpmLateServiceHandler( factory,
                                 CCA_C_U16_APP_SPM_SLV,
                                 CCA_C_U16_SRV_SPM_SLV,
                                 SPMSLV_FI_MAJOR_VERSION,
                                 SPMSLV_FI_MINOR_VERSION )
   , _poclCcaServiceHandler( NULL )
   , _poclStartupInvest( NULL )
   , _bMessageLeftToBeSend(FALSE )
   , _bSupervisionActive( TRUE )
   , _bLcmLateStarted( FALSE ){
   _ServiceFailedEntryList.clear( );
}

spm_tclSpmLateServiceHandler::~spm_tclSpmLateServiceHandler( ){
   _poclCcaServiceHandler = NULL;
}

tVoid spm_tclSpmLateServiceHandler::vGetReferences( ){
   spm_tclClientHandlerBase::vGetReferences( );

   SPM_GET_CLASS_REFERENCE_USE_VAR( _poclCcaServiceHandler, spm_tclCcaServiceHandlerConfig, ISpmCcaServiceServer );
   SPM_GET_IF_REFERENCE_USE_VAR   ( _poclStartupInvest, ISpmStartupInvestigationServer );
}

tVoid spm_tclSpmLateServiceHandler::vOnFirstServiceStateAvailable( ){
   ETG_TRACE_USR1( ( "spm_tclSpmLateServiceHandler::vOnFirstServiceStateAvailable: Start now late communication." ) );
   _crfFactory.vStartAllLateCommunication( );
   vStartupActivated( );

   SPM_GET_IF_REFERENCE_NEW_VAR( poclStartup, ISpmStartupSystem );
   poclStartup->vCheckForProcessToStart( );
   poclStartup->vOnSyncPointReached( SPM_STR_SYNC_POINT_LCMLATE_STARTED );
   _bLcmLateStarted = TRUE;
   if(_bMessageLeftToBeSend)
   {
      vSupervisionStateChange(_bSupervisionActive);
      _bMessageLeftToBeSend = FALSE;
   }
}

tVoid spm_tclSpmLateServiceHandler::vRegisterFids( ){
   vUpRegFid( LCM_DBUSCTRLFI_C_U16_APPSTARTREQUEST );
   vUpRegFid( LCM_DBUSCTRLFI_C_U16_APPINITIALIZED );
   vUpRegFid( LCM_DBUSCTRLFI_C_U16_APPSTATECHANGE );
   vUpRegFid( LCM_DBUSCTRLFI_C_U16_NEWSERVICELOADED );
   vUpRegFid( LCM_DBUSCTRLFI_C_U16_FAILEDSERVICESLIST );
   vUpRegFid( LCM_DBUSCTRLFI_C_U16_NEWPROCESSID );
   vUpRegFid( LCM_DBUSCTRLFI_C_U16_STARTUPFAILUREINDICATION );
   vUpRegFid( LCM_DBUSCTRLFI_C_U16_SUPERVISIONSTATECHANGE );

}

tVoid spm_tclSpmLateServiceHandler::vHandleStartProcessMResult( amt_tclServiceData *poMessage ){
   tU8 u8OpCode = poMessage->u8GetOpCode( );

   if ( u8OpCode == AMT_C_U8_CCAMSG_OPCODE_METHODRESULT ){
      fi_tclVisitorMessage                          oMsg( poMessage ); // create FI message from original message
      lcm_dbusctrlfi_tclMsgStartProcessMethodResult oMResult;

      (void) oMsg.s32GetData( oMResult );
   }
} // vHandleStartProcessMResult

tVoid spm_tclSpmLateServiceHandler::vHandleGetProcessIdMResult( amt_tclServiceData *poMessage ){
   tU8 u8OpCode = poMessage->u8GetOpCode( );

   if ( u8OpCode == AMT_C_U8_CCAMSG_OPCODE_METHODRESULT ){
      fi_tclVisitorMessage                          oMsg( poMessage ); // create FI message from original message
      lcm_dbusctrlfi_tclMsgGetProcessIdMethodResult oMResult;
      // According to lcm_dbusctl_fi.pdf the OPType MethodResult doesn't support.
      (void) oMsg.s32GetData( oMResult );
   } else {
      ETG_TRACE_ERRMEM( ( "spm_tclSpmLateServiceHandler::vHandleGetProcessIdMResult u8OpCode is not METHODRESULT" ) );
   }
} // vHandleGetProcessIdMResult

tVoid spm_tclSpmLateServiceHandler::vHandleAppStartReqUpdate( amt_tclServiceData *poMessage ){
   tU8 u8OpCode = poMessage->u8GetOpCode( );

   if ( u8OpCode == AMT_C_U8_CCAMSG_OPCODE_STATUS ){
      fi_tclVisitorMessage                       oMsg( poMessage ); // create FI message from original message
      lcm_dbusctrlfi_tclMsgAppStartRequestStatus oStatus;

      if ( OSAL_OK == oMsg.s32GetData( oStatus ) ) {
         if ( oStatus.u32AppId != 0 ){
            SPM_GET_IF_REFERENCE_NEW_VAR( poclLocalAppManager, ISpmLocalAppManager );
            SPM_GET_IF_REFERENCE_NEW_VAR( poclStartupCommon,   ISpmStartupCommon );
            SPM_GET_IF_REFERENCE_NEW_VAR( poclDatabase,        ISpmApplicationDatabase );

            spm_tclSoftwareBlock *poSwBlock = poclDatabase->poGetSoftwareBlock( oStatus.strSwBlockName.szValue );
            if ( poSwBlock == NULL ){
               poclStartupCommon->bAddAdditionalSwBlock( oStatus.strSwBlockName.szValue, oStatus.strServiceName.szValue, oStatus.u32AppId, SPM_STARTUP_VALUE_SYSTEMD_START_TYPE );
               poSwBlock = poclDatabase->poGetSoftwareBlock( oStatus.strSwBlockName.szValue );
            }
            if ( poSwBlock != NULL ){
               poSwBlock->vSetWholeBlockLoaded( );
               poclLocalAppManager->vOnWholeSwBlockIsLoaded( oStatus.strSwBlockName.szValue );
            } else {
               ETG_TRACE_ERRMEM( ( "No SW block found for name %s", oStatus.strSwBlockName.szValue ) );
            }
            poclLocalAppManager->bHandleAppStartReq( oStatus.u32AppId );
         }
      } else {
         ETG_TRACE_ERRMEM( ( "Could not read Fi-Msg element AppStartRequestStatus" ) );
      }
   }
} // vHandleAppStartReqUpdate

tVoid spm_tclSpmLateServiceHandler::vHandleAppInitUpdate( amt_tclServiceData *poMessage ){
   tU8 u8OpCode = poMessage->u8GetOpCode( );

   if ( u8OpCode == AMT_C_U8_CCAMSG_OPCODE_STATUS ){
      fi_tclVisitorMessage                      oMsg( poMessage ); // create FI message from original message
      lcm_dbusctrlfi_tclMsgAppInitializedStatus oStatus;

      if ( OSAL_OK == oMsg.s32GetData( oStatus ) ) {
         if ( oStatus.u32AppId != 0 ){
            SPM_GET_IF_REFERENCE_NEW_VAR( poclLocalAppManager, ISpmLocalAppManager );
            poclLocalAppManager->bHandleAppInitialized( oStatus.u32AppId );
         }
      } else {
         ETG_TRACE_ERRMEM( ( "Could not read Fi-Msg element AppStateChangeStatus" ) );
      }
      // todo, ksm2hi : Write EM-Trace in case of oMsg.s32GetData() returns OSAL_ERROR.
      //                This is also true for all other occurrences of oMsg.s32GetData() in this file.
   } else {
      ETG_TRACE_ERRMEM( ( "ERROR: vHandleAppInitUpdate() wrong OPCODE (%d)", ETG_ENUM( CCA_OPCODE, u8OpCode ) ) );
   }
} // vHandleAppInitUpdate

tVoid spm_tclSpmLateServiceHandler::vHandleAppStateChangeUpdate( amt_tclServiceData *poMessage ){
   tU8 u8OpCode = poMessage->u8GetOpCode( );

   if ( u8OpCode == AMT_C_U8_CCAMSG_OPCODE_STATUS ){
      fi_tclVisitorMessage                      oMsg( poMessage ); // create FI message from original message
      lcm_dbusctrlfi_tclMsgAppStateChangeStatus oStatus;

      if ( OSAL_OK == oMsg.s32GetData( oStatus ) ) {
         if ( oStatus.u32AppId != 0 ){

            SPM_GET_IF_REFERENCE_NEW_VAR( poclLocalAppManager, ISpmLocalAppManager );
            poclLocalAppManager->bHandleAppStateChangeAck( oStatus.u32AppId, oStatus.u32State, 0 );
         }
      } else {
         ETG_TRACE_ERRMEM( ( "Could not read Fi-Msg element AppStateChangeStatus" ) );
      }
   } else {
      ETG_TRACE_ERRMEM( ( "ERROR: vHandleAppStateChangeUpdate() wrong OPCODE (%d)", ETG_ENUM( CCA_OPCODE, u8OpCode ) ) );
   }
} // vHandleAppStateChangeUpdate

tVoid spm_tclSpmLateServiceHandler::vHandleNewServiceLoadedUpdate( amt_tclServiceData *poMessage ){
   tU8 u8OpCode = poMessage->u8GetOpCode( );

   if ( u8OpCode == AMT_C_U8_CCAMSG_OPCODE_STATUS ){
      fi_tclVisitorMessage                        oMsg( poMessage ); // create FI message from original message
      lcm_dbusctrlfi_tclMsgNewServiceLoadedStatus oStatus;

      if ( OSAL_OK == oMsg.s32GetData( oStatus ) ) {

         // add new app to database
         SPM_GET_IF_REFERENCE_NEW_VAR( poclStartupCommon, ISpmStartupCommon );
         SPM_GET_IF_REFERENCE_NEW_VAR( poAppCfg,          ISpmApplicationConfiguration );
         SPM_GET_IF_REFERENCE_NEW_VAR( _poclDatabase,     ISpmApplicationDatabase );

         std::string           strSwBlockName = oStatus.strBlockName.szValue;
         spm_tclSoftwareBlock *poSwBlock      = _poclDatabase->poGetSoftwareBlock( strSwBlockName );
         if ( poSwBlock == NULL ){
            // if this is a CCA application a SW should already exist
            if ( strSwBlockName != "none" ){
               // no bAddAdditionalSwBlock if this service is not configured in application
               // as also units would be added without LCM apps inside.
               tU32 u32AppId = oStatus.u32AppId;
               poclStartupCommon->bAddAdditionalSwBlock( strSwBlockName, oStatus.strServiceName.szValue, u32AppId, SPM_STARTUP_VALUE_SYSTEMD_START_TYPE );
               poSwBlock = _poclDatabase->poGetSoftwareBlock( strSwBlockName );
            }
         }
         if ( poSwBlock != NULL ){
            poSwBlock->vSetWholeBlockLoaded( );
            SPM_GET_IF_REFERENCE_NEW_VAR( poclLocalAppManager, ISpmLocalAppManager );
            poclLocalAppManager->vOnWholeSwBlockIsLoaded( strSwBlockName );
         } else {
            if ( oStatus.strServiceName.szValue[0] != 0 ){
               ETG_TRACE_ERRMEM( ( "No SW block '%40s' found for service '%40s'", strSwBlockName.c_str( ), oStatus.strServiceName.szValue ) );
            }
         }
      } else {
         ETG_TRACE_ERRMEM( ( "Could not read Fi-Msg element NewServiceLoadedStatus" ) );
      }
   } else {
            ETG_TRACE_ERRMEM( ( "ERROR: vHandleNewServiceLoadedUpdate() wrong OPCODE (%d)", ETG_ENUM( CCA_OPCODE, u8OpCode ) ) );
   }
} // vHandleNewServiceLoadedUpdate

tVoid spm_tclSpmLateServiceHandler::vHandleFailedServicesListUpdate( amt_tclServiceData *poMessage ){
   tU8 u8OpCode = poMessage->u8GetOpCode( );

   if ( u8OpCode == AMT_C_U8_CCAMSG_OPCODE_STATUS ){
      fi_tclVisitorMessage                          oMsg( poMessage ); // create FI message from original message
      lcm_dbusctrlfi_tclMsgFailedServicesListStatus oStatus;

      if ( OSAL_OK == oMsg.s32GetData( oStatus ) ) {

         _ServiceFailedEntryList.clear( );

         ::std::vector < lcm_fi_tcl_ServiceElement >::const_iterator cIterServices;
         for ( cIterServices = oStatus.tServices.tServices.begin( );
               cIterServices != oStatus.tServices.tServices.end( );
               ++cIterServices ){
            TServiceFailedEntry       ServiceFailedEntry;
            lcm_fi_tcl_ServiceElement ServiceElement = * cIterServices;
            ServiceFailedEntry.strServiceName        = ServiceElement.tUnitName.szValue;
            ServiceFailedEntry.strServiceDescription = ServiceElement.tDescription.szValue;
            ServiceFailedEntry.strServiceLoadState   = ServiceElement.tLoadState.szValue;
            ServiceFailedEntry.strServiceActiveState = ServiceElement.tActiveState.szValue;
            ServiceFailedEntry.strServiceSubState    = ServiceElement.tSubState.szValue;
            ServiceFailedEntry.strServiceFollower    = ServiceElement.tFollower.szValue;
            _ServiceFailedEntryList.push_back( ServiceFailedEntry );
         }

         ETG_TRACE_USR1( ( "spm_tclSpmLateServiceHandler::vHandleFailedServicesListUpdate() received %d failed services.", (tInt)oStatus.tServices.tServices.size( ) ) );
         SPM_GET_IF_REFERENCE_NEW_VAR( poclAppErrHandler, ISpmApplicationErrorHandler );
         poclAppErrHandler->vTriggerUnitFailedListReceived( );
      } else {
         ETG_TRACE_ERRMEM( ( "Could not read Fi-Msg element FailedServicesListStatus" ) );
      }
   } else {
      ETG_TRACE_ERRMEM( ( "ERROR: vHandleFailedServicesListUpdate() wrong OPCODE (%d)", ETG_ENUM( CCA_OPCODE, u8OpCode ) ) );
   }
} // vHandleFailedServicesListUpdate

tVoid spm_tclSpmLateServiceHandler::vHandleNewProcessIdUpdate( amt_tclServiceData *poMessage ){
   tU8 u8OpCode = poMessage->u8GetOpCode( );

   if ( u8OpCode == AMT_C_U8_CCAMSG_OPCODE_STATUS ){
      fi_tclVisitorMessage                    oMsg( poMessage ); // create FI message from original message
      lcm_dbusctrlfi_tclMsgNewProcessIdStatus oStatus;

      if ( OSAL_OK == oMsg.s32GetData( oStatus ) ) {

         ETG_TRACE_USR1( ( "vHandleNewProcessIdUpdate() ProcessId = %d for service %s",
                           oStatus.u32ProcessId,
                           oStatus.strServiceName.szValue ) );

         SPM_GET_IF_REFERENCE_NEW_VAR( poAppCfg,          ISpmApplicationConfiguration );
         std::string strSwBlockName = oStatus.strBlockName.szValue;

         SPM_GET_IF_REFERENCE_NEW_VAR( poclStartupCommon, ISpmStartupCommon );
         if ( strSwBlockName != SPM_APPLICATION_CONFIGURATION_NO_SWBLOCK ){
            poclStartupCommon->vSetProcId( strSwBlockName.c_str( ), oStatus.u32ProcessId );
         } else {
            const TMapProcessConfiguration         & oMapProcessConfig = poclStartupCommon->oGetProcessConfig( );
            TMapProcessConfiguration::const_iterator posProc;
            for ( posProc = oMapProcessConfig.begin( );
                  posProc != oMapProcessConfig.end( );
                  ++posProc ){
               std::string strServiceName1 = posProc->second.strProcessLocation;
               std::string strServiceName2 = std::string( oStatus.strServiceName.szValue );
               ETG_TRACE_USR4( ( "vHandleNewProcessIdUpdate() ->%40s<-, ->%40s<-", strServiceName1.c_str( ), strServiceName2.c_str( ) ) );
               if ( strServiceName1.compare( strServiceName2 ) == 0 ){
                  strSwBlockName = posProc->second.strProcessName;
                  poclStartupCommon->vSetProcId( strSwBlockName.c_str( ), oStatus.u32ProcessId );
                  break;
               }
            }
         }
         if ( strSwBlockName == SPM_APPLICATION_CONFIGURATION_NO_SWBLOCK ){
            ETG_TRACE_USR1( ( "vHandleNewProcessIdUpdate() no SW block found for service %s", oStatus.strServiceName.szValue ) );
         }
      } else {
         ETG_TRACE_ERRMEM( ( "Could not read Fi-Msg element NewProcessIdStatus" ) );
      }
   } else {
      ETG_TRACE_ERRMEM( ( "ERROR: vHandleNewProcessIdUpdate() wrong OPCODE (%d)", ETG_ENUM( CCA_OPCODE, u8OpCode ) ) );
   }
} // vHandleNewProcessIdUpdate

/**
  *  \brief called when StartupFailureIndication property of StartupController is updated
  *
  *  \param [in] poMessage contains StartupFailureInfo
  *  \return 
  *
  *  \details StartupInvestigation dumps FailureIndicaton to ErrMem
  */
tVoid spm_tclSpmLateServiceHandler::vHandleStartupFailureIndUpdate( amt_tclServiceData *poMessage ){
   tU8 u8OpCode = poMessage->u8GetOpCode( );
   if ( u8OpCode == AMT_C_U8_CCAMSG_OPCODE_STATUS ){
      fi_tclVisitorMessage                       oMsg( poMessage ); // create FI message from original message
      lcm_dbusctrlfi_tclMsgStartupFailureIndicationStatus oStatus;
      if ( OSAL_OK == oMsg.s32GetData( oStatus ) ) {
         const lcm_fi_tcl_StartupFailureInfo &failureInfo = oStatus.tStartupFailureInd;
         if ( failureInfo.tFailure.enType == lcm_fi_tcl_e32_LCM_STARTUP_FAILURE_TYPE::FI_EN_SPM_U32_STARTUP_FAILURE_NONE ) {
             return;
         }
         if( _poclStartupInvest != NULL ) {
             std::string StrFailureInd = strBuildStartupFailureMsg( failureInfo );
             ETG_TRACE_USR1( ( "spm_tclSpmLateServiceHandler::vHandleStartupFailureIndUpdate(), %s", StrFailureInd.c_str() ) );

             // add startup investigation information
             SPM_STARTUPINVEST_INIT_STARTUPITEM
             SPM_STARTUPINVEST_FORCE_ENTRY( "spm_tclSpmLateServiceHandler::vHandleStartupFailureIndUpdate", StrFailureInd )
             _poclStartupInvest->vDump();
         } else {
            ETG_TRACE_ERRMEM( ( "ERROR: vHandleStartupFailureIndUpdate() _poclStartupInvest is NULL " ) );
         }
      } else {
         ETG_TRACE_ERRMEM( ( "Could not read Fi-Msg element StartupFailureIndication" ) );
      }
   }
} // vHandleStartupFailureIndUpdate

/**
  *  \brief build StartupFailure message
  *
  *  \param [in] failureInfo contains StartupFailureInfo
  *  \return std::string of the failure message
  *
  *  \details build StartupFailure message as a string from failureInfo structure
  */
std::string spm_tclSpmLateServiceHandler::strBuildStartupFailureMsg( const lcm_fi_tcl_StartupFailureInfo &failureInfo ) {
    std::stringstream ret;
    ret << "StartupController failed, state: " << strGetStartupStatusString( failureInfo.tState );
    ret << ", Failure: " << strGetStartupFailureString( failureInfo.tFailure );
    switch( failureInfo.tFailure.enType ){
        case lcm_fi_tcl_e32_LCM_STARTUP_FAILURE_TYPE::FI_EN_SPM_U32_STARTUP_FAILURE_STARTUPTIME_AVAILABLE:
        case lcm_fi_tcl_e32_LCM_STARTUP_FAILURE_TYPE::FI_EN_SPM_U32_STARTUP_FAILURE_STARTUPTIME_EXPIRED:
        case lcm_fi_tcl_e32_LCM_STARTUP_FAILURE_TYPE::FI_EN_SPM_U32_STARTUP_FAILURE_DEBUGGING_STARTUPTIME_AVAILABLE:
        case lcm_fi_tcl_e32_LCM_STARTUP_FAILURE_TYPE::FI_EN_SPM_U32_STARTUP_FAILURE_DEBUGGING_STARTUPTIME_EXPIRED:
            ret << ", timer(second): ";
            break;
        case lcm_fi_tcl_e32_LCM_STARTUP_FAILURE_TYPE::FI_EN_SPM_U32_STARTUP_FAILURE_SHUTDOWN_FAILED:
            ret << ", pid: ";
            break;
        default:
            break;
    }
    ret << failureInfo.u32Data;
    return ret.str();
} // strBuildStartupFailureMsg

/**
  *  \brief get string value of StartupStatus
  *
  *  \param [in] type is an enum of StartupStatus
  *  \return std::string of StartupStatus
  *
  *  \details get string value of StartupStatus enum
  */
std::string spm_tclSpmLateServiceHandler::strGetStartupStatusString( const lcm_fi_tcl_e32_LCM_STARTUP_TYPE &type ) {
    std::string ret;
    switch( type.enType ) {
        case lcm_fi_tcl_e32_LCM_STARTUP_TYPE::FI_EN_SPM_U32_STARTUP_INIT:
            ret = "STARTUP_INIT";
            break;
        case lcm_fi_tcl_e32_LCM_STARTUP_TYPE::FI_EN_SPM_U32_STARTUP_ACTIVE:
            ret = "STARTUP_ACTIVE";
            break;
        case lcm_fi_tcl_e32_LCM_STARTUP_TYPE::FI_EN_SPM_U32_STARTUP_DONE:
            ret = "STARTUP_DONE";
            break;
        default:
            ret = "UNKNOWN";
            break;
    }
    return ret;
} // strGetStartupStatusString

/**
  *  \brief get string value of StartupFailure
  *
  *  \param [in] type is an enum of StartupFailure
  *  \return std::string of StartupFailure
  *
  *  \details get string value of StartupFailure enum
  */
std::string spm_tclSpmLateServiceHandler::strGetStartupFailureString( const lcm_fi_tcl_e32_LCM_STARTUP_FAILURE_TYPE &type ) {
    std::string ret;
    switch( type.enType ) {
        case lcm_fi_tcl_e32_LCM_STARTUP_FAILURE_TYPE::FI_EN_SPM_U32_STARTUP_FAILURE_STARTUPTIME_AVAILABLE:
            ret = "TimeoutStartSec=<sec>' is used";
            break;
        case lcm_fi_tcl_e32_LCM_STARTUP_FAILURE_TYPE::FI_EN_SPM_U32_STARTUP_FAILURE_STARTUPTIME_EXPIRED:
            ret = "Timer 'TimeoutStartSec=<sec>' expired";
            break;
        case lcm_fi_tcl_e32_LCM_STARTUP_FAILURE_TYPE::FI_EN_SPM_U32_STARTUP_FAILURE_SHUTDOWN_FAILED:
            ret = "Shutdown failed";
            break;
        case lcm_fi_tcl_e32_LCM_STARTUP_FAILURE_TYPE::FI_EN_SPM_U32_STARTUP_FAILURE_DEBUGGING_STARTUPTIME_AVAILABLE:
            ret = "'StartupTime=<sec>' is used";
            break;
        case lcm_fi_tcl_e32_LCM_STARTUP_FAILURE_TYPE::FI_EN_SPM_U32_STARTUP_FAILURE_DEBUGGING_STARTUPTIME_EXPIRED:
            ret = "Timer 'StartupTime=<sec>' expired";
            break;
        default:
            ret = "UNKNOWN";
            break;
    }
    return ret;
} // strGetStartupFailureString

tS32 spm_tclSpmLateServiceHandler::s32AppStateChange( tU32 u32AppId,
                                                      tU32 u32State ){
   tS32 s32Ret = - 1;

   SPM_NULL_POINTER_CHECK_VAL( _poclCcaMsgHandler );

   if ( _u16RegID != 0 ){
      lcm_dbusctrlfi_tclMsgAppStateChangeSet oSet;

      oSet.u32AppId = u32AppId;
      oSet.u32State = u32State;

      ETG_TRACE_USR1( ( "spm_tclSpmLateServiceHandler::s32AppStateChange: %u %u", ETG_ENUM( ail_u16AppId, oSet.u32AppId ), ETG_ENUM( SPM_APPSTATE_DEF, oSet.u32State ) ) );

      fi_tclVisitorMessage oSlvMsg( oSet );
      oSlvMsg.vInitServiceData( CCA_C_U16_APP_SPM,
                                _u16AppId,
                                AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,
                                0,
                                _u16RegID,
                                0,
                                _u16SrvId,
                                LCM_DBUSCTRLFI_C_U16_APPSTATECHANGE,
                                AMT_C_U8_CCAMSG_OPCODE_SET );

      (tVoid)_poclCcaMsgHandler->bPostMessage( &oSlvMsg );
   }

   return( s32Ret );
} // s32AppStateChange

tVoid spm_tclSpmLateServiceHandler::vGetProcessId( std::string strServiceName ){
   if ( strServiceName != "" ) {
      lcm_dbusctrlfi_tclMsgGetProcessIdMethodStart oMStart;
      oMStart.strServiceName = strServiceName.c_str( );

      fi_tclVisitorMessage oSlvMsg( oMStart );
      oSlvMsg.vInitServiceData( CCA_C_U16_APP_SPM,
                              _u16AppId,
                              AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,
                              0,
                              _u16RegID,
                              0,
                              _u16SrvId,
                              LCM_DBUSCTRLFI_C_U16_GETPROCESSID,
                              AMT_C_U8_CCAMSG_OPCODE_METHODSTART );

      (tVoid)_poclCcaMsgHandler->bPostMessage( &oSlvMsg );
   }
} // vGetProcessId

tVoid spm_tclSpmLateServiceHandler::vStartupActivated( ){
   SPM_NULL_POINTER_CHECK( _poclCcaMsgHandler );

   if ( _u16RegID != 0 ){
      lcm_dbusctrlfi_tclMsgStartupFinishedMethodStart oMStart;

      oMStart.eStartUpParam.enType = lcm_fi_tcl_e32_LCM_STARTUP_TYPE::FI_EN_SPM_U32_STARTUP_ACTIVE;

      ETG_TRACE_USR1( ( "spm_tclSpmLateServiceHandler::vStartupActivated()" ) );

      fi_tclVisitorMessage oSlvMsg( oMStart );
      oSlvMsg.vInitServiceData( CCA_C_U16_APP_SPM,
                                _u16AppId,
                                AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,
                                0,
                                _u16RegID,
                                0,
                                _u16SrvId,
                                LCM_DBUSCTRLFI_C_U16_STARTUPFINISHED,
                                AMT_C_U8_CCAMSG_OPCODE_METHODSTART );

      (tVoid)_poclCcaMsgHandler->bPostMessage( &oSlvMsg );
   }

} // vStartupActivated

/**
  *  \brief called when the list of failed units needs to be evaluated.
  *
  *  \return void
  *
  *  \details sends the GetFailedUnitList message
  *       Result is written to ErrMem, only successful writing is acknowledged
  */
tVoid spm_tclSpmLateServiceHandler::vTriggerGetFailedUnitList( ){
   SPM_NULL_POINTER_CHECK( _poclCcaMsgHandler );

   if ( _u16RegID != 0 ){
      lcm_dbusctrlfi_tclMsgTriggerGetUnitListMethodStart oMStart;

      ETG_TRACE_USR1( ( "spm_tclSpmLateServiceHandler::vTriggerGetFailedUnitList()" ) );

      fi_tclVisitorMessage                               oSlvMsg( oMStart );
      oSlvMsg.vInitServiceData( CCA_C_U16_APP_SPM,
                                _u16AppId,
                                AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,
                                0,
                                _u16RegID,
                                0,
                                _u16SrvId,
                                LCM_DBUSCTRLFI_C_U16_TRIGGERGETUNITLIST,
                                AMT_C_U8_CCAMSG_OPCODE_METHODSTART );

      (tVoid)_poclCcaMsgHandler->bPostMessage( &oSlvMsg );
   } else {
      ETG_TRACE_USR1( ( "spm_tclSpmLateServiceHandler::vTriggerGetFailedUnitList() failed. No ProcLcm registered yet" ) );
   }
} // vTriggerGetFailedUnitList

/**
  *  \brief called when startup is done - all processes have been started or there are no processes need to be started
  *
  *  \return TRUE if sending the startupFinished message successfully
  *
  *  \details sends the startupFinished message
  */
tBool spm_tclSpmLateServiceHandler::bStartupReady( ){
    SPM_NULL_POINTER_CHECK_VAL( _poclCcaMsgHandler );
    tBool bSuccess = FALSE;

   if ( _u16RegID != 0 ){
      lcm_dbusctrlfi_tclMsgStartupFinishedMethodStart oMStart;

      oMStart.eStartUpParam.enType = lcm_fi_tcl_e32_LCM_STARTUP_TYPE::FI_EN_SPM_U32_STARTUP_DONE;

      ETG_TRACE_USR1( ( "spm_tclSpmLateServiceHandler::bStartupReady()" ) );

      fi_tclVisitorMessage oSlvMsg( oMStart );
      oSlvMsg.vInitServiceData( CCA_C_U16_APP_SPM,
                                _u16AppId,
                                AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,
                                0,
                                _u16RegID,
                                0,
                                _u16SrvId,
                                LCM_DBUSCTRLFI_C_U16_STARTUPFINISHED,
                                AMT_C_U8_CCAMSG_OPCODE_METHODSTART );

      bSuccess = _poclCcaMsgHandler->bPostMessage( &oSlvMsg );
      if ( !bSuccess ) {
          ETG_TRACE_ERRMEM( ( "spm_tclSpmLateServiceHandler::bStartupReady( ) !!!!!! Error detected !!!!!! Sending startupFinished message returns FALSE" ) );
      }
   } else {
        ETG_TRACE_ERRMEM( ( "spm_tclSpmLateServiceHandler::bStartupReady( ) !!!!!! Error detected !!!!!! LATE SERVICE is not available (_u16RegID == 0)" ) );
   }

    return bSuccess;
} // bStartupReady

/**
  *  \brief check for procbaselate is start-up
  *
  *  \return TRUE if procbaselate starts up successfully
  *
  */
tBool spm_tclSpmLateServiceHandler::bIsLcmLateStarted( ){
   return _bLcmLateStarted;
} // bIsLcmLateStarted

/**
  *  \brief request to start a process
  *
  *  \param [in] strLocation service vile name or path to executable
  *  \param [in] u32Type OSAL/SYSTEMD
  *  \param [in] u32NiceLevel nicelevel at start-up
  *  \param [in] u32NiceLevelEnd nicelevel after startup
  *  \param [in] u32Prio priority of thread - currently not used
  *  \param [in] u32Affinity ??
  *  \param [in] u32AffinityEnd ??
  *  \return not defined yet
  *
  *  \details creates and sends the StartProcess request to LCM Late.
  *  \todo define return value correctly
  */
tS32 spm_tclSpmLateServiceHandler::s32StartProcess( const std::string& strLocation,
                                                    tU32               u32Type,
                                                    tU32               u32NiceLevel,
                                                    tU32               u32NiceLevelEnd,
                                                    tU32               u32Prio,
                                                    tU32               u32Affinity,
                                                    tU32               u32AffinityEnd ){

   tS32 s32Ret = - 1;

   SPM_NULL_POINTER_CHECK_VAL( _poclCcaMsgHandler );

   while ( _u16RegID == 0 ){
      OSAL_s32ThreadWait( 500 );
   }
   lcm_dbusctrlfi_tclMsgStartProcessMethodStart oStartProc;

   oStartProc.strLocation     = strLocation.c_str( );
   oStartProc.u32Type.enType  = (lcm_fi_tcl_e32_LCM_START_PROC_TYPE::tenType )u32Type;
   oStartProc.u32NiceLevel    = u32NiceLevel;
   oStartProc.u32NiceLevelEnd = u32NiceLevelEnd;
   oStartProc.u32Prio         = u32Prio;
   oStartProc.u32Affinity     = u32Affinity;
   oStartProc.u32AffinityEnd  = u32AffinityEnd;
   oStartProc.u32Info         = 0;

   ETG_TRACE_USR1( ( "spm_tclSpmLateServiceHandler::bStartProcess: Type: %d, Nice: %d, NiceEnd: %d, Location: '%s':",
                     ETG_ENUM( SPM_START_PROC_TYPE, u32Type),
                     (tS32)oStartProc.u32NiceLevel,
                     (tS32)oStartProc.u32NiceLevelEnd,
                     oStartProc.strLocation.szValue ) );

   fi_tclVisitorMessage oSlvMsg( oStartProc );
   oSlvMsg.vInitServiceData( CCA_C_U16_APP_SPM,
                             _u16AppId,
                             AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,
                             0,
                             _u16RegID,
                             0,
                             _u16SrvId,
                             LCM_DBUSCTRLFI_C_U16_STARTPROCESS,
                             AMT_C_U8_CCAMSG_OPCODE_METHODSTART );

   (tVoid)_poclCcaMsgHandler->bPostMessage( &oSlvMsg );
   return( s32Ret );
} // s32StartProcess

/**
  *  \brief request to stop a process
  *
  *  \param [in] strLocation service vile name or path to executable
  *  \param [in] u32ProcId ID of the process (defaults to 0)
  *  \return not defined yet
  *
  *  \details creates and sends the StopProcess request to LCM Late.
  *  \todo define return value correctly
  */
tBool spm_tclSpmLateServiceHandler::bStopProcess( const std::string& strLocation,
                                                  tU32               u32ProcId ){

   tBool                                       bRet = true;

   SPM_NULL_POINTER_CHECK_VAL( _poclCcaMsgHandler );

   lcm_dbusctrlfi_tclMsgStopProcessMethodStart oStopProc;

   oStopProc.strLocation = strLocation.c_str( );
   oStopProc.u32ProcId   = u32ProcId;

   ETG_TRACE_USR1( ( "spm_tclSpmLateServiceHandler::bStopProcess: u32ProcId: %d, Location: '%s':",
                     oStopProc.u32ProcId,
                     oStopProc.strLocation.szValue ) );

   fi_tclVisitorMessage oSlvMsg( oStopProc );
   oSlvMsg.vInitServiceData( CCA_C_U16_APP_SPM,
                             _u16AppId,
                             AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,
                             0,
                             _u16RegID,
                             0,
                             _u16SrvId,
                             LCM_DBUSCTRLFI_C_U16_STOPPROCESS,
                             AMT_C_U8_CCAMSG_OPCODE_METHODSTART );

   bRet = _poclCcaMsgHandler->bPostMessage( &oSlvMsg );
   return( bRet );
} // bStopProcess
/**
  *  \brief request to restart a process
  *
  *  \param [in] strLocation service vile name or path to executable
  *  \param [in] u32ProcId ID of the process (defaults to 0)
  *  \return true if message sent to procbaselate else it return false
  *
  *  \details creates and sends the RestartProcess request to LCM Late
  */
tBool spm_tclSpmLateServiceHandler::bRestartProcess( const std::string& strLocation,
                                                  tU32               u32ProcId ){

   tBool                                       bRet = true;

   SPM_NULL_POINTER_CHECK_VAL( _poclCcaMsgHandler );

   lcm_dbusctrlfi_tclMsgRestartProcessMethodStart oRestartProc;

   oRestartProc.strLocation = strLocation.c_str( );
   oRestartProc.u32ProcId   = u32ProcId;

   ETG_TRACE_USR1( ( "spm_tclSpmLateServiceHandler::bRestartProcess: u32ProcId: %d, Location: '%s':",
                     oRestartProc.u32ProcId,
                     oRestartProc.strLocation.szValue ) );

   fi_tclVisitorMessage oSlvMsg( oRestartProc );
   oSlvMsg.vInitServiceData( CCA_C_U16_APP_SPM,
                             _u16AppId,
                             AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,
                             0,
                             _u16RegID,
                             0,
                             _u16SrvId,
                             LCM_DBUSCTRLFI_C_U16_RESTARTPROCESS,
                             AMT_C_U8_CCAMSG_OPCODE_METHODSTART );

   bRet = _poclCcaMsgHandler->bPostMessage( &oSlvMsg );
   return( bRet );
} // bRestartProcess

/**
  *  \brief inform new value of Supervision from early to late
  *
  *  \param [in] bSupervisionActive value of Supervision is active or inactive
  *  \return void
  *
  *  \details  Check if bIsLcmLateStarted then send message to LCM late otherwise hold the message and then 
  *   vOnFirstServiceStateAvailable will call this function again when LCM late is ready.
  *  
  */
tVoid spm_tclSpmLateServiceHandler::vSupervisionStateChange( tBool               bSupervisionActive ){

   tBool                                       bRet = true;

   ETG_TRACE_USR4( ( "spm_tclSpmLateServiceHandler::vSupervisionStateChange: bSupervisionActive: %d",
                        ETG_ENUM( SPM_STATE_COMMON, bSupervisionActive) ) );

   SPM_NULL_POINTER_CHECK( _poclCcaMsgHandler );
   if(bIsLcmLateStarted())
   {
      if ( _u16RegID != 0 ){

         lcm_dbusctrlfi_tclMsgSupervisionStateChangeSet oStateChange;

         oStateChange.bSupervisionActive   = bSupervisionActive;

         fi_tclVisitorMessage oSlvMsg( oStateChange );
         
         oSlvMsg.vInitServiceData( CCA_C_U16_APP_SPM,
                                   _u16AppId,
                                   AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,
                                   0,
                                   _u16RegID,
                                   0,
                                   _u16SrvId,
                                   LCM_DBUSCTRLFI_C_U16_SUPERVISIONSTATECHANGE,
                                   AMT_C_U8_CCAMSG_OPCODE_SET );

         bRet = _poclCcaMsgHandler->bPostMessage( &oSlvMsg );
         if(bRet != true)
         {
            ETG_TRACE_ERRMEM( ( "spm_tclSpmLateServiceHandler::vSupervisionStateChange( ) !!!!!! Error detected !!!!!! Sending SupervisionStateChange message returns FALSE" ) );
         }
      }
      else {
         ETG_TRACE_USR4( ( "spm_tclSpmLateServiceHandler::vSupervisionStateChange( ) !!!!!! Error detected !!!!!! LATE SERVICE is not available (_u16RegID == 0)" ) );
      }
   }
   else {
      ETG_TRACE_USR4( ("LcmLateSerice is not ready yet, message will be saved and sent when lcmLateService is ready" ) );
      _bMessageLeftToBeSend = TRUE;
      _bSupervisionActive = bSupervisionActive;
   }
} // vSupervisionStateChange

// EOF

