/*!
  * \file spm_GlobalApplicationManager.cpp
  *  \brief
  *    This class control and configure different state of application.
  *
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  *  Date      | Author               | Modification
  * 20.01.12  | CM-AI/PJVW32 Kollai  | initial version
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

// SPM  configuration
#include "spm_Config.h"

// my class header
#include "spm_GlobalApplicationManager.h"

// interfaces class definitions
#include "spm_ISystemStateManager.h"
#include "spm_ISubStateClient.h"
#include "spm_ISystemPowerManager.h"
#include "spm_ICcaServiceServer.h"
#include "spm_ILamAppManager.h"
#include "spm_ILocalAppManager.h"

#include "spm_factory.h"
#include "spm_ILateServiceHandler.h"
#include "spm_IStartupCommon.h"

// spm helper

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM_LAM
#include "trcGenProj/Header/spm_GlobalApplicationManager.cpp.trc.h"
#endif

// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

#include "spm_rootdaemon.h"

/******************************************************************************
 | local #define (scope: module-local)
 |----------------------------------------------------------------------------*/
#ifndef SPM_PRJ_SYSTEM_STATE_DOWNLOAD
   #define SPM_PRJ_SYSTEM_STATE_DOWNLOAD    SPM_SYSTEM_DOWNLOAD
#endif

// #define SPM_TRACE_FILE_ID   SPM_FILE_GLOBALAPPLICATIONMANAGER

/**
  *  \brief constructor. initialize all variables
  *
  *  \param [in] factory: reference to LCM factory component
  *  \return
  */
spm_tclGlobalApplicationManager::spm_tclGlobalApplicationManager( const ISpmFactory& factory )
   : ISpmGlobalApplicationManager( factory ){
   _u32SystemState         = spm_fi_tcl_SPM_e32_SYSTEM_STATES::FI_EN_SPM_SYSTEM_ON;

   _u32SubState            = 0;

   _poclCcaServiceHandler  = NULL;
   _poclSubStateHandler    = NULL;
   _poclSystemPowerManager = NULL;
   _poclWorkerServer       = NULL;
   _poclAppManagerConfig   = NULL;
}

/**
  *  \brief destructor. Remove all connections to other LCM modules
  *
  *  \param [in]
  *  \return
  */
spm_tclGlobalApplicationManager::~spm_tclGlobalApplicationManager( ){
   if ( _poclWorkerServer != NULL ){
      _poclWorkerServer->vRemoveClient( this );
   }
   _poclCcaServiceHandler  = NULL;
   _poclSubStateHandler    = NULL;
   _poclSystemPowerManager = NULL;
   _poclWorkerServer       = NULL;
   _poclAppManagerConfig   = NULL;
}

/**
  *  \brief called from Factory to connect with other LCM modules
  *
  *  \param [in]
  *  \return
  */
tVoid spm_tclGlobalApplicationManager::vGetReferences( ){
   // get all needed references now -> all SPM objects are now available
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSystemPowerManager, ISpmSystemPowerManager );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclWorkerServer,       ISpmWorkerServer );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSubStateHandler,    ISpmSubStateClient );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclCcaServiceHandler,  ISpmCcaServiceServer );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclAppManagerConfig,   ISpmLamAppManager );
} // vGetReferences

/**
  *  \brief called from Factory to allow/start communication with other LCM modules
  *
  *  \param [in]
  *  \return
  */
tVoid spm_tclGlobalApplicationManager::vStartCommunication( ){
/*!
  * \fn
  *  \brief
  *    Synchronize the startup. Until now the FC_SPM is fully running.
  *
  *  \version
  *    1.0   - Initial
  ******
  */
   SPM_NULL_POINTER_CHECK( _poclWorkerServer );
   _poclWorkerServer->vAddClient( this );

   dp_tclSpmDpInternDataSystemState oNewSysState;
   if ( FALSE == bProcessSystemState( oNewSysState.tGetData( ) ) ){
      ETG_TRACE_COMP( ( "SPM: !!!!!! Warning detected !!!!!!" ) );
   }
}

/**
  *  \brief called from Factory to link the LAM to GlobalApplicationManager
  *
  *  \param [in] u32LamConfigId ID for the ApplicationManager (0 for local)
  *  \param [in] poRefLam pointer to instance of ApplicationManager
  *  \return always TRUE
  */
tBool spm_tclGlobalApplicationManager::bAddNewLam( tU32                 u32LamConfigId,
                                                   ISpmLocalAppManager *poRefLam ){
   _mapLam[u32LamConfigId] = poRefLam;
   return( TRUE );
}

tVoid spm_tclGlobalApplicationManager::vForceIntermediateState( tU32  u32Profile,
                                                                tBool bForceAll ){
   std::map < tU32, TLamState >::iterator it;

   for ( it = _mapLam.begin( ); it != _mapLam.end( ); ++it ){
      it->second._poRefLam->bForceAllApplicationsToProfile( u32Profile, bForceAll );
      it->second._poRefLam->bSetAllApplicationsInProfile( );
   }
}

/**
  *  \brief called when a system state is reached. It implements generic actions to be taken for a certain system state
  *  and calls the project specific function
  *
  *  \param [in] u32NewSystemState old system state
  *  \param [in] u32SystemStateReached new system state
  *  \return always TRUE
  *  \todo sync handling of service files - same as done for SW-Blocks
  *  \msc
  *   Fsm,GlobalApplicationManager,AppManagerConfig;
  *   Fsm->GlobalApplicationManager [label="bProcessSystemState()", URL="\ref GlobalApplicationManager::bProcessSystemState()"];
  *   GlobalApplicationManager->AppManagerConfig [label="vProcessSystemState()", URL="\ref vProcessSystemState()", ID="1"];
  *  \endmsc
  *
  *  \details implements the killing and starting of processes when DOWNLOAD state is reached and calls function
  *  vProcessSystemState to allow projects to implement specific handling
  */
tBool spm_tclGlobalApplicationManager::bProcessSystemState( tU32 u32NewSystemState,
                                                            tU32 u32SystemStateReached ){
   if ( u32SystemStateReached == SPM_PRJ_SYSTEM_STATE_DOWNLOAD ){
      #ifdef SPM_DOWNLOAD_SERVICES_2_STOP
         {
            const std::string astrServices2Stop[] = { SPM_DOWNLOAD_SERVICES_2_STOP };
            tU8               u8ArrayCount        = ELEMENTE( astrServices2Stop );

                  ETG_TRACE_USR1( ( "spm_tclGlobalApplicationManager::bProcessSystemState(): We are in state DOWNLOAD --> STOP %d services.", u8ArrayCount ) );

            for ( tU8 i = 0; i < u8ArrayCount; i++ ){
               ETG_TRACE_USR1( ( "spm_tclGlobalApplicationManager::bProcessSystemState(): STOP service '%s' now.", astrServices2Stop[i].c_str( ) ) );
               SPM_GET_CLIENT_HANDLER_IF_REFERENCE_NEW_VAR_VAL(poLateSrvHandler, spm_ISpmLateServiceHandler);
               ETG_TRACE_FATAL( ( "spm_tclGlobalApplicationManager::bProcessSystemState(): address %p", poLateSrvHandler ) );
               poLateSrvHandler->bStopProcess( astrServices2Stop[i].c_str( ) );
               ETG_TRACE_USR1( ( "spm_tclGlobalApplicationManager::bProcessSystemState(): poLateSrvHandler %s", astrServices2Stop[i].c_str( ) ) );
               // Get reference to spm_tclStartupCommon object
               SPM_GET_IF_REFERENCE_NEW_VAR_VAL( poclStartupCommon, ISpmStartupCommon );
               poclStartupCommon->vRemoveStoppedService( static_cast< std::string >( astrServices2Stop[i] ) );
            }
         }
               #endif // SPM_DOWNLOAD_SERVICES_2_STOP
               #ifdef SPM_DOWNLOAD_SERVICES_2_START
                  {
                     const std::string astrServices2Start[] = { SPM_DOWNLOAD_SERVICES_2_START };
                     tU8               u8ArrayCount         = ELEMENTE( astrServices2Start );

                           ETG_TRACE_USR1( ( "spm_tclGlobalApplicationManager::bProcessSystemState(): We are in state DOWNLOAD --> START %d services.", u8ArrayCount ) );

                     for ( tU8 i = 0; i < u8ArrayCount; i++ ){
                        ETG_TRACE_USR1( ( "spm_tclGlobalApplicationManager::bProcessSystemState(): START service '%s' now.", astrServices2Start[i].c_str( ) ) );
                        SPM_GET_CLIENT_HANDLER_IF_REFERENCE_NEW_VAR_VAL(poLateSrvHandler, spm_ISpmLateServiceHandler);
                        ETG_TRACE_FATAL( ( "spm_tclGlobalApplicationManager::bProcessSystemState(): address %p", poLateSrvHandler ) );
                        poLateSrvHandler->s32StartProcess( astrServices2Start[i].c_str( ), ( tU32 ) spm_fi_tcl_SPM_e32_START_PROC_TYPE::FI_EN_SPM_U32_START_PROC_TYPE_SYSTEMD );
                        // Get reference to spm_tclStartupCommon object
                        SPM_GET_IF_REFERENCE_NEW_VAR_VAL( poclStartupCommon, ISpmStartupCommon );
                        poclStartupCommon->vUpdateStartedService( static_cast< std::string >( astrServices2Start[i] ) );
                     }
                  }
                        #endif // SPM_DOWNLOAD_SERVICES_2_START
                        #ifdef SPM_DOWNLOAD_PROCS_2_KILL
                           {
                              std::map < tU32, TLamState >::iterator it;
                              for ( it = _mapLam.begin( ); it != _mapLam.end( ); ++it ){
                                 const std::string astrProcs2Kill[] = { SPM_DOWNLOAD_PROCS_2_KILL };
                                 tU8               u8ArrayCount     = ELEMENTE( astrProcs2Kill );

                                    ETG_TRACE_USR1( ( "spm_tclGlobalApplicationManager::bProcessSystemState(): We are in state DOWNLOAD --> KILL %d processes.", u8ArrayCount ) );

                                 for ( tU8 i = 0; i < u8ArrayCount; i++ ){
                                    ETG_TRACE_USR1( ( "spm_tclGlobalApplicationManager::bProcessSystemState(): terminate process '%s' now.", astrProcs2Kill[i].c_str( ) ) );
                                    it->second._poRefLam->vKillSoftwareBlocksOfProcess( astrProcs2Kill[i] );
                                 }
                              }
                           }
                        #else
                           SPM_NULL_POINTER_CHECK_VAL( _poclWorkerServer );
                           _poclWorkerServer->bPostMessage( "ISpmCcaServiceServer", SPM_U32_WORKER_CSH_SYSTEM_MODE_DL );
                        #endif // ifdef SPM_DOWNLOAD_PROCS_2_KILL
   }

   if ( _poclAppManagerConfig != NULL ){
      _poclAppManagerConfig->vProcessSystemState( u32NewSystemState, u32SystemStateReached );
   }
   return( TRUE );
} // bProcessSystemState

tVoid spm_tclGlobalApplicationManager::vTriggerAppStateChange( TTriggerMsg *tNewTrigger ){
   tBool bReturn;

   SPM_GET_IF_REFERENCE_NEW_VAR( poclSystemStateManager, ISpmSystemStateManager );

   if ( _poclAppManagerConfig != NULL ){
      _poclAppManagerConfig->vNewSubState( tNewTrigger );
   }
   std::map < tU32, TLamState >::iterator it;

   for ( it = _mapLam.begin( ); it != _mapLam.end( ); ++it ){
      it->second._u8State = SPM_GAM_LAM_STATE_CHANGE_IN_PROGRESS;
      bReturn             = it->second._poRefLam->bSetAllApplicationsInProfile( );
      if ( !bReturn ){
         ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
      }
   }

   // and now check if all requests are already processed
   tBool bAllRequestsProcessed = TRUE;
   for ( it = _mapLam.begin( ); it != _mapLam.end( ); ++it ){
      if ( it->second._poRefLam->bAllRequestsProcessed( ) ){
         it->second._u8State = SPM_GAM_LAM_STATE_CHANGE_PROCESSED;
      } else {
         bAllRequestsProcessed = FALSE;
      }
   }

   if ( bAllRequestsProcessed ){
      // reset initialization control timer for navi

      dp_tclSpmDpInternDataSystemState        oNewSysState;
      dp_tclSpmDpInternDataCurrentSystemState oCurSysState;

      if ( oNewSysState.tGetData( ) != oCurSysState.tGetData( ) ){
         poclSystemStateManager->vNewSystemStateReached( );

         if ( _poclAppManagerConfig != NULL ){
            _poclAppManagerConfig->vNewAppStateReached( );
         }
      }
   }
} // vTriggerAppStateChange

tBool spm_tclGlobalApplicationManager::bShutdownApplications( ){
   tBool                                  bReturn = FALSE;

   std::map < tU32, TLamState >::iterator it;

   for ( it = _mapLam.begin( ); it != _mapLam.end( ); ++it ){
      bReturn = it->second._poRefLam->bKillAllApplications( );
   }
   return( bReturn );
}

tVoid spm_tclGlobalApplicationManager::vNewSystemStateReached( tU32 u32LamId ){
   SPM_GET_IF_REFERENCE_NEW_VAR( poclSystemStateManager, ISpmSystemStateManager );

   std::map < tU32, TLamState >::iterator it;

   it = _mapLam.find( u32LamId );
   if ( it != _mapLam.end( ) ){
      if ( it->second._u8State != SPM_GAM_LAM_STATE_CHANGE_PROCESSED ){
         it->second._u8State = SPM_GAM_LAM_STATE_CHANGE_PROCESSED;
      } else {
         // nothing to do -> state change was already processed for LAM
         return;
      }
   }


   // and now check if all requests are already processed
   tBool bAllRequestsProcessed = TRUE;
   for ( it = _mapLam.begin( ); it != _mapLam.end( ); ++it ){
      if ( it->second._poRefLam->bAllRequestsProcessed( ) ){
         it->second._u8State = SPM_GAM_LAM_STATE_CHANGE_PROCESSED;
      } else {
         bAllRequestsProcessed = FALSE;
      }
   }

   if ( bAllRequestsProcessed ){
      // search for SW Blocks to kill
      for ( it = _mapLam.begin( ); it != _mapLam.end( ); ++it ){
         it->second._poRefLam->vCheckForSwBlocksToKill( );
      }
      poclSystemStateManager->vNewSystemStateReached( );
   }
} // vNewSystemStateReached

tVoid spm_tclGlobalApplicationManager::vTraceRemainingTime( ){
   if ( _poclAppManagerConfig != NULL ){
      _poclAppManagerConfig->vTraceRemainingTime( );
   }
}

tVoid spm_tclGlobalApplicationManager::vHandleMessage( tU32 u32Message,
                                                       tU32 u32Parm ){
   switch ( u32Message ){
      case SPM_U32_WORKER_LAM_APP_END_WANTED_TO:
      {
         std::map < tU32, TLamState >::iterator it;

         for ( it = _mapLam.begin( ); it != _mapLam.end( ); ++it ){
            it->second._poRefLam->vHandleApplicationEndFailureTimeout( );
         }
      }
      break;

      default:
         if ( _poclAppManagerConfig != NULL ){
            _poclAppManagerConfig->vHandleMessage( u32Message, u32Parm );
         }
         break;
   } // switch
}    // vHandleMessage

tVoid spm_tclGlobalApplicationManager::vAllSubStatesProcessed( ){

   if ( _poclAppManagerConfig != OSAL_NULL ){
      _poclAppManagerConfig->vAllSubStatesProcessed( );
   }
}

