/*!
  * \file spm_ApplicationErrorHandler.cpp
  *  \brief
  *    Handle errors in LCM, collect data and write to error memory
  *
  *  \note
  *  PROJECT: NextGen \n
  *  SW-COMPONENT: FC SPM \n
  *  COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * Date      | Author            | Modification
  * 03.04.12  | CM-AI/VW32 kollai | Adaptation for NISSAN LCN2 KAI
  ******
  */

#include <sstream>
#include <iomanip>
#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

// SPM  configuration
#include "spm_Config.h"

// my class header
#include "spm_ApplicationErrorHandler.h"

// interfaces class definitions
#include "spm_ISystemStateManager.h"
#include "spm_ISystemPowerManager.h"
#include "spm_IOsalProxy.h"
#include "spm_IApplicationDatabase.h"
#include "spm_ISupervisionEnableSupervisor.h"
#include "spm_ILateServiceHandler.h"
#include "spm_IFactory.h"

// spm helper
#include "spm_SoftwareBlock.h"
#include "spm_StringTokenizer.h"
#include "spm_ApplicationConfiguration.h"
#include "spm_Registry_pathes.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM_LAM
#include "trcGenProj/Header/spm_ApplicationErrorHandler.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/
#define SPM_MSGBOX_APPERR_MAX_MESSAGE_COUNT     10
#define SPM_MSGBOX_APPERR_MAX_MESSAGE_LENGTH    sizeof( TArrErrMsg )

#define SPM_MSGQ_APPERR_SERVER                  "SpmAppErrMsQ"

spm_tclApplicationErrorHandler::spm_tclApplicationErrorHandler( const ISpmFactory& factory )
   : ISpmApplicationErrorHandler( factory )
   , _poclWorkerServer( NULL ){
   if ( OSAL_ERROR == OSAL_s32MessageQueueCreate(
           SPM_MSGQ_APPERR_SERVER,
           SPM_MSGBOX_APPERR_MAX_MESSAGE_COUNT,
           SPM_MSGBOX_APPERR_MAX_MESSAGE_LENGTH,
           OSAL_EN_READWRITE,
           &_hAppErrQueue
           )
        ){
      _hAppErrQueue = OSAL_C_INVALID_HANDLE;
      ETG_TRACE_ERRMEM( ( "spm_tclApplicationErrorHandler::spm_tclApplicationErrorHandler(): Message queue creation error!!!!" ) );
   }
   _fWriteEmptyUnitFailedList     = FALSE;
   _fUnitFailedListResultReceived = FALSE;
}

spm_tclApplicationErrorHandler::~spm_tclApplicationErrorHandler( ){
   SPM_NULL_POINTER_CHECK( _poclWorkerServer );
   _poclWorkerServer->vRemoveClient( this );

   _poclWorkerServer = NULL;

}

tVoid spm_tclApplicationErrorHandler::vStartCommunication( ){
   SPM_NULL_POINTER_CHECK( _poclWorkerServer );
   _poclWorkerServer->vAddClient( this );
}

tVoid spm_tclApplicationErrorHandler::vGetReferences( ){
   SPM_GET_IF_REFERENCE_USE_VAR( _poclWorkerServer, ISpmWorkerServer );
}

/*!
  * \fn
  *  \brief
  *   Function retrieves additional thread information about the application
  *   based on its App ID.
  *  \param[in] u32AppId: Application ID
  *  \details
  *   Functions checks if the application ID is available in SPM_REG_PROCESS_BASE_SPM_THREADINFO_BASE_PATH
  *   and dumps process information i.e,a call stack for the process is dumped where the
  *   start type is "OSAL".
  ******
  */
tVoid spm_tclApplicationErrorHandler::vWriteAdditionalAppThreadInfo( tU32 u32AppId ){
      SPM_GET_IF_REFERENCE_NEW_VAR( poclSystemPowerManager,          ISpmSystemPowerManager );
      SPM_GET_IF_REFERENCE_NEW_VAR( poclOsalProxy,                   ISpmOsalProxy );
      SPM_GET_IF_REFERENCE_NEW_VAR( poclSupervisionEnableSupervisor, ISpmSupervisionEnableSupervisor );

   if ( TRUE == poclSupervisionEnableSupervisor->bIsSupervisionStopped( ) ){
      ETG_TRACE_ERRMEM( ( "spm_tclApplicationErrorHandler::vWriteAdditionalAppThreadInfo: Triggering Callstack disabled due to Supervision settings (WD_OFF/disable_reset.txt/TTFis-SpmResetSupervisionDisable)" ) );
      return;
   }
   std::string           strErrmemBuf;

   tU32                  as32Data[255];
   OSAL_trIOCtrlDir      rDirV = { 0,0,0 };
   OSAL_tIODescriptor    fd;
   OSAL_trIOCtrlRegistry rReg;

   fd = OSAL_IOOpen( SPM_REG_PROCESS_BASE_SPM_THREADINFO_BASE_PATH, OSAL_EN_READONLY );
   if ( fd != OSAL_ERROR ){
      rDirV.fd        = fd;
      rDirV.s32Cookie = 0;

      while ( OSAL_s32IOControl( fd, OSAL_C_S32_IOCTRL_REGENUMVALUE, ( intptr_t )&rDirV ) != OSAL_ERROR ){
         rReg.pcos8Name = rDirV.dirent.s8Name;
         rReg.ps8Value  = (tU8*)as32Data;
         rReg.u32Size   = sizeof( as32Data );

         if ( OSAL_s32IOControl( fd, OSAL_C_S32_IOCTRL_REGGETVALUE, ( intptr_t )&rReg ) != OSAL_ERROR ){

            if ( rReg.s32Type == OSAL_C_S32_VALUE_STRING ){
               std::string  s      = (const tChar*)rDirV.dirent.s8Name;
               tU32         u32Value;
               tChar       *endp; // not really used here
               const tChar *startp = s.c_str( );

               // get a number and evaluate it
               if ( ( s.length( ) > 2 ) && ( ( s.substr( 0, 2 ).compare( "0x" ) == 0 ) || ( s.substr( 0, 2 ).compare( "0X" ) == 0 ) ) ){
                  // found hex value
                  u32Value = (tU32)OSAL_u32StringToU32( startp + 2, &endp, 16 );
               } else {
                  u32Value = (tU32)OSAL_u32StringToU32( startp, &endp, 10 );
               }
               if ( std::size_t(endp - startp) != s.length( ) ){
                  break;
               }
               if ( u32Value == u32AppId ){

                  /* Handle local apps and additional process dumps for non-local apps*/
                  spm_tclStringTokenizer st( (const tChar*)as32Data, ", ", FALSE );
                  while ( st.bHasMoreTokens( ) ){
                     std::string strBuffer;

                     // need to use a buffer because parameter to bWdtTimeOutWarning is not const
                     strBuffer = st.oNextToken( );
                     {
                        if ( ( strBuffer.length( ) >= 2 ) && ( strBuffer.at( 0 ) == '#' ) ){
                           std::stringstream ssSendTriggerfor;
                           ssSendTriggerfor << "(Add.process info) Sending process callstack trigger for " << strBuffer.at( 1 );
                           strErrmemBuf = ssSendTriggerfor.str();
                           poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strErrmemBuf.c_str( ), (tU16)OSAL_u32StringLength( strErrmemBuf.c_str( ) ) );
                           poclOsalProxy->bDumpProcessInfo( &strBuffer[1], "OSAL" );
                        }

                     }
                  }
                  break; // stop iterating, application id was found
               }
            }
         }
      }
      OSAL_s32IOClose( fd );
   }
} // vWriteAdditionalAppThreadInfo

/*!
  * \fn
  *  \brief
  *   The function logs the queue status of the applications into Errmem.
  *  \param[in] strName: Queue name.
  *  \details
  *  Message Queue status of the application containing information like
  *  number of msgs, max msg length and max number of messages is logged
  *  into errmem.
  *  In case of failure,to open the message queue or to retrieve the message
  *   queue status the corresponding err msg is logged into Errmem.
  ******
  */
tVoid spm_tclApplicationErrorHandler::vWriteQueueStatus( const std::string& strName ){
   OSAL_tMQueueHandle hMessageQueueHandle  = OSAL_C_INVALID_HANDLE;
   tS32               s32Ret;
   tU32               nNumberOfMessages    = 0;
   tU32               nMaxNumberOfMessages = 0;
   tU32               nMaxMessageLenght    = 0;

   std::string        strErrmemBuf;

   SPM_GET_IF_REFERENCE_NEW_VAR( poclSystemPowerManager, ISpmSystemPowerManager );

   s32Ret = OSAL_s32MessageQueueOpen( strName.c_str( ), OSAL_EN_READWRITE, &hMessageQueueHandle );
   if ( s32Ret != OSAL_OK ){
      strErrmemBuf = "SPM: Error: Cant open queue " + strName;
      poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strErrmemBuf.c_str( ), (tU16)OSAL_u32StringLength( strErrmemBuf.c_str( ) ) );
   } else {
      if ( OSAL_s32MessageQueueStatus( hMessageQueueHandle, &nMaxNumberOfMessages, &nMaxMessageLenght, &nNumberOfMessages ) == OSAL_OK ){
         strErrmemBuf = "SPM: Number of messages in queue " + strName + " = " + std::to_string(nNumberOfMessages) + ", max_msg_lenght=" + std::to_string(nMaxMessageLenght) +", max_messages=" + std::to_string(nMaxNumberOfMessages);
         poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strErrmemBuf.c_str( ), (tU16)OSAL_u32StringLength( strErrmemBuf.c_str( ) ) );
      } else {
         strErrmemBuf = "SPM: Error: Cant get queue status for " + strName;
         poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strErrmemBuf.c_str( ), (tU16)OSAL_u32StringLength( strErrmemBuf.c_str( ) ) );
      }
      OSAL_s32MessageQueueClose( hMessageQueueHandle );
   }
} // vWriteQueueStatus

/*!
  * \fn
  *  \brief
  *   The function gives information about application for which the local
  *   application manager detected no response.
  *  \param[in] tAppErrMsg: Contains application details.
  *  \details
  *   The message queue status like the number of messages in queue for the
  *   application as well as SPM is logged into ErrMem. Information regarding
  *   the ready threads and call stack for the application is logged into errmem.
  ******
  */
tVoid spm_tclApplicationErrorHandler::vWriteAppThreadInfo( TArrErrMsg& tAppErrMsg ){
   tU8         u8Dummy = 0;

   std::string strErrmemBuf;

      SPM_GET_IF_REFERENCE_NEW_VAR( poclSystemPowerManager,          ISpmSystemPowerManager );
      SPM_GET_IF_REFERENCE_NEW_VAR( poclOsalProxy,                   ISpmOsalProxy );
      SPM_GET_IF_REFERENCE_NEW_VAR( poclSupervisionEnableSupervisor, ISpmSupervisionEnableSupervisor );

   if ( TRUE == poclSupervisionEnableSupervisor->bIsSupervisionStopped( ) ){
      ETG_TRACE_ERRMEM( ( "spm_tclApplicationErrorHandler::vWriteAppThreadInfo: Triggering Callstack disabled due to Supervision settings (WD_OFF/disable_reset.txt/TTFis-SpmResetSupervisionDisable)" ) );
      return;
   }

   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );

   std::stringstream ssDetectMissingResponse;
   ssDetectMissingResponse << "SPM: Local Application Manager detects missing response from app " << std::to_string(tAppErrMsg.u.tAppErr.u32AppId) << "(0x" << std::hex << tAppErrMsg.u.tAppErr.u32AppId << ")!!!!";
   strErrmemBuf = ssDetectMissingResponse.str();
   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strErrmemBuf.c_str( ), (tU16)OSAL_u32StringLength( strErrmemBuf.c_str( ) ) );

   std::stringstream ssMsgQueueStatus;
   ssMsgQueueStatus << "SPM: Getting now OSAL_s32MessageQueueStatus() for app " << std::to_string(tAppErrMsg.u.tAppErr.u32AppId) << "(0x" << std::hex << tAppErrMsg.u.tAppErr.u32AppId << ") (MQB_" << std::to_string(tAppErrMsg.u.tAppErr.u32AppId) << " & mbx_" << std::to_string(tAppErrMsg.u.tAppErr.u32AppId) << ")";
   strErrmemBuf = ssMsgQueueStatus.str();
   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strErrmemBuf.c_str( ), (tU16)OSAL_u32StringLength( strErrmemBuf.c_str( ) ) );

   /* Write queue message count for this app to errmem: */
   {
      /* array size includes max. of 5 char for AppId and 1 for string termination */
      std::string strName;
      strName      = "MQB_" + std::to_string(tAppErrMsg.u.tAppErr.u32AppId);
      vWriteQueueStatus( strName );
      strName      = "mbx_" + std::to_string(tAppErrMsg.u.tAppErr.u32AppId );
      vWriteQueueStatus( strName );

      strErrmemBuf = "SPM: Getting now OSAL_s32MessageQueueStatus() for SPM queues:";
      poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strErrmemBuf.c_str( ), (tU16)OSAL_u32StringLength( strErrmemBuf.c_str( ) ) );
      vWriteQueueStatus( "mbx_0" );
      vWriteQueueStatus( SPM_MSGQ_WORKER_SERVER );
      vWriteQueueStatus( SPM_MSGBOX_NAME );
      vWriteQueueStatus( SPM_MSGBOX_GPIO_NAME );
   }

   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );
   poclOsalProxy->bDumpThreadInfoReady( );
   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );

   if ( tAppErrMsg.u.tAppErr.strStartType[0] != 0 ){
      ETG_TRACE_FATAL( ( "spm_tclApplicationErrorHandler::vWriteAppThreadInfo(): FAILURE for app: %u(%d/0x%x), send trigger to '%80s' (starttype '%20s').", ETG_ENUM( ail_u16AppId, tAppErrMsg.u.tAppErr.u32AppId ), tAppErrMsg.u.tAppErr.u32AppId, tAppErrMsg.u.tAppErr.u32AppId, tAppErrMsg.u.tAppErr.strExecPath, tAppErrMsg.u.tAppErr.strStartType ) );

      std::stringstream ssSendingTriggerviaSh1;
      ssSendingTriggerviaSh1 << "Sending process callstack trigger via SH for '" << std::setw(80)<< tAppErrMsg.u.tAppErr.strExecPath <<"' (starttype '" << std::setw(20) << tAppErrMsg.u.tAppErr.strStartType << "')";
      strErrmemBuf = ssSendingTriggerviaSh1.str();
      poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strErrmemBuf.c_str( ), (tU16)OSAL_u32StringLength( strErrmemBuf.c_str( ) ) );
      if (NULL != strstr (tAppErrMsg.u.tAppErr.strExecPath,".service")) {
          //must be a systemd script
          poclOsalProxy->bDumpProcessInfo( tAppErrMsg.u.tAppErr.strExecPath, SPM_STARTUP_VALUE_SYSTEMD_START_TYPE );
      } else {
          poclOsalProxy->bDumpProcessInfo( tAppErrMsg.u.tAppErr.strExecPath, tAppErrMsg.u.tAppErr.strStartType );
      }
   }

   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );
   vWriteAdditionalAppThreadInfo( tAppErrMsg.u.tAppErr.u32AppId );
   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );

} // vWriteAppThreadInfo

/*!
  * \fn
  *  \brief
  *   Function is triggered when the application end request is received by AIL.
  *  \param[in] tAppErrMsg: Contains application details.
  *  \details
  *   Information like the Current and New System State,Current and Required App state,
  *   App ID is written.Application thread info like its queue status is logged, along
  *   with the time trace.
  ******
  */
tVoid spm_tclApplicationErrorHandler::vWriteApplicationErrorToErrmem( TArrErrMsg& tAppErrMsg ){
   tLcmUString strBuf;
   tU8         u8Dummy = 0;

   SPM_GET_IF_REFERENCE_NEW_VAR( poclSystemPowerManager, ISpmSystemPowerManager );
   SPM_GET_IF_REFERENCE_NEW_VAR( poclOsalProxy,          ISpmOsalProxy );

   strBuf.insert( 0, sizeof( tAppErrMsg.u8ShutdownReason ), tAppErrMsg.u8ShutdownReason );

   SPM_M_INSERT_STRING_AT_INDEX_T32( strBuf, 1,  tAppErrMsg.u.tAppErr.u32CurSystemState );
   SPM_M_INSERT_STRING_AT_INDEX_T32( strBuf, 5,  tAppErrMsg.u.tAppErr.u32NewSystemState );
   SPM_M_INSERT_STRING_AT_INDEX_T32( strBuf, 9,  tAppErrMsg.u.tAppErr.u32CurAppState );
   SPM_M_INSERT_STRING_AT_INDEX_T32( strBuf, 13, tAppErrMsg.u.tAppErr.u32ReqAppState );
   SPM_M_INSERT_STRING_AT_INDEX_T32( strBuf, 17, tAppErrMsg.u.tAppErr.u32AppId );
   SPM_M_INSERT_STRING_AT_INDEX_T32( strBuf, 21, tAppErrMsg.u32ErrorType );

   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );

   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( tAppErrMsg.u8ErrmemType ), strBuf.c_str( ) );
   if ( SPM_U8_ERRMEM_TYPE_APP_END_WANTED == tAppErrMsg.u8ErrmemType ){
      std::string strFirstEntry( "APP_END_WANTED: 1. Thread information  (Request received by AIL)." );
      poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strFirstEntry.c_str( ), (tU16)strFirstEntry.size( ) );
   }

   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );
   ETG_TRACE_USR1( ( "spm_tclApplicationErrorHandler::vWriteApplicationErrorToErrmem(): Start Time Tracer (for app %u(%d/0x%x)).", ETG_ENUM( ail_u16AppId, tAppErrMsg.u.tAppErr.u32AppId ), tAppErrMsg.u.tAppErr.u32AppId, tAppErrMsg.u.tAppErr.u32AppId ) );
   poclOsalProxy->bDumpTimeTrace( );
   ETG_TRACE_USR1( ( "spm_tclApplicationErrorHandler::vWriteApplicationErrorToErrmem(): Time Tracer done(for app %u(%d/0x%x)).", ETG_ENUM( ail_u16AppId, tAppErrMsg.u.tAppErr.u32AppId ), tAppErrMsg.u.tAppErr.u32AppId, tAppErrMsg.u.tAppErr.u32AppId ) );
   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );
   vWriteAppThreadInfo( tAppErrMsg );
   OSAL_s32ThreadWait( 200 ); // give exception handling to to store information
   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );
} // vWriteApplicationErrorToErrmem

/*!
  * \fn
  *  \brief
  *   The function is triggered if the Shutdwon reason is found to be SHUTDOWN_AIL_CCA_QUEUE_FULL.
  *   It logs the reasons for Application failure due to Queue full into ErrMem.
  *  \param[in] tAppErrMsg: Contains application details.
  *  \details
  *   Errmem is updated with information depending on if its the source or destination application
  *   that has experienced queue full.
  *   Information like the Current and New System State,Current and Required App state,
  *   App ID is written.Call Stack for the process is also logged along with the application
  *   time trace.
  *********
  */
tVoid spm_tclApplicationErrorHandler::vWriteApplicationQueueFullToErrmem( TArrErrMsg& tAppErrMsg ){
   tLcmUString strBuf;

   std::string strErrmemBuf;
   tU8         u8Dummy = 0;


   SPM_GET_IF_REFERENCE_NEW_VAR( poclSystemPowerManager, ISpmSystemPowerManager );
   SPM_GET_IF_REFERENCE_NEW_VAR( poclOsalProxy,          ISpmOsalProxy );

   strBuf.insert( 0, sizeof( tAppErrMsg.u8ShutdownReason ), tAppErrMsg.u8ShutdownReason );

   SPM_M_INSERT_STRING_AT_INDEX_T32( strBuf, 1,  tAppErrMsg.u.tAppErr.u32CurSystemState );
   SPM_M_INSERT_STRING_AT_INDEX_T32( strBuf, 5,  tAppErrMsg.u.tAppErr.u32NewSystemState );
   SPM_M_INSERT_STRING_AT_INDEX_T32( strBuf, 9,  tAppErrMsg.u.tAppErr.u32CurAppState );
   SPM_M_INSERT_STRING_AT_INDEX_T32( strBuf, 13, tAppErrMsg.u.tAppErr.u32ReqAppState );
   SPM_M_INSERT_STRING_AT_INDEX_T32( strBuf, 17, tAppErrMsg.u.tAppErr.u32AppId );
   SPM_M_INSERT_STRING_AT_INDEX_T32( strBuf, 21, tAppErrMsg.u32ErrorType );

   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_DOUBLE_SEPERATOR ), (const tU8*)&u8Dummy, sizeof( u8Dummy ) );

   std::string strFirstEntry( "QUEUE_FULL ERROR: Getting Thread&Queue information" );
   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strFirstEntry.c_str( ), (tU16)strFirstEntry.size( ) );

   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( tAppErrMsg.u8ErrmemType ), (const tU8*)strBuf.c_str( ) );
   if ( tAppErrMsg.u8ErrmemType == SPM_U8_ERRMEM_TYPE_AIL_MESSAGE_QUELL_FULL_S ){
      std::stringstream streamCallStackQueuesend;
      streamCallStackQueuesend << "QUEUE_FULL ERROR #1: Getting now Callstack&Queue information of sending app " << std::to_string(tAppErrMsg.u.tAppErr.u32AppId) << "(0x" << std::hex << tAppErrMsg.u.tAppErr.u32AppId << ")";
      strErrmemBuf = streamCallStackQueuesend.str();
      poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strErrmemBuf.c_str( ), (tU16)OSAL_u32StringLength( strErrmemBuf.c_str( ) ) );
   } else {
      std::stringstream streamCallStackQueueRev;
      streamCallStackQueueRev << "QUEUE_FULL ERROR #2: Getting now Callstack&Queue information of receiving app " << std::to_string(tAppErrMsg.u.tAppErr.u32AppId) << "(0x" << std::hex << tAppErrMsg.u.tAppErr.u32AppId << ")";
      strErrmemBuf = streamCallStackQueueRev.str();
      poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strErrmemBuf.c_str( ), (tU16)OSAL_u32StringLength( strErrmemBuf.c_str( ) ) );
   }

   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );
   vWriteAppThreadInfo( tAppErrMsg );
   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );

   ETG_TRACE_USR1( ( "spm_tclApplicationErrorHandler::vWriteApplicationQueueFullToErrmem(): Start Time Tracer (for app %u(%d/0x%x)).", ETG_ENUM( ail_u16AppId, tAppErrMsg.u.tAppErr.u32AppId ), tAppErrMsg.u.tAppErr.u32AppId, tAppErrMsg.u.tAppErr.u32AppId ) );
   poclOsalProxy->bDumpTimeTrace( );
   ETG_TRACE_USR1( ( "spm_tclApplicationErrorHandler::vWriteApplicationQueueFullToErrmem(): Time Tracer done(for app %u(%d/0x%x)).", ETG_ENUM( ail_u16AppId, tAppErrMsg.u.tAppErr.u32AppId ), tAppErrMsg.u.tAppErr.u32AppId, tAppErrMsg.u.tAppErr.u32AppId ) );
   OSAL_s32ThreadWait( 200 ); // give exception handling to to store information
   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );
} // vWriteApplicationQueueFullToErrmem

/*!
  * \fn
  *  \brief
  *   The function writes the reasons for application's failure caused due to Queue full
  *   into ErrMem.
  *  \param[in] tAppErrMsg: Contains application details.
  *  \details
  *   Errmem is updated with information depending on if its the source or destination
  *   application that has experienced queue full.
  *   Information like the Current and New System State, App ID is written.
  *   Call Stack for the process is also logged containing application
  *   thread information like its queue status etc is written into Errmem.
  *********
  */
tVoid spm_tclApplicationErrorHandler::vWriteQueueFullToErrmem( TArrErrMsg& tAppErrMsg ){
   tLcmUString strBuf;

   std::string strErrmemBuf;
   tU8         u8Dummy = 0;

   tAppErrMsg.u.tAppErr.u32AppId = tAppErrMsg.u32ErrorType;


   SPM_GET_IF_REFERENCE_NEW_VAR( poclSystemPowerManager, ISpmSystemPowerManager );
   SPM_GET_IF_REFERENCE_NEW_VAR( poclOsalProxy,          ISpmOsalProxy );

   strBuf.insert( 0, sizeof( tAppErrMsg.u8ShutdownReason ), tAppErrMsg.u8ShutdownReason );

   SPM_M_INSERT_STRING_AT_INDEX_T32( strBuf, 1, tAppErrMsg.u.tAppErr.u32CurSystemState );
   SPM_M_INSERT_STRING_AT_INDEX_T32( strBuf, 5, tAppErrMsg.u.tAppErr.u32NewSystemState );
   SPM_M_CLEAR_STRING_AT_INDEX_T32( strBuf, 9 );
   SPM_M_CLEAR_STRING_AT_INDEX_T32( strBuf, 13 );
   SPM_M_INSERT_STRING_AT_INDEX_T32( strBuf, 17, tAppErrMsg.u.tAppErr.u32AppId );
   SPM_M_CLEAR_STRING_AT_INDEX_T32( strBuf, 21 );

   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_DOUBLE_SEPERATOR ), (const tU8*)&u8Dummy, sizeof( u8Dummy ) );

   if ( tAppErrMsg.u8ErrmemType == SPM_U8_ERRMEM_TYPE_AIL_MESSAGE_QUELL_FULL_S ){
      std::stringstream streamSendingApp;
      streamSendingApp << "QUEUE_FULL ERROR: Cant get information for for sending app " << std::to_string(tAppErrMsg.u.tAppErr.u32AppId) << "(0x" << std::hex << tAppErrMsg.u.tAppErr.u32AppId << ")";
      strErrmemBuf = streamSendingApp.str();
      poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strErrmemBuf.c_str( ), (tU16)OSAL_u32StringLength( strErrmemBuf.c_str( ) ) );

   } else {
      std::stringstream streamReceivingApp;
      streamReceivingApp << "QUEUE_FULL ERROR: Cant get information for for receiving app " << std::to_string(tAppErrMsg.u.tAppErr.u32AppId) << "(0x" << std::hex << tAppErrMsg.u.tAppErr.u32AppId << ").";
      strErrmemBuf = streamReceivingApp.str();
      poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strErrmemBuf.c_str( ), (tU16)OSAL_u32StringLength( strErrmemBuf.c_str( ) ) );
      std::stringstream ssCallStacksNotConnectApp;
      ssCallStacksNotConnectApp << "Trying now to get MBX and App. callstacks for not connected app " <<  std::to_string(tAppErrMsg.u.tAppErr.u32AppId) << "(0x" << std::hex << tAppErrMsg.u.tAppErr.u32AppId << ")";
      strErrmemBuf = ssCallStacksNotConnectApp.str();
      poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strErrmemBuf.c_str( ), (tU16)OSAL_u32StringLength( strErrmemBuf.c_str( ) ) );

      poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( tAppErrMsg.u8ErrmemType ), strBuf.c_str( ) );
   }

   strErrmemBuf = "***CALLSTACK-TRIGGER start: **** ";
   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strErrmemBuf.c_str( ), (tU16)OSAL_u32StringLength( strErrmemBuf.c_str( ) ) );
   vWriteAppThreadInfo( tAppErrMsg );
   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );
   ETG_TRACE_USR1( ( "spm_tclApplicationErrorHandler::vWriteQueueFullToErrmem(): Start Time Tracer (for app %u(%d/0x%x)).", ETG_ENUM( ail_u16AppId, tAppErrMsg.u.tAppErr.u32AppId ), tAppErrMsg.u.tAppErr.u32AppId, tAppErrMsg.u.tAppErr.u32AppId ) );
   poclOsalProxy->bDumpTimeTrace( );
   ETG_TRACE_USR1( ( "spm_tclApplicationErrorHandler::vWriteQueueFullToErrmem(): Time Tracer done(for app %u(%d/0x%x)).", ETG_ENUM( ail_u16AppId, tAppErrMsg.u.tAppErr.u32AppId ), tAppErrMsg.u.tAppErr.u32AppId, tAppErrMsg.u.tAppErr.u32AppId ) );
   OSAL_s32ThreadWait( 200 ); // give exception handling to to store information
   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );
} // vWriteQueueFullToErrmem

/*!
  * \fn
  *  \brief
  *   Starts a new thread for SPM Application Handler.
  *  \param[in] strName: name of thread.
  *  \param[in] u32Prio: priority of thread.
  *  \param[in] u32Stack: stack size of thread.
  ******
  */
tVoid spm_tclApplicationErrorHandler::vStart( const std::string& strName,
                                              tU32               u32Prio,
                                              tU32               u32Stack ){
   spm_tclActive::vStartThread( strName, u32Prio, u32Stack );
} // vStart

tVoid spm_tclApplicationErrorHandler::main( ){
   tU32       u32Prio;
   tS32       s32Ret;
   TArrErrMsg tMsg;


   s32Ret = OSAL_s32MessageQueueWait( _hAppErrQueue, (tU8*)&tMsg, sizeof( TArrErrMsg ), &u32Prio, OSAL_C_U32_INFINITE );
   if ( s32Ret > 0 ){
      if ( tMsg.u32Cmd == SPM_APPERRHDL_TRIGGER_APP_ERROR ){
         if ( ( tMsg.u8ErrmemType == SPM_U8_ERRMEM_TYPE_STATE_REQUEST_IMM )
              || ( tMsg.u8ErrmemType == SPM_U8_ERRMEM_TYPE_STATE_REQUEST )
              || ( tMsg.u8ErrmemType == SPM_U8_ERRMEM_TYPE_APP_END_WANTED )
              ){
            ETG_TRACE_USR1( ( "spm_tclApplicationErrorHandler::main(): Write application info to errmem!" ) );
            vTriggerServiceFileErrorsGet( );
            vWriteApplicationErrorToErrmem( tMsg );

            /* If list is already received, this vWriteServiceFileErrorsToErrmem() stores it, if not,
               _fWriteEmptyUnitFailedList=true forces at (later incomming) SPM_U8_ERRMEM_TYPE_UNIT_FAILED_LIST_RESULT
               to write it*/
            _fWriteEmptyUnitFailedList = true;
            vWriteServiceFileErrorsToErrmem( true );
         } else if ( tMsg.u8ErrmemType == SPM_U8_ERRMEM_TYPE_AIL_MESSAGE_QUELL_FULL_T ){
            if ( tMsg.u8ShutdownReason == SPM_U32_SHUTDOWN_AIL_CCA_QUEUE_FULL ){
               vWriteApplicationQueueFullToErrmem( tMsg );
            } else {
               vWriteQueueFullToErrmem( tMsg );
            }
         } else if ( tMsg.u8ErrmemType == SPM_U8_ERRMEM_TYPE_AIL_MESSAGE_QUELL_FULL_S ){
            if ( tMsg.u8ShutdownReason == SPM_U32_SHUTDOWN_AIL_CCA_QUEUE_FULL ){
               vWriteApplicationQueueFullToErrmem( tMsg );
            } else {
               vWriteQueueFullToErrmem( tMsg );
            }
         } else if ( tMsg.u8ErrmemType == SPM_U8_ERRMEM_TYPE_UNIT_FAILED_LIST ){
            ETG_TRACE_USR1( ( "spm_tclApplicationErrorHandler::main():UNIT_FAILED_LIST -  vTriggerServiceFileErrorsGet %d", OSAL_ClockGetElapsedTime( ) ) );
            vTriggerServiceFileErrorsGet( );
         } else if ( tMsg.u8ErrmemType == SPM_U8_ERRMEM_TYPE_UNIT_FAILED_LIST_RESULT ){
            ETG_TRACE_USR1( ( "spm_tclApplicationErrorHandler::main():UNIT_FAILED_LIST_RESULT - vWriteServiceFileErrorsToErrmem %d", OSAL_ClockGetElapsedTime( ) ) );
            vWriteServiceFileErrorsToErrmem( _fWriteEmptyUnitFailedList );
            _fUnitFailedListResultReceived = TRUE;
         } else if ( tMsg.u8ErrmemType == SPM_U8_ERRMEM_TYPE_APP_WATCHDOG_ERROR_ERR ){
            ETG_TRACE_FATAL( ( "spm_tclApplicationErrorHandler::main():APP_WATCHDOG_ERROR - vWriteApplicationWdgErrorToErrmem %d, App: %u(%d/0x%x), process '%s'", OSAL_ClockGetElapsedTime( ), ETG_ENUM( ail_u16AppId, tMsg.u.tAppErr.u32AppId ), tMsg.u.tAppErr.u32AppId, tMsg.u.tAppErr.u32AppId, tMsg.u.tAppErr.strExecPath ) );
            vWriteApplicationWdgErrorToErrmem( tMsg );
         } else {
            ETG_TRACE_USR1( ( "spm_tclApplicationErrorHandler::main(): Unknown event!: Command=%d, Type=%d", tMsg.u32Cmd, tMsg.u8ErrmemType ) );
         }
      } else if ( tMsg.u32Cmd == SPM_APPERRHDL_TRIGGER_RESET ){
         SPM_NULL_POINTER_CHECK( _poclWorkerServer );
         _poclWorkerServer->bPostMessageToWorker( SPM_U32_WORKER_SPM_SHUTDOWN, tMsg.u32ErrorType );
      } else {
         ETG_TRACE_USR1( ( "spm_tclApplicationErrorHandler::main(): Unknown event!: Command=%d, Type=%d", tMsg.u32Cmd, tMsg.u8ErrmemType ) );
      }

   }

} // main

tBool spm_tclApplicationErrorHandler::bTriggerReset( tU32 u32ResetType ){
   tBool      bRet = TRUE;

   TArrErrMsg tMsg;

   tMsg.u32Cmd       = SPM_APPERRHDL_TRIGGER_RESET;
   tMsg.u32ErrorType = u32ResetType;

   tS32       s32Ret = OSAL_s32MessageQueuePost( _hAppErrQueue, (tCU8*)&tMsg, sizeof( tMsg ), OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST );
   if ( s32Ret != OSAL_OK ){
      bRet = FALSE;
   }
   tU32       u32MsgMax, u32MaxLen, u32MsgCount = 0;
   if ( OSAL_OK != OSAL_s32MessageQueueStatus( _hAppErrQueue, &u32MsgMax, &u32MaxLen, &u32MsgCount ) ){
      ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!!" ) );
   } else {
      if( u32MsgMax != 0 ) {
         tU32       u32FillLevel = ( u32MsgCount * 100 ) / u32MsgMax;
         if ( u32FillLevel > SPM_WARNING_QUEUE_FILL_LEVEL_PERCENTAGE_WARN ){
            std::string   strError = "SPM: !!!!!! Error detected !!!!!! Fill level of queue _hAppErrQueue (" + std::to_string(u32FillLevel) + "%) exceeded maximum of " + std::to_string(SPM_WARNING_QUEUE_FILL_LEVEL_PERCENTAGE_WARN) + "%";
            ETG_TRACE_ERRMEM( ( "%s", strError.c_str( ) ) );
         }
      }
   }

   return( bRet );
} // bTriggerReset

tBool spm_tclApplicationErrorHandler::bTriggerAppError( spm_tclConnectedApp app,
                                                        tU8                 u8ShutdownReason,
                                                        tU8                 u8ErrmemType,
                                                        tU32                u32ErrorType ){
   tBool bRet = TRUE;


   SPM_GET_IF_REFERENCE_NEW_VAR_VAL( poclSystemStateManager, ISpmSystemStateManager );
   SPM_GET_IF_REFERENCE_NEW_VAR_VAL( poclDb,                 ISpmApplicationDatabase );

   TArrErrMsg tMsg;
   tMsg.u32Cmd                      = SPM_APPERRHDL_TRIGGER_APP_ERROR;
   tMsg.u8ErrmemType                = u8ErrmemType;
   tMsg.u8ShutdownReason            = u8ShutdownReason;
   tMsg.u32ErrorType                = u32ErrorType;

   tMsg.u.tAppErr.u32CurSystemState = poclSystemStateManager->u32GetSystemState( );
   tMsg.u.tAppErr.u32NewSystemState = poclSystemStateManager->u32GetNewSystemState( );

   memset( tMsg.u.tAppErr.strExecPath,  0, sizeof(tMsg.u.tAppErr.strExecPath) );
   memset( tMsg.u.tAppErr.strStartType, 0, sizeof(tMsg.u.tAppErr.strStartType) );

   if ( app.u32GetAppId( ) != AMT_C_U16_APPID_INVALID ){
      tMsg.u.tAppErr.u32CurAppState = app.u32GetAppStateCurrent( );
      tMsg.u.tAppErr.u32ReqAppState = app.u32GetAppStateRequest( );
      tMsg.u.tAppErr.u32AppId       = app.u32GetAppId( );

      spm_tclSoftwareBlock *pSwBlock = poclDb->poGetSwBlockByAppID( app.u32GetAppId( ) );
      if ( pSwBlock != NULL ){
         const std::string strExecPath  = pSwBlock->oGetExecPath( );
         const std::string strStartType = pSwBlock->oGetStartType( );
         OSAL_szStringNCopy( tMsg.u.tAppErr.strExecPath,  strExecPath.c_str( ),  sizeof(tMsg.u.tAppErr.strExecPath) - 1 );
         OSAL_szStringNCopy( tMsg.u.tAppErr.strStartType, strStartType.c_str( ), sizeof(tMsg.u.tAppErr.strStartType) - 1 );
      }
   } else {
      tMsg.u.tAppErr.u32CurAppState = 0;
      tMsg.u.tAppErr.u32ReqAppState = 0;
      tMsg.u.tAppErr.u32AppId       = AMT_C_U16_APPID_INVALID;
   }

   tS32 s32Ret = OSAL_s32MessageQueuePost( _hAppErrQueue, (tCU8*)&tMsg, sizeof( tMsg ), OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST );
   if ( s32Ret != OSAL_OK ){
      bRet = FALSE;
   }
   tU32 u32MsgMax, u32MaxLen, u32MsgCount = 0;
   if ( OSAL_OK != OSAL_s32MessageQueueStatus( _hAppErrQueue, &u32MsgMax, &u32MaxLen, &u32MsgCount ) ){
      ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!! Can not get MessageQueue Status" ) );
   } else {
      if( u32MsgMax != 0 ) {
         tU32 u32FillLevel = ( u32MsgCount * 100 ) / u32MsgMax;
         if ( u32FillLevel > SPM_WARNING_QUEUE_FILL_LEVEL_PERCENTAGE_WARN ){
            std::string   strError = "SPM: !!!!!! Error detected !!!!!! Fill level of queue _hAppErrQueue (" + std::to_string(u32FillLevel) + "%) exceeded maximum of " + std::to_string(SPM_WARNING_QUEUE_FILL_LEVEL_PERCENTAGE_WARN) + "%";
            ETG_TRACE_ERRMEM( ( "%s", strError.c_str( ) ) );
         }
      } else {
         ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!! MsgMax == 0" ) );
      }
   }

   return( bRet );
} // bTriggerAppError

tVoid spm_tclApplicationErrorHandler::vStateChangeImmediately( spm_tclConnectedApp app,
                                                               tBool               bRestart ){
   bTriggerAppError( app, SPM_U32_SHUTDOWN_UNKOWN, SPM_U8_ERRMEM_TYPE_STATE_REQUEST_IMM );
   if ( bRestart ){
      bTriggerReset( SPM_U32_SHUTDOWN_APP_ERROR );
   }
}

tVoid spm_tclApplicationErrorHandler::vStateChangeFailed( spm_tclConnectedApp app,
                                                          tBool               bRestart ){
   bTriggerAppError( app, SPM_U32_SHUTDOWN_AFTER_NO_STATE_REQUEST, SPM_U8_ERRMEM_TYPE_STATE_REQUEST );
   if ( bRestart ){
      bTriggerReset( SPM_U32_SHUTDOWN_AFTER_NO_STATE_REQUEST );
   }
}

tVoid spm_tclApplicationErrorHandler::vAppEndFailure( spm_tclConnectedApp app,
                                                      tU32                u32ErrorType,
                                                      tBool               bRestart ){
   bTriggerAppError( app, SPM_U32_SHUTDOWN_APP_END_WANTED, SPM_U8_ERRMEM_TYPE_APP_END_WANTED, u32ErrorType );
   if ( bRestart ){
      bTriggerReset( SPM_U32_SHUTDOWN_APP_ERROR );
   }
}

tVoid spm_tclApplicationErrorHandler::vTerminateChangeFailed( spm_tclConnectedApp app,
                                                              tBool               bRestart ){
   bTriggerAppError( app, SPM_U32_SHUTDOWN_UNKOWN, SPM_U8_ERRMEM_TYPE_STATE_REQUEST_IMM );
   if ( bRestart ){
      bTriggerReset( SPM_U32_SHUTDOWN_APP_ERROR );
   }
}

tVoid spm_tclApplicationErrorHandler::vWdgNotifyProblem( spm_tclConnectedApp app,
                                                         tBool               bRestart ){
   bTriggerAppError( app, SPM_U32_SHUTDOWN_NOTIFY_PROBLEM, SPM_U8_ERRMEM_TYPE_APP_WATCHDOG_ERROR_ERR );
   if ( bRestart ){
      bTriggerReset( SPM_U32_SHUTDOWN_NOTIFY_PROBLEM );
   }

}

tVoid spm_tclApplicationErrorHandler::vMsgQueueFull( spm_tclConnectedApp app,
                                                     tBool               bSender,
                                                     tBool               bRestart ){
   if ( bSender ){
      bTriggerAppError( app, SPM_U32_SHUTDOWN_AIL_CCA_QUEUE_FULL, SPM_U8_ERRMEM_TYPE_AIL_MESSAGE_QUELL_FULL_S );
   } else {
      bTriggerAppError( app, SPM_U32_SHUTDOWN_AIL_CCA_QUEUE_FULL, SPM_U8_ERRMEM_TYPE_AIL_MESSAGE_QUELL_FULL_T );
   }
   if ( bRestart ){
      bTriggerReset( SPM_U32_SHUTDOWN_AIL_CCA_QUEUE_FULL );
   }
}

tVoid spm_tclApplicationErrorHandler::vMsgQueueFull( tU32  u32AppId,
                                                     tBool bSender,
                                                     tBool bRestart ){
   spm_tclConnectedApp app;

   if ( bSender ){
      bTriggerAppError( app, SPM_U8_ERRMEM_TYPE_AIL_MESSAGE_QUELL_FULL_T_ERR, SPM_U8_ERRMEM_TYPE_AIL_MESSAGE_QUELL_FULL_S, u32AppId );
   } else {
      bTriggerAppError( app, SPM_U8_ERRMEM_TYPE_AIL_MESSAGE_QUELL_FULL_T_ERR, SPM_U8_ERRMEM_TYPE_AIL_MESSAGE_QUELL_FULL_T, u32AppId );
   }
   if ( bRestart ){
      bTriggerReset( SPM_U32_SHUTDOWN_AIL_CCA_QUEUE_FULL );
   }
}

tVoid spm_tclApplicationErrorHandler::vGetUnitFailedList( ){
   spm_tclConnectedApp app;

   bTriggerAppError( app, SPM_U32_SHUTDOWN_UNKOWN, SPM_U8_ERRMEM_TYPE_UNIT_FAILED_LIST );
}

tVoid spm_tclApplicationErrorHandler::vTriggerUnitFailedListReceived( ){
   spm_tclConnectedApp app;

   ETG_TRACE_USR4( ( "spm_tclApplicationErrorHandler::vTriggerUnitFailedListReceived at %d", OSAL_ClockGetElapsedTime( ) ) );
   bTriggerAppError( app, SPM_U32_SHUTDOWN_UNKOWN, SPM_U8_ERRMEM_TYPE_UNIT_FAILED_LIST_RESULT );
}

tVoid spm_tclApplicationErrorHandler::vTriggerServiceFileErrorsGet( ){
   _fUnitFailedListResultReceived = FALSE;
   SPM_GET_CLIENT_HANDLER_IF_REFERENCE_NEW_VAR( poLateSrvHandler, spm_ISpmLateServiceHandler );
   poLateSrvHandler->vTriggerGetFailedUnitList( );
   ETG_TRACE_USR4( ( "vTriggerServiceFileErrorsGet() send at %d", OSAL_ClockGetElapsedTime( ) ) );
}

/*!
  * \fn
  *  \brief
  *   The function writes errors caused due to Service files into ErrMem.
  *  \param[in] bWriteIfListEmpty: If file list is empty or not.
  *  \details
  *   Lists the services failed along with other required information
  *   like the description of failed service, its LoadState etc.
  *   If the list is empty then the corresponding info is logged into
  *   errmem.  And if the Active State parameter of the service is found to
  *   be 'failed' then the status of the service is also logged.
  *********
  */
tVoid spm_tclApplicationErrorHandler::vWriteServiceFileErrorsToErrmem( tBool bWriteIfListEmpty ){
      SPM_GET_IF_REFERENCE_NEW_VAR( poclSystemPowerManager, ISpmSystemPowerManager );

   SPM_GET_CLIENT_HANDLER_IF_REFERENCE_NEW_VAR( poLateSrvHandler, spm_ISpmLateServiceHandler );
   const TServiceFailedEntryList ServiceFailedEntryList = poLateSrvHandler->GetUnitList( );

   ETG_TRACE_USR1( ( "spm_tclApplicationErrorHandler::vWriteServiceFileErrorsToErrmem(): found %d failed/incorrect units", (tInt)ServiceFailedEntryList.size( ) ) );

   std::string                      strError;
   if ( bWriteIfListEmpty || ( ServiceFailedEntryList.size( ) > 0 ) ){
      strError = "*** failed ServiceFileList start: **** ";
      poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ) );
   }

   if ( ServiceFailedEntryList.begin( ) != ServiceFailedEntryList.end( ) ){
      TServiceFailedEntryList::const_iterator FailedSerivceIter;

      tU16                                    u16Ctr = 1;
      for ( FailedSerivceIter = ServiceFailedEntryList.begin( );
            FailedSerivceIter != ServiceFailedEntryList.end( );
            FailedSerivceIter++ ){
         std::string strUnitCntr( 4, 0 ); // assuming we do not have more than 9999 failing units :-)
         strUnitCntr = std::to_string(u16Ctr++);
         strError    = (std::string)"************* Unit(" + strUnitCntr + ") failed/incorrect";
         poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ) );
         strError    = "UnitName        = " + FailedSerivceIter->strServiceName;
         poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ) );
         strError    = "--> Description = " + FailedSerivceIter->strServiceDescription;
         poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ) );
         strError    = "--> LoadState   = " + FailedSerivceIter->strServiceLoadState;
         poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ) );
         strError    = "--> ActiveState = " + FailedSerivceIter->strServiceActiveState;
         poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ) );
         strError    = "--> SubState    = " + FailedSerivceIter->strServiceSubState;
         poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ) );
         strError    = "--> Follower    = " + FailedSerivceIter->strServiceFollower;
         poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ) );
         if ( FailedSerivceIter->strServiceActiveState.compare( "failed" ) == 0 ){
            std::string strCmdLine = "systemctl status " + FailedSerivceIter->strServiceName + " > /dev/errmem";
            poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strCmdLine.c_str( ), (tU16)strCmdLine.length( ) );
            system( strCmdLine.c_str( ) ); // OK => This system() call runs WITHOUT root permission. Required group memberships : 'shell', 'adit_errmem_log'.
         }
      }
   } else {
      if ( bWriteIfListEmpty ){
         strError = "*** list is empty **** ";
         poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ) );
      }
   }

   if ( bWriteIfListEmpty || ( ServiceFailedEntryList.size( ) > 0 ) ){
      strError = "*** failed ServiceFileList end. **** ";
      poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ) );
   }
} // vWriteServiceFileErrorsToErrmem

/*!
  * \fn
  *  \brief
  *  Function logs the failure of application's watchdog into Errmem.
  *  \param[in] tAppErrMsg: Contains application details.
  *  \details
  *   Information like the Current and New System State,
  *   Current and Required Application state, App ID is written.
  *   Call Stack for the process is also logged along with the additional
  *   thread information.
  *********
  */
tVoid spm_tclApplicationErrorHandler::vWriteApplicationWdgErrorToErrmem( TArrErrMsg& tAppErrMsg ){
   tLcmUString strBuf;

   std::string strErrmemBuf;
   tU8         u8Dummy = 0;

      SPM_GET_IF_REFERENCE_NEW_VAR( poclSystemPowerManager,          ISpmSystemPowerManager );
      SPM_GET_IF_REFERENCE_NEW_VAR( poclOsalProxy,                   ISpmOsalProxy );
      SPM_GET_IF_REFERENCE_NEW_VAR( poclSupervisionEnableSupervisor, ISpmSupervisionEnableSupervisor );

   if ( TRUE == poclSupervisionEnableSupervisor->bIsSupervisionStopped( ) ){
      ETG_TRACE_ERRMEM( ( "spm_tclApplicationErrorHandler::vWriteApplicationWdgErrorToErrmem: Triggering Callstack disabled due to Supervision settings (WD_OFF/disable_reset.txt/TTFis-SpmResetSupervisionDisable)" ) );
      return;
   }

   ETG_TRACE_ERR( ( "spm_tclApplicationErrorHandler::vWriteApplicationWdgErrorToErrmem(): WATCHDOG ERROR for app: %u(%d/0x%x), process '%s'", ETG_ENUM( ail_u16AppId, tAppErrMsg.u.tAppErr.u32AppId ), tAppErrMsg.u.tAppErr.u32AppId, tAppErrMsg.u.tAppErr.u32AppId, tAppErrMsg.u.tAppErr.strExecPath ) );
   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_DOUBLE_SEPERATOR ), (tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );
   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_ERROR_ENTRY_START ), (tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );
   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_DOUBLE_SEPERATOR ), (tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );

   std::stringstream ssWatchdogerrForApp;
   ssWatchdogerrForApp << "WATCHDOG ERROR for App: " << std::to_string(ETG_ENUM( ail_u16AppId, tAppErrMsg.u.tAppErr.u32AppId )) << "(0x" << std::hex << tAppErrMsg.u.tAppErr.u32AppId << "), Process '" << tAppErrMsg.u.tAppErr.strExecPath;
   strErrmemBuf = ssWatchdogerrForApp.str();
   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strErrmemBuf.c_str( ), (tU16)OSAL_u32StringLength( strErrmemBuf.c_str( ) ) );

   strBuf.insert( 0, sizeof( tAppErrMsg.u8ShutdownReason ), tAppErrMsg.u8ShutdownReason );

   SPM_M_INSERT_STRING_AT_INDEX_T32( strBuf, 1,  tAppErrMsg.u.tAppErr.u32CurSystemState );
   SPM_M_INSERT_STRING_AT_INDEX_T32( strBuf, 5,  tAppErrMsg.u.tAppErr.u32NewSystemState );
   SPM_M_INSERT_STRING_AT_INDEX_T32( strBuf, 9,  tAppErrMsg.u.tAppErr.u32CurAppState );
   SPM_M_INSERT_STRING_AT_INDEX_T32( strBuf, 13, tAppErrMsg.u.tAppErr.u32ReqAppState );
   SPM_M_INSERT_STRING_AT_INDEX_T32( strBuf, 17, tAppErrMsg.u.tAppErr.u32AppId );
   SPM_M_INSERT_STRING_AT_INDEX_T32( strBuf, 21, tAppErrMsg.u32ErrorType );

   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( tAppErrMsg.u8ErrmemType ), strBuf.c_str( ) );
   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );

   if ( tAppErrMsg.u.tAppErr.strStartType[0] != 0 ){
      std::stringstream ssSendTriggerviaSh;
      ssSendTriggerviaSh << "Sending process callstack trigger via SH for '" << tAppErrMsg.u.tAppErr.strExecPath << "' (starttype '" << tAppErrMsg.u.tAppErr.strStartType << "')";
      strErrmemBuf = ssSendTriggerviaSh.str();
      poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strErrmemBuf.c_str( ), (tU16)OSAL_u32StringLength( strErrmemBuf.c_str( ) ) );
      poclOsalProxy->bDumpProcessInfo( tAppErrMsg.u.tAppErr.strExecPath, tAppErrMsg.u.tAppErr.strStartType );
   }

   vWriteAdditionalAppThreadInfo( tAppErrMsg.u.tAppErr.u32AppId );
   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_DOUBLE_SEPERATOR ), (tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );
   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_ERROR_ENTRY_END ), (tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );
   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_DOUBLE_SEPERATOR ), (tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );
   OSAL_s32ThreadWait( 200 ); // give exception handling to to store information
} // vWriteApplicationWdgErrorToErrmem

