/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_ApplicationConfiguration.h
  * @brief        This is header file contains functions for read Application state configuration, write application state configuration..
  * @copyright    (C) 2012 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

#include <map>
#include <set>
#include <string>
#include <vector>

#ifndef _SPM_CONFIG_H
   #error spm_Config.h must be included!
#endif



#include "spm_IApplicationConfiguration.h"

class ISpmCcaServiceServer;
class ISpmSystemPowerManager;
class ISpmSubStateClient;
class ISpmOsalProxy;
class ISpmApplicationDatabase;

#define SPM_PROFILE_CONTROL SPM_SYSTEM_STATE_MAX

#define SPM_STR_DEFAULT_NUMBER        "0"

/*!
  *  \class spm_tclApplicationConfiguration
  *  \brief This class is to read Application state configuration, write application state configuration.

          It takes and updates Application configuration from spm_tclSystemStateManagerConfig
        and spm_tclSystemPowerManagerConfig and gives this info to spm_tclSubStateHandlerConfig .
        It uses class spm_tclCcaServiceHandlerConfig to register new service.

     class spm_tclSoftwareBlock uses spm_tclApplicationConfiguration to get profile AppState.
  */
class spm_tclApplicationConfiguration : public ISpmApplicationConfiguration
{
protected:
std::map < tU32, tU32 >                    _tDefaultConfChangeable;
std::map < tU32, tU32 >                    _tDefaultConf;

std::map < tU32, tU32 >                    _tDefaultConfChangeableLcm;
std::map < tU32, tU32 >                    _tDefaultConfLcm;

std::map < tU32, std::map < tU32, tU32 > > _tAppConfigSpecific;
std::map < std::string, tU32 >             _mapAppStateNames;
std::map < std::string, tU32 >             _mapSystemStateNames;
std::map < tU32, std::string >             _mapSystemStateNamesReverse;

std::map < std::string, tU32 >             _mapAppNamesDefines;
std::map < std::string, std::string >      _mapAppNamesServiceName;

virtual tVoid vSetNewStateConfiguration( tU32               u32SystemState,
                                         const std::string& strContent );

virtual tVoid vGetSupervisionConfiguration( const std::string          & strContent,
                                            std::vector < std::string >& processes ) const;

virtual tVoid vSetNewDefaultValue( tU32 u32SystemState,
                                   tU32 u32AppState ){
   _tDefaultConfChangeable[u32SystemState]    = u32AppState;
   _tDefaultConfChangeableLcm[u32SystemState] = u32AppState;
}

virtual tVoid vSetDefaultValue( tU32 u32SystemState ){
   _tDefaultConfChangeable[u32SystemState]    = _tDefaultConf[u32SystemState];
   _tDefaultConfChangeableLcm[u32SystemState] = _tDefaultConfLcm[u32SystemState];
}

virtual tU32 u32GetValue( tU32 u32AppId,
                          tU32 u32SystemState );

virtual tVoid vTraceInfo( tU32 u32SystemState );

virtual tVoid vStartCommunication( );

virtual tVoid vGetReferences( );

virtual const tChar*getName( ) const { return( "spm_tclApplicationConfiguration" ); }

private:
public:
virtual ~spm_tclApplicationConfiguration( );
spm_tclApplicationConfiguration( const ISpmFactory& factory );

// trace handling
virtual tVoid vHandleTraceMessage( const tUChar *puchData );

virtual tU32 u32GetNewAppState( tU32 u32AppId );

virtual tU32 u32GetProfileAppState( tU32 u32Profile,
                                    tU32 u32AppId ){
   return( u32GetValue( u32AppId, u32Profile ) );
}

virtual tVoid vNewAppInSystem( tU32 u32AppId );

virtual tVoid vPrintAppInfo( tU32 u32AppId ){ (tVoid)u32AppId; }

virtual tVoid vReadStateConfigurations( );

virtual tBool bSetSpecificStateConfiguration( tU32 u32SystemState,
                                              tU32 u32Configuration );

// Get a list of processes which should not be supervised during this systemstate
virtual tVoid vGetSpecificSupervisionConfiguration( tU32                         u32SystemState,
                                                    tU32                         u32Configuration,
                                                    std::vector < std::string >& processes );


virtual tBool bSetNewAppStateTable( tU32  u32InternAppId,
                                    tU32  u32StateProfile,
                                    tU32  u32AppState,
                                    tBool bTriggerState = FALSE );

virtual tVoid vSetDefaultStateTable( tU32 u32SystemState );

virtual tVoid vSetValue( tU32 u32AppId,
                         tU32 u32SystemState,
                         tU32 u32Value );


ISpmSystemPowerManager  *_poclSystemPowerManager;
ISpmCcaServiceServer    *_poclCcaServiceHandler;
ISpmSubStateClient      *_poclSubStateHandler;
ISpmOsalProxy           *_poclPrx;
ISpmApplicationDatabase *_poclDb;

};

