/*!
  * \file spm_ApplicationConfiguration.cpp
  *  \brief
  *    This class is to read Application state , write application state , write application thread info and display application info.
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * 06.07.11  | TMS Fischer       | initial version
  * 03.04.12  | CM-AI/VW32 kollai | Adaptation for NISSAN LCN2 KAI
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

// SPM  configuration
#include "spm_Config.h"

// my class header
#include "spm_ApplicationConfiguration.h"

// interfaces class definitions
#include "spm_IOsalProxy.h"
#include "spm_ICcaServiceServer.h"
#include "spm_ISystemStateManager.h"
#include "spm_ISubStateClient.h"
#include "spm_ISystemPowerManager.h"
#include "spm_IApplicationDatabase.h"

#include "spm_IFactory.h"

// spm helper
#include "spm_StringTokenizer.h"
#include "spm_SoftwareBlock.h"
#include "spm_Registry_pathes.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM_LAM
#include "trcGenProj/Header/spm_ApplicationConfiguration.cpp.trc.h"
#endif

// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"


/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/
// #define SPM_TRACE_FILE_ID   SPM_FILE_APPLICATIONCONFIGURATION


/**
  *  \brief constructor. creates object and initializes all data
  *
  *  \param [in] factory reference to factory object
  *  \return
  */
spm_tclApplicationConfiguration::spm_tclApplicationConfiguration( const ISpmFactory& factory )
   : ISpmApplicationConfiguration( factory )
   , _poclSystemPowerManager( NULL )
   , _poclCcaServiceHandler( NULL )
   , _poclSubStateHandler( NULL )
   , _poclPrx( NULL )
   , _poclDb( NULL ){

}

/**
  *  \brief destructor. Remove all connections to other LCM modules
  *
  *  \param [in]
  *  \return
  */
spm_tclApplicationConfiguration::~spm_tclApplicationConfiguration( ){
   _poclSystemPowerManager = NULL;
   _poclCcaServiceHandler  = NULL;
   _poclSubStateHandler    = NULL;
   _poclPrx                = NULL;
   _poclDb                 = NULL;
}

/**
  *  \brief called from Factory to connect with other LCM modules
  *
  *  \param [in]
  *  \return
  */
tVoid spm_tclApplicationConfiguration::vGetReferences( ){
   // get all needed references now -> all SPM objects are now available
   SPM_GET_IF_REFERENCE_USE_VAR( _poclPrx,                ISpmOsalProxy );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclDb,                 ISpmApplicationDatabase );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSystemPowerManager, ISpmSystemPowerManager );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclCcaServiceHandler,  ISpmCcaServiceServer );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSubStateHandler,    ISpmSubStateClient );
} // vGetReferences

/**
  *  \brief called from Factory to allow/start communication with other LCM modules
  *
  *  \param [in]
  *  \return
  */
tVoid spm_tclApplicationConfiguration::vStartCommunication( ){
}

/**
  *  \brief called to get the requested ApplicationState for an application in a specific state
  *
  *   general access function. If values for app ids are specialized then they are held in _tAppConfigSpecific
  *   Therefore first app id and the configuration is checked and returned from specific map otherwise
  *   only the default value is returned.
  *
  *  \param [in] u32AppId ApplicationID
  *  \param [in]u32SystemState SystemState for which the data is retrieved
  *  \return
  */
tU32 spm_tclApplicationConfiguration::u32GetValue( tU32 u32AppId,
                                                   tU32 u32SystemState ){
   std::map < tU32, std::map < tU32, tU32 > >::const_iterator it;

   it = _tAppConfigSpecific.find( u32AppId );
   if ( it == _tAppConfigSpecific.end( ) ){
      if ( SPM_FRAMEWORK_IS_APP_CCA( u32AppId ) ){
         return( _tDefaultConfChangeable[u32SystemState] );
      } else if ( SPM_FRAMEWORK_IS_APP_LCM( u32AppId ) ){
         return( _tDefaultConfChangeableLcm[u32SystemState] );
      } else {
         ETG_TRACE_ERRMEM( ( "Illegal ApplicationType for AppId: %u", u32AppId ) );
         return AMT_C_U32_STATE_INVALID;
      }
   }
   std::map < tU32, tU32 >::const_iterator it2;
   it2 = it->second.find( u32SystemState );
   if ( it2 == it->second.end( ) ){
      if ( SPM_FRAMEWORK_IS_APP_CCA( u32AppId ) ){
         return( _tDefaultConfChangeable[u32SystemState] );

      } else if ( SPM_FRAMEWORK_IS_APP_LCM( u32AppId ) ){
         return( _tDefaultConfChangeableLcm[u32SystemState] );
      } else {
           ETG_TRACE_ERRMEM( ( "Illegal ApplicationType for AppId: %u", u32AppId ) );
           return AMT_C_U32_STATE_INVALID;
      }
   }
   return( it2->second );
} // u32GetValue

/**
  *  \brief called to get the requested ApplicationState for an application in a specific state
  *
  *   general routine to set a new application value. Only if different from default configuration
  *   a new specific value is stored.
  *
  *  \param [in] u32AppId        ApplicationID
  *  \param [in] u32SystemState  SystemState for which the data is changed
  *  \param [in] u32Value        new application state to be set
  *  \return
  */
tVoid spm_tclApplicationConfiguration::vSetValue( tU32 u32AppId,
                                                  tU32 u32SystemState,
                                                  tU32 u32Value ){
   std::map < tU32, std::map < tU32, tU32 > >::iterator it;

   it = _tAppConfigSpecific.find( u32AppId );
   if ( it != _tAppConfigSpecific.end( ) ){
      std::map < tU32, tU32 >::iterator it2;
      it2 = it->second.find( u32SystemState );
      if ( it2 != it->second.end( ) ){
         // now here a specialized value is available
         if ( SPM_FRAMEWORK_IS_APP_CCA( u32AppId ) ){
            if ( u32Value == _tDefaultConfChangeable[u32SystemState] ){
               // here the new value is like the default value, therefore remove the entry
               // now
               it->second.erase( it2 );
               return;
            }
         } else if ( SPM_FRAMEWORK_IS_APP_LCM( u32AppId ) ){
            if ( u32Value == _tDefaultConfChangeableLcm[u32SystemState] ){
               // here the new value is like the default value, therefore remove the entry
               // now
               it->second.erase( it2 );
               return;
            }
         }
      } else {
         if ( SPM_FRAMEWORK_IS_APP_CCA( u32AppId ) ){
            if ( _tDefaultConfChangeable[u32SystemState] == u32Value ){
               return; // nothing has changed
            }
         } else if ( SPM_FRAMEWORK_IS_APP_LCM( u32AppId ) ){
            if ( _tDefaultConfChangeableLcm[u32SystemState] == u32Value ){
               return; // nothing has changed
            }
         }
      }
   }
   // otherwise set a specific value
   _tAppConfigSpecific[u32AppId][u32SystemState] = u32Value;
} // vSetValue

/*!
  * \fn
  *  \brief
  *   Function is used to retrieve AppState of an application for the current SystemState
  *  \param[in] u32AppId: ApplicationId
  *
  *  \details
  *
  ******
  */
tU32 spm_tclApplicationConfiguration::u32GetNewAppState( tU32 u32AppId ){
   SPM_GET_IF_REFERENCE_NEW_VAR_VAL( poclSystemStateManager, ISpmSystemStateManager );
   return( u32GetValue( u32AppId, poclSystemStateManager->u32GetNewSystemState( ) ) );
}

/*!
  * \fn
  *  \brief
  *   Function is used to register an application in the system. It is called when
  *   an application registers (bHandleAppInitialized)
  *
  *  \param[in] u32AppId: ApplicationId
  *
  *  \details
  *
  ******
  */
tVoid spm_tclApplicationConfiguration::vNewAppInSystem( tU32 u32AppId ){
   if ( _poclCcaServiceHandler ){
      _poclCcaServiceHandler->bRegisterServices( u32AppId );
   }
}

/*!
  * \fn
  *  \brief
  *   Function is used to change the predefined application state of an app in
  *   the specified systemstate
  *
  *  \param[in] u32AppId:        ApplicationId
  *  \param[in] u32SystemState:  System state where the App-state ia changed
  *  \param[in] u32AppState:     new Application state for u32SystemState
  *  \param[in] bTriggerState:   unused
  *
  *  \details
  *
  ******
  */
tBool spm_tclApplicationConfiguration::bSetNewAppStateTable( tU32  u32AppId,
                                                             tU32  u32SystemState,
                                                             tU32  u32AppState,
                                                             tBool bTriggerState ){
   if ( u32GetValue( u32AppId, u32SystemState ) != u32AppState ){
      vSetValue( u32AppId, u32SystemState, u32AppState );

      if ( bTriggerState ){
         // _poclSubStateHandler->vSetInternalSubState(SPM_U32_INTERNAL_SUBSTATE_APP_STATE_CHANGE, 0);
      }
   }
   return( TRUE );
}

/*!
  * \fn
  *  \brief
  *   Function is used to remove all specific values for this systemstate.
  *
  *  \param[in] u32SystemState:  System state where the default app states are restored
  *
  *  \details
  *
  ******
  */
tVoid spm_tclApplicationConfiguration::vSetDefaultStateTable( tU32 u32SystemState ){
   std::map < tU32, std::map < tU32, tU32 > >::iterator it;

   for ( it = _tAppConfigSpecific.begin( ); it != _tAppConfigSpecific.end( ); ++it ){
      // iterate over all application ids
      std::map < tU32, tU32 >::iterator it2;

      it2 = it->second.find( u32SystemState );
      if ( it2 != it->second.end( ) ){
         it->second.erase( it2 );
      }
   }
}

/*!
  * \fn
  *  \brief
  *   Function is used to configure the application state based on the
  *   input provided.
  *  \param[in] u32SystemState: System state.
  *             strContent    : Application Configuration value.
  *  \details
  *  'strContent' is tokenized and based on the content the following action is taken:
  *  if " "(containing white space), then continue with next token
  *  else if "{", then application state is set to 1.
  ******
  */
tVoid spm_tclApplicationConfiguration::vSetNewStateConfiguration( tU32               u32SystemState,
                                                                  const std::string& strContent ){
   spm_tclStringTokenizer st( strContent, "{}[,] ", TRUE );
   tU32                   state              = 0;
   tBool                  bExtended          = FALSE;
   tU32                   u32AppState        = AMT_C_U32_STATE_NORMAL;
   tU32                   u32DefaultAppState = AMT_C_U32_STATE_NORMAL;

   while ( st.bHasMoreTokens( ) ){
      std::string s = st.oNextToken( );

      if ( s == " " ){
         // ignore whitespace
         continue;
      }
      switch ( state ){
         case 0:
            if ( s == "{" ){
               state = 1;
            } else {
               return; // error: do nothing
            }
            break;

         case 1:
         {
            // get application state
            std::map < std::string, tU32 >::const_iterator it;
            it = _mapAppStateNames.find( s );
            if ( it == _mapAppStateNames.end( ) ){
               return;     // app state not found
            }
            if ( bExtended ){
               u32DefaultAppState = u32AppState;
            }
            u32AppState = it->second;     // get the application state to be set
            state       = 2;
         }
         break;

         case 2:
            if ( s == "}" ){
               state = 3;
            } else if ( !bExtended && ( s == "," ) ){         // reverse behavior
               bExtended = TRUE;
               state     = 1;
            } else {
               return; // error: do nothing
            }
            break;

         case 3:
            vSetDefaultStateTable( u32SystemState );
            if ( bExtended ){
               _tDefaultConfChangeable[u32SystemState]    = u32DefaultAppState;
               _tDefaultConfChangeableLcm[u32SystemState] = _tDefaultConfLcm[u32SystemState];
            } else {
               // go back to the original default configuration
               _tDefaultConfChangeable[u32SystemState]    = _tDefaultConf[u32SystemState];
               _tDefaultConfChangeableLcm[u32SystemState] = _tDefaultConfLcm[u32SystemState];
            }
            if ( s == "[" ){
               state = 4;
            } else {
               return; // empty list: then only change the default state
            }
            break;

         case 4:
         {
            tU32         u32AppId;
            tChar       *endp;     // not really used here
            const tChar *startp = s.c_str( );

            // get a number and evaluate it
            if ( ( s.length( ) > 2 ) && ( ( s.substr( 0, 2 ).compare( "0x" ) == 0 ) || ( s.substr( 0, 2 ).compare( "0X" ) == 0 ) ) ){
               // found hex value
                u32AppId = (tU32)OSAL_u32StringToU32( startp + 2, &endp, 16 );
            } else {
                u32AppId = (tU32)OSAL_u32StringToU32( startp, &endp, 10 );
            }
            if ( std::size_t(endp - startp) != s.length( ) ){
               return;     // did not find a number, stop evaluation
            }
            if ( u32AppId < 0xffff ){
               // if it is a legal application value then set for this application
               // a new mapping from systemstate to application state.
               vSetValue( u32AppId, u32SystemState, u32AppState );
            }
            state = 5;
         }
         break;

         case 5:
            if ( s == "," ){
               state = 4;
            } else {
               return; // either } or any character finishes the evaluation
            }
            break;

         default:
            break;
      } // switch
   }
}       // vSetNewConfiguration

/*!
  * \fn
  *  \brief
  *   Function is used retrieve the supervision configuration required for process.
  *  \param[in] strContent: Supervision Configuration value
  *             processes : Process list
  *  \details
  *  'strContent' is tokenized and added into the process list.
  ******
  */
tVoid spm_tclApplicationConfiguration::vGetSupervisionConfiguration( const std::string          & strContent,
                                                                     std::vector < std::string >& processes ) const {
   spm_tclStringTokenizer st( strContent, ", ", FALSE );

   // Read in the process names. These should not be supervised in that state.
   while ( st.bHasMoreTokens( ) ){
      processes.push_back( st.oNextToken( ) );
   }
}

/*!
  * \fn
  *  \brief
  *   Function is used check if there are registry entries that overwrite the default
  *   application state configuration
  *
  *  \details
  *   parses HKEY_LOCAL_MACHINE\SOFTWARE\BLAUPUNKT\PROCESS\BASE\SPM\STATES for entries
  ******
  */
tVoid spm_tclApplicationConfiguration::vReadStateConfigurations( ){
   tU32                  as32Data[255];
   OSAL_trIOCtrlDir      rDirV = { 0,0,0 };
   OSAL_tIODescriptor    fd;
   OSAL_trIOCtrlRegistry rReg;

   fd = OSAL_IOOpen( SPM_REG_PROCESS_BASE_SPM_STATES_BASE_PATH, OSAL_EN_READONLY );
   if ( fd != OSAL_ERROR ){
      rDirV.fd        = fd;
      rDirV.s32Cookie = 0;

      while ( OSAL_s32IOControl( fd, OSAL_C_S32_IOCTRL_REGENUMVALUE, ( intptr_t )&rDirV ) != OSAL_ERROR ){
         rReg.pcos8Name = rDirV.dirent.s8Name;
         rReg.ps8Value  = (tU8*)as32Data;
         rReg.u32Size   = sizeof( as32Data );

         if ( OSAL_s32IOControl( fd, OSAL_C_S32_IOCTRL_REGGETVALUE, ( intptr_t )&rReg ) != OSAL_ERROR ){

            if ( rReg.s32Type == OSAL_C_S32_VALUE_STRING ){
               std::map < std::string, tU32 >::const_iterator it;

               // check the name with
               for ( it = _mapSystemStateNames.begin( ); it != _mapSystemStateNames.end( ); ++it ){
                  std::string s = it->first + SPM_STR_DEFAULT_NUMBER;
                  if ( s == std::string( (const tChar*)rDirV.dirent.s8Name ) ){
                     // found a matching default system state?
                     vSetNewStateConfiguration( it->second, (tChar*)as32Data );
                     break;
                  }
               }
            }
         }
      }
      OSAL_s32IOClose( fd );
   }
} // vReadConfigurations

tBool spm_tclApplicationConfiguration::bSetSpecificStateConfiguration( tU32 u32SystemState,
                                                                       tU32 u32Configuration ){
   tU32                                           as32Data[255];
   OSAL_trIOCtrlDir                               rDirV    = { 0,0,0 };
   OSAL_tIODescriptor                             fd;
   OSAL_trIOCtrlRegistry                          rReg;

   tBool                                          bUpdated = FALSE;

   std::map < tU32, std::string >::const_iterator it;

   // this result in "DOWNLOADxx"
   if(OSALUTIL_s32SaveNPrintFormat( (tChar*)as32Data, sizeof( as32Data ), "%d", u32Configuration ) == OSAL_ERROR) {
      tU32 u32ErrorReason = OSAL_u32ErrorCode( );
      ETG_TRACE_ERRMEM( ( "SPM: spm_tclApplicationConfiguration::bSetSpecificStateConfiguration !!!!!! Error detected !!!!!! cannot create string: error 0x%08X (%s)",
                          (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
   }

   it = _mapSystemStateNamesReverse.find( u32SystemState );
   if ( it == _mapSystemStateNamesReverse.end( ) ){
      return( FALSE );  // name not found
   }
   std::string confname = it->second + (const tChar*)as32Data;

   fd = OSAL_IOOpen( SPM_REG_PROCESS_BASE_SPM_STATES_BASE_PATH, OSAL_EN_READONLY );
   if ( fd != OSAL_ERROR ){
      rDirV.fd        = fd;
      rDirV.s32Cookie = 0;

      while ( OSAL_s32IOControl( fd, OSAL_C_S32_IOCTRL_REGENUMVALUE, ( intptr_t )&rDirV ) != OSAL_ERROR ){
         rReg.pcos8Name = rDirV.dirent.s8Name;
         rReg.ps8Value  = (tU8*)as32Data;
         rReg.u32Size   = sizeof( as32Data );

         if ( OSAL_s32IOControl( fd, OSAL_C_S32_IOCTRL_REGGETVALUE, ( intptr_t )&rReg ) != OSAL_ERROR ){
            if ( rReg.s32Type == OSAL_C_S32_VALUE_STRING ){
               if ( confname == std::string( (const tChar*)rDirV.dirent.s8Name ) ){
                  // found a matching default system state?
                  vSetNewStateConfiguration( u32SystemState, (tChar*)as32Data );
                  bUpdated = TRUE;
                  break;
               }
            }
         }
      }
      OSAL_s32IOClose( fd );
   }
   return( bUpdated );
} // bSetSpecificConfiguration

tVoid spm_tclApplicationConfiguration::vGetSpecificSupervisionConfiguration( tU32                         u32SystemState,
                                                                             tU32                         u32Configuration,
                                                                             std::vector < std::string >& processes ){
   tU32                                           as32Data[255];
   OSAL_trIOCtrlDir                               rDirV = { 0,0,0 };
   OSAL_tIODescriptor                             fd;
   OSAL_trIOCtrlRegistry                          rReg;

   std::map < tU32, std::string >::const_iterator it;

   // this result in "DOWNLOADxx"
   if(OSALUTIL_s32SaveNPrintFormat( (tChar*)as32Data, sizeof( as32Data ), "%d", u32Configuration ) == OSAL_ERROR) {
      tU32 u32ErrorReason = OSAL_u32ErrorCode( );
      ETG_TRACE_ERRMEM( ( "SPM: spm_tclApplicationConfiguration::vGetSpecificSupervisionConfiguration !!!!!! Error detected !!!!!! cannot create string: error 0x%08X (%s)",
                          (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
   }

   it = _mapSystemStateNamesReverse.find( u32SystemState );
   if ( it == _mapSystemStateNamesReverse.end( ) ){
      return;  // name not found
   }
   std::string confname = it->second + (const tChar*)as32Data;

   fd = OSAL_IOOpen( SPM_REG_PROCESS_BASE_SPM_NOTSUPERVISED_BASE_PATH, OSAL_EN_READONLY );
   if ( fd != OSAL_ERROR ){
      rDirV.fd        = fd;
      rDirV.s32Cookie = 0;

      while ( OSAL_s32IOControl( fd, OSAL_C_S32_IOCTRL_REGENUMVALUE, ( intptr_t )&rDirV ) != OSAL_ERROR ){
         rReg.pcos8Name = rDirV.dirent.s8Name;
         rReg.ps8Value  = (tU8*)as32Data;
         rReg.u32Size   = sizeof( as32Data );

         if ( OSAL_s32IOControl( fd, OSAL_C_S32_IOCTRL_REGGETVALUE, ( intptr_t )&rReg ) != OSAL_ERROR ){
            if ( rReg.s32Type == OSAL_C_S32_VALUE_STRING ){
               if ( confname == std::string( (const tChar*)rDirV.dirent.s8Name ) ){
                  // found a matching default system state?
                  vGetSupervisionConfiguration( (tChar*)as32Data, processes );
                  break;
               }
            }
         }
      }
      OSAL_s32IOClose( fd );
   }
} // vGetSpecificSupervisionConfiguration

tVoid spm_tclApplicationConfiguration::vTraceInfo( tU32 u32SystemState ){
   std::map < tU32, std::map < tU32, tU32 > >::const_iterator it;

   ETG_TRACE_FATAL( ( "ApplicationConfiguration for %u",
                      ETG_ENUM( SPM_SYSTEM_STATES, u32SystemState ) ) );
   ETG_TRACE_FATAL( ( "Original:     %u",
                      ETG_ENUM( SPM_APPSTATE_DEF, _tDefaultConf[u32SystemState] )
                      ) );
   ETG_TRACE_FATAL( ( "Defaultstate: %u",
                      ETG_ENUM( SPM_APPSTATE_DEF, _tDefaultConfChangeable[u32SystemState] )
                      ) );
   ETG_TRACE_FATAL( ( "Defaultstate LCM: %u",
                      ETG_ENUM( SPM_APPSTATE_DEF, _tDefaultConfChangeableLcm[u32SystemState] )
                      ) );
   for ( it = _tAppConfigSpecific.begin( ); it != _tAppConfigSpecific.end( ); ++it ){
      std::map < tU32, tU32 >::const_iterator it2;

      it2 = it->second.find( u32SystemState );
      if ( it2 != it->second.end( ) ){
         ETG_TRACE_FATAL( ( "Exception: AppId = %u, State = %u ",
                            ETG_ENUM( ail_u16AppId, it->first ),
                            ETG_ENUM( SPM_APPSTATE, it2->second )
                            ) );
      }
   }
} // vTraceInfo

