/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         LcmBaseAppComponent.cpp
  * @brief        Implementation of the LcmBaseAppComponent component (hand crafted code)
  * @copyright    (C) 2014 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#include "LcmBaseAppComponent.h"
#include "asf/core/Application.h"

#include <stdio.h>
#include <sys/types.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>

#ifndef LCM_UNIT_TESTS
#include <systemd/sd-daemon.h> // you also need to add -lsystemd-daemon to the linker settings
#include <signal.h>
#endif
// \todo: boost::shared_ptr is used here cause by ASF interface.
namespace org {
namespace bosch {
namespace cm {
namespace lcm {
namespace lcmbaseappcomponent {

using namespace ::org::genivi::NodeStateManager::Consumer;
using namespace ::org::genivi::NodeStateManager::LifeCycleConsumer;
using namespace ::org::genivi::NodeStateManager::LifecycleControl;
using namespace ::org::bosch::cm::lcm;
using namespace ::asf::core;

/*################################################
  ################################################
   SignalHandler
  ################################################
  ################################################ */
void LcmBaseApp_sig_handler( int signum ){
   #ifndef LCM_UNIT_TESTS
      if ( signum == SIGTERM ){
         sd_notifyf( 0, "STOPPING=1\n"                 // Tells the init system that service is beginning its shutdown.
                        "STATUS=Handling SIGTERM...\n" // Passes a single-line status string back to the init system that describes the daemon state.
                                                       // This is free-form and can be used for various purposes
                        "MAINPID=%lu\nPARENTID=%lu",
                     (unsigned long)getpid( ),
                     (unsigned long)getppid( ) );
      } else {
         sd_notifyf( 0, "STATUS=UNKNOWN SIGNAL %lu...\n" // Passes a single-line status string back to the init system that describes the daemon state.
                                                         // This is free-form and can be used for various purposes
                        "MAINPID=%lu\nPARENTID=%lu",
                     (unsigned long)signum,
                     (unsigned long)getpid( ),
                     (unsigned long)getppid( ) );
      }
   #endif // ifndef LCM_UNIT_TESTS
   if ( FALSE == asf::core::Application::getApplication( )->shutdownCurrentApplication( 0 /* exitcode */ ) ){
      _exit( 0 );
   }
   _exit( 0 );
} // LcmBaseApp_sig_handler

/*################################################
  ################################################
   LcmBaseAppComponent
  ################################################
  ################################################ */

LcmBaseAppComponent::LcmBaseAppComponent( const std::string      & busName,
                                          const std::string      & objName,
                                          LcmApplicationControlIF& ApplicationCb )
   : //stubs
   LifeCycleConsumerStub( "LifeCycleConsumerPort" ),
   // variables
   _sNotifySocket( "" ),
   _SystemdWdNotifyTimeMSec( 0 ),
   _SystemdWdPreTimeMSec( 0 ),
   _SystemdWdNotifyPreTimeout_Percentage( 90 ),
   _SystemdStartupPreTimeMSec( 0 ),
   _SystemdStartupPreTimeout_Percentage( 90 ),
   _ConsumerServiceRegistered( false ),
   _ShutdownClientRegistrationDone( false ),
   _AppStartReady( false ),
   _currentAppMode( _Nsm_Shutdown_Type_e__NSM_Shutdown_Type_Runup ),
   _busName( busName ),
   _objName( objName ),
   _ApplicationCb( &ApplicationCb ),
   // proxies
   _consumerProxy( ConsumerProxy::createProxy( "ConsumerPort", * this ) ){

   signal( SIGTERM, LcmBaseApp_sig_handler );

   char *EnvVar = getenv( "BOSCH_PRETIMEOUT_PERCENTAGE" );
   if ( EnvVar == NULL ){
      LOG_INFO( "LcmBaseAppComponent::LcmBaseAppComponent No BOSCH_PRETIMEOUT_PERCENTAGE defined. 90% will be used" );
   } else {
      _SystemdWdNotifyPreTimeout_Percentage = atoi( std::string( EnvVar ).c_str( ) );
      if ( _SystemdWdNotifyPreTimeout_Percentage >= 100 ){
         LOG_ERROR( "LcmBaseAppComponent::LcmBaseAppComponent _SystemdWdNotifyPreTimeout_Percentage must be < 100" );
         _SystemdWdNotifyPreTimeout_Percentage = 90;
      }
      _SystemdStartupPreTimeout_Percentage  = _SystemdWdNotifyPreTimeout_Percentage;
      LOG_INFO( "LcmBaseAppComponent::LcmBaseAppComponent _SystemdWdNotifyPreTimeout_Percentage = %i%", _SystemdStartupPreTimeMSec );
      LOG_INFO( "LcmBaseAppComponent::LcmBaseAppComponent _SystemdStartupPreTimeout_Percentage  = %i%", _SystemdStartupPreTimeout_Percentage );
   }

   EnvVar = getenv( "BOSCH_WD_PRETIMEOUT_PERCENTAGE" );
   if ( EnvVar == NULL ){
      LOG_WARN( "LcmBaseAppComponent::LcmBaseAppComponent No BOSCH_WD_PRETIMEOUT_PERCENTAGE defined. 90% will be used" );
   } else {
      _SystemdWdNotifyPreTimeout_Percentage = atoi( std::string( EnvVar ).c_str( ) );
      if ( _SystemdWdNotifyPreTimeout_Percentage >= 100 ){
         LOG_ERROR( "LcmBaseAppComponent::LcmBaseAppComponent _SystemdWdNotifyPreTimeout_Percentage must be < 100" );
         _SystemdWdNotifyPreTimeout_Percentage = 90;
      }
      LOG_INFO( "LcmBaseAppComponent::LcmBaseAppComponent _SystemdWdNotifyPreTimeout_Percentage = %i%", _SystemdStartupPreTimeMSec );
   }

   EnvVar = getenv( "BOSCH_STARTUP_PRETIMEOUT_PERCENTAGE" );
   if ( EnvVar == NULL ){
      LOG_WARN( "LcmBaseAppComponent::LcmBaseAppComponent No BOSCH_STARTUP_PRETIMEOUT_PERCENTAGE defined. 90% will be used" );
   } else {
      _SystemdStartupPreTimeout_Percentage = atoi( std::string( EnvVar ).c_str( ) );
      if ( _SystemdStartupPreTimeout_Percentage >= 100 ){
         LOG_ERROR( "LcmBaseAppComponent::LcmBaseAppComponent _SystemdStartupPreTimeout_Percentage must be < 100" );
         _SystemdStartupPreTimeout_Percentage = 90;
      }
      LOG_INFO( "LcmBaseAppComponent::LcmBaseAppComponent _SystemdStartupPreTimeout_Percentage = %i%", _SystemdStartupPreTimeMSec );
   }

   if ( !busName.empty( ) && !objName.empty( ) ){
      LOG_INFO( "LcmBaseAppComponent::LcmBaseAppComponent called\nbusname=%s\nobjname=%s", busName.c_str( ), objName.c_str( ) );

      _CommandLineArguments = asf::core::Application::getApplication( )->getArguments( );

      // get the set startuptime time in seconds
      EnvVar                = getenv( "BOSCH_STARTUPTIME_SEC" );
      if ( EnvVar == NULL ){
         LOG_WARN( "LcmBaseAppComponent::LcmBaseAppComponent No BOSCH_STARTUPTIME_SEC defined. Debug of startup behaviour limited" );
      } else {
         _SystemdStartupPreTimeMSec = ( atoi( std::string( EnvVar ).c_str( ) ) * ( 1000 ) ) * _SystemdStartupPreTimeout_Percentage / 100; // start a timer that runs out "shortly"
                                                                                                                                          // before timeout
         LOG_INFO( "LcmBaseAppComponent::LcmBaseAppComponent Startup pre-timeout = %i ms", _SystemdStartupPreTimeMSec );
         _SystemdStartupPreTimer.start( * this, _SystemdStartupPreTimeMSec, Timer::Once );
      }

      // get the set watchdog time in microseconds
      uint64 u64WatchdoguSec = 0;
      if ( sd_watchdog_enabled( 0, &u64WatchdoguSec ) <= 0 ){
         LOG_ERROR( "No Watchdog time defined. Watchdog is disabled" );
         _SystemdWdNotifyTimer.start( * this, _SystemdWdNotifyTimeMSec, Timer::Once );
      } else {

         _SystemdWdNotifyTimeMSec = (uint32)( u64WatchdoguSec / 1000 );

         _SystemdWdPreTimeMSec    = ( _SystemdWdNotifyTimeMSec * _SystemdWdNotifyPreTimeout_Percentage ) / 100;   // when time is almost up and still no sd_notify has been send,
                                                                                                                  // dump current state to ErrMem
         _SystemdWdNotifyPreTimer.start( * this, _SystemdWdPreTimeMSec, Timer::Once );

         LOG_INFO( "Watchdog timeout = %i ms", _SystemdWdNotifyTimeMSec );
         _SystemdWdNotifyTimeMSec = _SystemdWdNotifyTimeMSec / 3;   // use one third the time passed by systemd to retrigger the Watchdog, this also allows a one time miss of the
                                                                    // trigger (OnAppWatchdog==FALSE)
         _SystemdWdNotifyTimer.start( * this, _SystemdWdNotifyTimeMSec, Timer::Once );
      }

      // check for set notification socket
      EnvVar = getenv( "NOTIFY_SOCKET" );
      if ( EnvVar == NULL ){
         LOG_WARN( "LcmBaseAppComponent::LcmBaseAppComponent No NOTIFY_SOCKET defined" );
      } else {
         _sNotifySocket = std::string( EnvVar );
         LOG_INFO( "LcmBaseAppComponent::LcmBaseAppComponent NOTIFY_SOCKET = %s", _sNotifySocket.c_str( ) );
      }

   } else {
      LOG_ERROR( "LcmBaseAppComponent::LcmBaseAppComponent Busname or ObjectName is empty" );
   }
}

LcmBaseAppComponent::~LcmBaseAppComponent( ){
   LOG_INFO( "LcmBaseAppComponent::~LcmBaseAppComponent called" );
}

void LcmBaseAppComponent::setAppStartReady( ){
   LOG_INFO( "LcmBaseAppComponent::setAppStartReady ..........." );
   _AppStartReady = true;
   CheckAppStartReady( );
}

void LcmBaseAppComponent::CheckAppStartReady( ){
   if ( ( TRUE == _ConsumerServiceRegistered )
        && ( TRUE == _ShutdownClientRegistrationDone )
        && ( TRUE == _AppStartReady )
        ){
      #ifndef LCM_UNIT_TESTS
         int ErrorNumber = sd_notifyf( 0, "READY=1\n"                       // Tells the init system that daemon startup is finished.
                                                                            // This is only used by systemd if the service definition file has Type=notify set.
                                          "STATUS=Processing requests...\n" // Passes a single-line status string back to the init system that describes the daemon state.
                                                                            // This is free-form and can be used for various purposes
                                          "MAINPID=%lu\nPARENTID=%lu",
                                       (unsigned long)getpid( ),
                                       (unsigned long)getppid( ) );
         if ( 0 >= ErrorNumber ){
            PrintSdNotifyErrMemEntries( ErrorNumber, "READY" );
         }
      #endif
      LOG_INFO( "LcmBaseAppComponent::CheckAppStartReady send READY=1 ........." );
   } else {
      LOG_INFO(                                             "LcmBaseAppComponent::CheckAppStartReady Failed" );
      LOG_INFO(                                                       "LcmBaseAppComponent::busname    = %s", _busName.c_str( ) );
      LOG_INFO(                                                       "LcmBaseAppComponent::objectname = %s", _objName.c_str( ) );
      LOG_INFO( "_ConsumerServiceRegistered = %d, _ShutdownClientRegistrationDone = %d, _AppStartReady = %d", _ConsumerServiceRegistered, _ShutdownClientRegistrationDone, _AppStartReady );
   }
}              // CheckAppStartReady

void LcmBaseAppComponent::PrintTimeoutErrMemEntries( const std::string& strType ){
   int FileHandle;

   FileHandle = open( "/dev/errmem", O_WRONLY );
   if ( FileHandle != - 1 ){
      std::string Message = "LcmBaseApp: " + strType + " pre warning\n";
      if ( write( FileHandle, Message.c_str( ), Message.length( ) ) == - 1 ){
         LOG_ERROR( "LcmBaseAppComponent::PrintSdNotifyErrMemEntries: FAILED to write /dev/errmem. Error: %s", strerror( errno ) );
      }

      Message  = "busname    = " + _busName + "\n";
      if ( write( FileHandle, Message.c_str( ), Message.length( ) ) == - 1 ){
         LOG_ERROR( "LcmBaseAppComponent::PrintSdNotifyErrMemEntries: FAILED to write /dev/errmem. Error: %s", strerror( errno ) );
      }

      Message  = "objectname = " + _objName + "\n";
      if ( write( FileHandle, Message.c_str( ), Message.length( ) ) == - 1 ){
         LOG_ERROR( "LcmBaseAppComponent::PrintSdNotifyErrMemEntries: FAILED to write /dev/errmem. Error: %s", strerror( errno ) );
      }

      Message  = "sd_notify (WATCHDOG) pretimeout configured to ";
      Message += std::to_string( _SystemdWdNotifyPreTimeout_Percentage );
      Message += "% (timeout = " + std::to_string( _SystemdWdPreTimeMSec );
      Message += ", running = " + std::to_string( _SystemdWdNotifyPreTimer.isActive( ) ) + ")\n";
      if ( write( FileHandle, Message.c_str( ), Message.length( ) ) == - 1 ){
         LOG_ERROR( "LcmBaseAppComponent::PrintSdNotifyErrMemEntries: FAILED to write /dev/errmem. Error: %s", strerror( errno ) );
      }

      Message  = "sd_notify (READY) pretimeout configured to ";
      Message += std::to_string( _SystemdStartupPreTimeout_Percentage );
      Message += "% (timeout = " + std::to_string( _SystemdStartupPreTimeMSec );
      Message += ", running = " + std::to_string( _SystemdStartupPreTimer.isActive( ) ) + ")\n";
      if ( write( FileHandle, Message.c_str( ), Message.length( ) ) == - 1 ){
         LOG_ERROR( "LcmBaseAppComponent::PrintSdNotifyErrMemEntries: FAILED to write /dev/errmem. Error: %s", strerror( errno ) );
      }

      Message  = "_ConsumerServiceRegistered      = " + std::to_string( _ConsumerServiceRegistered ) + "\n";
      if ( write( FileHandle, Message.c_str( ), Message.length( ) ) == - 1 ){
         LOG_ERROR( "LcmBaseAppComponent::PrintSdNotifyErrMemEntries: FAILED to write /dev/errmem. Error: %s", strerror( errno ) );
      }

      Message  = "_ShutdownClientRegistrationDone = " + std::to_string( _ShutdownClientRegistrationDone ) + "\n";
      if ( write( FileHandle, Message.c_str( ), Message.length( ) ) == - 1 ){
         LOG_ERROR( "LcmBaseAppComponent::PrintSdNotifyErrMemEntries: FAILED to write /dev/errmem. Error: %s", strerror( errno ) );
      }

      Message  = "_AppStartReady                  = " + std::to_string( _AppStartReady ) + "\n";
      if ( write( FileHandle, Message.c_str( ), Message.length( ) ) == - 1 ){
         LOG_ERROR( "LcmBaseAppComponent::PrintSdNotifyErrMemEntries: FAILED to write /dev/errmem. Error: %s", strerror( errno ) );
      }

      if ( close( FileHandle ) == - 1 ){
         LOG_ERROR( "LcmBaseAppComponent::PrintSdNotifyErrMemEntries: FAILED to close /dev/errmem. Error: %s", strerror( errno ) );
      }
   }
}              // PrintTimeoutErrMemEntries

void LcmBaseAppComponent::PrintSdNotifyErrMemEntries( const int          ErrorNumber,
                                                      const std::string& strType ){
   // https://www.freedesktop.org/software/systemd/man/sd_notify.html
   int FileHandle;

   FileHandle = open( "/dev/errmem", O_WRONLY );
   if ( FileHandle != - 1 ){
      std::string Message = "LcmBaseApp: sd_notify for " + strType + " did not succeed\n   - busname=" + _busName + "\n   - objName=" + _objName + "\n";
      if ( write( FileHandle, Message.c_str( ), Message.length( ) ) == - 1 ){
         LOG_ERROR( "LcmBaseAppComponent::PrintSdNotifyErrMemEntries: FAILED to write /dev/errmem. Error: %s", strerror( errno ) );
      }

      // for ErrorNumber see https://www.freedesktop.org/software/systemd/man/sd_notify.html
      if ( ErrorNumber == 0 ){
         Message = "systemd: No systemd socket found.\n";
      } else {
         Message = "systemd: " + (std::string)( strerror( ErrorNumber ) );
      }
      if ( write( FileHandle, Message.c_str( ), Message.length( ) ) == - 1 ){
         LOG_ERROR( "LcmBaseAppComponent::PrintSdNotifyErrMemEntries: FAILED to write /dev/errmem. Error: %s", strerror( errno ) );
      }

      if ( close( FileHandle ) == - 1 ){
         LOG_ERROR( "LcmBaseAppComponent::PrintSdNotifyErrMemEntries: FAILED to close /dev/errmem. Error: %s", strerror( errno ) );
      }
   }
}              // PrintSdNotifyErrMemEntries

void LcmBaseAppComponent::SendLifeCycleRequestCompleteRequest( ){
   if ( _lifeCycleRequestRequest ){
      if ( _lifeCycleRequestRequest->hasRequestId( )
           && _lifeCycleRequestRequest->hasRequest( ) ){
         _consumerProxy->sendLifecycleRequestCompleteRequest( * this,
                                                              _lifeCycleRequestRequest->getRequestId( ),
                                                              _NsmErrorStatus_e__NsmErrorStatus_Ok );

         LOG_INFO( "LcmBaseAppComponent:: SendLifeCycleRequestCompleteRequest ..." );
      }
   }
}

// ServiceAvailableIF implementation
void LcmBaseAppComponent::onAvailable( const boost::shared_ptr < Proxy >& proxy,
                                       const ServiceStateChange         & stateChange ){
   if ( ( proxy == _consumerProxy ) && ( stateChange.hasCurrentState( ) ) ){
      LOG_DEBUG( "LcmBaseAppComponent::onAvailable::_consumerProxy service available: ServiceState: %s", ServiceState_Name( stateChange.getCurrentState( ) ) );

      if ( ServiceState__Available == stateChange.getCurrentState( ) ){
         _ConsumerServiceRegistered = true;
         _consumerProxy->sendRegisterShutdownClientRequest( * this,
                                                            _busName,
                                                            _objName,
                                                            _Nsm_Shutdown_Type_e__Nsm_Shutdown_Type_Normal | _Nsm_Shutdown_Type_e__NSM_Shutdown_Type_Fast, /*shutdownMode*/
                                                            10000 /*timeoutMs*/ );
         LOG_DEBUG( "LcmBaseAppComponent::sendRegisterShutdownClientRequest" );
         CheckAppStartReady( );
      } else {
         LOG_ERROR( "LcmBaseAppComponent::onAvailable:: Service is not available" );
      }
   } else {
      LOG_WARN( "LcmBaseAppComponent::onAvailable unknown service" );
   }
}              // onAvailable

void LcmBaseAppComponent::onUnavailable( const boost::shared_ptr < Proxy >& proxy,
                                         const ServiceStateChange         & stateChange ){
   if ( ( proxy == _consumerProxy ) && ( stateChange.hasCurrentState( ) ) ){
      LOG_DEBUG( "LcmBaseAppComponent::onUnavailable _consumerProxy service not available anym::onUnavailable: ServiceState: %s", ServiceState_Name( stateChange.getCurrentState( ) ) );
      _ConsumerServiceRegistered = false;
   } else {
      LOG_WARN( "LcmBaseAppComponent::onUnavailable unknown service" );
   }
}

// TimerCallbackIF implementation
void LcmBaseAppComponent::onExpired( asf::core::Timer                            & timer,
                                     boost::shared_ptr < asf::core::TimerPayload > payload ){
   LOG_INFO( "LcmBaseAppComponent::onExpired ..........." );
   if ( ( timer.getAct( ) == _SystemdWdNotifyTimer.getAct( ) )
        && ( &timer == &_SystemdWdNotifyTimer )
        ){
      if ( _ApplicationCb->OnAppWatchdog( ) ){
         #ifndef LCM_UNIT_TESTS
            int ErrorNumber = sd_notify( 0, "WATCHDOG=1" );
            if ( 0 >= ErrorNumber ){
               PrintSdNotifyErrMemEntries( ErrorNumber, "WATCHDOG" );
            }
         #endif
         LOG_DEBUG( "LcmBaseAppComponent::onExpired::sd_notify(WATCHDOG=1)" );
         _SystemdWdNotifyTimer.start( * this, _SystemdWdNotifyTimeMSec, Timer::Once );
         _SystemdWdNotifyPreTimer.start( );
      } else {
         LOG_ERROR( "LcmBaseAppComponent::onExpired: OnAppWatchdog returned FALSE" );
      }
   } else if ( ( timer.getAct( ) == _SystemdWdNotifyPreTimer.getAct( ) )
               && ( &timer == &_SystemdWdNotifyPreTimer )
               ){
      // write current state into ErrorMemory
      PrintTimeoutErrMemEntries( "Watchdog" );
   } else if ( ( timer.getAct( ) == _SystemdStartupPreTimer.getAct( ) )
               && ( &timer == &_SystemdStartupPreTimer )
               ){
      // write current state into ErrorMemory
      PrintTimeoutErrMemEntries( "StartupTime" );
   } else {
      LOG_ERROR( "LcmBaseAppComponent::onExpired: Unknown timer expired" );
   }

   if ( payload->hasReason( ) && payload->hasRepeatCount( ) ){
      LOG_DEBUG( "LcmBaseAppComponent::onExpired: TimerPayload Reason: %s, Repeat Count:%u", TimerPayload_Reason_Name( payload->getReason( ) ), payload->getRepeatCount( ) );
   }
}              // onExpired

// LifeCycleConsumerStub implementation
void LcmBaseAppComponent::onLifecycleRequestRequest( const ::boost::shared_ptr < LifecycleRequestRequest >& request ){
   _NsmErrorStatus_e AppChangeResult = _NsmErrorStatus_e__NsmErrorStatus_NotSet;

   LOG_INFO( "LcmBaseAppComponent::onLifecycleRequestRequest ..........." );
   if ( request->hasRequestId( ) && request->hasRequest( ) ){
      _lifeCycleRequestRequest = request;
      if ( uint32( _Nsm_Shutdown_Type_e__NSM_Shutdown_Type_Runup ) == request->getRequest( ) ){
         _ApplicationCb->onLoadPersistency( );
         AppChangeResult = _ApplicationCb->OnAppModeChange( _Nsm_Shutdown_Type_e__NSM_Shutdown_Type_Runup, _currentAppMode );
         _currentAppMode = _Nsm_Shutdown_Type_e__NSM_Shutdown_Type_Runup;
      } else if ( ( _Nsm_Shutdown_Type_e__Nsm_Shutdown_Type_Normal == request->getRequest( ) )
                  || ( _Nsm_Shutdown_Type_e__NSM_Shutdown_Type_Fast == request->getRequest( ) )
                  ){
         AppChangeResult = _ApplicationCb->OnAppModeChange( _Nsm_Shutdown_Type_e__Nsm_Shutdown_Type_Normal, _currentAppMode );
         _ApplicationCb->onSavePersistency( );
         _currentAppMode = _Nsm_Shutdown_Type_e__Nsm_Shutdown_Type_Normal;
      } else {
         LOG_ERROR( "LcmBaseAppComponent::onLifecycleRequestRequest::Unknown ApplicationModeRequest %i", request->getRequest( ) );
      }
      if ( ( _NsmErrorStatus_e__NsmErrorStatus_Ok == AppChangeResult ) || ( _NsmErrorStatus_e__NsmErrorStatus_ResponsePending == AppChangeResult ) ){
         LOG_DEBUG( "LcmBaseAppComponent::onLifecycleRequestRequest::sendLifecycleRequestResponse for Request % / RequestId %u",
                    request->getRequest( ),
                    request->getRequestId( )
                    );
         sendLifecycleRequestResponse( AppChangeResult );
      } else {
         LOG_ERROR( "LcmBaseAppComponent::onLifecycleRequestRequest::Not processed AppChangeResult %s", _NsmErrorStatus_e_Name( AppChangeResult ) );
      }
   }
}              // onLifecycleRequestRequest

void LcmBaseAppComponent::onLifecycleRequestCompleteError( const ::boost::shared_ptr < ConsumerProxy >                & proxy,
                                                           const ::boost::shared_ptr < LifecycleRequestCompleteError >& error ){
   if ( proxy == _consumerProxy ){
      LOG_DEBUG( "LcmBaseAppComponent::onLifecycleRequestCompleteError::_consumerProxy service available" );

      if ( error->hasName( ) && error->hasMessage( ) ){
         LOG_ERROR( "LcmBaseAppComponent::onLifecycleRequestCompleteError::ErrorName:%s,Errormessage:%s", error->getName( ).c_str( ), error->getMessage( ).c_str( ) );
         _ApplicationCb->OnAppLifecycleRequestCompleteError( _NsmErrorStatus_e__NsmErrorStatus_Dbus, error->getName( ), error->getName( ) );
      } else {
         LOG_INFO( "LcmBaseAppComponent::onLifecycleRequestCompleteError::No Error in D-Bus communication" );
      }
   } else {
      LOG_WARN( "LcmBaseAppComponent::onLifecycleRequestCompleteError:: unknown service" );
   }
}              // onLifecycleRequestCompleteError

void LcmBaseAppComponent::onLifecycleRequestCompleteResponse( const ::boost::shared_ptr < ConsumerProxy >                   & proxy,
                                                              const ::boost::shared_ptr < LifecycleRequestCompleteResponse >& response ){
   if ( proxy == _consumerProxy ){
      LOG_DEBUG( "LcmBaseAppComponent::onLifecycleRequestCompleteResponse::_consumerProxy service available" );

      if ( response->hasErrorCode( ) ){
         LOG_INFO( "LcmBaseAppComponent::onLifecycleRequestCompleteResponse::NsmErrorStatus:%s", _NsmErrorStatus_e_Name( response->getErrorCode( ) ) );

         if ( ( _NsmErrorStatus_e__NsmErrorStatus_Ok == response->getErrorCode( ) ) || ( _NsmErrorStatus_e__NsmErrorStatus_ResponsePending == response->getErrorCode( ) ) ){
            _ApplicationCb->OnAppLifecycleRequestCompleteResponse( response->getErrorCode( ) );
         } else {
            _ApplicationCb->OnAppLifecycleRequestCompleteError( response->getErrorCode( ) );
         }
      } else {
         LOG_INFO( "LcmBaseAppComponent::onLifecycleRequestCompleteResponse::Response has no error code " );
      }
   } else {
      LOG_WARN( "LcmBaseAppComponent::onLifecycleRequestCompleteResponse unknown service" );
   }
}              // onLifecycleRequestCompleteResponse

// ConsumerProxy
// RegisterShutdownClientCallbackIF
void LcmBaseAppComponent::onRegisterShutdownClientError( const ::boost::shared_ptr < ConsumerProxy >              & proxy,
                                                         const ::boost::shared_ptr < RegisterShutdownClientError >& error ){
   if ( proxy == _consumerProxy ){
      LOG_DEBUG( "LcmBaseAppComponent::onRegisterShutdownClientError::_consumerProxy service available" );

      if ( error->hasName( ) && error->hasMessage( ) ){
         LOG_ERROR( "LcmBaseAppComponent::onRegisterShutdownClientError::ErrorName:%s,Errormessage:%s", error->getName( ).c_str( ), error->getMessage( ).c_str( ) );
         _ShutdownClientRegistrationDone = false;
         _ApplicationCb->OnAppLifecycleRequestCompleteError( _NsmErrorStatus_e__NsmErrorStatus_Dbus, error->getName( ), error->getName( ) );
      } else {
         LOG_DEBUG( "LcmBaseAppComponent::onRegisterShutdownClientError::No Error in D-Bus communication" );
      }
   } else {
      LOG_WARN( "LcmBaseAppComponent::onRegisterShutdownClientError unknown service" );
   }
}              // onRegisterShutdownClientError

void LcmBaseAppComponent::onRegisterShutdownClientResponse( const ::boost::shared_ptr < ConsumerProxy >                 & proxy,
                                                            const ::boost::shared_ptr < RegisterShutdownClientResponse >& response ){
   if ( proxy == _consumerProxy ){
      LOG_DEBUG( "LcmBaseAppComponent::onRegisterShutdownClientResponse::_consumerProxy service available" );
      _ShutdownClientRegistrationDone = false;
      if ( response->hasErrorCode( ) ){
         LOG_INFO( "LcmBaseAppComponent::onRegisterShutdownClientResponse:: NsmErrorStatus: %s", _NsmErrorStatus_e_Name( response->getErrorCode( ) ) );

         if ( ( _NsmErrorStatus_e__NsmErrorStatus_Ok == response->getErrorCode( ) ) || ( _NsmErrorStatus_e__NsmErrorStatus_ResponsePending == response->getErrorCode( ) ) ){
            _ShutdownClientRegistrationDone = true;
            _ApplicationCb->OnAppLifecycleRequestCompleteResponse( response->getErrorCode( ) );
         } else {
            _ApplicationCb->OnAppLifecycleRequestCompleteError( response->getErrorCode( ) );
         }
      } else {
         LOG_INFO( "LcmBaseAppComponent::onRegisterShutdownClientResponse::Response has no error code " );
      }
      CheckAppStartReady( );
      _ApplicationCb->OnLcmRegistered( );

   } else {
      LOG_WARN( "LcmBaseAppComponent::onRegisterShutdownClientResponse unknown service" );
   }
}              // onRegisterShutdownClientResponse

// UnRegisterShutdownClientCallbackIF
void LcmBaseAppComponent::onUnRegisterShutdownClientError( const ::boost::shared_ptr < ConsumerProxy >                & proxy,
                                                           const ::boost::shared_ptr < UnRegisterShutdownClientError >& error ){
   if ( proxy == _consumerProxy ){
      LOG_DEBUG( "LcmBaseAppComponent::onUnRegisterShutdownClientError::_consumerProxy service available" );

      if ( error->hasName( ) && error->hasMessage( ) ){
         LOG_ERROR( "LcmBaseAppComponent::onUnRegisterShutdownClientError::ErrorName:%s,Errormessage:%s", error->getName( ).c_str( ), error->getMessage( ).c_str( ) );
         _ShutdownClientRegistrationDone = false;
         _ApplicationCb->OnAppLifecycleRequestCompleteError( _NsmErrorStatus_e__NsmErrorStatus_Dbus, error->getName( ), error->getName( ) );
      } else {
         LOG_INFO( "LcmBaseAppComponent::onUnRegisterShutdownClientError::No Error in D-Bus communication" );
      }
   } else {
      LOG_WARN( "LcmBaseAppComponent::onUnRegisterShutdownClientError unknown service" );
   }
}              // onUnRegisterShutdownClientError

void LcmBaseAppComponent::onUnRegisterShutdownClientResponse( const ::boost::shared_ptr < ConsumerProxy >                   & proxy,
                                                              const ::boost::shared_ptr < UnRegisterShutdownClientResponse >& response ){
   if ( proxy == _consumerProxy ){
      LOG_DEBUG( "LcmBaseAppComponent::onUnRegisterShutdownClientResponse::_consumerProxy service available" );
      _ShutdownClientRegistrationDone = false;

      if ( response->hasErrorCode( ) ){
         LOG_INFO( "LcmBaseAppComponent::onUnRegisterShutdownClientResponse:: NsmErrorStatus: %s", _NsmErrorStatus_e_Name( response->getErrorCode( ) ) );

         if ( ( _NsmErrorStatus_e__NsmErrorStatus_Ok == response->getErrorCode( ) ) || ( _NsmErrorStatus_e__NsmErrorStatus_ResponsePending == response->getErrorCode( ) ) ){
            _ApplicationCb->OnAppLifecycleRequestCompleteResponse( response->getErrorCode( ) );
         } else {
            _ApplicationCb->OnAppLifecycleRequestCompleteError( response->getErrorCode( ) );
         }
      } else {
         LOG_INFO( "LcmBaseAppComponent::onUnRegisterShutdownClientResponse::Response has no error code " );
      }
   } else {
      LOG_WARN( "LcmBaseAppComponent::onUnRegisterShutdownClientResponse unknown service" );
   }

}              // onUnRegisterShutdownClientResponse

/*################################################
  ################################################
   LcmApplicationControlIF
  ################################################
  ################################################ */
//Error and response Handing to be implmented by derived application
void LcmApplicationControlIF::OnAppLifecycleRequestCompleteError( _NsmErrorStatus_e  error,
                                                                  const std::string& name,
                                                                  const std::string& message ){
   LOG_WARN( "LcmApplicationControlIF::OnAppLifecycleRequestCompleteError ,function to be implemented in derived class ,\
            Error=%s, Name=%s, Message=%s", _NsmErrorStatus_e_Name( error ), name.c_str( ), message.c_str( ) );
}

void LcmApplicationControlIF::OnAppLifecycleRequestCompleteResponse( _NsmErrorStatus_e response ){
   LOG_WARN( "LcmApplicationControlIF::OnAppLifecycleRequestCompleteResponse,function to be implemented in derived class , Response=%s", _NsmErrorStatus_e_Name( response ) );
}

void LcmApplicationControlIF::onAppRegisterShutdownClientError( _NsmErrorStatus_e  error,
                                                                const std::string& name,
                                                                const std::string& message ){
   LOG_WARN( "LcmApplicationControlIF::onAppRegisterShutdownClientError, function to be implemented in derived class ,\
            Error=%s, Name=%s, Message=%s", _NsmErrorStatus_e_Name( error ), name.c_str( ), message.c_str( ) );
}

void LcmApplicationControlIF::onAppRegisterShutdownClientResponse( _NsmErrorStatus_e response ){
   LOG_WARN( "LcmApplicationControlIF::onAppRegisterShutdownClientResponse ,function to be implemented in derived class , Response=%s", _NsmErrorStatus_e_Name( response ) );
}

void LcmApplicationControlIF::onAppUnRegisterShutdownClientError( _NsmErrorStatus_e  error,
                                                                  const std::string& name,
                                                                  const std::string& message ){
   LOG_WARN( "LcmApplicationControlIF::onAppUnRegisterShutdownClientError, function to be implemented in derived class ,\
            Error=%s, Name=%s, Message=%s", _NsmErrorStatus_e_Name( error ), name.c_str( ), message.c_str( ) );
}

void LcmApplicationControlIF::onAppUnRegisterShutdownClientResponse( _NsmErrorStatus_e response ){
   LOG_WARN( "LcmApplicationControlIF::onAppUnRegisterShutdownClientResponse ,function to be implemented in derived class , Response=%s", _NsmErrorStatus_e_Name( response ) );
}

LcmApplicationControlIF::~LcmApplicationControlIF( ){
   LOG_INFO( "LcmApplicationControlIF::~LcmBaseAppComponent called" );
}

/*################################################
  ################################################
   LcmExtendedBaseAppComponent
  ################################################
  ################################################ */
// LifecycleControlProxy implementation
LcmExtendedBaseAppComponent::LcmExtendedBaseAppComponent( std::string busName,
                                                          std::string objName )
   :           // ASF Base
   BaseComponent( ),
   _lifecyclecontrolProxy( LifecycleControlProxy::createProxy( "LifecycleControlPort", * this ) ){
   if ( !busName.empty( ) && !objName.empty( ) ){
      LOG_INFO( "LcmExtendedBaseAppComponent::LcmExtendedBaseAppComponent called\nbusname=%s\nobjname=%s", busName.c_str( ), objName.c_str( ) );
   } else {
      LOG_ERROR( "LcmExtendedBaseAppComponent::BusName or ObjName is empty " );
   }
}

LcmExtendedBaseAppComponent::~LcmExtendedBaseAppComponent( ){
   LOG_INFO( "LcmExtendedBaseAppComponent::~LcmExtendedBaseAppComponent called" );
}

// ServiceAvailableIF implementation
void LcmExtendedBaseAppComponent::onAvailable( const boost::shared_ptr < Proxy >& proxy,
                                               const ServiceStateChange         & stateChange ){
   if ( ( proxy == _lifecyclecontrolProxy ) && ( stateChange.hasCurrentState( ) ) ){
      LOG_DEBUG( "LcmExtendedBaseAppComponent::onAvailable::_lifecyclecontrolProxy service available: ServiceState: %s", ServiceState_Name( stateChange.getCurrentState( ) ) );

      if ( ServiceState__Available == stateChange.getCurrentState( ) ){
         //code
      } else {
         LOG_ERROR( "LcmExtendedBaseAppComponent::onAvailable:: Service is not available" );
      }
   } else {
      LOG_WARN( "LcmExtendedBaseAppComponent::onAvailable:: unknown service" );
   }
}

void LcmExtendedBaseAppComponent::onUnavailable( const boost::shared_ptr < Proxy >& proxy,
                                                 const ServiceStateChange         & stateChange ){
   if ( ( proxy == _lifecyclecontrolProxy ) && ( stateChange.hasCurrentState( ) ) ){
      LOG_DEBUG( "LcmExtendedBaseAppComponent::onUnavailable::__lifecyclecontrolProxy service not available anymore: ServiceState: %s", ServiceState_Name( stateChange.getCurrentState( ) ) );
   } else {
      LOG_WARN( "LcmExtendedBaseAppComponent::onUnavailable::onUnavailable unknown service" );
   }
}

// TimerCallbackIF implementation
void LcmExtendedBaseAppComponent::onExpired( asf::core::Timer                            & timer,
                                             boost::shared_ptr < asf::core::TimerPayload > payload ){
   LOG_INFO( "LcmExtendedBaseAppComponent::onExpired::Timer Act:%u", timer.getAct( ) );

   if ( payload->hasReason( ) && payload->hasRepeatCount( ) ){
      LOG_DEBUG( "LcmExtendedBaseAppComponent::onExpired: TimerPayload Reason: %s, Repeat Count:%u", TimerPayload_Reason_Name( payload->getReason( ) ), payload->getRepeatCount( ) );
   }
}

void LcmExtendedBaseAppComponent::onSetAppHealthStatusError( const ::boost::shared_ptr < LifecycleControlProxy >  & proxy,
                                                             const ::boost::shared_ptr < SetAppHealthStatusError >& error ){
   if ( proxy == _lifecyclecontrolProxy ){
      LOG_DEBUG( "LcmExtendedBaseAppComponent::onSetAppHealthStatusError::_lifecyclecontrolProxy service available" );

      if ( error->hasName( ) && error->hasMessage( ) ){
         LOG_ERROR( "LcmExtendedBaseAppComponent::onSetAppHealthStatusError::ErrorName:%s,Errormessage:%s", error->getName( ).c_str( ), error->getMessage( ).c_str( ) );
         // _ExtApplicationCb->OnExtApponSetAppHealthStatusError(_NsmErrorStatus_e__NsmErrorStatus_Dbus, error->getName(), error->getName());
      } else {
         LOG_INFO( "LcmExtendedBaseAppComponent::onSetAppHealthStatusError::No Error in D-Bus communication" );
      }
   } else {
      LOG_WARN( "LcmExtendedBaseAppComponent::onSetAppHealthStatusError unknown service" );
   }

}              // onSetAppHealthStatusError

void LcmExtendedBaseAppComponent::onSetAppHealthStatusResponse( const ::boost::shared_ptr < LifecycleControlProxy >     & proxy,
                                                                const ::boost::shared_ptr < SetAppHealthStatusResponse >& response ){
   if ( proxy == _lifecyclecontrolProxy ){
      LOG_DEBUG( "LcmExtendedBaseAppComponent::onSetAppHealthStatusResponse::_lifecyclecontrolProxy service available" );

      if ( response->hasErrorCode( ) ){
         LOG_INFO( "LcmExtendedBaseAppComponent::onSetAppHealthStatusResponse: NsmErrorStatus: %s", _NsmErrorStatus_e_Name( response->getErrorCode( ) ) );

         if ( ( _NsmErrorStatus_e__NsmErrorStatus_Ok == response->getErrorCode( ) ) || ( _NsmErrorStatus_e__NsmErrorStatus_ResponsePending == response->getErrorCode( ) ) ){
            //  _ExtApplicationCb->OnExtAppSetAppHealthStatusResponse(response->getErrorCode());
         } else {
            //  _ExtApplicationCb->OnExtAppSetAppHealthStatusResponse(response->getErrorCode());
         }
      } else {
         LOG_INFO( "LcmExtendedBaseAppComponent::onSetAppHealthStatusResponse::Response has no error code " );
      }

   } else {
      LOG_WARN( "LcmExtendedBaseAppComponent::onSetAppHealthStatusResponse unknown service" );
   }
}              // onSetAppHealthStatusResponse

// SetNodeStateCallbackIF
void LcmExtendedBaseAppComponent::onSetApplicationModeError( const ::boost::shared_ptr < LifecycleControlProxy >  & proxy,
                                                             const ::boost::shared_ptr < SetApplicationModeError >& error ){
   if ( proxy == _lifecyclecontrolProxy ){
      LOG_DEBUG( "LcmExtendedBaseAppComponent::onSetApplicationModeError::_lifecyclecontrolProxy service available" );

      if ( error->hasName( ) && error->hasMessage( ) ){
         LOG_ERROR( "LcmExtendedBaseAppComponent::onSetApplicationModeError::ErrorName:%s,Errormessage:%s", error->getName( ).c_str( ), error->getMessage( ).c_str( ) );
         //_ExtApplicationCb->OnExtAppSetApplicationModeError(_NsmErrorStatus_e__NsmErrorStatus_Dbus, error->getName(), error->getName());
      } else {
         LOG_INFO( "LcmExtendedBaseAppComponent::onSetApplicationModeError::No Error in D-Bus communication" );
      }
   } else {
      LOG_WARN( "LcmExtendedBaseAppComponent::onSetApplicationModeError unknown service" );
   }
}              // onSetApplicationModeError

void LcmExtendedBaseAppComponent::onSetApplicationModeResponse( const ::boost::shared_ptr < LifecycleControlProxy >     & proxy,
                                                                const ::boost::shared_ptr < SetApplicationModeResponse >& response ){
   if ( proxy == _lifecyclecontrolProxy ){
      LOG_DEBUG( "LcmExtendedBaseAppComponent::onSetApplicationModeResponse::_lifecyclecontrolProxy service available" );

      if ( response->hasErrorCode( ) ){
         LOG_INFO( "LcmExtendedBaseAppComponent::onSetApplicationModeResponse: NsmErrorStatus: %s", _NsmErrorStatus_e_Name( response->getErrorCode( ) ) );

         if ( ( _NsmErrorStatus_e__NsmErrorStatus_Ok == response->getErrorCode( ) ) || ( _NsmErrorStatus_e__NsmErrorStatus_ResponsePending == response->getErrorCode( ) ) ){
            //  _ExtApplicationCb->OnExtAppSetAppHealthStatusResponse(response->getErrorCode());
         } else {
            //  _ExtApplicationCb->OnExtAppSetAppHealthStatusResponse(response->getErrorCode());
         }
      } else {
         LOG_INFO( "LcmExtendedBaseAppComponent::onSetApplicationModeResponse::Response has no error code " );
      }
   } else {
      LOG_WARN( "LcmExtendedBaseAppComponent::onSetApplicationModeResponse unknown service" );
   }
}              // onSetApplicationModeResponse

// SetApplicationModeCallbackIF
void LcmExtendedBaseAppComponent::onSetBootModeError( const ::boost::shared_ptr < LifecycleControlProxy >& proxy,
                                                      const ::boost::shared_ptr < SetBootModeError >     & error ){
   if ( proxy == _lifecyclecontrolProxy ){
      LOG_DEBUG( "LcmExtendedBaseAppComponent::onSetBootModeError::_lifecyclecontrolProxy service available" );

      if ( error->hasName( ) && error->hasMessage( ) ){
         LOG_ERROR( "LcmExtendedBaseAppComponent::onSetBootModeError::ErrorName:%s,Errormessage:%s", error->getName( ).c_str( ), error->getMessage( ).c_str( ) );
         // _ExtApplicationCb->OnExtAppSetBootModeError(_NsmErrorStatus_e__NsmErrorStatus_Dbus, error->getName(), error->getName());
      } else {
         LOG_INFO( "LcmExtendedBaseAppComponent::onSetBootModeError::No Error in D-Bus communication" );
      }
   } else {
      LOG_WARN( "LcmExtendedBaseAppComponent::onSetBootModeError unknown service" );
   }
}

void LcmExtendedBaseAppComponent::onSetBootModeResponse( const ::boost::shared_ptr < LifecycleControlProxy >& proxy,
                                                         const ::boost::shared_ptr < SetBootModeResponse >  & response ){
   if ( proxy == _lifecyclecontrolProxy ){
      LOG_DEBUG( "LcmExtendedBaseAppComponent::onSetBootModeResponse::_lifecyclecontrolProxy service available" );

      if ( response->hasErrorCode( ) ){
         LOG_INFO( "LcmExtendedBaseAppComponent::onSetBootModeResponse: NsmErrorStatus: %s", _NsmErrorStatus_e_Name( response->getErrorCode( ) ) );

         if ( ( _NsmErrorStatus_e__NsmErrorStatus_Ok == response->getErrorCode( ) ) || ( _NsmErrorStatus_e__NsmErrorStatus_ResponsePending == response->getErrorCode( ) ) ){
            //  _ExtApplicationCb->OnExtAppSetAppHealthStatusResponse(response->getErrorCode());
         } else {
            //  _ExtApplicationCb->OnExtAppSetAppHealthStatusResponse(response->getErrorCode());
         }
      } else {
         LOG_INFO( "LcmExtendedBaseAppComponent::onSetBootModeResponse::Response has no error code " );
      }
   } else {
      LOG_WARN( "LcmExtendedBaseAppComponent::onSetBootModeResponse unknown service" );
   }
}              // onSetBootModeResponse

// SetAppHealthStatusCallbackIF
void LcmExtendedBaseAppComponent::onSetNodeStateError( const ::boost::shared_ptr < LifecycleControlProxy >& proxy,
                                                       const ::boost::shared_ptr < SetNodeStateError >    & error ){
   if ( proxy == _lifecyclecontrolProxy ){
      LOG_DEBUG( "LcmExtendedBaseAppComponent::onSetNodeStateError::_lifecyclecontrolProxy service available" );

      if ( error->hasName( ) && error->hasMessage( ) ){
         LOG_ERROR( "LcmExtendedBaseAppComponent::onSetNodeStateError::ErrorName:%s,Errormessage:%s", error->getName( ).c_str( ), error->getMessage( ).c_str( ) );
         // _ExtApplicationCb->OnExtAppSetNodeStateError(_NsmErrorStatus_e__NsmErrorStatus_Dbus, error->getName(), error->getName());
      } else {
         LOG_INFO( "LcmExtendedBaseAppComponent::onSetNodeStateError::No Error in D-Bus communication" );
      }
   } else {
      LOG_WARN( "LcmExtendedBaseAppComponent::onSetNodeStateError unknown service" );
   }
}

void LcmExtendedBaseAppComponent::onSetNodeStateResponse( const ::boost::shared_ptr < LifecycleControlProxy >& proxy,
                                                          const ::boost::shared_ptr < SetNodeStateResponse > & response ){
   if ( proxy == _lifecyclecontrolProxy ){
      LOG_DEBUG( "LcmExtendedBaseAppComponent::onSetNodeStateResponse::_lifecyclecontrolProxy service available" );

      if ( response->hasErrorCode( ) ){
         LOG_INFO( "LcmExtendedBaseAppComponent::onSetNodeStateResponse: NsmErrorStatus: %s", _NsmErrorStatus_e_Name( response->getErrorCode( ) ) );

         if ( ( _NsmErrorStatus_e__NsmErrorStatus_Ok == response->getErrorCode( ) ) || ( _NsmErrorStatus_e__NsmErrorStatus_ResponsePending == response->getErrorCode( ) ) ){
            //  _ExtApplicationCb->OnExtAppSetAppHealthStatusResponse(response->getErrorCode());
         } else {
            //  _ExtApplicationCb->OnExtAppSetAppHealthStatusResponse(response->getErrorCode());
         }
      } else {
         LOG_INFO( "LcmExtendedBaseAppComponent::onSetNodeStateResponse::Response has no error code " );
      }
   } else {
      LOG_WARN( "LcmExtendedBaseAppComponent::onSetNodeStateResponse unknown service" );
   }
}              // onSetNodeStateResponse

/*################################################
  ##################################################
   LcmExtendedApplicationControlIF
  ##################################################
  ################################################## */
void LcmExtendedApplicationControlIF::onExtAppSetAppHealthStatusError( _NsmErrorStatus_e  error,
                                                                       const std::string& name,
                                                                       const std::string& message ){
   LOG_WARN( "LcmExtendedApplicationControlIF::onExtAppSetAppHealthStatusError, function to be implemented in derived class ,\
            Error=%s, Name=%s, Message=%s", _NsmErrorStatus_e_Name( error ), name.c_str( ), message.c_str( ) );
}

void LcmExtendedApplicationControlIF::onExtAppSetAppHealthStatusResponse( _NsmErrorStatus_e response ){
   LOG_WARN( "LcmExtendedApplicationControlIF::onExtAppSetAppHealthStatusResponse ,function to be implemented in derived class , Response=%s", _NsmErrorStatus_e_Name( response ) );
}

void LcmExtendedApplicationControlIF::onExtAppSetApplicationModeError( _NsmErrorStatus_e  error,
                                                                       const std::string& name,
                                                                       const std::string& message ){
   LOG_WARN( "LcmExtendedApplicationControlIF::onExtAppSetApplicationModeError, function to be implemented in derived class ,\
            Error=%s, Name=%s, Message=%s", _NsmErrorStatus_e_Name( error ), name.c_str( ), message.c_str( ) );
}

void LcmExtendedApplicationControlIF::onExtAppSetApplicationModeResponse( _NsmErrorStatus_e response ){
   LOG_WARN( "LcmExtendedApplicationControlIF::onExtAppSetApplicationModeResponse ,function to be implemented in derived class , Response=%s", _NsmErrorStatus_e_Name( response ) );
}

void LcmExtendedApplicationControlIF::onExtSetBootModeError( _NsmErrorStatus_e  error,
                                                             const std::string& name,
                                                             const std::string& message ){
   LOG_WARN( "LcmExtendedApplicationControlIF::onExtSetBootModeError, function to be implemented in derived class ,\
            Error=%s, Name=%s, Message=%s", _NsmErrorStatus_e_Name( error ), name.c_str( ), message.c_str( ) );
}

void LcmExtendedApplicationControlIF::onExtSetBootModeResponse( _NsmErrorStatus_e response ){
   LOG_WARN( "LcmExtendedApplicationControlIF::onExtSetBootModeResponse ,function to be implemented in derived class , Response=%s", _NsmErrorStatus_e_Name( response ) );
}

void LcmExtendedApplicationControlIF::onExtSetNodeStateError( _NsmErrorStatus_e  error,
                                                              const std::string& name,
                                                              const std::string& message ){
   LOG_WARN( "LcmExtendedApplicationControlIF::onExtSetNodeStateError, function to be implemented in derived class ,\
            Error=%s, Name=%s, Message=%s", _NsmErrorStatus_e_Name( error ), name.c_str( ), message.c_str( ) );
}

void LcmExtendedApplicationControlIF::onExtSetNodeStateResponse( _NsmErrorStatus_e response ){
   LOG_WARN( "LcmExtendedApplicationControlIF::onExtSetNodeStateResponse ,function to be implemented in derived class , Response=%s", _NsmErrorStatus_e_Name( response ) );
}

LcmExtendedApplicationControlIF::~LcmExtendedApplicationControlIF( ){
   LOG_INFO( "LcmExtendedApplicationControlIF::~LcmExtendedApplicationControlIF called" );
}

DEFINE_CLASS_LOGGER_AND_LEVEL(             "org.bosch.cm.lcm.lcmbaseappcomponent", LcmBaseAppComponent,             Info );
DEFINE_CLASS_LOGGER_AND_LEVEL(     "org.bosch.cm.lcm.lcmextendedbaseappcomponent", LcmExtendedBaseAppComponent,     Info );
DEFINE_CLASS_LOGGER_AND_LEVEL(         "org.bosch.cm.lcm.lcmapplicationcontrolif", LcmApplicationControlIF,         Info );
DEFINE_CLASS_LOGGER_AND_LEVEL( "org.bosch.cm.lcm.lcmextendedapplicationcontrolif", LcmExtendedApplicationControlIF, Info );

}
}
}
}
}

