/*!
  * \file     GenericRecoveryClientComponent.cpp
  * \brief    Base class to handle failures of systemd units. This class will be the
  *           base of the GenericRecoveryClient app that is started via
  *           tag "OnFailure="
  *
  * \author   rmi2hi
  *
  * \par Copyright:
  * (C) 2013 - 2016 Robert Bosch GmbH.
  *   The reproduction, distribution and utilization of this file as well as the
  *   communication of its contents to others without express authorization is prohibited.
  *   Offenders will be held liable for the payment of damages.
  *   All rights reserved in the event of the grant of a patent, utility model or design.
  ***************************************************************************/

#include "GenericRecoveryClientComponent.h"
#include "asf/core/Application.h"
#include <systemd/sd-daemon.h> // you also need to add -lnotify to the linker settings
#include <cstdlib>
#include <unistd.h>
#include <sys/types.h>
#include <dirent.h>
#include <locale>
#include <signal.h>
#include <errno.h>
#include <fstream>
#include <sstream>
// \todo: boost::shared_ptr is used here cause by ASF interface.
namespace org {
namespace bosch {
namespace cm  {
namespace lcm {
namespace genericrecoveryclient {

using namespace ::asf::core;
using namespace ::org::bosch::cm::lcm;
using namespace ::org::bosch::cm::lcm::genericrecoveryclient::Generic_RecoveryClient;
using namespace ::org::freedesktop::systemd1::Unit;
using namespace ::org::freedesktop::systemd1::Service;
using namespace ::org::freedesktop::systemd1::Manager;


#define OSAL_PROC_DIR               "/run/OSAL/Processes"
#define NON_OSAL_PROC_CALLSTACK_SIG SIGUSR2
#define UNKNOWN_PROC_ID             0xFFFFFFFF

/*! \fn
  *  \brief function to escape the unit names
  *    specified in
  *    copied from systemd sources at: systemd-217\src\shared\unit-name.c
  *  \param UnitName original name of the unit
  *  \exception none.
  *  \return new escaped unitname
  */
std::string GenericRecoveryClientComponent::unitNameEscape( const std::string UnitName ){
   std::locale loc;
   std::string EscapedUnitName;

   for ( std::string::const_iterator it = UnitName.begin( ); it != UnitName.end( ); ++it ){
      if ( std::isalnum( * it, loc ) ){
         EscapedUnitName += * it;
      } else {
         int               character = * it;
         std::stringstream strstream;
         strstream << "_" << std::hex << character;
         EscapedUnitName += strstream.str( );
      }
   }
   return( EscapedUnitName );
}              // unitNameEscape

/*! \fn
  *  \brief function to find whether
  *    the main process of the service is an OSAL process or not. It uses the MainId to try to find
  *    the process in Osal-ProcFileSystem
  *  \param none.
  *  \exception none.
  *  \return bool true or false.
  */
bool GenericRecoveryClientComponent::bIsOsalProcess( ){
   bool           bIsOsalProc   = false;

   #if 0          // \todo reactivate if C++11 features are allowed to use
      std::string OsalPath      = std::string( OSAL_PROC_DIR ) + "/" + to_string( _ExecMainPid );
   #else
      char        processId[16] = { 0 };
      snprintf( processId, 16, "%d", _ExecMainPid );
      std::string OsalPath      = std::string( OSAL_PROC_DIR ) + "/" + std::string( processId );
   #endif

   /* To decide if the process is an OSAL process,
      check in directory "OSAL_PROC_DIR".
      because for each OSAL process, an entry is made here.*/
   std::ifstream pidFile( OsalPath.c_str( ) );
   if ( pidFile.is_open( ) ){
      bIsOsalProc = true;
      pidFile.close( );
   }
   return( bIsOsalProc );
}              // bIsOsalProcess

/*! \fn
  *  \brief function to send callstack signal
  *    for all tasks under the process
  *  \param pid.
  *  \exception none.
  *  \return none.
  */
void GenericRecoveryClientComponent::vSendCallstackSignalToAllTasksOfProc( const uint32 u32Pid ){
   struct dirent *dp;
   DIR           *dfd;

   #if 0          // reactivate if C++11 features are allowed to use
      std::string TaskDir       = "/proc/" + to_string( u32Pid ) + "/task";
   #else
      char        processId[16] = { 0 };
      snprintf( processId, 16, "%d", u32Pid );
      std::string TaskDir       = "/proc/" + std::string( processId ) + "/task";
   #endif

   if ( ( dfd = opendir( TaskDir.c_str( ) ) ) != NULL ){
      while ( ( dp = readdir( dfd ) ) != NULL ){
         /* skip self and parent directories */
         if ( ( strcmp( dp->d_name, "." ) != 0 ) && ( strcmp( dp->d_name, ".." ) != 0 ) ){
            LOG_ERROR( "kill(%d, %d)", u32Pid, _CallStackSignal );
            int ret = kill( u32Pid, _CallStackSignal );
            if ( ret == - 1 ){
               if ( errno == EPERM ){
                  LOG_ERROR( "The process does not have permission to send the signal to any receiving process." );
               } else if ( errno == EINVAL ){
                  LOG_ERROR( "The value of the sig argument is an invalid or unsupported signal number." );
               } else if ( errno == ESRCH ){
                  LOG_ERROR( "No process or process group can be found corresponding to that specified by pid." );
               } else if ( errno == 0 ){
                  LOG_ERROR( "Kill=OK" );
               } else {
                  LOG_ERROR( "??????? = %d", ret );
               }
            }
         }
      }
      (void)closedir( dfd );
   } else {
      LOG_WARN( " cannot open dir: %s", TaskDir.c_str( ) );
   }
}              // vSendCallstackSignalToAllTasksOfProc

/*! \fn
  *  \brief constructor
  *
  *   evaluate command line parameter if any options are set
  *   create all servers and proxies
  *  \exception none.
  *  \return none.
  */
GenericRecoveryClientComponent::GenericRecoveryClientComponent( ) :
   #ifndef LCM_UNIT_TESTS
      BaseComponent( ),
   #endif
   _SystemdNotifyTimeMSec( 0 ),
   _ServiceState( UnitStates__UNIT_FAILED ),
   _ServiceAction( UnitActions__UNIT_KILL ),
   _CallStackSignal( NON_OSAL_PROC_CALLSTACK_SIG ),
   _ExecMainPid( UNKNOWN_PROC_ID ),
   _CallStackSignalCounter( 0 ),
   _managerProxyAvailable( false ),
   _unitProxyAvailable( false ),
   _serviceProxyAvailable( false ),
   _lcmRecoveryClientProxyAvailable( false ),
   _genericRecoveryClientResetAllowed( true ),
   _ServiceName( "none" ),
   _CheckTimer( * this, 1000, 1 ),
   _lcmRecoveryClientProxy( Generic_RecoveryClientProxy::createProxy( "RecoveryClientPort", * this ) ),
   _managerProxy( ManagerProxy::createProxy( "DBusManagerClientPort", * this ) ){
   LOG_INFO( "GenericRecoveryClientComponent called" );

   const std::vector < std::string >           arguments = asf::core::Application::getApplication( )->getArguments( );
   std::vector < std::string >::const_iterator cit;
   LOG_INFO( "ArgumentListSize = %i", arguments.size( ) );
   int                                         count     = 0;
   for ( cit = arguments.begin( ); cit != arguments.end( ); cit++ ){
      count++;
      std::string argument = ( * cit );
      std::string key      = "ServiceName=";
      LOG_INFO( "Argument%i = %s", count, argument.c_str( ) );
      if ( argument.find( key ) != std::string::npos ){ // string contains "ServiceName="
         argument     = argument.erase( 0, key.length( ) );
         LOG_WARN( "service name = '%s'", argument.c_str( ) );
         _ServiceName = argument;
      }
   }

   if ( _ServiceName == "none" ){
      LOG_ERROR( "No service name defined" );
   }
   std::string escapedServiceName = unitNameEscape( _ServiceName );
   std::string objectPath         = "/org/freedesktop/systemd1/unit/" + escapedServiceName;

   tU64        u64WatchdoguSec    = 0;
   if ( sd_watchdog_enabled( 0, &u64WatchdoguSec ) <= 0 ){
      LOG_ERROR( "No Watchdog time defined. Watchdog is disabled" );
   } else {
      _SystemdNotifyTimeMSec = (uint32)( u64WatchdoguSec / 1000 );

      LOG_INFO( "Watchdog timeout = %i ms", _SystemdNotifyTimeMSec );
      _SystemdNotifyTimeMSec = _SystemdNotifyTimeMSec / 2;   // use half the time passed by systemd to retrigger the Watchdog
      _SystemdNotifyTimer.start( * this, _SystemdNotifyTimeMSec, 1 );
   }


   LOG_INFO( "Create UnitProxy for objectpath - %s", objectPath.c_str( ) );
   _unitProxy = UnitProxy::createProxy( "DBusUnitClientPort",       // portName
                                        "org.freedesktop.systemd1", // busName
                                        objectPath,                 // objectPath
                                        DBUS_BUS_SYSTEM,            // busType
                                        * this                      // ServiceAvailableIF
                                        );

   LOG_INFO( "Create ServiceProxy for objectpath - %s", objectPath.c_str( ) );
   _serviceProxy = ServiceProxy::createProxy( "DBusServiceClientPort",    // portName
                                              "org.freedesktop.systemd1", // busName
                                              objectPath,                 // objectPath
                                              DBUS_BUS_SYSTEM,            // busType
                                              * this                      // ServiceAvailableIF
                                              );
}

/*! \fn
  *  \brief destructor
  *
  *  \exception none.
  *  \return none.
  */
GenericRecoveryClientComponent::~GenericRecoveryClientComponent( ){
   LOG_INFO( "~GenericRecoveryClientComponent called" );
}

// ServiceAvailableIF implementation

/*! \fn
  *  \brief function is called if any server connected to in constructor is connected
  *
  *    register for signals and properties
  *    exchange initial messages with new servers
  *
  *  \param proxy - pointer to proxy member where the server is now available
  *  \param stateChange - indicated old and new state of the connection
  *
  *  \exception none.
  *  \return none.
  */
void GenericRecoveryClientComponent::onAvailable( const boost::shared_ptr < Proxy >& proxy,
                                                  const ServiceStateChange         & stateChange ){
   if ( stateChange.hasCurrentState( ) ){
      LOG_INFO( "GenericRecoveryClientComponent::onAvailable: ServiceState: %s", ServiceState_Name( stateChange.getCurrentState( ) ) );
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onAvailable: service unknown" );
   }
   if ( proxy && ( _managerProxy == proxy ) ){
      LOG_INFO( "  - ManagerProxy is connected" );
      LOG_INFO( "  - register attributes and events" );
      _asfManagerProxy._u32JobNewRegisterId     = (tU32)_managerProxy->sendJobNewRegister( * this );
      _asfManagerProxy._u32JobRemovedRegisterId = (tU32)_managerProxy->sendJobRemovedRegister( * this );
      _managerProxyAvailable                    = true;
   }
   if ( proxy && ( _unitProxy == proxy ) ){
      LOG_INFO( "  - UnitProxy is connected" );
      _CheckTimer.start( );
      _unitProxy->sendActiveStateRegister( * this );
      _unitProxy->sendIdRegister( * this );
      _unitProxy->sendIdGet( * this );
      _unitProxy->sendActiveStateGet( * this );
      _unitProxyAvailable = true;
   }
   if ( proxy && ( _serviceProxy == proxy ) ){
      LOG_INFO( "  - ServiceProxy is connected" );
      _serviceProxy->sendExecMainPIDGet( * this );
      _serviceProxyAvailable = true;
   }
   if ( proxy && ( _lcmRecoveryClientProxy == proxy ) ){
      LOG_INFO( "  - Generic_RecoveryClientProxy is connected" );
      _lcmRecoveryClientProxy->sendSetFailedServiceNameRequest( * this, _ServiceName, _ServiceAction );
      LOG_INFO( "GenericRecoveryClientComponent::onAvailable send request to check GenericRecoveryClient can send reset command or not" );
      _lcmRecoveryClientProxy->sendCheckResetConditionRequest( * this, _ServiceName );
      _lcmRecoveryClientProxyAvailable = true;
   }

   if ( ( _managerProxyAvailable == true )
        && ( _unitProxyAvailable == true )
        && ( _serviceProxyAvailable == true )
        && ( _lcmRecoveryClientProxyAvailable == true )
        ){
      sd_notifyf( 0, "READY=1\n"                       // Tells the init system that daemon startup is finished.
                                                       // This is only used by systemd if the service definition file has Type=notify set.
                     "STATUS=Processing requests...\n" // Passes a single-line status string back to the init system that describes the daemon state.
                                                       // This is free-form and can be used for various purposes
                     "MAINPID=%lu\nPARENTID=%lu",
                  (unsigned long)getpid( ),
                  (unsigned long)getppid( ) );
      LOG_INFO( "GenericRecoveryClientComponent::onAvailable: sd_notifyf - READY=1 sent)" );
   }
}              // onAvailable

/*! \fn
  *  \brief function is called when a server is not connected anymore
  *
  *  \param proxy - pointer to proxy member where the server is now available
  *  \param stateChange - indicated old and new state of the connection
  *
  *  \exception none.
  *  \return none.
  */
void GenericRecoveryClientComponent::onUnavailable( const boost::shared_ptr < Proxy >& proxy,
                                                    const ServiceStateChange         & stateChange ){
   if ( stateChange.hasCurrentState( ) ){
      LOG_INFO( "GenericRecoveryClientComponent::onUnavailable: ServiceState: %s", ServiceState_Name( stateChange.getCurrentState( ) ) );
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onUnavailable: service unknown" );
   }

   if ( proxy && ( _managerProxy == proxy ) ){
      LOG_INFO( "  - ManagerProxy is disconnected" );
      _managerProxy->sendJobNewDeregister( _asfManagerProxy._u32JobNewRegisterId );
      _managerProxy->sendJobRemovedDeregister( _asfManagerProxy._u32JobRemovedRegisterId );
      _managerProxyAvailable = false;
   }
   if ( proxy && ( _unitProxy == proxy ) ){
      LOG_INFO( "  - UnitProxy is disconnected" );
      _unitProxy->sendActiveStateDeregisterAll( );
      _unitProxy->sendIdDeregisterAll( );
      _unitProxyAvailable = false;
   }
   if ( proxy && ( _serviceProxy == proxy ) ){
      LOG_INFO( "  - ServiceProxy is disconnected" );
      _serviceProxy->sendExecMainPIDDeregisterAll( );
      _serviceProxyAvailable = false;
   }
   if ( proxy && ( _lcmRecoveryClientProxy == proxy ) ){
      LOG_INFO( "  - Generic_RecoveryClientProxy is disconnected" );
      _lcmRecoveryClientProxyAvailable = false;
   }
}              // onUnavailable

// *************
// UserCallbacks
// *************

/*! \fn
  *  \brief Indicates that the Active state has been changed / requested state has been received
  *
  *  \param proxy - pointer to proxy member where the server is now available
  *  \param update - contains the current state
  *
  *  \exception none.
  *  \return none.
  */
void GenericRecoveryClientComponent::_onActiveStateUpdate( const ::boost::shared_ptr < UnitProxy >        & proxy,
                                                           const ::boost::shared_ptr < ActiveStateUpdate >& update,
                                                           uint8                                            CallStackSignalCounter ){
   (void)proxy;
   (void)update;
   (void)CallStackSignalCounter;
}

/*! \fn
  *  \brief Indicates that the Active state has some error
  *
  *  \param proxy - pointer to proxy member where the server is now available
  *  \param update - contains the current state
  *
  *  \exception none.
  *  \return none.
  */
void GenericRecoveryClientComponent::_onActiveStateError( const ::boost::shared_ptr < UnitProxy >       & proxy,
                                                          const ::boost::shared_ptr < ActiveStateError >& error,
                                                          uint8                                           CallStackSignalCounter ){
   (void)proxy;
   (void)error;
   (void)CallStackSignalCounter;
}

// *************
// ManagerProxy
// *************
// JobNewCallbackIF implementation
void GenericRecoveryClientComponent::onJobNewSignal( const ::boost::shared_ptr < ManagerProxy >& proxy,
                                                     const ::boost::shared_ptr < JobNewSignal >& pSignal ){
   if ( ( _managerProxy == proxy ) && pSignal->hasJob( ) ){
      LOG_INFO( "GenericRecoveryClientComponent::Received onJobNewSignal: Job=%s", pSignal->getJob( ).c_str( ) );
   } else {
      LOG_ERROR( " GenericRecoveryClientComponent::onJobNewSignal: proxy not available" );
   }
}

void GenericRecoveryClientComponent::onJobNewError( const ::boost::shared_ptr < ManagerProxy >& proxy,
                                                    const ::boost::shared_ptr < JobNewError > & error ){
   if ( ( _managerProxy == proxy ) && error->hasName( ) && error->hasMessage( ) ){
      LOG_ERROR( "GenericRecoveryClientComponent::Received onJobNewError: ErrorName=%s, ErrorMessage=%s", error->getName( ).c_str( ), error->getMessage( ).c_str( ) );
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onJobNewError: proxy not available" );
   }
}

// JobRemovedCallbackIF implementation
void GenericRecoveryClientComponent::onJobRemovedSignal( const ::boost::shared_ptr < ManagerProxy >    & proxy,
                                                         const ::boost::shared_ptr < JobRemovedSignal >& pSignal ){
   if ( ( _managerProxy == proxy ) && pSignal->hasJob( ) ){
      LOG_INFO( "GenericRecoveryClientComponent::Received onJobRemovedSignal: Job=%s", pSignal->getJob( ).c_str( ) );
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onJobRemovedSignal: proxy not available" );
   }
}

void GenericRecoveryClientComponent::onJobRemovedError( const ::boost::shared_ptr < ManagerProxy >   & proxy,
                                                        const ::boost::shared_ptr < JobRemovedError >& error ){
   if ( ( _managerProxy == proxy ) && error->hasName( ) && error->hasMessage( ) ){
      LOG_ERROR( "GenericRecoveryClientComponent::Received onJobRemovedError: ErrorName=%s, ErrorMessage=%s", error->getName( ).c_str( ), error->getMessage( ).c_str( ) );
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onJobRemovedError: proxy not available" );
   }
}

// GetUnitCallbackIF implementation
void GenericRecoveryClientComponent::onGetUnitResponse( const ::boost::shared_ptr < ManagerProxy >   & proxy,
                                                        const ::boost::shared_ptr < GetUnitResponse >& response ){
   if ( ( _managerProxy == proxy ) && response->hasUnit( ) ){
      LOG_INFO( "GenericRecoveryClientComponent::Received onGetUnitResponse: Unit=%s", response->getUnit( ).c_str( ) );
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onGetUnitResponse: proxy not available" );
   }
}

void GenericRecoveryClientComponent::onGetUnitError( const ::boost::shared_ptr < ManagerProxy >& proxy,
                                                     const ::boost::shared_ptr < GetUnitError >& error ){
   if ( ( _managerProxy == proxy ) && error->hasName( ) && error->hasMessage( ) ){
      LOG_ERROR( "GenericRecoveryClientComponent::Received onGetUnitError: ErrorName=%s, ErrorMessage=%s", error->getName( ).c_str( ), error->getMessage( ).c_str( ) );
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onGetUnitError: proxy not available" );
   }
}

// StartUnitCallbackIF implementation
void GenericRecoveryClientComponent::onStartUnitResponse( const ::boost::shared_ptr < ManagerProxy >     & proxy,
                                                          const ::boost::shared_ptr < StartUnitResponse >& response ){
   if ( ( _managerProxy == proxy ) && response->hasJob( ) ){
      LOG_INFO( "GenericRecoveryClientComponent::Received onStartUnitResponse: Job=%s", response->getJob( ).c_str( ) );
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onStartUnitResponse: proxy not available" );
   }
}

void GenericRecoveryClientComponent::onStartUnitError( const ::boost::shared_ptr < ManagerProxy >  & proxy,
                                                       const ::boost::shared_ptr < StartUnitError >& error ){
   if ( ( _managerProxy == proxy ) && error->hasName( ) && error->hasMessage( ) ){
      LOG_ERROR( "GenericRecoveryClientComponent::Received onStartUnitError: ErrorName=%s, ErrorMessage=%s", error->getName( ).c_str( ), error->getMessage( ).c_str( ) );
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onStartUnitError: proxy not available" );
   }
}

// StopUnitCallbackIF implementation
void GenericRecoveryClientComponent::onStopUnitResponse( const ::boost::shared_ptr < ManagerProxy >    & proxy,
                                                         const ::boost::shared_ptr < StopUnitResponse >& response ){
   if ( ( _managerProxy == proxy ) && response->hasJob( ) ){
      LOG_INFO( "GenericRecoveryClientComponent::Received onStopUnitResponse: Job=%s", response->getJob( ).c_str( ) );
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onStopUnitResponse: proxy not available" );
   }
}

void GenericRecoveryClientComponent::onStopUnitError( const ::boost::shared_ptr < ManagerProxy > & proxy,
                                                      const ::boost::shared_ptr < StopUnitError >& error ){
   if ( ( _managerProxy == proxy ) && error->hasName( ) && error->hasMessage( ) ){
      LOG_ERROR( "GenericRecoveryClientComponent::Received onStopUnitError: ErrorName=%s, ErrorMessage=%s", error->getName( ).c_str( ), error->getMessage( ).c_str( ) );
   } else {
      LOG_ERROR( " GenericRecoveryClientComponent::onStopUnitError: proxy not available" );
   }
}

// ListJobsCallbackIF implementation
void GenericRecoveryClientComponent::onListJobsResponse( const ::boost::shared_ptr < ManagerProxy >    & proxy,
                                                         const ::boost::shared_ptr < ListJobsResponse >& response ){
   if ( ( _managerProxy == proxy ) && response->hasJobs( ) ){
      LOG_INFO( "Received onListJobsResponse" );
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onListJobsResponse: proxy not available" );
   }
}

void GenericRecoveryClientComponent::onListJobsError( const ::boost::shared_ptr < ManagerProxy > & proxy,
                                                      const ::boost::shared_ptr < ListJobsError >& error ){
   if ( ( _managerProxy == proxy ) && error->hasName( ) && error->hasMessage( ) ){
      LOG_ERROR( "GenericRecoveryClientComponent::Received onListJobsError: ErrorName=%s, ErrorMessage=%s", error->getName( ).c_str( ), error->getMessage( ).c_str( ) );
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onListJobsError: proxy not available" );
   }
}

// **********
// UnitProxy
// **********
// ActiveStateCallbackIF implementation
void GenericRecoveryClientComponent::onActiveStateUpdate( const ::boost::shared_ptr < UnitProxy >        & proxy,
                                                          const ::boost::shared_ptr < ActiveStateUpdate >& update ){
   std::string strServiceState     = update->getActiveState( );

   LOG_INFO( "Received onActiveStateUpdate: %s", strServiceState.c_str( ) );

   std::string strServiceStateName = "unit_"; // unit state enum names are started with "UNIT_<ActiveState>" ex: UNIT_ACTIVE, UNIT_FAILED etc.
   strServiceStateName += strServiceState;

   if ( UnitStates_Parse( strServiceStateName.c_str( ), strServiceStateName.length( ), _ServiceState ) ){
      LOG_DEBUG( "GenericRecoveryClientComponent::onActiveStateUpdate UnitStates_Parse Success: Unit State %s/%d", strServiceStateName.c_str( ), _ServiceState );
   } else {
      LOG_WARN( "GenericRecoveryClientComponent::onActiveStateUpdate UnitStates_Parse Failed: Unit State %s/%d", strServiceStateName.c_str( ), _ServiceState );
   }

   if ( _CallStackSignalCounter > 5 * SPM_MAX_NUMBER_OF_RECOVERCLIENT_ACTIONS ){
      // If Lcm inform that GenericRecoveryClient can send reset command
      if ( _genericRecoveryClientResetAllowed ){
         LOG_INFO( "GenericRecoveryClientComponent::onActiveStateUpdate resetting" );
         FATAL_M_ASSERT_ALWAYS( );
      }
   } else {
      // user function
      _onActiveStateUpdate( proxy, update, _CallStackSignalCounter );

      if ( _lcmRecoveryClientProxyAvailable == true ){
         _lcmRecoveryClientProxy->sendSetFailedServiceStatusRequest( * this, _ServiceName, _ServiceState );
         _lcmRecoveryClientProxy->sendSetCallStackRequestCounterRequest( * this, _ServiceName, _CallStackSignalCounter );
      }
      _CallStackSignalCounter++;

      //make sure _ExecMainPid is updated
      if ( _serviceProxyAvailable && _serviceProxy->hasExecMainPID( ) ){
         _ExecMainPid = _serviceProxy->getExecMainPID( );
      }

      if ( _ExecMainPid != UNKNOWN_PROC_ID ){
         if ( bIsOsalProcess( ) ){
            _CallStackSignal = u32GetSigRtMinId( (tS32)_ExecMainPid );
            if ( _CallStackSignal != - 1 ){
               LOG_ERROR( "kill(%d, %d)", _ExecMainPid, (uint32)_CallStackSignal );
               int ret = kill( _ExecMainPid, _CallStackSignal );
               if ( ret == - 1 ){
                  if ( errno == EPERM ){
                     LOG_ERROR( "The process does not have permission to send the signal to any receiving process." );
                  } else if ( errno == EINVAL ){
                     LOG_ERROR( "The value of the sig argument is an invalid or unsupported signal number." );
                  } else if ( errno == ESRCH ){
                     LOG_ERROR( "No process or process group can be found corresponding to that specified by pid." );
                  } else if ( errno == 0 ){
                     LOG_ERROR( "Kill=OK" );
                  } else {
                     LOG_ERROR( "??????? = %d", ret );
                  }
               }
            } else {
               LOG_ERROR( "u32GetSigRtMinId could not determine Signal ID" );
            }
         } else {
            _CallStackSignal = NON_OSAL_PROC_CALLSTACK_SIG;
            vSendCallstackSignalToAllTasksOfProc( _ExecMainPid );
         }
      } else {
         if ( _ExecMainPid != UNKNOWN_PROC_ID ){
            LOG_ERROR( "onActiveStateUpdate: _ExecMainPid (%d) is not updated for service: %s", _ExecMainPid, _ServiceName.c_str( ) );
         }
      }
   }


}              // onActiveStateUpdate

void GenericRecoveryClientComponent::onActiveStateError( const ::boost::shared_ptr < UnitProxy >       & proxy,
                                                         const ::boost::shared_ptr < ActiveStateError >& error ){
   if ( error->hasName( ) && error->hasMessage( ) ){
      LOG_ERROR( "Received onActiveStateError: ErrorName=%s, ErrorMessage=%s", error->getName( ).c_str( ), error->getMessage( ).c_str( ) );
   } else {
      LOG_ERROR( "Received onActiveStateError" );
   }

   // Check _CallStackSignalCounter is more than 15 or not.
   if ( _CallStackSignalCounter > 5 * SPM_MAX_NUMBER_OF_RECOVERCLIENT_ACTIONS ){
      // If Lcm inform that GenericRecoveryClient can send reset command
      if ( _genericRecoveryClientResetAllowed ){
         LOG_INFO( "GenericRecoveryClientComponent::onActiveStateError resetting" );
         FATAL_M_ASSERT_ALWAYS( );
      }
   } else {
      // IF not, send Failed Service to Lcm
      if ( _lcmRecoveryClientProxyAvailable == true ){
         _lcmRecoveryClientProxy->sendSetFailedServiceStatusRequest( * this, _ServiceName, _ServiceState );
         _lcmRecoveryClientProxy->sendSetCallStackRequestCounterRequest( * this, _ServiceName, _CallStackSignalCounter );
      }
      _CallStackSignalCounter++;

      //make sure _ExecMainPid is updated
      if ( _serviceProxyAvailable && _serviceProxy->hasExecMainPID( ) ){
         _ExecMainPid = _serviceProxy->getExecMainPID( );
      }


      if ( _ExecMainPid != UNKNOWN_PROC_ID ){
         if ( bIsOsalProcess( ) ){
            _CallStackSignal = u32GetSigRtMinId( (tS32)_ExecMainPid );
            if ( _CallStackSignal != - 1 ){
               LOG_ERROR( "kill(%d, %d)", _ExecMainPid, (uint32)_CallStackSignal );
               int ret = kill( _ExecMainPid, _CallStackSignal );
               if ( ret == - 1 ){
                  if ( errno == EPERM ){
                     LOG_ERROR( "The process does not have permission to send the signal to any receiving process." );
                  } else if ( errno == EINVAL ){
                     LOG_ERROR( "The value of the sig argument is an invalid or unsupported signal number." );
                  } else if ( errno == ESRCH ){
                     LOG_ERROR( "No process or process group can be found corresponding to that specified by pid." );
                  } else if ( errno == 0 ){
                     LOG_ERROR( "Kill=OK" );
                  } else {
                     LOG_ERROR( "??????? = %d", ret );
                  }
               }
            } else {
               LOG_ERROR( "u32GetSigRtMinId could not determine Signal ID" );
            }
         } else {
            _CallStackSignal = NON_OSAL_PROC_CALLSTACK_SIG;
            vSendCallstackSignalToAllTasksOfProc( _ExecMainPid );

         }
      } else {
         LOG_ERROR( "onActiveStateUpdate: _ExecMainPid (%d) is not updated for service: %s", _ExecMainPid, _ServiceName.c_str( ) );
      }

      // user function
      _onActiveStateError( proxy, error, _CallStackSignalCounter );
   }

}              // onActiveStateError

// IdCallbackIF implementation
void GenericRecoveryClientComponent::onIdUpdate( const ::boost::shared_ptr < UnitProxy >& proxy,
                                                 const ::boost::shared_ptr < IdUpdate > & update ){
   if ( ( _unitProxy == proxy ) && update->hasId( ) ){
      LOG_INFO( "GenericRecoveryClientComponent::Received onIdUpdate: Id=%s", update->getId( ).c_str( ) );
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onIdUpdate: proxy not available" );
   }
}

void GenericRecoveryClientComponent::onIdError( const ::boost::shared_ptr < UnitProxy >& proxy,
                                                const ::boost::shared_ptr < IdError >  & error ){
   if ( ( _unitProxy == proxy ) && error->hasName( ) && error->hasMessage( ) ){
      LOG_ERROR( "GenericRecoveryClientComponent::Received onIdError: ErrorName=%s, ErrorMessage=%s", error->getName( ).c_str( ), error->getMessage( ).c_str( ) );
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onIdError: proxy not available" );
   }
}

// **********
// ServiceProxy
// **********
// ExecMainPIDCallbackIF implementation
void GenericRecoveryClientComponent::onExecMainPIDUpdate( const ::boost::shared_ptr < ServiceProxy >     & proxy,
                                                          const ::boost::shared_ptr < ExecMainPIDUpdate >& update ){
   if ( ( _serviceProxy == proxy ) && update->hasExecMainPID( ) ){
      _ExecMainPid = update->getExecMainPID( );
      if ( _ExecMainPid != UNKNOWN_PROC_ID ){
         LOG_INFO( "GenericRecoveryClientComponent::Received onExecMainPIDUpdate: _ExecMainPid: %d", _ExecMainPid );
      } else {
         LOG_ERROR( "GenericRecoveryClientComponent::onExecMainPIDUpdate: _ExecMainPid (%d) is not updated for service: %s", _ExecMainPid, _ServiceName.c_str( ) );
      }
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onExecMainPIDUpdate: proxy not available" );
   }
}

void GenericRecoveryClientComponent::onExecMainPIDError( const ::boost::shared_ptr < ServiceProxy >    & proxy,
                                                         const ::boost::shared_ptr < ExecMainPIDError >& error ){
   if ( ( _serviceProxy == proxy ) && error->hasName( ) && error->hasMessage( ) ){
      LOG_ERROR( "Received onExecMainPIDError: ErrorName=%s, ErrorMessage=%s", error->getName( ).c_str( ), error->getMessage( ).c_str( ) );
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onExecMainPIDError: proxy not available" );
   }
}

// *************
// Generic_RecoveryClientProxy
// *************
// FailedServiceNameCallbackIF implementation
void GenericRecoveryClientComponent::onFailedServiceNameUpdate( const ::boost::shared_ptr < Generic_RecoveryClientProxy >& proxy,
                                                                const ::boost::shared_ptr < FailedServiceNameUpdate >    & update ){
   if ( _lcmRecoveryClientProxy == proxy ){
      LOG_INFO( "GenericRecoveryClientComponent::Received onFailedServiceNameUpdate: ServiceName=%s", update->getFailedServiceName( ).c_str( ) );
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onFailedServiceNameUpdate: proxy not available" );
   }
}

void GenericRecoveryClientComponent::onFailedServiceNameError( const ::boost::shared_ptr < Generic_RecoveryClientProxy >& proxy,
                                                               const ::boost::shared_ptr < FailedServiceNameError >     & error ){
   if ( ( _lcmRecoveryClientProxy == proxy ) && error->hasName( ) && error->hasMessage( ) ){
      LOG_ERROR( "GenericRecoveryClientComponent::Received onFailedServiceNameError: ErrorName=%s, ErrorMessage=%s", error->getName( ).c_str( ), error->getMessage( ).c_str( ) );
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onFailedServiceNameError: proxy not available" );
   }
}

// FailedServiceStatusCallbackIF implementation
void GenericRecoveryClientComponent::onFailedServiceStatusUpdate( const ::boost::shared_ptr < Generic_RecoveryClientProxy >& proxy,
                                                                  const ::boost::shared_ptr < FailedServiceStatusUpdate >  & update ){
   if ( _lcmRecoveryClientProxy == proxy ){
      LOG_INFO( "GenericRecoveryClientComponent::Received onFailedServiceStatusUpdate: ServiceStatus= %s", UnitStates_Name( update->getFailedServiceStatus( ) ) );
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onFailedServiceStatusUpdate: proxy not available" );
   }
}

void GenericRecoveryClientComponent::onFailedServiceStatusError( const ::boost::shared_ptr < Generic_RecoveryClientProxy >& proxy,
                                                                 const ::boost::shared_ptr < FailedServiceStatusError >   & error ){
   if ( ( _lcmRecoveryClientProxy == proxy ) && error->hasName( ) && error->hasMessage( ) ){
      LOG_INFO( "GenericRecoveryClientComponent::Received onFailedServiceStatusError: ErrorName=%s, ErrorMessage=%s", error->getName( ).c_str( ), error->getMessage( ).c_str( ) );
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onFailedServiceStatusError: proxy not available" );
   }
}

// CallStackRequestCounterCallbackIF implementation
void GenericRecoveryClientComponent::onCallStackRequestCounterUpdate( const ::boost::shared_ptr < Generic_RecoveryClientProxy >  & proxy,
                                                                      const ::boost::shared_ptr < CallStackRequestCounterUpdate >& update ){
   if ( _lcmRecoveryClientProxy == proxy ){
      LOG_INFO( "GenericRecoveryClientComponent::Received onCallStackRequestCounterUpdate: CallStackRequestCounter=%i", update->getCallStackRequestCounter( ) );
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onCallStackRequestCounterUpdate: proxy not available" );
   }
}

void GenericRecoveryClientComponent::onCallStackRequestCounterError( const ::boost::shared_ptr < Generic_RecoveryClientProxy > & proxy,
                                                                     const ::boost::shared_ptr < CallStackRequestCounterError >& error ){
   if ( ( _lcmRecoveryClientProxy == proxy ) && error->hasName( ) && error->hasMessage( ) ){
      LOG_ERROR( "GenericRecoveryClientComponent::Received onCallStackRequestCounterError: ErrorName=%s, ErrorMessage=%s", error->getName( ).c_str( ), error->getMessage( ).c_str( ) );
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onCallStackRequestCounterError: proxy not available" );
   }
}

// ActionRequestCallbackIF implementation
void GenericRecoveryClientComponent::onActionRequestUpdate( const ::boost::shared_ptr < Generic_RecoveryClientProxy >& proxy,
                                                            const ::boost::shared_ptr < ActionRequestUpdate >        & update ){
   if ( _lcmRecoveryClientProxy == proxy ){
      LOG_INFO( "GenericRecoveryClientComponent::Received onActionRequestUpdate: ActionRequest= %s", UnitActions_Name( update->getActionRequest( ) ) );
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onActionRequestUpdate: proxy not available" );
   }
}

void GenericRecoveryClientComponent::onActionRequestError( const ::boost::shared_ptr < Generic_RecoveryClientProxy >& proxy,
                                                           const ::boost::shared_ptr < ActionRequestError >         & error ){
   if ( ( _lcmRecoveryClientProxy == proxy ) && error->hasName( ) && error->hasMessage( ) ){
      LOG_ERROR( "GenericRecoveryClientComponent::Received onActionRequestError: ErrorName=%s, ErrorMessage=%s", error->getName( ).c_str( ), error->getMessage( ).c_str( ) );
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onActionRequestError: proxy not available" );
   }
}

// SetFailedServiceNameCallbackIF implementation
void GenericRecoveryClientComponent::onSetFailedServiceNameResponse( const ::boost::shared_ptr < Generic_RecoveryClientProxy > & proxy,
                                                                     const ::boost::shared_ptr < SetFailedServiceNameResponse >& response ){
   if ( _lcmRecoveryClientProxy == proxy ){
      LOG_INFO( "GenericRecoveryClientComponent::Received onSetFailedServiceNameResponse: Response= %s", Response_Name( response->getResponse( ) ) );
   } else {
      LOG_INFO( "GenericRecoveryClientComponent::Received onSetFailedServiceNameResponse: Response= %s", Response_Name( response->getResponse( ) ) );
      LOG_ERROR( "GenericRecoveryClientComponent::onSetFailedServiceNameResponse" );
   }

}

// CheckResetCondition implementation

/*! \fn
  *  \brief response from LCM to indicate if for this service a reset should be allowed
  *
  *    There are usecases - see LCM - where no reset is supposed to be done. Via this interface
  *    the LCM can prevent reset via this call.
  *
  *  \param proxy - pointer to proxy member where the message was received for
  *  \param response - contains the send data
  *
  *  \exception none.
  *  \return none.
  */
void GenericRecoveryClientComponent::onCheckResetConditionResponse( const ::boost::shared_ptr < Generic_RecoveryClientProxy >& proxy,
                                                                    const ::boost::shared_ptr < CheckResetConditionResponse >& response ){
   if ( _lcmRecoveryClientProxy == proxy ){
      if ( response->hasIsOKToReset( ) ){
         if ( response->getIsOKToReset( ) != LcmResetAllowed__ALLOW_TO_RESET ){
            LOG_INFO( "GenericRecoveryClientComponent::onCheckResetConditionResponse: Lcm informs that reset is not allowed" );
            _genericRecoveryClientResetAllowed = false;
         } else {
            LOG_INFO( "GenericRecoveryClientComponent::onCheckResetConditionResponse: Lcm informs that reset is allowed" );
         }
      }
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onCheckResetConditionResponse: proxy not available " );
   }

}

// CheckResetCondition implementation

/*! \fn
  *  \brief response from LCM to indicate if for this service a reset should be allowed had an error
  *
  *    There are usecases - see LCM - where no reset is supposed to be done. In case of error the
  *    reset should always be allowed.
  *
  *  \param proxy - pointer to proxy member where the message was received for
  *  \param error - contains the error reason data
  *
  *  \exception none.
  *  \return none.
  */
void GenericRecoveryClientComponent::onCheckResetConditionError( const ::boost::shared_ptr < Generic_RecoveryClientProxy >& proxy,
                                                                 const ::boost::shared_ptr < CheckResetConditionError >   & error ){
   if ( ( _lcmRecoveryClientProxy == proxy ) && error->hasName( ) && error->hasMessage( ) ){
      LOG_ERROR( "GenericRecoveryClientComponent::Received onCheckResetConditionError: ErrorName=%s, ErrorMessage=%s", error->getName( ).c_str( ), error->getMessage( ).c_str( ) );
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onCheckResetConditionError: proxy not available" );
   }

}

void GenericRecoveryClientComponent::onSetFailedServiceNameError( const ::boost::shared_ptr < Generic_RecoveryClientProxy >& proxy,
                                                                  const ::boost::shared_ptr < SetFailedServiceNameError >  & error ){
   if ( ( _lcmRecoveryClientProxy == proxy ) && error->hasName( ) && error->hasMessage( ) ){
      LOG_ERROR( "GenericRecoveryClientComponent::Received onSetFailedServiceNameError: ErrorName=%s, ErrorMessage=%s", error->getName( ).c_str( ), error->getMessage( ).c_str( ) );
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onSetFailedServiceNameError: proxy not available" );
   }

}

// SetFailedServiceStatusCallbackIF implementation
void GenericRecoveryClientComponent::onSetFailedServiceStatusResponse( const ::boost::shared_ptr < Generic_RecoveryClientProxy >   & proxy,
                                                                       const ::boost::shared_ptr < SetFailedServiceStatusResponse >& response ){
   if ( _lcmRecoveryClientProxy == proxy ){
      LOG_INFO( "GenericRecoveryClientComponent::Received onSetFailedServiceStatusResponse: Response= %s", Response_Name( response->getResponse( ) ) );
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onSetFailedServiceStatusResponse: proxy not available" );
   }
}

void GenericRecoveryClientComponent::onSetFailedServiceStatusError( const ::boost::shared_ptr < Generic_RecoveryClientProxy >& proxy,
                                                                    const ::boost::shared_ptr < SetFailedServiceStatusError >& error ){
   if ( ( _lcmRecoveryClientProxy == proxy ) && error->hasName( ) && error->hasMessage( ) ){
      LOG_ERROR( "GenericRecoveryClientComponent::Received onSetFailedServiceStatusError: ErrorName=%s, ErrorMessage=%s", error->getName( ).c_str( ), error->getMessage( ).c_str( ) );
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onSetFailedServiceStatusError: proxy not available" );
   }
}

// SetCallStackRequestCounterCallbackIF implementation
void GenericRecoveryClientComponent::onSetCallStackRequestCounterResponse( const ::boost::shared_ptr < Generic_RecoveryClientProxy >       & proxy,
                                                                           const ::boost::shared_ptr < SetCallStackRequestCounterResponse >& response ){
   if ( _lcmRecoveryClientProxy == proxy ){
      LOG_INFO( "GenericRecoveryClientComponent::Received onSetCallStackRequestCounterResponse: Response= %s", Response_Name( response->getResponse( ) ) );
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onSetCallStackRequestCounterResponse: proxy not available" );
   }
}

void GenericRecoveryClientComponent::onSetCallStackRequestCounterError( const ::boost::shared_ptr < Generic_RecoveryClientProxy >    & proxy,
                                                                        const ::boost::shared_ptr < SetCallStackRequestCounterError >& error ){
   if ( ( _lcmRecoveryClientProxy == proxy ) && error->hasName( ) && error->hasMessage( ) ){
      LOG_ERROR( "GenericRecoveryClientComponent::Received onSetCallStackRequestCounterError: ErrorName=%s, ErrorMessage=%s", error->getName( ).c_str( ), error->getMessage( ).c_str( ) );
   } else {
      LOG_ERROR( "GenericRecoveryClientComponent::onSetCallStackRequestCounterError: proxy not available" );
   }
}

// *************
// Timer
// *************
// TimerCallbackIF implementation
void GenericRecoveryClientComponent::onExpired( asf::core::Timer                            & timer,
                                                boost::shared_ptr < asf::core::TimerPayload > payload ){
   LOG_INFO( "GenericRecoveryClientComponent::onExpired: Timer:%u", timer.getAct( ) );
   if ( payload->hasReason( ) && payload->hasRepeatCount( ) ){
      LOG_INFO( "GenericRecoveryClientComponent::onExpired: TimerPayload Reason: %s, Repeat Count:%u", TimerPayload_Reason_Name( payload->getReason( ) ), payload->getRepeatCount( ) );
   }
   if ( ( timer.getAct( ) == _SystemdNotifyTimer.getAct( ) )
        && ( &timer == &_SystemdNotifyTimer )
        ){
      if ( ( _managerProxyAvailable == TRUE )
           && ( _unitProxyAvailable == TRUE )
           && ( _serviceProxyAvailable == TRUE )
           && ( _lcmRecoveryClientProxyAvailable == TRUE )
           ){
         sd_notify( 0, "WATCHDOG=1" );
         LOG_INFO( "GenericRecoveryClientComponent::onExpired: _SystemdNotifyTimer sd_notify - WATCHDOG=1 sent)" );
      } else {
         LOG_WARN( "GenericRecoveryClientComponent::onExpired: Watchdog not retriggered. Proxy missing." );
      }
      // check if everything is OK. Then restart the timer
      _SystemdNotifyTimer.start( * this, _SystemdNotifyTimeMSec, 1 );
   } else if ( ( timer.getAct( ) == _CheckTimer.getAct( ) )
               && ( &timer == &_CheckTimer )
               ){
      LOG_DEBUG( "GenericRecoveryClientComponent::onExpired: _CheckTimer)" );
      _CheckTimer.start( );
      _unitProxy->sendActiveStateGet( * this );
   } else {
      LOG_WARN( "GenericRecoveryClientComponent::onExpired: unknown timer" );
   }
}              // onExpired

DEFINE_CLASS_LOGGER_AND_LEVEL( "org.bosch.cm.lcm.RecoveryClient", GenericRecoveryClientComponent, Info );

}
}
}
}
}

