/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         configreaderapp.cpp
  * @brief        
  * @copyright    (C) 2013 - 2018 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>

#ifdef LCM_KDS_READER_FEATURE_SD_NOTIFY
#include <systemd/sd-daemon.h> // you also need to add -lsystemd-daemon to the linker settings
#endif

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_if.h"
#include "dp_generic_if.h"

#include <fstream>
#include <string>

typedef enum {
   eFinal        = 0,
   eEqual        = 1,
   eNotEqual     = 2,
   eLess         = 3,
   eLessEqual    = 4,
   eBigger       = 5,
   eBiggerEqual  = 6,
   eAnd          = 7,
   eOr           = 8,
   eFailed       = 9
}EOperation;

typedef struct {
   tBool      bResult;
   EOperation eNextOperation;
   std::string strRemaining;
}TOperationResult;

TOperationResult eCheckNextOperation(std::string strCommad) {

   tU8 KdsResult = 0;
   TOperationResult eResult;

   eResult.bResult = FALSE;
   eResult.eNextOperation = eFailed;
   eResult.strRemaining = "";

   printf ("eCheckNextOperation(): Analyse command: '%s'\n",strCommad.c_str());

   // get KDS key
   size_t startPos = strCommad.find("<");
   size_t endPos = strCommad.find(">");
   size_t splitPos = strCommad.find(".");

   printf("%zu,%zu,%zu\n", startPos, endPos, splitPos);

   if ( (startPos != std::string::npos) && (splitPos != std::string::npos)) {
      startPos += 1;
      std::string strConfigElement = strCommad.substr(startPos, splitPos-startPos);
      std::string strConfigItem    = strCommad.substr(splitPos+1, endPos-splitPos-1);

      printf ("eCheckNextOperation(): Check for KDS Key: '%s.%s'\n",strConfigElement.c_str(), strConfigItem.c_str());

      if ( DP_S32_NO_ERR == DP_s32GetConfigItem(strConfigElement.c_str(), strConfigItem.c_str(), &KdsResult, 1, FALSE) ){
         // now we have the KDS value --> go on parsing command
         printf ("eCheckNextOperation(): KDS Key: '%d'\n",KdsResult);
         if (strCommad.length() > endPos+1) {
            std::string strRemainOp = strCommad.c_str()+endPos+1;
            std::string strOperand;
            std::string strValue;

            printf ("eCheckNextOperation(): Remaining string: '%s(%zu)'\n",strRemainOp.c_str(),strRemainOp.size());

            for(std::string::size_type i = 0; (i < strRemainOp.size()) && (strRemainOp[i] != ')'); ++i) {
               if ((strRemainOp[i] == '!') || (strRemainOp[i] == '=') || (strRemainOp[i] == '>') || (strRemainOp[i] == '<')) {
                  strOperand.push_back(strRemainOp[i]);
               }else if ((strRemainOp[i] >= '0') || (strRemainOp[i] >= '9')) {
                  strValue.push_back(strRemainOp[i]);
               }
            }

            if ((strValue.size() != 0) && (strOperand.size() != 0)) {
               int iValue = atoi(strValue.c_str());

               EOperation eOp = eFinal;
               // now we should have everything to execute command
               if (0 == strOperand.compare("==")) {
                  eOp = eEqual;
                  if ((int)KdsResult == iValue) eResult.bResult = TRUE;
               } else if (0 == strOperand.compare("!=")) {
                  eOp = eNotEqual;
                  if ((int)KdsResult != iValue) eResult.bResult = TRUE;
               } else if (0 == strOperand.compare(">")) {
                  eOp = eBigger;
                  if ((int)KdsResult > iValue) eResult.bResult = TRUE;
               } else if (0 == strOperand.compare(">=")) {
                  eOp = eBiggerEqual;
                  if ((int)KdsResult >= iValue) eResult.bResult = TRUE;
               } else if (0 == strOperand.compare("<")) {
                  eOp = eLess;
                  if ((int)KdsResult < iValue) eResult.bResult = TRUE;
               } else if (0 == strOperand.compare("<=")) {
                  eOp = eLessEqual;
                  if ((int)KdsResult <= iValue) eResult.bResult = TRUE;
               }

               printf ("eCheckNextOperation(): Operand: '%s(%d)', Value: '%s(%d), Result: %d'\n",strOperand.c_str(), eOp, strValue.c_str(), iValue, eResult.bResult);
               eResult.eNextOperation = eFinal;

               //and now check if there is an additional operation to check
               eResult.strRemaining = "";
               size_t posAnd = strRemainOp.find("&&");
               size_t posOr = strRemainOp.find("||");
               if ( (posAnd != std::string::npos) || (posOr != std::string::npos)) {
                  if (posAnd < posOr) {
                     eResult.eNextOperation = eAnd;
                  } else {
                     eResult.eNextOperation = eOr;
                  }
                  eResult.strRemaining = strRemainOp.c_str()+strRemainOp.find("(");;
               }

               printf ("eCheckNextOperation(): Result: '%d', NextOperation: '%d', nextCmd: '%s'\n",eResult.bResult, eResult.eNextOperation, eResult.strRemaining.c_str());
               return eResult;

            } else {
               printf ("eCheckNextOperation(): Failed --> remaining string mismatch, no args!\n");
            }

         } else {
            printf ("eCheckNextOperation(): Failed --> remaining string mismatch!\n");
            return eResult;
         }

      } else {
         printf ("eCheckNextOperation(): Failed to read KDS value!\n");
         return eResult;
      }
   } else {
      printf ("eCheckNextOperation(): KDS Key not configured!\n");
      return eResult;
   }

   return eResult;
}

int iGetConfigItem(const char* strKdsConfigElement, const char* strKdsConfigItem, int value2Check = -1, int dumpLength = 0 , const char*  dumpFile = NULL) {
   int  retVal = 0;
   int  fd;
   std::string  fileName;
   tU8  au8Data[256] = {0};

   fileName = "/var/log/ConfigItems/";
   int   fdDir   = open( fileName.c_str(), O_RDONLY );
   if ( fdDir == - 1 ){
       if ( mkdir( fileName.c_str(), 0744 ) != 0 ){
         printf ("iGetConfigItem: Can't create dir %s", fileName.c_str());
      }
   } else{
      if (close( fdDir ) == -1){
        printf("iGetConfigItem: Failed to close the file '%s. Error: %s", fileName.c_str(), strerror(errno));
      }
   }

   fileName += strKdsConfigElement;
   fileName += ".";
   fileName += strKdsConfigItem;


   if ( DP_S32_NO_ERR == DP_s32GetConfigItem(strKdsConfigElement, strKdsConfigItem, au8Data, 256, FALSE) ){
      char str[15] = {0};
      sprintf(str, "%d", au8Data[0]);
      fileName += ".";
      fileName += str;
      fd = open(fileName.c_str(), O_CREAT | O_WRONLY, 0744);
      if ( fd == -1 ) {
         printf ("iGetConfigItem: Can't open file %s", fileName.c_str());
      } else {
         if (write(fd,(const void*)str,strlen(str)) == -1){
           printf("iGetConfigItem: Failed to write the file %s. Error: %s", fileName.c_str(), strerror(errno));
         }
         if (close(fd) == -1){
           printf("iGetConfigItem: Failed to close the file %s. Error: %s", fileName.c_str(), strerror(errno));
         }
      }
      if (value2Check == -1) {
         retVal = au8Data[0];
      } else if (au8Data[0] == value2Check) {
         retVal = 0;
      } else {
         retVal = -2;
      }

      FILE * pFile = NULL;
      if(dumpFile != NULL){
         pFile = fopen (dumpFile,"w");
      }      
      // Dump au8Data with length = dumpLength
      for (int i = 0; i < dumpLength; i++)
      {   
         if(pFile)
            fprintf(pFile," %02x", au8Data[i]);
         else   
            printf(" %02x", au8Data[i]);
      }      
      if(pFile){
         fclose (pFile);
      }      
      printf("\n");
      
   } else {
      retVal = -3;
   }

   printf ("ConfigItem: Element = '%s', Item = '%s', value = %d\n", strKdsConfigElement, strKdsConfigItem, retVal);
   return retVal;
}

int s32RunEnv(int argc, char **argv) {

   int retVal = 0;

   int c;

   const int strKdsConfigFile_size    = 120;
   const int strKdsConfigElement_size = 120;
   const int strKdsConfigItem_size    = 120;
   const int strKdsConfigValue_size   = 20;
   const int strCommand_size          = 500;
   const int strDumpFile_size          = 120;
   
   char strKdsConfigFile[strKdsConfigFile_size + 1]       = {0};
   char strKdsConfigElement[strKdsConfigElement_size + 1] = {0};
   char strKdsConfigItem[strKdsConfigItem_size + 1]       = {0};
   char strKdsConfigValue[strKdsConfigValue_size + 1]     = {0};
   char strCommand[strCommand_size + 1]                   = {0};
   int value2Check = -1;
   char dumpFile[strDumpFile_size + 1] = {0};
   int dumpLength = 0;

   tBool bCmdCheck = FALSE;

   while ((c = getopt (argc, argv, "c:e:i:v:k:l:f:")) != -1)
      switch (c) {
      case 'c':
         strncpy(strKdsConfigFile, optarg, strKdsConfigFile_size);
         break;
      case 'e':
         strncpy(strKdsConfigElement, optarg, strKdsConfigElement_size);
         break;
      case 'i':
         strncpy(strKdsConfigItem, optarg, strKdsConfigItem_size);
         break;
      case 'k':
         printf("---------------------\n");
         strncpy(strCommand, optarg, strCommand_size);
         bCmdCheck = TRUE;
         printf("-----  %s--------\n", strCommand);
         break;
      case 'l':         
         dumpLength = atoi(optarg);
         break;
      case 'v':
         strncpy(strKdsConfigValue, optarg, strKdsConfigValue_size);
         value2Check = atoi(optarg);
         break;
      case 'f':
         strncpy(dumpFile, optarg, strDumpFile_size);
         break;
      default:
         return -1;
      }

   if (bCmdCheck) {
      tBool bFinalResult = FALSE;
      EOperation eNextOperation = eOr;

      printf("main(): Start checking: %d\n",bFinalResult);
      TOperationResult cmd = eCheckNextOperation(strCommand);
      while ((cmd.eNextOperation != eFinal) && (cmd.eNextOperation != eFailed) ) {
         if (cmd.eNextOperation != eFailed) {
            if (eNextOperation == eAnd) {
               bFinalResult = bFinalResult&&cmd.bResult;
               printf("main(): New result after AND: %d\n",bFinalResult);
            } else {
               bFinalResult = bFinalResult||cmd.bResult;
               printf("main(): New result after OR: %d\n",bFinalResult);
            }
            eNextOperation = cmd.eNextOperation;
         }
         cmd = eCheckNextOperation(cmd.strRemaining);
      }

      if (eNextOperation == eAnd) {
         bFinalResult = bFinalResult&&cmd.bResult;
         printf("main(): New result after AND: %d\n",bFinalResult);

      } else {
         bFinalResult = bFinalResult||cmd.bResult;
         printf("main(): New result after OR: %d\n",bFinalResult);
      }

   } else if (strKdsConfigFile[0] == 0) {
      retVal = iGetConfigItem(strKdsConfigElement, strKdsConfigItem, value2Check, dumpLength , dumpFile);
   } else {
      std::string line;
      std::ifstream myfile( strKdsConfigFile );
      if (myfile) {
         while (std::getline( myfile, line )) {

            size_t startPos = line.find("<") + 1;
            size_t endPos = line.find(">");
            size_t splitPos = line.find(".");
            if ( (startPos != std::string::npos) && (splitPos != std::string::npos)) {
               std::string strConfigElement = line.substr(startPos, splitPos-startPos);
               std::string strConfigItem    = line.substr(splitPos+1, endPos-splitPos-1);

               iGetConfigItem(strConfigElement.c_str(), strConfigItem.c_str());
            }
         }
         myfile.close();
      }
   }

#   ifdef LCM_KDS_READER_FEATURE_SD_NOTIFY
   sd_notifyf( 0, "READY=1\n"                       // Tells the init system that daemon startup is finished.
         // This is only used by systemd if the service definition file has Type=notify set.
         "STATUS=Processing requests...\n" // Passes a single-line status string back to the init system that describes the daemon state.
         // This is free-form and can be used for various purposes
         "MAINPID=%lu\nPARENTID=%lu",
         (unsigned long)getpid( ),
         (unsigned long)getppid( ) );
#   endif

   return retVal;
}

#ifdef __cplusplus
extern "C" {
#endif
tVoid vStartApp(int argc, char **argv) {
   printf("And now start proccess() --> config process.\n");
   _exit(s32RunEnv(argc, argv));
   return;
}
#ifdef __cplusplus
}
#endif
