/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         genericDBusClient.cpp
  * @brief
  * @copyright    (C) 2013 - 2018 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */
#include "asf/dbus/DBusDaemonProxy.h"
#include "asf/dbus/DBusPropertiesProxyWrapper.h"
#include "asf/dbus/DBusProxyDelegate.h"
#include "asf/dbus/DBusMessage.h"
#include <errno.h>

#include "genericDBusClient.h"

// \todo: boost::shared_ptr is used here cause by ASF interface.
using namespace ::asf::core;
using namespace ::asf::stream;

DEFINE_CLASS_LOGGER_AND_LEVEL( "org/bosch/cm/datacollector/genericDBusClient", genericDBusClient, Info );


genericDBusClient::genericDBusClient( const std::string& portName,
                                      int                sock,
                                      const std::string& strInterface ) : _bSrvAvailable( FALSE ),
   _genericProxy( genericAsfDBusProxy::createProxy( portName, * this, strInterface ) ){
   u16FuncIdCounter = 0;
   _sock            = sock;
   printf( "Constructor for genericDBusClient called\n" );
}

genericDBusClient::~genericDBusClient( ){
   printf( "~genericDBusClient called" );
}

void genericDBusClient::onAvailable( const ::boost::shared_ptr < asf::core::Proxy >& proxy,
                                     const asf::core::ServiceStateChange           & stateChange ){
   printf( "genericDBusClient::onAvailable(): CALLED\n" );
   (void)stateChange;
   if ( _genericProxy == proxy ){
      printf( "genericDBusClient : onAvailable _AudioProxy\n" );
      _bSrvAvailable = TRUE;
      std::map < uint16, TPropertyConf >::iterator it;
      for ( it = _mapSignalRegs.begin( ); it != _mapSignalRegs.end( ); it++ ){
         if ( it->second.eType == eSignal ){
            _genericProxy->sendSignalRegister( it->second.strFuncName.c_str( ), it->first, * this );
         } else {
            _genericProxy->sendPropertyRegister( it->second.strFuncName.c_str( ), it->first, * this );
            _genericProxy->sendPropertyGet( it->second.strFuncName.c_str( ), it->first, * this );
         }
      }
   }
} // genericDBusClient::onAvailable

void genericDBusClient::onUnavailable( const ::boost::shared_ptr < asf::core::Proxy >& proxy,
                                       const asf::core::ServiceStateChange           & stateChange ){
   printf( "genericDBusClient::onUnavailable(): CALLED\n" );
   (void)stateChange;
   if ( _genericProxy == proxy ){
      _bSrvAvailable = FALSE;
      printf( "genericDBusClient : onUnavailable _AudioProxy \n" );
   }
}

void genericDBusClient::addSignalRegistration( const std::string sigName ){
   printf( "genericDBusClient::addSignalRegistration(): '%s'\n", sigName.c_str( ) );
   TPropertyConf newSig = { sigName, eSignal };
   _mapSignalRegs[u16FuncIdCounter] = newSig;
   if ( _bSrvAvailable ){
      _genericProxy->sendSignalRegister( sigName.c_str( ), u16FuncIdCounter, * this );
   }
   u16FuncIdCounter++;
}

void genericDBusClient::addPropertyRegistration( const std::string propertyName ){
   printf( "genericDBusClient::addSignalRegistration(): '%s'\n", propertyName.c_str( ) );
   TPropertyConf newSig = { propertyName, eProperty };
   _mapSignalRegs[u16FuncIdCounter] = newSig;
   if ( _bSrvAvailable ){
      _genericProxy->sendPropertyRegister( propertyName.c_str( ), u16FuncIdCounter, * this );
      _genericProxy->sendPropertyGet( propertyName.c_str( ), u16FuncIdCounter, * this );
   }
   u16FuncIdCounter++;
}

void genericDBusClient::onSignalChangedError( const ::boost::shared_ptr < genericAsfDBusProxy >& proxy,
                                              uint16                                             funcId ){
   printf( "genericDBusClient::onMainSourceSoundPropertyChangedError(): CALLED\n" );
   (void)proxy;
   (void)funcId;
}

void genericDBusClient::getStructFromMsg( std::string    & strSockMsg,
                                          int              inType,
                                          DBusMessageIter *pPayloadIterator ){
   //Initialize iterator for struct
   printf( "genericDBusClient::getStructFromMsg(): CALLED\n" );
   (void)inType;
   DBusMessageIter tmpIter;
   dbus_message_iter_recurse( pPayloadIterator, &tmpIter );

   int             current_type;
   strSockMsg += "[";
   while ( ( current_type = dbus_message_iter_get_arg_type( &tmpIter ) ) != DBUS_TYPE_INVALID ){
      printf( "Struct Type: %d('%c')\n", current_type, current_type );
      if ( ( current_type == DBUS_TYPE_INT16 ) || ( current_type == DBUS_TYPE_UINT16 )
           || ( current_type == DBUS_TYPE_BYTE ) || ( current_type == DBUS_TYPE_BOOLEAN )
           || ( current_type == DBUS_TYPE_INT32 ) || ( current_type == DBUS_TYPE_UINT32 )
           ){
         getIntFromMsg( strSockMsg, current_type, &tmpIter );
      } else if ( DBUS_TYPE_STRING == current_type ){
         getStringFromMsg( strSockMsg, current_type, &tmpIter );
      } else if ( DBUS_TYPE_STRUCT == current_type ){
         getStructFromMsg( strSockMsg, current_type, &tmpIter );
      } else if ( DBUS_TYPE_ARRAY == current_type ){
         getArrayFromMsg( strSockMsg, current_type, &tmpIter );
      } else {
         strSockMsg += (char)current_type;
         strSockMsg += "-????";
      }
      dbus_message_iter_next( &tmpIter );//Go to next argument of structiter
      strSockMsg += ":";
   }
   strSockMsg += "]";
} // genericDBusClient::getStructFromMsg

void genericDBusClient::getArrayFromMsg( std::string    & strSockMsg,
                                         int              inType,
                                         DBusMessageIter *pPayloadIterator ){
   //Initialize iterator for struct
   printf( "genericDBusClient::getArrayFromMsg(): CALLED\n" );
   (void)inType;
   DBusMessageIter tmpIter;
   dbus_message_iter_recurse( pPayloadIterator, &tmpIter );

   int             current_type;
   strSockMsg += "(";
   while ( ( current_type = dbus_message_iter_get_arg_type( &tmpIter ) ) != DBUS_TYPE_INVALID ){
      printf( "Array Type: %d('%c')\n", current_type, current_type );
      if ( ( current_type == DBUS_TYPE_INT16 ) || ( current_type == DBUS_TYPE_UINT16 )
           || ( current_type == DBUS_TYPE_BYTE ) || ( current_type == DBUS_TYPE_BOOLEAN )
           || ( current_type == DBUS_TYPE_INT32 ) || ( current_type == DBUS_TYPE_UINT32 )
           ){
         getIntFromMsg( strSockMsg, current_type, &tmpIter );
      } else if ( DBUS_TYPE_STRING == current_type ){
         getStringFromMsg( strSockMsg, current_type, &tmpIter );
      } else if ( DBUS_TYPE_STRUCT == current_type ){
         getStructFromMsg( strSockMsg, current_type, &tmpIter );
      } else if ( DBUS_TYPE_ARRAY == current_type ){
         getArrayFromMsg( strSockMsg, current_type, &tmpIter );
      } else {
         strSockMsg += (char)current_type;
         strSockMsg += "-????";
      }
      dbus_message_iter_next( &tmpIter );//Go to next argument of structiter
      strSockMsg += ":";
   }
   strSockMsg += ")";
} // genericDBusClient::getArrayFromMsg

void genericDBusClient::getIntFromMsg( std::string    & strSockMsg,
                                       int              curType,
                                       DBusMessageIter *pPayloadIterator ){
   printf( "genericDBusClient::getIntFromMsg(): CALLED\n" );

   strSockMsg += (char)curType;
   int  data      = 0;
   dbus_message_iter_get_basic( pPayloadIterator, &data );
   printf( "data(int): %d\n", data );
   char cData[16] = { 0 };
   sprintf( cData, "%d", data );
   strSockMsg += "-";
   strSockMsg += cData;
}

void genericDBusClient::getStringFromMsg( std::string    & strSockMsg,
                                          int              curType,
                                          DBusMessageIter *pPayloadIterator ){
   printf( "genericDBusClient::getIntFromMsg(): CALLED\n" );

   strSockMsg += (char)curType;
   char *cData = NULL;
   dbus_message_iter_get_basic( pPayloadIterator, &cData );
   strSockMsg += "-'''";
   strSockMsg += cData;
   strSockMsg += "'''";
}

void genericDBusClient::onSignalChangedSignal( const ::boost::shared_ptr < genericAsfDBusProxy >& proxy,
                                               uint16                                             funcId,
                                               ::asf::dbus::DBusMessage                         & message ){
   printf( "genericDBusClient::onMainSourceSoundPropertyChangedSignal(): CALLED (%d)\n", funcId );
   (void)proxy;
   std::map < uint16, TPropertyConf >::iterator it = _mapSignalRegs.find( funcId );
   if ( it != _mapSignalRegs.end( ) ){
      printf( "genericDBusClient::onMainSourceSoundPropertyChangedSignal(): Update for '%s' received.\n", it->second.strFuncName.c_str( ) );
      std::string ss = "DBUS:NewSignal:";
      ss += it->second.strFuncName;
      ss += ":DATA:";

      if ( message.isError( ) ){
         printf( "onSignalChangedSignal: Message error detected!\n" );
      }
      int              current_type;
      DBusMessageIter *pPayloadIterator = NULL;
      DBusMessageIter  payloadIterator;

      if ( message.isProperty( ) ){
         printf( "onSignalChangedSignal: Property received!\n" );
         pPayloadIterator = message.getPropertyPayloadIter( );
      } else if ( message.isSignal( ) ){
         pPayloadIterator = &payloadIterator;
         printf( "onSignalChangedSignal: Signal received!\n" );
         if ( FALSE == dbus_message_iter_init( message.getLibDBusMessage( ), &payloadIterator ) ){
            printf( "onSignalChangedSignal: can not init payloadIterator\n" );
         }
      }

      printf( "onSignalChangedSignal: Iterator address: %p!\n", pPayloadIterator );
      #if 1

         while ( ( current_type = dbus_message_iter_get_arg_type( pPayloadIterator ) ) != DBUS_TYPE_INVALID ){
            printf( "Type: %d('%c')\n", current_type, current_type );
            if ( ( current_type == DBUS_TYPE_INT16 ) || ( current_type == DBUS_TYPE_UINT16 )
                 || ( current_type == DBUS_TYPE_BYTE ) || ( current_type == DBUS_TYPE_BOOLEAN )
                 || ( current_type == DBUS_TYPE_INT32 ) || ( current_type == DBUS_TYPE_UINT32 )
                 ){
               getIntFromMsg( ss, current_type, pPayloadIterator );
            } else if ( DBUS_TYPE_STRING == current_type ){
               getStringFromMsg( ss, current_type, pPayloadIterator );
            } else if ( DBUS_TYPE_STRUCT == current_type ){
               getStructFromMsg( ss, current_type, pPayloadIterator );
            } else if ( DBUS_TYPE_ARRAY == current_type ){
               getArrayFromMsg( ss, current_type, pPayloadIterator );
            } else {
               ss += (char)current_type;
               ss += "-????";
            }
            dbus_message_iter_next( pPayloadIterator );
            ss += ":";
         }
      #endif // if 1

      ss += "END";
      printf( "String to send: '%s'\n", ss.c_str( ) );
      if ( write( _sock, ss.c_str( ), ss.length( ) ) == - 1 ){
         printf( "genericDBusClient::onSignalChangedSignal(): FAILED to write the socket. Error: %s", strerror( errno ) );
      }
   }

} // genericDBusClient::onSignalChangedSignal

