/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         clienttestapp.cpp
  * @brief        
  * @copyright    (C) 2013 - 2018 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */
/*
    C socket server example, handles multiple clients using threads
 */

#include<stdio.h>
#include<string.h>      //strlen
#include<stdlib.h>      //strlen
#include<sys/socket.h>
#include<arpa/inet.h>   //inet_addr
#include<unistd.h>      //write
#include<pthread.h>     //for threading , link with lpthread
#include<errno.h>

#include "genericAsfStarter.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_if.h"
#include "dp_generic_if.h"

//#include <map>

#define ETG_S_IMPORT_INTERFACE_GENERIC
#define ETG_S_IMPORT_INTERFACE_PLATFORM
#include "etg_if.h"

#define SCD_S_IMPORT_INTERFACE_GENERIC
#include "scd_if.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"

#define AMT_S_IMPORT_INTERFACE_GENERIC
#include "amt_if.h"

#define REG_S_IMPORT_INTERFACE_GENERIC
#include "reg_if.h"

#include <map>

#define TR_CLASS_PROC_STARTUP_DYN          0x8201

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PROC_STARTUP_DYN
#include "trcGenProj/Header/clienttestapp.cpp.trc.h"
#endif

extern "C" void _exit(int status);


typedef struct {
    tU16  u16AppId;
    tBool bTerminate;
} TSockInfo;

std::map<int, TSockInfo> tSocketInfo;


//the thread function
void *connection_handler(void *);
void *vSocketSrvThread(void *pPort);
tVoid vTraceOutRegistry(const tChar *szBaseName, int sock);
tVoid vGetKdsEntry(tChar* strKdsConfigElement, tChar* strKdsConfigItem, int sock);
tVoid vStartClient(tU16 u16AppId, int sock);
tVoid vStopClient();

#define PROCIP_SOCKET_CMD_GET_KDS_ELEMENT         0
#define PROCIP_SOCKET_CMD_GET_REGISTRY            1
#define PROCIP_SOCKET_CMD_START_CCA_MSG_HDL       2
#define PROCIP_SOCKET_CMD_SEND_CCA_MSG            3
#define PROCIP_SOCKET_CMD_TERMINATE_CONNECTION    4
#define PROCIP_SOCKET_CMD_START_DBUS_CLIENT       5
#define PROCIP_SOCKET_CMD_DBUS_CLIENT_SIG_REGISTER 6
#define PROCIP_SOCKET_CMD_EXIT_PROC               7
#define PROCIP_SOCKET_CMD_DBUS_CLIENT_PROP_REGISTER 8

#define PROCIP_REGKEY_BLAUPUNKT   "/dev/registry/LOCAL_MACHINE/SOFTWARE/BLAUPUNKT/"

//int main(int args, char **argv) {
tS32 s32RunEnv(int port){

    //return startGenericAsfDbusClient("org.genivi.audiomanager", "/org/genivi/audiomanager/CommandInterface", "org.genivi.audiomanager.CommandInterface");

    int iPort = port;
    int socket_desc , client_sock , c , *new_sock;
    struct sockaddr_in server , client;

    // Start Trace
    vInitPlatformEtg();

    amt_bInit();
    scd_init();

    //Create socket
    socket_desc = socket(AF_INET , SOCK_STREAM , 0);
    if (socket_desc == -1)
    {
        ETG_TRACE_FATAL(("Could not create socket"));
        ETG_TRACE_FATAL(("%s", strerror(errno)));
        return OSAL_ERROR;
    }
    ETG_TRACE_FATAL(("Socket created"));

    //Prepare the sockaddr_in structure
    server.sin_family = AF_INET;
    server.sin_addr.s_addr = INADDR_ANY;
    server.sin_port = htons( iPort );
    std::memset(&server.sin_zero, 0, sizeof(server.sin_zero));
    //Bind
    if( bind(socket_desc,(struct sockaddr *)&server , sizeof(server)) < 0)
    {
        //print the error message
        ETG_TRACE_FATAL(("bind failed. Error"));
        ETG_TRACE_FATAL(("%s", strerror(errno)));
        if ( close(socket_desc) == -1 ){
           ETG_TRACE_FATAL( ( "s32RunEnv: FAILED to close the socket. Error: %s", strerror(errno) ) );
        }
        return OSAL_ERROR;
    }
    ETG_TRACE_FATAL(("bind done"));

    //Listen
    if ( listen(socket_desc, 3) != 0) {
        ETG_TRACE_FATAL(("listen fail"));
        ETG_TRACE_FATAL(("%s", strerror(errno)));
        if ( close(socket_desc) == -1 ){
           ETG_TRACE_FATAL( ( "s32RunEnv: FAILED to close the socket. Error: %s", strerror(errno) ) );
        }
        return OSAL_ERROR;
    }

    //Accept and incoming connection
    ETG_TRACE_FATAL(("Waiting for incoming connections..."));
    c = sizeof(struct sockaddr_in);


    //Accept and incoming connection
    ETG_TRACE_FATAL(("Waiting for incoming connections..."));
    c = sizeof(struct sockaddr_in);
    while( (client_sock = accept(socket_desc, (struct sockaddr *)&client, (socklen_t*)&c)) )
    {
        ETG_TRACE_FATAL(("Connection accepted"));

        pthread_t sniffer_thread;
        new_sock = (int*)malloc(1);
        *new_sock = client_sock;

        if( pthread_create( &sniffer_thread , NULL ,  connection_handler , (void*) new_sock) < 0)
        {
            ETG_TRACE_FATAL(("could not create thread"));
            ETG_TRACE_FATAL(("%s", strerror(errno)));
            if ( close(socket_desc) == -1 ){
               ETG_TRACE_FATAL( ( "s32RunEnv: FAILED to close the socket. Error: %s", strerror(errno) ) );
            }
            return OSAL_ERROR;
        }

        //Now join the thread , so that we dont terminate before the thread
        //pthread_join( sniffer_thread , NULL);
        ETG_TRACE_FATAL(("Handler assigned"));
    }

    if (client_sock < 0)
    {
        ETG_TRACE_FATAL(("accept failed"));
        ETG_TRACE_FATAL(("%s", strerror(errno)));
        if ( close(socket_desc) == -1 ){
           ETG_TRACE_FATAL( ( "s32RunEnv: FAILED to close the socket. Error: %s", strerror(errno) ) );
        }
        return OSAL_ERROR;
    }
    if ( close(socket_desc) == -1 ){
        ETG_TRACE_FATAL( ( "s32RunEnv: FAILED to close the socket. Error: %s", strerror(errno) ) );
    }
    return OSAL_OK;
}

class amt_tclBaseStreamMessage : public amt_tclBaseMessage
{
public:
    amt_tclBaseStreamMessage (tU8 *pStreamData){
        tU32 u32Size = * ( (tU32*)( pStreamData + AMT_C_U32_BASEMSG_SIZE ) ); // lint !e826 Suspicious pointer-to-pointer conversion (area too small)

        vSetDynMsgSize(0);
        vAddDynMsgSize(u32Size);

        // Allocate in Shared Memory
        bAllocateMessage();

        vSetStreamU8(0, pStreamData, u32Size, u32Size);
    }

};

tVoid vPostMessage(amt_tclBaseMessage *poMessage);

/*
 * This will handle connection for each client
 * */ 
void *connection_handler(void *socket_desc)
{
    //Get the socket descriptor
    int sock = *(int*)socket_desc;
    int read_size;
    char client_message[2000];

    //Send some messages to the client
    //message = "Greetings! I am your connection handler\n";
    //write(sock , message , strlen(message));

    //message = "Now type something and i shall repeat what you type \n";
    //write(sock , message , strlen(message));

    //Receive a message from client
    while( (read_size = (int)recv(sock , client_message , 2000 , 0)) > 0 )
    {
        char* pStartData = strchr(client_message, ':');
        int iCmd = 255;
        if (pStartData != NULL) {
            pStartData[0] = 0;
            pStartData += 1;
            iCmd = atoi(client_message);
            ETG_TRACE_FATAL(("ClientMessageReceived: Command: %d (sock: %d), data: '%s' ", iCmd, sock, pStartData));
            //printf("ClientMessageReceived: Command: %d (sock: %d), data: '%s' ", iCmd, sock, pStartData);
            switch (iCmd) {
            case PROCIP_SOCKET_CMD_GET_REGISTRY:
            { //get registry info with given basefolder
                tChar           szBaseName[256];
                OSAL_szStringNCopy( (tString)szBaseName, PROCIP_REGKEY_BLAUPUNKT, sizeof(PROCIP_REGKEY_BLAUPUNKT) );
                OSAL_szStringNConcat( (tString)szBaseName, pStartData, 255 );
                vTraceOutRegistry(szBaseName, sock);
            }
            break;
            case PROCIP_SOCKET_CMD_START_CCA_MSG_HDL:
            { // start CCA message handler
                vStartClient((tU16)atoi(pStartData),sock);
            }
            break;
            case PROCIP_SOCKET_CMD_SEND_CCA_MSG:
            {  // convert string stream to bin container
                tU8 au8Data[400] = {0};
                for (tU16 i=0; i<strlen(pStartData)/2;i++) {
                    char cData[3]={pStartData[2*i], pStartData[(2*i)+1], 0};
                    au8Data[i] = (tU8)strtol(cData, NULL, 16);
                }

                amt_tclBaseStreamMessage oMyBaseMsg( (tU8*)au8Data);
                vPostMessage(&oMyBaseMsg);
            }
            break;
            case PROCIP_SOCKET_CMD_TERMINATE_CONNECTION:
            {
                tSocketInfo[sock].bTerminate = TRUE;
            }
            break;
            case PROCIP_SOCKET_CMD_GET_KDS_ELEMENT:
            {
                tChar*  strKdsConfigElement = pStartData;
                tChar*  pSeperator = strchr(pStartData, ';');
                if (pSeperator!= NULL) {
                    pSeperator[0] = 0;
                    tChar*  strKdsConfigItem = pSeperator+1;
                    vGetKdsEntry(strKdsConfigElement, strKdsConfigItem,sock);
                }
            }
            break;
            case PROCIP_SOCKET_CMD_START_DBUS_CLIENT:
            {
                //"<org.genivi.audiomanager></org/genivi/audiomanager/CommandInterface><org.genivi.audiomanager.CommandInterface>"
                vStartDBusHandler(pStartData, sock);

                //OSAL_s32ThreadWait(5000);

                //addSignalRegister("VolumeChanged",3);
                //addSignalRegister("NewMainConnection",3);
            }
            break;
            case PROCIP_SOCKET_CMD_DBUS_CLIENT_SIG_REGISTER:
            {
                vAddDBusSignalRegister(pStartData,sock);
            }
            break;
            case PROCIP_SOCKET_CMD_DBUS_CLIENT_PROP_REGISTER:
            {
                vAddDBusPropertyRegister(pStartData,sock);
            }
            break;
            case PROCIP_SOCKET_CMD_EXIT_PROC:
            {

            }
            break;
            default:
                if ( write(sock , client_message , strlen(client_message)) == -1){
                   ETG_TRACE_FATAL( ( "ClientMessageReceived: FAILED to write the socket. Error: %s", strerror(errno) ) );
                }
                ETG_TRACE_FATAL(("ClientMessageReceived: Unknown command: %d", iCmd));
                break;
            }
        } else {
            //Send the message back to client
            if ( write(sock , client_message , strlen(client_message)) == -1){
               ETG_TRACE_FATAL( ( "ClientMessageReceived: FAILED to write the socket. Error: %s", strerror(errno) ) );
            }
            ETG_TRACE_FATAL(("ClientMessageReceived: Cannot interpret: '%s'", client_message));
        }
    }

    if(read_size == 0)
    {
        ETG_TRACE_FATAL(("Client disconnected"));
    }
    else if(read_size == -1)
    {
        ETG_TRACE_FATAL(("recv failed"));
    }

    //Free the socket pointer
    free(socket_desc);

    return 0;
}


#define PROCIP_BASE_NAME_SIZE       200

tVoid vTraceOutRegistry(const tChar *szBaseName, int sock){
    tS8                   as8Name[PROCIP_BASE_NAME_SIZE];
    tS32                  as32Data[255];

    OSAL_trIOCtrlDir      rDir;
    OSAL_trIOCtrlDir      rDirV;
    OSAL_trIOCtrlRegistry rReg;
    OSAL_tIODescriptor    fd;

    ETG_TRACE_FATAL( ( "PTH: %s", szBaseName ) );

    {
        char szFolder[256] = {0};
        sprintf(szFolder, "BaseFolder: '%s';",szBaseName);
        if ( write(sock , szFolder , strlen(szFolder)) == -1){
           ETG_TRACE_FATAL( ( "vTraceOutRegistry: FAILED to write the socket. Error: %s", strerror(errno) ) );
        }
    }

    fd = OSAL_IOOpen(szBaseName, OSAL_EN_READONLY);
    if (fd != OSAL_ERROR){

        /* First, recursively show all the subkeys */
        rDir.fd        = fd;
        rDir.s32Cookie = 0;

        while (OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_REGREADDIR, ( intptr_t ) & rDir) != OSAL_ERROR){
            std::string strBaseName( szBaseName );
            strBaseName += "/";
            strBaseName += (tString)rDir.dirent.s8Name;
            if ( strBaseName.size( ) < ( sizeof(as8Name) - 1 ) ) {
               OSAL_szStringNCopy( (tString)as8Name, strBaseName.c_str( ), strBaseName.size( ) );
               as8Name[strBaseName.size( )] = '\0';
            } else {
               ETG_TRACE_FATAL( ( "size of %s is not valid ", strBaseName.c_str( ) ) );
            }

            //vTraceOutRegistry( (tString)as8Name, sock);
            ETG_TRACE_FATAL( ( "VAL Folder: '%64s'", (const tChar*)as8Name) );
            {
                char szData[256] = {0};
                sprintf(szData, "Folder: '%s';",(const tChar*)as8Name);
                if ( write(sock , szData , strlen(szData)) == -1){
                   ETG_TRACE_FATAL( ( "vTraceOutRegistry: FAILED to write the socket. Error: %s", strerror(errno) ) );
                }
            }

        }

        /* Now show all the values of this key */
        rDirV.fd        = fd;
        rDirV.s32Cookie = 0;

        while (OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_REGENUMVALUE, ( intptr_t ) & rDirV) != OSAL_ERROR){
            rReg.pcos8Name = rDirV.dirent.s8Name;
            rReg.ps8Value  = (tU8*)as32Data;
            rReg.u32Size   = sizeof( as32Data );

            if (OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_REGGETVALUE, ( intptr_t ) & rReg) != OSAL_ERROR){
                // This is correct because not all traces come out if this is missing
                if (rReg.s32Type == OSAL_C_S32_VALUE_S32){
                    ETG_TRACE_FATAL( ( "VAL number: '%32s' 0x%08x (%d)", (const tChar*)rDirV.dirent.s8Name, as32Data[0], as32Data[0] ) );
                    {
                        char szData[256] = {0};
                        sprintf(szData, "Key: '%s', Value: '%d';",(const tChar*)rDirV.dirent.s8Name, as32Data[0]);
                        if ( write(sock , szData , strlen(szData)) == -1){
                           ETG_TRACE_FATAL( ( "vTraceOutRegistry: FAILED to write the socket. Error: %s", strerror(errno) ) );
                        }
                    }
                } else if (rReg.s32Type == OSAL_C_S32_VALUE_STRING){
                    ETG_TRACE_FATAL( ( "VAL string: '%32s' '%s'", (const tChar*)rDirV.dirent.s8Name, (const tChar*)as32Data ) );
                    {
                        char szData[256] = {0};
                        sprintf(szData, "Key: '%s', String: '%s';",(const tChar*)rDirV.dirent.s8Name, (const tChar*)as32Data);
                        if ( write(sock , szData , strlen(szData)) == -1){
                           ETG_TRACE_FATAL( ( "vTraceOutRegistry: FAILED to write the socket. Error: %s", strerror(errno) ) );
                        }
                    }
                }
            }
        }
        OSAL_s32IOClose(fd);
    }
} // vTraceOutRegistry

tVoid vGetKdsEntry(tChar* strKdsConfigElement, tChar* strKdsConfigItem, int sock) {

    tU8 au8Data[256] = {0};
    ETG_TRACE_FATAL(("vGetKdsEntry(): '%40s' , '%40s')", strKdsConfigElement, strKdsConfigItem));

    if ( DP_S32_NO_ERR == DP_s32GetConfigItem(strKdsConfigElement, strKdsConfigItem, au8Data, 256) ){
        char szData[400] = {0};
        OSAL_szStringConcat( szData, "KdsElementInfo:");
        memcpy((szData+strlen("KdsElementInfo:")+1), au8Data, 256);
        if ( write(sock , szData , 400) == -1){
           ETG_TRACE_FATAL( ( "vGetKdsEntry: FAILED to write the socket. Error: %s", strerror(errno) ) );
        }
    } else {
        char szData[400] = {0};
        OSAL_szStringConcat( szData, "KdsElementFailed");
        if ( write(sock , szData , strlen(szData)) == -1){
           ETG_TRACE_FATAL( ( "vGetKdsEntry: FAILED to write the socket. Error: %s", strerror(errno) ) );
        }
    }

}

tVoid vCcaMsgHandlerThread(tVoid *pvArg);

tVoid vStartClient(tU16 u16AppId, int sock) {


    TSockInfo tSockTmp = {u16AppId, FALSE};
    tSocketInfo[sock] = tSockTmp;

    tChar            szMyProcSpecName[32] = {0};


    // start receiving thread
    sprintf(szMyProcSpecName, "CcaSimu%04x", u16AppId);

    OSAL_trThreadAttribute rAttr;
    rAttr.szName       = (tString)szMyProcSpecName;
    rAttr.s32StackSize = 4096;
    rAttr.u32Priority  = 100;
    rAttr.pfEntry      = (OSAL_tpfThreadEntry)vCcaMsgHandlerThread;
    rAttr.pvArg        = ( tPVoid )((long int)sock);

    OSAL_tThreadID hThreadId = OSAL_ThreadSpawn(&rAttr);
    if ( hThreadId == OSAL_ERROR ){
        ETG_TRACE_ERR( ( "vStartClient(): Failed to spawn thread!" ) );
    }
}

tVoid vCcaMsgHandlerThread(tVoid *pvArg){

    int sock = (int)((intptr_t)pvArg);

    tU16 u16AppId = tSocketInfo[sock].u16AppId;

    // create message queue
    OSAL_tMQueueHandle _hCcaInQueue = OSAL_C_INVALID_HANDLE;

    tChar          szName[32];

    if(OSALUTIL_s32SaveNPrintFormat(szName, sizeof( szName ), "mbx_%d", u16AppId) == OSAL_ERROR) {
       tU32 u32ErrorReason = OSAL_u32ErrorCode( );
       ETG_TRACE_ERRMEM( ( "SPM: vCcaMsgHandlerThread !!!!!! Error detected !!!!!! cannot create string: error 0x%08X (%s)",
                           (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
    }

    _hCcaInQueue = OSAL_C_INVALID_HANDLE;
    if (OSAL_ERROR == OSAL_s32MessageQueueOpen(szName, OSAL_EN_READWRITE, &_hCcaInQueue)){
        if (OSAL_ERROR == OSAL_s32MessageQueueCreate(
                szName,
                20,
                SCD_MAILBOX_MAX_MESSAGE_LENGTH,
                OSAL_EN_READWRITE,
                &_hCcaInQueue
        )
        ){
            ETG_TRACE_FATAL( ( "vCcaMsgHandlerThread(): FAILED to open/create queue." ) );
        }
    }
    while (!tSocketInfo[sock].bTerminate && (_hCcaInQueue != OSAL_C_INVALID_HANDLE)){
        amt_tclBaseMessage oMsgObject;
        tS32               s32RetVal = OSAL_s32MessageQueueWait(
                _hCcaInQueue,
                (tU8*)oMsgObject.prGetOSALMsgHandle(), /*pointer of the MsgHandle field*/
                sizeof( OSAL_trMessage ),
                OSAL_NULL,
                1000
        );

        //ETG_TRACE_FATAL( ( "vCcaMsgHandlerThread(): s32RetVal: %x",  s32RetVal) );
        if (s32RetVal != OSAL_ERROR){
            // gets the pointer to shared memory
            if (sizeof( OSAL_trMessage ) == s32RetVal){
                oMsgObject.pu8SharedMemBase = OSAL_pu8MessageContentGet(* ( oMsgObject.prGetOSALMsgHandle() ), OSAL_EN_READWRITE);
                if (oMsgObject.pu8SharedMemBase != OSAL_NULL){
                    // send message to host via socket
                    if ( write(sock , oMsgObject.pu8SharedMemBase , oMsgObject.u32GetSize()) == -1){
                       ETG_TRACE_FATAL( ( "vCcaMsgHandlerThread(): FAILED to write the socket. Error: %s", strerror(errno) ) );
                    }

                    ETG_TRACE_FATAL(("vCcaMsgHandlerThread(): Send CCA message on sock: %d)", sock));

                    //ETG_TRACE_FATAL( ( "CCAMSG: RCV: %*x.!", ETG_LIST_LEN(oMsgObject.u32GetSize() ), ETG_LIST_PTR_T8(oMsgObject.pu8SharedMemBase) ) );
                    //ETG_TRACE_FATAL( ( "CCAMSG: END." ) );
                }
            } else {
                if (OSAL_u32ErrorCode() != OSAL_E_TIMEOUT){
                    // error
                } else {
                    // timeout
                }
            }
        } else {
            // ETG_TRACE_ERR_CLS((SPM_TRACE_CLASS_SPM, "bMessageQueueWait, receive error: %s", OSAL_coszErrorText(OSAL_u32ErrorCode())));
        }
    }

    //OSAL_s32MessageQueueDelete(szName);
} // vCcaMsgHandlerThread


tVoid vPostMessage(amt_tclBaseMessage *poMessage){

    tU16 u16TargetID = poMessage->u16GetTargetAppID();

    OSAL_tMQueueHandle hAppMsgQueue = scd_OpenQueue(u16TargetID);
    /* send message */
    if (hAppMsgQueue != OSAL_C_INVALID_HANDLE){
        tS32 s32Retval = OSAL_s32MessageQueuePost(hAppMsgQueue,
                (tCU8*)poMessage->prGetOSALMsgHandle(),
                sizeof( OSAL_trMessage ),
                OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST);
        if (s32Retval == OSAL_OK){
        } else {
            ETG_TRACE_FATAL(("vPostMessage: post to app '%d' failed.", u16TargetID));
            poMessage->bDelete();
        }
    } else {
        ETG_TRACE_FATAL(("vPostMessage: Cannot open queue for  app '%d'", u16TargetID));
    }

    return;
} // bPostMessage

tS32 s32ReadFileContent( const std::string& strFile, std::string& strBuf, tU32 u32Len){
    tS32        s32Ret = - 1;

    std::string strBuffer( u32Len + 1, 0 );
    FILE       *pFile  = fopen( strFile.c_str( ), "r" );
    if ( pFile ){
        s32Ret = (tS32)fread( ( tPS8 ) & strBuffer[0], 1, u32Len, pFile );
        if ( s32Ret > 0 ){
            strBuf.assign( strBuffer.c_str( ), (tU32)s32Ret );
        }
        fclose( pFile );
    }

    return( s32Ret );
}

void run_getenv (const char * name) {
    char * value;
    value = getenv (name);
    if (! value) {
        printf ("'%s' is not set.\n", name);
    }
    else {
        printf ("%s = %s\n", name, value);
    }
}

#ifdef __cplusplus
extern "C" {
#endif

//tVoid vStartApp(tVoid) {
tVoid vStartApp(int argc, char **argv) {

    int c;
    int port = 2355;
    char strPort[80]          = {0};

    printf("And now start proccess() --> clienttestapp.\n");
    while ((c = getopt (argc, argv, "p:")) != -1) {
        switch (c) {
        case 'p':
            strncpy(strPort, optarg, 20);
            port = atoi(optarg);
            break;
        default:
            port = 2355;
        }
    }
    printf("Start Socket on port %d\n", port);

    tS32 s32Ret = s32RunEnv(port);
    if (s32Ret == OSAL_OK) {
        _exit(1);
    }
    return;
}

#ifdef __cplusplus
}
#endif

