/**
 * @copyright (c) 2015-2021 Robert Bosch Car Multimedia GmbH
 * @addtogroup NavMiddleware
 * @{
 */

#include <string>
#include <boost/shared_ptr.hpp>

#include "Export.h"

#include "NavMiddleware.h"
#include "navmid/AIVIPresCtrlInterface.h"

using namespace ::navmiddleware;
using namespace ::aivi_pres_ctrl::adapter;

#include "navmid/InterfaceTraceConfig.h"
INTERFACE_LOGGER(inavmiddleware);
INTERFACE_LOGGER(position);
INTERFACE_LOGGER(map);
INTERFACE_LOGGER(location);
INTERFACE_LOGGER(route);
INTERFACE_LOGGER(guidance);
INTERFACE_LOGGER(positionsimulation);
INTERFACE_LOGGER(trafficmessage);
INTERFACE_LOGGER(engineering);
INTERFACE_LOGGER(destinationmemory);
INTERFACE_LOGGER(navdataupdate);
INTERFACE_LOGGER(configuration);
INTERFACE_LOGGER(iconcache);
INTERFACE_LOGGER(locationbasedservices);
INTERFACE_LOGGER(electricvehicle);

#define MESSAGE_TAG_IN "-> "
#define MESSAGE_TAG_OUT "<- "

static AIVIPresCtrlInterface presCtrlInterface;

#ifdef BUILD_FOR_COVERAGE
extern "C" void __gcov_flush();
#endif

NavMiddleware::NavMiddleware()
{
   LOG_INTERFACE(inavmiddleware, MESSAGE_TAG_IN"NavMiddleware()");
}

NavMiddleware::~NavMiddleware()
{
   LOG_INTERFACE(inavmiddleware, MESSAGE_TAG_IN"~NavMiddleware()");
}

bool NavMiddleware::initialize(const ConfigurationInfos& configurationInfos)
{
   LOG_INTERFACE(inavmiddleware, MESSAGE_TAG_IN"initialize()");
   LOG_INTERFACE_DEBUG(configuration, "\n ############ Configuration Info ############ - %s",
                       configurationInfos.toString().c_str());

   return presCtrlInterface.initialize(configurationInfos);
}

void NavMiddleware::stop()
{
   LOG_INTERFACE(inavmiddleware, MESSAGE_TAG_IN"stop()");
#ifdef BUILD_FOR_COVERAGE
   //dumping gcov data at runtime
   __gcov_flush();
#endif
   presCtrlInterface.stop();

}

void NavMiddleware::restart(const ConfigurationInfos& configurationInfos)
{
   LOG_INTERFACE(inavmiddleware, MESSAGE_TAG_IN"restart()");
   //currently, restart shall be used to re-initialize rather than stopping and starting. a previous stop-call is assumed to have happened before
   //stop();
   presCtrlInterface.initialize(configurationInfos);
}

void NavMiddleware::restoreFactorySettings(::navmiddleware::settings::RestoreType restoreType,
      ::navmiddleware::settings::UserProfileGroup userProfileGroup)
{
   LOG_INTERFACE(inavmiddleware, MESSAGE_TAG_IN"restoreFactorySettings(%d,%d)", restoreType, userProfileGroup);
   presCtrlInterface.restoreFactorySettings(restoreType, userProfileGroup);
}

void NavMiddleware::applySettings()
{
   LOG_INTERFACE(inavmiddleware, MESSAGE_TAG_IN"applySettings()");

   presCtrlInterface.applySettings();
}

void NavMiddleware::restoreDefaultSettings(::navmiddleware::settings::SettingsGroup settingsGroup)
{
   LOG_INTERFACE(inavmiddleware, MESSAGE_TAG_IN"restoreDefaultSettings(%d)", settingsGroup);
   presCtrlInterface.restoreDefaultSettings(settingsGroup);
}


void NavMiddleware::activateLimitedMode(const ::std::set<RequiredFeatures>& requiredFeatures)
{
   LOG_INTERFACE(inavmiddleware, MESSAGE_TAG_IN"activateLimitedMode()");
   presCtrlInterface.activateLimitedMode(requiredFeatures);
}

void NavMiddleware::deactivateLimitedMode()
{
   LOG_INTERFACE(inavmiddleware, MESSAGE_TAG_IN"deactivateLimitedMode()");
   presCtrlInterface.deactivateLimitedMode();
}

void NavMiddleware::registerLifeCyclePropertyUpdateCallback(PropertyUpdateCallback& callback)
{
   LOG_INTERFACE(inavmiddleware, MESSAGE_TAG_IN"registerLifeCyclePropertyUpdateCallback()");

   presCtrlInterface.registerLifeCyclePropertyUpdateCallback(callback);
}

void NavMiddleware::unregisterLifeCyclePropertyUpdateCallback(PropertyUpdateCallback& callback)
{
   LOG_INTERFACE(inavmiddleware, MESSAGE_TAG_IN"unregisterLifeCyclePropertyUpdateCallback()");

   presCtrlInterface.unregisterLifeCyclePropertyUpdateCallback(callback);
}

void NavMiddleware::registerPositionPropertyUpdateCallback(PropertyUpdateCallback& callback)
{
   LOG_INTERFACE(position, MESSAGE_TAG_IN"registerPositionPropertyUpdateCallback()");

   presCtrlInterface.registerPositionPropertyUpdateCallback(callback);
}

void NavMiddleware::unregisterPositionPropertyUpdateCallback(PropertyUpdateCallback& callback)
{
   LOG_INTERFACE(position, MESSAGE_TAG_IN"unregisterPositionPropertyUpdateCallback()");

   presCtrlInterface.unregisterPositionPropertyUpdateCallback(callback);
}

void NavMiddleware::registerGuidancePropertyUpdateCallback(PropertyUpdateCallback& callback)
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"registerGuidancePropertyUpdateCallback()");

   presCtrlInterface.registerGuidancePropertyUpdateCallback(callback);
}

void NavMiddleware::unregisterGuidancePropertyUpdateCallback(PropertyUpdateCallback& callback)
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"unregisterGuidancePropertyUpdateCallback()");

   presCtrlInterface.unregisterGuidancePropertyUpdateCallback(callback);
}

void NavMiddleware::registerLocationPropertyUpdateCallback(PropertyUpdateCallback& callback)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"registerLocationPropertyUpdateCallback()");

   presCtrlInterface.registerLocationPropertyUpdateCallback(callback);
}

void NavMiddleware::unregisterLocationPropertyUpdateCallback(PropertyUpdateCallback& callback)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"unregisterLocationPropertyUpdateCallback()");

   presCtrlInterface.unregisterLocationPropertyUpdateCallback(callback);
}

void NavMiddleware::registerMapPropertyUpdateCallback(PropertyUpdateCallback& callback)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"registerMapPropertyUpdateCallback()");

   presCtrlInterface.registerMapPropertyUpdateCallback(callback);
}

void NavMiddleware::unregisterMapPropertyUpdateCallback(PropertyUpdateCallback& callback)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"unregisterMapPropertyUpdateCallback()");

   presCtrlInterface.unregisterMapPropertyUpdateCallback(callback);
}

void NavMiddleware::registerRoutePropertyUpdateCallback(PropertyUpdateCallback& callback)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"registerRoutePropertyUpdateCallback()");

   presCtrlInterface.registerRoutePropertyUpdateCallback(callback);
}

void NavMiddleware::unregisterRoutePropertyUpdateCallback(PropertyUpdateCallback& callback)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"unregisterRoutePropertyUpdateCallback()");

   presCtrlInterface.unregisterRoutePropertyUpdateCallback(callback);
}

void NavMiddleware::registerTrafficMessagePropertyUpdateCallback(PropertyUpdateCallback& callback)
{
   LOG_INTERFACE(trafficmessage, MESSAGE_TAG_IN"registerTrafficMessagePropertyUpdateCallback()");

   presCtrlInterface.registerTrafficMessagePropertyUpdateCallback(callback);
}

void NavMiddleware::unregisterTrafficMessagePropertyUpdateCallback(PropertyUpdateCallback& callback)
{
   LOG_INTERFACE(trafficmessage, MESSAGE_TAG_IN"unregisterTrafficMessagePropertyUpdateCallback()");

   presCtrlInterface.unregisterTrafficMessagePropertyUpdateCallback(callback);
}

void NavMiddleware::registerEngineeringMenuPropertyUpdateCallback(PropertyUpdateCallback& callback)
{
   LOG_INTERFACE(engineering, MESSAGE_TAG_IN"registerEngineeringMenuPropertyUpdateCallback()");

   presCtrlInterface.registerEngineeringMenuPropertyUpdateCallback(callback);
}

void NavMiddleware::unregisterEngineeringMenuPropertyUpdateCallback(PropertyUpdateCallback& callback)
{
   LOG_INTERFACE(engineering, MESSAGE_TAG_IN"unregisterEngineeringMenuPropertyUpdateCallback()");

   presCtrlInterface.unregisterEngineeringMenuPropertyUpdateCallback(callback);
}

void NavMiddleware::registerDestinationMemoryPropertyUpdateCallback(PropertyUpdateCallback& callback)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"registerDestinationMemoryPropertyUpdateCallback()");

   presCtrlInterface.registerDestinationMemoryPropertyUpdateCallback(callback);
}

void NavMiddleware::unregisterDestinationMemoryPropertyUpdateCallback(PropertyUpdateCallback& callback)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"unregisterDestinationMemoryPropertyUpdateCallback()");

   presCtrlInterface.unregisterDestinationMemoryPropertyUpdateCallback(callback);
}

void NavMiddleware::registerNavDataUpdatePropertyUpdateCallback(PropertyUpdateCallback& callback)
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"registerNavDataUpdatePropertyUpdateCallback()");

   presCtrlInterface.registerNavDataUpdatePropertyUpdateCallback(callback);
}

void NavMiddleware::unregisterNavDataUpdatePropertyUpdateCallback(PropertyUpdateCallback& callback)
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"unregisterNavDataUpdatePropertyUpdateCallback()");

   presCtrlInterface.unregisterNavDataUpdatePropertyUpdateCallback(callback);
}

void NavMiddleware::registerConfigurationPropertyUpdateCallback(PropertyUpdateCallback& callback)
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"registerConfigurationPropertyUpdateCallback()");
   presCtrlInterface.registerConfigurationPropertyUpdateCallback(callback);
}
void NavMiddleware::unregisterConfigurationPropertyUpdateCallback(PropertyUpdateCallback& callback)
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"unregisterConfigurationPropertyUpdateCallback()");
   presCtrlInterface.unregisterConfigurationPropertyUpdateCallback(callback);
}

void NavMiddleware::registerIconCacheControlPropertyUpdateCallback(PropertyUpdateCallback& callback)
{
   LOG_INTERFACE(iconcache, MESSAGE_TAG_IN"registerIconCacheControlPropertyUpdateCallback()");
   presCtrlInterface.registerIconCacheControlPropertyUpdateCallback(callback);
}
void NavMiddleware::unregisterIconCacheControlPropertyUpdateCallback(PropertyUpdateCallback& callback)
{
   LOG_INTERFACE(iconcache, MESSAGE_TAG_IN"unregisterIconCacheControlPropertyUpdateCallback()");
   presCtrlInterface.unregisterIconCacheControlPropertyUpdateCallback(callback);
}

void NavMiddleware::registerLocationBasedServicesPropertyUpdateCallback(PropertyUpdateCallback& callback)
{
   LOG_INTERFACE(locationbasedservices, MESSAGE_TAG_IN"registerLocationBasedServicesPropertyUpdateCallback()");
   presCtrlInterface.registerLocationBasedServicesPropertyUpdateCallback(callback);
}
void NavMiddleware::unregisterLocationBasedServicesPropertyUpdateCallback(PropertyUpdateCallback& callback)
{
   LOG_INTERFACE(locationbasedservices, MESSAGE_TAG_IN"unregisterLocationBasedServicesPropertyUpdateCallback()");
   presCtrlInterface.unregisterLocationBasedServicesPropertyUpdateCallback(callback);
}

void NavMiddleware::registerElectricVehiclePropertyUpdateCallback(PropertyUpdateCallback& callback)
{
   LOG_INTERFACE(electricvehicle, MESSAGE_TAG_IN"registerElectricVehiclePropertyUpdateCallback()");
   presCtrlInterface.registerElectricVehiclePropertyUpdateCallback(callback);
}

void NavMiddleware::unregisterElectricVehiclePropertyUpdateCallback(PropertyUpdateCallback& callback)
{
   LOG_INTERFACE(electricvehicle, MESSAGE_TAG_IN"unregisterElectricVehiclePropertyUpdateCallback()");
   presCtrlInterface.unregisterElectricVehiclePropertyUpdateCallback(callback);
}

const WeatherFeaturesStatusInfo& NavMiddleware::getWeatherFeaturesStatusInfo() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"getWeatherFeaturesStatusInfo()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Weather Map Features Status info ############ - %s",
                       presCtrlInterface.getWeatherFeaturesStatusInfo().toString().c_str());

   return presCtrlInterface.getWeatherFeaturesStatusInfo();
}

//======================== Configuration
const ConfigurationInfos& NavMiddleware::getConfigurationInfos()
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"geConfigurationInfos()");
   LOG_INTERFACE_DEBUG(configuration, "\n ############ Configuration Info ############ - %s",
                       presCtrlInterface.getConfigurationInfos().toString().c_str());

   return presCtrlInterface.getConfigurationInfos();
}

void NavMiddleware::setLanguage(const ::std::string& languageCodeHMIAsString, LanguageCode languageCodeHMI,
                                const ::std::string& languageCodeSDSAsString, LanguageCode languageCodeSDS,
                                const ::std::string& languageCodeTrafficTextAsString, LanguageCode languageCodeTrafficText,
                                const ::std::string& languageCodeClusterAsString, LanguageCode languageCodeCluster)
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"setLanguage()");
   LOG_INTERFACE_DEBUG(configuration, "setLanguage as string (HMI: %s, SDS: %s,TrafficText: %s,Cluster: %s)",
                       languageCodeHMIAsString.c_str(),
                       languageCodeSDSAsString.c_str(),
                       languageCodeTrafficTextAsString.c_str(),
                       languageCodeClusterAsString.c_str()
                      );
   LOG_INTERFACE_DEBUG(configuration, "setLanguage (HMI: %d, SDS: %d, TrafficText: %d,Cluster: %d)", languageCodeHMI, languageCodeSDS, languageCodeTrafficText, languageCodeCluster);
   presCtrlInterface.setLanguage(languageCodeHMIAsString,languageCodeHMI,
                                 languageCodeSDSAsString,languageCodeSDS,
                                 languageCodeTrafficTextAsString, languageCodeTrafficText,
                                 languageCodeClusterAsString,languageCodeCluster);
}

void NavMiddleware::setDisplayTimeOffset(
   int32_t displayTimeOffset,
   ::navmiddleware::ConfigurationInfos::TimeZoneAndDSTConsideration timeZoneAndDSTConsideration)
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"setDisplayTimeOffset(displayTimeOffset : %u,"
                 " timeZoneAndDSTConsideration : %d )", displayTimeOffset, timeZoneAndDSTConsideration);
   presCtrlInterface.setDisplayTimeOffset(displayTimeOffset, timeZoneAndDSTConsideration);
}

void NavMiddleware::setDistanceUnitSystem(DistanceUnitSystem distanceUnitSystem)
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"setDistanceUnitSystem(%u)", distanceUnitSystem);
   presCtrlInterface.setDistanceUnitSystem(distanceUnitSystem);
}

void NavMiddleware::setDateFormat(DateFormat dateFormat)
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"setDateFormat(%u)", dateFormat);
   presCtrlInterface.setDateFormat(dateFormat);
}

void NavMiddleware::setTimeFormat(TimeFormat timeFormat)
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"setTimeFormat(%u)", timeFormat);
   presCtrlInterface.setTimeFormat(timeFormat);
}

void NavMiddleware::setDefaultPosition(const GeoCoordinateDegree& coordinate, float defaultPositionHeading)
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"setDefaultPosition(lat: %f, lon: %f, heading %f)", coordinate.getLatitude(), coordinate.getLongitude(), defaultPositionHeading);
   presCtrlInterface.setDefaultPosition(coordinate, defaultPositionHeading);
}

void NavMiddleware::configureLightMode(ConfigurationInfos::LightMode lightMode)
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"configureLightMode(lightMode = %d)", lightMode);

   presCtrlInterface.configureLightMode(lightMode);
}

void NavMiddleware::setSystemPowerMode(ConfigurationInfos::SystemPowerMode systemPowerMode)
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"setSystemPowerMode((SystemPowerMode = %d)", systemPowerMode);
   presCtrlInterface.setSystemPowerMode(systemPowerMode);
}

void  NavMiddleware::setTCUStatus(ConfigurationInfos::TCUStatus tcuStatus)
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"setTCUtoAvailable(TCUStatus = %d)", tcuStatus );
   presCtrlInterface.setTCUStatus(tcuStatus);
}

void NavMiddleware::setUserProfile(ConfigurationInfos::UserProfile userProfile)
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"setUserProfile(userProfile = %d)", userProfile);
   presCtrlInterface.setUserProfile(userProfile);
}

void NavMiddleware::copyUserProfileToUser(ConfigurationInfos::UserProfile targetUserProfile)
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"copyUserProfileToUser(targetUserProfile = %d)", targetUserProfile);
   presCtrlInterface.copyUserProfileToUser(targetUserProfile);
}

void NavMiddleware::deleteUserProfile(ConfigurationInfos::UserProfile userProfile)
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"deleteUserProfile(userProfile = %d)", userProfile);
   presCtrlInterface.deleteUserProfile(userProfile);
}

const UserProfileActionInfos& NavMiddleware::getUserProfileActionInfos() const
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"UserProfileActionInfos()");
   LOG_INTERFACE_DEBUG(configuration, "\n ############ Get UserProfileActionInfos ############ - ");
   return presCtrlInterface.getUserProfileActionInfos();
}

void NavMiddleware::setSystemState(ConfigurationInfos::SystemState systemState)
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"setSystemState(systemState = %d)", systemState);
   presCtrlInterface.setSystemState(systemState);
}

void NavMiddleware::addAvoidArea(const ::std::string& avoidAreaName, const GeoCoordinateDegree& centerGeoCoordinate,
                                 AvoidAreaInfo::AvoidAreaSize avoidAreaSize, uint32_t avoidAreaMapScale, uint32_t edgeLength,
                                 AvoidAreaInfo::AvoidAreaOptions avoidAreaOptions )
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"addAvoidArea()");
   LOG_INTERFACE_DEBUG(configuration, "addAvoidArea (Name: %s, CenterCoordinate: %s, Size: %d, Scale: %u, EdgeLength: %u, Options: %d)"
                       ,avoidAreaName.c_str(),
                       centerGeoCoordinate.toString().c_str(),
                       avoidAreaSize,
                       avoidAreaMapScale,
                       edgeLength,
                       avoidAreaOptions);
   presCtrlInterface.addAvoidArea(avoidAreaName,
                                  centerGeoCoordinate,
                                  avoidAreaSize,
                                  avoidAreaMapScale,
                                  edgeLength,
                                  avoidAreaOptions );
}

void NavMiddleware::addAvoidArea(const ::std::string& avoidAreaName,
                                 const ::std::vector< ::navmiddleware::ScreenCoordinate >& screenCoordinates,
                                 AvoidAreaInfo::AvoidAreaOptions avoidAreaOptions)
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"addAvoidArea()");
   LOG_INTERFACE_DEBUG(configuration, "addAvoidArea (Name: %s, screenCoordinatesSize: %d, Options: %d)",
                       avoidAreaName.c_str(),
                       screenCoordinates.size(),
                       avoidAreaOptions);
   presCtrlInterface.addAvoidArea(avoidAreaName,screenCoordinates,avoidAreaOptions);
}

void NavMiddleware::modifyAvoidArea(const AvoidAreaInfo& avoidAreaInfo)
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"modifyAvoidArea()");
   LOG_INTERFACE_DEBUG(configuration, "modifyAvoidArea (AvoidAreaInfo: %s)",avoidAreaInfo.toString().c_str());
   presCtrlInterface.modifyAvoidArea(avoidAreaInfo);
}

void NavMiddleware::modifyAvoidArea(const AvoidAreaInfo& avoidAreaInfo,
                                    const ::std::vector< ScreenCoordinate >& screenCoordinates)
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"modifyAvoidArea()");
   LOG_INTERFACE_DEBUG(configuration, "modifyAvoidArea (AvoidArea: %s, screenCoordinatesSize: %d)",
                       avoidAreaInfo.toString().c_str(),
                       screenCoordinates.size());
   presCtrlInterface.modifyAvoidArea(avoidAreaInfo, screenCoordinates);
}

void NavMiddleware::setAvoidAreaListFilter(const ::std::string& nameFilter)
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"setAvoidAreaListFilter(%s)", nameFilter.c_str());
   presCtrlInterface.setAvoidAreaListFilter(nameFilter);
}

const AvoidAreaInfos& NavMiddleware::getAvoidAreaInfos()const
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"getAvoidAreaInfos()");
   return presCtrlInterface.getAvoidAreaInfos();
}

void NavMiddleware::deleteAvoidArea( AvoidAreaId avoidAreaId)
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"deleteAvoidArea(Id: %llu)",avoidAreaId);
   presCtrlInterface.deleteAvoidArea(avoidAreaId);
}

void  NavMiddleware::deleteAllAvoidAreas()
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"deleteAllAvoidAreas()");
   presCtrlInterface.deleteAllAvoidAreas();
}

void NavMiddleware::addDynamicAreaUserStyle(const ::std::vector<DynamicAreaUserStyle>& dynamicAreaUserStyleList)
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"addDynamicAreaUserStyle()");
   presCtrlInterface.addDynamicAreaUserStyle(dynamicAreaUserStyleList);
}

AddDynamicAreaUserStyleResult NavMiddleware::getAddDynamicAreaUserStyleResult() const
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"getAddDynamicAreaUserStyleResult()");
   LOG_INTERFACE_DEBUG(configuration, "\n ############ AddDynamicAreaUserStyleResult ############ - %s",
                       toString(presCtrlInterface.getAddDynamicAreaUserStyleResult()).c_str());
   return presCtrlInterface.getAddDynamicAreaUserStyleResult();
}

void NavMiddleware::deleteAllDynamicAreaUserStyle()
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"deleteAllDynamicAreaUserStyle()");
   presCtrlInterface.deleteAllDynamicAreaUserStyle();
}

const RegisteredConfigItemsList& NavMiddleware::getConfigItemsToBeRegistered() const
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"getConfigItemsToBeRegistered()");
   LOG_INTERFACE_DEBUG(configuration, "\n ############ getConfigItemsToBeRegistered ############ - %s",
                       presCtrlInterface.getConfigItemsToBeRegistered().toString().c_str());
   return presCtrlInterface.getConfigItemsToBeRegistered();
}

const ConfigItemsGetResult&  NavMiddleware::getConfigItems(const ::std::string& configItemGroupName) const
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"getConfigItems(%s)",configItemGroupName.c_str());
   LOG_INTERFACE_DEBUG(configuration, "\n ############ getConfigItems ############ - %s",
                       presCtrlInterface.getConfigItems(configItemGroupName).toString().c_str());
   return presCtrlInterface.getConfigItems(configItemGroupName);
}

void  NavMiddleware::setConfigItems(const ::std::string& configItemGroupName, const ::std::vector< ConfigItem>& configItems)
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"setConfigItems()");
   LOG_INTERFACE_DEBUG(configuration, "\n ############ setConfigItems ############ - %s,%s",
                       configItemGroupName.c_str(),toString< ConfigItem>(configItems).c_str());
   presCtrlInterface.setConfigItems(configItemGroupName,configItems);
}

ConfigResultCode NavMiddleware::getSetConfigResult() const
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"getSetConfigResult()");
   LOG_INTERFACE_DEBUG(configuration, "\n ############ getSetConfigResult ############ - %s",
                       toString(presCtrlInterface.getSetConfigResult()).c_str());
   return presCtrlInterface.getSetConfigResult();
}
void NavMiddleware::requestNavigationVersion()
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"requestNavigationVersion()");
   presCtrlInterface.requestNavigationVersion();
}

const ::std::string& NavMiddleware::getNavigationVersion() const
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"getNavigationVersion()");
   LOG_INTERFACE_DEBUG(configuration, "\n ############ getNavigationVersion ############ - %s",
                       presCtrlInterface.getNavigationVersion().c_str());
   return(presCtrlInterface.getNavigationVersion());
}

void NavMiddleware::setPrivacyMode(ConfigurationInfos::PrivacyMode privacyMode)
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"setPrivacyMode()");
   LOG_INTERFACE_DEBUG(configuration, "\n ############ setPrivacyMode ############ - %d",
                       privacyMode);

   presCtrlInterface.setPrivacyMode(privacyMode);
}

//============================= POSITION =====================================
//============================================================================
const PositionInfo& NavMiddleware::getPositionInfo() const
{
   LOG_INTERFACE(position, MESSAGE_TAG_IN"getPositionInfo()");
   LOG_INTERFACE_DEBUG(position, "\n ############ Position Info ############ - %s",
                       presCtrlInterface.getPositionInfo().toString().c_str());

   return presCtrlInterface.getPositionInfo();
}

const SdsPositionInfo& NavMiddleware::getSdsPositionInfo() const
{
   LOG_INTERFACE(position, MESSAGE_TAG_IN"getSdsPositionInfo()");
   LOG_INTERFACE_DEBUG(position, "\n ############ SDS Position Info ############ - %s",
                       presCtrlInterface.getSdsPositionInfo().toString().c_str());
   return presCtrlInterface.getSdsPositionInfo();
}

const PositionStatusInfo& NavMiddleware::getPositionStatusInfo() const
{
   LOG_INTERFACE(position, MESSAGE_TAG_IN"getPositionStatus()");
   LOG_INTERFACE_DEBUG(position, "\n ############ Position Status Info ############ - %s",
                       presCtrlInterface.getPositionStatusInfo().toString().c_str());

   return presCtrlInterface.getPositionStatusInfo();
}

PositionQuality NavMiddleware::getPositionQuality() const
{
   LOG_INTERFACE(position, MESSAGE_TAG_IN"getPositionQuality()");
   LOG_INTERFACE_DEBUG(position, "\n ############ Position Quality Info ############ - %s",
                       ::navmiddleware::toString(presCtrlInterface.getPositionQuality()).c_str());

   return presCtrlInterface.getPositionQuality();
}

RnwDataAvailability NavMiddleware::getRnwDataAvailabilityAtCurrentPosition() const
{
   LOG_INTERFACE(position, MESSAGE_TAG_IN"getRnwDataAvailabilityAtCurrentPosition()");
   LOG_INTERFACE_DEBUG(position, "\n ############ Rnw data Availability Info ############ - %s",
                       ::navmiddleware::toString(presCtrlInterface.getRnwDataAvailabilityAtCurrentPosition()).c_str());

   return presCtrlInterface.getRnwDataAvailabilityAtCurrentPosition();
}

VehicleMovingStatus NavMiddleware::getVehicleMovingState() const
{
   LOG_INTERFACE(position, MESSAGE_TAG_IN"getVehicleMovingState()");
   LOG_INTERFACE_DEBUG(position, "\n ############ Vehicle Moving State Info ############ - %s",
                       ::navmiddleware::toString(presCtrlInterface.getVehicleMovingState()).c_str());

   return presCtrlInterface.getVehicleMovingState();
}

void NavMiddleware::requestPositionDebugInfo()
{
   LOG_INTERFACE(position, MESSAGE_TAG_IN"requestPositionDebugInfo()");
   presCtrlInterface.requestPositionDebugInfo();
}

void NavMiddleware::releasePositionDebugInfo()
{
   LOG_INTERFACE(position, MESSAGE_TAG_IN"releasePositionDebugInfo()");
   presCtrlInterface.releasePositionDebugInfo();
}

const PositionDebugInfo& NavMiddleware::getPositionDebugInfo() const
{
   LOG_INTERFACE(position, MESSAGE_TAG_IN"getPositionDebugInfo()");
   LOG_INTERFACE_DEBUG(position, "\n ############ Position debug Info ############ - %s",
                       presCtrlInterface.getPositionDebugInfo().toString().c_str());
   return presCtrlInterface.getPositionDebugInfo();
}

void NavMiddleware::setPositionByCoordinates(const GeoCoordinateDegree& geoCoordinate, float heading)
{
   LOG_INTERFACE(position, MESSAGE_TAG_IN"setPositionByCoordinates(lat=%f, long=%f, head=%f)",
                 geoCoordinate.getLatitude(),
                 geoCoordinate.getLongitude(),
                 heading);

   presCtrlInterface.setPositionByCoordinates(geoCoordinate, heading);
}

void NavMiddleware::enableWhereAmIInfo() const
{
   LOG_INTERFACE(position, MESSAGE_TAG_IN"enableWhereAmIInfo()");

   presCtrlInterface.enableWhereAmIInfo();
}

void NavMiddleware::disableWhereAmIInfo() const
{
   LOG_INTERFACE(position, MESSAGE_TAG_IN"disableWhereAmIInfo()");

   presCtrlInterface.disableWhereAmIInfo();
}

const WhereAmIInfo& NavMiddleware::getWhereAmIInfo() const
{
   LOG_INTERFACE(position, MESSAGE_TAG_IN"getWhereAmIInfo()");
   LOG_INTERFACE_DEBUG(position, "\n ############ WhereAmI Info ############ - %s",
                       presCtrlInterface.getWhereAmIInfo().toString().c_str());

   return presCtrlInterface.getWhereAmIInfo();
}

const GnssDataInfo& NavMiddleware::getGnssDataInfo() const
{
   LOG_INTERFACE(position, MESSAGE_TAG_IN"getGnssDataInfo()");
   LOG_INTERFACE_DEBUG(position, "\n ############ Gnss Data Info ############ - %s",
                       presCtrlInterface.getGnssDataInfo().toString().c_str());

   return presCtrlInterface.getGnssDataInfo();
}

void NavMiddleware::startTripReplay(const ::std::string& tripFileName)
{
   LOG_INTERFACE(position, MESSAGE_TAG_IN"startTripReplay(%s)", tripFileName.c_str());
   presCtrlInterface.startTripReplay(tripFileName);
}

void NavMiddleware::stopTripReplay()
{
   LOG_INTERFACE(position, MESSAGE_TAG_IN"stopTripReplay");
   presCtrlInterface.stopTripReplay();
}

void NavMiddleware::seekTripReplay(int16_t seekValue)
{
   LOG_INTERFACE(position, MESSAGE_TAG_IN"seekTripReplay(%d)", seekValue);
   presCtrlInterface.seekTripReplay(seekValue);
}

void NavMiddleware::setTripReplayMode(bool loopModeActive, uint16_t speed)
{
   LOG_INTERFACE(position, MESSAGE_TAG_IN"setTripReplayMode(loopModeActive=%s, speed=%d)",
                 (loopModeActive ? "true" : "false"), speed);
   presCtrlInterface.setTripReplayMode(loopModeActive, speed);
}

void NavMiddleware::writeTripMarker()
{
   LOG_INTERFACE(position, MESSAGE_TAG_IN"WriteTripMarker");
   presCtrlInterface.writeTripMarker();
}

const TripReplayStatusInfo& NavMiddleware::getTripReplayStatusInfo() const
{
   LOG_INTERFACE(position, MESSAGE_TAG_IN"getTripReplayStatusInfo");
   LOG_INTERFACE_DEBUG(position, "\n ############ Get Trip-Replay-Status info ############ - %s",
                       presCtrlInterface.getTripReplayStatusInfo().toString().c_str());

   return presCtrlInterface.getTripReplayStatusInfo();
}

const TripRecordingStatusInfo& NavMiddleware::getTripRecordingStatusInfo() const
{
   LOG_INTERFACE(position, MESSAGE_TAG_IN"getTripRecordingStatusInfo");
   LOG_INTERFACE_DEBUG(position, "\n ############ Get Trip-Recording-Status info ############ - %s",
                       presCtrlInterface.getTripRecordingStatusInfo().toString().c_str());

   return presCtrlInterface.getTripRecordingStatusInfo();
}

HomePositionStatus NavMiddleware::getHomeStatus() const
{
   LOG_INTERFACE(position, MESSAGE_TAG_IN"getHomeStatus(): %s",
                 ::navmiddleware::toString(presCtrlInterface.getHomeStatus()).c_str());

   return presCtrlInterface.getHomeStatus();
}

//======================== POSITION SIMULATION ===============================
//============================================================================
void NavMiddleware::activateDemoMode()
{
   LOG_INTERFACE(positionsimulation, MESSAGE_TAG_IN"activateDemoMode()");

   presCtrlInterface.activateDemoMode();
}

void NavMiddleware::deactivateDemoMode()
{
   LOG_INTERFACE(positionsimulation, MESSAGE_TAG_IN"deactivateDemoMode()");

   presCtrlInterface.deactivateDemoMode();
}

bool NavMiddleware::isDemoModeActive() const
{
   LOG_INTERFACE(positionsimulation, MESSAGE_TAG_IN"isDemoModeActive()");

   return presCtrlInterface.isDemoModeActive();
}

void NavMiddleware::setDemoModeSpeed(unsigned int speedInPercent)
{
   LOG_INTERFACE(positionsimulation, MESSAGE_TAG_IN"setDemoModeSpeed(%u%%)", speedInPercent);

   presCtrlInterface.setDemoModeSpeed(speedInPercent);
}

void NavMiddleware::setDemoModePosition(double latitude, double longitude, float heading)
{
   LOG_INTERFACE(positionsimulation, MESSAGE_TAG_IN"setDemoModePosition(lat=%f, long=%f, head=%f)", latitude, longitude, heading);

   presCtrlInterface.setDemoModePosition(latitude,longitude,heading);
}

void NavMiddleware::setPositionSimulationMode(::navmiddleware::PositionSimulationMode simulationMode)
{
   LOG_INTERFACE(positionsimulation, MESSAGE_TAG_IN"setPositionSimulationMode()");

   presCtrlInterface.setPositionSimulationMode(simulationMode);
}

bool NavMiddleware::isSimulationActive() const
{
   LOG_INTERFACE(positionsimulation, MESSAGE_TAG_IN"isSimulationActive()");

   return presCtrlInterface.isSimulationActive();
}

//============================================================================
const ArrivalInfos& NavMiddleware::getArrivalInfos() const
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"getArrivalInfos()");
   LOG_INTERFACE_DEBUG(route, "\n ############ Arrival Info ############ - %s",
                       presCtrlInterface.getArrivalInfos().toString().c_str());

   return presCtrlInterface.getArrivalInfos();
}

const DestinationInfos& NavMiddleware::getDestinationInfos() const
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"getDestinationInfos()");
   LOG_INTERFACE_DEBUG(route, "\n ############ Destination Info ############ - %s",
                       presCtrlInterface.getDestinationInfos().toString().c_str());

   return presCtrlInterface.getDestinationInfos();
}

const ArrivalInfos& NavMiddleware::getArrivalInfosGuidedRoute() const
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"getArrivalInfosGuidedRoute()");
   LOG_INTERFACE_DEBUG(route, "\n ############ Arrival Info for guided route ############ - %s",
                       presCtrlInterface.getArrivalInfosGuidedRoute().toString().c_str());

   return presCtrlInterface.getArrivalInfosGuidedRoute();
}

const DestinationInfos& NavMiddleware::getDestinationInfosGuidedRoute() const
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"getDestinationInfosGuidedRoute()");
   LOG_INTERFACE_DEBUG(route, "\n ############ Destination Info  for guided route ############ - %s",
                       presCtrlInterface.getDestinationInfosGuidedRoute().toString().c_str());

   return presCtrlInterface.getDestinationInfosGuidedRoute();
}

const ArrivalInfosAlternatives& NavMiddleware::getArrivalInfosAlternatives() const
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"getArrivalInfosAlternatives()");
   LOG_INTERFACE_DEBUG(route, "\n ############ Arrival Info alternatives ############ - %s",
                       presCtrlInterface.getArrivalInfosAlternatives().toString().c_str());

   return presCtrlInterface.getArrivalInfosAlternatives();
}

const TrafficImpactRouteAlternativeInfos& NavMiddleware::getTrafficImpactRouteAlternativeInfos() const
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"getTrafficImpactRouteAlternativeInfos()");
   LOG_INTERFACE_DEBUG(route, "\n ############ Traffic impact route alternatives ############ - %s",
                       presCtrlInterface.getTrafficImpactRouteAlternativeInfos().toString().c_str());

   return presCtrlInterface.getTrafficImpactRouteAlternativeInfos();
}

const RouteCalculationDurationInfos& NavMiddleware::getRouteCalcDurationInfos() const
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"getRouteCalcDurationInfos()");
   LOG_INTERFACE_DEBUG(route, "\n ############ Route Calculation Duration Infos ############ - %s",
                       presCtrlInterface.getRouteCalcDurationInfos().toString().c_str());
   return presCtrlInterface.getRouteCalcDurationInfos();
}

//======================== LOCATION INPUT ====================================
//============================================================================

void NavMiddleware::setLocationWithPickingResult(const PickingResults::PickingResult& pickingResult)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"setLocationWithPickingResult()");

   presCtrlInterface.setLocationWithPickingResult(pickingResult);
}

void NavMiddleware::setLocationWithPoiData(const PoiData& poiData)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"setLocationWithPoiData(poiData = %s)", poiData.toString().c_str());

   presCtrlInterface.setLocationWithPoiData(poiData);
}

void NavMiddleware::setLocationWithCoordinates(double latitude, double longitude)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"setLocationWithCoordinates(lat=%f, long=%f)", latitude, longitude);

   presCtrlInterface.setLocationWithCoordinates(latitude, longitude);
}

void NavMiddleware::setLocationWithExternalLocation(const ExternalLocation& externalLocation)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"setLocationWithExternalLocation()");
   LOG_INTERFACE_DEBUG(location, "\n ############ setLocationWithExternalLocation ############ - %s",
                       externalLocation.toString().c_str());
   presCtrlInterface.setLocationWithExternalLocation(externalLocation);
}

void NavMiddleware::setLocationWithDestinationMemoryId(const DestinationMemoryEntry::Id& destinationMemoryEntryId)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"setLocationWithDestinationMemoryId(destinationMemoryEntryId=%llu)", destinationMemoryEntryId);
   presCtrlInterface.setLocationWithDestinationMemoryId(destinationMemoryEntryId);
}

void NavMiddleware::setLocationWithWaypointIndex(unsigned int waypointIndex)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"setLocationWithWaypointIndex(waypointIndex=%u)", waypointIndex);
   presCtrlInterface.setLocationWithWaypointIndex(waypointIndex);
}

RequestId NavMiddleware::requestLocationAttributes()
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"requestLocationAttributes()");
   RequestId requestId = presCtrlInterface.requestLocationAttributes();
   LOG_INTERFACE_DEBUG(location, "\n ############ request Location Attributes ############ - %u", requestId);
   return requestId;
}

const LocationAttributeInfos& NavMiddleware::getLocationAttributes() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getLocationAttributes()");
   LOG_INTERFACE_DEBUG(location, "\n ############ Get Location Attributes ############ - %s",
                       presCtrlInterface.getLocationAttributes().toString().c_str());

   return presCtrlInterface.getLocationAttributes();
}

const LocationAttributeErrorInfo& NavMiddleware::getAddressAttributesOfCoordinateError() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getAddressAttributesOfCoordinateError()");
   const LocationAttributeErrorInfo& locationError = presCtrlInterface.getAddressAttributesOfCoordinateError();
   LOG_INTERFACE(location, "   returns %s", toString(locationError.getLocationAttributesError()).c_str());
   return locationError;
}

const LocationAttributeErrorInfo& NavMiddleware::getRequestLocationAttributesError() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getRequestLocationAttributesError()");
   LOG_INTERFACE(location, "\n ############ Get Location Attributes Error ############ - %s",
                 presCtrlInterface.getRequestLocationAttributesError().toString().c_str());

   return presCtrlInterface.getRequestLocationAttributesError();
}

RequestId NavMiddleware::requestAddressAttributesForCoordinate(const GeoCoordinateDegree& geoCoordinate, bool isOnBoardMatchingRequired)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"requestAddressAttributesForCoordinate(lat=%f, long=%f,isOnboardMatchingRequired=%d)",
                 geoCoordinate.getLatitude(), geoCoordinate.getLongitude(), isOnBoardMatchingRequired);
   return presCtrlInterface.requestResolvedCoordinate(geoCoordinate, isOnBoardMatchingRequired);
}

const LocationAttributeInfos& NavMiddleware::getAddressAttributesOfCoordinate() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getAddressAttributesOfCoordinate()");
   LOG_INTERFACE_DEBUG(location, "\n ############ Get Address Details of Coordinate ############ - %s",
                       presCtrlInterface.getResolvedCoordinateAttributeInfo().toString().c_str());
   return presCtrlInterface.getResolvedCoordinateAttributeInfo();
}

OnlineServiceAvailability NavMiddleware::getLocationOnlineSearchAvailability() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getLocationOnlineSearchAvailability()");
   LOG_INTERFACE_DEBUG(location, "\n ############ Get Request Online Search Availability ############ - %s",
                       toString(presCtrlInterface.getOnlineFreeTextSearchAvailability().getDefaultOnlineSearchAvailability()).c_str());
   return presCtrlInterface.getOnlineFreeTextSearchAvailability().getDefaultOnlineSearchAvailability();
}

void NavMiddleware::setAllowPositionBasedDistanceAndDirectionUpdates(bool allow)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"setAllowPositionBasedDistanceAndDirectionUpdates(%s)", (allow?"true":"false"));

   presCtrlInterface.setAllowPositionBasedDistanceAndDirectionUpdates(allow);
}

void NavMiddleware::requestAddressFormatForAvailableCountries()
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"requestAddressFormatForAvailableCountries");
   presCtrlInterface.requestAddressFormatForAvailableCountries();
}

const AddressFormatInfos& NavMiddleware::getAddressFormatInfos() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getAddressFormatInfos");
   LOG_INTERFACE_DEBUG(location, "\n ############ Get Address Format ############ - %s",
                       presCtrlInterface.getAddressFormatInfos().toString().c_str());
   return presCtrlInterface.getAddressFormatInfos();
}

void NavMiddleware::requestAddressesNearPosition(const GeoCoordinateDegree& geoCoordinate, unsigned int radiusInMeters, unsigned int numberOfLocations)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"requestAddressesNearPosition");
   LOG_INTERFACE_DEBUG(location, "requestAddressesNearPosition(geoCoordinate=%s, radiusInMeters=%u, numberOfLocations=%u)", geoCoordinate.toString().c_str(), radiusInMeters, numberOfLocations);
   presCtrlInterface.requestAddressesNearPosition(geoCoordinate,radiusInMeters,numberOfLocations);
}

const LocationInfos& NavMiddleware::getAddressesNearPositionInfos() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getAddressesNearPositionInfos");
   LOG_INTERFACE_DEBUG(location, "\n ############ Get Address Format ############ - %s",
                       presCtrlInterface.getAddressesNearPositionInfos().toString().c_str());

   return presCtrlInterface.getAddressesNearPositionInfos();
}

void NavMiddleware::requestLocationImage(const ::navmiddleware::LocationAttributeInfos::PictureReferenceInfo& pictureRefInfo,
      unsigned int requiredHeightInPixel, unsigned int requiredWidthInPixel)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"requestLocationImage(pictureRefInfo = %s requiredHeightInPixel = %d requiredWidthInPixel = %d)",
                 pictureRefInfo.toString().c_str(), requiredHeightInPixel, requiredWidthInPixel);

   return presCtrlInterface.requestLocationImage(pictureRefInfo, requiredHeightInPixel, requiredWidthInPixel);
}

const ::navmiddleware::Image& NavMiddleware::getLocationImageInfo() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getLocationImageInfo()");
   LOG_INTERFACE_DEBUG(location, "\n ############ Image info ############ - %s",
                       presCtrlInterface.getLocationImageInfo().toString().c_str());

   return presCtrlInterface.getLocationImageInfo();
}

settings::EvChargingStationFilterSettings NavMiddleware::getEvChargingStationFilterSettings() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getEvChargingStationFilterSettings()");
   LOG_INTERFACE_DEBUG(location, "\n ############ Get EV Charging Station Filter Settings ############ \n %s",
                       presCtrlInterface.getEvChargingStationFilterSettings().toString().c_str());

   return presCtrlInterface.getEvChargingStationFilterSettings();
}

void NavMiddleware::updateEvChargingStationFilterSettings(const settings::EvChargingStationFilterSettings& settings)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"updateEvChargingStationFilterSettings()");
   LOG_INTERFACE_DEBUG(location, "\n ############ Update EV Charging Station Filter Settings ############ - %s",
                       settings.toString().c_str());
   presCtrlInterface.updateEvChargingStationFilterSettings(settings);

}

settings::EvConnectorTypeFilterSettings NavMiddleware::getEvConnectorTypeFilterSettings() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getEvConnectorTypeFilterSettings()");
   LOG_INTERFACE_DEBUG(location, "\n ############ Get EV connector type filter Settings ############ \n %s",
                       presCtrlInterface.getEvConnectorTypeFilterSettings().toString().c_str());

   return presCtrlInterface.getEvConnectorTypeFilterSettings();
}

void NavMiddleware::updateEvConnectorTypeFilterSettings(const settings::EvConnectorTypeFilterSettings& settings)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"updateEvConnectorTypeFilterSettings()");
   LOG_INTERFACE_DEBUG(location, "\n ############ Update EV connector type filter Settings ############ - %s",
                       settings.toString().c_str());
   presCtrlInterface.updateEvConnectorTypeFilterSettings(settings);
}

settings::PaymentMethodSettings NavMiddleware::getEvChargingStationPaymentMethodSettings() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getEvChargingStationPaymentMethodSettings()");
   LOG_INTERFACE_DEBUG(location, "\n ############ Get payment method settings for EV charging stations ############ \n %s",
                       presCtrlInterface.getEvChargingStationPaymentMethodSettings().toString().c_str());

   return presCtrlInterface.getEvChargingStationPaymentMethodSettings();
}

void NavMiddleware::updateEvChargingStationPaymentMethodSettings(const settings::PaymentMethodSettings& settings)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"updateEvChargingStationPaymentMethodSettings()");
   LOG_INTERFACE_DEBUG(location, "\n ############ Update payment method settings for EV charging stations ############ - %s",
                       settings.toString().c_str());
   presCtrlInterface.updateEvChargingStationPaymentMethodSettings(settings);

}

settings::EvChargingSpeedFilterSettings NavMiddleware::getEvChargingSpeedFilterSettings() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getEvChargingSpeedFilterSettings()");
   LOG_INTERFACE_DEBUG(location, "\n ############ Get evChargingSpeed filter settings for EV charging stations ############ \n %s",
                       presCtrlInterface.getEvChargingSpeedFilterSettings().toString().c_str());

   return presCtrlInterface.getEvChargingSpeedFilterSettings();
}

void NavMiddleware::updateEvChargingSpeedFilterSettings(const settings::EvChargingSpeedFilterSettings& settings)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"updateEvChargingSpeedFilterSettings()");
   LOG_INTERFACE_DEBUG(location, "\n ############ Update evChargingSpeed filter settings for EV charging stations ############ - %s",
                       settings.toString().c_str());
   presCtrlInterface.updateEvChargingSpeedFilterSettings(settings);

}

settings::ParkingSettings NavMiddleware::getParkingSettings() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getParkingSettings()");
   LOG_INTERFACE_DEBUG(location, "\n ############ Get Parking Settings ############ \n %s",
                       presCtrlInterface.getParkingSettings().toString().c_str());

   return presCtrlInterface.getParkingSettings();
}

void NavMiddleware::updateParkingSettings(const settings::ParkingSettings& settings)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"updateParkingSettings()");
   LOG_INTERFACE_DEBUG(location, "\n ############ Update Parking Settings ############ - %s",
                       settings.toString().c_str());
   presCtrlInterface.updateParkingSettings(settings);

}

void NavMiddleware::startFormBasedAddressInput(const LocationContext& context, bool resetContents)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"startFormBasedAddressInput(context=%s, resetContents=%s)",
                 context.toString().c_str(), (resetContents?"true":"false"));

   presCtrlInterface.startFormBasedAddressInput(context, resetContents);
}

void NavMiddleware::startAddressInputClone(const LocationContext& context, unsigned int instanceNumberToBeCloned)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"startAddressInputClone(context=%s, instanceNumberToBeCloned=%u)",
                 context.toString().c_str(), instanceNumberToBeCloned);
   presCtrlInterface.startAddressInputClone(context, instanceNumberToBeCloned);
}

void NavMiddleware::startFormBasedAddressInputFromAddressDetails(const LocationContext& context, const LocationAttributeInfos::AddressDetailInfo& addressDetaiInfo)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"startFormBasedAddressInputFromAddressDetails(context=%s, addressDetails=%s)",
                 context.toString().c_str(),addressDetaiInfo.toString().c_str());

   presCtrlInterface.startFormBasedAddressInputFromAddressDetails(context, addressDetaiInfo);
}

const AddressInputInfo& NavMiddleware::getFormBasedAddressInputInfo() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getFormBasedAddressInputInfo()");
   LOG_INTERFACE_DEBUG(location, "\n ############ Form based address info ############ - %s",
                       presCtrlInterface.getFormBasedAddressInputInfo().toString().c_str());

   return presCtrlInterface.getFormBasedAddressInputInfo();
}

void NavMiddleware::selectAddressCategoryForInput(AddressCategory category, unsigned int numFirstMatches)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"selectAddressCategoryForInput(category=%s, numFirstMatches=%u)",
                 toString(category).c_str(), numFirstMatches);

   presCtrlInterface.selectAddressCategoryForInput(category, numFirstMatches);
}

const ElementSpellerInfo& NavMiddleware::getAddressSpellerInfo() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getAddressSpellerInfo()");
   LOG_INTERFACE_DEBUG(location, "\n ############ Element speller info ############ - %s",
                       presCtrlInterface.getAddressSpellerInfo().toString().c_str());

   return presCtrlInterface.getAddressSpellerInfo();
}

void NavMiddleware::setAddressSpellerString(const ::std::string& spellerString, unsigned int numFirstMatches)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"setAddressSpellerString(spellerString=\"%s\", numFirstMatches=%u)",
                 spellerString.c_str(), numFirstMatches);

   presCtrlInterface.setAddressSpellerString(spellerString, numFirstMatches);
}

void NavMiddleware::requestAddressElements(unsigned int startIndex, unsigned int numElements)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"requestAddressElements(startIndex=%u, numElements=%u)", startIndex, numElements);

   presCtrlInterface.requestAddressElements(startIndex, numElements);
}

const LocationInfos& NavMiddleware::getAddressElements() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getAddressElements()");
   LOG_INTERFACE_DEBUG(location, "\n ############ Address Elements ############ - %s",
                       presCtrlInterface.getAddressElements().toString().c_str());

   return presCtrlInterface.getAddressElements();
}

void NavMiddleware::selectAddressElement(const LocationInfos::LocationInfo& locationInfo)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"selectAddressElement(locationInfo.m_index=%u)", locationInfo.m_index);

   presCtrlInterface.selectAddressElement(locationInfo);
}

void NavMiddleware::requestLastSelectedAddressElements()
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"requestLastSelectedAddressElements()");

   presCtrlInterface.requestLastSelectedAddressElements();
}

const LocationInfos& NavMiddleware::getLastSelectedAddressElements()
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getLastSelectedAddressElements()");
   LOG_INTERFACE_DEBUG(location, "\n ############ Last Selected Address Elements ############ - %s",
                       presCtrlInterface.getLastSelectedAddressElements().toString().c_str());

   return presCtrlInterface.getLastSelectedAddressElements();
}

void NavMiddleware::selectFromLastSelectedAddressElements(const LocationInfos::LocationInfo& locationInfo)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"selectFromLastSelectedAddressElements(locationInfo.m_index=%u)", locationInfo.m_index);

   presCtrlInterface.selectFromLastSelectedAddressElements(locationInfo);
}

void NavMiddleware::undoLastAddressInputStep()
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"undoLastAddressInputStep()");

   presCtrlInterface.undoLastAddressInputStep();
}

void NavMiddleware::clearAddressCategory(AddressCategory category)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"clearAddressCategory(category=%s)", toString(category).c_str());

   presCtrlInterface.clearAddressCategory(category);
}

bool NavMiddleware::setLocationWithCurrentAddressInput()
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"setLocationWithCurrentAddressInput()");

   return presCtrlInterface.setLocationWithCurrentAddressInput();
}

void NavMiddleware::leaveFormBasedAddressInput()
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"leaveFormBasedAddressInput()");

   return presCtrlInterface.leaveFormBasedAddressInput();
}

void NavMiddleware::setPostValidationForHouseNumberInput(bool status)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"setPostValidationForHouseNumberInput(status=%d)",status);

   presCtrlInterface.setPostValidationForHouseNumberInput(status);
}

void NavMiddleware::setFormBasedAddressInputScope(AddressCategory /*scope*/)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"setFormBasedAddressInputScope()");
   LOG_INTERFACE_WARN(location, "not supported - only JP");
}

AddressCategory NavMiddleware::getFormBasedAddressInputScope() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getFormBasedAddressInputScope()");
   LOG_INTERFACE_WARN(location, "not supported - only JP");
   return COUNTRY;
}

const OnlinePoiAvailabilityInfo& NavMiddleware::getOnlinePoiAvailability() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getOnlinePoiAvailability()");
   LOG_INTERFACE_DEBUG(location, "\n ############ Get Online Poi Availability ############ - %s",
                       presCtrlInterface.getOnlinePoiAvailability().toString().c_str());

   return presCtrlInterface.getOnlinePoiAvailability();
}

void NavMiddleware::requestPoiOnlineServerAvailability(LocationSearchSource server)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"requestPoiOnlineServerAvailability (server=%s)", toString(server).c_str());

   presCtrlInterface.requestPoiOnlineServerAvailability(server);
}

void NavMiddleware::startHierarchicalPoiInput(
   const LocationSearchScope& searchScope,
   LocationSearchSource searchSource,
   const LocationContext& context)
{
   LOG_INTERFACE(location,
                 MESSAGE_TAG_IN"startHierarchicalPoiInput(\n"
                 "   searchScope=%s,\n"
                 "   searchSource=%s,\n"
                 "   context=%s)",
                 searchScope.toString().c_str(), toString(searchSource).c_str(), context.toString().c_str());

   presCtrlInterface.startHierarchicalPoiInput(searchScope, searchSource, context);
}

void NavMiddleware::startPoiInputWithCategoryShortcut(
   unsigned int shortcutNum,
   const LocationSearchScope& searchScope,
   LocationSearchSource searchSource,
   PoiCategoryFilter poiCategoryFilter,
   const LocationContext& context)
{
   LOG_INTERFACE(location,
                 MESSAGE_TAG_IN"startPoiInputWithCategoryShortcut(\n"
                 "   shortcutNum=%u,\n"
                 "   searchScope=%s,\n"
                 "   searchSource=%s,\n"
                 "   poiCategoryFilter=%s,\n"
                 "   context=%s)",
                 shortcutNum, searchScope.toString().c_str(), toString(searchSource).c_str(),
                 toString(poiCategoryFilter).c_str(),context.toString().c_str());

   presCtrlInterface.startPoiInputWithCategoryShortcut(
      shortcutNum, searchScope, searchSource, poiCategoryFilter, context);
}

void NavMiddleware::startPoiInputWithinAllShortcutCategories(
   const LocationSearchScope& searchScope,
   LocationSearchSource searchSource,
   const LocationContext& context)
{
   LOG_INTERFACE(location,
                 MESSAGE_TAG_IN"startPoiInputWithinAllShortcutCategories(\n"
                 "   searchScope=%s,\n"
                 "   searchSource=%s,\n"
                 "   context=%s)",
                 searchScope.toString().c_str(), toString(searchSource).c_str(), context.toString().c_str());

   presCtrlInterface.startPoiInputWithinAllShortcutCategories(
      searchScope, searchSource, context);
}

void NavMiddleware::startPoiInputWithPredefinedCategory(
   PredefinedPoiCategory predefinedPoiCategory,
   const LocationSearchScope& searchScope,
   LocationSearchSource searchSource,
   PoiCategoryFilter poiCategoryFilter,
   const LocationContext& context)
{
   LOG_INTERFACE(location, \
                 MESSAGE_TAG_IN"startPoiInputWithPredefinedCategory(\n"
                 "   predefinedPoiCategory=%s,\n"
                 "   searchScope=%s,\n"
                 "   searchSource=%s,\n"
                 "   poiCategoryFilter=%s,\n"
                 "   context=%s)",
                 toString(predefinedPoiCategory).c_str(), searchScope.toString().c_str(), toString(searchSource).c_str(),
                 toString(poiCategoryFilter).c_str(),context.toString().c_str());

   presCtrlInterface.startPoiInputWithPredefinedCategory(
      predefinedPoiCategory, searchScope, searchSource, poiCategoryFilter, context);
}

void NavMiddleware::startPoiInputWithInFtsChosenCategory(
   const LocationSearchScope& searchScope,
   LocationSearchSource searchSource,
   const LocationContext& context)
{
   LOG_INTERFACE(location, \
                 MESSAGE_TAG_IN"startPoiInputWithInFtsChosenCategory(\n"
                 "   searchScope=%s,\n"
                 "   searchSource=%s,\n"
                 "   context=%s)",
                 searchScope.toString().c_str(), toString(searchSource).c_str(),
                 context.toString().c_str());

   presCtrlInterface.startPoiInputWithInFtsChosenCategory(searchScope, searchSource, context);
}

void NavMiddleware::resumePoiInputInstance(const LocationContext& context)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"resumePoiInputInstance(context=%s)", context.toString().c_str());

   presCtrlInterface.resumePoiInputInstance(context);
}

const PoiInputInfo& NavMiddleware::getPoiInputInfo() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getPoiInputInfo()");
   LOG_INTERFACE_DEBUG(location, "\n ############ POI input info ############ - %s",
                       presCtrlInterface.getPoiInputInfo().toString().c_str());

   return presCtrlInterface.getPoiInputInfo();
}

void NavMiddleware::setPoiSortOrder(LocationSortOrder sortOrder)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"setPoiSortOrder(sortOrder=%s)", toString(sortOrder).c_str());

   presCtrlInterface.setPoiSortOrder(sortOrder);
}

void NavMiddleware::setPoiSearchScope(const LocationSearchScope& searchScope)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"setPoiSearchScope(searchScope=%s)", searchScope.toString().c_str());

   presCtrlInterface.setPoiSearchScope(searchScope);
}

void NavMiddleware::setPoiSearchSource(LocationSearchSource searchSource)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"setPoiSearchSource(searchSource=%s)", toString(searchSource).c_str());

   presCtrlInterface.setPoiSearchSource(searchSource);
}

void NavMiddleware::requestPoiEntries(unsigned int startIndex, unsigned int numEntries)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"requestPoiEntries(startIndex=%u, numElements=%u)", startIndex, numEntries);

   presCtrlInterface.requestPoiEntries(startIndex, numEntries);
}

const LocationInfos& NavMiddleware::getPoiEntries() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getPoiEntries()");
   LOG_INTERFACE_DEBUG(location, "\n ############ POI entries info ############ - %s",
                       presCtrlInterface.getPoiEntries().toString().c_str());

   return presCtrlInterface.getPoiEntries();
}

void NavMiddleware::requestPoiEntryAttributes(unsigned int listIndex)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"requestPoiEntryAttributes(listIndex = %u)", listIndex);

   presCtrlInterface.requestPoiEntryAttributes(listIndex);
}

const LocationAttributeInfos& NavMiddleware::getPoiEntryAttributes() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getPoiEntryAttributes()");
   LOG_INTERFACE_DEBUG(location, "\n ############ POI entries Attribute info ############ - %s",
                       presCtrlInterface.getPoiEntryAttributes().toString().c_str());

   return presCtrlInterface.getPoiEntryAttributes();
}

const LocationAttributeErrorInfo& NavMiddleware::getPoiEntryAttributesError() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getPoiEntryAttributesError()");
   LOG_INTERFACE(location, "\n ############ Get Poi Entry Attributes Error ############ - %s",
                 presCtrlInterface.getPoiEntryAttributesError().toString().c_str());

   return presCtrlInterface.getPoiEntryAttributesError();
}

void NavMiddleware::selectPoiEntry(const LocationInfos::LocationInfo& locationInfo)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"selectPoiEntry(locationInfo.m_index=%u)", locationInfo.m_index);

   presCtrlInterface.selectPoiEntry(locationInfo);
}

void NavMiddleware::selectPoiEntry(unsigned int listIndex)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"selectPoiEntry(listIndex=%u)", listIndex);

   presCtrlInterface.selectPoiEntry(listIndex);
}

void NavMiddleware::selectAllPoiCategory()
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"selectAllPoiCategory()");

   presCtrlInterface.selectAllPoiCategory();
}

void NavMiddleware::undoLastPoiEntrySelection()
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"undoLastPoiEntrySelection()");

   presCtrlInterface.undoLastPoiEntrySelection();
}

bool NavMiddleware::setLocationWithCurrentPoiInput()
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"setLocationWithCurrentPoiInput()");

   return presCtrlInterface.setLocationWithCurrentPoiInput();
}

void NavMiddleware::leaveHierarchicalPoiInput()
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"leaveHierarchicalPoiInput()");

   return presCtrlInterface.leaveHierarchicalPoiInput();
}

const ElementSpellerInfo& NavMiddleware::getPoiCategorySpellerInfo() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getPoiCategorySpellerInfo()");
   LOG_INTERFACE_DEBUG(location,"Japan Specific, Not Implemented");

   static ElementSpellerInfo elementSpellerInfo;
   return elementSpellerInfo;
}

void NavMiddleware::setPoiCategorySpellerString(const ::std::string& spellerString, unsigned int numFirstEntries)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"setPoiCategorySpellerString(%s,%d)",spellerString.c_str(),numFirstEntries);
   LOG_INTERFACE_WARN(location,"Japan Specific, Not Implemented");
}

const LocationCategoryInfos& NavMiddleware::getPoiShortcutCategoryInfos(const LocationContext& context) const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getPoiShortcutCategoryInfos(context=%s)", context.toString().c_str());
   LOG_INTERFACE_DEBUG(location, "\n ############ POI shortcut infos ############ - %s",
                       presCtrlInterface.getPoiShortcutCategoryInfos(context).toString().c_str());

   return presCtrlInterface.getPoiShortcutCategoryInfos(context);
}

void NavMiddleware::resetPoiShortcutsToFactoryDefaults()
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"resetPoiShortcutsToFactoryDefaults()");

   presCtrlInterface.resetPoiShortcutsToFactoryDefaults();
}

void NavMiddleware::storeCurrentPoiCategoryAsShortcut(unsigned int shortcutNum, const ::navmiddleware::LocationContext& context)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"storeCurrentPoiCategoryAsShortcut(shortcutNum=%u, context=%s)", shortcutNum, context.toString().c_str());

   presCtrlInterface.storeCurrentPoiCategoryAsShortcut(shortcutNum, context);
}

void NavMiddleware::deletePoiCategoryShortcut(unsigned int shortcutNum, const ::navmiddleware::LocationContext& context)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"deletePoiCategoryShortcut(shortcutNum=%u, context=%s)", shortcutNum, context.toString().c_str());

   presCtrlInterface.deletePoiShortcut(shortcutNum, context);
}

void NavMiddleware::movePoiShortcut(unsigned int shortCutNumDrag, unsigned int shortCutNumDrop, const LocationContext& context)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"movePoiShortcut(%u,%u,%s)", shortCutNumDrag,shortCutNumDrop,context.toString().c_str());

   presCtrlInterface.movePoiShortcut(shortCutNumDrag,shortCutNumDrop,context);
}

const ::std::map<ExitPOICategory, LocationCategoryInfos::LocationCategoryInfo>&
NavMiddleware::getExitPoiCategoryInfos() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getExitPoiCategoryInfos()");
   LOG_INTERFACE_DEBUG(location, "\n ############ Exit POI Category infos ############ - %s",
                       ::navmiddleware::toString<ExitPOICategory, LocationCategoryInfos::LocationCategoryInfo>
                       (presCtrlInterface.getExitPoiCategoryInfos()).c_str());
   return presCtrlInterface.getExitPoiCategoryInfos();
}

void NavMiddleware::requestPoiIconInfos(const LocationContext& context , const std::vector< ::navmiddleware::PoiIconInfos::ImageVariant>& variants,
                                        const std::vector< ::navmiddleware::PoiCategoryId>& poiCategories)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"requestPoiIconInfos(%s,%s,%s)", context.toString().c_str(),
                 ::navmiddleware::PoiIconInfos::toString(variants).c_str(),
                 ::navmiddleware::PoiIconInfos::toString(poiCategories).c_str());
   presCtrlInterface.requestPoiIconInfos(context,variants,poiCategories);
}

const PoiIconInfos& NavMiddleware::getPoiIconInfos(const LocationContext& context) const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getPoiIconInfos(%s)", context.toString().c_str());
   LOG_INTERFACE_DEBUG(location, "\n ############ Get Poi Icons Info ############ - %s",
                       presCtrlInterface.getPoiIconInfos(context).toString().c_str());
   return presCtrlInterface.getPoiIconInfos(context);
}

const OnlineFreeTextSearchAvailabilityInfo& NavMiddleware::getOnlineFreeTextSearchAvailability() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getOnlineFreeTextSearchAvailability()");
   LOG_INTERFACE_DEBUG(location, "\n ############ Get Online Free Text Search Availability ############ - %s",
                       presCtrlInterface.getOnlineFreeTextSearchAvailability().toString().c_str());
   return presCtrlInterface.getOnlineFreeTextSearchAvailability();
}

void NavMiddleware::requestFreeTextSearchOnlineServerAvailability(LocationSearchSource server)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"requestFreeTextSearchOnlineServerAvailability (server=%s)", toString(server).c_str());

   presCtrlInterface.requestFreeTextSearchOnlineServerAvailability(server);
}

void NavMiddleware::startFreeTextSearchInput(
   FreeTextSearchType searchType,
   const LocationSearchScope& searchScope,
   LocationSearchSource searchSource,
   const LocationContext& context,
   const ValidValue<LanguageEntryInfo>& languageEntryInfo)
{
   LOG_INTERFACE(location,
                 MESSAGE_TAG_IN"startFreeTextSearchInput(\n"
                 "   searchType=%s,\n"
                 "   searchScope=%s,\n"
                 "   searchSource=%s,\n"
                 "   context=%s, \n"
                 "   languageEntryInfo=%s",
                 toString(searchType).c_str(), searchScope.toString().c_str(), toString(searchSource).c_str(),
                 context.toString().c_str(),
                 (languageEntryInfo.isValid()) ? languageEntryInfo.getValue().toString().c_str() : "Invalid");

   presCtrlInterface.startFreeTextSearchInput(searchType, searchScope, searchSource, context);
}

void NavMiddleware::setFreeTextSearchString(const ::std::string& searchString)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"setFreeTextSearchString(searchString=\"%s\")", searchString.c_str());

   presCtrlInterface.setFreeTextSearchString(searchString);
}

const ::std::string& NavMiddleware::getFreeTextSearchString() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getFreeTextSearchString(): \"%s\"", presCtrlInterface.getFreeTextSearchString().c_str());

   return presCtrlInterface.getFreeTextSearchString();
}

void NavMiddleware::setFreeTextSearchScope(const LocationSearchScope& searchScope)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"setFreeTextSearchScope(scope=%s)", searchScope.toString().c_str());

   presCtrlInterface.setFreeTextSearchScope(searchScope);
}

void NavMiddleware::setFreeTextSearchSource(LocationSearchSource searchSource)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"setFreeTextSearchSource(source=%s)", toString(searchSource).c_str());

   presCtrlInterface.setFreeTextSearchSource(searchSource);
}

void NavMiddleware::setFreeTextSearchSortOrder(LocationSortOrder sortOrder)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"setFreeTextSearchSortOrder(sortOrder=%s)",
                 toString(sortOrder).c_str());
   presCtrlInterface.setFreeTextSearchSortOrder(sortOrder);
}

void NavMiddleware::setFreeTextSearchAutoCompletionMode(AutoCompletionMode autoCompletionMode)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"setFreeTextSearchAutoCompletionMode(autoCompletionMode=%s)",
                 toString(autoCompletionMode).c_str());
   presCtrlInterface.setFreeTextSearchAutoCompletionMode(autoCompletionMode);
}

void NavMiddleware::requestFreeTextSearchResults(unsigned int startIndex, unsigned int numElements, FreeTextSearchType filterType)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"requestFreeTextSearchResults(startIndex=%u, numElements=%u, filterType=%s)",
                 startIndex, numElements, toString(filterType).c_str());

   presCtrlInterface.requestFreeTextSearchResults(startIndex, numElements, filterType);
}

const LocationInfos& NavMiddleware::getFreeTextSearchResultInfos() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getFreeTextSearchResultInfos()");
   LOG_INTERFACE_DEBUG(location, "\n ############ Freetext search result info ############ - %s",
                       presCtrlInterface.getFreeTextSearchResultInfos().toString().c_str());

   return presCtrlInterface.getFreeTextSearchResultInfos();
}

void NavMiddleware::requestFreeTextSearchResultAttributes(unsigned int listIndex)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"requestFreeTextSearchResultAttributes(listIndex = %u)", listIndex);

   return presCtrlInterface.requestFreeTextSearchResultAttributes(listIndex);
}

const LocationAttributeInfos& NavMiddleware::getFreeTextSearchResultAttributes() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getFreeTextSearchResultAttributes()");
   LOG_INTERFACE_DEBUG(location, "\n ############ FTS Result Attributes info ############ - %s",
                       presCtrlInterface.getFreeTextSearchResultAttributes().toString().c_str());
   return presCtrlInterface.getFreeTextSearchResultAttributes();
}

const LocationAttributeErrorInfo& NavMiddleware::getFreeTextSearchResultAttributesError() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getFreeTextSearchResultAttributesError()");
   LOG_INTERFACE(location, "\n ############ Get Free Text Search Result Error ############ - %s",
                 presCtrlInterface.getFreeTextSearchResultAttributesError().toString().c_str());

   return presCtrlInterface.getFreeTextSearchResultAttributesError();
}

const ElementSpellerInfo& NavMiddleware::getFreeTextSearchSpellerInfo() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getFreeTextSearchSpellerInfo()");
   LOG_INTERFACE_DEBUG(location, "\n ############ Element speller info ############ - %s",
                       presCtrlInterface.getFreeTextSearchSpellerInfo().toString().c_str());

   return presCtrlInterface.getFreeTextSearchSpellerInfo();
}

void NavMiddleware::setLocationWithFreeTextSearchResult(const LocationInfos::LocationInfo& locationInfo)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"setLocationWithFreeTextSearchResult(locationInfo.m_index=%u)", locationInfo.m_index);

   presCtrlInterface.setLocationWithFreeTextSearchResult(locationInfo);
}

void NavMiddleware::setLocationWithFreeTextSearchResult(unsigned int listIndex)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"setLocationWithFreeTextSearchResult(listIndex=%u)", listIndex);

   presCtrlInterface.setLocationWithFreeTextSearchResult(listIndex);
}

void NavMiddleware::selectFreeTextSearchElement(unsigned int listIndex)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"selectFreeTextSearchElement(listIndex=%u)", listIndex);
   //[JP only]
   LOG_INTERFACE_WARN(location, "selectFreeTextSearchElement() currently unsupported");
}

void NavMiddleware::undoFreeTextSearchElementSelection()
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"undoFreeTextSearchElementSelection()");

   presCtrlInterface.undoFreeTextSearchElementSelection();
}

void NavMiddleware::leaveFreeTextSearchInput()
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"leaveFreeTextSearchInput()");

   presCtrlInterface.leaveFreeTextSearchInput();
}

void NavMiddleware::clearFreeTextSearchHistory()
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"clearFreeTextSearchHistory()");

   presCtrlInterface.clearFreeTextSearchHistory();
}

void NavMiddleware::requestRefreshForExpiredFTSResult(unsigned int listIndex)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"requestRefreshForExpiredFTSResult(listIndex = %u)", listIndex);

   presCtrlInterface.requestRefreshForExpiredFTSResult(listIndex);
}

void NavMiddleware::requestOnboardMatchingForExpiredFTSResult(unsigned int listIndex)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"requestOnboardMatchingForExpiredFTSResult(listIndex = %u)", listIndex);

   presCtrlInterface.requestOnboardMatchingForExpiredFTSResult(listIndex);
}

EntrySelectionStatus NavMiddleware::getEntrySelectionStatus() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getEntrySelectionStatus()");
   LOG_INTERFACE_DEBUG(location, MESSAGE_TAG_IN"getEntrySelectionStatus(%s)", toString(presCtrlInterface.getEntrySelectionStatus()).c_str());
   return presCtrlInterface.getEntrySelectionStatus();
}

const RefreshInfo& NavMiddleware::getRefreshStatus() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getRefreshStatus()");
   LOG_INTERFACE_DEBUG(location, "\n ############ Refresh status info ############ - %s", presCtrlInterface.getRefreshStatus().toString().c_str());
   return presCtrlInterface.getRefreshStatus();
}

void NavMiddleware::setFreeTextSearchAutoCompleteCategories(const AutoCompletionConfiguration& autoCompletionConfiguration)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"setFreeTextSearchAutoCompleteCategories(autoCompletionConfiguration=%s)",
                 autoCompletionConfiguration.toString().c_str());
   presCtrlInterface.setFreeTextSearchAutoCompleteCategories(autoCompletionConfiguration);
}

void NavMiddleware::resumeFreeTextSearchInstance(const LocationContext& context)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"resumeFreeTextSearchInstance(context=%s)", context.toString().c_str());
   return presCtrlInterface.resumeFreeTextSearchInstance(context);
}

const ::navmiddleware::ResumeFreeTextSearchStatusInfo& NavMiddleware::getResumeFreeTextSearchStatusInfo() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getResumeFreeTextSearchStatusInfo()");
   LOG_INTERFACE_DEBUG(location, "\n ############ Resume fts info ############ - %s", presCtrlInterface.getResumeFreeTextSearchStatusInfo().toString().c_str());
   return presCtrlInterface.getResumeFreeTextSearchStatusInfo();

}

const ::std::string& NavMiddleware::getCurrentLocationString() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getCurrentLocationString()");
   LOG_INTERFACE_DEBUG(location, "getCurrentLocationString returns (%s)", presCtrlInterface.getCurrentLocationString().c_str());
   return presCtrlInterface.getCurrentLocationString();
}

void NavMiddleware::sdsCheckAddress(const SDSAddress& sdsAddress,const ValidValue<LanguageEntryInfo>& languageEntryInfo)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"sdsCheckAddress()");
   LOG_INTERFACE_DEBUG(location, "\n ############ sdsCheckAddress payload ############ - %s \n %s",
                       sdsAddress.toString().c_str(),(languageEntryInfo.isValid()) ? languageEntryInfo.getValue().toString().c_str() : "languageEntryInfo is Invalid");

   presCtrlInterface.sdsCheckAddress(sdsAddress);
}

const SDSAddressWithOptions& NavMiddleware::getCurrentSDSAddressWithOptions() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getCurrentSDSAddressWithOptions()");
   LOG_INTERFACE_DEBUG(location, "\n ############ SDS address with option ############ - %s",
                       presCtrlInterface.getCurrentSDSAddressWithOptions().toString().c_str());

   return presCtrlInterface.getCurrentSDSAddressWithOptions();
}

void NavMiddleware::requestSdsRefinementList()
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"requestSdsRefinementList()");

   presCtrlInterface.requestSdsRefinementList();
}

void NavMiddleware::requestSdsRefinementList(SDSAddressElementType sdsAddressElementType)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"requestSdsRefinementList(%s)",toString(sdsAddressElementType).c_str());

   presCtrlInterface.requestSdsRefinementList(sdsAddressElementType);
}

const RefinementWithIdInfos& NavMiddleware::getSdsRefinementListInfo() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getSdsRefinementListInfo()");
   LOG_INTERFACE_DEBUG(location, "\n ############SDS Refinement info ############ - %s",
                       presCtrlInterface.getSdsRefinementListInfo().toString().c_str());

   return presCtrlInterface.getSdsRefinementListInfo();
}

void NavMiddleware::selectSdsRefinement(unsigned int id)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"selectSdsRefinement(id=%u)", id);

   presCtrlInterface.selectSdsRefinement(id);
}

void NavMiddleware::requestSdsHouseNumberPatterns()
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"requestSdsHouseNumberPatterns()");

   presCtrlInterface.requestSdsHouseNumberPatterns();
}

const SdsHouseNumberPatternsInfos& NavMiddleware::getSdsHouseNumberPatternsInfo() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getSdsHouseNumberPatternsInfo()");
   LOG_INTERFACE_DEBUG(location, "\n ############SDS House Number Pattern info ############ - %s",
                       presCtrlInterface.getSdsHouseNumberPatternsInfo().toString().c_str());

   return presCtrlInterface.getSdsHouseNumberPatternsInfo();
}

void NavMiddleware::denyMotorwayUsageInCurrentLocation()
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"denyMotorwayUsageInCurrentLocation()");
   presCtrlInterface.denyMotorwayUsageInCurrentLocation();
}

void NavMiddleware::setLocationWithSdsInput()
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"setLocationWithSdsInput()");

   presCtrlInterface.setLocationWithSdsInput();
}

void NavMiddleware::requestGlobalSdsData()
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"requestGlobalSdsData()");

   presCtrlInterface.requestGlobalSdsData();
}

void NavMiddleware::requestIdListOfCountryAndSubCountry()
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"requestIdListOfCountryAndSubCountry()");

   presCtrlInterface.requestIdListOfCountryAndSubCountry();
}

RequestId NavMiddleware::requestNamesForNdsIds(const ::std::vector<SdsGlobalNamedObjectIdInfo>& SdsGlobalNamedObjectIdInfos,
      const ::std::vector<LanguageEntryInfo>& languageEntryInfos, NdsFeatureType ndsFeatureType, bool stopOnFirstMatch)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"requestNamesForNdsIds(%lu,%lu,%u,%d)",
                 SdsGlobalNamedObjectIdInfos.size(),languageEntryInfos.size(),ndsFeatureType,stopOnFirstMatch);

   return presCtrlInterface.requestNamesForNdsIds(SdsGlobalNamedObjectIdInfos,languageEntryInfos,ndsFeatureType,stopOnFirstMatch);
}

RequestId NavMiddleware::requestPhonemesForNdsIds(const ::std::vector<SdsGlobalNamedObjectIdInfo>& SdsGlobalNamedObjectIdInfos,
      const ::std::vector<LanguageEntryInfo>& languageEntryInfos, NdsFeatureType ndsFeatureType, bool stopOnFirstMatch)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"requestPhonemesForNdsIds(%lu,%lu,%u,%d)",
                 SdsGlobalNamedObjectIdInfos.size(),languageEntryInfos.size(),ndsFeatureType,stopOnFirstMatch);

   return presCtrlInterface.requestPhonemesForNdsIds(SdsGlobalNamedObjectIdInfos,languageEntryInfos,ndsFeatureType,stopOnFirstMatch);
}

void NavMiddleware::requestCityHasStreets(const ::std::vector<SdsGlobalNamedObjectIdInfo>& SdsGlobalNamedObjectIdInfos)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"requestCityHasStreets(%lu)",SdsGlobalNamedObjectIdInfos.size());

   presCtrlInterface.requestCityHasStreets(SdsGlobalNamedObjectIdInfos);
}

const SdsGlobalDataInfo& NavMiddleware::getSdsGlobalDataInfo() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getSdsGlobalDataInfo()");
   LOG_INTERFACE_DEBUG(location, "\n ############SdsGlobalDataInfo ############ - %s",
                       presCtrlInterface.getSdsGlobalDataInfo().toString().c_str());

   return presCtrlInterface.getSdsGlobalDataInfo();
}

const SdsCountryAndSubCountryInfos& NavMiddleware::getSdsCountryAndSubCountryInfos() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getSdsCountryAndSubCountryInfos()");
   LOG_INTERFACE_DEBUG(location, "\n ############SdsCountryAndSubCountryInfos ############ - %s",
                       presCtrlInterface.getSdsCountryAndSubCountryInfos().toString().c_str());

   return presCtrlInterface.getSdsCountryAndSubCountryInfos();
}

const SdsNdsIdStringInfos& NavMiddleware::getSdsNdsIdNamedStringInfos() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getSdsNdsIdNamedStringInfos()");
   LOG_INTERFACE_DEBUG(location, "\n ############SdsNdsIdStringInfos ############ - %s",
                       presCtrlInterface.getSdsNdsIdNamedStringInfos().toString().c_str());

   return presCtrlInterface.getSdsNdsIdNamedStringInfos();
}

const SdsNdsIdStringInfos& NavMiddleware::getSdsNdsIdPhonemeStringInfos() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getSdsNdsIdPhonemeStringInfos()");
   LOG_INTERFACE_DEBUG(location, "\n ############SdsNdsIdStringInfos ############ - %s",
                       presCtrlInterface.getSdsNdsIdPhonemeStringInfos().toString().c_str());

   return presCtrlInterface.getSdsNdsIdPhonemeStringInfos();
}

const ::std::vector<SdsCityNamedObjectHasStreet>& NavMiddleware::getSdsCityNamedObjectHasStreetInfos() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getSdsCityNamedObjectHasStreetInfos() List Size %lu",
                 presCtrlInterface.getSdsCityNamedObjectHasStreetInfos().size());
   LOG_INTERFACE_DEBUG(location, "\n ############SdsCityNamedObjectHasStreet ############ - %s",
                       ::navmiddleware::toString<SdsCityNamedObjectHasStreet>(presCtrlInterface.getSdsCityNamedObjectHasStreetInfos()).c_str());

   return presCtrlInterface.getSdsCityNamedObjectHasStreetInfos();
}

void NavMiddleware::startGuidance()
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"startGuidance()");

   presCtrlInterface.startRouteGuidance();
}

void NavMiddleware::startRouteCalculationAndGuidance(bool isResume)
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"startRouteCalculationAndGuidance(%d)",isResume);

   presCtrlInterface.startRouteCalculationAndGuidance();
}

void NavMiddleware::restartGuidanceToLastDestination()
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"restartGuidanceToLastDestination()");

   presCtrlInterface.restartGuidanceToLastDestination();
}

const DestinationInfos& NavMiddleware::getDestinationInfosForLastMileNavigation() const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"getDestinationInfosForLastMileNavigation()");
   LOG_INTERFACE_DEBUG(guidance, "\n ############ DestinationInfos details ############ - %s",
                       presCtrlInterface.getDestinationInfosForLastMileNavigation().toString().c_str());
   return presCtrlInterface.getDestinationInfosForLastMileNavigation();
}

const DestinationMemoryItem& NavMiddleware::getLastGuidedItem()
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"getLastGuidedItem()");
   LOG_INTERFACE_DEBUG(guidance, "\n ############ LastGuidedItem details ############ - %s",
                       presCtrlInterface.getLastGuidedItem().toString().c_str());
   return presCtrlInterface.getLastGuidedItem();
}

void NavMiddleware::stopGuidance()
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"stopGuidance()");

   presCtrlInterface.stopRouteGuidance();
}

bool NavMiddleware::isGuidanceActive() const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"isGuidanceActive()");
   LOG_INTERFACE_DEBUG(guidance, "\n ############ Is Guidance Active  ############ - %s",
                       presCtrlInterface.isRouteGuidanceActive() ? "true" : "false");
   return presCtrlInterface.isRouteGuidanceActive();
}

bool NavMiddleware::isFinalDestinationReached() const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"isFinalDestinationReached()");
   LOG_INTERFACE_DEBUG(guidance, "\n ############ is Final Destination Reached  ############ - %s",
                       presCtrlInterface.isFinalDestinationReached() ? "true" : "false");
   return presCtrlInterface.isFinalDestinationReached();
}

void NavMiddleware::setSpeedLimitData(const ::navmiddleware::SpeedLimitData& speedLimitData)
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"setSpeedLimitData()");
   LOG_INTERFACE_DEBUG(guidance, "\n ############ Set speed limit data  ############ -%s ",
                       speedLimitData.toString().c_str());

   presCtrlInterface.setSpeedLimitData(speedLimitData);
}

const NextManeuverInfos& NavMiddleware::getNextManeuverInfos() const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"getNextManeuverInfo()");
   LOG_INTERFACE_DEBUG(guidance, "\n ############ Next Maneuver Info ############ - %s",
                       presCtrlInterface.getNextManeuverInfos().toString().c_str());

   return presCtrlInterface.getNextManeuverInfos();
}

const ManeuverViewInfo& NavMiddleware::getManeuverViewInfo() const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"getManeuverViewInfo()");
   LOG_INTERFACE_DEBUG(guidance, "\n ############ Maneuver View Info ############ - %s",
                       presCtrlInterface.getManeuverViewInfo().toString().c_str());

   return presCtrlInterface.getManeuverViewInfo();
}

const SignPostInfo& NavMiddleware::getSignPostInfo() const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"getSignPostInfo()");
   LOG_INTERFACE_DEBUG(guidance, "\n ############ SignPost Info ############ - %s",
                       presCtrlInterface.getSignPostInfo().toString().c_str());

   return presCtrlInterface.getSignPostInfo();
}

const LaneInfos& NavMiddleware::getLaneInfos() const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"getLaneInfos()");
   LOG_INTERFACE_DEBUG(guidance, "\n ############ Lane Info ############ - %s",
                       presCtrlInterface.getLaneInfos().toString().c_str());

   return presCtrlInterface.getLaneInfos();
}

const SpeedInfo& NavMiddleware::getSpeedInfo() const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"getSpeedInfo()");
   LOG_INTERFACE_DEBUG(guidance, "\n ############ Speed Info ############ - %s",
                       presCtrlInterface.getSpeedInfo().toString().c_str());
   return presCtrlInterface.getSpeedInfo();
}

const SpeedCamInfo& NavMiddleware::getSpeedCamInfo() const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"getSpeedCamInfo()");
   LOG_INTERFACE_DEBUG(guidance, "\n ############ SpeedCamInfo ############ - %s",
                       presCtrlInterface.getSpeedCamInfo().toString().c_str());
   return presCtrlInterface.getSpeedCamInfo();
}

const TrafficSignInfo& NavMiddleware::getTrafficSignInfo() const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"getTrafficSignInfo()");
   LOG_INTERFACE_DEBUG(guidance, "\n ############ TrafficSignInfo ############ - %s",
                       presCtrlInterface.getTrafficSignInfo().toString().c_str());
   return presCtrlInterface.getTrafficSignInfo();
}

const TrafficEnforcementInfo& NavMiddleware::getTrafficEnforcementInfo() const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"getTrafficEnforcementInfo()");
   LOG_INTERFACE_DEBUG(guidance, "\n ############ TrafficEnforcementInfo ############ - %s",
                       presCtrlInterface.getTrafficEnforcementInfo().toString().c_str());
   return presCtrlInterface.getTrafficEnforcementInfo();
}

const DestinationReachedInfo& NavMiddleware::getDestinationReachedInfo() const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"getDestinationReachedInfo()");
   LOG_INTERFACE_DEBUG(guidance, "\n ############ DestinationReachedInfo ############ - %s",
                       presCtrlInterface.getDestinationReachedInfo().toString().c_str());
   return presCtrlInterface.getDestinationReachedInfo();
}

const EVAlertInfo& NavMiddleware::getGuardianAngelAlertInfo() const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"getGuardianAngelAlertInfo()");
   LOG_INTERFACE_DEBUG(guidance, "\n ############ GuardianAngelAlertInfo ############ - %s",
                       presCtrlInterface.getGuardianAngelAlertInfo().toString().c_str());
   return presCtrlInterface.getGuardianAngelAlertInfo();
}

const EVAlertInfo& NavMiddleware::getUnreachableDestinationAlertInfo() const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"getUnreachableDestinationAlertInfo()");
   LOG_INTERFACE_DEBUG(guidance, "\n ############ UnreachableDestinationAlertInfo ############ - %s",
                       presCtrlInterface.getUnreachableDestinationAlertInfo().toString().c_str());
   return presCtrlInterface.getUnreachableDestinationAlertInfo();
}

const EVAlertInfo& NavMiddleware::getLowBatteryAlertInfo() const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"getLowBatteryAlertInfo()");
   LOG_INTERFACE_DEBUG(guidance, "\n ############ LowBatteryAlertInfo ############ - %s",
                       presCtrlInterface.getLowBatteryAlertInfo().toString().c_str());
   return presCtrlInterface.getLowBatteryAlertInfo();
}

const NearestChargingStationInfo& NavMiddleware::getNearestChargingStationInfo() const
{
   static const NearestChargingStationInfo nearestChargingStationInfo;
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"getNearestChargingStationInfo()");
   LOG_INTERFACE_DEBUG(guidance, "\n ############ NearestChargingStationInfo ############ - %s",
                       nearestChargingStationInfo.toString().c_str());
   return nearestChargingStationInfo;
}

const PredictedDestinationInfo& NavMiddleware::getPredictedDestinationInfo() const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"getPredictedDestinationInfo()");
   LOG_INTERFACE_DEBUG(guidance, "\n ############ PredictedDestinationInfo ############ - %s",
                       presCtrlInterface.getPredictedDestinationInfo().toString().c_str());
   return presCtrlInterface.getPredictedDestinationInfo();
}

void NavMiddleware::declinePredictedDestination(const PredictedDestinationInfo& /*predictedDestinationInfo*/)
{
   //JP Specific
}

const VehicleRestrictionsInfo& NavMiddleware::getVehicleRestrictionsInfo() const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"getVehicleRestrictionsInfo()");
   LOG_INTERFACE_DEBUG(guidance, "\n ############ VehicleRestrictionsInfo ############ - %s",
                       presCtrlInterface.getVehicleRestrictionsInfo().toString().c_str());
   return presCtrlInterface.getVehicleRestrictionsInfo();
}

VehicleRestrictionsStatus NavMiddleware::getVehicleRestrictionsStatus() const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"getVehicleRestrictionsStatus()");
   VehicleRestrictionsStatus vehicleRestrictionsStatus = presCtrlInterface.getVehicleRestrictionsStatus();
   LOG_INTERFACE_DEBUG(guidance, "\n ############ VehicleRestrictionsStatus ############ - %s",
                       toString(vehicleRestrictionsStatus).c_str());
   return vehicleRestrictionsStatus;
}

const CountryAndStateDetailsInfo& NavMiddleware::getCountryAndStateDetailsInfo() const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"getCountryAndStateDetailsInfo()");
   LOG_INTERFACE_DEBUG(guidance, "\n ############ CountryAndStateDetailsInfo ############ - %s",
                       presCtrlInterface.getCountryAndStateDetailsInfo().toString().c_str());
   return presCtrlInterface.getCountryAndStateDetailsInfo();
}

void NavMiddleware::retriggerAcousticOutput()
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"retriggerAcousticOutput()");

   presCtrlInterface.retriggerAcousticOutput();
}

void NavMiddleware::speakSamplePhrase(SamplePhrase samplePhraseType) const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"speakSamplePhrase(SamplePhrase = %d)", samplePhraseType);
   presCtrlInterface.speakSamplePhrase(samplePhraseType);
}

void NavMiddleware::speakPoiAlert(PoiAlertType poiAlertType) const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"speakPoiAlert(PoiAlertType = %d)", poiAlertType);
   presCtrlInterface.speakPoiAlert(poiAlertType);
}
void NavMiddleware::requestCountryAndStateDetails(const ::std::string& isoCountryCode, const ::std::string& isoSubCountryCode) const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"requestCountryAndStateDetails(isoCountryCode = %s, isoSubCountryCode = %s",
                 isoCountryCode.c_str(), isoSubCountryCode.c_str());
   presCtrlInterface.requestCountryAndStateDetails(isoCountryCode, isoSubCountryCode);
}

void NavMiddleware::requestRoadRestrictions(const ValidValue< LinkId >& linkId) const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"requestRoadRestrictions(linkId = %s)", linkId.toString().c_str());
   presCtrlInterface.requestRoadRestrictions(linkId);
}

void NavMiddleware::confirmPoiAlert(unsigned int alertIdentifier)
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"confirmPoiAlert(alertId = %d)", alertIdentifier);
   presCtrlInterface.confirmPoiAlert(alertIdentifier);
}

const HorizonEventInfo& NavMiddleware::getHorizonEventInfo() const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"getHorizonEventInfo()");
   LOG_INTERFACE_DEBUG(guidance, "\n ############ Horizon Event ############ - %s",
                       presCtrlInterface.getHorizonEventInfo().toString().c_str());
   return presCtrlInterface.getHorizonEventInfo();
}

bool NavMiddleware::isLocatorOnRoadWithAD2() const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"isLocatorOnRoadWithAD2()");
   return false;
}

bool NavMiddleware::isAutoZoomActive() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"isAutoZoomActive()");
   LOG(NOTSUPPORTED"");
   return false;
}

void NavMiddleware::switchToLinkedMode(MapViewId mapViewId, CameraAnimation cameraAnimation)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"switchToLinkedMode()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s ,cameraAnimation = %s", toString(mapViewId).c_str(), toString(cameraAnimation).c_str());

   presCtrlInterface.switchToLinkedMode(mapViewId, cameraAnimation);
}

bool NavMiddleware::isInLinkedMode(MapViewId mapViewId) const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"isInLinkedMode()");
   LOG_INTERFACE_DEBUG(map, "\n ############ isInLinkedMode ############ - mapViewId = %s, isInLinkedMode = %s",
                       toString(mapViewId).c_str(), presCtrlInterface.isInLinkedMode(mapViewId) ? "true" : "false");

   return presCtrlInterface.isInLinkedMode(mapViewId);
}

void NavMiddleware::switchToFreeMode(MapViewId mapViewId, CameraConfiguration cameraConfiguration)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"switchToFreeMode()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, cameraConfiguration = %s", toString(mapViewId).c_str(), toString(cameraConfiguration).c_str());

   presCtrlInterface.switchToFreeMode(mapViewId, cameraConfiguration);
}

void NavMiddleware::zoomInStep(MapViewId mapViewId, uint32_t stepCount, ZoomType zoomType, CameraAnimation cameraAnimation)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"zoomInStep()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, stepCount = %d, zoomType = %s, cameraAnimation = %s",
                       toString(mapViewId).c_str(), stepCount, toString(zoomType).c_str(), toString(cameraAnimation).c_str());

   presCtrlInterface.zoomInStep(mapViewId, stepCount, zoomType, cameraAnimation);
}

void NavMiddleware::zoomOutStep(MapViewId mapViewId, uint32_t stepCount, ZoomType zoomType, CameraAnimation cameraAnimation)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"zoomOutStep()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, stepCount = %d, zoomType = %s, cameraAnimation = %s",
                       toString(mapViewId).c_str(), stepCount, toString(zoomType).c_str(), toString(cameraAnimation).c_str());

   presCtrlInterface.zoomOutStep(mapViewId, stepCount, zoomType, cameraAnimation);
}

const MapCameraInfos& NavMiddleware::getMapCameraInfos() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"getMapCameraInfos()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Map camera info ############ - %s",
                       presCtrlInterface.getMapCameraInfos().toString().c_str());

   return presCtrlInterface.getMapCameraInfos();
}

const MapViewStatusInfos& NavMiddleware::getViewStatusInfos()const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"getViewStatusInfos()");
   LOG_INTERFACE_DEBUG(map, "\n ############ View Status info ############ - %s",
                       presCtrlInterface.getViewStatusInfos().toString().c_str());
   return( presCtrlInterface.getViewStatusInfos());
}

const MapRepresentationInfo& NavMiddleware::getMapRepresentationInfo() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"getMapRepresentationInfos()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Map representation info ############ - %s",
                       presCtrlInterface.getMapRepresentationInfo().toString().c_str());

   return presCtrlInterface.getMapRepresentationInfo();
}

void NavMiddleware::setMapRepresentation(MapViewId mapViewId,
      const MapRepresentation& mapRepresentation, CameraAnimation cameraAnimation)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"setMapRepresentation()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, mapPitch = %s, mapOrientation = %s, cameraAnimation = %s",
                       toString(mapViewId).c_str(), toString(mapRepresentation.m_mapPitch).c_str(),
                       toString(mapRepresentation.m_mapOrientation).c_str(), toString(cameraAnimation).c_str());

   presCtrlInterface.setMapRepresentation(mapViewId, mapRepresentation,
                                          cameraAnimation);
}

void NavMiddleware::freezeMap(MapViewId mapViewId, MapViewVisibility mapViewVisibility)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"freezeMap()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, mapViewVisibility = %s",toString(mapViewId).c_str(), toString(mapViewVisibility).c_str());

   presCtrlInterface.freezeMap(mapViewId, mapViewVisibility);
}

void NavMiddleware::freezeMap(MapViewId mapViewId,
                              ::navmiddleware::MapViewStatusInfos::DataRenderingStatus renderingStatus,
                              uint16_t duration,
                              ::navmiddleware::MapViewVisibility mapViewVisibility)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"freezeMap()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, DataRenderingStatus = %s, duration = %d, mapViewVisibility = %s",
                       toString(mapViewId).c_str(), ::navmiddleware::MapViewStatusInfos::toString(renderingStatus).c_str(), duration,
                       toString(mapViewVisibility).c_str());

   presCtrlInterface.freezeMap(mapViewId, renderingStatus, duration, mapViewVisibility);
}

void NavMiddleware::setMapFrameRate(MapViewId mapViewId, uint8_t frameRate, uint16_t duration)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"setMapFrameRate()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, frame rate = %d, duration = %d",
                       toString(mapViewId).c_str(), frameRate, duration);

   presCtrlInterface.setMapFrameRate(mapViewId, frameRate, duration);
}

void NavMiddleware::unFreezeMap(MapViewId mapViewId)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"unFreezeMap()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s", toString(mapViewId).c_str());

   presCtrlInterface.unFreezeMap(mapViewId);
}

bool NavMiddleware::isMapFrozen(MapViewId mapViewId) const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"isMapFrozen()");
   LOG_INTERFACE_DEBUG(map, "\n ############ is Map Frozen ############ - mapViewId = %s, isMapFrozen = %s", toString(mapViewId).c_str(), presCtrlInterface.isMapFrozen(mapViewId) ? "true" : "false");
   return presCtrlInterface.isMapFrozen(mapViewId);
}

ValidValue<RequestId> NavMiddleware::pickInMap(MapViewId mapViewId, unsigned int posX, unsigned int posY,
      const ValidValue<PickingResultFilter>& pickingResultFilter)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"pickInMap()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, posX = %u, posY = %u, pickingResultFilter = %s",
                       toString(mapViewId).c_str(), posX, posY, pickingResultFilter.isValid() ? pickingResultFilter.getValue().toString().c_str() : "Invalid");

   return presCtrlInterface.pickInMap(mapViewId, posX, posY, pickingResultFilter);
}

const MapScaleRange& NavMiddleware::getMapScaleRange() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"getMapScaleRange()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Map Scale Range ############ - %s",
                       presCtrlInterface.getMapScaleRange().toString().c_str());
   return presCtrlInterface.getMapScaleRange();
}

const PickingResults& NavMiddleware::getPickingResults() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"getPickingResults()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Picking results info ############ - %s",
                       presCtrlInterface.getPickingResults().toString().c_str());

   return presCtrlInterface.getPickingResults();
}

void NavMiddleware::requestPickedIndexFromPoiIcons(MapViewId mapViewId, unsigned int posX, unsigned int posY)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"requestPickedIndexFromPoiIcons()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, posX = %u, posY = %u", toString(mapViewId).c_str(), posX, posY);
   return presCtrlInterface.requestPickedIndexFromPoiIcons(mapViewId, posX, posY);
}

const PickedIndexFromPoiIcons& NavMiddleware::getPickedIndexFromPoiIcons() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"getPickedIndexFromPoiIcons()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Picked Index From Poi Icons info ############ - %s",
                       presCtrlInterface.getPickedIndexFromPoiIcons().toString().c_str());
   return presCtrlInterface.getPickedIndexFromPoiIcons();
}

void NavMiddleware::requestPickingDetails(MapViewId mapViewId,
      const ::std::vector< PickingResults::PickingResult >& elementsListForDetails)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"requestPickingDetails()");
   LOG_INTERFACE_DEBUG(map, "Num of elementsListForDetails = %d, mapViewId = %s",
                       elementsListForDetails.size(), toString(mapViewId).c_str());

   return presCtrlInterface.requestPickingDetails(mapViewId, elementsListForDetails);
}

void NavMiddleware::releasePickInMap(MapViewId mapViewId) const
{
   LOG_INTERFACE(map, "releasePickInMap()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s", toString(mapViewId).c_str());

   return presCtrlInterface.releasePickingDetails(mapViewId);
}

void NavMiddleware::showPickedPOIsInMap(MapViewId mapViewId,
                                        const VisibleListInfo& pickedPOIsVisibilityListInfo,
                                        const ValidValue<int32_t>& mapScale,
                                        CameraAnimation cameraAnimation)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"showPickedPOIsInMap()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, pickedPOIsVisibilityListInfo = %s, mapScale = %d, cameraAnimation = %s",
                       toString(mapViewId).c_str(),
                       pickedPOIsVisibilityListInfo.toString().c_str(),
                       mapScale.getValueOr(0),
                       toString(cameraAnimation).c_str());
   presCtrlInterface.showPickedPOIsInMap(mapViewId, pickedPOIsVisibilityListInfo, mapScale, cameraAnimation);
}

void NavMiddleware::highlightPickingResult(MapViewId mapViewId,
      const ::std::vector< PickingResults::PickingResult >& pickingResultList)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"highlightPickingResult()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, Num of PickingResult = %d",
                       toString(mapViewId).c_str(), pickingResultList.size());

   return presCtrlInterface.highlightPickingResult(mapViewId, pickingResultList);
}

void NavMiddleware::requestUserPoiIcons()
{
   LOG_INTERFACE(iconcache, MESSAGE_TAG_IN"requestUserPoiIcons()");
   presCtrlInterface.requestUserPoiIcons();
}

const ::std::vector< ::navmiddleware::Image >& NavMiddleware::getUserPoiIcons()
{
   LOG_INTERFACE(iconcache, MESSAGE_TAG_IN"getUserPoiIcons()");
   LOG_INTERFACE_DEBUG(iconcache, "\n ############ UserPoiIcons info ############ - %s",
                       ::navmiddleware::toString<Image>(presCtrlInterface.getUserPoiIcons()).c_str());
   return presCtrlInterface.getUserPoiIcons();
}

void NavMiddleware::startScrollingMap(MapViewId mapViewId, ScrollDirection scrollDirection,
                                      ScrollSpeed scrollSpeed, uint16_t maxScrollInPixels)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"startScrollingMap()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, direction = %s, speed = %s, maxPixels = %d", toString(mapViewId).c_str(),
                       toString(scrollDirection).c_str(), toString(scrollSpeed).c_str(), maxScrollInPixels);

   presCtrlInterface.startScrollingMap(mapViewId, scrollDirection, scrollSpeed, maxScrollInPixels);
}

void NavMiddleware::stopScrollingMap(MapViewId mapViewId)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"stopScrollingMap()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s", toString(mapViewId).c_str());

   presCtrlInterface.stopScrollingMap(mapViewId);
}

void NavMiddleware::startRotatingMap(MapViewId mapViewId, RotationType rotationType, RotationSpeed rotationSpeed, int32_t maxRotationAngle)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"startRotatingMap()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, rotationType = %s, rotationSpeed = %s, maxRotationAngle = %d",
                       toString(mapViewId).c_str(), toString(rotationType).c_str(), toString(rotationSpeed).c_str(), maxRotationAngle);
   presCtrlInterface.startRotatingMap(mapViewId, rotationType, rotationSpeed, maxRotationAngle);
}

void NavMiddleware::stopRotatingMap(MapViewId mapViewId)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"stopRotatingMap()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s", toString(mapViewId).c_str());

   presCtrlInterface.stopRotatingMap(mapViewId);
}

void NavMiddleware::rotateCamera(MapViewId mapViewId, CoordinateAxis coordinateAxis, RotationType rotationType, uint32_t angle)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"rotateCamera()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, coordinateAxis = %s, rotationType = %s, angle = %d",
                       toString(mapViewId).c_str(), toString(coordinateAxis).c_str(), toString(rotationType).c_str(), angle);
   presCtrlInterface.rotateCamera(mapViewId, coordinateAxis, rotationType, angle);
}

const MapCameraModeInfo& NavMiddleware::getMapCameraModeInfo() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"getMapCameraModeInfo()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Map camera mode info ############ - %s",
                       presCtrlInterface.getMapCameraModeInfo().toString().c_str());

   return presCtrlInterface.getMapCameraModeInfo();
}

void NavMiddleware::showRouteOverview(MapViewId mapViewId, ShowAlternatives showAlternatives,
                                      ::navmiddleware::settings::MoreRoutesTypeValue moreRouteTypesValue, DestinationPreference destinationPreference, bool showPlannedAndGuidedRoute)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"showRouteOverview()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, ShowAlternatives = %s, moreRoutesType = %s, destinationPreference = %s, showPlannedAndGuidedRoute = %d",
                       toString(mapViewId).c_str(), toString(showAlternatives).c_str(), toString(moreRouteTypesValue).c_str(), toString(destinationPreference).c_str(), showPlannedAndGuidedRoute);
   presCtrlInterface.showRouteOverview(mapViewId, showAlternatives, moreRouteTypesValue, destinationPreference, showPlannedAndGuidedRoute);
}

void NavMiddleware::showEditRouteOverviewInMap(MapViewId mapViewId, const ValidValue<int32_t>& mapScale,
      CameraAnimation cameraAnimation, const ValidValue<RouteVisibilityOptions>& routeVisibilityOptions)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"showEditRouteOverviewInMap()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, mapScale = %s, cameraAnimation = %s, routeVisibilityOptions = %s",
                       toString(mapViewId).c_str(), mapScale.isValid() ? mapScale.toString().c_str() : "Invalid", ::navmiddleware::toString(cameraAnimation).c_str(),
                       routeVisibilityOptions.isValid() ? (routeVisibilityOptions.getValue()).toString().c_str() : "Invalid");

   presCtrlInterface.showEditRouteOverviewInMap(mapViewId, mapScale, cameraAnimation, routeVisibilityOptions);
}

void NavMiddleware::requestTrafficMessageList(unsigned int startIndex, unsigned int numElements,
      TrafficMessageListType listType, TrafficMessageTextType textType)
{
   LOG_INTERFACE(trafficmessage, MESSAGE_TAG_IN"requestTrafficMessageList(%u, %u, %s, %s)", startIndex, numElements, ::navmiddleware::toString(listType).c_str(),
                 ::navmiddleware::toString(textType).c_str());

   presCtrlInterface.requestTrafficMessageList(startIndex, numElements, listType, textType);
}

void NavMiddleware::requestTrafficMessageListForRange(unsigned int maxDistanceInMeters, TrafficMessageListType listType,
      TrafficMessageTextType textType,  bool isFreeFlowMessagesRequired, bool isIncidentTrafficMessagesRequired)
{
   LOG_INTERFACE(trafficmessage, MESSAGE_TAG_IN"requestTrafficMessageListForRange(%u, %s, %s, %d, %d )",
                 maxDistanceInMeters, ::navmiddleware::toString(listType).c_str(), ::navmiddleware::toString(textType).c_str()
                 , isFreeFlowMessagesRequired, isIncidentTrafficMessagesRequired);

   presCtrlInterface.requestTrafficMessageListForRange(maxDistanceInMeters, listType, textType, isFreeFlowMessagesRequired, isIncidentTrafficMessagesRequired);
}

void NavMiddleware::requestTrafficMessageDetails(unsigned int trafficMessageID)
{
   LOG_INTERFACE(trafficmessage, MESSAGE_TAG_IN"requestTrafficMessageDetails(id=%u)", trafficMessageID);

   presCtrlInterface.requestTrafficMessageDetails(trafficMessageID);
}

const TrafficMessageListInfos& NavMiddleware::getTrafficMessageListInfos() const
{
   LOG_INTERFACE(trafficmessage, MESSAGE_TAG_IN"getTrafficMessageListInfos()");
   // log the payload
   LOG_INTERFACE_DEBUG(trafficmessage, "%s", presCtrlInterface.getTrafficMessageListInfos().toString().c_str());

   return presCtrlInterface.getTrafficMessageListInfos();
}

const ValidValue<TrafficMessageDetailsInfo>& NavMiddleware::getTrafficMessageDetailsInfo() const
{
   LOG_INTERFACE(trafficmessage, MESSAGE_TAG_IN"getTrafficMessageDetailsInfo()");
   // log the payload
   LOG_INTERFACE_DEBUG(trafficmessage, "%s", presCtrlInterface.getTrafficMessageDetailsInfo().getValue().toString().c_str());

   return presCtrlInterface.getTrafficMessageDetailsInfo();
}

void NavMiddleware::releaseTrafficMessageDetails(unsigned int trafficMessageID)
{
   LOG_INTERFACE(trafficmessage, MESSAGE_TAG_IN"releaseTrafficMessageDetails(id=%u)", trafficMessageID);

   presCtrlInterface.releaseTrafficMessageDetails(trafficMessageID);
}

const TrafficMessageListInfos& NavMiddleware::getXUrgentTrafficMessageListInfos() const
{
   LOG_INTERFACE(trafficmessage, MESSAGE_TAG_IN"getXUrgentTrafficMessageListInfos()");

   // log the payload
   LOG_INTERFACE_DEBUG(trafficmessage, "%s", presCtrlInterface.getXUrgentTrafficMessageListInfos().toString().c_str());

   return presCtrlInterface.getXUrgentTrafficMessageListInfos();
}

void NavMiddleware::releaseXUrgentTrafficMessage(unsigned int trafficMessageID)
{
   LOG_INTERFACE(trafficmessage, MESSAGE_TAG_IN"releaseXUrgentTrafficMessage(id=%u)", trafficMessageID);

   presCtrlInterface.releaseXUrgentTrafficMessage(trafficMessageID);
}

const ValidValue<TrafficServiceReceptionStatusInfo>& NavMiddleware::getTrafficServiceReceptionStatusInfo() const
{
   LOG_INTERFACE(trafficmessage, MESSAGE_TAG_IN"getTrafficServiceReceptionStatusInfo()");

   // log the payload
   if (presCtrlInterface.getTrafficServiceReceptionStatusInfo().isValid())
   {
      LOG_INTERFACE_DEBUG(trafficmessage, "%s", presCtrlInterface.getTrafficServiceReceptionStatusInfo().getValue().toString().c_str());
   }
   else
   {
      LOG_INTERFACE_DEBUG(trafficmessage, "TrafficServiceReceptionStatusInfo is invalid");
   }

   return presCtrlInterface.getTrafficServiceReceptionStatusInfo();
}

int NavMiddleware::getCountOfAllTrafficMessages() const
{
   LOG_INTERFACE(trafficmessage, MESSAGE_TAG_IN"getCountOfAllTrafficMessages()");
   // log the payload
   LOG_INTERFACE_DEBUG(trafficmessage, "%d", presCtrlInterface.getCountOfAllTrafficMessages());

   return presCtrlInterface.getCountOfAllTrafficMessages();
}

void NavMiddleware::setTrafficMessageListCurrentVisualRange(unsigned int startIndex, unsigned int numElements)
{
   LOG_INTERFACE(trafficmessage, MESSAGE_TAG_IN"setTrafficMessageListCurrentVisualRange(%u , %u)", startIndex, numElements);

   presCtrlInterface.setTrafficMessageListCurrentVisualRange(startIndex, numElements);
}

void NavMiddleware::setTrafficMessageListCurrentFocusedElementIndex(const ValidValue<unsigned int>& index)
{
   LOG_INTERFACE(trafficmessage, MESSAGE_TAG_IN"setTrafficMessageListCurrentFocusedElementIndex()");

   presCtrlInterface.setTrafficMessageListCurrentFocusedElementIndex(index);
}

void NavMiddleware::setTrafficMessageUserRule(unsigned int trafficMessageID, TrafficMessageUserRule userRule)
{
   LOG_INTERFACE(trafficmessage, MESSAGE_TAG_IN"setTrafficMessageUserRule(%u, %s)", trafficMessageID, ::navmiddleware::toString(userRule).c_str());

   presCtrlInterface.setTrafficMessageUserRule(trafficMessageID, userRule);
}

void NavMiddleware::releaseAllTrafficMessageLists()
{
   LOG_INTERFACE(trafficmessage, MESSAGE_TAG_IN"ReleaseAllTrafficMessageList()");

   presCtrlInterface.releaseAllTrafficMessageLists();
}

void NavMiddleware::detourAllTrafficMessagesAlongRoute()
{
   LOG_INTERFACE(trafficmessage, MESSAGE_TAG_IN"detourAllTrafficMessagesAlongRoute()");

   presCtrlInterface.detourAllTrafficMessagesAlongRoute();
}

void NavMiddleware::cancelDetourAllTrafficMessagesAlongRoute()
{
   LOG_INTERFACE(trafficmessage, MESSAGE_TAG_IN"cancelDetourAllTrafficMessagesAlongRoute()");

   presCtrlInterface.cancelDetourAllTrafficMessagesAlongRoute();
}

TrafficMessageDetourResult NavMiddleware::getTrafficMessageDetourResult() const
{
   LOG_INTERFACE(trafficmessage, MESSAGE_TAG_IN"getTrafficMessageDetourResult()");

   const TrafficMessageDetourResult result = presCtrlInterface.getTrafficMessageDetourResult();
   LOG_INTERFACE_DEBUG(trafficmessage, "%s", ::navmiddleware::toString(result).c_str());

   return result;
}

void NavMiddleware::requestTrafficIconSelectionList()
{
   LOG_INTERFACE(trafficmessage, MESSAGE_TAG_IN"requestTrafficIconSelectionList()");

   presCtrlInterface.requestTrafficIconSelectionList();
}

const TrafficIconSelectionListInfo& NavMiddleware::getTrafficIconSelectionListInfo() const
{
   LOG_INTERFACE(trafficmessage, MESSAGE_TAG_IN"getTrafficIconSelectionListInfo()");
   // log the payload
   LOG_INTERFACE_DEBUG(trafficmessage, "%s", presCtrlInterface.getTrafficIconSelectionListInfo().toString().c_str());

   return presCtrlInterface.getTrafficIconSelectionListInfo();
}

RequestId NavMiddleware::requestWeatherData(const ::std::vector< ReportConfiguration>& reportConfigurations)
{
   LOG_INTERFACE(locationbasedservices, MESSAGE_TAG_IN"requestWeatherData()");

   LOG_INTERFACE_DEBUG(locationbasedservices, "\n %s",
                       ::navmiddleware::toString<ReportConfiguration>(reportConfigurations).c_str());

   return presCtrlInterface.requestWeatherData(reportConfigurations);
}

const WeatherDataInfos& NavMiddleware::getWeatherDataInfos(RequestId requestId) const
{
   LOG_INTERFACE(locationbasedservices, MESSAGE_TAG_IN"getWeatherDataInfos()");
   LOG_INTERFACE_DEBUG(locationbasedservices, "%s",presCtrlInterface.getWeatherDataInfos(requestId).toString().c_str());

   return presCtrlInterface.getWeatherDataInfos(requestId);
}

settings::LocationBasedServicesSettings NavMiddleware::getLocationBasedServicesSettings() const
{
   LOG_INTERFACE(locationbasedservices, MESSAGE_TAG_IN"getLocationBasedServicesSettings()");
   LOG_INTERFACE_DEBUG(locationbasedservices, "%s", presCtrlInterface.getLocationBasedServicesSettings().toString().c_str());
   return presCtrlInterface.getLocationBasedServicesSettings();
}

void NavMiddleware::updateLocationBasedServicesSettings(const settings::LocationBasedServicesSettings& locationBasedServicesSettings) const
{
   LOG_INTERFACE(locationbasedservices, MESSAGE_TAG_IN"updateLocationBasedServicesSettings()");
   LOG_INTERFACE_DEBUG(locationbasedservices, "\n - %s",
                       locationBasedServicesSettings.toString().c_str());

   presCtrlInterface.updateLocationBasedServicesSettings(locationBasedServicesSettings);
}

void NavMiddleware::releaseWeatherData(RequestId requestId)
{
   LOG_INTERFACE(locationbasedservices, MESSAGE_TAG_IN"releaseWeatherData()");

   presCtrlInterface.releaseWeatherData(requestId);
}

::navmiddleware::WeatherReceptionState NavMiddleware::getWeatherDataReceptionState() const
{
   LOG_INTERFACE(locationbasedservices, MESSAGE_TAG_IN"getWeatherDataReceptionState()");
   LOG_INTERFACE_DEBUG(locationbasedservices, "%s", ::navmiddleware::toString(presCtrlInterface.getWeatherDataReceptionState()).c_str());
   return presCtrlInterface.getWeatherDataReceptionState();
}

const ::navmiddleware::OnlineServicesDiagnosticInfos& NavMiddleware::getOnlineServicesDiagnosticInfos() const
{
   LOG_INTERFACE(locationbasedservices, MESSAGE_TAG_IN"getOnlineServicesDiagnosticData()");
   return presCtrlInterface.getOnlineServicesDiagnosticInfos();
}

const ::navmiddleware::ConnectedServicesAvailabilityInfos& NavMiddleware::getConnectedServicesAvailabilityInfos() const
{
   LOG_INTERFACE(locationbasedservices, MESSAGE_TAG_IN"getConnectedServicesAvailabilityInfos()");
   LOG_INTERFACE_DEBUG(locationbasedservices, "%s", presCtrlInterface.getConnectedServicesAvailabilityInfos().toString().c_str());
   return presCtrlInterface.getConnectedServicesAvailabilityInfos();
}

void NavMiddleware::setAdvancedTrafficLoggingState(::navmiddleware::AdvancedLoggingState state)
{
   LOG_INTERFACE(locationbasedservices, MESSAGE_TAG_IN"setAdvancedTrafficLoggingState() %s", toString(state).c_str());
   return presCtrlInterface.setAdvancedTrafficLoggingState(state);
}

settings::RouteCriteriaSettings NavMiddleware::getRouteCriteriaSettings() const
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"getRouteCriteriaSettings");
   LOG_INTERFACE_DEBUG(route, "RouteCriteriaSettings - %d", presCtrlInterface.getRouteCriteriaSettings());

   return presCtrlInterface.getRouteCriteriaSettings();
}

settings::RouteOptionsSettings NavMiddleware::getRouteOptionSettings() const
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"getRouteOptionSettings");

   return presCtrlInterface.getRouteOptionSetting();
}

settings::RestoreStatus NavMiddleware::getLastRestoreStatusOfRouteCriteriaSettings() const
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"getLastRestoreStatusOfRouteCriteriaSettings");
   return presCtrlInterface.getLastRestoreStatusOfRouteOptionsSettings();
}

settings::RestoreStatus NavMiddleware::getLastRestoreStatusOfRouteOptionsSettings() const
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"getLastRestoreStatusOfRouteOptionsSettings");
   return presCtrlInterface.getLastRestoreStatusOfRouteOptionsSettings();
}

settings::VehicleProfileSettings NavMiddleware::getVehicleProfileSettings() const
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"getVehicleProfileSettings");

   return presCtrlInterface.getVehicleProfileSettings();
}

settings::RouteOptionsSettings NavMiddleware::getRouteOptionSettingsFromRoute() const
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"getRouteOptionSettingsFromRoute");

   return presCtrlInterface.getRouteOptionSettingsFromRoute();
}

void NavMiddleware::updateRouteCriteriaSettings(const settings::RouteCriteriaSettings& settings)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"updateRouteCriteriaSettings(RouteCriteriaSettings - %d)", settings.getRouteCriterion());

   presCtrlInterface.updateRouteCriteriaSettings(settings);
}

void NavMiddleware::updateRouteOptionSettings(const settings::RouteOptionsSettings& settings)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"updateRouteOptionSettings");

   presCtrlInterface.updateRouteOptionSettings(settings);
}

void NavMiddleware::updateVehicleProfileSettings(const settings::VehicleProfileSettings& settings)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"updateVehicleProfileSettings");

   presCtrlInterface.updateVehicleProfileSettings(settings);
}

void NavMiddleware::applySettingsToRoute(const settings::RouteOptionsSettings& settings)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"applySettingsToRoute");

   presCtrlInterface.applySettingsToRoute(settings);
}

void NavMiddleware::deleteLearnedRoutes()
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"deleteLearnedRoutes");

   presCtrlInterface.deleteLearnedRoutes();
}

void NavMiddleware::deletePredictedDestinations()
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"deletePredictedDestinations()");

   presCtrlInterface.deletePredictedDestinations();
}

void NavMiddleware::startRouteCalculation(RouteCalculationMode routeCalculationMode)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"startRouteCalculation(%s)", toString(routeCalculationMode).c_str());

   presCtrlInterface.startRouteCalculation(routeCalculationMode);
}

void NavMiddleware::stopRouteCalculation()
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"stopRouteCalculation()");

   presCtrlInterface.stopRouteCalculation();
}

void NavMiddleware::recalculateRoute()
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"recalculateRoute()");

   LOG_INTERFACE_DEBUG(route, "Not implemented, Japan specific interface");
}

void NavMiddleware::activatePrecalculatedRoute()
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"activatePrecalculatedRoute()");

   presCtrlInterface.activatePrecalculatedRoute();
}

bool NavMiddleware::hasActiveRoute() const
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"hasActiveRoute(): %s",
                 (presCtrlInterface.hasActiveRoute()? "true" : "false"));

   return presCtrlInterface.hasActiveRoute();
}

void NavMiddleware::calculateMoreRoutes()
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"DEPRECATED: calculateMoreRoutes()");

   presCtrlInterface.calculateMoreRoutes();
}

const ::navmiddleware::RouteCalculationInfo& NavMiddleware::getRouteCalculationInfo() const
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"getRouteCalculationInfo()");
   LOG_INTERFACE_DEBUG(route, "\n ############ route calculation info ############ - %s",
                       presCtrlInterface.getRouteCalculationInfo().toString().c_str());
   return presCtrlInterface.getRouteCalculationInfo();
}

const PlannedRouteHistoryInfos& NavMiddleware::getPlannedRouteHistoryInfos() const
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"getPlannedRouteHistoryInfos()");
   LOG_INTERFACE_DEBUG(route, "\n ############ planned route history infos ############ - %s",
                       presCtrlInterface.getPlannedRouteHistoryInfos().toString().c_str());
   return presCtrlInterface.getPlannedRouteHistoryInfos();
}

void NavMiddleware::activateHighlightedRoute()
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"DEPRECATED: activateHighlightedRoute()");

   presCtrlInterface.activateHighlightedRoute();
}

const WaypointInfos& NavMiddleware::getWaypointInfos() const
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"getWaypointInfos()");
   LOG_INTERFACE_DEBUG(route, "\n ############ waypoint info ############ - %s",
                       presCtrlInterface.getWaypointInfos().toString().c_str());

   return presCtrlInterface.getWaypointInfos();
}

void NavMiddleware::requestWaypointList(WaypointListMode waypointListMode)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"requestWaypointList(%u)", waypointListMode);

   presCtrlInterface.requestWaypointList(waypointListMode);
}

void NavMiddleware::requestWaypointListPlannedRoute()
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"requestWaypointListPlannedRoute()");

   presCtrlInterface.requestWaypointListPlannedRoute();
}

void NavMiddleware::applyWaypointListChange(WaypointEditMode waypointEditMode)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"applyWaypointListChange(waypointEditMode = %d)",waypointEditMode);

   presCtrlInterface.applyWaypointListChange(waypointEditMode);
}

void NavMiddleware::setStartPosition()
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"setStartPosition()");

   presCtrlInterface.setStartPosition();
}

void NavMiddleware::deleteStartPosition()
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"deleteStartPosition()");

   presCtrlInterface.deleteStartPosition();
}

void NavMiddleware::insertWaypoint(unsigned int newIndex, WaypointEditMode waypointEditMode)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"insertWaypoint(%u, %u)", newIndex, waypointEditMode);

   presCtrlInterface.insertWaypoint(newIndex, waypointEditMode);
}

void NavMiddleware::deleteWaypoints(::std::vector<unsigned int>& waypointIndexList, WaypointEditMode waypointEditMode)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"deleteWaypoint(WaypointEditMode: %u)", waypointEditMode);

   presCtrlInterface.deleteWaypoint(waypointIndexList, waypointEditMode);
}

void NavMiddleware::moveWaypoint(unsigned int fromOldIndex, unsigned int toNewIndex, WaypointEditMode waypointEditMode)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"moveWaypoint(%u , %u , %u)", fromOldIndex, toNewIndex, waypointEditMode);

   presCtrlInterface.moveWaypoint(fromOldIndex, toNewIndex, waypointEditMode);
}

void NavMiddleware::skipWaypoint(bool skipWaypoint, unsigned int waypointIndex)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"skipWaypoint(%s , %u)", (skipWaypoint?"true":"false"), waypointIndex);

   presCtrlInterface.skipWaypoint(skipWaypoint, waypointIndex);
}

void NavMiddleware::setRouteCriterionForWaypoint(unsigned int index, settings::RouteCriterion routeCriterion)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"setRouteCriterionForWaypoint(%u,  %d)", index, routeCriterion);

   presCtrlInterface.setRouteCriterionForWaypoint(index, routeCriterion);
}

void NavMiddleware::setRouteCriterionForOptimization(settings::RouteCriterion routeCriterion)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"setRouteCriterionForOptimization(%d)", routeCriterion);

   presCtrlInterface.setRouteCriterionForOptimization(routeCriterion);
}

void NavMiddleware::setTimeWindowsForAllWaypoints(const ::std::vector<WaypointOptimizationDetails>& waypointOptimizationDetails)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"setTimeWindowsForAllWaypoints");

   presCtrlInterface.setTimeWindowsForAllWaypoints(waypointOptimizationDetails);
}

void NavMiddleware::setFinalDestinationForOptimization(unsigned int waypointIndex, bool isFinalDest)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"setFinalDestinationForOptimization(%u, %d)", waypointIndex, isFinalDest);

   presCtrlInterface.setFinalDestinationForOptimization(waypointIndex, isFinalDest);
}

void NavMiddleware::setOptimizationRange(unsigned int optimizeFromSegment, unsigned int optimizeToSegment)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"setOptimizationRange(from %u to %u)", optimizeFromSegment, optimizeToSegment);

   presCtrlInterface.setOptimizationRange(optimizeFromSegment, optimizeToSegment);
}

void NavMiddleware::applyRouteOptimization()
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"applyRouteOptimization()");

   presCtrlInterface.applyRouteOptimization();
}

void NavMiddleware::cancelRouteOptimization()
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"cancelRouteOptimization()");

   presCtrlInterface.cancelRouteOptimization();
}

void NavMiddleware::cancelEditWaypointsMode()
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"cancelEditWaypointsMode()");

   presCtrlInterface.cancelEditWaypointsMode();
}

void NavMiddleware::replaceWaypointLocation(unsigned int index, WaypointEditMode waypointEditMode)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"replaceWaypointLocation(%u, %u)", index, waypointEditMode);

   presCtrlInterface.replaceWaypointLocation(index, waypointEditMode);
}

void NavMiddleware::acceptTrafficImpactRouteAlternative()
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"acceptTrafficImpactRouteAlternative()");

   presCtrlInterface.acceptTrafficImpactRouteAlternative();
}

void NavMiddleware::rejectTrafficImpactRouteAlternative()
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"rejectTrafficImpactRouteAlternative()");

   presCtrlInterface.rejectTrafficImpactRouteAlternative();
}

void NavMiddleware::startRouteListInfo(unsigned int indexAlternative, RouteListConfiguration routeListConfiguration)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"startRouteListInfo(%u, %u)", indexAlternative, routeListConfiguration);

   presCtrlInterface.startRouteListInfo(indexAlternative, routeListConfiguration);
}

void NavMiddleware::stopRouteListInfo(unsigned int indexAlternative)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"stopRouteListInfo(%u)", indexAlternative);

   presCtrlInterface.stopRouteListInfo(indexAlternative);
}

ValidValue<RequestId> NavMiddleware::requestRouteList(unsigned int indexAlternative, unsigned int startIndex, unsigned int numOfElements)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"requestRouteList(%u, %u, %u)", indexAlternative, startIndex, numOfElements);

   return presCtrlInterface.requestRouteList(indexAlternative, startIndex, numOfElements);
}

const RouteListInfos& NavMiddleware::getRouteListInfos() const
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"getRouteListInfos()");
   LOG_INTERFACE_DEBUG(route, "\n ############ Route list info ############ - %s",
                       presCtrlInterface.getRouteListInfos().toString().c_str());

   return presCtrlInterface.getRouteListInfos();
}

void NavMiddleware::avoidRouteListElements(unsigned int indexAlternative, unsigned int startIndex, unsigned int endIndex)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"avoidRouteListElements(%u, %u, %u)", indexAlternative, startIndex, endIndex);

   return presCtrlInterface.avoidRouteListElements(indexAlternative, startIndex, endIndex);
}

void NavMiddleware::cancelAvoidRouteListElements()
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"cancelAvoidRouteListElements()");

   return presCtrlInterface.cancelAvoidRouteListElements();
}

void NavMiddleware::requestJamAheadList()
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"requestJamAheadList()");

   presCtrlInterface.requestJamAheadList();
}

void NavMiddleware::releaseJamAheadList()
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"releaseJamAheadList()");

   presCtrlInterface.releaseJamAheadList();
}

void NavMiddleware::selectJamAheadListEntry(unsigned int index)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"selectJamAheadListEntry(%u)", index);

   presCtrlInterface.selectJamAheadListEntry(index);
}

const JamAheadListInfo& NavMiddleware::getJamAheadListInfo() const
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"getJamAheadListInfo()");
   LOG_INTERFACE_DEBUG(route, "\n ############ Jam ahead list info ############ - %s",
                       presCtrlInterface.getJamAheadListInfo().toString().c_str());

   return presCtrlInterface.getJamAheadListInfo();
}

void NavMiddleware::cancelJamAhead()
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"cancelJamAhead()");

   presCtrlInterface.cancelJamAhead();
}

void NavMiddleware::startStoredRouteCalculation(const ::std::vector<DestinationMemoryWaypoint>& waypointList)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"startStoredRouteCalculation()");

   presCtrlInterface.startStoredRouteCalculation(waypointList);
}

void NavMiddleware::startStoredRouteCalculationWithDestinationMemoryId(DestinationMemoryRoute::Id routeId, uint32_t numberOfWaypointsToSkip)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"startStoredRouteCalculationWithDestinationMemoryId(routeId = %llu, numberOfWaypointsToSkip = %u)", routeId, numberOfWaypointsToSkip);

   presCtrlInterface.startStoredRouteCalculationWithDestinationMemoryId(routeId, numberOfWaypointsToSkip);
}

void NavMiddleware::startStoredRouteCalculationForPlanning(DestinationMemoryRoute::Id routeId)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"startStoredRouteCalculationForPlanning()");

   presCtrlInterface.startStoredRouteCalculationForPlanning(routeId);
}

void NavMiddleware::startStoredRouteCalculationWithFtsSelection()
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"startStoredRouteCalculationWithFtsSelection()");

   presCtrlInterface.startStoredRouteCalculationWithFtsSelection();
}

void NavMiddleware::startRouteCalculationForPlanning()
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"startRouteCalculationForPlanning()");

   presCtrlInterface.startRouteCalculationForPlanning();
}

void NavMiddleware::startTourCalculation(const Tour& tour)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"startTourCalculation()");
   LOG_INTERFACE_DEBUG(route, "\n ############ Tour Info ############ - %s", tour.toString().c_str());

   presCtrlInterface.startTourCalculation(tour);
}

void NavMiddleware::removeUnusedRoutes()
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"DEPRECATED: removeUnusedRoutes()");

   presCtrlInterface.removeUnusedRoutes();
}

void NavMiddleware::requestRouteCountryAndStateInfo(unsigned int indexAlternative)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"requestRouteCountryAndStateInfo()");
   presCtrlInterface.requestRouteCountryAndStateInfo(indexAlternative);
}

const RouteCountryAndStateInfos& NavMiddleware::getRouteCountryAndStateInfos() const
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"getRouteCountryAndStateInfos()");
   LOG_INTERFACE_DEBUG(route, "\n ############ Country and State Info  ############ - %s",
                       presCtrlInterface.getRouteCountryAndStateInfos().toString().c_str());

   return presCtrlInterface.getRouteCountryAndStateInfos();
}

void NavMiddleware::startRangePredictionAtWaypointOrDestination(unsigned int indexAlternative, unsigned int waypointIndex)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"startRangePredictionAtWaypointOrDestination()");

   presCtrlInterface.startRangePredictionAtWaypointOrDestination(indexAlternative, waypointIndex);
}

void NavMiddleware::stopRangePredictionAtWaypointOrDestination()
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"stopRangePredictionAtWaypointOrDestination()");

   presCtrlInterface.stopRangePredictionAtWaypointOrDestination();
}

const RangePredictionInfos& NavMiddleware::getRangePredictionInfos() const
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"getRangePredictionInfos()");
   LOG_INTERFACE_DEBUG(route, "\n ############ Range Prediction Info  ############ - %s",
                       presCtrlInterface.getRangePredictionInfos().toString().c_str());

   return presCtrlInterface.getRangePredictionInfos();
}

void NavMiddleware::startAutomaticChargingStationDetection()
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"startAutomaticChargingStationDetection()");

   presCtrlInterface.startAutomaticChargingStationDetection();
}

const AutomaticChargingStationDetectionInfo& NavMiddleware::getAutomaticChargingStationDetectionInfo() const
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"getAutomaticChargingStationDetectionInfo()");
   LOG_INTERFACE_DEBUG(route, "\n ############ Automatic Charging Stations Detection Infos ############ - %s",
                       presCtrlInterface.getAutomaticChargingStationDetectionInfo().toString().c_str());
   return presCtrlInterface.getAutomaticChargingStationDetectionInfo();
}

void NavMiddleware::cancelAutomaticChargingStationDetection()
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"cancelAutomaticChargingStationDetection()");

   presCtrlInterface.cancelAutomaticChargingStationDetection();
}

void NavMiddleware::insertChargingStationsToRoute(RouteCalculationMode routeCalculationMode)
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"insertChargingStationsToRoute(%s)", toString(routeCalculationMode).c_str());

   presCtrlInterface.insertChargingStationsToRoute(routeCalculationMode);
}

void NavMiddleware::removeChargingStationsFromRoute()
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"removeChargingStationsFromRoute()");

   presCtrlInterface.removeChargingStationsFromRoute();
}

const ::navmiddleware::ShapePointsAvailabilityInfo& NavMiddleware::getShapePointsAvailabilityInfo() const
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"getShapePointsAvailabilityInfo()");

   LOG_INTERFACE_DEBUG(route, "\n ############ Shape Points Availability Info ############ - %s",
                       presCtrlInterface.getShapePointsAvailabilityInfo().toString().c_str());

   return presCtrlInterface.getShapePointsAvailabilityInfo();
}

void NavMiddleware::requestShapePoints()
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"requestShapePoints()");

   presCtrlInterface.requestShapePoints();
}

const ::navmiddleware::ShapePointsInfo& NavMiddleware::getShapePointsInfo() const
{
   LOG_INTERFACE(route, MESSAGE_TAG_IN"getShapePointsInfo()");

   LOG_INTERFACE_DEBUG(route, "\n ############ Shape Points Info ############ - %s",
                       presCtrlInterface.getShapePointsInfo().toString().c_str());

   return presCtrlInterface.getShapePointsInfo();
}

void NavMiddleware::setBatteryInfos(const BatteryInfos& batteryInfos)
{
   LOG_INTERFACE(electricvehicle, MESSAGE_TAG_IN"setBatteryInfos()");
   LOG_INTERFACE_DEBUG(electricvehicle, "\n Battery Infos - %s", batteryInfos.toString().c_str());
   presCtrlInterface.setBatteryInfos(batteryInfos);
}

void NavMiddleware::setChargingInfos(const ChargingInfos& chargingInfos)
{
   LOG_INTERFACE(electricvehicle, MESSAGE_TAG_IN"setChargingInfos()");
   LOG_INTERFACE_DEBUG(electricvehicle, "\n Charging Infos - %s", chargingInfos.toString().c_str());
   presCtrlInterface.setChargingInfos(chargingInfos);
}

void NavMiddleware::setDriveMode(DriveMode driveMode)
{
   LOG_INTERFACE(electricvehicle, MESSAGE_TAG_IN"setDriveMode(%s(%d))", ::navmiddleware::toString(driveMode).c_str(), driveMode);
   presCtrlInterface.setDriveMode(driveMode);
}

const ::navmiddleware::ElectricVehicleInfos& NavMiddleware::getElectricVehicleInfos() const
{
   LOG_INTERFACE(electricvehicle, MESSAGE_TAG_IN"getElectricVehicleInfos()");
   LOG_INTERFACE_DEBUG(electricvehicle, "\n ############ Electric Vehicle Infos ############ - %s",
                       presCtrlInterface.getElectricVehicleInfos().toString().c_str());
   return presCtrlInterface.getElectricVehicleInfos();
}

void NavMiddleware::storeDataBeforeShutDown()
{
   LOG_INTERFACE(electricvehicle, MESSAGE_TAG_IN"storeDataBeforeShutDown()");
   presCtrlInterface.storeDataBeforeShutDown();
}

settings::GuidanceSettings NavMiddleware::getGuidanceSettings() const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"getGuidanceSettings()");

   return presCtrlInterface.getGuidanceSettings();
}

void NavMiddleware::updateGuidanceSettings(const settings::GuidanceSettings& settings)
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"updateGuidanceSettings()");

   presCtrlInterface.updateGuidanceSettings(settings);
}

settings::SafetyGuidanceSettings NavMiddleware::getSafetyGuidanceSettings() const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"getSafetyGuidanceSettings()");

   return presCtrlInterface.getSafetyGuidanceSettings();
}

void NavMiddleware::updateSafetyGuidanceSettings(const settings::SafetyGuidanceSettings& settings)
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"updateSafetyGuidanceSettings()");

   presCtrlInterface.updateSafetyGuidanceSettings(settings);
}

settings::SafetyCameraSettings NavMiddleware::getSafetyCameraSettings() const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"getSafetyCameraSettings()");

   return presCtrlInterface.getSafetyCameraSettings();
}

void NavMiddleware::updateSafetyCameraSettings(const settings::SafetyCameraSettings& settings)
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"updateSafetyCameraSettings()");

   presCtrlInterface.updateSafetyCameraSettings(settings);
}

settings::EvBatteryStatusWarningSettings NavMiddleware::getEvBatteryStatusWarningSettings() const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"getEvBatteryStatusWarningSettings()");

   return presCtrlInterface.getEvBatteryStatusWarningSettings();
}

void NavMiddleware::updateEvBatteryStatusWarningSettings(const settings::EvBatteryStatusWarningSettings& settings)
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"updateEvBatteryStatusWarningSettings()");
   presCtrlInterface.updateEvBatteryStatusWarningSettings(settings);
}

settings::EvMiscellaneousSettings NavMiddleware::getEvMiscellaneousSettings() const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"getEvMiscellaneousSettings()");

   return presCtrlInterface.getEvMiscellaneousSettings();
}

void NavMiddleware::updateEvMiscellaneousSettings(const settings::EvMiscellaneousSettings& settings)
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"updateEvMiscellaneousSettings()");
   presCtrlInterface.updateEvMiscellaneousSettings(settings);
}

void NavMiddleware::setEcoNavSwitchState(EcoNavSwitchState ecoNavSwitchState)
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"setEcoNavSwitchState: %s",toString(ecoNavSwitchState).c_str());
   presCtrlInterface.setEcoNavSwitchState(ecoNavSwitchState);
}

EcoNavSwitchState NavMiddleware::getEcoNavSwitchState() const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"getEcoNavSwitchState");
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"\n ############ GetEcoNavSwitchState ############ - %d", presCtrlInterface.getEcoNavSwitchState());
   return presCtrlInterface.getEcoNavSwitchState();
}

EcoNavState NavMiddleware::getEcoNavStatus() const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"getEcoNavStatus");
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"\n ############ GetEcoNavStatus ############ - %d", presCtrlInterface.getEcoNavStatus());
   return presCtrlInterface.getEcoNavStatus();
}

void NavMiddleware::setUrbanZevDrivingOption(UrbanZevDrivingOption urbanZevDrivingOption)
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"setUrbanZevDrivingOption: %s",toString(urbanZevDrivingOption).c_str());
   presCtrlInterface.setUrbanZevDrivingOption(urbanZevDrivingOption);
}

UrbanZevDrivingOption NavMiddleware::getUrbanZevDrivingOptionStatus() const
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"getUrbanZevDrivingOptionStatus");
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"\n ############ GetUrbanZevDrivingOptionStatus ############ - %d", presCtrlInterface.getUrbanZevDrivingOptionStatus());
   return presCtrlInterface.getUrbanZevDrivingOptionStatus();
}

settings::MapSettings NavMiddleware::getMapSettings() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"getMapSettings()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Get Map Settings ############ - %s",
                       presCtrlInterface.getMapSettings().toString().c_str());

   return presCtrlInterface.getMapSettings();
}

void NavMiddleware::updateMapSettings(const settings::MapSettings& settings)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"updateMapSettings()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Update Map Settings ############ - %s",
                       settings.toString().c_str());

   presCtrlInterface.updateMapSettings(settings);
}

settings::MapTextFilterSettings NavMiddleware::getMapTextFilterSettings() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"getMapTextFilterSettings()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Get Map Text Filter Settings ############ - %s",
                       presCtrlInterface.getMapTextFilterSettings().toString().c_str());

   return presCtrlInterface.getMapTextFilterSettings();
}

void NavMiddleware::updateMapTextFilterSettings(const settings::MapTextFilterSettings& settings)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"updateMapTextFilterSettings()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Update Map Text Filter Settings ############ - %s",
                       settings.toString().c_str());

   presCtrlInterface.updateMapTextFilterSettings(settings);
}

void NavMiddleware::updateMapDynamicIconSettings(const settings::MapDynamicIconSettings& settings)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"updateMapDynamicIconSettings()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Update Map Dynamic Icon Settings ############ - %s",
                       settings.toString().c_str());

   presCtrlInterface.updateMapDynamicIconSettings(settings);
}

settings::MapDynamicIconSettings NavMiddleware::getMapDynamicIconSettings() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"getMapDynamicIconSettings()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Get Map DynamicIcon Settings ############ - %s",
                       presCtrlInterface.getMapDynamicIconSettings().toString().c_str());

   return presCtrlInterface.getMapDynamicIconSettings();
}

void NavMiddleware::resetMapDynamicIconSettings()
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"resetMapDynamicIconSettings()");
   presCtrlInterface.resetMapDynamicIconSettings();
}

void NavMiddleware::updateMapCarsorSettings(const settings::MapCarsorSettings& settings)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"updateMapCarsorSettings()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Update Map Carsor Settings ############ - %s",
                       settings.toString().c_str());

   presCtrlInterface.updateMapCarsorSettings(settings);
}

settings::MapCarsorSettings NavMiddleware::getMapCarsorSettings() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"getMapCarsorSettings()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Get Map Carsor Settings ############ - %s",
                       presCtrlInterface.getMapCarsorSettings().toString().c_str());

   return presCtrlInterface.getMapCarsorSettings();
}

settings::TrafficSettings NavMiddleware::getTrafficSettings() const
{
   LOG_INTERFACE(trafficmessage, MESSAGE_TAG_IN"getTrafficSettings()");
   LOG_INTERFACE_DEBUG(trafficmessage, "\n ############ getTrafficSettings ############ - %s",
                       presCtrlInterface.getTrafficSettings().toString().c_str());

   return presCtrlInterface.getTrafficSettings();
}

void NavMiddleware::updateTrafficSettings(const settings::TrafficSettings& settings) const
{
   LOG_INTERFACE(trafficmessage, MESSAGE_TAG_IN"updateTrafficSettings()");
   LOG_INTERFACE_DEBUG(trafficmessage, "\n ############ updateTrafficSettings ############ - %s",
                       settings.toString().c_str());

   presCtrlInterface.updateTrafficSettings(settings);
}

void NavMiddleware::highlightRoute(MapViewId mapViewId, unsigned int routeIndex)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"highlightRoute()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, routeIndex = %u", toString(mapViewId).c_str(), routeIndex);

   presCtrlInterface.highlightRoute(mapViewId, routeIndex);
}

const TouchEventInfo& NavMiddleware::getTouchEventInfo() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"getTouchEventInfo()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Touch event info ############ - %s",
                       presCtrlInterface.getTouchEventInfo().toString().c_str());

   return presCtrlInterface.getTouchEventInfo();
}

void NavMiddleware::selectHighlightedRoute(MapViewId mapViewId)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"selectHighlightedRoute()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s", toString(mapViewId).c_str());
   presCtrlInterface.selectHighlightedRoute(mapViewId);
}

void NavMiddleware::selectActiveRoute(MapViewId mapViewId)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"selectActiveRoute()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s", toString(mapViewId).c_str());
   presCtrlInterface.selectActiveRoute(mapViewId);
}

void NavMiddleware::setMapCameraAndViewModes(MapViewGroup mapViewGroup, const ::std::vector< ViewModeConfiguration >& viewModeConfiguration)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"setMapCameraAndViewModes()");
   LOG_INTERFACE_DEBUG(map, "mapViewGroup = %s, viewModeConfiguration = %s", toString(mapViewGroup).c_str(), toString<ViewModeConfiguration>(viewModeConfiguration).c_str());
   presCtrlInterface.setMapCameraAndViewModes(mapViewGroup, viewModeConfiguration);
}

void NavMiddleware::showWeatherOverviewMap(MapViewId mapViewId, CameraAnimation cameraAnimation)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"showWeatherOverviewMap()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, cameraAnimation = %s", toString(mapViewId).c_str(), toString(cameraAnimation).c_str());
   presCtrlInterface.showWeatherOverviewMap(mapViewId, cameraAnimation);
}

void NavMiddleware::showTrafficOverviewMap(MapViewId mapViewId, const ::std::vector<uint32_t>& trafficMessageID,
      const ValidValue<int32_t>& mapScale, CameraAnimation cameraAnimation)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"showTrafficOverviewMap()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, size of trafficMessageID = %d, mapScale = %d, cameraAnimation = %s",
                       toString(mapViewId).c_str(), trafficMessageID.size(), mapScale.getValueOr(0), toString(cameraAnimation).c_str());
   presCtrlInterface.showTrafficOverviewMap(mapViewId, trafficMessageID, mapScale, cameraAnimation);
}

const NominalScaleInfos& NavMiddleware::getNominalScaleInfos() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"getNominalScaleInfos()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Nominal scale info ############ - %s",
                       presCtrlInterface.getNominalScaleInfos().toString().c_str());

   return presCtrlInterface.getNominalScaleInfos();
}

void NavMiddleware::centerMapToGeoPosition(
   MapViewId mapViewId,
   const GeoCoordinateDegree& geoCoordinate,
   ValidValue<int32_t> mapScale,
   CameraAnimation cameraAnimation)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"centerMapToGeoPosition()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, cameraAnimation = %s, (latitude = %f, longitude = %f)", toString(mapViewId).c_str(), toString(cameraAnimation).c_str(),
                       geoCoordinate.getLatitude(), geoCoordinate.getLongitude());
   presCtrlInterface.centerMapToGeoPosition(mapViewId, geoCoordinate, mapScale, cameraAnimation);
}

void NavMiddleware::showSetLocationInMap(MapViewId mapViewId,int32_t mapScale,CameraAnimation cameraAnimation)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"showSetLocationInMap()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, mapScale = %d, cameraAnimation = %s", toString(mapViewId).c_str(), mapScale, toString(cameraAnimation).c_str());
   presCtrlInterface.showSetLocationInMap(mapViewId, mapScale, cameraAnimation);
}

void NavMiddleware::centerMapToViewCoordinate(
   MapViewId mapViewId,
   uint16_t posX,
   uint16_t posY,
   CameraAnimation cameraAnimation)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"centerMapToViewCoordinate()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, (posX = %d, posY = %d), cameraAnimation = %s", toString(mapViewId).c_str(), posX, posY, toString(cameraAnimation).c_str());
   presCtrlInterface.centerMapToViewCoordinate(mapViewId, posX, posY, cameraAnimation);
}

void NavMiddleware::moveMapCenterPoint(MapViewId mapViewId, int16_t posX, int16_t posY, CameraAnimation cameraAnimation)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"moveMapCenterPoint()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, posX = %d, posY = %d, cameraAnimation = %s",
                       toString(mapViewId).c_str(), posX, posY, ::navmiddleware::toString(cameraAnimation).c_str());
   presCtrlInterface.moveMapCenterPoint(mapViewId, posX, posY, cameraAnimation);
}

const MapMultiViewInfo& NavMiddleware::getMapMultiViewInfo() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"getMapMultiViewInfo()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Map multi view info ############ - %s",
                       presCtrlInterface.getMapMultiViewInfo().toString().c_str());

   return presCtrlInterface.getMapMultiViewInfo();
}

void NavMiddleware::storeEntry(DestinationMemoryEntryCategory category)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"storeEntry(%d)", category);
   presCtrlInterface.storeEntry(category);
}

void NavMiddleware::storeEntry(
   DestinationMemoryEntry::Id entryId,
   DestinationMemoryEntryCategory category)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"storeEntry(%llu, %d)", entryId, category);
   presCtrlInterface.storeEntry(entryId, category);
}

void NavMiddleware::storeEntry(
   DestinationMemoryEntryCategory category,
   const DestinationMemoryEntryAttributes& entryAttributes)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"storeEntry(%d, %s)", category, entryAttributes.toString().c_str());
   presCtrlInterface.storeEntry(category, entryAttributes);
}

DestinationMemoryStoreEntryStatus NavMiddleware::getStoreEntryStatus() const
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"getStoreEntryStatus()");
   LOG_INTERFACE_DEBUG(destinationmemory, "\n ############ StoreEntryStatus info ############ - %s",
                       presCtrlInterface.getStoreEntryStatus().toString().c_str());
   return presCtrlInterface.getStoreEntryStatus();
}

void NavMiddleware::requestEntryDetails(DestinationMemoryEntry::Id id)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"requestEntryDetails()");
   presCtrlInterface.requestEntryDetails(id);
}

const DestinationMemoryEntry& NavMiddleware::getEntryDetails() const
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"getEntryDetails()");
   return presCtrlInterface.getEntryDetails();
}

void NavMiddleware::storeEntries(DestinationMemoryEntryCategory category, const std::vector<DestinationMemoryEntry>& entries)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"storeEntries()");
   presCtrlInterface.storeEntries(category, entries);
}

const DestinationMemoryStoreEntriesStatus& NavMiddleware::getStoreEntriesStatus() const
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"getStoreEntriesStatus()");
   return presCtrlInterface.getStoreEntriesStatus();
}

void NavMiddleware::requestEntryList(
   DestinationMemoryEntryCategory category,
   const ValidValue< DestinationMemoryAttributesGroup>& subGroup,
   uint32_t startIndex,
   int32_t numElements,
   DestinationMemoryEntry::SortingAlgorithm sortingAlgorithm,
   const ::std::string& freeTextSearchString)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"requestEntryList()");
   presCtrlInterface.requestEntryList(category, subGroup, startIndex, numElements, sortingAlgorithm, freeTextSearchString);
}

const DestinationMemoryEntryList& NavMiddleware::getEntryList() const
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"getEntryList()");
   LOG_INTERFACE_DEBUG(destinationmemory, "\n ############ EntryList info ############ - %s",
                       presCtrlInterface.getEntryList().toString().c_str());

   return presCtrlInterface.getEntryList();
}

void NavMiddleware::replaceEntryLocation(DestinationMemoryEntry::Id entryId)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"replaceEntryLocation(%llu)", entryId);
   presCtrlInterface.replaceEntryLocation(entryId);
}

const DestinationMemoryStatus& NavMiddleware::getDestinationMemoryStatus() const
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"getDestinationMemoryStatus()");
   LOG_INTERFACE_DEBUG(destinationmemory, "\n ############ DestinationMemoryStatus info ############ - %s",
                       presCtrlInterface.getDestinationMemoryStatus().toString().c_str());
   return(presCtrlInterface.getDestinationMemoryStatus());
}

void NavMiddleware::modifyEntryAttributes(
   DestinationMemoryEntry::Id entryId,
   const DestinationMemoryEntryAttributes& entryAttributes)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"modifyEntryAttributes()");
   presCtrlInterface.modifyEntryAttributes(entryId, entryAttributes);
}

void NavMiddleware::setEntryPosition(DestinationMemoryEntry::Id entryId, uint64_t position)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"setEntryPosition(%llu, %llu)", entryId, position);
   presCtrlInterface.setEntryPosition(entryId, position);
}

ValidValue< ::std::string> NavMiddleware::getAddressbookMediaPath(DestinationMemoryEntryAttributes::Sound sound) const
{
   LOG_INTERFACE(destinationmemory, "getAddressbookMediaPath");
   return presCtrlInterface.getAddressbookMediaPath(sound);
}

void NavMiddleware::removeEntry(const ::std::vector< DestinationMemoryEntry::Id >& entryId)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"removeEntry()");
   presCtrlInterface.removeEntry(entryId);
}

void NavMiddleware::removeEntryByPoiId(const PoiId& poiId)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"removeEntryByPoiId(%s)", poiId.toString().c_str());
   presCtrlInterface.removeEntryByPoiId(poiId);
}

void NavMiddleware::removeAllEntries(DestinationMemoryEntryCategory category)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"removeAllEntries()");
   presCtrlInterface.removeAllEntries(category);
}

const DestinationMemoryAttributesGroupList& NavMiddleware::getAddressBookUsedGroups() const
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"getAddressBookUsedGroups()");
   return presCtrlInterface.getAddressBookUsedGroups();
}

void NavMiddleware::requestLastDestinationList(
   uint32_t startIndex,
   int32_t numElements,
   DestinationMemoryEntry::SortingAlgorithm sortingAlgorithm,
   DestinationMemoryItemList::Filter requestFilter,
   const ::std::string& freeTextSearchString)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"requestLastDestinationList(start index: %d, numElements: %d, "
                 "sortingAlgorithm: %d, requestFilter: %d, freeTextSearchString: %s)",
                 startIndex, numElements, sortingAlgorithm, requestFilter, freeTextSearchString.c_str());
   presCtrlInterface.requestLastDestinationList(startIndex, numElements, sortingAlgorithm, requestFilter, freeTextSearchString.c_str());
}

const DestinationMemoryItemList& NavMiddleware::getLastDestinationList() const
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"getLastDestinationList()");
   LOG_INTERFACE_DEBUG(destinationmemory, "\n ############ DM ItemList info ############ - %s",
                       presCtrlInterface.getLastDestinationList().toString().c_str());
   return presCtrlInterface.getLastDestinationList();
}

void NavMiddleware::updateLastDestinationHistoryList(const ::std::vector< DestinationMemoryLastDestinationHistory >& lastDestinationHistoryList)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"updateLastDestinationHistoryList()");
   presCtrlInterface.updateLastDestinationHistoryList(lastDestinationHistoryList);
}

const DestinationMemoryUpdatedLastDestinationList& NavMiddleware::getUpdatedLastDestinationList() const
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"getUpdatedLastDestinationList()");
   return presCtrlInterface.getUpdatedLastDestinationList();
}

void NavMiddleware::insertLastDestinationList(const ::std::vector< DestinationMemoryLastDestination >& lastDestinationList)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"insertLastDestinationList()");
   presCtrlInterface.insertLastDestinationList(lastDestinationList);
}

const DestinationMemoryUpdatedLastDestinationList& NavMiddleware::getInsertedLastDestinationList() const
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"getUpdatedLastDestinationList()");
   return presCtrlInterface.getInsertedLastDestinationList();
}

uint32_t NavMiddleware::getLastGuidedRouteNumberOfWaypointsReached() const
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"getLastGuidedRouteNumberOfWaypointsReached()");
   return presCtrlInterface.getLastGuidedRouteNumberOfWaypointsReached();
}

void NavMiddleware::removeLastDestinationListItems(
   const ::std::vector< DestinationMemoryItem::IdElement >& itemIdList)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"removeLastDestinationListItems()");
   presCtrlInterface.removeLastDestinationListItems(itemIdList);
}

void NavMiddleware::removeAllLastDestinationListItems(DestinationMemoryItemList::Filter requestFilter)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"removeAllLastDestinationListItems(requestFilter = %d)", requestFilter);
   presCtrlInterface.removeAllLastDestinationListItems(requestFilter);
}

void NavMiddleware::requestLastDeparture()
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"requestLastDeparture()");
   presCtrlInterface.requestLastDeparture();
}

const ::navmiddleware::DepartureInfo& NavMiddleware::getLastDeparture() const
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"getLastDeparture()");
   return presCtrlInterface.getLastDeparture();
}

void NavMiddleware::removeLastDeparture()
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"removeLastDeparture()");
   presCtrlInterface.removeLastDeparture();
}

void NavMiddleware::storeRoute(::std::string name)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"storeRoute()");
   return presCtrlInterface.storeRoute(name);
}

void NavMiddleware::storeRoute(DestinationMemoryRoute::Id routeId, DestinationMemoryRoute::Category category)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"storeRoute(routeId: %llu, category: %d)", routeId, category);
   return presCtrlInterface.storeRoute(routeId, category);
}

void NavMiddleware::storeRoute(const ::std::string& name, const Tour& tour, DestinationMemoryRoute::Category category)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"storeRoute(name: %s, category: %d)", name.c_str(), category);
   return presCtrlInterface.storeRoute(name, tour, category);
}

const DestinationMemoryStoreRouteStatus& NavMiddleware::getStoreRouteStatus() const
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"getStoreRouteStatus()");
   LOG_INTERFACE_DEBUG(destinationmemory, "\n ############ StoreRouteStatus info ############ - %s",
                       presCtrlInterface.getStoreRouteStatus().toString().c_str());
   return presCtrlInterface.getStoreRouteStatus();
}

void NavMiddleware::requestStoredRouteDetails(DestinationMemoryRoute::Id routeId)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"requestStoredRouteDetails()");
   presCtrlInterface.requestStoredRouteDetails(routeId);
}

const DestinationMemoryRoute& NavMiddleware::getStoredRouteInfo() const
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"getStoredRouteInfo()");
   return presCtrlInterface.getStoredRouteInfo();
}

void NavMiddleware::requestStoredRoutes(uint32_t startIndex, int32_t numElements, DestinationMemoryEntry::SortingAlgorithm sortingAlgorithm, const ::std::string& freeTextSearchString)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"requestStoredRoutes()");
   presCtrlInterface.requestStoredRoutes(startIndex, numElements, sortingAlgorithm, freeTextSearchString);
}

const DestinationMemoryRouteList& NavMiddleware::getStoredRoutes() const
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"getStoredRoutes()");
   return presCtrlInterface.getStoredRoutes();
}

void NavMiddleware::removeRoute(const ::std::vector< DestinationMemoryRoute::Id >& routeIdList)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"removeRoute(routeIdList.size(): %lu)", routeIdList.size());
   presCtrlInterface.removeRoute(routeIdList);
}

void NavMiddleware::removeAllRoutes()
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"removeAllRoutes()");
   presCtrlInterface.removeAllRoutes();
}

void NavMiddleware::replaceStoredRoute(DestinationMemoryRoute::Id routeId)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"replaceStoredRoute()");
   return presCtrlInterface.replaceStoredRoute(routeId);
}

void NavMiddleware::renameStoredRoute(DestinationMemoryRoute::Id routeId, const ::std::string& name)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"renameStoredRoute()");
   return presCtrlInterface.renameStoredRoute(routeId, name);
}

void NavMiddleware::setIncludeGooglePlacesInListRequests(bool includeGooglePlaces)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"setIncludeGooglePlacesInListRequests(includeGooglePlaces: %d)", includeGooglePlaces);
   presCtrlInterface.setIncludeGooglePlacesInListRequests(includeGooglePlaces);
}

void NavMiddleware::requestRefreshForExpiredDestinationEntry(DestinationMemoryEntry::Id entryId)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"requestRefreshForExpiredDestinationEntry(entryId = %llu)", entryId);

   presCtrlInterface.requestRefreshForExpiredDestinationEntry(entryId);
}

void NavMiddleware::requestOnboardMatchingForExpiredDestinationEntry(DestinationMemoryEntry::Id entryId)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"requestOnboardMatchingForExpiredDestinationEntry(entryId = %llu)", entryId);

   presCtrlInterface.requestOnboardMatchingForExpiredDestinationEntry(entryId);
}

EntrySelectionStatus NavMiddleware::getDestinationSelectionStatus() const
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"getDestinationSelectionStatus()");
   LOG_INTERFACE_DEBUG(destinationmemory, "\n ############ DestinationSelectionStatus info ############ - %s",
                       toString(presCtrlInterface.getDestinationSelectionStatus()).c_str());
   return presCtrlInterface.getDestinationSelectionStatus();
}

void NavMiddleware::clearVoiceTags()
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"clearVoiceTags()");
   presCtrlInterface.clearVoiceTags();
}

//TODO(pch8kor): Provide toString() in EvAutoSaveChargingStationSettings to log state of settings object.
settings::EvAutoSaveChargingStationSettings NavMiddleware::getEvAutoSaveChargingStationSettings() const
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"getEvAutoSaveChargingStationSettings()");
   return presCtrlInterface.getEvAutoSaveChargingStationSettings();
}

void NavMiddleware::updateEvAutoSaveChargingStationSettings(const settings::EvAutoSaveChargingStationSettings& settings)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"updateEvAutoSaveChargingStationSettings()");
   presCtrlInterface.updateEvAutoSaveChargingStationSettings(settings);
}

void NavMiddleware::requestLastDestinationsHistory(
   uint32_t startIndex,
   int32_t numElements,
   DestinationMemoryEntry::SortingAlgorithm sortingAlgorithm,
   DestinationMemoryItemList::Filter requestFilter)
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"requestLastDestinationsHistory()");
   presCtrlInterface.requestLastDestinationsHistory(startIndex, numElements, sortingAlgorithm, requestFilter);
}

const DestinationMemoryLastDestinationHistoryList& NavMiddleware::getLastDestinationsHistory() const
{
   LOG_INTERFACE(destinationmemory, MESSAGE_TAG_IN"getLastDestinationsHistory()");
   LOG_INTERFACE_DEBUG(destinationmemory, "\n ############ DM DestinationList info ############ - %s",
                       presCtrlInterface.getLastDestinationsHistory().toString().c_str());
   return presCtrlInterface.getLastDestinationsHistory();
}

settings::MapDetailTriggerSettings NavMiddleware::getMapDetailTriggerSettings() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"getMapDetailTriggerSettings()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Get Map Detail Trigger Settings ############ - %s",
                       presCtrlInterface.getMapDetailTriggerSettings().toString().c_str());

   return presCtrlInterface.getMapDetailTriggerSettings();
}

void NavMiddleware::updateMapDetailTriggerSettings(const settings::MapDetailTriggerSettings& settings)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"updateMapDetailTriggerSettings()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Update Map Detail Trigger Settings ############ - %s",
                       settings.toString().c_str());

   presCtrlInterface.updateMapDetailTriggerSettings(settings);
}

settings::ChangeMapViewSettings NavMiddleware::getChangeMapViewSettings() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"getChangeMapViewSettings()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Get Change Map View ############ - %s",
                       presCtrlInterface.getChangeMapViewSettings().toString().c_str());

   return presCtrlInterface.getChangeMapViewSettings();
}

void NavMiddleware::updateChangeMapViewSettings(const settings::ChangeMapViewSettings& settings)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"updateChangeMapViewSettings()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Update Change Map View ############ - %s",
                       settings.toString().c_str());

   presCtrlInterface.updateChangeMapViewSettings(settings);
}

settings::MapViewSettings NavMiddleware::getMapViewSettings() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"getMapViewSettings()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Get Map View Settings ############ - %s",
                       presCtrlInterface.getMapViewSettings().toString().c_str());

   return presCtrlInterface.getMapViewSettings();
}

void NavMiddleware::updateMapViewSettings(const settings::MapViewSettings& settings)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"updateMapViewSettings()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Update Map View Settings ############ - %s",
                       settings.toString().c_str());

   presCtrlInterface.updateMapViewSettings(settings);
}

settings::MapWeatherSettings NavMiddleware::getMapWeatherSettings() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"getMapWeatherSettings()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Get Map Weather Settings ############ - %s",
                       presCtrlInterface.getMapWeatherSettings().toString().c_str());

   return presCtrlInterface.getMapWeatherSettings();
}

void NavMiddleware::updateMapWeatherSettings(const settings::MapWeatherSettings& settings)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"updateMapWeatherSettings()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Update Map Weather Settings ############ - %s",
                       settings.toString().c_str());

   presCtrlInterface.updateMapWeatherSettings(settings);
}

void NavMiddleware::updateMapEvSettings(const settings::MapEvSettings& settings)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"updateMapEvSettings()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Update Map Ev Settings ############ - %s",
                       settings.toString().c_str());

   presCtrlInterface.updateMapEvSettings(settings);
}

settings::MapEvSettings NavMiddleware::getMapEvSettings() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"getMapEvSettings()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Get Map Ev Settings ############ - %s",
                       presCtrlInterface.getMapEvSettings().toString().c_str());

   return presCtrlInterface.getMapEvSettings();
}

void NavMiddleware::showWeatherInMap(MapViewId mapViewId)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"showWeatherInMap()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s", toString(mapViewId).c_str());

   presCtrlInterface.showWeatherInMap(mapViewId);
}

void NavMiddleware::configureTouch(MapViewId mapViewId,
                                   TouchActions touchActions,
                                   PitchRange pitchRange,
                                   MapScaleRange mapScaleRange)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"configureTouch()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, touchActions = %s, pitchRange = %s, mapScaleRange = %s",
                       toString(mapViewId).c_str(), touchActions.toString().c_str(), pitchRange.toString().c_str(), mapScaleRange.toString().c_str());

   presCtrlInterface.configureTouch(mapViewId, touchActions, pitchRange, mapScaleRange);
}

void NavMiddleware::showLocationsInMap(MapViewId mapViewId,
                                       const ::std::vector< GeoCoordinateDegree >& geoCoordinates,
                                       const ::std::vector< GeoCoordinateDegree >& optionalOverviewGeoCoordinates,
                                       const MapIconInfo& mapIconInfo,
                                       const ValidValue<int32_t>& mapScale,
                                       CameraAnimation cameraAnimation)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"showLocationsInMap()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, geoCoordinates = %s, "
                       "optionalOverviewGeoCoordinates = %s,"
                       "mapIconInfo = %s, mapScale = %d, cameraAnimation = %s)",
                       toString(mapViewId).c_str(),
                       toString<GeoCoordinateDegree>(geoCoordinates).c_str(),
                       toString<GeoCoordinateDegree>(optionalOverviewGeoCoordinates).c_str(),
                       mapIconInfo.toString().c_str(),
                       mapScale.getValueOr(0),
                       toString(cameraAnimation).c_str());

   presCtrlInterface.showLocationsInMap(mapViewId, geoCoordinates, optionalOverviewGeoCoordinates, mapIconInfo, mapScale,cameraAnimation);
}

void NavMiddleware::showLocationsInMap(MapViewId mapViewId, const VisibleListInfo& visibleListInfo,
                                       LocationInputType locationInputType,  const ValidValue<int32_t>& mapScale,
                                       CameraAnimation cameraAnimation)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"showLocationsInMap()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, visibleListInfo = %s, locationInputType = %s, mapScale = %d, cameraAnimation = %s",
                       toString(mapViewId).c_str(),
                       visibleListInfo.toString().c_str(),
                       toString(locationInputType).c_str(),
                       mapScale.getValueOr(0),
                       toString(cameraAnimation).c_str());

   presCtrlInterface.showLocationsInMap(mapViewId, visibleListInfo, locationInputType, mapScale, cameraAnimation);
}

void NavMiddleware::storeMapCapture(MapViewId mapViewId, const ::std::string& path)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN "storeMapCapture()");
   LOG_INTERFACE_DEBUG(map, "MapViewId = %s, path = %s", toString(mapViewId).c_str(), path.c_str());

   presCtrlInterface.storeMapCapture(mapViewId, path);
}

const MapCaptureResultInfo& NavMiddleware::getMapCaptureResultInfo() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN "getMapCaptureResultInfo()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Map Capture Result info ############ - %s",
                       presCtrlInterface.getMapCaptureResultInfo().toString().c_str());
   return presCtrlInterface.getMapCaptureResultInfo();
}

void NavMiddleware::addDynamicIconsOnMap(const ::std::vector< MapDynamicIconInfo>& dynamicIconList)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN "addDynamicIconsOnMap()");
   LOG_INTERFACE_DEBUG(map, "\n ############ dynamicIconList ############ - %s",
                       toString<MapDynamicIconInfo>(dynamicIconList).c_str());
   presCtrlInterface.addDynamicIconsOnMap(dynamicIconList);
}

void NavMiddleware::deleteDynamicIconsOnMap(const ::std::vector<MapDynamicIconId>& mapDynamicIconIds)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN "deleteDynamicIconsOnMap()");
   LOG_INTERFACE_DEBUG(map, "mapDynamicIconIds = %s", toString<MapDynamicIconId>(mapDynamicIconIds).c_str());
   presCtrlInterface.deleteDynamicIconsOnMap(mapDynamicIconIds);
}

const DynamicIconRequestResult& NavMiddleware::getDynamicIconRequestResult() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN "getDynamicIconRequestResult()");
   LOG_INTERFACE_DEBUG(map, "\n ############ DynamicIconRequestResult ############ - %s",
                       presCtrlInterface.getDynamicIconRequestResult().toString().c_str());
   return presCtrlInterface.getDynamicIconRequestResult();
}

//NavDataUpdate

void NavMiddleware::requestStopAvailableUpdates()
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"requestStopAvailableUpdates()");
   presCtrlInterface.requestStopAvailableUpdates();
}

void NavMiddleware::requestAvailableUpdates(uint32_t deviceId,AvailableUpdatesRequestType requestType)
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"requestAvailableUpdates(%u,%d)",deviceId,requestType);

   //todo(apr9kor) this shall be removed, with deprecated interface removal. Empty path, to align with new interface flow.
   const ::std::string path = "";
   presCtrlInterface.requestAvailableUpdates(deviceId, path, requestType);
}

void NavMiddleware::requestAvailableUpdates(uint32_t deviceId, const ::std::string& path, AvailableUpdatesRequestType requestType)
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"requestAvailableUpdates(%u, %s, %d)",deviceId,path.c_str(),requestType);
   presCtrlInterface.requestAvailableUpdates(deviceId, path, requestType);
}

const NavDataUpdateOverviewInfo& NavMiddleware::getAvailableUpdateInfo() const
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"getAvailableUpdateInfo()");

   const NavDataUpdateOverviewInfo& result = presCtrlInterface.getAvailableUpdateInfo();
   LOG_INTERFACE_DEBUG(navdataupdate, "\n ############ Get available update infos ############ - %s",
                       result.toString().c_str());

   return( result );
}

void NavMiddleware::declineAvailableUpdates(UpdateDeclineReason /*updateDeclineReason*/)
{
   // JP specfic
}

void NavMiddleware::requestAvailableUpdatesList(uint32_t startIndex, int32_t nbrOfElements)
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"requestAvailableUpdatesList(%u,%d)",startIndex,nbrOfElements);
   presCtrlInterface.requestAvailableUpdatesList(startIndex,nbrOfElements);
}

const NavDataUpdateInfos& NavMiddleware::getAvailableUpdatesList() const
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"getAvailableUpdatesList()");

   const NavDataUpdateInfos& result = presCtrlInterface.getAvailableUpdatesList();
   LOG_INTERFACE_DEBUG(navdataupdate, "\n ############ Get available updates list info ############ - %s",
                       result.toString().c_str());

   return( result );
}

void NavMiddleware::requestUpdateBySelection()
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"requestUpdateBySelection()");
   presCtrlInterface.requestUpdateBySelection();
}

void NavMiddleware::requestUpdateBySelection(const ::std::string& path)
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"requestUpdateBySelection(%s)", path.c_str());
   presCtrlInterface.requestUpdateBySelection(path);
}

void NavMiddleware::requestStopUpdate()
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"requestStopUpdate()");
   presCtrlInterface.requestStopUpdate();
}

void NavMiddleware::setUpdateSelection(uint32_t updateInfoId,bool select)
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"setUpdateSelection(%u,%s)",updateInfoId,(select?"true":"false"));
   presCtrlInterface.setUpdateSelection(updateInfoId,select);
}

void NavMiddleware::setSelectionOfAllUpdates(bool selected)
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"setSelectionOfAllUpdates(%s)",(selected?"true":"false"));
   presCtrlInterface.setSelectionOfAllUpdates(selected);
}

void NavMiddleware::requestPrepareUSBDevice(uint32_t deviceId)
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"requestPrepareUSBDevice(%u)",deviceId);
   presCtrlInterface.requestPrepareUSBDevice(deviceId);
}

NavDataUpdateUSBDeviceStatus NavMiddleware::getPreparedUSBUpdateDeviceResult()
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"getPreparedUSBUpdateDeviceResult");
   LOG_INTERFACE_DEBUG(navdataupdate, "\n prepared USB update device status = %d", presCtrlInterface.getPreparedUSBUpdateDeviceResult());
   return presCtrlInterface.getPreparedUSBUpdateDeviceResult();
}

const NavDataUpdateStatus& NavMiddleware::getUpdateStatus() const
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"getUpdateStatus()");

   const NavDataUpdateStatus& result = presCtrlInterface.getUpdateStatus();
   LOG_INTERFACE_DEBUG(navdataupdate, "\n ############ Nav data update status info ############ - %s",
                       result.toString().c_str());

   return( result );
}

const NavDataDeviceInfos& NavMiddleware::getNavDataDeviceInfos() const
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"getNavDataDeviceInfos() ");

   const NavDataDeviceInfos& result = presCtrlInterface.getNavDataDeviceInfos();
   LOG_INTERFACE_DEBUG(navdataupdate, "\n ############ Nav data update device info ############ - %s",
                       result.toString().c_str());

   return( result );
}

void NavMiddleware::checkNavDataCompatibility(const ::std::string& navigationVersionString,
      const ::std::vector<uint8_t>& compatibilityInfoData)
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"checkNavDataCompatibility(%s)", navigationVersionString.c_str());
   presCtrlInterface.checkNavDataCompatibility(navigationVersionString, compatibilityInfoData);
}

const NavDataCompatibilityInfo& NavMiddleware::getNavDataCompatibilityInfo() const
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"getNavDataCompatibilityInfo()");

   const NavDataCompatibilityInfo& result = presCtrlInterface.getNavDataCompatibilityInfo();
   LOG_INTERFACE_DEBUG(navdataupdate, "\n ############ Get compatibility info ############ - %s",
                       result.toString().c_str());

   return( result );
}

void NavMiddleware::requestNavigationDataVersionInfo()
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"requestNavigationDataVersionInfo()");
   presCtrlInterface.requestNavigationDataVersionInfo();
}

const NavDataVersionInfo& NavMiddleware::getNavigationDataVersionInfo() const
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"getNavigationDataVersionInfo()");

   const NavDataVersionInfo& result = presCtrlInterface.getNavigationDataVersionInfo();
   LOG_INTERFACE_DEBUG(navdataupdate, "\n ############ Get NavigationData Version Info ############ - %s",
                       result.toString().c_str());

   return( result );
}

const NavDataUpdateSelection& NavMiddleware::getUpdateSelectionStatus() const
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"getUpdateSelectionStatus()");

   const NavDataUpdateSelection& result = presCtrlInterface.getUpdateSelectionStatus();
   LOG_INTERFACE_DEBUG(navdataupdate, "\n ############ Get update selection status ############ - %s",
                       result.toString().c_str());

   return( result );
}

NavDataStatus NavMiddleware::getNavDataStatus() const
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"getNavDataStatus()");
   LOG_INTERFACE_DEBUG(navdataupdate, "\n NavDataStatus = %s", NavDataStatusToString(presCtrlInterface.getNavDataStatus()).c_str());
   return presCtrlInterface.getNavDataStatus();
}

CatalogueStatus NavMiddleware::getCatalogueStatus() const
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"getCatalogueStatus()");

   LOG_INTERFACE_DEBUG(navdataupdate, "\n CatalogueStatus = %s", CatalogueStatusToString(presCtrlInterface.getCatalogueStatus()).c_str());
   return presCtrlInterface.getCatalogueStatus();
}

void NavMiddleware::setScomoComponentRegistration(ScomoComponent scomoComponent, ScomoComponentRegistrationStatus registrationStatus)
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"setScomoComponentRegistration()");
   LOG_INTERFACE_DEBUG(navdataupdate, "\n scomoComponent = %s, registrationStatus = %s", ScomoComponentToString(scomoComponent).c_str(), toString(registrationStatus).c_str());
   presCtrlInterface.setScomoComponentRegistration(scomoComponent,registrationStatus);
}

void NavMiddleware::requestScomoInstallComponent(NavDataUpdateInstallStage installStage, ScomoComponent scomoComponent, const ::std::string& updateURL, const ::std::string& componentName)
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"requestScomoInstallComponent(installStage=%d, scomoComponent=%s, updateURL = %s, componentName = %s)",
                 installStage, ScomoComponentToString(scomoComponent).c_str(), updateURL.c_str(), componentName.c_str());
   presCtrlInterface.requestScomoInstallComponent(installStage, scomoComponent, updateURL, componentName);
}

void NavMiddleware::requestAbortScomoInstallComponent(ScomoComponent scomoComponent)
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"requestAbortScomoInstallComponent(scomoComponent=%s)",
                 ScomoComponentToString(scomoComponent).c_str());

   presCtrlInterface.requestAbortScomoInstallComponent(scomoComponent);
}

void NavMiddleware::requestScomoComponentStatus(ScomoComponent scomoComponent)
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"requestScomoComponentStatus: %s", ScomoComponentToString(scomoComponent).c_str());
   presCtrlInterface.requestScomoComponentStatus(scomoComponent);
}

void NavMiddleware::setFotaInventoryUpdateStatus(bool isFotaInventoryUpdated)
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"setFotaInventoryUpdateStatus: %d", isFotaInventoryUpdated);
   presCtrlInterface.setFotaInventoryUpdateStatus(isFotaInventoryUpdated);
}

void NavMiddleware::registerOnNavDataContextChanges()
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"registerOnNavDataContextChanges");
   presCtrlInterface.registerOnNavDataContextChanges();
}

void NavMiddleware::requestScomoDeinstallComponent(uint32_t /*productId*/, uint32_t /*supplierId*/, const ::std::vector< uint32_t >& /*updateRegionIds*/)
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"requestScomoDeinstallComponent is NOT_SUPPORTED");
   // Deinstallation of an updateRegion using locally stored packages is not supported.
   // Deinstall of an updateRegion is nothing but install with version 0. requestScomoInstallComponent() can be used for the same.
}

const ScomoComponentStatus& NavMiddleware::getScomoComponentStatus() const
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"getScomoComponentStatus");

   const ScomoComponentStatus& result = presCtrlInterface.getScomoComponentStatus();
   LOG_INTERFACE_DEBUG(navdataupdate, "\n ############ scomo component status ############ - %s",
                       result.toString().c_str());

   return( result );
}

const DeinstallScomoComponentResult& NavMiddleware::getScomoDeinstallComponentResult() const
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"getScomoDeinstallComponentResult");

   static DeinstallScomoComponentResult result(DEINSTALL_RESULT__FAILED);
   return( result );
}

const ScomoInstallComponentResult& NavMiddleware::getScomoInstallComponentResult() const
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"getScomoInstallComponentResult");

   const ScomoInstallComponentResult& result = presCtrlInterface.getScomoInstallComponentResult();
   LOG_INTERFACE_DEBUG(navdataupdate, "\n ############ install scomo component result ############ - %s",
                       result.toString().c_str());

   return( result );
}

const NavDataContext& NavMiddleware::getNavDataContext() const
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"getNavDataContext");

   const NavDataContext& result = presCtrlInterface.getScomoNavDataContext();
   LOG_INTERFACE_DEBUG(navdataupdate, "\n ############ scomo nav data context result ############ - %s",
                       result.toString().c_str());

   return( presCtrlInterface.getScomoNavDataContext() );
}

const ScomoAvailableUpdates& NavMiddleware::getAvailableScomoUpdates() const
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"getAvailableScomoUpdates");

   const ScomoAvailableUpdates& result = presCtrlInterface.getAvailableScomoUpdates();
   LOG_INTERFACE_DEBUG(navdataupdate, "\n ############ scomo available updates result ############ - %s",
                       result.toString().c_str());

   return( result );
}

void NavMiddleware::updateNavDataUpdateFavoriteRegionSettings(const ::navmiddleware::settings::NavDataUpdateFavoriteRegionSettings& favoriteRegionSetting)
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"updateNavDataUpdateFavoriteRegionSettings");
   LOG_INTERFACE_DEBUG(navdataupdate, "\n ############ setting infos ############ - %s",
                       favoriteRegionSetting.toString().c_str());
   presCtrlInterface.updateNavDataUpdateFavoriteRegion(favoriteRegionSetting);
}

void NavMiddleware::requestNavDataPartNumber()
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"requestNavDataPartNumber");
   presCtrlInterface.requestNavDataPartNumber();
}

const NavDataPartNumber& NavMiddleware::getNavDataPartNumber() const
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"getNavDataPartNumber");
   LOG_INTERFACE_DEBUG(navdataupdate, "\n ############ NavDataPartNumber infos ############ - %s",
                       presCtrlInterface.getNavDataPartNumber().toString().c_str());
   return presCtrlInterface.getNavDataPartNumber();
}

void NavMiddleware::requestNavDataUpdateHistory()
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"requestNavDataUpdateHistory");
   presCtrlInterface.requestNavDataUpdateHistory();
}

const NavDataUpdateHistory& NavMiddleware::getNavDataUpdateHistory() const
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"getNavDataUpdateHistory");
   LOG_INTERFACE_DEBUG(navdataupdate, "\n ############ NavDataUpdateHistory infos ############ - %s",
                       presCtrlInterface.getNavDataUpdateHistory().toString().c_str());
   return presCtrlInterface.getNavDataUpdateHistory();
}

const NavDatasetInfo& NavMiddleware::getNavDatasetInfo() const
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"getNavDatasetInfo");
   LOG_INTERFACE_DEBUG(navdataupdate, "\n ############ NavDatasetInfo payload ############ - %s",
                       presCtrlInterface.getNavDatasetInfo().toString().c_str());
   return(presCtrlInterface.getNavDatasetInfo());
}

void NavMiddleware::registerDatasetUser(NdsDatasetUser ndsDatasetUser, uint32_t datasetId)
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"registerDatasetUser %s %u",
                 ::navmiddleware::toString(ndsDatasetUser).c_str(), datasetId);
   presCtrlInterface.registerDatasetUser(ndsDatasetUser, datasetId);
}

void NavMiddleware::deregisterDatasetUser(NdsDatasetUser ndsDatasetUser, uint32_t datasetId)
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"deregisterDatasetUser %s %u",
                 ::navmiddleware::toString(ndsDatasetUser).c_str(), datasetId);
   presCtrlInterface.deregisterDatasetUser(ndsDatasetUser, datasetId);
}

const RegisterDatasetUserResult& NavMiddleware::getDatasetUserRegisterStatus() const
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"getDatasetUserRegisterStatus");
   LOG_INTERFACE_DEBUG(navdataupdate, "\n ############ DatasetUserRegisterStatus payload ############ - %s",
                       presCtrlInterface.getDatasetUserRegisterStatus().toString().c_str());
   return(presCtrlInterface.getDatasetUserRegisterStatus());
}

void NavMiddleware::datasetUserAccess(NdsDatasetUser ndsDatasetUser, uint32_t datasetId, NdsDataAccessType ndsDataAccessType)
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"datasetUserAccess %s %u %s",
                 ::navmiddleware::toString(ndsDatasetUser).c_str(), datasetId, ::navmiddleware::toString(ndsDataAccessType).c_str());
   presCtrlInterface.datasetUserAccess(ndsDatasetUser, datasetId, ndsDataAccessType);
}

void NavMiddleware::showGeoFenceInMap(MapViewId mapViewId, const GeoFence& geoFence)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN "showGeoFenceInMap()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, GeoCoordinate1 = (%f, %f)"
                       "GeoCoordinate2 = (%f, %f), Circle Radius = %d, geoFenceType = %s",
                       toString(mapViewId).c_str(), geoFence.m_geoCoordinate1.getLatitude(), geoFence.m_geoCoordinate1.getLongitude(),
                       geoFence.m_geoCoordinate2.isValid() ? geoFence.m_geoCoordinate2.getValue().getLatitude() : 0.0,
                       geoFence.m_geoCoordinate2.isValid() ? geoFence.m_geoCoordinate2.getValue().getLongitude() : 0.0,
                       geoFence.m_radius.getValueOr(0),
                       toString(geoFence.m_geoFenceType).c_str());

   presCtrlInterface.showGeoFenceInMap(mapViewId, geoFence);
}

void NavMiddleware::showRouteSegmentSelectionInMap(MapViewId mapViewId, const ValidValue<uint32_t>& indexInRouteList, CameraAnimation cameraAnimation)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN "showRouteSegmentSelectionInMap()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, indexInRouteList = %s, cameraAnimation = %s",
                       toString(mapViewId).c_str(), indexInRouteList.toString().c_str(), toString(cameraAnimation).c_str());

   presCtrlInterface.showRouteSegmentSelectionInMap(mapViewId, indexInRouteList, cameraAnimation);
}

::navmiddleware::settings::NavDataUpdateFavoriteRegionSettings NavMiddleware::getNavDataUpdateFavoriteRegionSettings() const
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"getNavDataUpdateFavoriteRegionSettings");
   LOG_INTERFACE_DEBUG(navdataupdate, "\n ############ NavDataUpdateFavoriteRegionSettings ############ - %s",
                       presCtrlInterface.getNavDataUpdateFavoriteRegion().toString().c_str());
   return presCtrlInterface.getNavDataUpdateFavoriteRegion();
}

::navmiddleware::UpdateFavoriteUpdateRegionsResult NavMiddleware::getUpdateNavDataUpdateFavoriteRegionSettingsResult() const
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"getUpdateNavDataUpdateFavoriteRegionSettingsResult");

   const ::navmiddleware::UpdateFavoriteUpdateRegionsResult result = presCtrlInterface.getUpdateNavDataUpdateFavoriteRegionSettingsResult();
   LOG_INTERFACE_DEBUG(navdataupdate, "\n ############ UpdateFavoriteUpdateRegionsResult ############ - %s",
                       ::navmiddleware::toString(result).c_str());
   return result;
}

void NavMiddleware::setAutomaticMapUpdateStatus(::navmiddleware::settings::AutomaticMapUpdateStatus automaticMapUpdateStatus)
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"setAutomaticMapUpdateStatus - %s",
                 ::navmiddleware::settings::toString(automaticMapUpdateStatus).c_str());

   presCtrlInterface.setAutomaticMapUpdateStatus(automaticMapUpdateStatus);
}

::navmiddleware::settings::AutomaticMapUpdateStatus NavMiddleware::getAutomaticMapUpdateStatus() const
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"getAutomaticMapUpdateStatus");

   ::navmiddleware::settings::AutomaticMapUpdateStatus status = presCtrlInterface.getAutomaticMapUpdateStatus();
   LOG_INTERFACE_DEBUG(navdataupdate, "\n ############ getAutomaticMapUpdateStatus ############ - %s",
                       ::navmiddleware::settings::toString(status).c_str());
   return status;
}

::navmiddleware::AutomaticMapUpdateStatusResult NavMiddleware::getAutomaticMapUpdateStatusResult() const
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"getAutomaticMapUpdateStatusResult");

   ::navmiddleware::AutomaticMapUpdateStatusResult result = presCtrlInterface.getAutomaticMapUpdateStatusResult();
   LOG_INTERFACE_DEBUG(navdataupdate, "\n ############ getAutomaticMapUpdateStatusResult ############ - %s",
                       ::navmiddleware::toString(result).c_str());
   return result;
}

::navmiddleware::AutomaticFullUpdateStatus NavMiddleware::getAutomaticFullUpdateStatus() const
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"getAutomaticFullUpdateStatus - %s", ::navmiddleware::toString(presCtrlInterface.getAutomaticFullUpdateStatus()).c_str());
   return presCtrlInterface.getAutomaticFullUpdateStatus();
}

void NavMiddleware::setAutomaticFullUpdateFailed(::navmiddleware::AutomaticFullUpdateFailureReason failureReason)
{
   LOG_INTERFACE(navdataupdate, MESSAGE_TAG_IN"setAutomaticFullUpdateStatus - %s", ::navmiddleware::toString(failureReason).c_str());
   presCtrlInterface.setAutomaticFullUpdateFailed(failureReason);
}

void NavMiddleware::showIntersectionMap(
   MapViewId mapViewId,
   CameraAnimation cameraAnimation,
   IntersectionMapMode mode)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"showIntersectionMap()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, cameraAnimation = %s, intersectionMode = %s", toString(mapViewId).c_str(), toString(cameraAnimation).c_str(), toString(mode).c_str());

   presCtrlInterface.showIntersectionMap(mapViewId, cameraAnimation, mode);
}

void NavMiddleware::showWeatherAlertsInMap(MapViewId mapViewId)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"showWeatherAlertsInMap()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s", toString(mapViewId).c_str());

   presCtrlInterface.showWeatherAlertsInMap(mapViewId);
}

void NavMiddleware::showMapWidget(MapViewId mapViewId)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"showMapWidget()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s", toString(mapViewId).c_str());

   presCtrlInterface.showMapWidget(mapViewId);
}

void NavMiddleware::showTrafficDetourInMap(MapViewId mapViewId, TrafficRouteOverviewMode trafficRouteOverviewMode, CameraAnimation cameraAnimation)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"showTrafficDetourInMap()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, trafficRouteOverviewMode = %s, cameraAnimation = %s",
                       toString(mapViewId).c_str(), ::navmiddleware::toString(trafficRouteOverviewMode).c_str(), toString(cameraAnimation).c_str());

   presCtrlInterface.showTrafficDetourInMap(mapViewId, trafficRouteOverviewMode, cameraAnimation);
}

void NavMiddleware::showAddressbookIconsInMap(MapViewId mapViewId,
      const ::std::vector< DestinationMemoryEntry::Id >& entryIds,
      int32_t selectedIndex,
      const ValidValue<int32_t>& mapScale,
      CameraAnimation cameraAnimation)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"showAddressbookIconsInMap()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, NumOfEntries =  %d, selectedIndex = %d, mapScale = %d, cameraAnimation = %s",
                       toString(mapViewId).c_str(),
                       entryIds.size(),
                       selectedIndex,
                       mapScale.getValueOr(0),
                       toString(cameraAnimation).c_str());
   presCtrlInterface.showAddressbookIconsInMap(mapViewId, entryIds, selectedIndex, mapScale, cameraAnimation);
}

void NavMiddleware::showReachableAreaOverviewInMap(MapViewId mapViewId,
      ReachableAreaVisibility reachableAreaVisibility,
      CameraAnimation cameraAnimation)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"showReachableAreaOverviewInMap()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, reachableAreaVisibility =  %s, cameraAnimation = %s",
                       toString(mapViewId).c_str(),
                       toString(reachableAreaVisibility).c_str(),
                       toString(cameraAnimation).c_str());
   presCtrlInterface.showReachableAreaOverviewInMap(mapViewId, reachableAreaVisibility, cameraAnimation);
}

void NavMiddleware::configureRouteVisibilityOnMap(MapViewId mapViewId, const ::std::vector<unsigned int >& routeIndices)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"configureRouteVisibilityOnMap()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, NumOfRoutesToBeHidden=  %d", toString(mapViewId).c_str(), routeIndices.size());
   presCtrlInterface.configureRouteVisibilityOnMap(mapViewId, routeIndices);
}

const IntersectionMapVectorDataInfos& NavMiddleware::getIntersectionMapVectorDataInfos() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"getIntersectionMapVectorDataList()");
   return presCtrlInterface.getIntersectionMapVectorDataInfos();
}

void NavMiddleware::deleteAvoidAreaPreview(MapViewId mapViewId)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"deleteAvoidAreaPreview()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s", toString(mapViewId).c_str());

   presCtrlInterface.deleteAvoidAreaPreview(mapViewId);
}

void NavMiddleware::addAvoidAreaPreview(MapViewId mapViewId, const GeoCoordinateDegree& centerPoint, uint32_t edgeLength, AvoidAreaPreviewStyle avoidAreaPreviewStyle)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"addAvoidAreaPreview()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, centerPoint = %s, edgeLength = %d, avoidAreaPreviewStyle = %s",
                       toString(mapViewId).c_str(), centerPoint.toString().c_str(), edgeLength, toString(avoidAreaPreviewStyle).c_str());
   presCtrlInterface.addAvoidAreaPreview(mapViewId, centerPoint, edgeLength, avoidAreaPreviewStyle);
}

void NavMiddleware::setConnectedClusterType(ConnectedClusterType connectedClusterType)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"setConnectedClusterType()");
   LOG_INTERFACE_DEBUG(map, "connectedClusterType = %s", toString(connectedClusterType).c_str());
   presCtrlInterface.setConnectedClusterType(connectedClusterType);
}

void NavMiddleware::configureCrossHair(::navmiddleware::MapViewId mapViewId, CrossHairVisibility crossHairVisibility)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"configureCrossHair()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, crossHairVisibility = %s)", toString(mapViewId).c_str(), toString(crossHairVisibility).c_str());

   presCtrlInterface.configureCrossHair(mapViewId, crossHairVisibility);
}

void NavMiddleware::configureMapLocationMarkers(MapViewId mapViewId, const MapLocationMarkers& mapLocationMarkers, MapLocationMarkerConfigurationOperation operation)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"configureMapLocationMarkers()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, Markers List = %s, MapLocationMarkerConfigurationOperation = %s",
                       toString(mapViewId).c_str(), mapLocationMarkers.toString().c_str(), ::navmiddleware::toString(operation).c_str());

   presCtrlInterface.configureMapLocationMarkers(mapViewId, mapLocationMarkers, operation);
}

void NavMiddleware::configureRoadNetworkIllustration(MapViewId mapViewId, RoadNetworkIllustration roadNetworkIllustration)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"configureRoadNetworkIllustration()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, roadNetworkIllustration = %s", toString(mapViewId).c_str(), toString(roadNetworkIllustration).c_str());
   LOG_INTERFACE_WARN(map, "NOT IMPLEMENTED");
}

void NavMiddleware::setMapScale(MapViewId mapViewId, int32_t mapScale, CameraAnimation cameraAnimation)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"setMapScale()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, mapScale = %d, cameraAnimation = %s", toString(mapViewId).c_str(), mapScale, toString(cameraAnimation).c_str());

   presCtrlInterface.setMapScale(mapViewId, mapScale, cameraAnimation);
}

void NavMiddleware::showStreetViewImagesInMap(MapViewId mapViewId, StreetViewMode streetViewMode, const GeoCoordinateDegree& geoCoordinate, const ValidValue< ::std::string >& address)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"showStreetViewImagesInMap()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, streetViewMode= %s, geoCoordinate = (%f, %f), address = %s",
                       toString(mapViewId).c_str(), toString(streetViewMode).c_str(), geoCoordinate.getLatitude(), geoCoordinate.getLongitude(), address.getValueOr("").c_str());

   presCtrlInterface.showStreetViewImagesInMap(mapViewId, streetViewMode, geoCoordinate, address);
}

void NavMiddleware::showAvoidAreaOverviewInMap(MapViewId mapViewId,
      const ValidValue<AvoidAreaId>& avoidAreaId,
      AvoidAreaVisibility avoidAreaVisibility,
      CameraAnimation cameraAnimation)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"showAvoidAreaOverviewInMap()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, avoidAreaId =  %llu, avoidAreaVisibility = %s, cameraAnimation = %s",
                       toString(mapViewId).c_str(),
                       avoidAreaId.getValueOr(0),
                       toString(avoidAreaVisibility).c_str(),
                       toString(cameraAnimation).c_str());

   presCtrlInterface.showAvoidAreaOverviewInMap(mapViewId, avoidAreaId, avoidAreaVisibility, cameraAnimation);
}

const AvoidAreaHMIOverlay& NavMiddleware::getAvoidAreaHMIOverlay() const
{
   LOG_INTERFACE(configuration, MESSAGE_TAG_IN"getAvoidAreaHMIOverlay() ");
   LOG_INTERFACE_DEBUG(configuration, "\n ############ AviodArea HMI Overlay ############ - %s",
                       presCtrlInterface.getAvoidAreaHMIOverlay().toString().c_str());
   return presCtrlInterface.getAvoidAreaHMIOverlay();
}

RequestId NavMiddleware::transformViewPositions(MapViewId mapViewId, const ::std::vector< ScreenCoordinate >& screenCoordinates)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"transformViewPositions()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, NumOfScreenCoordinates = %d", toString(mapViewId).c_str(), screenCoordinates.size());
   return presCtrlInterface.transformViewPositions(mapViewId, screenCoordinates);
}

const TransformedViewPositionsInfo& NavMiddleware::getTransformedViewPositionsInfo() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"getTransformedViewPositionsInfo()");
   LOG_INTERFACE_DEBUG(map, "\n ############ TransformedViewPositionsInfo ############ - %s",
                       presCtrlInterface.getTransformedViewPositionsInfo().toString().c_str());
   return presCtrlInterface.getTransformedViewPositionsInfo();
}

const IntersectionMapCarsorPosition& NavMiddleware::getIntersectionMapCarsorPosition() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"getIntersectionMapCarsorPosition()");
   LOG_INTERFACE_DEBUG(map, "\n ############ IntersectionMapCarsorPosition ############ - %s",
                       presCtrlInterface.getIntersectionMapCarsorPosition().toString().c_str());
   return presCtrlInterface.getIntersectionMapCarsorPosition();
}

void NavMiddleware::showSatelliteImagesInMap(MapViewId mapViewId, const GeoCoordinateDegree& geoCoordinate, int32_t mapScale)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"showSatelliteImagesInMap()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, latitude = %f, longitude = %f, map scale = %d",
                       toString(mapViewId).c_str(), geoCoordinate.getLatitude(), geoCoordinate.getLongitude(), mapScale);
   presCtrlInterface.showSatelliteImagesInMap(mapViewId, geoCoordinate, mapScale);
}

void NavMiddleware::requestStreetViewPreviewImage(uint32_t desiredWidth,
      uint32_t desiredHeight,
      const GeoCoordinateDegree& geoCoordinate,
      const ValidValue< ::std::string >& address)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"requestStreetViewPreviewImage()");
   LOG_INTERFACE_DEBUG(map, "desiredWidth=%d, desiredHeight=%d, latitude=%f, longitude=%f, address = %s",
                       desiredWidth, desiredHeight, geoCoordinate.getLatitude(), geoCoordinate.getLongitude(), address.getValueOr("").c_str());
   presCtrlInterface.requestStreetViewPreviewImage(desiredWidth, desiredHeight,geoCoordinate, address);
}

const StreetViewPreviewImageInfo& NavMiddleware::getStreetViewPreviewImage() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"getStreetViewPreviewImage()");
   LOG_INTERFACE_DEBUG(map, "\n ############ StreetView PreviewImage Info ############ - %s",
                       presCtrlInterface.getStreetViewPreviewImage().toString().c_str());
   return presCtrlInterface.getStreetViewPreviewImage();
}

void NavMiddleware::prepareSatelliteView(uint32_t desiredWidthInPixels,
      uint32_t desiredHeightInPixels,
      const GeoCoordinateDegree& geoCoordinate,
      const ::std::vector<int32_t>& mapScales)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"prepareSatelliteView()");
   LOG_INTERFACE_DEBUG(map, "desiredWidth = %d, desiredHeight = %d, latitude = %f, longitude = %f, number of mapscales = %d",
                       desiredWidthInPixels, desiredHeightInPixels, geoCoordinate.getLatitude(), geoCoordinate.getLongitude(), mapScales.size());
   presCtrlInterface.prepareSatelliteView(desiredWidthInPixels, desiredHeightInPixels, geoCoordinate, mapScales);
}

const PrepareSatelliteViewInfo& NavMiddleware::getPrepareSatelliteViewInfo() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"getPrepareSatelliteViewInfo()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Satellite View info ############ - %s",
                       presCtrlInterface.getPrepareSatelliteViewInfo().toString().c_str());
   return presCtrlInterface.getPrepareSatelliteViewInfo();
}

void NavMiddleware::prepareStreetView(uint32_t desiredWidthInPixels,
                                      uint32_t desiredHeightInPixels,
                                      const GeoCoordinateDegree& geoCoordinate,
                                      const ValidValue< ::std::string >& address)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"prepareStreetView()");
   LOG_INTERFACE_DEBUG(map, "desiredWidth=%d, desiredHeight=%d, latitude=%f, longitude=%f, address = %s",
                       desiredWidthInPixels, desiredHeightInPixels, geoCoordinate.getLatitude(), geoCoordinate.getLongitude(), address.getValueOr("").c_str());
   presCtrlInterface.prepareStreetView(desiredWidthInPixels, desiredHeightInPixels, geoCoordinate, address);
}

const PrepareStreetViewStatusInfo& NavMiddleware::getPrepareStreetViewStatusInfo() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"getPrepareStreetViewStatusInfo()");
   LOG_INTERFACE_DEBUG(map, "\n ############ StreetView Status Info ############ - %s",
                       presCtrlInterface.getPrepareStreetViewStatusInfo().toString().c_str());
   return presCtrlInterface.getPrepareStreetViewStatusInfo();
}

void NavMiddleware::cancelPrepareStreetView()
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"cancelPrepareStreetView()");
   presCtrlInterface.cancelPrepareStreetView();
}

void NavMiddleware::cancelRequestStreetViewPreviewImage()
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"cancelRequestStreetViewPreviewImage()");
   presCtrlInterface.cancelRequestStreetViewPreviewImage();
}

void NavMiddleware::cancelPrepareSatelliteView()
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"cancelPrepareSatelliteView()");
   presCtrlInterface.cancelPrepareSatelliteView();
}

void NavMiddleware::updateMapTrackingDotsSettings(const settings::MapTrackingDotsSettings& mapTrackingDotsSettings)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"updateMapTrackingDotsSettings()");
   LOG_INTERFACE_DEBUG(map, "TrackingDotDisplayOnMapEnabled = %s, trackingDotsDistance = %s",
                       (mapTrackingDotsSettings.getTrackingDotDisplayState() ? "Yes" : "No"),
                       navmiddleware::settings::MapTrackingDotsSettings::toString(mapTrackingDotsSettings.getMapTrackingDotDistance()).c_str());
   presCtrlInterface.updateMapTrackingDotsSettings(mapTrackingDotsSettings);
}

settings::MapTrackingDotsSettings NavMiddleware::getMapTrackingDotsSettings() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"getMapTrackingDotsSettings()");
   LOG_INTERFACE_DEBUG(map, "\n ############ MapTrackingDotsSettings ############ - %s",
                       presCtrlInterface.getMapTrackingDotsSettings().toString().c_str());
   return presCtrlInterface.getMapTrackingDotsSettings();
}

void NavMiddleware::deleteAllTrackingDots()
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"deleteAllTrackingDots()");
   presCtrlInterface.deleteAllTrackingDots();
}

const MapOnlineFeatureAvailabilityInfos& NavMiddleware::getMapOnlineFeatureAvailabilityInfos() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"getMapOnlineFeatureAvailabilityInfos()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Map online Features Status info ############ %s",
                       presCtrlInterface.getMapOnlineFeatureAvailabilityInfos().toString().c_str());

   return presCtrlInterface.getMapOnlineFeatureAvailabilityInfos();
}

const MapVideoStreamStatusInfo& NavMiddleware::getMapVideoStreamStatusInfo() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"getMapVideoStreamStatusInfo()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Map VideoStream Status Info ############ - %s",
                       presCtrlInterface.getMapVideoStreamStatusInfo().toString().c_str());
   return presCtrlInterface.getMapVideoStreamStatusInfo();
}

const MapCameraAndViewModesStatusInfo& NavMiddleware::getMapCameraAndViewModesStatusInfo() const
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"getMapCameraAndViewModesStatusInfo()");
   LOG_INTERFACE_DEBUG(map, "\n ############ Map Camera And ViewModes StatusInfo ############ - %s",
                       presCtrlInterface.getMapCameraAndViewModesStatusInfo().toString().c_str());
   return presCtrlInterface.getMapCameraAndViewModesStatusInfo();
}

void NavMiddleware::highlightRoad(MapViewId mapViewId, const LineId& lineId, bool isHighlighted)
{
   LOG_INTERFACE(map, MESSAGE_TAG_IN"highlightRoad()");
   LOG_INTERFACE_DEBUG(map, "mapViewId = %s, lineId = %s, isRoadHighlighted = %s",
                       ::navmiddleware::toString(mapViewId).c_str(), lineId.toString().c_str(), (isHighlighted ? "Yes" : "No"));

   presCtrlInterface.highlightRoad(mapViewId, lineId, isHighlighted);
}

const EngineeringInfos& NavMiddleware::getEngineeringInfos()
{
   LOG_INTERFACE(engineering, MESSAGE_TAG_IN"getEngineeringInfos()");
   return presCtrlInterface.getEngineeringInfos();
}

void NavMiddleware::selectEngineeringMenuEntry(unsigned int index)
{
   LOG_INTERFACE(engineering, MESSAGE_TAG_IN"selectEngineeringMenuEntry(%u)", index);
   presCtrlInterface.selectEngineeringMenuEntry(index);
}

void NavMiddleware::deselectEngineeringMenuEntry()
{
   LOG_INTERFACE(engineering, MESSAGE_TAG_IN"deselectEngineeringMenuEntry()");
   presCtrlInterface.deselectEngineeringMenuEntry();
}

void NavMiddleware::resetEngineeringMenu()
{
   LOG_INTERFACE(engineering, MESSAGE_TAG_IN"resetEngineeringMenu()");
   presCtrlInterface.resetEngineeringMenu();
}

void NavMiddleware::enterEngineeringMenu()
{
   LOG_INTERFACE(engineering, MESSAGE_TAG_IN"enterEngineeringMenu()");
   presCtrlInterface.enterEngineeringMenu();
}

void NavMiddleware::exitEngineeringMenu()
{
   LOG_INTERFACE(engineering, MESSAGE_TAG_IN"exitEngineeringMenu()");
   presCtrlInterface.exitEngineeringMenu();
}

void NavMiddleware::updateUserInputByValue(const ::std::string& userInputValue)
{
   LOG_INTERFACE(engineering, MESSAGE_TAG_IN"updateUserInputByValue()");
   LOG_INTERFACE_DEBUG(engineering, "%s", userInputValue.c_str());
   presCtrlInterface.updateUserInputByValue(userInputValue);
}

const EngineeringDebugInfos& NavMiddleware::getTrafficDebugInformation()
{
   LOG_INTERFACE(engineering, MESSAGE_TAG_IN"getTrafficDebugInformation()");
   // log the payload
   LOG_INTERFACE_DEBUG(engineering, "%s", presCtrlInterface.getTrafficDebugInformation().toString().c_str());

   return presCtrlInterface.getTrafficDebugInformation();
}

void NavMiddleware::requestTrafficDebugInformation()
{
   LOG_INTERFACE(engineering, MESSAGE_TAG_IN"requestTrafficDebugInformation()");
   presCtrlInterface.requestTrafficDebugInformation();
}

void NavMiddleware::releaseTrafficDebugInformation()
{
   LOG_INTERFACE(engineering, MESSAGE_TAG_IN"releaseTrafficDebugInformation()");
   presCtrlInterface.releaseTrafficDebugInformation();
}

//<-- INF4CV
//bool NavMiddleware::hasExtension(navmidextensions::AbstractExtension::ExtensionType /*type*/) const
//{
//   LOG_INTERFACE(inavmiddleware, MESSAGE_TAG_IN"hasExtension()");
//   LOG_INTERFACE_ERROR(inavmiddleware, "not supported - only JP");
//   return false;
//}

//navmidextensions::AbstractExtension& NavMiddleware::getExtension(
//   navmidextensions::AbstractExtension::ExtensionType /*type*/)
//{
//   LOG_INTERFACE(inavmiddleware, MESSAGE_TAG_IN"getExtension()");
//   LOG_INTERFACE_ERROR(inavmiddleware, "not supported - only JP");
//   static navmidextensions::InvalidExtension invalidExtension;
//   return invalidExtension;
//}
//->

void NavMiddleware::enableInterfaceTrace(InterfaceTraceClass interfaceTraceClass)
{
   LOG_INTERFACE(inavmiddleware, MESSAGE_TAG_IN"enableInterfaceTrace(%u)", (uint32)interfaceTraceClass);
   switch (interfaceTraceClass)
   {
   case INTERFACE_TRACECLASS_MIDDLEWARE:
      ENABLE_INTERFACE_LOGGER(inavmiddleware, ::asf::core::Logging::Level__Info);
      ENABLE_INTERFACE_LOGGER(prop_lifecycle, ::asf::core::Logging::Level__Info);
      break;
   case INTERFACE_TRACECLASS_POSITION:
      ENABLE_INTERFACE_LOGGER(position, ::asf::core::Logging::Level__Info);
      ENABLE_INTERFACE_LOGGER(prop_position, ::asf::core::Logging::Level__Info);
      break;
   case INTERFACE_TRACECLASS_POSITIONSIMULATION:
      ENABLE_INTERFACE_LOGGER(positionsimulation, ::asf::core::Logging::Level__Info);
      ENABLE_INTERFACE_LOGGER(prop_position, ::asf::core::Logging::Level__Info);
      break;
   case INTERFACE_TRACECLASS_MAP:
      ENABLE_INTERFACE_LOGGER(map, ::asf::core::Logging::Level__Info);
      ENABLE_INTERFACE_LOGGER(prop_map, ::asf::core::Logging::Level__Info);
      break;
   case INTERFACE_TRACECLASS_ROUTE:
      ENABLE_INTERFACE_LOGGER(route, ::asf::core::Logging::Level__Info);
      ENABLE_INTERFACE_LOGGER(prop_route, ::asf::core::Logging::Level__Info);
      break;
   case INTERFACE_TRACECLASS_GUIDANCE:
      ENABLE_INTERFACE_LOGGER(guidance, ::asf::core::Logging::Level__Info);
      ENABLE_INTERFACE_LOGGER(prop_guidance, ::asf::core::Logging::Level__Info);
      break;
   case INTERFACE_TRACECLASS_TRAFFICMESSAGE:
      ENABLE_INTERFACE_LOGGER(trafficmessage, ::asf::core::Logging::Level__Info);
      ENABLE_INTERFACE_LOGGER(prop_trafficmessage, ::asf::core::Logging::Level__Info);
      break;
   case INTERFACE_TRACECLASS_LOCATIONBASEDSERVICES:
      ENABLE_INTERFACE_LOGGER(locationbasedservices, ::asf::core::Logging::Level__Info);
      ENABLE_INTERFACE_LOGGER(prop_locationbasedservices, ::asf::core::Logging::Level__Info);
      break;
   default:
      LOG_INTERFACE_ERROR(inavmiddleware, "Logger (%u) to be enabled not found!", (uint32)interfaceTraceClass);
      break;
   }

}

void NavMiddleware::disableInterfaceTrace(InterfaceTraceClass interfaceTraceClass)
{
   LOG_INTERFACE(inavmiddleware, MESSAGE_TAG_IN"disableInterfaceTrace(): %u", (uint32)interfaceTraceClass);
   switch (interfaceTraceClass)
   {
   case INTERFACE_TRACECLASS_MIDDLEWARE:
      DISABLE_INTERFACE_LOGGER(inavmiddleware);
      DISABLE_INTERFACE_LOGGER(prop_lifecycle);
      break;
   case INTERFACE_TRACECLASS_POSITION:
      DISABLE_INTERFACE_LOGGER(position);
      DISABLE_INTERFACE_LOGGER(prop_position);
      break;
   case INTERFACE_TRACECLASS_POSITIONSIMULATION:
      DISABLE_INTERFACE_LOGGER(positionsimulation);
      DISABLE_INTERFACE_LOGGER(prop_position);
      break;
   case INTERFACE_TRACECLASS_MAP:
      DISABLE_INTERFACE_LOGGER(map);
      DISABLE_INTERFACE_LOGGER(prop_map);
      break;
   case INTERFACE_TRACECLASS_ROUTE:
      DISABLE_INTERFACE_LOGGER(route);
      DISABLE_INTERFACE_LOGGER(prop_route);
      break;
   case INTERFACE_TRACECLASS_GUIDANCE:
      DISABLE_INTERFACE_LOGGER(guidance);
      DISABLE_INTERFACE_LOGGER(prop_guidance);
      break;
   case INTERFACE_TRACECLASS_TRAFFICMESSAGE:
      DISABLE_INTERFACE_LOGGER(trafficmessage);
      DISABLE_INTERFACE_LOGGER(prop_trafficmessage);
      break;
   case INTERFACE_TRACECLASS_LOCATIONBASEDSERVICES:
      DISABLE_INTERFACE_LOGGER(locationbasedservices);
      DISABLE_INTERFACE_LOGGER(prop_locationbasedservices);
      break;
   default:
      LOG_INTERFACE_ERROR(inavmiddleware, "Logger (%u) to be disabled not found!", (uint32)interfaceTraceClass);
      break;
   }
}

void NavMiddleware::lock()
{
   LOG_INTERFACE_DEBUG(inavmiddleware, MESSAGE_TAG_IN"lock()");
   presCtrlInterface.lock();
}

void NavMiddleware::unlock()
{
   LOG_INTERFACE_DEBUG(inavmiddleware, MESSAGE_TAG_IN"unlock()");
   presCtrlInterface.unlock();
}

//<-- INF4CV
void NavMiddleware::importUserPois(const ::std::string& usbPath)
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"importUserPois(usbPath=%s)", usbPath.c_str());
   presCtrlInterface.importUserPois(usbPath);
}

const ImportUserPoisStatus& NavMiddleware::getImportUserPoisStatus() const
{
   LOG_INTERFACE(location, MESSAGE_TAG_IN"getImportUserPoisStatus()");
   LOG_INTERFACE_DEBUG(location, "\n ############ Import User POIs status ############ - %s",
                       presCtrlInterface.getImportUserPoisStatus().toString().c_str());

   return presCtrlInterface.getImportUserPoisStatus();
}

void NavMiddleware::deleteUserPois()
{
   LOG_INTERFACE_DEBUG(location, MESSAGE_TAG_IN "deleteUserPois()");
   presCtrlInterface.deleteUserPois();
}

void NavMiddleware::muteGuidance(bool guidanceMode)
{
   LOG_INTERFACE(guidance, MESSAGE_TAG_IN"muteGuidance()");
   LOG_INTERFACE_DEBUG(guidance, "guidanceMode = %d", guidanceMode);
   presCtrlInterface.muteGuidance(guidanceMode);
}
//->
