/***************************************************************************
* Copyright(c) 2018-2020 Robert Bosch Car Multimedia GmbH
* This software is property of Robert Bosch GmbH.
***************************************************************************/
/**
 * @addtogroup NavMiddleware
 * @{
 */

#ifndef NAV_MIDDLEWARE_H
#define NAV_MIDDLEWARE_H

#include <map>
#include <string>
#include <set>
#include <list>
//includes could have been forwarded but defined for usage convenience
#include "navmid/PropertyUpdateCallback.h"
#include "navmid/PropertyIdentifiers.h"
#include "navmid/ConfigurationInfos.h"
#include "navmid/DestinationMemoryInfos.h"
#include "navmid/GuidanceInfos.h"
#include "navmid/LocationInfos.h"
#include "navmid/LocationSettings.h"
#include "navmid/MapInfos.h"
#include "navmid/MapPickingResults.h"
#include "navmid/NavDataUpdateInfos.h"
#include "navmid/PositionInfos.h"
#include "navmid/RouteInfos.h"
#include "navmid/TrafficMessageInfos.h"
#include "navmid/RouteSettings.h"
#include "navmid/TrafficSettings.h"
#include "navmid/InfoTypes.h"
#include "navmid/DestinationMemorySettings.h"
#include "navmid/GuidanceSettings.h"
#include "navmid/MapSettings.h"
#include "navmid/EngineeringInfos.h"
#include "navmid/SettingTypes.h"
//#include "navmid/NavMiddlewareExtensions.h"   //<-- INF4CV
#include "navmid/WeatherInfos.h"
#include "navmid/NavDataUpdateSettings.h"
#include "navmid/SdsLocationInfos.h"
#include "navmid/LocationBasedServicesSettings.h"
#include "navmid/ElectricVehicleInfos.h"
#include "navmid/DiagnosticInfos.h"
#include "navmid/ConnectedServicesInfos.h"

#if defined(_MSC_VER)
#define NAV_MIDDLEWARE_EXPORT __declspec(dllexport)
#else
#define NAV_MIDDLEWARE_EXPORT __attribute__((visibility("default")))
#endif

namespace navmiddleware
{

/**
* \brief Navigation middleware interface
* This class provides a collection of methods to access and control navigation functionality.
* Retrieving information is conducted through Getters. Requests can be either asynchronously (ASYNC) or synchronously (no ASYNC).
* In case of ASYNC, the corresponding PropertyIdentifer (trigger) is noted.
* Triggers to retrieve new information are sent via the PropertyUpdate mechanism which require observer registration.
*/
class NAV_MIDDLEWARE_EXPORT NavMiddleware
{
public:
   NavMiddleware();
   ~NavMiddleware();

   ///@{ \name VERSION HANDLING
   ::std::string getVersionString() const;
   int getMajorVersion() const;
   int getMinorVersion() const;
   ::std::string getSwBaseVersion() const;

   enum Supplier
   {
      SUPPLIER__BOSCH,
      SUPPLIER__NEUSOFT
   };

   /**
   * Provides information about the owner of this NavMiddleware
   * @return Supplier (e.g. Bosch for ROW or Neusoft for JP)
   */
   Supplier getSupplier() const;
   ///@}

   ///@{ \name APPLICATION STATE
   /**
   * Starts Navigation Middleware. Initializes connections to navigation servers.
   * \async ::LIFECYCLE_STARTED
   */
   bool initialize(const ConfigurationInfos& configurationInfos);
   /**
   * Shuts down navigation middleware.
   * \async ::LIFECYCLE_STOPPED
   */
   void stop();
   /**
   * Restarts Navigation Middleware.
   * \async ::LIFECYCLE_STOPPED
   * \async ::LIFECYCLE_STARTED
   */
   void restart(const ConfigurationInfos& configurationInfos);

   /**
    *  Resets to Default Settings
    *  @param restoreType defines the kind of data to be restored
    *  @param userProfileGroup defines group of users to which data has to be restored
    */
   void restoreFactorySettings(settings::RestoreType restoreType,
                               settings::UserProfileGroup userProfileGroup);

   /**
    *  Applies all previously updated settings
    *  e.g. RouteOptionsSettings, MapSettings, GuidanceSettings, TrafficInformationSettings etc.
    */
   void applySettings();

   /**
    *  Resets to Default Settings where stored routes, Address book entries and last destinations are not deleted
    *  @param settingsGroup defines group of settings whose defaults has to be restored
    */
   void restoreDefaultSettings(settings::SettingsGroup settingsGroup);

   /**
    * Activates the limited mode
    * Allows to save CPU cycles and I/O bandwidth
    * by limiting functionality of components to what are required during activate limited mode.

    * @param RequiredFeatures - a set of features which are required during activate limited mode:
    * MAP_DISPLAY
    * LOCATION_INPUT
    * ROUTE_GUIDANCE
   */
   void activateLimitedMode(const ::std::set<RequiredFeatures>& requiredFeatures);
   /**
   * Deactivates the limited mode
   */
   void deactivateLimitedMode();

   ///@}

   ///@{ \name CALLBACK REGISTRATION
   void registerLifeCyclePropertyUpdateCallback(PropertyUpdateCallback& callback);
   void unregisterLifeCyclePropertyUpdateCallback(PropertyUpdateCallback& callback);

   void registerPositionPropertyUpdateCallback(PropertyUpdateCallback& callback);
   void unregisterPositionPropertyUpdateCallback(PropertyUpdateCallback& callback);

   void registerGuidancePropertyUpdateCallback(PropertyUpdateCallback& callback);
   void unregisterGuidancePropertyUpdateCallback(PropertyUpdateCallback& callback);

   void registerLocationPropertyUpdateCallback(PropertyUpdateCallback& callback);
   void unregisterLocationPropertyUpdateCallback(PropertyUpdateCallback& callback);

   void registerMapPropertyUpdateCallback(PropertyUpdateCallback& callback);
   void unregisterMapPropertyUpdateCallback(PropertyUpdateCallback& callback);

   void registerRoutePropertyUpdateCallback(PropertyUpdateCallback& callback);
   void unregisterRoutePropertyUpdateCallback(PropertyUpdateCallback& callback);

   void registerTrafficMessagePropertyUpdateCallback(PropertyUpdateCallback& callback);
   void unregisterTrafficMessagePropertyUpdateCallback(PropertyUpdateCallback& callback);

   void registerEngineeringMenuPropertyUpdateCallback(PropertyUpdateCallback& callback);
   void unregisterEngineeringMenuPropertyUpdateCallback(PropertyUpdateCallback& callback);

   void registerDestinationMemoryPropertyUpdateCallback(PropertyUpdateCallback& callback);
   void unregisterDestinationMemoryPropertyUpdateCallback(PropertyUpdateCallback& callback);

   void registerNavDataUpdatePropertyUpdateCallback(PropertyUpdateCallback& callback);
   void unregisterNavDataUpdatePropertyUpdateCallback(PropertyUpdateCallback& callback);

   void registerConfigurationPropertyUpdateCallback(PropertyUpdateCallback& callback);
   void unregisterConfigurationPropertyUpdateCallback(PropertyUpdateCallback& callback);

   void registerIconCacheControlPropertyUpdateCallback(PropertyUpdateCallback& callback);
   void unregisterIconCacheControlPropertyUpdateCallback(PropertyUpdateCallback& callback);

   void registerLocationBasedServicesPropertyUpdateCallback(PropertyUpdateCallback& callback);
   void unregisterLocationBasedServicesPropertyUpdateCallback(PropertyUpdateCallback& callback);

   void registerElectricVehiclePropertyUpdateCallback(PropertyUpdateCallback& callback);
   void unregisterElectricVehiclePropertyUpdateCallback(PropertyUpdateCallback& callback);

   ///@{ \name GENERAL CONFIGURATION

   /**
    * Retrieve startup configuration
    * @return
    */
   const ConfigurationInfos& getConfigurationInfos();

   /**
    * Sets a new language to be used
    * @param languageCodeHMIAsString language code as used on apphmi_navigation for HMI (e.g. "en_US")
    * @param languageCodeHMI language code as used on apphmi_navigation for HMI (e.g. LANGUAGE_CODE__ENGLISH_US)
    * @param languageCodeSDSAsString language code as used on apphmi_navigation for SDS (optional-> "")
    * @param languageCodeSDS language code as used on apphmi_navigation for SDS (optional-> LANGUAGE_CODE__UNUSED)
    * @param languageCodeTrafficTextAsString language code as used on apphmi_navigation for TrafficTextLanguage (optional-> LANGUAGE_CODE__UNUSED)
    * @param languageCodeClusterAsString language code as used on apphmi_navigation for Cluster (optional-> "")
    * @param langaugeCodeCluster language code as used on apphmi_navigation for Cluster (optional -> LANGUAGE_CODE__UNUSED)
    */
   void setLanguage(const ::std::string& languageCodeHMIAsString, LanguageCode languageCodeHMI,
                    const ::std::string& languageCodeSDSAsString, LanguageCode languageCodeSDS,
                    const ::std::string& languageCodeTrafficTextAsString, LanguageCode languageCodeTrafficText,
                    const ::std::string& languageCodeClusterAsString, LanguageCode languageCodeCluster);


   /**
    * Sets the values of displayTimeOffset and timeZoneAndDSTConsideration
    * @param displayTimeOffset - offset value between clock display time and system kernel time in seconds
    * @param timeZoneAndDSTConsideration - This value gives information on whether UtcOffset difference between
    * start position and destination should be considered while calculating ETA.
    * --Expected value in different clock modes:-
    * Manual and Timezone clock modes - TIMEZONE_AND_DST_OFFSET_CONSIDERATION_NOT_REQUIRED
    * Auto Clock mode                 - TIMEZONE_AND_DST_OFFSET_CONSIDERATION_REQUIRED
    */
   void setDisplayTimeOffset(int32_t displayTimeOffset,
                             ConfigurationInfos::TimeZoneAndDSTConsideration timeZoneAndDSTConsideration);

   /**
    * Sets the unit system for distance measures
    * @param unitSystem
    */
   void setDistanceUnitSystem(DistanceUnitSystem unitSystem);

   /**
    * Settings for displaying calendar date and time in the user favored setting
    */
   void setDateFormat(DateFormat dateFormat);
   void setTimeFormat(TimeFormat timeFormat);

   /**
    * Update default position used upon factory reset
    * @param coordinate position used for factory reset
    * @param defaultPositionHeading heading in degrees for default position [0;360)
    */
   void setDefaultPosition(const GeoCoordinateDegree& coordinate, float defaultPositionHeading);

   /**
   * @brief Configure Light Mode give you the option to switch map between DAY and NIGHT mode.
   * @param LightMode           - light mode type that is requested to be configured.
   */
   void configureLightMode(ConfigurationInfos::LightMode lightMode);

   /**
    * @brief Sets current system power mode.
    * @param SystemPowerMode -  current system power mode.
    */
   void setSystemPowerMode(ConfigurationInfos::SystemPowerMode systemPowerMode);

   /**
    * @brief Sets TCU status
    * @param tcuStatus - Current TCU status
    */
   void setTCUStatus(ConfigurationInfos::TCUStatus tcuStatus);

   /**
   * @brief sets the current user profile in the system. Settings and stored destinations specific to user will be restored.
   * @param userProfile - new user profile to be set
   */
   void setUserProfile(ConfigurationInfos::UserProfile userProfile);

   /**
   * @brief copies the data(e.g settings) to the specified target user profile from the current user profile set
   * @param targetUserProfile - the user profile to which the data is to be copied / duplicated
   */
   void copyUserProfileToUser(ConfigurationInfos::UserProfile targetUserProfile);

   /**
    * Retrieve userProfile action results
    * @return
    */
   const UserProfileActionInfos& getUserProfileActionInfos() const;

   /**
    * @brief Clears the user data for a particular user profile. I.e. the call resets the user data to default.
    * @param userProfile user profile to be resetted
    */
   void deleteUserProfile(ConfigurationInfos::UserProfile userProfile);

   /**
   * @brief Sets current system state.
   * @param systemState - current system state
   */
   void setSystemState(ConfigurationInfos::SystemState systemState);

   /**
    * adds a new avoid area
    * @param avoidAreaName
    * @param avoidAreaSize
    * @param avoidAreaMapScale
    * @param edgeLength
    * @param avoidMotorWays
    * @param centerGeoCoordinate
    * \async ::CONFIGURATION_AVOID_AREA_LIST_CHANGED
    */
   void addAvoidArea(const ::std::string& avoidAreaName, const GeoCoordinateDegree& centerGeoCoordinate,
                     AvoidAreaInfo::AvoidAreaSize avoidAreaSize, uint32_t avoidAreaMapScale, uint32_t edgeLength,
                     AvoidAreaInfo::AvoidAreaOptions avoidAreaOptions);

   /**
    * @breif adds a new avoid area
    * This interface is specific for Renault variant
    * @param avoidAreaName        - name of the avoid area
    * @param avoidAreaOptions     - avoid area type to either include or exclude Freeway/Motorways
    * @param screenCoordinates    - list of screen-coordinates with X and Y positions
    * \async ::CONFIGURATION_AVOID_AREA_LIST_CHANGED
    */
   void addAvoidArea(const ::std::string& avoidAreaName,
                     const ::std::vector< ScreenCoordinate >& screenCoordinates,
                     AvoidAreaInfo::AvoidAreaOptions avoidAreaOptions);

   /**
    * modify an existing avoid area
    * @param avoidAreaInfo
    * \async ::CONFIGURATION_AVOID_AREA_LIST_CHANGED
    */
   void modifyAvoidArea(const AvoidAreaInfo& avoidAreaInfo);

   /**
    * @brief modify an existing avoid area
    * This interface is specific for Renault variant
    * @param avoidAreaInfo     - Information related to avoid area
    * @param screenCoordinates - list of screen-coordinates with X and Y positions of the modified avoid area,
    *                            send an empty list in case the area is not changed but only the name or settings
    * \async ::CONFIGURATION_AVOID_AREA_LIST_CHANGED
    */
   void modifyAvoidArea(const AvoidAreaInfo& avoidAreaInfo,
                        const ::std::vector< ScreenCoordinate >& screenCoordinates);

   /**
   * @brief filters avoid area list using a search string
   * The filter applies until it is reset by passing an empty string
   * @param nameFilter pattern to filter list
   *
   * \async ::CONFIGURATION_AVOID_AREA_LIST_CHANGED
   */
   void setAvoidAreaListFilter(const ::std::string& nameFilter);

   /**
    * get the list of avoided areas
    * @param
    * @return AvoidAreaInfos obj
    */
   const AvoidAreaInfos& getAvoidAreaInfos() const;

   /**
    * delete an existing avoid area
    * @param avoidAreaId
    * \async ::CONFIGURATION_AVOID_AREA_LIST_CHANGED
    */
   void deleteAvoidArea(AvoidAreaId avoidAreaId);

   /**
    * delete all existing avoid areas
    * @param
    * \async ::CONFIGURATION_AVOID_AREA_LIST_CHANGED
    */
   void deleteAllAvoidAreas();

   /**
    * Draw dynamic areas with user-defined style on the map.
    * Any dynamic areas added by previous call will be removed before the requested dynamic areas are added.
    *
    * @param dynamicAreaUserStyleList list of dynamic areas specifying the user-defined styles
    * \async ::MAP_ADD_DYNAMIC_AREA_USER_STYLE_RESULT_AVAILABLE
    */
   void addDynamicAreaUserStyle(const ::std::vector<DynamicAreaUserStyle>& dynamicAreaUserStyleList);

   /**
    * @brief Getter to obtain the result of addDynamicAreaUserStyle().
    * Call after property update ::MAP_ADD_DYNAMIC_AREA_USER_STYLE_RESULT_AVAILABLE.
    * Returns if the previous addDynamicAreaUserStyle() call is successful or failed.
    */
   AddDynamicAreaUserStyleResult getAddDynamicAreaUserStyleResult() const;

   /**
    * Delete all dynamic areas of user-defined style from the map.
    */
   void deleteAllDynamicAreaUserStyle();

   /** Returns all config items to be registered
    */
   const RegisteredConfigItemsList& getConfigItemsToBeRegistered() const;

   /** Returns all configuration items as requested by getConfigItems.
    * All known config items are synchronously available, for unknown group names or other failures,
    * result structure offers a proper result code.
    * @param group name of the requested config items
    */
   const ConfigItemsGetResult& getConfigItems(const ::std::string& configItemGroupName) const;

   /** Whenever an update for a registered config item group is received setConfigItems shall be called to forward
    * received key/value pairs.
    * @param the name of the config item group
    * @param a vector list of key/value pairs.
    * \async ::CONFIGURATION_SET_CONFIG_RESULT as soon as the result is available.
    */
   void setConfigItems(const ::std::string& configItemGroupName, const ::std::vector< ConfigItem>& configItems);

   /** returns the result of previously sent set request.
    * \async ::CONFIGURATION_SET_CONFIG_RESULT as soon as the result is available.
    */
   ConfigResultCode getSetConfigResult() const;

   /**
    * Request to get the version tag of navigation delivery.
    */
   void requestNavigationVersion();

   /**
    * returns the previously requested navigation version tag
    * \async ::CONFIGURATION_NAVIGATION_VERSION_CHANGED as soon as the result is available.
    */
   const ::std::string& getNavigationVersion() const;

   /**
    * @brief Sets a Privacy mode
    * @param privacyMode - current Privacy mode
    */
   void setPrivacyMode(ConfigurationInfos::PrivacyMode privacyMode);

   /**
   * @brief Interface to inform NavMiddleware about the type of cluster that is connected.
   * @param[in] connectedClusterType     - Type of cluster that is connected
   */
   void setConnectedClusterType(ConnectedClusterType connectedClusterType);

   ///@}

   // ================================================================================================================
   ///@{ \name POSITION

   /**
    * Returns PositionInfo containing all position related info in current HMI language.
    * Update after property callback ::POSITION_INFO_CHANGED
    * \return reference to a PositionInfo object.
    */
   const PositionInfo& getPositionInfo() const;
   /**
    * Returns SdsPositionInfo containing all position related info in current SDS language.
    * Update after property callback ::POSITION_INFO_CHANGED
    * \return reference to a SdsPositionInfo object.
    */
   const SdsPositionInfo& getSdsPositionInfo() const;

   /**
    * Returns PositionStatusInfo containing status related info.
    * Update after property callback ::POSITION_STATUS_CHANGED
    */
   const PositionStatusInfo& getPositionStatusInfo() const;

   /**
    * Returns the info if the quality of the current position is sufficient or not.
    * Update after property callback ::POSITION_QUALITY_CHANGED
    * \return enum value of PositionQuality
    */
   PositionQuality getPositionQuality() const;

   /**
    * Returns the info about the availability of RNW (Road NetWork) data at current vehicle position.
    * Update after property callback ::POSITION_RNW_DATA_AVAILABILITY_CHANGED
    * \return enum value of RnwDataAvailability
    */
   RnwDataAvailability getRnwDataAvailabilityAtCurrentPosition() const;

   /**
    * Returns PositioniningDebugInfo containing all position related info like
    * GNSS, Odometer, Gyro, Acc, Map matching, Abs, Dr, Calibration Info required for Renault debug screen.
    * Update after property callback ::POSITION_DEBUG_INFO_CHANGED
    * \return reference to a PositionDebugInfo object.
    */
   const PositionDebugInfo& getPositionDebugInfo() const;

   /**
     * request for the position debug information.
     * requested information will be continuously updated until release is called.
     * \async ::POSITION_DEBUG_INFO_CHANGED
     */
   void requestPositionDebugInfo();

   /**
     * Release the position debug information.
     * Should be called when the debug information is not required anymore.
     */
   void releasePositionDebugInfo();

   /**
    * Sets the position
    */
   void setPositionByCoordinates(const GeoCoordinateDegree& geoCoordinate, float heading);

   /**
    * Enables Where am I feature. Afterwards property updates ::POSITION_WHERE_AM_I_CHANGED are given on each change
    * of Where am I info.
    */
   void enableWhereAmIInfo() const;
   /**
    * Disables Where am I feature, i.e. no further property updates are given
    */
   void disableWhereAmIInfo() const;
   /**
    * Returns current WhereAmIInfo info. (feature must be enabled by calling enableWhereAmIInfo before)
    *
    * Where am I info requires quite high cpu load. Therefore this feature must be enabled explicitly to receive
    * property updates and disabled explicitly if contents are not needed (displayed) any longer.
    *
    * Update after property callback ::POSITION_WHERE_AM_I_CHANGED
    */
   const WhereAmIInfo& getWhereAmIInfo() const;

   /**
   * Getter for the GNSS Data containing the Satellite Information
   * Update after property callback ::POSITION_SENSORINFO_CHANGED
   * \return reference to a SensorInfo object which holds Satellite information.
   */
   const GnssDataInfo& getGnssDataInfo() const;

   /**
    * Returns the info if the real vehicle is moving or not (independent of possibly active demo mode).
    * Update after property callback ::POSITION_VEHICLE_MOVING_STATE_CHANGED
    * \return the enumeration value of VehicleMovingStatus
    */
   VehicleMovingStatus getVehicleMovingState() const;

   /**
    * Starts the trip-replay for the given trip-file
    */
   void startTripReplay(const ::std::string& tripFileName);
   /**
    * Stops the trip-replay
    */
   void stopTripReplay();
   /**
    * Seeks to the specified location, in the trip-replay
    */
   void seekTripReplay(int16_t seekValue);
   /**
    * Sets the mode of the trip-replay
    */
   void setTripReplayMode(bool loopModeActive, uint16_t speed);
   /**
    * Getter for the trip-replay-status (TripReplayStatusInfo) info.
    * Update after property callback ::POSITION_TRIP_REPLAY_STATUS_CHANGED
    */
   const TripReplayStatusInfo& getTripReplayStatusInfo() const;

   /**
    * Getter for the trip-recording-status (TripRecordingStatusInfo) info.
    * Update after property callback ::POSITION_TRIP_RECORDING_STATUS_CHANGED
    */
   const TripRecordingStatusInfo& getTripRecordingStatusInfo() const;

   /**
    * This function is to be used by the HMI to request Navmiddleware
    * to store a user marker in the trip file.
    */
   void writeTripMarker();

   /*
    * Returns the current status of vehicle w.r.t HOME location.
    * Update after property callback ::POSITION_HOME_POSITION_STATUS_CHANGED
    * \return the enumeration value of HomePositionStatus */
   HomePositionStatus getHomeStatus() const;

   // ================================================================================================================
   ///@{ \name POSITION SIMULATION

   /**
    * Activates the demo mode
    */
   void activateDemoMode();

   /**
    * Deactivates the demo mode
    */
   void deactivateDemoMode();

   /**
    * Returns if the demo mode is currently enabled
    * Updated after property callback ::POSITION_SIMULATION_MODE_CHANGED
    */
   bool isDemoModeActive() const;

   /**
    * Sets the demo mode speed. Range is 0%-1000%, 0=stop, 100=normal speed, 1000=10x normal speed
    */
   void setDemoModeSpeed(unsigned int speedInPercent);

   /**
   * Sets the demo position for simulation
   * \async ::POSITION_INFO_CHANGED
   */
   void setDemoModePosition(double latitude, double longitude, float heading);

   /**
   * Sets the position simulation mode
   */
   void setPositionSimulationMode(PositionSimulationMode simulationMode);

   /**
   * Returns if the position simulation is active or inactive
   */
   bool isSimulationActive() const;
   ///@}

   ///@}

   // ================================================================================================================
   ///@{ \name LOCATION

   /**
    * Stores the given WGS84 coordinates for further processing
    */
   void setLocationWithCoordinates(double latitude, double longitude);

   /**
    * Stores the given externalLocation for further processing
    */
   void setLocationWithExternalLocation(const ExternalLocation& externalLocation);

   /**
    * Stores the given DestinationId for further processing
    */
   void setLocationWithDestinationMemoryId(const DestinationMemoryId& destinationMemoryId);

   /**
    * Stores the given waypoint index for further processing (re-creating a location from a waypoint)
    * @param waypointIndex  0 - 1st waypoint, 1 - 2ndwaypoint, ..., n - (final) destination
    */
   void setLocationWithWaypointIndex(unsigned int waypointIndex);

   /**
    * Stores the picking result element for further processing
    */
   void setLocationWithPickingResult(const PickingResults::PickingResult& pickingResult);

   /**
    * Stores the given POI data for further processing
    */
   void setLocationWithPoiData(const PoiData& poiData);

   /**
    * Request the location attributes for the last input location (setLocationWith...)
    * Returns the requestId to validate the response.
    * \async ::LOCATION_ATTRIBUTES_AVAILABLE
    * \async ::LOCATION_ATTRIBUTES_ERROR on Off_map Location
    */
   RequestId requestLocationAttributes();

   /**
    * Returns the location attributes which were requested last
    * Updates of the state are announced with property update LOCATION_ATTRIBUTES_AVAILABLE
    */
   const LocationAttributeInfos& getLocationAttributes() const;

   /**
    * Returns the error Information when requested for LocationAttributes.
    * Updates of the state are announced with property update LOCATION_ATTRIBUTES_ERROR
    */
   const LocationAttributeErrorInfo& getRequestLocationAttributesError() const;

   /**
    * Return the current state of the online search availability.
    * Updates of the state are announced with property update LOCATION_ONLINE_SEARCH_AVAILABILITY_CHANGED
    * @deprecated Please use getOnlineFreeTextSearchAvailability().getDefaultOnlineSearchAvailability() instead
    */
   OnlineServiceAvailability getLocationOnlineSearchAvailability() const;

   /**
    * Returns the current locationString (combination of country city street house-no) in string format to the HMI.
    *
    * Note: Current location must have been finalized before this method will return a (non-empty) string!!
    *       Finalization can be achieved by requestLocationDetails or starting route calculation or guidance.
    */
   const ::std::string& getCurrentLocationString() const;

   /** requests the location attribute information (e.g. country and city name) of the given geo-coordinate
    *
    * Returns the requestId to validate the response
    * @param geoCoordinate   - geographical coordinate
    * @param isOnBoardMatchingRequired  -  if true : the given coordinate is matched to on-board data and provides the details.
    *                                      if false: for the given coordinate the details are provided.
    * \async LOCATION_COORDINATE_RESOLVED
    */
   RequestId requestAddressAttributesForCoordinate(const GeoCoordinateDegree& geoCoordinate, bool isOnBoardMatchingRequired = false);

   /**
    * Returns the location attributes which were requested for a geo-coordinate
    * Updates of the state are announced with property update LOCATION_COORDINATE_RESOLVED
    */
   const LocationAttributeInfos& getAddressAttributesOfCoordinate() const;

   /**
    * Returns the error Information for the request requestAddressAttributesForCoordinate.
    * Updates of the state are announced with property update LOCATION_COORDINATE_RESOLVE_FAILED
    * \async LOCATION_COORDINATE_RESOLVE_FAILED
    */
   const LocationAttributeErrorInfo& getAddressAttributesOfCoordinateError() const;

   /**
    * Normally, the contained distance and direction values of requested location elements (e.g. city names,
    * poi elements, free text search results) are continuously updated with the current position. This function
    * allows to disable these updates. It is contained for testing purposes only.
    *
    * Note: The status is only taken into account with the call of the start...Input or the setSearchString function,
    *       respectively.
    *
    * @param allow if true updates are given; if false no updates will happen
    */
   void setAllowPositionBasedDistanceAndDirectionUpdates(bool allow);

   /** requests the address format table in NDS Data
    *
    * \async LOCATION_ADDRESS_FORMAT_AVAILABLE
    */
   void requestAddressFormatForAvailableCountries();

   /** gets the information about the address format and country code.
    *  - country code returned as ISO 3166-1-alpha-3.
    *  - Eg : "DEU"
    *  - addressFormat returned is empty address in xml tags in proper order as per the country.
    *  - Eg: "<ROAD/><HNR/><ZIP/><PLACE/><COUNTRY/>"
    * @return an object holding the above description addressformatInfo
    */
   const AddressFormatInfos& getAddressFormatInfos() const;

   // Addresses near Position Search  ------------------------------------------------------------------------------

   /** requests the addresses information for the given position and within given radius.
      * The list of addresses are returned in LocationInfo's.
      * The detailed information includes countryName/countryCode/state/city/street/postalcode and position.
      * @param geoCoordinate     - position to search at
      * @param radiusInMeters    - radius to be considered around the position
      *                          - maximum 1 km is allowed.
      * @param numberOfLocations - number of locations requested
      *                          - maximum 20 locations can be provided.
      * \async :: LOCATION_ADDRESS_INFOS_BASED_ON_POSITION_CHANGED as soon as the requested locations are available.
      */
   void requestAddressesNearPosition(const GeoCoordinateDegree& geoCoordinate, unsigned int radiusInMeters, unsigned int numberOfLocations);

   /** gets the information about the list of addresslocations.
    * @return an object holding the details of the requested locations around a position.
    */
   const LocationInfos& getAddressesNearPositionInfos() const;

   /**
    * Request to load the image for the specified reference
    * @param pictureRefInfo info containing the source id, reference and the max dimensions of the image to be loaded.
    * @param height the actual height of the needed image in Pixel (should be <= maximum Height in Pixel)
    * @param width the actual width of the needed image in Pixel (should be <= maximum Width in Pixel)
    */
   void requestLocationImage(const ::navmiddleware::LocationAttributeInfos::PictureReferenceInfo& pictureRefInfo,
                             unsigned int requiredHeightInPixel, unsigned int requiredWidthInPixel);

   /**
    * Returns the last requested location image
    * Should be called after receiving property update LOCATION_IMAGE_INFO_AVAILABLE.
    */
   const ::navmiddleware::Image& getLocationImageInfo() const;

   // LocationSettings -----------------------------------------------------------------------------------------------

   /**
    * Get the EvChargingStationFilterSettings settings object which is hosted by middleware
    */
   settings::EvChargingStationFilterSettings getEvChargingStationFilterSettings() const;

   /**
    * Send updated Hmi location settings back to the middleware.
    * After calling this method, changes on the local object will not affect
    * the passed reference (middleware creates a copy). To update middleware settings,
    * call this method again.
    */
   void updateEvChargingStationFilterSettings(const settings::EvChargingStationFilterSettings&);

   /**
    * Get the PaymentMethodSettings settings object for EV charging stations which is hosted by middleware
    */
   settings::PaymentMethodSettings getEvChargingStationPaymentMethodSettings() const;

   /**
    * Send updated PaymentMethodSettings for EV charging stations back to the middleware.
    * After calling this method, changes on the local object will not affect
    * the passed reference (middleware creates a copy). To update middleware settings,
    * call this method again.
    */
   void updateEvChargingStationPaymentMethodSettings(const settings::PaymentMethodSettings&);

   /**
    * Get the EvConnectorTypeFilterSettings settings object which is hosted by middleware
    */
   settings::EvConnectorTypeFilterSettings getEvConnectorTypeFilterSettings() const;

   /**
    * Send updated EvConnectorTypeFilterSettings settings back to the middleware.
    * After calling this method, changes on the local object will not affect
    * the passed reference (middleware creates a copy). To update middleware settings,
    * call this method again.
    */
   void updateEvConnectorTypeFilterSettings(const settings::EvConnectorTypeFilterSettings&);

   /**
    * Get the ParkingSettings object which is hosted by middleware
    */
   settings::ParkingSettings getParkingSettings() const;

   /**
    * Send updated Hmi location settings back to the middleware.
    * After calling this method, changes on the local object will not affect
    * the passed reference (middleware creates a copy). To update middleware settings,
    * call this method again.
    */
   void updateParkingSettings(const settings::ParkingSettings&);

   /**
    * Get the EvChargingSpeedSettings object which is hosted by middleware
    */
   settings::EvChargingSpeedFilterSettings getEvChargingSpeedFilterSettings() const;

   /**
    * Send updated Hmi location settings back to the middleware.
    * After calling this method, changes on the local object will not affect
    * the passed reference (middleware creates a copy). To update middleware settings,
    * call this method again.
    */
   void updateEvChargingSpeedFilterSettings(const settings::EvChargingSpeedFilterSettings&);

   // Freeway Destination --------------------------------------------------------------------------------------------

   /**
    * If there is a motorway in the selected location(by pick or by Coordinates),
    * the user's denial to take the motorway is indicated to the Middleware by the below call.
    * Property Update :: LOCATION_ATTRIBUTES_AVAILABLE to be expected from Middleware.
    * Property Update :: LOCATION_ATTRIBUTES_ERROR if no non motorway links available.
    */
   void denyMotorwayUsageInCurrentLocation();

   // Hierarchical / form-based address input ------------------------------------------------------------------------

   /**
    * Starts a form-based address input.
    *
    * @param resetContents if true starts a new (empty) input (possibly with country/city preselected).
    *         If false the last address input state is kept.
    * @param  context Defines the context in which the newly started address input will be used.
    * \async ::LOCATION_ADRESS_INPUT_INFO_CHANGED as soon as the address input is ready.
    */
   void startFormBasedAddressInput(const LocationContext& context, bool resetContents = false);

   /**
    * Starts a form-based address input by cloning the content of another input instance.
    * With this call the new instance becomes the active input instance.
    *
    * @param  context Defines the context in which the newly started address input will be used.
    * @param  instanceNumberToBeCloned  number of the input instance being cloned into this new instance,
    *         must be different from the (new) instance number passed in parameter "context"
    * \async ::LOCATION_ADRESS_INPUT_INFO_CHANGED as soon as the address input is ready.
    */
   void startAddressInputClone(const LocationContext& context, unsigned int instanceNumberToBeCloned);

   /**
    * Starts a form-based address input address details (e.g. an FTS search result)
    *
    * @param  context Defines the context in which the newly started address input will be used.
    * @param  addressDetaiInfo used to prefill the newly started address input.
    * \async ::LOCATION_ADRESS_INPUT_INFO_CHANGED as soon as the address input is ready.
    */
   void startFormBasedAddressInputFromAddressDetails(const LocationContext& context, const LocationAttributeInfos::AddressDetailInfo& addressDetaiInfo);

   /**
    * Get the information to fill a form-based address input screen:
    *  - Which address categories are valid for change,
    *  - what is the current content of the fields,
    *  - is the so far input location valid.
    *
    * @return an object holding the above described information.
    */
   const AddressInputInfo& getFormBasedAddressInputInfo() const;

   /**
    * Select the specified address category for input.
    *
    * @param category to be selected
    * @param numFirstMatches Number of first matching elements to be requested
    *
    * \async ::LOCATION_ADDRESS_SPELLER_INFO_CHANGED as soon as the address category is selected.
    * \async ::LOCATION_ADDRESS_ELEMENTS_CHANGED as soon as the address category is selected
    * Afterwards the address speller may be used and address elements can be requested.
    */
   void selectAddressCategoryForInput(AddressCategory category, unsigned int numFirstMatches = 0);

   /**
    * Return the current address input speller info.
    * @return current address input speller info
    */
   const ElementSpellerInfo& getAddressSpellerInfo() const;

   /**
    * Set the address speller string.
    *
    * @param spellerString to be set
    * @param numFirstMatches Number of first matching elements to be requested
    *
    * \async ::LOCATION_ADDRESS_SPELLER_INFO_CHANGED as soon as the speller changed is processed.
    * \async ::LOCATION_ADDRESS_ELEMENTS_CHANGED if numFirstMatches is > 0 and the requested elements are available.
    */
   void setAddressSpellerString(const ::std::string& spellerString, unsigned int numFirstMatches = 0);

   /**
    * Request the specified range of address elements (e.g. countries, cities, streets, ...) matching the set speller
    * string.
    * @param startIndex of the first element to be returned by getAddressElements()
    * @param numElements to be returned by getAddressElements()
    *
    * \async ::LOCATION_ADDRESS_ELEMENTS_CHANGED as soon as the requested elements are available.
    */
   void requestAddressElements(unsigned int startIndex, unsigned int numElements);

   /** Returns the list of address elements requested by requestAddressElements */
   const LocationInfos& getAddressElements() const;

   /**
    * Select an address element returned by getAddressElements() refining the address input.
    * @param locationInfo an element out of the LocationInfos list returned by getAddressElements()
    *
    * \async ::LOCATION_ADRESS_INPUT_INFO_CHANGED as soon as the given element is selected.
    */
   void selectAddressElement(const LocationInfos::LocationInfo& locationInfo);

   /**
    * Requests the list of last 5 cities.
    * \async: LOCATION_LAST_SELECTED_ADDRESS_ELEMENTS_CHANGED
    */
   void requestLastSelectedAddressElements();

   /**
    * Returns the list of last 5 cities requested by requestLastSelectedAddressElements
    */
   const LocationInfos& getLastSelectedAddressElements();

   /**
    * Select an element returned by getLastSelectedAddressElements() refining the address input.
    * @param locationInfo an element out of the LocationInfos list returned by getLastSelectedAddressElements()
    *
    * \async ::LOCATION_ADRESS_INPUT_INFO_CHANGED as soon as the given element is selected.
    */
   void selectFromLastSelectedAddressElements(const LocationInfos::LocationInfo& locationInfo);

   /**
    * Undo the last address input step:
    *  - If currently the speller screen is shown, this undoes the last category selection.
    *    \async ::LOCATION_ADRESS_INPUT_INFO_CHANGED if undoing to a category scope
    *  - If the form screen is shown, this undoes the last element selection.
    *    \async ::LOCATION_ADDRESS_SPELLER_INFO_CHANGED if undoing to an element scope
    *    \async ::LOCATION_ADDRESS_ELEMENTS_CHANGED if undoing to an element scope
    */
   void undoLastAddressInputStep();

   /**
    * Clears the current content (element) of the specified address category.
    * @param category to be cleared
    *
    * \async ::LOCATION_ADRESS_INPUT_INFO_CHANGED as soon as the address category's contents is cleared.
    */
   void clearAddressCategory(AddressCategory category);

   /**
    * Sets the current address input (as far as done) as the current working location, which may be used to get
    * details about, start guidance for, etc.
    *
    * @return true if address input is successfully set as current working location; false if the
    *         address input is not unique i.e. AddressInputInfo::isValidDestination() does not return true
    */
   bool setLocationWithCurrentAddressInput();

   /**
    * Used to notify that form-based location input has been left i.e. HMI switched to a different screen
    * Internally resources are cleaned up, deregistration from lower layers is done, etc.
    */
   void leaveFormBasedAddressInput();

   /**
    * This interface specifies the house number speller content should be evaluated or not.
    * @ param status if true, house number validation is done after a selection from list and the flags are updated.
    *                if false, a list of valid house numbers is returned to the client after a list request.
    * Once this Interface is called, the default behavior is overridden in NavMiddleware.
    */
   void setPostValidationForHouseNumberInput(bool status);

   /**
    * As japanese address input is not really form based, but rather sequential,
    * there is a need to tell middleware when it should consider address as complete.
    * This method sets scope, that middleware shall respect when it evaluates
    * whether current address input information is complete (i.e. destination is valid).
    * [JP]
    */
   void setFormBasedAddressInputScope(AddressCategory scope);

   /**
   * Returns current address input scope.
   * [JP]
   *
   * @return Address input scope.
   */
   AddressCategory getFormBasedAddressInputScope() const;


   // Hierarchical / form-based POI input ----------------------------------------------------------------------------

   /**
    * Return the current state of the online poi availability.
    * Updates of the state are announced with property update LOCATION_ONLINE_POI_AVAILABILITY_CHANGED.
    * In this a map is returned , which consists of predefined poi category and its corresponding online service availability.
    *
    * NOTE :
    * If a client needs to know the online service availability for a particular poi, which is not listed in predefined poi categories
    * the following enum PREDEFINED_POI_CATEGORY__UNDEFINED as generic can be used.
    */
   const OnlinePoiAvailabilityInfo& getOnlinePoiAvailability() const;

   /**
    * This call requests the server and informs about the online server availability.
    * @param server defines which server to be evaluated like TomTom(LOCATIONSEARCHSOURCE__ONLINE) or Google(LOCATIONSEARCHSOURCE__GOOGLE_PLACES).
    * Supported server TomTom(LOCATIONSEARCHSOURCE__ONLINE)
    * \async ::LOCATION_ONLINE_POI_AVAILABILITY_CHANGED
    */
   void requestPoiOnlineServerAvailability(LocationSearchSource server);

   /** If the requested search scope results in empty list, then a fall back executes for all poi start methods.
    * @param searchScope  : TYPE__ALONG_ROUTE/TYPE__ALONG_CURRENT_HIGHWAY
    * @param searchRadius : SEARCH_RADIUS__REACHABLE_AREA
    * @param fall back searchScope  : TYPE__AROUND_CURRENT_VEHICLE_POSITION
    * @param fall back searchRadius : SEARCH_RADIUS__REACHABLE_AREA
    */

   /**
    * Start POI input at top-level POI categories
    * @param searchScope  defines the search scope to be used initially
    * @param searchSource  defines the search source to be used initially
    * @param context  specifies the LocationContext in which the newly started POI input shall be used
    * \async ::LOCATION_POI_INPUT_INFO_CHANGED
    * \async ::LOCATION_POI_ENTRIES_CHANGED
    */
   void startHierarchicalPoiInput(
      const LocationSearchScope& searchScope = LocationSearchScope(),
      LocationSearchSource searchSource = LOCATIONSEARCHSOURCE__OFFLINE,
      const LocationContext& context = LocationContext(LocationContext::HMICONTEXT__GUI, APPLICATIONID__NAVIGATION));

   /**
    * Start POI input using one of n possible category shortcuts,
    * where n is defined by the number of CategoryShortcutInfos return by getPoiShortcutCategoryInfos().
    * @param shortcutNum  number of the category shortcut to take; must be 0 <= shortcutNum < n
    * @param searchScope  defines the search scope to be used initially
    * @param searchSource  defines the search source to be used initially
    * @param poiCategoryFilter defines the filter criteria to be applied for the given shortcutNum
    * @param context  specifies the LocationContext in which the newly started POI input shall be used
    * \async ::LOCATION_POI_INPUT_INFO_CHANGED
    * \async ::LOCATION_POI_ENTRIES_CHANGED
    */
   void startPoiInputWithCategoryShortcut(
      unsigned int shortcutNum,
      const LocationSearchScope& searchScope = LocationSearchScope(),
      LocationSearchSource searchSource = LOCATIONSEARCHSOURCE__OFFLINE,
      PoiCategoryFilter poiCategoryFilter = POI_CATEGORY_FILTER__NO_FILTER,
      const LocationContext& context = LocationContext(LocationContext::HMICONTEXT__GUI, APPLICATIONID__NAVIGATION));

   /**
    * Start POI input searching POI elements of all shortcut categories ("proximity POI search").
    * @param searchScope  defines the search scope to be used initially
    * @param searchSource  defines the search source to be used initially
    * @param context  specifies the LocationContext in which the newly started POI input shall be used
    * \async ::LOCATION_POI_INPUT_INFO_CHANGED
    * \async ::LOCATION_POI_ENTRIES_CHANGED
    */
   void startPoiInputWithinAllShortcutCategories(
      const LocationSearchScope& searchScope = LocationSearchScope(),
      LocationSearchSource searchSource = LOCATIONSEARCHSOURCE__OFFLINE,
      const LocationContext& context = LocationContext(LocationContext::HMICONTEXT__GUI, APPLICATIONID__NAVIGATION));

   /**
    * Start POI input for predefined POI categories
    * @param predefinedPoiCategory  predefined category to start search within (e.g. petrol stations, hospitals, etc)
    * @param searchScope  defines the search scope to be used initially
    * @param searchSource  defines the search source to be used initially
    * @param poiCategoryFilter defines the filter criteria to be applied for the given predefinedPoiCategory
    * @param context  specifies the LocationContext in which the newly started POI input shall be used
    * \async ::LOCATION_POI_INPUT_INFO_CHANGED
    * \async ::LOCATION_POI_ENTRIES_CHANGED
    */
   void startPoiInputWithPredefinedCategory(
      PredefinedPoiCategory predefinedPoiCategory,
      const LocationSearchScope& searchScope = LocationSearchScope(),
      LocationSearchSource searchSource = LOCATIONSEARCHSOURCE__OFFLINE,
      PoiCategoryFilter poiCategoryFilter = POI_CATEGORY_FILTER__NO_FILTER,
      const LocationContext& context = LocationContext(LocationContext::HMICONTEXT__GUI, APPLICATIONID__NAVIGATION));

   /** Start POI input with the POI category selected in last free text search
    * @param searchScope  defines the search scope to be used initially
    * @param searchSource  defines the search source to be used initially
    * @param context  specifies the LocationContext in which the newly started POI input shall be used
    * \async ::LOCATION_POI_INPUT_INFO_CHANGED
    * \async ::LOCATION_POI_ENTRIES_CHANGED
    */
   void startPoiInputWithInFtsChosenCategory(
      const LocationSearchScope& searchScope = LocationSearchScope(),
      LocationSearchSource searchSource = LOCATIONSEARCHSOURCE__OFFLINE,
      const LocationContext& context = LocationContext(LocationContext::HMICONTEXT__GUI, APPLICATIONID__NAVIGATION));

   /**
    * Resumes an existing POI input instance. The instance being resumed must have been created before by one of
    * the start POI input functions declared above. Unlike the start functions this resume function resumes the
    * input in the state (current input scope, search scope, search source, ...) as it was left before (means,
    * it does not reset the state of the resumed input).
    *
    * The specified instance becomes the active POI input (there can be only one active POI input at a time). A
    * possible other active POI input instance will be implicitly paused by this call.
    *
    * If the referenced instance is not existing, the possible currently active instance stays active.
    * If the referenced instance is already the active one, this call is silently ignored.
    *
    * @param context  specifies the LocationContext for the POI Input instance to be resumed
    * \async ::LOCATION_POI_INPUT_INFO_CHANGED
    * \async ::LOCATION_POI_ENTRIES_CHANGED
    */
   void resumePoiInputInstance(const LocationContext& context);

   /**
    * Return PoiInputInfo representing current state of POI input.
    * Should be called after receiving property update LOCATION_POI_INPUT_INFO_CHANGED.
    */
   const PoiInputInfo& getPoiInputInfo() const;

   /**
    * Set the sort order of the current list of POI entries (categories or elements).
    * Available sort orders and current sort order can be queried in the LocationInfos object returned by
    * getPoiElements().
    * @param sortOrder  to be set
    * \async ::LOCATION_POI_ENTRIES_CHANGED
    */
   void setPoiSortOrder(LocationSortOrder sortOrder);

   /**
    * Set the search scope for the Poi search.
    * @param searchScope  defines the search scope
    * \async ::LOCATION_POI_ENTRIES_CHANGED
    */
   void setPoiSearchScope(const LocationSearchScope& searchScope);

   /**
    * Set the search source of POI search
    * @param searchSource  defines the search source to be used
    * \async ::LOCATION_POI_ENTRIES_CHANGED
    */
   void setPoiSearchSource(LocationSearchSource searchSource);

   /**
    * Request location details for the specified POI element (POI input must be in element scope)
    * @param listIndex absolute index (0..LocationInfos::getCountOfAllLocations()-1) to request the attributes for
    * \async ::LOCATION_POI_ENTRY_ATTRIBUTES_AVAILABLE
    * \async ::LOCATION_POI_ENTRY_ATTRIBUTES_ERROR
    */
   void requestPoiEntryAttributes(unsigned int listIndex);

   /**
    * Return the POI attributes of last requested POI element entry.
    * Should be called after receiving property update LOCATION_POI_ENTRY_ATTRIBUTES_AVAILABLE.
    */
   const LocationAttributeInfos& getPoiEntryAttributes() const;

   /**
    * Returns the error Information when requested for PoiEntryAttributes.
    * Should be called after receiving property update LOCATION_POI_ENTRY_ATTRIBUTES_ERROR
    */
   const LocationAttributeErrorInfo& getPoiEntryAttributesError() const;

   /**
    * Request the specified range of entries in the current POI input scope (categories or elements)
    * @param startIndex  index of first entry to be requested
    * @param numEntries  number of entries to be requested
    * \async ::LOCATION_POI_ENTRIES_CHANGED
    */
   void requestPoiEntries(unsigned int startIndex, unsigned int numEntries);

   /**
    * Return the POI entries (categories or elements) of latest request.
    * Should be called after receiving property update LOCATION_POI_ENTRIES_CHANGED.
    */
   const LocationInfos& getPoiEntries() const;

   /**
    * Select an entry in current POI input scope (category or element)
    * @param locationInfo  entry to be selected (must be part of LocationInfos last returned by getPoiElements())
    * \async ::LOCATION_POI_INPUT_INFO_CHANGED
    * \async ::LOCATION_POI_ENTRIES_CHANGED
    * @DEPRECATED: Please use selectPoiEntry(unsigned int) instead
    */
   void selectPoiEntry(const LocationInfos::LocationInfo& locationInfo);

   /**
    * Select an entry in current POI input scope (category or element)
    * @param listIndex  absolute index (0..LocationInfos::getCountOfAllLocations()-1) to be selected
    * \async ::LOCATION_POI_INPUT_INFO_CHANGED
    * \async ::LOCATION_POI_ENTRIES_CHANGED
    */
   void selectPoiEntry(unsigned int listIndex);

   /**
    * Select the AllPOI Category.
    * \async ::LOCATION_POI_INPUT_INFO_CHANGED
    * \async ::LOCATION_POI_ENTRIES_CHANGED
    */
   void selectAllPoiCategory();

   /**
    * Request to undo the last entry selection.
    * If called at top-level this request is silently ignored.
    *
    * \async ::LOCATION_POI_INPUT_INFO_CHANGED
    * \async ::LOCATION_POI_ENTRIES_CHANGED
    */
   void undoLastPoiEntrySelection();

   /**
    * Set the current POI input as the current working location, which may be used to get details about, start
    * guidance for, etc.
    *
    * Note: A POI element must be finally selected to make this call successful.
    *
    * @return true if POI input is successfully set as current working location; false if the
    *         POI input is not unique i.e. PoiInputInfo::isValidDestination() does not return true
    */
   bool setLocationWithCurrentPoiInput();

   /**
    * Notify that hierarchical POI location input has been left, i.e. HMI switched to a different screen
    * Internally resources are cleaned up, deregistration from lower layers is done, etc.
    */
   void leaveHierarchicalPoiInput();

   /**
    * Return current POI category speller data.
    * Should be called after receiving property update LOCATION_POI_CATEGORY_SPELLER_INFO_CHANGED.
    */
   const ElementSpellerInfo& getPoiCategorySpellerInfo() const;

   /**
    * Request to set a (changed) speller string
    * @param spellerString  speller to be set
    * @param numFirstEntries  defines the number of first matching entries to be requested
    * \async ::LOCATION_POI_CATEGORY_SPELLER_INFO_CHANGED
    * \async ::LOCATION_POI_ENTRIES_CHANGED (if numFirstMatches > 0)
    */
   void setPoiCategorySpellerString(const ::std::string& spellerString, unsigned int numFirstEntries = 0);

   // POI shortcut configuration -------------------------------------------------------------------------------------

   /**
    * Get the current POI shortcut configuration (POI name and icon data per shortcut)
    *
    * This information is updated once on startup and may be gotten as soon as PresCtrl is initialized. On language
    * changes this information is updated and will trigger a property update LOCATION_POI_SHORTCUT_INFORMATION_UPDATED
    *
    * Supported shortcut combinations
    *  1. CONTEXT_GUI And APPID Navigation
    *  2. CONTEXT_SDS And APPID Navigation
    *  3. CONTEXT_MAP And APPID Navigation
    * Other combinations are not supported.
    */
   const LocationCategoryInfos& getPoiShortcutCategoryInfos(const LocationContext& context = LocationContext(
            LocationContext::HMICONTEXT__GUI, APPLICATIONID__NAVIGATION)) const;

   /**
    * Reset user-configurable POI shortcuts to factory defaults.
    * \async ::LOCATION_POI_SHORTCUT_INFORMATION_UPDATED
    */
   void resetPoiShortcutsToFactoryDefaults();

   /**
    * Store the currently selected POI category (category last selected by selectPoiEntry or top level category) as
    * the specified POI category shortcut.
    * @param shortcutNum the position of the new category shortcut in the list of category shortcuts
    * @param context specifies the context for the new shortcut category (e.g. GUI, SDS, MAP).
    * \async ::LOCATION_POI_SHORTCUT_INFORMATION_UPDATED
    */
   void storeCurrentPoiCategoryAsShortcut(unsigned int shortCutNum, const LocationContext& context = LocationContext(
         LocationContext::HMICONTEXT__GUI, APPLICATIONID__NAVIGATION));

   /**
    * Delete a POI category shortcut
    * @param shortcutNum the position of the shortcut to be removed in the list of category shortcuts
    * @param context specifies the context of the category shortcut list (e.g. GUI, SDS, MAP).
    * \async ::LOCATION_POI_SHORTCUT_INFORMATION_UPDATED
    */
   void deletePoiCategoryShortcut(unsigned int shortCutNum, const ::navmiddleware::LocationContext& context);

   /**
    * Drag the POIShortcut at shortCutNumDrag and drop at shortCutNumDrop in the given context.
    * All the other POIShortcuts to be readjusted accordingly.
    * @param :shortCutNumDrag and shortCutNumDrop should be in between 0 and getPoiShortcutCategoryInfos().size()
    * @param :context defines for Destination Screen / Map Screen based on m_hmiContext.
    * \async ::LOCATION_POI_SHORTCUT_INFORMATION_UPDATED
    */
   void movePoiShortcut(unsigned int shortCutNumDrag, unsigned int shortCutNumDrop, const LocationContext& context);

   /**
    * Get category information for the exit POI categories as defined by enum ExitPOICategory.
    * The returned category infos are of same order as defined by that enum.
    */
   const ::std::map<ExitPOICategory, LocationCategoryInfos::LocationCategoryInfo>& getExitPoiCategoryInfos() const;

   /**
    * Request the icons for a list of POI categories
    * @param context defines the HMI context for which the icons shall be provided
    * @param variants lists the icon variants that are required
    * @param poiCategories lists the POI categories for which the icons should be provided
    * \async ::LOCATION_POI_ICON_INFORMATION_UPDATED
    */
   void requestPoiIconInfos(const LocationContext& context , const std::vector< ::navmiddleware::PoiIconInfos::ImageVariant>& variants,
                            const std::vector< ::navmiddleware::PoiCategoryId>& poiCategories);

   /**
    * Get the list of POI category icon data
    * @param context for which the icon data is to be used
    * Should be called after receiving property update LOCATION_POI_ICON_INFORMATION_UPDATED.
    */
   const ::navmiddleware::PoiIconInfos& getPoiIconInfos(const LocationContext& context) const;


   // Free text search (FTS) -----------------------------------------------------------------------------------------

   /**
    * Return the current state of the online free text search availability.
    * Updates of the state are announced with property update LOCATION_ONLINE_FREE_TEXT_SEARCH_AVAILABILITY_CHANGED.
    * This property is available without prior starting a free text search.
    */
   const OnlineFreeTextSearchAvailabilityInfo& getOnlineFreeTextSearchAvailability() const;

   /**
    * This call requests the server and informs about the online server availability.
    * @param server defines which server to be evaluated like TomTom(LOCATIONSEARCHSOURCE__ONLINE) or Google(LOCATIONSEARCHSOURCE__GOOGLE_PLACES).
    * \async ::LOCATION_ONLINE_FREE_TEXT_SEARCH_AVAILABILITY_CHANGED
    */
   void requestFreeTextSearchOnlineServerAvailability(LocationSearchSource server);

   /**
    * Start a new free text search.
    * This call does not actually trigger some asynchronuous call. That is triggered by calling
    * requestFreeTextSearchResults
    * @param searchType  defines the kind of location type to be searched: Addresses, POIs, or both
    * @param searchScope  defines the search scope to be used initially
    * @param searchSource  defines the search source to be used initially
    * @param context  specifies the LocationContext in which the newly started POI input shall be used
    * @param languageEntryInfo  explicitly specifies the Language in which FTS search should be started (Used in SDS context only)
    *                           if invalid, the Current language setting depending on the passed locationContext is used.
    */
   void startFreeTextSearchInput(
      FreeTextSearchType searchType = FREETEXTSEARCHTYPE__COMBINED,
      const LocationSearchScope& searchScope = LocationSearchScope(),
      LocationSearchSource searchSource = LOCATIONSEARCHSOURCE__OFFLINE,
      const LocationContext& context = LocationContext(LocationContext::HMICONTEXT__GUI, APPLICATIONID__NAVIGATION),
      const ValidValue<LanguageEntryInfo>& languageEntryInfo = ValidValue<LanguageEntryInfo>());

   /**
    *  Set the current search string for the free text search.
    *  This call does not actually trigger search. That is triggered by calling requestFreeTextSearchResults
    *  @param spellerString search string to be used.
    *  To optimize the search results the search string can also be sent as tagged content.
    *  Example Usage : "<?xml><BRAND>POIBrandname</BRAND>"
    *  Supported Tags are as below :
    *  <NAME></NAME>: POI name (POI only)
    *  <CATEGORY></CATEGORY>: POI category (POI only)
    *  <CATEGORYID></CATEGORYID>: POI category id (POI only)
    *  <BRAND></BRAND>: Brand of POI (POI only)
    *  <PHONE></PHONE>: Phone number of POI (POI only)
    */
   void setFreeTextSearchString(const ::std::string& searchString);

   /**
    * Get the current free text search string
    * @return current free text search string
    */
   const ::std::string& getFreeTextSearchString() const;

   /**
    *  Sets the current search scope for the free text search.
    *  This call does not actually trigger search. That is triggered by calling requestFreeTextSearchResults
    *  @param searchScope defines the new search scope
    */
   void setFreeTextSearchScope(const LocationSearchScope& searchScope);

   /**
    * Set the search source of free text search
    * This call does not actually trigger search. That is triggered by calling requestFreeTextSearchResults
    * @param searchSource  defines the search source to be used
    */
   void setFreeTextSearchSource(LocationSearchSource searchSource);

   /**
    * Set the sort order for the current list of Fts entries
    * Current and supported sort orders can be queried in LocationInfos object returned by getFreeTextSearchResultInfos()
    * This call does not actually trigger sort. That is triggered by calling requestFreeTextSearchResults
    * @param sortOrder to be set
    */
   void setFreeTextSearchSortOrder(LocationSortOrder sortOrder);

   /**
    * Set the auto-completion mode of the free text search (currently relevant for Google Places search only)
    * This call does not actually trigger a new search. That is triggered by calling requestFreeTextSearchResults
    * @param autoCompletionMode  defines the auto-completion mode to be used (the default is AUTO_COMPLETION_MODE__RESULTS_ONLY)
    */
   void setFreeTextSearchAutoCompletionMode(AutoCompletionMode autoCompletionMode);

   /**
    *  Requests the specified range of free text search results for the search string given by calling
    *  setFreeTextSearchString
    *  @param startIndex of the first element to be returned by getFreeTextSearchResultInfos()
    *  @param numElements to be returned by getFreeTextSearchResultInfos()
    *  @param filterType defines the kind of location type to be used while filtering : Address, POI, Combined(No filtering).
    *  \async ::LOCATION_FREE_TEXT_SEARCH_RESULTS_CHANGED if requested results are available
    *  \async ::LOCATION_FREE_TEXT_SEARCH_SPELLER_INFO_CHANGED if suggestion characters are available
    */
   void requestFreeTextSearchResults(unsigned int startIndex, unsigned int numElements, FreeTextSearchType filterType);

   /**
    * Returns the list of fts results requested by requestFreeTextSearchResults
    */
   const LocationInfos& getFreeTextSearchResultInfos() const;

   /**
    * Returns the suggestion characters requested by requestFreeTextSearchResults
    */
   const ElementSpellerInfo& getFreeTextSearchSpellerInfo() const;

   /**
    * Request the location attributes for the specified free text search result
    * @param listIndex absolute index (0..LocationInfos::getCountOfAllLocations()-1) to request the attributes for
    * \async ::LOCATION_FREE_TEXT_SEARCH_RESULT_ATTRIBUTES_AVAILABLE
    * \async ::LOCATION_FREE_TEXT_SEARCH_RESULT_ATTRIBUTES_ERROR
    */
   void requestFreeTextSearchResultAttributes(unsigned int listIndex);

   /**
    * Return the location attributes of last requested free text search result.
    * Should be called after receiving property update LOCATION_FREE_TEXT_SEARCH_RESULT_ATTRIBUTES_AVAILABLE.
    */
   const LocationAttributeInfos& getFreeTextSearchResultAttributes() const;

   /**
    * Returns the error Information when requested for FreeTextSearchResultAttributes.
    * Should be called after receiving property update LOCATION_FREE_TEXT_SEARCH_RESULT_ATTRIBUTES_ERROR
    */
   const LocationAttributeErrorInfo& getFreeTextSearchResultAttributesError() const;

   /**
    * Chooses one element of the fts result to become the current working location
    * @DEPRECATED: Please use setLocationWithFreeTextSearchResult(unsigned int) instead
    */
   void setLocationWithFreeTextSearchResult(const LocationInfos::LocationInfo& locationInfo);

   /**
    * Chooses one element of the fts result to become the current working location
    * @param listIndex absolute index (0..LocationInfos::getCountOfAllLocations()-1) to be selected
    */
   void setLocationWithFreeTextSearchResult(unsigned int listIndex);

   /**
    * Japanese variant only. Valid only in case freetext search input was started with
    * search type FREETEXTSEARCHTYPE__POICATEGORY
    * Select one element of the fts result
    * @param listIndex absolute index (0...LocationInfos::getCountOfAllLocations() - 1) to be selected
    */
   void selectFreeTextSearchElement(unsigned int listIndex);

   /**
    * Undo the selection of a free text search result element.
    * \async ::LOCATION_FREE_TEXT_SEARCH_RESULTS_CHANGED
    */
   void undoFreeTextSearchElementSelection();

   /**
    * Used to notify that free-text-search input has been left i.e. HMI switched to a different screen
    * Internally resources are cleaned up, deregistration from lower layers is done, etc.
    */
   void leaveFreeTextSearchInput();

   /**
    * Clears all entries from the free text search history
    */
   void clearFreeTextSearchHistory();


   /**
    * Refresh an expired Google FTS result entry Request
    * This method needs to be called before selecting an expired entry in the FTS result list,
    * if the entry status is ENTRY_REFRESH_STATUS__EXPIRED.
    * @param listIndex absolute index (0..LocationInfos::getCountOfAllLocations()-1) to request for refresh
    * \async ::LOCATION_ENTRY_SELECTION_STATUS_CHANGED
    */
   void requestRefreshForExpiredFTSResult(unsigned int listIndex);

   /**
    * Request to transform an expired Google FTS result entry into On-board Matching entry
    * This method needs to be called before selecting an expired entry in the FTS result list,
    * if the entry status is ENTRY_REFRESH_STATUS__ONBOARD_MATHCING_NEEDED
    * @param listIndex absolute index (0..LocationInfos::getCountOfAllLocations()-1) to request for refresh
    * \async ::LOCATION_ENTRY_SELECTION_STATUS_CHANGED
    */
   void requestOnboardMatchingForExpiredFTSResult(unsigned int listIndex);

   /**
    * Returns the selection status for the entry.
    * If Status is set to UNKNOWN, the entry selection status is Unknown.
    * If Status is set to VALID, the entry can be selected.
    * If Status is EXPIRED, the entry needs to be refreshed before selection.
    * If Status is set to ONBOARD_MATHCING_NEEDED, the entry failed to refresh and needs to be matched to an on-board entry.
    * If Status is set to ONBOARD_MATCHING_FAILED, the entry could not be matched to an on-board entry and is deleted.
    * Should be called after receiving the property update LOCATION_ENTRY_SELECTION_STATUS_CHANGED.
    */
   EntrySelectionStatus getEntrySelectionStatus() const;

   /**
    * Returns the status and the count of the background Refresh process.
    * If status is inprogress, the background refresh process is going on
    * If status is interrupted, the background refresh is interrupted due to location input activity/connectivity issues.
    * If staus is completed, the background refresh is done.
    * \async :: LOCATION_REFRESH_STATUS_CHANGED
    */
   const RefreshInfo& getRefreshStatus() const;

   /**
    * Sets the current FTS AutoCompletion Categories to fetch the history entries.
    * This call does not actually trigger search. That is triggered by calling requestFreeTextSearchResults.
    * @param autoCompletionConfiguration defines the new configuration to be used.
    */
   void setFreeTextSearchAutoCompleteCategories(const AutoCompletionConfiguration& autoCompletionConfiguration);

   /**
    * Resumes an existing FTS input instance. The instance being resumed must have been created before by one of
    * the start Free text search input functions declared above. Unlike the start functions this resume function
    * resumes the input in the state (current input scope, search scope, search source, ...) as it was left before
    * (means, it does not reset the state of the resumed input).
    *
    * The specified instance becomes the active FTS input (there can be only one active FTS input at a time). A
    * possible other active FTS input instance will be implicitly paused by this call.
    *
    * If the referenced instance is not existing, the possible currently active instance stays active.
    * If the referenced instance is already the active one, this call is silently ignored.
    *
    * @param context  specifies the LocationContext for the Free Text Search instance to be resumed
    * \async :: LOCATION_FREE_TEXT_SEARCH_RESUME_STATUS_CHANGED
    */
   void resumeFreeTextSearchInstance(const LocationContext& context);


   /**
     * Returns the ResumeFreeTextSearchStatusInfo representing current state of FTS input.
     * Should be called after receiving property update LOCATION_FREE_TEXT_SEARCH_RESUME_STATUS_CHANGED.
     */
   const ResumeFreeTextSearchStatusInfo& getResumeFreeTextSearchStatusInfo() const;

   ///@}

   // ================================================================================================================
   ///@{ \name Speech Dialog System

   // SDS Functions: Addressinput -----------------------------------------------------------

   /** start checking the SDS-Address with the current Navigation Database
     *  @param sdsAddress - Address object that has to be checked
     *  @param languageEntryInfo  specifies the Language in which sdsAddress is given,By default Current SystemLanguage is considered
     *  \async ::LOCATION_SDS_ADDRESS_WITH_OPTIONS_CHANGED if requested results are available
     */
   void sdsCheckAddress(const SDSAddress& sdsAddress,
                        const ValidValue<LanguageEntryInfo>& languageEntryInfo = ValidValue<LanguageEntryInfo>());

   /**  returns the resulting address from sdsCheckAddress
     */
   const SDSAddressWithOptions& getCurrentSDSAddressWithOptions() const;

   /**  @DEPRECATED :Request for Ambiguity List, Instead use requestSdsRefinementList with Explicit Parameter
     */
   void requestSdsRefinementList();

   /**  Request RefinementList based on the Provided Address Category
     *  @param sdsAddressElementype- Element type for which list is requested
     *  \async ::LOCATION_SDS_ELEMENTS_CHANGED if result is available
     */
   void requestSdsRefinementList(SDSAddressElementType sdsAddressElementType);

   /**  returns the resulting refinementList on property update LOCATION_SDS_ELEMENTS_CHANGED
     *  from previous call of requestSdsRefinementList
     */
   const RefinementWithIdInfos& getSdsRefinementListInfo() const;

   /**  select the refinement Entry will be added to address and sdsCheckAddress will be called automatically
     *  \async ::LOCATION_SDS_CHECK_RESULTS_CHANGED
     */
   void selectSdsRefinement(unsigned int id);

   /**  request the house number patterns for the current sds address
     *  \async ::LOCATION_SDS_PATTERNS_CHANGED
     */
   void requestSdsHouseNumberPatterns();

   const SdsHouseNumberPatternsInfos& getSdsHouseNumberPatternsInfo() const;

   /**  Request will set the SDS Address as destination or waypoint
     *  @param  isWaypoint- if true: Address will be used as waypoint instead of destination
     */
   void setLocationWithSdsInput();

   /**
    *  Request will provide the meta data from NDS datacarrier
    *  \async ::LOCATION_SDS_GLOBAL_DATA_CHANGED
    */
   void requestGlobalSdsData();

   /** Request provides NamedObjectIds for Countries and SubCountries available in the datacrrier
    *  \async ::LOCATION_SDS_ID_LIST_COUNTRY_AND_SUBCOUNTRY_CHANGED
    * */
   void requestIdListOfCountryAndSubCountry();

   /**
    * Request provides Names for given NamedObjectIDs/POI NameIDs/POI Category NameIds/POI Attribute NameIDs
    * @param sdsGlobalNamedObjectIdInfos - List of SdsGlobalNamedObjectIdInfo (NDS internal data - Update Region ID,
    * Product ID, Supplier ID, NameID)
    * @param languageEntryInfos - Language info to filter the result
    * @param ndsFeatureType - To identify the request type if its Related to Address or POI
    * @param stopOnFirstMatch - Condition to search for all relevant matches to provided language
    * @return Provides unique identifier for the request which can be used to match the corresponding response
    * \async ::LOCATION_SDS_NAMES_FOR_NDS_IDS_CHANGED
    */
   RequestId requestNamesForNdsIds(const ::std::vector<SdsGlobalNamedObjectIdInfo>& sdsGlobalNamedObjectIdInfos,
                                   const ::std::vector<LanguageEntryInfo>& languageEntryInfos, NdsFeatureType ndsFeatureType, bool stopOnFirstMatch);

   /**
    * Request provides Phonemes for given NamedObjectIDs/POI NameIDs/POI Category NameIds/POI Attribute NameIDs
    * @param sdsGlobalNamedObjectIdInfos - List of SdsGlobalNamedObjectIdInfo (NDS internal data - Update Region ID,
    * Product ID, Supplier ID, NameID)
    * @param languageEntryInfos - Language info to filter the result
    * @param ndsFeatureType - To identify the request type if its Related to Address or POI
    * @param stopOnFirstMatch - Condition to search for all relevant matches to provided language
    * @return Provides unique identifier for the request which can be used to match the corresponding response
    * \async ::LOCATION_SDS_PHONEMES_FOR_NDS_IDS_CHANGED
    */
   RequestId requestPhonemesForNdsIds(const ::std::vector<SdsGlobalNamedObjectIdInfo>& sdsGlobalNamedObjectIdInfos,
                                      const ::std::vector<LanguageEntryInfo>& languageEntryInfos, NdsFeatureType ndsFeatureType, bool stopOnFirstMatch);

   /**
    * Request provides if given NamedObjectIDs of Cities have Streets or not
    * @param sdsGlobalNamedObjectIdInfos - List of sdsGlobalNamedObjectIdInfo (NDS internal data - UpdateRegionID,
    * ProductID, SupplierID, NameID)
    * \async ::LOCATION_SDS_CITY_HAS_STREETS_CHANGED
    */
   void requestCityHasStreets(const ::std::vector<SdsGlobalNamedObjectIdInfo>& sdsGlobalNamedObjectIdInfos);

   /**
    * Provides meta data required by SDS
    * Should be called on property update LOCATION_SDS_GLOBAL_DATA_CHANGED
    */
   const SdsGlobalDataInfo& getSdsGlobalDataInfo() const;

   /**
    * Provides NameIDs of available Countries and SubCountries in Datacarrier
    * Should be called on property update LOCATION_SDS_ID_LIST_COUNTRY_AND_SUBCOUNTRY_CHANGED
    */
   const SdsCountryAndSubCountryInfos& getSdsCountryAndSubCountryInfos() const;

   /**
    * Provides Names as strings for the given NamedObjectId
    * Should be called on property update LOCATION_SDS_NAMES_FOR_NDS_IDS_CHANGED
    */
   const SdsNdsIdStringInfos& getSdsNdsIdNamedStringInfos() const;

   /**
    * Provides Phonemes as strings for the given NamedObjectId
    * Should be called on property update LOCATION_SDS_PHONEMES_FOR_NDS_IDS_CHANGED
    */
   const SdsNdsIdStringInfos& getSdsNdsIdPhonemeStringInfos() const;

   /**
    * Provides If streets are available for given NamedObjectId of City
    * Should be called on property update LOCATION_SDS_CITY_HAS_STREETS_CHANGED
    */
   const ::std::vector<SdsCityNamedObjectHasStreet>& getSdsCityNamedObjectHasStreetInfos() const;


   ///@}

   // ================================================================================================================
   ///@{ \name GUIDANCE

   /** Starts the route guidance to an already calculated route
   * \async ::GUIDANCE_STATE_CHANGED
   */
   void startGuidance();

   /**
    * Calculates the route and starts the guidance to the destination/tour
    * previously set using a set location function
    * @param isResume [JP] Used to distinguish resume guidance scenario
    * (specifically e.g. after power down) for various cases. Used currently
    * only in Japanese navigation
    *
    * \async ::GUIDANCE_STATE_CHANGED
    */
   void startRouteCalculationAndGuidance(bool isResume = false);

   /**
    * Starts the guidance to the last previously guided destination/route.
    * Last destination or route is loaded from Destination Memory and then route calculation will be started.
    *
    * NOTE: In the case when a route is last guided, already reached waypoints will be skipped.
    * \async ::GUIDANCE_STATE_CHANGED
    */
   void restartGuidanceToLastDestination();

   /**
    * Return DestinationInfos containing last guided destinations details.
    * This call shall be made by HMI to get details of last guided destination for last mile navigation
    * DestinationInfos contains a non-empty list if the criteria for last mile navigation are met.
    * -------------------------------------------------------------------------------------------------------------------
    *  isGuidanceActive|   isFinalDestinationreached  |  air distance from curr pos to dest |  is DestinationInfos valid
    * -------------------------------------------------------------------------------------------------------------------
    *        1         |             x                |                 <= 3km              |        1
    *        1         |             x                |                 >  3km              |        0
    *        0         |             0                |                 x                   |        0
    *        0         |             1                |                 >  3km              |        0
    *        0         |             1                |                 <= 3km              |        1
    */
   const DestinationInfos& getDestinationInfosForLastMileNavigation() const;

   /**
    * Return DestinationMemoryItem structure containing last guided destination details.
    * This call shall be made by HMI to get details of last guided destination
    * after calling restartGuidanceToLastDestination()
    * \async ::GUIDANCE_LAST_GUIDED_ITEM_DETAILS_AVAILABLE
    */
   const DestinationMemoryItem& getLastGuidedItem();

   /**
    * Stops the route guidance if one is active
    * \async ::GUIDANCE_STATE_CHANGED
    */
   void stopGuidance();
   /**
    * Returns true if a route guidance is currently active.
    * Update after property callback ::GUIDANCE_STATE_CHANGED
    */
   bool isGuidanceActive() const;

   /**
    * Returns true if Final Destination Reached.
    * Update after property callback ::GUIDANCE_STATE_CHANGED
    */
   bool isFinalDestinationReached() const;

   /**
    * Sets speed limit data from external device like speed camera.
    * Results in property update GUIDANCE_SPEEDLIMIT_CHANGED if there is a change in speed limit.
    * Should be called when speed limit data is received from VD_VehicleData
    */
   void setSpeedLimitData(const SpeedLimitData& speedLimitData);

   /**
    * Returns the ArrivalInfo to the destination for a planned or guided route.
    * Update after property callback ::ROUTE_ARRIVALINFOS_CHANGED
    * Opening hours popup indication:
    * When a POI maybe isn't reachable during the opening hours a popup can be displayed.
    * This is triggered by ROUTE_OPENING_HOURS_REACHABILITY_INDICATION
    * After this property is triggered ArrivalInfos::ArrivalInfo::m_poiOpeningHoursStatus
    * gives more information about the reachability of the POI
    * Remark 1: This property isn't triggered with any change of the opening hours status.
    * Actually it should be triggered only once for a particular waypoint to avoid frequently
    * displayed popups.
    * Remark 2: The property ROUTE_OPENING_HOURS_REACHABILITY_INDICATION is always related to
    * the next waypoint.
    * Remark 3: When the next waypoint is an ePoi, the property update for ROUTE_APPROACHING_CHARGING_STATION
    * could be triggered. Based on this trigger, a pop-up can be displayed.
    */
   const ArrivalInfos& getArrivalInfos() const;

   /**
   * Returns the DestinationInfo to the destination for a planned or guided route.
   * Update after property callback ::ROUTE_DESTINATIONS_CHANGED
   */
   const DestinationInfos& getDestinationInfos() const;

   /**
    * Returns the ArrivalInfo to the destination, if a route guidance is currently running.
    * This property shall be used by external systems to obtain arrival infos for the guided route,
    * even if a planned route is displayed in HMI.
    * Update after property callback ::GUIDANCE_ARRIVALINFOS_CHANGED
    */
   const ArrivalInfos& getArrivalInfosGuidedRoute() const;

   /**
    * Returns the DestinationInfo to the destination, if a route guidance is currently running.
    * This property shall be used by external systems to obtain arrival infos for the guided route,
    * even if a planned route is displayed in HMI.
    * Update after property callback ::GUIDANCE_DESTINATIONS_CHANGED
    */
   const DestinationInfos& getDestinationInfosGuidedRoute() const;

   /**
    * Returns arrival info to next and final destination for all route alternatives (FAST, ECONOMIC, SHORT),
    * if a route guidance is currently running. Update after property callback ::ROUTE_ARRIVALINFOS_ALTERNATIVES_CHANGED
    * Opening hours popup indication:
    * When a POI maybe isn't reachable during the opening hours a popup can be displayed.
    * This is triggered by ROUTE_OPENING_HOURS_REACHABILITY_INDICATION
    * After this property is triggered
    * ArrivalInfosAlternatives::ArrivalInfosAlternative::m_poiOpeningHoursReachabilityStatusNextDest
    * ArrivalInfosAlternatives::ArrivalInfosAlternative::m_poiOpeningHoursReachabilityStatusFinalDest
    * gives more information about the reachability of the POI
    * Remark1 : This property isn't triggered with any change of the opening hours status.
    * Actually it should be triggered only once for a particular waypoint to avoid frequently
    * displayed popups.
    * Remark 2: The property ROUTE_OPENING_HOURS_REACHABILITY_INDICATION is always related to
    * the next waypoint of the currently highlighted respectively active route.
    */
   const ArrivalInfosAlternatives& getArrivalInfosAlternatives() const;

   /**
    * Returns real time user-confirmed-traffic info for the new traffic induced route
    * if a route guidance is currently running. Update after property callback ::ROUTE_TRAFFICIMPACT_ALTERNATIVE_CHANGED
    */
   const TrafficImpactRouteAlternativeInfos& getTrafficImpactRouteAlternativeInfos() const;

   /**
    * Returns route calculation duration info for completion of active route and alternative routes.
    * Update after property callback ::ROUTE_STATUS_CHANGED
    */
   const RouteCalculationDurationInfos& getRouteCalcDurationInfos() const;

   /**
    * Returns the NextManeuverInfo, if a guidance is active.
    * Update after property callback ::GUIDANCE_MANEUVERINFOS_CHANGED
    */
   const NextManeuverInfos& getNextManeuverInfos() const;

   /**
    * Returns the ManeuverViewInfo, if a guidance is active.
    * Update after property callback ::GUIDANCE_MANEUVERVIEW_CHANGED
    */
   const ManeuverViewInfo& getManeuverViewInfo() const;

   /**
    * Returns the SignPostInfo, if a guidance is active.
    * Update after property callback ::GUIDANCE_SIGNPOST_CHANGED
    */
   const SignPostInfo& getSignPostInfo() const;

   /**
     * Returns the Lane information list, if a guidance is active and lane information is available.
     * Update after property callback ::GUIDANCE_MANEUVERINFOS_CHANGED
     */
   const LaneInfos& getLaneInfos() const;

   /**
     * Returns the speed information, like speed limit, if a guidance is active and speed information is available.
     * Update after property callback ::GUIDANCE_SPEEDLIMIT_CHANGED
     */
   const SpeedInfo& getSpeedInfo() const;

   /**
     * Returns the speed camera information, like speed cam distance, speed cam count, speed limit when approaching a speedcam.
     * Update after property callback ::GUIDANCE_SPEEDCAM_CHANGED
     */
   const SpeedCamInfo& getSpeedCamInfo() const;

   /**
     * Returns the trafficSign information
     * Update after property callback ::GUIDANCE_TRAFFICSIGN_CHANGED
     */
   const TrafficSignInfo& getTrafficSignInfo() const;

   /**
     * Returns the traffic enforcement zone information, like zone type, distance to zone begin and end, alert state.
     * Update after property callback ::GUIDANCE_TRAFFIC_ENFORCEMENT_INFO_CHANGED
     */
   const TrafficEnforcementInfo& getTrafficEnforcementInfo() const;

   /**
     * Returns the destination reached information, like reached destination type is waypoint or final destination.
     * Update after property callback ::GUIDANCE_DESTINATION_REACHED
     */
   const DestinationReachedInfo&  getDestinationReachedInfo() const;

   /**
    * Returns EVAlertInfo with data related to GuardianAngelAlert
    * Update after property callback : GUIDANCE_GUARDIAN_ANGEL_ALERT_CHANGED
    */
   const EVAlertInfo& getGuardianAngelAlertInfo() const;

   /**
    * Returns EVAlertInfo with data related to UnReachableDestinationAlert
    * Update after property callback : GUIDANCE_UNREACHABLE_DESTINATION_ALERT_CHANGED
    */
   const EVAlertInfo& getUnreachableDestinationAlertInfo() const;

   /**
    * Returns EVAlertInfo with data related to Low Battery Alert
    * Update after property callback : GUIDANCE_LOW_BATTERY_ALERT_CHANGED
    */
   const EVAlertInfo& getLowBatteryAlertInfo() const;

   /**
    * Returns NearestChargingStationInfo with data related to nearest charging station
    * Update after property callback : GUIDANCE_NEAREST_CHARGING_STATION_INFO_CHANGED
    */
   const NearestChargingStationInfo& getNearestChargingStationInfo() const;

   /**
    * Returns PredictedDestinationInfo with destination memory entry  id and LocationAttributeInfos.
    * To set current location:
    * Use navmiddleware::setLocationWithDestinationMemoryId() if destination memory entry id is valid
    * Use navmiddleware::setLocationWithCoordinates() otherwise.
    * Update after property callback :
    * GUIDANCE_DESTINATION_DETECTED_BASED_ON_LEARNED_ROUTES, GUIDANCE_TRAFFIC_EVENTS_DETECTED_ON_LEARNED_ROUTES
    */
   const PredictedDestinationInfo& getPredictedDestinationInfo() const;

   /**
    * Decline the use of a predicted destination
    */
   void declinePredictedDestination(const PredictedDestinationInfo& predictedDestinationInfo);

   /**
    * Triggers the LastNavigation Announcement
    */
   void retriggerAcousticOutput();

   /**
    * Triggers the requested sample phrase to be spoken
    */
   void speakSamplePhrase(SamplePhrase samplePhraseType) const;

   /**
    * Triggers the request to speak poi alert type
    */
   void speakPoiAlert(PoiAlertType poiAlertType) const;

   /**
    *  Request road restrictions.
    *  @param linkId of the current car position.
    *  \async ::GUIDANCE_VEHICLE_RESTRICTIONS_AVAILABLE as soon as the requested elements are available.
    */
   void requestRoadRestrictions(const ValidValue< LinkId >& linkId) const;

   /**
    * Returns VehicleProhibitionInfo with restrictions on current road segment like
    * length, width, height weight, cargo type and vehicle type restrictions.
    * Update after property callback :
    * VEHICLE_RESTRICTIONS_AVAILABLE
    */
   const VehicleRestrictionsInfo& getVehicleRestrictionsInfo() const;

   /**
    * Returns VehicleRestrictionsStatus.
    * Updated after property callback :
    * GUIDANCE_VEHICLE_RESTRICTIONS_CHANGED
    */
   VehicleRestrictionsStatus getVehicleRestrictionsStatus() const;

   /**
    *  Request country and state details.
    *  @param isoCountryCode and isoSubCountryCode indicate iso country and subcountry codes.
    *  \async ::GUIDANCE_COUNTRY_DETAILS_AVAILABLE as soon as the requested elements are available.
    */
   void requestCountryAndStateDetails(const ::std::string& isoCountryCode, const ::std::string& isoSubCountryCode) const;

   /**
    *Triggers confirmPoiAlert request to indicate confirmation of alert by the user on poi alert pop-up.
    *@param alertIdentifier provided by EVAlertInfo.
    */
   void confirmPoiAlert(unsigned int alertIdentifier);

   /**
    * Returns country and state specific information like
    * country and subcountry names, country and sub country codes,
    * country and subcountry flag icon data, traffic sense,
    * supported iso language code list and country speed limit info which includes
    * speed limit value, road type applicable for the speed limit,
    * list of time, fuzzy time domain values, weather conditions for the speed limit
    * Update after property callback :
    * GUIDANCE_COUNTRY_DETAILS_AVAILABLE
    */
   const CountryAndStateDetailsInfo& getCountryAndStateDetailsInfo() const;

   const HorizonEventInfo& getHorizonEventInfo() const;

   /**
    * Getter to check if Road has Automated Driving Level 2 Information
    * Shall be used to get current status on every callback of GUIDANCE_AD2_AVAILABILITY_CHANGED
    */
   bool isLocatorOnRoadWithAD2() const;

   /**
    * Get the guidance settings object which is hosted by middleware
    */
   settings::GuidanceSettings getGuidanceSettings() const;

   /**
    * Send updated Hmi voice guidance settings back to the middleware.
    * After calling this method, changes on the local object will not affect
    * the passed reference (middleware creates a copy). To update middleware settings,
    * call this method again.
    */
   void updateGuidanceSettings(const settings::GuidanceSettings&);

   /**
   * Get the safety guidance settings object which is hosted by middleware
   */
   settings::SafetyGuidanceSettings getSafetyGuidanceSettings() const;

   /**
    * Send updated safety guidance settings back to the middleware.
    * After calling this method, changes on the local object will not affect
    * the passed reference (middleware creates a copy). To update middleware settings,
    * call this method again.
    */
   void updateSafetyGuidanceSettings(const settings::SafetyGuidanceSettings&);

   /**
    * Get the safety camera settings object which is hosted by middleware
    */
   settings::SafetyCameraSettings getSafetyCameraSettings() const;

   /**
    * Send updated safety camera settings back to the middleware.
    * After calling this method, changes on the local object will not affect
    * the passed reference (middleware creates a copy). To update middleware settings,
    * call this method again.
    */
   void updateSafetyCameraSettings(const settings::SafetyCameraSettings&);

   /**
    * Get the electric vehicle battery status warning settings object which is hosted by middleware
    */
   settings::EvBatteryStatusWarningSettings getEvBatteryStatusWarningSettings() const;

   /**
    * Send updated electric vehicle battery status warning settings back to the middleware.
    * After calling this method, changes on the local object will not affect
    * the passed reference (middleware creates a copy). To update middleware settings,
    * call this method again.
    */
   void updateEvBatteryStatusWarningSettings(const settings::EvBatteryStatusWarningSettings&);

   /**
    * Get the electric vehicle miscellaneous settings object which is hosted by middleware
    */
   settings::EvMiscellaneousSettings getEvMiscellaneousSettings() const;

   /**
    * Send updated electric vehicle miscellaneous settings back to the middleware.
    * After calling this method, changes on the local object will not affect
    * the passed reference (middleware creates a copy). To update middleware settings,
    * call this method again.
    */
   void updateEvMiscellaneousSettings(const settings::EvMiscellaneousSettings&);

   /**
    * Interface to Activate/Deactive EcoNavigation Functionality
    * GUIDANCE_ECONAV_STATUS_CHANGED update after change in EcoNavStatus
    */
   void setEcoNavSwitchState(EcoNavSwitchState ecoNavSwitchState);

   /*
    * Status of Whether EcoNavigation functionality is Activated/Deactivated
    * Valid EcoNavSwitchState with update on GUIDANCE_ECONAV_SWITCH_STATUS_CHANGED
    */
   EcoNavSwitchState getEcoNavSwitchState() const;

   /*
    * Status of the EcoNavStatus provided by the EcoNavSystem
    * Valid EcoNavSwitchState with update on GUIDANCE_ECONAV_STATUS_CHANGED
    */
   EcoNavState getEcoNavStatus() const;

   /**
   * Interface to Activate/Deactive UrbanZevZoneOption
   * GUIDANCE_URBAN_ZEV_DRIVING_OPTION_STATUS_CHANGED update after change in UrbanZevDrivingOption
   */
   void setUrbanZevDrivingOption(UrbanZevDrivingOption urbanZevDrivingOption);

   /*
    * Status of Whether UrbanZevZoneOption is Activated/Deactivated
    * Valid UrbanZevDrivingOption with update on GUIDANCE_URBAN_ZEV_DRIVING_OPTION_STATUS_CHANGED
    */
   UrbanZevDrivingOption getUrbanZevDrivingOptionStatus() const;

   ///@}

   // ================================================================================================================
   ///@{ \name MAP

   /**
    * Returns if the auto zoom is currently active
    */
   bool isAutoZoomActive() const;

   /**
    * Sets camera to linked carsor mode
    * @param[in] mapViewId - The map view id (PRIMARY or SECONDARY) to which this request is sent.
    */
   void switchToLinkedMode(MapViewId mapViewId, CameraAnimation cameraAnimation = CameraAnimation__IMMEDIATE);

   /**
    * Getter to Check if Camera is linked to Carsor
    * @param[in] mapViewId - The map view id (PRIMARY or SECONDARY) to which this request is sent.
    * Update after callback ::MAP_CAMERAMODE_CHANGED
    */
   bool isInLinkedMode(MapViewId mapViewId) const;

   /**
   * @brief Bring the map into the Camera mode "FREE"
   * Changes the camera mode and optionally restores the camera setup (position, map scale, pitch, heading)
   * that was active when last exiting the free mode.
   *
   * @param[in] mapViewId           - view id primary/secondary in the map
   * @param[in] cameraConfiguration - flag indicating whether to use the current camera configuration or to
   *                                  restore the last free mode camera configuration
   */
   void switchToFreeMode(MapViewId mapViewId, CameraConfiguration cameraConfiguration = CAMERA_CONFIGURATION__CURRENT);

   /**
    * Zooms in stepwise based on the zoomType and step count.
    * Based on the MapSettings, HMI should call this interface with appropriate zoomType
    * If zoomType is ZOOMTYPE_ON_NOMINALSCALE_VALUES, then map is zoomed on fixed steps(from nominal scale property).
    * If zoomType is ZOOM_TYPE_ON_SUBSTEPS_NOMINALSCALE_VALUES, then map is zoomed on smaller points of
    * fixed steps (from nominal scale property).
    * Note: After the zooming is performed on fixed steps(from nominal scale property), a timer is activated internally and
    * after the timer is expired, Map will be auto zoomed to a fixed scale(from nominal scale property) if the auto zoom
    * setting is switched to ON.
    * @param[in] mapViewId         - view id{primary or secondary} of the map on which zooming in will be performed.
    * @param[in] stepCount         - stepCount determines the number of zoom in steps to be performed in nominal scales.
    *                                Applicable only on ZOOMTYPE_ON_NOMINALSCALE_VALUES.
    * @param[in] zoomType          - ZOOMTYPE_ON_NOMINALSCALE_VALUES means map is zoomed on fixed steps(from nominal scale property).
    *                                ZOOM_TYPE_ON_SUBSTEPS_NOMINALSCALE_VALUES means map is zoomed on smaller points of
    *                                fixed steps(from nominal scale property). i.e. 20% zoom in from the current map scale.
    * @param[in] cameraAnimation   - type of animation used for the zooming in operation.
    * \async ::MAP_CAMERAINFOS_CHANGED.
    */
   void zoomInStep(MapViewId mapViewId, uint32_t stepCount = 1, ZoomType zoomType = ZOOM_TYPE_ON_NOMINALSCALE_VALUES,
                   CameraAnimation cameraAnimation = CameraAnimation__ADAPTIVE);

   /**
    * Zooms out stepwise based on the zoomType and step count.
    * Based on the MapSettings, HMI should call this interface with appropriate zoomType
    * If zoomType is ZOOMTYPE_ON_NOMINALSCALE_VALUES, then map is zoomed on fixed steps(from nominal scale property).
    * If zoomType is ZOOM_TYPE_ON_SUBSTEPS_NOMINALSCALE_VALUES, then map is zoomed on smaller points of
    * fixed steps (from nominal scale property).
    * Note: After the zooming is performed on fixed steps(from nominal scale property), a timer is activated internally and
    * after the timer is expired, Map will be auto zoomed to a fixed scale(from nominal scale property) if the auto zoom
    * setting is switched to ON.
    * @param[in] mapViewId         - view id{primary or secondary} of the map on which zooming in will be performed.
    * @param[in] stepCount         - stepCount determines the number of zoom in steps to be performed in nominal scales.
    *                                Applicable only on ZOOMTYPE_ON_NOMINALSCALE_VALUES.
    * @param[in] zoomType          - ZOOMTYPE_ON_NOMINALSCALE_VALUES means map is zoomed on fixed steps(from nominal scale property).
    *                                ZOOM_TYPE_ON_SUBSTEPS_NOMINALSCALE_VALUES means map is zoomed on smaller points of
    *                                fixed steps(from nominal scale property). i.e. 20% zoom out from the current map scale.
    * @param[in] cameraAnimation   - type of animation used for the zooming in operation.
    * \async ::MAP_CAMERAINFOS_CHANGED.
    */
   void zoomOutStep(MapViewId mapViewId, uint32_t stepCount = 1, ZoomType zoomType = ZOOM_TYPE_ON_NOMINALSCALE_VALUES,
                    CameraAnimation cameraAnimation = CameraAnimation__ADAPTIVE);

   /**
    * Returns the current scale level as string, the current camera heading to be used for the compass map to be shown on map overlay.
    * Update after property callback ::MAP_CAMERAINFOS_CHANGED
    */
   const MapCameraInfos& getMapCameraInfos() const;

   /*
    * get the data Loading Status and data rendering status
    * async :VIEW_STATUS_UPDATE
    */
   const MapViewStatusInfos& getViewStatusInfos() const;

   /**
    * Sets the map presentation mode. See enum mapPresentation for supported modes
    * @param[in] mapViewId           - view id of the map {primary or secondary}
    * @param[in] mapRepresentation   - type of map representation that is to be set.
    * \async ::MAP_PRESENTATION_MODE_CHANGED
    */
   void setMapRepresentation(MapViewId mapViewId,
                             const MapRepresentation& mapRepresentation,
                             CameraAnimation cameraAnimation = CameraAnimation__ADAPTIVE);

   /**
    * Returns the MapRepresentationInfo
    * Update after callback ::MAP_PRESENTATION_MODE_CHANGED
    */
   const MapRepresentationInfo& getMapRepresentationInfo() const;

   /* @DEPRECATED*
    * Sets map to freeze mode
    * The client can also specify the visibility of the view.
    * @param[in] mapViewId           - view id of the map {primary or secondary}
    * @param[in] mapViewVisibility   - the visibility of the map view {can be visible or hidden}
    * \async ::MAP_VIEWSTATUS_CHANGED
    */
   void freezeMap(MapViewId mapViewId, MapViewVisibility mapViewVisibility = MAP_VIEW_VISIBILITY__VISIBLE);

   /* Sets map to freeze mode
    * The client can also specify the visibility of the view.
    * @param[in] mapViewId           - view id of the map {primary or secondary}
    * @param[in] renderingStatus     - Should be Freeze/Freeze_Keep_Data to inform whether to clear the internal data or not,
    *                                  in case map freezes for a short duration.
    * @param[in] duration            - Specifies the freeze duration in milliseconds.
    *                                  Rendering will be active after the time out.
    *                                  The Value 0 means no timeout (Map is in freeze state till unfreeze() is called)
    * @param[in] mapViewVisibility   - the visibility of the map view {can be visible or hidden}
    * \async ::MAP_VIEWSTATUS_CHANGED
    */
   void freezeMap(MapViewId mapViewId,
                  ::navmiddleware::MapViewStatusInfos::DataRenderingStatus renderingStatus,
                  uint16_t duration,
                  ::navmiddleware::MapViewVisibility mapViewVisibility = MAP_VIEW_VISIBILITY__VISIBLE);

   /**
    * Sets map to active mode
    * @param[in] mapViewId           - view id of the map {primary or secondary}
    * \async ::MAP_VIEWSTATUS_CHANGED
    */
   void unFreezeMap(MapViewId mapViewId);
   /**
    * Returns if the map is frozen or activated.
    * @param[in] mapViewId           - view id of the map {primary or secondary}
    * Update after callback ::MAP_VIEWSTATUS_CHANGED
    */
   bool isMapFrozen(MapViewId mapViewId) const;

   /**
    * Method to set the frame rate of the rendering in the view
    * @param[in] mapViewId           - view id of the map {primary or secondary}
    * @param[in] frameRate           - valid range 0 to 100.
    *                                  If the value is 0, frame rate will be switched to default value of 30.
    * @param[in] duration              Duration in milliseconds for which frame rate needs to be set.
    *                                  After timeout frame rate will be set to default.
    *                                  If the duration value is 0, map frame rate will be set till new value is set.
    */
   void setMapFrameRate(MapViewId mapViewId, uint8_t frameRate, uint16_t duration);

   /**
    * Returns the map scale range for the current map screen.
    * If the current Map scale is outside the range of Map scale supplied by
    * this interface, then HMI can explicitly set the map scale to the nearest map
    * scale range value.
    * Update after callback ::MAP_RECOMMENDED_SCALE_RANGE_CHANGED
    */
   const MapScaleRange& getMapScaleRange() const;

   /**
    * Starts a map picking based in the given pixel position on the screen
    * IMPORTANT NOTE: The HMI ensures that next PickInMap call can be made only after completion of the one in progress.
    * @param[in] mapViewId             - view id {primary or secondary}
    * @param[in] posX                  - x Position on screen
    * @param[in] posY                  - y Position on screen
    * @param[in] pickingResultFilter   - If valid, the picked elements will be further filtered as per the specified filter.
    * @return ValidValue<RequestId>    - Unique Id to identify and map the request to response
    * \async ::MAP_PICKING_RESULTS_AVAILABLE
    */
   ValidValue<RequestId> pickInMap(MapViewId mapViewId, unsigned int posX, unsigned int posY, const ValidValue<PickingResultFilter>& pickingResultFilter);

   /**
    * Returns the string for the requested picking result element.
    * Access after callback ::MAP_PICKING_RESULTS_AVAILABLE
    */
   const PickingResults& getPickingResults() const;

   /**
    * @brief starts a map picking from the visible picking results or POI search results at the given pixel position.
    * This method can be used for the following use-cases:
    * - To pick indexed POIs from POI search results or
    * - To pick results from picking a multi POI, in this case HMI needs to call pickInMap() before!
    * @param[in] mapViewId    - view id {example: primary or secondary}
    * @param[in] posX         - x pixel position on the screen
    * @param[in] posY         - y pixel position on the screen
    * \async   ::MAP_PICKED_INDEX_FROM_POI_ICONS_CHANGED
    *
    * @pre  If this method is used for picking a multi POI, it expects a previous pickInMap() call and can be used in use case
    *       where user can pick a POI from the multi POI picking results.
    * Note: Besides picking a mutli POI, this method should be used for picking indexed POI from the POI search result
    *       and not for picking normal POIs as there won't be any indexed returned for normal POIs from map-engine.
    *
    */
   void requestPickedIndexFromPoiIcons(MapViewId mapViewId, unsigned int posX, unsigned int posY);

   /**
    * @brief   returns index of the POI icon that was picked from the map view using requestPickedIndexFromPoiIcons().
    * @return  - a valid pickedIndexFromPoiIcons     if the same picking element type exists in the result vector
    *                                                of previous pickInMap() call or in POI search results.
    *          - invalid pickedIndexFromPoiIcons     if the picking element type does not exist in the result vector
    *                                                of previous pickInMap() call or in POI search results.
    */
   const PickedIndexFromPoiIcons& getPickedIndexFromPoiIcons() const;

   /**
    * @brief returns details of picked map elements provided as argument (currently supported POI).
    * Calls are made to different components to render information; response sent once after all components respond.
    *
    * For address book elements, address information returned in `PickingResultDetailAddressBookPoi` structure
    * will be as follows:
    * - POI type: a single string containing the complete address in the `m_address` field.
    * - non-POI type: individual components of the address in `m_houseNumber`, `m_street`, `m_city`, `m_state`, `m_country`.
    *
    * @param[in] mapViewId              - view id {primary or secondary} of Map.
    * @param[in] elementsListForDetails - list of objects holding info of elements whose details need to be fetched.
    */
   void requestPickingDetails(MapViewId mapViewId, const ::std::vector< PickingResults::PickingResult >& elementsListForDetails);

   /**
    * Stops the property-update MAP_PICKING_RESULTS_DISTANCE_UPDATE.
    * @param[in] mapViewId The map view (PRIMARY or SECONDARY) to which this request is sent.
    * @attention HMI should call this method whenever leaving map picking context, or when air distance and direction
    *            update for picked elements is no longer needed.
    */
   void releasePickInMap(MapViewId mapViewId) const;

   /**
   * @brief showPickedPOIsInMap method will show all the POIs from the picked result list.
   * For a better overview all other POI's will be switched off in this camera mode(MAP_MODE_LOCATION_OVERVIEW).
   * MapScale must be set if range parameter is set to 1.
   * Only in this combination the map scale has any effect to show a single POI in 2D-north-up
   * if range = 0 all the POI Icons on the map will be deleted
   * If range > 1 the map engine will calculate the map scale automatically and the given map scale will be ignored.
   * To display list-item indexes with the POI-Icon, the Enum ListItemIndexDisplay (LIST_ITEM_INDEX_DISPLAY__ON) can be used.
   * @param[in] mapViewId         - view id of the map{primary or secondary}
   * @param[in] visibleListInfo   - specifies the startOffset of the list as well as the index of the
   *                                icon to be highlighted and the range of elements which will be displayed
   *                                with the listItemIndexDisplay as LIST_ITEM_INDEX_DISPLAY__ON.
   * @param[in] mapScale          - if only one element is to be shown, mapScale will be used, otherwise map scale is
   *                                determined automatically by map-engine and the value passed will have no effect.
   * @param.[in] cameraAnimation   - Type of camera animation to be used.
   * Note : 1. startOffset and range set in the visibleListInfo determines those POI icons that needs to be displayed in PRIORITY style.
   *        2. The POI icons which doesn't fall under the visibleListInfo range will displayed in NORMAL style.
   *        3. The dynamic icons added to map will be deleted
   *           1. on exiting the LocationOverview camera mode
   *           2. When HMI requests to delete (range = 0) or
   *              update the icons(StartOffset, range etc. updated in VisibleListInfo).
   */
   void showPickedPOIsInMap(MapViewId mapViewId, const VisibleListInfo& pickedPOIsVisibilityListInfo, const ValidValue<int32_t>& mapScale,
                            CameraAnimation cameraAnimation);

   /**
    * @brief highlights the picked map element provided as argument.
    * @param[in] mapViewId               - view id {primary or secondary} of Map.
    * @param[in] elementsToBeHighlighted - list of objects holding details of elements to be highlighted
    */
   void highlightPickingResult(MapViewId mapViewId,
                               const ::std::vector< PickingResults::PickingResult >& pickingResultList);
   /**
    * @brief request all user poi icons. See related method getUserPoiIcons().
    *  \async ::MAP_USER_POI_ICONS_AVAILABLE
    */
   void requestUserPoiIcons();

   /**
    * @brief getter for the requested user poi icons after method 'requestUserPoiIcons' was called.
    *        Returns a list of structures holding icon data and the internal map id.
    */
   const ::std::vector< ::navmiddleware::Image >& getUserPoiIcons();

   /**
    * Continuously scrolls the map in a given direction and with the given speed, specified by the scrollSpeed and direction parameter.
    * ScrollDirection can be of 8 types. The map will scroll in opposite direction of selected direction,
    * so that the user is able to see in the specified direction.
    * ScrollSpeed varies between 0 to 9, where 1 is slowest speed and 9 being the fastest CameraSpeed and 0 would stop the scrolling.
    * The scrolling can be limited by maxScrollInPixels parameter, the map camera would move by an amount in pixels.
    * In case of a non-zero value to maxScrollInPixels parameter, scrolling will be limited to that extent in pixels and stops automatically after that.
    * If the maxScrollInPixels value is set to zero, the scroll would not be limited. In this case HMI should use stopScrollingMap()
    * explicitly to stop scrolling.
    * @param[in] mapViewId               - view id {primary or secondary} of Map.
    * @param[in] scrollDirection         - The direction in which map will be scrolled.
    * @param[in] scrollSpeed             - Speed at which map will be scrolled.
    * @param[in] maxScrollInPixels       - If the maxScrollInPixels value is set to zero, the scroll
    *                                      would not be limited. In this case HMI should use stopScrollingMap()
    *                                      explicitly to stop scrolling.
    */
   void startScrollingMap(MapViewId mapViewId, ScrollDirection scrollDirection,
                          ScrollSpeed scrollSpeed, uint16_t maxScrollInPixels);
   /**
    * Stop Scrolling the MAP
    * @param[in] mapViewId               - view id {primary or secondary} of Map.
    */
   void stopScrollingMap(MapViewId mapViewId);

   /**
    * @brief Rotates the map around the up-axis.
    * If maxRotationAngle is given as 0 with clockwise/counter-clockwise, the rotation is continuous.
    * If the Rotation type is BOUNDED, rotates by the degree provided by maxRotationAngle
    * and stops automatically after rotation of maxRotationAngle is achieved.
    * If the rotation type is BOUNDED, then it means map rotates to a maximum of maxRotationAngle and
    * If the maxRotationAngle is negative, map rotates counter-clockwise to a maximum of maxRotationAngle.
    * If the maxRotationAngle is positive, map rotates clockwise to a maximum of maxRotationAngle.
    * If the rotation type is Clockwise or CounterClockwise then maxRotationAngle should be 0.
    * @param[in] mapViewId          - If mapViewId is MAP_VIEW_ID__PRIMARY, then rotation is performed on Primary view.
    *                                 If mapViewId is MAP_VIEW_ID__SECONDARY, then rotation is performed on Secondary view.
    * @param[in] rotationType       - Rotation is clockwise,counter-clockwise or bounded
    * @param[in] rotationSpeed      - Speed at which the camera will rotate Range-> 1 to 9 where
    *                                 ROTATION_SPEED_1 is slowest speed and
    *                                 ROTATION_SPEED_9 is fastest speed
    * @param[in] maxRotationAngle   - Degree of rotation(applied when RotationType is BOUNDED) Range-> -360 to +360
    */
   void startRotatingMap(MapViewId mapViewId, RotationType rotationType, RotationSpeed rotationSpeed, int32_t maxRotationAngle);

   /**
    * Stop Rotating the MAP
    * @param[in] mapViewId - If mapViewId is MAP_VIEW_ID__PRIMARY, then rotation on Primary view is stopped.
    *                        If mapViewId is MAP_VIEW_ID__SECONDARY, then rotation on Secondary view is stopped.
    */
   void stopRotatingMap(MapViewId mapViewId);

   /**
    * @brief Rotates the camera of the given map view around the axis specified. Rotation
    * can be clockwise and counter-clockwise. The passed in angle will always undergo a modulus by 360 operation.
    *
    * @param[in] mapViewId        - view id of Map.
    * @param[in] coordinateAxis   - axis of rotation
    * @param[in] rotationType     - rotation type. ROTATION_TYPE_BOUNDED is not supported for this method
    * @param[in] angle            - angle of rotation with respect to the current camera heading
    *
    */
   void rotateCamera(MapViewId mapViewId, CoordinateAxis coordinateAxis, RotationType rotationType, uint32_t angle);

   /* This is part of tracking dots feature which stores the positions with a fix distance on the driven roads
    * This method deletes all tracking dots till current position */
   void deleteAllTrackingDots();

   /**
    * Evaluate Map Related Settings
    */
   void updateMapSettings(const settings::MapSettings&);
   /** Get the map settings object which is hosted by middleware **/
   settings::MapSettings getMapSettings() const;

   void updateMapTextFilterSettings(const settings::MapTextFilterSettings&);
   settings::MapTextFilterSettings getMapTextFilterSettings() const;

   void updateMapDynamicIconSettings(const settings::MapDynamicIconSettings&);
   settings::MapDynamicIconSettings getMapDynamicIconSettings() const;

   /**
    * reset MapDynamicIconSettings to defaults.
    */
   void resetMapDynamicIconSettings();

   void updateMapDetailTriggerSettings(const settings::MapDetailTriggerSettings& );
   settings::MapDetailTriggerSettings getMapDetailTriggerSettings() const;

   void updateChangeMapViewSettings(const settings::ChangeMapViewSettings&);
   settings::ChangeMapViewSettings getChangeMapViewSettings() const;

   void updateMapViewSettings(const settings::MapViewSettings&);
   settings::MapViewSettings getMapViewSettings() const;

   void updateMapWeatherSettings(const settings::MapWeatherSettings&);
   settings::MapWeatherSettings getMapWeatherSettings() const;

   void updateMapEvSettings(const settings::MapEvSettings&);
   settings::MapEvSettings getMapEvSettings() const;

   /* @brief - This settings enables the tracking dots feature and
    * @stores the positions with a fix distance on the driven roads
    */
   void updateMapTrackingDotsSettings(const settings::MapTrackingDotsSettings&);
   navmiddleware::settings::MapTrackingDotsSettings getMapTrackingDotsSettings() const;

   void updateMapCarsorSettings(const settings::MapCarsorSettings&);
   settings::MapCarsorSettings getMapCarsorSettings() const;

   /**
    * Returns the MapCameraModeInfo
    * Update after callback ::MAP_CAMERAMODE_CHANGED
    */
   const MapCameraModeInfo& getMapCameraModeInfo() const;

   /** Set the RouteAlternatives for show overview
    * Currently only MAP_VIEW_ID__PRIMARY is supported
    *
    * @param[in] mapViewId                 - view id {primary or secondary} of the map for which alternatives are required
    * @param[in] showAlternatives          - indicates number of alternatives to show
    * @param[in] moreRouteTypesValue       - indicates which type of more routes (multiple or spatial alternatives) shall be used
    * @param[in] destinationPreference     - indicates which route segment to display
    * @param[in] showPlannedAndGuidedRoute - indicates the display of planned and guided routes on map.
    *                                        By default the current active route is shown.
    *                                        If set to true both planned and guided route will be shown on map.
    *                                        In case the value is true and if planned route is not available,
    *                                        then guided route will be shown in the overview.
    */
   void showRouteOverview(MapViewId mapViewId,
                          ShowAlternatives showAlternatives,
                          settings::MoreRoutesTypeValue moreRouteTypesValue,
                          DestinationPreference destinationPreference,
                          bool showPlannedAndGuidedRoute = false);

   /**
      * @brief Show an overview for the editing/creating of a route.
      * If a route has already been created and route display is enabled in routeVisibilityOptions,
      * a route overview (2D, north-up) will be shown that may automatically change during the editing of the route.
      * If no route exists when this method is called, the map view will show a carsor locked
      * map in the given mapscale in 2D, north-up.
      *
      * @param[in] mapViewId              - view id {primary or secondary} of the map
      * @param[in] mapScale               - The mapscale will be ignored if a route exists
      *                                     and a route overview will be shown
      * @param[in] cameraAnimation        - Type of camera animation to be used.
      * @param[in] routeVisibilityOptions - The route/carsor/destination flag visibility in route overview
      *
      * Update after callback ::MAP_CAMERAMODE_CHANGED
      */
   void showEditRouteOverviewInMap(MapViewId mapViewId, const ValidValue<int32_t>& mapScale,
                                   CameraAnimation cameraAnimation,
                                   const ValidValue<RouteVisibilityOptions>& routeVisibilityOptions);

   /** Set the MAP mode for route overview
    *
    * @param[in] mapViewId    - view id {primary or secondary} of Map.
    * @param routeIndex index of the route to be highlighted
    */
   void highlightRoute(MapViewId mapViewId, unsigned int routeIndex);

   /**
    * Returns the TouchEventInfo
    * Update after callback ::MAP_TOUCH_EVENT
    */
   const TouchEventInfo& getTouchEventInfo() const;

   /** Select the route that is currently highlighted in route overview
    * @param[in] mapViewId               - view id {primary or secondary} of Map.
    */
   void selectHighlightedRoute(MapViewId mapViewId);

   /** Select the route that was active when the route overview was entered.
    * @param[in] mapViewId               - view id {primary or secondary} of Map.
    */
   void selectActiveRoute(MapViewId mapViewId);

   /** Set camera mode and view size for one or more views.
   *  If an addressed view does not exist it will be created.
   *  An existing view that is not addressed will be destroyed.
   * \async ::MAP_*_CAMERAMODE_CHANGED after the corresponding feature call has been issued.
   *
   *  @param[in] mapViewGroup          - The view group that the given viewModeConfiguration is assigned to.
   *  @param[in] viewModeConfiguration - configuration information for the addressed views.
   */
   void setMapCameraAndViewModes(MapViewGroup mapViewGroup, const ::std::vector< ViewModeConfiguration >& viewModeConfiguration);

   /**
   * @brief show weather overview in traffic map with specific zoom level.
   * If guidance is not active, the weather overview of the current vehicle position is shown.
   * If guidance is active, the route overview along with weather icons for current vehicle position, intermediate destinations/waypoints and destination is shown.
   * Note : If the distance between two points like current vehicle position and destination or between two waypoints is less than 30km,
   *        then only one weather icon will be shown depending on the view attribute priority.
   *        Visibility of weather icons is dependent on weather icon settings.
   *
   * @param[in] mapViewId           - View id of the map (PRIMARY or SECONDARY)
   * @param[in] cameraAnimation     - Type of camera animation to be used.
   */
   void showWeatherOverviewMap(MapViewId mapViewId, CameraAnimation cameraAnimation);

   /**
   * @brief show traffic overview in map.
   * If the list of trafficMessageID is empty and the guidance is ON/OFF, map shows a route overview with fix map scale of 30km radius.
   * If only one trafficMessageId is provided in the list, a valid map scale is also expected.
   * If more than one trafficMessageIDs are received, the map engine will calculate the map scale and the provided map scale is ignored.
   * @param[in] mapViewId           - View id of the map (PRIMARY or SECONDARY)
   * @param[in] trafficMessageID    - List of indices corresponding to the traffic message to be shown
   * @param[in] mapScale            - Required mapScale value.
   * @param[in] cameraAnimation     - Type of camera animation to be used.
   */
   void showTrafficOverviewMap(MapViewId mapViewId,
                               const ::std::vector<uint32_t>& trafficMessageID,
                               const ValidValue<int32_t>& mapScale,
                               CameraAnimation cameraAnimation);

   /**
    * Returns the NominalScaleInfos
    * Update after callback ::MAP_NOMINAL_SCALES_CHANGED
    */
   const NominalScaleInfos& getNominalScaleInfos() const;

   /** Center the map around the specified geo-position
   *
   * @param[in] mapViewId       - view id {primary or secondary} of Map.
   * @param[in] geoCoordinate   - geographical coordinate
   * @param[in] mapScale        - required mapScale value. If this parameter is not set or provided,
   *                              the previous scale value will be retained.
   *                              The valid range is as per the NominalScaleInfo and any value outside
   *                              the range would lead to an error.
   * @param[in] cameraAnimation - required animation for camera
   */
   void centerMapToGeoPosition(MapViewId mapViewId,
                               const GeoCoordinateDegree& geoCoordinate,
                               ValidValue<int32_t> mapScale,
                               CameraAnimation cameraAnimation = CameraAnimation__IMMEDIATE);

   /**
    *  Center the map around the location
    *  Pre-requisite to showSetLocationInMap() call is that one *MUST* use the setLocationWithXY() interface to
    *  select the location which is going to be shown on map. Usage is similar to startGuidance().
    *  @param[in] mapViewId          - view id {primary or secondary} of Map.
    *  @param[in] MapScale          - required mapScale value
    *  @param[in] cameraAnimation   - required animation for camera
    */
   void showSetLocationInMap(MapViewId mapViewId, int32_t mapScale, CameraAnimation cameraAnimation = CameraAnimation__IMMEDIATE);

   /**
   * @brief Center the map around the specified view-coordinate
   * @param[in] mapViewId        - view id {primary or secondary} of Map.
   * @param[in] posX             - relative x coordinate of the screen based on either PRIMARY or SECONDARY view
   * @param[in] posY             - relative y coordinate of the screen based on either PRIMARY or SECONDARY view
   * @param[in] cameraAnimation  - required animation for camera
   * Note: coordinates are relative to a particular view.
   */
   void centerMapToViewCoordinate(MapViewId mapViewId,
                                  uint16_t posX,
                                  uint16_t posY,
                                  CameraAnimation cameraAnimation = CameraAnimation__IMMEDIATE);

   /**
    * @brief Move the map center point of the current map view by [posX, posY] values with the given camera animation.
    * These screen positions are based on the position [0,0] which is top-left corner of the screen
    * @param[in] mapViewId        - view id of the map{primary or secondary}
    * @param[in] posX             - x Position on screen
    * @param[in] posY             - y Position on screen
    * @param[in] cameraAnimation  - Type of camera animation to be used to move the center point.
    */
   void moveMapCenterPoint(MapViewId mapViewId, int16_t posX, int16_t posY, CameraAnimation cameraAnimation);

   /**
    * @brief show geographical weather overview in map corresponding to the provided mapViewId,
    * centered at current postion in 2D North-up mode at higher scale.
    * Zoom range will be limited to higher range ~(10km - 1000km), Nominal scale property updated with new scale range
    * This interface shows geographical weather overview for storm, winds, hurricane, radar and chart.
    * @param[in] mapViewType         - map view type (Primary / Secondary)
    */
   void showWeatherInMap(MapViewId mapViewId);

   /**
   * @brief Configure the type of touch event that should be handled by current map view.
   * It can also be used to switch off handling of all touch event by map.
   * @param[in] mapViewId        - view id {primary or secondary} of Map.
   * @param[in] touchActions     - indicates the kind of actions to be triggered by the touch event
   * @param[in] pitchRange       - minimum and maximum value of pitch to restrict map's camera movement
   * @param[in] mapScaleRange    - minimum and maximum value of map scale to restrict map's scale
   */
   void configureTouch(MapViewId mapViewId, TouchActions touchActions,
                       PitchRange pitchRange, MapScaleRange mapScaleRange);

   /**
   * @brief Show locations in map.
   * @param[in] mapViewId                      - view id of the map{primary or secondary}
   * @param[in] geoCoordinates                 - list of geographical coordinates
                                                 These will be taken into account for the overview map scale and -
                                                 depending on the mapIconInfo parameter, for the display of numbered icons.
   * @param[in] optionalOverviewGeoCoordinates - list of geographical coordinates
                                                 These will be taken into account for the overview, but not for the
                                                 display of numbered icons
   * @param[in] mapIconInfo                    - specifies the startOffset of the list as well as the index of the
   *                                             icon to be highlighted. startOffset value should lie between 1 to 200 [1,200].
   *                                             Using showNumberedIcons option, the user can also specify if they want to
   *                                             overview only the coordinates or they also want to show the numbered POIs.
   * @param[in] mapScale                       - this value is considered only when a single GeoCoordinateDegree is given.
   *                                             When a list of GeocoordinateDegrees are passed, the map scale will be automatically
   *                                             decided by map-engine and the value passed will have no effect.
   * @param[in] cameraAnimation                - Type of camera animation to be used.
   *
   * Note: The dynamic icons added to map will be deleted
   *       1. on exiting the LocationOverview camera mode
   *       2. when the HMI requests to clear the icons (empty GeoCoordinate list) or
   *          update the icons (change in the list entries)
   */
   void showLocationsInMap(MapViewId mapViewId,
                           const ::std::vector< GeoCoordinateDegree >& geoCoordinates,
                           const ::std::vector< GeoCoordinateDegree >& optionalOverviewGeoCoordinates,
                           const MapIconInfo& mapIconInfo,
                           const ValidValue<int32_t>& mapScale,
                           CameraAnimation cameraAnimation);

   /**
   * @brief Show locations overview in map with the requested Icons displayed.
   * @param[in] mapViewId         - view id of the map{primary or secondary}
   * @param[in] visibleListInfo   - specifies the startOffset of the list as well as the index of the
   *                                icon to be highlighted and the range of elements which will be displayed.
   * @param[in] locationInputType - current Location input type which is active.
   * @param[in] mapScale          - if only one element is to be shown, mapScale will be used, otherwise map scale is
   *                                determined automatically by map-engine and the value passed will have no effect.
   * @param[in] cameraAnimation   - Type of camera animation to be used.
   *
   * Note:
   * 1. Currently, there are no icons for ADDRESS and SDS_ADDRESS, so showing these locations on MAP will not work.
   *    HMI can use the other interface to show them as numbered icons instead.
   *
   * 2. The dynamic icons added to map will be deleted
   *    1. on exiting the LocationOverview camera mode or
   *    2. When HMI requests to delete (range = 0) or update the icons(StartOffset, range etc.
   *       updated in VisibleListInfo)
   */
   void showLocationsInMap(MapViewId mapViewId,
                           const VisibleListInfo& visibleListInfo,
                           LocationInputType locationInputType,
                           const ValidValue<int32_t>& mapScale,
                           CameraAnimation cameraAnimation);

   /**
    * Returns the MapMultiViewInfo
    * Update after callback ::MAP_MULTIVIEWMODE_CHANGED
    */
   const MapMultiViewInfo& getMapMultiViewInfo() const;

   /**
   * @brief Detailed information for Intersection Map.
   * @This mode is used for showing an intersection at a guidance maneuver.
   * @param[in] mapViewId        - view id of the map{primary or secondary}
   * @param[in] cameraAnimation  - animation type to use when switching to the intersection map
   * @param[in] mode             - use either the map engine internal automatic or the manual mode
   */
   void showIntersectionMap(MapViewId mapViewId, CameraAnimation cameraAnimation, IntersectionMapMode mode = INTERSECTION_MAP_MODE_MANUAL);

   /**
   * @brief show geographical weather alerts in map corresponding to the provided mapViewId.
   * zoom-level : 500km
   * orientation : 2D North-up
   * map location : current carsor position in map
   *
   * @param[in] mapViewId        - view id of the map{primary or secondary}
   */
   void showWeatherAlertsInMap(MapViewId mapViewId);

   /**
   * @brief Display the map view as a widget (like a button, for example)
   *
   * @param[in] mapViewId        - view id of the map{primary or secondary}
   */
   void showMapWidget(MapViewId mapViewId);

   /**
    * @brief If there is a traffic incident on the route,
    * Show route overview from current route(with incident) and one alternative(detour) route.
    * The TrafficRouteOverview mode with COMPLETE will show the whole route as a "detour all" feature or
    * with DETOUR, only a section where the route spreads due to a traffic incident on the route will be shown
    * and user can come back to the original route
    * @param[in] mapViewId                 - view id of the map{primary or secondary}
    * @param[in] trafficRouteOverviewMode  - TRAFFIC_ROUTE_OVERVIEW_MODE__COMPLETE to show the whole route or
    *                                        TRAFFIC_ROUTE_OVERVIEW_MODE__DETOUR  to show only a section of the route
    * @param[in] cameraAnimation           - required animation for camera for Traffic Route overview.
    */
   void showTrafficDetourInMap(MapViewId mapViewId, TrafficRouteOverviewMode trafficRouteOverviewMode, CameraAnimation cameraAnimation);

   /**
    * @brief Show address book icons in map.
    * @param[in] mapViewId        - view id of the map{primary or secondary}
    * @param[in] entryIds         - List of address book entryIds
    * @param[in] selectedIndex    - index in the list to be highlighted
    * @param[in] mapScale         - this value is considered only when a single entryId is given.
    *                               When a list of entryIds are passed, the map scale will be automatically
    *                               decided by map-engine and the value passed will have no effect.When mapScale
    *                               is invalid and we have a single entryId,we use the old mapScale.
    * @param[in] cameraAnimation  - Type of camera animation to be used.
    *
    * Note: All the dynamic address book icons will be deleted,
    *       1. On exiting the LocationOverview camera mode.(Address book icons will be added back,
    *          if next camera mode supports it)
    *       2. When the HMI requests to clear the list (by passing empty entryId list) or
    *          update the icons (with the updated entryIds).
    * This method is also applicable for work and home entries besides address book entries.
    */
   void showAddressbookIconsInMap(MapViewId mapViewId,
                                  const ::std::vector< DestinationMemoryEntry::Id >& entryIds,
                                  int32_t selectedIndex,
                                  const ValidValue<int32_t>& mapScale,
                                  CameraAnimation cameraAnimation);

   /**
    * @brief show reachable area overview in map.
    * @param[in] mapViewId                 - view id of the map{primary or secondary}
    * @param[in] reachableAreaVisibility   - Reachable area visibility {carsor or destination}
    * @param[in] cameraAnimation           - required camera animation for reachable area overview.
    * Note: If reachableAreaVisibility is REACHABLE_AREA_VISIBILITY__DESTINATION
    * case1: If there is no waypoint, the reachable area will be shown at destination.
    * case2: If there is one waypoint, the reachable area will be shown at the waypoint.
    * case3: If there are more than one waypoint, the reachable area from the first (nearest) waypoint shall be shown.
    */
   void showReachableAreaOverviewInMap(MapViewId mapViewId,
                                       ReachableAreaVisibility reachableAreaVisibility,
                                       CameraAnimation cameraAnimation);

   /**
    * @brief configures the visibility of routes on Map
    * @param[in] mapViewId              - view id of the map{primary or secondary}
    * @param[in] routeIndices           - list of route index to be hidden.
    * If the list is empty all the routes will be made visible.
    * Note: No property update will be sent for the request
    */
   void configureRouteVisibilityOnMap(MapViewId mapViewId, const ::std::vector<unsigned int >& routeIndices);

   /**
   * @brief Delete the avoid area previewed
   * @param[in] mapViewId              - view id of the map{primary or secondary}
   */
   void deleteAvoidAreaPreview(MapViewId mapViewId);

   /**
   * @brief Get the vector data for the upcoming intersections (for external display)
   *
   * Update after callback ::MAP_INTERSECTION_VECTOR_DATA_CHANGED
   */
   const IntersectionMapVectorDataInfos& getIntersectionMapVectorDataInfos() const;

   /**
    * @brief Adds and displays an avoid area preview on the map
    *
    * @param[in] mapViewId              - view id of the map{primary or secondary}
    * @param[in] centerPoint           - point around which the avoid area is to be displayed [WGS84]
    * @param[in] edgeLength            - length of the edges of the avoid area [m]
    * @param[in] avoidAreaPreviewStyle - provides information about if the avoid area preview includes freeways or not
    */
   void addAvoidAreaPreview(MapViewId mapViewId, const GeoCoordinateDegree& centerPoint, uint32_t edgeLength, AvoidAreaPreviewStyle avoidAreaPreviewStyle);

   /**
    * @brief Switch ON/OFF crosshair visibility on MAP
    * @param[in] mapViewId             - view id of the map{primary or secondary}
    * @param[in] crossHairVisibility   - ON or OFF
    */
   void configureCrossHair(MapViewId mapViewId, CrossHairVisibility crossHairVisibility);

   /**
    * @brief sets the list of Marker(MarkerDisplayMode & GeoCoordinateDegree) for each MarkerType with MapLocationMarkerConfigurationOperation passed from the HMI.
    * HMI calls this method with list of Marker(MarkerDisplayMode & GeoCoordinateDegree) and MapLocationMarkerConfigurationOperation is set to
    * MARKER_CONFIGURATION_OPERATION__ADD, the new markers are added and appended to existing makers if any of same type.
    * HMI calls this method with list of Marker(MarkerDisplayMode & GeoCoordinateDegree) and MapLocationMarkerConfigurationOperation is set to
    * MARKER_CONFIGURATION_OPERATION__REPLACE, the new markers will be added and replace/remove any existing makers of same type.
    * HMI calls this method with empty list of Marker(MarkerDisplayMode & GeoCoordinateDegree) and MapLocationMarkerConfigurationOperation is set to
    * MARKER_CONFIGURATION_OPERATION__REPLACE, all existing location markers of type MarkerType will be deleted.
    * @param[in] mapViewId             - view id of the map{primary or secondary}
    * @param[in] mapLocationMarkers    - has details like list of Marker(MarkerDisplayMode & GeoCoordinateDegree) and MarkerType.
    * @param[in] operation             - to add/replace/remove location markers.
    */
   void configureMapLocationMarkers(MapViewId mapViewId, const MapLocationMarkers& mapLocationMarkers, MapLocationMarkerConfigurationOperation operation);

   /**
    * @brief allows to modify the display/illustration of the roads and road network to be shown in the map view
    * Jpn only feature
    * @param[in] mapViewId - identifier for the particular mapview to be configured
    * @param[in] roadNetworkIllustration - configure level of detail of road network drawing
    *
    */
   void configureRoadNetworkIllustration(MapViewId mapViewId, RoadNetworkIllustration roadNetworkIllustration);

   /**
    * @brief set a required mapScale
    * @param[in] mapViewId     - view id of the map{primary or secondary}
    * @param[in] mapScale      - required mapScale value
    */
   void setMapScale(MapViewId mapViewId, int32_t mapScale, CameraAnimation cameraAnimation = CameraAnimation__IMMEDIATE);

   /**
   * @brief Get the carsor position for the current intersection(for external display)
   *
   * Update after callback ::MAP_INTERSECTION_MAP_CARSOR_POSITION_CHANGED
   */
   const IntersectionMapCarsorPosition& getIntersectionMapCarsorPosition() const;

   /**
    * Returns the WeatherFeaturesStatusInfo
    * Update after callback ::MAP_WEATHER_FEATURES_STATUS_CHANGED
    */
   const WeatherFeaturesStatusInfo& getWeatherFeaturesStatusInfo() const;

   /**
   * @brief Show satellite image on map.
   * @param[in] mapViewId        - view id of the map{primary or secondary}
   * @param[in] geoCoordinate    - geographical coordinate
   * @param[in] mapScale         - user should provide a valid map scale value, which is one of the requested map scales
   *                               set before in prepareSatelliteView().
   */
   void showSatelliteImagesInMap(MapViewId mapViewId, const GeoCoordinateDegree& geoCoordinate, int32_t mapScale);

   /**
    * @brief Switch on street view mode. Street view of the given position is shown
    * @param[in] mapViewId        - view id of the map{primary or secondary}
    * @param[in] streetViewMode   - display mode of the street view
    * @param[in] geoCoordinate    - geo position of where to show streetview
    * @param[in] address          - address of the location for which the street view
    *                               image is required
    */
   void showStreetViewImagesInMap(MapViewId mapViewId,
                                  StreetViewMode streetViewMode,
                                  const GeoCoordinateDegree& geoCoordinate,
                                  const ValidValue< ::std::string >& address);

   /**
    * @brief Show Avoid area overview in map.
    * The client can configure the avoid area visibility and screen-coordinate request type to be ON/OFF
    * @param[in] mapViewId                       - view id of the map{primary or secondary}
    * @param[in] avoidAreaId                     - when set,avoidAreaId will contain the Id associated to a particular avoid area.
    * @param[in] avoidAreaVisibility             - if avoidAreaVisibility set to "OFF" no avoid area will be visible,
    *                                              if avoidAreaVisibility set to "MAP" show avoid area via stored context
    *                                              coordinates on map,if avoidAreaVisibility set to "HMI_OVERLAY" HMI shows
    *                                              the avoid area overlay in map.
    * @param[in] cameraAnimation                 - Type of camera animation to be used to show avoid area overview.
    *
    * \async ::MAP_AVOIDAREA_HMI_OVERLAY_INFO_CHANGED
    */
   void showAvoidAreaOverviewInMap(MapViewId mapViewId,
                                   const ValidValue<AvoidAreaId>& avoidAreaId,
                                   AvoidAreaVisibility avoidAreaVisibility,
                                   CameraAnimation cameraAnimation);

   /**
    * @brief get the HMI overlay of avoided area
    * This interface is specific for Renault variant
    * @param
    * @return reference to AvoidAreaHMIOverlay
    */
   const AvoidAreaHMIOverlay& getAvoidAreaHMIOverlay() const;

   /**
    * @brief Show Geo Fence in Map
    * @param[in] mapViewId          - view id of the map{primary or secondary}
    * @param[in] geoFence           - Has two GeoCoordinates, radius and geoFenceType where
    *                                 GeoFence will be drawn as a circle(using the radius that is provided) if geoCoordinate2 is empty.
    *                                 If geoCoordinate2 is not empty, then a rectangular geo fence will be drawn where
    *                                 geoCoordinate1 and geoCoordinate2 are opposite corners of a rectangle.
    *                                 GeoFenceType_Inclusion indicates if the car is leaving the fence and
    *                                 GeoFenceType_Exclusion indicates if the car is entering the fence.
    */
   void showGeoFenceInMap(MapViewId mapViewId, const GeoFence& geoFence);

   /**
    * @brief Shows a selected route segment in map, when user selects a turnlist element
    * @param[in] mapViewId          - view id of the map (primary or secondary)
    * @param[in] indexInRouteList   - element index of route list for which the overview shall be shown
    *                                 where invalid indexInRouteList would deselect the selected route segment.
    * @param[in] cameraAnimation    - type of camera animation used.
    */
   void showRouteSegmentSelectionInMap(MapViewId mapViewId, const ValidValue<uint32_t>& indexInRouteList, CameraAnimation cameraAnimation);

   /**
    * @brief Transforms the given view positions on the map to its relative geographical coordinates
    * in reference to the map that is being displayed.
    * @param[in] mapViewId            - view id of the map{primary or secondary}
    * @param[in] screenCoordinates    - list of view positions to be transformed
    * @return unique RequestId
    * Note: The results won't be accurate if there any ongoing activities on map like moving,
    * zooming or animation.
    *
    * \async ::MAP_TRANSFORMED_VIEW_POSITIONS_AVAILABLE
    */
   RequestId transformViewPositions(MapViewId mapViewId, const ::std::vector< ScreenCoordinate >& screenCoordinates);

   /**
    * @brief getter for the requested geo coordinates after method 'transformViewPositions' was called.
    *        Returns transformed view positions holding list of geo coordinates and the request id.
    */
   const TransformedViewPositionsInfo& getTransformedViewPositionsInfo() const;

   /**
    * @brief Request the street view preview image of the specified width and height for the specified
    * geo-coordinate.
    * The address is an optional parameter. If it is not specified, then the only geo-coordinate
    * will be considered while fetching the street view preview image.
    * @param[in] desiredWidth               - width of the street view preview image
    * @param[in] desiredHeight              - height of the street view preview image
    * @param[in] geoCoordinate              - geo-coordinate of the location for which the street view
    *                                         preview image is required
    * @param[in] address                    - address of the location for which the street view preview
    *                                         image is required
    *
    * \async ::MAP_STREET_VIEW_PREVIEW_IMAGE_AVAILABLE
    */
   void requestStreetViewPreviewImage(uint32_t desiredWidth,
                                      uint32_t desiredHeight,
                                      const GeoCoordinateDegree& geoCoordinate,
                                      const ValidValue< ::std::string >& address);

   /**
    * @brief getter for the street view preview image after property update "MAP_STREET_VIEW_PREVIEW_IMAGE_AVAILABLE"
    * is received.
    * Note: For most cases, the dimensions of the image returned in the StreetViewPreviewImageInfo instance match the requested
    * desiredWidth and desiredHeight parameters. However, if the maximum supported image width/height is exceeded, the
    * StreetViewPreviewImageInfo::m_width and StreetViewPreviewImageInfo::m_height will still match the desired image dimensions,
    * but the actual dimensions of the image held by StreetViewPreviewImageInfo::m_image will be different.
    */
   const StreetViewPreviewImageInfo& getStreetViewPreviewImage() const;

   /**
    * @brief Method for pre-caching street view images BEFORE switching to street view with setMapCameraAndViewModes/showStreetViewImagesInMap.
    * Once the pre-caching is done, a property update will be triggered (MAP_PREPARE_STREET_VIEW_DONE). The result must be checked
    * before switching to street view using getPrepareStreetViewStatusInfo(). Only if the result is PREPARE_STREETVIEW_STATUS__COMPLETE
    * should the street view be activated.
    * The address is an optional parameter. If it is not specified, then the only geo-coordinate
    * will be considered while fetching the street view preview image.
    *
    * @param[in] desiredWidthInPixels     - width of the satellite view preview image in pixels
    * @param[in] desiredHeightInPixels    - height of the satellite view preview image in pixels
    * @param[in] geoCoordinate            - geo-coordinate of the location for which the street view
    *                                       preview image is required
    * @param[in] address                  - address of the location for which the street view preview
    *                                       image is required
    *
    * \async ::MAP_PREPARE_STREET_VIEW_DONE
    */
   void prepareStreetView(uint32_t desiredWidthInPixels,
                          uint32_t desiredHeightInPixels,
                          const GeoCoordinateDegree& geoCoordinate,
                          const ValidValue< ::std::string >& address);

   /**
    * @brief getter for the status of the pre-caching of the street view image after property update "MAP_PREPARE_STREET_VIEW_DONE"
    * is received.
    */
   const PrepareStreetViewStatusInfo& getPrepareStreetViewStatusInfo() const;

   /**
    * @brief Method to cancel all the on-going requests for prepareStreetView.
    */
   void cancelPrepareStreetView();

   /**
    * @brief Method to cancel all the on-going requests for requestStreetViewPreviewImage.
    */
   void cancelRequestStreetViewPreviewImage();

   /**
    * @brief Method to get the status of map online features (street view, satellite view)
    *
    * \async ::MAP_ONLINE_FEATURE_AVAILABILITY_CHANGED
    *
    */
   const MapOnlineFeatureAvailabilityInfos& getMapOnlineFeatureAvailabilityInfos() const;

   /**
    * @brief Method to pre-cache the satellite view images for the list of map scales provided by HMI.
    * BEFORE switching to satellite view MAP_MODE_SATELLITE_VIEW, this pre-caching is necessary.
    * Once the pre-caching is done, a property update will be triggered (MAP_PREPARE_SATELLITE_VIEW_DONE). The result must be checked
    * before getting the satellite view info using getPrepareSatelliteViewInfo() and switching to satellite view state.
    *
    * @param[in] desiredWidthInPixels     - width of the satellite view preview image in pixels
    * @param[in] desiredHeightInPixels    - height of the satellite view preview image in pixels
    * @param[in] geoCoordinate            - geo-coordinate of the location for which the satellite view
    *                                       preview image is required
    * @param[in] mapScales                - list of map scales for which the satellite images should be precached.
    *
    *
    * \async ::MAP_PREPARE_SATELLITE_VIEW_DONE
    */
   void prepareSatelliteView(uint32_t desiredWidthInPixels,
                             uint32_t desiredHeightInPixels,
                             const GeoCoordinateDegree& geoCoordinate,
                             const ::std::vector<int32_t>& mapScales);

   /**
    * @brief getter for the status of the pre-caching of the satellite view image after property update "MAP_PREPARE_SATELLITE_VIEW_STATUS_CHANGED"
    * is received.
    */
   const PrepareSatelliteViewInfo& getPrepareSatelliteViewInfo() const;

   /**
    * @brief Method to cancel all the on-going requests for prepareSatelliteView.
    */
   void cancelPrepareSatelliteView();

   /**
    * Returns the MapVideoStreamStatusInfo
    * Update after callback ::MAP_VIDEO_STREAM_STATUS_CHANGED
    */
   const  MapVideoStreamStatusInfo& getMapVideoStreamStatusInfo() const;

   /**
    * Returns the MapCameraAndViewModesStatusInfo
    * Update after callback ::MAP_SET_MAP_CAMERA_AND_VIEW_MODES_DONE
    */
   const MapCameraAndViewModesStatusInfo& getMapCameraAndViewModesStatusInfo() const;

   /**
    * @brief Set the highlight state of a road in the given map view. The lineId is
    * used to identify the road whose highlight state is supposed to be modified.
    * @param[in] mapViewId        - view id {primary or secondary} of Map.
    * @param[in] lineId           - identifier of the road to (de-)highlighted
    * @param[in] isHighlighted    - flag indicating whether highlighting is to turned on or off
    */
   void highlightRoad(MapViewId mapViewId, const LineId& lineId, bool isHighlighted);

   /**
    * @brief Store a capture/screenshot of the given map view to the specified file.
    */
   void storeMapCapture(MapViewId mapViewId, const ::std::string& captureFilePath);

   /**
    * @brief Getter to obtain the result of a capture operation.
    * Update after property update "MAP_CAPTURE_RESULT_AVAILABLE"
    */
   const MapCaptureResultInfo& getMapCaptureResultInfo() const;

   /**
    * request to add the dynamic icons on map.
    * @Param: MapDynamicIconInfo contains
    * MapDynamicIconId: structure which contains 64bit unique id and and intent for each icon to be added on map.
    * GeoCoordinate: Position for the icon to be added on map.
    * IconPath: Path for the icon as string.
    * Property update "MAP_DYNAMICICON_REQUEST_RESULT_AVAILABLE" will be triggered after execution
    */
   void addDynamicIconsOnMap(const ::std::vector<MapDynamicIconInfo>& dynamicIconList);

   /**
    * request to delete the dynamic icons from map.
    * @Param: List of MapDynamicIconId which contains unique 64 bit dynamic icon ids
    *         and the intent of icons to be deleted
    */
   void deleteDynamicIconsOnMap(const ::std::vector<MapDynamicIconId>& mapDynamicIconIds);

   /**
    * @brief Getter to obtain the result of addDynamicIconsOnMap()/deleteDynamicIconsOnMap()
    * Update after callback :: "MAP_DYNAMICICON_REQUEST_RESULT_AVAILABLE"
    * Returns DynamicIconRequestResult which contains the mapping of
    * MapDynamicIconId and the corresponding icon addition status
    */
   const DynamicIconRequestResult& getDynamicIconRequestResult() const;
   ///@}

   // ================================================================================================================
   ///@{ \name ROUTING

   /**
    * Get the route settings object which is hosted by middleware
    */
   settings::RouteCriteriaSettings getRouteCriteriaSettings() const;
   settings::RouteOptionsSettings getRouteOptionSettings() const;
   settings::VehicleProfileSettings getVehicleProfileSettings() const;

   /**
    * Get the last route restore status of RouteCriteriaSettings
    */
   settings::RestoreStatus getLastRestoreStatusOfRouteCriteriaSettings() const;

   /**
    * Get the last route restore status of RouteOptionSettings
    */
   settings::RestoreStatus getLastRestoreStatusOfRouteOptionsSettings() const;

   /**
    * Get the route settings object from the current route (guided route or planned route as available)
    */
   settings::RouteOptionsSettings getRouteOptionSettingsFromRoute() const;

   /**
    * Send updated Hmi route settings back to the middleware.
    * After calling this method, changes on the local object will not affect
    * the passed reference (middleware creates a copy). To update middleware settings,
    * call this method again.
    */
   void updateRouteCriteriaSettings(const settings::RouteCriteriaSettings&);
   void updateRouteOptionSettings(const settings::RouteOptionsSettings&);
   void updateVehicleProfileSettings(const settings::VehicleProfileSettings&);

   /**
    * Apply the changed route option setting to the current route(guided route or planned route as available)
    * and this route option setting will not be stored persistently.
    */
   void applySettingsToRoute(const settings::RouteOptionsSettings& settings);

   /**
    *  Deletes all learned routes
    *
    *  \async ::ROUTE_LEARNED_ROUTE_DELETED
    */
   void deleteLearnedRoutes();

   /**
    * Deletes all data regarding predicted destinations.
    *
    *  \async ::ROUTE_LEARNED_ROUTE_DELETED
    */
   void deletePredictedDestinations();

   /**
    * Set start point(origin) independent of the CVP provided by position component.
    * Remark: Start position can be set only for a planned route,
    * and starting of guidance is not possible.
    * Starting guidance is only possible when the actual CVP is used.
    */
   void setStartPosition();

   /**
    * Delete previously set start position.
    * This start point(origin) is independent of the CVP provided by position component.
    */
   void deleteStartPosition();

   /**
    * Starts the route calculation to the destination/tour previously set using a set location function  or
    * starts pre route calculation to the destination even before user has confirmed the route.
    * Remark 1: if routeCalculationMode is ROUTE_CALCULATION_MODE__DEFAULT, starts the route calculation to the destination/tour
    *           previously set using a set location function and the property updates ROUTE_ARRIVALINFOS_CHANGED,
    *           ROUTE_ARRIVALINFOS_ALTERNATIVES_CHANGED, ROUTE_CALCULATION_STARTED, ROUTE_ARRIVALINFOS_CHANGED are triggered.
    * Remark 2: if routeCalculationMode is ROUTE_CALCULATION_MODE__STEALTH, starts pre route calculation to the destination
    *           even before user has confirmed the route and only ROUTE_PRE_CALCULATION_STARTED property update is triggered.
    * @param RouteCalculationMode routeCalculationMode (route calculation started for default mode or for pre calculation mode)
    * \async ::ROUTE_ARRIVALINFOS_CHANGED(if routeCalculationMode = ROUTE_CALCULATION_MODE__DEFAULT)
    * \async ::ROUTE_ARRIVALINFOS_ALTERNATIVES_CHANGED (if routeCalculationMode = ROUTE_CALCULATION_MODE__DEFAULT)
    * \async ::ROUTE_CALCULATION_STARTED (if routeCalculationMode = ROUTE_CALCULATION_MODE__DEFAULT)
    * \async ::ROUTE_CALCULATION_INFO_CHANGED (if routeCalculationMode = ROUTE_CALCULATION_MODE__DEFAULT)
    * \async ::ROUTE_PRE_CALCULATION_STARTED (if routeCalculationMode = ROUTE_CALCULATION_MODE__STEALTH)
    */
   void startRouteCalculation(RouteCalculationMode routeCalculationMode = ROUTE_CALCULATION_MODE__DEFAULT);

   /**
    * Stops the route calculation.
    * \async ::ROUTE_CALCULATION_STOPPED
    */
   void stopRouteCalculation();

   /**
    * Recalculate the route using the current dynamic information
    * \async ::ROUTE_ARRIVALINFOS_CHANGED(if routeCalculationMode = ROUTE_CALCULATION_MODE__DEFAULT)
    * \async ::ROUTE_ARRIVALINFOS_ALTERNATIVES_CHANGED (if routeCalculationMode = ROUTE_CALCULATION_MODE__DEFAULT)
    * \async ::ROUTE_CALCULATION_STARTED (if routeCalculationMode = ROUTE_CALCULATION_MODE__DEFAULT)
    * \async ::ROUTE_CALCULATION_INFO_CHANGED (if routeCalculationMode = ROUTE_CALCULATION_MODE__DEFAULT)
    * \async ::ROUTE_PRE_CALCULATION_STARTED (if routeCalculationMode = ROUTE_CALCULATION_MODE__STEALTH)
    */
   void recalculateRoute();

   /**
    * activates the precalculated route.
    * \async ::ROUTE_CALCULATION_STARTED
    */
   void activatePrecalculatedRoute();
   /**
    * Returns true if there is an active route, means either a guided route or a planned route; false otherwise
    */
   bool hasActiveRoute() const;

   /**
    * @DEPRECATED :Calculates route alternatives to the current single route.
    * Depending on route option RouteOptionsSettings::m_moreRoutesType either multiple routes
    * (route criteria fast, economic, short) or spatial alternatives are calculated.
    * \async ::ROUTE_ARRIVALINFOS_ALTERNATIVES_CHANGED
    */
   void calculateMoreRoutes();//This method isn’t supported anymore and will be removed soon

   /**
    * Returns more details about the cause for a route calculation reason or error triggered by ::ROUTE_CALCULATION_INFO_CHANGED
    * Also provides the status of the route restrictions regarding seasonally closed roads,
    * time restricted roads and LCV restriction violated roads.
    * Property update for seasonally closed roads and time restricted roads is ::ROUTE_RESTRICTION_AVOIDED.
    * Property update for LCV restriction violated roads is ::ROUTE_LCV_RESTRICTION_VIOLATION_STATUS_CHANGED.
    */
   const RouteCalculationInfo& getRouteCalculationInfo() const;

   /**
    * Returns details about all waypoint operations performed on the planned route triggered by ::GUIDANCE_STARTED_TO_PLANNED_ROUTE
    */
   const PlannedRouteHistoryInfos& getPlannedRouteHistoryInfos() const;

   /**
    * Selects the route last highlighted in route overview as active route (active route: guided route,
    * if guidance is already active or assigned to be the guided route, if guidance is started afterwards).
    * \async ::ROUTE_ARRIVALINFOS_CHANGED
    */
   void activateHighlightedRoute();//@DEPRECATED This method isn’t supported anymore and will be removed soon

   /**
    * Sends request to remove unused route alternatives other than the currently active route
    */
   void removeUnusedRoutes();//@DEPRECATED This method isn’t supported anymore and will be removed soon

   /**
    * requestWaypointList functionality provides waypoint list of planned route(if exists) or guided route
    * based on the parameter waypointListMode
    * Remark 1: requestWaypointList is asynchronous - use getWaypointInfos to get way point list representation,
    *           upon receipt of ROUTE_WAYPOINTINFOS_CHANGED
    * Remark 2: if waypointListMode parameter is WAYPOINT_LIST_MODE__VIEW, consider waypoint list of planned
    *           route(if exists) Or guidedRoute
    * Remark 3: if waypointListMode is WAYPOINT_LIST_MODE__EDIT_WITH_RECALCULATION, consider waypoint list of planned
    *           route(if exists) Or clone the guided route, start route calculation on cloned route and consider its waypoint list
    * Remark 4: if waypointListMode is WAYPOINT_LIST_MODE__EDIT_WITHOUT_RECALCULATION, consider waypoint list of planned
    *          route(if exists) Or clone the guided route with no route calculation and consider its waypoint list
    *
    *@param WaypointListMode waypointListMode (view mode or clone with/without subsequent route calculation)
    *
    * \async ::ROUTE_WAYPOINTINFOS_CHANGED
    */
   void requestWaypointList(WaypointListMode waypointListMode);

   /**
    * requestWaypointListPlannedRoute functionality provides waypoint list of the planned (non guided) route if exists.
    * If there is no planned route an empty list will be provided.
    * Remark 1: requestWaypointList is asynchronous - use getWaypointInfos to get way point list representation,
    * upon receipt of ROUTE_WAYPOINTINFOS_CHANGED
    *
    * \async ::ROUTE_WAYPOINTINFOS_CHANGED
    */
   void requestWaypointListPlannedRoute();

   /**
    * Returns info (location string and route criterion) about way points, which are either currently edited
    * (e.g. during insert, move, delete way point operation) or already used in route calculation.
    * WaypointInfos is only the HMI representation of the real way point list, which is managed by a PresCtrl submodule.
    * Update after property callback ::ROUTE_WAYPOINTINFOS_CHANGED
    */
   const WaypointInfos& getWaypointInfos() const;

   /**
    * This method applies the current way point list to the route based on waypointEditMode.
    * Current waypoint list can be modified with waypoint edit methods insertWaypoint, deleteWaypoint,
    * moveWaypoint, setRouteCriterionForWaypoint.
    * Remark1: If the WaypointEditMode is WAYPOINT_EDIT_MODE__WITHOUT_RECALCULATION, then
    *          this method a flag, that the current way point list shall be taken in next startGuidance() request.
    *          Route will be calculated/recalculated as per waypoint list only after start guidance.
    * Remark2: If the WaypointEditMode is WAYPOINT_EDIT_MODE__WITH_RECALCULATION, then
    *          a planned route will be introduced and the the current way point list will be applied to the
    *          planned route i.e. route will be calculated/recalculated as per waypoint list.
    * Remark3: If this method is called while a planned route is active, the waypoint list will be directly applied to planned route
    *          and planned route will be recalculated.
    *
    * @param WaypointEditMode waypointEditMode (edit with or without subsequent route calculation)
    *          !!!IMPORTANT!!! Parameter will be used only in ROW navigation.
    *          For Japan Navigation irrespective of the parameter value, it will always be with Recaculation.
    *
    * \async ::ROUTE_WAYPOINTINFOS_CHANGED (if called with parameter WAYPOINT_EDIT_MODE__WITH_RECALCULATION)
    *
    * \async ::ROUTE_CALCULATION_STARTED (if called with parameter WAYPOINT_EDIT_MODE__WITH_RECALCULATION)
    */
   void applyWaypointListChange(WaypointEditMode waypointEditMode = WAYPOINT_EDIT_MODE__WITHOUT_RECALCULATION);

   /**
    * Inserts current (new created or last selected) location as new way point in current way point list.
    * Remark1: This method can also be called when neither planned nor guided route exists.
    * Remark2: This method is called with or without active guidance in background
    *          and if the WaypointEditMode is WAYPOINT_EDIT_MODE__WITH_RECALCULATION, the following actions are done:
    *          1. a planned route will be introduced by a clone of the currently guided route except there is already a planned route.
    *          2. the new way point is immediately inserted in the planned route and recalculation started for planned route.
    *          3. To take it over into the guided route StartGuidance() has to be called.
    * Remark3: If this method is called while active guidance in background
    *          and if the WaypointEditMode is WAYPOINT_EDIT_MODE__WITHOUT_RECALCULATION, the following actions are done:
    *          1. a planned route will be introduced by a clone of the currently guided route except there is already a planned route.
    *          2. The waypoint will be inserted into the planned route and not directly calculated
    *             The user can edit further waypoints without permanent recalculation (edit mode)
    *          Call of applyWaypointListChange() followed by startGuidance() is required to apply all way point list
    *          changes to the guided route.
    * Remark4: If this method is called while a planned route is active but no guidance is started
    *          and the WaypointEditMode is WAYPOINT_EDIT_MODE__WITH_RECALCULATION the waypoint list will be directly changed and route will be calculated.
    * Remark5: If this method is called while a planned route is active but no guidance is started
    *          and the WaypointEditMode is WAYPOINT_EDIT_MODE__WITHOUT_RECALCULATION:
    *          Nothing will happen. This mode isn't supported!
    * Remark6: insertWaypoint is asynchronous - use getWaypointInfos to get changed way point list representation,
    *          upon receipt of ROUTE_WAYPOINTINFOS_CHANGED.
    *
    * @param unsigned int newIndex of way point (after insertion)
    * @param WaypointEditMode waypointEditMode (edit with or without subsequent route calculation)
    *
    * \async ::ROUTE_WAYPOINTINFOS_CHANGED
    *
    * \async ::ROUTE_CALCULATION_STARTED (if called (with parameter WAYPOINT_EDIT_MODE__WITH_RECALCULATION) while a planned route is calculated/active with active guidance in background)
    *
    */
   void insertWaypoint(unsigned int newIndex, WaypointEditMode waypointEditMode);

   /**
    * Deletes all given way points in current way point list.
    * Remark1: This method can also be called for all existing waypoints including the final destination.
    * Remark2: If this method is called while active guidance in background
    *          and the WaypointEditMode is WAYPOINT_EDIT_MODE__WITH_RECALCULATION, the following actions are done:
    *          1. a planned route will be introduced by a clone of the currently guided route except there is already a planned route.
    *          2. the way point is immediately deleted from the planned route and recalculation started for planned route.
    *          3. To take it over into the guided route StartGuidance() has to be called.
    * Remark3: If this method is called while active guidance in background
    *          and the WaypointEditMode is WAYPOINT_EDIT_MODE__WITHOUT_RECALCULATION, the following actions are done:
    *          1. a planned route will be introduced by a clone of the currently guided route except there is already a planned route.
    *          2. The waypoint will be deleted from the planned route and not directly calculated
    *             The user can edit further waypoints without permanent recalculation (edit mode)
    *          Call of applyWaypointListChange() followed by startGuidance() is required to apply all way point list changes to the guided route.
    * Remark4: If this method is called while a planned route is active but no guidance is started
    *          and the WaypointEditMode is WAYPOINT_EDIT_MODE__WITH_RECALCULATION the waypoint list will be directly changed and route will be calculated.
    * Remark5: If this method is called while a planned route is active but no guidance is started
    *          and the WaypointEditMode is WAYPOINT_EDIT_MODE__WITHOUT_RECALCULATION:
    *          Nothing will happen. This mode isn't supported!
    * Remark6: deleteWaypoint is asynchronous - use getWaypointInfos to get changed way point list representation,
    *          upon receipt of ROUTE_WAYPOINTINFOS_CHANGED.
    *
    * @param unsigned int index of way point to delete
    * @param WaypointEditMode waypointEditMode (edit with or without subsequent route calculation)
    *
    * \async ::ROUTE_WAYPOINTINFOS_CHANGED
    *
    * \async ::ROUTE_CALCULATION_STARTED (if called (with parameter WAYPOINT_EDIT_MODE__WITH_RECALCULATION) while a planned route is calculated/active with active guidance in background)
    *
    */
   void deleteWaypoints(::std::vector<unsigned int>& waypointIndexList, WaypointEditMode waypointEditMode);

   /**
    * Moves way point in current way point list.
    * Remark1: If this method is called while active guidance in background
    *          and the WaypointEditMode is WAYPOINT_EDIT_MODE__WITH_RECALCULATION, the following actions are done:
    *          1. a planned route will be introduced by a clone of the currently guided route except there is already a planned route.
    *          2. the way point is immediately deleted from the planned route and recalculation started for planned route.
    *          3. To take it over into the guided route StartGuidance() has to be called.
    * Remark2: If this method is called while active guidance in background
    *          and the WaypointEditMode is WAYPOINT_EDIT_MODE__WITHOUT_RECALCULATION, the following actions are done:
    *          1. a planned route will be introduced by a clone of the currently guided route except there is already a planned route. In this case a clone of the planned route will be introduced.
    *          2. The waypoint will be deleted from the planned route and not directly calculated
    *             The user can edit further waypoints without permanent recalculation (edit mode)
    *          Call of applyWaypointListChange() followed by startGuidance() is required to apply all way point list changes to the guided route or the planned route if exist.
    * Remark3: If this method is called while a planned route is active but no guidance is started
    *          and the WaypointEditMode is WAYPOINT_EDIT_MODE__WITH_RECALCULATION the waypoint list will be directly changed and route will be calculated.
    * Remark4: If this method is called while a planned route is active but no guidance is started
    *          and the WaypointEditMode is WAYPOINT_EDIT_MODE__WITHOUT_RECALCULATION:
    *          A clone of the planned route route will be introduced.
    *          Call of applyWaypointListChange() is required to take over the changes as planned route. The previous planned route will be deleted.
    * Remark5: moveWaypoint() is asynchronous - use getWaypointInfos to get changed way point list representation,
    *          upon receipt of ROUTE_WAYPOINTINFOS_CHANGED.
    *
    * @param unsigned int fromOldIndex of way point (before move operation)
    * @param unsigned int toNewIndex of way point (after move operation)
    * @param WaypointEditMode waypointEditMode (edit with or without subsequent route calculation)
    *
    * \async ::ROUTE_WAYPOINTINFOS_CHANGED
    *
    * \async ::ROUTE_CALCULATION_STARTED (if called (with parameter WAYPOINT_EDIT_MODE__WITH_RECALCULATION) while a planned route is calculated/active with active guidance in background)
    *
    */
   void moveWaypoint(unsigned int fromOldIndex, unsigned int toNewIndex, WaypointEditMode waypointEditMode);

   /**
    * Skips current waypoint from the way point list.
    * This method is called when guidance is active
    * The current waypoint index will be updated in the route and accordingly route will be calculated.
    * @param bool skipWaypoint, user's response on confirming skip waypoint popup
    * @param unsigned int waypointIndex, waypoint index to be skipped
    *
    * \async ::ROUTE_WAYPOINTINFOS_CHANGED
    *
    */
   void skipWaypoint(bool skipWaypoint, unsigned int waypointIndex);

   /**
    * Sets route criterion for way point in current way point list.
    * Remark1: route criterion is not immediately changed in guided route (only temporarily modified in current way point list).
    * Remark2: setRouteCriterionForWaypoint is asynchronous - use getWaypointInfos to get changed way point list representation,
    *          upon receipt of ROUTE_WAYPOINTINFOS_CHANGED.
    * Remark3: Call of applyWaypointListChange() followed by startGuidance() is required to apply all way point list
    *          changes to the guided route.
    *
    * @param unsigned int index of way point
    * @param RouteCriterion routeCriterion
    *
    * \async ::ROUTE_WAYPOINTINFOS_CHANGED
    *
    */
   void setRouteCriterionForWaypoint(unsigned int index, settings::RouteCriterion routeCriterion);


   void setRouteCriterionForOptimization(settings::RouteCriterion routeCriterion);


   void setTimeWindowsForAllWaypoints(const ::std::vector<WaypointOptimizationDetails>& waypointOptimizationDetails);


   void setFinalDestinationForOptimization(unsigned int waypointIndex, bool isFinalDest);


   void setOptimizationRange(unsigned int optimizeFromSegment, unsigned int optimizeToSegment);


   void applyRouteOptimization();

   void cancelRouteOptimization();

   /**
    * Sets optimization settings for current route.
    * Remark1: route optimization settings are not immediately changed in guided route (only temporarily modified in current route).
    * Remark2: setOptimizationSettingsForRoute is asynchronous - use getWaypointInfos to get changed way point list representation,
    *          upon receipt of ROUTE_WAYPOINTINFOS_CHANGED.
    * Remark3: Call of applyWaypointListChange() followed by startGuidance() is required to apply all way point list
    *          changes to the guided route.
    *
    * @param unsigned int index of way point
    * @param settings::WaypointOptimizationSettings& waypointOptimizationSettings
    *
    * \async ::ROUTE_WAYPOINTINFOS_CHANGEDShow location overview for POI icons : \n \
    *
    */
   void setOptimizationSettingsForRoute(unsigned int optimizeFromSegment, unsigned int optimizeToSegment);

   /**
    * This method shall be called to cancel the way points edit mode.
    * Changes of current way point list will then be discarded and not taken over in guided route
    */
   void cancelEditWaypointsMode();

   /**
    * Uses current (new created, modified or last selected) location to replace way point location.
    * Remark1: If this method is called while active guidance in background
    *          and the WaypointEditMode is WAYPOINT_EDIT_MODE__WITH_RECALCULATION, the following actions are done:
    *          1. a planned route will be introduced by a clone of the currently guided route except there is already a planned route.
    *          2. the way point is immediately replaced in the planned route and recalculation started for planned route.
    *          3. To take it over into the guided route StartGuidance() has to be called.
    * Remark2: If this method is called while active guidance in background
    *          and the WaypointEditMode is WAYPOINT_EDIT_MODE__WITHOUT_RECALCULATION, the following actions are done:
    *          1. a planned route will be introduced by a clone of the currently guided route except there is already a planned route.
    *          2. The waypoint will be replaced in the planned route and not directly calculated
    *             The user can edit further waypoints without permanent recalculation (edit mode)
    *          Call of applyWaypointListChange() followed by startGuidance() is required to apply all way point list changes to the guided route.
    * Remark3: If this method is called while a planned route is active but no guidance is started
    *          and the WaypointEditMode is WAYPOINT_EDIT_MODE__WITH_RECALCULATION the waypoint list will be directly changed and route will be calculated.
    * Remark4: If this method is called while a planned route is active but no guidance is started
    *          and the WaypointEditMode is WAYPOINT_EDIT_MODE__WITHOUT_RECALCULATION:
    *          Nothing will happen. This mode isn't supported!
    * Remark5: replaceWaypointLocation is asynchronous - use getWaypointInfos to get changed way point list representation,
    *          upon receipt of ROUTE_WAYPOINTINFOS_CHANGED.
    *
    * @param unsigned int index of way point
    *
    * \async ::ROUTE_WAYPOINTINFOS_CHANGED
    *
    * \async ::ROUTE_CALCULATION_STARTED (if called (with parameter WAYPOINT_EDIT_MODE__WITH_RECALCULATION) while a planned route is calculated/active with active guidance in background)
    *
    */
   void replaceWaypointLocation(unsigned int index, WaypointEditMode waypointEditMode);

   /**
    * For accepting alternative route, which has been calculated in
    * background because of new incoming traffic events
    */
   void acceptTrafficImpactRouteAlternative();

   /**
    * For rejecting alternative route,which has been calculated in
    * background because of new incoming traffic events
    */
   void rejectTrafficImpactRouteAlternative();

   /**
    * This method starts the continuous generation of route list info for the given route alternative
    * and the given parameter.
    *
    * @param unsigned int index of route alternative
    * @param RouteListConfiguration routeListConfiguration, the content of the route list can be configured (enable freeway exits)
    *
    */
   void startRouteListInfo(unsigned int indexAlternative, RouteListConfiguration routeListConfiguration);

   /**
    * This method stops the continuous generation of route list info for the given route alternative.
    *
    * @param unsigned int indexAlternative
    *
    */
   void stopRouteListInfo(unsigned int indexAlternative);

   /**
    * Method for requesting the route list.
    * Remark1: property callback ::ROUTE_LIST_INVALIDATE indicates that a newer (complete) route list is available.
    * If the route list is currently displayed, HMI shall call this method to get the HMI relevant part of this new complete
    * route list. This method is asynchronous. The result shall be retrieved by calling getRouteListInfos upon receipt of
    * property callback ::ROUTE_LIST_INFO_CHANGED.
    * Remark2: After call of requestRouteList typically several ::ROUTE_LIST_INFO_CHANGED notifications are send to indicate
    * a change of first element in RouteListInfos (other elements are still valid).
    * Remark3: Method shall return a valid RequestId if the request can be processed, and an invalid RequestId if cannot be processed.
    * If an invalid RequestId is returned,MW will not trigger any property updateROUTE_LIST_INFOS_CHANGED or ROUTE_LIST_ERROR.
    *
    * @param unsigned int indexAlternative index of route alternative
    * @param unsigned int startIndex index of first route list element
    * @param unsigned int numOfElements number of requested route list elements
    *
    *  \async ::ROUTE_LIST_INFO_CHANGED
    *
    *  \async ::ROUTE_LIST_ERROR
    *
    */
   ValidValue<RequestId> requestRouteList(unsigned int indexAlternative, unsigned int startIndex, unsigned int numOfElements);

   /**
    * Method for getting the route list, which was requested before by requestRouteList.
    * Property callback ::ROUTE_LIST_INFO_CHANGED indicates that an (at least partial) update of RouteListInfos is available.
    * Property callback ::ROUTE_LIST_ERROR.
    */
   const RouteListInfos& getRouteListInfos() const;

   /*
    * Avoid route list elements. The route will be recalculated. This will result in property callback ROUTE_LIST_INVALIDATE.
    * This will result that route element(s) will be avoided in all route alternatives until they are released using
    * cancelAvoidRouteListElements()
    * @param unsigned int indexAlternative index of route alternative
    * @param unsigned int startIndex index of the first element in RouteListInfos that has to be avoided
    * @param unsigned int endIndex index of the last element in RouteListInfos that has to be avoided. Is the same index in case of only one element.
    */
   void avoidRouteListElements(unsigned int indexAlternative, unsigned int startIndex, unsigned int endIndex);

   /*
    * Cancel all avoided route list elements. The route will be recalculated. This will result in property callback ROUTE_LIST_INVALIDATE
    */
   void cancelAvoidRouteListElements();

   /**
    * This method starts the continuous update of jam ahead list enabling or disabling some entries in the list
    * those which can't be applied due to the remaining distance to next waypoint/destination.
    *
    */
   void requestJamAheadList();

   /**
    * This method stops the continuous update of jam ahead list.
    *
    */
   void releaseJamAheadList();

   /**
    * Send request for setting jam ahead based on selected entry in the list e.g. 200m, 500m, 1km, 5km, 10km
    * Or in miles: 1/8mi, 1/4mi, 1/2mi, 3mi, 5mi.
    * Starting position is current vehicle position.
    *
    * @param unsigned int index - index of an entry in the jam ahead list, starting with 0
    */
   void selectJamAheadListEntry(unsigned int index);

   /**
    * Method for getting the jam ahead list.
    * Property callback ::ROUTE_JAMAHEADINFO_CHANGED indicates that update of JamAheadListInfo is available.
    */
   const JamAheadListInfo& getJamAheadListInfo() const;

   /**
    * Send request to cancel jam ahead.
    *
    */
   void cancelJamAhead();


   /**
    * Start route calculation for stored route.
    *
    * @param waypointList Stored route waypointList.
    *
    * \async ::ROUTE_CALCULATION_STARTED
    */
   void startStoredRouteCalculation(const ::std::vector<DestinationMemoryWaypoint>& waypointList);

   /**
    * Start route calculation for stored route.
    *
    * @param routeId Stored route Id.
    * @param numberOfWaypointsToSkip when creating a route it is possible to skip number of waypoints (starting from the beginning).
    *
    * Remark:
    * It is not possible to have an empty route, so the numberOfWaypointsToSkip must be less than number of waypoints in the stored route.
    * E.g. For a restart a tour scenario it should be checked before calling this function if final waypoint have been reached.
    * \async ::ROUTE_CALCULATION_STARTED
    */
   void startStoredRouteCalculationWithDestinationMemoryId(DestinationMemoryRoute::Id routeId, uint32_t numberOfWaypointsToSkip);

   /**
    * Start route calculation for stored route from the first waypoint/destination and not the current vehicle position
    *
    * @param routeId Stored route Id.
    *
    * Remark:
    * This route is only for planning purpose and guidance cannot be started on it.
    * \asyn ::ROUTE_CALCULATION_STARTED
    *
    */
   void startStoredRouteCalculationForPlanning(DestinationMemoryRoute::Id routeId);

   /**
    * Start route calculation with freetextsearch selected (history) element
    *
    * \async ::ROUTE_CALCULATION_STARTED
    */
   void startStoredRouteCalculationWithFtsSelection();

   /**
    * Start route calculation from the first waypoint/destination and not the current vehicle position.
    *
    * Remark:
    * This route is only for planning purpose and guidance cannot be started on it.
    * \asyn ::ROUTE_CALCULATION_STARTED
    */
   void startRouteCalculationForPlanning();

   /**
    * Start route calculation for a given tour.
    * \async ::ROUTE_CALCULATION_STARTED
    * \async ::ROUTE_CALCULATION_INFO_CHANGED
    */
   void startTourCalculation(const Tour& tour);

   /**
    * Requests for the list of countries and states along the requested route.
    * This also provides the corresponding country and state flag icon information.
    *
    * @param index alternative of the given route
    * \async ::ROUTE_COUNTRY_AND_STATE_INFO_CHANGED
    */
   void requestRouteCountryAndStateInfo(unsigned int indexAlternative);

   /**
    * Returns list of countries and states along the requested active route or alternative routes.
    * Property callback ROUTE_COUNTRY_AND_STATE_INFO_CHANGED indicates RouteCountryAndStateInfos is available.
    */
   const RouteCountryAndStateInfos& getRouteCountryAndStateInfos() const;
   /**
    * Start range prediction for EVs at waypoint or destination.
    * @param unsigned int indexAlternative index of route alternative
    * @param unsigned int waypointIndex
    */
   void startRangePredictionAtWaypointOrDestination(unsigned int indexAlternative, unsigned int waypointIndex);

   /**
    * Stop range prediction for EVs at waypoint or destination.
    */
   void stopRangePredictionAtWaypointOrDestination();

   /**
    * Method for getting the RangePredictionInfos.
    * Property callback ROUTE_RANGEPREDICTION_INFO_CHANGED indicates that update of RangePredictionInfos is available.
    */
   const RangePredictionInfos& getRangePredictionInfos() const;

   /**
    * Starts automatic detection of charging stations(ePOIs) along route
    * by taking into account charging strategy, driving distance and
    * driving time to reach all the waypoints/destinations.
    * \async ::ROUTE_AUTOMATIC_CHARGING_STATION_DETECTION_INFO_CHANGED
    */
   void startAutomaticChargingStationDetection();

   /**
    * Method returns information about the detection status and insertion status of charging stations
    * which was earlier requested by startAutomaticChargingStationDetection() or
    * insertChargingStationsToRoute()
    *
    * \async ::ROUTE_AUTOMATIC_CHARGING_STATION_DETECTION_INFO_CHANGED
    */
   const AutomaticChargingStationDetectionInfo& getAutomaticChargingStationDetectionInfo() const;

   /**
    * Cancels automatic detection of charging stations to route.
    * \async ::ROUTE_AUTOMATIC_CHARGING_STATION_DETECTION_INFO_CHANGED
    */
   void cancelAutomaticChargingStationDetection();

   /**
    * Inserts automatically all the required charging stations to route to reach all the waypoints/destinations,
    * provided they do not exceed the maximum number of waypoints.
    * If the waypoints exceed the max number, they shall be removed explicitly.
    * case 1: if routeCalculationMode is ROUTE_CALCULATION_MODE__DEFAULT,
    *          Insert the detected charging stations, starts the route calculation, displays the calculated route and
    *          sends all the corresponding property updates mentioned below
    *          \async ::ROUTE_CALCULATION_STARTED
    *          \async ::ROUTE_ARRIVALINFOS_CHANGED
    *          \async ::ROUTE_WAYPOINTINFOS_CHANGED
    *          \async ::ROUTE_AUTOMATIC_CHARGING_STATION_DETECTION_INFO_CHANGED

    * case 2: if routeCalculationMode is ROUTE_CALCULATION_MODE__STEALTH
    *         Insert the detected charging stations, starts the route calculation, does not display the calculated route
    *         and sends the below property updates
    *          \async ::ROUTE_PRE_CALCULATION_WITH_AUTOMATIC_CHARGING_STATIONS_STATUS_CHANGED
    *          \async ::ROUTE_AUTOMATIC_CHARGING_STATION_DETECTION_INFO_CHANGED
    */
   void insertChargingStationsToRoute(RouteCalculationMode routeCalculationMode = ROUTE_CALCULATION_MODE__DEFAULT);

   /**
   * Stops the calculation of the route with inserted charging station waypoints. All automatically added
   * charging station waypoints are removed from the route, and the route calculation is restarted with
   * the original route definition.
   *          \async ::ROUTE_CALCULATION_STARTED
   *          \async ::ROUTE_ARRIVALINFOS_CHANGED
   *          \async ::ROUTE_WAYPOINTINFOS_CHANGED
   */
   void removeChargingStationsFromRoute();

   /**
    * Method for getting the Shape Points Availability Info
    * Property callback ::ROUTE_SHAPE_POINTS_AVAILABILITY_CHANGED indicates the update of ShapePointsAvailabilityInfo.
    */
   const ShapePointsAvailabilityInfo& getShapePointsAvailabilityInfo() const;

   /**
    * HMI shall call this method to get the shape points.
    * The result shall be retrieved by calling getShapePointsInfo upon receipt of
    * property callback ::ROUTE_SHAPE_POINTS_INFO_CHANGED.
    *
    *  \async ::ROUTE_SHAPE_POINTS_INFO_CHANGED
    */
   void requestShapePoints();

   /**
    * Getter for ShapePointsInfo
    * Property callback ::ROUTE_SHAPE_POINTS_INFO_CHANGED indicates that update of ShapePointsInfo.
    */
   const ShapePointsInfo& getShapePointsInfo() const;

   ///@}

   // ================================================================================================================
   ///@{ \name ELECTRIC VEHICLE
   /**
    * set the battery related information
    */
   void setBatteryInfos(const BatteryInfos& batteryInfos);

   /**
    * set charging related information
    */
   void setChargingInfos(const ChargingInfos& chargingInfos);

   /**
    * sets the driving mode of Vehicle
    */
   void setDriveMode(::navmiddleware::DriveMode driveMode);

   /**
    * Returns the current electric vehicle related info
    */
   const ElectricVehicleInfos& getElectricVehicleInfos() const;

   /**
    * Request to store necessary data, as a potential shutdown is indicated
    * Remark: On shutdown, there is a need to save current SOC
    */
   void storeDataBeforeShutDown();

   ///@}

   // ================================================================================================================
   ///@{ \name POI

   ///@}

   // ================================================================================================================
   ///@{ \name TRAFFIC

   /**
    * requests the given range of traffic messages
    */
   void requestTrafficMessageList(unsigned int startIndex, unsigned int numElements, TrafficMessageListType listType, TrafficMessageTextType textType);
   /**
    * requests all the traffic messages in a given distance.
    */
   void requestTrafficMessageListForRange(unsigned int maxDistanceInMeters, TrafficMessageListType listType,
                                          TrafficMessageTextType textType, bool isFreeFlowMessagesRequired, bool isIncidentTrafficMessagesRequired);
   /**
    * requests the details of the traffic messages
    */
   void requestTrafficMessageDetails(unsigned int trafficmessageID);
   /**
    * getter for traffic message infos, should be called after ::TRAFFICMESSAGE_INFO_LIST_CHANGED update
    */
   const TrafficMessageListInfos& getTrafficMessageListInfos() const;
   /**
    * getter for traffic message details infos, should be called after ::TRAFFICMESSAGE_INFO_DETAILS_CHANGED update
    */
   const ValidValue<TrafficMessageDetailsInfo>& getTrafficMessageDetailsInfo() const;
   /**
    * This method is called when user moves out of traffic message details screen context, for that traffic message ID.
    * Stops updates of traffic message details for that particular traffic message.
    */
   void releaseTrafficMessageDetails(unsigned int trafficMessageID);
   /**
    * getter for XUrgent traffic message infos, should be called after ::TRAFFICMESSAGE_XURGENT_CHANGED update
    */
   const TrafficMessageListInfos& getXUrgentTrafficMessageListInfos() const;
   /**
    * Informs about the confirmation of a XUrgent traffic message by user.
    */
   void releaseXUrgentTrafficMessage(unsigned int trafficMessageID);
   /**
    * getter for XUrgent traffic service reception status info, should be called after ::TRAFFICMESSAGE_RECEPTION_STATUS_CHANGED update
    * Return: Used as valid value. If invalid, reception status is not set.
    */
   const ValidValue<TrafficServiceReceptionStatusInfo>& getTrafficServiceReceptionStatusInfo() const;

   /**
    * Getter for count of all traffic messages.
    * The count is available after the system startup and will be updated based on the available traffic messages in the system.
    * No Property Identifier is added to get the count of all traffic messages.
    * Clients can call the function based on their requirements.
    */
   int getCountOfAllTrafficMessages() const;

   /**
    * Requests the traffic icon selection list.
    */
   void requestTrafficIconSelectionList();

   /**
    * Getter for traffic icon selection list info, should be called after ::TRAFFICMESSAGE_ICONFILTER_LIST_CHANGED update.
    */
   const TrafficIconSelectionListInfo& getTrafficIconSelectionListInfo() const;

   /**
    *  Informs PresCtrl about the current visual range of the related list widget.
    *
    *  Does not directly change the visual range, but is needed for PresCtrl to be able to recalculate a new visual range
    *  on traffic message changes from business_ctrl.
    *  Has to be called every time the related list widgets visual range changes.
    *
    *  @param startIndex startIndex of the list widgets visual range
    *  @param numElements size of the list widgets visual range
    */
   void setTrafficMessageListCurrentVisualRange(unsigned int startIndex, unsigned int numElements);
   /**
    * Informs PresCtrl about the index of the selected / focused list element if existing.
    * If the focus is gone, just send an invalid value.
    *
    * Does not directly change the focused element index, but is needed for PresCtrl to be able to recalculate a new visual range
    * on traffic message changes from business_ctrl.
    * Has to be called every time the related list widgets focused element index changes.
    */
   void setTrafficMessageListCurrentFocusedElementIndex(const ValidValue<unsigned int>& index);
   /**
    * Sets user rule for traffic message. E.g. blocks or ignores an existing traffic message by id from current traffic message list.
    *
    *  @param trafficMessageID id of the message to be manipulated
    *  @param userRule kind of manipulation. Use BLOCK for blocking (avoiding) and NONE for unblock
    *
    *  \async ::TRAFFICMESSAGE_DETOUR_RESULT_CHANGED as soon as the operation has been finished or an error occurred.
    */
   void setTrafficMessageUserRule(unsigned int trafficMessageID, TrafficMessageUserRule userRule);
   /**
    * Traffic Settings
    * get the traffic settings object
    */
   settings::TrafficSettings getTrafficSettings() const;
   /**
    * Send the updated Hmi traffic settings back to the middleware.
    */
   void updateTrafficSettings(const settings::TrafficSettings&) const;
   /**
    * To update middleware settings,apply settings method has to be called
    */
   void applyTrafficSettings();

   /**
    * Requests the deletion of the traffic message list,
    * when user moves out of the traffic message list context.
    */
   void releaseAllTrafficMessageLists();

   /**
    * Requests the detour of all traffic messages on current guided route.
    *
    * \async ::TRAFFICMESSAGE_DETOUR_RESULT_CHANGED as soon as the operation has been finished or an error occurred.
    */
   void detourAllTrafficMessagesAlongRoute();

   /**
    * Requests to cancel already defined detour using the method detourAllTrafficMessagesAlongRoute.
    *
    * \async ::TRAFFICMESSAGE_DETOUR_RESULT_CHANGED as soon as the operation has been finished or an error occurred.
    */
   void cancelDetourAllTrafficMessagesAlongRoute();

   /**
    * getter for result of detour/cancelDetour traffic message(s), should be called after ::TRAFFICMESSAGE_DETOUR_RESULT_CHANGED update
    */
   TrafficMessageDetourResult getTrafficMessageDetourResult() const;
   ///@}

   // ================================================================================================================
   ///@{ \name LocationBasedServices

   /**
    * Request for the WeatherData
    * If the requestId is 0, then the contents of the input parameter report configuration is not valid
    * and no property update will be provided in this case.
    */
   RequestId requestWeatherData(const ::std::vector< ReportConfiguration>& reportConfigurations);

   /**
    * Getter for WeatherData, should be called after ::WEATHER_DATA_CHANGED update
    */
   const WeatherDataInfos& getWeatherDataInfos(RequestId requestId) const;

   /**
    * This method will be called when the weather data is no longer required
    */
   void releaseWeatherData(RequestId requestId);

   /**
    * LocationBasedServices Settings
    * get the LocationBasedServices settings object
    */
   settings::LocationBasedServicesSettings getLocationBasedServicesSettings() const;

   /**
    * Send the updated Hmi LocationBasedServicesSettings back to the middleware.
    */
   void updateLocationBasedServicesSettings(const settings::LocationBasedServicesSettings&) const;

   /**
    * Reception state for the weather data.
    */
   ::navmiddleware::WeatherReceptionState getWeatherDataReceptionState() const;

   /**
    * Getter for OnlineServicesDiagnosticData, should be called after ONLINE_SERVICES_DIAGNOSITC_INFO_UPDATED
    */
   const OnlineServicesDiagnosticInfos& getOnlineServicesDiagnosticInfos() const;

   /**
    * Getter for ConnectedServicesAvailability, should be called after COUNTRY_SPECIFIC_CONNECTED_SERVICES_AVAILABILITY_CHANGED
    */
   const ConnectedServicesAvailabilityInfos& getConnectedServicesAvailabilityInfos() const;
   ///@}

   // ================================================================================================================
   ///@{ \name DestinationMemory

   /**
    *  Stores a new entry into the given category.
    *  Pre-requisite to storeEntry() call is that one *MUST* use the setLocationWithXY() interface to
    *  select the location which is going to end-up as part of the entry. Usage is the same as with startGuidance().
    *  @param category Entry will be stored in this category.
    *
    *  \async ::DESTINATIONMEMORY_STORE_ENTRY_STATUS_CHANGED as soon as the requested destination has been stored or an error occurred.
    *  \async ::DESTINATIONMEMORY_ADDRESSBOOK_STATUS_CHANGED
    */
   void storeEntry(DestinationMemoryEntryCategory category);

   /**
    *  Assigns given destination to the given category. Destination can be assigned to multiple categories.
    *  @param destinationId Destination to assign a given category to.
    *  @param category Destination will be assigned to this category.
    *
    *  \async ::DESTINATIONMEMORY_STORE_ENTRY_STATUS_CHANGED as soon as the operation has been finished or an error occurred.
    *  \async ::DESTINATIONMEMORY_ADDRESSBOOK_STATUS_CHANGED
    */
   void storeEntry(DestinationMemoryEntry::Id entryId, DestinationMemoryEntryCategory category);

   /**
    *  Stores a new entry into the given category. DestinationMemoryEntryAttributes can be filled with valid value.
    *  Pre-requisite to storeEntry() call is that one *MUST* use the setLocationWithXY() interface to
    *  select the location which is going to end-up as part of the entry. Usage is the same as with startGuidance().
    *  @param category Entry will be stored in this category.
    *  @param additionalAttributesInfo Attributes to overwrite default ones for the given entry..
    *
    *  \async ::DESTINATIONMEMORY_STORE_ENTRY_STATUS_CHANGED as soon as the requested destination has been stored or an error occurred.
    *  \async ::DESTINATIONMEMORY_ADDRESSBOOK_STATUS_CHANGED
    */
   void storeEntry(DestinationMemoryEntryCategory category, const DestinationMemoryEntryAttributes& destinationAttributesInfo);

   /**
    * Replaces the given entry to new destination location. Pre-requisite to replaceEntryLocation() call is that one *MUST* use
    * the setLocationWithXY() interface to select the location which is going to end-up as part of the entry. Usage is the same as with startGuidance().
    * @param entryId of the old entry to replace.
    *
    * \async ::DESTINATIONMEMORY_STORE_ENTRY_STATUS_CHANGED as soon as the requested entry location is replaced or an error occurred.
    */
   void replaceEntryLocation(const DestinationMemoryEntry::Id entryId);

   /**
    *  Returns the status of the storeEntry function
    *  If the status is STOREENTRYSTATUS__OK, entry ID of the entry is valid
    */
   DestinationMemoryStoreEntryStatus getStoreEntryStatus() const;

   /**
    *  Request the details about the entry.
    *  @param entryId Entry to get details of.
    *
    *  \async ::DESTINATIONMEMORY_ENTRY_DETAILS_AVAILABLE as soon as the requested elements are available.
    */
   void requestEntryDetails(DestinationMemoryEntry::Id entryId);

   /**
    *  Returns the details for entry requested by requestDestinationMemoryEntry().
    */
   const DestinationMemoryEntry& getEntryDetails() const;

   /**
    *  Assigns given list of destination entries to the given category.
    *  @param category Destination will be assigned to this category.
    *  @param entries List of entries to be stored
    *
    *  \async ::DESTINATIONMEMORY_STORE_ENTRIES_STATUS_CHANGED as soon as the operation has been finished or an error occurred.
    *  \async ::DESTINATIONMEMORY_ADDRESSBOOK_STATUS_CHANGED
    */
   void storeEntries(DestinationMemoryEntryCategory category, const std::vector<DestinationMemoryEntry>& entries);

   /**
    *  Returns the status of the storeEntries() method
    *  If the status is STOREENTRYSTATUS__OK, entry ID of the entry is valid
    */
   const DestinationMemoryStoreEntriesStatus& getStoreEntriesStatus() const;

   /**
    *  Request the list of destinations of particular category.
    *  @param category List will contain destinations only of this category.
    *  @param subgroup When set list will contain entries belonging to this sub group.
    *  @param startIndex of the first element to be returned by getEntryList()
    *  @param numElements to be returned by getEntryList(). If set to -1, all
    *  elements available will be returned.
    *  @param sortingAlgorithm  Resulting list will be sorted by the given order.
    *  Supported sorting algorithms are:
    *     SORTINGALGORITHM_UNSORTED
    *     SORTINGALGORITHM_ALPHABETICAL_BY_NAME_ASC
    *     SORTINGALGORITHM_ALPHABETICAL_BY_NAME_DESC
    *     SORTINGALGORITHM_ALPHABETICAL_BY_ALIAS_ASC
    *     SORTINGALGORITHM_ALPHABETICAL_BY_ALIAS_DESC
    *     SORTINGALGORITHM_BY_ENTRY_POSITION_ASC
    *     SORTINGALGORITHM_BY_ENTRY_POSITION_DESC
    *     SORTINGALGORITHM_BY_ICON_AND_ALPHABETICAL_BY_NAME_ASC
    *     SORTINGALGORITHM_BY_ICON_AND_ALPHABETICAL_BY_ALIAS_ASC
    *     SORTINGALGORITHM_BY_GROUP_AND_ALPHABETICAL_BY_NAME_ASC
    *     SORTINGALGORITHM_BY_GROUP_AND_ALPHABETICAL_BY_ALIAS_ASC
    *     SORTINGALGORITHM_BY_CREATION_TIME_NEWEST_FIRST
    *     SORTINGALGORITHM_BY_CREATION_TIME_OLDEST_FIRST
    *  If Sorting algorithm is one of
    *  SORTINGALGORITHM_BY_GROUP_AND_ALPHABETICAL_BY_NAME_ASC
    *  SORTINGALGORITHM_BY_GROUP_AND_ALPHABETICAL_BY_ALIAS_ASC
    *  Entries that belong to given subgroup will be sorted first.
    *  @param freeTextSearchString Entries with name which match the search string will be returned.
    *  If empty, FTS filtering will not be performed (complete list will be returned).
    *  \async ::DESTINATIONMEMORY_HOME_LIST_CHANGED if HOME category items have been requested.
    *  \async ::DESTINATIONMEMORY_WORK_LIST_CHANGED if WORK category items have been requested.
    *  \async ::DESTINATIONMEMORY_ADDR_BOOK_LIST_CHANGED if ADDR_BOOK category items have been requested.
    *  \async ::DESTINATIONMEMORY_ADDRESS_GADGET_LIST_CHANGED if ADDRESS_GADGET category items have been requested.
    *  \async ::DESTINATIONMEMORY_PERSONAL_EPOI_LIST_CHANGED if PERSONAL_EPOI category items have been requested.
    *  \async ::DESTINATIONMEMORY_FAVORITE_WEATHER_ALERTS_LIST_CHANGED if FAVORITE_WEATHER_ALERTS category items have been requested.
    */
   void requestEntryList(DestinationMemoryEntryCategory category,
                         const ValidValue< DestinationMemoryAttributesGroup>& subGroup,
                         uint32_t startIndex,
                         int32_t numElements,
                         DestinationMemoryEntry::SortingAlgorithm sortingAlgorithm,
                         const ::std::string& freeTextSearchString = "");

   /**
    *  Returns the list of destinations as requested by requestEntryList()
    */
   const DestinationMemoryEntryList& getEntryList() const;

   /**
    * Returns the status of destination memory which contain the number of stored entries and routes.
    * To be called when DESTINATIONMEMORY_ADDRESSBOOK_STATUS_CHANGED update is received for the following size-infos:
    *    - getNumberOfAddressBookEntries()
    *    - getNumberOfRoutes()
    *    - getNumberOfHomeEntries()
    *    - getNumberOfWorkEntries()
    *    - getNumberOfAddressGadgetEntries()
    *    - getNumberOfPersonalEPOIEntries()
    * To be called when DESTINATIONMEMORY_LAST_DESTINATION_LIST_STATUS_CHANGED update is received for the following size-infos:
    *    - getNumberOfPreviousDestionationItems(DestinationMemoryItemList::Filter filter)
    */
   const DestinationMemoryStatus& getDestinationMemoryStatus() const;

   /**
    *  Modifies the attributes of an existing entry (AddressBook, Work, Home).
    *  @param entryId Entry to be modified with the given attributes.
    *  @param additionalAttributesInfo Attributes to overwrite existing ones for the given entry.
    *
    *  \async ::DESTINATIONMEMORY_HOME_LIST_CHANGED if HOME category item(s) have been requested to be modified.
    *  \async ::DESTINATIONMEMORY_WORK_LIST_CHANGED if WORK category item(s) have been requested to be modified.
    *  \async ::DESTINATIONMEMORY_ADDR_BOOK_LIST_CHANGED if ADDR_BOOK category item(s) have been requested to be modified.
    *  \async ::DESTINATIONMEMORY_ADDRESS_GADGET_LIST_CHANGED if ADDRESS_GADGET item(s) have been requested to be modified.
    *  \async ::DESTINATIONMEMORY_PERSONAL_EPOI_LIST_CHANGED if PERSONAL_EPOI item(s) have been requested to be modified.
    *  \async ::DESTINATIONMEMORY_ADDRESSBOOK_STATUS_CHANGED if ADDR_BOOK category item(s) have been requested to be modified.
    *  \async ::DESTINATIONMEMORY_FAVORITE_WEATHER_ALERTS_LIST_CHANGED if FAVORITE_WEATHER_ALERTS category item(s) have been requested to be modified.
    */
   void modifyEntryAttributes(DestinationMemoryEntry::Id entryId, const DestinationMemoryEntryAttributes& destinationAttributesInfo);

   /*  Sets an entry to its new position.
    *  @param entryId Entry whose position is to be set.
    *
    *  \async ::DESTINATIONMEMORY_ENTRY_POSITION_CHANGED if position has been set successfully.
    */
   void setEntryPosition(DestinationMemoryEntry::Id entryId, uint64_t position);

   /*  Returns a path to the media file which corresponds to the given sound ID.
    *  @param sound soundID for which sound file path has to be fetched.
    */
   ValidValue< ::std::string> getAddressbookMediaPath(DestinationMemoryEntryAttributes::Sound sound) const;

   /**
    *  Removes an entry.
    *  @param entryId List of entries to be removed.
    *
    *  \async ::DESTINATIONMEMORY_HOME_ENTRY_REMOVED if HOME category item(s) have been requested to be removed.
    *  \async ::DESTINATIONMEMORY_WORK_ENTRY_REMOVED if WORK category item(s) have been requested to be removed.
    *  \async ::DESTINATIONMEMORY_ADDR_BOOK_ENTRY_REMOVED if ADDR_BOOK category item(s) have been requested to be removed.
    *  \async ::DESTINATIONMEMORY_ADDRESSBOOK_STATUS_CHANGED on successful removal regardless of category.
    */
   void removeEntry(const ::std::vector< DestinationMemoryEntry::Id >& entryId);

   /**
    *  Removes the destination by using the corresponding POI ID.
    *  @param poiId PoiId identifying the destination to be removed.
    *
    *  \async ::DESTINATIONMEMORY_HOME_ENTRY_REMOVED if HOME category item have been requested to be removed.
    *  \async ::DESTINATIONMEMORY_WORK_ENTRY_REMOVED if WORK category item have been requested to be removed.
    *  \async ::DESTINATIONMEMORY_ADDR_BOOK_ENTRY_REMOVED if ADDR_BOOK category item have been requested to be removed.
    *  \async ::DESTINATIONMEMORY_ADDRESSBOOK_STATUS_CHANGED on successful removal regardless of category.
    */
   void removeEntryByPoiId(const PoiId& poiId);

   /**
    *  Removes all the destinations from given category.
    *  @param category Category to be removed from.
    *
    *  \async ::DESTINATIONMEMORY_HOME_ENTRY_REMOVED if HOME category item(s) have been requested to be removed.
    *  \async ::DESTINATIONMEMORY_WORK_ENTRY_REMOVED if WORK category item(s) have been requested to be removed.
    *  \async ::DESTINATIONMEMORY_ADDR_BOOK_ENTRY_REMOVED if ADDR_BOOK category item(s) have been requested to be removed.
    *  \async ::DESTINATIONMEMORY_ADDRESSBOOK_STATUS_CHANGED on successful removal regardless of category.
    */
   void removeAllEntries(DestinationMemoryEntryCategory category);

   /**
    *  Returns the list of groups used in the address book.
    *  - The property will be updated on system start and whenever the list of groups has changed
    *       by to adding, removing or modifying address book entries.
    *  - The property will be notified on registration of the callback.
    *  - DESTINATION_MEMORY_ATTRIBUTES_GROUP__NONE may be included in the list of used
    *       groups if no group is assigned to any address book item.
    *  - If no address book items exist, the list of used groups will be empty
    *  To be called when ::DESTINATIONMEMORY_ADDRESSBOOK_USED_GROUPS_CHANGED has been received
    */
   const DestinationMemoryAttributesGroupList& getAddressBookUsedGroups() const;

   /**
    *  Request the list of last (guided) destinations and routes.
    *  @param startIndex of the first element to be returned by requestLastDestinationList()
    *  @param numElements to be returned by getLastDestinationList(). If set to -1, all
    *  elements available will be returned.
    *  @param sortingAlgorithm  Resulting list will be sorted by the given order.
    *  Supported sorting algorithms are:
    *     SORTINGALGORITHM_UNSORTED
    *     SORTINGALGORITHM_ALPHABETICAL_BY_NAME_ASC
    *     SORTINGALGORITHM_ALPHABETICAL_BY_NAME_DESC
    *     SORTINGALGORITHM_BY_START_GUIDANCE_TIME_NEWEST_FIRST
    *     SORTINGALGORITHM_BY_START_GUIDANCE_TIME_OLDEST_FIRST
    *  @param requestFilter Only items which satisfy filter criteria will be returned.
    *  @param freeTextSearchString Entries/Routes with name which match the search string will be returned.
    *  If empty, FTS filtering will not be performed (complete list will be returned).
    *
    *  \async ::DESTINATIONMEMORY_LAST_DESTINATION_LIST_CHANGED as soon as the requested elements are available.
    */
   void requestLastDestinationList(uint32_t startIndex,
                                   int32_t numElements,
                                   DestinationMemoryEntry::SortingAlgorithm sortingAlgorithm,
                                   DestinationMemoryItemList::Filter requestFilter,
                                   const ::std::string& freeTextSearchString = "");

   /**
    *  Returns the list of last (guided) destinations and routes as requested by requestLastDestinationList()
    */
   const DestinationMemoryItemList& getLastDestinationList() const;

   /**
    * Updates the previous Destination History data
    * @param lastDestinationHistoryList List of History data of destinations to be updated
    *
    * \async DESTINATIONMEMORY_UPDATE_LAST_DESTINATIONS_STATUS_CHANGED as soon as last destination histories has been updated
    */
   void updateLastDestinationHistoryList(const ::std::vector< DestinationMemoryLastDestinationHistory >& lastDestinationHistoryList);

   /**
    * Returns the list of last destinations which was updated by updateLastDestinationHistoryList()
    */
   const DestinationMemoryUpdatedLastDestinationList& getUpdatedLastDestinationList() const;

   /**
    *  Inserts last guided destination list
    *  @param lastDestinationList List of all the last destionation to be inserted
    *  \async ::DESTINATIONMEMORY_INSERT_LAST_DESTINATIONS_STATUS_CHANGED as soon as list has been inserted
    */
   void insertLastDestinationList(const ::std::vector< DestinationMemoryLastDestination >& lastDestinationList);

   /**
    * Returns the list of last destinations which was updated by insertLastDestinationList()
    */
   const DestinationMemoryUpdatedLastDestinationList& getInsertedLastDestinationList() const;

   /**
    *  Returns the number of reached waypoints for the last guided tour.
    *  Should be called after last guided item list requested with requestLastDestinationList function is available.
    *  Result is valid only if there is guided items in Destination memory and the tour was the last guided one.
    */
   uint32_t getLastGuidedRouteNumberOfWaypointsReached() const;

   /**
    *  Removes item(s) from the last (guided) destination/routes list.
    *
    *  \async ::DESTINATIONMEMORY_LAST_DESTINATION_LIST_ITEM_REMOVED as soon as item(s) has been removed.
    *  \async ::DESTINATIONMEMORY_LAST_DESTINATION_LIST_STATUS_CHANGED as soon as item(s) has been removed.
    */
   void removeLastDestinationListItems(const ::std::vector< DestinationMemoryItem::IdElement >& itemIdList);

   /**
    *  Removes all items from the last (guided) destination list with which satisfy filter criteria.
    *  @param requestFilter Only items which satisfy filter criteria will be removed.
    *  \async ::DESTINATIONMEMORY_LAST_DESTINATION_LIST_ITEM_REMOVED as soon as items have been removed.
    *  \async ::DESTINATIONMEMORY_LAST_DESTINATION_LIST_STATUS_CHANGED as soon as item(s) has been removed.
    */
   void removeAllLastDestinationListItems(DestinationMemoryItemList::Filter requestFilter);

   /**
    *  Request the details of the last departure.
    *  If no departure is available the details will be empty name, latitude/longitude set to 0
    *
    *  \async ::DESTINATIONMEMORY_LAST_DEPARTURE_CHANGED.
    */
   void requestLastDeparture();

   /**
    *  Returns the departureInfos as stored in the DestinationMemoryInfo
    */
   const DepartureInfo& getLastDeparture() const;

   /**
    *  Removes the details of the last departure.
    *
    *  \async ::DESTINATIONMEMORY_LAST_DEPARTURE_REMOVED.
    */
   void removeLastDeparture();

   /**
    * Stores the current route (at least one waypoint) and assigns the route to the given category.
    * @param name      The predefined name for a newly added tour. If set to "", name will be auto-generated by the PresCtrl.
    *
    * \async ::DESTINATIONMEMORY_STORE_ROUTE_STATUS_CHANGED as soon as the requested route has been stored.
    * \async ::DESTINATIONMEMORY_ADDRESSBOOK_STATUS_CHANGED
    */
   void storeRoute(::std::string name);

   /**
    *  Stores the existing route (specified by routeId) to the given category.
    *  @param routeId Route to assign a given category to.
    *  @param category Route will be assigned to this category.
    *
    *  \async ::DESTINATIONMEMORY_STORE_ROUTE_STATUS_CHANGED  as soon as the requested route has been assigned to the category.
    *  \async ::DESTINATIONMEMORY_ADDRESSBOOK_STATUS_CHANGED
    */
   void storeRoute(DestinationMemoryRoute::Id routeId, DestinationMemoryRoute::Category category);

   /**
    * Stores the Tour under the given category.
    * Validity of Tour will not be checked and tranformation to be done when set and created as route.
    * @param name The predefined name for a newly added tour. If set to "", name will be auto-generated by the PresCtrl.
    * @param tour A tour which consist of several (at least one) waypoint.
    * @param category Tour will be assigned to this category.
    *
    * \async ::DESTINATIONMEMORY_STORE_ROUTE_STATUS_CHANGED as soon as the requested route has been stored.
    * \async ::DESTINATIONMEMORY_ADDRESSBOOK_STATUS_CHANGED
    */
   void storeRoute(const ::std::string& name, const Tour& tour, DestinationMemoryRoute::Category category);

   /**
    *  Returns the status of the storeRoute function
    *  If the status is STOREROUTESTATUS__OK, Tour ID is valid
    *  If the status is STOREROUTESTATUS__ERROR, Tour ID is invalid
    */
   const DestinationMemoryStoreRouteStatus& getStoreRouteStatus() const;

   /**
    *  Request the details about the route.
    *  @param routeId Route to get details of.
    *
    *  \async ::DESTINATIONMEMORY_ROUTE_DETAILS_AVAILABLE as soon as the requested elements are available.
    */
   void requestStoredRouteDetails(DestinationMemoryRoute::Id routeId);

   /**
    *  Returns the details for destination requested by requestStoredRouteDetails().
    */
   const DestinationMemoryRoute& getStoredRouteInfo() const;

   /**
    *  Request the list of stored routes.
    *  @param startIndex  of the first element to be returned by requestStoredRoutes()
    *  @param numElements  to be returned by requestStoredRoutes(). If set to -1, all
    *  elements available will be returned.
    *  @param sortingAlgorithm  Resulting list will be sorted by the given order.
    *  Supported sorting algorithms are:
    *     SORTINGALGORITHM_UNSORTED
    *     SORTINGALGORITHM_ALPHABETICAL_BY_NAME_ASC
    *     SORTINGALGORITHM_ALPHABETICAL_BY_NAME_DESC
    *     SORTINGALGORITHM_BY_CREATION_TIME_NEWEST_FIRST
    *     SORTINGALGORITHM_BY_CREATION_TIME_OLDEST_FIRST
    *  @param freeTextSearchString Routes with name which match the search string will be returned.
    *  If empty, FTS filtering will not be performed (complete list will be returned).
    *
    *  \async ::DESTINATIONMEMORY_STORED_ROUTES_LIST_CHANGED as soon as the requested elements are available.
    */
   void requestStoredRoutes(uint32_t startIndex,
                            int32_t numElements,
                            DestinationMemoryEntry::SortingAlgorithm sortingAlgorithm,
                            const ::std::string& freeTextSearchString = "");

   /**
    *  Returns the list of stored routes as requested by requestStoredRoutes()
    */
   const DestinationMemoryRouteList& getStoredRoutes() const;

   /**
    *  Removes the list of routes.
    *  @param routeIdList list of the routes to be removed.
    *
    *  \async ::DESTINATIONMEMORY_ROUTE_REMOVED if routes are successfully removed.
    *  \async ::DESTINATIONMEMORY_ADDRESSBOOK_STATUS_CHANGED
    */
   void removeRoute(const ::std::vector< DestinationMemoryRoute::Id >& routeIdList);

   /**
    *  Removes all the routes.
    *
    *  \async ::DESTINATIONMEMORY_ROUTE_REMOVED if all routes have been successfully removed.
    *  \async ::DESTINATIONMEMORY_ADDRESSBOOK_STATUS_CHANGED
    */
   void removeAllRoutes();

   /**
    * Replaces the selected stored route with the current route. The new route name will be auto-generated.
    * @param routeId Route that is to be replaced.
    *
    * \async ::DESTINATIONMEMORY_STORE_ROUTE_STATUS_CHANGED as soon as the requested route has been replaced with the current one.
    */
   void replaceStoredRoute(DestinationMemoryRoute::Id routeId);

   /**
    * Renames the selected stored route.
    * @param routeId Route that is to be replaced.
    * @param name    The new name for a route.
    *
    * \async ::DESTINATIONMEMORY_ROUTE_MODIFIED as soon as the requested route has been renamed.
    */
   void renameStoredRoute(DestinationMemoryRoute::Id routeId, const ::std::string& name);

   /**
    * Get the electric vehicle charging station auto save settings object which is hosted by middleware
    */
   settings::EvAutoSaveChargingStationSettings getEvAutoSaveChargingStationSettings() const;

   /**
    * Send updated electric vehicle charging station auto save settings back to the middleware.
    * After calling this method, changes on the local object will not affect
    * the passed reference (middleware creates a copy). To update middleware settings,
    * call this method again.
    */
   void updateEvAutoSaveChargingStationSettings(const settings::EvAutoSaveChargingStationSettings&);

   /**
    *  request a list of last destination entry Ids along with their 'create' and 'last guided' timestamps
    *  @param startIndex of the first element to be returned by requestLastDestinationsHistory()
    *  @param numElements to be returned by getLastDestinationsHistory(). If set to -1, all
    *  elements available will be returned.
    *  @param sortingAlgorithm  Resulting list will be sorted by the given order.
    *  Supported sorting algorithms are:
    *     SORTINGALGORITHM_UNSORTED
    *     SORTINGALGORITHM_ALPHABETICAL_BY_NAME_ASC
    *     SORTINGALGORITHM_ALPHABETICAL_BY_NAME_DESC
    *     SORTINGALGORITHM_BY_START_GUIDANCE_TIME_NEWEST_FIRST
    *     SORTINGALGORITHM_BY_START_GUIDANCE_TIME_OLDEST_FIRST
    *  @param requestFilter Only items which satisfy filter criteria will be returned.
    *
    *  \async ::DESTINATIONMEMORY_LAST_DESTINATIONS_HISTORY_AVAILABLE as soon as the requested elements are avail
    */
   void requestLastDestinationsHistory(uint32_t startIndex,
                                       int32_t numElements,
                                       ::navmiddleware::DestinationMemoryEntry::SortingAlgorithm sortingAlgorithm,
                                       ::navmiddleware::DestinationMemoryItemList::Filter requestFilter);
   /**
    * Returns the list of last destinations entry Ids along with create' and 'last guided' timestamps
    */
   const DestinationMemoryLastDestinationHistoryList& getLastDestinationsHistory() const;

   /**
    * If Google places have to be included/Excluded in Destination memory lists, HMI has to activate the option accordingly
    * Note: Option is not persistent but remembered throughout the power cycle
    * Note: There is no getter of this option, It is expected that HMI shall not depend on the previous state to use this request
    */
   void setIncludeGooglePlacesInListRequests(bool includeGooglePlaces);

   /**
    * Refresh an expired Google destination memory entry request
    * This method needs to be called before selecting an expired entry in any destination memory entry list,
    * if the entry status is ENTRY_REFRESH_STATUS__EXPIRED.
    * @param entryId entry id for the entry to request for refresh
    * \async ::DESTINATIONMEMORY_SELECTION_STATUS_CHANGED
    */

   void requestRefreshForExpiredDestinationEntry(DestinationMemoryEntry::Id entryId);

   /**
    * Request to transform an expired Google destination memory entry into an On-board Matching entry
    * This method needs to be called before selecting an expired entry in any destination memory entry list,
    * if the entry status is ENTRY_REFRESH_STATUS__ONBOARD_MATHCING_NEEDED.
    * @param entryId entry id for the destination to request for refresh
    * \async ::DESTINATIONMEMORY_SELECTION_STATUS_CHANGED
    */

   void requestOnboardMatchingForExpiredDestinationEntry(DestinationMemoryEntry::Id entryId);

   /**
    * Returns the selection status for the destination entry.
    * If Status is set to UNKNOWN, the entry selection status is Unknown.
    * If Status is set to VALID, the entry can be selected.
    * If Status is EXPIRED, the entry needs to be refreshed before selection.
    * If Status is set to ONBOARD_MATHCING_NEEDED, the entry failed to refresh and needs to be matched to an on-board entry.
    * If Status is set to ONBOARD_MATCHING_FAILED, the entry could not be matched to an on-board entry and is deleted.
    * Should be called after receiving the property update LOCATION_ENTRY_SELECTION_STATUS_CHANGED.
    */
   EntrySelectionStatus getDestinationSelectionStatus() const;

   /**
    * Clears all voice tags from destination memory.
    * \async ::DESTINATIONMEMORY_ADDRESSBOOK_STATUS_CHANGED
    */
   void clearVoiceTags();
   ///@}

   // ================================================================================================================
   ///@{ \name NavDataUpdate


   /**
    * To abort the initialization
   * \async ::NAVDATAUPDATE_UPDATE_STATUS_CHANGED property update and status as DATAUPDATESTATUS__AVAILABLE_UPDATES_ABORTED_BY_USER.
    */
   void requestStopAvailableUpdates();

   /**
    * \@DEPRECATED This will be removed soon.
    *             New Interface requestAvailableUpdates(uint32_t deviceId, const ::std::string& downloadURL,
    *             AvailableUpdatesRequestType requestType) should be used instead.
    * Request a list of all available updates to be downloaded and installed
    * @param deviceId - This is generic device Identifier.
    * @param requestType - Set to INITIAL_REQUEST if it is the first request after entering the update screen,
    *                      Set to MODIFY_SELECTION_REQUEST to update the data using setSelection
    *                      Set to CANCEL_SELECTION_REQUEST for canceling the updated data using setSelection
    *
    * \async ::NAVDATAUPDATE_UPDATE_INFO_CHANGED as soon as the UpdateOverviewInfo data available.
    */
   void requestAvailableUpdates(uint32_t deviceId, AvailableUpdatesRequestType requestType);

   /**
    * Request a list of all available updates to be downloaded and installed
    * @param deviceId - This is generic device Identifier.
    * @param path - This is path/url where updates/catalogues can be stored.
    * @param requestType - Set to INITIAL_REQUEST if it is the first request after entering the update screen,
    *                      Set to MODIFY_SELECTION_REQUEST to update the data using setSelection
    *                      Set to CANCEL_SELECTION_REQUEST for canceling the updated data using setSelection
    *
    * \async ::NAVDATAUPDATE_UPDATE_INFO_CHANGED as soon as the UpdateOverviewInfo data available.
    */
   void requestAvailableUpdates(uint32_t deviceId, const ::std::string& path, AvailableUpdatesRequestType requestType);

   /**
    * Returns the last update overview Info selected in Store Area
    *
    */
   const NavDataUpdateOverviewInfo& getAvailableUpdateInfo() const;

   /**
    * decline the provided automatic update and inform about the reason for the decline
    * @param the reason to decline the update (Button NO is pushed/ Button LATER is pushed)
    */
   void declineAvailableUpdates(UpdateDeclineReason updateDeclineReason);

   /**
    * Request a list of all higher versions available updates which can be downloaded and installed
    * @param startIndex of the first element to be returned by getAvailableUpdatesList()
    * @param numElements to be returned by getAvailableUpdatesList().
    *        If set to -1: actual end of the list is considered
    *
    * \async ::NAVDATAUPDATE_UPDATE_LIST_UPDATE as soon as the requested elements are available.
    */
   void requestAvailableUpdatesList(uint32_t startIndex, int32_t nbrOfElements);

   /**
    * Returns the list of available updates as requested by requestAvailableUpdates()
    */
   const NavDataUpdateInfos& getAvailableUpdatesList() const;

   /**
    * \@DEPRECATED This will be removed soon.
    *             New Interface requestUpdateBySelection(const ::std::string& path) should be used instead.
    * To start download and installation.
    * \async ::NAVDATAUPDATE_UPDATE_STATUS_CHANGED for each stages at last you get DATAUPDATESTATUS__PROCESS_COMPLETE.
    */
   void requestUpdateBySelection();

   /**
    *To start download and installation.
    * @param path - This is path/url where updates/catalogues can be stored.
    *               This path is required only for USB partial updates and not relevant for USB full updates.
    * \async ::NAVDATAUPDATE_UPDATE_STATUS_CHANGED for each stages at last you get DATAUPDATESTATUS__PROCESS_COMPLETE.
    */
   void requestUpdateBySelection(const ::std::string& path);

   /**
    * To abort the download and install
    * \async ::NAVDATAUPDATE_UPDATE_STATUS_CHANGED property update and status as DATAUPDATESTATUS__INSTALLATION_ABORTED_BY_USER
    */
   void requestStopUpdate();

   /**
    * By default, selection is true for each updateInfoId
    * @param updateInfoId: updateInfoId received in NavDataUpdateInfos
    * @param select : false makes them not to download and install
    *
    */
   void setUpdateSelection(uint32_t updateInfoId, bool select);

   /** In case the selection state of all updates has to be changed at once,
    * this method can be used. It sets the update selection of 'all' entries,
    * not of the currently shown entries.
    * @param selected : false makes them not to download and install
    */
   void setSelectionOfAllUpdates(bool selected);

   /**
    * Prepare the USB device to dump the meta data
    * \async : UPDATE_PREPARATION_STATUS_CHANGED
    * @param deviceId :Device Id of the USB to be passed where meta data will be downloaded
    */
   void requestPrepareUSBDevice(uint32_t deviceId);

   /**
    * Latest result of prepareUSBDevice is returned
    */
   NavDataUpdateUSBDeviceStatus getPreparedUSBUpdateDeviceResult();

   /**
    * Returns the last update status triggered by the requestDownloadUpdate() or requestInstallUpdate().
    */
   const NavDataUpdateStatus& getUpdateStatus() const;

   /**
    * Returns the last device status as received from core navi.
    */
   const NavDataDeviceInfos& getNavDataDeviceInfos() const;

   /**
    * Request a compatibility check of the current navigation software and the navigation dataset (NDS)
    * @param navigationVersionString version string of the navigation service
    * @param compatibilityInfoData a detailed description of compatible SW revisions and dataset formats (raw format)
    *
    *
    * \async ::NAVDATAUPATE_COMPATIBILITY_CHANGED as soon as the result of the compatibility check is available
    */
   void checkNavDataCompatibility(const ::std::string& navigationVersionString, const ::std::vector<uint8_t>& compatibilityInfoData);

   /**
    * Returns the compatibility use getCompatibility to get one of ::NavDataCompatibility::CompatibilityState (COMPATIBLE/INCOMPATIBLE/UNKNOWN)
    */
   const NavDataCompatibilityInfo& getNavDataCompatibilityInfo() const;

   /**
    * Requests details about the currently used navigation data (aka map data) like current version.
    *
    * \async ::NAVDATAUPDATE_VERSION_INFO_CHANGED
    */
   void requestNavigationDataVersionInfo();

   /**
    * Returns current navigation data version information including the current version of the navigation data (map).
    * Do not mix up with the map engine version responsible for graphical display of the navigation data.
    */
   const NavDataVersionInfo& getNavigationDataVersionInfo() const;

   /**
    * Returns current update selection information including selected and total available count.
    */
   const NavDataUpdateSelection& getUpdateSelectionStatus() const;

   /**
    * Request to return back the OEM part number of the current navigation data (aka map data) in use.
    * * \async ::NAVDATAUPDATE_PART_NUMBER_INFO_CHANGED
    */
   void requestNavDataPartNumber();

   /**
    * Returns a OEM part number (aka map reference number). This number identifies the map data in use by the device.
    * Information can be used to differentiate for license fee.
    * NavDataPartNumber contains both result code and in case of success the part number (ascii alphanumeric) itself.
    * PropertyUpdate: NAVDATAUPDATE_PART_NUMBER_INFO_CHANGED
    */
   const NavDataPartNumber& getNavDataPartNumber() const;

   /**
    * Request to return back the stored navigation data update history.
    * * \async ::NAVDATAUPDATE_UPDATE_HISTORY_CHANGED
    */
   void requestNavDataUpdateHistory();

   /** Returns the navigation data update installation history.
    * PropertyUpdate: NAVDATAUPDATE_UPDATE_HISTORY_CHANGED
    */
   const NavDataUpdateHistory& getNavDataUpdateHistory() const;

   // ========================================== External direct NDS database user  ==================================

   /**
    * Returns all known and required informations about the navigation dataset in use (if successfully installed).
    * NdsDirectUser requires the information to open the databases in the right mode and to enable themself to register
    * and de-register as a direct user.
    * PropertyUpdate: NAVDATAUPDATE_DATASET_INFO_CHANGED
    */
   const NavDatasetInfo& getNavDatasetInfo() const;

   /** Registers the user as a NdsDirectDatasetUser for the given dataset. User is allowed to open a sqlite connection
    * directly on the dataset databases.
    * @param ndsDatasetUser a enumeration representing the current user
    * @param datasetId the datasetId as received in getNavDatasetInfo
    * \async : NAVDATAUPDATE_DATASET_USER_REG_CHANGED
    */
   void registerDatasetUser(NdsDatasetUser ndsDatasetUser, uint32_t datasetId);

   /** Deregisters the user as a NdsDirectDatasetUser for the given dataset. User is not longer allowed to use a sqlite connection
    * directly on the dataset databases.
    * @param ndsDatasetUser a enumeration representing the current user
    * @param datasetId the datasetId as received in getNavDatasetInfo
    * \async : NAVDATAUPDATE_DATASET_USER_REG_CHANGED
    */
   void deregisterDatasetUser(NdsDatasetUser ndsDatasetUser, uint32_t datasetId);

   /**
    * Returns the result of a registration request as response on registerDatasetUser and deregisterDatasetUser
    * PropertyUpdate: NAVDATAUPDATE_DATASET_USER_REG_CHANGED
    */
   const RegisterDatasetUserResult& getDatasetUserRegisterStatus() const;

   /** Informs about the current access the user opened a sqlite connection (either in shared or in not shared mode)
    * @param ndsDatasetUser a enumeration representing the current user
    * @param datasetId the datasetId as received in getNavDatasetInfo
    * @param ndsDataAccessType the tyoe of database connection (shared/not shared)
    * \async : no asynchronous response
    */
   void datasetUserAccess(NdsDatasetUser ndsDatasetUser, uint32_t datasetId, NdsDataAccessType ndsDataAccessType);

   // ========================================== NavDataUpdate via FOTA (ST2.1 only) =================================

   /** Is called whenever the registration of a scomo component was successfull or failed to indicate the state to
    * navigation middleware
    */
   void setScomoComponentRegistration(ScomoComponent scomoComponent, ScomoComponentRegistrationStatus registrationStatus);

   /**
    * Returns the current state of the navigation data itself. It can be unknown, ok or installation in progress.
    * PropertyUpdate: NAVDATAUPDATE_NAVDATA_STATUS_CHANGED
    */
   NavDataStatus getNavDataStatus() const;

   /**
    * Returns the status of the catalogue.
    * PropertyUpdate: NAVDATAUPDATE_CATALOGUE_STATUS_CHANGED
    */
   CatalogueStatus getCatalogueStatus() const;

   /**
    * To start installation of an update for SOTA source update, for given URL.
    * @param installStage indicating different phases of the same installation request. It's up to the MW to decide what to execute
    * @param component indicates the component the given update relates to.
    * @param url pointing to the location where to find the update package.
    * @param componentName full string as received from scomoCmp property InitState update.
    * \async NAVDATAUPDATE_COMPONENT_INSTALL_RESULT_UPDATED for each stages .
    */
   void requestScomoInstallComponent(NavDataUpdateInstallStage installStage, ScomoComponent scomoComponent, const ::std::string& updateURL, const ::std::string& componentName = "");

   /**
    * To abort an ongoing installation of an update for SOTA source.
    * @param scomoComponent indicates the component the ongoing installation relates to.
    * \async NAVDATAUPDATE_COMPONENT_INSTALL_RESULT_UPDATED and the result as SCOMO_INSTALL_RESULT__ABORTED_BY_REQUEST .
    */
   void requestAbortScomoInstallComponent(ScomoComponent scomoComponent);

   /**
    * Returns the current version status of one, multiple or all update components. On startup usually all component
    * versions are required.
    *
    * This interface shall be used to fetch the inventory details after receiving the property update NAVDATAUPDATE_INVENTORY_INFO_CHANGED
    * @param list of component names (aka componentId); in case all components are required the vector size is zero
    *
    * \async ::NAVDATAUPDATE_COMPONENT_STATUS_CHANGED
    * PropertyUpdate: NAVDATAUPDATE_INVENTORY_INFO_CHANGED
    */
   void requestScomoComponentStatus(ScomoComponent scomoComponent);

   /**
    * Used by HMI to inform MW about the Fota Inventory Update status
    *
    * NAVDATAUPDATE_INVENTORY_INFO_CHANGED will be triggered when the inventory details has been changed and has to be updated to FOTA
    * Additionally, this property update shall be triggered on start-up,
    * when the inventory information is not properly posted to FOTA in the previous power cycle, due to some error situations.
    *
    * After property update trigger, FOTA could request the inventory with interface requestScomoComponentStatus(ScomoComponent scomoComponent)
    *
    * @param isFotaInventoryUpdated: true, if the Inventory information is Updated successfully to FOTA
    *                                false, if the Fota has failed to update the Inventory details
    *
    * PropertyUpdate: NAVDATAUPDATE_INVENTORY_INFO_CHANGED
    */
   void setFotaInventoryUpdateStatus(bool isFotaInventoryUpdated);

   /**
    * Activates notification of changes on the NavDataContext.
    * Depending on the project different products of the NavData will be part of the context.
    * Product selection happens in MW.
    *
    * \async ::NAVDATAUPDATE_CONTEXT_CHANGED
    */
   void registerOnNavDataContextChanges();

   /**
    * Deinstallation of an updateRegion using locally stored packages is NOT SUPPORTED.
    * Deinstall of an updateRegion is nothing but install with version 0. requestScomoInstallComponent() can be used for the same.
    *
    * Removes a certain update region by calling deinstallation the region from navigation data.
    * Same parameters as published in NavDataContext are expected to be set by calling client.
    * @param productId as published in NavDataContext
    * @param supplierId as published in NavDataContext
    * @param updateRegionIds a list of all update regions to be deinstalled
    *
    * \async ::NAVDATAUPDATE_COMPONENT_DEINSTALL_RESULT_CHANGED
    */
   void requestScomoDeinstallComponent(uint32_t productId, uint32_t supplierId, const ::std::vector< uint32_t >& updateRegionIds);

   /**
    * Returns a list of component/version pairs to be used to register the inventory if the map update installer
    * PropertyUpdate: NAVDATAUPDATE_COMPONENT_STATUS_CHANGED
    */
   const ScomoComponentStatus& getScomoComponentStatus() const;

   /**
    * Returns the result of the last requested scomo component de-installation.
    * PropertyUpdate: NAVDATAUPDATE_COMPONENT_DEINSTALL_RESULT_CHANGED
    */
   const DeinstallScomoComponentResult& getScomoDeinstallComponentResult() const;

   /**
    * Returns the result of the last requested scomo component installation.
    * PropertyUpdate: NAVDATAUPDATE_COMPONENT_INSTALL_RESULT_UPDATED
    */
   const ScomoInstallComponentResult& getScomoInstallComponentResult() const;

   /**
    * Returns the current navdata context which represents a list of all installed products and its regions
    * for a human readable presentation and it also contains details required for further communication with external
    * device by using the unique identifiers of each product/region as provided by the nds data carrier.
    * This is indeed required for SOTA update only.
    * PropertyUpdate: NAVDATAUPDATE_CONTEXT_CHANGED
    */
   const NavDataContext& getNavDataContext() const;

   /**
    * Returns a list of all update regions with its current update status.
    * Available updates have to be requested by calling requestAvailableUpdates as described above.
    * PropertyUpdate: NAVDATAUPDATE_AVAILABLE_SCOMO_UPDATE_INFO_CHANGED
    */
   const ScomoAvailableUpdates& getAvailableScomoUpdates() const;

   /**
    * Set the favorite regions of the Navigation data update
    * @param favoriteRegionSetting favorite regions details
    */
   void updateNavDataUpdateFavoriteRegionSettings(const ::navmiddleware::settings::NavDataUpdateFavoriteRegionSettings& favoriteRegionSetting);

   /**
    * Returns the currently set favoriteRegions details(Ids)
    * PropertyUpdate: NAVDATAUPDATE_FAVORITE_REGION_SETTINGS_CHANGED
    */
   ::navmiddleware::settings::NavDataUpdateFavoriteRegionSettings getNavDataUpdateFavoriteRegionSettings() const;

   /**
    * Returns the result of the  Update NavDataUpdate Favorite Region Settings request
    * PropertyUpdate: NAVDATAUPDATE_UPDATE_FAVORITE_REGION_SETTINGS_RESULT_CHANGED
    */
   UpdateFavoriteUpdateRegionsResult getUpdateNavDataUpdateFavoriteRegionSettingsResult() const;

   /**
    * sets the automatic map update status(Enabled/Disabled).
    * @param automaticMapUpdate automatic map update status.
    * \async ::NAVDATAUPDATE_AUTOMATIC_UPDATE_STATUS_CHANGED
    */
   void setAutomaticMapUpdateStatus(::navmiddleware::settings::AutomaticMapUpdateStatus automaticMapUpdateStatus);

   /**
    * Returns the currently set automatic map update status
    * PropertyUpdate: NAVDATAUPDATE_AUTOMATIC_MAP_UPDATE_STATUS_CHANGED
    */
   ::navmiddleware::settings::AutomaticMapUpdateStatus getAutomaticMapUpdateStatus() const;

   /**
    * Returns the result of the setAutomaticMapUpdateStatus request
    * PropertyUpdate: NAVDATAUPDATE_AUTOMATIC_MAP_UPDATE_STATUS_RESULT_CHANGED
    */
   ::navmiddleware::AutomaticMapUpdateStatusResult getAutomaticMapUpdateStatusResult() const;

   /**
    * Returns the status of automatic full update status in case of a yellow stick update (IVI + Map + IVC)
    * Should be called after PropertyUpdate: NAVDATAUPDATE_AUTOMATIC_FULL_UPDATE_STATUS_CHANGED
    */
   ::navmiddleware::AutomaticFullUpdateStatus getAutomaticFullUpdateStatus() const;

   /**
    * Sets the Automatic Full Update Status in case of a yellow stick update (IVI + Map + IVC)
    * Required to trigger writing the Map Update result to USB
    * Should be used in cases where Automatic Full Update failed from HMI due to Lock failure, etc..
    */
   void setAutomaticFullUpdateFailed(::navmiddleware::AutomaticFullUpdateFailureReason failureReason);

   ///@}
   // ================================================================================================================
   ///@{ \name EngineeringMenu

   /**
   * Retrieves the information about the currently selected menu item of the engineering menu
   */
   const EngineeringInfos& getEngineeringInfos();

   /**
   * Navigate forward: Selects, index-based, an item of the currently selected menu level if, available
   * \async ::ENGINEERING_MENU_CHANGED
   */
   void selectEngineeringMenuEntry(unsigned int index);

   /**
   * Navigation backward: Deselects the current menu level and sets the current level to the parent, if available
   * \async ::ENGINEERING_MENU_CHANGED
   */
   void deselectEngineeringMenuEntry();

   /**
   * Navigation reset: Resets the menu to the root level.
   * \async ::ENGINEERING_MENU_CHANGED
   */
   void resetEngineeringMenu();

   /**
    * Register to the Engineering Mode Properties
    * To be called whenever the Engineering menu is entered.
    */
   void enterEngineeringMenu();

   /**
    * Deregister from the Engineering Mode Properties
    * To be called whenever the Engineering menu is exited.
    */
   void exitEngineeringMenu();

   /**
    * Updates the current menu with user input.
    * In case current menu step does not accept user input it'll be silently ignored.
    */
   void updateUserInputByValue(const ::std::string& userInputValue);

   ///@}

   // ================================================================================================================
   ///@{ \name EngineeringDebugInformation
   /**
     * Retrieves the information about the traffic debug information
     */
   const EngineeringDebugInfos& getTrafficDebugInformation();

   /**
     * request for the traffic debug information
     * \async ::ENGINEERING_DEBUG_INFO_CHANGED
     */
   void requestTrafficDebugInformation();

   /**
     * Release the traffic debug information.
     * Should be called when the debug information is not required anymore.
     */
   void releaseTrafficDebugInformation();

   /**
    * This method can be used to enable/disable advanced logging of TPEG data for Traffic and LocationBasedServices
    */
   void setAdvancedTrafficLoggingState(AdvancedLoggingState state);

   ///@}
   // ================================================================================================================
   ///@{ \name Extensions

   /**
    * Checks if given extension type is available. We could also check e.g.
    * extension availability per region/SW variant, check ConfigurationInfos.
    */
   //bool hasExtension(navmidextensions::AbstractExtension::ExtensionType type) const;   //<-- INF4CV

   /**
    * Returns extension of given type. If extension is not available, method
    * will always return InvalidExtension.
    *
    * TODO: Function might also assert and/or throw if we want to be more
    * strict and catch any programming errors early on.
    */
   //<-- INF4CV
   //navmidextensions::AbstractExtension& getExtension(
   //   navmidextensions::AbstractExtension::ExtensionType type);
   //->

   // ================================================================================================================
   ///@{ \name DEBUG

   enum InterfaceTraceClass
   {
      INTERFACE_TRACECLASS_MIDDLEWARE,
      INTERFACE_TRACECLASS_LOCATION,
      INTERFACE_TRACECLASS_POSITION,
      INTERFACE_TRACECLASS_POSITIONSIMULATION,
      INTERFACE_TRACECLASS_ROUTE,
      INTERFACE_TRACECLASS_MAP,
      INTERFACE_TRACECLASS_GUIDANCE,
      INTERFACE_TRACECLASS_TRAFFICMESSAGE,
      INTERFACE_TRACECLASS_LOCATIONBASEDSERVICES
   };

   /**
   * Enables interface traces for a particular trace group (API calls and property updates)
   */
   void enableInterfaceTrace(InterfaceTraceClass interfaceTraceClass);

   /**
   * Disables interface traces for a particular trace group
   */
   void disableInterfaceTrace(InterfaceTraceClass interfaceTraceClass);

   /**
    * Enables external locking of NavMiddleware (internal message processing will be blocked)
    * to be used only in scenarios where this class is used in different threads
    */
   void lock();

   /**
    * Unlocks internal message queue
    * to be used only in scenarios where this class is used in different threads
    */
   void unlock();

   ///@}

   //<-- INF4CV
   // ================================================================================================================
   ///@{ \name USB/User POI

   /**
    * Request import of POI data from USB stick to POI server
    * @param usbPath  USB device path
    * \async ::LOCATION_UPOIS_IMPORT_STATUS_CHANGED
    */
   void importUserPois(const ::std::string& usbPath);

   /**
    * Returns ImportUserPoisStatus containing status related info.
    * Should be called after receiving property update LOCATION_UPOIS_IMPORT_STATUS_CHANGED
    */
   const ImportUserPoisStatus& getImportUserPoisStatus() const;

   /**
    * Request to delete the User POIs imported from USB device.
    */
   void deleteUserPois();

   ///@}

   /**
    * Request to mute/unmute voice guidance. This voice guidance state is not stored persistently.
    * @param guidanceMode - True for mute guidance and False for unmute guidance
    */
   void muteGuidance(bool guidanceMode);
   //->
};

}

#endif // NAV_MIDDLEWARE_H
