/**************************************************************************************
* @file         : VehicleConfigurationReader.cpp
* @author       :
* @addtogroup   : HMI-COMMON
* @brief        :
* @copyright    : (C) 2016 Robert Bosch GmbH
*                 (C) 2016 Robert Bosch Engineering and Business Solutions Limited
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "hall_std_if.h"
#include "hmi_trace_if.h"
#include "../CommonTrace.h"
#include "VehicleConfigurationReader.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_COMMON
#include "trcGenProj/Header/VehicleConfigurationReader.cpp.trc.h"
#endif

using namespace ::vehicle_main_fi_types;
using namespace ::VEHICLE_MAIN_FI;

#define DEFAULT_ACT_ID 0
#define INVALID_REGION -1
#define DOOR_CLOSE 0
#define DOOR_OPEN 1

namespace App {
namespace Core {

VehicleConfigurationReader* VehicleConfigurationReader::_pVehicleConfigurationReader = NULL;

VehicleConfigurationReader::VehicleConfigurationReader() : _vehicleProxy(::VEHICLE_MAIN_FIProxy::createProxy("vehicleMainFiPort", *this))
{
   ETG_TRACE_USR1(("VehicleConfigurationReader Constructor"));
   _regionValue = INVALID_REGION;
   if (NULL != _vehicleProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _vehicleProxy->getPortName());
   }
}


VehicleConfigurationReader::~VehicleConfigurationReader()
{
   _pVehicleConfigurationReader = NULL;
}


VehicleConfigurationReader* VehicleConfigurationReader::getInstance()
{
   if (_pVehicleConfigurationReader == NULL)
   {
      _pVehicleConfigurationReader = new VehicleConfigurationReader();
   }
   return _pVehicleConfigurationReader;
}


void VehicleConfigurationReader::onAvailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR1(("VehicleConfigurationReader::onAvailable entered"));
   if ((proxy == _vehicleProxy) && (NULL != _vehicleProxy.get()))
   {
      _swivMonAutoOpen_RegId = _vehicleProxy->sendSwivMonAutoOpenUpReg(*this);
      _marketRegionId = _vehicleProxy->sendRegionStatusUpReg(*this);
      _vehicleProxy->sendRegionStatusGet(*this);
   }
}


void VehicleConfigurationReader::onUnavailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR1(("VehicleConfigurationReader::onUnavailable entered"));
   if ((proxy == _vehicleProxy) && (NULL != _vehicleProxy.get()))
   {
      _vehicleProxy->sendSwivMonAutoOpenRelUpReg(_swivMonAutoOpen_RegId);
      _swivMonAutoOpen_RegId = DEFAULT_ACT_ID;
      _vehicleProxy->sendRegionStatusRelUpReg(_marketRegionId);
      _marketRegionId = DEFAULT_ACT_ID;
   }
}


void VehicleConfigurationReader::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& /*proxy*/, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR1(("VehicleConfigurationReader::registerProperties entered"));
}


void VehicleConfigurationReader::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& /*proxy*/, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR1(("VehicleConfigurationReader::deregisterProperties entered"));
}


void VehicleConfigurationReader::registerForRegionUpdate(IMarketRegionValue* obj)
{
   std::vector<IMarketRegionValue*>::iterator itr = std::find(_clients.begin(), _clients.end(), obj);
   if (itr == _clients.end())
   {
      _clients.push_back(obj);
      ETG_TRACE_USR4(("VehicleConfigurationReader::registerForRegionUpdate client added"));
      ETG_TRACE_USR4(("VehicleConfigurationReader::market region: %d", _regionValue));
      if (_regionValue != INVALID_REGION)
      {
         sendUpdateToClients();
      }
   }
}


void VehicleConfigurationReader::deRegisterForRegionUpdate(IMarketRegionValue* obj)
{
   std::vector<IMarketRegionValue*>::iterator itr = std::find(_clients.begin(), _clients.end(), obj);
   if (itr != _clients.end())
   {
      ETG_TRACE_USR4(("VehicleConfigurationReader::deRegisterForRegionUpdate client removed"));
      _clients.erase(itr);
   }
}


void VehicleConfigurationReader::onSwivMonAutoOpenError(const ::boost::shared_ptr< VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& /*proxy*/, const ::boost::shared_ptr< VEHICLE_MAIN_FI::SwivMonAutoOpenError >& error)
{
   if ((NULL != error.get()) && (error->getAct() == _swivMonAutoOpen_RegId))
   {
      ETG_TRACE_USR1(("VehicleConfigurationReader::onSwivMonAutoOpenError"));
   }
}


void VehicleConfigurationReader::onSwivMonAutoOpenStatus(const ::boost::shared_ptr< VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& /*proxy*/, const ::boost::shared_ptr< VEHICLE_MAIN_FI::SwivMonAutoOpenStatus >& status)
{
   if ((NULL != status.get()) && (status->getAct() == _swivMonAutoOpen_RegId))
   {
      ETG_TRACE_USR1(("VehicleConfigurationReader::onSwivMonAutoOpenStatus %d", status->getSwivMonAutoOpenValue()));
      uint8 swivMonAutoOpenValue = status->getSwivMonAutoOpenValue();
      switch (swivMonAutoOpenValue)
      {
         case DOOR_CLOSE:
            ETG_TRACE_USR1(("VehicleConfigurationReader::onSwivMonAutoOpenStatus Door close::%d", swivMonAutoOpenValue));
            break;
         case DOOR_OPEN:
            ETG_TRACE_USR1(("VehicleConfigurationReader::onSwivMonAutoOpenStatus Door open::%d", swivMonAutoOpenValue));
            break;

         default: //Unknown
            ETG_TRACE_USR1(("VehicleConfigurationReader::onSwivMonAutoOpenStatus unknown::%d", swivMonAutoOpenValue));
            break;
      }
   }
}


void VehicleConfigurationReader::onRegionStatusError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/, const ::boost::shared_ptr< RegionStatusError >& /*error*/)
{
   ETG_TRACE_USR4(("VehicleDataClientHandler::onRegionStatusError"));
}


void VehicleConfigurationReader::onRegionStatusStatus(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/, const ::boost::shared_ptr< RegionStatusStatus >& status)
{
   ETG_TRACE_USR4(("VehicleDataClientHandler::onRegionStatusStatus: "));
   if ((NULL != status.get()) && (status->getAct() == _marketRegionId) && (status->hasRegionStatusValue()))
   {
      _regionValue = status->getRegionStatusValue();
      ETG_TRACE_USR4(("VehicleDataClientHandler::onRegionStatusStatus: %d ", _regionValue));
   }
   sendUpdateToClients();
}


void VehicleConfigurationReader::sendUpdateToClients() const
{
   std::vector<IMarketRegionValue*>::const_iterator itr = _clients.begin();
   for (; itr != _clients.end(); itr++)
   {
      ETG_TRACE_USR4(("VehicleDataClientHandler::sendUpdateToClients: %d ", _regionValue));
      (*itr)->updateRegionStatusValue(_regionValue);
   }
}


} //namespace Core
} //namespace App
