/****************************************************************************
* Copyright (C) Robert Bosch Car Multimedia GmbH, 2018
* This software is property of Robert Bosch GmbH. Unauthorized
* duplication and disclosure to third parties is prohibited.
*
* FILE                : PluginClientHandler.h
* COMPONENT Name      : PluginClientHandler
* DESCRIPTION         : DBus client for plugins
*                       Receives command requests from HMI and sends it to plugin
*                       Receives updates from plugin and sends it to HMI
* AUTHOR              : Preethi Alagappan
* Date                : 14.09.2018
* Revision History    : 0.1
* Date 14.09.2018     : Initial version
* Revision History    : 0.2
* Date 25.10.2018     : Removed condition check for plugin and control block name.
*                       Moved the code, which converts EventDataUtility to Dbus variant
*                       and vice versa, to PluginDataConverter
* Revision History    : 0.3
* Date 15.11.2018     : Implementation for PluginListCommandRequest and PluginListUpdateSignal
*                       Register for plugin signal updates based on plugin and control block names
****************************************************************************/


/*****************************************************************
| includes
|----------------------------------------------------------------*/
#include "PluginClientHandler.h"
#include "PluginDataConverter.h"
#include "hmi_trace_if.h"
#include "PluginClientHandlerTrace.h"

/*****************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------*/

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_PLUGIN_CLIENT_HANDLER
#include "trcGenProj/Header/PluginClientHandler.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

#define PLUGIN_DBUS_PORTNAME "PluginDbusInterfacePort"
#define MEDIA_USB_DB_CRC_UPDATE 41
#define CMR_USB_DB_CRC_UPDATE 42


/************************************************************************
*NAME        : getInstance
*DESCRIPTION : Design section 8.2.5 - PluginClientHandler
************************************************************************/
PluginClientHandler* PluginClientHandler::getInstance()
{
   static PluginClientHandler pluginClientHandler;

   return &pluginClientHandler;
}


/************************************************************************
*NAME        : PluginClientHandler
*DESCRIPTION : Design section 8.2.5 - PluginClientHandler
************************************************************************/
PluginClientHandler::PluginClientHandler()
   : m_poPluginDbusProxy(PluginDbusInterfaceProxy::createProxy(PLUGIN_DBUS_PORTNAME, *this))
{
   ETG_TRACE_USR4(("PluginClientHandler::PluginClientHandler()"));
   StartupSync::getInstance().registerPropertyRegistrationIF(this, PLUGIN_DBUS_PORTNAME);
   vRequestCamportStatusUpdate();
   vRequestRouterStatusUpdate();
}


/************************************************************************
*NAME        : PluginClientHandler
*DESCRIPTION : Design section 8.2.5 - PluginClientHandler
************************************************************************/
PluginClientHandler::~PluginClientHandler()
{
   ETG_TRACE_USR4(("PluginClientHandler::~PluginClientHandler()"));
}


/************************************************************************
*NAME        : registerProperties
*DESCRIPTION : Design section 8.2.5 - PluginClientHandler
************************************************************************/
void PluginClientHandler::registerProperties(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("PluginClientHandler::registerProperties()"));
   if ((NULL != m_poPluginDbusProxy) && (proxy == m_poPluginDbusProxy))
   {
      m_poPluginDbusProxy->sendPluginUpdateRegister(*this);
      m_poPluginDbusProxy->sendPluginListUpdateRegister(*this);
      m_poPluginDbusProxy->sendEndStationListUpdateRegister(*this);
      m_poPluginDbusProxy->sendEndStationListUpdateGet(*this);
      m_poPluginDbusProxy->sendEntityDetailsUpdateRegister(*this);
      m_poPluginDbusProxy->sendEntityDetailsUpdateGet(*this);
   }
}


/************************************************************************
*NAME        : deregisterProperties
*DESCRIPTION : Design section 8.2.5 - PluginClientHandler
************************************************************************/
void PluginClientHandler::deregisterProperties(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("PluginClientHandler::deregisterProperties()"));
   if ((NULL != m_poPluginDbusProxy) && (proxy == m_poPluginDbusProxy))
   {
      m_poPluginDbusProxy->sendPluginUpdateDeregisterAll();
      m_poPluginDbusProxy->sendPluginListUpdateDeregisterAll();
      m_poPluginDbusProxy->sendEndStationListUpdateDeregisterAll();
      m_poPluginDbusProxy->sendEntityDetailsUpdateDeregisterAll();
   }
}


/************************************************************************
*NAME        : registerPluginSignals
*DESCRIPTION : Design section 8.2.5 - PluginClientHandler, 8.2.5.2 - Registration for plugin signals
************************************************************************/
bool PluginClientHandler::registerPluginSignals(const std::map<std::string, std::vector<std::string> >& signalMap)
{
   ETG_TRACE_USR4(("PluginClientHandler::registerPluginSignals()"));

   bool isSuccess = false;
   if (0 == pluginCtrlBlockMap.size())
   {
      pluginCtrlBlockMap = signalMap;
      isSuccess = true;
   }
   return isSuccess;
}


/************************************************************************
*NAME        : onPluginCommandError
*DESCRIPTION : Design section 8.2.5 - PluginClientHandler, 8.2.1 - PluginDbusInterface
************************************************************************/
void PluginClientHandler::onPluginCommandError(const ::boost::shared_ptr< PluginDbusInterfaceProxy >& proxy, const ::boost::shared_ptr< PluginCommandError >& error)
{
   ETG_TRACE_USR4(("PluginClientHandler::onPluginCommandError()"));

   if (NULL != error.get())
   {
      if (true == error->hasName())
      {
         ETG_TRACE_USR4(("PluginClientHandler::onPluginCommandError(), Error name : %s", error->getName().c_str()));
      }
      if (true == error->hasMessage())
      {
         ETG_TRACE_USR4(("PluginClientHandler::onPluginCommandError(), Error message : %s", error->getMessage().c_str()));
      }
   }
}


/************************************************************************
*NAME        : onPluginCommandResponse
*DESCRIPTION : Design section 8.2.5 - PluginClientHandler, 8.2.1 - PluginDbusInterface
************************************************************************/
void PluginClientHandler::onPluginCommandResponse(const ::boost::shared_ptr< PluginDbusInterfaceProxy >& proxy, const ::boost::shared_ptr< PluginCommandResponse >& response)
{
   ETG_TRACE_USR4(("PluginClientHandler::onPluginCommandResponse()"));
}


/************************************************************************
*NAME        : onPluginCommandRequest
*DESCRIPTION : Design section 8.2.5 - PluginClientHandler, 8.2.1 - PluginDbusInterface
************************************************************************/
void PluginClientHandler::onPluginCommandRequest
(
   const std::string& pluginName,
   const std::string& ctrlBlockName,
   const boost::shared_ptr<EventDataUtility>& eventData
)
{
   ETG_TRACE_USR4(("PluginClientHandler::onPluginCommandRequest() Plugin name: %s", pluginName.c_str()));
   ETG_TRACE_USR4(("PluginClientHandler::onPluginCommandRequest() Control block name: %s", ctrlBlockName.c_str()));

   if (NULL != m_poPluginDbusProxy)
   {
      ::asf::dbus::DBusVariant variantData;
      bool convertStatus = PluginDataConverter::convertToVariant(eventData, variantData);
      if (true == convertStatus)
      {
         m_poPluginDbusProxy->sendPluginCommandRequest(*this, pluginName, ctrlBlockName, variantData);
      }
      else
      {
         ETG_TRACE_USR4(("PluginClientHandler::onPluginCommandRequest(), Error in writing data into Dbus Variant"));
      }
   }
}


/************************************************************************
*NAME        : onPluginListCommandError
*DESCRIPTION : Design section 8.2.5 - PluginClientHandler, 8.2.1 - PluginDbusInterface
************************************************************************/
void PluginClientHandler::onPluginListCommandError(const ::boost::shared_ptr< PluginDbusInterfaceProxy >& proxy, const ::boost::shared_ptr< PluginListCommandError >& error)
{
   ETG_TRACE_USR4(("PluginClientHandler::onPluginListCommandError()"));

   if (NULL != error.get())
   {
      if (true == error->hasName())
      {
         ETG_TRACE_USR4(("PluginClientHandler::onPluginListCommandError(), Error name : %s", error->getName().c_str()));
      }
      if (true == error->hasMessage())
      {
         ETG_TRACE_USR4(("PluginClientHandler::onPluginListCommandError(), Error message : %s", error->getMessage().c_str()));
      }
   }
}


/************************************************************************
*NAME        : onPluginListCommandResponse
*DESCRIPTION : Design section 8.2.5 - PluginClientHandler, 8.2.1 - PluginDbusInterface
************************************************************************/
void PluginClientHandler::onPluginListCommandResponse(const ::boost::shared_ptr< PluginDbusInterfaceProxy >& proxy, const ::boost::shared_ptr< PluginListCommandResponse >& response)
{
   ETG_TRACE_USR4(("PluginClientHandler::onPluginListCommandResponse()"));
}


/************************************************************************
*NAME        : onPluginListCommandRequest
*DESCRIPTION : Design section 8.2.5 - PluginClientHandler, 8.2.1 - PluginDbusInterface
************************************************************************/
void PluginClientHandler::onPluginListCommandRequest
(
   const std::string& pluginName,
   const std::string& ctrlBlockName,
   const boost::shared_ptr<EventDataUtility>& eventData,
   const boost::shared_ptr<EventListDataUtility>& listData
)
{
   ETG_TRACE_USR4(("PluginClientHandler::onPluginListCommandRequest() Plugin name: %s", pluginName.c_str()));
   ETG_TRACE_USR4(("PluginClientHandler::onPluginListCommandRequest() Control block name: %s", ctrlBlockName.c_str()));

   if (NULL != m_poPluginDbusProxy)
   {
      ::asf::dbus::DBusVariant variantData;
      bool convertStatus = PluginDataConverter::convertToVariant(eventData, variantData);

      ::asf::dbus::DBusVariant listVariantData;
      convertStatus = convertStatus && PluginDataConverter::convertToListVariant(listData, listVariantData);
      if (true == convertStatus)
      {
         m_poPluginDbusProxy->sendPluginListCommandRequest(*this, pluginName, ctrlBlockName, variantData, listVariantData);
      }
      else
      {
         ETG_TRACE_USR4(("PluginClientHandler::onPluginListCommandRequest(), Error in writing data into Dbus Variant"));
      }
   }
}


/************************************************************************
*NAME        : onPluginUpdateError
*DESCRIPTION : Design section 8.2.5 - PluginClientHandler, 8.2.1 - PluginDbusInterface
************************************************************************/
void PluginClientHandler::onPluginUpdateError(const ::boost::shared_ptr< PluginDbusInterfaceProxy >& proxy, const ::boost::shared_ptr< PluginUpdateError >& error)
{
   ETG_TRACE_USR4(("PluginClientHandler::onPluginUpdateError()"));

   if (NULL != error.get())
   {
      if (true == error->hasName())
      {
         ETG_TRACE_USR4(("PluginClientHandler::onPluginUpdateError(), Error name : %s", error->getName().c_str()));
      }
      if (true == error->hasMessage())
      {
         ETG_TRACE_USR4(("PluginClientHandler::onPluginUpdateError(), Error message : %s", error->getMessage().c_str()));
      }
   }
}


/************************************************************************
*NAME        : onPluginUpdateSignal
*DESCRIPTION : Design section 8.2.5 - PluginClientHandler, 8.2.5.1 - Courier messages for plugin data
*              Design section 8.2.1 - PluginDbusInterface
************************************************************************/
void PluginClientHandler::onPluginUpdateSignal(const ::boost::shared_ptr< PluginDbusInterfaceProxy >& proxy, const ::boost::shared_ptr< PluginUpdateSignal >& signal)
{
   ETG_TRACE_USR4(("PluginClientHandler::onPluginUpdateSignal()"));

   if ((NULL != signal.get())
         && (true == signal->hasPlugin_name())
         && (true == signal->hasControlblock_name()))
   {
      std::string pluginName    = signal->getPlugin_name();
      std::string ctrlBlockName = signal->getControlblock_name();

      if ((true == isSignalRegistered(pluginName, ctrlBlockName))
            && (true == signal->hasData()))
      {
         ::asf::dbus::DBusVariant variantData = signal->getData();
         boost::shared_ptr<EventDataUtility> eventData;
         bool convertStatus = PluginDataConverter::convertToEventDataUtility(variantData, eventData);
         if (true == convertStatus)
         {
            POST_MSG((COURIER_MESSAGE_NEW(PluginUpdateRespMsg)(pluginName.c_str(), ctrlBlockName.c_str(), eventData)));
         }
         else
         {
            ETG_TRACE_USR4(("PluginClientHandler::onPluginUpdateSignal(), Error in reading data from DBus Variant"));
         }
      }
   }
}


/************************************************************************
*NAME        : onPluginListUpdateError
*DESCRIPTION : Design section 8.2.5 - PluginClientHandler, 8.2.1 - PluginDbusInterface
************************************************************************/
void PluginClientHandler::onPluginListUpdateError(const ::boost::shared_ptr< PluginDbusInterfaceProxy >& proxy, const ::boost::shared_ptr< PluginListUpdateError >& error)
{
   ETG_TRACE_USR4(("PluginClientHandler::onPluginListUpdateError()"));

   if (NULL != error.get())
   {
      if (true == error->hasName())
      {
         ETG_TRACE_USR4(("PluginClientHandler::onPluginListUpdateError(), Error name : %s", error->getName().c_str()));
      }
      if (true == error->hasMessage())
      {
         ETG_TRACE_USR4(("PluginClientHandler::onPluginListUpdateError(), Error message : %s", error->getMessage().c_str()));
      }
   }
}


/************************************************************************
*NAME        : onPluginListUpdateSignal
*DESCRIPTION : Design section 8.2.5 - PluginClientHandler, 8.2.1 - PluginDbusInterface
************************************************************************/
void PluginClientHandler::onPluginListUpdateSignal(const ::boost::shared_ptr< PluginDbusInterfaceProxy >& proxy, const ::boost::shared_ptr< PluginListUpdateSignal >& signal)
{
   ETG_TRACE_USR4(("PluginClientHandler::onPluginListUpdateSignal()"));

   if ((NULL != signal.get())
         && (true == signal->hasPlugin_name())
         && (true == signal->hasControlblock_name()))
   {
      std::string pluginName    = signal->getPlugin_name();
      std::string ctrlBlockName = signal->getControlblock_name();

      if ((true == isSignalRegistered(pluginName, ctrlBlockName))
            && (true == signal->hasData()) && (true == signal->hasListData()))
      {
         ::asf::dbus::DBusVariant variantData = signal->getData();
         ::asf::dbus::DBusVariant listVariantData = signal->getListData();

         boost::shared_ptr<EventDataUtility> eventData;
         bool convertStatus = PluginDataConverter::convertToEventDataUtility(variantData, eventData);

         boost::shared_ptr<EventListDataUtility> listData;
         convertStatus = convertStatus && PluginDataConverter::convertToEventListDataUtility(listVariantData, listData);
         if (true == convertStatus)
         {
            POST_MSG((COURIER_MESSAGE_NEW(PluginListUpdateRespMsg)(pluginName.c_str(), ctrlBlockName.c_str(), eventData, listData)));
         }
         else
         {
            ETG_TRACE_USR4(("PluginClientHandler::onPluginListUpdateSignal(), Error in reading data from DBus Variant"));
         }
      }
   }
}


/************************************************************************
*NAME        : onCourierMessage(PluginCommandReqMsg)
*DESCRIPTION : Design section 8.2.5 - PluginClientHandler, 8.2.5.1 - Courier messages for plugin data
************************************************************************/
bool PluginClientHandler::onCourierMessage(const PluginCommandReqMsg& oMsg)
{
   ETG_TRACE_USR4(("PluginClientHandler::onCourierMessage(PluginCommandReqMsg)"));

   std::string pluginName = oMsg.GetPluginName().GetCString();
   std::string ctrlBlockName = oMsg.GetCtrlBlockName().GetCString();
   boost::shared_ptr<EventDataUtility> eventData = oMsg.GetPluginData();

   onPluginCommandRequest(pluginName, ctrlBlockName, eventData);

   return true;
}


/************************************************************************
*NAME        : onCourierMessage(PluginListCommandReqMsg)
*DESCRIPTION : Design section 8.2.5 - PluginClientHandler, 8.2.5.1 - Courier messages for plugin data
************************************************************************/
bool PluginClientHandler::onCourierMessage(const PluginListCommandReqMsg& oMsg)
{
   ETG_TRACE_USR4(("PluginClientHandler::onCourierMessage(PluginListCommandReqMsg)"));

   std::string pluginName = oMsg.GetPluginName().GetCString();
   std::string ctrlBlockName = oMsg.GetCtrlBlockName().GetCString();
   boost::shared_ptr<EventDataUtility> eventData = oMsg.GetPluginData();
   boost::shared_ptr<EventListDataUtility> listData = oMsg.GetPluginListData();

   onPluginListCommandRequest(pluginName, ctrlBlockName, eventData, listData);

   return true;
}


/************************************************************************
*NAME        : isSignalRegistered
*DESCRIPTION : Design section 8.2.5 - PluginClientHandler, 8.2.5.2 - Registration for plugin signals
************************************************************************/
bool PluginClientHandler::isSignalRegistered(const std::string& pluginName, const std::string& ctrlBlockName)
{
   ETG_TRACE_USR4(("PluginClientHandler::isSignalRegistered()"));
   ETG_TRACE_USR4(("PluginClientHandler::isSignalRegistered(), Plugin name : %s", pluginName.c_str()));
   ETG_TRACE_USR4(("PluginClientHandler::isSignalRegistered(), Control block name : %s", ctrlBlockName.c_str()));

   bool matchFound = false;
   std::map<std::string, std::vector<std::string> >::iterator iter = pluginCtrlBlockMap.find(pluginName);
   if (iter != pluginCtrlBlockMap.end())
   {
      std::vector<std::string> ctrlBlockNames = iter->second;
      std::vector<std::string>::iterator ctrlBlockIter = ctrlBlockNames.begin();
      for (; ctrlBlockIter != ctrlBlockNames.end(); ctrlBlockIter++)
      {
         if (0 == (*ctrlBlockIter).compare(ctrlBlockName))
         {
            matchFound = true;
            break;
         }
      }
   }
   ETG_TRACE_USR4(("PluginClientHandler::isSignalRegistered(), Status: %d", matchFound));
   return matchFound;
}


/************************************************************************
*NAME        : onEndStationListUpdateError
*DESCRIPTION : Design section 8.2.5 - PluginClientHandler, 8.2.5.1 - Courier messages for endsattionlist count
*              Design section 8.2.1 - PluginDbusInterface
************************************************************************/
void  PluginClientHandler::onEndStationListUpdateError(const ::boost::shared_ptr< PluginDbusInterfaceProxy >& proxy, const ::boost::shared_ptr< EndStationListUpdateError >& error)
{
   ETG_TRACE_USR4(("PluginClientHandler::onEndStationListUpdateError()"));
}


/************************************************************************
*NAME        : onEndStationListUpdateUpdate
*DESCRIPTION : Design section 8.2.5 - PluginClientHandler, 8.2.5.1 - Courier messages for endsattionlist count
*              Design section 8.2.1 - PluginDbusInterface
************************************************************************/

void PluginClientHandler::onEndStationListUpdateUpdate(const ::boost::shared_ptr< PluginDbusInterfaceProxy >& proxy, const ::boost::shared_ptr< EndStationListUpdateUpdate >& update)
{
   ETG_TRACE_USR4(("PluginClientHandler::onEndStationListUpdateUpdate()"));

   if (true == update->hasEndStationListUpdate())
   {
      ETG_TRACE_USR4(("PluginClientHandler::onEndStationListUpdateUpdate() Enteered"));
      ::asf::dbus::DBusVariant variantData = update->getEndStationListUpdate();
      boost::shared_ptr<EventDataUtility> eventData;

      bool bConvertStatus = PluginDataConverter::convertToEventDataUtility(variantData, eventData);
      ETG_TRACE_USR4(("PluginClientHandler::onEndStationListUpdateUpdate() bConvertStatus  %d", bConvertStatus));

      if (true == bConvertStatus)
      {
         POST_MSG((COURIER_MESSAGE_NEW(PluginEndStationListUpdateMsg)(eventData)));
      }
      else
      {
         ETG_TRACE_USR4(("PluginClientHandler::onEndStationListUpdateUpdate(), Error in reading data from DBus Variant"));
      }
   }
}


/************************************************************************
*NAME        : onEntityDetailsUpdateError
*DESCRIPTION : Design section 8.2.5 - PluginClientHandler, 8.2.5.1 - Courier messages for EntityDetailsUpdate
*              Design section 8.2.1 - PluginDbusInterface
************************************************************************/
void  PluginClientHandler::onEntityDetailsUpdateError(const ::boost::shared_ptr< PluginDbusInterfaceProxy >& proxy, const ::boost::shared_ptr< EntityDetailsUpdateError >& error)
{
   ETG_TRACE_USR4(("PluginClientHandler::onEntityDetailsUpdateError()"));
}


/************************************************************************
*NAME        : onEntityDetailsUpdate
*DESCRIPTION : Design section 8.2.5 - PluginClientHandler, 8.2.5.1 - Courier messages for EntityDetailsUpdate
*              Design section 8.2.1 - PluginDbusInterface
************************************************************************/

void PluginClientHandler::onEntityDetailsUpdateUpdate(const ::boost::shared_ptr< PluginDbusInterfaceProxy >& proxy, const ::boost::shared_ptr< EntityDetailsUpdateUpdate >& update)
{
   ETG_TRACE_USR4(("PluginClientHandler::onEntityDetailsUpdateUpdate()"));

   if (true == update->hasEntityDetailsUpdate())
   {
      ETG_TRACE_USR4(("PluginClientHandler::onEntityDetailsUpdate() Enteered"));
      //std::vector< EntityDetailsUpdateStruct > EntityDetails = update->getEntityDetailsUpdate();

      ::asf::dbus::DBusVariant listVariantData = update->getEntityDetailsUpdate();
      boost::shared_ptr<EventListDataUtility> entityData;
      bool convertStatus = PluginDataConverter::convertToEventListDataUtility(listVariantData, entityData);
      if (true == convertStatus)
      {
         POST_MSG((COURIER_MESSAGE_NEW(PluginEntityDetailsUpdateMsg)(entityData)));
      }
      else
      {
         ETG_TRACE_USR4(("PluginClientHandler::onEntityDetailsUpdate(), Error in reading data from DBus Variant"));
      }
   }
}


void PluginClientHandler::vRequestCamportStatusUpdate()
{
   boost::shared_ptr<EventDataUtility> pluginData(EventDataUtility::newEventDataUtility());
   uint16 ControlId = MEDIA_USB_DB_CRC_UPDATE;
   uint16 ControlValue = 0;
   uint8 activeRegion = 0;
   uint8 listValue = 0;
   uint16 positionValue = 0;
   ETG_TRACE_USR4(("PluginClientHandler::vRequestCamportStatusUpdate %d", ControlId));

   if (NULL != pluginData.get())
   {
      pluginData->addEventData(ControlId);
      pluginData->addEventData(ControlValue);
      pluginData->addEventData(activeRegion);
      pluginData->addEventData(listValue);
      pluginData->addEventData(positionValue);
      onPluginCommandRequest("Media", "", pluginData);
   }
}


void PluginClientHandler::vRequestRouterStatusUpdate()
{
   boost::shared_ptr<EventDataUtility> pluginData(EventDataUtility::newEventDataUtility());
   uint8 ControlId = CMR_USB_DB_CRC_UPDATE;
   uint8 ControlValue = 0;
   uint8 activeRegion = 0;
   uint8 listValue = 0;
   uint16 positionValue = 0;
   ETG_TRACE_USR4(("PluginClientHandler::vRequestRouterStatusUpdate %d", ControlId));

   if (NULL != pluginData.get())
   {
      pluginData->addEventData(ControlId);
      pluginData->addEventData(ControlValue);
      pluginData->addEventData(activeRegion);
      pluginData->addEventData(listValue);
      pluginData->addEventData(positionValue);
      onPluginCommandRequest("Router", "", pluginData);
   }
}


/* ----------------------------------------------------------*/
