/**
 * @file         : LanguageHandlerInterface.h
 * @author       :
 * @addtogroup   : AppHmi_App_PRJ_Common
 * @brief        : LanguageHandler is to handle the LanguageHandler feature interfaces implementation
 * @copyright    : (C) 2016 Robert Bosch GmbH
 *                 (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#ifndef LANGUAGE_HANDLER_INTERFACE_H
#define LANGUAGE_HANDLER_INTERFACE_H


#include <map>
#include <vector>
#include <string>
#include <asf/core/Types.h>
#include <hmibase/util/StrUtf8.h>


/**
 *  Classes Forward Declaration
 */

namespace languageHandler {


class ILanguageISOCodeHandler;
class ILanguageStatusPropertyUpdateCB;
} //namespace languageHandler


/**
 *  Classes Declaration
 */

namespace languageHandler {


/**
 *  IServiceClient is used to
 *  - Provide interface impl between service client and LanguageHandler
 */
class IServiceClient
{
   public:
      /**
       *  Member Function Declaration
       */
      IServiceClient() {}
      virtual ~IServiceClient() {}
      virtual bool fetchLanguageStatusInfo(uint32& languageId) const = 0;
      virtual void init() = 0;
      virtual void registerLanguageStatusPropertyUpdate(ILanguageStatusPropertyUpdateCB& imp) = 0;
      virtual void deregisterLanguageStatusPropertyUpdate(ILanguageStatusPropertyUpdateCB& imp) = 0;
};


/**
 *  ILanguageDpHandler is used to
 *  - Provide interface impl between LanguageDpHandler and Users
 */
class ILanguageDpHandler
{
   public:
      /**
       *  Member Function Declaration
       */
      ILanguageDpHandler() {}
      virtual ~ILanguageDpHandler() {}
      virtual uint32 getLanguageId() const = 0;
      virtual void init() = 0;
      virtual void setLanguageId(const uint32 languageId) = 0;
      virtual void setLanguageISOCodeHandler(const ILanguageISOCodeHandler& imp) = 0;
};


/**
 *  ILanguageISOCodeHandler is used to
 *  - Provide interface impl between LanguageISOCodeHandler and Users
 */
class ILanguageISOCodeHandler
{
   public:
      /**
       *  Member Functions Declaration
       */
      ILanguageISOCodeHandler() {}
      virtual ~ILanguageISOCodeHandler() {}
      virtual uint32 getLanguageId(const ::std::string& code) const = 0;
      virtual ::std::string getLanguageISOCode(const uint32 languageId) const = 0;
      virtual void init() = 0;
};


/**
 *  IServiceClientConfigurator is used to
 *  - Provide interface impl between service client and its Configurator
 */
class IServiceClientConfigurator
{
   public:
      /**
       *  Member Function Declaration
       */
      IServiceClientConfigurator() {}
      virtual ~IServiceClientConfigurator() {}
};


/**
 *  ILanguageStatusPropertyUpdateCB is used to
 *  - Provide interface to communicate the HmiInfo / Vehicle service LanguageStatusProperty
 *      update to service client users
 */
class ILanguageStatusPropertyUpdateCB
{
   public:
      /**
       *  Member Function Declaration
       */
      ILanguageStatusPropertyUpdateCB() {}
      virtual ~ILanguageStatusPropertyUpdateCB() {}
      virtual void onLanguageStatusUpdate(const uint32 languageId) = 0;
};


/**
 *  IHmiInfoServiceClient is used to
 *  - Provide interface impl between HmiInfoServiceClient and Users
 */
class IHmiInfoServiceClient : public IServiceClient
{
   public:
      /**
       *  Member Function Declaration
       */
      IHmiInfoServiceClient() {}
      virtual ~IHmiInfoServiceClient() {}
};


/**
 *  IVehicleServiceClient is used to
 *  - Provide interface impl between VehicleServiceClient and Users
 */
class IVehicleServiceClient : public IServiceClient
{
   public:
      /**
       *  Member Function Declaration
       */
      IVehicleServiceClient() {}
      virtual ~IVehicleServiceClient() {}
};


/**
 *  IHmiInfoServiceClientConfigurator is used to
 *  - Provide interface impl between HmiInfoServiceClient and its Configurator
 */
class IHmiInfoServiceClientConfigurator : virtual public IServiceClientConfigurator
{
   public:
      /**
       *  Member Function Declaration
       */
      IHmiInfoServiceClientConfigurator() {}
      virtual ~IHmiInfoServiceClientConfigurator() {}
      virtual void configureHmiInfoServicePortName(::std::string& name, bool& isWiredRunTime) = 0;
};


/**
 *  IVehicleServiceClientConfigurator is used to
 *  - Provide interface impl between VehicleServiceClient and its Configurator
 */
class IVehicleServiceClientConfigurator : virtual public IServiceClientConfigurator
{
   public:
      /**
       *  Member Function Declaration
       */
      IVehicleServiceClientConfigurator() {}
      virtual ~IVehicleServiceClientConfigurator() {}
      virtual void configureVehicleServicePortName(::std::string& name, bool& isWiredRunTime) = 0;
};


} //namespace languageHandler


#endif /* LANGUAGE_HANDLER_INTERFACE_H */
