/**
 * @file         : LanguageHandler.cpp
 * @author       :
 * @addtogroup   : AppHmi_App_PRJ_Common
 * @brief        : LanguageHandler is to handle the language / culture update implementation
 * @copyright    : (C) 2016 Robert Bosch GmbH
 *                 (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#include "LanguageHandler.h"
#include "unistd.h"
#include "CommonTrace.h"
#include "hmi_trace_if.h"
#include "LanguageISOCodeHandler.h"
#include <Trace/TraceUtils.h>
#include <BaseContract/generated/BaseMsgs.h>


using namespace ::hmibase::trace;


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_COMMON
#include "trcGenProj/Header/LanguageHandler.cpp.trc.h"
#endif /* VARIANT_S_FTR_ENABLE_TRC_GEN */


/**
 *  Static Member Initialization
 */

namespace languageHandler {


LanguageHandler* LanguageHandler::_languageHandler = NULL;
} //namespace languageHandler


namespace languageHandler {


LanguageHandler::LanguageHandler() : _serviceClient(NULL)
   , _languageDpHandler(NULL)
   , _languageISOCodeHandler(NULL)
{
   ETG_TRACE_USR1(("LanguageHandler: INFO: APP: %25s : PID: %d : CTOR",  getAppName().c_str(), getpid()));
   clear();
   initLanguageISOCodeHandler();
}


LanguageHandler::~LanguageHandler()
{
   ETG_TRACE_USR1(("LanguageHandler: INFO: APP: %25s : PID: %d : DTOR",  getAppName().c_str(), getpid()));
   clear();
}


LanguageHandler::LanguageHandler(const LanguageHandler& obj) : _serviceClient(NULL)
   , _languageDpHandler(NULL)
   , _languageISOCodeHandler(NULL)
{
   ETG_TRACE_USR1(("LanguageHandler: INFO: APP: %25s : PID: %d : CTOR",  getAppName().c_str(), getpid()));
   clear();
   *this = obj;
   initLanguageISOCodeHandler();
}


LanguageHandler& LanguageHandler::operator= (const LanguageHandler& obj)
{
   if (this != &obj)
   {
      clear();
   }
   return *this;
}


void LanguageHandler::clear()
{
   _guiInitialized = false;
   if (NULL != _serviceClient)
   {
      _serviceClient->deregisterLanguageStatusPropertyUpdate((*this));
      delete _serviceClient;
      _serviceClient = NULL;
   }
   if (NULL != _languageDpHandler)
   {
      delete _languageDpHandler;
      _languageDpHandler = NULL;
   }
   if (NULL != _languageISOCodeHandler)
   {
      delete _languageISOCodeHandler;
      _languageISOCodeHandler = NULL;
   }
}


void LanguageHandler::initLanguageISOCodeHandler()
{
   if (NULL == _languageISOCodeHandler)
   {
      _languageISOCodeHandler = new LanguageISOCodeHandler();
      if (NULL != _languageISOCodeHandler)
      {
         _languageISOCodeHandler->init();
      }
   }
}


bool LanguageHandler::OnMessage(const ::Courier::Message& Msg)
{
   bool isConsumed = false;
   switch (Msg.GetId())
   {
      case ::Courier::StartupMsg::ID:
      {
         _guiInitialized = true;
         performStartupLanguageUpdate();
         break;
      }
      case ::Courier::ShutdownMsg::ID:
      {
         _guiInitialized = false;
         break;
      }
      default:
         break;
   }
   return isConsumed;
}


void LanguageHandler::onLanguageStatusUpdate(const uint32 languageId)
{
   sendLocalizationRequest(languageId);
   if (NULL != _languageDpHandler)
   {
      _languageDpHandler->setLanguageId(languageId);
   }
}


void LanguageHandler::performStartupLanguageUpdate()
{
   uint32 tLanguageId = 0;
   if ((NULL != _serviceClient) && (_serviceClient->fetchLanguageStatusInfo(tLanguageId)))
   {
      sendLocalizationRequest(tLanguageId);
   }
   else if (NULL != _languageDpHandler)
   {
      tLanguageId = _languageDpHandler->getLanguageId();
      sendLocalizationRequest(tLanguageId);
   }
   else
   {}
}


void LanguageHandler::sendLocalizationRequest(const uint32 languageId)
{
   ETG_TRACE_USR1(("LanguageHandler: INFO: APP: %25s : PID: %d : sendLocalizationRequest: <-: LanguageId: %d", getAppName().c_str(), getpid(), languageId));
   if (_guiInitialized)
   {
      ::std::string tISOCode = (NULL != _languageISOCodeHandler) ? _languageISOCodeHandler->getLanguageISOCode(languageId) : "";
      if (!tISOCode.empty())
      {
         ::Courier::Message* msg = COURIER_MESSAGE_NEW(LocalizationReqMsg)(Candera::String(tISOCode.c_str(), tISOCode.length()));
         if (NULL != msg)
         {
            ETG_TRACE_USR1(("LanguageHandler: INFO: APP: %25s : PID: %d : sendLocalizationRequest: ->: LanguageId: %d : ISOCode: %s", \
                            getAppName().c_str(), getpid(), languageId, tISOCode.c_str()));
            (void)msg->Post();
         }
      }
   }
}


} // namespace languageHandler
