/* ***************************************************************************************
* FILE:          DefaultListRotaryController.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  DefaultListRotaryController.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"

#include "DefaultListRotaryController.h"
#include "ProjectBaseTypes.h"
#include "ProjectBaseMsgs.h"

#include <Courier/Visualization/ViewHandler.h>

#include <View/CGI/CgiExtensions/AppViewHandler.h>
#include <Focus/FCommon.h>
#include <Focus/FConfigInfo.h>
#include <Focus/FController.h>
#include <Focus/FDataSet.h>
#include <Focus/FSession.h>
#include <Focus/FActiveViewGroup.h>
#include <Focus/FManager.h>
#include <Focus/FGroupTraverser.h>
#include <Focus/Default/FDefaultAvgManager.h>

#include "Widgets/2D/WidgetFinder2D.h"
#include "Widgets/2D/List/ListWidget2D.h"

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "Widgets/widget_etg_if.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_FOCUS
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/DefaultListRotaryController.cpp.trc.h"
#endif

COURIER_LOG_SET_REALM(Candera::Diagnostics::LogRealm::User);

Courier::IViewHandler* DefaultListRotaryController::_viewHandler = NULL;

DefaultListRotaryController::DefaultListRotaryController(bool invertedRotary, bool pagewiseScrolling)
   : _pagewiseScrolling(pagewiseScrolling), _invertedRotary(invertedRotary)
{
}


bool DefaultListRotaryController::configureWidget(Focus::FSession&, Focus::FWidgetConfig&, Focus::FFrameworkWidget& widget)
{
   //FlexListWidget2D* listWidget = dynamic_cast<FlexListWidget2D*>(&widget);
   //if ((listWidget == NULL) || (listWidget->GetParentView() == NULL))
   //{
   //   return false;
   //}

   _viewHandler = (widget.GetParentView() != NULL) ? widget.GetParentView()->GetViewHandler() : NULL;

   ListContentUpdater::SetContentUpdateCallback(onListWidgetContentUpdated);

   return true;
}


static bool isWidgetValid(Focus::Focusable& f)
{
   Focus::FWidgetVisitor visitor;
   f.visit(visitor);
   //if it is valid (count == 1)
   return (visitor.getCount() > 0);
}


bool DefaultListRotaryController::onMessage(Focus::FSession& session, Focus::FGroup& group, const Focus::FMessage& msg)
{
   bool result = false;
   switch (msg.GetId())
   {
      case ListFocusResetReqMsg::ID:
      {
         const ListFocusResetReqMsg* focusResetMsg = Courier::message_cast<const ListFocusResetReqMsg*>(&msg);
         if (focusResetMsg != NULL)
         {
            result = onFocusResetReqMessage(session, group, *focusResetMsg);
         }
         break;
      }
      case ListFocusChangeReqMsg::ID:
      {
         const ListFocusChangeReqMsg* focusChangeMsg = Courier::message_cast<const ListFocusChangeReqMsg*>(&msg);
         if (focusChangeMsg != NULL)
         {
            result = onFocusChangeReqMessage(session, group, *focusChangeMsg);
         }
         break;
      }

      case EncoderStatusChangedUpdMsg::ID:
      {
         const EncoderStatusChangedUpdMsg* encoderMsg = Courier::message_cast<const EncoderStatusChangedUpdMsg*>(&msg);
         if (encoderMsg != NULL)
         {
            result = onRotaryMessage(session, group, *encoderMsg);
         }
         break;
      }

      default:
         break;
   }

   return result;
}


bool DefaultListRotaryController::onFocusResetReqMessage(Focus::FSession& session, Focus::FGroup& group, const ListFocusResetReqMsg& msg)
{
   bool result = false;

   ETG_TRACE_USR1_CLS((TR_CLASS_HMI_FW_FOCUS, "DefaultListRotaryController::onFocusResetReqMessage listId=%u, action=%d, childCount=%u",
                       msg.GetListId(), msg.GetAction(), group.Children.count()));

   Focus::FListData* listData = group.getData<Focus::FListData>();
   if ((listData != NULL) && (msg.GetListId() == static_cast<unsigned int>(listData->ListId))
         && (group.Children.count() > 0))
   {
      Focus::Focusable* newFocus = NULL;
      int steps = 0;//used to move focus in case new focus is not valid
      switch (msg.GetAction())
      {
         case LIST_FOCUS_RESET_TO_DEFAULT:
         case LIST_FOCUS_RESET_TO_FIRST_VISIBLE:
         {
            newFocus = group.Children.get(0);
            steps = 1;
            break;
         }
         case LIST_FOCUS_RESET_TO_LAST_VISIBLE:
         {
            newFocus = group.Children.get(group.Children.count() - 1);
            steps = -1;
            break;
         }
         default:
            break;
      }
      if (newFocus != NULL)
      {
         Focus::FManager& manager = Focus::FManager::getInstance();
         Focus::FDefaultAvgManager avgManager(manager);
         avgManager.setFocus(session, newFocus);

         bool isValid = isWidgetValid(*newFocus);

         ETG_TRACE_USR1_CLS((TR_CLASS_HMI_FW_FOCUS, "DefaultListRotaryController::onFocusResetReqMessage isValid=%u, newFocus=%s",
                             isValid, FID_STR(newFocus->getId())));

         int actualSteps = 0;
         //if item is not valid we need to skip it
         if (!isValid)
         {
            actualSteps = avgManager.moveFocus(session, group, steps);
         }
         if ((isValid || (actualSteps != 0)) && !manager.isFocusVisible())
         {
            manager.setFocusVisible(true);
         }
      }
      result = true;
   }

   return result;
}


bool DefaultListRotaryController::onFocusChangeReqMessage(Focus::FSession& session, Focus::FGroup& group, const ListFocusChangeReqMsg& msg)
{
   bool result = false;

   ETG_TRACE_USR1_CLS((TR_CLASS_HMI_FW_FOCUS, "DefaultListRotaryController::onFocusChangeReqMessage listId=%u, steps=%d",
                       msg.GetListId(), msg.GetSteps()));

   Focus::FListData* listData = group.getData<Focus::FListData>();
   if ((listData != NULL) && (msg.GetListId() == static_cast<unsigned int>(listData->ListId)))
   {
      Focus::FManager& manager = Focus::FManager::getInstance();
      Focus::FDefaultAvgManager avgManager(manager);
      int actualSteps = avgManager.moveFocus(session, group, msg.GetSteps());
      if ((actualSteps != 0) && (!manager.isFocusVisible()))
      {
         manager.setFocusVisible(true);
      }

      result = true;
   }

   return result;
}


bool DefaultListRotaryController::onRotaryMessage(Focus::FSession& session, Focus::FGroup& group, const EncoderStatusChangedUpdMsg& msg)
{
   bool result = false;
   Focus::FManager& manager = Focus::FManager::getInstance();
   if (manager.getActivityTimer() != NULL)
   {
      manager.getActivityTimer()->restart();
   }

   Focus::FDefaultAvgManager avgManager(manager);

   int steps = msg.GetEncSteps();
   if (_invertedRotary)
   {
      steps = -steps;
   }

   bool crtFocusVisibleAndValid = manager.isFocusVisible() && avgManager.isCurrentFocusAvailable(session, &group);

   //if focus is not visible check old focus
   if (!crtFocusVisibleAndValid)
   {
      //we need a new sequence for the preserve focus
      manager.generateNewSequenceId();

      //get default/current focus
      Focus::Focusable* f = avgManager.getFocus(session, &group);
      if (f != NULL)
      {
         if (isWidgetValid(*f))
         {
            //make focus visible
            manager.setFocusVisible(true);
            avgManager.setFocus(session, f);
            result = true;
         }
         //otherwise proceed with 1 step movement of the focus which skips disabled items
         else
         {
            steps = (steps < 0) ? -1 : 1;
         }
      }
   }

   if (!result)
   {
      int actualSteps = avgManager.moveFocus(session, group, steps);
      //if focus is not visible yet make it visible
      if ((actualSteps != 0) && (!crtFocusVisibleAndValid))
      {
         manager.setFocusVisible(true);
      }
      if (steps != actualSteps)
      {
         result = handleIncompleteFocusMove(session, group, steps - actualSteps);
      }
      else
      {
         result = true;
      }
   }
   return result;
}


//lint -esym(1764, session) TODO: Remove this line as soon as parameter session is used in this function
bool DefaultListRotaryController::handleIncompleteFocusMove(Focus::FSession& session, Focus::FGroup& group, int correction)
{
   (void)session;
   Focus::FManager& manager = Focus::FManager::getInstance();
   //no correction necessary
   if (correction == 0)
   {
      return false;
   }

   Focus::FGroupData* groupData = group.getData<Focus::FGroupData>();
   Focus::FListData* listData = group.getData<Focus::FListData>();
   if ((groupData == NULL) || (listData == NULL))
   {
      ETG_TRACE_USR1_CLS((TR_CLASS_HMI_FW_FOCUS, "DefaultListRotaryController::handleIncompleteFocusMove no list data or no group data"));
      return false;
   }

   ETG_TRACE_USR1_CLS((TR_CLASS_HMI_FW_FOCUS, "DefaultListRotaryController::handleIncompleteFocusMove correction=%d, listId=%d, firstVisibleIndex=%d, visibleItems=%d, totalItems=%d",
                       correction, listData->ListId, listData->FirstVisibleIndex, listData->VisibleItemCount, listData->TotalItemCount));

   //invalid list data or no scroll possible due to few elements in the list
   if ((listData->ListId < 0) || (listData->FirstVisibleIndex < 0)
         || (listData->VisibleItemCount <= 0) || (listData->TotalItemCount <= 0)
         || (listData->TotalItemCount <= listData->VisibleItemCount))
   {
      return false;
   }

   bool result = false;
   ListChangeMsg* listChangeMsg = NULL;
   ListFocusResetReqMsg* listFocusResetMsg = NULL;
   if (correction > 0)
   {
      //can scroll down
      if (listData->FirstVisibleIndex + listData->VisibleItemCount < listData->TotalItemCount)
      {
         result = true;
         if (_pagewiseScrolling)
         {
            listChangeMsg = COURIER_MESSAGE_NEW(ListChangeMsg)(listData->ListId, ListChangePageDown, 1, ListChangeMsgSourceEncoder);
            listFocusResetMsg = COURIER_MESSAGE_NEW(ListFocusResetReqMsg)(listData->ListId, LIST_FOCUS_RESET_TO_FIRST_VISIBLE);
            --correction;//one step will be "consumed" by the ListFocusResetReqMsg
         }
         else
         {
            listChangeMsg = COURIER_MESSAGE_NEW(ListChangeMsg)(listData->ListId, ListChangeDown, correction, ListChangeMsgSourceEncoder);
         }
      }
      //last item visible and wrap around
      else if (groupData->WrapAround)
      {
         result = true;
         listChangeMsg = COURIER_MESSAGE_NEW(ListChangeMsg)(listData->ListId, ListChangeSet, 0, ListChangeMsgSourceEncoder);
         --correction;//one step will be "consumed" by the ListFocusResetReqMsg
      }
      else
      {
         //nothing to do
      }
   }
   else //if correction < 0
   {
      //can scroll up
      if (listData->FirstVisibleIndex > 0)
      {
         result = true;
         if (_pagewiseScrolling)
         {
            listChangeMsg = COURIER_MESSAGE_NEW(ListChangeMsg)(listData->ListId, ListChangePageUp, -1, ListChangeMsgSourceEncoder);
            listFocusResetMsg = COURIER_MESSAGE_NEW(ListFocusResetReqMsg)(listData->ListId, LIST_FOCUS_RESET_TO_LAST_VISIBLE);
            ++correction;//one step will be "consumed" by the ListFocusResetReqMsg
         }
         else
         {
            listChangeMsg = COURIER_MESSAGE_NEW(ListChangeMsg)(listData->ListId, ListChangeUp, -correction, ListChangeMsgSourceEncoder);
         }
      }
      //first item visible and wrap around
      else if (groupData->WrapAround)
      {
         result = true;
         int position = listData->TotalItemCount - listData->VisibleItemCount;
         listChangeMsg = COURIER_MESSAGE_NEW(ListChangeMsg)(listData->ListId, ListChangeSet, position, ListChangeMsgSourceEncoder);
         listFocusResetMsg = COURIER_MESSAGE_NEW(ListFocusResetReqMsg)(listData->ListId, LIST_FOCUS_RESET_TO_LAST_VISIBLE);
         ++correction;//one step will be "consumed" by the ListFocusResetReqMsg
      }
      else
      {
         //nothing to do
      }
   }

   if (result)
   {
      if (manager.getSessionManager() != NULL)
      {
         manager.getSessionManager()->requestSessionSuspend();
      }
      if (listChangeMsg != NULL)
      {
         listChangeMsg->Post();
      }
      if (listFocusResetMsg != NULL)
      {
         listFocusResetMsg->Post();
      }
      if (correction != 0)
      {
         ListFocusChangeReqMsg* listFocusChangeMsg = COURIER_MESSAGE_NEW(ListFocusChangeReqMsg)(listData->ListId, correction);
         if (listFocusChangeMsg != NULL)
         {
            listFocusChangeMsg->Post();
         }
      }
   }
   return result;
}


void DefaultListRotaryController::onListWidgetContentUpdated(Candera::UInt32 listId, bool success)
{
   (void)listId;
   (void)success;

   Focus::FManager& manager = Focus::FManager::getInstance();
   Focus::FSessionManager* sessionManager = manager.getSessionManager();
   if (sessionManager != NULL)
   {
      //if visible content is updated while the session is suspended
      //=>we clear the suspend flag and refresh the session
      if (sessionManager->isSessionSuspended())
      {
         ETG_TRACE_USR1_CLS((TR_CLASS_HMI_FW_FOCUS, "DefaultListRotaryController::onListWidgetContentUpdated Session is suspended!"));
         sessionManager->clearSessionSuspend();

         RefreshFocusSessionReqMsg refreshMsg;
         bool status = sessionManager->onMessage(refreshMsg);
      }
      //if visible content is updated while no session exists
      //=>check if the current focus is still valid (widget exists)
      else if ((sessionManager->getSession() == NULL) && (manager.getConsistencyChecker() != NULL))
      {
         manager.getConsistencyChecker()->checkFocus();
      }
      else
      {
         //nothing to do
      }
   }
}
