/* ***************************************************************************************
* FILE:          ListModel.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ListModel.h is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#if !defined(_LIST_MODEL_H)
#define _LIST_MODEL_H

#include "hmi_trace_if.h"
#include "List/Data/ListDataProviderDistributor.h"
#include "Common/DataModel/ListRegistry.h"

/**
* Provides the data for the list and manages the selected item index based on the button reaction messages from the list items.
* To ensure that those button messages are received by this object it should be added in the dispatch table of the main data model component as shown below:
*      COURIER_MSG_MAP_BEGIN(...)
*      ...
*      COURIER_MSG_MAP_DELEGATE_DEF_BEGIN()
*      COURIER_MSG_DELEGATE_TO_OBJ(_XYZSimpleListModel)
*      COURIER_MSG_MAP_DELEGATE_DEF_END()
*      COURIER_MSG_MAP_DELEGATE_END()
*/
class AbstractListModel : public ListImplementation
{
      typedef ListImplementation Base;

   public:
      AbstractListModel(unsigned int listId, DataItemContext listItemTemplate);
      virtual ~AbstractListModel();

      COURIER_MSG_MAP_BEGIN(0)
      ON_COURIER_MESSAGE(ButtonListItemUpdMsg)
      COURIER_MSG_MAP_DELEGATE_DEF_BEGIN()
      /** Dispatch the messages to the base class */
      COURIER_MSG_DELEGATE_TO_CLASS(ListImplementation)
      COURIER_MSG_MAP_DELEGATE_DEF_END()
      COURIER_MSG_MAP_DELEGATE_END()

      /** Returns the list id */
      unsigned int getListId() const;

      /** Returns the number of items in the list. */
      virtual size_t getItemCount() const = 0;

      /** Returns the index of the selected item. */
      size_t getSelectedItemIndex() const;
      /** Changes the index of the selected item and calls onSelectedItemChanged(). */
      void setSelectedItemIndex(size_t index);

      /** Sends the list data update message. It should be called when the list content is changed or when the selected item index changes. */
      virtual void sendUpdate();

   protected:
      /** Adds the data for the specified list item. It is called during sending of the content update. */
      virtual void addListItem(ListDataProviderBuilder& listBuilder, size_t index, bool isSelected) const = 0;

      /** Called when the selected item index is changed as a result of an external setSelectedItemIndex() or from the list item button messages. */
      virtual void onSelectedItemChanged();

      /** Creates a list data provider containing an entry for each item in order to be sent to the widget. */
      virtual tSharedPtrDataProvider getListDataProvider(unsigned int listId, int reqItemIndex, unsigned int reqItemCount);
      virtual tSharedPtrDataProvider getListDataProvider(const ListDateProviderReqMsg& oMsg);

      /** Processes the list item button messages and determines the new selected index. */
      virtual bool onCourierMessage(const ButtonListItemUpdMsg& oMsg);

   private:
      /** Identifies the list. */
      unsigned int _listId;
      /** List item template. */
      DataItemContext _listItemTemplate;
      /** Index of the selected item. */
      size_t _selectedItemIndex;
};


/**
* Extends AbstractListModel with additional helper functionality.
* This template must be specialized with 2 types:
* - TListDataBindingSource - databinding source containing a field named "SelectedIndex" used for the selected item index and a field named "Count" for the total number of items in the list. Additional fields can be stored in it like ListId.
* - TItemData - data used for each item.
* Contains a vector where item data is added.
* Updates the databinding source when the selected tab changes.
*
* This is how it should be used.
* Add a databinding source in the HMI contract. Only SelectedIndex and Count will be used by this class.
*   <bindingSource name="FMTabControlDemoList" readonly="false" access="asynchronous" uid="{589C00EE-FF3D-4D6B-BB84-054AAC551936}">
*     <item name="ListId" type="Candera::UInt32" readonly="true" uid="{4A882B3F-516E-4F69-9AC2-A3F7D0E1961C}" defaultValue="LIST_ID_FM_TAB_CONTROL_DEMO"/>
*     <item name="Count" type="Candera::UInt32" readonly="false" uid="{6A12618B-3AD2-43C3-A4B6-C022397C685A}" defaultValue="0"/>
*     <item name="SelectedIndex" type="Candera::UInt32" readonly="false" uid="{B2E7E0CE-2E2A-4175-9758-D1F3592197DE}" defaultValue="0"/>
*   </bindingSource>
*
* Instantiate the list model and populate it with items.
*  _FMTabControlDemo = new SimpleListModel<FMTabControlDemoListDataBindingSource, unsigned int>(LIST_ID_FM_TAB_CONTROL_DEMO, "TabControl_ListItem");
*  if (_FMTabControlDemo != NULL)
*  {
*     ListRegistry::s_getInstance().addListImplementation(_FMTabControlDemo->getListId(), _FMTabControlDemo);
*     _FMTabControlDemo->addItem(TextId_TUNER_SetupList_SeekMode_Station_Text);
*     _FMTabControlDemo->addItem(TextId_TUNER_SetupList_SeekMode_Preset_Text);
*     _FMTabControlDemo->addItem(TextId_TUNER_FM_Bank_Text1);
*     _FMTabControlDemo->addItem(TextId_TUNER_FM_Bank_Text2);
*     _FMTabControlDemo->addItem(TextId_TUNER_FM_Bank_Text3);
*     _FMTabControlDemo->sendUpdate();
*  }
*/
template<typename TListDataBindingSource, typename TItemData>
class SimpleListModel : public AbstractListModel
{
      typedef AbstractListModel Base;

   public:
      SimpleListModel(unsigned int listId, DataItemContext listItemTemplate)
         : Base(listId, listItemTemplate), _dataBindingSource(), _itemData()
      {
      }

      /** Returns the number of items. */
      virtual size_t getItemCount() const
      {
         return _itemData.size();
      }

      /** Returns the data for the specified index. */
      const TItemData& getItem(size_t index) const
      {
         return _itemData.at(index);
      }

      /** Adds a new item at the end of the list. The notifications should be done from the user code.  */
      void addItem(const TItemData& itemData)
      {
         _itemData.push_back(itemData);
      }

      /** Inserts a new item at the specified position. The notifications should be done from the user code.  */
      void insertItem(size_t index, const TItemData& itemData)
      {
         _itemData.insert(_itemData.begin() + index, itemData);
      }

      /** Removes an item. The notifications should be done from the user code.  */
      void removeItem(size_t index)
      {
         _itemData.erase(_itemData.begin() + index);
      }

      /** Removes all items. The notifications should be done from the user code.  */
      void removeAllItems()
      {
         _itemData.clear();
      }

      /** Returns the data binding source and allows to change various field before sending the data. The fields Count and SelectedIndex are managed internally.  */
      Courier::DataItemContainer<TListDataBindingSource>& getDataBindingSource()
      {
         return _dataBindingSource;
      }

      /** Sends the update of the data binding source and also the list content update message. */
      virtual void sendUpdate()
      {
         (*_dataBindingSource).mCount = getItemCount();
         (*_dataBindingSource).mSelectedIndex = getSelectedItemIndex();
         _dataBindingSource.SendUpdate(true);

         Base::sendUpdate();
      }

   private:
      /** Databinding source which contains a field for the selected item index and for total item count. */
      Courier::DataItemContainer<TListDataBindingSource> _dataBindingSource;
      /** Vector with list item data. */
      std::vector<TItemData> _itemData;
};


template<typename TListDataBindingSource>
class StringListModel : public SimpleListModel<TListDataBindingSource, Candera::String>
{
      typedef SimpleListModel<TListDataBindingSource, Candera::String> Base;

   public:
      using Base::getItem;
      StringListModel(unsigned int listId, DataItemContext listItemTemplate)
         : Base(listId, listItemTemplate)
      {
      }

   protected:
      /** Adds a list item for the specified index. 3 values are added: button id, text and isSelected. */
      virtual void addListItem(ListDataProviderBuilder& listBuilder, size_t index, bool isSelected) const
      {
         const Candera::String& item = getItem(index);
         listBuilder.AddItem(index).AddData(item).AddData(isSelected);
      }
};


template<typename TListDataBindingSource>
class LocStringListModel : public SimpleListModel<TListDataBindingSource, Candera::UInt32>
{
      typedef SimpleListModel<TListDataBindingSource, Candera::UInt32> Base;
      using Base::getItem;

   public:
      LocStringListModel(unsigned int listId, DataItemContext listItemTemplate)
         : Base(listId, listItemTemplate)
      {
      }

   protected:
      /** Adds a list item for the specified index. 3 values are added: button id, text and isSelected. */
      virtual void addListItem(ListDataProviderBuilder& listBuilder, size_t index, bool isSelected) const
      {
         Candera::String item(getItem(index));
         listBuilder.AddItem(index).AddData(item).AddData(isSelected);
      }
};


#endif // _LIST_MODEL_H
