/* ***************************************************************************************
* FILE:          KeyMapping.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  KeyMapping.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "sys_std_if.h"
#include "hmi_trace_if.h"

#include "Courier/Version.h"

#include "BaseContract/generated/BaseMsgs.h"
#include "ProjectBaseTypes.h"
#include "KeyMapping.h"

//////// TRACE IF ///////////////////////////////////////////////////
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW_INPUT
#include "trcGenProj/Header/KeyMapping.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

//project specific key IDs additional to the key IDs defined in Linux input.h
//these IDs are send by the V850 key driver via INC to Linux. First to
//input_inc kernel driver then via evdev to Wayland/ILM

//valid key codes are ((>= 0x300) && (< 0x3ff))
//HMI-Base keys on A-IVI target
//fascia keys // HMI base reference
/* #define KEY_FAC_XM                          0x300
#define KEY_FAC_CD                          0x301
#define KEY_FAC_CAMERA                      0x302
#define KEY_FAC_AUX                         0x303
#define KEY_FAC_NAV                         0x304
#define KEY_FAC_MAP                         0x305
#define KEY_FAC_FMAM                        0x306
#define KEY_FAC_DAY_NIGHT                   0x307
#define KEY_FAC_NEXT                        0x308
#define KEY_FAC_APP                         0x309
#define KEY_FAC_BACK                        0x30a
#define KEY_FAC_PREV                        0x30b
#define KEY_FAC_PHONE                       0x30c
#define KEY_FAC_ENTER                       0x30d
#define KEY_FAC_POWER                       0x30f
#define KEY_FAC_EJECT                       0x310 */

//fascia keys // AIVI reference
#define KEY_FAC_PHONE                    0x300
#define KEY_FAC_CD                       0x301
#define KEY_FAC_CAMERA                   0x302
#define KEY_FAC_AUX                      0x303
#define KEY_FAC_PREV                     0x304
#define KEY_FAC_NEXT                     0x305
#define KEY_FAC_RADIO                    0x306
#define KEY_FAC_INFO                     0x307
#define KEY_FAC_SETUP                    0x308
#define KEY_FAC_MAP                      0x309
#define KEY_FAC_BACK                     0x30A
#define KEY_FAC_DAY_NIGHT                0x30B
#define KEY_FAC_NAV                      0x30C
#define KEY_FAC_ENTER                    0x30D
#define KEY_FAC_XM                       0x30E
#define KEY_FAC_POWER                    0x30F
#define KEY_FAC_EJECT                    0x310
#define KEY_FAC_FMAM                     0x311
#define KEY_FAC_MENU                     0x312
#define KEY_FAC_APP                      0x313
#define KEY_FAC_AUDIO                    0x314
#define KEY_FAC_DISP                     0x315
#define KEY_FAC_VOLUME_UP                0x316
#define KEY_FAC_VOLUME_DOWN              0x317
#define KEY_FAC_VOLUME                   0x318
#define KEY_FAC_HOME                     0x319
#define KEY_FAC_EV                       0x31B
#define KEY_FAC_FOLDER_NEXT              0x31C
#define KEY_FAC_FOLDER_PREV              0x31D
#define KEY_FAC_MUTE                     0x31E
#define KEY_FAC_CLIMATE                  0x31F
#define KEY_FAC_MEX                      0x320
#define KEY_FAC_TRAFFIC_INFO             0x321
#define KEY_FAC_POWER_LONG_DIAG          0x2D2 //RTC1183622


#define KEY_INF4CV_MCP_SRC_COCKPIT	    0x03d0
#define KEY_INF4CV_MCP_NAV	            0x03d1
#define KEY_INF4CV_MCP_MAP	            0x03d2
#define KEY_INF4CV_MCP_SRC_CABIN	    0x03d3
#define KEY_INF4CV_MCP_HOME_CABIN	    0x03d4
#define KEY_INF4CV_MCP_SET_COCKPIT	    0x03d5
#define KEY_INF4CV_MCP_CAM	            0x03d6
#define KEY_INF4CV_MCP_PHONE_ACCEPT	    0x03d7
#define KEY_INF4CV_MCP_PHONE_REJECT	    0x03d8
#define KEY_INF4CV_MCP_MON_CABIN        0x03d9
#define KEY_INF4CV_MCP_SET_CABIN	    0x03dA

#define Encoder_KEY_INF4CV_VOL_CABIN	0x03DC

//#ifndef VARIANT_S_FTR_ENABLE_CABIN_A_SUPPORT
//steering wheel INF4CV
#define KEY_INF4CV_MCP_HOME_COCKPIT	    0x03cf
#define Encoder_KEY_INF4CV_VOL_COCKPIT	0x03DB
#define KEY_SWC_VOLUME_DOWN              0x6A   //0x03DD
#define KEY_SWC_VOLUME_UP                0x6B  //0x03DE
#define KEY_SWC_CALL_PICKUP				 0x68  //0x03E1

//#endif
#define KEY_SWC_CALL_REJECT				0x6C //0x03E2
/*Commenting the PTT Longpress event in SMART since same id used for SWC Demute for CMC
Seperate dynamic library to be created during code merge*/
//#define KEY_SWC_SMART_PTT_LONGPRESS		0x7A

//steering wheel keys
#define KEY_SWC_NEXT                        0x35C
#define KEY_SWC_PREV                        0x360
#define KEY_SWC_ROTARY_ENCODER_BUTTON       0x364
#define KEY_SWC_OK                          0x368
#define KEY_SWC_MENU_BACK                   0x36C
#define KEY_SWC_SWITCH_AUDIO                0x370
#define KEY_SWC_PTT                         0x378
#define KEY_SWC_PHONE                       0x37C
#define KEY_SWC_MENU                        0x380
#define KEY_SWC_MENU_UP                     0x384
#define KEY_SWC_MENU_DOWN                   0x388
#define KEY_SWC_UP                          0x394
#define KEY_SWC_DOWN                        0x398
#define KEY_SWC_VOLUME_BUTTON               0x39C
#define KEY_SWC_ARROW_A_UP                  0x3A0
#define KEY_SWC_ARROW_A_DOWN                0x3A4
#define KEY_SWC_ARROW_B_UP                  0x3A8
#define KEY_SWC_ARROW_B_DOWN                0x3AC
#define KEY_SWC_INIT                        0x3B0

//Additional steering wheel keys CMC Generic variant
#define KEY_SWC_MUTE                        0x71
#define KEY_SWC_DEMUTE                        0x7A

//RemoteControl Keys
#define KEY_REMOTECONTROL_HASH				0X6D
#define KEY_REMOTECONTROL_STAR				0X6E
#define KEY_REMOTECONTROL_VOICE				0X6F
#define KEY_REMOTECONTROL_NUM_9				0X70
#define KEY_REMOTECONTROL_NUM_8				0X72
#define KEY_REMOTECONTROL_NUM_7				0X73
#define KEY_REMOTECONTROL_NUM_6				0X74
#define KEY_REMOTECONTROL_NUM_5				0X75
#define KEY_REMOTECONTROL_NUM_4				0X76
#define KEY_REMOTECONTROL_NUM_3				0X77
#define KEY_REMOTECONTROL_NUM_2				0X78
#define KEY_REMOTECONTROL_NUM_1				0X79
#define KEY_REMOTECONTROL_NUM_0				0X7B
#define KEY_REMOTECONTROL_TRAFFIC			0X7C
#define KEY_REMOTECONTROL_MAP				0X7D
#define KEY_REMOTECONTROL_NAVI				0X7E
#define KEY_REMOTECONTROL_PHONE				0X7F
#define KEY_REMOTECONTROL_MEDIA				0X80
#define KEY_REMOTECONTROL_TUNER				0X81
#define KEY_REMOTECONTROL_PREVIOUS			0X82
#define KEY_REMOTECONTROL_NEXT				0X83
#define KEY_REMOTECONTROL_BACK				0X84
#define KEY_REMOTECONTROL_SETUP				0X85
#define KEY_REMOTECONTROL_HOME				0X86
#define KEY_REMOTECONTROL_VOLUME			0X87
#define KEY_REMOTECONTROL_MUTE				0X88
#define KEY_REMOTECONTROL_VOLUMEDOWN		        0X89
#define KEY_REMOTECONTROL_VOLUMEUP			0X8A
#define KEY_REMOTECONTROL_ONOFF				0X8B

//virtual keys
#define KEY_VIRT_ENG_TESTMODE               0x340
#define KEY_VIRT_SCREENSHOT                 0x341
#define KEY_VIRT_DISPLAY_OFF                0x342

//HMI-Base keys on Suzuki target
#define KEY_SPEECH							0x2FA
//all other keys are taken from linux input.h
//KEY_POWER, KEY_HOME, KEY_BRL_DOT4
/* #ifdef VARIANT_S_FTR_ENABLE_CABIN_A_SUPPORT
#define KEY_INF4CV_MCP_HOME_COCKPIT	    0x03cf //short press home and long press settings
#define Encoder_KEY_INF4CV_VOL_COCKPIT	0x03DB
#define KEY_SWC_VOLUME_DOWN                 0x03DD
#define KEY_SWC_VOLUME_UP                   0x03DE
#define KEY_SWC_CALL_PICKUP					0x03E1
#endif */

// ------------------------------------------------------------------------
KeyMapping::KeyMapping()
   : _mAnyKeyIsAbortKey(false),
     _mRepeatTimeout(200)
{
   CreateKeyMap();
   CreateEncoderMap();
   CreateAbortKeyList();
}


// ------------------------------------------------------------------------
uint32_t KeyMapping::GetHmiKeyCode(uint32_t key, uint32_t userData) const
{
   // userData not used in this implementation
   (void)userData;

   uint32_t unifiedKeyCode = hmibase::app::base::keymapbase::KEY_CODE_INVALID;
   std::map<uint32_t, HkLongPress>::const_iterator itr;

   itr = _mKeyMap.find(key);
   if (itr != _mKeyMap.end())
   {
      unifiedKeyCode = itr->second.key;
   }

   if (unifiedKeyCode == hmibase::app::base::keymapbase::KEY_CODE_INVALID)
   {
      ETG_TRACE_ERR(("KeyMapping::GetHmiKeyCode hmi-keycode not found for orig-keycode '%d'!", key));
   }
   else
   {
      ETG_TRACE_USR4(("KeyMapping::GetHmiKeyCode Mapped orig-keycode: '%d' to hmi-keycode: %d!",
                      key,
                      unifiedKeyCode));
   }

   return unifiedKeyCode;
}


// ------------------------------------------------------------------------
uint32_t KeyMapping::GetOrigKeyCode(uint32_t key, uint32_t userData) const
{
   // userData not used in this implementation
   (void)userData;

   uint32_t origKeyCode = 0;
   std::map<uint32_t, HkLongPress>::const_iterator itr;

   for (itr = _mKeyMap.begin(); itr != _mKeyMap.end(); ++itr)
   {
      if ((itr->second.key) == key)
      {
         origKeyCode = itr->first;
         break;
      }
   }

   if (origKeyCode == 0)
   {
      ETG_TRACE_ERR(("KeyMapping::GetOrigKeyCode orig-keycode not found for hmi-keycode '%d'!", key));
   }
   else
   {
      ETG_TRACE_USR4(("KeyMapping::GetOrigKeyCode Mapped hmi-keycode: '%d' to orig-keycode: %d!",
                      key,
                      origKeyCode));
   }

   return origKeyCode;
}


// ------------------------------------------------------------------------
uint32_t KeyMapping::GetHmiEncoderCode(uint32_t encoder, uint32_t userData) const
{
   // userData not used in this implementation
   (void)userData;

   uint32_t unifiedEncoderCode = hmibase::app::base::keymapbase::ENCODER_CODE_INVALID;
   std::map<uint32_t, uint32_t>::const_iterator itr;

   itr = _mEncoderMap.find(encoder);
   if (itr != _mEncoderMap.end())
   {
      unifiedEncoderCode = itr->second;
   }

   if (unifiedEncoderCode == hmibase::app::base::keymapbase::ENCODER_CODE_INVALID)
   {
      ETG_TRACE_ERR(("KeyMapping::GetHmiEncoderCode hmi-encoder-code not found for orig-encoder-code '%d'!", encoder));
   }
   else
   {
      ETG_TRACE_USR4(("KeyMapping::GetHmiEncoderCode Mapped orig-encoder-code: '%d' to hmi-encoder-code: %d!",
                      encoder,
                      unifiedEncoderCode));
   }
   return unifiedEncoderCode;
}


// ------------------------------------------------------------------------
uint32_t KeyMapping::GetOrigEncoderCode(uint32_t encoder, uint32_t userData) const
{
   // userData not used in this implementation
   (void)userData;

   uint32_t origEncoderCode = 0;
   bool bMappingFound = false;
   std::map<uint32_t, uint32_t>::const_iterator itr;

   for (itr = _mEncoderMap.begin(); itr != _mEncoderMap.end(); ++itr)
   {
      if ((itr->second) == encoder)
      {
         origEncoderCode = itr->first;
         bMappingFound = true;
         break;
      }
   }

   if (!bMappingFound)
   {
      ETG_TRACE_ERR(("KeyMapping::GetOrigEncoderCode orig-encoder-code not found for hmi-encoder-code '%d'!", encoder));
   }
   else
   {
      ETG_TRACE_USR4(("KeyMapping::GetOrigEncoderCode Mapped hmi-encoder-code: '%d' to orig-encoder-code: %d!",
                      encoder,
                      origEncoderCode));
   }
   return origEncoderCode;
}


// ------------------------------------------------------------------------
void KeyMapping::CreateKeyMap()
{
   //HMI-Base keys on A-IVI target
   /*    UpdateMapKey(KEY_FAC_XM, (uint32_t)HARDKEYCODE_HK_XMTUNER, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
      UpdateMapKey(KEY_FAC_CD, (uint32_t)HARDKEYCODE_HK_CD, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
      UpdateMapKey(KEY_FAC_CAMERA, (uint32_t)HARDKEYCODE_HK_CAMERA, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
      UpdateMapKey(KEY_FAC_AUX, (uint32_t)HARDKEYCODE_HK_AUX, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
      UpdateMapKey(KEY_FAC_NAV, (uint32_t)HARDKEYCODE_HK_NAV, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
      UpdateMapKey(KEY_FAC_MAP, (uint32_t)HARDKEYCODE_HK_MAP, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
      UpdateMapKey(KEY_FAC_FMAM, (uint32_t)HARDKEYCODE_HK_FM_AM, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
      UpdateMapKey(KEY_FAC_DAY_NIGHT, (uint32_t)HARDKEYCODE_HK_DAY_NIGHT, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
      UpdateMapKey(KEY_FAC_NEXT, (uint32_t)HARDKEYCODE_HK_NEXT, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
      UpdateMapKey(KEY_FAC_APP, (uint32_t)HARDKEYCODE_HMI_BASE_1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
      UpdateMapKey(KEY_FAC_BACK, (uint32_t)HARDKEYCODE_HMI_BASE_4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
      UpdateMapKey(KEY_FAC_PREV, (uint32_t)HARDKEYCODE_HK_PREVIOUS, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
      UpdateMapKey(KEY_FAC_PHONE, (uint32_t)HARDKEYCODE_HK_PHONE, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
      UpdateMapKey(KEY_FAC_ENTER, (uint32_t)HARDKEYCODE_HK_SELECT, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
      UpdateMapKey(KEY_FAC_POWER, (uint32_t)HARDKEYCODE_HK_POWER, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
      UpdateMapKey(KEY_FAC_EJECT, (uint32_t)HARDKEYCODE_HK_EJECT, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
       */
   // final key mapping on V850
   UpdateMapKey(KEY_FAC_RADIO, (uint32_t)HARDKEYCODE_HK_FM_AM, 500, 3500, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_FAC_FMAM, (uint32_t)HARDKEYCODE_HK_FM_AM, 500, 3500, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_FAC_XM, (uint32_t)HARDKEYCODE_HK_XMTUNER, 500, 3500, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_FAC_CD, (uint32_t)HARDKEYCODE_HK_CD, 500, 3500, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_FAC_AUX, (uint32_t)HARDKEYCODE_HK_AUX, 500, 3500, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_FAC_EJECT, (uint32_t)HARDKEYCODE_HK_EJECT, 500, 3500, 0, 0, 0, 0, 0, 0, 0, 0, false);

   UpdateMapKey(KEY_INF4CV_MCP_PHONE_ACCEPT, (uint32_t)HARDKEYCODE_HK_PHONE_ACCEPT, 600, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_INF4CV_MCP_PHONE_REJECT, (uint32_t)HARDKEYCODE_HK_PHONE_REJECT, 600, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_INF4CV_MCP_NAV, (uint32_t)HARDKEYCODE_HK_NAV, 500, 3500, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_FAC_MENU, (uint32_t)HARDKEYCODE_HK_MENU, 500, 15000, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_FAC_INFO, (uint32_t)HARDKEYCODE_HK_INFO, 500, 3500, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_FAC_APP, (uint32_t)HARDKEYCODE_HK_INFO, 500, 3500, 0, 0, 0, 0, 0, 0, 0, 0, false);
   //UpdateMapKey(KEY_INF4CV_MCP_MAP, (uint32_t)HARDKEYCODE_HK_MAP, 500, 3500, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_INF4CV_MCP_MAP, (uint32_t)HARDKEYCODE_HK_MAP, 500, 3500, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_FAC_BACK, (uint32_t)HARDKEYCODE_HK_ESC, 500, 3500, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_FAC_DAY_NIGHT, (uint32_t)HARDKEYCODE_HK_ILLUM, 500, 3500, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_INF4CV_MCP_CAM, (uint32_t)HARDKEYCODE_HK_CAMERA, 500, 3500, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_FAC_POWER, (uint32_t)HARDKEYCODE_HK_POWER, 2000, 4000, 10000, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_FAC_POWER_LONG_DIAG, (uint32_t)HARDKEYCODE_HK_POWER_LONGPRESS, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_FAC_ENTER, (uint32_t)HARDKEYCODE_HK_SELECT, 500, 3500, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_FAC_NEXT, (uint32_t)HARDKEYCODE_HK_NEXT, 500, 3500, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_FAC_PREV, (uint32_t)HARDKEYCODE_HK_PREVIOUS, 500, 3500, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_FAC_DISP, (uint32_t)HARDKEYCODE_HK_DISP, 500, 3500, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_FAC_AUDIO, (uint32_t)HARDKEYCODE_HK_AUDIO, 500, 15000, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_FAC_VOLUME_UP, (uint32_t)HARDKEYCODE_HK_VOLUME_UP, 500, 3500, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_FAC_VOLUME_DOWN, (uint32_t)HARDKEYCODE_HK_VOLUME_DOWN, 500, 3500, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_FAC_VOLUME, (uint32_t)HARDKEYCODE_HK_VOLUME, 500, 3500, 0, 0, 0, 0, 0, 0, 0, 0, false);

   UpdateMapKey(KEY_INF4CV_MCP_SET_CABIN, (uint32_t)HARDKEYCODE_HK_CABIN_SETTINGS, 500, 3500, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_INF4CV_MCP_HOME_CABIN, (uint32_t)HARDKEYCODE_HK_HOME_CABIN, 500, 3500, 0, 0, 0, 0, 0, 0, 0, 0, false);

   UpdateMapKey(KEY_INF4CV_MCP_MON_CABIN, (uint32_t)HARDKEYCODE_HK_MCP_MON_CABIN, 500, 3500, 0, 0, 0, 0, 0, 0, 0, 0, false);

   UpdateMapKey(KEY_INF4CV_MCP_SRC_COCKPIT, (uint32_t)HARDKEYCODE_HK_SRC_COCKPIT, 500, 3500, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_INF4CV_MCP_SRC_CABIN, (uint32_t)HARDKEYCODE_HK_SRC_CABIN, 500, 3500, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(Encoder_KEY_INF4CV_VOL_COCKPIT, (uint32_t)ENCCODE_LEFT_ENCODER, 500, 3500, 0, 0, 0, 0, 0, 0, 0, 0, false); //Encoder Mute/unmute for cockpit
   UpdateMapKey(Encoder_KEY_INF4CV_VOL_CABIN, (uint32_t)ENCCODE_RIGHT_ENCODER, 500, 3500, 0, 0, 0, 0, 0, 0, 0, 0, false);//Encoder Mute/unmute for cabin

   UpdateMapKey(KEY_INF4CV_MCP_HOME_COCKPIT, (uint32_t)HARDKEYCODE_HK_HOME, 500, 3500, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_INF4CV_MCP_SET_COCKPIT, (uint32_t)HARDKEYCODE_HK_SETTINGS, 500, 3500, 0, 0, 0, 0, 0, 0, 0, 0, false);

   UpdateMapKey(KEY_VIRT_ENG_TESTMODE, (uint32_t)HARDKEYCODE_VIRT_TESTMODE, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_VIRT_SCREENSHOT, (uint32_t)HARDKEYCODE_VIRT_SCREENSHOT, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_VIRT_DISPLAY_OFF, (uint32_t)HARDKEYCODE_VIRT_DISPLAY_OFF, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);

   UpdateMapKey(KEY_SWC_NEXT, (uint32_t)HARDKEYCODE_SWC_NEXT, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_SWC_PREV, (uint32_t)HARDKEYCODE_SWC_PREV, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_SWC_ROTARY_ENCODER_BUTTON, (uint32_t)HARDKEYCODE_SWC_ROTARY_ENCODER_BUTTON, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_SWC_OK, (uint32_t)HARDKEYCODE_SWC_OK, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_SWC_MENU_BACK, (uint32_t)HARDKEYCODE_SWC_MENU_BACK, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_SWC_SWITCH_AUDIO, (uint32_t)HARDKEYCODE_SWC_SWITCH_AUDIO, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_SWC_MUTE, (uint32_t)HARDKEYCODE_SWC_MUTE, 500, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_SWC_DEMUTE, (uint32_t)HARDKEYCODE_SWC_DEMUTE, 500, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_SWC_PTT, (uint32_t)HARDKEYCODE_SWC_PTT, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_SWC_CALL_REJECT, (uint32_t)HARDKEYCODE_SWC_PHONE_REJECT, 600, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   //UpdateMapKey(KEY_SWC_SMART_PTT_LONGPRESS, (uint32_t)HARDKEYCODE_SWC_PHONE_REJECT, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_SWC_CALL_PICKUP, (uint32_t)HARDKEYCODE_SWC_PHONE_PICKUP, 600, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   //UpdateMapKey(KEY_SWC_MENU, (uint32_t)HARDKEYCODE_SWC_MENU, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_SWC_MENU_UP, (uint32_t)HARDKEYCODE_SWC_MENU_UP, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_SWC_MENU_DOWN, (uint32_t)HARDKEYCODE_SWC_MENU_DOWN, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_SWC_VOLUME_UP, (uint32_t)HARDKEYCODE_SWC_VOLUME_UP, 1000, 0, 0, 0, 0, 0, 0, 0, 0, 0, true);
   UpdateMapKey(KEY_SWC_VOLUME_DOWN, (uint32_t)HARDKEYCODE_SWC_VOLUME_DOWN, 1000, 0, 0, 0, 0, 0, 0, 0, 0, 0, true);
   UpdateMapKey(KEY_SWC_UP, (uint32_t)HARDKEYCODE_SWC_UP, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_SWC_DOWN, (uint32_t)HARDKEYCODE_SWC_DOWN, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_SWC_VOLUME_BUTTON, (uint32_t)HARDKEYCODE_SWC_VOLUME_BUTTON, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_SWC_ARROW_A_UP, (uint32_t)HARDKEYCODE_SWC_ARROW_A_UP, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_SWC_ARROW_A_DOWN, (uint32_t)HARDKEYCODE_SWC_ARROW_A_DOWN, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_SWC_ARROW_B_UP, (uint32_t)HARDKEYCODE_SWC_ARROW_B_UP, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_SWC_ARROW_B_DOWN, (uint32_t)HARDKEYCODE_SWC_ARROW_B_DOWN, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_SWC_INIT, (uint32_t)HARDKEYCODE_SWC_INIT, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   //HMI-Base keys on Suzuki target
   UpdateMapKey(KEY_POWER, (uint32_t)HARDKEYCODE_HMI_BASE_2, 1000, 2000, 3000, 4000, 5000, 6000, 7000, 8000, 9000, 10000, true);
   UpdateMapKey(KEY_HOME, (uint32_t)HARDKEYCODE_HMI_BASE_1, 1000, 2000, 3000, 4000, 5000, 6000, 7000, 8000, 9000, 10000, false);
   UpdateMapKey(KEY_SPEECH, (uint32_t)HARDKEYCODE_HMI_BASE_3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_BRL_DOT4, (uint32_t)HARDKEYCODE_HMI_BASE_4, 1000, 2000, 3000, 4000, 5000, 6000, 7000, 8000, 9000, 10000, true);
   //HMI-Base keys on normal PC keyboard
   //for testing purpose for targets without keys
   UpdateMapKey(KEY_LEFTMETA, (uint32_t)HARDKEYCODE_HMI_BASE_1, 1000, 2000, 3000, 4000, 5000, 6000, 7000, 8000, 9000, 10000, false);
   UpdateMapKey(KEY_A, (uint32_t)HARDKEYCODE_HMI_BASE_1, 1000, 2000, 3000, 4000, 5000, 6000, 7000, 8000, 9000, 10000, false);
   UpdateMapKey(KEY_S, (uint32_t)HARDKEYCODE_HMI_BASE_2, 1000, 2000, 3000, 4000, 5000, 6000, 7000, 8000, 9000, 10000, true);
   UpdateMapKey(KEY_D, (uint32_t)HARDKEYCODE_HMI_BASE_3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_F, (uint32_t)HARDKEYCODE_HMI_BASE_4, 1000, 2000, 3000, 4000, 5000, 6000, 7000, 8000, 9000, 10000, true);
   UpdateMapKey(KEY_Z, (uint32_t)HARDKEYCODE_HMI_BASE_5, 1000, 2000, 3000, 4000, 5000, 6000, 7000, 8000, 9000, 10000, true);
   UpdateMapKey(KEY_X, (uint32_t)HARDKEYCODE_HMI_BASE_6, 1000, 2000, 3000, 4000, 5000, 6000, 7000, 8000, 9000, 10000, true);
   UpdateMapKey(KEY_C, (uint32_t)HARDKEYCODE_HMI_BASE_7, 1000, 2000, 3000, 4000, 5000, 6000, 7000, 8000, 9000, 10000, true);
   UpdateMapKey(KEY_V, (uint32_t)HARDKEYCODE_HMI_BASE_8, 1000, 2000, 3000, 4000, 5000, 6000, 7000, 8000, 9000, 10000, true);
   UpdateMapKey(KEY_B, (uint32_t)HARDKEYCODE_HMI_BASE_9, 1000, 2000, 3000, 4000, 5000, 6000, 7000, 8000, 9000, 10000, true);
   UpdateMapKey(KEY_M, (uint32_t)HARDKEYCODE_HMI_BASE_9, 1000, 2000, 3000, 4000, 5000, 6000, 7000, 8000, 9000, 10000, true);

   //RemoteControl Keys
   UpdateMapKey(KEY_REMOTECONTROL_HASH, (uint32_t)HARDKEYCODE_REMOTECONTROL_HASH, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_REMOTECONTROL_STAR, (uint32_t)HARDKEYCODE_REMOTECONTROL_STAR, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_REMOTECONTROL_VOICE, (uint32_t)HARDKEYCODE_REMOTECONTROL_VOICE, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_REMOTECONTROL_NUM_9, (uint32_t)HARDKEYCODE_REMOTECONTROL_NUM_9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_REMOTECONTROL_NUM_8, (uint32_t)HARDKEYCODE_REMOTECONTROL_NUM_8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_REMOTECONTROL_NUM_7, (uint32_t)HARDKEYCODE_REMOTECONTROL_NUM_7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_REMOTECONTROL_NUM_6, (uint32_t)HARDKEYCODE_REMOTECONTROL_NUM_6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_REMOTECONTROL_NUM_5, (uint32_t)HARDKEYCODE_REMOTECONTROL_NUM_5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_REMOTECONTROL_NUM_4, (uint32_t)HARDKEYCODE_REMOTECONTROL_NUM_4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_REMOTECONTROL_NUM_3, (uint32_t)HARDKEYCODE_REMOTECONTROL_NUM_3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_REMOTECONTROL_NUM_2, (uint32_t)HARDKEYCODE_REMOTECONTROL_NUM_2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_REMOTECONTROL_NUM_1, (uint32_t)HARDKEYCODE_REMOTECONTROL_NUM_1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_REMOTECONTROL_NUM_0, (uint32_t)HARDKEYCODE_REMOTECONTROL_NUM_0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_REMOTECONTROL_TRAFFIC, (uint32_t)HARDKEYCODE_REMOTECONTROL_TRAFFIC, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_REMOTECONTROL_MAP, (uint32_t)HARDKEYCODE_REMOTECONTROL_MAP, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_REMOTECONTROL_NAVI, (uint32_t)HARDKEYCODE_REMOTECONTROL_NAVI, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_REMOTECONTROL_PHONE, (uint32_t)HARDKEYCODE_REMOTECONTROL_PHONE, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_REMOTECONTROL_MEDIA, (uint32_t)HARDKEYCODE_REMOTECONTROL_MEDIA, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_REMOTECONTROL_TUNER, (uint32_t)HARDKEYCODE_REMOTECONTROL_TUNER, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_REMOTECONTROL_PREVIOUS, (uint32_t)HARDKEYCODE_REMOTECONTROL_PREVIOUS, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_REMOTECONTROL_NEXT, (uint32_t)HARDKEYCODE_REMOTECONTROL_NEXT, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_REMOTECONTROL_BACK, (uint32_t)HARDKEYCODE_REMOTECONTROL_BACK, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_REMOTECONTROL_SETUP, (uint32_t)HARDKEYCODE_REMOTECONTROL_SETUP, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_REMOTECONTROL_HOME, (uint32_t)HARDKEYCODE_REMOTECONTROL_HOME, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_REMOTECONTROL_VOLUME, (uint32_t)HARDKEYCODE_REMOTECONTROL_VOLUME, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_REMOTECONTROL_MUTE, (uint32_t)HARDKEYCODE_REMOTECONTROL_MUTE, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
   UpdateMapKey(KEY_REMOTECONTROL_VOLUMEDOWN, (uint32_t)HARDKEYCODE_REMOTECONTROL_VOLUMEDOWN, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, true);
   UpdateMapKey(KEY_REMOTECONTROL_VOLUMEUP, (uint32_t)HARDKEYCODE_REMOTECONTROL_VOLUMEUP, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, true);
   UpdateMapKey(KEY_REMOTECONTROL_ONOFF, (uint32_t)HARDKEYCODE_REMOTECONTROL_ONOFF, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);
}


// ------------------------------------------------------------------------
/************************************************
Function 	: CreateEncoderMap
Parameters 	: void
Description : create a map of keys which have direct mapping in V850 and do not have mapping in VD Input

SYSFL/SWFL : Implemented SWFL 2631/SYSFL 4695
************************************************/
void KeyMapping::CreateEncoderMap()
{
   _mEncoderMap[REL_WHEEL] = (uint32_t)ENCCODE_HMI_BASE_1;
   _mEncoderMap[REL_DIAL]  = (uint32_t)ENCCODE_LEFT_ENCODER;

   _mEncoderMap[1] = (uint32_t)ENCCODE_LEFT_ENCODER;//to be checked why it has to be 1 index
   _mEncoderMap[REL_HWHEEL]  = (uint32_t)ENCCODE_LEFT_ENCODER;

   _mEncoderMap[REL_RX]  = (uint32_t)ENCCODE_RIGHT_ENCODER;
}


// ------------------------------------------------------------------------
void KeyMapping::CreateAbortKeyList()
{
   _mAbortKeyList.push_back((uint32_t)HARDKEYCODE_HMI_BASE_2);
   _mAbortKeyList.push_back((uint32_t)HARDKEYCODE_HMI_BASE_4);
}


// ------------------------------------------------------------------------
void KeyMapping::UpdateMapKey(uint32_t OrigKey,
                              uint32_t Appkey,
                              unsigned int long1,
                              unsigned int long2,
                              unsigned int long3,
                              unsigned int long4,
                              unsigned int long5,
                              unsigned int long6,
                              unsigned int long7,
                              unsigned int long8,
                              unsigned int long9,
                              unsigned int long10,
                              bool isRepeatKey)
{
   HkLongPress keyUpdate;
   keyUpdate.key = Appkey;
   keyUpdate.timers.long1 = long1;
   keyUpdate.timers.long2 = long2;
   keyUpdate.timers.long3 = long3;
   keyUpdate.timers.long4 = long4;
   keyUpdate.timers.long5 = long5;
   keyUpdate.timers.long6 = long6;
   keyUpdate.timers.long7 = long7;
   keyUpdate.timers.long8 = long8;
   keyUpdate.timers.long9 = long9;
   keyUpdate.timers.long10 = long10;
   keyUpdate.isRepeatKey = isRepeatKey;
   _mKeyMap[OrigKey] = keyUpdate;
}


// ------------------------------------------------------------------------
KeyMappingBase::HkTimers KeyMapping::GetTimerValuesForKey(uint32_t key, uint32_t userData) const
{
   // userData not used in this implementation
   (void)userData;

   HkTimers lpTimers;
   memset(&lpTimers, 0, sizeof(lpTimers));
   bool keyFound = false;
   std::map<uint32_t, HkLongPress>::const_iterator itr;

   for (itr = _mKeyMap.begin(); itr != _mKeyMap.end(); ++itr)
   {
      if ((itr->second.key) == key)
      {
         lpTimers = itr->second.timers;
         keyFound = true;
         break;
      }
   }

   if (keyFound)
   {
      ETG_TRACE_USR4(("KeyMapping::GetTimerValuesForKey key %d found with these timer values: Long1 = %d, Long2 = %d, Long3 = %d, Long4 = %d, Long5 = %d.",
                      key, lpTimers.long1, lpTimers.long2, lpTimers.long3, lpTimers.long4, lpTimers.long5));
      ETG_TRACE_USR4(("KeyMapping::GetTimerValuesForKey key %d found with these timer values: Long6 = %d, Long7 = %d, Long8 = %d, Long9 = %d, Long10 = %d.",
                      key, lpTimers.long6, lpTimers.long7, lpTimers.long8, lpTimers.long9, lpTimers.long10));
   }
   else
   {
      ETG_TRACE_ERR(("KeyMapping::GetTimerValuesForKey key not found '%d'!", key));
   }

   return lpTimers;
}


// ------------------------------------------------------------------------
unsigned int KeyMapping::GetRepeatTimeoutForKey(uint32_t key, uint32_t userData) const
{
   // userData not used in this implementation
   (void)userData;

   unsigned int repeatTimeout = 0;
   bool keyFound = false;
   bool isRepeatKey = false;
   std::map<uint32_t, HkLongPress>::const_iterator itr;

   for (itr = _mKeyMap.begin(); itr != _mKeyMap.end(); ++itr)
   {
      if ((itr->second.key) == key)
      {
         if (itr->second.isRepeatKey)
         {
            //yes, this is a repeat key
            repeatTimeout = _mRepeatTimeout;
            isRepeatKey = true;
         }
         keyFound = true;
         break;
      }
   }

   if (keyFound)
   {
      if (isRepeatKey)
      {
         ETG_TRACE_USR4(("KeyMapping::GetRepeatTimeoutForKey key %d is a repeat key with this timer value: %d.",
                         key, repeatTimeout));
      }
      else
      {
         ETG_TRACE_USR4(("KeyMapping::GetRepeatTimeoutForKey key %d is no repeat key.",
                         key));
      }
   }
   else
   {
      ETG_TRACE_ERR(("KeyMapping::GetRepeatTimeoutForKey key not found '%d'!", key));
   }

   return repeatTimeout;
}


// ------------------------------------------------------------------------
bool KeyMapping::IsAbortKey(uint32_t key, uint32_t userData) const
{
   // userData not used in this implementation
   (void)userData;
   bool bReturn = false;

   if (_mAnyKeyIsAbortKey)
   {
      //every key shell do an abort, independent of the key code
      bReturn = true;
   }
   else
   {
      //only specific keys are abort keys
      if (key != hmibase::app::base::keymapbase::KEY_CODE_INVALID)
      {
         for (unsigned int i = 0; i < _mAbortKeyList.size(); ++i)
         {
            if ((_mAbortKeyList[i]) == key)
            {
               bReturn = true;
               break;
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("KeyMapping::IsAbortKey invalid key code: '%d'!",
                        key));
      }
   }
   return bReturn;
}
