/**************************************************************************************
* @file         : TunerDataPoolConfig.cpp
* @author       : ECV2-AGN7KOR
* @addtogroup   : AppHmi_Tuner
* @brief        :
* @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "hall_std_if.h"

#define DATAPOOL_WRITE_DISABLED 1

#ifdef DP_DATAPOOL_ID
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_hmi_02_if.h"
#include "dp_generic_if.h"
#include "dp_tclDefault.h"
#endif

#include "TunerDataPoolConfig.h"
#include "App/Core/Tuner_Defines.h" //@todo: remove once settings for scope2 is available.

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_TUNER_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_TUNER
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_TUNER_"
#define ETG_I_FILE_PREFIX                 App::Core::TunerHall::
#include "trcGenProj/Header/TunerDataPoolConfig.cpp.trc.h"
#endif

// Instantiate the static class object
TunerDataPoolConfig* TunerDataPoolConfig::_DpTuner = NULL;


/**
 * Description     : Constructor of TunerDataPoolConfig
 *
 *
 * @param[in]      : none
 * @param[in]      : None
 * @return         : void
 */
TunerDataPoolConfig::TunerDataPoolConfig()
   : _regionType(0),
     _DABMounted(0),
     _DABMountedSystem(0),
     _AMActivatedValue(0),
     _SXMMounted(0),
     _testModeValue(0),
     _BlockTestMode(1)
{
   DP_vCreateDatapool();
   /* Read KDS for Region type*/
   readKdsForRegion();
   readDABMounted();
   readDABMountedSystem();
   readAMActivation();
   readSXMMounted();
   readTestmode();
}


/**
 * Description     : setLanguage, sets the language if there is a change
 *
 * @param[in]      : LanguageIndexTuner : Language Id
 * @param[in]      : LanguageIndexTuner : uint8
 * @return         : void
 */
void TunerDataPoolConfig::setLanguage(const uint8 LanguageIndexTuner)
{
   _DpLanguage.vSetData(LanguageIndexTuner);
   ETG_TRACE_USR4(("TunerDataPoolConfig::setLanguage  :%d", LanguageIndexTuner));
}


/**
 * Description     : getLanguage, reads the language id on startup
 *
 * @param[in]      : none
 * @param[in]      : none
 * @return         : uint8
 */
uint8 TunerDataPoolConfig::getLanguage()
{
   uint8 languageStatus = static_cast<uint8>(_DpLanguage.tGetData());
   ETG_TRACE_USR4(("TunerDataPoolConfig::getLanguage :%d", languageStatus));
   return languageStatus;
}


/**
 * Description     : function called to get radio text on/off from datapool
 *
 * @param[in]      : none
 * @return         : bool
 */
bool TunerDataPoolConfig::getRadioTextStatus()
{
   dp_tclAppHmi_TunerDPRadiotext _mDPradioText;
   ETG_TRACE_USR4(("TunerDataPoolConfig::getRadioTextStatus() %d", _mDPradioText.tGetData()));
   return _mDPradioText.tGetData();
}


/**
 * Description     : function called to set radio text on/off to datapool
 *
 * @param[in]      : radioTextStatus
 * @return         : void
 */
void TunerDataPoolConfig::setRadioTextStatus(bool radioTextStatus)
{
   ETG_TRACE_USR4(("TunerDataPoolConfig::setRadioTextStatus() %d", radioTextStatus));
   dp_tclAppHmi_TunerDPRadiotext _mDPradioText;
   _mDPradioText.s32SetData(radioTextStatus);
}


/**
 * Description     : getInstance, returns the instance of the class
 *                   TunerDataPoolConfig
 * @param[in]      : none
 * @param[in]      : none
 * @return         : TunerDataPoolConfig*
 */
TunerDataPoolConfig* TunerDataPoolConfig::getInstance()
{
   if (_DpTuner == NULL)
   {
      _DpTuner = new TunerDataPoolConfig();
   }
   return _DpTuner;
}


/**
 * Description     : Destructor of TunerDataPoolConfig
 *
 * @param[in]      : none
 * @param[in]      : None
 * @return         : void
 */
TunerDataPoolConfig::~TunerDataPoolConfig()
{
   if (_DpTuner != NULL)
   {
      delete _DpTuner;
   }
   _DpTuner = NULL;
}


/**
 * Description     : readKdsForRegion, reads the KDS for region type
 *                   and stores on a class member
 * @param[in]      : none
 * @param[in]      : None
 * @return         : void
 */
void TunerDataPoolConfig::readKdsForRegion()
{
   dp_tclKdsVehicleInformation _KdsVehicleInformationInst;
   _KdsVehicleInformationInst.u8GetDestinationRegion1(_regionType);
}


/**
* Description     : readDABMounted, reads whether DAB is mounted / not
*                   and stores on a class member
* @param[in]      : none
* @param[in]      : None
* @return         : void
*/
void TunerDataPoolConfig::readDABMounted()
{
   dp_tclKdsCMVariantCoding _DABMountedInformation;
   _DABMountedInformation.u8GetDAB(_DABMounted);
}


/**
* Description     : readDABMounted, reads whether DAB(system) is mounted / not
*                   and stores on a class member
* @param[in]      : none
* @param[in]      : None
* @return         : void
*/
void TunerDataPoolConfig::readDABMountedSystem()
{
   dp_tclKdsSystemConfiguration1 _DABMountedInformationSystem;
   _DABMountedInformationSystem.u8GetDAB(_DABMountedSystem);
}


/**
* Description     : readDABMounted, reads whether AM(system) is activated / not
*                   and stores on a class member
* @param[in]      : none
* @param[in]      : None
* @return         : void
*/
void TunerDataPoolConfig::readAMActivation()
{
   dp_tclKdsSystemConfiguration1 _AMActivatedInformationSystem;
   _AMActivatedInformationSystem.u8GetAMTunerActivation(_AMActivatedValue);
}


/**
* Description     : readRegionType, provides whether AM is activated or not
*
* @param[in]      : none
* @param[in]      : None
* @return         : (_AMActivatedValue) uint8
*/
uint8 TunerDataPoolConfig::readAMActivatedValue()
{
   return _AMActivatedValue;
}


/**
* Description     : readRegionType, provides whether DAB id mounted or not
*
* @param[in]      : none
* @param[in]      : None
* @return         : (_DABMounted) uint8
*/
uint8 TunerDataPoolConfig::readDABMountedValue()
{
   return _DABMounted;
}


/**
* Description     : readDABMountedSystemValue, provides whether DAB(system) id mounted or not
*
* @param[in]      : none
* @param[in]      : None
* @return         : (_DABMounted) uint8
*/
uint8 TunerDataPoolConfig::readDABMountedSystemValue()
{
   return _DABMountedSystem;
}


/**
 * Description     : readRegionType, provides the region type to other classes
 *
 * @param[in]      : none
 * @param[in]      : None
 * @return         : (_regionType) uint8
 */
uint8 TunerDataPoolConfig::readRegionType()
{
   return _regionType;
}


void TunerDataPoolConfig::vSetDabDynamicContentStatus(bool bOptStatus)
{
   ETG_TRACE_USR3(("tclTuunerAppLogic::vSetDabDynamicContentStatus(%d)", bOptStatus));
   dp_tclAppHmi_TunerDAB_DynamicText obj;
   tS32 s32Status = obj.s32SetData(bOptStatus);
   ETG_TRACE_USR3(("s32Status(%d)", (tU32)s32Status));
}


void TunerDataPoolConfig::vSetDabFmFollowingStatus(bool bOptStatus)
{
   ETG_TRACE_USR3(("tclTuunerAppLogic::vSetDabFmFollowingStatus(%d)", bOptStatus));
   dp_tclAppHmi_TunerDAB_FMFollowing obj;
   tS32 s32Status = obj.s32SetData(bOptStatus);
   ETG_TRACE_USR3(("s32Status(%d)", (tU32)s32Status));
}


void TunerDataPoolConfig::vSetDabDabFollowingStatus(bool bOptStatus)
{
   ETG_TRACE_USR3(("tclTuunerAppLogic::vSetDabDabFollowingStatus(%d)", bOptStatus));
#ifndef DATAPOOL_WRITE_DISABLED
   dp_tclAppHmi_TunerDAB_DABLinking obj;
   tS32 s32Status = obj.s32SetData(bOptStatus);
   ETG_TRACE_USR3(("s32Status(%d)", (tU32)s32Status));
#endif
}


bool TunerDataPoolConfig::isNARVariant()
{
   return ((readRegionType() == NAM_REGION)
           || (readRegionType() == CAN_REGION)
           || (readRegionType() == MEX_REGION));
}


/**
* Description     : readSXMMounted, reads whether SXM is mounted / not
*                   and stores on a class member
* @param[in]      : none
* @param[in]      : None
* @return         : void
*/
void TunerDataPoolConfig::readSXMMounted()
{
   dp_tclKdsCMVariantCoding _SXMMountedInformation;
   _SXMMountedInformation.u8GetXMTuner(_SXMMounted);
}


/**
* Description     : readRegionType, provides whether SXM id mounted or not
*
* @param[in]      : none
* @param[in]      : None
* @return         : (_SXMMounted) uint8
*/
uint8 TunerDataPoolConfig::readSXMMountedValue()
{
   return _SXMMounted;
}


/**
* Description     : readTestmodeMounted, reads whether Testmode Screen is blocked / not
*                   and stores on a class member
* @param[in]      : none
* @param[in]      : None
* @return         : void
*/
void TunerDataPoolConfig::readTestmode()
{
   // tU8 val = 0;
   //_testModeValue = &val;
   // dp_tclDefaultBlockingTestModeApplication _TestmodeBlockInformation;
   // _TestmodeBlockInformation.s32GetData(_testModeValue, 1);

   dp_tclKdsBlockingTestMode _blockTestmode;
   _blockTestmode.u8GetBlockingTestModeApplicationSwitch(_BlockTestMode);
   ETG_TRACE_USR4(("TunerDataPoolConfig::u8GetBlockingTestModeApplicationSwitch  :%d", _BlockTestMode));
}


/**
* Description     : readRegionType, provides whether Testmode Screen is blocked / not
*
* @param[in]      : none
* @param[in]      : None
* @return         : ( _testModeValue) tS32
*/
tU8 TunerDataPoolConfig::readTestmodeValue()
{
   //return * _testModeValue;
   ETG_TRACE_USR4(("TunerDataPoolConfig::u8GetBlockingTestModeApplicationSwitch reading :%d", _BlockTestMode));
   return _BlockTestMode;
}
