/**************************************************************************************
* @file         : WeatherBand.cpp
* @author       : ECV2-Priyanka B
* @addtogroup   : Tuner HALL
* @brief        : Class contains necessary business logic required to handle AM FM Testmode data.
* @copyright    : (c) 2019-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "hall_std_if.h"
#include "WeatherBandMain.h"
#include "App/Core/Utility/TunerUtility.h"
#include "CourierTunnelService/CourierMessageReceiverStub.h"
#include "MIDW_TUNER_FIProxy.h"
#include "TunerDataBindingHandler.h"
#include "Tuner_Defines.h"
#include "App/Core/LanguageDefines.h"
#include "AppHmi_MasterBase/AudioInterface/AudioDefines.h"
#include "../VehicleSrvClient/VehicleSrvClient.h"

#define INITIALISE_SUBSRC_ID 0
#define  DIVIDE_BY_THOUSAND 1000
#define TUN_BAND_WB 18

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_TUNER_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_TUNER
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_Tuner_"
#define ETG_I_FILE_PREFIX                 App::Core::WeatherBandMain::
#include "trcGenProj/Header/WeatherBandMain.cpp.trc.h"
#endif

#ifdef DP_DATAPOOL_ID
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"
#endif


using namespace ::MIDW_TUNER_FI;
using namespace ::MIDW_TUNERMASTER_FI;
using namespace ::midw_tuner_fi_types;
using namespace ::midw_tunermaster_fi_types;
//using namespace ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange;
using namespace ::bosch::cm::ai::hmi::masteraudioservice::SoundProperties;

namespace App {
namespace Core {
DEFINE_CLASS_LOGGER_AND_LEVEL("App/Core/WeatherBand", WeatherBandMain, Info);

WeatherBandMain* WeatherBandMain::_WeatherBandMainInst = NULL;

struct WeatherBandData
{
   uint32 long frequency;
   std::string officialName;
   uint32 prestno;
};


WeatherBandData WBD[10] =
{
   { 162400, "WX2 36B PAC :1", 1 },
   { 162425, "WX4 96B PAC :2", 2 },
   { 162450, "WX5 37B PAC :3", 3 },
   { 162475, "WX3 97B PAC :4", 4 },
   { 162500, "WX6 38B PAC :5", 5 },
   { 162525, "WX7 98B PAC :6", 6 },
   { 162550, "WX1 39B PAC :7", 7 },
   { 161650, "WX8 21B", 8 },
   { 161775, "WX9 83B", 9 },
   { 163275, "WX10 113B", 10 }
};


/**
* Description     : Destructor of class WeatherBandMain
*/
WeatherBandMain::~WeatherBandMain()
{
   _tunerMasterFiProxy.reset();
   _tunerFiProxy.reset();

   ETG_I_UNREGISTER_FILE();
}


/**
* Description     : Constructor of class WeatherBandMain
*/
WeatherBandMain::WeatherBandMain()
   : _tunerFiProxy(MIDW_TUNER_FIProxy::createProxy("tunerFiPort", *this)),
     _tunerMasterFiProxy(MIDW_TUNERMASTER_FIProxy::createProxy("tunermasterFiPort", *this)),
     _soundPropertiesProxy(SoundPropertiesProxy::createProxy("soundPropertiesPort", *this))


{
   if (_tunerFiProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _tunerFiProxy->getPortName());
   }
   if (_tunerMasterFiProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _tunerMasterFiProxy->getPortName());
   }
   //_audioSrcChgProxy(AudioSourceChangeProxy::createProxy("audioSourceChangePort", *this))
   //if (_audioSrcChgProxy.get())
   //{
   //    StartupSync::getInstance().registerPropertyRegistrationIF(this, _audioSrcChgProxy->getPortName());
   //    //ETG_TRACE_USR4(("audioSourceChangePort proxy created.......%d", _audioSrcChgProxy));
   //}
   if (_soundPropertiesProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _soundPropertiesProxy->getPortName());
   }
   _WeatherBandMainInst = this;
   m_wxFrequencyvalue = "";
   m_WBFrequency = 0;
   m_wxPACvalue = "";

   ETG_I_REGISTER_FILE();
}


/**
* Description     : Request from Framework to start registering for properties. This request means GUI is ready now.
*
* @param[in]      : proxy : reference to the base class of all proxies
* @param[in]      : stateChange : represents a state transition of a service
* @return         : void
*/
void WeatherBandMain::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   if ((proxy == _tunerFiProxy) && (_tunerFiProxy.get()))
   {
      _tunerFiProxy->sendFID_TUN_G_ATSTATIONINFOUpReg(*this);
   }
   /* if ((proxy == _audioSrcChgProxy) && (_audioSrcChgProxy.get()))
    {
        _audioSrcChgProxy->sendActiveSourceGet(*this);
        _audioSrcChgProxy->sendActiveSourceListRegister(*this);
    }*/
   if ((proxy == _soundPropertiesProxy) && (_soundPropertiesProxy.get()))
   {
      _soundPropertiesProxy->sendMuteStateRegister(*this);
   }
}


/**
* Description     : Request from Framework to start de-registering for properties.
*
* @param[in]      : proxy : reference to the base class of all proxies
* @param[in]      : stateChange : represents a state transition of a service
* @return         : void
*/
void WeatherBandMain::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   if ((proxy == _tunerFiProxy) && (_tunerFiProxy.get()))
   {
      _tunerFiProxy->sendFID_TUN_G_ATSTATIONINFORelUpRegAll();
   }
   /*if (_audioSrcChgProxy && _audioSrcChgProxy == proxy)
   {
       _audioSrcChgProxy->sendDeregisterAll();
   }*/
   else if (_soundPropertiesProxy && _soundPropertiesProxy == proxy)
   {
      _soundPropertiesProxy->sendVolumeDeregisterAll();
      _soundPropertiesProxy->sendMuteStateDeregisterAll();
   }
}


void WeatherBandMain::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   if ((proxy == _tunerFiProxy) && (_tunerFiProxy.get()))
   {
      _tunerFiProxy->sendFID_TUN_G_ATSTATIONINFORelUpRegAll();
   }
   /*if ((proxy == _audioSrcChgProxy) && (_audioSrcChgProxy.get()))
   {
       _audioSrcChgProxy->sendActiveSourceDeregisterAll();
       _audioSrcChgProxy->sendActiveSourceListDeregisterAll();
   _audioSrcChgProxy->sendActiveSourceDeregisterAll();
   }*/
   else if (_soundPropertiesProxy && _soundPropertiesProxy == proxy)
   {
      _soundPropertiesProxy->sendVolumeDeregisterAll();
      _soundPropertiesProxy->sendMuteStateDeregisterAll();
   }
}


void WeatherBandMain::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("WeatherBandMain : onAvailable"));
   if ((proxy == _tunerFiProxy) && (_tunerFiProxy.get()))
   {
      _tunerFiProxy->sendFID_TUN_G_ATSTATIONINFOUpReg(*this);
   }
   /*if ((proxy == _audioSrcChgProxy) && (_audioSrcChgProxy.get()))
   {
       ETG_TRACE_USR4(("WeatherBandMain::onAvailable Sucess audioSrcChgProxy"));
       _audioSrcChgProxy->sendActiveSourceGet(*this);
       _audioSrcChgProxy->sendActiveSourceListRegister(*this);
   _audioSrcChgProxy->sendActiveSourceRegister(*this);
   }*/
   if ((proxy == _soundPropertiesProxy) && (_soundPropertiesProxy.get()))
   {
      _soundPropertiesProxy->sendMuteStateRegister(*this);
   }
}


//void WeatherBandMain::onActivateSourceError(const ::boost::shared_ptr< ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
//    const ::boost::shared_ptr< ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange::ActivateSourceError >& /*error*/)
//{
//    ETG_TRACE_USR4(("WeatherBandMain::onActivateSourceError"));
//}

//void WeatherBandMain::onActivateSourceResponse(const ::boost::shared_ptr< ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
//    const ::boost::shared_ptr< ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange::ActivateSourceResponse >& /*response*/)
//{
//    ETG_TRACE_USR4(("WeatherBandMain::onActivateSourceResponse"));
//}

//void WeatherBandMain::onActiveSourceError(const ::boost::shared_ptr< ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
//    const ::boost::shared_ptr< ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange::ActiveSourceError >& /*error*/)
//{
//    ETG_TRACE_USR4(("WeatherBandMain::onActiveSourceError"));
//}

//void WeatherBandMain::onActiveSourceUpdate(const ::boost::shared_ptr< ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
//    const ::boost::shared_ptr< ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange::ActiveSourceUpdate >& update)
//{
//    ETG_TRACE_USR4(("WeatherBandMain::onActiveSourceUpdate"));
//}

//void WeatherBandMain::onActiveSourceListError(const ::boost::shared_ptr< ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
//    const ::boost::shared_ptr< ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange::ActiveSourceListError >& /*error*/)
//{
//    ETG_TRACE_USR4(("WeatherBandMain::onActiveSourceListError"));
//}

//void WeatherBandMain::onActiveSourceListUpdate(const ::boost::shared_ptr< ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
//    const ::boost::shared_ptr< ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange::ActiveSourceListUpdate >& update)
//{
//    ETG_TRACE_USR4(("WeatherBandMain::onActiveSourceListUpdate"));
//}

//////////////////////////////
void WeatherBandMain::onFID_TUN_S_DIRECT_FREQUENCYError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& proxy, const boost::shared_ptr< FID_TUN_S_DIRECT_FREQUENCYError >& error)
{
   ETG_TRACE_USR3(("WeatherBandMain::Direct_Frequency_Error()"));
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void WeatherBandMain::onFID_TUN_S_DIRECT_FREQUENCYResult(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& proxy, const boost::shared_ptr< FID_TUN_S_DIRECT_FREQUENCYResult >& result)
{
}


/**
* Description     : Virtual function implemented to get update of StationInfo Error
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
* @param[in]      : error : The error message of 'StationInfo'
* @return         : void
*/
void WeatherBandMain::onFID_TUN_G_ATSTATIONINFOError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUN_G_ATSTATIONINFOError >& /*error*/)
{
}


void WeatherBandMain::onMuteStateError(const ::boost::shared_ptr< SoundPropertiesProxy >& /*proxy*/, const ::boost::shared_ptr< MuteStateError >& /*error*/)
{
   ETG_TRACE_USR4((" WeatherBandMain::onMuteStateError"));
}


/**
* onMuteStateUpdate - This status is received from HMI_Master if there is any change in property mutestate.
* @param[in] proxy
* @parm[in] status
* @return void
*/

void WeatherBandMain::onMuteStateUpdate(const ::boost::shared_ptr< SoundPropertiesProxy >& /*proxy*/, const ::boost::shared_ptr< MuteStateUpdate >& update)
{
   bool bMuteStatus = update->getMuteState();
   ETG_TRACE_USR4((" WeatherBandMain::onMuteStateUpdate"));
}


/**
* Description     : Virtual function implemented to get update of StationInfo Status
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
* @param[in]      : status : The status message of the property "StationInfo"
* @return         : void
*/
void WeatherBandMain::onFID_TUN_G_ATSTATIONINFOStatus(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUN_G_ATSTATIONINFOStatus >& status)
{
   ETG_TRACE_USR3(("WeatherBandMain: onFID_TUN_G_ATSTATIONINFOStatus"));
   std::string freqStr = "";
   tChar szFrequency[M_FREQ_STR_BUFFER_SIZE] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
   tU8 u8CurrentBand = status->getAtstationinfo().getE8Band();
   ETG_TRACE_USR3(("WeatherBandMain: u8CurrentBand %d", u8CurrentBand));
   uint32 currentFreq = status->getAtstationinfo().getU32Frequency();
   if (u8CurrentBand == TUN_BAND_WB)
   {
      for (int i = 0; i < sizeof(WBD) / sizeof(WBD[0]); i++)
      {
         if (currentFreq == WBD[i].frequency)
         {
            update_WB_PresetFocused(i);
            freqStr = formatTestmodeFrequency(currentFreq);
            (*oPresetInfoWB).mWBFrequency = freqStr.c_str();
            (*oPresetInfoWB).mWBmode = WBD[i].officialName.c_str();
            m_wxFrequencyvalue = freqStr;
            m_wxPACvalue = WBD[i].officialName;
            TunerDataBindingHandler::getDataBindingHandlerInstance()->WXHomeScreen();
            oPresetInfoWB.MarkAllItemsModified();
            oPresetInfoWB.SendUpdate();
         }
      }
   }
}


std::string WeatherBandMain::getfreqvalue()
{
   return m_wxFrequencyvalue;
}


std::string WeatherBandMain::getPacvalue()
{
   return m_wxPACvalue;
}


std::string WeatherBandMain::formatTestmodeFrequency(const uint32& frequency)
{
   char freqString[ARRAY_SIZE] = "\0";
   int mhz = frequency / DIVIDE_BY_THOUSAND;
   int fraction = 0;
   fraction = (frequency % DIVIDE_BY_THOUSAND);
   snprintf(freqString, sizeof freqString, "%d.%02d", mhz, fraction);
   return freqString;
}


bool WeatherBandMain::update_WB_PresetFocused(uint16 PresetNumber)
{
   ETG_TRACE_USR3(("WeatherBandMain::update_WB_PresetFocused(%d)", PresetNumber));
   VehicleSrvClient::poGetInstance()->updateTunerPresetStatus(PresetNumber); //CAN Imp
   bool* pPresetState[] =
   {
      &((*oWBPresetInfo).mPreset_1_Active),
      &((*oWBPresetInfo).mPreset_2_Active),
      &((*oWBPresetInfo).mPreset_3_Active),
      &((*oWBPresetInfo).mPreset_4_Active),
      &((*oWBPresetInfo).mPreset_5_Active),
      &((*oWBPresetInfo).mPreset_6_Active),
      &((*oWBPresetInfo).mPreset_7_Active),
      &((*oWBPresetInfo).mPreset_8_Active),
      &((*oWBPresetInfo).mPreset_9_Active),
      &((*oWBPresetInfo).mPreset_10_Active)
   };
   //clear the presets
   for (tU8 u8Index = 0; u8Index < (sizeof(pPresetState) / sizeof(pPresetState[0])); ++u8Index)
   {
      *(pPresetState[u8Index]) = false;
   }
   // if ((PresetNumber >= 0) && (PresetNumber < 10)) //coverity issue
   if (PresetNumber < 10)
   {
      *(pPresetState[PresetNumber]) = true;
   }

   oWBPresetInfo.MarkAllItemsModified();
   oWBPresetInfo.SendUpdate();
   return true;
}


uint32 WeatherBandMain::getFrequencyFromTable(int index)
{
   m_WBFrequency = WBD[index].frequency;
   return m_WBFrequency;
}


bool WeatherBandMain::onCourierMessage(const Tuner_WxBLMsg& corfoMsg)
{
   uint32 value;
   ETG_TRACE_USR3(("WeatherBandMain::Tuner_WxBLMsg()"));
   bool bMsgConsumed = false; // Holds true if Model consumes the message
   const Tuner_WxBLMsg* poTunerBLMsg = Courier::message_cast<const Tuner_WxBLMsg*>(&corfoMsg);

   ETG_TRACE_USR4(("WeatherBandMain::Tuner_WxBLMsg( %d)", poTunerBLMsg->GetEventId()));
   value = poTunerBLMsg->GetEventId();
   m_WBFrequency = getFrequencyFromTable(value - 1);
   if (_tunerFiProxy != NULL)
   {
      _tunerFiProxy->sendFID_TUN_S_DIRECT_FREQUENCYStart(*this, m_WBFrequency, (::midw_tuner_fi_types::T_e8_Tun_TunerBand)TUN_BAND_WB /*...T_e8_Tun_TunerBand__TUN_BAND_WB */);   //Band should be Wx
   }

   return true;
}


void WeatherBandMain::presetData()
{
   (*oPresetInfoWB).mWBName1 = "162.400";
   (*oPresetInfoWB).mWBName2 = "162.425";
   (*oPresetInfoWB).mWBName3 = "162.450";
   (*oPresetInfoWB).mWBName4 = "162.475";
   (*oPresetInfoWB).mWBName5 = "162.500";
   (*oPresetInfoWB).mWBName6 = "162.525";
   (*oPresetInfoWB).mWBName7 = "162.550";
   (*oPresetInfoWB).mWBName8 = "161.650";
   (*oPresetInfoWB).mWBName9 = "161.775";
   (*oPresetInfoWB).mWBName10 = "163.275";
   oPresetInfoWB.MarkAllItemsModified();
   oPresetInfoWB.SendUpdate();
}


ETG_I_CMD_DEFINE((TraceCmd_onPresetData, "onPresetData"))
void WeatherBandMain::TraceCmd_onPresetData()
{
   ETG_TRACE_USR4(("WeatherBandMain::TraceCmd_onPresetData"));
   if (_WeatherBandMainInst != NULL)
   {
      _WeatherBandMainInst->presetData();
   }
}


}
} // namespace Core
// namespace App
