/* ***************************************************************************************
* FILE:          TunerHall.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TunerHall.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2017-2020 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "hall_std_if.h"
#include "TunerHall.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_TUNER_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_TUNER
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_TUNER_"
#define ETG_I_FILE_PREFIX                 App::Core::TunerHall::
#include "trcGenProj/Header/TunerHall.cpp.trc.h"
#endif

//using namespace ::App::Core;
using namespace ::CourierTunnelService::CourierMessageReceiver;


namespace App {
namespace Core {

DEFINE_CLASS_LOGGER_AND_LEVEL("App/Core", TunerHall, Info);

TunerHall::TunerHall()
   : HallComponentBase("", "App.Core.AppHmi_Tuner", "/org/genivi/NodeStateManager/LifeCycleConsumer/AppHmi_Tuner"),
     _AmFmInfo(NULL),
     _presetList(NULL),
     _srcChg(NULL),
     _tunerDataBindingHandler(NULL),
     _AnnouncementHandler(NULL),
     _ATStationInfo(NULL),
     _AMFMStationList(NULL),
     _RadioSetting(NULL),
     _DABStationList(NULL),
     _VehicleSrvClient(NULL),
     _WeatherBandMain(NULL),
     _DabInfo(NULL),
     _AMFMTestMode(NULL),
     _DiagnosticsClientHandler(NULL)
{
   ETG_TRACE_USR4(("TunerHall : Constructor"));
   ETG_I_REGISTER_FILE();

   _DiagnosticsClientHandler = DiagnosticsClientHandler::poGetInstance();
   _tunerDataBindingHandler = TunerDataBindingHandler::getDataBindingHandlerInstance();
   _srcChg = SrcChangeHandler::poGetInstance();
   _DabInfo = DabInfo::poGetInstance();
   _DABStationList = DABStationList::poGetInstance();
   _presetList = PresetListHandling::poGetInstance();
   _ATStationInfo = ATStationInfo::poGetInstance();
   _AMFMStationList = AMFMStationList::poGetInstance();
   _RadioSetting = RadioSetting::poGetInstance();
   _AnnouncementHandler = AnnoHandler::poGetInstance();
   _AmFmInfo = AMFMInfo::poGetInstance();
   _VehicleSrvClient = VehicleSrvClient::poGetInstance();
   _WeatherBandMain = WeatherBandMain::poGetInstance();
   _applicationSwitchClientComponent = ApplicationSwitchClientComponent::poGetInstance();
   _AMFMTestMode = AMFMTestMode::poGetInstance();
   (void)NavigationModeDBHandler::getInstance();
   //for factory reset
   DefSetServiceBase::s_Intialize("AppHmiTunerServicePort");//port name needs to be same as port name given in cmc file
   _mdefSetServiceBase = DefSetServiceBase::GetInstance();
   if (_mdefSetServiceBase != NULL)
   {
      _mdefSetServiceBase->vRegisterforUpdate(this);
   }

   if (_applicationSwitchClientComponent != NULL)
   {
      _applicationSwitchClientComponent->setApplicationID(APPID_APPHMI_TUNER);
   }
}


TunerHall::~TunerHall()
{
   ETG_TRACE_USR4(("TunerHall : Destructor"));
   if (_AmFmInfo)
   {
      delete _AmFmInfo;
   }
   if (_RadioSetting)
   {
      delete _RadioSetting;
   }
   if (_AMFMTestMode)
   {
      delete _AMFMTestMode;
   }
   if (_AnnouncementHandler)
   {
      delete _AnnouncementHandler;
   }
   if (_presetList)
   {
      delete _presetList;
   }
   if (_tunerDataBindingHandler)
   {
      delete _tunerDataBindingHandler;
   }
   if (_srcChg)
   {
      delete _srcChg;
   }
   if (_AMFMStationList)
   {
      delete _AMFMStationList;
   }
   if (_DABStationList)
   {
      delete _DABStationList;
   }
   if (_applicationSwitchClientComponent)
   {
      delete _applicationSwitchClientComponent;
   }
   if (_ATStationInfo)
   {
      delete _ATStationInfo;
   }
   if (_DabInfo)
   {
      delete _DabInfo;
   }

   if (_VehicleSrvClient)
   {
      delete _VehicleSrvClient;
   }
   if (_WeatherBandMain != NULL)
   {
      delete	_WeatherBandMain;
   }
   if (_DiagnosticsClientHandler)
   {
      delete	_DiagnosticsClientHandler;
   }
   DefSetServiceBase::s_Destrory(); //to destroy DefSetServiceBase instance
   if (_mdefSetServiceBase != NULL)
   {
      _mdefSetServiceBase->vUnRegisterforUpdate(this);
      //donot delete this instance as its not created by us.
      _mdefSetServiceBase = NULL;
   }
}


void TunerHall::onExpired(asf::core::Timer& /*timer*/, boost::shared_ptr<asf::core::TimerPayload> /*data*/)
{
}


/**
* Description     : This method will be called on system startup for language translation
*
* @return         : uint8
*/
uint8 TunerHall::readLanguageOnStartUp()
{
   uint8 languageIndex = TunerDataPoolConfig::getInstance()->getLanguage();
   ETG_TRACE_USR4(("TunerHall::setLanguageOnSystemLoad requested :%d", languageIndex));
   return languageIndex;
}


bool TunerHall::onCourierMessage(const SwitchApplicationReq& oMsg)
{
   ETG_TRACE_USR3(("TunerHall->onCourierMessage:SwitchApplicationReq(): %d", oMsg.GetApplicationID()));
   //_applicationSwitchClientComponent->requestSwitchApplication(APPID_APPHMI_MASTER);
   int32_t ApplicationID = oMsg.GetApplicationID();
   switch (ApplicationID)
   {
      case APPID_APPHMI_MEDIA:
      {
         uint8 activeRegion = SrcChangeHandler::poGetInstance()->getActiveRegionValue();
         _applicationSwitchClientComponent->requestContextSwitchApplication(activeRegion, APPID_APPHMI_TUNER, enActivityIDs__eActivityID_TUNER_SOURCE_SELECTION,
               APPID_APPHMI_MEDIA, enActivityIDs__eActivityID_MEDIA_SOURCE_SELECTION);
         break;
      }
      default:
         _applicationSwitchClientComponent->requestSwitchApplication(APPID_APPHMI_MASTER);
   }

   return true;
}


/**
* reqPrepareResponse: To know the response of the respective APP when Restore factory setting is pressed.
* @param[in]: None
* @param[in] :None
*/
void TunerHall::reqPrepareResponse(const PrepareResDiagMsgData& /*oPrepareResDiagMsgData*/)
{
   ETG_TRACE_USR4(("TunerHall:: reqPrepareResponse"));
   if (_mdefSetServiceBase)
   {
      _mdefSetServiceBase->sendPrepareResponse(0, this);
      ETG_TRACE_USR4(("TunerHall:: Prepared"));
   }
}


/**
 * reqExecuteResponse: To Execute the response when restore factory setting has been executed..
 * @param[in]: None
 * @param[in] :None
 */
void TunerHall::reqExecuteResponse(const ExecuteResDiagMsgData& /*oExecuteResDiagMsgData*/)
{
   ETG_TRACE_USR4(("TunerHall:: reqExecuteResponse"));
   if (_mdefSetServiceBase)
   {
      _mdefSetServiceBase->sendExecuteResponse(0, this);
      ETG_TRACE_USR4(("TunerHall:: Executed"));
   }
}


/**
 * reqExecuteResponse: To Finalize the response when restore factory setting has been executed.
 * @param[in]: None
 * @param[in] :None
 */
void TunerHall:: reqFinalizeResponse(const FinalizeResDiagMsgData& /*oFinalizeResDiagMsgData*/)
{
   ETG_TRACE_USR4(("TunerHall:: reqFinalizeResponse"));
   if (_mdefSetServiceBase)
   {
      _mdefSetServiceBase->sendFinalizeResponse(0, this);
      ETG_TRACE_USR4(("TunerHall:: Finalized"));
      if (_tunerDataBindingHandler)
      {
         _tunerDataBindingHandler->SetSXMActive(false);
      }
      if (_srcChg)
      {
         _srcChg->setSXMStatus(false);
         _srcChg->setBackBtnStatus(true);
      }
      POST_MSG((COURIER_MESSAGE_NEW(TunerViewReqMsg)(EN_VIEW_FM1)));
      TunerDataPoolConfig* pTunerDataPoolConfig = TunerDataPoolConfig::getInstance();
      if (pTunerDataPoolConfig)
      {
         pTunerDataPoolConfig->setRadioTextStatus(1);
         pTunerDataPoolConfig->vSetDabFmFollowingStatus(1);
      }
      if (_RadioSetting)
      {
         _RadioSetting->vUpdate_Settings_Options(true);
         _RadioSetting->setHMI_REGstatus();
         _RadioSetting->onSelectHMI_AFFollowing();
         _RadioSetting->vSetDpDABPtyFilter(62); // 0x3E
         _RadioSetting->InitDABPTYDisableForFirstTime();
      }

      if (_AnnouncementHandler)
      {
         _AnnouncementHandler->setHMI_TAflag();
      }
      if (_AmFmInfo)
      {
         _AmFmInfo->bSendMsgToPL(EV_MSG_TO_PL_OPT_RADIO_TEXT, 1, "");
         _AmFmInfo->bSendMsgToPL(EV_MSG_TO_PL_OPT_DAB_FM_LINKING, 1, "");
      }
      if (_VehicleSrvClient)
      {
         _VehicleSrvClient->updateRDSStatus(true); //CAN Imp
         _VehicleSrvClient->updateDABFMStatus(true); //CAN Imp
      }
   }
}


bool TunerHall::OnAppModeChange(hmibase::app::base::lcm::ShutdownType newState, hmibase::app::base::lcm::ShutdownType oldState)
{
   ETG_TRACE_USR4(("TunerHall::OnAppModeChange(newState: %d, oldState: %d)", newState, oldState));

   if ((hmibase::app::base::lcm::SHUTDOWN_NORMAL == newState) || (hmibase::app::base::lcm::SHUTDOWN_FAST == newState))
   {
      if (SrcChangeHandler::poGetInstance())
      {
         SrcChangeHandler::poGetInstance()->isShutDown();
      }
   }
   else
   {
   }
   return true;
}


ETG_I_CMD_DEFINE((TraceCmd_onTestTraceCommand, "TraceCmd_onTestTraceCommand %d", int))
void TunerHall::TraceCmd_onTestTraceCommand(uint8 u8Value)
{
   ETG_TRACE_USR4(("TraceCmd_onTestTraceCommand : %d", u8Value));
}


bool TunerHall::onCourierMessage(const NavigationCarModeInfo& oMsg)
{
   ETG_TRACE_USR4(("TunerHall::onCourierMessage carMode :%d coachMode :%d", oMsg.GetCarMode(),  oMsg.GetCoachMode()));

   (*_carModeVisibilityStatus).mVisibleStatus = oMsg.GetCarMode();
   _carModeVisibilityStatus.MarkItemModified(ItemKey::CarModeVisibility::VisibleStatusItem);
   _carModeVisibilityStatus.SendUpdate(true);
   return true;
}


} // namespace Core
} // namespace App
