/**************************************************************************************
* @file         : SrcChangeHandler.cpp
* @author       : ECV2-Priyanka B
* @addtogroup   : Tuner HALL
* @brief        : Class contains necessary business logic required to handle SourceChange data.
* @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "hall_std_if.h"
#include "WeatherBandMain.h"
#include "SrcChangeHandler.h"
#include "AMFMStationList.h"
#include "CourierTunnelService/CourierMessageReceiverStub.h"
#include "AppHmi_MasterBase/AudioInterface/AudioDefines.h"
#include "App/Core/Tuner_Defines.h"
#include "TunerDataBindingHandler.h"
#include "RadioSetting.h"
#include "../AnnouncementHandler/AnnoHandler.h"
#include "spm_core_fi_typesConst.h"
#define OSAL_S_IMPORT_INTERFACE_THREADING
#include "osal_if.h"
#include "../VehicleSrvClient/VehicleSrvClient.h"  //CAN Imp


#define INITIALISE_SUBSRC_ID -1
#define INITIALISE_SUBSRC_ID_WX 0
#define DEFAULTCASE 10
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_TUNER_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_TUNER
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_TUNER_"
#define ETG_I_FILE_PREFIX                 App::Core::SrcChangeHandler::
#include "trcGenProj/Header/SrcChangeHandler.cpp.trc.h"
#endif

using namespace ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange;
using namespace ::bosch::cm::ai::hmi::masteraudioservice::SoundProperties;
using namespace ::bosch::cm::ai::hmi::hmiinfoservice::HmiInfoService;
using namespace ::spm_core_fi_types;

namespace App {
namespace Core {

DEFINE_CLASS_LOGGER_AND_LEVEL("App/Core", SrcChangeHandler, Info);

SrcChangeHandler* SrcChangeHandler::m_poSrcChangeHanlder = NULL;


/**
* Description     : Constructor of SrcChangeHandler
*/
SrcChangeHandler::SrcChangeHandler()
   : _audioSrcChgProxy(AudioSourceChangeProxy::createProxy("audioSourceChangePort", *this)),
     _soundPropertiesProxy(SoundPropertiesProxy::createProxy("soundPropertiesPort", *this)),
     m_poApplicationSwitchProxy(ApplicationSwitchProxy::createProxy("applicationSwitchPort", *this)),
     /*_srcChgdataBinder(NULL),
     _srcChgCourierMsg(NULL),*/
     _currSourceID(SRC_TUNER_FM),
     _CurrentSourceID(SRC_INVALID),
     _AudioSourceID(SRC_INVALID),
     sxm_selected_val(false),
     backbtn_val(false),
     m_cockpitSrc(SRC_TUNER_FM),
     m_cabinSrc(SRC_TUNER_FM),
     disconnect_src(false),
     fmConnected(false),
     amConnected(false),
     dabConnected(false),
     wxConnected(false)
     /*_transitToMainScreen(false),
      _switchId(0),
      _srcId(0),
      #ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1_R
      _dabOrDrmAudioSource(BAND_INVALID),
      #endif
      _appMode(APP_MODE_UNDEFINED)*/
{
   ETG_TRACE_USR3(("SrcChangeHandler::SrcChangeHandler"));
   ETG_I_REGISTER_FILE();
   if (_audioSrcChgProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _audioSrcChgProxy->getPortName());
   }
   if (_soundPropertiesProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _soundPropertiesProxy->getPortName());
   }
   int32 _sourceID = SRC_TUNER_FM;
   _sourceinfo = 0;
   prevSrcId = 0;
   u8Src_ConnectionState = STATE_UNKNOWN ;

   m_SinksPendingSrcChg = 0;
   for (uint8 i = 0; i < CABIN_B_SINK_ID; i++)
   {
      m_CurrentSrc[i] = 0xFF;
   }
   m_u8activeRegion = 2;
   _tunerSourceConnStatus = DISCONNECT;
   _tunerDataBindingHandler = TunerDataBindingHandler::getDataBindingHandlerInstance();
   _applicationSwitchClientComponent = ApplicationSwitchClientComponent::poGetInstance();
   _DiagnosticsClientHandler = DiagnosticsClientHandler::poGetInstance();

   _hmiInfoServiceProxyClient = (::HmiInfoServiceProxy::createProxy("hmiinfoservicePort", *this));

   if (_hmiInfoServiceProxyClient.get())
   {
      ETG_TRACE_USR4(("SrcChangeHandler::SrcChangeHandler() StartupSync _hmiInfoProxyClient"));
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _hmiInfoServiceProxyClient->getPortName());
   }
}


/**
* Description     : Destructor of SrcChangeHandler
*/
SrcChangeHandler::~SrcChangeHandler()
{
}


/**
* Description     : Request from Framework to start registering for properties.
*                   This request means GUI is ready now.
*
* @param[in]      : proxy : reference to the base class of all proxies
* @param[in]      : stateChange : represents a state transition of a service
* @return         : void
*/
void SrcChangeHandler::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR3(("SrcChangeHandler::registerProperties"));
   if ((proxy == _audioSrcChgProxy) && (_audioSrcChgProxy.get()))
   {
      _audioSrcChgProxy->sendActiveSourceGet(*this);
      _audioSrcChgProxy->sendActiveSourceListRegister(*this);
      _audioSrcChgProxy->sendActiveSourceRegister(*this);
   }
   if ((proxy == _soundPropertiesProxy) && (_soundPropertiesProxy.get()))
   {
      _soundPropertiesProxy->sendMuteStateRegister(*this);
      _soundPropertiesProxy->sendMuteStateMapRegister(*this);
   }
   if (_hmiInfoServiceProxyClient && _hmiInfoServiceProxyClient == proxy)
   {
      _hmiInfoServiceProxyClient->sendToggleGadgetAudioSourceRequestRegister(*this);
      _hmiInfoServiceProxyClient->sendCabinsSystemsInfoRegister(*this);
      _hmiInfoServiceProxyClient->sendCabinsSourceRestorationInfoRegister(*this);
   }
}


/**
* Description     : Request from Framework to start de-registering for properties.
*
* @param[in]      : proxy : reference to the base class of all proxies
* @param[in]      : stateChange : represents a state transition of a service
* @return         : void
*/
void SrcChangeHandler::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR3(("SrcChangeHandler::deregisterProperties"));
   if (_audioSrcChgProxy && _audioSrcChgProxy == proxy)
   {
      _audioSrcChgProxy->sendDeregisterAll();
   }
   else if (_soundPropertiesProxy && _soundPropertiesProxy == proxy)
   {
      _soundPropertiesProxy->sendVolumeDeregisterAll();
      _soundPropertiesProxy->sendMuteStateDeregisterAll();
      _soundPropertiesProxy->sendMuteStateMapDeregisterAll();
   }
   if (_hmiInfoServiceProxyClient && _hmiInfoServiceProxyClient == proxy)
   {
      _hmiInfoServiceProxyClient->sendToggleGadgetAudioSourceRequestDeregisterAll();
      _hmiInfoServiceProxyClient->sendCabinsSystemsInfoDeregisterAll();
      _hmiInfoServiceProxyClient->sendCabinsSourceRestorationInfoDeregisterAll();
   }
}


/**
* Description     : Request from Framework to start de-registering for properties.
*
* @param[in]      : proxy : reference to the base class of all proxies
* @param[in]      : stateChange : represents a state transition of a service
* @return         : void
*/
void SrcChangeHandler::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   if ((proxy == _audioSrcChgProxy) && (_audioSrcChgProxy.get()))
   {
      _audioSrcChgProxy->sendActiveSourceDeregisterAll();
      _audioSrcChgProxy->sendActiveSourceListDeregisterAll();
   }
   if ((proxy == _soundPropertiesProxy) && (_soundPropertiesProxy.get()))
   {
      _soundPropertiesProxy->sendMuteStateMapDeregisterAll();
      _soundPropertiesProxy->sendMuteStateDeregisterAll();
      _soundPropertiesProxy->sendAudioStateInfoDeregisterAll();
   }
   if (m_poApplicationSwitchProxy && m_poApplicationSwitchProxy == proxy)
   {
      m_poApplicationSwitchProxy->sendActiveRegionIdDeregisterAll();
      m_poApplicationSwitchProxy->sendSig_ActivateContextDeregisterAll();
   }
   if (_hmiInfoServiceProxyClient && _hmiInfoServiceProxyClient == proxy)
   {
      _hmiInfoServiceProxyClient->sendToggleGadgetAudioSourceRequestDeregisterAll();
      _hmiInfoServiceProxyClient->sendCabinsSystemsInfoDeregisterAll();
      _hmiInfoServiceProxyClient->sendCabinsSourceRestorationInfoDeregisterAll();
   }
}


void SrcChangeHandler::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR3(("SrcChangeHandler::onAvailable"));
   if ((proxy == _audioSrcChgProxy) && (_audioSrcChgProxy.get()))
   {
      _audioSrcChgProxy->sendActiveSourceGet(*this);
      _audioSrcChgProxy->sendActiveSourceListRegister(*this);
      _audioSrcChgProxy->sendActiveSourceRegister(*this);
   }
   if ((proxy == _soundPropertiesProxy) && (_soundPropertiesProxy.get()))
   {
      _soundPropertiesProxy->sendMuteStateMapRegister(*this);
      _soundPropertiesProxy->sendMuteStateRegister(*this);
      _soundPropertiesProxy->sendAudioStateInfoRegister(*this);
   }
   if ((proxy == _audioSrcChgProxy) && (_audioSrcChgProxy.get()))
   {
      _audioSrcChgProxy->sendActiveSourceGet(*this);
      _audioSrcChgProxy->sendActiveSourceListRegister(*this);
   }

   if (m_poApplicationSwitchProxy && m_poApplicationSwitchProxy == proxy)
   {
      m_poApplicationSwitchProxy->sendActiveRegionIdRegister(*this);
      m_poApplicationSwitchProxy->sendActiveRegionIdGet(*this);
      m_poApplicationSwitchProxy->sendSig_ActivateContextRegister(*this);
   }
   if (_hmiInfoServiceProxyClient && _hmiInfoServiceProxyClient == proxy)
   {
      _hmiInfoServiceProxyClient->sendToggleGadgetAudioSourceRequestRegister(*this);
      _hmiInfoServiceProxyClient->sendCabinsSystemsInfoRegister(*this);
      _hmiInfoServiceProxyClient->sendCabinsSourceRestorationInfoRegister(*this);
   }
}


void SrcChangeHandler::onToggleGadgetAudioSourceError(const ::boost::shared_ptr< HmiInfoServiceProxy >& proxy, const ::boost::shared_ptr< ToggleGadgetAudioSourceError >& error)
{
   ETG_TRACE_USR4(("SrcChangeHandler::onToggleGadgetAudioSourceError()"));
}


void SrcChangeHandler::onToggleGadgetAudioSourceResponse(const ::boost::shared_ptr< HmiInfoServiceProxy >& proxy, const ::boost::shared_ptr< ToggleGadgetAudioSourceResponse >& response)
{
   ETG_TRACE_USR4(("SrcChangeHandler::onToggleGadgetAudioSourceResponse()"));
}


/**
* Description     : Virtual function implemented to get update of onSig_ActivateContextError Error
*
* @param[in]      : proxy: the client side representation of the ContextSwitch IF
* @param[in]      : error : The error message of 'onSig_ActivateContextError'
* @return         : void
*/

void SrcChangeHandler::onSig_ActivateContextError(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< Sig_ActivateContextError >& error)
{
   ETG_TRACE_USR4(("SrcChangeHandler::onSig_ActivateContextError"));
}


/**
* Description     : Virtual function implemented to get update of onSig_ActivateContextSignal
*
* @param[in]      : proxy: the client side representation of the ContextSwitch IF
* @param[in]      : response : The response message of 'onSig_ActivateContextSignal'
* @return         : void
*/

void SrcChangeHandler::onSig_ActivateContextSignal(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< Sig_ActivateContextSignal >& signal)
{
   ETG_TRACE_USR4(("SrcChangeHandler::onSig_ActivateContextSignal %d", signal->getTargetactivityId()));
   uint8 activityID = signal->getTargetactivityId();
   uint8 switchScreen = 0;
   int32 audioSource = 0;  //default is FM
   switch (activityID)
   {
      case enActivityIDs__eActivityID_TUNER:										// 100 Request from App-Flyins
         POST_MSG((COURIER_MESSAGE_NEW(ButtonMainSrcActivate)(_sourceinfo)));
         break;
      case enActivityIDs__eActivityID_TUNER_SOURCE_SELECTION:           			// 101 Source Selection
         switchScreen = EN_VIEW_SOURCE_SELECTION;
         POST_MSG((COURIER_MESSAGE_NEW(TunerViewReqMsg)(switchScreen)));
         break;
      case enActivityIDs__eActivityID_TUNER_PLAY: 									// Play request from App-Fyins.
         activeTunerAudioAllocationRequest();
         break;
      case enActivityIDs__eActivityID_TUNER_FM1:	                                // 102 = FM Main Screen
         sm_gAudioSource = SRC_TUNER_FM;
         switchScreen = EN_VIEW_FM1;
         audioSource = SRC_TUNER_FM;
         activateSource(audioSource);
         POST_MSG((COURIER_MESSAGE_NEW(TunerViewReqMsg)(switchScreen)));
         break;
      case enActivityIDs__eActivityID_TUNER_AM:										// 105 = activate AM Main screen
         sm_gAudioSource = SRC_TUNER_AM;
         switchScreen = EN_VIEW_AM;
         audioSource = SRC_TUNER_AM;
         activateSource(audioSource);
         POST_MSG((COURIER_MESSAGE_NEW(TunerViewReqMsg)(switchScreen)));
         break;
      case enActivityIDs__eActivityID_TUNER_DAB:												// 107 = activate DAB Main screen
         sm_gAudioSource = SRC_DAB;
         switchScreen = EN_VIEW_DAB;
         audioSource = SRC_DAB;
         activateSource(audioSource);
         POST_MSG((COURIER_MESSAGE_NEW(TunerViewReqMsg)(switchScreen)));
         break;
      case enActivityIDs__eActivityID_TUNER_WX:												// 115 = activate WX Main screen
         sm_gAudioSource = SRC_TUNER_WX;
         switchScreen = EN_VIEW_WEATHERBAND;
         audioSource = SRC_TUNER_WX;
         activateSource(audioSource);
         POST_MSG((COURIER_MESSAGE_NEW(TunerViewReqMsg)(switchScreen)));
         break;
      case enActivityIDs__eActivityID_TUNER_SETTINGS_VIEW:                                         // 111 = activate Settings screen
         switchScreen = EN_VIEW_OPT_FMAM;
         POST_MSG((COURIER_MESSAGE_NEW(TunerViewReqMsg)(switchScreen)));
         break;
      case enActivityIDs__eActivityID_HOME:                                      // 80 = activate Home screen
         AMFMStationList::poGetInstance()->ListExitMsg();
         break;
      case enActivityIDs__eActivityID_TUNER_TESTMODESCREEN:                                         //  = activate Sngs screen
         switchScreen = EN_VIEW_TESTMODESCREENS;
         POST_MSG((COURIER_MESSAGE_NEW(TunerViewReqMsg)(switchScreen)));
         break;
      case enActivityIDs__eActivityID_TUNER_FM_LIST:                                         // 113 = activate FM List screen
         switchScreen = EN_VIEW_FM_SL;
         POST_MSG((COURIER_MESSAGE_NEW(TunerViewReqMsg)(switchScreen)));
         break;
      case enActivityIDs__eActivityID_TUNER_DAB_LIST:                                         // 114 = activate DAB List screen
         switchScreen = EN_VIEW_DAB_SL;
         POST_MSG((COURIER_MESSAGE_NEW(TunerViewReqMsg)(switchScreen)));
         break;
      default:
         break;
   }
}


void SrcChangeHandler::activeTunerAudioAllocationRequest()
{
   ETG_TRACE_USR4(("SrcChangeHandler::activeTunerAudioAllocationRequest %d", prevSrcId));
   uint8 switchScreen = 0;
   sm_gAudioSource = prevSrcId;
   activateSource(prevSrcId);

   if (prevSrcId == SRC_TUNER_FM)
   {
      switchScreen = EN_VIEW_FM1;
      POST_MSG((COURIER_MESSAGE_NEW(TunerViewReqMsg)(switchScreen)));
   }
   else if (prevSrcId == SRC_TUNER_AM)
   {
      switchScreen = EN_VIEW_AM;
      POST_MSG((COURIER_MESSAGE_NEW(TunerViewReqMsg)(switchScreen)));
   }
   else if (prevSrcId == SRC_DAB)
   {
      switchScreen = EN_VIEW_DAB;
      POST_MSG((COURIER_MESSAGE_NEW(TunerViewReqMsg)(switchScreen)));
   }
   else if (prevSrcId == SRC_TUNER_WX)
   {
      switchScreen = EN_VIEW_WEATHERBAND;
      POST_MSG((COURIER_MESSAGE_NEW(TunerViewReqMsg)(switchScreen)));
   }
   else if (prevSrcId == SRC_TUNER_XM)
   {
      POST_MSG((COURIER_MESSAGE_NEW(SXMContextMsg)()));
   }
   else
   {
      POST_MSG((COURIER_MESSAGE_NEW(TunerViewReqMsg)(switchScreen)));
   }
}


void SrcChangeHandler::onToggleGadgetAudioSourceRequestError(const ::boost::shared_ptr<bosch::cm::ai::hmi::hmiinfoservice::HmiInfoService::HmiInfoServiceProxy >& proxy, const ::boost::shared_ptr<ToggleGadgetAudioSourceRequestError >& error)
{
   ETG_TRACE_COMP(("SrcChangeHandler::onToggleGadgetAudioSourceRequestError(const ::boost::shared_ptr< HmiInfoServiceProxy >& proxy, const ::boost::shared_ptr< ToggleGadgetAudioSourceRequestError >& error) not yet implemented"));
}


void  SrcChangeHandler::onToggleGadgetAudioSourceRequestSignal(const ::boost::shared_ptr<bosch::cm::ai::hmi::hmiinfoservice::HmiInfoService::HmiInfoServiceProxy >& proxy, const ::boost::shared_ptr<ToggleGadgetAudioSourceRequestSignal >& signal)
{
   ETG_TRACE_COMP(("SrcChangeHandler::onToggleGadgetAudioSourceRequestSignal"));
   ETG_TRACE_USR4(("SrcChangeHandler::onToggleGadgetAudioSourceRequestSignal destapplication %d", signal->getDestApplicationId()));
   uint8 destapplication = signal->getDestApplicationId();
   if (destapplication == APPID_APPHMI_TUNER)
   {
      switchToTunerHomeScreenGadget();
   }
}


void SrcChangeHandler::vRequestTunerSourceDeactivation(int srcid)
{
   ETG_TRACE_USR4(("SrcChangeHandler::vRequestTunerSourceDeactivation srcid = %d", srcid));
   sourceData srcData;
   srcData.setSrcId(srcid);
   if (srcid == SRC_TUNER_WX)
   {
      srcData.setSubSrcId(INITIALISE_SUBSRC_ID_WX);//for weatherband
   }
   else
   {
      srcData.setSubSrcId(INITIALISE_SUBSRC_ID);
   }
   srcData.setConnectionState(0);
   srcData.setSinkId(CABIN_A_SINK_ID);
   _audioSrcChgProxy->sendDeactivateSourceRequest(*this, srcData);
}


void SrcChangeHandler::onCabinsSystemsInfoError(const ::boost::shared_ptr< HmiInfoServiceProxy >& proxy, const ::boost::shared_ptr< CabinsSystemsInfoError >& error)
{
   ETG_TRACE_USR4(("SrcChangeHandler::onCabinsSystemsInfoError"));
}


void  SrcChangeHandler::onCabinsSystemsInfoUpdate(const ::boost::shared_ptr< HmiInfoServiceProxy >& proxy, const ::boost::shared_ptr< CabinsSystemsInfoUpdate >& update)
{
   ETG_TRACE_USR4(("SrcChangeHandler::onCabinsSystemsInfoUpdate"));

   if (update != NULL)
   {
      ::std::vector< SystemInfo >systeminfovector = update->getCabinsSystemsInfo();

      if (!systeminfovector.empty())
      {
         ::std::vector< SystemInfo >::const_iterator systemInfo_iter = systeminfovector.begin();

         if (proxy == _hmiInfoServiceProxyClient)
         {
            while (systemInfo_iter != systeminfovector.end())
            {
               uint8 systemId = systemInfo_iter->getSystemId();
               uint8 systemPosition = systemInfo_iter->getSystemPosition();
               uint8 systemGroup = systemInfo_iter->getSystemGroup();
               uint8 systemStatus = systemInfo_iter->getSystemStatus();
               uint8 systemCategory = systemInfo_iter->getSystemCategory();

               ETG_TRACE_USR4(("SrcChangeHandler::onCabinsSystemsInfoUpdate systemId %d, systemPosition %d, systemGroup %d, systemStatus %d, systemCategory %d, m_cabinSrc %d", systemId, systemPosition, systemGroup, systemStatus, systemCategory, m_cabinSrc));

               if (SYSTEM_CATEGORY_CMA == systemCategory)
               {
                  if ((SYSTEM_STATUS_DISCONNECTED == systemStatus) ||
                        (SYSTEM_STATUS_INACTIVE_DISCONNECTED == systemStatus) ||
                        (SYSTEM_STATUS_PNM_DISCONNECTED == systemStatus) ||
                        (SYSTEM_STATUS_USER_OFF_DISCONNECTED == systemStatus) ||
                        (SYSTEM_STATUS_HOURLOGIC_DISCONNECTED == systemStatus) ||
                        (SYSTEM_STATUS_HMI_OFF_DISCONNECTED == systemStatus) ||
                        (SYSTEM_STATUS_COCKPIT_DISCONNECTED == systemStatus) ||
                        (SYSTEM_STATUS_GOODBYE_DISCONNECTED == systemStatus))
                  {
                     if (((m_cabinSrc == SRC_TUNER_FM) || (m_cabinSrc == SRC_TUNER_AM) || (m_cabinSrc == SRC_DAB) || (m_cabinSrc == SRC_TUNER_WX)) && (disconnect_src == false))
                     {
                        disconnect_src = true;
                        ETG_TRACE_USR4(("SrcChangeHandler::onCabinsSystemsInfoUpdate Disconnected: systemStatus %d,systemCategory %d,m_cabinSrc %d", systemStatus, systemCategory, m_cabinSrc));
                        vRequestTunerSourceDeactivation(m_cabinSrc);
                        break;
                     }
                  }
                  else if (SYSTEM_STATUS_CONNECTED == systemStatus)
                  {
                     disconnect_src = false;
                  }
               }
               systemInfo_iter++;
            }
         }
      }
   }
}


//Source Restoration

void SrcChangeHandler::onCabinsSourceRestorationInfoError(const ::boost::shared_ptr< HmiInfoServiceProxy >& proxy,
      const ::boost::shared_ptr< CabinsSourceRestorationInfoError >& error)
{
   ETG_TRACE_USR4(("SrcChangeHandler::onCabinsSourceRestorationInfoError"));
}


void SrcChangeHandler::onCabinsSourceRestorationInfoUpdate(const ::boost::shared_ptr< HmiInfoServiceProxy >& proxy,
      const ::boost::shared_ptr< CabinsSourceRestorationInfoUpdate >& update)
{
   ETG_TRACE_USR4(("SrcChangeHandler::onCabinsSourceRestorationInfoUpdate"));

   if (update != NULL)
   {
      ::std::vector< CabinSourceRestorationInfo >restorevector = update->getCabinsSourceRestorationInfo();

      if (!restorevector.empty())
      {
         ::std::vector< CabinSourceRestorationInfo >::const_iterator cabinsSourceRestore_iter = restorevector.begin();

         if (proxy == _hmiInfoServiceProxyClient)
         {
            while (cabinsSourceRestore_iter != restorevector.end())
            {
               uint32 cabinId = cabinsSourceRestore_iter->getCabinId();
               // uint32 status = cabinsSourceRestore_iter->getStatus();

               // ETG_TRACE_USR4(("SrcChangeHandler::onCabinsSourceRestorationInfoUpdate : status %d, cabinId %d ", status, cabinId));

               ::std::vector< SourceInfoHolder> SourcesInfoArrayData = cabinsSourceRestore_iter->getLastActiveSourcesInfo();
               ::std::vector<SourceInfoHolder>::const_iterator sourcesInfo_iter = SourcesInfoArrayData.begin();

               while (sourcesInfo_iter != SourcesInfoArrayData.end())
               {
                  int16 lastActiveSource = sourcesInfo_iter->getSourceId();
                  int16 sinkId = sourcesInfo_iter->getSinkId();
                  uint32 status = sourcesInfo_iter->getStatus();

                  ETG_TRACE_USR4(("SrcChangeHandler::onCabinsSourceRestorationInfoUpdate lastActiveSource = %d,sinkId = %d, status = %d ", lastActiveSource, sinkId, status));
                  if ((status == CABIN_RESTORE_STATUS_RESTORE) && (sinkId == CABIN_A_SINK_ID)
                        && ((lastActiveSource == SRC_TUNER_FM) || (lastActiveSource == SRC_TUNER_AM) || (lastActiveSource == SRC_DAB) || (lastActiveSource == SRC_TUNER_WX)))
                  {
                     //Send connection req
                     disconnect_src = false;
                     vRequestTunerSourceActivation(lastActiveSource, sinkId);
                     break;
                  }
                  sourcesInfo_iter++;
               }
               cabinsSourceRestore_iter++;
            }
         }
      }
   }
}


void SrcChangeHandler::vRequestTunerSourceActivation(int16 srcid, int16 sinkId)
{
   ETG_TRACE_USR4(("SrcChangeHandler::vRequestTunerSourceActivation srcid = %d, sinkId = %d", srcid, sinkId));

   sourceData srcData;
   srcData.setSrcId(srcid);
   if (srcid == SRC_TUNER_WX)
   {
      srcData.setSubSrcId(INITIALISE_SUBSRC_ID_WX);//for weatherband
   }
   else
   {
      srcData.setSubSrcId(INITIALISE_SUBSRC_ID);
   }
   srcData.setConnectionState(0);
   srcData.setSinkId(sinkId);
   ((_audioSrcChgProxy.get()) && (_audioSrcChgProxy->sendActivateSourceRequest(*this, srcData, true)));
}


void SrcChangeHandler::vSourceCommandActivation(int16 sourceId)
{
   ETG_TRACE_USR3(("SrcChangeHandler::vSourceCommandActivation sourceId = %d", sourceId));
   uint8 dabVal = TunerDataPoolConfig::getInstance()->readDABMountedValue();
   uint8 sxmVal = TunerDataPoolConfig::getInstance()->readSXMMountedValue();

   switch (sourceId)
   {
      case CAN_SOURCE_FM:
      {
         if (m_u8activeRegion == REGION_COCKPIT)
         {
            _applicationSwitchClientComponent->requestContextSwitchApplication(2, APPID_APPHMI_UNKNOWN, enActivityIDs__eActivityID_NONE, APPID_APPHMI_TUNER, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_TUNER_FM1);
         }
         else
         {
            sourceData srcData;
            srcData.setSrcId(SRC_TUNER_FM);
            srcData.setSubSrcId(INITIALISE_SUBSRC_ID);
            srcData.setConnectionState(0);
            srcData.setSinkId(COCKPIT_SINK_ID);
            ((_audioSrcChgProxy.get()) && (_audioSrcChgProxy->sendActivateSourceRequest(*this, srcData, true)));
         }
      }
      break;
      case CAN_SOURCE_AM:
      {
         if (m_u8activeRegion == REGION_COCKPIT)
         {
            _applicationSwitchClientComponent->requestContextSwitchApplication(2, APPID_APPHMI_UNKNOWN, enActivityIDs__eActivityID_NONE, APPID_APPHMI_TUNER, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_TUNER_AM);
         }
         else
         {
            sourceData srcData;
            srcData.setSrcId(SRC_TUNER_AM);
            srcData.setSubSrcId(INITIALISE_SUBSRC_ID);
            srcData.setConnectionState(0);
            srcData.setSinkId(COCKPIT_SINK_ID);
            ((_audioSrcChgProxy.get()) && (_audioSrcChgProxy->sendActivateSourceRequest(*this, srcData, true)));
         }
      }
      break;
      case CAN_SOURCE_DAB:
      {
         if ((TunerDataPoolConfig::getInstance()->readRegionType() == EUR_REGION) && (dabVal == 1))
         {
            if (m_u8activeRegion == REGION_COCKPIT)
            {
               _applicationSwitchClientComponent->requestContextSwitchApplication(2, APPID_APPHMI_UNKNOWN, enActivityIDs__eActivityID_NONE, APPID_APPHMI_TUNER, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_TUNER_DAB);
            }
            else
            {
               sourceData srcData;
               srcData.setSrcId(SRC_DAB);
               srcData.setSubSrcId(INITIALISE_SUBSRC_ID);
               srcData.setConnectionState(0);
               srcData.setSinkId(COCKPIT_SINK_ID);
               ((_audioSrcChgProxy.get()) && (_audioSrcChgProxy->sendActivateSourceRequest(*this, srcData, true)));
            }
         }
      }
      break;
      case CAN_SOURCE_WX:
      {
         if (TunerDataPoolConfig::getInstance()->readRegionType() != EUR_REGION)
         {
            if (m_u8activeRegion == REGION_COCKPIT)
            {
               _applicationSwitchClientComponent->requestContextSwitchApplication(2, APPID_APPHMI_UNKNOWN, enActivityIDs__eActivityID_NONE, APPID_APPHMI_TUNER, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_TUNER_WX);
            }
            else
            {
               sourceData srcData;
               srcData.setSrcId(SRC_TUNER_WX);
               srcData.setSubSrcId(INITIALISE_SUBSRC_ID_WX);
               srcData.setConnectionState(0);
               srcData.setSinkId(COCKPIT_SINK_ID);
               ((_audioSrcChgProxy.get()) && (_audioSrcChgProxy->sendActivateSourceRequest(*this, srcData, true)));
            }
         }
      }
      break;
      case CAN_SOURCE_XM:
      {
         if (sxmVal == 1)
         {
            if (m_u8activeRegion == REGION_COCKPIT)
            {
               ETG_TRACE_USR4(("SrcChangeHandler::vSourceCommandActivation sourceId = %d", sourceId));
               setSXMStatus(true);
               VehicleSrvClient::poGetInstance()->updateTunerBandStatus(CAN_XM); //CAN Imp
               TunerDataBindingHandler::getDataBindingHandlerInstance()->SetSXMActive(true);
               _applicationSwitchClientComponent->requestContextSwitchApplication(2, APPID_APPHMI_UNKNOWN, enActivityIDs__eActivityID_NONE, APPID_APPHMI_SXM, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_SXM_MAIN);
            }
            else
            {
               sourceData srcData;
               srcData.setSrcId(SRC_TUNER_XM);
               srcData.setSubSrcId(INITIALISE_SUBSRC_ID);
               srcData.setConnectionState(0);
               srcData.setSinkId(COCKPIT_SINK_ID);
               ((_audioSrcChgProxy.get()) && (_audioSrcChgProxy->sendActivateSourceRequest(*this, srcData, true)));
            }
         }
      }
      break;
      default:
      {
         ETG_TRACE_USR4(("SrcChangeHandler::vSourceCommandActivation Default case option"));
      }
      break;
   }
}


void SrcChangeHandler::vCabinSourceCommandActivation(int16 sourceId)
{
   ETG_TRACE_USR3(("SrcChangeHandler::vCabinSourceCommandActivation sourceId = %d", sourceId));
   uint8 dabVal = TunerDataPoolConfig::getInstance()->readDABMountedValue();
   uint8 sxmVal = TunerDataPoolConfig::getInstance()->readSXMMountedValue();

   switch (sourceId)
   {
      case CAN_SOURCE_FM:
      {
         if (m_u8activeRegion == REGION_CABIN_A)
         {
            _applicationSwitchClientComponent->requestContextSwitchApplication(0, APPID_APPHMI_UNKNOWN, enActivityIDs__eActivityID_NONE, APPID_APPHMI_TUNER, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_TUNER_FM1);
         }
         else
         {
            sourceData srcData;
            srcData.setSrcId(SRC_TUNER_FM);
            srcData.setSubSrcId(INITIALISE_SUBSRC_ID);
            srcData.setConnectionState(0);
            srcData.setSinkId(CABIN_A_SINK_ID);
            ((_audioSrcChgProxy.get()) && (_audioSrcChgProxy->sendActivateSourceRequest(*this, srcData, true)));
         }
      }
      break;
      case CAN_SOURCE_AM:
      {
         if (m_u8activeRegion == REGION_CABIN_A)
         {
            _applicationSwitchClientComponent->requestContextSwitchApplication(0, APPID_APPHMI_UNKNOWN, enActivityIDs__eActivityID_NONE, APPID_APPHMI_TUNER, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_TUNER_AM);
         }
         else
         {
            sourceData srcData;
            srcData.setSrcId(SRC_TUNER_AM);
            srcData.setSubSrcId(INITIALISE_SUBSRC_ID);
            srcData.setConnectionState(0);
            srcData.setSinkId(CABIN_A_SINK_ID);
            ((_audioSrcChgProxy.get()) && (_audioSrcChgProxy->sendActivateSourceRequest(*this, srcData, true)));
         }
      }
      break;
      case CAN_SOURCE_DAB:
      {
         if ((TunerDataPoolConfig::getInstance()->readRegionType() == EUR_REGION) && (dabVal == 1))
         {
            if (m_u8activeRegion == REGION_CABIN_A)
            {
               _applicationSwitchClientComponent->requestContextSwitchApplication(0, APPID_APPHMI_UNKNOWN, enActivityIDs__eActivityID_NONE, APPID_APPHMI_TUNER, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_TUNER_DAB);
            }
            else
            {
               sourceData srcData;
               srcData.setSrcId(SRC_DAB);
               srcData.setSubSrcId(INITIALISE_SUBSRC_ID);
               srcData.setConnectionState(0);
               srcData.setSinkId(CABIN_A_SINK_ID);
               ((_audioSrcChgProxy.get()) && (_audioSrcChgProxy->sendActivateSourceRequest(*this, srcData, true)));
            }
         }
      }
      break;
      case CAN_SOURCE_WX:
      {
         if (TunerDataPoolConfig::getInstance()->readRegionType() != EUR_REGION)
         {
            if (m_u8activeRegion == REGION_CABIN_A)
            {
               _applicationSwitchClientComponent->requestContextSwitchApplication(0, APPID_APPHMI_UNKNOWN, enActivityIDs__eActivityID_NONE, APPID_APPHMI_TUNER, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_TUNER_WX);
            }
            else
            {
               sourceData srcData;
               srcData.setSrcId(SRC_TUNER_WX);
               srcData.setSubSrcId(INITIALISE_SUBSRC_ID_WX);
               srcData.setConnectionState(0);
               srcData.setSinkId(CABIN_A_SINK_ID);
               ((_audioSrcChgProxy.get()) && (_audioSrcChgProxy->sendActivateSourceRequest(*this, srcData, true)));
            }
         }
      }
      break;
      case CAN_SOURCE_XM:
      {
         if (sxmVal == 1)
         {
            if (m_u8activeRegion == REGION_CABIN_A)
            {
               setSXMStatus(true);
               VehicleSrvClient::poGetInstance()->updateCabinTunerBandStatus(CAN_XM);
               TunerDataBindingHandler::getDataBindingHandlerInstance()->SetSXMActive(true);
               _applicationSwitchClientComponent->requestContextSwitchApplication(0, APPID_APPHMI_UNKNOWN, enActivityIDs__eActivityID_NONE, APPID_APPHMI_SXM, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_SXM_MAIN);
            }
            else
            {
               sourceData srcData;
               srcData.setSrcId(SRC_TUNER_XM);
               srcData.setSubSrcId(INITIALISE_SUBSRC_ID);
               srcData.setConnectionState(0);
               srcData.setSinkId(CABIN_A_SINK_ID);
               ((_audioSrcChgProxy.get()) && (_audioSrcChgProxy->sendActivateSourceRequest(*this, srcData, true)));
            }
         }
      }
      break;
      default:
      {
         ETG_TRACE_USR4(("SrcChangeHandler::vCabinSourceCommandActivation Default case option"));
      }
      break;
   }
}


void SrcChangeHandler::switchToTunerHomeScreenGadget()
{
   ETG_TRACE_USR4(("SrcChangeHandler::switchToTunerHomeScreenGadget"));
   int32 audioSource = 0;
   activateSource(_currSourceID);
}


/************************************************************************
*FUNCTION		: reqMediaScreenToggle
*DESCRIPTION	: func call context switch to Media Home screen
*PARAMETER		: courier msg
*PARAMETER		: NA
*RETURNVALUE	: bool
************************************************************************/

void SrcChangeHandler::reqMediaScreenToggle()
{
   ETG_TRACE_USR4(("SrcChangeHandler::reqMediaScreenToggle"));
   _hmiInfoServiceProxyClient->sendToggleGadgetAudioSourceRequest(*this, m_u8activeRegion, APPID_APPHMI_TUNER, APPID_APPHMI_MEDIA);
}


/**
* Description     : Virtual function implemented to get update of ActivateSource Error
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'AudioSourceChangeProxy'
* @param[in]      : error : The error message of 'ActivateSource'
* @return         : void
*/
void SrcChangeHandler::onActivateSourceError(const ::boost::shared_ptr< AudioSourceChangeProxy >& /*proxy*/, const ::boost::shared_ptr< ActivateSourceError >& /*error*/)
{
   ETG_TRACE_USR3(("SrcChangeHandler::onActivateSourceError"));
}


/**
* Description     : Virtual function implemented to get update of ActivateSource Response
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'AudioSourceChangeProxy'
* @param[in]      : response : The response message of 'ActivateSource'
* @return         : void
*/
void SrcChangeHandler::onActivateSourceResponse(const ::boost::shared_ptr< AudioSourceChangeProxy >& /*proxy*/, const ::boost::shared_ptr< ActivateSourceResponse >& /*response*/)
{
   ETG_TRACE_USR3(("SrcChangeHandler::onActivateSourceResponse"));
}


/**
* Description     : Virtual function implemented to get update of onActiveSourceError error
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'AudioSourceChangeProxy'
* @param[in]      : error : The error message of 'onActiveSourceError'
* @return         : void
*/
void SrcChangeHandler::onActiveSourceError(const ::boost::shared_ptr< AudioSourceChangeProxy >& /*proxy*/, const ::boost::shared_ptr< ActiveSourceError >& /*error*/)
{
   ETG_TRACE_USR3(("SrcChangeHandler::onActiveSourceError"));
}


void SrcChangeHandler::onDeactivateSourceError(const ::boost::shared_ptr< AudioSourceChangeProxy >& /*proxy*/, const ::boost::shared_ptr< DeactivateSourceError >& /*error*/)
{
   ETG_TRACE_USR3(("SrcChangeHandler::onDeactivateSourceError"));
}


void SrcChangeHandler::onDeactivateSourceResponse(const ::boost::shared_ptr< AudioSourceChangeProxy >& /*proxy*/, const ::boost::shared_ptr< DeactivateSourceResponse >& /*response*/)
{
   ETG_TRACE_USR3(("SrcChangeHandler::onDeactivateSourceResponse"));
}


/**
* Description     : Virtual function implemented to get update of onActiveSourceUpdate
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'AudioSourceChangeProxy'
* @param[in]      : update : The update message of 'onActiveSourceUpdate'
* @return         : void
*/
void SrcChangeHandler::onActiveSourceUpdate(const ::boost::shared_ptr< AudioSourceChangeProxy >& /*proxy*/, const ::boost::shared_ptr< ActiveSourceUpdate >& update)
{
   ETG_TRACE_USR3(("SrcChangeHandler::onActiveSourceUpdate"));
   sourceData source = update->getActiveSource();
   _sourceinfo = source.getSrcId();
   int32 activeSource = source.getSrcId();
   u8Src_ConnectionState = update->getActiveSource().getConnectionState();
   ETG_TRACE_USR4(("SrcChangeHandler::onActiveSourceUpdate SrcId: == %d  ConnectionState == %d", source.getSrcId(), u8Src_ConnectionState));
   ETG_TRACE_USR4(("SrcChangeHandler::onActiveSourceUpdate SrcId: %d  SinkId: %d  m_SinksPendingSrcChg: %d", source.getSrcId(), source.getSinkId(), m_SinksPendingSrcChg));
   sm_gAudioSource = activeSource;

   setCockpitCabinValues(source.getSinkId(), activeSource, u8Src_ConnectionState);
   if ((source.getSinkId() == COCKPIT_SINK_ID) || (source.getSinkId() == CABIN_A_SINK_ID) || (source.getSinkId() == CABIN_B_SINK_ID))
   {
      if (update->getActiveSource().getSrcId() != SRC_NAVI_SPEECH)
      {
         m_CurrentSrc[source.getSinkId()] = source.getSrcId();
      }
   }

   if (isTunerSourceRange(update->getActiveSource().getSrcId()))
   {
      prevSrcId = source.getSrcId();
   }

   if ((activeSource == SRC_TUNER_FM) || (activeSource == SRC_TUNER_AM) || (activeSource == SRC_DAB) || (activeSource == SRC_TUNER_WX))
   {
      if (m_SinksPendingSrcChg != 0)
      {
         sourceData srcData;
         if (m_SinksPendingSrcChg & (1 << CABIN_A_SINK_ID))
         {
            disconnect_src = false;
            srcData.setSrcId(activeSource);
            srcData.setSubSrcId(INITIALISE_SUBSRC_ID);
            srcData.setConnectionState(0);
            srcData.setSinkId(CABIN_A_SINK_ID);
            m_SinksPendingSrcChg = ((m_SinksPendingSrcChg) & (~(1 << CABIN_A_SINK_ID)));
            ((_audioSrcChgProxy.get()) && (_audioSrcChgProxy->sendActivateSourceRequest(*this, srcData, true)));
         }
         else if (m_SinksPendingSrcChg & (1 << CABIN_B_SINK_ID))
         {
            srcData.setSrcId(activeSource);
            srcData.setSubSrcId(INITIALISE_SUBSRC_ID);
            srcData.setConnectionState(0);
            srcData.setSinkId(CABIN_B_SINK_ID);
            m_SinksPendingSrcChg = ((m_SinksPendingSrcChg) & (~(1 << CABIN_B_SINK_ID)));
            ((_audioSrcChgProxy.get()) && (_audioSrcChgProxy->sendActivateSourceRequest(*this, srcData, true)));
         }
         else if (m_SinksPendingSrcChg & (1 << COCKPIT_SINK_ID))
         {
            srcData.setSrcId(activeSource);
            srcData.setSubSrcId(INITIALISE_SUBSRC_ID);
            srcData.setConnectionState(0);
            srcData.setSinkId(COCKPIT_SINK_ID);
            m_SinksPendingSrcChg = ((m_SinksPendingSrcChg) & (~(1 << COCKPIT_SINK_ID)));
            ((_audioSrcChgProxy.get()) && (_audioSrcChgProxy->sendActivateSourceRequest(*this, srcData, true)));
         }
         else
         {}
      }
   }

   //added for the bug 533719
   // if ((activeSource == SRC_MEDIA_PLAYER) || (activeSource == SRC_PHONE_BTAUDIO))
   if (isMediaSourceRange(update->getActiveSource().getSrcId()))
   {
      ETG_TRACE_USR4(("SrcChangeHandler::activeSource is SRC_MEDIA_PLAYER / SRC_PHONE_BTAUDIO / HDMI"));
      TunerDataBindingHandler::getDataBindingHandlerInstance()->UpdateSourceIcon((tU8)DEFAULTCASE);
   }
   if (((activeSource == SRC_PHONE) || (activeSource == SRC_PHONE_OUTBAND_RING) || (activeSource == SRC_SPI_ALERT) || (activeSource == SRC_SPI_PHONE) || (activeSource == SRC_SPI_SPEECHRECOGNITION) || (activeSource == SRC_PHONE_SPEECHRECOGNITION)) && (source.getSinkId() != PRIVATE_SPEAKER_SINK_ID))
   {
      ETG_TRACE_USR4(("SrcChangeHandler::activeSource is src_phone_ring "));
      POST_MSG((COURIER_MESSAGE_NEW(Popup_PFO_TAclose)()));
      VehicleSrvClient::poGetInstance()->updateTAStatus(false); //CAN Imp New
      POST_MSG((COURIER_MESSAGE_NEW(Popup_PFO_PTY31close)()));
      POST_MSG((COURIER_MESSAGE_NEW(Popup_PFO_WBclose)()));
   }
   ETG_TRACE_USR4(("SrcChangeHandler::onActiveSourceListUpdate m_SinksPendingSrcChg: %d", m_SinksPendingSrcChg));
   if (((m_u8activeRegion == 2) && (source.getSinkId() == COCKPIT_SINK_ID)) || ((m_u8activeRegion == 0) && (source.getSinkId() == CABIN_A_SINK_ID)))
   {
      if (u8Src_ConnectionState == STATE_CONNECTED)
      {
         switch (activeSource)
         {
            case SRC_TUNER_AM:
            {
               TunerDataBindingHandler::getDataBindingHandlerInstance()->AMHomeScreen();
               _currSourceID = activeSource;
               _tunerSourceConnStatus = update->getActiveSource().getConnectionState();
               TunerDataBindingHandler::getDataBindingHandlerInstance()->UpdateSourceIcon((tU8)VIEW_AM);
               ETG_TRACE_USR4(("onActiveSourceUpdate SRC_TUNER_AM: %d", activeSource));
               Courier::Message* lMsg = COURIER_MESSAGE_NEW(TunerViewMsg)();
               if ((0 != lMsg) && (VIEW_AM != sm_gCurrentView))
               {
                  lMsg->Post();
               }
            }
            break;
            case SRC_TUNER_FM:
            {
               TunerDataBindingHandler::getDataBindingHandlerInstance()->FMAMHomeScreen();
               TunerDataBindingHandler::getDataBindingHandlerInstance()->UpdateSourceIcon((tU8)VIEW_FM1);
               _currSourceID = activeSource;
               _tunerSourceConnStatus = update->getActiveSource().getConnectionState();
               ETG_TRACE_USR4(("onActiveSourceUpdate SRC_TUNER_FM: %d", activeSource));
               Courier::Message* lMsg = COURIER_MESSAGE_NEW(TunerViewMsg)();
               if ((0 != lMsg) && (VIEW_FM1 != sm_gCurrentView))
               {
                  lMsg->Post();
               }
            }
            break;
            case SRC_DAB:
            {
               TunerDataBindingHandler::getDataBindingHandlerInstance()->DABHomeScreen();
               TunerDataBindingHandler::getDataBindingHandlerInstance()->UpdateSourceIcon((tU8)VIEW_DAB);
               _currSourceID = activeSource;
               DABSource = 1;
               _tunerSourceConnStatus = update->getActiveSource().getConnectionState();
               ETG_TRACE_USR4(("onActiveSourceUpdate SRC_DAB: %d", activeSource));
               Courier::Message* lMsg = COURIER_MESSAGE_NEW(TunerViewMsg)();
               if ((0 != lMsg) && (VIEW_DAB != sm_gCurrentView))
               {
                  lMsg->Post();
               }
            }
            break;
            case SRC_TUNER_WX:
            {
               WeatherBandMain::poGetInstance()->presetData();
               TunerDataBindingHandler::getDataBindingHandlerInstance()->WXHomeScreen();
               TunerDataBindingHandler::getDataBindingHandlerInstance()->UpdateSourceIcon((tU8)VIEW_WEATHERBAND);
               _currSourceID = activeSource;
               _tunerSourceConnStatus = update->getActiveSource().getConnectionState();
               ETG_TRACE_USR4(("onActiveSourceUpdate SRC_WX: %d", activeSource));
               Courier::Message* lMsg = COURIER_MESSAGE_NEW(TunerViewMsg)();
               if ((0 != lMsg) && (VIEW_WEATHERBAND != sm_gCurrentView))
               {
                  lMsg->Post();
               }
            }
            break;
            case SRC_TUNER_DRM:
            {
               _currSourceID = activeSource;
               _tunerSourceConnStatus = update->getActiveSource().getConnectionState();
            }
            break;
            case SRC_DTV:
            {
               _currSourceID = activeSource;
               _tunerSourceConnStatus = update->getActiveSource().getConnectionState();
            }
            break;
            case SRC_TUNER_TI_AM:
            {
               _currSourceID = activeSource;
               _tunerSourceConnStatus = update->getActiveSource().getConnectionState();
            }
            break;
            default:
               break;
         }
      }
   }
   if ((((activeSource == SRC_TUNER_TA_DAB) || (activeSource == SRC_TUNER_TA_FM) || (activeSource == SRC_WEATHER_ALERT)) && (u8Src_ConnectionState == STATE_CONNECTED)) \
         || ((activeSource != SRC_PHONE) && (activeSource != SRC_PHONE_OUTBAND_RING) && (activeSource != SRC_SPI_ALERT) && (activeSource != SRC_SPI_PHONE) && (activeSource != SRC_SPI_SPEECHRECOGNITION) && (activeSource != SRC_PHONE_SPEECHRECOGNITION)))
   {
      if (AnnoHandler::poGetInstance()->checkAnnoStatus())
      {
         uint8 popupstatus = AnnoHandler::poGetInstance()->GetPopupStatus();
         ETG_TRACE_USR4(("TunerDataBindingHandler::Anno Status popupstatus %d", popupstatus));
         if (popupstatus == ANNOTYPE_TA)
         {
            AnnoHandler::poGetInstance()->showHideAnnouncement(true, ANNOTYPE_TA); //TA
         }
         else if (popupstatus == ANNOTYPE_PTY)
         {
            AnnoHandler::poGetInstance()->showHideAnnouncement(true, ANNOTYPE_PTY);   //PTY31
         }
         else if (popupstatus == ANNOTYPE_WB)
         {
            AnnoHandler::poGetInstance()->showHideAnnouncement(true, ANNOTYPE_WB);   //WX Alert
         }
      }
   }
   int activesrc = update->getActiveSource().getSrcId();
   if (activesrc == SRC_SPI_ENTERTAIN)
   {
      TunerDataBindingHandler::getDataBindingHandlerInstance()->updateSpidataVisibilityonTunerGadget(true);
   }
   else if (isTunerSourceRange(activesrc))
   {
      TunerDataBindingHandler::getDataBindingHandlerInstance()->updateSpidataVisibilityonTunerGadget(false);
   }

   if (((m_u8activeRegion == 2) && (source.getSinkId() == COCKPIT_SINK_ID)) || ((m_u8activeRegion == 0) && (source.getSinkId() == CABIN_A_SINK_ID)))
   {
      int activesrcSXM = update->getActiveSource().getSrcId();
      bool sxm_value;
      if (((activesrcSXM == SRC_TUNER_XM) || (activesrcSXM == SRC_SXM_NEWS)) && (u8Src_ConnectionState == STATE_CONNECTED))
      {
         _currSourceID = SRC_TUNER_XM;
         sxm_value = true;
         TunerDataBindingHandler::getDataBindingHandlerInstance()->SetSXMActive(true);
      }
      else
      {
         sxm_value = false;
      }
      setSXMStatus(sxm_value);
   }
   //first time activation do nothing
}


void SrcChangeHandler::setCockpitCabinValues(int sinkid, int srcid, int connectionState)
{
   ETG_TRACE_USR4(("SrcChangeHandler::setCockpitCabinValues sinkid: %d, srcid: %d, connectionState: %d", sinkid, srcid, connectionState));

   if (connectionState == STATE_CONNECTED)
   {
      if (sinkid == COCKPIT_SINK_ID)
      {
         m_cockpitSrc = srcid;
         if (((srcid == SRC_TUNER_FM) || (srcid == SRC_TUNER_AM) || (srcid == SRC_DAB) || (srcid == SRC_TUNER_WX)) && ((m_cabinSrc == SRC_TUNER_FM) || (m_cabinSrc == SRC_TUNER_AM) || (m_cabinSrc == SRC_DAB) || (m_cabinSrc == SRC_TUNER_WX)))
         {
            ETG_TRACE_USR4(("SrcChangeHandler::setCockpitCabinValues COCKPIT_SINK_ID srcid: %d", srcid));
            m_cabinSrc = srcid;
            CabinCANSourceData(m_cabinSrc); //CAN Imp
         }
         else if (isMediaSourceRange(m_cockpitSrc))
         {
            CANSourceData(m_cockpitSrc); //CAN Imp
         }
      }
      if (sinkid == CABIN_A_SINK_ID)
      {
         m_cabinSrc = srcid;
         if (((srcid == SRC_TUNER_FM) || (srcid == SRC_TUNER_AM) || (srcid == SRC_DAB) || (srcid == SRC_TUNER_WX)) && ((m_cockpitSrc == SRC_TUNER_FM) || (m_cockpitSrc == SRC_TUNER_AM) || (m_cockpitSrc == SRC_DAB) || (m_cockpitSrc == SRC_TUNER_WX)))
         {
            ETG_TRACE_USR4(("SrcChangeHandler::setCockpitCabinValues CABIN_A_SINK_ID srcid: %d", srcid));
            m_cockpitSrc = srcid;
            CANSourceData(m_cockpitSrc); //CAN Imp
         }
         else if (isMediaSourceRange(m_cabinSrc))
         {
            CabinCANSourceData(m_cabinSrc); //CAN Imp
         }
      }
   }
   if ((srcid == SRC_TUNER_FM) || (srcid == SRC_TUNER_AM) || (srcid == SRC_DAB) || (srcid == SRC_TUNER_WX) || (srcid == SRC_WEATHER_ALERT) || (srcid == SRC_TUNER_TA_FM) || (srcid == SRC_TUNER_TA_DAB))
   {
      ETG_TRACE_USR4(("SrcChangeHandler::setCockpitCabinValues  srcid: %d", srcid));
      if ((srcid == SRC_TUNER_FM) && (fmConnected == false))
      {
         if (amConnected == true)
         {
            amConnected = false;
            RunTimeSrcAM(amConnected);
         }
         else if (dabConnected == true)
         {
            dabConnected = false;
            RunTimeSrcDAB(dabConnected);
         }
         else if (wxConnected == true)
         {
            wxConnected = false;
            RunTimeSrcWX(wxConnected);
         }
         fmConnected = true;
         RunTimeSrcFM(fmConnected);
      }
      else if ((srcid == SRC_TUNER_AM) && (amConnected == false))
      {
         if (fmConnected == true)
         {
            fmConnected = false;
            RunTimeSrcFM(fmConnected);
         }
         else if (dabConnected == true)
         {
            dabConnected = false;
            RunTimeSrcDAB(dabConnected);
         }
         else if (wxConnected == true)
         {
            wxConnected = false;
            RunTimeSrcWX(wxConnected);
         }
         amConnected = true;
         RunTimeSrcAM(amConnected);
      }
      else if ((srcid == SRC_DAB) && (dabConnected == false))
      {
         if (fmConnected == true)
         {
            fmConnected = false;
            RunTimeSrcFM(fmConnected);
         }
         else if (amConnected == true)
         {
            amConnected = false;
            RunTimeSrcAM(amConnected);
         }
         else if (wxConnected == true)
         {
            wxConnected = false;
            RunTimeSrcWX(wxConnected);
         }
         dabConnected = true;
         RunTimeSrcDAB(dabConnected);
      }
      else if ((srcid == SRC_TUNER_WX) && (wxConnected == false))
      {
         if (fmConnected == true)
         {
            fmConnected = false;
            RunTimeSrcFM(fmConnected);
         }
         else if (amConnected == true)
         {
            amConnected = false;
            RunTimeSrcAM(amConnected);
         }
         else if (dabConnected == true)
         {
            dabConnected = false;
            RunTimeSrcDAB(dabConnected);
         }
         wxConnected = true;
         RunTimeSrcWX(wxConnected);
      }
   }
   else
   {
      if (fmConnected == true)
      {
         fmConnected = false;
         RunTimeSrcFM(fmConnected);
      }
      else if (amConnected == true)
      {
         amConnected = false;
         RunTimeSrcAM(amConnected);
      }
      else if (dabConnected == true)
      {
         dabConnected = false;
         RunTimeSrcDAB(dabConnected);
      }
      else if (wxConnected == true)
      {
         wxConnected = false;
         RunTimeSrcWX(wxConnected);
      }
   }
   ETG_TRACE_USR4(("SrcChangeHandler::setCockpitCabinValues m_cockpitSrc: %d, m_cabinSrc: %d", m_cockpitSrc, m_cabinSrc));
}


void SrcChangeHandler::RunTimeSrcFM(bool fmStatus)
{
   static uint32_t fmStartTime = 0;
   static uint32_t fmStopTime = 0;
   bool fmConnStatus = fmStatus;
   if (true == fmConnStatus)
   {
      fmStartTime = OSAL_ClockGetElapsedTime();
      ETG_TRACE_USR4(("SrcChangeHandler::RunTimeSrcFM(), fmStartTime start time : %d ms", fmStartTime));
   }
   else
   {
      fmStopTime = OSAL_ClockGetElapsedTime();
      ETG_TRACE_USR4(("SrcChangeHandler::RunTimeSrcFM(), fmStopTime stop time : %d ms", fmStopTime));
      uint32_t duration = (fmStopTime - fmStartTime) / 1000;   // Convert milliseconds to seconds
      fmStartTime = fmStopTime = 0;

      if (NULL != _DiagnosticsClientHandler)
      {
         _DiagnosticsClientHandler->updateFMRuntimeInfo(duration);
      }
   }
}


void SrcChangeHandler::RunTimeSrcAM(bool amStatus)
{
   static uint32_t amStartTime = 0;
   static uint32_t amStopTime = 0;
   bool amConnStatus = amStatus;
   if (true == amConnStatus)
   {
      amStartTime = OSAL_ClockGetElapsedTime();
      ETG_TRACE_USR4(("SrcChangeHandler::RunTimeSrcAM(), amStartTime start time : %d ms", amStartTime));
   }
   else
   {
      amStopTime = OSAL_ClockGetElapsedTime();
      ETG_TRACE_USR4(("SrcChangeHandler::RunTimeSrcAM(), amStopTime stop time : %d ms", amStopTime));
      uint32_t duration = (amStopTime - amStartTime) / 1000;   // Convert milliseconds to seconds
      amStartTime = amStopTime = 0;

      if (NULL != _DiagnosticsClientHandler)
      {
         _DiagnosticsClientHandler->updateAMRuntimeInfo(duration);
      }
   }
}


void SrcChangeHandler::RunTimeSrcDAB(bool dabStatus)
{
   static uint32_t dabStartTime = 0;
   static uint32_t dabStopTime = 0;
   bool dabConnStatus = dabStatus;
   if (true == dabConnStatus)
   {
      dabStartTime = OSAL_ClockGetElapsedTime();
      ETG_TRACE_USR4(("SrcChangeHandler::RunTimeSrcDAB(), dabStartTime start time : %d ms", dabStartTime));
   }
   else
   {
      dabStopTime = OSAL_ClockGetElapsedTime();
      ETG_TRACE_USR4(("SrcChangeHandler::RunTimeSrcDAB(), dabStopTime stop time : %d ms", dabStopTime));
      uint32_t duration = (dabStopTime - dabStartTime) / 1000;   // Convert milliseconds to seconds
      dabStartTime = dabStopTime = 0;

      if (NULL != _DiagnosticsClientHandler)
      {
         _DiagnosticsClientHandler->updateDABRuntimeInfo(duration);
      }
   }
}


void SrcChangeHandler::RunTimeSrcWX(bool wxStatus)
{
   static uint32_t wxStartTime = 0;
   static uint32_t wxStopTime = 0;
   bool wxConnStatus = wxStatus;
   if (true == wxConnStatus)
   {
      wxStartTime = OSAL_ClockGetElapsedTime();
      ETG_TRACE_USR4(("SrcChangeHandler::RunTimeSrcWX(), wxStartTime start time : %d ms", wxStartTime));
   }
   else
   {
      wxStopTime = OSAL_ClockGetElapsedTime();
      ETG_TRACE_USR4(("SrcChangeHandler::RunTimeSrcWX(), wxStopTime stop time : %d ms", wxStopTime));
      uint32_t duration = (wxStopTime - wxStartTime) / 1000;   // Convert milliseconds to seconds
      wxStartTime = wxStopTime = 0;

      if (NULL != _DiagnosticsClientHandler)
      {
         _DiagnosticsClientHandler->updateWXRuntimeInfo(duration);
      }
   }
}


bool SrcChangeHandler::isShutDown()
{
   ETG_TRACE_USR4(("SrcChangeHandler::isShutDown"));

   if (fmConnected == true)
   {
      fmConnected = false;
      RunTimeSrcFM(fmConnected);
   }
   else if (amConnected == true)
   {
      amConnected = false;
      RunTimeSrcAM(amConnected);
   }
   else if (dabConnected == true)
   {
      dabConnected = false;
      RunTimeSrcDAB(dabConnected);
   }
   else if (wxConnected == true)
   {
      wxConnected = false;
      RunTimeSrcWX(wxConnected);
   }
   return true;
}


bool SrcChangeHandler::isMediaSourceRange(int srcid)
{
   ETG_TRACE_USR4(("SrcChangeHandler::isMediaSourceRange:sourceid %d ", srcid));

   if (((srcid >= SRC_MEDIA_AUX) && (srcid <= SRC_PHONE_BTAUDIO)) || ((srcid >= SRC_CAMERAPORT1_CAM1) && (srcid <= SRC_CAMERAPORT4_HDMI)))
   {
      return true;
   }
   else
   {
      return false;
   }
}


void SrcChangeHandler::setSXMStatus(bool sxm_status)
{
   ETG_TRACE_USR3(("SrcChangeHandler::setSXMStatus() sxm_status:%d", sxm_status));
   sxm_selected_val = sxm_status;
}


bool SrcChangeHandler::getSXMStatus()
{
   ETG_TRACE_USR3(("SrcChangeHandler::getSXMStatus sxm_selected_val:%d", sxm_selected_val));
   return sxm_selected_val;
}


void SrcChangeHandler::setBackBtnStatus(bool backbtn_status)
{
   ETG_TRACE_USR3(("SrcChangeHandler::setBackBtntatus() backbtn_status:%d", backbtn_status));
   backbtn_val = backbtn_status;
}


bool SrcChangeHandler::getBackBtnStatus()
{
   ETG_TRACE_USR3(("SrcChangeHandler::getBackBtntatus backbtn_val:%d", backbtn_val));
   return backbtn_val;
}


uint8 SrcChangeHandler::getsourceInfo()
{
   return _sourceinfo;
}


uint8 SrcChangeHandler::getConnectionState()
{
   return u8Src_ConnectionState;
}


bool SrcChangeHandler::isTunerSourceRange(int srcid)
{
   if ((srcid >= SRC_TUNER_FM) && (srcid <= SRC_TUNER_WX))
   {
      return true;
   }
   else
   {
      return false;
   }
}


/**
* Description     : Virtual function implemented to get update of onActiveSourceListError
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'AudioSourceChangeProxy'
* @param[in]      : error : The error message of onActiveSourceListError
*/
void SrcChangeHandler::onActiveSourceListError(const ::boost::shared_ptr< AudioSourceChangeProxy >& /*proxy*/, const ::boost::shared_ptr< ActiveSourceListError >& /*error*/)
{
   ETG_TRACE_USR3(("SrcChangeHandler::onActiveSourceListError"));
}


/**
* Description     : Virtual function implemented to get update of onActiveSourceListUpdate
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'AudioSourceChangeProxy'
* @param[in]      : error : The signal message of onActiveSourceListSignal
*/
void SrcChangeHandler::onActiveSourceListUpdate(const ::boost::shared_ptr< AudioSourceChangeProxy >& /*proxy*/, const ::boost::shared_ptr< ActiveSourceListUpdate >& update)
{
   ETG_TRACE_USR3(("SrcChangeHandler::onActiveSourceListUpdate()"));

   ::std::vector< sourceData >SourceList = update->getActiveSourceList();
   ::std::vector<sourceData>::const_iterator iter = SourceList.begin(); // stack top is list bottom
   bool isSourceRequired = false;
   _tunerSourceConnStatus = DISCONNECT;
   int32 audioSourceConnStatus;
   while (iter != SourceList.end())
   {
      _sourceinfo = _AudioSourceID = iter->getSrcId();
      audioSourceConnStatus = iter->getConnectionState();

      ETG_TRACE_USR3(("SrcChangeHandler::_AudioSourceID:%d audioSourceConnStatus:%d _sourceinfo %d", _AudioSourceID, audioSourceConnStatus, _sourceinfo));

      if (((_AudioSourceID == SRC_TUNER_FM) || (_AudioSourceID == SRC_TUNER_AM) || (_AudioSourceID == SRC_DAB) ||
            (_AudioSourceID == SRC_TUNER_WX) || (_AudioSourceID == SRC_TUNER_XM)) && (audioSourceConnStatus == CONNECTED_TO_SINK))
      {
         _tunerSourceConnStatus = audioSourceConnStatus;
         ETG_TRACE_USR4(("vcurrActiveAudioSource=%d", _AudioSourceID));
         isSourceRequired = true;

         if (_AudioSourceID != _currSourceID)
         {
            _currSourceID = _AudioSourceID;
         }
         break;
      }

      iter++;
   }
   if (NULL != _tunerDataBindingHandler)
   {
      ETG_TRACE_USR4(("onActiveSourceListUpdate Send Update"))
      _tunerDataBindingHandler->updatedata();
   }
   else
   {
      ETG_TRACE_USR4(("onActiveSourceListUpdate _tunerDataBindingHandler pointer Null"))
   }
}


void SrcChangeHandler::CANSourceData(int32 sourceID)
{
   ETG_TRACE_USR4(("SrcChangeHandler::CANSourceData(%d)", sourceID));

   switch (sourceID)
   {
      case SRC_TUNER_FM:
         VehicleSrvClient::poGetInstance()->updateTunerBandStatus(CAN_FM); //CAN Imp
         break;

      case SRC_TUNER_AM:
         VehicleSrvClient::poGetInstance()->updateTunerBandStatus(CAN_MW); //CAN Imp
         break;

      case SRC_DAB:
         VehicleSrvClient::poGetInstance()->updateTunerBandStatus(CAN_DAB); //CAN Imp
         break;

      case SRC_TUNER_WX:
         VehicleSrvClient::poGetInstance()->updateTunerBandStatus(CAN_WB); //CAN Imp
         break;

      case SRC_MEDIA_AUX:
      case SRC_MEDIA_PLAYER:
      case SRC_MEDIA_CDDA:
      case SRC_PHONE_BTAUDIO:
      case SRC_CAMERAPORT1_CAM1:
      case SRC_CAMERAPORT1_CAM2:
      case SRC_CAMERAPORT1_CAM3:
      case SRC_CAMERAPORT1_CAM4:
      case SRC_CAMERAPORT1_IPCAM:
      case SRC_CAMERAPORT1_SPLITSCREEN:
      case SRC_CAMERAPORT1_USB_VIDEO:
      case SRC_CAMERAPORT1_HDMI:
      case SRC_CAMERAPORT2_CAM1:
      case SRC_CAMERAPORT2_CAM2:
      case SRC_CAMERAPORT2_CAM3:
      case SRC_CAMERAPORT2_CAM4:
      case SRC_CAMERAPORT2_IPCAM:
      case SRC_CAMERAPORT2_SPLITSCREEN:
      case SRC_CAMERAPORT2_USB_VIDEO:
      case SRC_CAMERAPORT2_HDMI:
      case SRC_CAMERAPORT3_CAM1:
      case SRC_CAMERAPORT3_CAM2:
      case SRC_CAMERAPORT3_CAM3:
      case SRC_CAMERAPORT3_CAM4:
      case SRC_CAMERAPORT3_IPCAM:
      case SRC_CAMERAPORT3_SPLITSCREEN:
      case SRC_CAMERAPORT3_USB_VIDEO:
      case SRC_CAMERAPORT3_HDMI:
      case SRC_CAMERAPORT4_CAM1:
      case SRC_CAMERAPORT4_CAM2:
      case SRC_CAMERAPORT4_CAM3:
      case SRC_CAMERAPORT4_CAM4:
      case SRC_CAMERAPORT4_IPCAM:
      case SRC_CAMERAPORT4_SPLITSCREEN:
      case SRC_CAMERAPORT4_USB_VIDEO:
      case SRC_CAMERAPORT4_HDMI:
         VehicleSrvClient::poGetInstance()->updateTunerBandStatus(CAN_SNA); //CAN Imp
         break;

      default:
         break;
   }
}


void SrcChangeHandler::CabinCANSourceData(int32 sourceID)
{
   ETG_TRACE_USR4(("SrcChangeHandler::CabinCANSourceData(%d)", sourceID));

   switch (sourceID)
   {
      case SRC_TUNER_FM:
         VehicleSrvClient::poGetInstance()->updateCabinTunerBandStatus(CAN_FM); //CAN Imp
         break;

      case SRC_TUNER_AM:
         VehicleSrvClient::poGetInstance()->updateCabinTunerBandStatus(CAN_MW); //CAN Imp
         break;

      case SRC_DAB:
         VehicleSrvClient::poGetInstance()->updateCabinTunerBandStatus(CAN_DAB); //CAN Imp
         break;

      case SRC_TUNER_WX:
         VehicleSrvClient::poGetInstance()->updateCabinTunerBandStatus(CAN_WB); //CAN Imp
         break;

      case SRC_MEDIA_AUX:
      case SRC_MEDIA_PLAYER:
      case SRC_MEDIA_CDDA:
      case SRC_PHONE_BTAUDIO:
      case SRC_CAMERAPORT1_CAM1:
      case SRC_CAMERAPORT1_CAM2:
      case SRC_CAMERAPORT1_CAM3:
      case SRC_CAMERAPORT1_CAM4:
      case SRC_CAMERAPORT1_IPCAM:
      case SRC_CAMERAPORT1_SPLITSCREEN:
      case SRC_CAMERAPORT1_USB_VIDEO:
      case SRC_CAMERAPORT1_HDMI:
      case SRC_CAMERAPORT2_CAM1:
      case SRC_CAMERAPORT2_CAM2:
      case SRC_CAMERAPORT2_CAM3:
      case SRC_CAMERAPORT2_CAM4:
      case SRC_CAMERAPORT2_IPCAM:
      case SRC_CAMERAPORT2_SPLITSCREEN:
      case SRC_CAMERAPORT2_USB_VIDEO:
      case SRC_CAMERAPORT2_HDMI:
      case SRC_CAMERAPORT3_CAM1:
      case SRC_CAMERAPORT3_CAM2:
      case SRC_CAMERAPORT3_CAM3:
      case SRC_CAMERAPORT3_CAM4:
      case SRC_CAMERAPORT3_IPCAM:
      case SRC_CAMERAPORT3_SPLITSCREEN:
      case SRC_CAMERAPORT3_USB_VIDEO:
      case SRC_CAMERAPORT3_HDMI:
      case SRC_CAMERAPORT4_CAM1:
      case SRC_CAMERAPORT4_CAM2:
      case SRC_CAMERAPORT4_CAM3:
      case SRC_CAMERAPORT4_CAM4:
      case SRC_CAMERAPORT4_IPCAM:
      case SRC_CAMERAPORT4_SPLITSCREEN:
      case SRC_CAMERAPORT4_USB_VIDEO:
      case SRC_CAMERAPORT4_HDMI:
         VehicleSrvClient::poGetInstance()->updateCabinTunerBandStatus(CAN_SNA); //CAN Imp
         break;

      default:
         break;
   }
}


///**
// * Description     : Function implemented to activate source of given sourceID.
// *
// * @param[in]      : sourceID : source ID
// * @return         : void
// */
void SrcChangeHandler::activateSource(int32& sourceID)
{
   ETG_TRACE_USR3(("SrcChangeHandler::activateSource(%d)", sourceID));
   ETG_TRACE_USR3(("SrcChangeHandler::CurrentactivateSource(%d)", _currSourceID));
   ETG_TRACE_USR3(("SrcChangeHandler::ActiveRegionID(%d)", m_u8activeRegion));
   if (m_u8activeRegion == REGION_COCKPIT)
   {
      CANSourceData(sourceID); //CAN Imp
   }
   else if (m_u8activeRegion == REGION_CABIN_A)
   {
      CabinCANSourceData(sourceID); //CAN Imp
   }
   sm_gAudioSource = sourceID;
   _sourceinfo = sourceID;
   int sinkId = 0;
   ((m_u8activeRegion == 2) ? sinkId = COCKPIT_SINK_ID : ((m_u8activeRegion == 0) ?  sinkId = CABIN_A_SINK_ID :  sinkId = CABIN_B_SINK_ID));
   sourceData srcData;

   srcData.setSrcId(sourceID);
   if (_sourceinfo == SRC_TUNER_WX)
   {
      srcData.setSubSrcId(INITIALISE_SUBSRC_ID_WX);//for weatherband
   }
   else
   {
      srcData.setSubSrcId(INITIALISE_SUBSRC_ID);
   }
   srcData.setConnectionState(0);
   // uint8 SrcChgInSink = IsSrcChgReqdInOtherSinks(sourceID, sinkId);
   /* uint8 SrcChgInSink = 0;
    if (SrcChgInSink & (1 << CABIN_A_SINK_ID))
    {
       disconnect_src = false;
       ETG_TRACE_USR3(("check 1"));
       srcData.setSinkId(CABIN_A_SINK_ID);
       ((_audioSrcChgProxy.get()) && (_audioSrcChgProxy->sendActivateSourceRequest(*this, srcData, true)));
       m_SinksPendingSrcChg = ((SrcChgInSink | (1 << sinkId)) & (~(1 << CABIN_A_SINK_ID)));
    }
    else if (SrcChgInSink & (1 << CABIN_B_SINK_ID))
    {
       ETG_TRACE_USR3(("check 2"));
       srcData.setSinkId(CABIN_B_SINK_ID);
       ((_audioSrcChgProxy.get()) && (_audioSrcChgProxy->sendActivateSourceRequest(*this, srcData, true)));
       m_SinksPendingSrcChg = ((SrcChgInSink | (1 << sinkId)) & (~(1 << CABIN_B_SINK_ID)));
    }
    else if (SrcChgInSink & (1 << COCKPIT_SINK_ID))
    {
       ETG_TRACE_USR3(("check 3"));
       srcData.setSinkId(COCKPIT_SINK_ID);
       ((_audioSrcChgProxy.get()) && (_audioSrcChgProxy->sendActivateSourceRequest(*this, srcData, true)));
       m_SinksPendingSrcChg = ((SrcChgInSink | (1 << sinkId)) & (~(1 << COCKPIT_SINK_ID)));
    }
    else
    {
       ETG_TRACE_USR3(("check 4"));*/
   if (sinkId == CABIN_A_SINK_ID)
   {
      disconnect_src = false;
   }
   srcData.setSinkId(sinkId);
   ((_audioSrcChgProxy.get()) && (_audioSrcChgProxy->sendActivateSourceRequest(*this, srcData, true)));
//  }
//  ETG_TRACE_USR4(("SrcChangeHandler::activateSource m_SinksPendingSrcChg: %d", m_SinksPendingSrcChg));
}


uint8 SrcChangeHandler::IsSrcChgReqdInOtherSinks(int32 ReqSrcID, int32 ReqSinkId)
{
   uint8 SrcChgreqInSink = 0x00;
   ETG_TRACE_USR4(("SrcChangeHandler::IsSrcChgReqdInOtherSinks ReqSrcId: %d  ReqSinkId: %d", ReqSrcID, ReqSinkId));
   ETG_TRACE_USR3(("SrcChangeHandler::IsSrcChgReqdInOtherSinks m_CurrentSrc[0]:%d m_CurrentSrc[1]:%d  m_CurrentSrc[2]:%d", m_CurrentSrc[0], m_CurrentSrc[1], m_CurrentSrc[2]));

   for (uint8 i = 0; i < CABIN_B_SINK_ID; i++)

   {
      ETG_TRACE_USR3(("SrcChangeHandler::IsSrcChgReqdInOtherSinks m_CurrentSrc[i]:%d ", m_CurrentSrc[i]));
      if ((i != ReqSinkId)  && ((ReqSrcID < 10) == (m_CurrentSrc[i] < 10)))

         //Order changes to send last activation type
      {
         if (ReqSrcID != m_CurrentSrc[i])

         {
            SrcChgreqInSink = SrcChgreqInSink | (1 << i);
         }
         //alreadyActivated = true;
      }
      //ignore
   }
   ETG_TRACE_USR4(("SrcChangeHandler::IsSrcChgReqdInOtherSinks SrcChgreqInSink: %d", SrcChgreqInSink));
   return SrcChgreqInSink;
}


/**
* onMuteStateError - This status is received from HMI_Master if there are any errors in muteState.
* @param[in] proxy
* @parm[in] status
* @return void
*/

void SrcChangeHandler::onMuteStateError(const ::boost::shared_ptr< SoundPropertiesProxy >& /*proxy*/, const ::boost::shared_ptr< MuteStateError >& /*error*/)
{
   ETG_TRACE_USR3((" SrcChangeHandler::onMuteStateError"));
}


/**
* onMuteStateUpdate - This status is received from HMI_Master if there is any change in property mutestate.
* @param[in] proxy
* @parm[in] status
* @return void
*/

void SrcChangeHandler::onMuteStateUpdate(const ::boost::shared_ptr< SoundPropertiesProxy >& /*proxy*/, const ::boost::shared_ptr< MuteStateUpdate >& update)
{
   bool bMuteStatus = update->getMuteState();
   ETG_TRACE_USR3((" SrcChangeHandler::onMuteStateUpdate %d", bMuteStatus));
   //if (bMuteStatus && (NULL != _srcChgdataBinder))
   //{
   //   //Update consumers on active mute state
   //   _srcChgdataBinder->updateEntMuteActive();
   //}
   //if ((_AudioSourceID == SRC_TUNER_FM) || (_AudioSourceID == SRC_TUNER_AM) || (_AudioSourceID == SRC_DAB) || (_AudioSourceID == SRC_TUNER_DRM) || (_AudioSourceID == SRC_TUNER_TI_AM))
   //{
   //   POST_MSG((COURIER_MESSAGE_NEW(EntertainmentMute)(bMuteStatus, _AudioSourceID)));
   //}
}


void SrcChangeHandler::onMuteStateMapError(const ::boost::shared_ptr< SoundPropertiesProxy >& /*proxy*/, const ::boost::shared_ptr< MuteStateMapError >& error)
{
   ETG_TRACE_USR4(("SrcChangeHandler:: onMuteStateMapError() \n"));
}


void SrcChangeHandler::onMuteStateMapUpdate(const ::boost::shared_ptr< SoundPropertiesProxy >& /*proxy*/, const ::boost::shared_ptr< MuteStateMapUpdate >& update)
{
   m_MuteStatusMap = update->getMuteStateMap();
   std::map<uint16, bool>::iterator itr;

   for (std::map<uint16, bool>::iterator itr = m_MuteStatusMap.begin(); itr != m_MuteStatusMap.end(); ++itr)
   {
      ETG_TRACE_USR4(("SrcChangeHandler::onMuteStateMapUpdate %d", itr->first));
      ETG_TRACE_USR4(("SrcChangeHandler::onMuteStateMapUpdate %d", itr->second));
   }

   bool _bMutestatus = getCurrentRegionMuteStatus();
   TunerDataBindingHandler::getDataBindingHandlerInstance()->TunerPLMsg(EV_MSG_TO_PL_MUTE, _bMutestatus, "");
}


bool SrcChangeHandler::getCurrentRegionMuteStatus()
{
   ETG_TRACE_USR4(("SrcChangeHandler::getCurrentRegionMuteStatus enetered "));
   bool bMuteStatus = false;
   uint8 iActiveRegion = REGION_INVALID;

   uint8 u8activeRegion = getActiveRegionValue();
   ETG_TRACE_USR4(("SrcChangeHandler::getCurrentRegionMuteStatus u8activeRegion cabin  %d ", u8activeRegion));

   std::map<uint16, bool>::iterator itr;
   for (std::map<uint16, bool>::iterator itr = m_MuteStatusMap.begin(); itr != m_MuteStatusMap.end(); ++itr)
   {
      ETG_TRACE_USR4(("SrcChangeHandler::Mutestatus %d", itr->second));
      if (itr->second == true)
      {
         ETG_TRACE_USR4(("SrcChangeHandler::getCurrentRegionMuteStatusMutestatus %d", itr->first));
         int iSinkId = itr->first;
         iActiveRegion = ((iSinkId == COCKPIT_SINK_ID) ? REGION_COCKPIT : ((iSinkId == CABIN_A_SINK_ID) ? REGION_CABIN_A : REGION_CABIN_B));
         ETG_TRACE_USR4(("SrcChangeHandler::getCurrentRegionMuteStatus iActiveRegion  %d ", iActiveRegion));
         if (iActiveRegion == u8activeRegion)
         {
            bMuteStatus = true;
            break;
         }
      }
   }
   ETG_TRACE_USR4(("SrcChangeHandler::getCurrentRegionMuteStatusMutestatus bMuteStatus %d", bMuteStatus));
   return bMuteStatus;
}


void SrcChangeHandler::onAudioStateInfoError(const ::boost::shared_ptr< SoundPropertiesProxy >& /*proxy*/, const ::boost::shared_ptr< AudioStateInfoError >& /*error*/)
{
   ETG_TRACE_USR4(("SrcChangeHandler::onAudioStateInfoError"));
}


void SrcChangeHandler::onAudioStateInfoUpdate(const ::boost::shared_ptr< SoundPropertiesProxy >& /*proxy*/, const ::boost::shared_ptr< AudioStateInfoUpdate >& update)
{
   //uint8 _bMuteStatus = update->getAudioStateInfo().getAudioOnOffState();
   bool _bMutestatus = update->getAudioStateInfo().getMuteState();
   ETG_TRACE_USR4(("SrcChangeHandler::onAudioStateInfoUpdate:Mutestatus %d", _bMutestatus));

   // TunerDataBindingHandler::getDataBindingHandlerInstance()->TunerPLMsg(EV_MSG_TO_PL_MUTE, _bMutestatus, "");
}


void SrcChangeHandler::onActiveRegionIdError(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< ActiveRegionIdError >& error)
{
   ETG_TRACE_USR4(("SrcChangeHandler::onActiveRegionIdError"));
}


void SrcChangeHandler::onActiveRegionIdUpdate(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< ActiveRegionIdUpdate >& update)
{
   m_u8activeRegion = update->getActiveRegionId();
   ETG_TRACE_USR4(("SrcChangeHandler::m_u8activeRegion: %d", m_u8activeRegion));

   bool _bMutestatus = getCurrentRegionMuteStatus();
   ETG_TRACE_USR4(("SrcChangeHandler::Mutestatus %d", _bMutestatus));

   TunerDataBindingHandler::getDataBindingHandlerInstance()->TunerPLMsg(EV_MSG_TO_PL_MUTE, _bMutestatus, "");
   ETG_TRACE_USR4(("SrcChangeHandler::m_cockpitSrc: %d, m_cabinSrc: %d", m_cockpitSrc, m_cabinSrc));

   bool sxm_value;
   if (m_u8activeRegion == 2)
   {
      if (m_cockpitSrc == SRC_TUNER_FM)
      {
         sxm_value = false;
         setSXMStatus(sxm_value);
         TunerDataBindingHandler::getDataBindingHandlerInstance()->UpdateSourceIcon((tU8)VIEW_FM1);
      }
      else if (m_cockpitSrc == SRC_TUNER_AM)
      {
         sxm_value = false;
         setSXMStatus(sxm_value);
         TunerDataBindingHandler::getDataBindingHandlerInstance()->UpdateSourceIcon((tU8)VIEW_AM);
      }
      else if (m_cockpitSrc == SRC_DAB)
      {
         sxm_value = false;
         setSXMStatus(sxm_value);
         TunerDataBindingHandler::getDataBindingHandlerInstance()->UpdateSourceIcon((tU8)VIEW_DAB);
      }
      else if (m_cockpitSrc == SRC_TUNER_WX)
      {
         sxm_value = false;
         setSXMStatus(sxm_value);
         TunerDataBindingHandler::getDataBindingHandlerInstance()->UpdateSourceIcon((tU8)VIEW_WEATHERBAND);
      }
      else if ((m_cockpitSrc == SRC_TUNER_XM) || (m_cockpitSrc == SRC_SXM_NEWS))
      {
         sxm_value = true;
         TunerDataBindingHandler::getDataBindingHandlerInstance()->SetSXMActive(true);
         setSXMStatus(sxm_value);
      }
      else
      {
         TunerDataBindingHandler::getDataBindingHandlerInstance()->UpdateSourceIcon((tU8)DEFAULTCASE);
      }
   }

   if (m_u8activeRegion == 0)
   {
      if (m_cabinSrc == SRC_TUNER_FM)
      {
         sxm_value = false;
         setSXMStatus(sxm_value);
         TunerDataBindingHandler::getDataBindingHandlerInstance()->UpdateSourceIcon((tU8)VIEW_FM1);
      }
      else if (m_cabinSrc == SRC_TUNER_AM)
      {
         sxm_value = false;
         setSXMStatus(sxm_value);
         TunerDataBindingHandler::getDataBindingHandlerInstance()->UpdateSourceIcon((tU8)VIEW_AM);
      }
      else if (m_cabinSrc == SRC_DAB)
      {
         sxm_value = false;
         setSXMStatus(sxm_value);
         TunerDataBindingHandler::getDataBindingHandlerInstance()->UpdateSourceIcon((tU8)VIEW_DAB);
      }
      else if (m_cabinSrc == SRC_TUNER_WX)
      {
         sxm_value = false;
         setSXMStatus(sxm_value);
         TunerDataBindingHandler::getDataBindingHandlerInstance()->UpdateSourceIcon((tU8)VIEW_WEATHERBAND);
      }
      else if ((m_cabinSrc == SRC_TUNER_XM) || (m_cabinSrc == SRC_SXM_NEWS))
      {
         sxm_value = true;
         TunerDataBindingHandler::getDataBindingHandlerInstance()->SetSXMActive(true);
         setSXMStatus(sxm_value);
      }
      else
      {
         TunerDataBindingHandler::getDataBindingHandlerInstance()->UpdateSourceIcon((tU8)DEFAULTCASE);
      }
   }
}


uint8 SrcChangeHandler::getActiveRegionValue()
{
   return m_u8activeRegion;
}


/**
* Description     : Function called by GUI, when user requests for next band.
*
* @return         : void
*/
void SrcChangeHandler::onRequestNextBand()
{
   int32 _sourceID;
   if (_currSourceID == SRC_TUNER_FM)
   {
      _sourceID = SRC_TUNER_AM;
      _currSourceID  = SRC_TUNER_AM;
   }
   else
   {
      _sourceID = SRC_TUNER_FM;
      _currSourceID  = SRC_TUNER_FM;
   }
   activateSource(_sourceID);
}


} // namespace Core
} // namespace App
