
/**************************************************************************************
* @file         : TunerDataBindingHandler.cpp
* @author       : ECV2-Rohini S R
* @addtogroup   : Tuner HALL
* @brief        : Class contains utility functions to handle DataBinding messages required for AM FM Info and HD data.
* @copyright    : (c) 2019-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "hall_std_if.h"
#include "WeatherBandMain.h"
#include "TunerDataBindingHandler.h"
#include "CgiExtensions/ImageLoader.h"
#include "App/Core/AMFMInfo/AMFMStationListTimer.h"
#include "AMFMInfo.h"
#include "AMFMHDdata.h"
#include "AMFMInfoCourierMsgHandler.h"
#include "../VehicleSrvClient/VehicleSrvClient.h"
#include "../RadioSettings/RadioSetting.h"
#include "../SourceChangeHandler/SrcChangeHandler.h"
#include "AppHmi_MasterBase/AudioInterface/AudioDefines.h"
#include "../AnnouncementHandler/AnnoHandler.h"
#include "Common/VariantHandling/VariantHandling.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_TUNER_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_TUNER
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_Tuner_"
#define ETG_I_FILE_PREFIX                 App::Core::TunerDataBindingHandler::
#include "trcGenProj/Header/TunerDataBindingHandler.cpp.trc.h"
#endif

//extern int sm_gCurrentView;

/*****************************************************************
| defines and macros (scope: local)
|----------------------------------------------------------------*/


#define C_FM_FREQ_UNIT "MHz"
#define C_AM_FREQ_UNIT "kHz"
#define C_WAVE_BAND_INVALID -1

#define TIMER_SLIDER_VAL_INTERVAL  300

#define TIMER_VAL_INTERVAL  30
#define TOTAL_TIME_COUNT    50
#define MAX_POPUP_TEXT      50

#define MUTE 1
#define TA_ICON      "TunerModule#Images#Common#Icon_TA"
#define TA_CROSSED   "TunerModule#Images#Common#Icon_TA_CrossedOut"
#define SOURCE_RADIO   "TunerModule#Imports#Images#Icon_Radio"
#define SOURCE_MEDIA   "TunerModule#Imports#Images#Icon_Source"

//EU
#define EU_9k                       "TunerModule#Imports#Images#Icon_TuningScale_AM_522-1611_Scale"
#define AM_SLIDER_1611              1611
#define AM_BG_1611                  "TunerModule#Imports#Images#Icon_TuningScale_AM_522-1611_BackGround"
//India Australia NewZealand Thai
#define GEX_9k                      "TunerModule#Imports#Images#Icon_TuningScale_AM_531-1629_scale"
#define AM_SLIDER_1629              1629
#define AM_BG_1629                  "TunerModule#Imports#Images#Icon_TuningScale_AM_531-1629_BackGround"
//Mexico
#define GEX_10k                     "TunerModule#Imports#Images#Icon_TuningScale_AM_530-1710_Scale"
#define AM_SLIDER_1710              1710
#define AM_BG_1710                  "TunerModule#Imports#Images#Icon_TuningScale_AM_530-1710_BackGround"

//China Variant SUZUKI
#define AM_SLIDER_1620              1620

#define MAX_FM_PRESETS              12
#define MAX_DAB_PRESETS             12
#define MAX_AM_PRESETS              12

//testing
#define MAX_FM_PRESETS_BANK         6

using namespace ::midw_tuner_fi_types;

namespace App {
namespace Core {

TunerDataBindingHandler* TunerDataBindingHandler::m_tunerDataBindingHandler = NULL;

tU8 g_e8BrowseViewID = 0xFF;

/*****************************************************************
| Template function implementation (scope: global)

|----------------------------------------------------------------*/

/************************************************************************
*FUNCTION: 		TTMonitorSet1DataUpdate
*RETURNVALUE: 	void
*
*HISTORY:
*05.03.13 Karthik
*Rev 1.0 Initial Revision.
************************************************************************/

template <class type>
void TTMonitorSet1DataUpdate(type& oTTMonitor, const TripleTunerMonitorDataSet1* poTunerEngineeringModeMsg)
{
   tChar szData[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
   ETG_TRACE_USR3(("TTMonitorSet1DataUpdate updation"));
   /* Retrieving the frequency from the message and updating the binding source */

   ETG_TRACE_USR3(("TTMonitorSet1DataUpdate GetFrequency: %s", (poTunerEngineeringModeMsg->GetFrequency()).GetCString()));
   (*oTTMonitor).mFreq = poTunerEngineeringModeMsg->GetFrequency();

   ETG_TRACE_USR3(("TTMonitorSet1DataUpdate GetPSName: %s", (poTunerEngineeringModeMsg->GetPSName()).GetCString()));
   (*oTTMonitor).mPS = poTunerEngineeringModeMsg->GetPSName();

   sprintf(szData, "%d", poTunerEngineeringModeMsg->GetPI());
   (*oTTMonitor).mPI = szData;

   sprintf(szData, "%d/%d", poTunerEngineeringModeMsg->GetU8Fieldstrength(), poTunerEngineeringModeMsg->GetU8Quality());
   (*oTTMonitor).mFSQ = szData;
   ETG_TRACE_USR3(("TTMonitorSet1DataUpdate GetU8Fieldstrength: %s", szData));

   sprintf(szData, "%d", poTunerEngineeringModeMsg->GetU8Sharx());
   (*oTTMonitor).mSharx = szData;
   ETG_TRACE_USR3(("TTMonitorSet1DataUpdate GetU8Sharx: %s", szData));

   sprintf(szData, "%d", poTunerEngineeringModeMsg->GetU8HiCut());
   (*oTTMonitor).mHicut = szData;

   sprintf(szData, "%d", poTunerEngineeringModeMsg->GetU8FMDeviation());
   (*oTTMonitor).mDev = szData;

   sprintf(szData, "%d", poTunerEngineeringModeMsg->GetU8ChannelSeparation());
   (*oTTMonitor).mCS = szData;

   /* Mark all items for updation */
   oTTMonitor.MarkAllItemsModified();

   bool bDataUpdated = oTTMonitor.SendUpdate(true);
   if (false == bDataUpdated)
   {
      ETG_TRACE_USR3(("ERROR: TTMonitorSet1Data Data updation failed"));
   }
}


/*****************************************************************
| Template function implementation (scope: global)

|----------------------------------------------------------------*/

/************************************************************************
*FUNCTION: 		TTMonitorSet2DataUpdate
*RETURNVALUE: 	void
*
*HISTORY:
*05.03.13 Karthik
*Rev 1.0 Initial Revision.
************************************************************************/
template <class type1>
void TTMonitorSet2DataUpdate(type1& oTTMonitor, const TripleTunerMonitorDataSet2* poTunerEngineeringModeMsg)
{
   tChar szData[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

   /* Retrieving the frequency from the message and updating the binding source */
   sprintf(szData, "%d", poTunerEngineeringModeMsg->GetFMRds());
   (*oTTMonitor).mRDS = szData;

   sprintf(szData, "%d", poTunerEngineeringModeMsg->GetFMTPFlag());
   (*oTTMonitor).mTP = szData;

   sprintf(szData, "%d", poTunerEngineeringModeMsg->GetFMTAFlag());
   (*oTTMonitor).mTA = szData;

   sprintf(szData, "%d", poTunerEngineeringModeMsg->GetFMAlterFreq());
   (*oTTMonitor).mAF = szData;

   sprintf(szData, "%d/%d", poTunerEngineeringModeMsg->GetFMMultipath(), poTunerEngineeringModeMsg->GetFMNeighbour());
   (*oTTMonitor).mMPNC = szData;
   ETG_TRACE_USR3(("TTMonitorSet1DataUpdate GetFrequency: %s", szData));

   /* Mark all items for updation */
   oTTMonitor.MarkAllItemsModified();

   bool bDataUpdated = oTTMonitor.SendUpdate(true);
   if (false == bDataUpdated)
   {
      ETG_TRACE_USR3(("ERROR: TTMonitorSet2DataUpdate Update Data failed"));
   }
}


/*****************************************************************
| function implementation (scope: global)
|----------------------------------------------------------------*/

/************************************************************************
*FUNCTION: 		TunerDataBindingHandler
*DESCRIPTION: 	Constructor
*PARAMETER:		None
*RETURNVALUE: 	void
*
*HISTORY:
*05.03.13 Shrivani
*Rev 1.0 Initial Revision.
************************************************************************/

TunerDataBindingHandler::TunerDataBindingHandler()
   : m_oTimerPreset(*this, TIMER_VAL_INTERVAL)
   , _hdStationLogoTimer(*this, HD_STATION_LOGO_TIMER)
   , m_oTimerSlider(*this, TIMER_SLIDER_VAL_INTERVAL)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::TunerDataBindingHandler()"));
   // Initialize _isCabinOnly based on the VariantType
   if (VariantHandling::getInstance() != NULL)
   {
      _isCabinOnly = (VariantHandling::getInstance()->getVariantType() == VARIANTTYPE_CC);
   }
   else
   {
      _isCabinOnly = false; // Default to false if VariantHandling is not available
   }
   ETG_TRACE_USR3(("TunerDataBindingHandler::TunerDataBindingHandler _isCabinOnly=%d", _isCabinOnly));
   m_u8popupState = (tU8)EN_POPUP_DISPLAY; //Value is 0
   m_u8FMFocusedPresestNumber = (tU8)EN_PRESET_NONE;
   m_u8AMFocusedPresestNumber = (tU8)EN_PRESET_NONE;
   m_u8timerCount = 0;
   m_u8CurrentPresetNr = 0;
   m_bIsTASourceDAB = false;
   m_bIsPTYSourceDAB = false;
   m_sFMTASourcePSName = "";
   m_sDABTASourcePSName = "";
   m_sfrequencyvalue = "";
   m_amFrequencyvalue = "";
   m_FMStationName = "";
   m_u8popupState = (tU8)EN_POPUP_DISPLAY;//Value is 0
   _hdAlbumArtVisibilityStatus = false;
   _isHDAlbumArtSet = false;
   _radioMainText = "";
   (*oTunerPopupData).mVisible_Txt_Wgt_1 = false;
   (*oTunerPopupData).mVisible_Txt_Wgt_2 = false;
   (*oRadioBandInfoData).mVisible_Txt_Wgt_TA = false;
   (*oRadioBandInfoData).mVisible_Txt_Wgt_TA_In_AM = false;
   (*oRadioBandInfoData).mVisible_Txt_Wgt_TA_FM = false;
   (*oRadioBandInfoData).mVisible_Txt_Wgt_TA_DAB = false;
   (*oTunerInfoData).mAMFrequency = "";
   (*oTunerInfoData).mFMFrequency = "";
   (*oHomeScreenInfo).mHomeFrequency = "";
   (*oTunerInfoData).mUnit = "";
   (*oTunerInfoData).mTPInfo = false;
   (*oTunerInfoData).mDabTPInfo = false;
   (*oTunerInfoData).mCurrentStationTPInfo = false;
   (*oTunerInfoData).mPSName = "";
   (*oTunerInfoData).mViewName = ""; /*"%1";*/
   (*oTunerInfoData).mRadioText = "";
   (*oTunerInfoData).mVisible_Txt_Wgt_RDS = false;//Display RDS info
   (*oTunerInfoData).mVisible_Txt_Wgt_NoRDS = false;//No RDS info
   (*oAppFlyIn).mPhoneSPILink = false;
   (*oTunerInfoData).mVisible_Txt_Wgt_Navi = false;
   (*oTunerInfoData).mVariant_Type = false;
   (*oTunerInfoData).mVariant_Type_NonEU = true;
   (*oHomeScreenInfo).mHomeScreen_MuteIconStatus = false;
   (*oDABChannelInfo).mTpStatus = "";
   (*oDABChannelInfo).mDabStatus = "";
   (*oDABChannelInfo).mEnsembleLabel = "---";
   (*oDABChannelInfo).mServiceLabel = "---";
   (*oDABChannelInfo).mRadioText = "---";
   (*oDABChannelInfo).mRadioTextVisible = false;
   (*oChannelInfo).mAdvisoryText = "";
   (*oChannelInfo).mArtist = "";
   (*oChannelInfo).mAlbum = "";
   (*oChannelInfo).mTrack = "";
   (*oChannelInfo).mCategory = "";
   (*oChannelInfo).mArtistAlbumSwitch = static_cast<Courier::UInt8>(TUNER_HD_DATA_NONE);
   /*
   * Active bank information
   */
   (*oActiveBankInfo).mFM1_Active = false;
   (*oActiveBankInfo).mFM2_Active = false;
   (*oActiveBankInfo).mFMAST_Active = false;
   (*oActiveBankInfo).mAM_Active = false;
   (*oActiveBankInfo).mAMAST_Active = false;
   (*oActiveBankInfo).mDAB_Active = false;
   (*oActiveBankInfo).mDAB_AST_Active = false;

   /*
   * Variant information
   */
   (*oVariantInfo).mDAB_Available = false;
   (*oVariantInfo).mHD_Available = false;
   (*oVariantInfo).mSXM_Mounted = false;
   (*oVariantInfo).mWX_Available = false;
   (*oDABPresetInfo).mPreset_1_Active = false;
   (*oDABPresetInfo).mPreset_2_Active = false;
   (*oDABPresetInfo).mPreset_3_Active = false;
   (*oDABPresetInfo).mPreset_4_Active = false;
   (*oDABPresetInfo).mPreset_5_Active = false;
   (*oDABPresetInfo).mPreset_6_Active = false;

   (*oScanASTStatusData).mIsDAB_AST_Active = false;
   (*oScanASTStatusData).mIsFM_AST_Active = false;
   (*oScanASTStatusData).mIsAM_AST_Active = false;
   (*oAppFlyIn).mVisible_Media = false;
   (*oAppFlyIn).mVisible_Navi = false;
   (*oActiveSourceInfo).mBitmapforSource = (Candera::String)SOURCE_RADIO;
   (*oActiveSourceInfo).mActiveSource = true;
   m_u32FMHiRange = 108000;
   m_u32FMLowRange = 0;
   m_u32AMHiRange = 0;
   m_u32AMLowRange = 0;
   m_u32SliderValue = 0;
   previousFreqvalue = 0;

   (*oCheckBoxData).misChecked_TA = 0;
   (*oCheckBoxData).misChecked_RadioTxt = 0;
   (*oCheckBoxData).misChecked_RDSRegPgm = 0;
   (*oCheckBoxData).misChecked_DAB_DynamicContent = 0;
   (*oCheckBoxData).misChecked_DAB_FM_Following = 0;
   (*oCheckBoxData).misChecked_DAB_DAB_Linking = 0;

   (*oDabOptionsData).misChecked_EnablePTYFilter = 0;
   (*oDabOptionsData).misChecked_News = 0;
   (*oDabOptionsData).misChecked_Culture = 0;
   (*oDabOptionsData).misChecked_Sports = 0;
   (*oDabOptionsData).misChecked_Music = 0;
   (*oDabOptionsData).misChecked_Special = 0;
   (*oMenuSelectionData).mMainMenu_Selection = 0;
   (*oMenuSelectionData).mDABPTYOption_Selection = 0;
   (*oMenu_NumOptionsData).mMainMenu_NumOptions = 1;
   (*oBrowseListSelectedElement).mTun_StationSelected = 0;
   (*oBrowseListSelectedElement).mDAB_ServiceSelected = 0;
   (*oBrowseListSelectedElement).mAF_ServiceSelected = 0;
   (*oBrowseListSelectedElement).mTMC_ServiceSelected = 0;

   (*oTTTunerSource_Band).mTunerSource1 = true;
   (*oTTTunerSource_Band).mTunerSource2 = false;
   (*oTTTunerSource_Band).mTunerSource3 = false;
   (*oTTTunerSource_Band).mTunerSource4 = false;
   (*oTTTunerSource_Band).mBandEnable1 = true;
   (*oTTTunerSource_Band).mBandEnable2 = false;
   (*oTTTunerSource_Band).mBandEnable3 = false;
   (*oTTTunerSource_Band).mBandEnable4 = false;
   (*oTTTunerSource_Band).mBand1 = "FM";
   (*oTTTunerSource_Band).mBand2 = "FM";
   (*oTTTunerSource_Band).mBand3 = "DAB";
   (*oTTTunerSource_Band).mBand4 = "FM";

   (*oEngineeringMenuData).m_szFrequency = "";
   (*oEngineeringMenuData).m_szPSName = "";
   (*oEngineeringMenuData).m_szPI = "";
   (*oEngineeringMenuData).m_szFieldstrength = "";
   (*oEngineeringMenuData).m_szQuality = "";
   (*oEngineeringMenuData).m_szSharx = "";
   (*oEngineeringMenuData).m_szHicut = "";
   (*oEngineeringMenuData).m_szFMDeviation = "";
   (*oEngineeringMenuData).m_szChannelSeparation = "";

   (*oEngineeringMenuData1).mMultipathQuality = "";
   (*oEngineeringMenuData1).mRDS = "";
   (*oEngineeringMenuData1).mAntennaStatus = "";
   (*oEngineeringMenuData1).mReceptionQuality = "";
   (*oEngineeringMenuData1).mSzSharx = "";
   (*oEngineeringMenuData1).mSzHicut = "";
   (*oEngineeringMenuData1).mTPFlag = "";

   (*oEngineeringMenuData2).mTAFlag = "";
   (*oEngineeringMenuData2).mAltFrequency = "";
   (*oEngineeringMenuData2).mAFonOff = 0;
   (*oEngineeringMenuData2).mBandscanOnOff = 0;
   (*oEngineeringMenuData2).mMeasureOnOff = 0;
   (*oEngineeringMenuData2).mFreqStepFM = "";
   (*oEngineeringMenuData2).mFreqStepAM = "";
   (*oEngineeringMenuData2).mRFLevelAM = "";
   (*oEngineeringMenuData2).mLinearAudio = 0;
   (*oEngineeringMenuData2).mAMTuner = "";
   (*oEngineeringMenuData2).mFMTuner1 = "";
   (*oEngineeringMenuData2).mFMTuner2 = "";
   (*oEngineeringMenuData2).mFMTuner3 = "";
   (*oEngineeringMenuData2).mFMTuner4 = "";
   (*oEngineeringMenuData2).mDABTuner = "";

   (*oDABSetupMonitor).mServiceLinking = "DAB";
   (*oDABSetupMonitor).mConcealmentLevel = "0";

   (*oDABTestModePointer).mEnsemble_Lbl = "";
   (*oDABTestModePointer).mEnsemble_ID = "";
   (*oDABTestModePointer).mService_ID = "";
   (*oDABTestModePointer).mEnsemble_freq = "";
   (*oDABTestModePointer).mService_Lbl = "";
   (*oDABTestModePointer).mAud_Data_Bit_Rate = "";
   (*oDABTestModePointer).mMSC = "";
   (*oDABTestModePointer).mFIC = "";
   (*oDABTestModePointer).mNET = "";
   (*oDABTestModePointer).mRS_FEC = "";
   (*oDABTestModePointer).mSync_flag = "";
   (*oDABTestModePointer).mQA = "";
   (*oDABTestModePointer).mQS = "";
   (*oDABTestModePointer).mField_Strength = "";
   (*oDABTestModePointer).mMute_flag = "";
   (*oDABTestModePointer).mBg_tuner_ens_freq = "";
   (*oDABTestModePointer).mBg_Ensemble_ID = "";
   (*oDABTestModePointer).mBg_FIC = "";
   (*oDABTestModePointer).mAudio_Mode = "";
   (*oDABTestModePointer).mFreq_Table = "";
   (*oDABTestModePointer).mBGMode = "";
   (*oDABTestModePointer).mBGSync_flag = "";
   (*oDABTestModePointer).mTASrc = "";

   (*oDABTestModeActivertunerInfo).mDRC = "";
   (*oDABTestModeActivertunerInfo).mPS = "";
   (*oDABTestModeActivertunerInfo).mSM_ACC = "";
   (*oDABTestModeActivertunerInfo).mSR = "";
   (*oDABTestModeActivertunerInfo).mTP = "";

   (*oDABTestModeActiveEnsembleInfo).mNoOfAud_DataServiceComp = "";
   (*oDABTestModeActiveEnsembleInfo).mNoOfAud_DataService = "";
   (*oDABTestModeActiveEnsembleInfo).mNoOfEnsembles = "";

   (*oDABTestModeGTestmodeInfo).mFM = "";
   (*oDABTestModeGTestmodeInfo).mPI = "";
   (*oDABTestModeGTestmodeInfo).mQFM = "";
   (*oDABTestModeGTestmodeInfo).mPL = "";

   (*oDABTestModeAnnoStatusInfo).mASW = "";
   (*oDABTestModeAnnoStatusInfo).mASU = "";

   (*oDABTestModeActiveCompInfo).mSC = "";
   (*oDABTestModeActiveCompInfo).mSCType = "";

   (*oDABTestModeGAvailabiltyInfo).mTA = "";
   (*oDABTestModeGAvailabiltyInfo).mIndicator = "";
   (*oDABTestModeGAvailabiltyInfo).mDABDAB = "";

   (*oDABTestServiceFollowingInfo).mTSU = "";

   (*oTmcInformationData).mNumberOfMsg = "";
   (*oTmcInformationData).mSelectedMsg = "";
   (*oTmcInformationData).mLastMsgRecd = "";

   (*oTmcStationInformationData).mPS = "";
   (*oTmcStationInformationData).mPI = "";
   (*oTmcStationInformationData).mCC = "";
   (*oTmcStationInformationData).mSID = "";
   (*oTmcStationInformationData).mLTN = "";
   (*oTmcStationInformationData).mQuality = "";

   (*oTestModeAvailabilityInfo).mFMTestModeAvail = false;
   (*oTestModeAvailabilityInfo).mDABTestModeAvail = false;
   (*oTunerScanInfoData).mIsActiveScan = false;
   (*oTunerSourceIconData).mFM1Active = false;
   (*oTunerSourceIconData).mFM2Active = false;
   (*oTunerSourceIconData).mFMASTActive = false;
   (*oTunerSourceIconData).mAMActive = false;
   (*oTunerSourceIconData).mAMASTActive = false;
   (*oTunerSourceIconData).mDABActive = false;
   (*oTunerSourceIconData).mDABASTActive = false;
   (*oTunerSourceIconData).mSXMActive = false;
   _AMPSName = "";
   //NORMAL_M_ASSERT(NULL == m_poSelf);
   //m_poSelf = this;
}


/************************************************************************
*FUNCTION: 		~TunerDataBindingHandler
*DESCRIPTION:   Destructor
*PARAMETER:		None
*RETURNVALUE: 	NA
*
*HISTORY:
*05.03.13 Shrivani
*Rev 1.0 Initial Revision.
************************************************************************/

TunerDataBindingHandler::~TunerDataBindingHandler()
{
   //added for coverity
   if (m_tunerDataBindingHandler)
   {
      delete m_tunerDataBindingHandler;
      m_tunerDataBindingHandler = 0;
   }
   ETG_TRACE_USR3(("TunerDataBindingHandler::~TunerDataBindingHandler()"));
}


TunerDataBindingHandler* TunerDataBindingHandler::getDataBindingHandlerInstance()
{
   if (NULL == m_tunerDataBindingHandler)
   {
      m_tunerDataBindingHandler = new TunerDataBindingHandler();
      ETG_TRACE_USR3(("TunerDataBindingHandler::TunerDataBindingHandler Instance"));
   }
   return m_tunerDataBindingHandler;
}


/************************************************************************
*FUNCTION: 		cobOnShutdownMsg
*DESCRIPTION: 	Handles Shutdown Courier message
*PARAMETER:
* 				corfoMsg:	(I)
* 					Courier message received
*
*RETURNVALUE: 	tBool
*					TRUE, if the message is consumed
*					FALSE, if the message should be forwarded
*
*HISTORY:
*13.05.13 Shrivani
*Rev. 0.2 Removed lint warnings
*06.05.13 Shrivani
*Rev 0.1 Initial Revision.
************************************************************************/

bool TunerDataBindingHandler::cobOnShutdownMsg(const Courier::Message& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::cobOnShutdownMsg()"));

   bool bMsgConsumed = true; // Holds true if Model consumes the message

   /* Nothing to be done yet */
   (void)corfoMsg; // To avoid lint warning (unused variable)

   return bMsgConsumed;
}


/************************************************************************
*FUNCTION: 		cobOnUpdateTunerStatusMsg
*DESCRIPTION: 	Handles UpdateTunerStatusMsg Courier message
*PARAMETER:
* 				corfoMsg:	(I)
* 					Courier message received
*
*RETURNVALUE: 	tBool
*					TRUE, if the message is consumed
*					FALSE, if the message should be forwarded
*
*HISTORY:
*06.05.13 Shrivani
*Rev 0.1 Initial Revision.
************************************************************************/

bool TunerDataBindingHandler::cobOnUpdateTunerStatusMsg(const Courier::Message& corfoMsg) const
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::cobOnUpdateTunerStatusMsg()"));

   bool bMsgConsumed = false; // Holds true if Model consumes the message

   const UpdateTunerStatusMsg* copoStatusMsg =
      message_cast<const UpdateTunerStatusMsg*>(&corfoMsg);
   if (NULL != copoStatusMsg)
   {
      /* Retrieving the tuner status from the message */
      UInt32 u32ActivateStatus = copoStatusMsg->GetActivateStatus();
      ETG_TRACE_USR3(("Tuner Activation Status : %d", u32ActivateStatus));
   }
   else
   {
      ETG_TRACE_USR3(("TunerDataBindingHandler : copoStatusMsg NULL Pointer"));
   }

   bMsgConsumed = true;
   return bMsgConsumed;
}


/************************************************************************
*FUNCTION: 		bOnUpdateEngineeringMenuMsg
*DESCRIPTION: 	Handles UpdateFrequencyMsg Courier message.
*               Updates Frequency item of Tuner Info binding source
*PARAMETER:
* 				corfoMsg:	(I)
* 					Courier message received
*
*RETURNVALUE: 	tBool
*					TRUE, if the message is consumed
*					FALSE, if the message should be forwarded
*
*HISTORY:
*06.05.13 Shrivani
*Rev 0.1 Initial Revision.
************************************************************************/

bool TunerDataBindingHandler::bOnUpdateEngineeringMenuMsg(const Courier::Message& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::bOnUpdateEngineeringMenuMsg()"));

   bool bMsgConsumed = false; // Holds true if Model consumes the message

   /* Update the frequency value in HMI for UpdateFrequencyMsg */
   const TripleTunerMonitorDataSet1* poTripleTunerMonitorDataSet1 =
      message_cast<const TripleTunerMonitorDataSet1*>(&corfoMsg);
   if (NULL != poTripleTunerMonitorDataSet1)
   {
      tChar szData[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

      /* Retrieving the frequency from the message and updating the binding source */
      (*oEngineeringMenuData).m_szFrequency = poTripleTunerMonitorDataSet1->GetFrequency();
      (*oEngineeringMenuData).m_szPSName = poTripleTunerMonitorDataSet1->GetPSName();

      sprintf(szData, "%d", poTripleTunerMonitorDataSet1->GetPI());
      (*oEngineeringMenuData).m_szPI = szData;

      sprintf(szData, "%d", poTripleTunerMonitorDataSet1->GetU8Fieldstrength());
      (*oEngineeringMenuData).m_szFieldstrength = szData;

      sprintf(szData, "%d", poTripleTunerMonitorDataSet1->GetU8Quality());
      (*oEngineeringMenuData).m_szQuality = szData;

      /*
      Reception Status = poor: x <= 37
      Reception Status = mono: 37 < x <= 76
      Reception Status = stereo: x > 76
      */
      if (poTripleTunerMonitorDataSet1->GetU8Quality() <= 37)
      {
         (*oEngineeringMenuData1).mReceptionQuality = "POOR";
      }
      else if (poTripleTunerMonitorDataSet1->GetU8Quality() <= 76)
      {
         (*oEngineeringMenuData1).mReceptionQuality = "MONO";
      }
      else
      {
         (*oEngineeringMenuData1).mReceptionQuality = "STEREO";
      }

      sprintf(szData, "%d", poTripleTunerMonitorDataSet1->GetU8Sharx());
      (*oEngineeringMenuData).m_szSharx = szData;

      sprintf(szData, "%d", poTripleTunerMonitorDataSet1->GetU8HiCut());
      (*oEngineeringMenuData).m_szHicut = szData;

      sprintf(szData, "%d", poTripleTunerMonitorDataSet1->GetU8FMDeviation());
      (*oEngineeringMenuData).m_szFMDeviation = szData;

      sprintf(szData, "%d", poTripleTunerMonitorDataSet1->GetU8ChannelSeparation());
      (*oEngineeringMenuData).m_szChannelSeparation = szData;

      /* Mark all items for updation */
      oEngineeringMenuData.MarkAllItemsModified();

      bool bDataUpdated = oEngineeringMenuData.SendUpdate(true);
      if (false == bDataUpdated)
      {
         ETG_TRACE_USR3(("ERROR: Data updation failed oEngineeringMenuData"));
      }
   }
   else
   {
      ETG_TRACE_USR3(("ERROR: poTunerEngineeringModeMsg NULL Pointer"));
   }

   bMsgConsumed = true;
   return bMsgConsumed;
}


/************************************************************************
*FUNCTION: 		bOnUpdateEngineeringMenuMsg1
*DESCRIPTION: 	Handles UpdateFrequencyMsg Courier message.
*               Updates Frequency item of Tuner Info binding source
*PARAMETER:
* 				corfoMsg:	(I)
* 					Courier message received
*
*RETURNVALUE: 	tBool
*					TRUE, if the message is consumed
*					FALSE, if the message should be forwarded
*
*HISTORY:
*06.05.13 Shrivani
*Rev 0.1 Initial Revision.
************************************************************************/

bool TunerDataBindingHandler::bOnUpdateEngineeringMenuMsg1(const Courier::Message& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::bOnUpdateEngineeringMenuMsg1()"));

   /* Update the frequency value in HMI for UpdateFrequencyMsg */
   const TripleTunerMonitorDataSet2* poTunerEngineeringModeMsg1 =
      message_cast<const TripleTunerMonitorDataSet2*>(&corfoMsg);
   if (NULL != poTunerEngineeringModeMsg1)
   {
      tChar szData[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

      //Take the quality value from the previous update of oEngineeringMenuData
      sprintf(szData, "%d / %s", poTunerEngineeringModeMsg1->GetFMMultipath(), (*oEngineeringMenuData).m_szQuality.GetCString());
      (*oEngineeringMenuData1).mMultipathQuality = szData;

      //sprintf(szData, "%d", poTunerEngineeringModeMsg1->GetFMNeighbour());
      //(*oEngineeringMenuData1).mNChannel = szData;

      sprintf(szData, "%d", poTunerEngineeringModeMsg1->GetFMRds());
      (*oEngineeringMenuData1).mRDS = szData;

      sprintf(szData, "%d", poTunerEngineeringModeMsg1->GetFMTPFlag());
      (*oEngineeringMenuData1).mTPFlag = szData;

      sprintf(szData, "%d", poTunerEngineeringModeMsg1->GetFMTAFlag());
      (*oEngineeringMenuData2).mTAFlag = szData;

      sprintf(szData, "%d", poTunerEngineeringModeMsg1->GetFMAlterFreq());
      (*oEngineeringMenuData2).mAltFrequency = szData;

      /* Mark all items for updation */
      oEngineeringMenuData.MarkAllItemsModified();
      oEngineeringMenuData1.MarkAllItemsModified();
      oEngineeringMenuData2.MarkAllItemsModified();

      bool bDataUpdated = oEngineeringMenuData1.SendUpdate(true);
      if (false == bDataUpdated)
      {
         ETG_TRACE_USR3(("ERROR: Data updation failed oEngineeringMenuData1"));
      }

      bDataUpdated = oEngineeringMenuData2.SendUpdate(true);
      if (false == bDataUpdated)
      {
         ETG_TRACE_USR3(("ERROR: Data updation failed oEngineeringMenuData2"));
      }
      bDataUpdated = oEngineeringMenuData.SendUpdate(true);
      if (false == bDataUpdated)
      {
         ETG_TRACE_USR3(("ERROR: Data updation failed oEngineeringMenuData"));
      }
   }
   else
   {
      ETG_TRACE_USR3(("ERROR: poTunerEngineeringModeMsg NULL Pointer"));
   }

   return true;
}


/************************************************************************
*FUNCTION: 		bOnUpdateEngineeringMenuMsg2
*DESCRIPTION: 	Handles UpdateFrequencyMsg Courier message.
*               Updates Frequency item of Tuner Info binding source
*PARAMETER:
* 				corfoMsg:	(I)
* 					Courier message received
*
*RETURNVALUE: 	tBool
*					TRUE, if the message is consumed
*					FALSE, if the message should be forwarded
*
*HISTORY:
*06.05.13 Shrivani
*Rev 0.1 Initial Revision.
************************************************************************/

bool TunerDataBindingHandler::bOnUpdateEngineeringMenuMsg2(const Courier::Message& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::bOnUpdateEngineeringMenuMsg2()"));

   /* Update the frequency value in HMI for UpdateFrequencyMsg */
   const TunerEngineeringModeMsg2* poTunerEngineeringModeMsg2 =
      message_cast<const TunerEngineeringModeMsg2*>(&corfoMsg);
   if (NULL != poTunerEngineeringModeMsg2)
   {
      tChar szData[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

      //(*oEngineeringMenuData1).mReceptionQuality = poTunerEngineeringModeMsg2->GetFMReception();

      sprintf(szData, "%d", poTunerEngineeringModeMsg2->GetFMFreqStep());
      (*oEngineeringMenuData2).mFreqStepFM = szData;
      sprintf(szData, "%d", poTunerEngineeringModeMsg2->GetAMFreqStep());
      (*oEngineeringMenuData2).mFreqStepAM = szData;
      sprintf(szData, "%d", poTunerEngineeringModeMsg2->GetAMTuner());
      (*oEngineeringMenuData2).mAMTuner = szData;
      sprintf(szData, "%d", poTunerEngineeringModeMsg2->GetFMTuner1());
      (*oEngineeringMenuData2).mFMTuner1 = szData;
      sprintf(szData, "%d", poTunerEngineeringModeMsg2->GetFMTuner2());
      (*oEngineeringMenuData2).mFMTuner2 = szData;
      sprintf(szData, "%d", poTunerEngineeringModeMsg2->GetDABTuner());
      (*oEngineeringMenuData2).mDABTuner = szData;

      /*sprintf(szData, "%d", poTunerEngineeringModeMsg2->GetAMRFLevel());
      (*oEngineeringMenuData2).mRFLevelAM = szData;*/

      /* Mark all items for updation */
      oEngineeringMenuData1.MarkAllItemsModified();
      oEngineeringMenuData2.MarkAllItemsModified();

      bool bDataUpdated = oEngineeringMenuData1.SendUpdate(true);
      if (false == bDataUpdated)
      {
         ETG_TRACE_USR3(("ERROR: Data updation failed oEngineeringMenuData1"));
      }

      bDataUpdated = oEngineeringMenuData2.SendUpdate(true);
      if (false == bDataUpdated)
      {
         ETG_TRACE_USR3(("ERROR: Data updation failed oEngineeringMenuData2"));
      }
   }
   else
   {
      ETG_TRACE_USR3(("ERROR: poTunerEngineeringModeMsg NULL Pointer"));
   }

   return true;
}


/************************************************************************
*FUNCTION: 		bOnUpdateAntennaStatus
*DESCRIPTION: 	Handles antenna status
*PARAMETER:
* 				corfoMsg:	(I)
* 					Courier message received
*
*RETURNVALUE: 	tBool
*					TRUE, if the message is consumed
*					FALSE, if the message should be forwarded
*
*HISTORY:
*06.05.13 Shrivani
*Rev 0.1 Initial Revision.
************************************************************************/

bool TunerDataBindingHandler::bOnUpdateAntennaStatus(const Courier::Message& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::bOnUpdateAntennaStatus()"));

   /* Update the frequency value in HMI for UpdateFrequencyMsg */
   const  AntennaStatus* poAntennaStatus =
      message_cast<const AntennaStatus*>(&corfoMsg);
   if (NULL != poAntennaStatus)
   {
      const char* szAntennaStatusStr[] =
      {
         "UNKNOWN",
         "OK",
         "OPEN",
         "SHORT",
         "DEACTIVATED",
         "DEACTIVATED_LOW_VG",
         "DEACTIVATED_HIGH_VG",
         "OPEN_AMPLIFIER",
         "NUMBER_OF_ITEM"/*8UL*/
      };

      if (poAntennaStatus->GetBAntennaStatus() < 9 /*MAX AntennaStatus Enum*/)
      {
         (*oEngineeringMenuData1).mAntennaStatus =
            szAntennaStatusStr[poAntennaStatus->GetBAntennaStatus()];
      }

      /* Mark all items for updation */
      oEngineeringMenuData1.MarkAllItemsModified();

      bool bDataUpdated = oEngineeringMenuData1.SendUpdate(true);
      if (false == bDataUpdated)
      {
         ETG_TRACE_USR3(("ERROR: Data updation failed oEngineeringMenuData1"));
      }
   }
   else
   {
      ETG_TRACE_USR3(("ERROR: poAntennaStatus NULL Pointer"));
   }

   return true;
}


/************************************************************************
*FUNCTION: 		bOnUpdateBandscanStatus
*DESCRIPTION: 	Handles background band scan status
*PARAMETER:
* 				corfoMsg:	(I)
* 					Courier message received
*
*RETURNVALUE: 	tBool
*					TRUE, if the message is consumed
*					FALSE, if the message should be forwarded
*
*HISTORY:
*06.05.13 Shrivani
*Rev 0.1 Initial Revision.
************************************************************************/

bool TunerDataBindingHandler::bOnUpdateBandscanStatus(const Courier::Message& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::bOnUpdateAntennaStatus()"));

   /* Update the frequency value in HMI for UpdateFrequencyMsg */
   const  BandscanStatus* poBandscanStatus =
      message_cast<const BandscanStatus*>(&corfoMsg);
   if (NULL != poBandscanStatus)
   {
      (*oEngineeringMenuData2).mBandscanOnOff = poBandscanStatus->GetBBandscan();

      /* Mark all items for updation */
      oEngineeringMenuData2.MarkAllItemsModified();

      bool bDataUpdated = oEngineeringMenuData2.SendUpdate(true);
      if (false == bDataUpdated)
      {
         ETG_TRACE_USR3(("ERROR: Data updation failed oEngineeringMenuData2"));
      }
   }
   else
   {
      ETG_TRACE_USR3(("ERROR: poBandscanStatus NULL Pointer"));
   }

   return true;
}


bool TunerDataBindingHandler::vUpdate_DAB_PresetFocused(tU16 PresetNumber)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::vUpdate_DAB_PresetFocused(%d)", PresetNumber));
   VehicleSrvClient::poGetInstance()->updateTunerPresetStatus(PresetNumber); //CAN Imp
   bool* pPresetState[] =
   {
      &((*oDABPresetInfo).mPreset_1_Active),
      &((*oDABPresetInfo).mPreset_2_Active),
      &((*oDABPresetInfo).mPreset_3_Active),
      &((*oDABPresetInfo).mPreset_4_Active),
      &((*oDABPresetInfo).mPreset_5_Active),
      &((*oDABPresetInfo).mPreset_6_Active),
      &((*oDABPresetInfo).mPreset_7_Active),
      &((*oDABPresetInfo).mPreset_8_Active),
      &((*oDABPresetInfo).mPreset_9_Active),
      &((*oDABPresetInfo).mPreset_10_Active),
      &((*oDABPresetInfo).mPreset_11_Active),
      &((*oDABPresetInfo).mPreset_12_Active)
   };
   //clear the presets
   for (tU8 u8Index = 0; u8Index < (sizeof(pPresetState) / sizeof(pPresetState[0])); ++u8Index)
   {
      *(pPresetState[u8Index]) = false;
   }
   if (PresetNumber != 0)
   {
      tU16 u8ActivePreset = PresetNumber - 1;
      *(pPresetState[u8ActivePreset]) = true;
   }
   oDABPresetInfo.MarkAllItemsModified();
   oDABPresetInfo.SendUpdate(true);
   return true;
}


bool TunerDataBindingHandler::vUpdate_FM_PresetFocused(tU16 PresetNumber)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::vUpdate_FM_PresetFocused(%d)", PresetNumber));
   m_u8FMFocusedPresestNumber = PresetNumber;
   VehicleSrvClient::poGetInstance()->updateTunerPresetStatus(m_u8FMFocusedPresestNumber); //CAN Imp

   bool* pPresetState[] =
   {
      &((*oFMAMPresetInfo).mPreset_1_Active),
      &((*oFMAMPresetInfo).mPreset_2_Active),
      &((*oFMAMPresetInfo).mPreset_3_Active),
      &((*oFMAMPresetInfo).mPreset_4_Active),
      &((*oFMAMPresetInfo).mPreset_5_Active),
      &((*oFMAMPresetInfo).mPreset_6_Active),
      &((*oFMAMPresetInfo).mPreset_7_Active),
      &((*oFMAMPresetInfo).mPreset_8_Active),
      &((*oFMAMPresetInfo).mPreset_9_Active),
      &((*oFMAMPresetInfo).mPreset_10_Active),
      &((*oFMAMPresetInfo).mPreset_11_Active),
      &((*oFMAMPresetInfo).mPreset_12_Active)
   };
   //clear the presets
   for (tU8 u8Index = 0; u8Index < (sizeof(pPresetState) / sizeof(pPresetState[0])); ++u8Index)
   {
      *(pPresetState[u8Index]) = false;
   }
   if (PresetNumber != 0)
   {
      if (sm_gCurrentView == VIEW_FM1)
      {
         tU16 u8ActivePreset = PresetNumber - 1;
         *(pPresetState[u8ActivePreset]) = true;
      }
   }
   oFMAMPresetInfo.MarkAllItemsModified();
   oFMAMPresetInfo.SendUpdate(true);
   return true;
}


bool TunerDataBindingHandler::vUpdate_AM_PresetFocused(tU16 PresetNumber)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::vUpdate_AM_PresetFocused(%d)", PresetNumber));
   m_u8AMFocusedPresestNumber = PresetNumber;
   VehicleSrvClient::poGetInstance()->updateTunerPresetStatus(m_u8AMFocusedPresestNumber); //CAN Imp

   bool* pPresetState[] =
   {
      &((*oAMPresetInfo).mPreset_1_Active),
      &((*oAMPresetInfo).mPreset_2_Active),
      &((*oAMPresetInfo).mPreset_3_Active),
      &((*oAMPresetInfo).mPreset_4_Active),
      &((*oAMPresetInfo).mPreset_5_Active),
      &((*oAMPresetInfo).mPreset_6_Active),
      &((*oAMPresetInfo).mPreset_7_Active),
      &((*oAMPresetInfo).mPreset_8_Active),
      &((*oAMPresetInfo).mPreset_9_Active),
      &((*oAMPresetInfo).mPreset_10_Active),
      &((*oAMPresetInfo).mPreset_11_Active),
      &((*oAMPresetInfo).mPreset_12_Active)
   };
   //clear the presets
   for (tU8 u8Index = 0; u8Index < (sizeof(pPresetState) / sizeof(pPresetState[0])); ++u8Index)
   {
      *(pPresetState[u8Index]) = false;
   }
   if (PresetNumber != 0)
   {
      if (sm_gCurrentView == VIEW_AM)
      {
         tU16 u8ActivePreset = PresetNumber - 1;
         *(pPresetState[u8ActivePreset]) = true;
      }
   }
   oAMPresetInfo.MarkAllItemsModified();
   oAMPresetInfo.SendUpdate(true);
   return true;
}


void TunerDataBindingHandler::updateFMpresets(::std::vector<std::string>& rfoPresetList)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::updateFMpresets(size: %d)", rfoPresetList.size()));
   Candera::String* pTunerPresetState[MAX_FM_PRESETS] =
   {
      &((*oTunerPresetFM1).mPSName1),
      &((*oTunerPresetFM1).mPSName2),
      &((*oTunerPresetFM1).mPSName3),
      &((*oTunerPresetFM1).mPSName4),
      &((*oTunerPresetFM1).mPSName5),
      &((*oTunerPresetFM1).mPSName6),
      &((*oTunerPresetFM2).mPSName1),
      &((*oTunerPresetFM2).mPSName2),
      &((*oTunerPresetFM2).mPSName3),
      &((*oTunerPresetFM2).mPSName4),
      &((*oTunerPresetFM2).mPSName5),
      &((*oTunerPresetFM2).mPSName6)
   };

   for (tU8 u8Index = 0; (u8Index < rfoPresetList.size()) && (u8Index < MAX_FM_PRESETS); ++u8Index)
   {
      *pTunerPresetState[u8Index] = (Candera::String)rfoPresetList[u8Index].c_str();
   }

   oTunerPresetFM2.MarkAllItemsModified();
   tBool bDataUpdated = oTunerPresetFM2.SendUpdate(true);
   oTunerPresetFM1.MarkAllItemsModified();
   bDataUpdated = oTunerPresetFM1.SendUpdate(true);

   if (false == bDataUpdated)
   {
      ETG_TRACE_USR3(("tclTunerModelComponent : Data updation failed"));
   }
}


void TunerDataBindingHandler::updateAMpresets(::std::vector<std::string>& rfoPresetList, tU8 u8BankID)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::updateAMpresets(size: %d)", rfoPresetList.size()));
   ETG_TRACE_USR3(("TunerDataBindingHandler : updateAMpresets Bank %d", u8BankID));

   Candera::String* pTunerPresetState[MAX_AM_PRESETS] =
   {
      &((*oTunerPresetAM).mPSName1),
      &((*oTunerPresetAM).mPSName2),
      &((*oTunerPresetAM).mPSName3),
      &((*oTunerPresetAM).mPSName4),
      &((*oTunerPresetAM).mPSName5),
      &((*oTunerPresetAM).mPSName6),
      &((*oTunerPresetAM).mPSName7),
      &((*oTunerPresetAM).mPSName8),
      &((*oTunerPresetAM).mPSName9),
      &((*oTunerPresetAM).mPSName10),
      &((*oTunerPresetAM).mPSName11),
      &((*oTunerPresetAM).mPSName12)
   };

   for (tU8 u8Index = 0; (u8Index < rfoPresetList.size()) && (u8Index < MAX_AM_PRESETS); ++u8Index)
   {
      *pTunerPresetState[u8Index] = (Candera::String)rfoPresetList[u8Index].c_str();
   }

   oTunerPresetAM.MarkAllItemsModified();
   tBool bDataUpdated = oTunerPresetAM.SendUpdate(true);

   if (false == bDataUpdated)
   {
      ETG_TRACE_USR3(("TunerDataBindingHandler : Data updation failed"));
   }
}


void TunerDataBindingHandler::updateDABpresets(::std::vector<std::string>& rfoPresetList)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::updateDABpresets(size: %d)", rfoPresetList.size()));
   Candera::String* pTunerPresetState[MAX_DAB_PRESETS] =
   {
      &((*oTunerPresetDAB).mPSName1),
      &((*oTunerPresetDAB).mPSName2),
      &((*oTunerPresetDAB).mPSName3),
      &((*oTunerPresetDAB).mPSName4),
      &((*oTunerPresetDAB).mPSName5),
      &((*oTunerPresetDAB).mPSName6),
      &((*oTunerPresetDAB).mPSName7),
      &((*oTunerPresetDAB).mPSName8),
      &((*oTunerPresetDAB).mPSName9),
      &((*oTunerPresetDAB).mPSName10),
      &((*oTunerPresetDAB).mPSName11),
      &((*oTunerPresetDAB).mPSName12)
   };

   for (tU8 u8Index = 0; (u8Index < rfoPresetList.size()) && (u8Index < MAX_DAB_PRESETS); ++u8Index)
   {
      *pTunerPresetState[u8Index] = (Candera::String)rfoPresetList[u8Index].c_str();
   }

   oTunerPresetDAB.MarkAllItemsModified();
   tBool bDataUpdated = oTunerPresetDAB.SendUpdate(true);

   if (false == bDataUpdated)
   {
      ETG_TRACE_USR3(("tclTunerModelComponent : Data updation failed"));
   }
}


void TunerDataBindingHandler::updatedata(void)
{
   oTunerPresetFM1.MarkAllItemsModified();
   oTunerPresetFM1.SendUpdate(true);

   oTunerPresetFM2.MarkAllItemsModified();
   oTunerPresetFM2.SendUpdate(true);

   oTunerPresetAM.MarkAllItemsModified();
   oTunerPresetAM.SendUpdate(true);
}


void TunerDataBindingHandler::FMAMHomeScreen()
{
   ETG_TRACE_USR3(("TunerDataBindingHandler: FMAMHomeScreen"));
   //bool bDataUpdated = false;
   if (!_isCabinOnly)
   {
      int targetRegion = TunerDataPoolConfig::getInstance()->readRegionType();
      (*oRadioBandInfoData).mVisible_Txt_Wgt_TA_FM = false;
      (*oRadioBandInfoData).mVisible_Txt_Wgt_TA_DAB = false;
      if (AnnoHandler::poGetInstance()->isTaActive() == true)
      {
         if (targetRegion == EUR_REGION)  //Only for EU
         {
            (*oRadioBandInfoData).mVisible_Txt_Wgt_TA_FM = true;
            //	oRadioBandInfoData.MarkItemModified(ItemKey::RadioBandInfo::Visible_Txt_Wgt_TA_FMItem);
         }
      }
      oRadioBandInfoData.MarkItemModified(ItemKey::RadioBandInfo::Visible_Txt_Wgt_TA_FMItem);
      oRadioBandInfoData.MarkItemModified(ItemKey::RadioBandInfo::Visible_Txt_Wgt_TA_DABItem);
      oRadioBandInfoData.SendUpdate(true);
   }

   (*oHomeScreenInfo).mHomeScreen_ScanInvisible = 1;
   (*oHomeScreenInfo).mHomeScreen_DABServicevisiblity = 0;
   // uint8 Presetavailablevalue = PresetListHandling::poGetInstance()->PresetnameAvailable();
   (*oHomeScreenInfo).mHomeFrequency = m_sfrequencyvalue.c_str();
   (*oHomeScreenInfo).mHomeFrequencyStationName = m_FMStationName.c_str();
   (*oHomeScreenInfo).mHomeScreen_PACStatus = false;
   (*oHomeScreenInfo).mHomeFrequencyStationNameVisibility = true;
   (*oHomeScreenInfo).mFrequencyUnit = (Candera::String)C_FM_FREQ_UNIT;
   (*oHomeScreenInfo).mHomeScreen_ActiveSourceText = "FM";
   (*oHomeScreenInfo).mHomeScreen_NextPreviousIconStatus = true;
   (*oHomeScreenInfo).mHomeScreen_ActiveIconIndex = 0;
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeScreen_ScanInvisibleItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeFrequencyStationNameVisibilityItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeScreen_DABServicevisiblityItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeFrequencyStationNameItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeFrequencyItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::FrequencyUnitItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeScreen_ActiveSourceTextItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeScreen_NextPreviousIconStatusItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeScreen_PACStatusItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeScreen_ActiveIconIndexItem);
   oHomeScreenInfo.SendUpdate(true);
}


void TunerDataBindingHandler::AMHomeScreen()
{
   ETG_TRACE_USR3(("TunerDataBindingHandler: AMHomeScreen"));
   if (!_isCabinOnly)
   {
      (*oRadioBandInfoData).mVisible_Txt_Wgt_TA_FM = false;
      (*oRadioBandInfoData).mVisible_Txt_Wgt_TA_DAB = false;
      oRadioBandInfoData.MarkItemModified(ItemKey::RadioBandInfo::Visible_Txt_Wgt_TA_FMItem);
      oRadioBandInfoData.MarkItemModified(ItemKey::RadioBandInfo::Visible_Txt_Wgt_TA_DABItem);
      oRadioBandInfoData.SendUpdate(true);
   }
   (*oHomeScreenInfo).mHomeScreen_ScanInvisible = 1;
   (*oHomeScreenInfo).mHomeScreen_DABServicevisiblity = 0;
   (*oHomeScreenInfo).mHomeScreen_PACStatus = false;
   (*oHomeScreenInfo).mHomeFrequency = m_amFrequencyvalue.c_str();
   (*oHomeScreenInfo).mHomeFrequencyStationNameVisibility = false;
   (*oHomeScreenInfo).mFrequencyUnit = (Candera::String)C_AM_FREQ_UNIT;
   (*oHomeScreenInfo).mHomeScreen_ActiveSourceText = "AM";
   (*oHomeScreenInfo).mHomeScreen_NextPreviousIconStatus = true;
   (*oHomeScreenInfo).mHomeScreen_ActiveIconIndex = 1;
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeScreen_ScanInvisibleItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeFrequencyStationNameVisibilityItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeScreen_DABServicevisiblityItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeFrequencyItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::FrequencyUnitItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeScreen_ActiveSourceTextItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeScreen_NextPreviousIconStatusItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeScreen_PACStatusItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeScreen_ActiveIconIndexItem);
   oHomeScreenInfo.SendUpdate(true);
}


void TunerDataBindingHandler::WXHomeScreen()
{
   if (!_isCabinOnly)
   {
      (*oRadioBandInfoData).mVisible_Txt_Wgt_TA_FM = false;
      (*oRadioBandInfoData).mVisible_Txt_Wgt_TA_DAB = false;
      oRadioBandInfoData.MarkItemModified(ItemKey::RadioBandInfo::Visible_Txt_Wgt_TA_FMItem);
      oRadioBandInfoData.MarkItemModified(ItemKey::RadioBandInfo::Visible_Txt_Wgt_TA_DABItem);
      oRadioBandInfoData.SendUpdate(true);
   }
   ETG_TRACE_USR3(("TunerDataBindingHandler: WXHomeScreen"));
   (*oHomeScreenInfo).mHomeScreen_ScanInvisible = 0;
   (*oHomeScreenInfo).mHomeScreen_DABServicevisiblity = 0;
   (*oHomeScreenInfo).mHomeScreen_ActiveSourceText = "WX";
   (*oHomeScreenInfo).mHomeScreen_ActiveIconIndex = 3;
   (*oHomeScreenInfo).mHomeScreen_NextPreviousIconStatus = false;
   (*oHomeScreenInfo).mHomeFrequencyStationNameVisibility = false;
   (*oHomeScreenInfo).mHomeScreen_PACStatus = true;
   std::string wxfreqval = WeatherBandMain::poGetInstance()->getfreqvalue();
   std::string wxPacval = WeatherBandMain::poGetInstance()->getPacvalue();
   (*oHomeScreenInfo).mHomeFrequency =  wxfreqval.c_str();
   (*oHomeScreenInfo).mHomeWeatherbandText = wxPacval.c_str();
   (*oHomeScreenInfo).mFrequencyUnit = (Candera::String)C_FM_FREQ_UNIT;
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeFrequencyItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeScreen_ScanInvisibleItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeFrequencyStationNameVisibilityItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeScreen_DABServicevisiblityItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeScreen_ActiveSourceTextItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::FrequencyUnitItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeScreen_ActiveIconIndexItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeScreen_NextPreviousIconStatusItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeScreen_PACStatusItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeWeatherbandTextItem);
   oHomeScreenInfo.SendUpdate(true);
}


void TunerDataBindingHandler::DABHomeScreen()
{
   ETG_TRACE_USR3(("TunerDataBindingHandler: DABHomeScreen"));
   if (!_isCabinOnly)
   {
      int targetRegion = TunerDataPoolConfig::getInstance()->readRegionType();
      (*oRadioBandInfoData).mVisible_Txt_Wgt_TA_FM = false;
      (*oRadioBandInfoData).mVisible_Txt_Wgt_TA_DAB = false;
      if (AnnoHandler::poGetInstance()->isTaActive() == true)
      {
         if (targetRegion == EUR_REGION)  //Only for EU
         {
            (*oRadioBandInfoData).mVisible_Txt_Wgt_TA_DAB = true;
            //  oRadioBandInfoData.MarkItemModified(ItemKey::RadioBandInfo::Visible_Txt_Wgt_TA_DABItem);
         }
      }
      oRadioBandInfoData.MarkItemModified(ItemKey::RadioBandInfo::Visible_Txt_Wgt_TA_FMItem);
      oRadioBandInfoData.MarkItemModified(ItemKey::RadioBandInfo::Visible_Txt_Wgt_TA_DABItem);
      oRadioBandInfoData.SendUpdate(true);
   }

   (*oHomeScreenInfo).mHomeScreen_DABServicevisiblity = 1;
   (*oHomeScreenInfo).mHomeScreen_ScanInvisible = 0;
   (*oHomeScreenInfo).mHomeFrequency = "";
   (*oHomeScreenInfo).mFrequencyUnit = "";
   (*oHomeScreenInfo).mHomeScreen_ActiveSourceText = "DAB";
   (*oHomeScreenInfo).mHomeScreen_NextPreviousIconStatus = true;
   (*oHomeScreenInfo).mHomeFrequencyStationNameVisibility = false;
   (*oHomeScreenInfo).mHomeScreen_PACStatus = false;
   DABSource = 1;
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeScreen_ScanInvisibleItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeScreen_DABServicevisiblityItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeFrequencyStationNameVisibilityItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeFrequencyItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::FrequencyUnitItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeScreen_ActiveSourceTextItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeScreen_NextPreviousIconStatusItem);
   oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeScreen_PACStatusItem);
   oHomeScreenInfo.SendUpdate();
}


void TunerDataBindingHandler::updateFrequency(uint8& currentBand, uint32& currentFreqvalue, std::string& FrequencyUnit, std::string& psName)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler: updateFrequency %d %d", currentFreqvalue, currentBand));
   tChar szFrequency[M_FREQ_STR_BUFFER_SIZE] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

   switch (currentBand)
   {
      case T_e8_Tun_TunerBand__TUN_BAND_FM:
      {
         if (TUNER_ZERO_VALUE == (currentFreqvalue % M_DIVIDE_BY_HUNDRED))
         {
            sprintf(szFrequency, "%.01f", ((tF32)currentFreqvalue / M_CONVERSION_KHZ_TO_MHZ));
         }
         else
         {
            sprintf(szFrequency, "%.02f", ((tF32)currentFreqvalue / M_CONVERSION_KHZ_TO_MHZ));
         }
         ETG_TRACE_USR3(("Frequency : %s", szFrequency));
         (*oTunerInfoData).mVisible_Txt_Wgt_RDS = true;
         if (previousFreqvalue != currentFreqvalue)
         {
            (*oTunerInfoData).mFMFrequency = szFrequency;
            previousFreqvalue = currentFreqvalue;
         }
         updateFmPsName(psName);
         if (!psName.empty())
         {
            m_FMStationName = psName;
            VehicleSrvClient::poGetInstance()->pushTunerCANLongData(TUNER_MAINLINE, psName);
         }
         else
         {
            m_FMStationName = "";
            std::string FMStringUnit = " MHz";
            std::string FMString = szFrequency + FMStringUnit;
            VehicleSrvClient::poGetInstance()->pushTunerCANLongData(TUNER_MAINLINE, FMString);
         }
         m_sfrequencyvalue = szFrequency;
         oTunerInfoData.MarkItemModified(ItemKey::StationInfoSource::FMFrequencyItem);
         bool bDataUpdated = oTunerInfoData.SendUpdate(true);
         FMAMHomeScreen();
      }
      break;
      case T_e8_Tun_TunerBand__TUN_BAND_MW:
      {
         ETG_TRACE_USR3(("TunerDataBindingHandler, Frequency : %s", ((*oTunerInfoData).mAMFrequency).GetCString()));
         sprintf(szFrequency, "%d", currentFreqvalue);
         ETG_TRACE_USR3(("Tuner Band AM Frequency : %s", szFrequency));
         if (previousFreqvalue != currentFreqvalue)
         {
            (*oTunerInfoData).mAMFrequency = szFrequency;
            previousFreqvalue = currentFreqvalue;
         }
         if (!_AMPSName.empty())
         {
            VehicleSrvClient::poGetInstance()->pushTunerCANLongData(TUNER_MAINLINE, _AMPSName);
         }
         else
         {
            std::string AMStringUnit = " kHz";
            std::string AMString = szFrequency + AMStringUnit;
            VehicleSrvClient::poGetInstance()->pushTunerCANLongData(TUNER_MAINLINE, AMString);
         }
         m_amFrequencyvalue = szFrequency;
         oTunerInfoData.MarkItemModified(ItemKey::StationInfoSource::AMFrequencyItem);
         bool bDataUpdated = oTunerInfoData.SendUpdate(true);
         AMHomeScreen();
      }
      break;
      case T_e8_Tun_TunerBand__TUN_BAND_WB:
      {
         ETG_TRACE_USR3(("TunerDataBindingHandler,IN WX "));
         WXHomeScreen();
      }
      break;
      default:
      {
         ETG_TRACE_USR3(("ERROR:tclTunerModelComponent::request.ItemKey()"));
      }
      break;
   }
}


/**
* Description     : Function called to update FM frequency or PS name
*
* @param[in]      : psName: The PS Name to be updated to GUI.
* @return         : void
*/
void TunerDataBindingHandler::updateFmPsName(std::string& psName)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler:updateFmPsName %s", psName.c_str()));
   /*Check if there is any change in the content before updating UI */
   if (strcmp(((*oTunerInfoData).mPSName).GetCString(), psName.c_str()) != 0)
   {
      (*oTunerInfoData).mPSName = psName.c_str();
      oTunerInfoData.MarkItemModified(ItemKey::StationInfoSource::PSNameItem);
      oTunerInfoData.SendUpdate();
   }
}


/**
* Description     : Function called to update AM frequency or PS name
*
* @param[in]      : psName: The PS Name to be updated to GUI.
* @return         : void
*/
void TunerDataBindingHandler::updateAmPsName(std::string& psName)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler:updateAmPsName %s", psName.c_str()));
   _AMPSName = psName;
   /*Check if there is any change in the content before updating UI */
   if (strcmp(((*oTunerInfoData).mAMPSName).GetCString(), psName.c_str()) != 0)
   {
      (*oTunerInfoData).mAMPSName = psName.c_str();
      oTunerInfoData.MarkItemModified(ItemKey::StationInfoSource::AMPSNameItem);
      oTunerInfoData.SendUpdate();
   }
}


/************************************************************************
*FUNCTION: 		getfrequencyvalue
*DESCRIPTION: 	Returns the currently tuned frequency in FM.
************************************************************************/
std::string TunerDataBindingHandler::getfrequencyvalue()
{
   return m_sfrequencyvalue;
}


/************************************************************************
*FUNCTION: 		bOnPresetPopupMsg
*DESCRIPTION: 	Handles PresetPopupMsg  Courier message.
*               Starts the timer to update the Pfo_StorePreset view  binding
source i.e the progress bar
*PARAMETER:
* 				corfoMsg:	(I)
* 					Courier message received
*
*RETURNVALUE: 	tBool
*					TRUE, if the message is consumed
*					FALSE, if the message should be forwarded
*
*HISTORY:
*24.09.2013 Shrivani
*Rev 0.1 Initial Revision.
************************************************************************/

bool TunerDataBindingHandler::bOnPresetPopupMsg(const Courier::Message& corfoMsg)
{
   AMFMInfo::poGetInstance()->vAbortTunerFunctionality();
   ETG_TRACE_USR3(("TunerDataBindingHandler::bOnPresetPopupMsg()"));
   char sPopup_text[MAX_POPUP_TEXT];
   bool bMsgConsumed = false; // Holds true if Model consumes the message
   m_u8popupState = EN_POPUP_DISPLAY;
   /* Update the frequency value in HMI for PresetPopupMsg */
   const PresetPopupMsg* copoPresetPopupMsg = message_cast<const PresetPopupMsg*>(&corfoMsg);
   if (NULL != copoPresetPopupMsg)
   {
      bool bDataUpdated = false;

      //Print "Hold to store Preset # "message
      m_u8CurrentPresetNr = copoPresetPopupMsg->GetPresetNumber();
      tU8 l_u8CurrentPresetNr = m_u8CurrentPresetNr;
      //if (VIEW_FM2 == sm_gCurrentView)
      //{
      //l_u8CurrentPresetNr = m_u8CurrentPresetNr + 6;
      //}
      sprintf(sPopup_text, "%d", l_u8CurrentPresetNr);
      (*oTunerPopupData).mVisible_Txt_Wgt_2 = false;
      (*oTunerPopupData).mVisible_Txt_Wgt_1 = true;
      (*oTunerPopupData).mPopup_text_1 = sPopup_text;
      ETG_TRACE_USR3((" TunerDataBindingHandlerl :bTimeout: sPopup_text %s ", sPopup_text));
      ETG_TRACE_USR3(("TunerDataBindingHandler:: POPUP Text%s ", sPopup_text));
      oTunerPopupData.MarkItemModified(ItemKey::Preset_Popup::Visible_Txt_Wgt_1Item);
      oTunerPopupData.MarkItemModified(ItemKey::Preset_Popup::Visible_Txt_Wgt_2Item);
      oTunerPopupData.MarkItemModified(ItemKey::Preset_Popup::Popup_text_1Item);
      bDataUpdated = oTunerPopupData.SendUpdate(true);

      m_u8timerCount = 0;
      (*oTunerPopupData).mCurrentValue = (Candera::UInt16) m_u8timerCount;
      oTunerPopupData.MarkItemModified(ItemKey::Preset_Popup::CurrentValueItem);
      ETG_TRACE_USR3(("TunerDataBindingHandler :bOnPresetPopupMsg, CurrentValue (*oTunerPopupData).mCurrentValue = %f ",
                      (*oTunerPopupData).mCurrentValue));
      bDataUpdated = oTunerPopupData.SendUpdate(true);

      m_oTimerPreset.start();
   }
   else
   {
      ETG_TRACE_USR3(("TunerDataBindingHandler : copoUpdatePresetMsg NULL Pointer"));
   }
   bMsgConsumed = true;
   return bMsgConsumed;
}


tVoid TunerDataBindingHandler::vHandleMainMenuOption(Courier::UInt32 u32SelectedItem)
{
   if (u32SelectedItem == EN_OPT_SOUNDSETTINGS)
   {
      //	Perform_Tuner_fn(EV_SOUND_SETTING, 0, 0);
   }
   else if (u32SelectedItem == EN_OPT_DABPTY)
   {
      Courier::Message* lMsg = COURIER_MESSAGE_NEW(TunerViewReqMsg)(EN_VIEW_OPT_DABPTY);
      if (0 != lMsg)
      {
         lMsg->Post();
         ETG_TRACE_USR3(("TunerDataBindingHandler::vHandleMainMenuOption() TunerViewReqMsg POSTED SUCCESSFULLY"));
      }
      else
      {
         ETG_TRACE_USR3(("ERROR: TunerDataBindingHandler::vHandleMainMenuOption() TunerViewReqMsg Failed"));
      }
   }
   else
   {
      vSendCheckBoxMsg(EN_MENU_TYPE_MAIN, u32SelectedItem);
   }
}


tVoid TunerDataBindingHandler::vHandleDABPTYFilterOption(Courier::UInt32 u32SelectedItem)
{
   vSendCheckBoxMsg(EN_MENU_TYPE_DABPTY_FILTER, u32SelectedItem - 1);
}


/* -------------------------------------------------------------------------------------------------
* Definition of	:	Courier::bool TunerDataBindingHandler::vHandleSliderChange(const Courier::Request&)
*
* Description		:	Update the binding source for a change in scroll direction
* Parameters		:	Courier Request
* Return type		:	void
*
* --------------------------------------------------------------------------------------------------*/

tVoid TunerDataBindingHandler::vHandleSliderChange(const Request& request)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler : vHandleSliderChange()"));

   static Candera::Float g_u32SliderValue = 1;

   ETG_TRACE_USR3(("RequestItemKey: %d..!!", request.ItemKey()));
   switch (request.ItemKey())
   {
      case ItemKey::tuningScale_SliderMarkerPosition::DragCurrentValueItem:
      {
         oTunerSliderMarkerPosition.SetValue(request.ItemKey(), request.GetItemValue());
         g_u32SliderValue = (Candera::Float)(*oTunerSliderMarkerPosition).mDragCurrentValue;
      }
      break;
      case ItemKey::tuningScale_SliderMarkerPosition::CurrentValueItem:
      {
         oTunerSliderMarkerPosition.SetValue(request.ItemKey(), request.GetItemValue());
         g_u32SliderValue = (Candera::Float)(*oTunerSliderMarkerPosition).mCurrentValue;

         (*oTunerSliderMarkerPosition).mCurrentValue = g_u32SliderValue;
         /* Mark only the CurrentValue item for updation */
         oTunerSliderMarkerPosition.MarkItemModified
         (ItemKey::tuningScale_SliderMarkerPosition::CurrentValueItem);
         oTunerSliderMarkerPosition.SendUpdate(true);

         if (TRUE == m_oTimerSlider.isActive())
         {
            m_oTimerSlider.start();
         }
      }
      break;
      default:
      {
         ETG_TRACE_USR3(("ERROR:TunerDataBindingHandler::request.ItemKey()"));
      }
      break;
   }
   ETG_TRACE_USR3(("SLIDER DRAG VALUE:  %f", (Candera::Float)(*oTunerSliderMarkerPosition).mDragCurrentValue));
   ETG_TRACE_USR3(("SLIDER CURRENT VALUE: %f", (Candera::Float)(*oTunerSliderMarkerPosition).mCurrentValue));
   ETG_TRACE_USR3(("g_u32SliderValue: %f", g_u32SliderValue));
   ETG_TRACE_USR3(("TunerDataBindingHandler::(sm_gCurrentView:%d)", sm_gCurrentView));

   if ((sm_gCurrentView == VIEW_FM1) || (sm_gCurrentView == VIEW_FM2) || (sm_gCurrentView == VIEW_FM_AST))
   {
      g_u32SliderValue = g_u32SliderValue * 1000;
      ETG_TRACE_USR3(("g_u32SliderValue:  %f", g_u32SliderValue));
      if ((g_u32SliderValue) > m_u32FMHiRange)
      {
         g_u32SliderValue = (Candera::Float)(m_u32FMHiRange);
         ETG_TRACE_USR3(("TunerDataBindingHandler, FM Frequency g_u32SliderValue > m_u32FMHiRange = %f", g_u32SliderValue));
      }
      else
      {
         ETG_TRACE_USR3(("TunerDataBindingHandler, FM Frequency g_u32SliderValue  in RANGE!!!!!!!!!!!!!!!"));
      }
   }
   if ((sm_gCurrentView == VIEW_AM) || (sm_gCurrentView == VIEW_AM_AST))
   {
      if (g_u32SliderValue < m_u32AMLowRange)
      {
         g_u32SliderValue = (Candera::Float)(m_u32AMLowRange);
         ETG_TRACE_USR3(("TunerDataBindingHandler,AM Frequency g_u32SliderValue < m_u32AMLowRange = %f", g_u32SliderValue));
      }
      else if (g_u32SliderValue > m_u32AMHiRange)
      {
         g_u32SliderValue = (Candera::Float)(m_u32AMHiRange);
         ETG_TRACE_USR3(("TunerDataBindingHandler, AM Frequency g_u32SliderValue > m_u32AMHiRange = %f", g_u32SliderValue));
      }
      else
      {
         ETG_TRACE_USR3(("TunerDataBindingHandler, Frequency g_u32SliderValue  in AM RANGE!!!!!!!!!!!!!!!"));
      }
   }

   m_u32SliderValue = (tU32)(g_u32SliderValue);
   //commented for build error
   if (FALSE == m_oTimerSlider.isActive())
   {
      bSendSliderRequestToBL();
      m_oTimerSlider.start();
   }
}


bool TunerDataBindingHandler::bSendSliderRequestToBL()
{
   tBool bChange = FALSE;
   static tU32 g_u32LastFreqValueSent = 0;
   ETG_TRACE_USR3(("TunerDataBindingHandler::bSendSliderRequestToBL(%d, %d)", m_u32SliderValue, g_u32LastFreqValueSent));

   if (m_u32SliderValue != g_u32LastFreqValueSent)
   {
      bChange = TRUE;
      g_u32LastFreqValueSent = m_u32SliderValue;
   }

   Slider_TunerFrequencyMsg* poSlider_TunerFrequency = COURIER_MESSAGE_NEW(Slider_TunerFrequencyMsg)(m_u32SliderValue);

   if (NULL != poSlider_TunerFrequency)
   {
      tU32 u32Value = poSlider_TunerFrequency->GetCurrent_Frequency_Value();
      ETG_TRACE_USR3(("SliderValue : %d", u32Value));

      tBool bResult = poSlider_TunerFrequency->Post();
      if (0 == bResult)
      {
         ETG_TRACE_USR3(("ERROR Slider_TunerFrequencyMsg SEND FAILDED "));
      }
   }
   else
   {
      ETG_TRACE_USR3(("ERROR: Slider_TunerFrequencyMsg CREATE FAILED"));
   }

   return bChange;
}


/************************************************************************
*FUNCTION: 		vUpdateCurrentbankIcon
*DESCRIPTION: 	Handles Updating the current active bank info in the source selection view
*PARAMETER:None
*
*RETURNVALUE: 	void
*HISTORY:
*25.09.14 shrivani
*Rev 0.1 Initial Revision.
************************************************************************/

void TunerDataBindingHandler::vUpdateCurrentbankIcon()
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::vUpdateCurrentbankIcon(sm_gSwitchState:%d)", sm_gSwitchState));
   /* Active bank information*/

   bool* pActiveBankIcon[] =
   {
      &((*oActiveBankInfo).mFM1_Active),
      &((*oActiveBankInfo).mFM2_Active),
      &((*oActiveBankInfo).mFMAST_Active),
      &((*oActiveBankInfo).mAM_Active),
      &((*oActiveBankInfo).mAMAST_Active),
      &((*oActiveBankInfo).mDAB_Active),
      &((*oActiveBankInfo).mDAB_AST_Active)
   };
   //Set all the active icon databinding elements to false
   for (tU8 u8Index = 0; u8Index < (sizeof(pActiveBankIcon) / sizeof(pActiveBankIcon[0])); ++u8Index)
   {
      *(pActiveBankIcon[u8Index]) = false;
   }

   // Handle test mode availabilityl
   (*oTestModeAvailabilityInfo).mFMTestModeAvail = false;
   (*oTestModeAvailabilityInfo).mDABTestModeAvail = false;

   if (!((*oActiveSourceInfo).mActiveSource))
   {
      oActiveBankInfo.MarkAllItemsModified();
      oActiveBankInfo.SendUpdate(true);
      oTestModeAvailabilityInfo.MarkAllItemsModified();
      oTestModeAvailabilityInfo.SendUpdate(true);
      return;
   }

   if ((sm_gSwitchState > 0) && ((tU32)sm_gSwitchState <= (sizeof(pActiveBankIcon) / sizeof(pActiveBankIcon[0]))))
   {
      *(pActiveBankIcon[sm_gSwitchState - 1]) = true;
   }
   oActiveBankInfo.MarkAllItemsModified();
   oActiveBankInfo.SendUpdate(true);

   if ((sm_gSwitchState >= EN_TUN_BANK_FM1) && (sm_gSwitchState <= EN_TUN_BANK_AM_AST))
   {
      (*oTestModeAvailabilityInfo).mFMTestModeAvail = true;
   }
   else if ((sm_gSwitchState == EN_TUN_BANK_DAB) || (sm_gSwitchState == EN_TUN_BANK_DAB_AST))
   {
      (*oTestModeAvailabilityInfo).mDABTestModeAvail = true;
   }
   else
   {
      ETG_TRACE_USR4(("Not in any tuner bank. FM and DAB test mode will not be available"));
   }
   oTestModeAvailabilityInfo.MarkAllItemsModified();
   oTestModeAvailabilityInfo.SendUpdate(true);
}


/************************************************************************
*FUNCTION: 		bOnUpdateRDSSupport
*DESCRIPTION: 	Handles Updating the visibility of the RDS layer info
*PARAMETER:
* 				corfoMsg:	(I)
* 					Courier message received
*
*RETURNVALUE: 	tBool
*					TRUE, if the message is consumed
*					FALSE, if the message should be forwarded
*
*HISTORY:
*06.05.13 Preethi Muthu
*Rev 0.1 Initial Revision.
************************************************************************/

bool TunerDataBindingHandler::bOnUpdateRDSSupport(const Courier::Message& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::bOnUpdateRDSSupport()"));

   bool bMsgConsumed = false; // Holds true if Model consumes the message

   const RDSSupport* poOnUpdateRDSSupport =
      message_cast<const RDSSupport*>(&corfoMsg);
   if (NULL != poOnUpdateRDSSupport)
   {
      bool bDataUpdated = false;
      ETG_TRACE_USR3(("TunerDataBindingHandler::bOnUpdateRDSSupport(%d)", poOnUpdateRDSSupport->GetRDSStatus()));
      //set the visibility of the node accordingly
      if (TRUE == poOnUpdateRDSSupport->GetRDSStatus())//RDS is active
      {
         (*oTunerInfoData).mVisible_Txt_Wgt_RDS = true;//Display RDS info
         (*oTunerInfoData).mVisible_Txt_Wgt_NoRDS = false;
         ETG_TRACE_USR3(("TunerDataBindingHandler::bOnUpdateRDSSupport():mVisible_Txt_Wgt_RDS=true"));
      }
      else //RDS is inactive
      {
         (*oTunerInfoData).mVisible_Txt_Wgt_NoRDS = true;//DO not Display RDS info
         (*oTunerInfoData).mVisible_Txt_Wgt_RDS = false;
         ETG_TRACE_USR3(("TunerDataBindingHandler::bOnUpdateRDSSupport():mVisible_Txt_Wgt_RDS=false"));
      }

      oTunerInfoData.MarkItemModified(ItemKey::StationInfoSource::Visible_Txt_Wgt_RDSItem);
      oTunerInfoData.MarkItemModified(ItemKey::StationInfoSource::Visible_Txt_Wgt_NoRDSItem);
      bDataUpdated = oTunerInfoData.SendUpdate(true);
   }
   bMsgConsumed = true;
   return bMsgConsumed;
}


/************************************************************************
*FUNCTION: 		vSendCheckBoxMsg
*DESCRIPTION: 	Sends the menu selection to BL
*
*RETURNVALUE: 	tVoid
************************************************************************/

tVoid TunerDataBindingHandler::vSendCheckBoxMsg(tenMenuID enType, Courier::UInt32 u32SelectedItem)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::vSendCheckBoxMsg()"));
   Courier::Message* lMsg = COURIER_MESSAGE_NEW(UpdateCheckBoxMsg)(enType, u32SelectedItem);
   if (0 != lMsg)
   {
      lMsg->Post();
      ETG_TRACE_USR3(("TunerDataBindingHandler::vSendCheckBoxMsg() UpdateCheckBoxMsg invoked"));
   }
   else
   {
      // UpdateCheckBoxMsg Msg creation failure
      ETG_TRACE_USR3(("TunerDataBindingHandler::vSendCheckBoxMsg() UpdateCheckBoxMsg failed "));
   }
}


void TunerDataBindingHandler::vSliderTimeoutHandler()
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::vSliderTimeoutHandler()"));
   if (TRUE == bSendSliderRequestToBL())
   {
      m_oTimerSlider.start();
   }
   else
   {
      m_oTimerSlider.stop();
   }
   //bSendSliderRequestToBL();
   //m_oTimerSlider.stop();
}


/************************************************************************
*FUNCTION: 		vPresetTimeoutHandler
*DESCRIPTION:   Time out function - function called from Timer
*PARAMETER:		None
*RETURNVALUE: 	NA
*
*HISTORY:
*10.08.13  Shrivani
*Rev 1.0 Initial Revision.
************************************************************************/
void TunerDataBindingHandler::vPresetTimeoutHandler()
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::vPresetTimeoutHandler()"));

   char sPopup_text[MAX_POPUP_TEXT];
   std::string l_sViewID(C_PFO_STOREPRESET_NAME);
   bool bDataUpdated = false;
   //update the progress bar when there is timeout
   m_u8timerCount++;
   switch (m_u8popupState)
   {
      case EN_POPUP_DISPLAY:
      {
         (*oTunerPopupData).mCurrentValue = (Candera::UInt16) m_u8timerCount;
         ETG_TRACE_USR3(("CurrentValue (*oTunerPopupData).mCurrentValue = %f ", (*oTunerPopupData).mCurrentValue));
         oTunerPopupData.MarkItemModified(ItemKey::Preset_Popup::CurrentValueItem);
         bDataUpdated = oTunerPopupData.SendUpdate(true);
         ETG_TRACE_USR3(("m_u8timerCount : %d", m_u8timerCount));
         if (TOTAL_TIME_COUNT == m_u8timerCount)
         {
            tU8 l_u8CurrentPresetNr = m_u8CurrentPresetNr;
            if (VIEW_FM2 == sm_gCurrentView)
            {
               l_u8CurrentPresetNr = m_u8CurrentPresetNr + 6;
            }
            sprintf(sPopup_text, "%d", l_u8CurrentPresetNr);
            (*oTunerPopupData).mVisible_Txt_Wgt_1 = false;
            (*oTunerPopupData).mVisible_Txt_Wgt_2 = true;
            (*oTunerPopupData).mPopup_text_2 = sPopup_text;
            ETG_TRACE_USR3(("sPopup_text %s ", sPopup_text));
            ETG_TRACE_USR3(("POPUP Text%s ", sPopup_text));
            oTunerPopupData.MarkItemModified(ItemKey::Preset_Popup::Visible_Txt_Wgt_1Item);
            oTunerPopupData.MarkItemModified(ItemKey::Preset_Popup::Visible_Txt_Wgt_2Item);
            oTunerPopupData.MarkItemModified(ItemKey::Preset_Popup::Popup_text_2Item);
            ETG_TRACE_USR3(("timer_expiry, CurrentValue (*oTunerPopupData).mCurrentValue = %f ", (*oTunerPopupData).mCurrentValue));
            bDataUpdated = oTunerPopupData.SendUpdate(true);
            ETG_TRACE_USR3(("bdataupdated = %d ", bDataUpdated));
            PresetStoreMsg* poPresetStoreMsg = COURIER_MESSAGE_NEW(PresetStoreMsg)(m_u8CurrentPresetNr);
            //commented for coverity
            //poPresetStoreMsg->Post(); // Posting Courier message with Band level to HMI Model
            m_u8popupState = (tU8)EN_POPUP_HIDE;
            m_u8timerCount = 0;
            if (NULL != poPresetStoreMsg)
            {
               poPresetStoreMsg->Post();
               m_oTimerPreset.start();
            }
         }

         else
         {
            m_oTimerPreset.start();
         }
      }
      break;

      case EN_POPUP_HIDE:
      {
         if (m_u8timerCount >= TOTAL_TIME_COUNT)
         {
            //Courier::Message * lMsg = COURIER_MESSAGE_NEW(DisablePopup)(Courier::ViewId(l_sViewID.c_str()));
            POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Hide, Courier::ViewId(l_sViewID.c_str()/*"TunerPopups#Popups#RADIO__TPOP_STATION_LIST_UPDATE_INFO"*/))));
            //if (0 != lMsg)
            //{
            //   //lMsg->Post();
            //   ETG_TRACE_USR3(("disable popup invoked" ));
            //   m_u8popupState = (tU8)EN_POPUP_DISPLAY;
            //}
            //else
            //{
            //   // DisablePopup Msg creation failure
            //   ETG_TRACE_USR3(("disable popup failed "));
            //   m_u8popupState = (tU8)EN_POPUP_DISPLAY;
            //}
            m_oTimerPreset.stop();
         }
         else
         {
            m_oTimerPreset.start();
            // DisablePopup Msg creation failure
         }
      }
      break;
      default:
      {
         ETG_TRACE_USR3(("TunerDataBindingHandler::vPresetTimeoutHandler() default"));
      }
      break;
   }
}


tVoid TunerDataBindingHandler::vUpdateTASourceProvider(void)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::vUpdateTASourceProvider"));
   if (m_bIsTASourceDAB)
   {
      (*oTAInfo).mTAPSName = m_sDABTASourcePSName;
   }
   else
   {
      (*oTAInfo).mTAPSName = m_sFMTASourcePSName;
   }
   ETG_TRACE_USR3(("TunerDataBindingHandler::vUpdateTASourceProvider((*oTAInfo).mTAPSName: %s)", (*oTAInfo).mTAPSName.GetCString()));
   oTAInfo.MarkItemModified(ItemKey::TAInfo::TAPSNameItem);
   oTAInfo.SendUpdate(true);
}


tVoid TunerDataBindingHandler::vUpdatePTY31SourceProvider(void)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::vUpdatePTY31SourceProvider"));
   if (m_bIsPTYSourceDAB)
   {
      (*oTAInfo).mPTY31PSName = m_sDABTASourcePSName;
   }
   else
   {
      (*oTAInfo).mPTY31PSName = m_sFMTASourcePSName;
   }
   ETG_TRACE_USR3(("TunerDataBindingHandler::vUpdatePTY31SourceProvider((*oTAInfo).mPTY31PSName: %s)", (*oTAInfo).mPTY31PSName.GetCString()));
   oTAInfo.MarkItemModified(ItemKey::TAInfo::PTY31PSNameItem);
   oTAInfo.SendUpdate(true);
}


/************************************************************************
*FUNCTION: 		bOnUpdateBandscanStatus
*DESCRIPTION: 	Handles background band scan status
*PARAMETER:
* 				corfoMsg:	(I)
* 					Courier message received
*
*RETURNVALUE: 	tBool
*					TRUE, if the message is consumed
*					FALSE, if the message should be forwarded
*
*HISTORY:
*06.05.13 Shrivani
*Rev 0.1 Initial Revision.
************************************************************************/

//bool TunerDataBindingHandler::onCourierMessage(const TunerPLMsg& corfoMsg)
void TunerDataBindingHandler::TunerPLMsg(tU8 u8EventType, tU32 u32Para, Candera::String const& szStrPara)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::bOnTunerPLMsg(%d %d)", u8EventType, u32Para));
   int targetRegion = TunerDataPoolConfig::getInstance()->readRegionType();
   bool bDataUpdated = false;

   tChar szData[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

   switch (u8EventType)
   {
      case EV_MSG_TO_PL_TA:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg:: EV_MSG_TO_PL_TA"));
         if (strcmp(szStrPara.GetCString(), "DAB") == 0)
         {
            m_bIsTASourceDAB = true;
         }
         else
         {
            m_bIsTASourceDAB = false;
         }
         vUpdateTASourceProvider();
         //ReqToDisplayPopup(PID_SYS_C_PFO_TA_NAME);
      }
      break;

      case EV_MSG_TO_PL_PTY31:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg:: EV_MSG_TO_PL_PTY31"));
         if (strcmp(szStrPara.GetCString(), "DAB") == 0)
         {
            m_bIsPTYSourceDAB = true;
         }
         else
         {
            m_bIsPTYSourceDAB = false;
         }
         vUpdatePTY31SourceProvider();
         //ReqToDisplayPopup(PID_SYS_C_PFO_PTY31_NAME);
      }
      break;
      case EV_MSG_TO_PL_RADIO_TEXT:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg:: EV_MSG_TO_PL_RADIO_TEXT (%s)", szStrPara));
         ETG_TRACE_USR3(("bOnTunerPLMsg(%s)", ((*oTunerInfoData).mRadioText).GetCString()));
      }
      break;
      case EV_MSG_TO_PL_TP:  // TP info shall be shown if TP info is set and it is RDS variant
      {
         if (!_isCabinOnly)
         {
            ETG_TRACE_USR3(("bOnTunerPLMsg:: EV_MSG_TO_PL_TP (%d)", u32Para));
            ETG_TRACE_USR3(("TunerDataBindingHandler::bOnTunerPLMsg(%d)", (*oTunerInfoData).mTPInfo));
            (*oTunerInfoData).mTPInfo = u32Para;
            (*oTunerInfoData).mTPInfo = (*oTunerInfoData).mTPInfo && (*oTunerInfoData).mVisible_Txt_Wgt_RDS;
            oTunerInfoData.MarkItemModified(ItemKey::StationInfoSource::TPInfoItem);
            bDataUpdated = oTunerInfoData.SendUpdate(true);
         }
      }
      break;
      case EV_MSG_TO_PL_HICUTABS_STATUS:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg:: EV_MSG_TO_PL_HICUTABS_STATUS (%d)", u32Para));
         sprintf(szData, "%d", u32Para);
         (*oEngineeringMenuData1).mSzHicut = szData;
         oEngineeringMenuData1.MarkItemModified(ItemKey::EngineeringInfo1::SzHicutItem);
         bDataUpdated = oEngineeringMenuData1.SendUpdate(true);
      }
      break;
      case EV_MSG_TO_PL_SHARXABS_STATUS:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg:: EV_MSG_TO_PL_SHARXABS_STATUS (%d)", u32Para));
         sprintf(szData, "%d", u32Para);
         (*oEngineeringMenuData1).mSzSharx = szData;
         oEngineeringMenuData1.MarkItemModified(ItemKey::EngineeringInfo1::SzSharxItem);
         bDataUpdated = oEngineeringMenuData1.SendUpdate(true);
      }
      break;
      case EV_MSG_TO_PL_MEASUREMODE_STATUS:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg:: EV_MSG_TO_PL_MEASUREMODE_STATUS (%d)", u32Para));
         //(*oFMSetupMenuInfo).mToggleMM = u32Para;
         //ETG_TRACE_USR3(("mMeasureOnOff: %d", (*oFMSetupMenuInfo).mToggleMM));
         //oFMSetupMenuInfo.MarkAllItemsModified();
         //bDataUpdated = oFMSetupMenuInfo.SendUpdate(true);
      }
      break;
      case EV_MSG_TO_PL_LINEAR_AUDIO:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg:: EV_MSG_TO_PL_LINEAR_AUDIO (%d)", u32Para));
         (*oEngineeringMenuData2).mLinearAudio = u32Para;
         ETG_TRACE_USR3(("mLinearAudio: %d", (*oEngineeringMenuData2).mLinearAudio));
         oEngineeringMenuData2.MarkItemModified(ItemKey::EngineeringInfo2::LinearAudioItem);
         bDataUpdated = oEngineeringMenuData2.SendUpdate(true);
      }
      break;
      case EV_MSG_TO_PL_OPT_TA:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_OPT_TA new(%d)", u32Para));
         ETG_TRACE_USR3(("misChecked_TA old = %d ", (*oCheckBoxData).misChecked_TA));
         (*oCheckBoxData).misChecked_TA = u32Para;
         oCheckBoxData.MarkItemModified(ItemKey::TunerOptions::isChecked_TAItem);
         bDataUpdated = oCheckBoxData.SendUpdate(true);
      }
      break;
      case EV_MSG_TO_PL_OPT_RADIO_TEXT:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_OPT_RADIO_TEXT new (%d)", u32Para));
         (*oCheckBoxData).misChecked_RadioTxt = u32Para;
         oCheckBoxData.MarkItemModified(ItemKey::TunerOptions::isChecked_RadioTxtItem);
         bDataUpdated = oCheckBoxData.SendUpdate(true);
         ETG_TRACE_USR3(("misChecked_RadioTxt after update= %d ", (*oCheckBoxData).misChecked_RadioTxt));
      }
      break;

      case EV_MSG_TO_PL_OPT_REGPGM:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_OPT_REGPGM(%d)", u32Para));
         ETG_TRACE_USR3(("misChecked_RDSRegPgm = %d ", (*oCheckBoxData).misChecked_RDSRegPgm));
         (*oCheckBoxData).misChecked_RDSRegPgm = u32Para;
         oCheckBoxData.MarkAllItemsModified();
         bDataUpdated = oCheckBoxData.SendUpdate(true);
      }
      break;

      case EV_MSG_TO_PL_OPT_ALT_FREQ:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_OPT_ALT_FREQ(%d)", u32Para));
         //(*oFMSetupMenuInfo).mToggleAF = u32Para;
         //oFMSetupMenuInfo.MarkAllItemsModified();
         //bDataUpdated = oFMSetupMenuInfo.SendUpdate(true);
      }
      break;
      case EV_MSG_TO_PL_MUTE:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_MUTE"));
         if (MUTE == u32Para)
         {
            (*oTunerInfoData).mViewName = (Candera::String)"%2";
            (*oHomeScreenInfo).mHomeScreen_MuteIconStatus = (_isCabinOnly) ? false : true;
         }
         else
         {
            (*oTunerInfoData).mViewName = (Candera::String)"%1";
            (*oHomeScreenInfo).mHomeScreen_MuteIconStatus = false;
         }
         oTunerInfoData.MarkItemModified(ItemKey::StationInfoSource::ViewNameItem);
         oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeScreen_MuteIconStatusItem);
         oHomeScreenInfo.SendUpdate();
         bDataUpdated = oTunerInfoData.SendUpdate(true);
      }
      break;

      case EV_MSG_TO_PL_RDS_CONFIG:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_RDS_CONFIG"));

         //(*oFMSetupMenuInfo).mToggleRDS = u32Para;//Display RDS info
         //oFMSetupMenuInfo.MarkAllItemsModified();
         //bDataUpdated = oFMSetupMenuInfo.SendUpdate(true);
      }
      break;
      case EV_MSG_TO_PL_AF_CONFIG:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_AF_CONFIG(%d)", u32Para));
         (*oRDSOptions).mVisible_AF = (u32Para == 1);
         oRDSOptions.MarkItemModified(ItemKey::RDSOptions::Visible_AFItem);
         bDataUpdated = oRDSOptions.SendUpdate(true);
         (*oMenu_NumOptionsData).mMainMenu_NumOptions += 1;
         oMenu_NumOptionsData.MarkItemModified(ItemKey::Menu_NumOptions::MainMenu_NumOptionsItem);
         bDataUpdated = oMenu_NumOptionsData.SendUpdate(true);
      }
      break;
      case EV_MSG_TO_PL_REG_CONFIG:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_REG_CONFIG(%d)", u32Para));
         (*oRDSOptions).mVisible_Reg = (u32Para == 1);
         oRDSOptions.MarkItemModified(ItemKey::RDSOptions::Visible_RegItem);
         bDataUpdated = oRDSOptions.SendUpdate(true);
         (*oMenu_NumOptionsData).mMainMenu_NumOptions += 1;
         oMenu_NumOptionsData.MarkItemModified(ItemKey::Menu_NumOptions::MainMenu_NumOptionsItem);
         bDataUpdated = oMenu_NumOptionsData.SendUpdate(true);
      }
      break;
      case EV_MSG_TO_PL_TA_CONFIG:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_TA_CONFIG(%d)", u32Para));
         //TA is active
      }
      break;
      case EV_MSG_TO_PL_AM_SLIDER_RANGE:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_AM_SLIDER_RANGE(%d)", u32Para));
         switch (u32Para)
         {
            case AM_SLIDER_1611:
            {
               (*otuningSlider).mImage_SliderAM = (Candera::String)EU_9k;
               (*otuningSlider).mImage_Slider_BG = (Candera::String)AM_BG_1611;
               (*otuningSlider).mMinValueAM = (Candera::Float)514.00;
               (*otuningSlider).mMaxValueAM = (Candera::Float)1614.00;
            }
            break;
            case AM_SLIDER_1629:
            case AM_SLIDER_1620:
            {
               (*otuningSlider).mImage_SliderAM = (Candera::String)GEX_9k;
               (*otuningSlider).mImage_Slider_BG = (Candera::String)AM_BG_1629;
               (*otuningSlider).mMinValueAM = (Candera::Float)529.00;
               (*otuningSlider).mMaxValueAM = (Candera::Float)1629.00;
            }
            break;
            case AM_SLIDER_1710:
            {
               (*otuningSlider).mImage_SliderAM = (Candera::String)GEX_10k;
               (*otuningSlider).mImage_Slider_BG = (Candera::String)AM_BG_1710;
               (*otuningSlider).mMinValueAM = (Candera::Float)523.00;
               (*otuningSlider).mMaxValueAM = (Candera::Float)1710.00;
            }
            break;
            default:
            {
               (*otuningSlider).mImage_SliderAM = (Candera::String)GEX_9k;
               (*otuningSlider).mImage_Slider_BG = (Candera::String)AM_BG_1629;
               (*otuningSlider).mMinValueAM = (Candera::Float)529.00;
               (*otuningSlider).mMaxValueAM = (Candera::Float)1629.00;
            }
            break;
         }
         otuningSlider.MarkItemModified(ItemKey::tuningSlider::Image_SliderAMItem);
         otuningSlider.MarkItemModified(ItemKey::tuningSlider::Image_Slider_BGItem);
         otuningSlider.MarkItemModified(ItemKey::tuningSlider::MinValueAMItem);
         otuningSlider.MarkItemModified(ItemKey::tuningSlider::MaxValueAMItem);
         bDataUpdated = otuningSlider.SendUpdate(true);
      }
      break;
      case EV_MSG_TO_PL_AM_SLIDER_STEPSIZE:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_AM_SLIDER_STEPSIZE(%d)", u32Para));
         (*otuningSlider).mStepSizeAM = u32Para;
         otuningSlider.MarkItemModified(ItemKey::tuningSlider::StepSizeAMItem); /* Mark only the StepSize item for updation */
         bDataUpdated = otuningSlider.SendUpdate(true);
      }
      break;
      case EV_MSG_TO_PL_AM_SLIDERNOOFSTEPS:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_AM_SLIDERNOOFSTEPS(%d)", u32Para));
         (*otuningSlider).mNoOfStepsAM = u32Para;
         otuningSlider.MarkItemModified(ItemKey::tuningSlider::NoOfStepsAMItem); /* Mark only the NoOfSteps item for updation */
         bDataUpdated = otuningSlider.SendUpdate(true);
      }
      break;
      case EV_MSG_TO_PL_AM_SLIDER_MAXVALUE:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg:EV_MSG_TO_PL_AM_SLIDER_MAXVALUE(%d)", u32Para));
         m_u32AMHiRange = u32Para;
      }
      break;
      case EV_MSG_TO_PL_AM_SLIDER_MINVALUE:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg:EV_MSG_TO_PL_AM_SLIDER_MINVALUE(%d)", u32Para));
         m_u32AMLowRange = u32Para;
      }
      break;
      case EV_MSG_TO_PL_FM_SLIDER_STEPSIZE:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_FM_SLIDER_STEPSIZE (%d)", u32Para));
         (*otuningSlider).mStepSizeFM = u32Para;
         otuningSlider.MarkItemModified(ItemKey::tuningSlider::StepSizeFMItem); /* Mark only the StepSize item for updation */
         bDataUpdated = otuningSlider.SendUpdate(true);
         updatedata();
      }
      break;
      case EV_MSG_TO_PL_FM_SLIDERNOOFSTEPS:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_FM_SLIDERNOOFSTEPS (%d)", u32Para));
         //(*otuningSlider).mNoOfStepsFM = u32Para;
         //(*otuningSlider).mNoOfStepsFM = 87.62;
         otuningSlider.MarkItemModified(ItemKey::tuningSlider::NoOfStepsFMItem); /* Mark only the NoOfSteps item for updation */
         bDataUpdated = otuningSlider.SendUpdate(true);
      }
      break;
      case EV_MSG_TO_PL_FM_SLIDER_MAXVALUE:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_FM_SLIDER_MAXVALUE (%d)", u32Para));
         //m_u32FMHiRange = u32Para;
         //(*otuningSlider).mMaxValueFM = 108.375;
         //otuningSlider.MarkItemModified(ItemKey::tuningSlider::MaxValueFMItem); /* Mark only the MaxValue item for updation */
         //bDataUpdated = otuningSlider.SendUpdate(true);
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_FM_SLIDER_MAXVALUE (%d)", m_u32FMHiRange));
      }
      break;
      case EV_MSG_TO_PL_TA_ICON:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_TA_ICON (%d)", u32Para));
         if (!_isCabinOnly)
         {
            if (u32Para)
            {
               if (targetRegion == EUR_REGION)  //Only for EU
               {
                  ETG_TRACE_USR3(("bOnTunerPLMsg:TA Button active"));
                  (*oRadioBandInfoData).mVisible_Txt_Wgt_TA = u32Para;
                  (*oRadioBandInfoData).mVisible_Txt_Wgt_TA_In_AM = u32Para;
                  uint8 bandinfo = SrcChangeHandler::poGetInstance()->getsourceInfo();
                  if ((bandinfo == SRC_TUNER_AM))
                  {
                     (*oRadioBandInfoData).mVisible_Txt_Wgt_TA_FM = false;
                     (*oRadioBandInfoData).mVisible_Txt_Wgt_TA_DAB = false;
                  }
                  else if ((bandinfo == SRC_TUNER_FM))
                  {
                     (*oRadioBandInfoData).mVisible_Txt_Wgt_TA_FM = true;
                     (*oRadioBandInfoData).mVisible_Txt_Wgt_TA_DAB = false;
                  }
                  else if ((bandinfo == SRC_DAB))
                  {
                     (*oRadioBandInfoData).mVisible_Txt_Wgt_TA_DAB = true;
                     (*oRadioBandInfoData).mVisible_Txt_Wgt_TA_FM = false;
                  }
                  else if ((bandinfo == SRC_TUNER_WX))
                  {
                     (*oRadioBandInfoData).mVisible_Txt_Wgt_TA_FM = false;
                     (*oRadioBandInfoData).mVisible_Txt_Wgt_TA_DAB = false;
                  }

                  oRadioBandInfoData.MarkItemModified(ItemKey::RadioBandInfo::Visible_Txt_Wgt_TAItem);
                  oRadioBandInfoData.MarkItemModified(ItemKey::RadioBandInfo::Visible_Txt_Wgt_TA_In_AMItem);
                  oRadioBandInfoData.MarkItemModified(ItemKey::RadioBandInfo::Visible_Txt_Wgt_TA_FMItem);
                  oRadioBandInfoData.MarkItemModified(ItemKey::RadioBandInfo::Visible_Txt_Wgt_TA_DABItem);
               }
               bDataUpdated = oRadioBandInfoData.SendUpdate(true);
               // (*oFMSetupMenuInfo).mToggleTA_status = (u32Para);
            }
            else
            {
               ETG_TRACE_USR3(("bOnTunerPLMsg:TA Button inactive"));
               (*oRadioBandInfoData).mVisible_Txt_Wgt_TA_FM = false;
               (*oRadioBandInfoData).mVisible_Txt_Wgt_TA_DAB = false;
               (*oRadioBandInfoData).mVisible_Txt_Wgt_TA = u32Para;
               (*oRadioBandInfoData).mVisible_Txt_Wgt_TA_In_AM = u32Para;
               //   (*oFMSetupMenuInfo).mToggleTA_status = (u32Para);
               oRadioBandInfoData.MarkItemModified(ItemKey::RadioBandInfo::Visible_Txt_Wgt_TAItem);
               oRadioBandInfoData.MarkItemModified(ItemKey::RadioBandInfo::Visible_Txt_Wgt_TA_In_AMItem);
               oRadioBandInfoData.MarkItemModified(ItemKey::RadioBandInfo::Visible_Txt_Wgt_TA_FMItem);
               oRadioBandInfoData.MarkItemModified(ItemKey::RadioBandInfo::Visible_Txt_Wgt_TA_DABItem);
               bDataUpdated = oRadioBandInfoData.SendUpdate(true); //added for coverity may 4
            }
            // oFMSetupMenuInfo.MarkAllItemsModified();
         }
      }
      break;
      case EV_MSG_TO_PL_TA_ICON_TYPE:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_TA_ICON_TYPE (%d)", u32Para));
         if (!_isCabinOnly)
         {
            if (u32Para)
            {
               (*oRadioBandInfoData).mImage_TA = (Candera::String)TA_ICON;
               //(*oTunerInfoData).mTPInfo = u32Para;
               (*oTunerInfoData).mDabTPInfo = u32Para;
               (*oDABChannelInfo).mTpStatus = (Candera::String)"%1";
               oRadioBandInfoData.MarkItemModified(ItemKey::RadioBandInfo::Image_TAItem);
               oDABChannelInfo.MarkItemModified(ItemKey::DABChannelInfo::TpStatusItem);
            }
            else
            {
               (*oRadioBandInfoData).mImage_TA = (Candera::String)TA_CROSSED;
               //(*oTunerInfoData).mTPInfo = u32Para;
               (*oTunerInfoData).mDabTPInfo = u32Para;
               (*oDABChannelInfo).mTpStatus = (Candera::String)"%2";
               oRadioBandInfoData.MarkItemModified(ItemKey::RadioBandInfo::Image_TAItem);
               oDABChannelInfo.MarkItemModified(ItemKey::DABChannelInfo::TpStatusItem);
            }

            //oTunerInfoData.MarkItemModified(ItemKey::StationInfoSource::TPInfoItem);
            oTunerInfoData.MarkItemModified(ItemKey::StationInfoSource::DabTPInfoItem);
            oTunerInfoData.SendUpdate();
            oDABChannelInfo.SendUpdate();
            bDataUpdated = oRadioBandInfoData.SendUpdate();
         }
      }
      break;
      case EV_MSG_TO_PL_NAVI_CONFIG:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_NAVI_CONFIG "));
         (*oAppFlyIn).mVisible_Media = false;
         (*oAppFlyIn).mVisible_Navi = false;
         if (TRUE == u32Para)//Map active
         {
            (*oTunerInfoData).mVisible_Txt_Wgt_Navi = true;
            (*oAppFlyIn).mVisible_Navi = true;
            ETG_TRACE_USR3(("bOnTunerPLMsg:bOnUpdateRDSSupport():mVisible_Txt_Wgt_Navi=true"));
         }
         else
         {
            (*oTunerInfoData).mVisible_Txt_Wgt_Navi = false;
            (*oAppFlyIn).mVisible_Media = true;
            ETG_TRACE_USR3(("bOnTunerPLMsg:bOnUpdateRDSSupport():mVisible_Txt_Wgt_Navi=false"));
         }

         oTunerInfoData.MarkItemModified(ItemKey::StationInfoSource::Visible_Txt_Wgt_NaviItem);
         bDataUpdated = oTunerInfoData.SendUpdate(true);
         oAppFlyIn.MarkItemModified(ItemKey::AppFlyIn::Visible_NaviItem);
         oAppFlyIn.MarkItemModified(ItemKey::AppFlyIn::Visible_MediaItem);
         bDataUpdated = oAppFlyIn.SendUpdate(true);
      }
      break;
      case EV_MSG_TO_PL_VARIANT_TYPE:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_VARIANT_TYPE(%d)", u32Para));
         if (EUR_REGION == u32Para)
         {
            (*oVariantInfo).mDAB_Available = true;
            (*oVariantInfo).mHD_Available = false;
            (*oVariantInfo).mWX_Available = false;
         }
         else
         {
            (*oVariantInfo).mDAB_Available = false;
            (*oVariantInfo).mHD_Available = true;
            (*oVariantInfo).mWX_Available = true;
         }
         oVariantInfo.MarkAllItemsModified();
         bDataUpdated = oVariantInfo.SendUpdate(true);
         break;
      }
      case EV_MSG_TO_PL_SEL_ELEM:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_SEL_ELEM (%d) ", u32Para));
         if (SM_TUNER_LISTTYPE_FM_STATIONLIST == g_e8BrowseViewID)
         {
            (*oBrowseListSelectedElement).mTun_StationSelected = u32Para;
            oBrowseListSelectedElement.MarkItemModified(ItemKey::BrowseListSelectedLine::Tun_StationSelectedItem);
            bDataUpdated = oBrowseListSelectedElement.SendUpdate(true);
         }
#if 0
         if (SM_DAB_LISTTYPE_SERVICELIST == g_e8BrowseViewID)
         {
            (*oBrowseListSelectedElement).mDAB_ServiceSelected = u32Para;
            oBrowseListSelectedElement.MarkItemModified(ItemKey::BrowseListSelectedLine::DAB_ServiceSelectedItem);
            bDataUpdated = oBrowseListSelectedElement.SendUpdate(true);
         }
#endif
         break;
      }
      case EV_MSG_TO_PL_DABUPDATE_POPUP:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_DABUPDATE_POPUP "));
         //ReqToDisplayPopup(PID_APP_POPUP_DABSRVLIST);
         //Update the databinding text...
         (*oTunerPopupData).mAST_CannotStore = (Candera::String)"%1";
         oTunerPopupData.MarkItemModified(ItemKey::Preset_Popup::AST_CannotStoreItem);
         bDataUpdated = oTunerPopupData.SendUpdate(true);
      }
      break;
      case EV_MSG_TO_PL_TA_PSNAME:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_TA_PSNAME (%s)", szStrPara));
         m_sFMTASourcePSName = szStrPara; //vAnnouncementInfoStatus() and vTA_Announcement_Status() order is not consistent such that AM FM PSName is  overwriiten hence store into member variable
         //SUZUKI-21332
         vUpdatePTY31SourceProvider();
         vUpdateTASourceProvider();
      }
      break;
      case EV_MSG_TO_PL_OPT_DAB_DYNCONTENT:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_OPT_DAB_DYNCONTENT (%d)", u32Para));
         (*oCheckBoxData).misChecked_DAB_DynamicContent = u32Para;
         (*oDABChannelInfo).mRadioTextVisible = (bool)u32Para;
         oCheckBoxData.MarkItemModified(ItemKey::TunerOptions::isChecked_DAB_DynamicContentItem);
         oDABChannelInfo.MarkItemModified(ItemKey::DABChannelInfo::RadioTextVisibleItem);
         bDataUpdated = oCheckBoxData.SendUpdate(true);
         bDataUpdated = oDABChannelInfo.SendUpdate(true);
         ETG_TRACE_USR3(("misChecked_DAB_DynamicContent = %d ", (*oCheckBoxData).misChecked_DAB_DynamicContent));
      }
      break;
      case EV_MSG_TO_PL_OPT_DAB_FM_LINKING:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_OPT_DAB_FM_LINKING (%d)", u32Para));
         (*oCheckBoxData).misChecked_DAB_FM_Following = u32Para;
         oCheckBoxData.MarkItemModified(ItemKey::TunerOptions::isChecked_DAB_FM_FollowingItem);
         bDataUpdated = oCheckBoxData.SendUpdate(true);
         ETG_TRACE_USR3(("misChecked_DAB_FM_Following = %d ", (*oCheckBoxData).misChecked_DAB_FM_Following));
      }
      break;
      case EV_MSG_TO_PL_OPT_DAB_DAB_LINKING:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_OPT_DAB_DAB_LINKING (%d)", u32Para));
         (*oCheckBoxData).misChecked_DAB_DAB_Linking = u32Para;
         oCheckBoxData.MarkItemModified(ItemKey::TunerOptions::isChecked_DAB_DAB_LinkingItem);
         bDataUpdated = oCheckBoxData.SendUpdate(true);
         ETG_TRACE_USR3(("misChecked_DAB_DAB_Linking = %d ", (*oCheckBoxData).misChecked_DAB_DAB_Linking));
      }
      break;
      case EV_MSG_TO_PL_DABPTY_FILTER_SETTINGS:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_DABPTY_FILTER_SETTINGS (%d)", u32Para));
         tU32 u32PtyMask = u32Para;
         ETG_TRACE_USR3(("Filter settings = %d ", u32PtyMask));
         (*oDabOptionsData).misChecked_EnablePTYFilter = u32PtyMask & (1 << EN_PTY_FILTER_ENABLED);
         (*oDabOptionsData).misChecked_News = u32PtyMask & (1 << EN_PTY_TYPE_NEWS);
         (*oDabOptionsData).misChecked_Culture = u32PtyMask & (1 << EN_PTY_TYPE_CULTURE);
         (*oDabOptionsData).misChecked_Sports = u32PtyMask & (1 << EN_PTY_TYPE_SPORTS);
         (*oDabOptionsData).misChecked_Music = u32PtyMask & (1 << EN_PTY_TYPE_MUSIC);
         (*oDabOptionsData).misChecked_Special = u32PtyMask & (1 << EN_PTY_TYPE_SPECIAL);
         oDabOptionsData.MarkAllItemsModified();
         bDataUpdated = oDabOptionsData.SendUpdate(true);
      }
      break;
      case EV_MSG_TO_PL_SOURCE_SEL_ICON:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_SOURCE_SEL_ICON (%d)", u32Para));
         (*oActiveSourceInfo).mActiveSource = u32Para;
         //commented because the source selection icon is Radio everytime
         //if(0 == sm_gKDS_Src_sel)
         // {
         //    (*oActiveSourceInfo).mBitmapforSource =(Candera::String)SOURCE_RADIO;
         // }
         // else
         // {
         if (u32Para)
         {
            (*oActiveSourceInfo).mBitmapforSource = (Candera::String)SOURCE_RADIO;
         }
         else
         {
            (*oActiveSourceInfo).mBitmapforSource = (Candera::String)SOURCE_MEDIA;
         }
         oActiveSourceInfo.MarkAllItemsModified();
         bDataUpdated = oActiveSourceInfo.SendUpdate(true);
         vUpdateCurrentbankIcon();
      }
      break;
      case EV_MSG_TO_PL_AUTOSTORE_POPUP:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_AUTOSTORE_POPUP(%d)", u32Para));
         //ReqToDisplayPopup(PID_APP_POPUP_DABSRVLIST);
         //Update the databinding text...
         (*oTunerPopupData).mAST_CannotStore = (Candera::String)"%2";
         oTunerPopupData.MarkItemModified(ItemKey::Preset_Popup::AST_CannotStoreItem);
         bDataUpdated = oTunerPopupData.SendUpdate(true);
      }
      break;
      case EV_MSG_TO_PL_SPI_ICON:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_SPI_ICON (%d)", u32Para));
         if (u32Para)
         {
            (*oAppFlyIn).mPhoneSPILink = true;
         }
         else
         {
            (*oAppFlyIn).mPhoneSPILink = false;
         }
         oAppFlyIn.MarkAllItemsModified();
         bDataUpdated = oAppFlyIn.SendUpdate(true);
      }
      break;
      case EV_MSG_TO_PL_TP_CURRENT_STATION:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg: EV_MSG_TO_PL_TP_CURRENT_STATION (%d)", u32Para));
         (*oTunerInfoData).mCurrentStationTPInfo = (bool)(u32Para);
         oTunerInfoData.MarkItemModified(ItemKey::StationInfoSource::CurrentStationTPInfoItem);
         bDataUpdated = oTunerInfoData.SendUpdate(true);
      }
      break;
      default:
      {
         ETG_TRACE_USR3(("bOnTunerPLMsg:default "));
      }
      break;
   }

   if (false == bDataUpdated)
   {
      ETG_TRACE_USR3(("ERROR: Data updation failed oTunerInfoData/oCheckBoxData"));
   }
}


/************************************************************************
* 					Courier message from the Message Router
*RETURNVALUE: 	bool
*HISTORY:
*06.06.14 Rev. 0.1 Kiran S
* 		  Initial Revision.
************************************************************************/
bool TunerDataBindingHandler::bOnUpdateBrowseMsg(const Courier::Message& corfoMsg)
{
   bool bMsgConsumed = false; // Holds true if Model consumes the message

   ETG_TRACE_USR3(("TunerDataBindingHandler : bOnUpdateBrowseMsg()"));

   bool bUpdateStatus = false;
   const BrowseListRespMsg* poBrowseListRespMsg =
      message_cast<const BrowseListRespMsg*>(&corfoMsg);
   if (NULL != poBrowseListRespMsg)
   {
      bMsgConsumed = true;
      Courier::UInt8 u8ListType = poBrowseListRespMsg->GetG_SM_ListType();

      ETG_TRACE_USR3(("TunerDataBindingHandler : bOnUpdateBrowseMsg(%d, %d)", u8ListType,
                      poBrowseListRespMsg->GetListSize()));

      switch (u8ListType)
      {
         case SM_TUNER_LISTTYPE_FM_STATIONLIST:
         {
            /* g_e8BrowseViewID = u8ListType;
             ETG_TRACE_USR3(("TunerDataBindingHandler :FM StationList"));
             Candera::MemoryManagement::SharedPointer<WindowListWidgetBackEndInterface> poStationListHead;
             poStationListHead = poBrowseListRespMsg->GetDataPointer();
             if( poStationListHead != NULL )
             {
             ETG_TRACE_USR3(("StationList:  Element 1 : %s", poStationListHead->stgGetStringData(0).GetCString()));
             ETG_TRACE_USR3(("StationList:  Element 2 : %s", poStationListHead->stgGetStringData(1).GetCString()));
             ETG_TRACE_USR3(("StationList:  Element 3 : %s", poStationListHead->stgGetStringData(2).GetCString()));
             ETG_TRACE_USR3(("StationList:  Element 4 : %s", poStationListHead->stgGetStringData(3).GetCString()));
             }
             else
             {
             ETG_TRACE_USR3(("poStationListHead head pointer is null"));
             }
             (*oBrowseListPointer).mTun_StationListDataPtr = poBrowseListRespMsg->GetDataPointer();
             (*oBrowseListSize).mTun_StationListSize = poBrowseListRespMsg->GetListSize();
             (*oBrowseListSize).mTun_StationWindowListSize = poBrowseListRespMsg->GetWindowListSize();

             ETG_TRACE_USR3(("TunerDataBindingHandler : SM_TUNER_LISTTYPE_FM_STATIONLIST List Size : (%d %d)",poBrowseListRespMsg->GetListSize(), poBrowseListRespMsg->GetWindowListSize()));
             oBrowseListPointer.MarkItemModified(ItemKey::BrowseListPointer::Tun_StationListDataPtrItem);
             oBrowseListSize.MarkItemModified(ItemKey::BrowseListSize::Tun_StationWindowListSizeItem);
             oBrowseListSize.MarkItemModified(ItemKey::BrowseListSize::Tun_StationListSizeItem);
             bUpdateStatus = oBrowseListPointer.SendUpdate();
             bUpdateStatus = bUpdateStatus &&oBrowseListSize.SendUpdate();*/
         }
         break;
         case SM_TUNER_LISTTYPE_AFLIST:
         {
            /* g_e8BrowseViewID = u8ListType;
             ETG_TRACE_USR3(("TunerDataBindingHandler :AF List"));
             Candera::MemoryManagement::SharedPointer<WindowListWidgetBackEndInterface> poStationListHead;
             poStationListHead = poBrowseListRespMsg->GetDataPointer();
             if( poStationListHead != NULL )
             {
             ETG_TRACE_USR3(("AFList:  Element 1 : %s", poStationListHead->stgGetStringData(0).GetCString()));
             }
             else
             {
             ETG_TRACE_USR3(("poStationListHead head pointer is null"));
             }
             (*oBrowseListPointer).mAF_ServiceListDataPtr = poBrowseListRespMsg->GetDataPointer();
             (*oBrowseListSize).mAF_ServiceListSize = poBrowseListRespMsg->GetListSize();
             (*oBrowseListSize).mAF_ServiceWindowListSize = poBrowseListRespMsg->GetWindowListSize();

             ETG_TRACE_USR3(("TunerDataBindingHandler : SM_TUNER_LISTTYPE_AFLIST List Size : (%d %d)",poBrowseListRespMsg->GetListSize(), poBrowseListRespMsg->GetWindowListSize()));
             oBrowseListPointer.MarkItemModified(ItemKey::BrowseListPointer::AF_ServiceListDataPtrItem);
             oBrowseListSize.MarkItemModified(ItemKey::BrowseListSize::AF_ServiceWindowListSizeItem);
             oBrowseListSize.MarkItemModified(ItemKey::BrowseListSize::AF_ServiceListSizeItem);
             bUpdateStatus = oBrowseListPointer.SendUpdate();
             bUpdateStatus = bUpdateStatus &&oBrowseListSize.SendUpdate();*/
         }
         break;
         case SM_TUNER_LISTTYPE_TMCLIST:
         {
            /*g_e8BrowseViewID = u8ListType;
            ETG_TRACE_USR3(("TunerDataBindingHandler :TMC List"));
            Candera::MemoryManagement::SharedPointer<WindowListWidgetBackEndInterface> poStationListHead;
            poStationListHead = poBrowseListRespMsg->GetDataPointer();
            if( poStationListHead != NULL )
            {
            ETG_TRACE_USR3(("TMCList:  Element 1 : %s", poStationListHead->stgGetStringData(0).GetCString()));
            }
            else
            {
            ETG_TRACE_USR3(("poStationListHead head pointer is null"));
            }
            (*oBrowseListPointer).mTMC_ServiceListDataPtr = poBrowseListRespMsg->GetDataPointer();
            (*oBrowseListSize).mTMC_ServiceListSize = poBrowseListRespMsg->GetListSize();
            (*oBrowseListSize).mTMC_ServiceWindowListSize = poBrowseListRespMsg->GetWindowListSize();

            ETG_TRACE_USR3(("TunerDataBindingHandler : SM_TUNER_LISTTYPE_TMCLIST List Size : (%d %d)",poBrowseListRespMsg->GetListSize(), poBrowseListRespMsg->GetWindowListSize()));
            oBrowseListPointer.MarkItemModified(ItemKey::BrowseListPointer::TMC_ServiceListDataPtrItem);
            oBrowseListSize.MarkItemModified(ItemKey::BrowseListSize::TMC_ServiceWindowListSizeItem);
            oBrowseListSize.MarkItemModified(ItemKey::BrowseListSize::TMC_ServiceListSizeItem);
            bUpdateStatus = oBrowseListPointer.SendUpdate();
            bUpdateStatus = bUpdateStatus &&oBrowseListSize.SendUpdate(); */
         }
         break;

         case SM_DAB_LISTTYPE_SERVICELIST:
         {
            /* g_e8BrowseViewID = u8ListType;
             ETG_TRACE_USR3(("TunerDataBindingHandler :DAB Service List"));
             Candera::MemoryManagement::SharedPointer<WindowListWidgetBackEndInterface> poStationListHead;
             poStationListHead = poBrowseListRespMsg->GetDataPointer();
             if( poStationListHead != NULL )
             {
             ETG_TRACE_USR3(("ServiceList:  Element 1 : %s", poStationListHead->stgGetStringData(0).GetCString()));
             ETG_TRACE_USR3(("ServiceList:  Element 2 : %s", poStationListHead->stgGetStringData(1).GetCString()));
             ETG_TRACE_USR3(("ServiceList:  Element 3 : %s", poStationListHead->stgGetStringData(2).GetCString()));
             }
             else
             {
             ETG_TRACE_USR3(("poStationListHead head pointer is null"));
             }
             (*oBrowseListPointer).mDAB_ServiceListDataPtr = poBrowseListRespMsg->GetDataPointer();
             (*oBrowseListSize).mDAB_ServiceListSize = poBrowseListRespMsg->GetListSize();
             (*oBrowseListSize).mDAB_ServiceWindowListSize = poBrowseListRespMsg->GetWindowListSize();

             ETG_TRACE_USR3(("TunerDataBindingHandler : SM_DAB_LISTTYPE_SERVICELIST List Size : (%d %d)",poBrowseListRespMsg->GetListSize(), poBrowseListRespMsg->GetWindowListSize()));
             oBrowseListPointer.MarkItemModified(ItemKey::BrowseListPointer::DAB_ServiceListDataPtrItem);
             oBrowseListSize.MarkItemModified(ItemKey::BrowseListSize::DAB_ServiceWindowListSizeItem);
             oBrowseListSize.MarkItemModified(ItemKey::BrowseListSize::DAB_ServiceListSizeItem);
             bUpdateStatus = oBrowseListPointer.SendUpdate();
             bUpdateStatus = bUpdateStatus &&oBrowseListSize.SendUpdate(); */
         }
         break;

         default:
            ETG_TRACE_USR3(("TunerDataBindingHandler :Unknown list type in bOnUpdateBrowseMsg"));
            bMsgConsumed = false;
            break;
      }
   }
   else
   {
      ETG_TRACE_USR3(("TunerDataBindingHandler : poBrowseListRespMsg NULL Pointer"));
   }
   return bMsgConsumed;
}


/************************************************************************
* 					Courier message from the Message Router
*RETURNVALUE: 	bool
*HISTORY:
*06.06.14 Rev. 0.1 Kiran S
* 		  Initial Revision.
************************************************************************/
bool TunerDataBindingHandler::bOnUpdateWindowData(const Courier::Message& corfoMsg)
{
   /*const UpdateWindowData* pUpdateWindow = message_cast<const UpdateWindowData*>(&corfoMsg);

   if(NULL != pUpdateWindow)
   {
   UInt8 u8ListID = pUpdateWindow->GetListIdentifier();
   tU16 u16WindowDistance = pUpdateWindow->GetWindowDistance();
   PopulateScrollListWindowRequest *pPopulateScrollListWindowRequest =
   COURIER_MESSAGE_NEW(::PopulateScrollListWindowRequest)(u8ListID, pUpdateWindow->GetScrollDirOnWindowTrigger(), u16WindowDistance);
   pPopulateScrollListWindowRequest->Post();
   }*/

   return true;
}


bool TunerDataBindingHandler::bOnDABChannelInfoMsg(const Courier::Message& corfoMsg)
{
   tChar szStatusText[3] = { 0 };
   ETG_TRACE_USR3(("TunerDataBindingHandler::bOnDABChannelInfoMsg()"));
   bool bMsgConsumed = false; // Holds true if Model consumes the message

   /* Update the HMI to indicate the current waveband */
   const DABChannelInfoMsg* copoSourceMsg =
      message_cast<const DABChannelInfoMsg*>(&corfoMsg);
   if (NULL != copoSourceMsg)
   {
      if (!_isCabinOnly)
      {
         (*oDABChannelInfo).mTpStatus = (Candera::String)(copoSourceMsg->GetTpAvailability() ? "%1" : "");
      }
      tenDabStatus enStatus = (tenDabStatus)copoSourceMsg->GetDabStatus();
      if (enStatus > EN_DAB_NO_RECEPTION)
      {
         enStatus = EN_DAB_ALL_OK;
      }

      if (enStatus > EN_DAB_ALL_OK)
      {
         sprintf(szStatusText, "%%%d", enStatus);
      }
      (*oDABChannelInfo).mDabStatus = (Candera::String)szStatusText;
      (*oDABChannelInfo).mEnsembleLabel = copoSourceMsg->GetEnsembleLabel();
      (*oDABChannelInfo).mServiceLabel = copoSourceMsg->GetServiceLabel();
      (*oDABChannelInfo).mRadioText = copoSourceMsg->GetRadioText();
      oDABChannelInfo.MarkAllItemsModified();
      oDABChannelInfo.SendUpdate(true);
      bMsgConsumed = true;
   }
   else
   {
      ETG_TRACE_USR3(("ERROR: Message pointer is NULL"));
   }
   return bMsgConsumed;
}


bool TunerDataBindingHandler::bOnCannotStorePopupMsg(const Courier::Message& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::bOnCannotStorePopupMsg()"));
   bool bMsgConsumed = false; // Holds true if Model consumes the midw_ext_fi_tcl_DAB_ChannelInfo

   /* Update the HMI to indicate the current waveband */
   const CannotStorePopupMsg* copoSourceMsg =
      message_cast<const CannotStorePopupMsg*>(&corfoMsg);
   if (NULL != copoSourceMsg)
   {
      ETG_TRACE_USR3(("Cannot store preset popup display request"));
      //ReqToDisplayPopup(PID_APP_POPUP_CANNOT_STORE);
      bMsgConsumed = true;
   }
   else
   {
      ETG_TRACE_USR3(("ERROR: Message pointer is NULL"));
   }
   return bMsgConsumed;
}


bool TunerDataBindingHandler::updateASTStatus(bool status)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::UpdateASTStatus(%d)", status));
   bool bMsgConsumed = false;
   if (sm_gCurrentView == VIEW_DAB_AST)
   {
      (*oScanASTStatusData).mIsDAB_AST_Active = status;
      oScanASTStatusData.MarkItemModified(ItemKey::Scan_AST::IsDAB_AST_ActiveItem);
      oScanASTStatusData.SendUpdate(true);
      bMsgConsumed = true;
   }
   else if (sm_gCurrentView == VIEW_FM_AST)
   {
      (*oScanASTStatusData).mIsFM_AST_Active = status;
      oScanASTStatusData.MarkItemModified(ItemKey::Scan_AST::IsFM_AST_ActiveItem);
      oScanASTStatusData.SendUpdate(true);
      bMsgConsumed = true;
   }
   else if (sm_gCurrentView == VIEW_AM_AST)
   {
      (*oScanASTStatusData).mIsAM_AST_Active = status;
      oScanASTStatusData.MarkItemModified(ItemKey::Scan_AST::IsAM_AST_ActiveItem);
      oScanASTStatusData.SendUpdate(true);
      bMsgConsumed = true;
   }
   return bMsgConsumed;
}


bool TunerDataBindingHandler::bOnUpdateAutoStoreMsg(const Courier::Message& corfoMsg)
{
   //ETG_TRACE_USR3(("TunerDataBindingHandler::bOnUpdateAutoStoreMsg()"));
   //bool bMsgConsumed = false; // Holds true if Model consumes the message

   ///* Update the HMI to indicate the current waveband */
   //const UpdateAutoStoreMsg *copoSourceMsg =
   //	message_cast<const UpdateAutoStoreMsg*>(&corfoMsg);
   //if (NULL != copoSourceMsg)
   //{
   //	/**
   //	 * Do not consider current bank (sm_gSwitchState) if
   //	 * autostore status is 0. It means autostore is 0 for
   //	 * all AST banks
   //	 */
   //	if (copoSourceMsg->GetIsAutostoreRunning() == false)
   //	{
   //		(*oScanASTStatusData).mIsDAB_AST_Active = false;
   //		(*oScanASTStatusData).mIsFM_AST_Active = false;
   //		(*oScanASTStatusData).mIsAM_AST_Active = false;
   //	}
   //	ETG_TRACE_USR3(("Auto store status update(%d) in Bank %d", copoSourceMsg->GetIsAutostoreRunning(), sm_gSwitchState));
   //	if (sm_gSwitchState == EN_TUN_BANK_DAB_AST)
   //	{
   //		(*oScanASTStatusData).mIsDAB_AST_Active = copoSourceMsg->GetIsAutostoreRunning();
   //		oScanASTStatusData.MarkItemModified(ItemKey::Scan_AST::IsDAB_AST_ActiveItem);
   //		oScanASTStatusData.MarkAllItemsModified();
   //	}
   //	else if(sm_gSwitchState == EN_TUN_BANK_FM_AST)
   //	{
   //		ETG_TRACE_USR3(("TunerDataBindingHandler::FMASTACTIVE()"));
   //		(*oScanASTStatusData).mIsFM_AST_Active = copoSourceMsg->GetIsAutostoreRunning();
   //		oScanASTStatusData.MarkItemModified(ItemKey::Scan_AST::IsFM_AST_ActiveItem);
   //		oScanASTStatusData.SendUpdate(true);
   //		bMsgConsumed = true;
   //	}
   //	else if(sm_gSwitchState == EN_TUN_BANK_AM_AST)
   //	{
   //		(*oScanASTStatusData).mIsAM_AST_Active = copoSourceMsg->GetIsAutostoreRunning();
   //		oScanASTStatusData.MarkItemModified(ItemKey::Scan_AST::IsAM_AST_ActiveItem);
   //		oScanASTStatusData.SendUpdate(true);
   //		bMsgConsumed = true;
   //	}
   //	else
   //	{
   //		ETG_TRACE_USR3(("TUN_ERROR ::Auto store status update"));
   //	}
   //	oScanASTStatusData.SendUpdate(true);
   //	ETG_TRACE_USR3(("Auto store status update(%d) in Bank %d", copoSourceMsg->GetIsAutostoreRunning(), sm_gSwitchState));
   //	bMsgConsumed = true;
   //}
   //else
   //{
   //	ETG_TRACE_USR3(("ERROR: Message pointer is NULL"));
   //}
   //return bMsgConsumed;
   return true; ///added for coverity june 25
}


bool TunerDataBindingHandler::cobSelectedListItemInfo(const Courier::Message& corfoMsg)
{
   ETG_TRACE_USR4(("TunerDataBindingHandler :cobSelectedListItemInfo()"));
   bool bMsgConsumed = false;
   //   bool bUpdateListFlag = FALSE;
   //   BrowseListItemSelReq *poBrowseListItemSelReq = NULL;
   //   const SelectedListItemInfo *copoSelectedListItemInfo =
   //   message_cast<const SelectedListItemInfo *>(&corfoMsg);
   //   if(NULL != copoSelectedListItemInfo)
   //   {
   //      UInt8 u8ListID = copoSelectedListItemInfo->GetListIdentifier();
   //      UInt8 u8SelecteionIndex =
   //      copoSelectedListItemInfo->GetSelectedLineNumber();
   //      Candera::String szWidgetName =copoSelectedListItemInfo->GetWidgetName();
   //      ETG_TRACE_USR4(("List Identifier : %d ", u8ListID));
   //      ETG_TRACE_USR4(("Selection Index : %d ", u8SelecteionIndex));
   //      ETG_TRACE_USR4(("Interacted Widget : %s ", szWidgetName.GetCString()));
   //
   //      if ((SM_TUNER_LISTTYPE_FM_STATIONLIST == u8ListID)||(SM_DAB_LISTTYPE_SERVICELIST == u8ListID)||
   //              (SM_TUNER_LISTTYPE_AFLIST == u8ListID)||(SM_TUNER_LISTTYPE_TMCLIST == u8ListID))
   //      {
   //         poBrowseListItemSelReq = COURIER_MESSAGE_NEW(BrowseListItemSelReq)
   //         (u8ListID,u8SelecteionIndex);
   //         bUpdateListFlag = true;
   //      }
   ////      if (SM_DAB_LISTTYPE_SERVICELIST == u8ListID)
   ////      {
   ////         poBrowseListItemSelReq = COURIER_MESSAGE_NEW(BrowseListItemSelReq)
   ////         (SM_DAB_LISTTYPE_SERVICELIST,u8SelecteionIndex);
   ////         bUpdateListFlag = true;
   ////      }
   //      if(bUpdateListFlag)
   //      {
   //         if(poBrowseListItemSelReq)
   //         {
   //            poBrowseListItemSelReq->Post();
   //            bMsgConsumed = true;
   //            bUpdateListFlag = false;
   //         }
   //         else
   //         {
   //            ETG_TRACE_USR3(("cobSelectedListItemInfo - poBrowseListItemSelReq is NULL"));
   //         }
   //      }
   //   }
   return bMsgConsumed;
}


/*****************************************************************
| function implementation

|----------------------------------------------------------------*/

/************************************************************************
*FUNCTION: 		bOnUpdateTTMonitorTuner1
*RETURNVALUE: 	bool
*
*HISTORY:
*05.03.13 Karthik
*Rev 1.0 Initial Revision.
************************************************************************/
bool TunerDataBindingHandler::bOnUpdateTTMonitorTuner1(const Courier::Message& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::bOnUpdateTTMonitorTuner1()"));

   bool bMsgConsumed = false; // Holds true if Model consumes the message

   const TripleTunerMonitorDataSet1* poTripleTunerMonitorDataSet1 =
      message_cast<const TripleTunerMonitorDataSet1*>(&corfoMsg);

   if (NULL != poTripleTunerMonitorDataSet1)
   {
      //FM_MONITOR_TESTMODE screen values to be updated for every variant
      if (poTripleTunerMonitorDataSet1->GetTunerType() == TUNER2)
      {
         (void)bOnUpdateEngineeringMenuMsg(corfoMsg);
      }

      switch (poTripleTunerMonitorDataSet1->GetTunerType())
      {
         case TUNER1:
         {
            TTMonitorSet1DataUpdate(oTTMonitor1, poTripleTunerMonitorDataSet1);
         }
         break;
         case TUNER2:
            TTMonitorSet1DataUpdate(oTTMonitor2, poTripleTunerMonitorDataSet1);
            break;
         case TUNER3:
            TTMonitorSet1DataUpdate(oTTMonitor3, poTripleTunerMonitorDataSet1);
            break;
         default:
            ETG_TRACE_USR3(("TunerDataBindingHandler::bOnUpdateTTMonitorTuner1 invalid tuner type"));
            break;
      }
   }
   else
   {
      ETG_TRACE_USR3(("ERROR:bOnUpdateTTMonitorTuner1 poTunerEngineeringModeMsg NULL Pointer"));
   }

   bMsgConsumed = true;
   return bMsgConsumed;
}


/*****************************************************************
| function implementation

|----------------------------------------------------------------*/

/************************************************************************
*FUNCTION: 		bOnUpdateTTMonitorTuner2
*RETURNVALUE: 	bool
*
*HISTORY:
*05.03.13 Karthik
*Rev 1.0 Initial Revision.
************************************************************************/

bool TunerDataBindingHandler::bOnUpdateTTMonitorTuner2(const Courier::Message& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::bOnUpdateTTMonitorTuner1()"));

   bool bMsgConsumed = false; // Holds true if Model consumes the message

   const TripleTunerMonitorDataSet2* poTripleTunerMonitorDataSet2 =
      message_cast<const TripleTunerMonitorDataSet2*>(&corfoMsg);

   if (NULL != poTripleTunerMonitorDataSet2)
   {
      ETG_TRACE_USR3(("bOnUpdateTTMonitorTuner2 GetTunerType: %d", poTripleTunerMonitorDataSet2->GetTunerType()));
      switch (poTripleTunerMonitorDataSet2->GetTunerType())
      {
         case TUNER1:
            TTMonitorSet2DataUpdate(oTTMonitor1, poTripleTunerMonitorDataSet2);
            break;
         case TUNER2:
            TTMonitorSet2DataUpdate(oTTMonitor2, poTripleTunerMonitorDataSet2);
            break;
         case TUNER3:
            TTMonitorSet2DataUpdate(oTTMonitor3, poTripleTunerMonitorDataSet2);
            break;
         default:
            ETG_TRACE_USR3(("TunerDataBindingHandler::bOnUpdateTTMonitorTuner2() invalid tuner type"));
            break;
      }
   }
   else
   {
      ETG_TRACE_USR3(("ERROR:bOnUpdateTTMonitorTuner2 poTripleTunerMonitorDataSet2 NULL Pointer"));
   }
   bMsgConsumed = true;
   return bMsgConsumed;
}


/*****************************************************************
| function implementation

|----------------------------------------------------------------*/

/************************************************************************
*FUNCTION: 		bOnUpdateTTMonitorTunerSource_Band
*RETURNVALUE: 	bool
*
*HISTORY:
*05.03.13 Karthik
*Rev 1.0 Initial Revision.
************************************************************************/
bool TunerDataBindingHandler::bOnUpdateTTMonitorTunerSource_Band(const Courier::Message& corfoMsg)
{
   // to do
   ETG_TRACE_USR3(("TunerDataBindingHandler::bOnUpdateTTMonitorTunerSource_Band"));

   const TripleTunerMonitorDataSet3* poTripleTunerMonitorDataSet3 =
      message_cast<const TripleTunerMonitorDataSet3*>(&corfoMsg);

   tU8 u8ActiveTuner = poTripleTunerMonitorDataSet3->GetActiveTuner();
   tU8 u8ActiveBand = poTripleTunerMonitorDataSet3->GetActiveBand();
   tU8 u8TunerType = poTripleTunerMonitorDataSet3->GetTunerType();

   ETG_TRACE_USR3(("bOnUpdateTTMonitorTunerSource_Band GetActiveTuner: %d", poTripleTunerMonitorDataSet3->GetActiveTuner()));
   ETG_TRACE_USR3(("bOnUpdateTTMonitorTunerSource_Band GetActiveBand: %d", poTripleTunerMonitorDataSet3->GetActiveBand()));

   if (NULL != poTripleTunerMonitorDataSet3)
   {
      (*oTTTunerSource_Band).mTunerSource1 = false;
      (*oTTTunerSource_Band).mTunerSource2 = false;
      (*oTTTunerSource_Band).mTunerSource3 = false;
      (*oTTTunerSource_Band).mTunerSource4 = false;
      (*oTTTunerSource_Band).mBandEnable1 = false;
      (*oTTTunerSource_Band).mBandEnable2 = false;
      (*oTTTunerSource_Band).mBandEnable3 = false;
      (*oTTTunerSource_Band).mBandEnable4 = false;

      switch (u8ActiveTuner)
      {
         case ACTIVETUNER1:
         {
            (*oTTTunerSource_Band).mTunerSource1 = true;
            (*oTTTunerSource_Band).mBandEnable1 = true;
         }
         break;

         case ACTIVETUNER2:
         {
            (*oTTTunerSource_Band).mTunerSource2 = true;
            (*oTTTunerSource_Band).mBandEnable2 = true;
         }
         break;
         case ACTIVETUNER3:
         {
            (*oTTTunerSource_Band).mTunerSource3 = true;
            (*oTTTunerSource_Band).mBandEnable3 = true;
         }
         break;
         case ACTIVETUNER4:
         {
            (*oTTTunerSource_Band).mTunerSource4 = true;
            (*oTTTunerSource_Band).mBandEnable4 = true;
         }
         break;
         default:
            ETG_TRACE_USR3(("TunerDataBindingHandler::bOnUpdateTTMonitorTunerSource_Band() invalid Active Tuner Type type"));
            break;
      }
      switch (u8TunerType)
      {
         case 0:
         {
            if (u8ActiveBand == FMBAND2)
            {
               (*oTTTunerSource_Band).mBand1 = "FM";
            }
            else if (u8ActiveBand == AMBAND)
            {
               (*oTTTunerSource_Band).mBand1 = "AM";
            }
         }
         break;
         case 1:
         {
            if (u8ActiveBand == FMBAND2)
            {
               (*oTTTunerSource_Band).mBand2 = "FM";
            }
            else if (u8ActiveBand == AMBAND)
            {
               (*oTTTunerSource_Band).mBand2 = "AM";
            }
         }
         break;
         //        case 2:
         //        {
         //            if(u8ActiveBand==FMBAND2)
         //                   (*oTTTunerSource_Band).mBand3="FM";
         //            else if(u8ActiveBand==AMBAND)
         //                   (*oTTTunerSource_Band).mBand3="AM";
         //        }
         //            break;
         default:
            break;
      }

      /* Mark all items for updation */
      oTTTunerSource_Band.MarkAllItemsModified();

      bool bDataUpdated = oTTTunerSource_Band.SendUpdate(true);
      if (false == bDataUpdated)
      {
         ETG_TRACE_USR3(("ERROR: Data updation failed oEngineeringMenuData1"));
      }
   }
   else
   {
      ETG_TRACE_USR3(("ERROR: bOnUpdateTTMonitorTunerSource_Band NULL Pointer"));
   }

   return true;
}


/************************************************************************
*FUNCTION: 		bonUpdateDABServiceLinkingMsg
*RETURNVALUE: 	Void
*
*HISTORY:
*05.03.13 Karthik
*Rev 1.0 Initial Revision.
************************************************************************/
void TunerDataBindingHandler::UpdateDABServiceLinkingMsg(uint8 ServiceLinkingValue)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::UpdateDABServiceLinkingMsg"));

   if (ServiceLinkingValue == EN_DAB_LINKING_DAB)
   {
      (*oDABSetupMonitor).mServiceLinking = "DAB";
   }
   if (ServiceLinkingValue == EN_DAB_LINKING_DAB_FM)
   {
      (*oDABSetupMonitor).mServiceLinking = "DAB+FM";
   }
   if (ServiceLinkingValue == EN_DAB_LINKING_FM)
   {
      (*oDABSetupMonitor).mServiceLinking = "FM";
   }
   if (ServiceLinkingValue == EN_DAB_LINKING_OFF)
   {
      (*oDABSetupMonitor).mServiceLinking = "OFF";
   }
   ///* Mark all items for updation */
   //oDABSetupMonitor.MarkAllItemsModified();
   //bool bDataUpdated = oDABSetupMonitor.SendUpdate(true);

   oDABSetupMonitor.MarkItemModified(ItemKey::DAB_Setup_Screen::ServiceLinkingItem);
   oDABSetupMonitor.SendUpdate();

   //if (false == bDataUpdated)
   //{
   //	ETG_TRACE_USR3(("ERROR: Data updation failed UpdateDABServiceLinkingMsg"));
   //}
}


/************************************************************************
*FUNCTION: 		bOnUpdateDABConcealmentLevelMsg
*RETURNVALUE: 	bool
*
*HISTORY:
*05.03.13 Karthik
*Rev 1.0 Initial Revision.
************************************************************************/
bool TunerDataBindingHandler::bOnUpdateDABConcealmentLevelMsg(const Courier::Message& corfoMsg)
{
   //to do
   ETG_TRACE_USR3(("TunerDataBindingHandler::bOnUpdateDABConcealmentLevelMsg"));

   const DABConcealmentLevelMsg* poDABConcealmentLevelMsg =
      message_cast<const DABConcealmentLevelMsg*>(&corfoMsg);

   if (NULL != poDABConcealmentLevelMsg)
   {
      tChar szData[] = { 0, 0, 0, };
      ETG_TRACE_USR3(("bOnUpdateDABConcealmentLevelMsg CLevel: %d", poDABConcealmentLevelMsg->GetCLevel()));
      sprintf(szData, "%d", poDABConcealmentLevelMsg->GetCLevel());
      //(*oDABSetupMonitor).mConcealmentLevel = szData;
      /* Mark all items for updation */
      //oDABSetupMonitor.MarkAllItemsModified();

      bool bDataUpdated = oDABSetupMonitor.SendUpdate(true);
      if (false == bDataUpdated)
      {
         ETG_TRACE_USR3(("ERROR: Data updation failed bOnUpdateDABConcealmentLevelMsg"));
      }
   }
   else
   {
      ETG_TRACE_USR3(("ERROR: bOnUpdateDABConcealmentLevelMsg NULL Pointer"));
   }

   return true;
}


/************************************************************************
*FUNCTION: 		ScanActive
************************************************************************/

void TunerDataBindingHandler::ScanActive(bool scan_status)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::ScanActive(%d)", scan_status));
   (*oTunerScanInfoData).mIsActiveScan = scan_status;
   oTunerScanInfoData.MarkAllItemsModified();
   oTunerScanInfoData.SendUpdate(true);
   ETG_TRACE_USR3(("TunerDataBindingHandler::added for scan active check"));
}


/************************************************************************
*FUNCTION: 		Source_Selection_Icon_Update
************************************************************************/

void TunerDataBindingHandler::UpdateSourceIconData(tU8 Previous_View)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::UpdateSourceIconData(%d)", Previous_View));
   switch (Previous_View)
   {
      case VIEW_FM1:
      {
         (*oTunerSourceIconData).mFM1Active = true;
         (*oTunerSourceIconData).mAMActive = false;
         (*oTunerSourceIconData).mDABActive = false;
         (*oTunerSourceIconData).mSXMActive = false;
         (*oTunerSourceIconData).mWXActive = false;
         /* (*oTunerSourceIconData).mFM2Active = false;
          (*oTunerSourceIconData).mFMASTActive = false;
          (*oTunerSourceIconData).mAMASTActive = false;
          (*oTunerSourceIconData).mDABASTActive = false;
          SetSXMActive(false);*/

         sm_gSwitchState = EN_TUN_BANK_FM1;
         break;
      }
      case VIEW_FM2:
      {
         /*  (*oTunerSourceIconData).mFM2Active = true;

           (*oTunerSourceIconData).mFM1Active = false;
           (*oTunerSourceIconData).mFMASTActive = false;
           (*oTunerSourceIconData).mAMActive = false;
           (*oTunerSourceIconData).mAMASTActive = false;
           (*oTunerSourceIconData).mDABActive = false;
           (*oTunerSourceIconData).mDABASTActive = false;
           //SetSXMActive(false);
           (*oTunerSourceIconData).mSXMActive = false;
           sm_gSwitchState = EN_TUN_BANK_FM2;*/
         break;
      }
      case VIEW_FM_AST:
      {
         /* (*oTunerSourceIconData).mFMASTActive = true;

          (*oTunerSourceIconData).mFM1Active = false;
          (*oTunerSourceIconData).mFM2Active = false;
          (*oTunerSourceIconData).mAMActive = false;
          (*oTunerSourceIconData).mAMASTActive = false;
          (*oTunerSourceIconData).mDABActive = false;
          (*oTunerSourceIconData).mDABASTActive = false;
          sm_gSwitchState = EN_TUN_BANK_FM_AST;*/
         break;
      }
      case VIEW_AM:
      {
         (*oTunerSourceIconData).mAMActive = true;
         (*oTunerSourceIconData).mFM1Active = false;
         (*oTunerSourceIconData).mDABActive = false;
         (*oTunerSourceIconData).mSXMActive = false;
         (*oTunerSourceIconData).mWXActive = false;
         /* (*oTunerSourceIconData).mFM2Active = false;
          (*oTunerSourceIconData).mFMASTActive = false;
          (*oTunerSourceIconData).mAMASTActive = false;
          (*oTunerSourceIconData).mDABASTActive = false;
          SetSXMActive(false);*/

         sm_gSwitchState = EN_TUN_BANK_AM;
         break;
      }
      case VIEW_WEATHERBAND:
      {
         (*oTunerSourceIconData).mAMActive = false;
         (*oTunerSourceIconData).mWXActive = true;
         (*oTunerSourceIconData).mFM1Active = false;
         (*oTunerSourceIconData).mDABActive = false;
         (*oTunerSourceIconData).mSXMActive = false;
         /* (*oTunerSourceIconData).mFM2Active = false;
          (*oTunerSourceIconData).mFMASTActive = false;
          (*oTunerSourceIconData).mAMASTActive = false;
          (*oTunerSourceIconData).mDABASTActive = false;
          SetSXMActive(false);*/

         sm_gSwitchState = EN_TUN_BANK_WX;
         break;
      }
      case VIEW_AM_AST:
      {
         /* (*oTunerSourceIconData).mAMASTActive = true;

          (*oTunerSourceIconData).mFM1Active = false;
          (*oTunerSourceIconData).mFM2Active = false;
          (*oTunerSourceIconData).mFMASTActive = false;
          (*oTunerSourceIconData).mAMActive = false;
          (*oTunerSourceIconData).mDABActive = false;
          (*oTunerSourceIconData).mDABASTActive = false;
          sm_gSwitchState = EN_TUN_BANK_AM_AST;*/
         break;
      }
      case VIEW_DAB:
      {
         (*oTunerSourceIconData).mDABActive = true;
         (*oTunerSourceIconData).mFM1Active = false;
         (*oTunerSourceIconData).mAMActive = false;
         (*oTunerSourceIconData).mSXMActive = false;
         (*oTunerSourceIconData).mWXActive = false;
         /* (*oTunerSourceIconData).mFM2Active = false;
          (*oTunerSourceIconData).mFMASTActive = false;
          (*oTunerSourceIconData).mAMASTActive = false;
          (*oTunerSourceIconData).mDABASTActive = false;
          SetSXMActive(false);*/

         sm_gSwitchState = EN_TUN_BANK_DAB;
         break;
      }
      case VIEW_DAB_AST:
      {
         /*  (*oTunerSourceIconData).mDABASTActive = true;

           (*oTunerSourceIconData).mFM1Active = false;
           (*oTunerSourceIconData).mFM2Active = false;
           (*oTunerSourceIconData).mFMASTActive = false;
           (*oTunerSourceIconData).mAMActive = false;
           (*oTunerSourceIconData).mAMASTActive = false;
           (*oTunerSourceIconData).mDABActive = false;
           sm_gSwitchState = EN_TUN_BANK_DAB_AST;*/
         break;
      }
      default:
         break;
   }
   oTunerSourceIconData.MarkAllItemsModified();
   oTunerSourceIconData.SendUpdate(true);
   ETG_TRACE_USR3(("TunerDataBindingHandler::Updatedone"));
}


/************************************************************************
*FUNCTION: 		bOnUpdateDABTestModeActiveServiceInfo
*RETURNVALUE: 	bool
*
*HISTORY:
*05.03.13 Karthik
*Rev 1.0 Initial Revision.
************************************************************************/

bool TunerDataBindingHandler::bOnUpdateDABTestModeActiveServiceInfo(const Courier::Message& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::bOnUpdateDABTestModeActiveServiceInfo "));
   bool bMsgConsumed = false;
   const DABTestModeActiveServiceInfo* poDABTestModeActiveServiceInfo =
      message_cast<const DABTestModeActiveServiceInfo*>(&corfoMsg);

   if (NULL != poDABTestModeActiveServiceInfo)
   {
      bMsgConsumed = true;
      tChar szData[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
      (*oDABTestModePointer).mAudio_Mode = poDABTestModeActiveServiceInfo->GetSAudioCodec();
      (*oDABTestModePointer).mService_Lbl = poDABTestModeActiveServiceInfo->GetServiceLabel();

      sprintf(szData, "%d", poDABTestModeActiveServiceInfo->GetU16BitRate());
      (*oDABTestModePointer).mAud_Data_Bit_Rate = szData;
      memset(szData, 0, sizeof(szData));

      sprintf(szData, "%d", poDABTestModeActiveServiceInfo->GetU8DRC());
      (*oDABTestModeActivertunerInfo).mDRC = szData;
      memset(szData, 0, sizeof(szData));

      tU8 PS = poDABTestModeActiveServiceInfo->GetU8PS();
      if (PS == ONE)
      {
         (*oDABTestModeActivertunerInfo).mPS = "1";
      }
      else
      {
         (*oDABTestModeActivertunerInfo).mPS = "0";
      }

      sprintf(szData, "%x", poDABTestModeActiveServiceInfo->GetU32ServId());
      (*oDABTestModePointer).mService_ID = szData;
      memset(szData, 0, sizeof(szData));

      sprintf(szData, "%s/%s", poDABTestModeActiveServiceInfo->GetSAudioMode().GetCString(), poDABTestModeActiveServiceInfo->GetSAdvancedAudioCodec().GetCString());
      (*oDABTestModeActivertunerInfo).mSM_ACC = szData;
      memset(szData, 0, sizeof(szData));

      sprintf(szData, "%d", poDABTestModeActiveServiceInfo->GetU8SampleRate());
      (*oDABTestModeActivertunerInfo).mSR = szData;
      memset(szData, 0, sizeof(szData));

      sprintf(szData, "%d", poDABTestModeActiveServiceInfo->GetB16AvailableAnnoTypesMask());
      (*oDABTestModeActivertunerInfo).mTP = szData;
      memset(szData, 0, sizeof(szData));

      oDABTestModePointer.MarkAllItemsModified();
      oDABTestModeActivertunerInfo.MarkAllItemsModified();

      bool bDataUpdated = oDABTestModePointer.SendUpdate(true);
      if (false == bDataUpdated)
      {
         ETG_TRACE_USR3(("ERROR: Data updation failed oDABTestModePointer"));
      }

      bDataUpdated = oDABTestModeActivertunerInfo.SendUpdate(true);
      if (false == bDataUpdated)
      {
         ETG_TRACE_USR3(("ERROR: Data updation failed oDABTestModeActivertunerInfo"));
      }
   }
   else
   {
      ETG_TRACE_USR3(("ERROR: bOnUpdateTTMonitorTunerSource_Band NULL Pointer"));
   }

   return bMsgConsumed;
}


/************************************************************************
*FUNCTION: 		bOnUpdateDABTestModeActiveEnsembleInfo
*RETURNVALUE: 	bool
*
*HISTORY:
*05.03.13 Karthik
*Rev 1.0 Initial Revision.
************************************************************************/
bool TunerDataBindingHandler::bOnUpdateDABTestModeActiveEnsembleInfo(const Courier::Message& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::bOnUpdateDABTestModeActiveEnsembleInfo"));
   bool bMsgConsumed = false;
   const DABTestModeActiveEnsembleInfo* poDABTestModeActiveEnsembleInfo =
      message_cast<const DABTestModeActiveEnsembleInfo*>(&corfoMsg);

   if (NULL != poDABTestModeActiveEnsembleInfo)
   {
      bMsgConsumed = true;
      tChar szData[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
      (*oDABTestModePointer).mEnsemble_Lbl = poDABTestModeActiveEnsembleInfo->GetEnsembleLabel();

      sprintf(szData, "%x", poDABTestModeActiveEnsembleInfo->GetEnsembleID());
      (*oDABTestModePointer).mEnsemble_ID = szData;
      memset(szData, 0, sizeof(szData));

      sprintf(szData, "%d/%d", poDABTestModeActiveEnsembleInfo->GetU8NoOfAudServiceComp(), poDABTestModeActiveEnsembleInfo->GetU8NoOfDataServiceComp());
      (*oDABTestModeActiveEnsembleInfo).mNoOfAud_DataServiceComp = szData;
      memset(szData, 0, sizeof(szData));

      sprintf(szData, "%d/%d", poDABTestModeActiveEnsembleInfo->GetU8NoOfAudService(), poDABTestModeActiveEnsembleInfo->GetU8NoOfDataService());
      (*oDABTestModeActiveEnsembleInfo).mNoOfAud_DataService = szData;
      memset(szData, 0, sizeof(szData));

      sprintf(szData, "%d", poDABTestModeActiveEnsembleInfo->GetU8NoOfEnsembles());
      (*oDABTestModeActiveEnsembleInfo).mNoOfEnsembles = szData;
      memset(szData, 0, sizeof(szData));

      oDABTestModePointer.MarkAllItemsModified();
      oDABTestModeActiveEnsembleInfo.MarkAllItemsModified();

      bool bDataUpdated = oDABTestModePointer.SendUpdate(true);
      if (false == bDataUpdated)
      {
         ETG_TRACE_USR3(("ERROR: Data updation failed oDABTestModePointer"));
      }

      bDataUpdated = oDABTestModeActiveEnsembleInfo.SendUpdate(true);
      if (false == bDataUpdated)
      {
         ETG_TRACE_USR3(("ERROR: Data updation failed oDABTestModeActivertunerInfo"));
      }
   }
   else
   {
      ETG_TRACE_USR3(("ERROR: bOnUpdateDABTestModeActiveEnsembleInfo NULL Pointer"));
   }

   return bMsgConsumed;
}


/*****************************************************************
| function implementation

|----------------------------------------------------------------*/

/************************************************************************
*FUNCTION: 		bOnUpdateDABTestModeTunerStatusInfo
*RETURNVALUE: 	bool
*
*HISTORY:
*05.03.13 Karthik
*Rev 1.0 Initial Revision.
************************************************************************/
bool TunerDataBindingHandler::bOnUpdateDABTestModeTunerStatusInfo(const Courier::Message& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::bOnUpdateDABTestModeTunerStatusInfo"));
   bool bMsgConsumed = false;
   const DABTestModeTunerStatusInfo* poDABTestModeTunerStatusInfo =
      message_cast<const DABTestModeTunerStatusInfo*>(&corfoMsg);

   tU32 u32EnsembleFreg;

   if (NULL != poDABTestModeTunerStatusInfo)
   {
      bMsgConsumed = true;
      tChar szData[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
      (*oDABTestModePointer).mchannel_Num = poDABTestModeTunerStatusInfo->GetChannel_Num();
      (*oDABTestModePointer).mFreq_Table = poDABTestModeTunerStatusInfo->GetFrequency_Table();

      sprintf(szData, "%d", poDABTestModeTunerStatusInfo->GetMute());
      (*oDABTestModePointer).mMute_flag = szData;
      memset(szData, 0, sizeof(szData));
      sprintf(szData, "%d", poDABTestModeTunerStatusInfo->GetAudio_Quality());
      (*oDABTestModePointer).mQA = szData;
      memset(szData, 0, sizeof(szData));
      sprintf(szData, "%d", poDABTestModeTunerStatusInfo->GetSignal_Quality());
      (*oDABTestModePointer).mQS = szData;
      memset(szData, 0, sizeof(szData));
      sprintf(szData, "%d", poDABTestModeTunerStatusInfo->GetSync());
      (*oDABTestModePointer).mSync_flag = szData;
      memset(szData, 0, sizeof(szData));

      u32EnsembleFreg = poDABTestModeTunerStatusInfo->GetEnsemble_Frequency();
      if (TUNER_ZERO_VALUE == (u32EnsembleFreg % M_DIVIDE_BY_HUNDRED))
      {
         sprintf(szData, "%.1f", ((tF32)u32EnsembleFreg / M_CONVERSION_KHZ_TO_MHZ));
      }
      else
      {
         sprintf(szData, "%.2f", ((tF32)u32EnsembleFreg / M_CONVERSION_KHZ_TO_MHZ));
      }
      (*oDABTestModePointer).mEnsemble_freq = szData;
      memset(szData, 0, sizeof(szData));

      oDABTestModePointer.MarkAllItemsModified();

      bool bDataUpdated = oDABTestModePointer.SendUpdate(true);
      if (false == bDataUpdated)
      {
         ETG_TRACE_USR3(("ERROR: Data updation failed oDABTestModePointer"));
      }
   }
   else
   {
      ETG_TRACE_USR3(("ERROR: bOnUpdateDABTestModeTunerStatusInfo NULL Pointer"));
   }

   return bMsgConsumed;
}


/*****************************************************************
| function implementation

|----------------------------------------------------------------*/

/************************************************************************
*FUNCTION: 		bOnUpdateDABTestModeGTestmodeInfo
*RETURNVALUE: 	bool
*
*HISTORY:
*05.03.13 Karthik
*Rev 1.0 Initial Revision.
************************************************************************/
bool TunerDataBindingHandler::bOnUpdateDABTestModeGTestmodeInfo(const Courier::Message& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::bOnUpdateDABTestModeGTestmodeInfo"));
   bool bMsgConsumed = false;
   const DABTestModeGTestmodeInfo* poDABTestModeGTestmodeInfo =
      message_cast<const DABTestModeGTestmodeInfo*>(&corfoMsg);

   if (NULL != poDABTestModeGTestmodeInfo)
   {
      bMsgConsumed = true;
      tChar szData[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

      sprintf(szData, "%d", poDABTestModeGTestmodeInfo->GetGFM());
      (*oDABTestModeGTestmodeInfo).mFM = szData;
      memset(szData, 0, sizeof(szData));
      sprintf(szData, "%d", poDABTestModeGTestmodeInfo->GetGPI());
      (*oDABTestModeGTestmodeInfo).mPI = szData;
      memset(szData, 0, sizeof(szData));
      sprintf(szData, "%d", poDABTestModeGTestmodeInfo->GetGQFM());
      (*oDABTestModeGTestmodeInfo).mQFM = szData;
      memset(szData, 0, sizeof(szData));
      sprintf(szData, "%d", poDABTestModeGTestmodeInfo->GetGPL());
      (*oDABTestModeGTestmodeInfo).mPL = szData;
      memset(szData, 0, sizeof(szData));
      sprintf(szData, "%d", poDABTestModeGTestmodeInfo->GetGFIC());
      (*oDABTestModePointer).mFIC = szData;
      memset(szData, 0, sizeof(szData));
      sprintf(szData, "%d", poDABTestModeGTestmodeInfo->GetGField_Strength());
      (*oDABTestModePointer).mField_Strength = szData;
      memset(szData, 0, sizeof(szData));
      sprintf(szData, "%d", poDABTestModeGTestmodeInfo->GetGMSC());
      (*oDABTestModePointer).mMSC = szData;
      memset(szData, 0, sizeof(szData));
      sprintf(szData, "%d", poDABTestModeGTestmodeInfo->GetGRSFEC());
      (*oDABTestModePointer).mRS_FEC = szData;
      memset(szData, 0, sizeof(szData));

      sprintf(szData, "%d", poDABTestModeGTestmodeInfo->GetGNET());
      (*oDABTestModePointer).mNET = szData;
      memset(szData, 0, sizeof(szData));
      oDABTestModePointer.MarkAllItemsModified();
      oDABTestModeGTestmodeInfo.MarkAllItemsModified();

      bool bDataUpdated = oDABTestModePointer.SendUpdate(true);
      if (false == bDataUpdated)
      {
         ETG_TRACE_USR3(("ERROR: Data updation failed oDABTestModePointer"));
      }
      bDataUpdated = oDABTestModeGTestmodeInfo.SendUpdate(true);
      if (false == bDataUpdated)
      {
         ETG_TRACE_USR3(("ERROR: Data updation failed oDABTestModePointer"));
      }
   }
   else
   {
      ETG_TRACE_USR3(("ERROR: bOnUpdateDABTestModeGTestmodeInfo NULL Pointer"));
   }
   return bMsgConsumed;
}


/*****************************************************************
| function implementation

|----------------------------------------------------------------*/

/************************************************************************
*FUNCTION: 		bOnUpdateDABTestModeBGTestmodeInfo
*RETURNVALUE: 	bool
*
*HISTORY:
*05.03.13 Karthik
*Rev 1.0 Initial Revision.
************************************************************************/
bool TunerDataBindingHandler::bOnUpdateDABTestModeBGTestmodeInfo(const Courier::Message& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::bOnUpdateDABTestModeBGTestmodeInfo"));
   const DABTestModeBGTestmodeInfo* poDABTestModeBGTestmodeInfo =
      message_cast<const DABTestModeBGTestmodeInfo*>(&corfoMsg);
   bool bMsgConsumed = false;
   tU32 u32BGEnsembleFreg;
   if (NULL != poDABTestModeBGTestmodeInfo)
   {
      bMsgConsumed = true;
      tChar szData[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
      u32BGEnsembleFreg = poDABTestModeBGTestmodeInfo->GetBGEensembleFreg();
      (*oDABTestModePointer).mBGMode = poDABTestModeBGTestmodeInfo->GetBGMode();
      if (TUNER_ZERO_VALUE == (u32BGEnsembleFreg % M_DIVIDE_BY_HUNDRED))
      {
         sprintf(szData, "%.1f", ((tF32)u32BGEnsembleFreg / M_CONVERSION_KHZ_TO_MHZ));
      }
      else
      {
         sprintf(szData, "%.2f", ((tF32)u32BGEnsembleFreg / M_CONVERSION_KHZ_TO_MHZ));
      }

      // sprintf(szData, "%d", poDABTestModeBGTestmodeInfo->GetBGEensembleFreg());
      (*oDABTestModePointer).mBg_tuner_ens_freq = szData;
      memset(szData, 0, sizeof(szData));
      sprintf(szData, "%d", poDABTestModeBGTestmodeInfo->GetBGEensembleID());
      (*oDABTestModePointer).mBg_Ensemble_ID = szData;
      memset(szData, 0, sizeof(szData));
      sprintf(szData, "%d", poDABTestModeBGTestmodeInfo->GetBGFIC());
      (*oDABTestModePointer).mBg_FIC = szData;
      memset(szData, 0, sizeof(szData));
      sprintf(szData, "%d", poDABTestModeBGTestmodeInfo->GetBGSyncFlag());
      (*oDABTestModePointer).mBGSync_flag = szData;
      memset(szData, 0, sizeof(szData));

      oDABTestModePointer.MarkAllItemsModified();

      bool bDataUpdated = oDABTestModePointer.SendUpdate(true);
      if (false == bDataUpdated)
      {
         ETG_TRACE_USR3(("ERROR: Data updation failed oDABTestModePointer"));
      }
   }
   else
   {
      ETG_TRACE_USR3(("ERROR: bOnUpdateDABTestModeBGTestmodeInfo NULL Pointer"));
   }
   return bMsgConsumed;
}


void TunerDataBindingHandler::updateFmTAStnName(const std::string& serviceLabel, const uint8 currentAnnoType)
{
   /*Check if there is any change in the content before updating UI */
   std::string secureString = TunerUtility::getSecureString((*oTAInfo).mTAPSName);
   std::string sServiceLabel = serviceLabel;
   ETG_TRACE_USR3(("updateFmTAStnName of sServiceLabel %s", sServiceLabel));
   if (secureString.compare(sServiceLabel) != 0)
   {
      ETG_TRACE_USR3(("annoText OF updateFmTAStnName is sent to GUI"));
      if (sServiceLabel != "")
      {
         std::string sServiceLabelstringTA = "(" + sServiceLabel + ")";
         (*oTAInfo).mTAPSName = sServiceLabelstringTA.c_str();
      }
      else
      {
         std::string sServiceLabelstringTA = sServiceLabel;
         (*oTAInfo).mTAPSName = sServiceLabelstringTA.c_str();
      }
      oTAInfo.MarkItemModified(ItemKey::TAInfo::TAPSNameItem);
      oTAInfo.SendUpdate();
   }
}


void TunerDataBindingHandler::updateDabOtherAnnouncementsName(const std::string& serviceLabel)
{
   std::string secureString = TunerUtility::getSecureString((*oTAInfo).mTAPSName);
   std::string sServiceLabel = serviceLabel;
   ETG_TRACE_USR3(("updateDabOtherAnnouncementsName of sServiceLabel %s", sServiceLabel));
   if (secureString.compare(sServiceLabel) != 0)
   {
      ETG_TRACE_USR3(("annoText updateDabOtherAnnouncementsName sent to GUI"));
      if (sServiceLabel != "")
      {
         std::string sServiceLabelstringDabTA = "(" + sServiceLabel + ")";
         (*oTAInfo).mTAPSName = sServiceLabelstringDabTA.c_str();
      }
      else
      {
         std::string sServiceLabelstringDabTA = sServiceLabel ;
         (*oTAInfo).mTAPSName = sServiceLabelstringDabTA.c_str();
      }
      oTAInfo.MarkItemModified(ItemKey::TAInfo::TAPSNameItem);
      oTAInfo.SendUpdate();
   }
}


/*****************************************************************
| function implementation

|----------------------------------------------------------------*/

/************************************************************************
*FUNCTION: 		bOnUpdateDABTestModeAnnoStatus
*RETURNVALUE: 	bool
*
*HISTORY:
*05.03.13 Karthik
*Rev 1.0 Initial Revision.
************************************************************************/
bool TunerDataBindingHandler::bOnUpdateDABTestModeAnnoStatus(const Courier::Message& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::bOnUpdateDABTestModeAnnoStatus"));
   const DABTestModeAnnoStatus* poDABTestModeAnnoStatusInfo =
      message_cast<const DABTestModeAnnoStatus*>(&corfoMsg);
   bool bMsgConsumed = false;
   if (NULL != poDABTestModeAnnoStatusInfo)
   {
      bMsgConsumed = true;
      tChar szData[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

      sprintf(szData, "%d", poDABTestModeAnnoStatusInfo->GetASW());
      (*oDABTestModeAnnoStatusInfo).mASW = szData;
      memset(szData, 0, sizeof(szData));
      sprintf(szData, "%d", poDABTestModeAnnoStatusInfo->GetASU());
      (*oDABTestModeAnnoStatusInfo).mASU = szData;
      memset(szData, 0, sizeof(szData));
      m_sDABTASourcePSName = poDABTestModeAnnoStatusInfo->GetAnnoServiceLabel();

      oDABTestModeAnnoStatusInfo.MarkAllItemsModified();

      bool bDataUpdated = oDABTestModeAnnoStatusInfo.SendUpdate(true);
      if (false == bDataUpdated)
      {
         ETG_TRACE_USR3(("ERROR: Data updation failed oDABTestModePointer"));
      }
      vUpdateTASourceProvider();
      vUpdatePTY31SourceProvider();
   }
   else
   {
      ETG_TRACE_USR3(("ERROR: bOnUpdateDABTestModeAnnoStatus NULL Pointer"));
   }
   return bMsgConsumed;
}


/*****************************************************************
| function implementation

|----------------------------------------------------------------*/

/************************************************************************
*FUNCTION: 		bOnUpdateDABGTestModeActiveComp
*RETURNVALUE: 	bool
*
*HISTORY:
*05.03.13 Karthik
*Rev 1.0 Initial Revision.
************************************************************************/
bool TunerDataBindingHandler::bOnUpdateDABGTestModeActiveComp(const Courier::Message& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::bOnUpdateDABGTestModeActiveComp"));
   const DABTestModeActiveCompInfo* poDABTestModeActiveCompInfo =
      message_cast<const DABTestModeActiveCompInfo*>(&corfoMsg);
   bool bMsgConsumed = false;
   if (NULL != poDABTestModeActiveCompInfo)
   {
      bMsgConsumed = true;
      tChar szData[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

      sprintf(szData, "%d", poDABTestModeActiveCompInfo->GetSC());
      (*oDABTestModeActiveCompInfo).mSC = szData;
      memset(szData, 0, sizeof(szData));
      sprintf(szData, "%d", poDABTestModeActiveCompInfo->GetSCType());
      (*oDABTestModeActiveCompInfo).mSCType = szData;
      memset(szData, 0, sizeof(szData));

      oDABTestModeActiveCompInfo.MarkAllItemsModified();

      bool bDataUpdated = oDABTestModeActiveCompInfo.SendUpdate(true);
      if (false == bDataUpdated)
      {
         ETG_TRACE_USR3(("ERROR: Data updation failed oDABTestModePointer"));
      }
   }
   else
   {
      ETG_TRACE_USR3(("ERROR: bOnUpdateDABGTestModeActiveComp NULL Pointer"));
   }
   return bMsgConsumed;
}


/*****************************************************************
| function implementation

|----------------------------------------------------------------*/

/************************************************************************
*FUNCTION: 		bOnUpdateDABTestModeGAvailabilty
*RETURNVALUE: 	bool
*
*HISTORY:
*05.03.13 Karthik
*Rev 1.0 Initial Revision.
************************************************************************/
bool TunerDataBindingHandler::bOnUpdateDABTestModeGAvailabilty(const Courier::Message& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::bOnUpdateDABTestModeGAvailabilty"));
   const DABTestModeGAvailabilty* poDABTestModeGAvailabilty =
      message_cast<const DABTestModeGAvailabilty*>(&corfoMsg);
   bool bMsgConsumed = false;
   if (NULL != poDABTestModeGAvailabilty)
   {
      bMsgConsumed = true;
      tChar szData[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

      sprintf(szData, "%d", poDABTestModeGAvailabilty->GetTA());
      (*oDABTestModeGAvailabiltyInfo).mTA = szData;
      memset(szData, 0, sizeof(szData));

      oDABTestModeGAvailabiltyInfo.MarkAllItemsModified();

      bool bDataUpdated = oDABTestModeGAvailabiltyInfo.SendUpdate(true);
      if (false == bDataUpdated)
      {
         ETG_TRACE_USR3(("ERROR: Data updation failed oDABTestModePointer"));
      }
   }
   else
   {
      ETG_TRACE_USR3(("ERROR: bOnUpdateDABTestModeGAvailabilty NULL Pointer"));
   }
   return bMsgConsumed;
}


/*****************************************************************
| function implementation

|----------------------------------------------------------------*/

/************************************************************************
*FUNCTION: 		bOnUpdateDABTestModeServiceFollowingStatus
*RETURNVALUE: 	bool
*
*HISTORY:
*05.03.13 Karthik
*Rev 1.0 Initial Revision.
************************************************************************/
bool TunerDataBindingHandler::bOnUpdateDABTestModeServiceFollowingStatus(const Courier::Message& corfoMsg)
{
   bool bMsgConsumed = false;
   ETG_TRACE_USR3(("TunerDataBindingHandler::bOnUpdateDABTestModeServiceFollowingStatus"));
   const DABTestModeServiceFollowingStatus* poDABTestModeServiceFollowingStatus =
      message_cast<const DABTestModeServiceFollowingStatus*>(&corfoMsg);

   if (NULL != poDABTestModeServiceFollowingStatus)
   {
      bMsgConsumed = true;
      tChar szData[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

      sprintf(szData, "%d", poDABTestModeServiceFollowingStatus->GetTSU());
      (*oDABTestServiceFollowingInfo).mTSU = szData;
      memset(szData, 0, sizeof(szData));

      oDABTestServiceFollowingInfo.MarkAllItemsModified();

      bool bDataUpdated = oDABTestServiceFollowingInfo.SendUpdate(true);
      if (false == bDataUpdated)
      {
         ETG_TRACE_USR3(("ERROR: Data updation failed oDABTestModePointer"));
      }
   }
   else
   {
      ETG_TRACE_USR3(("ERROR: bOnUpdateDABTestModeServiceFollowingStatus NULL Pointer"));
   }
   return bMsgConsumed;
}


void TunerDataBindingHandler::updateRadioText(std::string& radiotxt)
{
   _radioMainText = radiotxt;

   ETG_TRACE_USR3(("TunerDataBindingHandler: updateRadioText %s", radiotxt.c_str()));

   (*oTunerInfoData).mRadioText = radiotxt.c_str();
   oTunerInfoData.MarkItemModified(ItemKey::StationInfoSource::RadioTextItem);
   oTunerInfoData.SendUpdate(true);
}


/*****************************************************************
| function implementation

|----------------------------------------------------------------*/

/************************************************************************
*FUNCTION: 		bOnUpdateDABTestModeTASource
*RETURNVALUE: 	bool
*
*HISTORY:
*05.03.13 Karthik
*Rev 1.0 Initial Revision.
************************************************************************/
bool TunerDataBindingHandler::bOnUpdateDABTestModeTASource(const Courier::Message& corfoMsg)
{
   bool bMsgConsumed = false;
   ETG_TRACE_USR3(("TunerDataBindingHandler::bOnUpdateDABTestModeTASource"));
   const DABTestModeTASource* poDABTestModeTASourceInfo =
      message_cast<const DABTestModeTASource*>(&corfoMsg);

   if (NULL != poDABTestModeTASourceInfo)
   {
      bMsgConsumed = true;
      tU8 u8TASource = poDABTestModeTASourceInfo->GetTASource();
      if (u8TASource == EN_TA_SOURCE_FM)
      {
         (*oDABTestModePointer).mTASrc = "FM";
      }
      else if (u8TASource == EN_TA_SOURCE_DAB)
      {
         (*oDABTestModePointer).mTASrc = "DAB";
      }
      else if (u8TASource == EN_TA_SOURCE_NODATA)
      {
         (*oDABTestModePointer).mTASrc = "--";
      }
      else if (u8TASource == EN_TA_SOURCE_NA)
      {
         (*oDABTestModePointer).mTASrc = "N.A";
      }
      else
      {
         ETG_TRACE_USR3(("ERROR: Invalid TA Source Value "));
      }
      u8TASource = poDABTestModeTASourceInfo->GetDAB_DAB();
      if (u8TASource == 1)
      {
         (*oDABTestModeGAvailabiltyInfo).mDABDAB = "ON";
      }
      else
      {
         (*oDABTestModeGAvailabiltyInfo).mDABDAB = "OFF";
      }

      oDABTestModePointer.MarkAllItemsModified();

      bool bDataUpdated = oDABTestModePointer.SendUpdate(true);
      if (false == bDataUpdated)
      {
         ETG_TRACE_USR3(("ERROR: Data updation failed oDABTestModePointer"));
      }
      oDABTestModeGAvailabiltyInfo.MarkAllItemsModified();

      bDataUpdated = oDABTestModeGAvailabiltyInfo.SendUpdate(true);
      if (false == bDataUpdated)
      {
         ETG_TRACE_USR3(("ERROR: Data updation failed oDABTestModePointer"));
      }
   }
   else
   {
      ETG_TRACE_USR3(("ERROR: bOnUpdateDABTestModeTASource NULL Pointer"));
   }
   return true;
}


/************************************************************************
*FUNCTION: 		bOnbOnTmcStnInfoData
*RETURNVALUE: 	bool
*
*HISTORY:
*05.03.13 Karthik
*Rev 1.0 Initial Revision.
************************************************************************/
bool TunerDataBindingHandler::bOnTmcStnInfoData(const Courier::Message& corfoMsg)
{
   bool bMsgConsumed = false;
   ETG_TRACE_USR3(("TunerDataBindingHandler::bOnbOnTmcStnInfoData"));
   const TmcStnInfoData* poTmcStnInfoData =
      message_cast<const TmcStnInfoData*>(&corfoMsg);

   if (NULL != poTmcStnInfoData)
   {
      bMsgConsumed = true;
      tChar szData[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

      (*oTmcStationInformationData).mPS = poTmcStnInfoData->GetPSData();
      ETG_TRACE_USR3(("TunerDataBindingHandler::bOnTmcStnInfoData GetPSData: %s", poTmcStnInfoData->GetPSData().GetCString()));
      sprintf(szData, "%d", poTmcStnInfoData->GetPIData());
      ETG_TRACE_USR3(("TunerDataBindingHandler::bOnTmcStnInfoData GetPIData: %d", poTmcStnInfoData->GetPIData()));
      (*oTmcStationInformationData).mPI = szData;
      sprintf(szData, "%d", poTmcStnInfoData->GetCCData());
      ETG_TRACE_USR3(("TunerDataBindingHandler::bOnTmcStnInfoData GetCCData: %d", poTmcStnInfoData->GetCCData()));
      (*oTmcStationInformationData).mCC = szData;
      sprintf(szData, "%d", poTmcStnInfoData->GetSIDData());
      ETG_TRACE_USR3(("TunerDataBindingHandler::bOnTmcStnInfoData GetSIDData: %d", poTmcStnInfoData->GetSIDData()));
      (*oTmcStationInformationData).mSID = szData;
      sprintf(szData, "%d", poTmcStnInfoData->GetLTNData());
      ETG_TRACE_USR3(("TunerDataBindingHandler::bOnTmcStnInfoData GetLTNData: %d", poTmcStnInfoData->GetLTNData()));
      (*oTmcStationInformationData).mLTN = szData;
      ETG_TRACE_USR3(("TunerDataBindingHandler::bOnTmcStnInfoData GetQData: %d", poTmcStnInfoData->GetQData()));
      if (poTmcStnInfoData->GetQData() <= 37)
      {
         (*oTmcStationInformationData).mQuality = "POOR";
      }
      else if (poTmcStnInfoData->GetQData() <= 76)
      {
         (*oTmcStationInformationData).mQuality = "MONO";
      }
      else
      {
         (*oTmcStationInformationData).mQuality = "STEREO";
      }

      oTmcStationInformationData.MarkAllItemsModified();
      bool bDataUpdated = oTmcStationInformationData.SendUpdate(true);
      if (false == bDataUpdated)
      {
         ETG_TRACE_USR3(("ERROR: Data updation failed poTmcStnInfoData"));
      }
   }
   else
   {
      ETG_TRACE_USR3(("ERROR: bOnbOnTmcStnInfoData NULL Pointer"));
   }
   return bMsgConsumed;
}


/************************************************************************
*FUNCTION			:updateChannelData -------->wst4kor
*DESCRIPTION		:Function called to update ChannelData
*RETURNVALUE		:void
************************************************************************/
bool TunerDataBindingHandler::bOnChannelInfoMsg(const Courier::Message& corfoMsg)
{
   tChar szStatusTextHD[3] = { 0 };
   ETG_TRACE_USR3(("TunerDataBindingHandler::bOnChannelInfoMsg()"));
   bool bMsgConsumed = false; // Holds true if Model consumes the message

   /* Update the HMI to indicate the current waveband */
   const ChannelInfoMsg* copoSourceMsg =
      message_cast<const ChannelInfoMsg*>(&corfoMsg);
   if (NULL != copoSourceMsg)
   {
      // (*oDABChannelInfo).mTpStatus = (Candera::String)(copoSourceMsg->GetTpAvailability() ? "%1" : "");
      HdAdvisory enHDStatus = (HdAdvisory)copoSourceMsg->GetAdvisoryText();
      if (enHDStatus > EN_HD_NO_CHANNEL)
      {
         enHDStatus = EN_HD_ALL_OK;
      }

      if (enHDStatus > EN_HD_ALL_OK)
      {
         sprintf(szStatusTextHD, "%%%d", enHDStatus);
      }
      (*oChannelInfo).mAdvisoryText = (Candera::String)szStatusTextHD;
      oChannelInfo.MarkAllItemsModified();
      oChannelInfo.SendUpdate(true);
      bMsgConsumed = true;
   }
   else
   {
      ETG_TRACE_USR3(("ERROR: Message pointer is NULL"));
   }
   return bMsgConsumed;
}


/************************************************************************
*FUNCTION			:updateDabChannelData
*DESCRIPTION		:Function called to update DabChannelData
*RETURNVALUE		:void
************************************************************************/
void TunerDataBindingHandler::updateDabChannelData(std::string psName, std::string shortPsName, std::string activePresetNumText, uint8 presetIndicatorIndex)
{
   int8 bandinfo = PresetListHandling::poGetInstance()->getbandInfo();
   ETG_TRACE_USR3(("TunerDataBindingHandler::bandinfo %d", bandinfo));
   if (bandinfo == 1)
   {
      ETG_TRACE_USR3(("TunerDataBindingHandler::updateDabChannelData"));
      ETG_TRACE_USR3(("TunerDataBindingHandler::ServiceName %s", psName.c_str()));
      (*oDABChannelInfo).mServiceLabel = (Candera::String)psName.c_str();
      (*oHomeScreenInfo).mHomeScreen_ActiveIconIndex = 2;
      (*oHomeScreenInfo).mHomeScreen_DABServiceName = (Candera::String)psName.c_str();;
      oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeScreen_ActiveIconIndexItem);
      oHomeScreenInfo.MarkItemModified(ItemKey::HomeScreenInfo::HomeScreen_DABServiceNameItem);
      oDABChannelInfo.MarkItemModified(ItemKey::DABChannelInfo::ServiceLabelItem);
      oDABChannelInfo.SendUpdate(true);
      oHomeScreenInfo.SendUpdate(true);
   }
}


/************************************************************************
*DESCRIPTION		:Function called to set DAB advisories,RadioText and switch index
*RETURNVALUE		: Void
************************************************************************/
void TunerDataBindingHandler::setDLSelements(std::string dabAdv, std::string radioText, uint8 switchIndex, uint8 enDabStatus)
{
   tChar szStatusText[3] = { 0 };
   sprintf(szStatusText, "%%%d", enDabStatus);
   ETG_TRACE_USR3(("TunerDataBindingHandler::setDLSelements"));
   ETG_TRACE_USR3(("TunerDataBindingHandler::Radio Text %s", radioText.c_str()));
   (*oDABChannelInfo).mRadioText = (Candera::String)radioText.c_str();
   if (enDabStatus == 0)
   {
      (*oDABChannelInfo).mDabStatus = "";
   }
   else
   {
      (*oDABChannelInfo).mDabStatus = (Candera::String)szStatusText;
   }
   oDABChannelInfo.MarkAllItemsModified();
   oDABChannelInfo.SendUpdate(true);
}


/************************************************************************
*DESCRIPTION		:Function called to set DAB Ensemble Name
*RETURNVALUE		: Void
************************************************************************/
void TunerDataBindingHandler::setEnsembleName(std::string ensembleText)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::setEnsembleName"));
   (*oDABChannelInfo).mEnsembleLabel = (Candera::String)ensembleText.c_str();
   oDABChannelInfo.MarkAllItemsModified();
   oDABChannelInfo.SendUpdate(true);
}


/************************************************************************
*FUNCTION			:bOnUpdateActiveLineIndexInfo
*DESCRIPTION		:update the selectedline property in List widget
*PARAMETER			:
*RETURNVALUE		:true
************************************************************************/

bool TunerDataBindingHandler::bOnUpdateActiveLineIndexInfo(const Courier::Message& corfoMsg)
{
   // const ActiveLineIndexInfoMsg *copoActiveLineIndexInfoMsg =
   //       message_cast<const ActiveLineIndexInfoMsg *>(&corfoMsg);
   // tU32 MsgId = corfoMsg.GetId();
   //if(copoActiveLineIndexInfoMsg != NULL)
   //{
   //m_ActiveLineIndexInfoMap.clear();
   // Candera::MemoryManagement::SharedPointer<WindowListSelectedLineInterface> poLineIndex(NULL);
   //	 poLineIndex = WindowListSelectedLineInterface::Create();
   //tU32 u32Listtype = copoActiveLineIndexInfoMsg->GetListType();
   //UInt8 u8CurrentRelativeItemPosition = copoActiveLineIndexInfoMsg->GetActiveLineIndex();
   //UInt8 u8CurrentPositionIconType = copoActiveLineIndexInfoMsg->GetActiveLineIconIndex();
   //UInt8 u8LastRelativeItemPosition = copoActiveLineIndexInfoMsg->GetLastLineIndex();
   //UInt8 u8LastPositionIconType = copoActiveLineIndexInfoMsg->GetLastLineIconIndex();
   //ETG_TRACE_USR4(("bOnUpdateActiveLineIndexInfo u32listtype = %d u8CurrentRelativeItemPosition = %d u8CurrentPositionIconType = %d u8LastRelativeItemPosition = %d, u8LastPositionIconType = %d", u32Listtype, u8CurrentRelativeItemPosition, u8CurrentPositionIconType,u8LastRelativeItemPosition,u8LastPositionIconType));
   //  m_ActiveLineIndexInfoMap.insert(std::pair<UInt8,UInt8>(u8LastRelativeItemPosition, u8LastPositionIconType));
   //  m_ActiveLineIndexInfoMap.insert(std::pair<UInt8,UInt8>(u8CurrentRelativeItemPosition, u8CurrentPositionIconType));
   //poLineIndex->vSetSelectedLineMapInfo(m_ActiveLineIndexInfoMap);
   //if(u32Listtype == SM_DAB_LISTTYPE_SERVICELIST)
   //{
   //	ETG_TRACE_USR3(("bOnUpdateActiveLineIndexInfo:: SelectedLineInfo, mapsize = %d", m_ActiveLineIndexInfoMap.size()));
   //	(*oBrowseListSelectedElement).mDAB_ServiceSelectedlineIndexInfo = poLineIndex;
   //	oBrowseListSelectedElement.MarkAllItemsModified();
   //	//oBrowseListSelectedElement.MarkItemModified(ItemKey::BrowseListSelectedLine::DAB_ServiceSelectedlineIndexInfoItem);
   //	oBrowseListSelectedElement.SendUpdate(true);
   //}
   //}
   //else
   //ETG_TRACE_USR4(("media_tclMediaModel::bOnUpdateActiveLineIndexInfo() MsgId %d", MsgId));
   return true;
}


bool TunerDataBindingHandler::onCourierMessage(const TunerEngineeringModeMsg2& corfoMsg)
{
   return true;
}


bool TunerDataBindingHandler::onCourierMessage(const Courier::StartupMsg& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: StartupMsg"));
   bool msgConsumed = false;
   (void)corfoMsg; // To avoid lint warning (unused variable)

   AMFMInfo::poGetInstance()->vGetKDSValue();

   /* Sets tuner information binding sources to null string */
   bool bDataUpdated = false;
   oTunerInfoData.MarkAllItemsModified();
   bDataUpdated = oTunerInfoData.SendUpdate(true);
   oCheckBoxData.MarkAllItemsModified();
   bDataUpdated = oCheckBoxData.SendUpdate(true);
   oDabOptionsData.MarkAllItemsModified();
   bDataUpdated = oDabOptionsData.SendUpdate(true);
   oMenuSelectionData.MarkAllItemsModified();
   bDataUpdated = oMenuSelectionData.SendUpdate(true);
   oMenu_NumOptionsData.MarkAllItemsModified();
   bDataUpdated = oMenu_NumOptionsData.SendUpdate(true);
   oRadioBandInfoData.MarkAllItemsModified();
   bDataUpdated = oRadioBandInfoData.SendUpdate(true);
   oAppFlyIn.MarkAllItemsModified();
   bDataUpdated = oAppFlyIn.SendUpdate(true);
   oDABChannelInfo.MarkAllItemsModified();
   bDataUpdated = oDABChannelInfo.SendUpdate(true);
   oActiveBankInfo.MarkAllItemsModified();
   bDataUpdated = oActiveBankInfo.SendUpdate(true);
   oVariantInfo.MarkAllItemsModified();
   bDataUpdated = oVariantInfo.SendUpdate(true);

   oDABPresetInfo.MarkAllItemsModified();
   bDataUpdated = oDABPresetInfo.SendUpdate(true);

   oFMAMPresetInfo.MarkAllItemsModified();
   bDataUpdated = oFMAMPresetInfo.SendUpdate(true);

   oAMPresetInfo.MarkAllItemsModified();
   bDataUpdated = oAMPresetInfo.SendUpdate(true);

   oScanASTStatusData.MarkAllItemsModified();
   bDataUpdated = oScanASTStatusData.SendUpdate(true);

   oActiveSourceInfo.MarkAllItemsModified();
   bDataUpdated = oActiveSourceInfo.SendUpdate(true);

   oTAInfo.MarkAllItemsModified();
   bDataUpdated = oTAInfo.SendUpdate(true);

   oTTMonitor1.MarkAllItemsModified();
   bDataUpdated = oTTMonitor1.SendUpdate(true);

   oTTMonitor2.MarkAllItemsModified();
   bDataUpdated = oTTMonitor2.SendUpdate(true);

   oTTMonitor3.MarkAllItemsModified();
   bDataUpdated = oTTMonitor3.SendUpdate(true);

   oTTTunerSource_Band.MarkAllItemsModified();
   bDataUpdated = oTTTunerSource_Band.SendUpdate(true);

   oDABSetupMonitor.MarkAllItemsModified();
   bDataUpdated = oDABSetupMonitor.SendUpdate(true);

   oDABTestModePointer.MarkAllItemsModified();
   bDataUpdated = oDABTestModePointer.SendUpdate(true);

   oDABTestModeActivertunerInfo.MarkAllItemsModified();
   bDataUpdated = oDABTestModeActivertunerInfo.SendUpdate(true);

   oDABTestModeActiveEnsembleInfo.MarkAllItemsModified();
   bDataUpdated = oDABTestModeActiveEnsembleInfo.SendUpdate(true);

   oDABTestModeGTestmodeInfo.MarkAllItemsModified();
   bDataUpdated = oDABTestModeGTestmodeInfo.SendUpdate(true);

   oDABTestModeAnnoStatusInfo.MarkAllItemsModified();
   bDataUpdated = oDABTestModeAnnoStatusInfo.SendUpdate(true);

   oDABTestModeGAvailabiltyInfo.MarkAllItemsModified();
   bDataUpdated = oDABTestModeGAvailabiltyInfo.SendUpdate(true);

   oDABTestModeActiveCompInfo.MarkAllItemsModified();
   bDataUpdated = oDABTestModeActiveCompInfo.SendUpdate(true);

   oDABTestServiceFollowingInfo.MarkAllItemsModified();
   bDataUpdated = oDABTestServiceFollowingInfo.SendUpdate(true);

   oTmcInformationData.MarkAllItemsModified();
   bDataUpdated = oTmcInformationData.SendUpdate(true);

   oTestModeAvailabilityInfo.MarkAllItemsModified();
   oTestModeAvailabilityInfo.SendUpdate(true);

   ETG_TRACE_USR3(("TunerDataBindingHandler::bOnStartupMsg(), Data updation failed"));
   //External state variable to switch the state in the entry tot he application and also back from source selection view
   //sm_gSwitchState = tclDataClass::poGetDataClassInstance()->u8GetCurrentBank();
   //ETG_TRACE_USR3(("TunerDataBindingHandler::bOnStartupMsg(), Data updation for sm_gSwitchState : %d",sm_gSwitchState));

   if (false == bDataUpdated)
   {
      ETG_TRACE_USR3(("TunerDataBindingHandler::bOnStartupMsg(), Data updation failed"));
   }

   msgConsumed = true;
   return msgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const ShutdownMsg& corfoMsg)
{
   return true;
}


bool TunerDataBindingHandler::onCourierMessage(const UpdateTunerStatusMsg& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: UpdateTunerStatusMsg"));
   bool msgConsumed = false;
   msgConsumed = cobOnUpdateTunerStatusMsg(corfoMsg);
   return msgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const UpdateFrequencyMsg& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: UpdateFrequencyMsg"));
   bool msgConsumed = false;
   // msgConsumed = bOnUpdateFrequencyMsg(corfoMsg);
   return msgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const UpdatePSNameTPMsg& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: UpdatePSNameTPMsg"));
   bool msgConsumed = false;
   //msgConsumed = bOnUpdatePSNameTPMsg(corfoMsg);
   return msgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const UpdatePresetMsg& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: UpdatePresetMsg"));
   bool msgConsumed = false;
   //msgConsumed = bOnUpdatePresetMsg(corfoMsg);
   return msgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const UpdateFMPresetFocused& /*corfoMsg*/)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: UpdateFMPresetFocused"));
   vUpdate_FM_PresetFocused(m_u8FMFocusedPresestNumber);
   return true;
}


bool TunerDataBindingHandler::onCourierMessage(const UpdateAMPresetFocused& /*corfoMsg*/)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: UpdateAMPresetFocused"));
   vUpdate_AM_PresetFocused(m_u8AMFocusedPresestNumber);
   return true;
}


bool TunerDataBindingHandler::onCourierMessage(const PresetPopupMsg& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: PresetPopupMsg"));
   bool msgConsumed = false;
   msgConsumed = bOnPresetPopupMsg(corfoMsg);
   return msgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const LPReleaseMsg& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: LPReleaseMsg"));

   /* Update the frequency value in HMI for LPReleaseMsg */
   const LPReleaseMsg* copoLPReleaseMsg = message_cast<const LPReleaseMsg*>(&corfoMsg);
   if (NULL != copoLPReleaseMsg)
   {
      /*Set the Current value for the  progress bar in the popup view to 0*/
      m_u8timerCount = 0;
      m_u8popupState = EN_POPUP_CANNOTSTORE;
      /*Stop the timer  */
      m_oTimerPreset.stop();
   }
   else
   {
      ETG_TRACE_USR3(("TunerDataBindingHandler : copoUpdatePresetMsg NULL Pointer"));
   }
   return true;
}


bool TunerDataBindingHandler::onCourierMessage(const Courier::UpdateModelMsg& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: UpdateModelMsg"));
   bool bMsgConsumed = false;
   ETG_TRACE_USR3(("TunerDataBindingHandler::vOnUpdateModelMsg()"));
   bool bUpdateListFlag = FALSE;
   const UpdateModelMsg* poUpdateModelMsg = message_cast<const UpdateModelMsg*>(&corfoMsg);

   if (NULL == poUpdateModelMsg)
   {
      ETG_TRACE_USR3(("ERROR: TunerDataBindingHandler::vOnUpdateModelMsg(INVALID MESSAGE) "));
   }
   else
   {
      bool bRetBindingVal = FALSE;

      for (UInt32 u32Index = 0; (poUpdateModelMsg->RequestCount() > u32Index); ++u32Index)
      {
         Request request(poUpdateModelMsg->GetRequest(u32Index));
         ETG_TRACE_USR3(("bOnUpdateModelMsg - request.ItemKey(): %d ", request.ItemKey()));
         if (ItemKey::tuningScale_SliderMarkerPositionItem == request.BindingSourceKey())
         {
            bMsgConsumed = true;
            vHandleSliderChange(request);
         }
         else if (ItemKey::MenuSelectionItem == request.BindingSourceKey())
         {
            bMsgConsumed = true;
            if (ItemKey::MenuSelection::MainMenu_SelectionItem == request.ItemKey())
            {
               oMenuSelectionData.SetValue(request.ItemKey(), request.GetItemValue());
               ETG_TRACE_USR3(("bOnUpdateModelMsg - Main menu item selected = %d", (*oMenuSelectionData).mMainMenu_Selection));
               vHandleMainMenuOption((*oMenuSelectionData).mMainMenu_Selection);
               break;
            }
            else if (ItemKey::MenuSelection::DABPTYOption_SelectionItem == request.ItemKey())
            {
               oMenuSelectionData.SetValue(request.ItemKey(), request.GetItemValue());
               ETG_TRACE_USR3(("bOnUpdateModelMsg - DAB PTY menu item selected = %d", (*oMenuSelectionData).mDABPTYOption_Selection));
               vHandleDABPTYFilterOption((*oMenuSelectionData).mDABPTYOption_Selection);
               break;
            }
            else if (ItemKey::BrowseListSelectedLine::AF_ServiceSelectedItem == request.ItemKey())
            {
               bRetBindingVal = oBrowseListSelectedElement.SetValue(request.ItemKey(), request.GetItemValue());
               ETG_TRACE_USR3(("bOnUpdateModelMsg - AF_ServiceSelected trigger: %d", (*oBrowseListSelectedElement).mAF_ServiceSelected));
               bUpdateListFlag = true;
            }
            else if (ItemKey::BrowseListSelectedLine::TMC_ServiceSelectedItem == request.ItemKey())
            {
               bRetBindingVal = oBrowseListSelectedElement.SetValue(request.ItemKey(), request.GetItemValue());
               ETG_TRACE_USR3(("bOnUpdateModelMsg - TMC_ServiceSelected trigger: %d", (*oBrowseListSelectedElement).mTMC_ServiceSelected));
               bUpdateListFlag = true;
            }
         } // End of else if
         else
         {
            // Catch all else
         }
      }//  End of for ( UInt32 u32Index = 0;( copoMsg->RequestCount() > u32Index ); ++u32Index)
   }
   return bMsgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const Update_Slider_TunerFrequencyMsg& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: bonUpdate_Slider_TunerFrequencyMsg"));
   bool bMsgConsumed = false; // Holds true if Model consumes the message

   //if(TRUE == m_oTimerSlider.isActive())
   //{
   //ETG_TRACE_USR3(("NO UPDATE IN THE SLIDER AS TIMER IS RUNNING TO AVOID JUMPS"));
   //return bMsgConsumed;
   //}

   const Update_Slider_TunerFrequencyMsg* copoUpdate_Slider_TunerFrequencyMsg = message_cast<const Update_Slider_TunerFrequencyMsg*>(&corfoMsg);
   if (NULL != copoUpdate_Slider_TunerFrequencyMsg)
   {
      bool bDataUpdated = false;

      ETG_TRACE_USR3(("TunerDataBindingHandler, Frequency" "copoUpdate_Slider_TunerFrequencyMsg->GetCurrent_Frequency_Value() = %f",
                      copoUpdate_Slider_TunerFrequencyMsg->GetCurrent_Frequency_Value()));
      (*oTunerSliderMarkerPosition).mCurrentValue = (Candera::Float) copoUpdate_Slider_TunerFrequencyMsg->GetCurrent_Frequency_Value();
      ETG_TRACE_USR3(("TunerDataBindingHandler, Frequency (*oTunerSliderMarkerPosition).mCurrentValue = %f", (*oTunerSliderMarkerPosition).mCurrentValue));
      oTunerSliderMarkerPosition.MarkItemModified(ItemKey::tuningScale_SliderMarkerPosition::CurrentValueItem);
      bDataUpdated = oTunerSliderMarkerPosition.SendUpdate(true);
      if (true == bDataUpdated)
      {
         ETG_TRACE_USR3(("TunerDataBindingHandler,(*oTunerSliderMarkerPosition).mCurrentValue UPDATED"));
      }
      else
      {
         ETG_TRACE_USR3(("ERROR: TunerDataBindingHandler:Data updation failed oTunerSliderMarkerPosition"));
      }
   }
   else
   {
      ETG_TRACE_USR3(("TunerDataBindingHandler : copoUpdateFreqMsg NULL Pointer"));
   }
   bMsgConsumed = true;
   return bMsgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const UpdateCurrentBank& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::UpdateCurrentBank()"));
   bool bMsgConsumed = false;
   const UpdateCurrentBank* poUpdateCurrentBank = message_cast<const UpdateCurrentBank*>(&corfoMsg);
   if (NULL != poUpdateCurrentBank)
   {
      ETG_TRACE_USR3(("TunerDataBindingHandler::UpdateCurrentBank(%d)", poUpdateCurrentBank->GetCurentBank()));
      ETG_TRACE_USR3(("TunerDataBindingHandler::UpdateCurrentBank(%d)", sm_gSwitchState));
      vUpdateCurrentbankIcon();
   }
   bMsgConsumed = true;
   return bMsgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const RDSSupport& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: RDSSupport"));
   bool msgConsumed = false;
   msgConsumed = bOnUpdateRDSSupport(corfoMsg);
   return msgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const AntennaStatus& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: AntennaStatus"));
   bool msgConsumed = false;
   msgConsumed = bOnUpdateAntennaStatus(corfoMsg);
   return msgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const BandscanStatus& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: BandscanStatus"));
   bool msgConsumed = false;
   msgConsumed = bOnUpdateBandscanStatus(corfoMsg);
   return msgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const BrowseListRespMsg& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: BrowseListRespMsg"));
   bool msgConsumed = false;
   msgConsumed = bOnUpdateBrowseMsg(corfoMsg);
   return msgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const DABChannelInfoMsg& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: DABChannelInfoMsg"));
   bool msgConsumed = false;
   msgConsumed = bOnDABChannelInfoMsg(corfoMsg);
   return msgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const CannotStorePopupMsg& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: CannotStorePopupMsg"));
   bool msgConsumed = false;
   msgConsumed = bOnCannotStorePopupMsg(corfoMsg);
   return msgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const UpdateAutoStoreMsg& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: UpdateAutoStoreMsg"));
   bool msgConsumed = false;
   msgConsumed = bOnUpdateAutoStoreMsg(corfoMsg);
   return msgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const TripleTunerMonitorDataSet1& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: TripleTunerMonitorDataSet1"));
   bool msgConsumed = false;
   msgConsumed = bOnUpdateTTMonitorTuner1(corfoMsg);
   return msgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const TripleTunerMonitorDataSet2& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: TripleTunerMonitorDataSet2"));
   bool msgConsumed = false;
   msgConsumed = bOnUpdateTTMonitorTuner2(corfoMsg);
   return msgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const TripleTunerMonitorDataSet3& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: TripleTunerMonitorDataSet3"));
   bool msgConsumed = false;
   msgConsumed = bOnUpdateTTMonitorTunerSource_Band(corfoMsg);
   return msgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const DABServiceLinkingMsg& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: DABServiceLinkingMsg"));
   bool msgConsumed = false;
   //msgConsumed = bOnUpdateDABServiceLinkingMsg(corfoMsg);
   return msgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const DABConcealmentLevelMsg& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: DABConcealmentLevelMsg"));
   bool msgConsumed = false;
   msgConsumed = bOnUpdateDABConcealmentLevelMsg(corfoMsg);
   return msgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const DABTestModeActiveServiceInfo& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: DABTestModeActiveServiceInfo"));
   bool msgConsumed = false;
   msgConsumed = bOnUpdateDABTestModeActiveServiceInfo(corfoMsg);
   return msgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const DABTestModeActiveEnsembleInfo& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: DABTestModeActiveEnsembleInfo"));
   bool msgConsumed = false;
   msgConsumed = bOnUpdateDABTestModeActiveEnsembleInfo(corfoMsg);
   return msgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const DABTestModeTunerStatusInfo& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: DABTestModeTunerStatusInfo"));
   bool msgConsumed = false;
   msgConsumed = bOnUpdateDABTestModeTunerStatusInfo(corfoMsg);
   return msgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const DABTestModeBGTestmodeInfo& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: DABTestModeBGTestmodeInfo"));
   bool msgConsumed = false;
   msgConsumed = bOnUpdateDABTestModeBGTestmodeInfo(corfoMsg);
   return msgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const DABTestModeGTestmodeInfo& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: DABTestModeGTestmodeInfo"));
   bool msgConsumed = false;
   msgConsumed = bOnUpdateDABTestModeGTestmodeInfo(corfoMsg);
   return msgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const DABTestModeAnnoStatus& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: DABTestModeAnnoStatus"));
   bool msgConsumed = false;
   msgConsumed = bOnUpdateDABTestModeAnnoStatus(corfoMsg);
   return msgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const DABTestModeActiveCompInfo& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: DABTestModeActiveCompInfo"));
   bool msgConsumed = false;
   msgConsumed = bOnUpdateDABGTestModeActiveComp(corfoMsg);
   return msgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const DABTestModeGAvailabilty& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: DABTestModeGAvailabilty"));
   bool msgConsumed = false;
   msgConsumed = bOnUpdateDABTestModeGAvailabilty(corfoMsg);
   return msgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const DABTestModeServiceFollowingStatus& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: DABTestModeServiceFollowingStatus"));
   bool msgConsumed = false;
   msgConsumed = bOnUpdateDABTestModeServiceFollowingStatus(corfoMsg);
   return msgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const DABTestModeTASource& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: DABTestModeTASource"));
   bool msgConsumed = false;
   msgConsumed = bOnUpdateDABTestModeTASource(corfoMsg);
   return msgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const TmcStnInfoData& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: TmcStnInfoData"));
   bool msgConsumed = false;
   msgConsumed = bOnTmcStnInfoData(corfoMsg);
   return msgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const ActiveLineIndexInfoMsg& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: ActiveLineIndexInfoMsg"));
   bool msgConsumed = false;
   msgConsumed = bOnUpdateActiveLineIndexInfo(corfoMsg);
   return msgConsumed;
}


//wst4kor
bool TunerDataBindingHandler::onCourierMessage(const ChannelInfoMsg& corfoMsg)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onCourierMessage:: ChannelInfoMsg"));
   bool msgConsumed = false;
   msgConsumed = bOnChannelInfoMsg(corfoMsg);
   return msgConsumed;
}


// bool TunerDataBindingHandler::onCourierMessage(const UpdateWaveBandStatusMsg& corfoMsg){return true;}
// bool TunerDataBindingHandler::onCourierMessage(const TunerEngineeringModeMsg& corfoMsg){return true;}
// bool TunerDataBindingHandler::onCourierMessage(const TunerEngineeringModeMsg1& corfoMsg){return true;}
// bool TunerDataBindingHandler::onCourierMessage(const UpdateWindowData& corfoMsg){return true;}
// bool TunerDataBindingHandler::onCourierMessage(const SelectedListItemInfo& corfoMsg){return true;}

void TunerDataBindingHandler::startPresetTimer()
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::startPresetTimer()"));
   m_oTimerPreset.start();
}


void TunerDataBindingHandler::onExpired(asf::core::Timer& timer, boost::shared_ptr<asf::core::TimerPayload> data)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::onExpired(Reason: %d)", data->getReason()));
   //If expired, check for the reason and fire SM message
   if ((timer.getAct() == m_oTimerPreset.getAct()) && (data->getReason() == asf::core::TimerPayload_Reason__Completed))
   {
      m_tunerDataBindingHandler->vPresetTimeoutHandler();
   }
}


/**
* Description     : Function called to update Quad Monitor Column 1 Data
*
* @param[in]      : band: band in the corresponding physical tuner
* @param[in]      : freq: frequency
* @param[in]      : PI: PI code of the tuned frequency
* @param[in]      : psName: PS name of the tuned station
* @param[in]      : FS_Q: field strength and quality of the station
* @param[in]      : hCut: HiCut value
* @param[in]      : sharx: Sharx Value
* @param[in]      : PD: Phase diversity
* @param[in]      : TP: TP Station
* @param[in]      : TA: Traffic Announcement
* @param[in]      : RDS: RDS Error Rate
* @param[in]      : AFlength: AF List length
* @param[in]      : MP_NC: MultiPath and Neighbour Channel
* @param[in]      : deviation: Frequency deviation
* @param[in]      : CS: Channel Separation
* @return         : void
*/
void TunerDataBindingHandler::updateTun1QuadData(std::string band, std::string freq, std::string PI, std::string psName, std::string FS_Q, std::string hCut, std::string sharx,
      std::string PD, std::string TP, std::string TA, std::string RDS, std::string AFlength, std::string MP_NC, std::string deviation, std::string CS)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::updateTun1QuadData 1"));

   (*oTTMonitor1).mFreq = freq.c_str();
   (*oTTMonitor1).mPS = psName.c_str();
   (*oTTMonitor1).mPI = PI.c_str();
   (*oTTMonitor1).mFSQ = FS_Q.c_str();
   (*oTTMonitor1).mSharx = sharx.c_str();
   (*oTTMonitor1).mHicut = hCut.c_str();
   (*oTTMonitor1).mDev = deviation.c_str();
   (*oTTMonitor1).mCS = CS.c_str();
   (*oTTMonitor1).mRDS = RDS.c_str();
   (*oTTMonitor1).mTP =  TP.c_str();
   (*oTTMonitor1).mTA = TA.c_str();
   (*oTTMonitor1).mAF = AFlength.c_str();
   (*oTTMonitor1).mMPNC = MP_NC.c_str();

   oTTMonitor1.MarkAllItemsModified();
   oTTMonitor1.SendUpdate();
}


/**
* Description     : Function called to update Quad Monitor Column 2 Data
*
* @param[in]      : band: band in the corresponding physical tuner
* @param[in]      : freq: frequency
* @param[in]      : PI: PI code of the tuned frequency
* @param[in]      : psName: PS name of the tuned station
* @param[in]      : FS_Q: field strength and quality of the station
* @param[in]      : hCut: HiCut value
* @param[in]      : sharx: Sharx Value
* @param[in]      : PD: Phase diversity
* @param[in]      : TP: TP Station
* @param[in]      : TA: Traffic Announcement
* @param[in]      : RDS: RDS Error Rate
* @param[in]      : AFlength: AF List length
* @param[in]      : MP_NC: MultiPath and Neighbour Channel
* @param[in]      : deviation: Frequency deviation
* @param[in]      : CS: Channel Separation
* @return         : void
*/
void TunerDataBindingHandler::updateTun2QuadData(std::string band, std::string freq, std::string PI, std::string psName, std::string FS_Q, std::string hCut, std::string sharx,
      std::string PD, std::string TP, std::string TA, std::string RDS, std::string AFlength, std::string MP_NC, std::string deviation, std::string CS)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::updateTun1QuadData 2"));

   (*oTTMonitor2).mFreq = freq.c_str();
   (*oTTMonitor2).mPS = psName.c_str();
   (*oTTMonitor2).mPI = PI.c_str();
   (*oTTMonitor2).mFSQ = FS_Q.c_str();
   (*oTTMonitor2).mSharx = sharx.c_str();
   (*oTTMonitor2).mHicut = hCut.c_str();
   (*oTTMonitor2).mDev = deviation.c_str();
   (*oTTMonitor2).mCS = CS.c_str();
   (*oTTMonitor2).mRDS = RDS.c_str();
   (*oTTMonitor2).mTP =  TP.c_str();
   (*oTTMonitor2).mTA = TA.c_str();
   (*oTTMonitor2).mAF = AFlength.c_str();
   (*oTTMonitor2).mMPNC = MP_NC.c_str();

   oTTMonitor2.MarkAllItemsModified();
   oTTMonitor2.SendUpdate();
}


/**
* Description     : Function called to update Quad Monitor Column 3 Data
*
* @param[in]      : band: band in the corresponding physical tuner
* @param[in]      : freq: frequency
* @param[in]      : PI: PI code of the tuned frequency
* @param[in]      : psName: PS name of the tuned station
* @param[in]      : FS_Q: field strength and quality of the station
* @param[in]      : hCut: HiCut value
* @param[in]      : sharx: Sharx Value
* @param[in]      : PD: Phase diversity
* @param[in]      : TP: TP Station
* @param[in]      : TA: Traffic Announcement
* @param[in]      : RDS: RDS Error Rate
* @param[in]      : AFlength: AF List length
* @param[in]      : MP_NC: MultiPath and Neighbour Channel
* @param[in]      : deviation: Frequency deviation
* @param[in]      : CS: Channel Separation
* @return         : void
*/
void TunerDataBindingHandler::updateTun3QuadData(std::string band, std::string freq, std::string PI, std::string psName, std::string FS_Q, std::string hCut, std::string sharx,
      std::string PD, std::string TP, std::string TA, std::string RDS, std::string AFlength, std::string MP_NC, std::string deviation, std::string CS)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::updateTun1QuadData 3"));

   (*oTTMonitor3).mFreq = freq.c_str();
   (*oTTMonitor3).mPS = psName.c_str();
   (*oTTMonitor3).mPI = PI.c_str();
   (*oTTMonitor3).mFSQ = FS_Q.c_str();
   (*oTTMonitor3).mSharx = sharx.c_str();
   (*oTTMonitor3).mHicut = hCut.c_str();
   (*oTTMonitor3).mDev = deviation.c_str();
   (*oTTMonitor3).mCS = CS.c_str();
   (*oTTMonitor3).mRDS = RDS.c_str();
   (*oTTMonitor3).mTP =  TP.c_str();
   (*oTTMonitor3).mTA = TA.c_str();
   (*oTTMonitor3).mAF = AFlength.c_str();
   (*oTTMonitor3).mMPNC = MP_NC.c_str();

   oTTMonitor3.MarkAllItemsModified();
   oTTMonitor3.SendUpdate();
}


/**
* Description     : Function called to update Quad Monitor Column 4 Data
*
* @param[in]      : band: band in the corresponding physical tuner
* @param[in]      : freq: frequency
* @param[in]      : PI: PI code of the tuned frequency
* @param[in]      : psName: PS name of the tuned station
* @param[in]      : FS_Q: field strength and quality of the station
* @param[in]      : hCut: HiCut value
* @param[in]      : sharx: Sharx Value
* @param[in]      : PD: Phase diversity
* @param[in]      : TP: TP Station
* @param[in]      : TA: Traffic Announcement
* @param[in]      : RDS: RDS Error Rate
* @param[in]      : AFlength: AF List length
* @param[in]      : MP_NC: MultiPath and Neighbour Channel
* @param[in]      : deviation: Frequency deviation
* @param[in]      : CS: Channel Separation
* @return         : void
*/
void TunerDataBindingHandler::updateTun4QuadData(std::string band, std::string freq, std::string PI, std::string psName, std::string FS_Q, std::string hCut, std::string sharx,
      std::string PD, std::string TP, std::string TA, std::string RDS, std::string AFlength, std::string MP_NC, std::string deviation, std::string CS)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::updateTun1QuadData 4"));
   (*oTTMonitor4).mFreq = freq.c_str();
   (*oTTMonitor4).mPS = psName.c_str();
   (*oTTMonitor4).mPI = PI.c_str();
   (*oTTMonitor4).mFSQ = FS_Q.c_str();
   (*oTTMonitor4).mSharx = sharx.c_str();
   (*oTTMonitor4).mHicut = hCut.c_str();
   (*oTTMonitor4).mDev = deviation.c_str();
   (*oTTMonitor4).mCS = CS.c_str();
   (*oTTMonitor4).mRDS = RDS.c_str();
   (*oTTMonitor4).mTP = TP.c_str();
   (*oTTMonitor4).mTA = TA.c_str();
   (*oTTMonitor4).mAF = AFlength.c_str();
   (*oTTMonitor4).mMPNC = MP_NC.c_str();

   oTTMonitor4.MarkAllItemsModified();
   oTTMonitor4.SendUpdate();
}


void TunerDataBindingHandler::updateQuadMonitorStatus(bool tuner1Status, bool tuner2Status, bool tuner3Status, bool tuner4Status,
      bool band1Status, bool band2Status, bool band3Status, bool band4Status,
      bool isBand1Enabled, bool isBand2Enabled, bool isBand3Enabled, bool isBand4Enabled, bool isTuner1Enabled, bool isTuner2Enabled, bool isTuner3Enabled, bool isTuner4Enabled)
{
   /*(*_dtmQuadMonData).mIsTuner1Active = tuner1Status;
   (*_dtmQuadMonData).mIsTuner2Active = tuner2Status;
   (*_dtmQuadMonData).mIsTuner3Active = tuner3Status;
   (*_dtmQuadMonData).mIsTuner4Active = tuner4Status;
   (*_dtmQuadMonData).mIsBand1Active = band1Status;
   (*_dtmQuadMonData).mIsBand2Active = band2Status;
   (*_dtmQuadMonData).mIsBand3Active = band3Status;
   (*_dtmQuadMonData).mIsBand4Active = band4Status;
   (*_dtmQuadMonData).mIsBand1Enable = isBand1Enabled;
   (*_dtmQuadMonData).mIsBand2Enable = isBand2Enabled;
   (*_dtmQuadMonData).mIsBand3Enable = isBand3Enabled;
   (*_dtmQuadMonData).mIsBand4Enable = isBand4Enabled;
   (*_dtmQuadMonData).mIsTuner1Enable = isTuner1Enabled;
   (*_dtmQuadMonData).mIsTuner2Enable = isTuner2Enabled;
   (*_dtmQuadMonData).mIsTuner3Enable = isTuner3Enabled;
   (*_dtmQuadMonData).mIsTuner4Enable = isTuner4Enabled;

   _dtmQuadMonData.MarkAllItemsModified();
   _dtmQuadMonData.SendUpdate();*/
}


/**
* Description     : Function called to update Calibration Data
*
* @param[in]      : calibFMTun1:Calibration Data of FM Tuner 1
* @param[in]      : calibFMTun1:Calibration Data of FM Tuner 2
* @param[in]      : calibFMTun1:Calibration Data of FM Tuner 3
* @param[in]      : calibFMTun1:Calibration Data of FM Tuner 4
* @param[in]      : calibFMTun1:Calibration Data of AM tuner
* @return         : void
*/
void TunerDataBindingHandler::updateCalibrationData(std::string calibFMTun1, std::string calibFMTun2, std::string calibFMTun3, std::string calibFMTun4, std::string calibAMTun)
{
   /*(*_dtmCalibData).mFmTuner1Txt = calibFMTun1.c_str();
   (*_dtmCalibData).mFmTuner2Txt = calibFMTun2.c_str();
   (*_dtmCalibData).mFmTuner3Txt = calibFMTun3.c_str();
   (*_dtmCalibData).mFmTuner4Txt = calibFMTun4.c_str();
   (*_dtmCalibData).mAmTunerTxt = calibAMTun.c_str();

   _dtmCalibData.MarkAllItemsModified();
   _dtmCalibData.SendUpdate();*/
   bool bDataUpdated = false;

   (*oEngineeringMenuData2).mAMTuner = calibAMTun.c_str();

   (*oEngineeringMenuData2).mFMTuner1 = calibFMTun1.c_str();

   (*oEngineeringMenuData2).mFMTuner2 = calibFMTun2.c_str();

   (*oEngineeringMenuData2).mFMTuner3 = calibFMTun3.c_str();

   (*oEngineeringMenuData2).mFMTuner4 = calibFMTun4.c_str();

   //(*oEngineeringMenuData2).mDABTuner = szData;

   /* Mark all items for updation */

   oEngineeringMenuData2.MarkAllItemsModified();

   bDataUpdated = oEngineeringMenuData2.SendUpdate(true);
}


/**
* Description     : Function called to update Active Tuner Data in Quad monitor
*
* @param[in]      : activeBand: Active Physical Tuner
* @return         : void
*/
void TunerDataBindingHandler::updateTMactiveTunerdata(uint8 activeBand)
{
   bool bDataUpdated = false;
   (*oTTTunerSource_Band).mTunerSource1 = false;
   (*oTTTunerSource_Band).mTunerSource2 = false;
   (*oTTTunerSource_Band).mTunerSource3 = false;
   (*oTTTunerSource_Band).mTunerSource4 = false;
   (*oTTTunerSource_Band).mBandEnable1 = false;
   (*oTTTunerSource_Band).mBandEnable2 = false;
   (*oTTTunerSource_Band).mBandEnable3 = false;
   (*oTTTunerSource_Band).mBandEnable4 = false;
   if (activeBand == 0)
   {
      (*oTTTunerSource_Band).mTunerSource1 = true;
      (*oTTTunerSource_Band).mBandEnable1 = true;
   }
   else if (activeBand == 1)
   {
      (*oTTTunerSource_Band).mTunerSource2 = true;
      (*oTTTunerSource_Band).mBandEnable2 = true;
   }
   else if (activeBand == 2)
   {
      (*oTTTunerSource_Band).mTunerSource3 = true;
      (*oTTTunerSource_Band).mBandEnable3 = true;
   }
   else if (activeBand == 3)
   {
      (*oTTTunerSource_Band).mTunerSource4 = true;
      (*oTTTunerSource_Band).mBandEnable4 = true;
   }
   oTTTunerSource_Band.MarkAllItemsModified();
   bDataUpdated = oTTTunerSource_Band.SendUpdate(true);
}


void TunerDataBindingHandler::updateEngineeringMenu(std::string freq, std::string psName, std::string PI, std::string FS_Q, std::string Quality, std::string hCut, std::string sharx,
      std::string deviation, std::string CS)
{
   (*oEngineeringMenuData).m_szFrequency = freq.c_str();
   (*oEngineeringMenuData).m_szPSName = psName.c_str();
   (*oEngineeringMenuData).m_szPI = PI.c_str();
   (*oEngineeringMenuData).m_szFieldstrength = FS_Q.c_str();
   (*oEngineeringMenuData).m_szQuality = Quality.c_str();
   (*oEngineeringMenuData).m_szSharx = sharx.c_str();
   (*oEngineeringMenuData).m_szHicut = hCut.c_str();
   (*oEngineeringMenuData).m_szFMDeviation = deviation.c_str();
   (*oEngineeringMenuData).m_szChannelSeparation = CS.c_str();

   oEngineeringMenuData.MarkAllItemsModified();
   oEngineeringMenuData.SendUpdate();
}


void TunerDataBindingHandler::setAMSliderValues(uint32 u32AMStartFreq, uint16 u16AMmaxvalue, uint8 u8AMnoofsteps, uint32 u32AMFrequencyStepSize)
{
   //added for coverity may 4
   ETG_TRACE_USR3(("TunerDataBindingHandler::setAMSliderValues()"));
   bool bDataUpdated = false;
   //update member value.
   m_u32AMHiRange = u16AMmaxvalue;
   m_u32AMLowRange = u32AMStartFreq;

   (*otuningSlider).mMinValueAM = (Candera::Float)u32AMStartFreq;
   (*otuningSlider).mMaxValueAM = (Candera::Float)u16AMmaxvalue;
   (*otuningSlider).mNoOfStepsAM = (Candera::Float)u8AMnoofsteps;

   ETG_TRACE_USR3(("TunerDataBindingHandler::getRegion(%d)", TunerDataPoolConfig::getInstance()->readRegionType()));
   //set the values in databinding for update in GUI Screens based on region.
   if ((TunerDataPoolConfig::getInstance()->readRegionType() == NAM_REGION))
   {
      (*otuningSlider).mImage_SliderAM = (Candera::String)GEX_10k;
      (*otuningSlider).mImage_Slider_BG = (Candera::String)AM_BG_1710;
      (*otuningSlider).mStepSizeAM = (Candera::Float)u32AMFrequencyStepSize;
   }
   else if ((TunerDataPoolConfig::getInstance()->readRegionType() == EUR_REGION))
   {
      (*otuningSlider).mImage_SliderAM = (Candera::String)EU_9k;
      (*otuningSlider).mImage_Slider_BG = (Candera::String)AM_BG_1611;
      (*otuningSlider).mStepSizeAM = (Candera::Float)u32AMFrequencyStepSize;
   }
   otuningSlider.MarkAllItemsModified();
   bDataUpdated = otuningSlider.SendUpdate(true);
}


void TunerDataBindingHandler::setFMSliderValues(uint32 u32FMLowRange, uint32 u32FMHiRange, uint32 u32FMFrequencyStep)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::setFMSliderValues()"));
   //update member variables.
   m_u32FMLowRange = u32FMLowRange;
   m_u32FMHiRange = u32FMHiRange;

   (*otuningSlider).mMinValueAM = (Candera::Float)(u32FMLowRange / 1000);
   (*otuningSlider).mMaxValueFM = (Candera::Float)(u32FMHiRange / 1000);
   (*otuningSlider).mStepSizeFM = (Candera::Float)u32FMFrequencyStep;
   otuningSlider.MarkAllItemsModified();
   otuningSlider.SendUpdate();
}


bool TunerDataBindingHandler::onCourierMessage(const NextSelectMsg_FM& corfoMsg)
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::NextSelectMsg_FM()"));

   tBool bMsgConsumed = FALSE; //Holds TRUE if the message is consumed
   (*oLayerNextPresets_FM).mNextPreviousIndex_FM = 1;
   (*oNextPrevious_Status_FM).mIsPreviousActive_FM = 1;
   oLayerNextPresets_FM.MarkAllItemsModified();
   oNextPrevious_Status_FM.MarkAllItemsModified();
   oLayerNextPresets_FM.SendUpdate();
   oNextPrevious_Status_FM.SendUpdate();
   return bMsgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const PreviousSelectMsg_FM& corfoMsg)
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::PreviousSelectMsg_FM()"));

   tBool bMsgConsumed = FALSE; //Holds TRUE if the message is consumed
   (*oLayerNextPresets_FM).mNextPreviousIndex_FM = 0;
   (*oNextPrevious_Status_FM).mIsNextActive_FM = 1;
   oLayerNextPresets_FM.MarkAllItemsModified();
   oNextPrevious_Status_FM.MarkAllItemsModified();
   oLayerNextPresets_FM.SendUpdate();
   oNextPrevious_Status_FM.SendUpdate();

   return bMsgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const NextSelectMsg_AM& corfoMsg)   ////fayaz
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::NextSelectMsg_AM()"));

   tBool bMsgConsumed = FALSE; //Holds TRUE if the message is consumed
   (*oLayerNextPresets_AM).mNextPreviousIndex_AM = 1;
   (*oNextPrevious_Status_AM).mIsPreviousActive_AM = 1;
   oLayerNextPresets_AM.MarkAllItemsModified();
   oNextPrevious_Status_AM.MarkAllItemsModified();
   oLayerNextPresets_AM.SendUpdate();
   oNextPrevious_Status_AM.SendUpdate();
   return bMsgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const PreviousSelectMsg_AM& corfoMsg) ///fayaz
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::PreviousSelectMsg_AM()"));

   tBool bMsgConsumed = FALSE; //Holds TRUE if the message is consumed
   (*oLayerNextPresets_AM).mNextPreviousIndex_AM = 0;
   (*oNextPrevious_Status_AM).mIsNextActive_AM = 1;
   oLayerNextPresets_AM.MarkAllItemsModified();
   oNextPrevious_Status_AM.MarkAllItemsModified();
   oLayerNextPresets_AM.SendUpdate();
   oNextPrevious_Status_AM.SendUpdate();
   return bMsgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const NextSelectMsg_DAB& corfoMsg)   ////fayaz
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::NextSelectMsg_DAB()"));

   tBool bMsgConsumed = FALSE; //Holds TRUE if the message is consumed
   (*oLayerNextPresets_DAB).mNextPreviousIndex_DAB = 1;
   (*oNextPrevious_Status_DAB).mIsPreviousActive_DAB = 1;
   oLayerNextPresets_DAB.MarkAllItemsModified();
   oNextPrevious_Status_DAB.MarkAllItemsModified();
   oLayerNextPresets_DAB.SendUpdate();
   oNextPrevious_Status_DAB.SendUpdate();
   return bMsgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const PreviousSelectMsg_DAB& corfoMsg) ///fayaz
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::PreviousSelectMsg_DAB()"));

   tBool bMsgConsumed = FALSE; //Holds TRUE if the message is consumed
   (*oLayerNextPresets_DAB).mNextPreviousIndex_DAB = 0;
   (*oNextPrevious_Status_DAB).mIsNextActive_DAB = 1;
   oLayerNextPresets_DAB.MarkAllItemsModified();
   oNextPrevious_Status_DAB.MarkAllItemsModified();
   oLayerNextPresets_DAB.SendUpdate();
   oNextPrevious_Status_DAB.SendUpdate();

   return bMsgConsumed;
}


void TunerDataBindingHandler::InitialIconHighlight()
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::InitialIconHighlight()"));

   (*oNextPrevious_Status_FM).mIsNextActive_FM = 1;
   (*oNextPrevious_Status_AM).mIsNextActive_AM = 1;
   (*oNextPrevious_Status_DAB).mIsNextActive_DAB = 1;
   oNextPrevious_Status_FM.MarkAllItemsModified();
   oNextPrevious_Status_AM.MarkAllItemsModified();
   oNextPrevious_Status_DAB.MarkAllItemsModified();
   oNextPrevious_Status_FM.SendUpdate();
   oNextPrevious_Status_AM.SendUpdate();
   oNextPrevious_Status_DAB.SendUpdate();
}


bool TunerDataBindingHandler::onCourierMessage(const PreviousIconSelectMsg& corfoMsg)
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::PreviousIconSelectMsg()"));

   tBool bMsgConsumed = FALSE; //Holds TRUE if the message is consumed
   (*oSearchNextPrevious_IconActive).mIsPreviousIconActive = 1;
   (*oSearchNextPrevious_IconActive).mIsNextIconActive = 0;
   oSearchNextPrevious_IconActive.MarkAllItemsModified();
   oSearchNextPrevious_IconActive.SendUpdate();

   return bMsgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const NextIconSelectMsg& corfoMsg)
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::NextIconSelectMsg()"));

   tBool bMsgConsumed = FALSE; //Holds TRUE if the message is consumed
   (*oSearchNextPrevious_IconActive).mIsNextIconActive = 1;
   (*oSearchNextPrevious_IconActive).mIsPreviousIconActive = 0;
   oSearchNextPrevious_IconActive.MarkAllItemsModified();
   oSearchNextPrevious_IconActive.SendUpdate();

   return bMsgConsumed;
}


bool TunerDataBindingHandler::onCourierMessage(const AbortNextPreviousIcon& corfoMsg)
{
   ETG_TRACE_USR3(("AMFMInfoCourierMsgHandler::AbortNextPreviousIcon()"));

   tBool bMsgConsumed = FALSE; //Holds TRUE if the message is consumed
   (*oSearchNextPrevious_IconActive).mIsNextIconActive = 0;
   (*oSearchNextPrevious_IconActive).mIsPreviousIconActive = 0;
   oSearchNextPrevious_IconActive.MarkAllItemsModified();
   oSearchNextPrevious_IconActive.SendUpdate();

   return bMsgConsumed;
}


void TunerDataBindingHandler::vOnUpdateSearchNextPreviousIconMsg(bool iconState)
{
   (*oSearchNextPrevious_IconActive).mIsNextIconActive = iconState;
   (*oSearchNextPrevious_IconActive).mIsPreviousIconActive = iconState;
   oSearchNextPrevious_IconActive.MarkAllItemsModified();
   oSearchNextPrevious_IconActive.SendUpdate();
}


bool TunerDataBindingHandler::onCourierMessage(const GuiStartupFinishedUpdMsg& oMsg)
{
   ETG_TRACE_USR4(("TunerDataBindingHandler::GuiStartupFinishedUpdMsg()"));
   VehicleSrvClient::poGetInstance()->readLanguageOnStartUp();
   RadioSetting::poGetInstance()->InitDABPTYDisableForFirstTime();
   AMFMInfo::poGetInstance()->getRegionType();  //wst4kor-added for HD Radio button visiblity
   uint8 bandinfo = SrcChangeHandler::poGetInstance()->getsourceInfo();
   ETG_TRACE_USR4(("TunerDataBindingHandler::bandinfo %d", bandinfo));
   uint8 connectionstate = SrcChangeHandler::poGetInstance()->getConnectionState();
   TunerPLMsg(EV_MSG_TO_PL_VARIANT_TYPE, TunerDataPoolConfig::getInstance()->readRegionType(), "");
   //533683  ->bug id
   if ((bandinfo == SRC_TUNER_AM) && (connectionstate == STATE_CONNECTED))
   {
      UpdateSourceIcon((tU8)VIEW_AM);
   }
   else if ((bandinfo == SRC_TUNER_FM) && (connectionstate == STATE_CONNECTED))
   {
      UpdateSourceIcon((tU8)VIEW_FM1);
   }
   else if ((bandinfo == SRC_DAB) && (connectionstate == STATE_CONNECTED))
   {
      UpdateSourceIcon((tU8)VIEW_DAB);
   }
   else if ((bandinfo == SRC_TUNER_WX) && (connectionstate == STATE_CONNECTED))
   {
      UpdateSourceIcon((tU8)VIEW_WEATHERBAND);
   }
   else if ((bandinfo == SRC_TUNER_XM) && (connectionstate == STATE_CONNECTED))
   {
      SetSXMActive(true);
      SrcChangeHandler::poGetInstance()->setSXMStatus(true);
   }
   else
   {
      //Do Nothing
   }
   uint8 popupstatus = AnnoHandler::poGetInstance()->GetPopupStatus();
   ETG_TRACE_USR4(("TunerDataBindingHandler::GuiStartupFinishedUpdMsg::popupstatus %d", popupstatus));
   if ((popupstatus == TA_POPUP) && (!_isCabinOnly))
   {
      AnnoHandler::poGetInstance()->showHideAnnouncement(true, ANNOTYPE_TA); //TA
   }
   else if (popupstatus == PTY31_POPUP)
   {
      AnnoHandler::poGetInstance()->showHideAnnouncement(true, ANNOTYPE_PTY);   //PTY31
   }
   CheckDABMounted();
   CheckAMActivation();
   bool FMHDStatus = AMFMHDdata::poGetInstance()->getFMHDActiveStatus();
   bool AMHDStatus = AMFMHDdata::poGetInstance()->getAMHDActiveStatus();
   updateFMHDIndicatorStatus(FMHDStatus);
   updateAMHDIndicatorStatus(AMHDStatus);
   if (bandinfo != SRC_SPI_ENTERTAIN)
   {
      updateSpidataVisibilityonTunerGadget(false);
   }
   else
   {
      updateSpidataVisibilityonTunerGadget(true);
   }
   CheckSXMMounted();

   return false;
}


void TunerDataBindingHandler::CheckDABMounted()
{
   uint8 dabVal = TunerDataPoolConfig::getInstance()->readDABMountedValue();
   uint8 dabValSystem = TunerDataPoolConfig::getInstance()->readDABMountedSystemValue();
   ETG_TRACE_USR3(("TunerDataBindingHandler::CheckDABMounted()vdabVal %d", dabVal));
   ETG_TRACE_USR3(("TunerDataBindingHandler::CheckDABMounted() dabValSystem %d", dabValSystem));

   if ((TunerDataPoolConfig::getInstance()->readRegionType() == EUR_REGION))
   {
      VehicleSrvClient::poGetInstance()->updateDABAvailStatus(dabVal); //CAN Imp
      if ((dabVal == 1) && (dabValSystem == 1))
      {
         (*oVariantInfo).mDAB_Available = true;
         (*oVariantInfo).mDAB_Mounted = true;
      }
      else if ((dabVal == 1) && (dabValSystem == 0))
      {
         (*oVariantInfo).mDAB_Available = true;
         (*oVariantInfo).mDAB_Mounted = false;
      }
      else if (dabVal == 0)
      {
         (*oVariantInfo).mDAB_Available = false;
         (*oVariantInfo).mDAB_Mounted = false;
      }
   }
   oVariantInfo.MarkItemModified(ItemKey::VariantInfo::DAB_AvailableItem);
   oVariantInfo.MarkItemModified(ItemKey::VariantInfo::DAB_MountedItem);
   oVariantInfo.SendUpdate(true);
}


//WST4KOR
void TunerDataBindingHandler::CheckSXMMounted()
{
   uint8 sxmVal = TunerDataPoolConfig::getInstance()->readSXMMountedValue();
//	uint8 sxmValSystem = TunerDataPoolConfig::getInstance()->readSXMMountedSystemValue();
   ETG_TRACE_USR3(("TunerDataBindingHandler::CheckSXMMounted() SXMVal %d", sxmVal));
   //ETG_TRACE_USR3(("TunerDataBindingHandler::CheckSXMMounted() SXMValSystem %d", sxmValSystem));
   if (sxmVal == 1)
   {
      (*oVariantInfo).mSXM_Mounted = true;
   }
   else if (sxmVal == 0)
   {
      (*oVariantInfo).mSXM_Mounted = false;
   }
   oVariantInfo.MarkItemModified(ItemKey::VariantInfo::SXM_MountedItem);
   oVariantInfo.SendUpdate(true);
}


void TunerDataBindingHandler::CheckAMActivation()
{
   uint8 AMval = TunerDataPoolConfig::getInstance()->readAMActivatedValue();

   ETG_TRACE_USR3(("TunerDataBindingHandler::CheckAMMounted()AMval %d", AMval));

   if (AMval == AMFMActivated)
   {
      (*oVariantInfo).mAM_Activation = true;
      (*oVariantInfo).mFM_Activation = true;
      int AmFmVal = 1;
      VehicleSrvClient::poGetInstance()->updateAMFMAvailStatus(AmFmVal); //CAN Imp
   }
   else if (AMval == LWDeactivatedMWActivated)
   {
      (*oVariantInfo).mAM_Activation = true;
      (*oVariantInfo).mFM_Activation = false;
   }
   else if (AMval == AMFMDeactivated)
   {
      (*oVariantInfo).mAM_Activation = false;
      (*oVariantInfo).mFM_Activation = false;
      int AmFmVal = 0;
      VehicleSrvClient::poGetInstance()->updateAMFMAvailStatus(AmFmVal); //CAN Imp
   }
   oVariantInfo.MarkItemModified(ItemKey::VariantInfo::AM_ActivationItem);
   oVariantInfo.MarkItemModified(ItemKey::VariantInfo::FM_ActivationItem);
   oVariantInfo.SendUpdate(true);
}


//======HD Functions wst4kor======

/**
* Description     : Function called to update HD display
*
* @param[in]      : regionType: Region Type.
* @return         : void
*/
void TunerDataBindingHandler::updateFMHDLogo(uint8 HDlogoStatus)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::updateFMHDLogo(%d)", HDlogoStatus));
   (*oFmHDlogo).mFmHdLogoIndex = HDlogoStatus;
   oFmHDlogo.MarkItemModified(ItemKey::FmHdLogo::FmHdLogoIndexItem);
   oFmHDlogo.SendUpdate();
}


/**
* Description     : Function called to update HD display
*
* @param[in]      : regionType: Region Type.
* @return         : void
*/
void TunerDataBindingHandler::updateAMHDLogo(uint8 HDlogoStatus)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::updateAMHDLogo(%d)", HDlogoStatus));
   (*oAmHDlogo).mAmHdLogoIndex = HDlogoStatus;
   oAmHDlogo.MarkItemModified(ItemKey::AmHdLogo::AmHdLogoIndexItem);
   oAmHDlogo.SendUpdate();
}


/**
* Description     : Function called to update HD button visibility
*
* @param[in]      : HDButtonStatus: true if region is NAR or MEX .false for others
* @return         : void
*/
void TunerDataBindingHandler::setHdButtonVisibilityStatus(bool HDButtonStatus)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::setHdButtonVisibilityStatus() HDButtonStatus=%d", HDButtonStatus));
   (*oHDButtonVisibility).misHdVisible = HDButtonStatus;
   oHDButtonVisibility.MarkAllItemsModified();
   oHDButtonVisibility.SendUpdate();
}


/**
* Description     : Function called to update FM HD ON/OFF button.
*
* @param[in]      : HDdata: HD ON/OFF text.
* @param[in]      : isActive: HD ON/OFF highlight.
* @return         : void
*/
void TunerDataBindingHandler::updateFMHDIndicatorStatus(bool isActive)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::updateFMHDIndicatorStatus() isActive:%d", isActive));
   if (isActive == true)
   {
      std::string _radiotxt = "";
      updateRadioText(_radiotxt);
   }
   (*oFmHdOnOffInfo).mIsActive = isActive;
   oFmHdOnOffInfo.MarkItemModified(ItemKey::FmHdOnOff::IsActiveItem);
   oFmHdOnOffInfo.SendUpdate();
}


/**
* Description     : Function called to update AM HD ON/OFF button.
*
* @param[in]      : HDdata: HD ON/OFF text.
* @param[in]      : isActive: HD ON/OFF highlight.
* @return         : void
*/
void TunerDataBindingHandler::updateAMHDIndicatorStatus(bool isActive)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::updateAMHDIndicatorStatus(%d)", isActive));
   (*oAmHdOnOffInfo).mIsActive = isActive;
   oAmHdOnOffInfo.MarkItemModified(ItemKey::AmHdOnOff::IsActiveItem);
   oAmHdOnOffInfo.SendUpdate();
}


/**
* Description     : Function called to update HD Audio program visibility
*
* @param[in]      : HDButtonStatus: true if Hd has been acquired else false
* @return         : void
*/
void TunerDataBindingHandler::setHdAudioProgramVisibilityStatus(bool HDProgramvisibilityStatus)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::setHdAudioProgramVisibilityStatus %d", HDProgramvisibilityStatus));
   (*oHDAudioProgramVisibility).misHdProgramListVisible = HDProgramvisibilityStatus;
   oHDAudioProgramVisibility.MarkAllItemsModified();
   oHDAudioProgramVisibility.SendUpdate();
}


void TunerDataBindingHandler::updateAdvisoryData(uint8 enHDRStatus)
{
   tChar szStatusTextHD[3] = { 0 };
   sprintf(szStatusTextHD, "%%%d", enHDRStatus);
   ETG_TRACE_USR3(("TunerDataBindingHandler::updateAdvisoryData %d", enHDRStatus));

   if (enHDRStatus == 0)
   {
      (*oChannelInfo).mAdvisoryText = "";
   }
   else
   {
      (*oChannelInfo).mAdvisoryText = (Candera::String)szStatusTextHD;
   }
   oChannelInfo.MarkAllItemsModified();
   oChannelInfo.SendUpdate(true);
}


/**
* Description     : Function called to update HD MPS Audio Program databinding
* @param[in]      : active : the status for the switch index 0->blank 1->grey 2->white
* @return         : void
*/

void TunerDataBindingHandler::setHDProgSizeAndSwitch(uint8 audioProgram, float size, uint8 active)
{
   ETG_TRACE_USR4(("TunerDataBindingHandler::setHDProgSizeAndSwitch audioProgram:%d size:%f active:%d", audioProgram, size, active));
   switch (audioProgram)
   {
      case 0:
      {
         (*oHDAudioProgMps).mHD_Prog_MPS_switch = active;
         (*oHDAudioProgMps).mHD_Prog_MPS_size.SetX(size);
         (*oHDAudioProgMps).mHD_Prog_MPS_size.SetY(size);
         oHDAudioProgMps.MarkAllItemsModified();
         oHDAudioProgMps.SendUpdate();
         break;
      }
      case 1:
      {
         (*oHDAudioProgSps1).mHD_Prog_SPS1_switch = active;
         (*oHDAudioProgSps1).mHD_Prog_SPS1_size.SetX(size);
         (*oHDAudioProgSps1).mHD_Prog_SPS1_size.SetY(size);
         oHDAudioProgSps1.MarkAllItemsModified();
         oHDAudioProgSps1.SendUpdate();
         break;
      }
      case 2:
      {
         (*oHDAudioProgSps2).mHD_Prog_SPS2_switch = active;
         (*oHDAudioProgSps2).mHD_Prog_SPS2_size.SetX(size);
         (*oHDAudioProgSps2).mHD_Prog_SPS2_size.SetY(size);
         oHDAudioProgSps2.MarkAllItemsModified();
         oHDAudioProgSps2.SendUpdate();
         break;
      }
      case 3:
      {
         (*oHDAudioProgSps3).mHD_Prog_SPS3_switch = active;
         (*oHDAudioProgSps3).mHD_Prog_SPS3_size.SetX(size);
         (*oHDAudioProgSps3).mHD_Prog_SPS3_size.SetY(size);
         oHDAudioProgSps3.MarkAllItemsModified();
         oHDAudioProgSps3.SendUpdate();
         break;
      }
      case 4:
      {
         (*oHDAudioProgSps4).mHD_Prog_SPS4_switch = active;
         (*oHDAudioProgSps4).mHD_Prog_SPS4_size.SetX(size);
         (*oHDAudioProgSps4).mHD_Prog_SPS4_size.SetY(size);
         oHDAudioProgSps4.MarkAllItemsModified();
         oHDAudioProgSps4.SendUpdate();
         break;
      }
      case 5:
      {
         (*oHDAudioProgSps5).mHD_Prog_SPS5_switch = active;
         (*oHDAudioProgSps5).mHD_Prog_SPS5_size.SetX(size);
         (*oHDAudioProgSps5).mHD_Prog_SPS5_size.SetY(size);
         oHDAudioProgSps5.MarkAllItemsModified();
         oHDAudioProgSps5.SendUpdate();
         break;
      }

      case 6:
      {
         (*oHDAudioProgSps6).mHD_Prog_SPS6_switch = active;
         (*oHDAudioProgSps6).mHD_Prog_SPS6_size.SetX(size);
         (*oHDAudioProgSps6).mHD_Prog_SPS6_size.SetY(size);
         oHDAudioProgSps6.MarkAllItemsModified();
         oHDAudioProgSps6.SendUpdate();
         break;
      }
      case 7:
      {
         (*oHDAudioProgSps7).mHD_Prog_SPS7_switch = active;
         (*oHDAudioProgSps7).mHD_Prog_SPS7_size.SetX(size);
         (*oHDAudioProgSps7).mHD_Prog_SPS7_size.SetY(size);
         oHDAudioProgSps7.MarkAllItemsModified();
         oHDAudioProgSps7.SendUpdate();
         break;
      }
      default:
         break;
   }
}


void TunerDataBindingHandler::updateHDIconpresets(uint8 presetNo, uint8 bHDStatus, uint32 SPSvalue)
{
   ETG_TRACE_USR4(("TunerDataBindingHandler::updateHDIconpresets presetNo:%d  bHDStatus:%d string_value %d", presetNo, bHDStatus, SPSvalue));

   tChar szData[] = { 0, 0 };
   sprintf(szData, "%d", SPSvalue);
   switch (presetNo)
   {
      case 0:
      {
         (*oHDIconPreset1).mHD_Preset_1 = bHDStatus;
         if (SPSvalue != 0)
         {
            (*oHDIconPreset1).mHD_Preset_1_value = szData;
         }
         else
         {
            (*oHDIconPreset1).mHD_Preset_1_value = "";
         }
         oHDIconPreset1.MarkItemModified(ItemKey::HDIconPreset1::HD_Preset_1Item);
         oHDIconPreset1.MarkItemModified(ItemKey::HDIconPreset1::HD_Preset_1_valueItem);
         oHDIconPreset1.SendUpdate();
         break;
      }
      case 1:
      {
         (*oHDIconPreset2).mHD_Preset_2 = bHDStatus;
         if (SPSvalue != 0)
         {
            (*oHDIconPreset2).mHD_Preset_2_value = szData;
         }
         else
         {
            (*oHDIconPreset2).mHD_Preset_2_value = "";
         }
         oHDIconPreset2.MarkItemModified(ItemKey::HDIconPreset2::HD_Preset_2Item);
         oHDIconPreset2.MarkItemModified(ItemKey::HDIconPreset2::HD_Preset_2_valueItem);
         oHDIconPreset2.SendUpdate();
         break;
      }
      case 2:
      {
         (*oHDIconPreset3).mHD_Preset_3 = bHDStatus;
         if (SPSvalue != 0)
         {
            (*oHDIconPreset3).mHD_Preset_3_value = szData;
         }
         else
         {
            (*oHDIconPreset3).mHD_Preset_3_value = "";
         }
         oHDIconPreset3.MarkItemModified(ItemKey::HDIconPreset3::HD_Preset_3Item);
         oHDIconPreset3.MarkItemModified(ItemKey::HDIconPreset3::HD_Preset_3_valueItem);
         oHDIconPreset3.SendUpdate();
         break;
      }
      case 3:
      {
         (*oHDIconPreset4).mHD_Preset_4 = bHDStatus;
         if (SPSvalue != 0)
         {
            (*oHDIconPreset4).mHD_Preset_4_value = szData;
         }
         else
         {
            (*oHDIconPreset4).mHD_Preset_4_value = "";
         }
         oHDIconPreset4.MarkItemModified(ItemKey::HDIconPreset4::HD_Preset_4Item);
         oHDIconPreset4.MarkItemModified(ItemKey::HDIconPreset4::HD_Preset_4_valueItem);
         oHDIconPreset4.SendUpdate();
         break;
      }
      case 4:
      {
         (*oHDIconPreset5).mHD_Preset_5 = bHDStatus;
         if (SPSvalue != 0)
         {
            (*oHDIconPreset5).mHD_Preset_5_value = szData;
         }
         else
         {
            (*oHDIconPreset5).mHD_Preset_5_value = "";
         }
         oHDIconPreset5.MarkItemModified(ItemKey::HDIconPreset5::HD_Preset_5Item);
         oHDIconPreset5.MarkItemModified(ItemKey::HDIconPreset5::HD_Preset_5_valueItem);
         oHDIconPreset5.SendUpdate();
         break;
      }
      case 5:
      {
         (*oHDIconPreset6).mHD_Preset_6 = bHDStatus;
         if (SPSvalue != 0)
         {
            (*oHDIconPreset6).mHD_Preset_6_value = szData;
         }
         else
         {
            (*oHDIconPreset6).mHD_Preset_6_value = "";
         }
         oHDIconPreset6.MarkItemModified(ItemKey::HDIconPreset6::HD_Preset_6Item);
         oHDIconPreset6.MarkItemModified(ItemKey::HDIconPreset6::HD_Preset_6_valueItem);
         oHDIconPreset6.SendUpdate();
         break;
      }
      case 6:
      {
         (*oHDIconPreset7).mHD_Preset_7 = bHDStatus;
         if (SPSvalue != 0)
         {
            (*oHDIconPreset7).mHD_Preset_7_value = szData;
         }
         else
         {
            (*oHDIconPreset7).mHD_Preset_7_value = "";
         }
         oHDIconPreset7.MarkItemModified(ItemKey::HDIconPreset7::HD_Preset_7Item);
         oHDIconPreset7.MarkItemModified(ItemKey::HDIconPreset7::HD_Preset_7_valueItem);
         oHDIconPreset7.SendUpdate();
         break;
      }
      case 7:
      {
         (*oHDIconPreset8).mHD_Preset_8 = bHDStatus;
         if (SPSvalue != 0)
         {
            (*oHDIconPreset8).mHD_Preset_8_value = szData;
         }
         else
         {
            (*oHDIconPreset8).mHD_Preset_8_value = "";
         }
         oHDIconPreset8.MarkItemModified(ItemKey::HDIconPreset8::HD_Preset_8Item);
         oHDIconPreset8.MarkItemModified(ItemKey::HDIconPreset8::HD_Preset_8_valueItem);
         oHDIconPreset8.SendUpdate();
         break;
      }
      case 8:
      {
         (*oHDIconPreset9).mHD_Preset_9 = bHDStatus;
         if (SPSvalue != 0)
         {
            (*oHDIconPreset9).mHD_Preset_9_value = szData;
         }
         else
         {
            (*oHDIconPreset9).mHD_Preset_9_value = "";
         }
         oHDIconPreset9.MarkItemModified(ItemKey::HDIconPreset9::HD_Preset_9Item);
         oHDIconPreset9.MarkItemModified(ItemKey::HDIconPreset9::HD_Preset_9_valueItem);
         oHDIconPreset9.SendUpdate();
         break;
      }
      case 9:
      {
         (*oHDIconPreset10).mHD_Preset_10 = bHDStatus;
         if (SPSvalue != 0)
         {
            (*oHDIconPreset10).mHD_Preset_10_value = szData;
         }
         else
         {
            (*oHDIconPreset10).mHD_Preset_10_value = "";
         }
         oHDIconPreset10.MarkItemModified(ItemKey::HDIconPreset10::HD_Preset_10Item);
         oHDIconPreset10.MarkItemModified(ItemKey::HDIconPreset10::HD_Preset_10_valueItem);
         oHDIconPreset10.SendUpdate();
         break;
      }
      case 10:
      {
         (*oHDIconPreset11).mHD_Preset_11 = bHDStatus;
         if (SPSvalue != 0)
         {
            (*oHDIconPreset11).mHD_Preset_11_value = szData;
         }
         else
         {
            (*oHDIconPreset11).mHD_Preset_11_value = "";
         }
         oHDIconPreset11.MarkItemModified(ItemKey::HDIconPreset11::HD_Preset_11Item);
         oHDIconPreset11.MarkItemModified(ItemKey::HDIconPreset11::HD_Preset_11_valueItem);
         oHDIconPreset11.SendUpdate();
         break;
      }
      case 11:
      {
         (*oHDIconPreset12).mHD_Preset_12 = bHDStatus;
         if (SPSvalue != 0)
         {
            (*oHDIconPreset12).mHD_Preset_12_value = szData;
         }
         else
         {
            (*oHDIconPreset12).mHD_Preset_12_value = "";
         }
         oHDIconPreset12.MarkItemModified(ItemKey::HDIconPreset12::HD_Preset_12Item);
         oHDIconPreset12.MarkItemModified(ItemKey::HDIconPreset12::HD_Preset_12_valueItem);
         oHDIconPreset12.SendUpdate();
         break;
      }
      default:
         break;
   }
}


/**
* Description     : Function called to update Hd Info screen data.
*
* @param[in]      : radioMainData
* @return         : void
*/
void TunerDataBindingHandler::updateHdInfoData(NARRadioMainData& radioMainData)
{
   std::ostringstream hdInfoBuffer;
   std::string HdInfoTxt;
   //bool speedLockState = TunerCommonHandler::instance()->getSpeedLockStatus();

//#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2

   if (strcmp(radioMainData._trackName.c_str(), "") != 0)
   {
      hdInfoBuffer << "\\c1Song Name\n   \\c2 " << radioMainData._trackName.c_str();
   }
   if (strcmp(radioMainData._artistName.c_str(), "") != 0)
   {
      hdInfoBuffer << "\n\\c1Artist Name\n   \\c2 " << radioMainData._artistName.c_str();
   }
   if (strcmp(radioMainData._albumName.c_str(), "") != 0)
   {
      hdInfoBuffer << "\n\\c1Album\n   \\c2 " << radioMainData._albumName.c_str();
   }
   if (strcmp(radioMainData._HDInfoGenre.c_str(), "") != 0)
   {
      hdInfoBuffer << "\n\\c1Genre\n   \\c2 " << radioMainData._HDInfoGenre.c_str();
   }
   if ((strcmp(radioMainData._commerialPriceString.c_str(), "") != 0))
      //&& ((speedLockState == false) || (!TunerCommonHandler::isDualDisplay())))
   {
      hdInfoBuffer << "\n\\c1Commercial Price String\n   \\c2 " << radioMainData._commerialPriceString.c_str();
   }
   else if ((strcmp(radioMainData._commerialPriceString.c_str(), "") != 0))
      //&& ((speedLockState == true) && (TunerCommonHandler::isDualDisplay())))
   {
      hdInfoBuffer << "\n\\c1Commercial Price String\n   \\c2 ";
   }
   if ((strcmp(radioMainData._commerialValidity.c_str(), "") != 0))
      //&& ((speedLockState == false) || (!TunerCommonHandler::isDualDisplay())))
   {
      hdInfoBuffer << "\n\\c1Commercial Valid Until\n   \\c2 " << radioMainData._commerialValidity.c_str();
   }
   else if ((strcmp(radioMainData._commerialValidity.c_str(), "") != 0))
      //&& ((speedLockState == true) && (TunerCommonHandler::isDualDisplay())))
   {
      hdInfoBuffer << "\n\\c1Commercial Valid Until\n   \\c2 ";
   }
   if ((strcmp(radioMainData._commerialContactURL.c_str(), "") != 0))
      //&& ((speedLockState == true) || (!TunerCommonHandler::isDualDisplay())))
   {
      hdInfoBuffer << "\n\\c1Commercial Contact URL\n   \\c2 " << radioMainData._commerialContactURL.c_str();
   }
   else if ((strcmp(radioMainData._commerialContactURL.c_str(), "") != 0))
      //&& ((speedLockState == true) && (TunerCommonHandler::isDualDisplay())))
   {
      hdInfoBuffer << "\n\\c1Commercial Contact URL\n   \\c2 ";
   }
   if ((strcmp(radioMainData._commerialRecievedAs.c_str(), "") != 0))
      //&& ((speedLockState == true) || (!TunerCommonHandler::isDualDisplay())))
   {
      hdInfoBuffer << "\n\\c1Commercial Received as\n   \\c2 " << radioMainData._commerialRecievedAs.c_str();
   }
   else if ((strcmp(radioMainData._commerialRecievedAs.c_str(), "") != 0))
      //&& ((speedLockState == true) && (TunerCommonHandler::isDualDisplay())))
   {
      hdInfoBuffer << "\n\\c1Commercial Received as\n   \\c2 ";
   }
   if ((strcmp(radioMainData._commerialSellerName.c_str(), "") != 0))
      //	&& ((speedLockState == true) || (!TunerCommonHandler::isDualDisplay())))
   {
      hdInfoBuffer << "\n\\c1Commercial Name of Seller\n   \\c2 " << radioMainData._commerialSellerName.c_str();
   }
   else if ((strcmp(radioMainData._commerialSellerName.c_str(), "") != 0))
      //&& ((speedLockState == true) && (TunerCommonHandler::isDualDisplay())))
   {
      hdInfoBuffer << "\n\\c1Commercial Name of Seller\n   \\c2 ";
   }
   if ((strcmp(radioMainData._commerialDesc.c_str(), "") != 0))
      //&& ((speedLockState == false) || (!TunerCommonHandler::isDualDisplay())))
   {
      hdInfoBuffer << "\n\\c1Commercial Description\n   \\c2 " << radioMainData._commerialDesc.c_str();
   }
   else if ((strcmp(radioMainData._commerialDesc.c_str(), "") != 0))
      //	&& ((speedLockState == true) && (TunerCommonHandler::isDualDisplay())))
   {
      hdInfoBuffer << "\n\\c1Commercial Description\n   \\c2 ";
   }
   if (strcmp(radioMainData._commentLanguage.c_str(), "") != 0)
   {
      hdInfoBuffer << "\n\\c1Comment Language\n   \\c2 " << radioMainData._commentLanguage.c_str();
   }
   if (strcmp(radioMainData._commentShortDec.c_str(), "") != 0)
   {
      hdInfoBuffer << "\n\\c1Comment Short Description\n  \\c2 " << radioMainData._commentShortDec.c_str();
   }
   if (strcmp(radioMainData._commentActualTxt.c_str(), "") != 0)
   {
      hdInfoBuffer << "\n\\c1Comment Actual Text\n   \\c2 " << radioMainData._commentActualTxt.c_str();
   }
   if ((strcmp(radioMainData._slogan.c_str(), "") != 0))
      //&& ((speedLockState == true) || (!TunerCommonHandler::isDualDisplay())))
   {
      hdInfoBuffer << "\n\\c1Slogan / Service Display Name\n  \\c2 " << radioMainData._slogan.c_str();
   }
   else if ((strcmp(radioMainData._slogan.c_str(), "") != 0))
      //&& ((speedLockState == true) && (TunerCommonHandler::isDualDisplay())))
   {
      hdInfoBuffer << "\n\\c1Slogan / Service Display Name\n  \\c2 ";
   }

//#endif
   //Advisory
   if (hdInfoBuffer.str().length() == 0)
   {
      if (strlen(_radioMainText.c_str()) != 0)
      {
         hdInfoBuffer << _radioMainText;
      }
      else
      {
         hdInfoBuffer << TunerUtility::getSecureString(Candera::String(0x4E929E55));

         /* #ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2
         _radioMainText = "No info available";
         hdInfoBuffer << _radioMainText;
         			hdInfoBuffer << TunerUtility::getSecureString(Candera::String(TextId_AUDIO_9119));
         #elif defined(VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1_R)
         			_radioMainText = "No info available";
         			hdInfoBuffer << _radioMainText;
         #endif*/
      }
   }
   HdInfoTxt = hdInfoBuffer.str();
   std::string secureString = TunerUtility::getSecureString((*_rdsRadioTextInfo).mText);
   if (strcmp(secureString.c_str(), HdInfoTxt.c_str()) != 0)
   {
      (*_rdsRadioTextInfo).mText = HdInfoTxt.c_str();
      _rdsRadioTextInfo.MarkItemModified(ItemKey::RdsRadioText::TextItem);
      _rdsRadioTextInfo.SendUpdate();
   }
}


//For HD Test Mode Screens wst4kor
void TunerDataBindingHandler::updateHdDTMScreenInfo(std::string hdStatus, std::string sisStatus, std::string audStatus, std::string blendStatus)
{
   bool isUpdated = false;
   if ((*oDtmHD).mHDStatus != hdStatus.c_str())
   {
      (*oDtmHD).mHDStatus = hdStatus.c_str();
      oDtmHD.MarkItemModified(ItemKey::DtmHD::HDStatusItem);
      isUpdated = true;
   }
   if ((*oDtmHD).mSISStatus != sisStatus.c_str())
   {
      (*oDtmHD).mSISStatus = sisStatus.c_str();
      oDtmHD.MarkItemModified(ItemKey::DtmHD::SISStatusItem);
      isUpdated = true;
   }
   if ((*oDtmHD).mAUDStatus != audStatus.c_str())
   {
      (*oDtmHD).mAUDStatus = audStatus.c_str();
      oDtmHD.MarkItemModified(ItemKey::DtmHD::AUDStatusItem);
      isUpdated = true;
   }
   if ((*oDtmHD).mBlendStatus != blendStatus.c_str())
   {
      (*oDtmHD).mBlendStatus = blendStatus.c_str();
      oDtmHD.MarkItemModified(ItemKey::DtmHD::BlendStatusItem);
      isUpdated = true;
   }
   if (isUpdated)
   {
      oDtmHD.SendUpdate();
   }
}


void TunerDataBindingHandler::updateHdDTMAudiblePhysicalTunerInfo(std::string HD10Freq, std::string FST)
{
   bool isUpdated = false;
   if ((*oDtmHD).mHD10Freq != HD10Freq.c_str())
   {
      (*oDtmHD).mHD10Freq = HD10Freq.c_str();
      oDtmHD.MarkItemModified(ItemKey::DtmHD::HD10FreqItem);
      isUpdated = true;
   }
   if ((*oDtmHD).mHD10FST != FST.c_str())
   {
      (*oDtmHD).mHD10FST = FST.c_str();
      oDtmHD.MarkItemModified(ItemKey::DtmHD::HD10FSTItem);
      isUpdated = true;
   }
   if (isUpdated)
   {
      oDtmHD.SendUpdate();
   }
}


void TunerDataBindingHandler::updateHdDTMSecPhysicalTunerInfo(std::string HD05Freq, std::string HD05FST)
{
   bool isUpdated = false;
   if ((*oDtmHD).mHD05Freq != HD05Freq.c_str())
   {
      (*oDtmHD).mHD05Freq = HD05Freq.c_str();
      oDtmHD.MarkItemModified(ItemKey::DtmHD::HD05FreqItem);
      isUpdated = true;
   }
   if ((*oDtmHD).mHD05FST != HD05FST.c_str())
   {
      (*oDtmHD).mHD05FST = HD05FST.c_str();
      oDtmHD.MarkItemModified(ItemKey::DtmHD::HD05FSTItem);
      isUpdated = true;
   }
   if (isUpdated)
   {
      oDtmHD.SendUpdate();
   }
}


void TunerDataBindingHandler::updateDTMHdAudibleTunerCdNo(std::string Hd10cd_No)
{
   if ((*oDtmHD).mHD05CdNo != Hd10cd_No.c_str())
   {
      (*oDtmHD).mHD05CdNo = Hd10cd_No.c_str();
      oDtmHD.MarkItemModified(ItemKey::DtmHD::HD05CdNoItem);
      oDtmHD.SendUpdate();
   }
}


void TunerDataBindingHandler::updateDTMHdInfo(std::string hdCallSign, std::string cdNo, std::string audQuality)
{
   bool isUpdated = false;
   if ((*oDtmHD).mHDCallSign != hdCallSign.c_str())
   {
      (*oDtmHD).mHDCallSign = hdCallSign.c_str();
      oDtmHD.MarkItemModified(ItemKey::DtmHD::HDCallSignItem);
      isUpdated = true;
   }
   if ((*oDtmHD).mHD10CdNo != cdNo.c_str())
   {
      (*oDtmHD).mHD10CdNo = cdNo.c_str();
      oDtmHD.MarkItemModified(ItemKey::DtmHD::HD10CdNoItem);
      isUpdated = true;
   }
   if ((*oDtmHD).mHD10AudQual != audQuality.c_str())
   {
      (*oDtmHD).mHD10AudQual = audQuality.c_str();
      oDtmHD.MarkItemModified(ItemKey::DtmHD::HD10AudQualItem);
      isUpdated = true;
   }
   if (isUpdated)
   {
      oDtmHD.SendUpdate();
   }
}


void TunerDataBindingHandler::updateDTMHdTransitionCount(std::string transitionCount)
{
   if ((*oDtmHD).mTransitions != transitionCount.c_str())
   {
      (*oDtmHD).mTransitions = transitionCount.c_str();
      oDtmHD.MarkItemModified(ItemKey::DtmHD::TransitionsItem);
      oDtmHD.SendUpdate();
   }
}


void TunerDataBindingHandler::setHDProgSizeTestmode(uint8 TestaudioProgram, uint8 Testactive)
{
   ETG_TRACE_USR4(("TunerDataBindingHandler::TestaudioProgram:%d, Testactive:%d", TestaudioProgram, Testactive));
   switch (TestaudioProgram)
   {
      case 0:
         (*oTMHDAudioMPS).mTMHD_Prog_MPS_switch = Testactive;
         oTMHDAudioMPS.MarkItemModified(ItemKey::TMHDAudioMPS::TMHD_Prog_MPS_switchItem);
         oTMHDAudioMPS.SendUpdate();
         break;
      case 1:
         (*oTMHDAudioSPS1).mTMHD_Prog_SPS1_switch = Testactive;
         oTMHDAudioSPS1.MarkItemModified(ItemKey::TMHDAudioSPS1::TMHD_Prog_SPS1_switchItem);
         oTMHDAudioSPS1.SendUpdate();
         break;
      case 2:
         (*oTMHDAudioSPS2).mTMHD_Prog_SPS2_switch = Testactive;
         oTMHDAudioSPS2.MarkItemModified(ItemKey::TMHDAudioSPS2::TMHD_Prog_SPS2_switchItem);
         oTMHDAudioSPS2.SendUpdate();
         break;
      case 3:
         (*oTMHDAudioSPS3).mTMHD_Prog_SPS3_switch = Testactive;
         oTMHDAudioSPS3.MarkItemModified(ItemKey::TMHDAudioSPS3::TMHD_Prog_SPS3_switchItem);
         oTMHDAudioSPS3.SendUpdate();
         break;
      case 4:
         (*oTMHDAudioSPS4).mTMHD_Prog_SPS4_switch = Testactive;
         oTMHDAudioSPS4.MarkItemModified(ItemKey::TMHDAudioSPS4::TMHD_Prog_SPS4_switchItem);
         oTMHDAudioSPS4.SendUpdate();
         break;
      case 5:
         (*oTMHDAudioSPS5).mTMHD_Prog_SPS5_switch = Testactive;
         oTMHDAudioSPS5.MarkItemModified(ItemKey::TMHDAudioSPS5::TMHD_Prog_SPS5_switchItem);
         oTMHDAudioSPS5.SendUpdate();
         break;
      case 6:
         (*oTMHDAudioSPS6).mTMHD_Prog_SPS6_switch = Testactive;
         oTMHDAudioSPS6.MarkItemModified(ItemKey::TMHDAudioSPS6::TMHD_Prog_SPS6_switchItem);
         oTMHDAudioSPS6.SendUpdate();
         break;
      case 7:
         (*oTMHDAudioSPS7).mTMHD_Prog_SPS7_switch = Testactive;
         oTMHDAudioSPS7.MarkItemModified(ItemKey::TMHDAudioSPS7::TMHD_Prog_SPS7_switchItem);
         oTMHDAudioSPS7.SendUpdate();
         break;
      default:
         break;
   }
}


void TunerDataBindingHandler::setHDProgAvailableTestmode(uint8 TestaudioProgramAvailable, uint8 Testavailable)
{
   ETG_TRACE_USR4(("TunerDataBindingHandler::setHDProgAvailableTestmode TestaudioProgramAvailable:%d, Testavailable:%d", TestaudioProgramAvailable, Testavailable));
   switch (TestaudioProgramAvailable)
   {
      case 0:
         (*oTMHDAudioMPS).mTMHD_Prog_MPS_available = Testavailable;
         oTMHDAudioMPS.MarkItemModified(ItemKey::TMHDAudioMPS::TMHD_Prog_MPS_availableItem);
         oTMHDAudioMPS.SendUpdate();
         break;
      case 1:
         (*oTMHDAudioSPS1).mTMHD_Prog_SPS1_available = Testavailable;
         oTMHDAudioSPS1.MarkItemModified(ItemKey::TMHDAudioSPS1::TMHD_Prog_SPS1_availableItem);
         oTMHDAudioSPS1.SendUpdate();
         break;
      case 2:
         (*oTMHDAudioSPS2).mTMHD_Prog_SPS2_available = Testavailable;
         oTMHDAudioSPS2.MarkItemModified(ItemKey::TMHDAudioSPS2::TMHD_Prog_SPS2_availableItem);
         oTMHDAudioSPS2.SendUpdate();
         break;
      case 3:
         (*oTMHDAudioSPS3).mTMHD_Prog_SPS3_available = Testavailable;
         oTMHDAudioSPS3.MarkItemModified(ItemKey::TMHDAudioSPS3::TMHD_Prog_SPS3_availableItem);
         oTMHDAudioSPS3.SendUpdate();
         break;
      case 4:
         (*oTMHDAudioSPS4).mTMHD_Prog_SPS4_available = Testavailable;
         oTMHDAudioSPS4.MarkItemModified(ItemKey::TMHDAudioSPS4::TMHD_Prog_SPS4_availableItem);
         oTMHDAudioSPS4.SendUpdate();
         break;
      case 5:
         (*oTMHDAudioSPS5).mTMHD_Prog_SPS5_available = Testavailable;
         oTMHDAudioSPS5.MarkItemModified(ItemKey::TMHDAudioSPS5::TMHD_Prog_SPS5_availableItem);
         oTMHDAudioSPS5.SendUpdate();
         break;
      case 6:
         (*oTMHDAudioSPS6).mTMHD_Prog_SPS6_available = Testavailable;
         oTMHDAudioSPS6.MarkItemModified(ItemKey::TMHDAudioSPS6::TMHD_Prog_SPS6_availableItem);
         oTMHDAudioSPS6.SendUpdate();
         break;
      case 7:
         (*oTMHDAudioSPS7).mTMHD_Prog_SPS7_available = Testavailable;
         oTMHDAudioSPS7.MarkItemModified(ItemKey::TMHDAudioSPS7::TMHD_Prog_SPS7_availableItem);
         oTMHDAudioSPS7.SendUpdate();
         break;
      default:
         break;
   }
}


/**
* Description     : Function called to update Category name/PTY
*
* @param[in]      : category: Current PTY/Category name to be updated to GUI.
* @return         : void
*/
void TunerDataBindingHandler::setCategory(std::string& category)
{
   ETG_TRACE_USR4(("TunerDataBindingHandler::setCategory category ==%s", category.c_str()));
   std::string secureString = TunerUtility::getSecureString((*oChannelInfo).mCategory);
   if (strcmp(secureString.c_str(), category.c_str()) != 0)
   {
      (*oChannelInfo).mCategory = category.c_str();
      oChannelInfo.MarkItemModified(ItemKey::ChannelInfo::CategoryItem);
      oChannelInfo.SendUpdate();
   }
}


/**
* Description     : Function called to update FM Main screen
*
* @param[in]      : radioMainData: The structure holds Main screen data to be updated to GUI.
* @return         : void
*/

void TunerDataBindingHandler::updatePSDData(NARRadioMainData& radioMainData)
{
   ETG_TRACE_USR4(("TunerDataBindingHandler::updatePSDData"));
   bool UpdateData = false;
   Courier::UInt8 hdDataAvailability = static_cast<Courier::UInt8>(TUNER_HD_DATA_NONE);
   std::string secureString = TunerUtility::getSecureString((*oChannelInfo).mAlbum);
   if (strcmp(secureString.c_str(), radioMainData._albumName.c_str()) != 0)
   {
      if (radioMainData._albumName.size() > 0)
      {
         hdDataAvailability += static_cast<Courier::UInt8>(TUNER_HD_DATA_ALBUM);
      }
      ETG_TRACE_USR4(("TunerDataBindingHandler::updatePSDData Album == %s", radioMainData._albumName.c_str()));
      (*oChannelInfo).mAlbum = radioMainData._albumName.c_str();
      oChannelInfo.MarkItemModified(ItemKey::ChannelInfo::AlbumItem);
      UpdateData = true;
   }
   secureString = TunerUtility::getSecureString((*oChannelInfo).mArtist);
   if (strcmp(secureString.c_str(), radioMainData._artistName.c_str()) != 0)
   {
      if (radioMainData._artistName.size() > 0)
      {
         hdDataAvailability += static_cast<Courier::UInt8>(TUNER_HD_DATA_ARTIST);
      }
      ETG_TRACE_USR4(("TunerDataBindingHandler::updatePSDData Artist ==%s", radioMainData._artistName.c_str()));
      (*oChannelInfo).mArtist = radioMainData._artistName.c_str();
      oChannelInfo.MarkItemModified(ItemKey::ChannelInfo::ArtistItem);
      UpdateData = true;
   }
   secureString = TunerUtility::getSecureString((*oChannelInfo).mTrack);
   if (strcmp(secureString.c_str(), radioMainData._trackName.c_str()) != 0)
   {
      ETG_TRACE_USR4(("TunerDataBindingHandler::updatePSDData Track %s", radioMainData._trackName.c_str()));
      (*oChannelInfo).mTrack = radioMainData._trackName.c_str();
      oChannelInfo.MarkItemModified(ItemKey::ChannelInfo::TrackItem);
      UpdateData = true;
   }
   if (UpdateData)
   {
      (*oChannelInfo).mArtistAlbumSwitch = hdDataAvailability;
      oChannelInfo.MarkItemModified(ItemKey::ChannelInfo::ArtistAlbumSwitchItem);
      oChannelInfo.SendUpdate();
   }
   if (!UpdateData)
   {
      ETG_TRACE_USR4(("TunerDataBindingHandler::No PSD Data Available"));
      (*oChannelInfo).mTrack = radioMainData._trackName.c_str();
      (*oChannelInfo).mArtist = radioMainData._artistName.c_str();
      oChannelInfo.MarkItemModified(ItemKey::ChannelInfo::TrackItem);
      oChannelInfo.SendUpdate();
   }
}


/**
* Description     : Function called to update BallGame Mode active message to GUI.
*
* @param[in]      : BallgameMode Status
* @return         : void
*/
void TunerDataBindingHandler::setBallGameMode(bool& status)
{
   if ((*oChannelInfo).mIsBallGameModeVisible != status)
   {
      (*oChannelInfo).mIsBallGameModeVisible = status;
      oChannelInfo.MarkItemModified(ItemKey::ChannelInfo::IsBallGameModeVisibleItem);
      oChannelInfo.SendUpdate();
   }
}


/**
* Description     : Function called to update Album/Cover art .
* There are 2 types of images that can be available for a channel
* 1. Station Logo
* 2. CoverArt
*
* And there are 2 place holders for these images
* 1. Left corner
* 2. Screen Background
*
* Screen Background image will ALWAYS display stationLogo. if StationLogo is unavailable then show defaultLogo.
*
* The changes happen only to the "Left Corner" based on the following conditions.
* Case 1 : Both Station logo and CoverArt are available
* Requirement : Show station logo for 5 seconds and then show Coverart
*
* Case 2 : Only cover art is available
* Requirement : Show cover art
*
* @param[in]      : u8ImageData: Image Data to be updated to GUI.
* @param[in]      : u32ImageDataSize : Image Size.
* @return         : void
*/

void TunerDataBindingHandler::updateHDAlbumArt(const ::std::vector< uint8 >& u8ImageData, uint32 u32ImageDataSize, bool displayStatus)
{
   Candera::Bitmap::SharedPointer bmp;
   _hdAlbumArtVisibilityStatus = displayStatus;
   _u8HdImageData = u8ImageData;
   if (_hdAlbumArtVisibilityStatus && u32ImageDataSize > 0)
   {
      ETG_TRACE_USR4(("AMFMInfoDataBindingHandler::u32ImageDataSize==%d", u32ImageDataSize));

      bmp = ImageLoader::loadBitmapData(const_cast<uint8*>(&(u8ImageData)[0]), u32ImageDataSize);

      if (bmp != NULL)
      {
         _isHDAlbumArtSet = true;
         (*_amfmLogoImage).mAMFMLogo = ImageLoader::createImage(bmp);
         clearDefaultAlbumArt();
      }
   }
   else
   {
      _isHDAlbumArtSet = false;
      if (!_imageData.empty())
      {
         bmp = ImageLoader::loadBitmapFile(_imageData.c_str());
         if (bmp != NULL)
         {
            (*_amfmLogoImage).mAMFMLogo = ImageLoader::createImage(bmp);
            clearDefaultAlbumArt();
         }
      }
      else
      {
         ETG_TRACE_USR4(("AMFMInfoDataBindingHandler::updateHDAlbumArt_else_Case_Executed"));
         clearAlbumArt();
         //(*_amfmLogoImage).mAMFMLogo = ImageLoader::getAssetBitmapImage("TunerModule#Images#HD_FM_AM_NEW_BITMAPS#Icon_albumBmp");
         (*_amfmLogoImage).mAMFMDefalutLogo = ImageLoader::getAssetBitmapImage("TunerModule#Images#HomeScreen_latest_images#Icon_FM_album");
      }
   }
   ETG_TRACE_USR4(("AMFMInfoDataBindingHandler::updateHDAlbumArt _hdStationLogoTimer status %d", _hdStationLogoTimer.isActive()));
   if (!_hdStationLogoTimer.isActive())
   {
      (*_amfmLogoImage).mAMFMLogoVisibility = true;
      _amfmLogoImage.MarkAllItemsModified();
      _amfmLogoImage.SendUpdate();
   }
}


/**
* Description     : Function called to clear previous album art
*
* @return         : void
*/
void TunerDataBindingHandler::clearAlbumArt()
{
   Candera::Bitmap::SharedPointer bmp;
   bmp = ImageLoader::loadBitmapData(0, 0);
   (*_amfmLogoImage).mAMFMLogo = ImageLoader::createImage(bmp);
}


void TunerDataBindingHandler::clearDefaultAlbumArt()
{
   Candera::Bitmap::SharedPointer bmp;
   bmp = ImageLoader::loadBitmapData(0, 0);
   (*_amfmLogoImage).mAMFMDefalutLogo = ImageLoader::createImage(bmp);
}


void TunerDataBindingHandler::UpdateSourceIcon(tU8 Previous_View)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::UpdateSourceIcon(%d)", Previous_View));
   switch (Previous_View)
   {
      case VIEW_FM1:
      {
         (*oTunerSourceIconData).mFM1Active = true;
         (*oTunerSourceIconData).mAMActive = false;
         (*oTunerSourceIconData).mDABActive = false;
         (*oTunerSourceIconData).mWXActive = false;
         (*oTunerSourceIconData).mSXMActive = false;
         /*(*oTunerSourceIconData).mFM2Active = false;
         (*oTunerSourceIconData).mFMASTActive = false;
         (*oTunerSourceIconData).mAMASTActive = false;
         (*oTunerSourceIconData).mDABASTActive = false;*/
         break;
      }
      case VIEW_WEATHERBAND:
      {
         (*oTunerSourceIconData).mFM1Active = false;
         (*oTunerSourceIconData).mAMActive = false;
         (*oTunerSourceIconData).mDABActive = false;
         (*oTunerSourceIconData).mWXActive = true;
         //SetSXMActive(false);
         (*oTunerSourceIconData).mSXMActive = false;

         /*(*oTunerSourceIconData).mFM2Active = false;
         (*oTunerSourceIconData).mFMASTActive = false;
         (*oTunerSourceIconData).mAMASTActive = false;
         (*oTunerSourceIconData).mDABASTActive = false;*/
         break;
      }
      case VIEW_FM2:
      {
         /*(*oTunerSourceIconData).mFM2Active = true;
         (*oTunerSourceIconData).mAMActive = false;
         	 (*oTunerSourceIconData).mDABActive = false;
         	 (*oTunerSourceIconData).mFM1Active = false;
               (*oTunerSourceIconData).mFMASTActive = false;
               (*oTunerSourceIconData).mAMASTActive = false;
               (*oTunerSourceIconData).mDABASTActive = false;*/
         break;
      }
      case VIEW_FM_AST:
      {
         /*(*oTunerSourceIconData).mFMASTActive = true;
         (*oTunerSourceIconData).mAMActive = false;
         	 (*oTunerSourceIconData).mDABActive = false;
               (*oTunerSourceIconData).mFM1Active = false;
               (*oTunerSourceIconData).mFM2Active = false;
               (*oTunerSourceIconData).mAMASTActive = false;
               (*oTunerSourceIconData).mDABASTActive = false;*/
         break;
      }
      case VIEW_AM:
      {
         (*oTunerSourceIconData).mAMActive = true;
         (*oTunerSourceIconData).mFM1Active = false;
         (*oTunerSourceIconData).mDABActive = false;
         (*oTunerSourceIconData).mWXActive = false;
         //SetSXMActive(false);
         (*oTunerSourceIconData).mSXMActive = false;

         /*(*oTunerSourceIconData).mFM2Active = false;
           (*oTunerSourceIconData).mFMASTActive = false;
           (*oTunerSourceIconData).mAMASTActive = false;
           (*oTunerSourceIconData).mDABASTActive = false; */
         break;
      }
      case VIEW_AM_AST:
      {
         /*(*oTunerSourceIconData).mAMASTActive = true;
         (*oTunerSourceIconData).mFM1Active = false;
         (*oTunerSourceIconData).mFM2Active = false;
         (*oTunerSourceIconData).mFMASTActive = false;
         (*oTunerSourceIconData).mAMActive = false;
         (*oTunerSourceIconData).mDABActive = false;
         (*oTunerSourceIconData).mDABASTActive = false;
         sm_gSwitchState = EN_TUN_BANK_AM_AST;*/
         break;
      }
      case VIEW_DAB:
      {
         (*oTunerSourceIconData).mDABActive = true;
         (*oTunerSourceIconData).mFM1Active = false;
         (*oTunerSourceIconData).mAMActive = false;
         //SetSXMActive(false);
         (*oTunerSourceIconData).mSXMActive = false;
         (*oTunerSourceIconData).mWXActive = false;
         /*(*oTunerSourceIconData).mFM2Active = false;
           (*oTunerSourceIconData).mFMASTActive = false;
           (*oTunerSourceIconData).mAMASTActive = false;
           (*oTunerSourceIconData).mDABASTActive = false;*/

         break;
      }
      case VIEW_DAB_AST:
      {
         /*(*oTunerSourceIconData).mDABASTActive = true;
         (*oTunerSourceIconData).mFM1Active = false;
         (*oTunerSourceIconData).mFM2Active = false;
         (*oTunerSourceIconData).mFMASTActive = false;
         (*oTunerSourceIconData).mAMActive = false;
         (*oTunerSourceIconData).mAMASTActive = false;
         (*oTunerSourceIconData).mDABActive = false;*/
         break;
      }
      default:
      {
         //BUG ID:533719
         ETG_TRACE_USR3(("TunerDataBindingHandler::default case to handle highlight "));
         (*oTunerSourceIconData).mDABASTActive = false;
         (*oTunerSourceIconData).mFM1Active = false;
         (*oTunerSourceIconData).mFM2Active = false;
         (*oTunerSourceIconData).mFMASTActive = false;
         (*oTunerSourceIconData).mAMActive = false;
         (*oTunerSourceIconData).mAMASTActive = false;
         (*oTunerSourceIconData).mDABActive = false;
         (*oTunerSourceIconData).mWXActive = false;
         //SetSXMActive(false);
         (*oTunerSourceIconData).mSXMActive = false;
         break;
      }
   }
   oTunerSourceIconData.MarkAllItemsModified();
   oTunerSourceIconData.SendUpdate(true);
   ETG_TRACE_USR3(("TunerDataBindingHandler::Updatedone"));
}


void TunerDataBindingHandler::HomeScreenFMHDLogo(uint8 HomeScreenHDlogo)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::HomeScreenFMHDLogo(%d)", HomeScreenHDlogo));
   (*oHomeScreenFmHdLogo).mHomeScreen_FmHdLogo = HomeScreenHDlogo;
   oHomeScreenFmHdLogo.MarkItemModified(ItemKey::HomeScreenFmHdLogo::HomeScreen_FmHdLogoItem);
   oHomeScreenFmHdLogo.SendUpdate();
}


void TunerDataBindingHandler::HomeScreenAMHDLogo(uint8 HomeScreenHDlogo)
{
   ETG_TRACE_USR3(("TunerDataBindingHandler::HomeScreenAMHDLogo(%d)", HomeScreenHDlogo));
   (*oHomeScreenAmHdLogo).mHomeScreen_AmHdLogo = HomeScreenHDlogo;
   oHomeScreenAmHdLogo.MarkItemModified(ItemKey::HomeScreenAmHdLogo::HomeScreen_AmHdLogoItem);
   oHomeScreenAmHdLogo.SendUpdate();
}


/************************************************************************
*FUNCTION: 		SPIDataVisibility
*DESCRIPTION: 	Handles SPI visibility of icon and text.
*PARAMETER:     bool val
*
*
*RETURNVALUE:void
*
************************************************************************/
void TunerDataBindingHandler::updateSpidataVisibilityonTunerGadget(bool val)
{
   ETG_TRACE_USR4(("TunerDataBindingHandler::SPIDataVisibility() %d", val));
   if (val == true)
   {
      (*oSPIVisibilityDataTuner).mSPIIsvisibleTuner = true;
      (*oSPIVisibilityDataTuner).mSPIIsvisibleTunerFMTuner = false;
   }
   else
   {
      (*oSPIVisibilityDataTuner).mSPIIsvisibleTuner = false;
      (*oSPIVisibilityDataTuner).mSPIIsvisibleTunerFMTuner = true;
   }
   oSPIVisibilityDataTuner.MarkAllItemsModified();
   oSPIVisibilityDataTuner.SendUpdate();
}


/**
* Description     : Function called to update SXM Mode active message to GUI.
*
* @param[in]      : SXM Active Status
* @return         : void
*/
void TunerDataBindingHandler::SetSXMActive(bool active)
{
   (*oTunerSourceIconData).mAMActive = false;
   (*oTunerSourceIconData).mFM1Active = false;
   (*oTunerSourceIconData).mFM2Active = false;
   (*oTunerSourceIconData).mFMASTActive = false;
   (*oTunerSourceIconData).mAMASTActive = false;
   (*oTunerSourceIconData).mDABActive = false;
   (*oTunerSourceIconData).mDABASTActive = false;
   (*oTunerSourceIconData).mWXActive = false;
   (*oTunerSourceIconData).mSXMActive = active;
   oTunerSourceIconData.MarkAllItemsModified();
   oTunerSourceIconData.SendUpdate(true);
}


} // namespace Core
} // namespace
