/**************************************************************************************
* @file         : DabInfoCourierMsgHandler.cpp
* @author       : ECG5-nma4kor
* @addtogroup   : Tuner HALL
* @brief        : Class contains utility functions to handle courier messages required for DAB data.
* @copyright    : (c) 2018-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "util/StrUtf8.h"
#include "hmi_trace_if.h"
#include "AppHmi_TunerMessages.h"
#include "DabInfoCourierMsgHandler.h"
#include "DabInfo.h"
#include "DABStationList.h"
#include "App/Core/AMFMInfo/AMFMStationListTimer.h"
#include "AppHmi_MasterBase/AudioInterface/AudioDefines.h"
#include "AMFMTestMode.h"
#include "PresetListHandling.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_TUNER_DM
#include "trcGenProj/Header/DabInfoCourierMsgHandler.cpp.trc.h"
#endif

namespace App {
namespace Core {
/**
* Description     : Constructor of class DabInfoCourierMsgHandler
*/
DabInfoCourierMsgHandler::DabInfoCourierMsgHandler(DabInfo* dabInfo, DABStationList* dabStationList)
   : _dabInfo(dabInfo),
     _dabList(dabStationList)
{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR3(("DabInfoCourierMsgHandler:: Constructor called"));

   m_u8ServiceLinkingValue = 0;
}


/**
* Description     : Destructor of class DabInfoCourierMsgHandler
*/
DabInfoCourierMsgHandler::~DabInfoCourierMsgHandler()
{
   _dabList = NULL;
   _dabInfo = NULL;

   ETG_I_UNREGISTER_FILE();
}


/**
* Callback to handle Courier Message Courier::StartupMsg Request
* @param [in] : StartupMsg
* @return     : bool true if message is consumed
*/
bool DabInfoCourierMsgHandler::onCourierMessage(const Courier::StartupMsg& /*oMsg*/)
{
   ETG_TRACE_USR3(("DabInfoCourierMsgHandler::StartupMsg()"));
   FocusChangedUpdMsg::Subscribe(Courier::ComponentType::Model, Courier::ComponentType::View);
   return false;
}


/************************************************************************
*FUNCTION: 		DABServiceReqMsg

*RETURNVALUE: 	Bool
*HISTORY:
*25.07.13 Narahari M S
*Rev 1.0 Initial Revision.
************************************************************************/
bool DabInfoCourierMsgHandler::onCourierMessage(const DABServiceReqMsg& corfoMsg)
{
   ETG_TRACE_USR3(("DabInfoCourierMsgHandler::DABServiceReqMsg()"));

   bool bMsgConsumed = FALSE; //Holds TRUE if the message is consumed
   const DABServiceReqMsg* copoTunerReqMsg =
      Courier::message_cast<const DABServiceReqMsg*>(&corfoMsg);

   if (NULL != copoTunerReqMsg)
   {
      bMsgConsumed = TRUE;
      tU8 u8ServiceLinkingvalue = copoTunerReqMsg->GetService_Value();
      switch (u8ServiceLinkingvalue)
      {
         case TUN_INC:
         {
            ETG_TRACE_USR3(("DabInfoCourierMsgHandler::DABServiceReqMsg==Seek Request Next"));
            DABSource = 1;
            _dabList->onSeekService(TUN_INC);
            ETG_TRACE_USR3(("DabInfoCourierMsgHandler::DABServiceReqMsg==Seek Request Next"));
         }
         break;
         case TUN_DEC:
         {
            ETG_TRACE_USR3(("DabInfoCourierMsgHandler::DABServiceReqMsg==Seek Request Previous"));
            DABSource = 1;
            _dabList->onSeekService(TUN_DEC);
            ETG_TRACE_USR3(("DabInfoCourierMsgHandler::DABServiceReqMsg==Seek Request Previous"));
         }
         break;
         default:
         {
            ETG_TRACE_USR3(("ERROR: DABServiceReqMsg is INVALID"));
         }
         break;
      }
   }
   return bMsgConsumed;
}


/************************************************************************
*FUNCTION: 		DABServiceReqMsg

*RETURNVALUE: 	Bool
*HISTORY:
*25.07.13 Narahari M S
*Rev 1.0 Initial Revision.
************************************************************************/
bool DabInfoCourierMsgHandler::onCourierMessage(const DABServiceLinkReqMsg& corfoMsg)
{
   ETG_TRACE_USR3(("DabInfoCourierMsgHandler::DABServiceLinkReqMsg()"));

   bool bMsgConsumed = FALSE; //Holds TRUE if the message is consumed
   const DABServiceLinkReqMsg* copoTunerReqMsg =
      Courier::message_cast<const DABServiceLinkReqMsg*>(&corfoMsg);

   if (NULL != copoTunerReqMsg)
   {
      bMsgConsumed = TRUE;
      AMFMTestMode::poGetInstance()->toggleLinkingMode();
   }
   return bMsgConsumed;
}


/************************************************************************
*FUNCTION: 		DABCLevelReqMsg

*RETURNVALUE: 	Bool
*HISTORY:
*25.07.13 Narahari M S
*Rev 1.0 Initial Revision.
************************************************************************/
bool DabInfoCourierMsgHandler::onCourierMessage(const DABCLevelReqMsg& corfoMsg)
{
   ETG_TRACE_USR3(("DabInfoCourierMsgHandler::DABCLevelReqMsg()"));

   bool bMsgConsumed = FALSE; //Holds TRUE if the message is consumed
   const DABCLevelReqMsg* copoTunerReqMsg =
      Courier::message_cast<const DABCLevelReqMsg*>(&corfoMsg);
   uint8 operation = TUN_DIRECTION_INVALID;
   if (NULL != copoTunerReqMsg)
   {
      bMsgConsumed = TRUE;
      tU8 u8Concealmentvalue = copoTunerReqMsg->GetCLevel_Value();
      switch (u8Concealmentvalue)
      {
         case TUN_INC:
         {
            ETG_TRACE_USR3(("DabInfoCourierMsgHandler::DABCLevelReqMsg==Conceal Inc"));
            operation = TUN_DIRECTION_UP;
         }
         break;
         case TUN_DEC:
         {
            ETG_TRACE_USR3(("DabInfoCourierMsgHandler::DABCLevelReqMsg==Conceal Dec"));
            operation = TUN_DIRECTION_DOWN;
         }
         break;
         default:
         {
            ETG_TRACE_USR3(("ERROR: DABCLevelReqMsg is INVALID"));
         }
         break;
      }
   }
   AMFMTestMode::poGetInstance()->updateConcealmentValue(AMFMTestMode::poGetInstance()->_concealmentLevel, operation);
   return true;
}


/************************************************************************
*FUNCTION: 		DABTMSeekReqMsg

*RETURNVALUE: 	Bool
*HISTORY:
*25.07.13 Narahari M S
*Rev 1.0 Initial Revision.
************************************************************************/
bool DabInfoCourierMsgHandler::onCourierMessage(const DABTMSeekReqMsg& corfoMsg)
{
   ETG_TRACE_USR3(("DabInfoCourierMsgHandler::DABTMSeekReqMsg()"));

   bool bMsgConsumed = FALSE; //Holds TRUE if the message is consumed
   const DABTMSeekReqMsg* copoTunerReqMsg = Courier::message_cast<const DABTMSeekReqMsg*>(&corfoMsg);

   if (NULL != copoTunerReqMsg)
   {
      bMsgConsumed = TRUE;
      tU8 u8ServiceReqvalue = copoTunerReqMsg->GetDABTMSeekMode();
      switch (u8ServiceReqvalue)
      {
         case SEEK_DAB_TM_SERVICE_NEXT:
         {
            ETG_TRACE_USR3(("DabInfoCourierMsgHandler::DABTMSeekReqMsg== Inc"));
            _dabList->onSeekReqService(SEEK_NEXT);
         }
         break;
         case SEEK_DAB_TM_SERVICE_PREVIOUS:
         {
            ETG_TRACE_USR3(("DabInfoCourierMsgHandler::DABTMSeekReqMsg== Dec"));
            _dabList->onSeekReqService(SEEK_PREVIOUS);
         }
         break;
         case SEEK_DAB_TM_ENSEMBLE_NEXT:
         {
            ETG_TRACE_USR3(("DabInfoCourierMsgHandler::DABTMSeekReqMsg== Inc"));
            _dabList->onSeekEnsemble(SEEK_NEXT);
         }
         break;
         case SEEK_DAB_TM_ENSEMBLE_PREVIOUS:
         {
            ETG_TRACE_USR3(("DabInfoCourierMsgHandler::DABTMSeekReqMsg== Dec"));
            _dabList->onSeekEnsemble(SEEK_PREVIOUS);
         }
         break;
         default:
         {
            ETG_TRACE_USR3(("ERROR: DABTMSeekReqMsg is INVALID"));
         }
         break;
      }
   }
   return bMsgConsumed;
}


} // namespace Core
} // namespace App
