/**************************************************************************************
 * @file         : ATStationInfo.cpp
 * @author       : ECV- pbe5kor
 * @addtogroup   : Tuner HALL
 * @brief        : Class contains necessary business logic required to handle AM FM AtStationinfo data.
 * @copyright    : (c) 2017-2019 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 **************************************************************************************/


#include "hall_std_if.h"
#include "ATStationInfo.h"
#include "App/Core/AMFMInfo/AMFMInfoIF.h"
#include "App/Core/PresetListHandler/PresetListHandling.h"
#include "AMFMInfo.h"
#include "App/Core/Tuner_Defines.h"
#include "App/datapool/TunerDataPoolConfig.h"
#include "App/Core/AMFMInfo/AMFMStationIF.h"
#include "AMFMTestMode.h"
#include "../AnnouncementHandler/AnnoHandler.h"
#include "../VehicleSrvClient/VehicleSrvClient.h"  //CAN Imp

/*#include "App/Core/AMFMInfo/AMFMStationIF.h"
#include "App/Core/TestMode/TestModeIF.h"
#include "App/Core/Tuner_Defines.h"
#include "App/datapool/TunerDataPoolConfig.h"
#include "App/Core/Utility/TunerCommonHandler.h" */

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_TUNER_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_TUNER
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_Tuner_"
#define ETG_I_FILE_PREFIX                 App::Core::ATStationInfo::
#include "trcGenProj/Header/ATStationInfo.cpp.trc.h"
#endif


namespace App {
namespace Core {

using namespace ::MIDW_TUNER_FI;
using namespace ::midw_tuner_fi_types;

DEFINE_CLASS_LOGGER_AND_LEVEL("App/Core/Tuner_AMFM", ATStationInfo, Info);


ATStationInfo* ATStationInfo::m_poATStationInfo = NULL;

/**
 * Description     : Destructor of class ATStationInfo
 */
ATStationInfo::~ATStationInfo()
{
   _AMFMStationListIF = NULL;
   _AMFMInfoIF = NULL;
   _AMFMHDdata = NULL;
}


/**
 * Description     : Constructor of class ATStationInfo
 */
ATStationInfo::ATStationInfo()
   : _tunerFiProxy(MIDW_TUNER_FIProxy::createProxy("tunerFiPort", *this)),
     _AMFMHDdata(NULL)
{
   ETG_TRACE_USR3(("ATStationInfo: Constructor"));
   if (_tunerFiProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _tunerFiProxy->getPortName());
   }

   _AMFMInfoIF = AMFMInfo::poGetInstance();
   _AMFMStationListIF = AMFMStationList::poGetInstance()
                        ETG_I_REGISTER_FILE();
   if (TunerCommonHandler::instance()->isHDFeatureEnabled())     //wst4kor
   {
      _AMFMHDdata = AMFMHDdata::poGetInstance()
                    ETG_I_REGISTER_FILE();
   }
   _AMFMTestMode = AMFMTestMode::poGetInstance();
}


// ServiceAvailableIF implementation
void ATStationInfo::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   _tunerFiProxy->sendFID_TUN_G_ATSTATIONINFORelUpRegAll();
}


void ATStationInfo::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR3(("ATStationInfo: onAvailable"));
   if ((proxy == _tunerFiProxy) && (_tunerFiProxy.get()))
   {
      _tunerFiProxy->sendFID_TUN_G_ATSTATIONINFOUpReg(*this);
   }
}


/**
 * Description     : Request from Framework to start registering for properties. This request means GUI is ready now.
 *
 * @param[in]      : proxy : reference to the base class of all proxies
 * @param[in]      : stateChange : represents a state transition of a service
 * @return         : void
 */
void ATStationInfo::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR3(("ATStationInfo: registerProperties"));
   if ((proxy == _tunerFiProxy) && (_tunerFiProxy.get()))
   {
      _tunerFiProxy->sendFID_TUN_G_ATSTATIONINFOUpReg(*this);
   }
}


/**
 * Description     : Request from Framework to start de-registering for properties.
 *
 * @param[in]      : proxy : reference to the base class of all proxies
 * @param[in]      : stateChange : represents a state transition of a service
 * @return         : void
 */
void ATStationInfo::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR3(("ATStationInfo: deregisterProperties"));
   if ((proxy == _tunerFiProxy) && (_tunerFiProxy.get()))
   {
      _tunerFiProxy->sendFID_TUN_G_ATSTATIONINFORelUpRegAll();
   }
}


/**
 * Description     : Virtual function implemented to get update of StationInfo Status
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : status : The status message of the property "StationInfo"
 * @return         : void
 */
void ATStationInfo::onFID_TUN_G_ATSTATIONINFOStatus(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUN_G_ATSTATIONINFOStatus >& status)
{
   ETG_TRACE_USR3(("ATStationInfo: onFID_TUN_G_ATSTATIONINFOStatus"));

   tU8 u8CurrentBand = status->getAtstationinfo().getE8Band();
   ETG_TRACE_USR3(("ATStationInfo: u8CurrentBand %d", u8CurrentBand));

   T_e8_Band e8Band = (T_e8_Band)u8CurrentBand;
   //AMFMInfo::poGetInstance()->vSetCurrentBand(e8Band);

   uint16 presetListID = status->getAtstationinfo().getU16PresetListID();
   ETG_TRACE_USR3(("ATStationInfo: presetListID %d", presetListID));

   uint32 currentFreq = status->getAtstationinfo().getU32Frequency();
   bool AFbitStatus = status->getAtstationinfo().getU8AtStationInfoBits().getALTERNATE_FREQUENCY_BIT();
   std::string psName = status->getAtstationinfo().getSPSName();
   //std::string PsName = std::string psName;
   AnnoHandler::poGetInstance()->setPSName(psName, currentFreq);
   uint16 presetID = status->getAtstationinfo().getU16StaticListElementID();
   ETG_TRACE_USR3(("ATStationInfo: presetListID %d", status->getAtstationinfo().getU16StaticListElementID()));
   int16 configListElemID = (int)(status->getAtstationinfo().getU16ConfigListElementID());
   configListElemID = configListElemID - 1;

   uint16 PI = status->getAtstationinfo().getU16PI();
   //  std::string HDcallSign = status->getAtstationinfo().getSHDCallSign();
   tU8 u8TPStatus = status->getAtstationinfo().getE8AtStationTPinfo();

   bool HDDataDecoding = status->getAtstationinfo().getB8HDStationInfoBits().getHD_DATA_DECODING();
   bool HDAudioDecoding = status->getAtstationinfo().getB8HDStationInfoBits().getHD_AUDIO_DECODING();
   uint8 activeHDAudioPrg = status->getAtstationinfo().getE8ActiveHDAudPrgm();
   bool BallGameMode = status->getAtstationinfo().getB8HDStationInfoBits().getHD_BALLGAME_MODE();
   uint8 HDStationGenre = status->getAtstationinfo().getU8HDStationGenre();
   std::string HDcallSign = status->getAtstationinfo().getSHDCallSign();

   T_e8_AtStationTPinfo e8AtStationTPinfo = (T_e8_AtStationTPinfo)u8TPStatus;
   if (T_e8_AtStationTPinfo__SELF_TP == e8AtStationTPinfo)
   {
      TunerDataBindingHandler::getDataBindingHandlerInstance()->TunerPLMsg(EV_MSG_TO_PL_TP, 1, "");
      TunerDataBindingHandler::getDataBindingHandlerInstance()->TunerPLMsg(EV_MSG_TO_PL_TP_CURRENT_STATION, 1, "");
      VehicleSrvClient::poGetInstance()->updateTPStatus(1); //CAN Imp
   }
   else
   {
      TunerDataBindingHandler::getDataBindingHandlerInstance()->TunerPLMsg(EV_MSG_TO_PL_TP, 0, "");
      TunerDataBindingHandler::getDataBindingHandlerInstance()->TunerPLMsg(EV_MSG_TO_PL_TP_CURRENT_STATION, 0, "");
      VehicleSrvClient::poGetInstance()->updateTPStatus(0); //CAN Imp
   }
   uint8 ptyIndex = status->getAtstationinfo().getU8PTY();
   if ((_AMFMInfoIF != NULL) && (AMFMInfo::poGetInstance()->e8GetCurrentBand() != T_e8_Band__TUN_MSTR_BAND_DAB))
   {
      ETG_TRACE_USR3(("ATStationInfo: Print data [%d %d %d %s]", u8CurrentBand, currentFreq, presetID, psName.c_str()));
      ETG_TRACE_USR3(("PSNAME==: %s", psName.c_str()));
      _AMFMInfoIF->setAtStationInfoRadioMainData(u8CurrentBand, currentFreq, presetID, psName);
   }
   else
   {
      ETG_TRACE_USR3(("ATStationInfo: Pointer is Null"));
   }
   ETG_TRACE_USR4(("isHDFeatureEnabled():%d", TunerCommonHandler::instance()->isHDFeatureEnabled()));
   if ((NULL != AMFMHDdata::poGetInstance()) && (TunerCommonHandler::instance()->isHDFeatureEnabled()))
   {
      ETG_TRACE_USR4(("TunerCommonHandler::instance()->isHDFeatureEnabled():%d", TunerCommonHandler::instance()->isHDFeatureEnabled()));

      _AMFMHDdata->setHDInfoData(HDcallSign, activeHDAudioPrg, currentFreq, u8CurrentBand, psName, HDStationGenre);
      _AMFMHDdata->setHDData(HDDataDecoding, HDAudioDecoding);
      _AMFMHDdata->setBallGameModeData(BallGameMode);
   }
   if (NULL != _AMFMStationListIF)
   {
      _AMFMStationListIF->setAtStationInfoDataToAMFMStationList(status);
   }
   if (NULL != AMFMTestMode::poGetInstance())
   {
      ETG_TRACE_USR3(("Calling setSTMData function"));
      AMFMTestMode::poGetInstance()->setSTMData(status);
   }
   /*Update the preset focus*/
   if (u8CurrentBand == 0)
   {
      ETG_TRACE_USR3(("ATStationInfo: presetcurrentID for FM %d", status->getAtstationinfo().getU16StaticListElementID()));
      TunerDataBindingHandler::getDataBindingHandlerInstance()->vUpdate_FM_PresetFocused(status->getAtstationinfo().getU16StaticListElementID());
   }
   else if (u8CurrentBand == 2)
   {
      ETG_TRACE_USR3(("ATStationInfo: presetcurrentID for AM %d", status->getAtstationinfo().getU16StaticListElementID()));
      TunerDataBindingHandler::getDataBindingHandlerInstance()->vUpdate_AM_PresetFocused(status->getAtstationinfo().getU16StaticListElementID());
   }
}


/**
 * Description     : Virtual function implemented to get update of StationInfo Error
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : error : The error message of 'StationInfo'
 * @return         : void
 */
void ATStationInfo::onFID_TUN_G_ATSTATIONINFOError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUN_G_ATSTATIONINFOError >& /*error*/)
{
}


} // namespace Core
} // namespace App
