/**************************************************************************************
* @file         : AMFMStationList.cpp
* @author       : ECV2-Deepak
* @addtogroup   : Tuner HALL
* @brief        : Class contains necessary business logic required to handle AM FM List data.
* @copyright    : (c) 2017-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/
#include "hall_std_if.h"
#include "AMFMStationList.h"
#include "List/Data/ListDataProviderDistributor.h"
#include "CourierTunnelService/CourierMessageReceiverStub.h"
#include "App/Core/Utility/TunerUtility.h"
#include "App/datapool/TunerDataPoolConfig.h"
#include "App/Core/AMFMInfo/AMFMStationListTimer.h"
#include "App/Core/Tuner_Defines.h"
#include "App/Core/LanguageDefines.h"
#include "AppHmi_MasterBase/AudioInterface/AudioDefines.h"
#include "App/Core/DABInfo/DABStationList.h"
#include "App/Core/Utility/TunerCommonHandler.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_TUNER_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_TUNER
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_Tuner_"
#define ETG_I_FILE_PREFIX                 App::Core::AMFMStationList::
#include "trcGenProj/Header/AMFMStationList.cpp.trc.h"
#endif

using namespace ::MIDW_TUNER_FI;
using namespace ::midw_tuner_fi_types;
using namespace ::MIDW_TUNERMASTER_FI;
using namespace ::midw_tunermaster_fi_types;


namespace App {
namespace Core {

AMFMStationList* AMFMStationList::m_poAMFMStationList = NULL;

/**
 * Description     : Destructor of AMFMStationList
 */
AMFMStationList::~AMFMStationList()
{
   ETG_I_UNREGISTER_FILE();
   // _tunInfoDataBinder = NULL; removed for coverity may 4

   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_FM_LIST);
   ListRegistry::s_getInstance().removeListImplementation(LISTID_FM);
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_HD_INFO_LIST);
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_AM_LIST);
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_AM_LIST);
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_FM_MENU);
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_AM_MENU);
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_AREASELECTION);
#endif
   clearFMList();
   _tunerFiProxy.reset();
   _tunerMasterFiProxy.reset();
}


/**
 * Description     : Constructor of AMFMStationList
 */
AMFMStationList::AMFMStationList()
   : _tunerFiProxy(MIDW_TUNER_FIProxy::createProxy("tunerFiPort", *this))
   , _tunerMasterFiProxy(MIDW_TUNERMASTER_FIProxy::createProxy("tunermasterFiPort", *this))
   , _stationListElemCount(0)
   , _selectedElemID(0)
   , _listEnter(false)
   , _isListDataRequested(false)
   , _isFMStationListOpened(false)
   , _AMStationListElemCount(0)
   , _scanStatus(false)
   , _AMListSize(0)
   , _FMListSize(0)
   , _AtStationFreq(0)
   , _band(0)
     //wst4kor
   , _regionType(0)
   , _activeHDAudioPrg(HD_AUDIO_PROGRAM_UNAVAIABLE)
   , _HDDataDecodingStatus(false)
   , _HDDataDecodingBit(false)
   , _HDAudioDecodingBit(false)
   , _HDAudioDecodingStatus(false)
   , _AMHDActiveStatus(false)
   , _FMHDActiveStatus(false)
   , _HDStationID(0)

{
   if (_tunerFiProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _tunerFiProxy->getPortName());
   }

   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR3(("AMFMStationList : Constructor"));
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_FM_LIST, this);
   ListRegistry::s_getInstance().addListImplementation(LISTID_FM, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_HD_INFO_LIST, this);
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_AM_LIST, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_FM_MENU, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_AM_MENU, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_AREASELECTION, this);
#endif

   _tunInfoDataBinder = TunerDataBindingHandler::getDataBindingHandlerInstance();//new TunerDataBindingHandler();

   for (uint i = 0; i < MAX_HD_AUDIO_PROGRAMS; i++)
   {
      _HdProgramsAvailable[i] = false;
   }
}


/**
 * Description     : Request from Framework to start registering for properties.
 *                   This request means GUI is ready now.
 *
 * @param[in]      : proxy : reference to the base class of all proxies
 * @param[in]      : stateChange : represents a state transition of a service
 * @return         : void
 */
void AMFMStationList::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR3(("AMFMStationList : registerProperties"));
   if ((proxy == _tunerFiProxy) && (_tunerFiProxy.get()))
   {
      _tunerFiProxy->sendFID_TUN_G_GET_CONFIG_LISTUpReg(*this);
      _tunerFiProxy->sendFID_TUN_G_ATSEEKUpReg(*this);

#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1_R
      _tunerFiProxy->sendFID_TUN_G_LISTUPDATE_NOTIFICATIONUpReg(*this);
#endif
   }
}


/**
 * Description     : Request from Framework to start de-registering for properties.
 *
 * @param[in]      : proxy : reference to the base class of all proxies
 * @param[in]      : stateChange : represents a state transition of a service
 * @return         : void
 */
void AMFMStationList::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& /*proxy*/, const asf::core::ServiceStateChange& /*stateChange*/)
{
}


/**
 * Description     : Request from Framework to start registering for properties.
 *
 * @param[in]      : proxy : reference to the base class of all proxies
 * @param[in]      : stateChange : represents a state transition of a service
 * @return         : void
 */

void AMFMStationList::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR3(("AMFMStationList::onAvailable()"));

   if ((proxy == _tunerFiProxy) && (_tunerFiProxy.get()))
   {
      _tunerFiProxy->sendFID_TUN_G_GET_CONFIG_LISTUpReg(*this);
      _tunerFiProxy->sendFID_TUN_G_ATSEEKUpReg(*this);
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1_R
      _tunerFiProxy->sendFID_TUN_G_LISTUPDATE_NOTIFICATIONUpReg(*this);
#endif
   }
}


/**
 * Description     : Request from Framework to start de-registering for properties.
 *
 * @param[in]      : proxy : reference to the base class of all proxies
 * @param[in]      : stateChange : represents a state transition of a service
 * @return         : void
 */
void AMFMStationList::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR3(("AMFMStationList : onUnavailable"));
   if ((proxy == _tunerFiProxy) && (_tunerFiProxy.get()))
   {
      _tunerFiProxy->sendFID_TUN_G_GET_CONFIG_LISTRelUpRegAll();
      _tunerFiProxy->sendFID_TUN_G_ATSEEKRelUpRegAll();
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1_R
      _tunerFiProxy->sendFID_TUN_G_LISTUPDATE_NOTIFICATIONRelUpRegAll();
#endif
   }
}


/**
 * Description     : Funtion creates Lists corresponding to the list IDs that it receives
 *
 * @param[in]      : listDataInfo : unique ID for lists
 * @return         : tSharedPtrDataProvider
 */
tSharedPtrDataProvider AMFMStationList::getListDataProvider(const ListDateProviderReqMsg& listDataInfo)
{
   ETG_TRACE_USR3(("AMFMStationList : getListDataProvider"));
   tSharedPtrDataProvider ptr = tSharedPtrDataProvider();
   ETG_TRACE_USR3(("AMFMStationList : list ID %d", listDataInfo.GetListId()));
   switch (listDataInfo.GetListId())
   {
      case LIST_ID_FM_LIST:
      case LISTID_FM:
      {
         ptr = getTunerFMStationList();
         break;
      }

      case LIST_ID_HD_INFO_LIST:
      {
         ptr = getTunerHDInfoList();
         break;
      }
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2
      case LIST_ID_AM_LIST:
      {
         ptr = getTunerAMStationList();
         break;
      }
      case LIST_ID_FM_MENU:
      {
         ptr = getTunerFmMenuList();
         break;
      }
      case LIST_ID_AM_MENU:
      {
         ptr = getTunerAmMenuList();
         break;
      }
      case LIST_ID_AREASELECTION:
      {
         ptr = getAreaSelectionList();
      }
      break;
#endif
      default:
      {
         break;
      }
   }
   return ptr;
}


//wst4kor(For HD Info list)
/**
* Description     : Function implemented to populate HD Info List
*
* @return         : tSharedPtrDataProvider
*/
tSharedPtrDataProvider AMFMStationList::getTunerHDInfoList()
{
   const char* const HD_INFO_LIST_ITEM = "HDInfoTemplateDark";
   ListDataProviderBuilder listBuilder(LIST_ID_HD_INFO_LIST, HD_INFO_LIST_ITEM);
   std::string buffer = ":";
   std::string  trackName = _radioMainData._trackName;
   listBuilder.AddItem(0, 0, "HDInfoProgramTemplate").AddData(TunerUtility::getSecureString(Candera::String("Song Name")).c_str()).AddData(trackName.c_str());
   std::string  artistName = _radioMainData._artistName;
   listBuilder.AddItem(1, 0, "HDInfoTemplateDark").AddData(TunerUtility::getSecureString(Candera::String("Artist Name")).c_str()).AddData(artistName.c_str());
   std::string  albumName = _radioMainData._albumName;
   listBuilder.AddItem(2, 0, "HDInfoProgramTemplate").AddData(TunerUtility::getSecureString(Candera::String("Album")).c_str()).AddData(albumName.c_str());;
   std::string commentActualTxt = _radioMainData._commentActualTxt;
   listBuilder.AddItem(3, 0, "HDInfoTemplateDark").AddData(TunerUtility::getSecureString(Candera::String("Comment")).c_str()).AddData(commentActualTxt.c_str());
   std::string  commentShortDec = _radioMainData._commentShortDec;
   listBuilder.AddItem(4, 0, "HDInfoProgramTemplate").AddData(TunerUtility::getSecureString(Candera::String("Commercial")).c_str()).AddData(commentShortDec.c_str());
   std::string slogan = _radioMainData._slogan;
   listBuilder.AddItem(5, 0, "HDInfoTemplateDark").AddData(TunerUtility::getSecureString(Candera::String("Slogan")).c_str()).AddData(slogan.c_str());

   return listBuilder.CreateDataProvider();
}


/**
 * Description     : Function implemented to populate FM Station list
 *
 * @return         : tSharedPtrDataProvider
 */
tSharedPtrDataProvider AMFMStationList::getTunerFMStationList()
{
   ETG_TRACE_USR3(("AMFMStationList : getTunerFMStationList"));
   const char* const FM_STATION_LIST_ITEM = "Layer_StationListEntry_1";
   ListDataProviderBuilder listBuilder(LISTID_FM, FM_STATION_LIST_ITEM);
   std::string strBuffer;

   StationInfoSourceData item;
   for (unsigned int itr = 0; itr < _stationListElemCount; itr++)
   {
      /* ETG_TRACE_USR3(("AMFMStationList : array of frequencies %s", AMFMStationList::poGetInstance()->frequencyarray[itr].c_str()));
         ETG_TRACE_USR3(("AMFMStationList : Initial Freq %s", TunerDataBindingHandler::getDataBindingHandlerInstance()->getfrequencyvalue().c_str()));
        if (strcmp(frequencyarray[itr].c_str(), TunerDataBindingHandler::getDataBindingHandlerInstance()->getfrequencyvalue().c_str()) == 0)
        {
          ETG_TRACE_USR3(("Icon highlight,index = %d", itr));
          item.mIcon_station = 1;
        }
        else
        {
          ETG_TRACE_USR3(("Icon not  highlight ,index = %d", itr));
          item.mIcon_station = 0;
        }
      */
      ETG_TRACE_USR3(("AMFMStationList : Selected Index %d, frequencies %d", _fmStationList[itr]._elementSelected, _fmStationList[itr]._frequency));
      item.mIcon_station = _fmStationList[itr]._elementSelected;

      strBuffer.clear();

      if (TunerCommonHandler::instance()->isHDFeatureEnabled())
      {
         strBuffer.append(_fmStationList[itr]._psName.c_str()).append(_fmStationList[itr]._hdFMAudioPrg.c_str()).append(_fmStationList[itr]._Reg.c_str());
         listBuilder.AddItem(itr, 0, FM_STATION_LIST_ITEM).
         AddData(strBuffer.c_str()).
         AddData(_fmStationList[itr]._preset.c_str()).
         AddData(_fmStationList[itr]._TPName.c_str()).
         AddDataBindingUpdater<StationInfoSourceDataBindingSource>(item);
      }
      else
      {
         strBuffer.append(_fmStationList[itr]._psName.c_str()).append(_fmStationList[itr]._Reg.c_str());
         listBuilder.AddItem(itr, 0, FM_STATION_LIST_ITEM).
         AddData(strBuffer.c_str()).
         AddData(_fmStationList[itr]._preset.c_str()).
         AddData(_fmStationList[itr]._TPName.c_str()).
         AddDataBindingUpdater<StationInfoSourceDataBindingSource>(item);
      }
   }
   return listBuilder.CreateDataProvider();
}


/**
 * Description     : Virtual function implemented to get update of FID_TUN_S_GET_CONFIG_LIST Error
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : error : The error message of 'FID_TUN_S_GET_CONFIG_LIST'
 * @return         : void
 */
void AMFMStationList::onFID_TUN_S_GET_CONFIG_LISTError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_S_GET_CONFIG_LISTError >& /*error*/)
{
}


/**
 * Description     : Virtual function implemented to get update of FID_TUN_S_GET_CONFIG_LIST Result
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : result : The Result message of the property "FID_TUN_S_GET_CONFIG_LIST"
 * @return         : void
 */
void AMFMStationList::onFID_TUN_S_GET_CONFIG_LISTResult(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_S_GET_CONFIG_LISTResult >& /*result*/)
{
}


/**
 * Description     : Virtual function implemented to get update of Direct AtFrequency Error
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : error : The error message of 'Direct AtFrequency'
 * @return         : void
 */
void AMFMStationList::onFID_TUN_S_DIRECT_FREQUENCYError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUN_S_DIRECT_FREQUENCYError >& /*error*/)
{
}


/**
 * Description     : Virtual function implemented to get the result of method start DirectFrequency
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : result : The result message of the method "DirectFrequency"
 * @return         : void
 */
void AMFMStationList::onFID_TUN_S_DIRECT_FREQUENCYResult(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUN_S_DIRECT_FREQUENCYResult >& /*result*/)
{
}


/**
 * Description     : Virtual function implemented to get update of FID_TUN_S_STATIONLIST_EXIT Error
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : error : The error message of 'FID_TUN_S_STATIONLIST_EXIT'
 * @return         : void
 */
void AMFMStationList::onFID_TUN_S_STATIONLIST_EXITError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_S_STATIONLIST_EXITError >& /*error*/)
{
}


/**
 * Description     : Virtual function implemented to get update of FID_TUN_S_STATIONLIST_EXIT Result
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : result : The Result message of the property "FID_TUN_S_STATIONLIST_EXIT"
 * @return         : void
 */
void AMFMStationList::onFID_TUN_S_STATIONLIST_EXITResult(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_S_STATIONLIST_EXITResult >& /*result*/)
{
}


/**
 * Description     : Virtual function implemented to get update of FID_TUN_S_SEL_LIST_ELEM Error
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : error : The error message of 'FID_TUN_S_SEL_LIST_ELEM'
 * @return         : void
 */
void AMFMStationList::onFID_TUN_S_SEL_LIST_ELEMError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_S_SEL_LIST_ELEMError >& /*error*/)
{
}


/**
 * Description     : Virtual function implemented to get update of FID_TUN_S_SEL_LIST_ELEM Result
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : result : The Result message of the property "FID_TUN_S_SEL_LIST_ELEM"
 * @return         : void
 */
void AMFMStationList::onFID_TUN_S_SEL_LIST_ELEMResult(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_S_SEL_LIST_ELEMResult >& /*result*/)
{
}


/**
 * Description     : Virtual function implemented to get update of FID_TUN_G_GET_CONFIG_LIST Error
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : error : The error message of 'FID_TUN_G_GET_CONFIG_LIST'
 * @return         : void
 */
void AMFMStationList::onFID_TUN_G_GET_CONFIG_LISTError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUN_G_GET_CONFIG_LISTError >& /*error*/)
{
}


/**
* Description     : Virtual function implemented to get update of updated station selection Error
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNERMASTER_FIProxy'
* @param[in]      : error : The error message of 'FID_TUNMSTR_S_STATION_SELECTION'
* @return         : void
*/
void AMFMStationList::onFID_TUNMSTR_S_STATION_SELECTIONError(const ::boost::shared_ptr< ::MIDW_TUNERMASTER_FI::MIDW_TUNERMASTER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< ::MIDW_TUNERMASTER_FI::FID_TUNMSTR_S_STATION_SELECTIONError >& /*error*/)
{
}


/**
* Description     : Virtual function implemented to get update of updated station selection result
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNERMASTER_FIProxy'
* @param[in]      : result : The result message of 'FID_TUNMSTR_S_STATION_SELECTION'
* @return         : void
*/
void AMFMStationList::onFID_TUNMSTR_S_STATION_SELECTIONResult(const ::boost::shared_ptr< ::MIDW_TUNERMASTER_FI::MIDW_TUNERMASTER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< ::MIDW_TUNERMASTER_FI::FID_TUNMSTR_S_STATION_SELECTIONResult >& /*result*/)
{
}


//=============For HD these two properties added===========wst4kor

/**
* Description     : Virtual function implemented to get update of SELECT HD AUDIO PRG Error
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
* @param[in]      : error : The error message of 'SELECT HD AUDIO PRG'
* @return         : void
*/
void AMFMStationList::onFID_TUN_S_SELECT_HD_AUDIOPRGMError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_S_SELECT_HD_AUDIOPRGMError >& /*error*/)
{
   ETG_TRACE_USR4(("onFID_TUN_S_SELECT_HD_AUDIOPRGMError"));
}


/**
* Description     : Virtual function implemented to set the result of method start SELECT HD AUDIO PRG
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
* @param[in]      : result : The result message of the method "SELECT HD AUDIO PRG"
* @return         : void
*/
void AMFMStationList::onFID_TUN_S_SELECT_HD_AUDIOPRGMResult(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_S_SELECT_HD_AUDIOPRGMResult >& /*result*/)
{
   //ETG_TRACE_USR4(("onFID_TUN_S_SELECT_HD_AUDIOPRGMResult"));
}


void AMFMStationList::onFID_TUN_S_SEL_HD_AUDIOPRGMDIRECTError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >&, const ::boost::shared_ptr< FID_TUN_S_SEL_HD_AUDIOPRGMDIRECTError >&)
{
}


void AMFMStationList::onFID_TUN_S_SEL_HD_AUDIOPRGMDIRECTResult(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >&, const ::boost::shared_ptr< FID_TUN_S_SEL_HD_AUDIOPRGMDIRECTResult >&)
{
}


/**
 * Description     : Virtual function implemented to get update of FID_TUN_G_GET_CONFIG_LIST Status
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : status : The status message of the property "FID_TUN_G_GET_CONFIG_LIST"
 * @return         : void
 */
void AMFMStationList::onFID_TUN_G_GET_CONFIG_LISTStatus(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/,
      const boost::shared_ptr< FID_TUN_G_GET_CONFIG_LISTStatus >& status)
{
   uint16 listSize = status->getTun_ConfigListElements().size();

   ETG_TRACE_USR3(("AMFMStationList : onlist size %d", listSize));

   if (listSize)
   {
      if (status->getE8ConfigListID() == T_e8_Tun_ConfigListID__TUN_TUN_CONFIG_LIST_FM)
      {
         _FMListSize = listSize;
         createFMList(status);
      }
      _listEnter = true;
      _isListDataRequested = false;
   }
}


//todo start
void AMFMStationList::setStartIndex()
{
   ETG_TRACE_USR3(("AMFMStationList : setStartIndex"));
   uint8 mlistSize = _FMListSize;
   uint32 mlistID = LIST_ID_FM_LIST;
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2
   if (_band == T_e8_Tun_TunerBand__TUN_BAND_MW)
   {
      mlistSize = _AMListSize;
      mlistID = LIST_ID_AM_LIST;
   }
#endif
   uint32 startInd = TunerUtility::getListStartIndex(mlistSize, _selectedElemID);
   POST_MSG((COURIER_MESSAGE_NEW(ListChangeMsg)(mlistID, ListChangeSet, startInd)));
}


/**
 * Description     : Function returns List ID for respective band
 *
 * @param[in]      : band : current band
 * @return         : uint8 : Config List ID
 */
uint AMFMStationList:: getListIDfromBand(uint8 band)
{
   ETG_TRACE_USR3(("AMFMStationList : getListIDfromBand"));
   uint retval = 0;
   switch (band)
   {
      case T_e8_Tun_TunerBand__TUN_BAND_FM:
         retval = LIST_ID_FM_LIST;
         break;
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2
      case T_e8_Tun_TunerBand__TUN_BAND_MW:
         retval = LIST_ID_AM_LIST;
         break;
#endif
      default :
         break;
   }

   return retval;
}


/**
 * Description     : Function updates AMFMStationList with AtStationInfo details.
 *
 * @return         : void
 */
void AMFMStationList::setAtStationInfoDataToAMFMStationList(const boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_G_ATSTATIONINFOStatus >& status)
{
   ETG_TRACE_USR4(("AMFMStationList::setAtStationInfoDataToAMFMStationList"));

   //wst4kor
   if (TunerCommonHandler::instance()->isHDFeatureEnabled())
   {
      setAtStationInfoHDdataToAMFMSTationList(status);
   }
   _band = status->getAtstationinfo().getE8Band();

   _selectedElemID = (int)(status->getAtstationinfo().getU16ConfigListElementID());
   _selectedElemID = _selectedElemID - 1;
   ETG_TRACE_USR4(("AMFMStationList::Selected Element_ID == %d", _selectedElemID));
   std::string PSName = status->getAtstationinfo().getSPSName();
   uint16 pi = status->getAtstationinfo().getU16PI();

   if (_listEnter)
   {
      if (_band == T_e8_Tun_TunerBand__TUN_BAND_FM)
      {
         _FMListSize = _stationListElemCount;
      }
      else if (_band == T_e8_Tun_TunerBand__TUN_BAND_MW)
      {
         _AMListSize = _AMStationListElemCount;
      }
//#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2   //ded6kor uncommented for testing.

      if (_band == T_e8_Tun_TunerBand__TUN_BAND_FM)
      {
         fmStationListHighlight(_stationListElemCount, _selectedElemID, PSName, pi);
      }
      else if (_band == T_e8_Tun_TunerBand__TUN_BAND_MW)
      {
         //      amStationListHighlight(_AMStationListElemCount, _selectedElemID);
      }
//#endif
   }
   if (_band == T_e8_Tun_TunerBand__TUN_BAND_FM)
   {
      // updateList(LIST_ID_FM_LIST);   //To fix Flicker Issue on the Home Screen.
   }
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2
   else if (_band == T_e8_Tun_TunerBand__TUN_BAND_MW)
   {
      updateList(LIST_ID_AM_LIST);
   }
#endif
}


/**
 * Description    :Function for updating FM list highlight details.
 *
 *@param[in]      :listCount: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 *@param[in]      :configListElemID:
 *@param[in]      :psName
 *@param[in]      :HDCallSign
 *@param[in]      :PI
 * @return        :void
 */
void AMFMStationList::fmStationListHighlight(uint32& listCount, uint32& configListElemID, std::string& psName, uint16& pi)
{
   ETG_TRACE_USR3(("AMFMStationList : fmStationListHighlight==%d, %d", listCount, configListElemID));
   for (uint8 itr = 0; itr < listCount; itr++)
   {
      _fmStationList[itr]._elementSelected = false;
      if (itr == configListElemID)
      {
         _fmStationList[itr]._elementSelected = true;
      }
   }
   updateList(LISTID_FM);
}


//todo ends

/**
 * Description     : Virtual function implemented to get update of FID_TUN_G_ATSEEK Error
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : error : The error message of 'FID_TUN_G_ATSEEK'
 * @return         : void
 */
void AMFMStationList::onFID_TUN_G_ATSEEKError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_G_ATSEEKError >& /*error*/)
{
}


/**
 * Description     : Virtual function implemented to get update of FID_TUN_G_ATSEEK Status
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : status : The status message of the property "FID_TUN_G_ATSEEK"
 * @return         : void
 */
void AMFMStationList::onFID_TUN_G_ATSEEKStatus(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_G_ATSEEKStatus >& status)
{
   ETG_TRACE_USR3(("AMFMStationList : onFID_TUN_G_ATSEEKStatus "));
   if ((_scanStatus == 1) && (status->getE8SeekState() == 0))  //removed for coverity may 4
   {
      _scanStatus = 0;
      //    _tunInfoDataBinder->updateScanHidePopUp();
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2
      startStopListTimer(true);
      if (_band == T_e8_Tun_TunerBand__TUN_BAND_FM)
      {
         updateList(LIST_ID_FM_MENU);
      }
      else if (_band == T_e8_Tun_TunerBand__TUN_BAND_MW)
      {
         updateList(LIST_ID_AM_MENU);
      }
#endif
   }
}


/**
 * Description     : Virtual function implemented to get update of AtSeek with TunerHeader Error
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : error : The error message of 'AtSeek with TunerHeader'
 * @return         : void
 */
void AMFMStationList::onFID_TUN_S_ATSEEK_WITH_TUNERHEADERError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUN_S_ATSEEK_WITH_TUNERHEADERError >& /*error*/)
{
}


/**
 * Description     : Virtual function implemented to get the result of method start AtSeek with TunerHeader
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : result : The result message of the method "AtSeek with TunerHeader"
 * @return         : void
 */
void AMFMStationList::onFID_TUN_S_ATSEEK_WITH_TUNERHEADERResult(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUN_S_ATSEEK_WITH_TUNERHEADERResult >& /*result*/)
{
}


/**
 * Description     : Virtual function implemented to get update of FID_TUN_S_CONFIGLIST_UPDATE error
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : error : The error message of 'FID_TUN_S_CONFIGLIST_UPDATE'
 * @return         : void
 */
void AMFMStationList::onFID_TUN_S_CONFIGLIST_UPDATEError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_S_CONFIGLIST_UPDATEError >& /*error*/)
{
}


/**
 * Description     : Virtual function implemented to get the result of method start FID_TUN_S_CONFIGLIST_UPDATE
 *
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
 * @param[in]      : result : The result message of the method "FID_TUN_S_CONFIGLIST_UPDATE"
 * @return         : void
 */
void AMFMStationList::onFID_TUN_S_CONFIGLIST_UPDATEResult(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_S_CONFIGLIST_UPDATEResult >& /*result*/)
{
}


/**
 * Description     : Function implemented to getSelectedElementId using the frequency
 *
 * @param[in]      : void
 * @return         : void
 */
uint8 AMFMStationList::getSelectedElementId()
{
   ETG_TRACE_USR3(("AMFMStationList : getSelectedElementId"));
   uint itr;
   for (itr = 0 ; itr < _fmStationList.size(); itr++)
   {
      if (_AtStationFreq == _fmStationList.at(itr)._frequency)
      {
         break;
      }
   }
   return itr;
}


/**
 * Description     : Function implemented to simulate FM list for testing purpose
 *
 * @param[in]      : void
 * @return         : void
 */
void AMFMStationList::simulateFMlist()
{
   ETG_TRACE_USR3(("AMFMStationList : simulateFMlist"));
   _fmStationList.clear();
   _fmStationList.push_back(FM_StationList_Elem(TunerUtility::formatFrequency(_AtStationFreq), _AtStationFreq, ""));
   _fmStationList.push_back(FM_StationList_Elem("90.0", 90000, ""));
   _fmStationList.push_back(FM_StationList_Elem("91.0", 91000, "2"));
   _fmStationList.push_back(FM_StationList_Elem("92.0", 92000, "1"));
   _fmStationList.push_back(FM_StationList_Elem("93.0", 93000, ""));
   _fmStationList.push_back(FM_StationList_Elem("94.0", 94000, ""));
   _fmStationList.push_back(FM_StationList_Elem("95.0", 95000, "3"));
   _fmStationList.push_back(FM_StationList_Elem("96.0", 96000, ""));
   _fmStationList.push_back(FM_StationList_Elem("97.0", 97000, ""));
   _fmStationList.push_back(FM_StationList_Elem("98.0", 98000, ""));
}


/**
 * Description     : Function implemented to get update of FID_TUN_G_GET_CONFIG_LISTStatus Status
 *
 * @param[in]      : status : The status message of the property "FID_TUN_G_GET_CONFIG_LISTStatus"
 * @return         : void
 */
void AMFMStationList::createFMList(const boost::shared_ptr< FID_TUN_G_GET_CONFIG_LISTStatus >& status)
{
   std::string freqStr = "";
   std::string psName = "";
   uint32 frequency = 0;
   volatile uint8 bTPStatus = 0 ;
   T_TunerConfigListElementList tunerConfigListElementList = status->getTun_ConfigListElements();
   bool hdUpdateRequired = false;
//  uint8 activeHDAudioPrg = status->getAtstationinfo().getE8ActiveHDAudPrgm();

   _fmStationList.clear();
   frequencyarray.clear();
   _stationListElemCount = tunerConfigListElementList.size();
   ETG_TRACE_USR3(("List size in FM %d", _stationListElemCount));
   for (uint8 itr = 0 ; itr < tunerConfigListElementList.size() ; itr++)
   {
      ETG_TRACE_USR3(("AMFMStationList:ElementID value == %d", tunerConfigListElementList[itr].getU8ElementID()));
      ETG_TRACE_USR3(("AMFMStationList:ElementSelected value == %d", tunerConfigListElementList[itr].getU8ElementSelected()));
      FM_StationList_Elem listItem;
      listItem._frequency = frequency = tunerConfigListElementList[itr].getU32Frequency();
      bTPStatus = tunerConfigListElementList[itr].getBTpInfo();
      ETG_TRACE_USR3(("AMFMStationList :TPStatus value == %d", bTPStatus));
      bTPStatus = (bTPStatus & 0x01);
      if (bTPStatus)
      {
         listItem._TPName = "TP";
         ETG_TRACE_USR3(("AMFMStationList :TP ON bTPStatus value == %d", bTPStatus));
      }
      else
      {
         listItem._TPName = "";
         ETG_TRACE_USR3(("AMFMStationList :TP OFF bTPStatus value == %d", bTPStatus));
      }
      ETG_TRACE_USR3(("AMFMStationList : check frequency from midleware freqStr %d", frequency));
      freqStr = TunerUtility::formatFrequency(frequency);

      frequencyarray.push_back(freqStr);
      ETG_TRACE_USR3(("AMFMStationList : createFMList freqStr %s", freqStr.c_str()));

      psName = tunerConfigListElementList[itr].getSPSName();
      ETG_TRACE_USR3(("AMFMStationList : psName  %s", psName.c_str()));
      if (psName.empty())
      {
         listItem._psName = freqStr.append("  MHz");
      }
      else
      {
         listItem._psName = psName;
         //wst4kor
         if ((tunerConfigListElementList[itr].getE8ModulationType() == T_e8_Tun_ModulationType__TUN_MODULATION_HD)
               && (TunerCommonHandler::instance()->isHDFeatureEnabled()))
         {
            listItem._psName = freqStr + "  " + psName + "    " + "HD-";
         }
      }

      if ((tunerConfigListElementList[itr].getU8PositionInPresetList()) > 0)
      {
         char presetString[ARRAY_SIZE] = {'\0'};
         snprintf(presetString, sizeof(presetString), "%d", tunerConfigListElementList[itr].getU8PositionInPresetList());
         listItem._preset = std::string(presetString);
      }
      listItem._PI = tunerConfigListElementList[itr].getU32PI();
      listItem._elementSelected = (_selectedElemID == itr);
      listItem._ishdFMStatus = false;  //wst4kor

      if (T_e8_Tun_ModulationType__TUN_MODULATION_HD == tunerConfigListElementList[itr].getE8ModulationType())
      {
         listItem._ishdFMStatus = true;
         hdUpdateRequired = true;
      }

      ETG_TRACE_USR3(("AMFMStationList : createFMList list ps name %s", listItem._psName.c_str()));
      _fmStationList.push_back(listItem);
      calcRegVal(itr);
   }//for loop

   if (hdUpdateRequired)
   {
      updateHDAudioProg(status);
   }
   updateList(LISTID_FM);
}


/**
* Description     : Function called to set RadioMaindata.
*
* @param[in]      : NARRadioMainData
* @return         : void
*/

void AMFMStationList::setradioMainData(NARRadioMainData& radioMainData)
{
   _radioMainData = radioMainData;
}


/**
 * Description     : Function implemented to calculate Region value
 *
 * @param[in]      : i : itr of Station list item.
 * @return         : void
 */
void AMFMStationList::calcRegVal(const uint8& i)
{
   ETG_TRACE_USR3(("AMFMStationList : calcRegVal"));
   unsigned long PI = _fmStationList[i]._PI;
   unsigned long PI_2;
   int regval;
   char reg[ARRAY_SIZE] = {"\0"};
   if ((PI != 0) && (((PI & PI_REG_CODE) >> 8) >= 4) && (i > 0))
   {
      for (int itr = i ; itr != 0 ; itr--)
      {
         PI_2 = _fmStationList[itr - 1]._PI;
         if ((PI_2 != 0) && ((PI_2 & (~PI_REG_CODE)) == (PI   & (~PI_REG_CODE))) && (_fmStationList[i]._psName == _fmStationList[itr - 1]._psName))
         {
            if (((PI_2 &  PI_REG_CODE)   >> 8) < 4)
            {
               regval = (tU8)((PI   &  PI_REG_CODE)   >> 8) - 4;
               snprintf(reg, sizeof reg, " R%d", regval + 1);
               _fmStationList[i]._Reg = std::string(reg);
            }
            else
            {
               regval = (tU8)((PI_2 &  PI_REG_CODE)   >> 8) - 4;
               snprintf(reg, sizeof reg, " R%d", regval + 1);
               _fmStationList[itr - 1]._Reg   =  reg;
               regval = (tU8)((PI   &  PI_REG_CODE)   >> 8) - 4;
               snprintf(reg, sizeof reg, " R%d", regval + 1);
               _fmStationList[i]._Reg = std::string(reg);
            }
         }
         else if ((!_fmStationList[i]._Reg.empty()) && (!_fmStationList[itr - 1]._Reg.empty()) && (_fmStationList[i]._psName != _fmStationList[itr - 1]._psName))
         {
            _fmStationList[i]._Reg = "";
            _fmStationList[itr - 1]._Reg = "";
         }
      }
   }
}


/**
 * Description     : Function implements to clear the FM Station List variables.
 *
 * @return         : void
 */
void AMFMStationList::clearFMList()
{
   ETG_TRACE_USR3(("AMFMStationList : clearFMList"));
   for (uint itr = 0 ; itr < _stationListElemCount ; itr++)
   {
      _fmStationList[itr]._psName = "";
      _fmStationList[itr]._frequency = 0;
      _fmStationList[itr]._preset = "";
      _fmStationList[itr]._Reg = "";
      _fmStationList[itr]._PI = 0;
      _fmStationList[itr]._elementSelected = false;
      _fmStationList[itr]._ishdFMStatus = false;
   }
}


/**
 * Description     : Function implemented to update the list of the specified list ID
 *
 * @param[in]      : listID : List ID of the list to be updated
 * @return         : void
 */
void AMFMStationList::updateList(const uint32& listID)
{
   ETG_TRACE_USR3(("AMFMStationList : updateList"));
   //ListDateProviderReqMsg listDataInfo;
   //listDataInfo.GetListId() = listID;
   ListRegistry::s_getInstance().updateList(listID);
}


/**
* onCourierMessage - This message is received from the flexlist widget when it requires new data when the
*               list is displayed or scrolled
* @param[in] ListDateProviderReqMsg& oMsg
* @parm[out] bool
*/
bool AMFMStationList::onCourierMessage(const ListDateProviderReqMsg& oMsg)
{
   ETG_TRACE_USR3(("AMFMStationList : onCourierMessage ListDateProviderReqMsg"));
   return ListRegistry::s_getInstance().updateList(oMsg.GetListId(), oMsg.GetStartIndex(), oMsg.GetWindowElementSize());//TODO : Confirm if this should be included after list data is received
}


bool AMFMStationList::onCourierMessage(const StationListReqMsg& corfoMsg)
{
   ETG_TRACE_USR3(("AMFMStationList : onCourierMessage StationListReqMsg"));
   T_Tun_MakeConfigList  tunMakeConfigList;
   tunMakeConfigList.setEConfigListID(T_e8_Tun_ConfigListID__TUN_TUN_CONFIG_LIST_FM);
   tunMakeConfigList.setESortingCriteria((T_e8_Tun_SortingCriteria)1);
   //commented for coverity issue
   T_b8_Tun_Digital_Filters u8DigitalFilters;
   if (TunerCommonHandler::instance()->isHDFeatureEnabled())
   {
      u8DigitalFilters.setSTATIONLIST_WITH_HDMPS_SPS(_FMHDActiveStatus);
      tunMakeConfigList.setU8DigitalFilters(u8DigitalFilters);
   }
   ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_GET_CONFIG_LISTStart(*this, tunMakeConfigList)));
   return true;
}


bool AMFMStationList::onCourierMessage(const BrowseListExitMsg& corfoMsg)
{
   ETG_TRACE_USR3(("AMFMStationList : onCourierMessage BrowseListExitMsg"));
   ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_STATIONLIST_EXITStart(*this, T_e8_Tun_ConfigListID__TUN_TUN_CONFIG_LIST_FM)));
   return true;
}


void AMFMStationList::ListExitMsg()
{
   ETG_TRACE_USR3(("AMFMStationList : onCourierMessage ListExitMsg"));
   ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_STATIONLIST_EXITStart(*this, T_e8_Tun_ConfigListID__TUN_TUN_CONFIG_LIST_FM)));
}


/**
* Description     : Function called on any button press in the application
*
* @param[in]      : ButtonReactionMsg
* @return         : bool
*/
bool AMFMStationList::onCourierMessage(const ButtonReactionMsg& oMsg)
{
   bool isRetVal = false;
   ListProviderEventInfo info;

   if ((ListProviderEventInfo::GetItemIdentifierInfo(oMsg.GetSender(), info) == true))
   {
      unsigned int listId = info.getListId();     // the list id for generic access
      uint32 hdlRow = info.getHdlRow();     // normaly the itr
      unsigned int hdlCol = info.getHdlCol();     // if more than 1 active element in one list row, e.g. Button in a button line

      ETG_TRACE_USR4(("AMFMStationList::onCourierMessage::ButtonReactionMsg::%d", oMsg.GetEnReaction()));

      if (oMsg.GetEnReaction() == enRelease)
      {
         switch (info.getListId())
         {
            case LISTID_FM:
            {
               ETG_TRACE_USR4(("AMFMStationList::onCourierMessage::ButtonReactionMsg LISTID_FM %d", hdlRow));
               fmListSelectedIndex(hdlRow);
               isRetVal = true;
            }
            break;
            case LISTID_DAB:
            {
               ETG_TRACE_USR4(("AMFMStationList::onCourierMessage::ButtonReactionMsg LISTID_DAB %d", hdlRow));
               DABStationList::poGetInstance()->onDabServiceSelect(hdlRow);
               isRetVal = true;
            }
            break;
            default:
               break;
         }
         if (isRetVal)
         {
            POST_MSG((COURIER_MESSAGE_NEW(ButtonListItemUpdMsg)(listId, hdlRow, hdlCol, enRelease)));
         }
      }
   }

   return isRetVal;
}


void AMFMStationList::fmListSelectedIndex(uint32& selectedIndex)
{
   if (selectedIndex < _fmStationList.size())
   {
      if (TunerCommonHandler::instance()->isHDFeatureEnabled() && (_FMHDActiveStatus))
      {
         if (_tunerMasterFiProxy.get())
         {
            _tunerMasterFiProxy->sendFID_TUNMSTR_S_STATION_SELECTIONStart(*this, midw_tunermaster_fi_types::T_e8_StationSelection_Option__StationName_Input,
                  midw_tunermaster_fi_types::T_e8_Band__TUN_MSTR_BAND_FM, _fmStationList.at(selectedIndex)._frequency, selectedIndex, midw_tunermaster_fi_types::T_e8_List__BANK_FM1, 0, 0, midw_tunermaster_fi_types::T_e8_ModulationType__TUN_MSTR_MODULATION_HD, _fmStationList.at(selectedIndex)._hdProgramNumber);
         }
      }
      else
      {
         if (_tunerMasterFiProxy.get())
         {
            _tunerMasterFiProxy->sendFID_TUNMSTR_S_STATION_SELECTIONStart(*this, midw_tunermaster_fi_types::T_e8_StationSelection_Option__Frequency_Input,
                  midw_tunermaster_fi_types::T_e8_Band__TUN_MSTR_BAND_FM, _fmStationList.at(selectedIndex)._frequency, selectedIndex, midw_tunermaster_fi_types::T_e8_List__BANK_FM1, _fmStationList.at(selectedIndex)._PI, 0, midw_tunermaster_fi_types::T_e8_ModulationType__TUN_MSTR_MODULATION_ANALOG, 0);
         }
      }
   }
}


uint32 AMFMStationList::getCoverFlowStartIndex(uint8 selectedItemID)
{
   uint32 listSize = _fmStationList.size();
   uint32 startItemIndex = 0;

   if (selectedItemID < listSize)
   {
      startItemIndex = (selectedItemID + (listSize - 2)) % listSize;
   }
   return startItemIndex;
}


//==================wst4kor=====================

/**
* Description     : Function implemented to check demod switch based on currently tuned band.
*
* @return         : void
*/
bool AMFMStationList::bIsHDActive()
{
   if (_band == T_e8_Tun_TunerBand__TUN_BAND_FM)
   {
      return _FMHDActiveStatus;
   }
   else if (_band == T_e8_Tun_TunerBand__TUN_BAND_MW)
   {
      return _AMHDActiveStatus;
   }
   return false;
}


/**
* Description     : Function reads region variant values from KDS configuration.
*
* @return         : void
*/
void AMFMStationList::getRegionType()
{
   _regionType = TunerDataPoolConfig::getInstance()->readRegionType();
   //ETG_TRACE_USR4(("AMFMStationList::getRegionType :%d", _regionType));
}


void AMFMStationList::setFMHDActiveStatus(bool& FMHDActiveStatus)
{
   _FMHDActiveStatus = FMHDActiveStatus;
}


void AMFMStationList::setAMHDActiveStatus(bool& AMHDActiveStatus)
{
   _AMHDActiveStatus = AMHDActiveStatus;
}


void AMFMStationList::setHDDecodingData(bool& HDDataDecoding, bool& HDAudioDecoding)
{
   _HDDataDecodingStatus = HDDataDecoding;
   _HDAudioDecodingStatus = HDAudioDecoding;
   ETG_TRACE_USR4(("AMFMStationList:: setHDDecodingData %d %d", _HDDataDecodingStatus, _HDAudioDecodingStatus));
}


/**
* Description     : Virtual function implemented to set update of StationInfo Status
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
* @param[in]      : status : The status message of the property "StationInfo"
* @return         : void
*/
void AMFMStationList::setAtStationInfoHDdataToAMFMSTationList(const boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_G_ATSTATIONINFOStatus >& status)
{
   ETG_TRACE_USR4(("AMFMStationList::setAtStationInfoHDdataToAMFMSTationList() %d", status->getAtstationinfo().getU32StationInfoHDStationID()));
   //ETG_TRACE_USR4(("getB8HDAudioPrgmAvail %d", status->getAtstationinfo().getB8HDAudioPrgmAvail()));
   ETG_TRACE_USR4(("getE8ActiveHDAudPrgm %d", status->getAtstationinfo().getE8ActiveHDAudPrgm()));
   ETG_TRACE_USR4(("_HDDataDecodingStatus: %d, _HDAudioDecodingStatus: %d ", _HDDataDecodingStatus, _HDAudioDecodingStatus));

   bool isSPSAvailable = false;
   bool HdProgramsAvailable[MAX_HD_AUDIO_PROGRAMS] = { false };
   _HDStationID = status->getAtstationinfo().getU32StationInfoHDStationID();

   HdProgramsAvailable[0] = status->getAtstationinfo().getB8HDAudioPrgmAvail().getMPS_AVAIL();
   HdProgramsAvailable[1] = status->getAtstationinfo().getB8HDAudioPrgmAvail().getSPS_1_AVAIL();
   HdProgramsAvailable[2] = status->getAtstationinfo().getB8HDAudioPrgmAvail().getSPS_2_AVAIL();
   HdProgramsAvailable[3] = status->getAtstationinfo().getB8HDAudioPrgmAvail().getSPS_3_AVAIL();
   HdProgramsAvailable[4] = status->getAtstationinfo().getB8HDAudioPrgmAvail().getSPS_4_AVAIL();
   HdProgramsAvailable[5] = status->getAtstationinfo().getB8HDAudioPrgmAvail().getSPS_5_AVAIL();
   HdProgramsAvailable[6] = status->getAtstationinfo().getB8HDAudioPrgmAvail().getSPS_6_AVAIL();
   HdProgramsAvailable[7] = status->getAtstationinfo().getB8HDAudioPrgmAvail().getSPS_7_AVAIL();

   ETG_TRACE_USR4(("getMPS_AVAIL %d", status->getAtstationinfo().getB8HDAudioPrgmAvail().getMPS_AVAIL()));
   ETG_TRACE_USR4(("getSPS_1_AVAIL %d", status->getAtstationinfo().getB8HDAudioPrgmAvail().getSPS_1_AVAIL()));

   if (HdProgramsAvailable[0])
   {
      ETG_TRACE_USR4(("inside HD Audio Program:if "));

      for (uint8 index = 1; index < (sizeof(HdProgramsAvailable)) / sizeof(bool); index++)
      {
         if (HdProgramsAvailable[index])
         {
            isSPSAvailable = true;
            break;
         }
      }
   }
   for (uint8 index = 0; index < (sizeof(HdProgramsAvailable)) / sizeof(bool); index++)
   {
      if (_HdProgramsAvailable[index] != HdProgramsAvailable[index])
      {
         ETG_TRACE_USR4(("_HdProgramsAvailable[index] "));

         std::memcpy(_HdProgramsAvailable, HdProgramsAvailable, sizeof(HdProgramsAvailable));
         break;
      }
   }
   uint8 activeHDProgram = status->getAtstationinfo().getE8ActiveHDAudPrgm();
   ETG_TRACE_USR4(("_activeHDAudioPrg:%d, activeHDAudioPrg:%d ", _activeHDAudioPrg, activeHDProgram));
   if (_activeHDAudioPrg != activeHDProgram)
   {
      _activeHDAudioPrg = activeHDProgram;
   }

   if (((_HDDataDecodingStatus) || (_HDAudioDecodingStatus)) && isSPSAvailable) // remove once fix from MIDW is available
   {
      ETG_TRACE_USR4(("inside HD Audio isSPSAvailable : %d", isSPSAvailable));

      setHDProgramList(HdProgramsAvailable);
   }
   else if (NULL != _tunInfoDataBinder && ((_AtStationFreq != status->getAtstationinfo().getU32Frequency()) || (!bIsHDActive())))
   {
      ETG_TRACE_USR4(("getU32Frequency %d", status->getAtstationinfo().getU32Frequency()));

      for (uint8 index = 0; index < (sizeof(HdProgramsAvailable)) / sizeof(bool); index++)
      {
         _tunInfoDataBinder->setHDProgSizeAndSwitch(index, HDPROGINACTIVE, HD_AUDIO_PROGRAM_BLANK);
      }
      _AtStationFreq = status->getAtstationinfo().getU32Frequency();
   }
}


/**
* Description     : function to assign size and active status to the HD Audio Programs
*
* @param[in]      : HdProgramsAvailable[] the array of HD Programs
* @return         : void
*/
void AMFMStationList::setHDProgramList(bool HdProgramsAvailable[])
{
   ETG_TRACE_USR4(("AMFMStationList::setHDProgramList "));
   if (NULL != _tunInfoDataBinder)
   {
      for (uint8 index = 0; index < MAX_HD_AUDIO_PROGRAMS; index++)
      {
         ETG_TRACE_USR4((" index %d, _activeHDAudioPrg%d, _HDAudioDecodingStatus%d", index, _activeHDAudioPrg, _HDAudioDecodingStatus));
         if (HdProgramsAvailable[index]) //HD Audio Program is available and it is the current active Program
         {
            if (index == _activeHDAudioPrg && (_HDDataDecodingStatus) && (_HDAudioDecodingStatus))
            {
               _tunInfoDataBinder->setHDProgSizeAndSwitch(index, HDPROGACTIVE, HD_AUDIO_PROGRAM_ACTIVE);
            }
            else  //HD Audio Program is available and it is not the current active Program
            {
               _tunInfoDataBinder->setHDProgSizeAndSwitch(index, HDPROGACTIVE, HD_AUDIO_PROGRAM_GREY);
            }
         }
         else //HD Audio Program is not available
         {
            _tunInfoDataBinder->setHDProgSizeAndSwitch(index, HDPROGINACTIVE, HD_AUDIO_PROGRAM_BLANK);
         }
      }
   }
}


/**
* Description     : Function called when HD Lost timer is set/reset
*
* @param[in]      : HDTimerStatus
* @return         : bool
*/
void AMFMStationList::setHDTimerStatus(bool HDTimerStatus)
{
   ETG_TRACE_USR4(("AMFMStationList::setHDTimerStatus %d", HDTimerStatus));

   if (((!HDTimerStatus) && ((!_HDDataDecodingStatus) && (!_HDAudioDecodingStatus))) && (NULL != _tunInfoDataBinder))
   {
      for (uint8 index = 0; index < (sizeof(_HdProgramsAvailable)) / sizeof(bool); index++)
      {
         _tunInfoDataBinder->setHDProgSizeAndSwitch(index, HDPROGINACTIVE, HD_AUDIO_PROGRAM_BLANK);
      }
   }
}


/**
* Description     : Function implemented to get update of HD Audio program
*
* @param[in]      : status : The status message of the property "FID_TUN_G_GET_CONFIG_LISTStatus"
* @return         : void
*/
void AMFMStationList::updateHDAudioProg(const boost::shared_ptr< FID_TUN_G_GET_CONFIG_LISTStatus >& status)
{
   T_TunerConfigListElementList tunerConfigListElementList = status->getTun_ConfigListElements();

   if (_band == T_e8_Tun_TunerBand__TUN_BAND_FM)
   {
      for (unsigned itr = 0; itr < _stationListElemCount; itr++)
      {
         if (tunerConfigListElementList[itr].getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_MPS())
         {
            _fmStationList[itr]._hdFMAudioPrg = HD_AUDIOPROGRAM_MPS;
            _fmStationList[itr]._hdProgramNumber = T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_ONE;
         }
         else if (tunerConfigListElementList[itr].getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_SPS1())
         {
            _fmStationList[itr]._hdFMAudioPrg = HD_AUDIOPROGRAM_SPS1;
            _fmStationList[itr]._hdProgramNumber = T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_TWO;
         }
         else if (tunerConfigListElementList[itr].getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_SPS2())
         {
            _fmStationList[itr]._hdFMAudioPrg = HD_AUDIOPROGRAM_SPS2;
            _fmStationList[itr]._hdProgramNumber = T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_THREE;
         }
         else if (tunerConfigListElementList[itr].getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_SPS3())
         {
            _fmStationList[itr]._hdFMAudioPrg = HD_AUDIOPROGRAM_SPS3;
            _fmStationList[itr]._hdProgramNumber = T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_FOUR;
         }
         else if (tunerConfigListElementList[itr].getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_SPS4())
         {
            _fmStationList[itr]._hdFMAudioPrg = HD_AUDIOPROGRAM_SPS4;
            _fmStationList[itr]._hdProgramNumber = T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_FIVE;
         }
         else if (tunerConfigListElementList[itr].getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_SPS5())
         {
            _fmStationList[itr]._hdFMAudioPrg = HD_AUDIOPROGRAM_SPS5;
            _fmStationList[itr]._hdProgramNumber = T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_SIX;
         }
         else if (tunerConfigListElementList[itr].getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_SPS6())
         {
            _fmStationList[itr]._hdFMAudioPrg = HD_AUDIOPROGRAM_SPS6;
            _fmStationList[itr]._hdProgramNumber = T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_SEVEN;
         }
         else if (tunerConfigListElementList[itr].getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_SPS7())
         {
            _fmStationList[itr]._hdFMAudioPrg = HD_AUDIOPROGRAM_SPS7;
            _fmStationList[itr]._hdProgramNumber = T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_EIGHT;
         }
      }
   }
   else if (_band == T_e8_Tun_TunerBand__TUN_BAND_MW)
   {
      /*  for (unsigned itr = 0; itr < _AMStationListElemCount; itr++)
        {
           if (tunerConfigListElementList[itr].getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_MPS())
           {
              _amStationList[itr]._hdAMAudioPrg = HD_AUDIOPROGRAM_MPS;
           }
           else if (tunerConfigListElementList[itr].getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_SPS1())
           {
              _amStationList[itr]._hdAMAudioPrg = HD_AUDIOPROGRAM_SPS1;
           }
           else if (tunerConfigListElementList[itr].getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_SPS2())
           {
              _amStationList[itr]._hdAMAudioPrg = HD_AUDIOPROGRAM_SPS2;
           }
           else if (tunerConfigListElementList[itr].getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_SPS3())
           {
              _amStationList[itr]._hdAMAudioPrg = HD_AUDIOPROGRAM_SPS3;
           }
           else if (tunerConfigListElementList[itr].getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_SPS4())
           {
              _amStationList[itr]._hdAMAudioPrg = HD_AUDIOPROGRAM_SPS4;
           }
           else if (tunerConfigListElementList[itr].getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_SPS5())
           {
              _amStationList[itr]._hdAMAudioPrg = HD_AUDIOPROGRAM_SPS5;
           }
           else if (tunerConfigListElementList[itr].getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_SPS6())
           {
              _amStationList[itr]._hdAMAudioPrg = HD_AUDIOPROGRAM_SPS6;
           }
           else if (tunerConfigListElementList[itr].getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_SPS7())
           {
              _amStationList[itr]._hdAMAudioPrg = HD_AUDIOPROGRAM_SPS7;
           }
        }*/
   }
}


//For HD need to check if it needs to be included
/**
* Description     : Function returns PS name if available, else returns frequency.
*
* @param[in]      : psName : Station Name /HD call-sign
* @param[in]      : freq : formatted current frequency
* @param[in]      : activeHDAudioPrg : Current active HD audio program number
* @return         : string
*/
/*std::string AMFMStationList::getPSname(const std::string& psName, const std::string& freq, const uint8 activeHDAudioPrg)
{
   std::string PSName = psName;

   if (bIsHDActive() && _HDDataDecodingStatus && _HDAudioDecodingStatus)
   {
      PSName = PSName + "    " + "HD-" + TunerUtility::u16ConversionToString(activeHDAudioPrg + 1);
   }
   else if (psName.empty() == 0)
   {
      PSName = freq;
   }
   return PSName;
}


*/

/**
 * onCourierMessage - This message is received from the flexlist widget when it requires new data when the
 *               list is displayed or scrolled
 * @param[in] ListDateProviderReqMsg& oMsg
 * @parm[out] bool
 */
/*bool AMFMStationList::onCourierMessage(const ListDateProviderReqMsg& oMsg)
{
   ETG_TRACE_USR3(("AMFMStationList : onCourierMessage ListDateProviderReqMsg"));
   if (oMsg.GetListId())
   {
      ListDataInfo listDataInfo;
      listDataInfo.listId = oMsg.GetListId();
      listDataInfo.startIndex =  oMsg.GetStartIndex();
      listDataInfo.windowSize = oMsg.GetWindowElementSize();
      if (oMsg.GetListId() == LIST_ID_FM_LIST)
      {
         ListDateProviderReqItemEnumerator it(oMsg, _fmStationList.size());
         uint32_t itemIndex(0);
         it.getItemIndex(itemIndex);
         listDataInfo.startIndex = itemIndex;
      }
      ListRegistry::s_getInstance().updateList(listDataInfo);
      return true;
   }
   return false;
} */
}


}
