/**************************************************************************************
* @file         : AMFMInfo.cpp
* @author       : ECV- Rohini S R
* @addtogroup   : Tuner HALL
* @brief        : Class contains necessary business logic required to handle AM FM Info.
* @copyright    : (c) 2017-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "hall_std_if.h"
#include "AMFMInfo.h"
#include "Tuner_Defines.h"
#include "midw_fi_if.h"
#include "App/Core/Utility/TunerUtility.h"
#include "App/Core/AMFMInfo/AMFMStationListTimer.h"
#include "PresetListHandling.h"
#include "TunerDataBindingHandler.h"
#include "Common/ApplicationSwitchClientComponent/ApplicationSwitchClientComponent.h"
#include "App/Core/Utility/TunerCommonHandler.h" //added for HD wst4kor
#include "AnnoHandler.h"
#include "../VehicleSrvClient/VehicleSrvClient.h"  //CAN Imp
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_TUNER_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_TUNER
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_Tuner_"
#define ETG_I_FILE_PREFIX                 App::Core::AMFMInfo::
#include "trcGenProj/Header/AMFMInfo.cpp.trc.h"
#endif

#define IS_LIST_HANDLER_VALID() (NULL != _tunerFiProxy->m_poListHandler)
#define VIEW_CHANGE_IN_PROGRESS() ( (sm_gTargetView != sm_gCurrentView) )
#define IS_TUNER_VALID() ((_tunerFiProxy != NULL))
//#define IS_DAB_VALID() ((m_poTunerDABAppLogic != NULL) && (m_poDABClientHandler != NULL))
#define ROUND_TO_HUNDRED  50

#define AM_START_FREQ_1629          (504)
#define AM_START_FREQ_1611          (513)
#define AM_START_FREQ_1710          (500)

#define AM_SLIDER_STEP_9k           (9)
#define AM_SLIDER_1611              1611
#define AM_SLIDER_NOOFSTEPS_9K      1209
#define AM_SLIDER_MAX_9K            1721

#define AM_SLIDER_1629              1629
#define AM_SLIDER_NOOFSTEPS_EU_9K   1209
#define AM_SLIDER_MAX_EU_9K         1721

#define AM_SLIDER_STEP_10K          (10)
#define AM_SLIDER_1710              1710
#define AM_SLIDER_NOOFSTEPS_10K     1209
#define AM_SLIDER_MAX_10K           1721


namespace App {
namespace Core {

using namespace ::MIDW_TUNER_FI;
using namespace ::midw_tuner_fi_types;
using namespace ::midw_tunermaster_fi_types;

DEFINE_CLASS_LOGGER_AND_LEVEL("App/Core/Tuner_AMFM", AMFMInfo, Info);

AMFMInfo* AMFMInfo::m_poAMFMInfo = NULL;
AMFMInfo::AMFMInfo() :
   _tunerFiProxy(MIDW_TUNER_FIProxy::createProxy("tunerFiPort", *this)),
   _radiotxt(""),
   _tunerStatus(APPSTATUS_BACKGROUND)

{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR3(("AMFMInfo : Param Constructor entered"));
   if (_tunerFiProxy.get())
   {
      ETG_TRACE_USR3(("AMFMInfo : StartupSync"));
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _tunerFiProxy->getPortName());
   }

   _AmFmInfoDataBinder = TunerDataBindingHandler::getDataBindingHandlerInstance();//new TunerDataBindingHandler();
   _AmFmInfoCourierMsgHandler = new AMFMInfoCourierMsgHandler(this, PresetListHandling::poGetInstance());

   m_bIsScanActive = FALSE;
   m_enSeekType = T_e8_Tun_TunerSeek__TUN_SEEKTYPE_OFF;

   m_rAtStationInfo.u32Frequency = 0;
   m_rAtStationInfo.u16PI = 0;
   m_rAtStationInfo.sPSName = "";
   m_rAtStationInfo.e8Band = T_e8_Tun_TunerBand__TUN_BAND_FM;
   m_rAtStationInfo.e8AtStationTPinfo = T_e8_AtStationTPinfo__Unknown;
   m_rAtStationInfo.u16PresetListID = 0;
   m_rAtStationInfo.u16PresetElementId = 0;
   m_rAtStationInfo.u16ConfigListElementID = 0;

   m_bRDSConfigEnable = FALSE;
   m_u32FMLowRange = 0;
   m_u32FMHiRange = 0;
   m_u32AMLowRange = 0;
   m_u32AMHiRange = 0;
   m_u32AMFrequencyStep = 0;
   m_u32FMFrequencyStep = 1;
   m_bIsSeekStatus = 0;
   m_u32CurrentTunedFreq = 0;
   m_bOptDabDabFollowing = false;
   m_bOptDabFmFollowing = false;
   m_bOptDabDynamicText = false;
   m_u32AMStartFreq = 0;
   m_u8KDSConfigTA_Enable = 0;
   m_u16AMmaxvalue = 0;
   m_u8AMnoofsteps = 0;
   m_bRDSConfigEnable = 0;
   m_bOptRadioText = false;
   m_TAActive = false;
   m_PTY31Active = false;
   m_u8KDSConfigAF_Enable = 0;
   m_u8KDSConfigREG_Enable = 0;
   m_EnggModeActive = false;
   m_u8activeRegion = REGION_INVALID;
   _band = T_e8_Band__TUN_MSTR_BAND_FM;
   HDActive = false;
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
AMFMInfo::~AMFMInfo()
{
   ETG_TRACE_USR3(("AMFMInfo : Destructor entered"));
   if (NULL != _AmFmInfoDataBinder)
   {
      delete _AmFmInfoDataBinder;
      _AmFmInfoDataBinder = NULL;
   }
   if (NULL != _AmFmInfoCourierMsgHandler)
   {
      delete _AmFmInfoCourierMsgHandler;
      _AmFmInfoCourierMsgHandler = NULL;
   }
}


tclPresetInfoList::tclPresetInfoList()
{
   ETG_TRACE_USR3(("tclPresetInfoList::tclPresetInfoList()"));
   for (tU8 u8Index = 0; u8Index < MAX_BANK_PRESET_LIST; ++u8Index)
   {
      m_arPresetInfoList[u8Index].m_u32Frequency = 0;
      m_arPresetInfoList[u8Index].m_u8PresetNumber = 0;
   }
}


/**
* Description     : Request from Framework to start registering for properties. This request means GUI is ready now.
*
* @param[in]      : proxy : reference to the base class of all proxies
* @param[in]      : stateChange : represents a state transition of a service
* @return         : void
*/
void AMFMInfo::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR3(("AMFMInfo : registerProperties "));

   if ((proxy == _tunerFiProxy) && (_tunerFiProxy.get()))
   {
      _tunerFiProxy->sendFID_TUN_G_RADIO_TEXTUpReg(*this);
      _tunerFiProxy->sendFID_TUN_G_ATSEEKUpReg(*this);
   }
}


/**
* Description     : Request from Framework to start de-registering for properties.
*
* @param[in]      : proxy : reference to the base class of all proxies
* @param[in]      : stateChange : represents a state transition of a service
* @return         : void
*/
void AMFMInfo::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& /*proxy*/, const asf::core::ServiceStateChange& /*stateChange*/)
{
}


/**
* Description     : Request from Framework to start de-registering for properties.
*
* @param[in]      : proxy : reference to the base class of all proxies
* @param[in]      : stateChange : represents a state transition of a service
* @return         : void
*/
void AMFMInfo::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   if ((proxy == _tunerFiProxy) && (_tunerFiProxy.get()))
   {
      _tunerFiProxy->sendFID_TUN_G_RADIO_TEXTRelUpRegAll();
      _tunerFiProxy->sendFID_TUN_G_ATSEEKRelUpRegAll();
   }
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void AMFMInfo::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR3(("AMFMInfo::onAvailable()"));

   if ((proxy == _tunerFiProxy) && (_tunerFiProxy.get()))
   {
      _tunerFiProxy->sendFID_TUN_G_RADIO_TEXTUpReg(*this);
      _tunerFiProxy->sendFID_TUN_G_ATSEEKUpReg(*this);
   }
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void AMFMInfo::onFID_TUN_S_ATFREQUENCYError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& proxy, const boost::shared_ptr< FID_TUN_S_ATFREQUENCYError >& error)
{
   ////Add your code here
}


void AMFMInfo::onFID_TUN_S_ATFREQUENCYResult(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& proxy, const boost::shared_ptr< FID_TUN_S_ATFREQUENCYResult >& result)
{
   //Add your code here

   //if (NULL != m_ptrInterfaceClass)
   //{
   //	m_ptrInterfaceClass->onFID_TUN_S_ATFREQUENCYResult(tunerserver_statusresponse);
   //}
}


void AMFMInfo::setFMHDIndicatorStatus(bool isActive)
{
   ETG_TRACE_USR3(("AMFMInfo::setFMHDIndicatorStatus() isActive:%d", isActive));
   HDActive = isActive;
}


bool AMFMInfo::getFMHDIndicatorStatus()
{
   ETG_TRACE_USR3(("AMFMInfo::getFMHDIndicatorStatus() HDActive:%d", HDActive));
   return HDActive;
}


/**
* Description     : Virtual function implemented to get update of RadioText Error
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
* @param[in]      : error : The error message of 'RadioText'
* @return         : void
*/
void AMFMInfo::onFID_TUN_G_RADIO_TEXTError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUN_G_RADIO_TEXTError >& /*error*/)
{
}


void AMFMInfo::onFID_TUN_G_RADIO_TEXTStatus(const ::boost::shared_ptr< ::MIDW_TUNER_FI::MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< ::MIDW_TUNER_FI::FID_TUN_G_RADIO_TEXTStatus >& status)
{
   ETG_TRACE_USR4(("onFID_TUN_G_RADIO_TEXTStatus: RadioText > "));

   if (_radiotxt != status->getTunerRadioText().getSRadioTextData())
   {
      _radiotxt = status->getTunerRadioText().getSRadioTextData();
      ETG_TRACE_USR4(("onFID_TUN_G_RADIO_TEXTStatus: RadioText %s", _radiotxt.c_str()));

      if ((NULL != _AmFmInfoDataBinder) && (getFMHDIndicatorStatus() != true))
      {
         _AmFmInfoDataBinder->updateRadioText(_radiotxt);
      }
   }
}


/**
* Description     : Virtual function implemented to get update of AtSeek with TunerHeader Error
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
* @param[in]      : error : The error message of 'AtSeek with TunerHeader'
* @return         : void
*/
void AMFMInfo::onFID_TUN_S_ATSEEK_WITH_TUNERHEADERError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const boost::shared_ptr< FID_TUN_S_ATSEEK_WITH_TUNERHEADERError >& /*error*/)
{
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void AMFMInfo::onFID_TUN_S_ATSEEK_WITH_TUNERHEADERResult(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& proxy, const boost::shared_ptr< FID_TUN_S_ATSEEK_WITH_TUNERHEADERResult >& result)
{
   ETG_TRACE_USR3(("AMFMInfo::ATSeek_With_Tunerheader_Result()"));
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void AMFMInfo::onFID_TUN_S_DIRECT_FREQUENCYError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& proxy, const boost::shared_ptr< FID_TUN_S_DIRECT_FREQUENCYError >& error)
{
   ETG_TRACE_USR3(("AMFMInfo::Direct_Frequency_Error()"));
   //	vUpdateSliderValue();
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void AMFMInfo::onFID_TUN_S_DIRECT_FREQUENCYResult(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& proxy, const boost::shared_ptr< FID_TUN_S_DIRECT_FREQUENCYResult >& result)
{
   ////Add your code here
   //if (NULL != m_ptrInterfaceClass)
   //{
   //	m_ptrInterfaceClass->onFID_TUN_S_DIRECT_FREQUENCYResult(tunerserver_statusresponse);
   //}
}


/**
* Description     : Virtual function implemented to get update of onFID_TUN_G_ATSEEKError
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
* @param[in]      : error : The error message of 'onFID_TUN_G_ATSEEKError'
* @return         : void
*/
void AMFMInfo::onFID_TUN_G_ATSEEKError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_G_ATSEEKError >& /*error*/)
{
}


/**
* Description     : Virtual function implemented to get update of onFID_TUN_G_ATSEEKStatus
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
* @param[in]      : status : The status message of the property "onFID_TUN_G_ATSEEKStatus"
* @return         : void
*/
void AMFMInfo::onFID_TUN_G_ATSEEKStatus(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_G_ATSEEKStatus >& status)
{
   T_e8_Tun_TunerSeek seekstatus = status->getE8SeekState();
   ETG_TRACE_USR3(("AMFMInfo::onFID_TUN_G_ATSEEKStatus(%d)", seekstatus));

   if (m_enSeekType != seekstatus)
   {
      m_enSeekType = seekstatus;
      if (T_e8_Tun_TunerSeek__TUN_SEEKTYPE_UP_SCAN_FREQUENCY != m_enSeekType)
      {
         m_bIsScanActive = FALSE;
         ETG_TRACE_USR3(("AMFMInfo::ATSEEKSCANSET(%d)", m_bIsScanActive));
      }
      else
      {
         m_bIsScanActive = TRUE;
         ETG_TRACE_USR3(("AMFMInfo::ATSEEKSCANCLEARED(%d)", m_bIsScanActive));
      }
      vOnUpdateScanMsg(m_bIsScanActive);
   }
   if (0 == status->getE8SeekState())
   {
      ETG_TRACE_USR3(("AMFMInfo::check"));
      TunerDataBindingHandler::getDataBindingHandlerInstance()->vOnUpdateSearchNextPreviousIconMsg(0);
   }
}


/**
* Description     : Virtual function implemented to set PI Error
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
* @param[in]      : error : The error message of 'Set PI'
* @return         : void
*/
void AMFMInfo::onFID_TUN_S_SET_PIError(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_S_SET_PIError >& /*error*/)
{
}


/**
* Description     : Virtual function implemented to set the result of method start Set PI
*
* @param[in]      : proxy: the client side representation of the CCA Functional Interface 'MIDW_TUNER_FIProxy'
* @param[in]      : result : The result message of the method "Set PI"
* @return         : void
*/
void AMFMInfo::onFID_TUN_S_SET_PIResult(const ::boost::shared_ptr< MIDW_TUNER_FIProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUN_S_SET_PIResult >& /*result*/)
{
}


/************************************************************************
*FUNCTION:     vTuneFrequencyStart
*DESCRIPTION:  Sends a CCA message to start the Tuning of Frequency
*PARAMETER:
*               manualFreqAdjustmentDirection and u8NumberOfTicks
*Rev 0.1 Initial Revision
************************************************************************/
void AMFMInfo::vTuneFrequencyStart(tU8 manualFreqAdjustmentDirection, tU8 u8NumberOfTicks)
{
   ETG_TRACE_USR3(("AMFMInfo : vTuneFrequencyStart "));

   T_TunerManualSingleStep  ManualSingleStep;

   switch (manualFreqAdjustmentDirection)
   {
      case TUN_DIRECTION_UP:
      {
         ManualSingleStep.setManualFreqAdjustment(T_e8_Tun_ManualFreqAdjustment__TUN_SEEK_DIRECTION_UP);
         break;
      }
      case TUN_DIRECTION_DOWN:
      {
         ManualSingleStep.setManualFreqAdjustment(T_e8_Tun_ManualFreqAdjustment__TUN_SEEK_DIRECTION_DOWN);
         break;
      }
      default:
         break;
   }
   ManualSingleStep.setU8NumberOfTicks(u8NumberOfTicks);
   _tunerFiProxy->sendFID_TUN_S_ATFREQUENCYStart(*this, T_e8_Tun_TunerHeader__TUN_FOREGROUND, ManualSingleStep, T_e8_Tun_AcceptAF__TUN_NOT_ACCEPT_AF);
}


void AMFMInfo::seek(tU8 mode)
{
   ETG_TRACE_USR3(("AMFMInfo : seek request "));

   tU8 u8MappedSeekMode = 0;

   switch (enSeekMode(mode))
   {
      case SEEK_OFF:
      {
         u8MappedSeekMode = (tU8)T_e8_Tun_TunerSeek__TUN_SEEKTYPE_OFF;
      }
      break;
      case SEEK_UPAUTO:
      {
         u8MappedSeekMode = (tU8)T_e8_Tun_TunerSeek__TUN_SEEKTYPE_UP_AUTO;
      }
      break;
      case SEEK_UP_CONTINUOUS_MANUAL_STEP_MUTED:
      {
         u8MappedSeekMode = (tU8)T_e8_Tun_TunerSeek__TUN_SEEKTYPE_UP_CONTINUOUS_MUTED;
      }
      break;
      case SEEK_DOWNAUTO:
      {
         u8MappedSeekMode = (tU8)T_e8_Tun_TunerSeek__TUN_SEEKTYPE_DOWN_AUTO;
      }
      break;
      case SEEK_DOWN_CONTINUOUS_MANUAL_STEP_MUTED:
      {
         u8MappedSeekMode = (tU8)T_e8_Tun_TunerSeek__TUN_SEEKTYPE_DOWN_CONTINUOUS_MUTED;
      }
      break;
      case SEEK_UPPRESET:
      {
         u8MappedSeekMode = (tU8)T_e8_Tun_TunerSeek__TUN_SEEKTYPE_UP_PRESET;
      }
      break;
      case SEEK_DOWNPRESET:
      {
         u8MappedSeekMode = (tU8)T_e8_Tun_TunerSeek__TUN_SEEKTYPE_DOWN_PRESET;
      }
      break;
      // case SEEK_UP_DRM_JUMP:
      // {
      // u8MappedSeekMode = (tU8)T_e8_Tun_TunerSeek__TUN_SEEKTYPE_UP_DRM_FREQUENCY;
      // }
      // break;

      // case SEEK_DOWN_DRM_JUMP:
      // {
      // u8MappedSeekMode = (tU8)T_e8_Tun_TunerSeek__TUN_SEEKTYPE_DOWN_DRM_FREQUENCY;
      // }
      // break;
      default:
      {
         ETG_TRACE_USR3(("TUN_ERROR: seek mode is INVALID"));
      }
      break;
   }

   ETG_TRACE_USR3(("TunerAppLogic::vSeekReq:Start Seek, mode: %d freq: %d", u8MappedSeekMode, m_u32CurrentTunedFreq));
   vSeekStart(u8MappedSeekMode, m_u32CurrentTunedFreq);
}


/************************************************************************
*FUNCTION: 		vAbortTunerFunctionality
*DESCRIPTION: 	update the seek status
*PARAMETER:
* 				corfoMsg:	(I)
* 					Courier message received
*
*RETURNVALUE: 	tVoid
*
*HISTORY:
*19.09.13 Pushpa Talawar
*Rev 1.0 Initial Revision.
************************************************************************/
tVoid AMFMInfo::vAbortTunerFunctionality()
{
   ETG_TRACE_USR3(("AMFMInfo::vAbortTunerFunctionality"));
   PresetListHandling::poGetInstance()->vAutostore_Stop();
   tU8 seekMode = T_e8_Tun_TunerSeek__TUN_SEEKTYPE_OFF;
   if ((m_enSeekType != T_e8_Tun_TunerSeek__TUN_SEEKTYPE_OFF) ||
         (TRUE == m_bIsScanActive))
   {
      m_bIsScanActive = FALSE;
      m_enSeekType = T_e8_Tun_TunerSeek__TUN_SEEKTYPE_OFF;
      ETG_TRACE_USR3(("AMFMInfo::vAbortTunerFunctionality to stop the seek"));
      vSeekStart(seekMode, m_u32CurrentTunedFreq);
      vOnUpdateScanMsg(m_bIsScanActive);
   }
}


/************************************************************************
*FUNCTION:     vSeekStart
*DESCRIPTION:  Sends a CCA message to start the Seek of stations
************************************************************************/
void AMFMInfo::vSeekStart(tU8 seekMode, tU32 u32Frequency)
{
   ETG_TRACE_USR3(("AMFMInfo::vSeekStart(%d, %d)", (T_e8_Tun_TunerSeek)seekMode, u32Frequency));
   if (IS_TUNER_VALID())
   {
      _tunerFiProxy->sendFID_TUN_S_ATSEEK_WITH_TUNERHEADERStart(*this, (T_e8_Tun_TunerSeek)seekMode, T_e8_Tun_TunerHeader__TUN_FOREGROUND, u32Frequency);
   }
}


void AMFMInfo::vScanRequestMessage()
{
   ETG_TRACE_USR3(("AMFMInfo::vScanRequestMessage()"));

   //stop the scan if ongoing.
   tU8 seekMode = (tU8)T_e8_Tun_TunerSeek__TUN_SEEKTYPE_OFF;
   m_bIsScanActive = !m_bIsScanActive;

   if (m_enSeekType != T_e8_Tun_TunerSeek__TUN_SEEKTYPE_OFF)
   {
      vSeekStart(seekMode, m_u32CurrentTunedFreq);
   }

   if (T_e8_Tun_TunerSeek__TUN_SEEKTYPE_UP_SCAN_FREQUENCY == m_enSeekType)
   {
      m_bIsScanActive = FALSE;
   }

   ETG_TRACE_USR3(("m_bIsScanActive= %d", m_bIsScanActive));

   if (m_bIsScanActive)
   {
      //start the scan
      seekMode = (tU8)T_e8_Tun_TunerSeek__TUN_SEEKTYPE_UP_SCAN_FREQUENCY;
   }

   vOnUpdateScanMsg(m_bIsScanActive);
   vSeekStart(seekMode, m_u32CurrentTunedFreq);
}


void AMFMInfo::vSlider_TunerFrequencyMsg(uint32 u32SliderValue)
{
   ETG_TRACE_USR3(("AMFMInfo : vSlider_TunerFrequencyMsg %d", u32SliderValue));

   tU32 u32FrequencyToBeTuned = 0;

   u32FrequencyToBeTuned = u32SliderValue;

   if (PresetListHandling::poGetInstance()->IsBandFM())
   {
      if (u32FrequencyToBeTuned == 87620)
      {
         u32FrequencyToBeTuned = m_u32FMLowRange;
      }
      else
      {
         u32FrequencyToBeTuned = (u32FrequencyToBeTuned + ROUND_TO_HUNDRED) / 100 * 100;
      }
      if (u32FrequencyToBeTuned > m_u32FMHiRange)
      {
         u32FrequencyToBeTuned = m_u32FMHiRange;
      }
      else if (u32FrequencyToBeTuned < m_u32FMLowRange)
      {
         u32FrequencyToBeTuned = m_u32FMLowRange;
      }
   }
   else
   {
      if (((u32FrequencyToBeTuned % 100) < m_u32AMFrequencyStep) && (AM_SLIDER_STEP_9k == m_u32AMFrequencyStep))
      {
         u32FrequencyToBeTuned += m_u32AMFrequencyStep;
      }
      u32FrequencyToBeTuned = u32FrequencyToBeTuned - (u32FrequencyToBeTuned % m_u32AMFrequencyStep);

      if (u32FrequencyToBeTuned > m_u32AMHiRange)
      {
         u32FrequencyToBeTuned = m_u32AMHiRange;
      }
      else if (u32FrequencyToBeTuned < m_u32AMLowRange)
      {
         u32FrequencyToBeTuned = m_u32AMLowRange;
      }
   }

   if (0 != u32FrequencyToBeTuned)
   {
      vDirect_Frequency_Change_Start(T_e8_Tun_TunerBand__TUN_BAND_FM,/*m_enCurrentTunBand,*/ u32FrequencyToBeTuned);
      ETG_TRACE_USR3(("AMFMInfo : u32Frequencytuned to %d", u32FrequencyToBeTuned));
   }
   else
   {
      ETG_TRACE_USR3(("vSlider_TunerFrequencyMsg::vSlider_TunerFrequencyMsg() TunClient NULL Pointer"));
   }
}


/************************************************************************
*FUNCTION:     vDirect_Frequency_Change_Start
*DESCRIPTION:
*PARAMETER:     Band:   (I), u32Frequency:   (I)
*Rev 0.1 Initial Revision
************************************************************************/
void AMFMInfo::vDirect_Frequency_Change_Start(T_e8_Tun_TunerBand u8Band, tU32 u32Frequency)
{
   ETG_TRACE_USR3(("AMFMInfo: vDirect_Frequency_Change_Start"));
   _tunerFiProxy->sendFID_TUN_S_DIRECT_FREQUENCYStart(*this, u32Frequency, (::midw_tuner_fi_types::T_e8_Tun_TunerBand)u8Band);
}


/************************************************************************
*FUNCTION:     setAtStationInfoRadioMainData
*DESCRIPTION:
*PARAMETER:   currentBand, currentFreq, presetID, psName
*Rev 0.1 Initial Revision
************************************************************************/

void AMFMInfo::setAtStationInfoRadioMainData(uint8& currentBand, uint32& currentFreq, uint16& presetID, std::string& psName)
{
   ETG_TRACE_USR3(("AMFMInfo: setAtStationInfoRadioMainData"));
   std::string FreqUnit = getFrequencyUnit();

   _AmFmInfoDataBinder->updateFrequency(currentBand, currentFreq, FreqUnit, psName);
   m_u32CurrentTunedFreq = ((tF32)currentFreq);
}


void AMFMInfo::vUpdateFreqConversion()
{
   tChar szFMFrequency[M_FREQ_STR_BUFFER_SIZE] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
   tChar szAMFrequency[M_FREQ_STR_BUFFER_SIZE] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
   ETG_TRACE_USR3(("AMFMInfo::vUpdateFreqConversion()"));

   if (TUNER_ZERO_VALUE == m_rAtStationInfo.u32Frequency)
   {
      ETG_TRACE_USR3(("AMFMInfo:: INVALID FREQUENCY : 0"));
      return;
   }

   if (m_u32CurrentTunedFreq != m_rAtStationInfo.u32Frequency)
   {
      m_u32CurrentTunedFreq = m_rAtStationInfo.u32Frequency;

      switch (m_rAtStationInfo.e8Band)
      {
         case T_e8_Tun_TunerBand__TUN_BAND_FM:
         {
            if (TUNER_ZERO_VALUE == (m_u32CurrentTunedFreq % M_DIVIDE_BY_HUNDRED))
            {
               sprintf(szFMFrequency, "%.1f", ((tF32)m_u32CurrentTunedFreq / M_CONVERSION_KHZ_TO_MHZ));
            }
            else
            {
               sprintf(szFMFrequency, "%.2f", ((tF32)m_u32CurrentTunedFreq / M_CONVERSION_KHZ_TO_MHZ));
            }
            //Update the  Frequency to the HMI Model
            vUpdateFrequency(szFMFrequency, EN_BAND_FM);
         }
         break;

         case T_e8_Tun_TunerBand__TUN_BAND_MW:
         {
            //Frequency Unit for the AM Band
            sprintf(szAMFrequency, "%d", m_u32CurrentTunedFreq);

            //Update the  Frequency to the HMI Model
            vUpdateFrequency(szAMFrequency, EN_BAND_MW);
         }
         break;

         default:
            break;
      }

      vUpdateSliderValue();
   }

   //if ((m_rAtStationInfo.e8Band == T_e8_Tun_TunerBand__TUN_BAND_FM)/* && (bIsValidString(m_rAtStationInfo.sPSName))*/)
   //{
   //   vUpdatePSName(m_rAtStationInfo.sPSName.c_str());
   //}
}


///* ***************************************************************
//* Function 	: vUpdatePSName
//* Description 	: Sends the Frequency Update  to HMI Model component
//* Parameters 	: szFrequency
//* Return value	: None
//*25.07.13 Pushpa Talawar
//*Rev 1.0 Initial Revision.
//* ***************************************************************/
//void AMFMInfo::vUpdatePSName(const tChar* szPSName)const
//{
//   ETG_TRACE_USR3(("AMFMInfo::vUpdatePSName(%s)", szPSName));
//
//   Candera::String oPSName(szPSName);
//
//   UpdatePSNameTPMsg* pUpdatePSNameTPMsg = COURIER_MESSAGE_NEW(UpdatePSNameTPMsg)(oPSName);
//   if (pUpdatePSNameTPMsg)
//   {
//      pUpdatePSNameTPMsg->Post();    // Posting Courier message with Frequency value to HMI Model
//   }
//   else
//   {
//      ETG_TRACE_USR3(("TUN_ERROR: AMFMInfo::vUpdatePSName() CourierMsg Create Failed"));
//   }
//}


/* ***************************************************************
* Function 	: vUpdateFrequency
* Description 	: Sends the Frequency Update  to HMI Model component
* Parameters 	: szFrequency
* Return value	: None
*25.07.13 Pushpa Talawar
*Rev 1.0 Initial Revision.
* ***************************************************************/
void AMFMInfo::vUpdateFrequency(const tChar* szFrequency, tU8 u8Band)const
{
   ETG_TRACE_USR3(("AMFMInfo::vUpdateFrequency(%s)", szFrequency));

   Candera::String oFrequency(szFrequency);

   UpdateFrequencyMsg* poFrequencyMsg = COURIER_MESSAGE_NEW(UpdateFrequencyMsg)(oFrequency, u8Band);

   if (poFrequencyMsg)
   {
      poFrequencyMsg->Post();    // Posting Courier message with Frequency value to HMI Model
   }
   else
   {
      ETG_TRACE_USR3(("TUN_ERROR: AMFMInfo::vUpdateFrequency() CourierMsg Create Failed"));
   }
}


/**
* Description     : Function returns Frequency Unit for NAR or band for EU
*
* @return         : std::string
*/
std::string AMFMInfo::getFrequencyUnit()
{
   std::string FreqUnit = "";
   //added enum type for coverity
   if (T_e8_Tun_TunerBand(m_rAtStationInfo.e8Band) == T_e8_Tun_TunerBand__TUN_BAND_FM)
   {
      FreqUnit = "MHz";
   }
   else if (T_e8_Tun_TunerBand(m_rAtStationInfo.e8Band) == T_e8_Tun_TunerBand__TUN_BAND_MW)
   {
      FreqUnit = "kHz";
   }

   return FreqUnit;
}


void AMFMInfo::setTunerAppStatus(uint8 appStatus)
{
}


/* ***********************************************************
* Function 	: poGetPresetList
* Description 	: Get the current preset list pointer
*
* Return value	: tclPresetInfoList*
*10.09.13 Pushpa Talawar
*Rev 1.0 Initial Revision.
* ***********************************************************/
tclPresetInfoList* AMFMInfo::poGetPresetList(T_e8_Tun_ListID enPrestType)
{
   ETG_TRACE_USR3(("AMFMInfo::poGetPresetList()"));
   tclPresetInfoList* poPresetInfo = NULL;

   switch (enPrestType)
   {
      case T_e8_Tun_ListID__TUN_BANK_FM1:
      {
         poPresetInfo = &m_oFM1PresetInfo;
      }
      break;
      case T_e8_Tun_ListID__TUN_BANK_FM2:
      {
         poPresetInfo = &m_oFM2PresetInfo;
      }
      break;
      case T_e8_Tun_ListID__TUN_BANK_FM_AST:
      {
         poPresetInfo = &m_oFM_ASTPresetInfo;
      }
      break;

      case T_e8_Tun_ListID__TUN_BANK_AM_MW1:
      {
         poPresetInfo = &m_oAMPresetInfo;
      }
      break;

      case T_e8_Tun_ListID__TUN_BANK_AM_AST:
      {
         poPresetInfo = &m_oAM_ASTPresetInfo;
      }
      break;
      default:
      {
         ETG_TRACE_USR3(("ERROR-Invalid Parameter : %d", enPrestType));
      }
      break;
   }

   return poPresetInfo;
}


/************************************************************************
*FUNCTION: 		vOnUpdateScanMsg
*DESCRIPTION: 	Handles the Scan icon updation
*RETURNVALUE: 	tBool
*					TRUE, if the message is consumed
*					FALSE, if the message should be forwarded
*Rev 1.0 Initial Revision.
************************************************************************/
void AMFMInfo::vOnUpdateScanMsg(tBool bIsScanActive)
{
   //added for coverity may 4
   ETG_TRACE_USR3(("AMFMInfo::bOnUpdateScanMsg(%d)", bIsScanActive));
   tBool bseekstatus = false;
   if (T_e8_Tun_TunerSeek__TUN_SEEKTYPE_OFF == m_enSeekType)
   {
      bseekstatus = false;
   }
   else
   {
      bseekstatus = true;
   }
   //UpdateScanMsg *poUpdateScanMsg = COURIER_MESSAGE_NEW(UpdateScanMsg)(bIsScanActive, bseekstatus);

   TunerDataBindingHandler::getDataBindingHandlerInstance()->ScanActive(bIsScanActive); //added for coverity
   bOnUpdateScanMsg(bIsScanActive, bseekstatus);
   // if (NULL != poUpdateScanMsg)
   // {
   // TunerDataBindingHandler::getDataBindingHandlerInstance()->ScanActive(bIsScanActive); //added for coverity
   // bOnUpdateScanMsg(bIsScanActive, bseekstatus);
   // poUpdateScanMsg->Post();    // Posting Courier message with Frequency value to HMI Model
   // }
}


/************************************************************************
*FUNCTION: 		bOnUpdateScanMsg
*RETURNVALUE: 	Bool
*
*HISTORY:
*05.03.13 Narahari
*Rev 1.0 Initial Revision.
************************************************************************/
bool AMFMInfo::bOnUpdateScanMsg(bool ScanActive, bool Seekstatus)
{
   bool bMsgConsumed = false;
   ETG_TRACE_USR3(("AMFMInfo::bOnUpdateScanMsg"));
   if (false != ScanActive)  //changed for compiler warning
   {
      m_bIsSeekStatus = Seekstatus;
      ETG_TRACE_USR3(("m_bIsSeekStatus( %d)", m_bIsSeekStatus));
   }
   else
   {
      ETG_TRACE_USR3(("ERROR: bOnUpdateScanMsg NULL Pointer"));
   }
   bMsgConsumed = true;
   return bMsgConsumed;
}


/* ***************************************************************
* Function 	: vUpdateSliderValue
* Description 	: Sends the Frequency Update  to HMI Model component
* Parameters 	: szFrequency
* Return value	: None
*25.07.13 Pushpa Talawar
*Rev 1.0 Initial Revision.
* ***************************************************************/
void AMFMInfo::vUpdateSliderValue()
{
   ETG_TRACE_USR3(("AMFMInfo::vUpdateSliderValue()"));

   tU32 u32Slidervalue = 0;

   if (e8GetCurrentBand() == T_e8_Band__TUN_MSTR_BAND_FM)
   {
      u32Slidervalue = m_u32CurrentTunedFreq / M_CONVERSION_KHZ_TO_MHZ;
   }
   else
   {
      u32Slidervalue = m_u32CurrentTunedFreq;
   }

   ETG_TRACE_USR3(("AMFMInfo::vUpdateSliderValue(%d)", u32Slidervalue));

   Update_Slider_TunerFrequencyMsg* poSliderUpdateMsg = COURIER_MESSAGE_NEW(Update_Slider_TunerFrequencyMsg)(u32Slidervalue);
   //_AmFmInfoDataBinder->bOnUpdate_Slider_TunerFrequencyMsg(u32Slidervalue);
   if (poSliderUpdateMsg)
   {
      poSliderUpdateMsg->Post(); // Posting Courier message with Band level to HMI Model
   }
   else
   {
      ETG_TRACE_USR3(("TUN_ERROR: tclTunerAppLogic::vUpdateSliderValue() CourierMsg Create Failed"));
   }
}


/* ***************************************************************
* Function 	: bSendMsgToPL
* Description 	: Sends the message to PL
* Return value	: None
*25.04.2014 Shrivani K Chakreshwar
*Rev 1.0 Initial Revision.
* ***************************************************************/
bool AMFMInfo::bSendMsgToPL(tU8 u8EventType, tU32 u32Para, Candera::String const& szStrPara) const
{
   ETG_TRACE_USR3(("tclTunerAppLogic::bSendMsgToPL(Event: %d, FirstPara: %d)", u8EventType, u32Para));
   _AmFmInfoDataBinder->TunerPLMsg(u8EventType, u32Para, szStrPara);
   return FALSE;
}


/************************************************************************
*FUNCTION: 		bIsValidString
*DESCRIPTION: 	Returns true if the string is valid
*PARAMETER :
*
*
*RETURNVALUE: 	tBool
*
*HISTORY:
*25.07.13 Pushpa Talawar
*Rev 1.0 Initial Revision.
************************************************************************/

bool AMFMInfo::bIsValidString(std::string& rfoString, tBool bEmptyString /*= TRUE*/)
{
   tBool bIsValid = false;

   if (NULL != rfoString.c_str())
   {
      if (('\0' != rfoString[0]) || bEmptyString)
      {
         //If RDS configuration then display the PSName instead of Frequency.
         if (m_bRDSConfigEnable)
         {
            bIsValid = true;
         }
         else
         {
            bIsValid = false;
         }
      }
   }
   return bIsValid;
}


/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////                                                                TRACE FUNCTIONS                                                      //////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


void AMFMInfo::manualStepSeek(uint8& seekDirection)
{
   T_TunerManualSingleStep  ManualSingleStep;
   switch (seekDirection)
   {
      case TUN_DIRECTION_UP:
      {
         ManualSingleStep.setManualFreqAdjustment(T_e8_Tun_ManualFreqAdjustment__TUN_SEEK_DIRECTION_UP);
         break;
      }
      case TUN_DIRECTION_DOWN:
      {
         ManualSingleStep.setManualFreqAdjustment(T_e8_Tun_ManualFreqAdjustment__TUN_SEEK_DIRECTION_DOWN);
         break;
      }
      default:
         break;
   }
   ManualSingleStep.setU8NumberOfTicks(1);
   ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_ATFREQUENCYStart(*this, T_e8_Tun_TunerHeader__TUN_FOREGROUND, ManualSingleStep, T_e8_Tun_AcceptAF__TUN_NOT_ACCEPT_AF)));
}


void AMFMInfo::targetSeek(uint8& seekDirection)
{
   bool _isTargetSeekActive = false;

   switch (seekDirection)
   {
      case TUN_DIRECTION_DOWN:
      {
         ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_ATSEEK_WITH_TUNERHEADERStart(*this, T_e8_Tun_TunerSeek__TUN_SEEKTYPE_DOWN_CONTINUOUS_MUTED, T_e8_Tun_TunerHeader__TUN_FOREGROUND, m_u32CurrentTunedFreq)));
         break;
      }
      case TUN_DIRECTION_UP:
      {
         ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_ATSEEK_WITH_TUNERHEADERStart(*this, T_e8_Tun_TunerSeek__TUN_SEEKTYPE_UP_CONTINUOUS_MUTED, T_e8_Tun_TunerHeader__TUN_FOREGROUND, m_u32CurrentTunedFreq)));
         break;
      }
      default:
      {
         break;
      }
   }
   _isTargetSeekActive = true;
   PresetListHandling::poGetInstance()->setActiveSeekStatus(_isTargetSeekActive);
}


void AMFMInfo::vUpdateKDSValueToModel() const
{
   ETG_TRACE_USR4(("tclTunerAppLogic::vUpdateKDSValueToModel()"));
   //Update the slider bitmap for different variants by chnging the
   //bSendMsgToPL(EV_MSG_TO_PL_RDS_CONFIG, m_bRDSConfigEnable, "");       // RDS CONFIG
   bSendMsgToPL(EV_MSG_TO_PL_AF_CONFIG, m_u8KDSConfigAF_Enable, "");    // ALTERNATE FREQUENCY
   bSendMsgToPL(EV_MSG_TO_PL_REG_CONFIG, m_u8KDSConfigREG_Enable, "");    // REG AVAILABLE
   bSendMsgToPL(EV_MSG_TO_PL_TA_CONFIG, m_u8KDSConfigTA_Enable, "");    //TRAFFIC ANNOUNCEMENT AVAILABLE

   //Settings screen related updates. TODO: check for DAB related updates.
   bSendMsgToPL(EV_MSG_TO_PL_OPT_RADIO_TEXT, m_bOptRadioText, "");
   bSendMsgToPL(EV_MSG_TO_PL_OPT_DAB_DAB_LINKING, m_bOptDabDabFollowing, "");
   bSendMsgToPL(EV_MSG_TO_PL_OPT_DAB_FM_LINKING, m_bOptDabFmFollowing, "");
   //bSendMsgToPL(EV_MSG_TO_PL_DABPTY_FILTER_SETTINGS, m_poTunerDABAppLogic->m_u32DabPtyMask, "");
   bSendMsgToPL(EV_MSG_TO_PL_OPT_DAB_DYNCONTENT, m_bOptDabDynamicText, "");
   bSendMsgToPL(EV_MSG_TO_PL_VARIANT_TYPE, TunerDataPoolConfig::getInstance()->readRegionType(), "");
   bSendMsgToPL(EV_MSG_TO_PL_MUTE, 0, ""); // Retaining Mute State is Still not Done from Master.

   VehicleSrvClient::poGetInstance()->updateRDSStatus(m_bOptRadioText); //CAN Imp
   VehicleSrvClient::poGetInstance()->updateDABFMStatus(m_bOptDabFmFollowing); //CAN Imp
   int targetRegion = TunerDataPoolConfig::getInstance()->readRegionType();
   if (targetRegion == EUR_REGION)
   {
      VehicleSrvClient::poGetInstance()->updateDABAvailStatus(true); //CAN Imp
   }
   else
   {
      VehicleSrvClient::poGetInstance()->updateDABAvailStatus(false); //CAN Imp
   }
}


void AMFMInfo::vGetKDSValue()
{
   ETG_TRACE_USR4(("tclTunerAppLogic::vGetKDSValue()"));

   dp_tclKdsAMFMTunerParameter oAMFMTunerPara;
   tU8 u8Temp = 0;
   //RDS enable/Disable
   oAMFMTunerPara.u8GetRDS(u8Temp);
   m_bRDSConfigEnable = u8Temp;

   oAMFMTunerPara.u8GetManualStepSizeFM(u8Temp);
   m_u32FMFrequencyStep = u8Temp;

   oAMFMTunerPara.u8GetStepSizeMW(u8Temp);
   m_u32AMFrequencyStep = u8Temp;
   ETG_TRACE_USR4(("FMStep = (%d), AMStep = (%d) ", m_u32FMFrequencyStep, m_u32AMFrequencyStep));

   oAMFMTunerPara.u8GetAF(u8Temp);//AF in the Option AM/FM view
   m_u8KDSConfigAF_Enable = u8Temp;

   oAMFMTunerPara.u8GetREG(u8Temp);// REG in the Option AM/FM view
   m_u8KDSConfigREG_Enable = u8Temp;

   oAMFMTunerPara.u8GetTA(u8Temp);// TA in the Option AM/FM view
   m_u8KDSConfigTA_Enable = u8Temp;

   //@ded6kor
   dp_tclAppHmi_TunerDPRadiotext obj;
   tS32 s32Status = obj.tGetData();
   m_bOptRadioText = s32Status;
   ETG_TRACE_USR4(("s32Status= %d,%d ", s32Status, m_bOptRadioText));

   dp_tclAppHmi_TunerDAB_DynamicText objDabDynText;
   s32Status = objDabDynText.tGetData();
   m_bOptDabDynamicText = s32Status;
   ETG_TRACE_USR4(("s32Status= %d,%d ", s32Status, m_bOptDabDynamicText));

   dp_tclAppHmi_TunerDAB_FMFollowing objDabFmLinking;
   s32Status = objDabFmLinking.tGetData();
   m_bOptDabFmFollowing = s32Status;
   ETG_TRACE_USR4(("s32Status= %d,%d ", s32Status, m_bOptDabFmFollowing));

   dp_tclAppHmi_TunerDAB_DABLinking objDabDabLinking;
   s32Status = objDabDabLinking.tGetData();
   m_bOptDabDabFollowing = s32Status;
   ETG_TRACE_USR4(("s32Status= %d,%d ", s32Status, m_bOptDabDabFollowing));

   //redundant code, needs to be checked, if required.
   tU8 au8Data[4];
   oAMFMTunerPara.u8GetLowFM(au8Data, 4);
   m_u32FMLowRange = au8Data[3] | (au8Data[2] << 8) | (au8Data[1] << 16) | (au8Data[0] << 24);

   oAMFMTunerPara.u8GetHighFM(au8Data, 4);
   m_u32FMHiRange = au8Data[3] | (au8Data[2] << 8) | (au8Data[1] << 16) | (au8Data[0] << 24);

   oAMFMTunerPara.u8GetLowMW(au8Data, 4);
   m_u32AMLowRange = au8Data[3] | (au8Data[2] << 8) | (au8Data[1] << 16) | (au8Data[0] << 24);

   oAMFMTunerPara.u8GetHighMW(au8Data, 4);
   m_u32AMHiRange = au8Data[3] | (au8Data[2] << 8) | (au8Data[1] << 16) | (au8Data[0] << 24);

   //redundant code, needs to be checked, if required.

   if (AM_SLIDER_STEP_9k == m_u32AMFrequencyStep &&  AM_SLIDER_1611 == m_u32AMHiRange)
   {
      m_u8AMnoofsteps = AM_SLIDER_NOOFSTEPS_9K;
      m_u16AMmaxvalue = AM_SLIDER_MAX_9K;
      m_u32AMStartFreq = AM_START_FREQ_1611;//513
   }
   else if (AM_SLIDER_STEP_9k == m_u32AMFrequencyStep && ((AM_SLIDER_1629 == m_u32AMHiRange) /*|| (AM_SLIDER_1620 == m_u32AMHiRange))*/))
   {
      m_u8AMnoofsteps = AM_SLIDER_NOOFSTEPS_EU_9K;
      m_u16AMmaxvalue = AM_SLIDER_MAX_EU_9K;
      m_u32AMStartFreq = AM_START_FREQ_1629;//504
   }
   else if (AM_SLIDER_STEP_10K == m_u32AMFrequencyStep &&  AM_SLIDER_1710 == m_u32AMHiRange)
   {
      m_u8AMnoofsteps = AM_SLIDER_NOOFSTEPS_10K;
      m_u16AMmaxvalue = AM_SLIDER_MAX_10K;
      m_u32AMStartFreq = AM_START_FREQ_1710; //495
   }
   else
   {
      ETG_TRACE_USR3(("Not AM Frequency"));
   }

   TunerUtility::setFMFreqMinMax(m_u32FMLowRange, m_u32FMHiRange);
   _AmFmInfoDataBinder->setFMSliderValues(m_u32FMLowRange, m_u32FMHiRange, m_u32FMFrequencyStep);
   _AmFmInfoDataBinder->setAMSliderValues(m_u32AMStartFreq, m_u16AMmaxvalue, m_u8AMnoofsteps, m_u32AMFrequencyStep);

   ETG_TRACE_USR4(("Print FM SLIDER DATA := %d,%d, %d ", m_u32FMLowRange, m_u32FMHiRange, m_u32FMFrequencyStep));
   ETG_TRACE_USR4(("Print AM SLIDER DATA := %d,%d, %d %d", m_u32AMStartFreq, m_u16AMmaxvalue, m_u8AMnoofsteps, m_u32AMFrequencyStep));

   //Check DAB variables.
   /*dp_tclKdsCMVariantCoding HwConfig;
   dp_tclKdsSystemConfiguration1 SysConfig;
   tU8 dabConfigType = 0xFF;
   tU8 dabEnabledConfigType = 0xFF;
   HwConfig.u8GetDAB(dabConfigType);
   SysConfig.u8GetDAB(dabEnabledConfigType);*/

   //@ded6kor commneted for testing.
   vUpdateKDSValueToModel();
}


void AMFMInfo::onActiveRegionIdError(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< ActiveRegionIdError >& error)
{
   ETG_TRACE_USR4(("AMFMInfo::onActiveRegionIdError"));
}


void AMFMInfo::onActiveRegionIdUpdate(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< ActiveRegionIdUpdate >& update)
{
   m_u8activeRegion = update->getActiveRegionId();
   ETG_TRACE_USR4(("AMFMInfo::m_u8activeRegion: %d", m_u8activeRegion));

   bool _bMutestatus = SrcChangeHandler::poGetInstance()->getCurrentRegionMuteStatus();
   ETG_TRACE_USR4(("AMFMInfo::onActiveRegionIdUpdate:Mutestatus %d", _bMutestatus));

   TunerDataBindingHandler::getDataBindingHandlerInstance()->TunerPLMsg(EV_MSG_TO_PL_MUTE, _bMutestatus, "");
}


/**
* Description     : Function provides Tuner App Status(Foreground/Background).
*
* @return         : void
*/
void AMFMInfo::setTunerAppStatus(uint8& appStatus)
{
   ETG_TRACE_USR4(("AMFMInfo::setTunerAppStatus(%d)", appStatus));
   _tunerStatus = appStatus;
   if (_tunerStatus == hmibase::IN_FOREGROUND /*APPSTATUS_FOREGROUND*/)
   {
      //perform foreground tasks
   }
   else if (_tunerStatus == hmibase::IN_BACKGROUND/*APPSTATUS_BACKGROUND*/)
   {
      /*if ((_isAutoSeekActive) || (_isTargetSeekActive))
      {
      ((_tunerFiProxy.get()) && (_tunerFiProxy->sendFID_TUN_S_ATSEEK_WITH_TUNERHEADERStart(*this, T_e8_Tun_TunerSeek__TUN_SEEKTYPE_OFF, T_e8_Tun_TunerHeader__TUN_FOREGROUND, _AtStationFreq)));
      }*/
   }
}


void AMFMInfo::vRequestView1(tenViewID u8TestNumber)
{
   ApplicationSwitchClientComponent* _applicationSwitchClientComponent = ApplicationSwitchClientComponent::poGetInstance();
   _applicationSwitchClientComponent->requestSwitchApplication(APPID_APPHMI_TUNER);
   /* if (_AmFmInfoCourierMsgHandler != NULL)
   {
   	_AmFmInfoCourierMsgHandler->vRequestView(u8TestNumber);
   } */
}


/**
* Trace command to call onSeek
*
* @param[in]      : Seek type
* @param[in]      : Seek direction
* @return         : void
*/
ETG_I_CMD_DEFINE((TraceCmd_onSeek, "Seek %d %d %d", int, int, int))
void AMFMInfo::TraceCmd_onSeek(uint8 seekType, uint8 seekDirection, uint8 keytype)
{
   if (m_poAMFMInfo)
   {
      //	m_poAMFMInfo->seek(seekType, seekDirection, keytype);
   }
}


/**
* Trace command to call onTargetSeek
*
* @param[in]      : Seek direction
* @return         : void
*/
ETG_I_CMD_DEFINE((TraceCmd_onTargetSeek, "TargetSeek %d", int))
void AMFMInfo::TraceCmd_onTargetSeek(uint8 seekDirection)
{
   ETG_TRACE_USR4(("AMFMInfo::TraceCmd_onTargetSeek"));
   if (m_poAMFMInfo)
   {
      m_poAMFMInfo->targetSeek(seekDirection);
   }
}


/**
      * Description     : Function reads region variant values from KDS configuration.
      *
      * @return         : void
      */
void AMFMInfo::getRegionType() // wst4kor
{
   bool isHDButtonVisible = false;
   uint32 regionVal = 1;
   ETG_TRACE_USR4(("HD Availablility::%d", TunerCommonHandler::instance()->isHDFeatureEnabled()));
   ETG_TRACE_USR4(("HD NAR Variant::%d", TunerCommonHandler::instance()->isNARVariant()));

   if (TunerCommonHandler::instance()->isHDFeatureEnabled() || TunerCommonHandler::instance()->isNARVariant())
   {
      isHDButtonVisible = true;
      regionVal = 0;
   }
   /* else if (TunerCommonHandler::instance()->isNARVariant())
    {
       regionVal = 0;
    }*/

   if (NULL != _AmFmInfoDataBinder)
   {
      ETG_TRACE_USR4(("isHDButtonVisible::%d", isHDButtonVisible));
      _AmFmInfoDataBinder->setHdButtonVisibilityStatus(isHDButtonVisible);
   }
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1_R
   if (TunerDataPoolConfig::getInstance()->readRegionType() == IND_REGION)
   {
      regionVal = TunerDataPoolConfig::getInstance()->readRegionType();
   }
#endif

#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2
   bool isAutoPresetBtnVisible = false;
   // Japan Variant AutoPreset Button Visibility
   if (TunerDataPoolConfig::getInstance()->readRegionType() == JPN_REGION)
   {
      isAutoPresetBtnVisible = true;
      regionVal = TunerDataPoolConfig::getInstance()->readRegionType();
   }
   if (NULL != _AmFmInfoDataBinder)
   {
      _AmFmInfoDataBinder->setAutoPresetButtonVisibilty(isAutoPresetBtnVisible);
   }
#endif
   if (NULL != _AmFmInfoDataBinder)
   {
      // _AmFmInfoDataBinder->updateEncoderRegionType(regionVal);  //need to check for data binding
   }
}


/**
* Trace command to call onTargetSeek
*
* @param[in]      : Auto Seek direction
* @return         : void
*/
ETG_I_CMD_DEFINE((TraceCmd_onAutoSeek, "AutoSeek %d", int))
void AMFMInfo::TraceCmd_onAutoSeek(uint8 seekDirection)
{
   ETG_TRACE_USR4(("AMFMInfo::TraceCmd_onAutoSeek"));
   if (m_poAMFMInfo)
   {
      m_poAMFMInfo->seek(seekDirection);
      //m_poAMFMInfo->autoSeek(seekDirection);
   }
}


/**
* Trace command to call onManualSeek
*
* @param[in]      : Manual Seek direction
* @return         : void
*/
ETG_I_CMD_DEFINE((TraceCmd_onManualStepSeek, "ManualSeek %d", int))
void AMFMInfo::TraceCmd_onManualStepSeek(uint8 seekDirection)
{
   ETG_TRACE_USR4(("AMFMInfo::TraceCmd_onManualStepSeek"));
   if (m_poAMFMInfo)
   {
      m_poAMFMInfo->manualStepSeek(seekDirection);
   }
}


/**
* Trace command to call onManualSeek
*
* @param[in]      : Manual Seek direction
* @return         : void
*/
ETG_I_CMD_DEFINE((TraceCmd_onTestCommand, "TestCommand %d", int))
void AMFMInfo::TraceCmd_onTestCommand(uint8 u8TestNumber)
{
   ETG_TRACE_USR4(("Test Command : %d", u8TestNumber));
   SrcChangeHandler::poGetInstance()->switchToTunerHomeScreenGadget();
}


/**
* Trace command to call onManualSeek
*
* @param[in]      : Manual Seek direction
* @return         : void
*/
ETG_I_CMD_DEFINE((TraceCmd_onSwitchPlayView, "SwitchPlayView %d", int))
void AMFMInfo::TraceCmd_onSwitchPlayView(uint8 u8TestNumber)
{
   ETG_TRACE_USR4(("Switch Play View : %d", u8TestNumber));
   if (m_poAMFMInfo != NULL)
   {
      m_poAMFMInfo->vRequestView1((tenViewID)u8TestNumber);
   }
   else
   {
      ETG_TRACE_USR4(("Switch Play View Null Pointer "));
   }
}


void AMFMInfo::onTunerCommandReq(uint32 u8tunerCommand)
{
   ETG_TRACE_USR4(("AMFMInfo::onTunerCommandReq u8tunerCommand: %d", u8tunerCommand));

   switch (u8tunerCommand)
   {
      case Auto_Seek_Up:
      {
         int seekDirection = SEEK_UPAUTO;
         POST_MSG((COURIER_MESSAGE_NEW(NextIconSelectMsg)()));
         m_poAMFMInfo->seek(seekDirection);
      }
      break;
      case Auto_Seek_Down:
      {
         int seekDirection = SEEK_DOWNAUTO;
         POST_MSG((COURIER_MESSAGE_NEW(PreviousIconSelectMsg)()));
         m_poAMFMInfo->seek(seekDirection);
      }
      break;
      case Manual_Step_Up:
      {
         uint8 seekDirection = TUN_DIRECTION_UP;
         m_poAMFMInfo->manualStepSeek(seekDirection);
      }
      break;
      case Manual_Step_Down:
      {
         uint8 seekDirection = TUN_DIRECTION_DOWN;
         m_poAMFMInfo->manualStepSeek(seekDirection);
      }
      break;
      case Manual_Seek_Up:
      {
         int seekDirection = SEEK_UP_CONTINUOUS_MANUAL_STEP_MUTED;
         m_poAMFMInfo->seek(seekDirection);
      }
      break;
      case Manual_Seek_Down:
      {
         int seekDirection = SEEK_DOWN_CONTINUOUS_MANUAL_STEP_MUTED;
         m_poAMFMInfo->seek(seekDirection);
      }
      break;
      case Preset_Up:
      {
         int seekDirection = SEEK_UPPRESET;
         m_poAMFMInfo->seek(seekDirection);
      }
      break;
      case Preset_Down:
      {
         int seekDirection = SEEK_DOWNPRESET;
         m_poAMFMInfo->seek(seekDirection);
      }
      break;
      case Dont_CareCmd:
      {
         POST_MSG((COURIER_MESSAGE_NEW(AbortNextPreviousIcon)()));
         vAbortTunerFunctionality();
      }
      break;
      default:
      {
         ETG_TRACE_USR1(("AMFMInfo::onTunerCommandReq Default case "));
      }

      break;
   }
}


void AMFMInfo::onTACommandReq(uint32 u8taCommand)
{
   ETG_TRACE_USR4(("AMFMInfo::onTACommandReq u8taCommand: %d", u8taCommand));

   switch (u8taCommand)
   {
      case Switch_TA_On:
      {
         if (AnnoHandler::poGetInstance()->isTaActive() == false)
         {
            POST_MSG((COURIER_MESSAGE_NEW(ButtonListItemUpdMsg)(1000, 2, 0, enRelease)));
         }
         else
         {
            //Do nothing
         }
      }
      break;
      case Switch_TA_Off:
      {
         if (AnnoHandler::poGetInstance()->isTaActive() == true)
         {
            POST_MSG((COURIER_MESSAGE_NEW(ButtonListItemUpdMsg)(1000, 2, 0, enRelease)));
         }
         else
         {
            //Do nothing
         }
      }
      break;
      default:
      {
         ETG_TRACE_USR1(("AMFMInfo::onTACommandReq Default case "));
      }

      break;
   }
}


void AMFMInfo::onRDSCommandReq(uint32 u8rdsCommand)
{
   ETG_TRACE_USR4(("AMFMInfo::onRDSCommandReq u8rdsCommand: %d", u8rdsCommand));
   bool radiostatus = TunerDataPoolConfig::getInstance()->getRadioTextStatus();
   switch (u8rdsCommand)
   {
      case Switch_RDS_On:
      {
         if (radiostatus == 0)
         {
            POST_MSG((COURIER_MESSAGE_NEW(ButtonListItemUpdMsg)(1000, 1, 0, enRelease)));
         }
         else
         {
            //Do nothing
         }
      }
      break;
      case Switch_RDS_Off:
      {
         if (radiostatus == 1)
         {
            POST_MSG((COURIER_MESSAGE_NEW(ButtonListItemUpdMsg)(1000, 1, 0, enRelease)));
         }
         else
         {
            //Do nothing
         }
      }
      break;
      default:
      {
         ETG_TRACE_USR1(("AMFMInfo::onRDSCommandReq Default case"));
      }
      break;
   }
}


}
}
