/**************************************************************************************
* @file         : TunerInfo.cpp
* @author       : ECV-Priyanka B
* @addtogroup   : Tuner HALL
* @brief        : Class contains necessary business logic required to handle utility functions.
* @copyright    : (c) 2017-2018 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "AppUtils/HmiMacros.h"
#include <stdio.h>
#include<sstream>
#include "TunerUtility.h"

#define FM_FREQ_CONVERT       1000
#define FM_FREQ_FRACTION      100
#define ARRAY_SIZE            20
#define START_WINDOW_FMLIST   3
#define THIRD_POS_FMLIST      2
#define LAST_WINDOW_FMLIST    6

namespace App {
namespace Core {
/**
 * Description     : Destructor of class TunerUtility
 */
TunerUtility::~TunerUtility()
{
}


/**
 * Description     : Constructor of class TunerUtility
 */
TunerUtility::TunerUtility()
{
   //ETG_I_REGISTER_FILE();
}


uint32 TunerUtility::_AMHighFrequency = 0;
uint32 TunerUtility::_AMLowFrequency = 0;
uint32 TunerUtility::_FMHighFrequency = 0;
uint32 TunerUtility::_FMLowFrequency = 0;
uint8 TunerUtility::_AMStepSize = 0;
uint8 TunerUtility::_FMStepSize = 0;

void TunerUtility::setFMFreqMinMax(const uint32& FMLowFrequency, const uint32& FMHighFrequency)
{
   _FMLowFrequency = FMLowFrequency;
   _FMHighFrequency = FMHighFrequency;
}


void TunerUtility::setAMFreqMinMax(const uint32& AMLowFrequency, const uint32& AMHighFrequency)
{
   _AMLowFrequency = AMLowFrequency;
   _AMHighFrequency = AMHighFrequency;
}


/**
 * Description     : Function implements to convert the given frequency into required format for the display.
 *
 * @param[in]      : frequency : Frequency selected by the user.
 * @return         : std::string
 */
std::string TunerUtility::formatFrequency(const uint32& frequency)
{
   char freqString[ARRAY_SIZE] = "\0";

   if ((frequency >= _FMLowFrequency) && (frequency <= _FMHighFrequency))
   {
      uint8 frequencyFraction = 100;
      int mhz = frequency / FM_FREQ_CONVERT;
      int fraction = 0;
      // ETG_TRACE_USR3(("TunerUtility : check before _FMStepSize in tuner utility  %d", _FMStepSize));
      //if (_FMStepSize == 50)
      if (1)
      {
         frequencyFraction = 10;     // Required to show two digits after decimal if step size is 50
         fraction = (frequency % FM_FREQ_CONVERT) / frequencyFraction;
         //snprintf(freqString, sizeof freqString, "%d.%02d MHz", mhz, fraction);
         snprintf(freqString, sizeof freqString, "%d.%02d", mhz, fraction);
      }
      else
      {
         fraction = (frequency % FM_FREQ_CONVERT) / frequencyFraction;
         // snprintf(freqString, sizeof freqString, "%d.%-2dMHz", mhz, fraction);
         snprintf(freqString, sizeof freqString, "%d.%-2d", mhz, fraction);
      }
   }
   else if ((frequency >= _AMLowFrequency) && (frequency <= _AMHighFrequency))
   {
      snprintf(freqString, sizeof freqString, "%d kHz", frequency);
   }
   return freqString;
}


/**
 * Description     : Function implements to convert the given frequency into required format for the display for testmode classes.
 *
 * @param[in]      : frequency : Frequency selected by the user.
 * @param[in]      : band : the band for whcih the frequency is selected.
 * @return         : std::string
 */


std::string TunerUtility::formatTestmodeFrequency(const uint32& frequency, std::string band)
{
   char freqString[ARRAY_SIZE] = "\0";

   uint8 frequencyFraction = 100;
   int mhz = frequency / FM_FREQ_CONVERT;
   int fraction = 0;

   if (band == "FM")
   {
      if (_FMStepSize == 50)
      {
         frequencyFraction = 10;     // Required to show two digits after decimal if step size is 50
         fraction = (frequency % FM_FREQ_CONVERT) / frequencyFraction;
         snprintf(freqString, sizeof freqString, "%d.%02d", mhz, fraction);
      }
      else
      {
         fraction = (frequency % FM_FREQ_CONVERT) / frequencyFraction;
         snprintf(freqString, sizeof freqString, "%d.%-2d", mhz, fraction);
      }
   }
   else if (band == "AM")
   {
      snprintf(freqString, sizeof freqString, "%d", frequency);
   }
   else if (band == "DAB")
   {
      fraction = (frequency % FM_FREQ_CONVERT);
      snprintf(freqString, sizeof freqString, "%d.%03d", mhz, fraction);
   }

   return freqString;
}


/**
 * Description     : Function implements to convert the given EU frequency into required format for the display.
 *
 * @param[in]      : frequency : Frequency selected by the user.
 * @return         : std::string
 */
std::string TunerUtility::formatGadgetFrequency(const uint32& frequency)
{
   char freqString[ARRAY_SIZE] = "\0";

   if ((frequency >= _FMLowFrequency) && (frequency <= _FMHighFrequency))
   {
      uint8 frequencyFraction = 100;
      int mhz = frequency / FM_FREQ_CONVERT;
      int fraction = 0;
      if (_FMStepSize == 50)
      {
         frequencyFraction = 10;     // Required to show two digits after decimal if step size is 50
         fraction = (frequency % FM_FREQ_CONVERT) / frequencyFraction;
         snprintf(freqString, sizeof freqString, "%d.%02d", mhz, fraction);
      }
      else
      {
         fraction = (frequency % FM_FREQ_CONVERT) / frequencyFraction;
         snprintf(freqString, sizeof freqString, "%d.%d", mhz, fraction);
      }
   }
   else if ((frequency >= _AMLowFrequency) && (frequency <= _AMHighFrequency))
   {
      snprintf(freqString, sizeof freqString, "%d", frequency);
   }
   return freqString;
}


/**
 * Description     : Function implements to convert the given frequency into required format for the display.
 *
 * @param[in]      : frequency : Frequency selected by the user.
 * @return         : std::string
 */
std::string TunerUtility::formatFreqwithBand(const uint32& frequency)
{
   char freqString[ARRAY_SIZE] = "\0";
   if ((frequency >= _FMLowFrequency) && (frequency <= _FMHighFrequency))
   {
      uint8 frequencyFraction = 100;
      int mhz = frequency / FM_FREQ_CONVERT;
      int fraction = 0;
      if (_FMStepSize == 50)
      {
         frequencyFraction = 10;     // Required to show two digits after decimal if step size is 50
         fraction = (frequency % FM_FREQ_CONVERT) / frequencyFraction;
         snprintf(freqString, sizeof freqString, "%d.%02d FM", mhz, fraction);
      }
      else
      {
         fraction = (frequency % FM_FREQ_CONVERT) / frequencyFraction;
         snprintf(freqString, sizeof freqString, "%d.%-2d FM", mhz, fraction);
      }
   }
   else if ((frequency >= _AMLowFrequency) && (frequency <= _AMHighFrequency))
   {
      snprintf(freqString, sizeof freqString, "%d AM", frequency);
   }
   return freqString;
}


/**
 * Description     : Function implements to convert uint8 value to string.
 *
 * @param[in]      : u8Value : uint8 value.
 * @return         : std::string
*/
std::string TunerUtility::u8ConversionToString(uint8 u8Value)
{
   char stringToConv[ARRAY_SIZE] = "\0";

   snprintf(stringToConv, sizeof stringToConv, "%d", u8Value);

   return stringToConv;
}


/**
 * Description     : Function implements to convert uint16 value to string.
 *
 * @param[in]      : u16Value : uint16 value.
 * @return         : std::string
 */
std::string TunerUtility::u16ConversionToString(uint16 u16Value)
{
   char stringToConv[ARRAY_SIZE] = "\0";

   snprintf(stringToConv, sizeof stringToConv, "%d", u16Value);

   return stringToConv;
}


/**
 * Description     : Function implements to convert uint32 value to string.
 *
 * @param[in]      : u16Value : uint16 value.
 * @return         : std::string
 */
std::string TunerUtility::u32ConversionToString(uint32 u32Value)
{
   char stringToConv[ARRAY_SIZE] = "\0";

   snprintf(stringToConv, sizeof stringToConv, "%d", u32Value);

   return stringToConv;
}


/**
* Description     : Function implements to convert uint32 value to Hex value string.
*
* @param[in]      : u32Value : uint32 value.
* @return         : std::string
*/
std::string TunerUtility::u32ConversionToHexString(uint32 u32Value)
{
   char stringToConv[ARRAY_SIZE] = "\0";

   snprintf(stringToConv, sizeof stringToConv, "%x", u32Value);

   return stringToConv;
}


/**
 * Description     : Function implements to convert uint16 value to Hex value string.
 *
 * @param[in]      : u16Value : uint16 value.
 * @return         : std::string
 */
std::string TunerUtility::u16ConversionToHexString(uint16 u16Value)
{
   char stringToConv[ARRAY_SIZE] = "\0";

   snprintf(stringToConv, sizeof stringToConv, "%x", u16Value);

   return stringToConv;
}


/**
 * Description     : Function implements to concatenate two values.
 *
 * @param[in]      : u8Value : uint8 value.
 * @param[in]      : u8Value : uint8 value.
 * @return         : std::string
*/
std::string TunerUtility::quadConcatenate(const uint8& u8Val1, const uint8& u8Val2)
{
   char sQuadConcat[ARRAY_SIZE] = "\0";

   snprintf(sQuadConcat, sizeof sQuadConcat, "%d/%d", u8Val1, u8Val2);

   return sQuadConcat;
}


void TunerUtility::setAMFMStepSizeFromKDS(const uint8& KDSAMStepSize, const uint8& KDSFMStepSize)
{
   _AMStepSize = KDSAMStepSize;
   _FMStepSize = KDSFMStepSize;
}


/**
 * Description     : Function implements to format FM direct frequency.
 *
 * @param[in]      : float : currentFreq value.
 * @return         : uint32
*/
uint32 TunerUtility::formatFMDirectFreq(const float& currentFreq)
{
   uint32 fmfreq = 0;
   if ((currentFreq >= _FMLowFrequency) && (currentFreq <= _FMHighFrequency))
   {
      fmfreq = (uint32)(currentFreq / FM_FREQ_FRACTION);
      fmfreq = fmfreq * FM_FREQ_FRACTION;
   }
   return fmfreq;
}


/**
 * Description     : Function implements to format AM direct frequency.
 *
 * @param[in]      : float : currentFreq value.
 * @return         : uint32
*/
uint32 TunerUtility::formatAMDirectFreq(const float& currentFreq)
{
   uint32 frequency = (uint32)currentFreq;
   if ((currentFreq >= _AMLowFrequency) && (currentFreq <= _AMHighFrequency))
   {
      uint32 mod = 0;
      uint32 val = 0;
      if (_AMStepSize != 0)
      {
         mod = frequency % _AMStepSize;
         val = frequency / _AMStepSize;
      }
      frequency = (uint32)val * _AMStepSize;

      if (mod > 0)
      {
         frequency = frequency + _AMStepSize;
      }
   }
   return frequency;
}


/**
 * Description     : Function implements to calculate FM list Start Index.
 *
 * @param[in]      : listSize : FM list size.
 * @param[in]      : selectedElemID : Tuned station position in FM list.
 * @return         : uint32
 */
uint32 TunerUtility::getListStartIndex(const uint8& listSize, const uint32& selectedElemID)
{
   uint8 lastWindow = listSize - START_WINDOW_FMLIST;
   uint8 startIndex = 0;
   if (listSize > LAST_WINDOW_FMLIST)
   {
      if (selectedElemID < START_WINDOW_FMLIST)
      {
         startIndex = 0;
      }
      else if ((selectedElemID >= START_WINDOW_FMLIST) && (selectedElemID < lastWindow))
      {
         startIndex = selectedElemID - THIRD_POS_FMLIST;
      }
      else
      {
         startIndex = listSize - LAST_WINDOW_FMLIST;
      }
   }
   return startIndex;
}


std::string TunerUtility::getSecureString(const Candera::String& string)
{
   static std::string secureString = "";
   SECURE_FEATSTD_STRING_ACCESS_BEGIN(string); // locks the access of GetCString within this scope.
   secureString = string.GetCString();
   SECURE_FEATSTD_STRING_ACCESS_END()
   return secureString;
}


/**
 * Description     : Function to convert Carrier Noise value to valid string format
 *
 * @param[in]      : value : Carrier Noise value
 * @return         : std::string
 */
std::string TunerUtility::formatDTVCNdata(const uint16& value)
{
   std::ostringstream s;
   s << value;
   std::string str = s.str();
   str = str + " dB";
   return str;
}


/**
 * Description     : Function to convert PER value to valid string format
 *
 * @param[in]      : value : PER value
 * @param[in]      : layer : A or B or C
 * @return         : std::string
 */
std::string TunerUtility::formatDTVPERdata(const uint16& value, char layer)
{
   std::ostringstream s;
   s << value;
   std::string str = s.str();
   str = str + " %" + " (" + layer + ")";
   return str;
}


} // namespace Core
} // namespace App
