/**************************************************************************************
* @file         : <XMLParser.cpp>
* @author       : <Ramesh Kesavan> <ECH> <INF4CV>
* @addtogroup   : <AppHmi_System>
* @brief        :
* @copyright    : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "hall_std_if.h"
#include "App/Core/Utils/XMLParser/XMLParser.h"
#include "App/Core/Utils/SWUpdateUtility.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_SYSTEM_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_SYSTEM
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_SYSTEM_"
#define ETG_I_FILE_PREFIX                 App::Core::XMLDocParser::
#include "trcGenProj/Header/XMLParser.cpp.trc.h"
#endif

const char* SWDL_CMC_UPDATE_RESULTS_XML = "/var/opt/bosch/persistent/fcswupdate/doipcmcupdateresult.xml";
const char* SWDL_CMC_INSTAL_RESULTS_XML = "/var/opt/bosch/persistent/fcswupdate/doipcmcinstallresult.xml";
const char* SWDL_CMC_SYSTEM_INSTAL_RESULTS_XML = "/var/opt/bosch/persistent/avdecc/doipcmcinstallresult.xml";
const char* SWDL_CMC_DTC_CODE_STATUS_XML = "/tmp/doipcmcerrmemresult.xml"; // new xml
const char* SWDL_CMC_CONFG_VER_CIS_NET_XML = "/var/opt/bosch/persistent/avdecc/cisinstallation/CIS_network.xml"; // new xml CIS_network_A004448XX62_Setra.xml
const char* SWDL_CMC_CONFG_VER_INST_RES_XML = "/var/opt/bosch/persistent/avdecc/doipcmcinstallresult.xml"; // doipcmcinstallresult.xml //doipinstallresults.xml /var/opt/bosch/persistent/avdecc/doipcmcinstallresult.xml”
const char* SWDL_CMC_DTC_INSTALLED_CODE_STATUS_XML = "/tmp/doipcmcerrmemresult.xml"; // new xml


const char* TAG_TESTER = "tester";
const int LEN_TAG_TESTER = 7;

const char* TAG_RESULT = "result";
const int LEN_TAG_RESULT = 7;

const char* ATTRIB_TESTER_EST_TIME = "timeestim";
const int LEN_ATTRIB_TESTER_EST_TIME = 10;

const char* ATTRIB_RESULT_STATUS = "status";
const int LEN_ATTRIB_RESULT_STATUS = 7;

const char* TAG_NODE = "node";
const int LEN_TAG_NODE = 5;

const char* TAG_VEH_CONFG_ASACH_CIS_NET = "ASachnummer"; //  2.1 Asachnumme --> Vehicle configuration identifier  --> Refer CIS_network.xml
const int LEN_TAG_VEH_ConfG_ASACH_CIS_NET = 12;

const char*  TAG_VEH_CONFG_VER_CIS_NET = "CfgVersion"; //  2.2 CfgVersion--> Configuration version --> Refer CIS_network.xml"
const int LEN_TAG_VEH_CONFG_VER_CIS_NET = 11;


const char*  ATTRIB_NODE_SCRIPTS = "scripts"; //  2.3 Configuration Engine version --> Refer doipinstallresults.xml
const int LEN_ATTRIB_NODE_SCRIPTS = 8;

const char* ATTRIB_NODE_GID = "GID";
const int LEN_ATTRIB_NODE_GID = 4;
//Task 1012776 and CRQ 952224
const char* TAG_ERRMEM = "errmem";
const int LEN_TAG_ERRMEM = 7;

const char* ATTRIB_DTC_CODE = "code";
const int LEN_ATTRIB_DTC_CODE = 5;

const char* ATTRIB_DTC_STATUS = "status";
const int LEN_ATTRIB_DTC_STATUS = 7;

const char* TAG_DTC = "DTC";
const int LEN_TAG_DTC = 4;
//Task 1012776 and CRQ 952224
const char* ATTRIB_NODE_UPDATERESULT = "updateresult";
const int LEN_ATTRIB_NODE_UPDATERESULT = 14;

const char* ATTRIB_NODE_SWVERSION = "newSWversion";
const int LEN_ATTRIB_NODE_SWVERSION = 13;

const char* ATTRIB_NODE_DEVICENAME = "name";
const int LEN_ATTRIB_NODE_DEVICENAME = 5;

const char* ATTRIB_NODE_SERIALNUMBER = "serialnum";
const int LEN_ATTRIB_NODE_SERIALNUMBER = 10;

const char* ATTRIB_NODE_MACADDRESS = "mac";
const int LEN_ATTRIB_NODE_MACADDRESS = 4;

const char* SUCCESSFUL = "Successful";
const char* FAILED     = "Failed";

const char* EMPTYSTR = "";
const char* EQUAL = "Equal";
const char* UPDATED = "Updated";
const char* UNKNOWN = "Unknown";


using namespace std;
namespace App {
namespace Core {

const int UPDATE_RESULTS_MAX_INDEX = 11;
const int ENTITY_NAME_MAX_ROW_INDEX = 11;
string UpdateResultStr[UPDATE_RESULTS_MAX_INDEX] =
{
   "Equal",
   "Updated",
   "ConnectFail",
   "SecAccFail",
   "PrepFail",
   "TransferFail",
   "SignalFail",
   "InstFail",
   "Timedout",
   "ReconnectFail",
   "UnKnown"
};


string EntityName[ENTITY_NAME_MAX_ROW_INDEX][2] =
{
   { "0001", "Center" },
   { "0002", "Amplifier" },
   { "0003", "Router" },
   { "0004", "HDTV" },
   { "0005", "CameraPort" },
   { "0006", "Glass" },
   { "0007", "ControlPanel" },
   { "0008", "Intercom" },
   { "0009", "SwivelHolder" },
   { "0010", "Testbed" },
   { "0011", "KDS" }
};


XMLDocParser* XMLDocParser::m_xmlParser = NULL;

/**
* @Constructor
*/
XMLDocParser::XMLDocParser() :
   _xmlDoc(SWDL_CMC_UPDATE_RESULTS_XML)
   , _xmlInstallResultDoc(SWDL_CMC_INSTAL_RESULTS_XML)
   , _xmlSystemInstallResultDoc(SWDL_CMC_SYSTEM_INSTAL_RESULTS_XML)
   //Task 1012776 and CRQ 952224
   , _xmlDtcCodeStatusDoc(SWDL_CMC_DTC_CODE_STATUS_XML)
   , xmlConfigVerCISnetworkDoc(SWDL_CMC_CONFG_VER_CIS_NET_XML) //new xml CIS_network.xml
   , xmlConfigVerDoIpcInstallResultDoc(SWDL_CMC_CONFG_VER_INST_RES_XML) // doipcmcinstallresult.xml
   , _xmlDtcInstallaedCodeStatusDoc(SWDL_CMC_DTC_INSTALLED_CODE_STATUS_XML)
   //Task 1012776 and CRQ 952224
   , _stageOneEstimatedTime(0)
   , _Status_Str("")
   , _nodeTagCounter(0)
   , NodeIndex(0)
   ////New XML Adaption////
   , _gidAtrribCounter(0)
   , _updateResultAtrribCounter(0)
   , _SWVersionAtrribCounter(0)
   , _DeviceNameAtrribCounter(0)
   //Task 1012776 and CRQ 952224
   , _deviceName("")
   , _isNodeTagAvailable(false)
   , _dtcCounter(0)
   //Task 1012776 and CRQ 952224
   , _isTesterTagAvailable(false)
   , _isSuccess(false)
   , isValidItem(false)
{
   _confgVerCisNetworkVecItem.clear();
   _confgVerdoipcinstallVecItem.clear();
   ETG_TRACE_USR4(("XMLDocParser Constructor is called"));
}


/**
*  @Singleton Instance
*/
XMLDocParser* XMLDocParser::GetInstance()
{
   if (NULL == m_xmlParser)
   {
      m_xmlParser = new XMLDocParser();
   }
   return m_xmlParser;
}


/**
* @Destructor
*/
XMLDocParser::~XMLDocParser()
{
   _statusReportVec.clear();
   _CISstatusReportVec.clear();
   ETG_TRACE_USR4(("XMLDocParser Destructor is called"));
}


/**
* GetNecessaryInfo - For Extracting information from XML doc
*/
int XMLDocParser::GetNecessaryInfo(TiXmlNode* pParent, int NodeEnum)
{
   if (!pParent)
   {
      ETG_TRACE_USR4(("int XMLDocParser::GetNecessaryInfo(TiXmlNode* pParent, int NodeEnum) !pParent"));
      return 0;
   }

   TiXmlNode* pChild;

   int NodeType = pParent->Type();
   ETG_TRACE_USR4(("int XMLDocParser::GetNecessaryInfo(TiXmlNode* pParent, int NodeEnum)NodeType head is pParent %d ", NodeType));
   switch (NodeType)
   {
      case TiXmlNode::TINYXML_ELEMENT:
      {
         string TagName = pParent->Value();
         ETG_TRACE_USR4(("int XMLDocParser::GetNecessaryInfo(TiXmlNode* pParent, int NodeEnum) inside case TagName head is pParent %s ", TagName.c_str()));
         //int NodeTypeEnum = 0;       // 0- def, 1 - Node Tag, 2 - Tester Tag
         if (strncmp(TagName.c_str(), TAG_NODE, LEN_TAG_NODE) == 0)
         {
            _nodeTagCounter++;
            ExractTagInformation(pParent->ToElement());
            vecDTCInformation.push_back(_DTCinfo);
            memset(&_DTCinfo, 0, sizeof(_DTCinfo));
            //NodeTypeEnum = 1;
            _isNodeTagAvailable = true;
            ETG_TRACE_USR4(("int XMLDocParser::GetNecessaryInfo(TiXmlNode* pParent, int NodeEnum) inside case TAG_NODE _nodeTagCounter %d ", _nodeTagCounter));
         }
         else if (strncmp(TagName.c_str(), TAG_VEH_CONFG_ASACH_CIS_NET, LEN_TAG_VEH_ConfG_ASACH_CIS_NET) == 0)   // 1 --> ASACH_CIS_NET
         {
            TiXmlNode*  asachChild = pParent->FirstChild();
            congVerCisNetworkObj.ConfigStr = TagName;              // ASachnummer
            ETG_TRACE_USR4(("int XMLDocParser::GetNecessaryInfo TAG_VEH_CONFG_ASACH_CIS_NET iExractTagInformation congVerCisNetworkObj.ConfigStr::%s \n", congVerCisNetworkObj.ConfigStr.c_str()));
            if (asachChild != NULL)
            {
               congVerCisNetworkObj.ConfigVal = asachChild->Value();	 // ASachnummer Value
            }
            else
            {
               congVerCisNetworkObj.ConfigVal = "Unknown";	 // ASachnummer Value
            }
            ETG_TRACE_USR4(("int XMLDocParser::GetNecessaryInfo TAG_VEH_CONFG_ASACH_CIS_NET iExractTagInformation congVerCisNetworkObj.ConfigVal::%s \n", congVerCisNetworkObj.ConfigVal.c_str()));
            _confgVerCisNetworkVecItem.push_back(congVerCisNetworkObj);
         }
         else if (strncmp(TagName.c_str(), TAG_VEH_CONFG_VER_CIS_NET, LEN_TAG_VEH_CONFG_VER_CIS_NET) == 0)      //2 --> CONFG_VER_CIS_NET
         {
            TiXmlNode*  configVerChild = pParent->FirstChild();
            congVerCisNetworkObj.ConfigStr = TagName;                // ASachnummer string
            ETG_TRACE_USR4(("int XMLDocParser::GetNecessaryInfo TAG_VEH_CONFG_VER_CIS_NET iExractTagInformation congVerCisNetworkObj.ConfigStr:: %s \n", congVerCisNetworkObj.ConfigStr.c_str()));
            if (configVerChild != NULL)
            {
               congVerCisNetworkObj.ConfigVal = configVerChild->Value();	 	 // ASachnummer Value
            }
            else
            {
               congVerCisNetworkObj.ConfigVal = "Unknown";	 // ASachnummer Value is unknown
            }
            ETG_TRACE_USR4(("int XMLDocParser::GetNecessaryInfo TAG_VEH_CONFG_VER_CIS_NET iExractTagInformation congVerCisNetworkObj.ConfigVal:: %s \n", congVerCisNetworkObj.ConfigVal.c_str()));
            _confgVerCisNetworkVecItem.push_back(congVerCisNetworkObj);
         }
         else if (strncmp(TagName.c_str(), TAG_TESTER, LEN_TAG_TESTER) == 0)
         {
            ETG_TRACE_USR4(("int XMLDocParser::GetNecessaryInfo(TiXmlNode* pParent, int NodeEnum) inside case TAG_TESTER  "));
            ExractTagInformation(pParent->ToElement());
            _isTesterTagAvailable = true;
         }
         else if (strncmp(TagName.c_str(), TAG_RESULT, LEN_TAG_RESULT) == 0)
         {
            //Task 1012776 and CRQ 952224
            //xmlscript for dtc -- 22.12.2020
            ETG_TRACE_USR4(("int XMLDocParser::GetNecessaryInfo(TiXmlNode* pParent, int NodeEnum) TAG_RESULT \n"));
            ExractTagInformation(pParent->ToElement());
         }
         else if (strncmp(TagName.c_str(), TAG_ERRMEM, LEN_TAG_ERRMEM) == 0)
         {
            ETG_TRACE_USR4(("int XMLDocParser::GetNecessaryInfo(TiXmlNode* pParent, int NodeEnum) TAG_ERRMEM \n"));
            //xmlscript for dtc -- 22.12.2020
            //Task 1012776 and CRQ 952224
            ExractTagInformation(pParent->ToElement());
         }
         else
         {
            ETG_TRACE_USR4(("int XMLDocParser::GetNecessaryInfo(TiXmlNode* pParent, int NodeEnum) ELSE CONDITION \n"));
         }
      }
      break;
      case TiXmlNode::TINYXML_TEXT:
      {
         TiXmlText* pText;
         if (NodeEnum == 1)
         {
            pText = pParent->ToText();
            ETG_TRACE_USR4(("int XMLDocParser::GetNecessaryInfo(TiXmlNode* pParent, int NodeEnum) pText 1 ==> %d\n", pText));
         }
         else if (NodeEnum == 2)
         {
            pText = pParent->ToText();
            ETG_TRACE_USR4(("int XMLDocParser::GetNecessaryInfo(TiXmlNode* pParent, int NodeEnum) pText 2 ==> %d\n", pText));
         }
         else if (NodeEnum == 3)
         {
            ETG_TRACE_USR4(("int XMLDocParser::GetNecessaryInfo(TiXmlNode* pParent, int NodeEnum) pText 3 ==> %d\n", pText));
         }
         else
         {
            ETG_TRACE_USR4(("int XMLDocParser::GetNecessaryInfo(TiXmlNode* pParent, int NodeEnum) pText else ==> %d\n", pText));
         }
         ETG_TRACE_USR4(("int XMLDocParser::GetNecessaryInfo(TiXmlNode* pParent, int NodeEnum)pText NodeEnum is %s ", pText->Value()));
      }
      break;

      default:
         ETG_TRACE_USR4(("int XMLDocParser::GetNecessaryInfo(TiXmlNode* pParent, int NodeEnum) default CONDITION \n"));
         break;
   }
   ETG_TRACE_USR4(("int XMLDocParser::GetNecessaryInfo(TiXmlNode* pParent, int NodeEnum) pChild =%d ", pChild));
   for (pChild = pParent->FirstChild(); pChild != 0; pChild = pChild->NextSibling())
   {
      string childdata = pChild->Value();
      ETG_TRACE_USR4(("int XMLDocParser::GetNecessaryInfo(TiXmlNode* pParent, int NodeEnum) pChild =%d ", pChild));
      ETG_TRACE_USR4(("int XMLDocParser::GetNecessaryInfo(TiXmlNode* pParent, int NodeEnum) Next Child is  %s", childdata.c_str()));
      GetNecessaryInfo(pChild, NodeType); // chagned NodeTypeEnum to NodeType for testing
   }
   return 1;
}


/**
* ExractTagInformation - For Extracting information from XML doc
* @param[in] TiXmlElement
* @return   : extracted string
*/
int XMLDocParser::ExractTagInformation(TiXmlElement* pElement)
{
   ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement) is called"));

   UpdReport tempObj;
   UpdReport CISObj;
   SWUpdateReport SWUpdDetailsObj;
   isValidItem = false;

   if (!pElement)
   {
      ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement) is called !pElement "));
      return 0;
   }

   if (pElement->FirstAttribute() != NULL)
   {
      TiXmlAttribute* pAttrib = pElement->FirstAttribute();

      //ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement) pAttrib head %d", pAttrib));
      while (pAttrib)
      {
         // const char* pAttribName = pAttrib->Name();
         // const char* pAttribValue = pAttrib->Value();
         const char* pAttribName = "";
         const char* pAttribValue = "";
         if (pAttrib->Name() != NULL)
         {
            pAttribName = pAttrib->Name();
         }
         if (pAttrib->Value() != NULL)
         {
            pAttribValue = pAttrib->Value();
         }

         if (strncmp(pAttribName, ATTRIB_NODE_GID, LEN_ATTRIB_NODE_GID) == 0)
         {
            tempObj.nameStr = ComponentMappingStr(pAttribValue, false);
            isValidItem = true;
            _gidAtrribCounter++;
            ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement) Exracted ATTRIB_NODE_GID nameStr is %s", tempObj.nameStr.c_str()));
            ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement) _gidAtrribCounter is %d", _gidAtrribCounter));
         }
         else if (strncmp(pAttribName, ATTRIB_NODE_SCRIPTS, LEN_ATTRIB_NODE_SCRIPTS) == 0)   // 3---> SCRIPTS
         {
            congVerdoipcinstallObj.ConfigStr = pAttribName;
            ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement) Exracted ATTRIB_NODE_SCRIPTS LEN_ATTRIB_NODE_SCRIPTS congVerdoipcinstallObj.ConfigStr :: %s", congVerdoipcinstallObj.ConfigStr));
            //int strlenvalue=strlen(pAttribValue);
            //ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement) Exracted ATTRIB_NODE_SCRIPTS LEN_ATTRIB_NODE_SCRIPTS  strlenvalue==> %s",strlenvalue));
            //ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement) Exracted ATTRIB_NODE_SCRIPTS LEN_ATTRIB_NODE_SCRIPTS congVerdoipcinstallObj.ConfigVal outside pAttribValue :: %s",pAttribValue));
            //if(pAttribValue!=NULL)

            if (strncmp(pAttribValue, "", 1) == 0)
            {
               congVerdoipcinstallObj.ConfigVal = "Unknown";
               ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement) Exracted ATTRIB_NODE_SCRIPTS LEN_ATTRIB_NODE_SCRIPTS congVerdoipcinstallObj.ConfigVal inside if(strlen(pAttribValue)==0) :: %s", congVerdoipcinstallObj.ConfigVal));
            }
            else
            {
               congVerdoipcinstallObj.ConfigVal = pAttribValue;
               ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement) Exracted ATTRIB_NODE_SCRIPTS LEN_ATTRIB_NODE_SCRIPTS congVerdoipcinstallObj.ConfigVal inside else :: %s", congVerdoipcinstallObj.ConfigVal));
            }
            _confgVerdoipcinstallVecItem.push_back(congVerdoipcinstallObj);
         }
         else if (strncmp(pAttribName, ATTRIB_NODE_UPDATERESULT, LEN_ATTRIB_NODE_UPDATERESULT) == 0)
         {
            // resetting for current module status
            _isSuccess = false;
            tempObj.resultStr = ComponentMappingStr(pAttribValue, true);
            _updateResultAtrribCounter++;
            isValidItem = true;

            // this bool is update inside component Mapping string for update result string
            tempObj.isUpdateSuccess = _isSuccess;

            ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement) Exracted ATTRIB_NODE_UPDATERESULT resultStr is %s", tempObj.resultStr.c_str()));
            ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement) _updateResultAtrribCounter is %d", _updateResultAtrribCounter));
         }

         else if (strncmp(pAttribName, ATTRIB_NODE_SWVERSION, LEN_ATTRIB_NODE_SWVERSION) == 0)
         {
            CISObj.resultStr = pAttribValue;
            isValidItem = true;
            _SWVersionAtrribCounter++;
            ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement) Exracted ATTRIB_NODE_SWVERSION resultStr is %s", CISObj.resultStr.c_str()));
            ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement) _SWVersionAtrribCounter is %d", _SWVersionAtrribCounter));
         }

         else if (strncmp(pAttribName, ATTRIB_NODE_DEVICENAME, LEN_ATTRIB_NODE_DEVICENAME) == 0)
         {
            CISObj.nameStr = pAttribValue;
            SWUpdDetailsObj.nameStr = pAttribValue;
            //	DTCInfo objDTCinfo;
            //	objDTCinfo.subSystemName=pAttribValue;
            _DTCinfo.subSystemName = pAttribValue;
            //	vecDTCInformation.push_back(objDTCinfo);
            _deviceName = pAttribValue;
            isValidItem = true;
            _DeviceNameAtrribCounter++;
            ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement)Exracted ATTRIB_NODE_DEVICENAME nameStr _deviceName is %s ", CISObj.nameStr.c_str()));
            ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement) _DeviceNameAtrribCounter %d", _DeviceNameAtrribCounter));
         }
         else if (strncmp(pAttribName, ATTRIB_NODE_SERIALNUMBER, LEN_ATTRIB_NODE_SERIALNUMBER) == 0)
         {
            SWUpdDetailsObj.serialnumberStr = pAttribValue;
            isValidItem = true;
            _SerialnumberCounter++;
            ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement) Exracted ATTRIB_NODE_SERIALNUMBER serialnumberStr is %s", SWUpdDetailsObj.serialnumberStr.c_str()));
            ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement) _SerialnumberCounter is %d", _SerialnumberCounter));
         }
         else if (strncmp(pAttribName, ATTRIB_NODE_MACADDRESS, LEN_ATTRIB_NODE_MACADDRESS) == 0)
         {
            SWUpdDetailsObj.macAddressStr = pAttribValue;
            isValidItem = true;
            _MACaddressCounter++;
            ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement) Exracted macAddressStr macAddressStr is %s", SWUpdDetailsObj.macAddressStr.c_str()));
            ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement) _MACaddressCounter is %d", _MACaddressCounter));
         }

         else if (strncmp(pAttribName, ATTRIB_TESTER_EST_TIME, LEN_ATTRIB_TESTER_EST_TIME) == 0)
         {
            std::string timeestimStr = pAttribValue;
            if (!timeestimStr.empty())
            {
               _stageOneEstimatedTime = std::atol(timeestimStr.c_str());
               // isValidItem = true;
               ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement) Exracted ATTRIB_TESTER_EST_TIME Estimated Time is %u", _stageOneEstimatedTime));
            }
         }
         else if (strncmp(pAttribName, ATTRIB_RESULT_STATUS, LEN_ATTRIB_RESULT_STATUS) == 0)
         {
            _Status_Str = pAttribValue;
            // isValidItem = true;
            ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement) Exracted ATTRIB_RESULT_STATUS Result Status %s", _Status_Str.c_str()));
         }
         else
         {
            //nothing
         }
         pAttrib = pAttrib->Next();
      }//end of while loop
   }//end of if after while

   if (_nodeTagCounter > _gidAtrribCounter)
   {
      tempObj.nameStr = UNKNOWN;
      _gidAtrribCounter++;
      ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement)--\"GID\" _gidAtrribCounter %d", _gidAtrribCounter));
      isValidItem = true;
   }
   if (_nodeTagCounter > _updateResultAtrribCounter)
   {
      tempObj.resultStr = UNKNOWN;
      _updateResultAtrribCounter++;
      ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement)--\"GID\" _gidAtrribCounter %d", _updateResultAtrribCounter));
      isValidItem = true;
   }

   if (_nodeTagCounter > _SWVersionAtrribCounter)
   {
      CISObj.resultStr = UNKNOWN;
      _SWVersionAtrribCounter++;
      ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement)-- \"SWVersion\" attribute is missing in Node Tag. subsituting Unknown string %d", _SWVersionAtrribCounter));
      isValidItem = true;
   }
   if (_nodeTagCounter > _DeviceNameAtrribCounter)
   {
      CISObj.nameStr = UNKNOWN;
      SWUpdDetailsObj.nameStr = UNKNOWN;
      _DeviceNameAtrribCounter++;
      ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement)-- \"SWVersion\" attribute is missing in Node Tag. subsituting Unknown string==> %d", _DeviceNameAtrribCounter));
      isValidItem = true;
   }
   if (_nodeTagCounter > _SerialnumberCounter)
   {
      SWUpdDetailsObj.serialnumberStr = UNKNOWN;
      _SerialnumberCounter++;
      ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement)-- \"SerialNumber\" attribute is missing in Node Tag. subsituting Unknown string==> %d", _SerialnumberCounter));
      isValidItem = true;
   }
   if (_nodeTagCounter > _MACaddressCounter)
   {
      SWUpdDetailsObj.macAddressStr = UNKNOWN;
      _MACaddressCounter++;
      ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement)-- \"MacAddress\" attribute is missing in Node Tag. subsituting Unknown string==> %d", _MACaddressCounter));
      isValidItem = true;
   }

   if (isValidItem)
   {
      if (tempObj.nameStr.empty())
      {
         tempObj.nameStr = UNKNOWN;
         ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement)-- Temp obj Name string is having empty str. subsituting Unknown string==> =%s", tempObj.nameStr.c_str()));
      }
      if (tempObj.resultStr.empty())
      {
         tempObj.resultStr = UNKNOWN;
         ETG_TRACE_USR3(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement)-- Temp obj Name Result is having empty str. subsituting Unknown string==>%s", tempObj.resultStr.c_str()));
      }

      if (CISObj.nameStr.empty())
      {
         CISObj.nameStr = UNKNOWN;
         ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement)-- CIS obj Name string is having empty str. subsituting Unknown string==> %s", CISObj.nameStr.c_str()));
      }
      if (CISObj.resultStr.empty())
      {
         CISObj.resultStr = UNKNOWN;
         ETG_TRACE_USR3(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement)-- CIS obj Result string is having empty str. subsituting Unknown string==> %s", CISObj.resultStr.c_str()));
      }
      if (SWUpdDetailsObj.nameStr.empty())
      {
         SWUpdDetailsObj.nameStr = UNKNOWN;
         ETG_TRACE_USR3(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement)-- SWUpdDetailsObj.nameStr obj Result string is having empty str. subsituting Unknown string==> %s", SWUpdDetailsObj.nameStr.c_str()));
      }
      if (SWUpdDetailsObj.serialnumberStr.empty())
      {
         SWUpdDetailsObj.serialnumberStr = UNKNOWN;
         ETG_TRACE_USR3(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement)-- SWUpdDetailsObj.serialnumberStr obj Result string is having empty str. subsituting Unknown string==> %s", SWUpdDetailsObj.serialnumberStr.c_str()));
      }
      if (SWUpdDetailsObj.macAddressStr.empty())
      {
         SWUpdDetailsObj.macAddressStr = UNKNOWN;
         ETG_TRACE_USR3(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement)-- SWUpdDetailsObj.macAddressStr obj Result string is having empty str. subsituting Unknown string==> %s", SWUpdDetailsObj.macAddressStr.c_str()));
      }
      // This condition is used to avoid empty node being pushed to the vector
      _statusReportVec.push_back(tempObj);
      _CISstatusReportVec.push_back(CISObj);
      _SerialandMACAddressReportVec.push_back(SWUpdDetailsObj);
      ETG_TRACE_USR3(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement)-- changes pushed to vector _status and _cis"));
   }
   //Task 1012776 and CRQ 952224
   TiXmlNode* pChild;
   int NodeType = pElement->Type(); 						//If node is element node==>the nodeType property will return 1.//If node is attribute node==>the nodeType property will return 2.
   //If node is text node==>the nodeType property will return 3.//If node is comment node==>the nodeType property will return 8.
   int GetNecessaryInfo_DTCStatus = -1;
   // point to first child i.e DTC ... condition check is child element is not null ..... and increament to next pchild to next siblings

   for (pChild = pElement->FirstChild(); pChild != 0; pChild = pChild->NextSibling())
   {
      // Invoke for every DTC - one by one
      // send pointer of child , nodetype , and devicename
      ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement)-- GetNecessaryInfo_DTC checking CODE and STATUS in DTC--pChild and NodeType %d \t %d \n  ", pChild, NodeType));
      //  GetNecessaryInfo_DTCStatus = GetNecessaryInfo_DTC(pChild, NodeType, _deviceName,NodeIndex);
      // chagned NodeTypeEnum to NodeType for testing
      GetNecessaryInfo_DTCStatus = GetNecessaryInfo_DTC(pChild, NodeType, _deviceName, _nodeTagCounter);

      //	NodeIndex++;
      ETG_TRACE_USR4(("int XMLDocParser::ExractTagInformation(TiXmlElement* pElement)--GetNecessaryInfo_DTCStatus %d", GetNecessaryInfo_DTCStatus));
   }
   //Task 1012776 and CRQ 952224
   return 1;
}


//Task 1012776 and CRQ 952224--> Get DTC Information
int XMLDocParser::GetNecessaryInfo_DTC(TiXmlNode* pParent, int NodeEnum, const char* _deviceName, int NodeIndex)
{
   static int AmpCodeloop = 0;
   int retval = 0;
   ETG_TRACE_USR4(("int XMLDocParser::GetNecessaryInfo_DTC(TiXmlNode* pParent, int NodeEnum, const char* _deviceName) ENTERING- NodeInde=%d)", NodeIndex));

   if (!pParent)
   {
      ETG_TRACE_USR4(("int XMLDocParser::GetNecessaryInfo_DTC(TiXmlNode* pParent, int NodeEnum, const char* _deviceName) pParent Status"));
      return 0;
   }
   if (_deviceName != NULL)
   {
      const char* _deviceNameInDTCNode = _deviceName;
      int NodeType = pParent->Type();

      ETG_TRACE_USR4(("int XMLDocParser::GetNecessaryInfo_DTC(TiXmlNode* pParent, int NodeEnum, const char* _deviceName)  _deviceNameInDTCNode  %s  ", _deviceNameInDTCNode));
      ETG_TRACE_USR4(("int XMLDocParser::GetNecessaryInfo_DTC(TiXmlNode* pParent, int NodeEnum, const char* _deviceName)  NodeType= %d ", NodeType));

      switch (NodeType)
      {
         case TiXmlNode::TINYXML_ELEMENT:
         {
            string TagName = "";
            if (pParent->Value() != NULL)
            {
               TagName = pParent->Value();
            }		 // tagName property returns the tag name of the element-->Return a string in UPPERCASE like "DTC, CODE, STATUS"
            ETG_TRACE_USR4(("int XMLDocParser::GetNecessaryInfo_DTC(TiXmlNode* pParent, int NodeEnum, const char* _deviceName) TagName= %s", TagName));
            if (strncmp(TagName.c_str(), TAG_DTC, LEN_TAG_DTC) == 0)		// Compare dtc for each subsystem
            {
               ETG_TRACE_USR4(("int XMLDocParser::GetNecessaryInfo_DTC(TiXmlNode* pParent, int NodeEnum, const char* _deviceName) NodeIndex= %d", NodeIndex));
               ETG_TRACE_USR4(("int XMLDocParser::GetNecessaryInfo_DTC(TiXmlNode* pParent, int NodeEnum, const char* _deviceName) _deviceNameInDTCNode=%s", _deviceNameInDTCNode));
               ExtractDTCTagInformation(pParent->ToElement(), _deviceNameInDTCNode, NodeIndex);
               retval = 1;
            }
            else
            {
               retval = 0;
               ETG_TRACE_USR4(("int XMLDocParser::GetNecessaryInfo_DTC(TiXmlNode* pParent, int NodeEnum, const char* _deviceName) ELSE CONDITIN "));
            }
         }
         break;
         default:
            ETG_TRACE_USR4(("int XMLDocParser::GetNecessaryInfo(TiXmlNode* pParent, int NodeEnum) default CONDITION \n"));
            break;
      }
   }
   return retval;
}


//Task 1012776 and CRQ 952224
int XMLDocParser::ExtractDTCTagInformation(TiXmlElement* pElement, const char* _deviceNameInExtTagInfo, int NodeIndex)
{
   ETG_TRACE_USR4(("int XMLDocParser::ExtractDTCTagInformation(TiXmlElement* pElement, const char* _deviceNameInExtTagInfo)= %s", _deviceNameInExtTagInfo));
   if (!pElement)
   {
      ETG_TRACE_USR4(("int XMLDocParser::ExtractDTCTagInformation(TiXmlElement* pElement, const char* _deviceNameInExtTagInfo) \n"));
      return 0;
   }

   if (pElement->FirstAttribute() != NULL)
   {
      TiXmlAttribute* pAttrib = pElement->FirstAttribute(); // here first pElement->FirstAttribute() is pointing to the code (i.e name)
      //ETG_TRACE_USR4(("int XMLDocParser::ExtractDTCTagInformation(TiXmlElement* pElement, const char* _deviceNameInExtTagInfo) pAttrib==> %d", pAttrib));
      while (pAttrib)
      {
         const char* pAttribName = "";
         const char* pAttribValue = "";
         if (pAttrib->Name() != NULL)
         {
            pAttribName = pAttrib->Name();	// code or status --> pointer for the attribute name
         }
         if (pAttrib->Value() != NULL)
         {
            pAttribValue = pAttrib->Value();  //--> pointer for the attribute value
         }
         ETG_TRACE_USR4(("int XMLDocParser::ExtractDTCTagInformation(TiXmlElement* pElement, const char* _deviceNameInExtTagInfo) pAttribName==> %d", pAttribName));
         ETG_TRACE_USR4(("int XMLDocParser::ExtractDTCTagInformation(TiXmlElement* pElement, const char* _deviceNameInExtTagInfo) pAttribValue==> %d", pAttribValue));
         if ((strncmp(pAttribName, "code", LEN_ATTRIB_DTC_CODE) == 0))//pointer pAttribName is comparing with "code "
         {
            _dtc.dtcCode = pAttribValue ;
            ETG_TRACE_USR4(("int XMLDocParser::ExtractDTCTagInformation(TiXmlElement* pElement, const char* _deviceNameInExtTagInfo)==>  _dtc.dtcCode===> %d",  pAttribValue));
         }// End of if - Based on the Code Type

         else if (strncmp(pAttribName, "status", 7) == 0) 		// Comparsion of Attirbute based on Status Type
         {
            _dtc.dtcStatus = pAttribValue;
            _DTCinfo.dtc.push_back(_dtc);
            ETG_TRACE_USR4(("int XMLDocParser::ExtractDTCTagInformation(TiXmlElement* pElement, const char* _deviceNameInExtTagInfo)==>  _dtc.dtcStatus===> %d",  pAttribValue));

            _dtcCounter++;
         }// End of ELSE if - Based on the Status Type
         else
         {
            ETG_TRACE_USR4(("int XMLDocParser::ExtractDTCTagInformation(TiXmlElement* pElement, const char* _deviceNameInExtTagInfo) ELSE Condition of CODE AND STATUS")); //do nothing
         }//end of ELSE for code and status type
         pAttrib = pAttrib->Next();
      }//end of while loop
   }//END OF if after while
   return 1;
}


/**
* ComponentMappingStr - For Maping string to predefined Enumeration
* @param[in] instr
* @param[in] is the instr represent an Enumeration
* @return strings : extracted string content
*/
string XMLDocParser::ComponentMappingStr(string instr, bool IsEnumStr)
{
   ETG_TRACE_USR4(("XMLDocParser ComponentMappingStr is called"));
   string valueStr("");

   if (!instr.empty())
   {
      bool isFound = false;

      if (!IsEnumStr)
      {
         string componentId = instr.substr(0, 4);

         for (int idx = 0; ((idx < ENTITY_NAME_MAX_ROW_INDEX) && (!isFound)); idx++)
         {
            string matchId = EntityName[idx][0].substr(0, 4);

            ETG_TRACE_USR4(("XMLDocParser : componentId : %s", componentId.c_str()));
            ETG_TRACE_USR4(("XMLDocParser : matchId : %s", matchId.c_str()));

            if (strncmp(componentId.c_str(), matchId.c_str(), componentId.length()) == 0)
            {
               isFound = true;
               valueStr = EntityName[idx][1].c_str();
               ETG_TRACE_USR4(("XMLDocParser : GID : %s", EntityName[idx][0].c_str()));
               ETG_TRACE_USR4(("XMLDocParser : Component name : %s", EntityName[idx][1].c_str()));
            }
         }
      }
      else
      {
         string tempStr = instr.c_str();

         ETG_TRACE_USR4(("XMLDocParser : XML Result string : %s", tempStr.c_str()));

         if (strncmp(EQUAL, tempStr.c_str(), tempStr.length()) == 0)
         {
            isFound = true;
            _isSuccess = true;
            valueStr = EQUAL;
         }
         else if (strncmp(UPDATED, tempStr.c_str(), tempStr.length()) == 0)
         {
            isFound = true;
            _isSuccess = true;
            valueStr = SUCCESSFUL;
         }
         else
         {
            for (int idx = 2; ((idx < UPDATE_RESULTS_MAX_INDEX) && (!isFound)); idx++)
            {
               if (strncmp(UpdateResultStr[idx].c_str(), tempStr.c_str(), tempStr.length()) == 0)
               {
                  // check for other update Result
                  isFound = true;
                  _isSuccess = false;
                  valueStr = FAILED;
               }
            }
         }
      }

      if (!isFound)
      {
         // if the expected strings are not found, return the string as unknown
         valueStr = UNKNOWN;
         ETG_TRACE_USR4(("XMLDocParser : returning  : %s", valueStr.c_str()));
      }
   } // end of if (!instr.empty())

   return valueStr;
}


/**
* GetResultUpdates - Loads the xml for information extraction
* @return   - success or failure
*/
bool XMLDocParser::FetchUpdateResults()
{
   ETG_TRACE_USR4(("XMLDocParser - Enter"));
   _isTesterTagAvailable = false;
   // _isNodeTagAvailable=false;
   _nodeTagCounter = 0;
   _gidAtrribCounter = 0;
   _updateResultAtrribCounter = 0;
   _SWVersionAtrribCounter = 0;
   _DeviceNameAtrribCounter = 0;
   _SerialnumberCounter = 0;
   _MACaddressCounter = 0;

   bool result = _xmlDoc.LoadFile();

   if (result)
   {
      ETG_TRACE_USR4(("XMLDocParser - XML load Successful"));
      _statusReportVec.clear();
      _CISstatusReportVec.clear();
      _SerialandMACAddressReportVec.clear();
      GetNecessaryInfo(&_xmlDoc, false);
      if (!_isTesterTagAvailable)
      {
         ETG_TRACE_USR4(("XMLDocParser - XML load Successful. But <tester> Tag is not Found. Estimate update is Missing"));
         result = false;
      }
   }
   else
   {
      ETG_TRACE_ERR(("XMLDocParser - XML load Failed %s", _xmlDoc.ErrorDesc()));
   }

   ETG_TRACE_USR4(("XMLDocParser - Exit"));
   return result;
}


////New XML Adaption ////
int XMLDocParser::FetchTotalNodeNumberInfo()
{
   int totalnode = 0;
   bool result = _xmlDtcCodeStatusDoc.LoadFile();
   if (result)
   {
      ETG_TRACE_ERR(("XMLDocParser - XML load Successful\n"));
      totalnode = GetTotalNodeInfo(&_xmlDtcCodeStatusDoc, false);
   }
   else
   {
      ETG_TRACE_ERR(("XMLDocParser - XML load Failed\n"));
   }

   ETG_TRACE_ERR(("void XMLDocParser::FetchTotalNodeNumberInfo(); total number of nodes are %d", totalnode));
   return totalnode;
}


int XMLDocParser::GetTotalNodeInfo(TiXmlNode* pParent, int NodeEnum)
{
   static int totnodecounter = 0;
   if (!pParent)
   {
      ETG_TRACE_ERR(("int XMLDocParser::GetTotalNodeInfo(TiXmlNode* pParent, int NodeEnum) !pParent"));
   }
   else
   {
      TiXmlNode* pChild;
      int NodeType = pParent->Type();
      ETG_TRACE_ERR(("int XMLDocParser::GetTotalNodeInfo(TiXmlNode* pParent, int NodeEnum)NodeType head is pParent %d", NodeType));
      switch (NodeType)
      {
         case TiXmlNode::TINYXML_ELEMENT:
         {
            string TagName = pParent->Value();
            ETG_TRACE_ERR(("int XMLDocParser::GetTotalNodeInfo(TiXmlNode* pParent, int NodeEnum) inside case TagName head is pParent %s", TagName));
            //int NodeTypeEnum = 0;       // 0- def, 1 - Node Tag, 2 - Tester Tag
            if (strncmp(TagName.c_str(), TAG_NODE, LEN_TAG_NODE) == 0)
            {
               totnodecounter ++;
               ETG_TRACE_ERR(("int XMLDocParser::GetTotalNodeInfo(TiXmlNode* pParent, int NodeEnum) inside case TAG_NODE totnodecounter %d ", totnodecounter));
            }
         }
         break;
         default:
            ETG_TRACE_ERR(("int XMLDocParser::GetTotalNodeInfo(TiXmlNode* pParent, int NodeEnum) default CONDITION "));
            break;
      }
      // ETG_TRACE_USR4(("int XMLDocParser::GetTotalNodeInfo(TiXmlNode* pParent, int NodeEnum) pChild =%d ", pChild));
      for (pChild = pParent->FirstChild(); pChild != 0; pChild = pChild->NextSibling())
      {
         GetTotalNodeInfo(pChild, NodeType); // chagned NodeTypeEnum to NodeType for testing
      }
      ETG_TRACE_ERR(("int XMLDocParser::GetTotalNodeInfo(TiXmlNode* pParent, int NodeEnum) Total totnodecounter %d", totnodecounter));
   }

   return totnodecounter;
// return 1;
}


bool XMLDocParser::FetchConfigurationVersionCisNetworkXml()
{
   bool  configVerCISnetLoadFile = xmlConfigVerCISnetworkDoc.LoadFile();
   _confgVerCisNetworkVecItem.clear();

   if (configVerCISnetLoadFile)
   {
      int retval = GetNecessaryInfo(&xmlConfigVerCISnetworkDoc, false);
      ETG_TRACE_ERR(("bool XMLDocParser::FetchConfigurationVersionCisNetworkXml() configVerCISnetLoadFile- configVerCISnetLoadFile retval==> %d %d", configVerCISnetLoadFile, retval));
   }
   else
   {
      ETG_TRACE_ERR(("bool XMLDocParser::FetchConfigurationVersionCisNetworkXml() configVerCISnetLoadFile- XML load Failed ==>%s", xmlConfigVerCISnetworkDoc.ErrorDesc()));
   }

   return configVerCISnetLoadFile;
}


bool XMLDocParser::FetchConfigurationVersionDoIpcInstallResultXml()
{
   bool configInstResLoadFile = xmlConfigVerDoIpcInstallResultDoc.LoadFile();
   _confgVerdoipcinstallVecItem.clear();
   if (configInstResLoadFile)
   {
      int retval = GetNecessaryInfo(&xmlConfigVerDoIpcInstallResultDoc, false);
      ETG_TRACE_ERR(("XMLDocParser::FetchConfigurationVersionDoIpcInstallResultXml() XML load Successful configInstResLoadFile retval==>%d %d", configInstResLoadFile, retval));
   }
   else
   {
      ETG_TRACE_ERR(("XMLDocParser::FetchConfigurationVersionDoIpcInstallResultXml()  XML load Failed ==>%s ", xmlConfigVerDoIpcInstallResultDoc.ErrorDesc()));
   }
   return configInstResLoadFile;
}


/********************************************

* NAME    : FetchDtcXmlUpdateResults Information From DTCXML
* <!--Task 1012776 and CRQ 952224  -->
****************************************************/
bool XMLDocParser::FetchDtcXmlUpdateResults()
{
   ETG_TRACE_USR4(("bool XMLDocParser::FetchDtcXmlUpdateResults()"));
   _isTesterTagAvailable = false;
   _isNodeTagAvailable = false;
   _nodeTagCounter = 0;
   _gidAtrribCounter = 0;
   _updateResultAtrribCounter = 0;
   _SWVersionAtrribCounter = 0;
   _DeviceNameAtrribCounter = 0;

   ETG_TRACE_USR4(("bool XMLDocParser::FetchDtcXmlUpdateResults()"));

   bool  result = _xmlDtcCodeStatusDoc.LoadFile();
   ETG_TRACE_USR4(("bool XMLDocParser::FetchDtcXmlUpdateResults() =%d", result));

   if (result)
   {
      ETG_TRACE_USR4(("bool XMLDocParser::FetchDtcXmlUpdateResults() XML load Successful"));
      _statusReportVec.clear();
      _CISstatusReportVec.clear();

      vecDTCInformation.clear();

      GetNecessaryInfo(&_xmlDtcCodeStatusDoc, false);

      if ((!_isTesterTagAvailable) && (!_isNodeTagAvailable))
      {
         ETG_TRACE_USR4(("XMLDocParser::FetchDtcXmlUpdateResults XML load Successful. But <tester> Tag is not Found. Estimate update is Missing"));
         result = false;
      }
   }
   else
   {
      ETG_TRACE_ERR(("bool XMLDocParser::FetchDtcXmlUpdateResults() XML load Failed %s", _xmlDtcCodeStatusDoc.ErrorDesc()));

      ETG_TRACE_USR4(("bool XMLDocParser::FetchDtcXmlUpdateResults() result %d", result));
   }
   ETG_TRACE_USR4(("bool XMLDocParser::FetchDtcXmlUpdateResults() - Exit result %d", result));

   return result;
}


bool XMLDocParser::FetchInstallResult()
{
   ETG_TRACE_USR4(("XMLDocParser - FetchInstallResult is called"));
   bool result = _xmlInstallResultDoc.LoadFile();
   if (result)
   {
      ETG_TRACE_USR4(("XMLDocParser - XML load Successful"));
      _CISstatusReportVec.clear();
      GetNecessaryInfo(&_xmlInstallResultDoc, false);
   }
   else
   {
      ETG_TRACE_ERR(("XMLDocParser - XML load Failed %s", _xmlDoc.ErrorDesc()));
   }
   return result;
}


bool XMLDocParser::FetchInstalledDTCResult()
{
   ETG_TRACE_USR4(("bool XMLDocParser::FetchInstalledDTCResult() is called"));
   bool result = _xmlDtcInstallaedCodeStatusDoc.LoadFile();
   if (result)
   {
      ETG_TRACE_USR4(("bool XMLDocParser::FetchInstalledDTCResult()- XML load Successful"));
      _CISstatusReportVec.clear();
      GetNecessaryInfo(&_xmlDtcInstallaedCodeStatusDoc, false);
   }
   else
   {
      ETG_TRACE_ERR(("bool XMLDocParser::FetchInstalledDTCResult() load Failed %s", _xmlDtcInstallaedCodeStatusDoc.ErrorDesc()));
   }
   return result;
}


bool XMLDocParser::FetchSystemInstallResult()
{
   ETG_TRACE_USR4(("XMLDocParser - FetchSystemInstallResult is called"));
   bool result = _xmlSystemInstallResultDoc.LoadFile();
   if (result)
   {
      ETG_TRACE_USR4(("XMLDocParser - XML load Successful"));
      GetNecessaryInfo(&_xmlSystemInstallResultDoc, false);
   }
   else
   {
      ETG_TRACE_ERR(("XMLDocParser - XML load Failed %s", _xmlDoc.ErrorDesc()));
   }
   return result;
}


/**
* GetResultUpdates - Fetches the extracted result information
* @return   - Returns the vector of status report
*/
std::vector<UpdReport> XMLDocParser::GetResultUpdates()
{
   ETG_TRACE_USR4(("XMLDocParser::GetResultUpdates"));
   return _statusReportVec;
}


std::vector<UpdReport> XMLDocParser::GetCISResultUpdates()
{
   ETG_TRACE_USR4(("XMLDocParser::GetCISResultUpdates"));
   return _CISstatusReportVec;
}


std::vector<SWUpdateReport> XMLDocParser::GetSerialandMACAddress()
{
   ETG_TRACE_USR4(("XMLDocParser::GetSerialandMACAddress"));
   return _SerialandMACAddressReportVec;
}


/*******************DTC Const Char* "Code or Status" Data for Devices "Amp - CMP - CMGAVB - CMG1" ************************/


std::vector<DTCInfo> XMLDocParser::GetDTCInfo()
{
   ETG_TRACE_USR4(("XMLDocParser::_DTCCMG1StatusReportVec"));
   return vecDTCInformation;
}


//Task 1012776 and CRQ 952224
/**
* GetStageOneEstimatedTime - provides estimated time info from xml
* @return  --  Returns the estimate time from XML
*/
uint32 XMLDocParser::GetStageOneEstimatedTime()
{
   ETG_TRACE_USR4(("XMLDocParser::GetStageOneEstimatedTime %u", _stageOneEstimatedTime));

   // Clear residual information in XML if any
   _statusReportVec.clear();

   return _stageOneEstimatedTime;
}


std::string XMLDocParser::GetInstallationResultStatus()
{
   ETG_TRACE_USR4(("XMLDocParser::GetInstallationResultStatus %s", _Status_Str.c_str()));
   return _Status_Str;
}


std::vector<ConfigurationVersionItems> XMLDocParser::GetConfigurationVersionCisNetworkData()
{
   return _confgVerCisNetworkVecItem;
}


std::vector<ConfigurationVersionItems> XMLDocParser::GetConfigurationVersionDoIpcInstallData()
{
   return _confgVerdoipcinstallVecItem;
}


}
}
