/**
 * @addtogroup <AppHmi_System>
 * @{
 * @file        NavigationServiceClient.cpp
 * @brief       NavigationServiceClient is to handle ServiceClient impl of Navigation services.
 * @author      Senthilkumar Venugopal - A-IVI
 * @copyright   (C) 2016 Robert Bosch GmbH.
 *              (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *              The reproduction, distribution and utilization of this file as
 *              well as the communication of its contents to others without express
 *              authorization is prohibited. Offenders will be held liable for the
 *              payment of damages. All rights reserved in the event of the grant
 *              of a patent, utility model or design.
 * @}
 */


#include "hall_std_if.h"
#include "NavigationServiceClient.h"
#include "App/Core/AppLogic/SystemSettingListhandler/SystemSettingsListHandler.h"
#include "App/Core/AppLogic/CabinA/CabinASettingListhandler/CabinASettingListhandler.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_SYSTEM_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_SYSTEM
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_SYSTEM_"
#define ETG_I_FILE_PREFIX                 App::Core::NavigationServiceClient::
#include "trcGenProj/Header/NavigationServiceClient.cpp.trc.h"
#endif


using namespace NAVIGATION_INTERFACE;


namespace App {
namespace Core {

/**
 * @Constructor
 */
NavigationServiceClient* NavigationServiceClient::_navigationSrvClient = NULL;
NavigationServiceClient::NavigationServiceClient()
{
   ETG_TRACE_USR4(("IN NavigationServiceClient INIT"));

   _navigationServiceProxy                       = ::NavigationServiceProxy::createProxy("navigationServicePort", *this); //Creating the proxy for navigation interface to access navigation service
   //Register the NavigationServiceClient instance in StartupSync for
   // register/deregister call back whenever gui is started.
   if (_navigationServiceProxy)
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _navigationServiceProxy->getPortName());
   }
}


/**
 * @Destructor
 */
NavigationServiceClient::~NavigationServiceClient()
{
   ETG_TRACE_USR4(("IN NavigationServiceClient DTOR"));
   delete _navigationSrvClient;
   _navigationSrvClient = NULL;
}


/**
 * onAvailable - To Handle Service Availability
 * @param[in]    : proxy
 * @param[in]    : stateChange
 * @param[out]   : None
 * @return       : void
 */
void NavigationServiceClient::onAvailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   StartupSync::getInstance().onAvailable(proxy, stateChange);
}


/**
 * onUnavailable - To Handle Service Unavailability
 * @param[in]    : proxy
 * @param[in]    : stateChange
 * @param[out]   : None
 * @return       : void
 */
void NavigationServiceClient::onUnavailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   StartupSync::getInstance().onUnavailable(proxy, stateChange);
}


/**
 * registerProperties - To Register Service Properties
 * @param[in]    : proxy
 * @param[in]    : stateChange
 * @param[out]   : None
 * @return       : void
 */
void NavigationServiceClient::registerProperties(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& /*stateChange*/)
{
   if (_navigationServiceProxy == proxy)
   {
      //ETG_TRACE_USR4_DCL((TR_CLASS_APPHMI_SYSTEM_NAVIGATION_DEBUG, "NavigationServiceClient:Info:Register properties is called for Navigation"));
      _navigationServiceProxy->sendDatasetVersionRegister(*this);
      //Get the startup data in service because dBus will not send
      //   startup data update during register
      _navigationServiceProxy->sendDatasetVersionGet(*this);
   }
}


/**
 * deregisterProperties - To DeRegister Service Properties
 * @param[in]    : proxy
 * @param[in]    : stateChange
 * @param[out]   : None
 * @return       : void
 */
void NavigationServiceClient::deregisterProperties(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& /*stateChange*/)
{
   if (_navigationServiceProxy == proxy)
   {
      //ETG_TRACE_USR4_DCL((TR_CLASS_APPHMI_SYSTEM_NAVIGATION_DEBUG, "NavigationServiceClient:Info:Deregister properties is called for Navigation"));
      _navigationServiceProxy->sendDatasetVersionDeregisterAll();
   }
}


/**
 * onDatasetVersionError - Error handling for Map data version
 * @param[in]    : proxy
 * @param[in]    : error
 * @param[out]   : None
 * @return       : void
 */
void NavigationServiceClient::onDatasetVersionError(const ::boost::shared_ptr< NavigationServiceProxy >& proxy, const ::boost::shared_ptr< DatasetVersionError >& /* error */)
{
   if ((proxy == _navigationServiceProxy))
   {
      ETG_TRACE_ERR(("NavigationServiceClient:ERROR:onDatasetVersionError is called"));
   }
}


/**
 * onDatasetVersionUpdate - Update handling for Map data version update
 * @param[in]    : proxy
 * @param[in]    : update
 * @param[out]   : None
 * @return       : void
 */
void NavigationServiceClient::onDatasetVersionUpdate(const ::boost::shared_ptr< NavigationServiceProxy >& proxy, const ::boost::shared_ptr< DatasetVersionUpdate >& update)
{
   if ((proxy == _navigationServiceProxy) && update->hasDatasetVersion())
   {
      std::string dataSetVersion = update->getDatasetVersion();
      SystemSettingsListHandler::poGetInstance()->vDataversion(dataSetVersion);
#ifdef VARIANT_S_FTR_ENABLE_CABIN_A_SUPPORT
      if (CabinASettingListhandler::poGetInstance() != NULL)
      {
         CabinASettingListhandler::poGetInstance()->setDatacarrierVersion(dataSetVersion);
      }
#endif
   }
}


} //Core
} //App
