/**
 * @addtogroup <AppHmi_System>
 * @{
 * @file        DiagnosisSrvClient.cpp
 * @brief       DiagnosisSrvClient is to handle ServiceClient impl of Diagnosis Services.
 *              (-> Control1, ...)
 * @author      Sathiya Jayanth - ECG
 * @copyright   (c) 2018-2019 Robert Bosch Car Multimedia GmbH
 *              The reproduction, distribution and utilization of this file as
 *              well as the communication of its contents to others without express
 *              authorization is prohibited. Offenders will be held liable for the
 *              payment of damages. All rights reserved in the event of the grant
 *              of a patent, utility model or design.
 * @}
 */

#include <hmibase/hall_std_if.h>
#include "DiagnosisSrvClient.h"
#include "App/Core/ClientHandler/VehicleSrvClient/VehicleSrvClient.h"

#include "gui_std_if.h"
#include "util/StrUtf8.h"
#include "AppHmi_SystemStateMachine.h"
#include "hmi_trace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_SYSTEM_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_SYSTEM
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_SYSTEM_"
#define ETG_I_FILE_PREFIX                  App::DataModel::DiagnosisSrvClient
#include "trcGenProj/Header/DiagnosisSrvClient.cpp.trc.h"
#endif

using namespace org::bosch::cm::diagnosis::dbus::Diagnosis1::Control1;

namespace App {
namespace Core {

DiagnosisSrvClient* DiagnosisSrvClient::_diaglient = NULL;
/**
 * @Constructor
 */

DiagnosisSrvClient::DiagnosisSrvClient()
{
   ETG_TRACE_USR4(("IN DiagnosisSrvClient CTOR"))
   _controlProxy = Control1Proxy::createProxy("control1Port", *this);
   //Register the DiagnosisSrvClient instance in StartupSync for
   // register/deregister call back whenever gui is started.
   privacyResetRequest = false;
   if (_controlProxy)
   {
      ETG_TRACE_USR4(("IN DiagnosisSrvClient CTOR after creating proxy"))
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _controlProxy->getPortName());
   }
}


/**
 * @Destructor
 */
DiagnosisSrvClient::~DiagnosisSrvClient()
{
   ETG_TRACE_USR4(("IN DiagnosisSrvClient DTOR"));
}


void DiagnosisSrvClient::destroy()
{
   delete _diaglient;
   _diaglient = NULL;
}


/**
 * onAvailable - To Handle Service Availability
 * @param[in]    : proxy
 * @param[in]    : stateChange
 * @param[out]   : None
 * @return       : void
 */
void DiagnosisSrvClient::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
   if (proxy == _controlProxy)
   {
      ETG_TRACE_USR4(("IN DiagnosisSrvClient onAvailable"));
      StartupSync::getInstance().onAvailable(proxy, stateChange);
   }
}


/**
 * onUnavailable - To Handle Service Unavailability
 * @param[in]    : proxy0
 * @param[in]    : stateChange
 * @param[out]   : None
 * @return       : void
 */
void DiagnosisSrvClient::onUnavailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("IN DiagnosisSrvClient onUnavailable"));
   StartupSync::getInstance().onUnavailable(proxy, stateChange);
}


/**
 * registerProperties - To Register Service Properties
 * @param[in]    : proxy
 * @param[in]    : stateChange
 * @param[out]   : None
 * @return       : void
 */
void DiagnosisSrvClient::registerProperties(const ::boost::shared_ptr< ::asf::core::Proxy >& /*proxy*/, const ::asf::core::ServiceStateChange& /*stateChange*/)
{
   //Info: Currently not in use
}


/**
 * deregisterProperties - To DeRegister Service Properties
 * @param[in]    : proxy
 * @param[in]    : stateChange
 * @param[out]   : None
 * @return       : void
 */
void DiagnosisSrvClient::deregisterProperties(const ::boost::shared_ptr< ::asf::core::Proxy >& /*proxy*/, const ::asf::core::ServiceStateChange& /*stateChange*/)
{
   //Info: Currently not in use
}


void DiagnosisSrvClient::FactoryReset()
{
   ETG_TRACE_USR4(("IN DiagnosisSrvClient FactoryReset"));
   std::vector< StartSystemSettingRequestExtendedDataStruct > extendedData;
   uint32 req_cookie = 12; //as of now set a 14..hardcoded
   const ::std::string sysSetType = "DIA_SYSSET_TYPE_HMI"; //for hmi reset
   if (_controlProxy)
   {
      _controlProxy->sendStartSystemSettingRequest(*this, sysSetType, extendedData, req_cookie);
      ETG_TRACE_USR4(("IN DiagnosisSrvClient FactoryReset Sent Request"));
   }
}


void DiagnosisSrvClient::privacyResetValue()
{
   ETG_TRACE_USR4(("IN DiagnosisSrvClient privacyResetValue"));
   privacyResetRequest = true;
}


/**
 * onStartSystemSettingError - Error handling for StartSystemSetting.
 * @param[in]    : proxy
 * @param[in]    : error
 * @param[out]   : None
 * @return       : void
 */
void DiagnosisSrvClient::onStartSystemSettingError(const ::boost::shared_ptr< Control1Proxy >& proxy, const ::boost::shared_ptr< StartSystemSettingError >& /* error */)
{
   ETG_TRACE_USR4(("IN DiagnosisSrvClient onStartSystemSettingError 1"));
   if (proxy == _controlProxy) //for safety
   {
      ETG_TRACE_USR4(("IN DiagnosisSrvClient onStartSystemSettingError"));
      POST_MSG((COURIER_MESSAGE_NEW(Popup_Reset_System)()));
   }
}


/**
 * onStartSystemSettingResponse - Respose handling for StartSystemSetting.
 * @param[in]    : proxy
 * @param[in]    : response
 * @param[out]   : None
 * @return       : void
 */
void DiagnosisSrvClient::onStartSystemSettingResponse(const ::boost::shared_ptr< Control1Proxy >& proxy, const ::boost::shared_ptr< StartSystemSettingResponse >& response)
{
   ETG_TRACE_USR4(("IN DiagnosisSrvClient onStartSystemSettingResponse %d", response->getResultCode()));
   if (proxy == _controlProxy) //for safety
   {
      ETG_TRACE_USR4(("IN DiagnosisSrvClient onStartSystemSettingResponse"));
      POST_MSG((COURIER_MESSAGE_NEW(Popup_Reset_System)()));
      if (privacyResetRequest == true)
      {
         VehicleSrvClient::poGetInstance()->sendPrivacyResetAck();
         privacyResetRequest = false;
      }
   }
}


bool DiagnosisSrvClient::onCourierMessage(const	 FactoryResetSM& /* oMsg */)
{
   ETG_TRACE_USR4(("DiagnosisSrvClient->onCourierMessage:ResetSystemSettings()"));

   std::vector< StartSystemSettingRequestExtendedDataStruct > extendedData;
   uint32 req_cookie = 12;     //as of now set a 14..hardcoded
   const ::std::string sysSetType = "DIA_SYSSET_TYPE_HMI"; //for hmi reset

   if (_controlProxy)
   {
      _controlProxy->sendStartSystemSettingRequest(*this, sysSetType, extendedData, req_cookie);
      ETG_TRACE_USR4(("IN DiagnosisSrvClient FactoryReset Sent Request"));
   }

   else
   {
      ETG_TRACE_USR4(("IN DiagnosisSrvClient FactoryReset No Request Sent"));
   }

   return true;
}


} // namespace Core
} // namespace App
