/**
 * @addtogroup <AppHmi_System>
 * @{
 * @file        AudioServiceClient.cpp
 * @brief       AudioServiceClient is to handle ServiceClient impl of Audio Services.
 *              (-> Genivi audio manager, Mascf_main_fi and Master audio)
 * @author       Sathiya Jayanth- ECG
 * @copyright   (c) 2018-2019 Robert Bosch Car Multimedia GmbH
 *              The reproduction, distribution and utilization of this file as
 *              well as the communication of its contents to others without express
 *              authorization is prohibited. Offenders will be held liable for the
 *              payment of damages. All rights reserved in the event of the grant
 *              of a patent, utility model or design.
 * @}
 */

#include "hall_std_if.h"
#include "AudioServiceClient.h"
#include "AudioEqlPrstRBtn.h"
#include "AudioSpdDepdentVolCntrlRBtn.h"
//#include "CabinAAudioSettingListHandler.h"
#include "AppHmi_MasterBase/AudioInterface/AudioDefines.h"
#include "bosch/cm/ai/hmi/masteraudioservice/AudioSourceChange.h"
#include "App/Core/AppLogic/CabinA/CabinASettingListhandler/CabinAAudioSettingListHandler/CabinAAudioSettingListHandler.h"
#include "CabinAGlassSettingListhandler.h"
#include "CabinASettingListhandler.h"
#include "ProjectBaseTypes.h"
#include "App/Core/AppLogic/CabinA/CabinASettingListhandler/CabinAGlassSettingListhandler/CabinAGlassSettingListhandler.h"
#include "App/Core/AppLogic/SystemSettingListhandler/SystemSettingsListHandler.h"
#include "App/Core/AppLogic/MicSettings/MicSetting.h"

#include "AudioPrivateModeRBtn.h"
#include "AppHmi_MasterBase/AudioInterface/AudioDefines.h"
#include "Common/ApplicationSwitchClientComponent/ApplicationSwitchClientComponent.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_SYSTEM_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_SYSTEM
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_SYSTEM_"
#define ETG_I_FILE_PREFIX                 App::Core::AudioServiceClient::
#include "trcGenProj/Header/AudioServiceClient.cpp.trc.h"
#endif

using namespace ::org::genivi::audiomanager::CommandInterface;
using namespace ::bosch::cm::ai::hmi::masteraudioservice::SoundProperties;
using namespace ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange;
using namespace ::bosch::cm::ai::hmi::hmiinfoservice::HmiInfoService;		//	 HMIINFOSERVICE Mapout Service-26.05.2020
using namespace ::asf::core;

namespace App {
namespace Core {

AudioServiceClient* AudioServiceClient::_AudioSrvClient = NULL;

AudioServiceClient::AudioServiceClient()
{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR4(("AudioServiceClient Constructor "));
   init();
   m_SpeedDepVolCntrlval = 0;
   m_Equaliserval = 0;
   SinkID_cc = COCKPITSINKID;

   activesourceId = 0;
   mediasinkidstatus = 0;
}


AudioServiceClient::~AudioServiceClient()
{
   clear();
   destroy();

   ETG_TRACE_USR4(("AudioServiceClient Distructor "));
}


void AudioServiceClient::destroy()
{
   delete _AudioSrvClient;
   _AudioSrvClient = NULL;
}


void AudioServiceClient::init()
{
   _mainSinkSoundChanged_RegId       			= 0;
   _requestMainSinkSoundPropertyList_SPReg_Id 	= 0;
   _soundPropertyList_SPRegId					= 0;
   _requestSystemPropertyList_SPReg_Id			= 0;
   _mainSinkSoundSet_RegId          			= 0;
   _setSystemProperty_SPRegId					= 0;
   _setMainSinkSoundProperty_SPRegId			= 0;
   _systemPropertyListUpdate_SPRegId			= 0;
   //_systemPropertyChanged_RegId     			= 0;
   _sendsystempropertylistget_SPReg_Id			= 0;
   _soundPropertieslistget_SpReg_ID			= 0;
   _getListMainSinkSound_RegId       			= 0;
   _getListSystemProperty_RegId      			= 0;
   _muteStateSet_RegId               			= 0;
   _muteStateGet_RegId               			= 0;
   _muteStateChanged_RegId           			= 0;
   _systemPropertyChangedSP_RegId	 			= 0;
   _getListSystemPropertySP_RegId	 			= 0;
   _ampAvailFunctionStatus_RegId     			= 0;
   _activeSourceListUpdate_RegId     			= 0;
   _regionsMapOutStatusInfo_RegId    			= 0;		//	Creating the Proxy for audio source change to HMIINFOSERVICE Mapout Service Properties-26.05.2020
   _cabinsSystemsInfo_RegId						= 0;
   _sendsystempropertylistregister			= 0;
   m_typeproperty       						= 9;
   cockpit_propertyupdate                       = FALSE;
   cabinA_propertyupdate                        = FALSE;

   _commandIFProxy                   = CommandInterfaceProxy::createProxy("commandInterfacePort", *this); //Creating the proxy for command interface to access genivi audio manager service

   _audioSoundPropertiesProxy        = SoundPropertiesProxy::createProxy("soundPropertiesPort", *this); //Creating the proxy for sound properties interface to access master audio sound properties service

   _audioSourceChangeProxy           = AudioSourceChangeProxy::createProxy("audioSourceChangePort", *this); //Creating the proxy for audio source change to access master audio source change service
   ETG_TRACE_USR4(("AudioServiceClient INIT "));
   _hmiInfoServiceProxy			 = HmiInfoServiceProxy::createProxy("hmiinfoservicePort", *this);//	Creating the Proxy for HMIINFOSERVICE Mapout Service Properties-26.05.2020

   registerPropertyRegistrationIF();
}


//sendRequestSystemPropertyListRequest
void AudioServiceClient::clear()
{
   _mainSinkSoundChanged_RegId       			= 0;
   _requestMainSinkSoundPropertyList_SPReg_Id 	= 0;
   _soundPropertyList_SPRegId					= 0;
   _requestSystemPropertyList_SPReg_Id			= 0;
   _mainSinkSoundSet_RegId           			= 0;
   _setSystemProperty_SPRegId					= 0;
   _setMainSinkSoundProperty_SPRegId			= 0;
   _systemPropertyListUpdate_SPRegId			= 0;
   _soundPropertieslistget_SpReg_ID			= 0;
   _getListSystemProperty_RegId      			= 0;
   _getListMainSinkSound_RegId       			= 0;
   _muteStateSet_RegId               			= 0;
   _muteStateGet_RegId              			= 0;
   _muteStateChanged_RegId           			= 0;
   _ampAvailFunctionStatus_RegId     			= 0;
   _activeSourceListUpdate_RegId     			= 0;
   _systemPropertyChangedSP_RegId	 			= 0;
   _getListSystemPropertySP_RegId	 			= 0;
   _regionsMapOutStatusInfo_RegId    			= 0;
   _cabinsSystemsInfo_RegId						= 0;
   _sendsystempropertylistget_SPReg_Id			= 0;
   _sendsystempropertylistregister			= 0;
}


void AudioServiceClient::registerPropertyRegistrationIF()
{
   ETG_TRACE_USR4(("AudioServiceClient registerPropertyRegistrationIF "));
   if (_commandIFProxy)
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _commandIFProxy->getPortName());
      ETG_TRACE_USR4(("AudioServiceClient registerPropertyRegistrationIF 12 "));
   }
   ETG_TRACE_USR4(("AudioServiceClient registerPropertyRegistrationIF "));
   if (_audioSoundPropertiesProxy)
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _audioSoundPropertiesProxy->getPortName());
      ETG_TRACE_USR4(("AudioServiceClient registerPropertyRegistrationIF 22"));
   }
   if (_audioSourceChangeProxy)
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _audioSourceChangeProxy->getPortName());
      ETG_TRACE_USR4(("AudioServiceClient registerPropertyRegistrationIF 23"));
   }
   if (_hmiInfoServiceProxy)
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _hmiInfoServiceProxy->getPortName());
      ETG_TRACE_USR4(("AudioServiceClient registerPropertyRegistrationIF 24"));
   }
}


void AudioServiceClient::registerProperties(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("AudioServiceClient registerProperties"));

   if (_commandIFProxy == proxy)
   {
      ETG_TRACE_USR4(("AudioServiceClient registerProperties _commandIFProxy "));
   }
   if (_audioSoundPropertiesProxy == proxy)
   {
      _muteStateChanged_RegId = _audioSoundPropertiesProxy->sendMuteStateRegister(*this);
      _mainSinkSoundChanged_RegId = _audioSoundPropertiesProxy->sendMainSinkSoundPropertyChangedRegister(*this);

      SinkID_cc = COCKPITSINKID;
      _requestMainSinkSoundPropertyList_SPReg_Id = _audioSoundPropertiesProxy->sendRequestMainSinkSoundPropertyListRequest(COCKPITSINKID);
      _soundPropertieslistget_SpReg_ID = _audioSoundPropertiesProxy->sendSoundPropertyListGet(*this);
      _soundPropertyList_SPRegId = _audioSoundPropertiesProxy-> sendSoundPropertyListRegister(*this);

      //onSoundPropertyListUpdate
      _systemPropertyChangedSP_RegId = _audioSoundPropertiesProxy->sendSystemPropertyChangedRegister(*this);
      _sendsystempropertylistregister = _audioSoundPropertiesProxy-> sendSystemPropertyListRegister(*this);
      _sendsystempropertylistget_SPReg_Id = _audioSoundPropertiesProxy->sendSystemPropertyListGet(*this);

      /*  SinkID_cc = CABINSASINKID;
        _requestMainSinkSoundPropertyList_SPReg_Id = _audioSoundPropertiesProxy->sendRequestMainSinkSoundPropertyListRequest(CABINSASINKID); */
      //  _requestSystemPropertyList_SPReg_Id = _audioSoundPropertiesProxy->sendRequestSystemPropertyListRequest(*this);

      //	SinkID_cc = COCKPITSINKID;
      //	_getListMainSinkSound_RegId=_audioSoundPropertiesProxy->sendGetListMainSinkSoundPropertiesRequest(*this, COCKPITSINKID);
      //	SinkID_cc = CABINSASINKID;
      //_getListMainSinkSound_RegId=_audioSoundPropertiesProxy->sendGetListMainSinkSoundPropertiesRequest(*this, CABINSASINKID);
      //	_getListSystemPropertySP_RegId=_audioSoundPropertiesProxy->sendGetListSystemPropertiesRequest(*this);
      //new sound property from hmi base---20.05.2020

      ETG_TRACE_USR4(("AudioServiceClient registerProperties _audioSoundPropertiesProxy "));
   }
   if (_audioSourceChangeProxy == proxy)
   {
      _activeSourceListUpdate_RegId   = _audioSourceChangeProxy->sendActiveSourceListRegister(*this);// source list instance will added to the hmi audio base client list
      _audioSourceChangeProxy->sendActiveSourceListGet(*this);// trigger the call back
      ETG_TRACE_USR4(("AudioServiceClient registerProperties _audioSourceChangeProxy "));
   }
   if (_hmiInfoServiceProxy == proxy)
   {
      _regionsMapOutStatusInfo_RegId = _hmiInfoServiceProxy->sendRegionsMapOutStatusInfoRegister(*this);//Registering RegionMapout ---26.05.2020
      _cabinsSystemsInfo_RegId = _hmiInfoServiceProxy->sendCabinsSystemsInfoRegister(*this);
      ETG_TRACE_USR4(("AudioServiceClient registerProperties _hmiInfoServiceProxy "));
   }
}


void AudioServiceClient::deregisterProperties(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("AudioServiceClient DeregisterProperties"));
   if (_commandIFProxy == proxy)
   {
      _commandIFProxy->sendMainSinkSoundPropertyChangedDeregisterAll();
   }
   if (_audioSoundPropertiesProxy == proxy)
   {
      _audioSoundPropertiesProxy->sendMuteStateDeregisterAll();
      _audioSoundPropertiesProxy->sendMainSinkSoundPropertyChangedDeregisterAll();//new sound property from hmi base---20.05.2020
      ETG_TRACE_USR4(("AudioServiceClient DeregisterProperties _audioSoundPropertiesProxy "));
   }
   if (_audioSourceChangeProxy == proxy)
   {
      _audioSourceChangeProxy->sendActiveSourceListDeregisterAll();
      ETG_TRACE_USR4(("AudioServiceClient deregisterProperties _audioSourceChangeProxy "));
   }
   if (_hmiInfoServiceProxy == proxy)
   {
      _hmiInfoServiceProxy->sendRegionsMapOutStatusInfoDeregisterAll();		//Deregistering RegionMapout ---26.05.2020
      ETG_TRACE_USR4(("AudioServiceClient deregisterProperties _hmiInfoServiceProxy "));
   }
}


/*New Sound Properties update from HMI Base audio---20.05.2020*/
/* void AudioServiceClient::onGetListSystemPropertiesError(const ::boost::shared_ptr< AUDIO_SOUND_PROPERTIES::SoundPropertiesProxy >& proxy, const ::boost::shared_ptr<  AUDIO_SOUND_PROPERTIES::GetListSystemPropertiesError >& error)
{
		ETG_TRACE_USR4(("AudioServiceClient::onGetListSystemPropertiesError Error Message....."));
		if ((proxy == _audioSoundPropertiesProxy) && (error->getAct() == _getListSystemPropertySP_RegId)) //for safety
		{
		  ETG_TRACE_USR4(("AudioServiceClient::onGetListSystemPropertiesError Error Message in CondCheck."));
		}
}


void AudioServiceClient::onGetListSystemPropertiesResponse(const ::boost::shared_ptr< AUDIO_SOUND_PROPERTIES::SoundPropertiesProxy >& proxy, const ::boost::shared_ptr<  AUDIO_SOUND_PROPERTIES::GetListSystemPropertiesResponse >& response)
{
	ETG_TRACE_USR4(("AudioServiceClient:onGetListSystemPropertiesResponse is called"));
   std::vector< SystemProperty > m_hmiaudiosystemproperty;
   m_hmiaudiosystemproperty = response->getSystemProperties();

   if ((proxy == _audioSoundPropertiesProxy) && (response->getAct() == _getListSystemPropertySP_RegId)) //for safety
   {
      ETG_TRACE_USR4(("AudioServiceClient:onGetListSystemPropertiesResponse is IFF LOOP"));
	  for (std::vector< SystemProperty >::iterator iter = m_hmiaudiosystemproperty.begin() ; iter != m_hmiaudiosystemproperty.end(); iter++)
      {
         uint16 soundproperty_ckpit_pvtmod = static_cast<uint16>(iter->getPropertyType());
         uint16 value_ckpit_pvtmod         = static_cast<uint16>(iter->getValue());
         ETG_TRACE_USR4(("AudioServiceClient::onGetListMainSinkSoundPropertiesResponse::value_ckpit_pvtmod, SinkID_cc, soundproperty_ckpit_pvtmod %d %d %d", value_ckpit_pvtmod, SinkID_cc, soundproperty_ckpit_pvtmod));

         if (soundproperty_ckpit_pvtmod == 13)
         {
            ETG_TRACE_USR4(("AudioServiceClient:onGetListSystemPropertiesResponse IF soundproperty_ckpit_pvtmod is %d", soundproperty_ckpit_pvtmod));
            ETG_TRACE_USR4(("AudioServiceClient:onGetListSystemPropertiesResponse IF value_ckpit_pvtmod is %d", value_ckpit_pvtmod));
            AudioPrivateModeRBtn ::poGetInstance()->UpdatePrivateModeCntrlStatus(value_ckpit_pvtmod);
         }//end of if
      }//end of for
   }//end of if
}//end of property
 */


/* void AudioServiceClient::onGetListMainSinkSoundPropertiesError(const ::boost::shared_ptr< AUDIO_SOUND_PROPERTIES:: SoundPropertiesProxy >& proxy, const ::boost::shared_ptr<AUDIO_SOUND_PROPERTIES::GetListMainSinkSoundPropertiesError >& error)
{
		 ETG_TRACE_USR4(("AudioServiceClient::onGetListMainSinkSoundPropertiesError Error Message....."));
		if ((proxy == _audioSoundPropertiesProxy) && (error->getAct() == _getListMainSinkSound_RegId)) //for safety
		{
		   ETG_TRACE_USR4(("AudioServiceClient::onGetListMainSinkSoundPropertiesError Error Message in CondCheck."));
		}
}


void AudioServiceClient::onGetListMainSinkSoundPropertiesResponse(const ::boost::shared_ptr< AUDIO_SOUND_PROPERTIES::SoundPropertiesProxy >& proxy, const ::boost::shared_ptr<AUDIO_SOUND_PROPERTIES::GetListMainSinkSoundPropertiesResponse >& response)
{
   ::std::vector<SoundProperty>m_audiosoundproperty;
   m_audiosoundproperty = response->getSoundProperties();

   ETG_TRACE_USR4(("AudioServiceClient:onGetListMainSinkSoundPropertiesResponse is called"));

		  if ((proxy == _audioSoundPropertiesProxy) && (response->getAct() == _getListMainSinkSound_RegId)) //for safety
		  {
			  ETG_TRACE_USR4(("AudioServiceClient:onGetListMainSinkSoundPropertiesResponse is called IFF LOOP"));
			     ETG_TRACE_USR4(("AudioServiceClient:onGetListMainSinkSoundPropertiesResponse Entered in IFF LOOP"));
				for (std::vector< SoundProperty >::iterator iter = m_audiosoundproperty.begin() ; iter != m_audiosoundproperty.end(); iter++)
				{
					uint16 soundproperty =static_cast<uint16>(iter->getPropertyType());
					int value= static_cast<int>(iter->getValue());

					ETG_TRACE_USR4(("AudioServiceClient::onGetListMainSinkSoundPropertiesResponse::value, SinkID_cc, soundproperty %d %d %d", value, SinkID_cc, soundproperty));

						if (SinkID_cc == COCKPITSINKID)
						{
								if (soundproperty == 14)
								{
								ETG_TRACE_USR4(("AudioServiceClient::onMainSinkSoundPropertyChangedSignal soundproperty == 14::%d", value));
								AudioEquilizerPresetRBtn ::poGetInstance()->UpdateEqlPrstTglStatus(value);
								}
								else if (soundproperty == 15)
								{
								ETG_TRACE_USR4(("AudioServiceClient::onMainSinkSoundPropertyChangedSignal soundproperty == 15::%d", value));
								AudioSpdDepdentVolCntrlRBtn ::poGetInstance()->UpdateSpdDepVolCntrlTglStatus(value);
								}
								else
								{
								ETG_TRACE_USR4(("AudioServiceClient::onMainSinkSoundPropertyChangedSignal::AudioServiceSoundPropertyCheckOnPowerUpdate :UpdatePrivateModeCntrlStatus=%d", value));
								AudioSettingListHandler::poGetInstance()->vPerformMainSinkSoundPropertyBassUpdate(COCKPITSINKID, soundproperty, value);
								}//end of if-else for soundproperty Condition
	 					}//end of SINKID Cockpit If condition Check
						else if (SinkID_cc == CABINSASINKID)
						{
								ETG_TRACE_USR4(("AudioServiceClient::onGetListMainSinkSoundPropertiesResponse::value, SinkID_cc, soundproperty %d %d %d", value, SinkID_cc, soundproperty));
								#ifndef VARIANT_S_FTR_ENABLE_AUX_SUPPORT
								CabinAAudioSettingListHandler::poGetInstance()->vPerformCabinASinkSoundPropertyBassUpdate(CABINSASINKID, soundproperty, value);
								#endif
						}
						else
						{
							ETG_TRACE_USR4(("AudioServiceClient::onGetListMainSinkSoundPropertiesResponse::Nothing just else %d %d %d", value, SinkID_cc, soundproperty));
						}
				}//end of for Loop Condition
			}//end of proxy if condition check
} */
/************************************************************************
*FUNCTION: 		 AudioServiceClient - onMainSinkSoundPropertyChangedError
*DESCRIPTION: 	 Handle the Error Message in Sound Properties Update
*PARAMETER:
*RETURNVALUE: 	 None
*
*HISTORY:

*20.05.2020 Rev. 1.0 Prashant BG

*1.0 Initial Revision.
************************************************************************/

void AudioServiceClient::onMainSinkSoundPropertyChangedError(const ::boost::shared_ptr<AUDIO_SOUND_PROPERTIES::SoundPropertiesProxy >& proxy, const ::
      boost::shared_ptr<AUDIO_SOUND_PROPERTIES::MainSinkSoundPropertyChangedError >& error)
{
   ETG_TRACE_USR4(("AudioServiceClient::onMainSinkSoundPropertyChangedError Error Message....."));

   if ((proxy == _audioSoundPropertiesProxy) && (error->getAct() == _mainSinkSoundChanged_RegId)) //for safety
   {
      ETG_TRACE_USR4(("AudioServiceClient::onMainSinkSoundPropertyChangedError Error Message in CondCheck....."));
   }
}


/************************************************************************
*FUNCTION: 		 AudioServiceClient - onMainSinkSoundPropertyChangedSignal
*DESCRIPTION: 	 Handle the Update change in Sound Properties
*PARAMETER:
*RETURNVALUE: 	 None
*
*HISTORY:

*20.05.2020 Rev. 1.0 Prashant BG

*1.0 Initial Revision.
************************************************************************/
void AudioServiceClient::onMainSinkSoundPropertyChangedSignal(const ::boost::shared_ptr<AUDIO_SOUND_PROPERTIES::SoundPropertiesProxy >& proxy, const ::boost::shared_ptr<AUDIO_SOUND_PROPERTIES::MainSinkSoundPropertyChangedSignal >& changedSignal)
{
   ETG_TRACE_USR4(("AudioServiceClient:onMainSinkSoundPropertyChangedSignal is called"));
   if ((proxy == _audioSoundPropertiesProxy) && (changedSignal->getAct() == _mainSinkSoundChanged_RegId)) //for safety
   {
      uint16 value			 = static_cast<uint16>(changedSignal->getValue());
      uint8 sinkId		 = static_cast<uint8>(changedSignal->getSinkId());
      uint16 soundproperty = static_cast<uint16>(changedSignal->getPropertyType());
      m_typeproperty       = soundproperty ;
      ETG_TRACE_USR4(("AudioServiceClient::onMainSinkSoundPropertyChangedSignal::value, sinkId, soundproperty %d %d %d", value, sinkId, soundproperty));

      if (sinkId == COCKPITSINKID)
      {
         ETG_TRACE_USR4(("AudioServiceClient::onMainSinkSoundPropertyChangedSignal COCKPITSINKID sinkId,soundproperty,value =%d %d %d", sinkId, soundproperty, value));
         if (soundproperty == 14)
         {
            ETG_TRACE_USR4(("AudioServiceClient::onMainSinkSoundPropertyChangedSignal sinkId=1 soundproperty == 14::%d", value));
            AudioEquilizerPresetRBtn ::poGetInstance()->UpdateEqlPrstTglStatus(value);
         }
         else if (soundproperty == 15)
         {
            ETG_TRACE_USR4(("AudioServiceClient::onMainSinkSoundPropertyChangedSignal sinkId=1 soundproperty == 15::%d", value));
            AudioSpdDepdentVolCntrlRBtn ::poGetInstance()->UpdateSpdDepVolCntrlTglStatus(value);
         }
         else
         {
            ETG_TRACE_USR4(("AudioServiceClient::onMainSinkSoundPropertyChangedSignal sinkId=1::AudioServiceSoundPropertyCheckOnValUpdate :UpdatePrivateModeCntrlStatus=%d", value));

            AudioSettingListHandler::poGetInstance()->vPerformMainSinkSoundPropertyBassUpdate(sinkId, soundproperty, value);
         }
      }
      else if (sinkId == CABINSASINKID)
      {
         ETG_TRACE_USR4(("AudioServiceClient::onMainSinkSoundPropertyChangedSignal CABINSASINKID sinkId,soundproperty,value =%d %d %d", sinkId, soundproperty, value));
#ifndef VARIANT_S_FTR_ENABLE_AUX_SUPPORT
         CabinAAudioSettingListHandler::poGetInstance()->vPerformCabinASinkSoundPropertyBassUpdate(sinkId, soundproperty, value);
         MicSetting::poGetInstance()->vPerformMainSinkSoundPropertyBassUpdate(sinkId, soundproperty, value);
#endif
      }
      else
      {
         ETG_TRACE_USR4(("AudioServiceClient::onMainSinkSoundPropertyChangedSignal ELSE sinkId,soundproperty,value =%d %d %d", sinkId, soundproperty, value));
      }
   }
}


void AudioServiceClient::onSystemPropertyChangedError(const ::boost::shared_ptr<AUDIO_SOUND_PROPERTIES::SoundPropertiesProxy >& proxy, const ::boost::shared_ptr<AUDIO_SOUND_PROPERTIES::SystemPropertyChangedError >& error)
{
   ETG_TRACE_USR4(("AudioServiceClient::onSystemPropertyChangedError Error Message....."));
   if ((proxy == _audioSoundPropertiesProxy) && (error->getAct() == _systemPropertyChangedSP_RegId)) //for safety
   {
      ETG_TRACE_USR4(("AudioServiceClient::onSystemPropertyChangedError Error Message in CondCheck."));
   }
}


void AudioServiceClient::onSystemPropertyChangedSignal(const ::boost::shared_ptr<AUDIO_SOUND_PROPERTIES::SoundPropertiesProxy >& proxy, const ::boost::shared_ptr< AUDIO_SOUND_PROPERTIES::SystemPropertyChangedSignal >& signal)
{
   ETG_TRACE_USR4(("AudioServiceClient:onSystemPropertyChangedSignal is called"));
   if ((proxy == _audioSoundPropertiesProxy))
   {
      ETG_TRACE_USR4(("AudioServiceClient:onSystemPropertyChangedSignal in IFF Loop"));
      uint16 soundproperty_ckpit_pvtmod = static_cast<uint16>(signal->getPropertyType());
      uint16 value_ckpit_pvtmod		 =  static_cast<uint16>(signal->getValue());

      /*   m_hmiaudiosystemproperty = response->getSystemProperties();
      uint16 soundproperty_ckpit_pvtmod = static_cast<uint16>(iter->getPropertyType());
         uint16 value_ckpit_pvtmod         = static_cast<uint16>(iter->getValue());
      */
      //  ETG_TRACE_USR4(("AudioServiceClient::onMainSinkSoundPropertyChangedSignal::AudioPrivateModeRBtn :UpdatePrivateModeCntrlStatus=%d", value));
      if (soundproperty_ckpit_pvtmod == 13)
      {
         ETG_TRACE_USR4(("AudioServiceClient:onSystemPropertyChangedSignal IF soundproperty_ckpit_pvtmod is %d", soundproperty_ckpit_pvtmod));
         ETG_TRACE_USR4(("AudioServiceClient:onSystemPropertyChangedSignal IF value is %d", value_ckpit_pvtmod));
         AudioPrivateModeRBtn ::poGetInstance()->UpdatePrivateModeCntrlStatus(value_ckpit_pvtmod);
      }
      else
      {
         ETG_TRACE_USR4(("AudioServiceClient:onSystemPropertyChangedSignal ELSE soundproperty_ckpit_pvtmod is %d", soundproperty_ckpit_pvtmod));
         ETG_TRACE_USR4(("AudioServiceClient:onSystemPropertyChangedSignal ELSE value_ckpit_pvtmod is %d", value_ckpit_pvtmod));
      }
   }
}


/*New Sound Properties update from HMI Base audio---20.05.2020*/

/************added for cabin audio ************/
void AudioServiceClient::onMainSourceSoundPropertyChangedError(const ::boost::shared_ptr< CommandInterfaceProxy >& proxy, const ::boost::shared_ptr< MainSourceSoundPropertyChangedError >& error)
{
   ETG_TRACE_USR4(("AudioServiceClient::onMainSourceSoundPropertyChangedError Error Message....."));

   if ((proxy == _commandIFProxy))  //for safety
   {
      ETG_TRACE_USR4(("AudioServiceClient::onMainSourceSoundPropertyChangedError Error Message in CondCheck....."));
   }
}


void AudioServiceClient::onMainSourceSoundPropertyChangedSignal(const ::boost::shared_ptr< CommandInterfaceProxy >& proxy, const ::boost::shared_ptr< MainSourceSoundPropertyChangedSignal >& signal)
{
   ETG_TRACE_USR4(("AudioServiceClient::onMainSourceSoundPropertyChangedError Signal Message....."));

   if ((proxy == _commandIFProxy))  //for safety
   {
      ETG_TRACE_USR4(("AudioServiceClient::onMainSourceSoundPropertyChangedError Signal Message in CondCheck....."));
   }
}


void AudioServiceClient::onSetMuteStateError(const ::boost::shared_ptr< SoundPropertiesProxy >& proxy, const ::boost::shared_ptr< SetMuteStateError >& error)
{
   //if ((proxy == _audioSoundPropertiesProxy))
   {
   }
}


void AudioServiceClient::onSetMuteStateResponse(const ::boost::shared_ptr< SoundPropertiesProxy >& proxy, const ::boost::shared_ptr< SetMuteStateResponse >& response)
{
   //if ((proxy == _audioSoundPropertiesProxy))
   {
   }
}


void AudioServiceClient::onMuteStateError(const ::boost::shared_ptr< SoundPropertiesProxy >& proxy, const ::boost::shared_ptr< MuteStateError >& error)
{
   //if ((proxy == _audioSoundPropertiesProxy))
   {
   }
}


void AudioServiceClient::onMuteStateUpdate(const ::boost::shared_ptr< SoundPropertiesProxy >& proxy, const ::boost::shared_ptr< MuteStateUpdate >& update)
{
   //TODO: Discuss with goc1kor regarding sink id in this status update -sve2cob
   //if ((proxy == _audioSoundPropertiesProxy))
   {
   }
}


void AudioServiceClient::onAvailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   StartupSync::getInstance().onAvailable(proxy, stateChange);
   ETG_TRACE_USR4(("AudioServiceClient onAvailable "));
}


void AudioServiceClient::onUnavailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   StartupSync::getInstance().onUnavailable(proxy, stateChange);
   ETG_TRACE_USR4(("AudioServiceClient onUnavailable  "));
}


/************************************************************************
*FUNCTION: 		 AudioServiceClient - performMainSinkSoundPropertyUserChange
*DESCRIPTION: 	 Handle to perform the MainSinkSoundProperty user change and set the new value to Audio Service.
*PARAMETER:		 MainSinkSoundUserChangeInfo& obj
*RETURNVALUE: 	 None
*
*HISTORY:

*24.4.2016  Rev. 0.1 Sheena Mathew

*0.1 Initial Revision.
************************************************************************/
void AudioServiceClient::performMainSinkSoundPropertyUserChange(const uint8& msinkid, const uint16& mtypeproperty, const uint32& mvalue)
{
   ETG_TRACE_USR4(("AudioServiceClient:performMainSinkSoundPropertyUserChange is called =%d %d %d ", msinkid, mtypeproperty, mvalue));
   //SetMainSinkSoundPropertyRequest obj_SetMainSinkSoundPropertyRequest=SetMainSinkSoundPropertyRequest(m_typeproperty, mvalue, msinkid);
   uint16 v_soundproperty = mtypeproperty;
   uint8 v_sinkId = msinkid;
   uint32 v_soundvalue = mvalue;

   _audioSoundPropertiesProxy->sendSetMainSinkSoundPropertyRequest(*this, v_soundproperty, v_soundvalue, v_sinkId);
   ETG_TRACE_USR4(("AudioServiceClient:performMainSinkSoundPropertyUserChange is called assigned =%d %d %d ", v_sinkId, v_soundproperty, v_soundvalue));
}


/************added for cabin audio ************/
void AudioServiceClient::OnStartupRequestSinkId4CAaudio()
{
   ETG_TRACE_USR4(("AudioServiceClient:performCabinASinkSoundPropertyUserChange is called"));
}


/************************************************************************
*FUNCTION: 		 AudioServiceClient - performSystemPropertyChange
*DESCRIPTION: 	 Handle to perform the SystemProperty user change and set the new value to Audio Service.
*PARAMETER:		 MainSinkSoundUserChangeInfo& obj
*RETURNVALUE: 	 None
*
*HISTORY:

*13.01.2019  Rev. 0.1 Sathiya

*0.1 Initial Revision.
************************************************************************/
void AudioServiceClient::performSystemPropertyChange(const uint16& mtypeproperty, const uint32& mvalue)
{
   //ETG_TRACE_USR4(("AudioServiceClient:performMainSinkSoundPropertyUserChange is called"));

   m_typeproperty       = mtypeproperty ;
   ETG_TRACE_USR4(("AudioServiceClient::SetSystemPropertyRequestPropertyStruct:mtypeproperty:%d::mvalue:%d", mtypeproperty, mvalue));

   //SetMainSinkSoundPropertyRequest obj_SetMainSinkSoundPropertyRequest=SetMainSinkSoundPropertyRequest(m_typeproperty, mvalue, msinkid);
   SetSystemPropertyRequestPropertyStruct obj = SetSystemPropertyRequestPropertyStruct(mtypeproperty, mvalue);
//ETG_TRACE_USR4(("AudioServiceClient::SetSystemPropertyRequestPropertyStruct:msink:%d::mvalue:%d", msink, obj));
   //_commandIFProxy->sendSetSystemPropertyRequest(*this, obj);  //Set the the new value to service
//sendSystemPropertyChangedSignal
   //sendSetSystemPropertyRequest
   _audioSoundPropertiesProxy->sendSetSystemPropertyRequest(*this, m_typeproperty, mvalue) ;
}


void AudioServiceClient::onActiveSourceListUpdate(const ::boost::shared_ptr< AUDIO_SOURCE_CHANGE::AudioSourceChangeProxy >& proxy,
      const ::boost::shared_ptr< AUDIO_SOURCE_CHANGE::ActiveSourceListUpdate >& signal)
{
   ETG_TRACE_USR4(("AudioServiceClient:: onActiveSourceListUpdate() \n"));

   ::std::vector< sourceData>SourceList = signal->getActiveSourceList();
   ::std::vector<sourceData>::const_iterator iter = SourceList.begin(); // stack top is list bottom
   int32 mediasourceConnStatus;
   ETG_TRACE_USR4(("AudioServiceClient:: onActiveSourceListUpdate size = %d", SourceList.size()));
   bool isCabinAAudioSourceActive = false;
   bool isCabinA1VideoSourceActive = false;
   bool isCabinA2VideoSourceActive = false;

   bool isCabinBAudioSourceActive = false;
   bool isCabinB1VideoSourceActive = false;
   bool isCabinB2VideoSourceActive = false;
   bool isMic1Active = false;
   static int  prevActivesource = 0;
   while (iter != SourceList.end())
   {
      activesourceId   = iter->getSrcId();
      int32   mediasourceConnStatus = iter->getConnectionState();
      mediasinkidstatus = iter->getSinkId();
      ETG_TRACE_USR4(("AudioServiceClient:: onActiveSourceListUpdate activesourceId = %d, mediasourceConnStatus = %d, mediasinkidstatus = %d", activesourceId, mediasourceConnStatus, mediasinkidstatus));

      /*   if (activesourceId == SRC_MIC1_ANNOUNCEMENT)
        {
           // Mic1 is active
           isMic1Active = true;
           ETG_TRACE_USR4(("AudioServiceClient:: onActiveSourceListUpdate Mic1 active"));
        }
      */

      if (mediasinkidstatus == 3)
      {
         if ((activesourceId == SRC_MIC1_ANNOUNCEMENT) && (mediasourceConnStatus == MEDIA_STATE_CONNECTED))
         {
            ETG_TRACE_USR4(("AudioServiceClient:: onActiveSourceListUpdate Mic1 is  active"));
            prevActivesource = SRC_MIC1_ANNOUNCEMENT;
            isMic1Active = true;
         }
      }
      if ((mediasourceConnStatus == MEDIA_STATE_CONNECTING) || (mediasourceConnStatus == MEDIA_STATE_CONNECTED) || (mediasourceConnStatus == MEDIA_STATE_SUSPENDED))
      {
         switch (mediasinkidstatus)
         {
            case CABIN_A_SINK_ID:
            {
               isCabinAAudioSourceActive = true;
            }
            break;

            case CABIN_A_MEDIAGLASS1_SINK_ID:
            {
               isCabinA1VideoSourceActive = true;
               break;
            }

            case CABIN_A_MEDIAGLASS2_SINK_ID:
            {
               isCabinA2VideoSourceActive = true;
               break;
            }

            case CABIN_B_SINK_ID:
            {
               isCabinBAudioSourceActive = true;
               break;
            }

            case CABIN_B_MEDIAGLASS1_SINK_ID:
            {
               isCabinB1VideoSourceActive = true;
               break;
            }

            case CABIN_B_MEDIAGLASS2_SINK_ID:
            {
               isCabinB2VideoSourceActive = true;
               break;
            }

            default:
               break;
         }
      }

      iter++;
   }
   //for mic deactivation
   ETG_TRACE_USR4(("AudioServiceClient:: onActiveSourceListUpdate Mic1 active prevActivesource=%d", prevActivesource));
   ETG_TRACE_USR4(("AudioServiceClient:: onActiveSourceListUpdate g_sm_Context=%d", g_sm_Context));
   if ((isMic1Active  == false) && (prevActivesource == SRC_MIC1_ANNOUNCEMENT))
   {
      prevActivesource = 0;
      if (g_sm_Context == enActivityIDs__eActivityID_MIC_SETTINGS)
      {
         ETG_TRACE_USR4(("AudioServiceClient:: onActiveSourceListUpdate Mic1 is not active"));
         ApplicationSwitchClientComponent* _applicationSwitchClientComponent = ApplicationSwitchClientComponent::poGetInstance();
         bool isSettingAppInForeground = SystemSettingsListHandler::poGetInstance()->getSettingsAppState();
         ETG_TRACE_USR3(("AudioServiceClient:: onActiveSourceListUpdate isSettingAppInForeground= %d", isSettingAppInForeground));
         if (isSettingAppInForeground == true)
         {
            ETG_TRACE_USR3(("AudioServiceClient:: onActiveSourceListUpdate Sending Homescreen context"));
            _applicationSwitchClientComponent->requestContextSwitchApplication(2, APPID_APPHMI_SYSTEM, enActivityIDs__eActivityID_MIC_SETTINGS,
                  APPID_APPHMI_MASTER, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)0);
         }
         else
         {
            ETG_TRACE_USR3(("AudioServiceClient:: onActiveSourceListUpdate Sending back context"));
            _applicationSwitchClientComponent->requestContextSwitchApplication(2, APPID_APPHMI_SYSTEM, enActivityIDs__eActivityID_MIC_SETTINGS,
                  0, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)0);
         }
      }
   }
   ETG_TRACE_USR4(("AudioServiceClient:: onActiveSourceListUpdate Mic1 is not active prevActivesource =%d", prevActivesource));
   evaluateSourceUpdate(isCabinA1VideoSourceActive, isCabinAAudioSourceActive, CABIN_A_MEDIAGLASS1_SINK_ID, isMic1Active);
   evaluateSourceUpdate(isCabinA2VideoSourceActive, isCabinAAudioSourceActive, CABIN_A_MEDIAGLASS2_SINK_ID, isMic1Active);
   // evaluateSourceUpdate(isCabinB1VideoSourceActive, isCabinBAudioSourceActive, CABIN_B_MEDIAGLASS1_SINK_ID, isMic1Active);
   // evaluateSourceUpdate(isCabinB2VideoSourceActive, isCabinBAudioSourceActive, CABIN_B_MEDIAGLASS2_SINK_ID, isMic1Active);
}


void AudioServiceClient::evaluateSourceUpdate(bool isVideoSourceActive, bool isAudioSourceActive, int sinkid, bool isMic1Active)
{
   bool isUserStandbyOffState = HmiStateHandler::poGetInstance()->isUserStandbyOff();
   ETG_TRACE_USR4(("AudioServiceClient:: evaluateSourceUpdate() isUserStandbyOffState=%d  isVideoSourceActive=%d, isAudioSourceActive=%d,isMic1Active=%d \n", isUserStandbyOffState, isVideoSourceActive, isAudioSourceActive, isMic1Active));
   if (isVideoSourceActive)
   {
      ETG_TRACE_USR4(("AudioServiceClient:: evaluateSourceUpdate: Video source active in sink %d", sinkid));
      CabinAGlassSettingListhandler::poGetInstance()->sendContentStatusRequestToPlugin(true, sinkid, false); //[1] Status [2] Sink id [3] source should not be ignored
   }
   else if (isAudioSourceActive)
   {
      ETG_TRACE_USR4(("AudioServiceClient:: evaluateSourceUpdate: Audio source active in sink %d", sinkid));
      CabinAGlassSettingListhandler::poGetInstance()->sendContentStatusRequestToPlugin(false, sinkid, false);//[1] Status [2] Sink id [3]  source should not be ignored
   }
   else if (isUserStandbyOffState) //tipper off state - 5 second hold - swivel should off and glass off
   {
      ETG_TRACE_USR4(("AudioServiceClient:: evaluateSourceUpdate: No Source active in user off State %d", sinkid));
      CabinAGlassSettingListhandler::poGetInstance()->sendContentStatusRequestToPlugin(false, sinkid, false);//[1] Status [2] Sink id [3]  source should not be ignored
   }
   else
   {
      ETG_TRACE_USR4(("AudioServiceClient:: evaluateSourceUpdate: No source active in sink %d", sinkid));
      CabinAGlassSettingListhandler::poGetInstance()->sendContentStatusRequestToPlugin(false, sinkid, true); //[1] Status [2] Sink id [3]source should be ignored(by sending it has true)
      // Show confirmation pop-up
   }
}


void AudioServiceClient::onActiveSourceListError(const ::boost::shared_ptr< AUDIO_SOURCE_CHANGE::AudioSourceChangeProxy >& proxy,
      const ::boost::shared_ptr< AUDIO_SOURCE_CHANGE::ActiveSourceListError >& error)
{
   ETG_TRACE_USR4(("AudioServiceClient:: onActiveSourceListError() \n"));
}


void AudioServiceClient::onRegionsMapOutStatusInfoError(const ::boost::shared_ptr<HMI_INFO_SERVICE_CHANGE:: HmiInfoServiceProxy >& proxy, const ::boost::shared_ptr<HMI_INFO_SERVICE_CHANGE::RegionsMapOutStatusInfoError >& error)
{
   ETG_TRACE_USR4(("AudioServiceClient:: onRegionsMapOutStatusInfoError() \n"));

   if ((proxy == _hmiInfoServiceProxy) && (error->getAct() == _regionsMapOutStatusInfo_RegId)) //for safety
   {
      ETG_TRACE_USR4(("AudioServiceClient::onRegionsMapOutStatusInfoError Error Message in CondCheck."));
   }
}


/*  */
void AudioServiceClient::onCabinsSystemsInfoError(const ::boost::shared_ptr<HMI_INFO_SERVICE_CHANGE::HmiInfoServiceProxy >& proxy, const ::boost::shared_ptr<HMI_INFO_SERVICE_CHANGE::CabinsSystemsInfoError >& error)
{
   if ((proxy == _hmiInfoServiceProxy))
   {
      ETG_TRACE_USR4(("AudioServiceClient::onCabinsSystemsInfoError Error Message in CondCheck."));
   }
}


void AudioServiceClient::onCabinsSystemsInfoUpdate(const ::boost::shared_ptr<HMI_INFO_SERVICE_CHANGE::HmiInfoServiceProxy >& proxy, const ::boost::shared_ptr<HMI_INFO_SERVICE_CHANGE::CabinsSystemsInfoUpdate >& update)
{
   ETG_TRACE_USR4(("AudioServiceClient::onCabinsSystemsInfoUpdate CondCheck."));
   if ((proxy == _hmiInfoServiceProxy))
   {
#ifdef VARIANT_S_FTR_ENABLE_CABIN_A_SUPPORT
      App::Core::CabinASettingListhandler::poGetInstance()->enableDisableGlass(false);
#endif
      ::std::vector< SystemInfo > systemCategoryArrayData = update->getCabinsSystemsInfo();
      ::std::vector<SystemInfo>::const_iterator systemCategoryArrayData_iter = systemCategoryArrayData.begin();
      while (systemCategoryArrayData_iter != systemCategoryArrayData.end())
      {
         uint8 systemCategory = systemCategoryArrayData_iter->getSystemCategory();
         uint32 entityid = systemCategoryArrayData_iter->getEntityId();
         uint8 systemStatus = systemCategoryArrayData_iter->getSystemStatus();
         uint8 systemPosition = systemCategoryArrayData_iter->getSystemPosition();
         uint8 systemGroup = systemCategoryArrayData_iter->getSystemGroup();
         ETG_TRACE_USR4(("AudioServiceClient::onCabinsSystemsInfoUpdate Error Message in CondCheck systemCategory entityid systemStatus systemPosition systemGroup =%d \t%d \t%d \t%d \t%d \t", systemCategory, entityid, systemStatus, systemPosition, systemGroup));
         ::std::vector< uint32> cabinInfoArrayData = systemCategoryArrayData_iter->getCabinsInfo();
         ::std::vector<uint32>::const_iterator cabinInfoArrayData_iter = cabinInfoArrayData.begin();

         if ((systemCategory ==  SYSTEM_CATEGORY_CMG) && (systemStatus == 1)) //  SYSTEM_CATEGORY_CMG and disconnected)
         {
#ifdef VARIANT_S_FTR_ENABLE_CABIN_A_SUPPORT
            App::Core::CabinASettingListhandler::poGetInstance()->enableDisableGlass(true);
#endif
         }
         while (cabinInfoArrayData_iter != cabinInfoArrayData.end())
         {
            cabinInfoArrayData_iter++;
         }
         systemCategoryArrayData_iter++;
      }
   }
}


void AudioServiceClient::onRegionsMapOutStatusInfoUpdate(const ::boost::shared_ptr<HMI_INFO_SERVICE_CHANGE::HmiInfoServiceProxy >& proxy, const ::boost::shared_ptr<HMI_INFO_SERVICE_CHANGE::RegionsMapOutStatusInfoUpdate >& update)
{
   ETG_TRACE_USR4(("AudioServiceClient:: onRegionsMapOutStatusInfoUpdate() \n"));

   //RegionMapOutStatusInfo Iteration to know the RegionID
   ::std::vector< RegionMapOutStatusInfo> RegionMapOutStatusInfoArrayData = update->getRegionsMapOutStatusInfo();
   ::std::vector<RegionMapOutStatusInfo>::const_iterator regionmapoutstatusinfo_iter = RegionMapOutStatusInfoArrayData.begin();

   if ((proxy == _hmiInfoServiceProxy) && (update->getAct() == _regionsMapOutStatusInfo_RegId)) //for safety
   {
      ETG_TRACE_USR4(("AudioServiceClient:: onRegionsMapOutStatusInfoUpdate IFF LOOP \n"));

      while (regionmapoutstatusinfo_iter != RegionMapOutStatusInfoArrayData.end())
      {
         int32  regionidstatus = regionmapoutstatusinfo_iter->getRegionId();
         ETG_TRACE_USR4(("AudioServiceClient:: onRegionsMapOutStatusInfoUpdate regionidstatus regionmapoutstatusinfo_iter =%d   \n", regionidstatus));

         //MapOutStatusInfo Iteration to get glass id and Connection status getRegionsMapOutStatusInfo
         ::std::vector< MapOutStatusInfo> GlassMapOutStatusInfoArrayData = regionmapoutstatusinfo_iter->getMapOutStatusInfo();
         ::std::vector<MapOutStatusInfo>::const_iterator glassmapoutstatus_iter = GlassMapOutStatusInfoArrayData.begin();

         while (glassmapoutstatus_iter != GlassMapOutStatusInfoArrayData.end())
         {
            int32   glassidstatus = glassmapoutstatus_iter->getGlassId();
            int32   activeglassstatus = glassmapoutstatus_iter->getActiveStatus();
            ETG_TRACE_USR4(("AudioServiceClient:: onRegionsMapOutStatusInfoUpdate glassidstatus activeglassstatus=%d %d\n", glassidstatus, activeglassstatus));

            CabinAGlassSettingListhandler::poGetInstance()->sendMapoutStatusRequestToPlugin(activeglassstatus, glassidstatus);

            glassmapoutstatus_iter++;
         }
         regionmapoutstatusinfo_iter++;
      }
   }
}


////////////////// Removed from HMI BASE Recently
/* void AudioServiceClient::onRequestMainSinkSoundPropertyListError(const ::boost::shared_ptr<AUDIO_SOUND_PROPERTIES::SoundPropertiesProxy >& proxy, const ::boost::shared_ptr< AUDIO_SOUND_PROPERTIES::RequestMainSinkSoundPropertyListError >& error)
{
   if ((proxy == _audioSoundPropertiesProxy) && (error->getAct() == _requestMainSinkSoundPropertyList_SPReg_Id)) //for safety
   {
      ETG_TRACE_USR4(("AudioServiceClient::onRequestMainSinkSoundPropertyListError Error Message in CondCheck....."));
   }
}


//RequestMainSinkSoundPropertyList
void AudioServiceClient:: onRequestMainSinkSoundPropertyListResponse(const ::boost::shared_ptr<AUDIO_SOUND_PROPERTIES::SoundPropertiesProxy >& proxy, const ::boost::shared_ptr< AUDIO_SOUND_PROPERTIES::RequestMainSinkSoundPropertyListResponse >& response)
{
   if ((proxy == _audioSoundPropertiesProxy)) //for safety
   {
      ETG_TRACE_USR4(("AudioServiceClient::onRequestMainSinkSoundPropertyListError Error Message in CondCheck....."));
   }
}


*/

void AudioServiceClient::onSoundPropertyListError(const ::boost::shared_ptr< AUDIO_SOUND_PROPERTIES::SoundPropertiesProxy >& proxy, const ::boost::shared_ptr<AUDIO_SOUND_PROPERTIES::SoundPropertyListError >& error)
{
   if ((proxy == _audioSoundPropertiesProxy) && (error->getAct() == _soundPropertyList_SPRegId)) //for safety
   {
      ETG_TRACE_USR4(("AudioServiceClient::onSoundPropertyListError Error Message in CondCheck....."));
   }
}


void  AudioServiceClient::onSoundPropertyListUpdate(const ::boost::shared_ptr< AUDIO_SOUND_PROPERTIES::SoundPropertiesProxy >& proxy, const ::boost::shared_ptr< AUDIO_SOUND_PROPERTIES::SoundPropertyListUpdate >& update)
{
   ETG_TRACE_USR4(("AudioClientHandler::onSoundPropertyListUpdate()"));

   if ((_audioSoundPropertiesProxy == proxy) && (update->hasSoundPropertyList()))
   {
      const SoundPropertyData& soundPropertyData = update->getSoundPropertyList();
      SinkID_cc = soundPropertyData.getSinkId();
      std::vector< SoundProperty > audioPropertyVector = soundPropertyData.getSoundData();

      std::vector< SoundProperty >::iterator itr = audioPropertyVector.begin();
      for (; itr < audioPropertyVector.end(); ++itr)
      {
         int v_soundpropertytype = itr->getPropertyType();
         int v_soundValue = itr->getValue();

         ETG_TRACE_USR4(("AudioServiceClient::onSoundPropertyListUpdate v_soundpropertytype  v_soundValue =%d %d ", v_soundpropertytype, v_soundValue));

         if ((v_soundpropertytype == 14) && (SinkID_cc == COCKPITSINKID))
         {
            ETG_TRACE_USR4(("AudioServiceClient::onSoundPropertyListUpdate v_soundpropertytype == 14::%d", v_soundValue));
            AudioEquilizerPresetRBtn ::poGetInstance()->UpdateEqlPrstTglStatus(v_soundValue);
         }
         else if ((v_soundpropertytype == 15) && (SinkID_cc == COCKPITSINKID))
         {
            ETG_TRACE_USR4(("AudioServiceClient::onSoundPropertyListUpdate v_soundpropertytype == 15::%d", v_soundValue));
            AudioSpdDepdentVolCntrlRBtn ::poGetInstance()->UpdateSpdDepVolCntrlTglStatus(v_soundValue);
         }
         else if (SinkID_cc == COCKPITSINKID)
         {
            ETG_TRACE_USR4(("AudioServiceClient::onSoundPropertyListUpdate::AudioServiceSoundPropertyCheckOnPowerUpdate  COCKPITSINKID v_soundpropertytype v_soundValue=%d %d %d ", COCKPITSINKID, v_soundpropertytype, v_soundValue));

            AudioSettingListHandler::poGetInstance()->vPerformMainSinkSoundPropertyBassUpdate(COCKPITSINKID, v_soundpropertytype, v_soundValue);
         }//end of if-else for v_soundpropertytype Condition
         else if (SinkID_cc == CABINSASINKID)
         {
            ETG_TRACE_USR4(("CabinAAudioSettingListHandler::onSoundPropertyListUpdate::CabinAAudioSettingListHandler CABINSASINKID v_soundpropertytype v_soundValue=%d %d %d ", CABINSASINKID, v_soundpropertytype, v_soundValue));
#ifndef VARIANT_S_FTR_ENABLE_AUX_SUPPORT
            CabinAAudioSettingListHandler::poGetInstance()->vPerformCabinASinkSoundPropertyBassUpdate(CABINSASINKID, v_soundpropertytype, v_soundValue);
            MicSetting::poGetInstance()->vPerformMainSinkSoundPropertyBassUpdate(CABINSASINKID, v_soundpropertytype, v_soundValue);
#endif
         }
      }//end of for
      //cockpit_propertyupdate = true;
      if (cabinA_propertyupdate == false)
      {
         SinkID_cc = CABINSASINKID;
         //_getListMainSinkSound_RegId = _audioSoundPropertiesProxy->sendGetListMainSinkSoundPropertiesRequest(*this, CABINSASINKID);

         ETG_TRACE_USR4(("AudioServiceClient::onSoundPropertyListUpdate:: CABINSASINKID Registering "));
         _requestMainSinkSoundPropertyList_SPReg_Id = _audioSoundPropertiesProxy->sendRequestMainSinkSoundPropertyListRequest(CABINSASINKID);

         ETG_TRACE_USR4(("AudioServiceClient::onSoundPropertyListUpdate:: CABINSASINKID Registered"));
         cabinA_propertyupdate = true;
      }
   }//end of if
}


/*    ::std::vector< SoundProperty> o_SoundPropertyListArrayData ;
   o_SoundPropertyListArrayData = update->getSoundPropertyList();

   ETG_TRACE_USR4(("AudioServiceClient::onSoundPropertyListUpdate Main "));

   if ((proxy == _audioSoundPropertiesProxy) && (update->getAct() == _soundPropertyList_SPRegId)) //for safety
   {
      ETG_TRACE_USR4(("AudioServiceClient::onSoundPropertyListUpdate Main --IFF LOOP"));
      for (std::vector< SoundProperty >::iterator iter = o_SoundPropertyListArrayData.begin() ; iter != o_SoundPropertyListArrayData.end(); iter++)
      {
         uint32 v_soundpropertytype = iter->getPropertyType();
         uint32 v_soundValue = iter->getValue();

         ETG_TRACE_USR4(("AudioServiceClient::onSoundPropertyListUpdate v_soundpropertytype  v_soundValue =%d %d ", v_soundpropertytype, v_soundValue));

         ETG_TRACE_USR4(("AudioServiceClient::onSoundPropertyListUpdate::v_soundValue, SinkID_cc, v_soundpropertytype %d %d %d", v_soundValue, SinkID_cc, v_soundpropertytype));

         if (SinkID_cc == COCKPITSINKID)
         {
            if (v_soundpropertytype == 14)
            {
               ETG_TRACE_USR4(("AudioServiceClient::onSoundPropertyListUpdate v_soundpropertytype == 14::%d", v_soundValue));
               AudioEquilizerPresetRBtn ::poGetInstance()->UpdateEqlPrstTglStatus(v_soundValue);
            }
            else if (v_soundpropertytype == 15)
            {
               ETG_TRACE_USR4(("AudioServiceClient::onSoundPropertyListUpdate v_soundpropertytype == 15::%d", v_soundValue));
               AudioSpdDepdentVolCntrlRBtn ::poGetInstance()->UpdateSpdDepVolCntrlTglStatus(v_soundValue);
            }
            else
            {
               ETG_TRACE_USR4(("AudioServiceClient::onSoundPropertyListUpdate::AudioServiceSoundPropertyCheckOnPowerUpdate :UpdatePrivateModeCntrlStatus=%d", v_soundValue));
               AudioSettingListHandler::poGetInstance()->vPerformMainSinkSoundPropertyBassUpdate(COCKPITSINKID, v_soundpropertytype, v_soundValue);
            }//end of if-else for v_soundpropertytype Condition
         }//end of SINKID Cockpit If condition Check
         else if (SinkID_cc == CABINSASINKID)
         {
            ETG_TRACE_USR4(("AudioServiceClient::onSoundPropertyListUpdate::v_soundValue, SinkID_cc, v_soundpropertytype %d %d %d", v_soundValue, SinkID_cc, v_soundpropertytype));
#ifndef VARIANT_S_FTR_ENABLE_AUX_SUPPORT
            CabinAAudioSettingListHandler::poGetInstance()->vPerformCabinASinkSoundPropertyBassUpdate(CABINSASINKID, v_soundpropertytype, v_soundValue);
#endif
         }
         else
         {
            ETG_TRACE_USR4(("AudioServiceClient::onSoundPropertyListUpdate::Nothing just else %d %d %d", v_soundValue, SinkID_cc, v_soundpropertytype));
         }
      }
   }
}*/


void AudioServiceClient::onSetMainSinkSoundPropertyError(const::boost::shared_ptr< AUDIO_SOUND_PROPERTIES::SoundPropertiesProxy >& proxy, const::boost::shared_ptr< AUDIO_SOUND_PROPERTIES::SetMainSinkSoundPropertyError >& error)
{
   ETG_TRACE_USR4(("AudioServiceClient::onSetMainSinkSoundPropertyError Error Message....."));
   if ((proxy == _audioSoundPropertiesProxy) && (error->getAct() == _setMainSinkSoundProperty_SPRegId)) //for safety //for safety
   {
      ETG_TRACE_USR4(("AudioServiceClient::onSetMainSinkSoundPropertyError Error Message in CondCheck."));
   }
}


void AudioServiceClient::onSetMainSinkSoundPropertyResponse(const ::boost::shared_ptr< AUDIO_SOUND_PROPERTIES::SoundPropertiesProxy >& proxy, const ::boost::shared_ptr< AUDIO_SOUND_PROPERTIES::SetMainSinkSoundPropertyResponse >& response)
{
   ETG_TRACE_USR4(("AudioServiceClient::onSetMainSinkSoundPropertyResponse Error Message in CondCheck."));

   if ((proxy == _audioSoundPropertiesProxy) && (response->getAct() == _setMainSinkSoundProperty_SPRegId)) //for safety
   {
      ETG_TRACE_USR4(("AudioServiceClient::onSetMainSinkSoundPropertyResponse Error Message in CondCheck."));
   }
}


////////////////// Removed from HMI BASE Recently
/* void AudioServiceClient::onRequestSystemPropertyListError(const ::boost::shared_ptr<AUDIO_SOUND_PROPERTIES:: SoundPropertiesProxy >& proxy, const ::boost::shared_ptr<AUDIO_SOUND_PROPERTIES:: RequestSystemPropertyListError >& error)
{
   if ((proxy == _audioSoundPropertiesProxy) && (error->getAct() == _requestSystemPropertyList_SPReg_Id))
   {
   }
}


void AudioServiceClient::onRequestSystemPropertyListResponse(const ::boost::shared_ptr<AUDIO_SOUND_PROPERTIES:: SoundPropertiesProxy >& proxy, const ::boost::shared_ptr<AUDIO_SOUND_PROPERTIES::RequestSystemPropertyListResponse >& response)
{
   if ((proxy == _audioSoundPropertiesProxy) && (response->getAct() == _requestSystemPropertyList_SPReg_Id))
   {
   }

   ETG_TRACE_USR4(("AudioServiceClient:onGetListSystemPropertiesResponse is called"));
}


*/

void AudioServiceClient::onSystemPropertyListError(const ::boost::shared_ptr<AUDIO_SOUND_PROPERTIES::SoundPropertiesProxy >& proxy, const ::boost::shared_ptr< AUDIO_SOUND_PROPERTIES::SystemPropertyListError >& error)
{
   ETG_TRACE_USR4(("AudioServiceClient:onSystemPropertyListError is called"));

   if ((proxy == _audioSoundPropertiesProxy) && (error->getAct() == _systemPropertyListUpdate_SPRegId)) //for safety
   {
      ETG_TRACE_USR4(("AudioServiceClient:onSystemPropertyListError inside if is called"));
   }
}


void AudioServiceClient::onSystemPropertyListUpdate(const ::boost::shared_ptr<AUDIO_SOUND_PROPERTIES::SoundPropertiesProxy >& proxy, const ::boost::shared_ptr<AUDIO_SOUND_PROPERTIES::SystemPropertyListUpdate >& update)
{
   ETG_TRACE_USR4(("AudioServiceClient:onSystemPropertyListUpdate is called"));

   ::std::vector< SystemProperty > o_SystemPropertyListArrayData ;
   o_SystemPropertyListArrayData = update->getSystemPropertyList();
//(update->hasSystemPropertyList())
   if ((proxy == _audioSoundPropertiesProxy) && (update->hasSystemPropertyList()))//(update->getAct() == _systemPropertyListUpdate_SPRegId)) //for safety
   {
      ETG_TRACE_USR4(("AudioServiceClient:onGetListSystemPropertiesResponse is IFF LOOP"));

      for (std::vector< SystemProperty >::iterator iter = o_SystemPropertyListArrayData.begin() ; iter != o_SystemPropertyListArrayData.end(); iter++)
      {
         uint32 systemproperty_ckpit_pvtmod = static_cast<uint16>(iter->getPropertyType());
         uint32 systemvalue_ckpit_pvtmod         = static_cast<uint16>(iter->getValue());
         ETG_TRACE_USR4(("AudioServiceClient::onGetListMainSinkSoundPropertiesResponse::systemvalue_ckpit_pvtmod, SinkID_cc, systemproperty_ckpit_pvtmod %d %d %d", systemvalue_ckpit_pvtmod, SinkID_cc, systemproperty_ckpit_pvtmod));

         if (systemproperty_ckpit_pvtmod == 13)
         {
            ETG_TRACE_USR4(("AudioServiceClient:onGetListSystemPropertiesResponse IF systemproperty_ckpit_pvtmod is %d", systemproperty_ckpit_pvtmod));
            ETG_TRACE_USR4(("AudioServiceClient:onGetListSystemPropertiesResponse IF systemvalue_ckpit_pvtmod is %d", systemvalue_ckpit_pvtmod));
            AudioPrivateModeRBtn ::poGetInstance()->UpdatePrivateModeCntrlStatus(systemvalue_ckpit_pvtmod);
         }//end of if
      }//end of for
   }//end of if
}


/************************************************************************
*FUNCTION: 		 AudioServiceClient - onSetSystemPropertyResponse
*DESCRIPTION: 	 Handle to perform the MainSinkSoundProperty user change and set the new value to Audio Service.
*PARAMETER:
*RETURNVALUE: 	 None
*
*HISTORY:

*13.01.2019   Rev. 0.1 Sathiya

*0.1 Initial Revision.
************************************************************************/
void AudioServiceClient::onSetSystemPropertyResponse(const ::boost::shared_ptr< AUDIO_SOUND_PROPERTIES::SoundPropertiesProxy >& proxy, const ::boost::shared_ptr< AUDIO_SOUND_PROPERTIES::SetSystemPropertyResponse >& response)
{
   ETG_TRACE_USR4(("AudioServiceClient:onSetSystemPropertyResponse is called"));

   /* if ((proxy == _audioSoundPropertiesProxy) )//&& (response->getAct() == _setSystemProperty_SPRegId)) //for safety
       {
   	   ETG_TRACE_USR4(("AudioServiceClient:onSetSystemPropertyResponse inside if is called"));
       }  */
}


/************************************************************************
*FUNCTION: 		 AudioServiceClient - onSetSystemPropertyError
*DESCRIPTION: 	 Handle to perform the MainSinkSoundProperty user change and set the new value to Audio Service.
*PARAMETER:
*RETURNVALUE: 	 None
*
*HISTORY:

*13.01.2019   Rev. 0.1 Sathiya

*0.1 Initial Revision.
************************************************************************/
void AudioServiceClient:: onSetSystemPropertyError(const ::boost::shared_ptr< AUDIO_SOUND_PROPERTIES::SoundPropertiesProxy >& proxy, const ::boost::shared_ptr< AUDIO_SOUND_PROPERTIES::SetSystemPropertyError >& error)
{
   ETG_TRACE_USR4(("AudioServiceClient:onSetSystemPropertyError is called"));

   /*   if ((error == _audioSoundPropertiesProxy) )//&& (error->getAct() == _setSystemProperty_SPRegId)) //for safety
     {
      ETG_TRACE_USR4(("AudioServiceClient:onSetSystemPropertyError inside if is called"));
     }   */
}


}
}
