/**************************************************************************************
* @file         : SystemSettingListHandler.cpp
* @author       : Sheena Mathew
* @addtogroup   : AppHmi_System
* @brief        :
* @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/
#include "gui_std_if.h"
#include <string>
//#include "util/StrUtf8.h"
#include "SystemSettingsListHandler.h"
#include "App/Core/ClientHandler/VehicleSrvClient/VehicleSrvClient.h"
#include "App/Core/ClientHandler/DownloadSrvClient/SWUpdateClient.h"
#include "App/Core/ClientHandler/SPM/SpmSrvClient.h"
#ifdef VARIANT_S_FTR_ENABLE_NAVIGATION
#include "App/Core/ClientHandler/BTSrvClient/ServiceBT.h"
#endif
#include "App/Core/ClientHandler/Diagnosis/DiagnosisSrvClient/DiagnosisSrvClient.h"
#include "App/Core/AppLogic/CabinA/CabinASettingListhandler/CabinASettingListhandler.h"
#include "hmi_trace_if.h"
#include "CgiExtensions/ImageLoader.h"
#include "App/Core/Utils/XMLParser/XMLParser.h"

#define DP_S_IMPORT_INTERFACE_KDS_FI
#include "dp_kds_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_SYSTEM_HALL
#define ETG_I_TRACE_CHANNEL TR_TTFIS_APPHMI_SYSTEM
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_SYSTEM_"
#define ETG_I_FILE_PREFIX                 App::Core::SystemSettingsListHandler::
#include "trcGenProj/Header/SystemSettingsListHandler.cpp.trc.h"
#endif
//#define VARIANT_S_FTR_ENABLE_COACH_FEATURE
//#define VARIANT_S_FTR_ENABLE_AUX
//#define VARIANT_S_FTR_ENABLE_SPEED_LIMIT
//#define VARIANT_S_FTR_ENABLE_NAVIGATION
namespace App {
namespace Core {
SystemSettingsListHandler* SystemSettingsListHandler::m_posystemSettingsListHandler = NULL;
// Template used for simple button items FlexListWidget2D. The DataItem has the following structure:
// 0 - identifier - Name(Id) of the button which can be used to retrieve the ListId, HdlRow, HdlCol, Flags from the Sender parameter of the ButtonReactionMsg.
// 1 - string     - Text of the button.
// 2 - int        - IsSelected (0 - not selected, 1 - selected) used to toggle the visibility of an icon.
static const char* const DATA_CONTEXT_TEXT_FIRST_LIST_ITEM = "Sound_Setting";
static const char* const DATA_CONTEXT_TEXT_BUTTON_LIST_ITEM = "TextButton_ListItem";
static const char* const DATA_CONTEXT_COMBOBOX_LIST_ITEM = "ComboBox_ListItem";
static const char* const DATA_CONTEXT_TEXT_LIST_ITEM = "Layer_Sound";
static const char* const DATA_CONTEXT_TEXT_ITEM = "Layer_Language";
//static const char* const DATA_CONTEXT_TEXT_ITEM_SERIAL_NUMBER = "Serial_Number";
static const char* const DATA_CONTEXT_TEXT_ITEM_DATA_VERSION = "Layer_DataVersion";
static const char* const DATA_CONTEXT_TEXT_ITEM_SERIAL_NUMB = "Layer_Serial";
static const char* const DATA_CONTEXT_TEXT_ITEM_SOFTWARE_VERSION = "Layer_SoftwareVersion";
static const char* const DATA_CONTEXT_TEXT_BUTTON_SWIPETOGGLE_LIST_ITEM = "Button_Text_SwipeToggle_Icon";

// Template used for simple toggle button items FlexListWidget2D. The DataItem has the following structure:
// 0 - identifier - Name(Id) of the button which can be used to retrieve the ListId, HdlRow, HdlCol, Flags from the Sender parameter of the ButtonReactionMsg.
// 1 - string     - Text of the button.
// 2 - int        - IsSelected (0 - not selected, 1 - selected) used to toggle the visibility of an icon. DATA_ITEM_SELECTED_IGNORED can be used if IsSelected should be ignored.
static const char* const DATA_CONTEXT_TEXT_BUTTON_TOGGLE_LIST_ITEM = "Layer_Checkbox";
//const char* MeasurementUnitStrings[2]  = { "Imperial", "Metric"};
const uint32 MAX_STEPCOUNT = 100;
Candera::String  MeasurementUnitStrings[2]  = {Candera::String(TextId_0x025F), Candera::String(TextId_0x025E)};
Candera::String SystemsettingMain[3] = {Candera::String(TextId_0x007A), Candera::String(TextId_0x0262), Candera::String(TextId_0x10DB)};
const char* MeasurementUnit_ListItemText[1][2] =
{
   { "Measurement Unit", "Metric" },
};


/************************************************************************
*FUNCTION: 		 SystemSettingListHandler - Constructor
*DESCRIPTION: 	 default constructor of the System setting Listhadler will be created
*PARAMETER:		 None
*RETURNVALUE: 	 None
*
*HISTORY:
*27.3.2018:Sheena Mathew

*0.1 Initial Revision.
************************************************************************/

SystemSettingsListHandler::SystemSettingsListHandler()
{
   ETG_TRACE_COMP(("SystemSettingListHandler:: Constructor is called"));
   VariantType();
   vCreateTemplateList();
   _speedlockStatus = true;
   rowIdx = 0;
   _currentMeasurementOption = Metric;
   _RxdCurrentLanguageStatus4mVehData = 23;
   _currentSoftwareVersion = "";
   SYSTEMMAINTEXT = Candera::String(TextId_0x007A);
   _hmidatasrvhandler = HmiDataServiceClientHandler::poGetInstance();
   _currentMeasurementOptionStr = "Imperial";
   _CurrentLanguageStatusStringData2Home = "UK English";
   intializalanguagedata();
#if defined VARIANT_S_FTR_ENABLE_NAVIGATION
   _bluetoothStatus = false;
#endif
   listcreated = false;
   _dataversion = "";
   //my_vec.clear();
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_SYSTEM_MAIN_LIST, this);
   // ListRegistry::s_getInstance().addListImplementation(LIST_ID_SYSTEM_MEASUREMENTUNIT_LIST, this);
   _serialnumber = "";
   vGetSerialNumber();
   // intializalanguagedata();
   //my_vec.clear();
   //_hmidatasrvhandler->SetSpeedlock(true);
   _isSettingAppInForeground = false;
}


/************************************************************************
*FUNCTION: 		 SystemSettingsListHandler- intializalanguagedata
*DESCRIPTION: 	 Populate the mpLanguageStringStatus vector map
*PARAMETER:		 None
*RETURNVALUE: 	 None
*
*HISTORY:
*14.6.2019:Sushmita Dikshit

*0.2 added 2 more mapping values Revision.
************************************************************************/


void SystemSettingsListHandler::intializalanguagedata()
{
   ETG_TRACE_COMP(("SystemSettingsListHandler::intializalanguagedata"));
   mpLanguageStringStatus.insert(make_pair(12, "UK English"));
   mpLanguageStringStatus.insert(make_pair(47, "Español"));
   mpLanguageStringStatus.insert(make_pair(22, "Français"));
   mpLanguageStringStatus.insert(make_pair(23, "Deutsch"));
   mpLanguageStringStatus.insert(make_pair(10, "Nederlands"));
   mpLanguageStringStatus.insert(make_pair(21, "Français"));
   mpLanguageStringStatus.insert(make_pair(29, "Italiano"));
   mpLanguageStringStatus.insert(make_pair(46, "Español"));
   mpLanguageStringStatus.insert(make_pair(39, "Português"));
   mpLanguageStringStatus.insert(make_pair(27, "Magyar"));
#ifndef VARIANT_S_FTR_ENABLE_AUX_SUPPORT
   mpLanguageStringStatus.insert(make_pair(42, "Русский"));
   mpLanguageStringStatus.insert(make_pair(13, "US English"));
#endif
   mpLanguageStringStatus.insert(make_pair(8, "Čeština"));
   mpLanguageStringStatus.insert(make_pair(9, "Dansk"));
   mpLanguageStringStatus.insert(make_pair(35, "Norsk"));
   mpLanguageStringStatus.insert(make_pair(38, "Polski"));
   mpLanguageStringStatus.insert(make_pair(49, "Svenska"));
   mpLanguageStringStatus.insert(make_pair(24, "Ελληνικά"));
   mpLanguageStringStatus.insert(make_pair(52, "Türkçe"));

#if defined VARIANT_S_FTR_ENABLE_AUX_SUPPORT
   mpLanguageStringStatus.insert(make_pair(3, "български"));
   mpLanguageStringStatus.insert(make_pair(45, "Slovenščina"));
   mpLanguageStringStatus.insert(make_pair(44, "Slovenčina"));
   mpLanguageStringStatus.insert(make_pair(41, "Română"));
   mpLanguageStringStatus.insert(make_pair(19, "Hrvatski"));
   mpLanguageStringStatus.insert(make_pair(7, "Hrvatski"));
#endif
}


void SystemSettingsListHandler::vCreateTemplateList()
{
   ETG_TRACE_COMP(("SystemSettingListHandler:: vCreateTemplateList is called"));
   SMDI.clear();
   SMDI.push_back("Sound");
   SMDI.push_back("Display");
   SMDI.push_back("Clock");
   SMDI.push_back("Language");
   SMDI.push_back("ComboBox_ListItem");
   SMDI.push_back("Lock_Keypad");
   SMDI.push_back("Open_SRC_SW");
   SMDI.push_back("Reset_to_Factory");
   SMDI.push_back("Software_Version");
   SMDI.push_back("Configuration_Version");
   SMDI.push_back("Data_Carrier");
   SMDI.push_back("Serial_Number");
   SMDI.push_back("SW_Update");
   SMDI.push_back("SW_Update_History");
   SMDI.push_back("Bluetooth");
   SMDI.push_back("Service_Menu");
}


/************************************************************************
*FUNCTION: 		 SystemSettingListHandler - Destructor
*DESCRIPTION: 	 Allocated resources such as memory will be released here
*PARAMETER:		 None
*RETURNVALUE: 	 None
*
*HISTORY:
*27.3.2018:Sheena Mathew

*0.1 Initial Revision.
************************************************************************/

SystemSettingsListHandler::~SystemSettingsListHandler()
{
   ETG_TRACE_COMP(("SystemSettingListHandler:: Destructor is called"));
   my_vec.clear();
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_SYSTEM_MAIN_LIST);
   //ListRegistry::s_getInstance().removeListImplementation(LIST_ID_SYSTEM_MEASUREMENTUNIT_LIST);
   listcreated = false;
   delete _hmidatasrvhandler;
}


bool SystemSettingsListHandler::onCourierMessage(const Courier::StartupMsg& /*oMsg*/)
{
   ETG_TRACE_COMP(("SystemSettingListHandler:: StartupMsg is called"));
   return true;
}


Candera::String SystemSettingsListHandler::vGetSerialNumber()
{
   ETG_TRACE_USR1(("SystemSettingListHandler:vGetSerialNumber is called"));
   const uint8 TEXT_BUFFER_LENGTH = 8;
   unsigned char version[TEXT_BUFFER_LENGTH] = {0};
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("ECUSerialNumber", "", version, TEXT_BUFFER_LENGTH))
   {
      ETG_TRACE_USR4(("SystemSettingListHandler:: serialnumber =%s", version));
      std::string serialnum(version, version + TEXT_BUFFER_LENGTH);
      ETG_TRACE_USR4(("SystemSettingListHandler:: serialnum =%s", serialnum.c_str()));
      _serialnumber = Candera::String(serialnum.c_str());
      ETG_TRACE_USR4(("SystemSettingListHandler:: _serialnumber =%s", _serialnumber));
   }
   return _serialnumber;
}


/**********************************************************************************************
*FUNCTION: 		 SystemSettingListHandler - getListDataProvider
*DESCRIPTION: 	 Creates a list data provider which would contain unique list id
*PARAMETER:		 oMsg - Courier message received
*RETURNVALUE: 	 tSharedPtrDataProvider()
*
*HISTORY:
*27.3.2018:Sheena Mathew

*0.1 Initial Revision.
**********************************************************************************************/

tSharedPtrDataProvider SystemSettingsListHandler::getListDataProvider(const ListDateProviderReqMsg& oMsg)
{
   ETG_TRACE_COMP(("SystemSettingListHandler:: getListDataProvider: List Id = %d", oMsg.GetListId()));
   switch (oMsg.GetListId())
   {
      case LIST_ID_SYSTEM_MAIN_LIST:
         return getMainViewListDataProvider();
      default:
      {
         break;
      }
   }
   return tSharedPtrDataProvider();
}


tSharedPtrDataProvider SystemSettingsListHandler::getMainViewListDataProvider()
{
   ListDataProviderBuilder listBuilder(LIST_ID_SYSTEM_MAIN_LIST);
   _currentSoftwareVersion = SWUpdateClient::GetInstance()->getCurrentSytemVersion();
#if defined VARIANT_S_FTR_ENABLE_NAVIGATION
   for (int index = 0; index < SMDI.size(); index++)
   {
      ETG_TRACE_USR1(("SystemSettingListHandler::getMainViewListDataProvider: index = %d", index));
      if ((index != ((int)LANGUAGE)) && (index != ((int) MEASUREMENT_UNIT)) && (index != ((int) SPEEDLOCK)) && (index != ((int)SERIALNUMBER)) && (index != ((int) DATACARIERVERSION)) && (index != ((int)BLUETOOTHSETTING)))
      {
         listBuilder.AddItem(index, 0UL, SMDI[index].c_str());
      }
      else if (index == ((int)LANGUAGE))
      {
         listBuilder.AddItem(index, 0UL, SMDI[index].c_str()).AddData(_CurrentLanguageStatusStringData2Home.c_str());
      }
      else if (index == ((int)SERIALNUMBER))
      {
         listBuilder.AddItem(index, 0UL, SMDI[index].c_str()).AddData(_serialnumber);
      }
      else if (index == ((int)DATACARIERVERSION))
      {
         // Data carrier version list item should be displayed only for Navi variant.
         if (VariantType() == NON_NAVI)
         {
            ETG_TRACE_USR1(("SystemSettingListHandler::getMainViewListDataProvider DATACARIERVERSION not available for Non Navi variant"));
         }
         else
         {
            ETG_TRACE_USR1(("SystemSettingListHandler::getMainViewListDataProvider DATACARIERVERSION required for Navi variant"));
            listBuilder.AddItem(index, 0UL, SMDI[index].c_str()).AddData(_dataversion);
         }
      }
      else if (index == ((int)BLUETOOTHSETTING))
      {
         listBuilder.AddItem(index, 0UL, SMDI[index].c_str()).AddData(_bluetoothStatus);
      }
      else if (index == ((int)SPEEDLOCK))
      {
         listBuilder.AddItem(index, 0UL, SMDI[index].c_str()).AddData(_speedlockStatus);
      }
      else if (index == ((int)MEASUREMENT_UNIT))
      {
         //Feature 1937873 : [SYSFL-10186] Measurement Unit is required only for NAVI variant
         if (VariantType() == NAVI)
         {
            ETG_TRACE_USR1(("SystemSettingListHandler::getMainViewListDataProvider MEASUREMENT_UNIT is required for Navi variant"));
            listBuilder.AddItem(index, 0UL, SMDI[index].c_str())
            .AddData(Candera::String(TextId_0x0262))
            .AddId(index, 1).AddData(MeasurementUnitStrings[0]).AddData(false)
            .AddId(index, 2).AddData(MeasurementUnitStrings[1]).AddData(false)
            .AddData(_currentMeasurementOptionStr);
         }
      }
   }
#endif
   listcreated = true;
   return listBuilder.CreateDataProvider();
}


string SystemSettingsListHandler::vUpdateLanguageStatusString(uint32 Languageindex)
{
   string langstr = " ";
   map<int, string>::iterator it = mpLanguageStringStatus.find(Languageindex);
   if (it != mpLanguageStringStatus.end())
   {
      langstr = it->second;
   }
   ETG_TRACE_USR1(("SystemSettingsListHandler::vUpdateLanguageStatusString: Languageindex = %d, langstr = %s", Languageindex, langstr.c_str()));
   return langstr;
}


bool SystemSettingsListHandler::onCourierMessage(const ButtonListItemUpdMsg& oMsg)
{
   bool retval = false;
   rowIdx = (unsigned int)oMsg.GetHdl();
   ListDataProviderBuilder modifiedListItems(LIST_ID_SYSTEM_MAIN_LIST);
   CheckBoxListItemData item;
   if (oMsg.GetListId() == LIST_ID_SYSTEM_MAIN_LIST)
   {
      ETG_TRACE_USR1(("SystemSettingsListHandler::ButtonListItemUpdMsg: LIST_ID_SYSTEM_MAIN_LIST: Action = %d, rowIdx = %d", oMsg.GetReaction(), rowIdx));

      if (oMsg.GetReaction() == enRelease)
      {
         switch (oMsg.GetHdl())
         {
#if defined VARIANT_S_FTR_ENABLE_NAVIGATION
            case MEASUREMENT_UNIT:
            {
               ETG_TRACE_USR1(("SystemSettingsListHandler::ButtonListItemUpdMsg: LIST_ID_SYSTEM_MAIN_LIST: MEASUREMENT_UNIT"));
               changeMeasurementSettings(oMsg.GetSubHdl());
            }
            break;
            case SPEEDLOCK:
            {
               ETG_TRACE_USR1(("SystemSettingsListHandler::ButtonListItemUpdMsg: LIST_ID_SYSTEM_MAIN_LIST: SPEEDLOCK"));
               _speedlockStatus = !_speedlockStatus;
               _hmidatasrvhandler->SetSpeedlock(_speedlockStatus);
               retval = true;
            }
            break;
            case BLUETOOTHSETTING:
            {
               showWaitAnimation(true);
               _bluetoothStatus = !_bluetoothStatus;
               ETG_TRACE_USR4(("SystemSettingsListHandler::ButtonListItemUpdMsg: LIST_ID_SYSTEM_MAIN_LIST: _bluetoothStatus=%d ", _bluetoothStatus));
               if (_bluetoothStatus == true)
               {
                  ServiceBT::poGetInstance()->SetBluetoothState(true);
               }
               else
               {
                  ServiceBT::poGetInstance()->SetBluetoothState(false);
               }
               retval = true;
            }
            break;
            case SOFTWAREVERSION:
            {
               POST_MSG(COURIER_MESSAGE_NEW(SoftwareVersionCISMsg)());
               retval = true;
            }
            break;
            case STARTSOFTWAREUPDATE:
            {
               POST_MSG(COURIER_MESSAGE_NEW(StartSoftwareUpdatePressMsg)());
               retval = true;
            }
            break;
            case OEM_SETTING:
            {
               POST_MSG(COURIER_MESSAGE_NEW(OEMSettingMsg)());
               retval = true;
            }
            break;
#endif
            case OPENSOURCESOFTWARE:
            {
               POST_MSG(COURIER_MESSAGE_NEW(OpenSourceSoftwareMsg)());
               retval = true;
            }
            break;
            case RESETTOFACTORYSETTING:
            {
               POST_MSG(COURIER_MESSAGE_NEW(ResetFactorySettingsMsg)());
               retval = true;
            }
            break;
            case SOFTWAREUPDATEHISTORY:
            {
               POST_MSG(COURIER_MESSAGE_NEW(StartSoftwareUpdateHistoryMsg)());
               retval = true;
            }
            break;
            default:
               break;
         }
      }
      if (oMsg.GetReaction() == enLongPress)
      {
         switch (oMsg.GetHdl())
         {
#if defined VARIANT_S_FTR_ENABLE_NAVIGATION
            case STARTSOFTWAREUPDATE:
            {
               POST_MSG(COURIER_MESSAGE_NEW(StartSoftwareUpdateLongPressMsg)());
            }
            break;
#endif
            default:
               break;
         }
      }
   }
   return retval;
}


void SystemSettingsListHandler::changeMeasurementSettings(Candera::UInt32 selectedRowId)
{
   ETG_TRACE_USR4(("SystemSettingsListHandler::void SystemSettingsListHandler::changeMeasurementSettings(Candera::UInt32 selectedRowId)(), Row Id : %d", selectedRowId));

   if (selectedRowId <= 2)
   {
      // Subtract 1 to match the selection and enumeration index,
      // handle for first option in dropdown is 1. However, the enumeration value for same is 0
      _currentMeasurementOption = static_cast<enMenu>(selectedRowId);
   }

   VehicleSrvClient::poGetInstance()->setdistanceunit(_currentMeasurementOption);
}


void SystemSettingsListHandler::sendDistanceUnitUpdateToGUI(uint32 DistanceUnitCode)
{
   if (DistanceUnitCode == 1)
   {
      _currentMeasurementOptionStr =	MeasurementUnitStrings[0];
   }
   else
   {
      _currentMeasurementOptionStr =	MeasurementUnitStrings[1];
   }
   ETG_TRACE_USR1(("SystemSettingsListHandler::sendDistanceUnitUpdateToGUI: DistanceUnitCode =%d, _currentMeasurementOptionStr = %s\n", DistanceUnitCode, _currentMeasurementOptionStr.GetCString()));
   ::hmibase::widget::combobox::ComboBoxReqMsg* cbToggleMsg = COURIER_MESSAGE_NEW(::hmibase::widget::combobox::ComboBoxReqMsg)(0, ::hmibase::widget::combobox::enComboBoxAction::Close); //ComboBoxAction enComboBoxAction::Enum::ComboBoxClose:
   if (cbToggleMsg != NULL)
   {
      cbToggleMsg->Post();
   }

   ListRegistry::s_getInstance().updateList(LIST_ID_SYSTEM_MAIN_LIST);
}


void SystemSettingsListHandler::showHidePrivacyPopUp(uint8 PrivacyResetRequestUnit)
{
   ETG_TRACE_COMP(("SystemSettingsListHandler::showHidePrivacyPopUp: PrivacyResetRequestUnit is %d ", PrivacyResetRequestUnit));
   if (PrivacyResetRequestUnit == 1)
   {
      POST_MSG((COURIER_MESSAGE_NEW(Popup_Privacy_open)()));
      ETG_TRACE_USR3(("In Privacy reset popup"));
      DiagnosisSrvClient::poGetInstance()->privacyResetValue();
   }
}


void SystemSettingsListHandler::SetspeedlockVisibilityStatus(bool status)
{
   _speedlockStatus = status;
   ETG_TRACE_USR1(("SystemSettingsListHandler::SetspeedlockVisibilityStatus: _speedlockStatus = %d ", _speedlockStatus));
   ListRegistry::s_getInstance().updateList(LIST_ID_SYSTEM_MAIN_LIST);
   // ListRegistry::s_getInstance().updateList(LIST_ID_SYSTEM_MEASUREMENTUNIT_LIST);
}


void SystemSettingsListHandler::SetBluetoothStatus(bool status)
{
   showWaitAnimation(false);
   _bluetoothStatus = status;
   ETG_TRACE_USR1(("SystemSettingsListHandler::SetBluetoothStatus: _bluetoothStatus = %d ", _bluetoothStatus));
   if (listcreated == true)
   {
      ListRegistry::s_getInstance().updateList(LIST_ID_SYSTEM_MAIN_LIST);
   }
}


void SystemSettingsListHandler::SetLanguageStatus(uint32 languagecode)
{
   _RxdCurrentLanguageStatus4mVehData = languagecode;
   _CurrentLanguageStatusStringData2Home = vUpdateLanguageStatusString(_RxdCurrentLanguageStatus4mVehData);
   ETG_TRACE_USR1(("SystemSettingsListHandler::SetLanguageStatus: languagecode = %d, _CurrentLanguageStatusStringData2Home = %s", languagecode, _CurrentLanguageStatusStringData2Home.c_str()));
#ifdef VARIANT_S_FTR_ENABLE_CABIN_A_SUPPORT
   if (CabinASettingListhandler::poGetInstance() != NULL)
   {
      CabinASettingListhandler::poGetInstance()->setSelectedLanguage(_CurrentLanguageStatusStringData2Home);
   }
#endif
   if (listcreated == true)
   {
      ListRegistry::s_getInstance().updateList(LIST_ID_SYSTEM_MAIN_LIST);
   }
}


void SystemSettingsListHandler::vDataversion(std::string dataversion)
{
   _dataversion = Candera::String(dataversion.c_str());
   ETG_TRACE_USR1(("SystemSettingsListHandler::vDataversion: _dataversion = %s", _dataversion.GetCString()));
}


/************************************************************************
*FUNCTION: 		 SystemSettingListHandler - variantType
*DESCRIPTION: 	 Checks if the current variant is Non Navi or Navigation supported.
*PARAMETER:		 None
*RETURNVALUE: 	 bool
*
*HISTORY:
*30.10.2023:Sriganesh
************************************************************************/
bool SystemSettingsListHandler::VariantType()
{
   ETG_TRACE_USR4(("SystemSettingsListHandler::variantType is called"));
   uint8 value = 0xFF;
   uint16 u16Variant = 0;
   bool Variant_Type;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("AIVIVariantCoding", "VariantSelection", &value, 1))
   {
      u16Variant = value;
   }
   ETG_TRACE_USR1(("SystemSettingsListHandler::VariantType u16Variant = %d", u16Variant));

   Variant_Type = (u16Variant == 0x02) ? true : false; // 0x02 - Non Navi, 0x01 - Navi.
   ETG_TRACE_USR1(("SystemSettingsListHandler::VariantType Variant_Type = %d", Variant_Type));
   return Variant_Type;
}


bool SystemSettingsListHandler::onCourierMessage(const ApplicationStateUpdMsg& oMsg)
{
   ETG_TRACE_COMP(("SystemSettingsListHandler::ApplicationStateUpdMsg: applicationState=%d", oMsg.GetState()));
   if (oMsg.GetState() == hmibase::IN_FOREGROUND)
   {
      _isSettingAppInForeground = true;
   }
   else
   {
      _isSettingAppInForeground = false;
   }
   ETG_TRACE_COMP(("SystemSettingsListHandler::ApplicationStateUpdMsg: _isSettingAppInForeground=%d", _isSettingAppInForeground));
   return false;
}


bool SystemSettingsListHandler::getSettingsAppState()
{
   ETG_TRACE_COMP(("SystemSettingsListHandler::getSettingsAppState: _isSettingAppInForeground=%d", _isSettingAppInForeground));
   return _isSettingAppInForeground;
}


void SystemSettingsListHandler::isSWUpdateviaDiagTool(bool status)
{
   ETG_TRACE_COMP(("SystemSettingsListHandler::isSWUpdateviaDiagTool: status=%d , %d", status, !status));
   (*_SWUpdateListItemEnable).mButtonEnable = !status;
   _SWUpdateListItemEnable.MarkAllItemsModified();
   _SWUpdateListItemEnable.SendUpdate(true);
   if (listcreated == true)
   {
      ListRegistry::s_getInstance().updateList(LIST_ID_SYSTEM_MAIN_LIST);
   }
}


void SystemSettingsListHandler::showWaitAnimation(bool status)
{
   ETG_TRACE_COMP(("SystemSettingsListHandler::showWaitAnimation: status=%d", status));
   (*_systemSettingsNodeSwitch).misWaitAnimationEnabled = status;
   _systemSettingsNodeSwitch.MarkAllItemsModified();
   _systemSettingsNodeSwitch.SendUpdate(true);
}


}
}
