/**************************************************************************************
* @file         : MicSetting.cpp
* @author       : Sriganesh
* @addtogroup   : AppHmi_System
* @brief        :
* @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/
#include "gui_std_if.h"
#include <string>
#include "MicSetting.h"
#include "App/Core/ClientHandler/Audio/AudioServiceClient.h"
#include "AppHmi_MasterBase/AudioInterface/AudioDefines.h"
#include <cstdlib>
#include "hmi_trace_if.h"
#include <vector>
#include <algorithm>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_SYSTEM_HALL
#define ETG_I_TRACE_CHANNEL TR_TTFIS_APPHMI_SYSTEM
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_SYSTEM_"
#define ETG_I_FILE_PREFIX                 App::Core::MicSetting::
#include "trcGenProj/Header/MicSetting.cpp.trc.h"
#endif

namespace App {
namespace Core {

MicSetting* MicSetting::m_poMicSetting = NULL;

#define COLUMN_BUTTON_MINUS 1
#define COLUMN_BUTTON_PLUS  2


/************************************************************************
*FUNCTION: 		 MicSetting - Constructor
*DESCRIPTION: 	 default constructor of the System setting Listhadler will be created
*PARAMETER:		 None
*RETURNVALUE: 	 None
************************************************************************/

MicSetting::MicSetting()
{
   ETG_TRACE_COMP(("MicSetting:: Constructor is called"));
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_SYSTEM_MIC_SETTINGS, this);
   onCreateMaplistUpdate();
   bass_value = 0;
   treble_value = 0;
   listcreated = false;
   m_propertytype = 0;
   signtype = 0;
   _bass = "0";
   _treble = "0";
}


/************************************************************************
*FUNCTION: 		 MicSetting - Destructor
*DESCRIPTION: 	 Allocated resources such as memory will be released here
*PARAMETER:		 None
*RETURNVALUE: 	 None
************************************************************************/

MicSetting::~MicSetting()
{
   ETG_TRACE_COMP(("MicSetting:: Destructor is called"));
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_SYSTEM_MIC_SETTINGS);
   listcreated = false;
}


void MicSetting::onCreateMaplistUpdate()
{
   ETG_TRACE_COMP(("MicSetting:: onCreateMaplistUpdate is called"));
   m_ListAudSpectoServAudSpec[MIC_BASS] = MSP_SOUND_BASS_RELATIVE;
   m_ListAudSpectoServAudSpec[MIC_TREBBLE] = MSP_SOUND_TREBBLE_RELATIVE;
}


/**********************************************************************************************
*FUNCTION: 		 MicSetting - getListDataProvider
*DESCRIPTION: 	 Creates a list data provider which would contain unique list id
*PARAMETER:		 oMsg - Courier message received
*RETURNVALUE: 	 tSharedPtrDataProvider()
**********************************************************************************************/
tSharedPtrDataProvider MicSetting::getListDataProvider(const ListDateProviderReqMsg& oMsg)
{
   ETG_TRACE_COMP(("MicSetting:: getListDataProvider: List Id = %d", oMsg.GetListId()));
   switch (oMsg.GetListId())
   {
      case LIST_ID_SYSTEM_MIC_SETTINGS:
         return getMainViewListDataProvider();
   }
   return tSharedPtrDataProvider();
}


tSharedPtrDataProvider MicSetting::getMainViewListDataProvider()
{
   ListDataProviderBuilder listBuilder(LIST_ID_SYSTEM_MIC_SETTINGS);

   listBuilder.AddItem(MIC_BASS, 0UL, "Bass")
   .AddId(MIC_BASS, COLUMN_BUTTON_MINUS)     // -
   .AddId(MIC_BASS, COLUMN_BUTTON_PLUS)// +
   .AddData(_bass);

   listBuilder.AddItem(MIC_TREBBLE, 0UL, "Treble")
   .AddId(MIC_TREBBLE, COLUMN_BUTTON_MINUS)     // -
   .AddId(MIC_TREBBLE, COLUMN_BUTTON_PLUS)// +
   .AddData(_treble);

   listcreated = true;
   return listBuilder.CreateDataProvider();
}


bool MicSetting::onCourierMessage(const ButtonListItemUpdMsg& oMsg)
{
   ListDataProviderBuilder modifiedListItems(LIST_ID_SYSTEM_MIC_SETTINGS);
   bool msgProcessed = false;
   unsigned int rowIdx = (unsigned int)oMsg.GetHdl(); //Check For the Row Clicked

   if (oMsg.GetListId() == LIST_ID_SYSTEM_MIC_SETTINGS)
   {
      ETG_TRACE_USR2(("MicSetting::ButtonListItemUpdMsg:LIST_ID_SYSTEM_MIC_SETTINGS: Reaction = %d, rowIdx = %d, Column Id = %d", oMsg.GetReaction(), rowIdx, oMsg.GetSubHdl()));
      if (oMsg.GetReaction() == enRelease)
      {
         switch (rowIdx)
         {
            case MIC_BASS:
            {
               if (oMsg.GetSubHdl() != 0)
               {
                  CheckButtonType(oMsg.GetSubHdl(), MIC_BASS) ; // Function to check the Btn Click "+" OR "-"
               }
            }
            break;

            case MIC_TREBBLE:
            {
               if (oMsg.GetSubHdl() != 0)
               {
                  CheckButtonType(oMsg.GetSubHdl(), MIC_TREBBLE) ; // Function to check the Btn Click "+" OR "-"
               }
            }
            break;

            default:
            {
               // Do Nothing
            }
            break;
         }
         msgProcessed = true;
      }
   }
   return msgProcessed;
}


void MicSetting::CheckButtonType(unsigned int RxdColumnBtn, unsigned int Type)
{
//Function to Identify the Column Click (ie BTN + OR BTN -)
   ETG_TRACE_USR1(("MicSetting::CheckButtonType: RxdColumnBtn = %d, Type = %d", RxdColumnBtn, Type));
   if (RxdColumnBtn == COLUMN_BUTTON_MINUS)
   {
      signtype = MINUS_SIGN;
   }
   else if (RxdColumnBtn == COLUMN_BUTTON_PLUS)
   {
      signtype = PLUS_SIGN;
   }

   m_propertytype = m_ListAudSpectoServAudSpec.find(Type)->second; //Find the Sound Property

   ETG_TRACE_USR4(("MicSetting::vAudSpecBassincrementdecrementvalue sinkid signtype m_propertytype=%d nd %d nd %d ", CABIN_A_SINK_ID, signtype, m_propertytype));
   AudioServiceClient::poGetInstance()->performMainSinkSoundPropertyUserChange(CABIN_A_SINK_ID, m_propertytype, signtype);
}


/************************************************************************
*FUNCTION: 		 MicSetting - vPerformMainSinkSoundPropertyBassUpdate
*DESCRIPTION: 	 Update the chnaged values to HMI from the service

*PARAMETER:
*RETURNVALUE: 	 None
************************************************************************/
void MicSetting::vPerformMainSinkSoundPropertyBassUpdate(const uint16& sinkId, const uint16& soundproperty, const uint16& value)
{
   ETG_TRACE_COMP(("MicSetting::vPerformMainSinkSoundPropertyBassUpdate:PropertyType:%d::Value:%d", soundproperty, value));
   ListDataProviderBuilder listBuilder(LIST_ID_SYSTEM_MIC_SETTINGS);
   if (soundproperty == MSP_SOUND_BASS)
   {
      bass_value = value;
      stringstream bout;
      bout << bass_value ;
      string m_bass = bout.str();
      ETG_TRACE_USR1(("MicSetting::vPerformMainSinkSoundPropertyBassUpdate: m_bass = %s", m_bass.c_str()));
      _bass = m_bass.c_str();
   }
   else if (soundproperty == MSP_SOUND_TREBBLE)
   {
      treble_value = value;
      stringstream bout;
      bout << treble_value ;
      string m_treble = bout.str();
      ETG_TRACE_USR1(("MicSetting::vPerformMainSinkSoundPropertyBassUpdate: m_treble = %s", m_treble.c_str()));
      _treble = m_treble.c_str();
   }
   if ((soundproperty == MSP_SOUND_BASS) || (soundproperty == MSP_SOUND_TREBBLE))
   {
      ListRegistry::s_getInstance().updateList(LIST_ID_SYSTEM_MIC_SETTINGS);
   }
}


}
}
