/**************************************************************************************
* @file         : CabinAGlassSettingListhandler.cpp
* @author       : Prashant BG
* @addtogroup   : AppHmi_System
* @brief        :
* @copyright    : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/
#include "gui_std_if.h"
#include <string>
//#include "util/StrUtf8.h"
#include "CabinAGlassSettingListhandler.h"
#include "App/Core/ClientHandler/Audio/AudioServiceClient.h"
#include "App/Core/AppLogic/LanguageSettingListHandler/LanguageSettingListHandler.h"
#include "CGIAppController.h"
//#include "PluginDataType.h"
#include "MediaGlass.h"
#include "AudioDefines.h"
#include <cstdlib>
#include "hmi_trace_if.h"
#include <vector>
#include <iterator>
#include <algorithm>
#include "CgiExtensions/ImageLoader.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_SYSTEM_HALL
#define ETG_I_TRACE_CHANNEL TR_TTFIS_APPHMI_SYSTEM
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_SYSTEM_"
#define ETG_I_FILE_PREFIX                 App::Core::CabinAGlassSettingListhandler::
#include "trcGenProj/Header/CabinAGlassSettingListhandler.cpp.trc.h"
#endif

#define ZERO 0
#define COMMAND_LCD_ON 1
#define COMMAND_LCD_OFF 0

#define COMMAND_SWIVEL_NONE 0x00
#define COMMAND_SWIVEL_DOWN 0x01
#define COMMAND_SWIVEL_UP   0x02
#define COMMAND_SWIVEL_NONE_LCD_ON 0xFF

#define SWIVEL_SELECTED_NONE 0
#define SWIVEL_SELECTED_ONLY_PASSENGER_DOOR 1
#define SWIVEL_SELECTED_ONLY_DRIVER_DOOR 2
#define SWIVEL_SELECTED_ALL_EXCEPT_PASSENGER_DOOR 3
#define SWIVEL_SELECTED_ALL_EXCEPT_DRIVER_DOOR 4
#define SWIVEL_SELECTED_ALL_EXCEPT_DOOR 5
#define SWIVEL_SELECTED_ALL 6

#define AUTO_INP 00
#define CAN_PASSENGER_DOOR_INP 01
#define CAN_DRIVER_DOOR_INP 02
#define MON_INP 03
#define HMI_INP 04
#define PNM_INP 05

#define SWIVEL_DELAY 10
#define SWIVEL_STARTUPDELAY 120


// HMI setting of LCN On/Off
#define HMI_LCD_ON 0
#define HMI_LCD_OFF 1


#define HMI_SWIVEL_AUTO 0
#define HMI_SWIVEL_DOWN 1
#define HMI_SWIVEL_UP 2
#define HMI_SWIVEL_OPTIONS 3

#define INACTIVITY_TIMER_STOPPED 0
#define INACTIVITY_TIMER_RUNNING 1
#define INACTIVITY_TIMER_EXPIRED 2

#define INITIAL_STATE   0
#define USER_OFF		1
#define USER_ON			2


namespace App {
namespace Core {

CabinAGlassSettingListhandler* CabinAGlassSettingListhandler::m_poCabinAGlassSettingListhandler = NULL;


static const char* const CABINA_GAMMA_LCD_COMBOBOX_LIST_ITEM = "Combobox_LCD_CA_ListItem";
static const char* const CABINA_GAMA_MNTR_COMBOBOX_LIST_ITEM = "Combobox_MNTR_CA_ListItem";
static const char* const CABINA_GAMA_SWIVEL_CALIB_LIST_ITEM = "TextButton_ListItem";

Candera::String GlassAsceptRatStrings[3]  = {Candera::String(TextId_0x1591), Candera::String(TextId_0x1592), Candera::String(TextId_0x1593)}; //Wide, Zoom, Panoramic
Candera::String GlassLCDStrings[2]  = {Candera::String(TextId_0x1596), Candera::String(TextId_0x1597)};		//On, Off
Candera::String GlassMonitorFlipStrings[HMI_SWIVEL_OPTIONS]  = {Candera::String(TextId_0x1958), Candera::String(TextId_0x1956), Candera::String(TextId_0x1957)};        // Automatic, On, Off


//const char* LISTITEMTEMPLATE[2] = {"listTplate_BtnBassBar", "listTplate_BtnBassBar"};

//[1]Gamma correction setting, [2]Contrast, [3]Brightness, [4]Aspect ratio mode, [5]LCD panel setting [6]Monitor flip setting
//Candera::String  GamaCASettingText[6] = {Candera::String(TextId_0x158A), Candera::String(TextId_0x158C), Candera::String(TextId_0x158E), Candera::String(TextId_0x1590), Candera::String(TextId_0x1595), Candera::String(TextId_0x1599)};

// "LCD ON AND OFF" Feature based on MCP LONG PRESS IS Removed
Candera::String  GamaCASettingText[1] = {Candera::String(TextId_0x1969)};

#define COLUMN_BUTTON_MINUS 1
#define COLUMN_BUTTON_PLUS  2

std::string IconNormalSwivel[3] = {"AppHmi_System#Images#PopUp_three_line_three_button#TouchArea_Auto_normalBmp",
                                   "AppHmi_System#Images#PopUp_three_line_three_button#TouchArea_Down_normalBmp",
                                   "AppHmi_System#Images#PopUp_three_line_three_button#TouchArea_Up_normalBmp"
                                  };

std::string IconActiveSwivel[3] = {"AppHmi_System#Images#PopUp_three_line_three_button#TouchArea_Auto_touchedBmp",
                                   "AppHmi_System#Images#PopUp_three_line_three_button#TouchArea_Down_touchedBmp",
                                   "AppHmi_System#Images#PopUp_three_line_three_button#TouchArea_Up_touchedBmp"
                                  };

/************************************************************************
*FUNCTION: 		 CabinAGlassSettingListhandler - Constructor
*DESCRIPTION: 	 default constructor of the System setting Listhadler will be created
*PARAMETER:		 None
*RETURNVALUE: 	 None
*
*HISTORY:
*10.09.2019:Prashant BG

*0.1 Initial Revision.
************************************************************************/

CabinAGlassSettingListhandler::CabinAGlassSettingListhandler()
{
   ETG_TRACE_COMP(("CabinAGlassSettingListhandler:: Constructor is called"));
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_CABIN_A_GAMMA_MAIN_LIST, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_CABIN_A_GLASS_MONITOR_FLIP_PANEL_DD_LIST, this);

   _pluginClientHandler = PluginClientHandler::getInstance();
   PopupState = false;
   updatedHMISwivelMode = HMI_SWIVEL_AUTO;
   currentHMISwivelMode = HMI_SWIVEL_AUTO;
   _currentmonitorflip4caOption =   static_cast<enMonitorUpDownGammaCASettingModeOption>(currentHMISwivelMode + 1);

   _currentmonitorflip4caOptionStr = GlassMonitorFlipStrings[currentHMISwivelMode];

   // Door status is common for all zones
   isPassengerDoorOpen = false;
   isDriverDoorOpen = false;

   ignoreSourceUpdate = false; // Any Condition when source can be ignored Such as mic / no source condition

   // Status is separate for each zone
   for (int i = 0; i < SINK_INVALID; i++)
   {
      currentSwivelStatus[i] = COMMAND_SWIVEL_NONE;
      currentLCDStatus[i] = COMMAND_LCD_OFF;
      isContentActiveInGlass[i] = false;
      isSourceActiveInGlass[i] = false;
      isMapoutActiveInGlass[i] = false;
      isInactivityTimerRunning[i] = INACTIVITY_TIMER_STOPPED;
   }

   rowIdx = 0;
   _currentlcdpanel4caOption = static_cast<enLCDOnOffGammaCASettingModeOption>(HMI_LCD_OFF);
   lcmUserState = INITIAL_STATE;
   //startInactivityTimer();
}


/************************************************************************
*FUNCTION: 		 CabinAGlassSettingListhandler - Destructor
*DESCRIPTION: 	 Allocated resources such as memory will be released here
*PARAMETER:		 None
*RETURNVALUE: 	 None
*
*HISTORY:
*10.09.2019:Prashant BG

*0.1 Initial Revision.
************************************************************************/

CabinAGlassSettingListhandler::~CabinAGlassSettingListhandler()
{
   ETG_TRACE_COMP(("CabinAGlassSettingListhandler:: Destructor is called"));
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_CABIN_A_GAMMA_MAIN_LIST);
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_CABIN_A_GLASS_MONITOR_FLIP_PANEL_DD_LIST);

   if (NULL != _pluginClientHandler)
   {
      delete _pluginClientHandler;
      _pluginClientHandler = NULL;
   }
}


/**********************************************************************************************
*FUNCTION: 		 CabinAGlassSettingListhandler - getListDataProvider
*DESCRIPTION: 	 Creates a list data provider which would contain unique list id
*PARAMETER:		 oMsg - Courier message received
*RETURNVALUE: 	 tSharedPtrDataProvider()
*
*HISTORY:
*10.09.2019:Prashant BG

*0.1 Initial Revision.
**********************************************************************************************/

tSharedPtrDataProvider CabinAGlassSettingListhandler::getListDataProvider(const ListDateProviderReqMsg& oMsg)
{
   ETG_TRACE_COMP(("CabinAGlassSettingListhandler:: getListDataProvider: List Id = %d", oMsg.GetListId()));
   switch (oMsg.GetListId())
   {
      case LIST_ID_CABIN_A_GAMMA_MAIN_LIST:
         return getMainViewListDataProvider();
      case LIST_ID_CABIN_A_GLASS_MONITOR_FLIP_PANEL_DD_LIST:
         return GetComboGammaMonitorFlipListDataProvider(oMsg.GetListId());
   }
   return tSharedPtrDataProvider();
}


tSharedPtrDataProvider CabinAGlassSettingListhandler::getMainViewListDataProvider()
{
   ListDataProviderBuilder listBuilder(LIST_ID_CABIN_A_GAMMA_MAIN_LIST);
   int updatedval;
   listBuilder.AddExpandableItem(CA_MONITORFLIPCNTRL,            //  enum name available in systemtypes.h - identifies the row
                                 LIST_ID_CABIN_A_GLASS_MONITOR_FLIP_PANEL_DD_LIST,                          //  Main list name  - used to automatically toggle the visibility of the combobox dropdown list when the combox button is clicked
                                 CABINA_GAMA_MNTR_COMBOBOX_LIST_ITEM)         //SceneComposer Dock Layout name   - overwrites the default data context
   .AddData(GamaCASettingText[0])  // Text of the combobox button.
   .AddData(LIST_ID_CABIN_A_GLASS_MONITOR_FLIP_PANEL_DD_LIST)				 // Main list name - ListId of the combobox dropdown list.
   .AddData(_currentmonitorflip4caOptionStr);	//3 - Text of the label displayed in front of the combobox button, should be obtained
   return listBuilder.CreateDataProvider();
}


/**********************************************************************************************
*FUNCTION:          sendLCDOnOffCommandToPlugin
*DESCRIPTION:       Sends requested LCD on/off command to LCD
*PARAMETER:
*RETURNVALUE:       None
*
*HISTORY:
*05.05.2020:Prashant BG

*0.1 Initial Revision.
**********************************************************************************************/

// "LCD ON AND OFF" Feature based on MCP LONG PRESS and HMI IS Removed
/*
void CabinAGlassSettingListhandler::sendLCDOnOffCommandToPlugin(uint8 sinkID, uint8 lcdStatus)
{
   ETG_TRACE_USR4(("CabinAGlassSettingListhandler::sendLCDOnOffCommandToPlugin(), sink ID : %d, lcdStatus : %d", sinkID, lcdStatus));
   boost::shared_ptr<EventDataUtility> pluginData(EventDataUtility::newEventDataUtility());
   if (NULL != pluginData.get())
   {
      pluginData->addEventData(sinkID);
      pluginData->addEventData(lcdStatus);//value
      POST_MSG((COURIER_MESSAGE_NEW(PluginCommandReqMsg)(PLUGIN_NAME_SETTINGS, CTRLBLOCK_NAME_SETTINGS_GLASSLCD, pluginData)));
   }
}


*/


/**********************************************************************************************
*FUNCTION:          isSinkSupported
*DESCRIPTION:       Checks if sinkid is supported
*PARAMETER:         sinkid - sink id
*RETURNVALUE:       Returns true if sink is supported, else false
*
*HISTORY:
*05.05.2020:Prashant BG

*0.1 Initial Revision.
**********************************************************************************************/
bool CabinAGlassSettingListhandler::isSinkSupported(int sinkid)
{
   bool isSinkSupported = false;

   if (
      (sinkid == CABIN_A_MEDIAGLASS1_SINK_ID)
      || (sinkid == CABIN_A_MEDIAGLASS2_SINK_ID)
      // || ( sinkid == CABIN_B_MEDIAGLASS1_SINK_ID )
      // || ( sinkid == CABIN_B_MEDIAGLASS2_SINK_ID )
   )
   {
      isSinkSupported = true;
   }

   return isSinkSupported;
}


/**********************************************************************************************
*FUNCTION:          sendContentStatusRequestToPlugin
*DESCRIPTION:       Processes the cabin source information and evalutes swivel operation
*PARAMETER:         status - video source active or inactive
*RETURNVALUE:       None
*
*HISTORY:
*05.05.2020:Prashant BG

*0.1 Initial Revision.
**********************************************************************************************/
void CabinAGlassSettingListhandler::sendContentStatusRequestToPlugin(int status, int sinkid, bool isSourceIgnored)
{
   ETG_TRACE_COMP(("CabinAGlassSettingListhandler ::sendContentStatusRequestToPlugin: status = %d, sinkid = %d, isSourceIgnored = %d", status, sinkid, isSourceIgnored));
   bool contentActive = false;

   // Check only current cabin sources-------> check based on sink id
   // To do - Add logic to check if video source is active
   // Evaluate content state and send swivel command
   if (isSinkSupported(sinkid))
   {
      // Status changed for Media Glass sink
      isSourceActiveInGlass[sinkid] = status;
      ignoreSourceUpdate = isSourceIgnored; // Any Condition when source can be ignored Such as mic / no source condition

      contentActive = isSourceActiveInGlass[sinkid] | isMapoutActiveInGlass[sinkid];

      ETG_TRACE_USR1(("CabinAGlassSettingListhandler ::sendContentStatusRequestToPlugin contentActive=%d isSourceActiveInGlass[%d]= %d, isMapoutActiveInGlass[%d]= %d,isSourceIgnored=%d ", contentActive, sinkid, isSourceActiveInGlass[sinkid], sinkid, isMapoutActiveInGlass[sinkid], isSourceIgnored));
      if (isSourceIgnored == false)
      {
         CabinAGlassSettingListhandler::poGetInstance()->evaluateAndSendSwivelCommand(sinkid, contentActive, AUTO_INP);
      }
      //Below is commented for
      //Bug 1642528: Swivel does not move to close position after audio source selected
      //Bug 1629406: Swivel does not close during radio source playback (After PNM is turned ON)
      /*      else // Save this Glass Active for the next user Interaction  Bug 1331299
           {
              isContentActiveInGlass[sinkid] = contentActive;
           } */
   }
}


void CabinAGlassSettingListhandler::sendMapoutStatusRequestToPlugin(int status, int sinkid)
{
   ETG_TRACE_COMP(("CabinAGlassSettingListhandler ::sendMapoutStatusRequestToPlugin: sinkid = %d, status = %d", sinkid, status));
   bool contentActive = false;

   if (isSinkSupported(sinkid))
   {
      // Status changed for Media Glass sink
      isMapoutActiveInGlass[sinkid] = status;

      contentActive = isSourceActiveInGlass[sinkid] | isMapoutActiveInGlass[sinkid];
      ETG_TRACE_USR1(("CabinAGlassSettingListhandler ::sendMapoutStatusRequestToPlugin contentActive =%d isSourceActiveInGlass[%d]= %d, isMapoutActiveInGlass[%d]= %d ", contentActive, sinkid, isSourceActiveInGlass[sinkid], sinkid, isMapoutActiveInGlass[sinkid]));

      CabinAGlassSettingListhandler::poGetInstance()->evaluateAndSendSwivelCommand(sinkid, contentActive, AUTO_INP);
   }
}


/**********************************************************************************************
*FUNCTION:          startInactivityTimer
*DESCRIPTION:       Starts inactivity timer
*PARAMETER:         None
*RETURNVALUE:       None
*
*HISTORY:
*05.05.2020:Prashant BG

*0.1 Initial Revision.
**********************************************************************************************/
void CabinAGlassSettingListhandler::startInactivityTimer(int sinkid)
{
   if (isSinkSupported(sinkid))
   {
      ETG_TRACE_COMP(("CabinAGlassSettingListhandler ::startInactivityTimer sinkid = %d, isInactivityTimerRunning = %d ", sinkid, isInactivityTimerRunning[sinkid]));
      if (isInactivityTimerRunning[sinkid] == INACTIVITY_TIMER_STOPPED)		//default value of isInactivityTimerRunning is false
      {
         _swiveltimer[sinkid].start(*this, 1000, SWIVEL_DELAY);
         isInactivityTimerRunning[sinkid] = INACTIVITY_TIMER_RUNNING;
         ETG_TRACE_USR1(("CabinAGlassSettingListhandler ::startInactivityTimer: isInactivityTimerRunning=INACTIVITY_TIMER_RUNNING"));
      }
   }
}


/**********************************************************************************************
*FUNCTION:          startInactivityTimerONStartup
*DESCRIPTION:       Starts inactivity timer
*PARAMETER:         None
*RETURNVALUE:       None
*
*HISTORY:
*05.05.2020:Prashant BG

*0.1 Initial Revision.
**********************************************************************************************/
void CabinAGlassSettingListhandler::startInactivityTimerONStartup(int sinkid)
{
   if (isSinkSupported(sinkid))
   {
      ETG_TRACE_COMP(("CabinAGlassSettingListhandler ::startInactivityTimerONStartup sinkid = %d, isInactivityTimerRunning = %d ", sinkid, isInactivityTimerRunning[sinkid]));
      if (isInactivityTimerRunning[sinkid] == INACTIVITY_TIMER_STOPPED)		//default value of isInactivityTimerRunning is false
      {
         _swiveltimer[sinkid].start(*this, 1000, SWIVEL_STARTUPDELAY);
         isInactivityTimerRunning[sinkid] = INACTIVITY_TIMER_RUNNING;
         ETG_TRACE_USR1(("CabinAGlassSettingListhandler ::startInactivityTimerONStartup: isInactivityTimerRunning=INACTIVITY_TIMER_RUNNING"));
      }
   }
}


/**********************************************************************************************
*FUNCTION:          stopInactivityTimer
*DESCRIPTION:       Stops inactivity timer
*PARAMETER:         None
*RETURNVALUE:       None
*
*HISTORY:
*05.05.2020:Prashant BG

*0.1 Initial Revision.
**********************************************************************************************/
void CabinAGlassSettingListhandler::stopInactivityTimer(int sinkid)
{
   if (isSinkSupported(sinkid))
   {
      ETG_TRACE_COMP(("CabinAGlassSettingListhandler ::stopInactivityTimer sinkid = %d, isInactivityTimerRunning=%d ", sinkid, isInactivityTimerRunning[sinkid]));
      if (isInactivityTimerRunning[sinkid] == INACTIVITY_TIMER_RUNNING)
      {
         _swiveltimer[sinkid].stop();
         ETG_TRACE_USR1(("CabinAGlassSettingListhandler ::stopInactivityTimer: isInactivityTimerRunning=INACTIVITY_TIMER_STOPPED"));
      }
      isInactivityTimerRunning[sinkid] = INACTIVITY_TIMER_STOPPED;
   }
}


/**********************************************************************************************
*FUNCTION:          sendPassengerDoorStatusRequestToPlugin
*DESCRIPTION:       Processes the passenger door status information and evalutes swivel operation
*PARAMETER:         Door status received over CAN
*RETURNVALUE:       None
*
*HISTORY:
*05.05.2020:Prashant BG

*0.1 Initial Revision.
**********************************************************************************************/
void CabinAGlassSettingListhandler::sendPassengerDoorStatusRequestToPlugin(uint8 doorStatus)
{
   ETG_TRACE_COMP(("CabinAGlassSettingListhandler ::sendPassengerDoorStatusRequestToPlugin: doorStatus = %d", doorStatus));

   // Evaluate door status and send swivel command for all zones
   CabinAGlassSettingListhandler::poGetInstance()->evaluateAndSendSwivelCommandToAllZones(doorStatus, CAN_PASSENGER_DOOR_INP);
}


/**********************************************************************************************
*FUNCTION:          sendDriverDoorStatusRequestToPlugin
*DESCRIPTION:       Processes the driver door status information and evalutes swivel operation
*PARAMETER:         Door status received over CAN
*RETURNVALUE:       None
*
*HISTORY:
*05.05.2020:Prashant BG

*0.1 Initial Revision.
**********************************************************************************************/
void CabinAGlassSettingListhandler::sendDriverDoorStatusRequestToPlugin(uint8 doorStatus)
{
   ETG_TRACE_COMP(("CabinAGlassSettingListhandler ::sendDriverDoorStatusRequestToPlugin: doorStatus = %d", doorStatus));

   // Evaluate door status and send swivel command for all zones
   CabinAGlassSettingListhandler::poGetInstance()->evaluateAndSendSwivelCommandToAllZones(doorStatus, CAN_DRIVER_DOOR_INP);
}


void CabinAGlassSettingListhandler::CheckPopupActive(int hardkeyState)
{
   ETG_TRACE_COMP(("CabinAGlassSettingListhandler::CheckPopupActive: HKStatus = %d", hardkeyState));
   if (PopupState == false)
   {
      ETG_TRACE_USR1(("CabinAGlassSettingListhandler::CheckPopupActive: inside PopupState = false"));
      if (hardkeyState == HARDKEYUP)
      {
         POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_System#SoftwareDownload#PopUp_three_line_three_button"))));
         updatedHMISwivelMode = currentHMISwivelMode;
         ETG_TRACE_USR2(("CabinAGlassSettingListhandler::CheckPopupActive: currentHMISwivelMode = %d", currentHMISwivelMode));
         if (currentHMISwivelMode == HMI_SWIVEL_AUTO)
         {
            (*oSwivelPopup).mActiveSwivelStateAuto = ImageLoader::getAssetBitmapImage(IconActiveSwivel[0].c_str());
            (*oSwivelPopup).mActiveSwivelStateDown = ImageLoader::getAssetBitmapImage(IconNormalSwivel[1].c_str());
            (*oSwivelPopup).mActiveSwivelStateUp = ImageLoader::getAssetBitmapImage(IconNormalSwivel[2].c_str());
         }
         else if (currentHMISwivelMode == HMI_SWIVEL_UP)
         {
            (*oSwivelPopup).mActiveSwivelStateAuto = ImageLoader::getAssetBitmapImage(IconNormalSwivel[0].c_str());
            (*oSwivelPopup).mActiveSwivelStateDown = ImageLoader::getAssetBitmapImage(IconNormalSwivel[1].c_str());
            (*oSwivelPopup).mActiveSwivelStateUp = ImageLoader::getAssetBitmapImage(IconActiveSwivel[2].c_str());
         }
         else if (currentHMISwivelMode == HMI_SWIVEL_DOWN)
         {
            (*oSwivelPopup).mActiveSwivelStateAuto = ImageLoader::getAssetBitmapImage(IconNormalSwivel[0].c_str());
            (*oSwivelPopup).mActiveSwivelStateDown = ImageLoader::getAssetBitmapImage(IconActiveSwivel[1].c_str());
            (*oSwivelPopup).mActiveSwivelStateUp = ImageLoader::getAssetBitmapImage(IconNormalSwivel[2].c_str());
         }

         else
         {
            /* do nothing */
         }

         oSwivelPopup.MarkAllItemsModified();
         oSwivelPopup.SendUpdate(true);
      }
   }

   else if (PopupState == true)
   {
      ETG_TRACE_USR1(("CabinAGlassSettingListhandler::CheckPopupActive: PopupState = true"));
      if (hardkeyState == HARDKEYLONGPRESS)
      {
         POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Hide, Courier::ViewId("AppHmi_System#SoftwareDownload#PopUp_three_line_three_button"))));
         sendMcpStatusRequestToPlugin(hardkeyState); // this will select the option highlighted
      }

      else if (hardkeyState == HARDKEYUP)
      {
         sendMcpStatusRequestToPlugin(hardkeyState); // this will go to next option of the popup and be there
      }

      else
      {
         ETG_TRACE_USR2(("CabinAGlassSettingListhandler::CheckPopupActive: inside PopupState = true , else case"));/* do nothing */
      }
   }

   else
   {
      ETG_TRACE_USR1(("CabinAGlassSettingListhandler::CheckPopupActive: inside PopupState  else case"));/* do nothing */
   }
}


/**********************************************************************************************
*FUNCTION:          sendMcpStatusRequestToPlugin
*DESCRIPTION:       Processes the MCP MON button press
*PARAMETER:         MCP MON button hard key state
*RETURNVALUE:       None
*
*HISTORY:
*05.05.2020:Prashant BG

*0.1 Initial Revision.
**********************************************************************************************/
void CabinAGlassSettingListhandler::sendMcpStatusRequestToPlugin(int hardkeyState)
{
   ETG_TRACE_COMP(("CabinAGlassSettingListhandler ::sendMcpStatusRequestToPlugin:hardkeyState = %d", hardkeyState));
   if (HARDKEYUP == hardkeyState)
   {
      // Change HMI swivel status
      if (HMI_SWIVEL_AUTO == updatedHMISwivelMode)
      {
         // AUTO --> DOWN
         ETG_TRACE_USR2(("CabinAGlassSettingListhandler: sendMcpStatusRequestToPlugin: HMI_SWIVEL_DOWN highlighted"));
         updatedHMISwivelMode = HMI_SWIVEL_DOWN;
         (*oSwivelPopup).mActiveSwivelStateAuto = ImageLoader::getAssetBitmapImage(IconNormalSwivel[0].c_str());
         (*oSwivelPopup).mActiveSwivelStateDown = ImageLoader::getAssetBitmapImage(IconActiveSwivel[1].c_str());
         (*oSwivelPopup).mActiveSwivelStateUp = ImageLoader::getAssetBitmapImage(IconNormalSwivel[2].c_str());
      }
      else if (HMI_SWIVEL_DOWN == updatedHMISwivelMode)
      {
         // DOWN --> UP
         ETG_TRACE_USR2(("CabinAGlassSettingListhandler: sendMcpStatusRequestToPlugin: HMI_SWIVEL_UP highlighted"));
         updatedHMISwivelMode = HMI_SWIVEL_UP;
         (*oSwivelPopup).mActiveSwivelStateAuto = ImageLoader::getAssetBitmapImage(IconNormalSwivel[0].c_str());
         (*oSwivelPopup).mActiveSwivelStateDown = ImageLoader::getAssetBitmapImage(IconNormalSwivel[1].c_str());
         (*oSwivelPopup).mActiveSwivelStateUp = ImageLoader::getAssetBitmapImage(IconActiveSwivel[2].c_str());
      }
      else if (HMI_SWIVEL_UP == updatedHMISwivelMode)
      {
         // UP --> AUTO
         ETG_TRACE_USR2(("CabinAGlassSettingListhandler: sendMcpStatusRequestToPlugin: HMI_SWIVEL_AUTO highlighted"));
         updatedHMISwivelMode = HMI_SWIVEL_AUTO;
         (*oSwivelPopup).mActiveSwivelStateAuto = ImageLoader::getAssetBitmapImage(IconActiveSwivel[0].c_str());
         (*oSwivelPopup).mActiveSwivelStateDown = ImageLoader::getAssetBitmapImage(IconNormalSwivel[1].c_str());
         (*oSwivelPopup).mActiveSwivelStateUp = ImageLoader::getAssetBitmapImage(IconNormalSwivel[2].c_str());
      }

      oSwivelPopup.MarkAllItemsModified();
      oSwivelPopup.SendUpdate(true);
   }

   else if (HARDKEYLONGPRESS == hardkeyState)
   {
      // Subtract 1 to match the selection and enumeration index,
      // handle for first option in dropdown is 1. However, the enumeration value for same is 0
      _currentmonitorflip4caOption = static_cast<enMonitorUpDownGammaCASettingModeOption>(updatedHMISwivelMode + 1);
      _currentmonitorflip4caOptionStr = (GlassMonitorFlipStrings[updatedHMISwivelMode]);
      currentHMISwivelMode = updatedHMISwivelMode;
      ETG_TRACE_USR2(("CabinAGlassSettingListhandler ::sendMcpStatusRequestToPlugin: evaluateAndSendSwivelCommand _currentmonitorflip4caOption=%d, _currentmonitorflip4caOptionStr = %s", _currentmonitorflip4caOption, _currentmonitorflip4caOptionStr.GetCString()));

      // Evaluate HMI swivel status and send swivel command for all zones
      CabinAGlassSettingListhandler::poGetInstance()->evaluateAndSendSwivelCommandToAllZones(static_cast<uint8>(updatedHMISwivelMode), MON_INP);
   }

   else
   {
      ETG_TRACE_USR1(("CabinAGlassSettingListhandler ::sendMcpStatusRequestToPlugin: inside else case "));
      //do nothing
   }
}


bool CabinAGlassSettingListhandler::onCourierMessage(const ChangeMonitorUpDown4GammaCASettingModevalReqMsg& oMsg)
{
   uint8 GlassMonitorFlipCntrl4CAOpt = oMsg.GetMonitorUpDownGammaCASettingModeOptions();
   ETG_TRACE_USR1(("CabinAGlassSettingListhandler::ChangeMonitorUpDown4GammaCASettingModevalReqMsg: GlassMonitorFlipCntrl4CAOpt=%d", GlassMonitorFlipCntrl4CAOpt));
   if (GlassMonitorFlipCntrl4CAOpt <= HMI_SWIVEL_OPTIONS)
   {
      // Subtract 1 to match the selection and enumeration index,
      // handle for first option in dropdown is 1. However, the enumeration value for same is 0
      // Evaluate and send swivel command
      currentHMISwivelMode = static_cast<uint8>(GlassMonitorFlipCntrl4CAOpt - 1);
      _currentmonitorflip4caOption = static_cast<enMonitorUpDownGammaCASettingModeOption>(currentHMISwivelMode + 1);
      _currentmonitorflip4caOptionStr = (GlassMonitorFlipStrings[currentHMISwivelMode]);
      ETG_TRACE_USR2(("CabinAGlassSettingListhandler::ChangeMonitorUpDown4GammaCASettingModevalReqMsg: _currentmonitorflip4caOption=%d, _currentmonitorflip4caOptionStr =%s", _currentmonitorflip4caOption, _currentmonitorflip4caOptionStr.GetCString()));

      // Evaluate HMI swivel status and send swivel command for all zones
      CabinAGlassSettingListhandler::poGetInstance()->evaluateAndSendSwivelCommandToAllZones(currentHMISwivelMode, HMI_INP);

      //add the intereface here
      ListRegistry::s_getInstance().updateList(LIST_ID_CABIN_A_GLASS_MONITOR_FLIP_PANEL_DD_LIST);
      ListRegistry::s_getInstance().updateList(LIST_ID_CABIN_A_GAMMA_MAIN_LIST);
// Add the Inter face here
   }

   return true;
}


bool CabinAGlassSettingListhandler::onCourierMessage(const CockpitModeMsg& oMsg)
{
   bool ret = false;
   bool CockpitModeStatus = oMsg.GetCockpitModeStatus();
   ETG_TRACE_USR4(("CabinAGlassSettingListhandler::onCourierMessage(CockpitModeMsg)=%d", CockpitModeStatus));

   if (CockpitModeStatus == Battery_ON)
   {
      POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Hide, Courier::ViewId("AppHmi_System#SoftwareDownload#PopUp_three_line_two_button_Swivel_GS_CA"))));
      ETG_TRACE_USR4(("CabinAGlassSettingListhandler::onCourierMessage(CockpitModeMsg) PNM_INP=%d  ", PNM_INP));

      // Evaluate PNM input and send swivel command for all zones
      CabinAGlassSettingListhandler::poGetInstance()->evaluateAndSendSwivelCommandToAllZones(ZERO, PNM_INP);

      ret = true;
   }
   else if (CockpitModeStatus == Battery_OFF)
   {
      POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Hide, Courier::ViewId("AppHmi_System#SoftwareDownload#PopUp_three_line_two_button_Swivel_GS_CA"))));
   }

   return ret;
}


void CabinAGlassSettingListhandler::UpdateCockpitModeStatus(const uint8 HMIState)
{
   ETG_TRACE_COMP(("CabinAGlassSettingListhandler::UpdateCockpitModeStatus HMIState currentHMISwivelMode= %d %d ", HMIState, currentHMISwivelMode));

   if (currentHMISwivelMode == HMI_SWIVEL_AUTO)
   {
      // Fix for the Bug 1175875: Driver Confirmation popup to close the Monitors is shown on even when Glass settings are set
      if (
         (currentSwivelStatus[CABIN_A_MEDIAGLASS1_SINK_ID] == COMMAND_SWIVEL_DOWN)
         || (currentSwivelStatus[CABIN_A_MEDIAGLASS2_SINK_ID] == COMMAND_SWIVEL_DOWN)
         // || ( currentSwivelStatus[CABIN_B_MEDIAGLASS1_SINK_ID] == COMMAND_SWIVEL_DOWN )
         // ( currentSwivelStatus[CABIN_B_MEDIAGLASS2_SINK_ID] == COMMAND_SWIVEL_DOWN )
      )
      {
         /*
          * Fix for Bug 1220896
          * Timer could have been already started when
          * PNM signal is received. Hence, stop the timer
          * before showing pop-up
          */
         // Stop inactivity timer
         // stopInactivityTimer(CABIN_A_MEDIAGLASS1_SINK_ID);
         //stopInactivityTimer(CABIN_A_MEDIAGLASS2_SINK_ID);
         //	stopInactivityTimer(CABIN_B_MEDIAGLASS1_SINK_ID);
         //	stopInactivityTimer(CABIN_B_MEDIAGLASS2_SINK_ID);
         ETG_TRACE_COMP(("CabinAGlassSettingListhandler::UpdateCockpitModeStatus AppHmi_System#SoftwareDownload#PopUp_three_line_two_button_Swivel_GS_CA"));
         POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_System#SoftwareDownload#PopUp_three_line_two_button_Swivel_GS_CA"))));
      }
      //Bug 1175875: Driver Confirmation popup to close the Monitors is shown on even when Glass settings are set
      else
      {
         ETG_TRACE_COMP(("CabinAGlassSettingListhandler::UpdateCockpitModeStatus ifelseAppHmi_System#SoftwareDownload#PopUp_three_line_Swivel_GS_CA"));
         POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_System#SoftwareDownload#PopUp_three_line_Swivel_GS_CA"))));
      }
   }
   else
   {
      ETG_TRACE_COMP(("CabinAGlassSettingListhandler::UpdateCockpitModeStatus AppHmi_System#SoftwareDownload#PopUp_three_line_Swivel_GS_CA"));
      POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_System#SoftwareDownload#PopUp_three_line_Swivel_GS_CA"))));
   }
}


bool CabinAGlassSettingListhandler::onCourierMessage(const SwivelModeMsg& oMsg)
{
   uint8 SelectedSwivelMode = oMsg.GetSelectedModeItem();
   ETG_TRACE_USR1(("CabinAGlassSettingListhandler::SwivelModeMsg: SelectedSwivelMode=%d", SelectedSwivelMode));
   if (SelectedSwivelMode <= HMI_SWIVEL_OPTIONS)
   {
      currentHMISwivelMode = static_cast<uint8>(SelectedSwivelMode - 1);

      _currentmonitorflip4caOption = static_cast<enMonitorUpDownGammaCASettingModeOption>(SelectedSwivelMode);
      _currentmonitorflip4caOptionStr = (GlassMonitorFlipStrings[SelectedSwivelMode - 1U]);
      ETG_TRACE_USR2(("CabinAGlassSettingListhandler::SwivelModeMsg: _currentmonitorflip4caOption=%d, _currentmonitorflip4caOptionStr =%s", _currentmonitorflip4caOption, _currentmonitorflip4caOptionStr.GetCString()));

      // Evaluate HMI swivel status and send swivel command
      CabinAGlassSettingListhandler::poGetInstance()->evaluateAndSendSwivelCommandToAllZones(currentHMISwivelMode, HMI_INP);

      ListRegistry::s_getInstance().updateList(LIST_ID_CABIN_A_GLASS_MONITOR_FLIP_PANEL_DD_LIST);
      ListRegistry::s_getInstance().updateList(LIST_ID_CABIN_A_GAMMA_MAIN_LIST);
   }
   return true;
}


bool CabinAGlassSettingListhandler::onCourierMessage(const PopupStatusMsg& oMsg)
{
   ETG_TRACE_COMP(("CabinAGlassSettingListhandler::PopupStatusMsg: List Id = %u, PopupState = %d", oMsg.GetActiveViewId(), oMsg.GetPopupStatus()));
   bool ret = false;
   uint32 ListId = oMsg.GetActiveViewId();
   if (SWIVELPOPUPID == ListId)
   {
      PopupState = oMsg.GetPopupStatus();
      ret = true;
   }
   return ret;
}


/**********************************************************************************************
*FUNCTION:          evaluateAndSendSwivelCommandToAllZones
*DESCRIPTION:       Processes the status for all supported zones and sends appropriate command to swivel
*PARAMETER:
*RETURNVALUE:       None

** status					= 	true / false
** key 						= 	CAN /AUTO /HMI/ MON
*HISTORY:
*05.05.2020:Prashant BG

*0.1 Initial Revision.
**********************************************************************************************/
void CabinAGlassSettingListhandler::evaluateAndSendSwivelCommandToAllZones(uint8 status, uint8 key)
{
   evaluateAndSendSwivelCommand(CABIN_A_MEDIAGLASS1_SINK_ID, status, key);
   evaluateAndSendSwivelCommand(CABIN_A_MEDIAGLASS2_SINK_ID, status, key);
   // evaluateAndSendSwivelCommand(CABIN_B_MEDIAGLASS1_SINK_ID, status, key);
   // evaluateAndSendSwivelCommand(CABIN_B_MEDIAGLASS2_SINK_ID, status, key);
}


/**********************************************************************************************
*FUNCTION:          evaluateAndSendSwivelCommand
*DESCRIPTION:       Processes the status and sends appropriate command to swivel
*PARAMETER:
*RETURNVALUE:       None

** sink 					= 	cabin sink ID
** status					= 	true / false
** key 						= 	CAN /AUTO /HMI/ MON
** true						=	con/con-ing/susp
** false   					=	dis-con	/dis-con-ing
** currentHMISwivelMode	=	basically a variable value assigned during the drop down (and May be MON Options too)
*HISTORY:
*05.05.2020:Prashant BG

*0.1 Initial Revision.
**********************************************************************************************/
void CabinAGlassSettingListhandler::evaluateAndSendSwivelCommand(int sinkid, uint8 status, uint8 key)
{
   if (isSinkSupported(sinkid))
   {
      int swivelRequest = currentSwivelStatus[sinkid];
      int lcdOnOffRequest = currentLCDStatus[sinkid];
      int swivelSelected = SWIVEL_SELECTED_NONE;
      bool sendSwivelCommand = true;
      bool currenthmistate = HmiStateHandler::poGetInstance()->isPartialUserModeOn();
      bool isUserStandbyOffState = HmiStateHandler::poGetInstance()->isUserStandbyOff();
      ETG_TRACE_USR1(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: sinkid = %d, status = %d, key = %d currenthmistate=%d", sinkid, status, key, currenthmistate));
      ETG_TRACE_USR1(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: isUserStandbyOffState=%d currentSwivelStatus[%d] = %d, currentLCDStatus[%d] = %d", isUserStandbyOffState,  sinkid, currentSwivelStatus[sinkid], sinkid, currentLCDStatus[sinkid]));
      switch (key)
      {
         case AUTO_INP:
         {
            //MON Mode= Auto
            if (HMI_SWIVEL_AUTO == currentHMISwivelMode)
            {
               if (status)// Video or Logo is active
               {
                  if ((isContentActiveInGlass[sinkid] == false) || (currentSwivelStatus[sinkid] != COMMAND_SWIVEL_DOWN))
                  {
                     // Content is active --Connected / Connecting  -- Swivel is down LCD IS ON (Non Door Swivel)ie 6f, 6g 8f , 8g
                     ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: content is active - swivel DOWN"));
                     isContentActiveInGlass[sinkid] = true;
                     swivelRequest = COMMAND_SWIVEL_DOWN;
                  }
                  else
                  {
                     // Do nothing
                     ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: content already active, swivel in correct state"));
                     sendSwivelCommand = false;
                  }

                  // Stop inactivity timer
                  stopInactivityTimer(sinkid);
               }
               else// Content is InActive
               {
                  // Check if previously content was active
                  if (isContentActiveInGlass[sinkid])		//enters when glass is disconnecting/diconnected and previous currentglassactive is true
                  {
                     // Content active --> inactive
                     ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: content inactive after being active"));
                     isContentActiveInGlass[sinkid] = false;
                     // Start inactivity timer
                     startInactivityTimer(sinkid);		//start the timer and  make the (isInactivityTimerRunning=true)
                     sendSwivelCommand = false;
                  }
                  //On Receiving the PNM OFF in hmi state  during source is already active --> timer should off and popup should be poped asking for swivel closure - Bug 1261221 and Bug 1220896
                  else if (true == currenthmistate)//||(true==ignoreSourceUpdate)) // Any Condition when source can be ignored Such as mic / no source condition
                  {
                     ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: PNM is off or Ignore source Update currenthmistate =%d ", currenthmistate));
                     //During PNM is off, stop inactivity timer, do nothing
                     stopInactivityTimer(sinkid);
                     sendSwivelCommand = false;
                  }
                  else if (isInactivityTimerRunning[sinkid] == INACTIVITY_TIMER_EXPIRED)
                  {
                     //Bug 2008405: CMG Turn on after MIC 1 activated during User off Mode
                     // Inactivity timer expired, Mic 1 is not active, swivel up
                     ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: inactivity timer expired - swivel UP"));
                     swivelRequest = COMMAND_SWIVEL_UP;
                     isInactivityTimerRunning[sinkid] = INACTIVITY_TIMER_STOPPED;
                  }
                  //Bug 2008405: CMG Turn on after MIC 1 activated during User off Mode
                  else if (isInactivityTimerRunning[sinkid] == INACTIVITY_TIMER_STOPPED)
                  {
                     // Content active --> inactive
                     ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: content inactive during source is changed audio to audiio"));
                     isContentActiveInGlass[sinkid] = false;
                     // Start inactivity timer
                     startInactivityTimer(sinkid);		//start the timer and  make the (isInactivityTimerRunning=true)
                     sendSwivelCommand = false;
                  }
                  else
                  {
                     // Inactivity timer running, do nothing
                     //isContentActiveInGlass is false and // isInactivityTimerRunning== true and //sendSwivelCommand = false;
                     ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: inactivity timer running or not started"));
                     sendSwivelCommand = false;
                  }
               }
            }
            //Bug 1397359: CM-G Backlight is active (screen is black) in case when no video source is connected in the cabin
            //As per Initial and current Requirement Bug 1397359 --> In ON MODE ==> Irrespective video or audio(logo should be ON) is played --> swivel Remains always ON AND LCD should be always ON
            //In Between Requirement Requesst(Is Invalid Now)    --> In ON Mode ==> SWIVEL AND LCD is ON based on video avaialblity and SWIVEL AND LCD is OFF on Video UnAVAIALBLITY AND AUDIO source Availablity
            //In ON Mode, is always Remains Swivel Down and LCD Remains ON
            else if (HMI_SWIVEL_DOWN == currentHMISwivelMode)
            {
               // Save the content status - needed when mode is changed to AUTO
               isContentActiveInGlass[sinkid] = status;
               //Bug 1480637: When monitor is set to ON & USER OFF is performed, glasses are not swiveled up
               if ((isUserStandbyOffState == true) && (lcmUserState == USER_ON)) //In Mode ON, User OFF - Swivel UP and LCD Remains OFF
               {
                  ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: User ON TO USER OFF , SWIVEL UP AND LCD OFF "));
                  swivelRequest = COMMAND_SWIVEL_UP;
               }
               //In Mode ON, USER ON- Swivel Down and LCD Remains ON
               //IN Mode ON, for audio/no sourse --> always logo is shown and for video/camera/map--> Swivel is down with respective streaming
               else if ((isUserStandbyOffState == false) && (lcmUserState == USER_OFF))
               {
                  ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: User OFF TO USER ON , SWIVEL DOWN AND LCD ON "));
                  swivelRequest = COMMAND_SWIVEL_DOWN;
               }
            }

            //Bug 1194103: Video is displayed on the glasses even when the glass settings are set to OFF after doing USER OFF/ON
            //In OFF Mode , during UserOff/User On is always Remains Swivel UP and LCD Remains OFF
            else if (HMI_SWIVEL_UP == currentHMISwivelMode)
            {
               // Save the content status - needed when mode is changed to AUTO
               isContentActiveInGlass[sinkid] = status;
               if (isInactivityTimerRunning[sinkid] == INACTIVITY_TIMER_EXPIRED) //In Mode OFF, After StartupTimer is Expired is always Remains Swivel Down and LCD Remains ON
               {
                  ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: inactivity timer expired in OFF mode- swivel UP"));
                  swivelRequest = COMMAND_SWIVEL_UP;
                  isInactivityTimerRunning[sinkid] = INACTIVITY_TIMER_STOPPED;
               }
            }
            else
            {
               // AUTO_INP to be considered only when AUTO mode is set
               ETG_TRACE_USR3(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: content status only considered in AUTO and ON mode"));
               sendSwivelCommand = false;

               // Save the content status - needed when mode is changed to AUTO
               isContentActiveInGlass[sinkid] = status;
            }
            break;
         }

         case CAN_PASSENGER_DOOR_INP:
         {
            // status = door open status
            if (status == 1)
            {
               // door is opened
               ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: passenger door open - swivel UP"));
               isPassengerDoorOpen = true;
               swivelRequest = COMMAND_SWIVEL_UP;
            }
            else if (status == 0) //Bug 1177225: Door Swivel unit remains closed and Glass remains off after the opening and closing the Door via CAN
            {
               // door is closed, follow all other swivels
               ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: passenger door closed"));
               isPassengerDoorOpen = false;
               swivelRequest = currentSwivelStatus[sinkid];
            }
            else //Bug 1177225: Door Swivel unit remains closed and Glass remains off after the opening and closing the Door via CAN
            {
               // Do nothing
               ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: invalid passenger door status - %d", status));
               sendSwivelCommand = false;
            }
            break;
         }

         case CAN_DRIVER_DOOR_INP:
         {
            // status = door open status
            if (status == 1)
            {
               // door is opened
               ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: driver door open - swivel UP"));
               isDriverDoorOpen = true;
               swivelRequest = COMMAND_SWIVEL_UP;
            }
            else if (status == 0) //Bug 1177225: Door Swivel unit remains closed and Glass remains off after the opening and closing the Door via CAN
            {
               // door is closed, follow all other swivels
               ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: driver door closed"));
               isDriverDoorOpen = false;
               swivelRequest = currentSwivelStatus[sinkid];
            }
            else //Bug 1177225: Door Swivel unit remains closed and Glass remains off after the opening and closing the Door via CAN
            {
               // Do nothing
               ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: invalid driver door status - %d", status));
               sendSwivelCommand = false;
            }
            break;
         }

         case PNM_INP:
         {
            ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: PNM Group10 ON =%d", PNM_INP));
            if (currentSwivelStatus[sinkid] != COMMAND_SWIVEL_UP)
            {
               // PNM Group 10 On, swivel up

               swivelRequest = COMMAND_SWIVEL_UP;
               ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: mode changed to UP currentSwivelStatus swivelRequest=%d %d", currentSwivelStatus[sinkid], swivelRequest));
            }
            break;
         }

         case MON_INP:

         case HMI_INP:
         {
            // status = HMI swivel status
            ETG_TRACE_USR3(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: HMI swivel status = %d", status));
            if (HMI_SWIVEL_DOWN == status)
            {
               swivelRequest = COMMAND_SWIVEL_DOWN;

               // Status could have changed from AUTO, stop inactivity timer
               stopInactivityTimer(sinkid);
            }
            else if (HMI_SWIVEL_UP == status)
            {
               swivelRequest = COMMAND_SWIVEL_UP;

               // Status could have changed from AUTO, stop inactivity timer
               stopInactivityTimer(sinkid);
            }
            else if (HMI_SWIVEL_AUTO == status)
            {
               if (isContentActiveInGlass[sinkid])
               {
                  // Content is active, swivel down
                  ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: content active hence swivel down"));
                  swivelRequest = COMMAND_SWIVEL_DOWN;
               }
               else
               {
                  // No content is active, start inactivity timer
                  ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: content not active"));
                  startInactivityTimer(sinkid);

                  // No command to send to swivel
                  sendSwivelCommand = false;
               }
            }
            break;
         }

         default:
         {
            ETG_TRACE_USR3(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: key input not handled - %d", key));
            sendSwivelCommand = false;
            break;
         }
      }
      // is always true by default whenever it enters.
      if (sendSwivelCommand)
      {
         // Select the swivel to send the command
         if (key == CAN_PASSENGER_DOOR_INP)
         {
            // Request to only passenger door asociated swivels
            ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: only passenger door swivels selected"));
            swivelSelected = SWIVEL_SELECTED_ONLY_PASSENGER_DOOR;
         }
         else if (key == CAN_DRIVER_DOOR_INP)
         {
            // Request to only driver door asociated swivels
            ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: only driver door swivels selected"));
            swivelSelected = SWIVEL_SELECTED_ONLY_DRIVER_DOOR;
         }
         else if (
            (swivelRequest == COMMAND_SWIVEL_DOWN) &&
            (isPassengerDoorOpen) &&
            (isDriverDoorOpen)
         )
         {
            // Request to swivel down but both passenger and driver door is open
            ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: all except door swivels selected"));
            swivelSelected = SWIVEL_SELECTED_ALL_EXCEPT_DOOR;
         }
         else if (
            (swivelRequest == COMMAND_SWIVEL_DOWN) &&
            (isPassengerDoorOpen) &&
            (!isDriverDoorOpen)
         )
         {
            // Request to swivel down but passenger door is open
            ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: all except passenger door swivels selected"));
            swivelSelected = SWIVEL_SELECTED_ALL_EXCEPT_PASSENGER_DOOR;
         }
         else if (
            (swivelRequest == COMMAND_SWIVEL_DOWN) &&
            (!isPassengerDoorOpen) &&
            (isDriverDoorOpen)
         )
         {
            // Request to swivel down but driver door is open
            ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: all except driver door swivels selected"));
            swivelSelected = SWIVEL_SELECTED_ALL_EXCEPT_DRIVER_DOOR;
         }
         else
         {
            // Request to all swivels
            ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: all swivels selected"));
            swivelSelected = SWIVEL_SELECTED_ALL;
         }

         // Swivel movement is associated with LCD On/Off, set the LCD On/Off status
         if (COMMAND_SWIVEL_DOWN == swivelRequest)
         {
            ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: swivel down and hence LCD on"));
            lcdOnOffRequest = COMMAND_LCD_ON;
         }
         else if (COMMAND_SWIVEL_UP == swivelRequest)
         {
            ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: swivel up and hence LCD off"));
            lcdOnOffRequest = COMMAND_LCD_OFF;
         }
         else if (COMMAND_SWIVEL_NONE_LCD_ON == swivelRequest)
         {
            ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: no swivel movement, only LCD on"));
            lcdOnOffRequest = COMMAND_LCD_ON;
         }
         else
         {
            ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: no swivel movement, only LCD off"));
            lcdOnOffRequest = COMMAND_LCD_OFF;
         }

         // Send swivel command along with LCD On/Off to plug-in
         if (CabinAGlassSettingListhandler::poGetInstance()->sendSwivelCommandToPlugin(sinkid, static_cast<uint8>(swivelSelected), static_cast<uint8>(swivelRequest), static_cast<uint8>(lcdOnOffRequest)))
         {
            ETG_TRACE_USR4(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: sendSwivelCommandToPlugin succeeded swivelRequest%d lcdOnOffRequest%d", swivelRequest, lcdOnOffRequest));

            // Checks swivel not near the door
            if ((key != CAN_PASSENGER_DOOR_INP) && (key != CAN_DRIVER_DOOR_INP))
            {
               //maintain swivel and lcd status other then door
               currentSwivelStatus[sinkid] = swivelRequest;
               currentLCDStatus[sinkid] = lcdOnOffRequest;
            }
         }
      }

      // lcmUserState Maintains previous state of User State
      if (isUserStandbyOffState == true)
      {
         lcmUserState =  USER_OFF;
      }
      else
      {
         lcmUserState = USER_ON;
      }

      ETG_TRACE_USR1(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: lcmUserState==>%d", lcmUserState));
      //Add the intereface here
      ListRegistry::s_getInstance().updateList(LIST_ID_CABIN_A_GLASS_MONITOR_FLIP_PANEL_DD_LIST);
      ListRegistry::s_getInstance().updateList(LIST_ID_CABIN_A_GAMMA_MAIN_LIST);
   }
   else
   {
      ETG_TRACE_USR1(("CabinAGlassSettingListhandler ::evaluateAndSendSwivelCommand: unsupported sinkid = %d", sinkid));
   }
}


/**********************************************************************************************
*FUNCTION:          sendSwivelCommandToPlugin
*DESCRIPTION:       Sends requested command to swivel
*PARAMETER:
*RETURNVALUE:       true - successfully sent command to swivel
*                   false - failed to send command to swivel
*
*HISTORY:
*05.05.2020:Prashant BG

*0.1 Initial Revision.
**********************************************************************************************/
bool CabinAGlassSettingListhandler::sendSwivelCommandToPlugin(uint8 sinkID, uint8 swivelSelected, uint8 swivelRequest, uint8 lcdOnOffRequest)
{
   bool retValue = false;
   boost::shared_ptr<EventDataUtility> pluginData(EventDataUtility::newEventDataUtility());
   ETG_TRACE_USR1(("CabinAGlassSettingListhandler sendSwivelCommandToPlugin: sinkID : %d, swivelSelected: %d, swivelRequest: %d, lcdOnOffRequest = %d", sinkID, swivelSelected, swivelRequest, lcdOnOffRequest));
   if (NULL != pluginData.get())
   {
      pluginData->addEventData(sinkID);
      pluginData->addEventData(swivelSelected);
      pluginData->addEventData(swivelRequest);
      pluginData->addEventData(lcdOnOffRequest);
      POST_MSG((COURIER_MESSAGE_NEW(PluginCommandReqMsg)(PLUGIN_NAME_SETTINGS, CTRLBLOCK_NAME_SETTINGS_SWIVELUPDOWN, pluginData)));
      retValue = true;
   }

   return retValue;
}


tSharedPtrDataProvider CabinAGlassSettingListhandler::GetComboGammaMonitorFlipListDataProvider(::Courier::UInt32 listID)
{
   const char* DATA_CONTEXT_MNTRCA_DROPDOWN_LIST_ITEM = "DropdownMNTRCAListButton";

   ListDataProviderBuilder listBuilder(LIST_ID_CABIN_A_GLASS_MONITOR_FLIP_PANEL_DD_LIST, DATA_CONTEXT_MNTRCA_DROPDOWN_LIST_ITEM);

   listBuilder.AddItem(1UL).AddData(GlassMonitorFlipStrings[0]);
   listBuilder.AddItem(2UL).AddData(GlassMonitorFlipStrings[1]);
   listBuilder.AddItem(3UL).AddData(GlassMonitorFlipStrings[2]);

   return listBuilder.CreateDataProvider();
}


bool CabinAGlassSettingListhandler::onCourierMessage(const ButtonListItemUpdMsg& oMsg)
{
   unsigned int rowIdx = 0;
   if (oMsg.GetListId() == LIST_ID_CABIN_A_GAMMA_MAIN_LIST)
   {
      ETG_TRACE_USR1(("CabinAGlassSettingListhandler:: ButtonListItemUpdMsg:LIST_ID_CABIN_A_GAMMA_MAIN_LIST: rowIdx = %d", (unsigned int)oMsg.GetHdl()));
      rowIdx = (unsigned int)oMsg.GetHdl();
      Candera::DropdownChangeType CabinAGammaSingleRowUpd = Candera::DropdownChangeClose;
      if (((rowIdx == 4)))
      {
         CabinAGammaSingleRowUpd = Candera::DropdownChangeClose;
         DropdownChangeMsg* ddToggleMsg_MonitorUpd = COURIER_MESSAGE_NEW(DropdownChangeMsg)(CabinAGammaSingleRowUpd, 507);
         ETG_TRACE_USR2(("CabinAGlassSettingListhandler:: ButtonListItemUpdMsg:CabinAGammaSingleRowUpd = %d, ddToggleMsg_MonitorUpd = %d", CabinAGammaSingleRowUpd, ddToggleMsg_MonitorUpd));//needed?
         if (ddToggleMsg_MonitorUpd != NULL)
         {
            ddToggleMsg_MonitorUpd->Post();
         }
      }
      else if (((rowIdx == 5)))
      {
         CabinAGammaSingleRowUpd = Candera::DropdownChangeClose;

         DropdownChangeMsg* ddToggleMsg_LCDUpd = COURIER_MESSAGE_NEW(DropdownChangeMsg)(CabinAGammaSingleRowUpd, 506);
         ETG_TRACE_USR2(("CabinAGlassSettingListhandler:: ButtonListItemUpdMsg:CabinAGammaSingleRowUpd = %d, ddToggleMsg_LCDUpd = %d", CabinAGammaSingleRowUpd, ddToggleMsg_LCDUpd));//needed?
         if (ddToggleMsg_LCDUpd != NULL)
         {
            ddToggleMsg_LCDUpd->Post();
         }
      }
      else
      {
         DropdownChangeMsg* ddToggleMsg_LCDUpd = COURIER_MESSAGE_NEW(DropdownChangeMsg)(CabinAGammaSingleRowUpd, 506);
         if (ddToggleMsg_LCDUpd != NULL)
         {
            ddToggleMsg_LCDUpd->Post();
         }
         DropdownChangeMsg* ddToggleMsg_MonitorUpd = COURIER_MESSAGE_NEW(DropdownChangeMsg)(CabinAGammaSingleRowUpd, 507);
         if (ddToggleMsg_MonitorUpd != NULL)
         {
            ddToggleMsg_MonitorUpd->Post();
         }
         ETG_TRACE_USR2(("CabinAGlassSettingListhandler:: ButtonListItemUpdMsg: ddToggleMsg_MonitorUpd = %d, ddToggleMsg_LCDUpd = %d", ddToggleMsg_MonitorUpd, ddToggleMsg_LCDUpd));//needed?
      }
      return true;
   }
   return false;
}


void CabinAGlassSettingListhandler::onExpired(asf::core::Timer& timer, boost::shared_ptr<asf::core::TimerPayload> data)

{
   ETG_TRACE_COMP(("CabinAGlassSettingListhandler ::onExpired isInactivityTimerRunning=%d ", isInactivityTimerRunning));
   //unique timer comparsion
   int sinkid = SINK_INVALID;
   for (int i = 0; i < SINK_INVALID; i++)
   {
      if (&timer == &_swiveltimer[i])
      {
         sinkid = i;
         break;
      }
   }

   if (isSinkSupported(sinkid))
   {
      ETG_TRACE_COMP(("CabinAGlassSettingListhandler ::onExpired isInactivityTimerRunning=%d ", isInactivityTimerRunning[sinkid]));

      // call swivel door after time out here
      //on expiry of 10 min and isInactivityTimerRunning is made to false
      if (data->getReason() == asf::core::TimerPayload_Reason__Completed)
      {
         isInactivityTimerRunning[sinkid] = INACTIVITY_TIMER_EXPIRED;
         ETG_TRACE_USR2(("CabinAGlassSettingListhandler ::onExpired isInactivityTimerRunning=%d ", isInactivityTimerRunning[sinkid]));
         // Evaluate content state and send swivel command
         CabinAGlassSettingListhandler::poGetInstance()->evaluateAndSendSwivelCommand(sinkid, isContentActiveInGlass[sinkid], AUTO_INP);
      }
   }
}


}
}
