/**
 * @file        :  SxmUtilsListHandler.cpp
 * @addtogroup  :  AppHmi_Sxm
 * @brief       :
 * @copyright   : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
 *                The reproduction, distribution and utilization of this file as
 *                well as the communication of its contents to others without express
 *                authorization is prohibited. Offenders will be held liable for the
 *                payment of damages. All rights reserved in the event of the grant
 *                of a patent, utility model or design.
 */
#include "hall_std_if.h"   // Including HALL framework code
#include "SxmUtilsListHandler.h"
#include "App/Core/SxmUtils/SxmUtils.h"
#include "App/Core/SxmCommon/SXMCommonInterface.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS       TR_CLASS_APPHMI_SXMUTILS
#define ETG_I_TRACE_CHANNEL           TR_TTFIS_APPHMI_SXM
#define ETG_I_TTFIS_CMD_PREFIX        "APPHMI_SXM_"
#define ETG_I_FILE_PREFIX             App::Core::SxmUtilsListHandler::
#include "trcGenProj/Header/SxmUtilsListHandler.cpp.trc.h"
#endif


namespace App {
namespace Core {

/**
 * SxmUtilsListHandler Class constructor
 */
SxmUtilsListHandler::SxmUtilsListHandler()
   : _speedlockState(false)
   , _isFocused(false)
{
   ETG_TRACE_USR4(("SxmUtilsListHandler() Constructor"));
   _speedlockBviewName = Courier::ViewId("");
   _mbackListInfo.clear();
}


/**
 * SxmUtilsListHandler Class destructor
 */
SxmUtilsListHandler::~SxmUtilsListHandler()
{
   ETG_TRACE_USR4(("~SxmUtilsListHandler() Destructor"));
   _speedlockState = false;
   _speedlockBviewName = Courier::ViewId("");
   _isFocused = false;
}


/**
 * Callback to handle Courier Message ButtonReactionMsg Request
 * @param [in] : Reference of Courier::ButtonReactionMsg
 * @return     :  bool true if message is consumed
 */
bool SxmUtilsListHandler::onCourierMessage(const ButtonReactionMsg& oMsg)
{
   ListProviderEventInfo info;
   ::Courier::Identifier idSender = oMsg.GetSender();
   enReaction msgEnReaction = oMsg.GetEnReaction();
   if (ListProviderEventInfo::GetItemIdentifierInfo(idSender, info) && Enum_enRelease == msgEnReaction)
   {
      unsigned int listId = info.getListId();     // the list id for generic access
      unsigned int hdlRow = info.getHdlRow();     // normally the index
      ETG_TRACE_USR3(("ButtonReactionMsg listId:%d,hdlRow:%d", listId, hdlRow));
      if ((listId == LIST_ID_SXM_AUDIO_MENU && 11 == hdlRow) || ((listId > LIST_ID_SXM_START) &&
            (listId != LIST_ID_SXM_SPORTS__AFFILIATE_DETAILS_LIST) && (listId < LIST_ID_SXM_END) &&
            (listId != LIST_ID_SXM_AUDIO_FAVORITE_ARTISTLIST) && (listId != LIST_ID_SXM_AUDIO_FAVORITE_SONGLIST)))
      {
         ETG_TRACE_USR3(("ButtonReactionMsg _activelistinfo.startIndex :%d", _activelistinfo.startIndex));
         _mbackListInfo[listId] = _activelistinfo;
      }
   }
   return false;
}


/**
 * Callback to handle Courier Message ListChangedUpdMsg Request
 * @param [in] : Reference of Courier::ListChangedUpdMsg
 * @return     :  bool true if message is consumed
 */
bool SxmUtilsListHandler::onCourierMessage(const ListChangedUpdMsg& oMsg)
{
   ETG_TRACE_USR3(("ListChangedUpdMsg StartIndex:%d", oMsg.GetFirstVisibleIndex()));
   bool status = false;
   if (_activelistinfo.currentListId == oMsg.GetListId())
   {
      if ((_mbackListInfo.find(oMsg.GetListId()) != _mbackListInfo.end()) &&
            (_mbackListInfo[oMsg.GetListId()].startIndex != oMsg.GetFirstVisibleIndex()))
      {
         status = false;
      }
      else if ((_mbackListInfo.find(oMsg.GetListId()) != _mbackListInfo.end()) &&
               (_mbackListInfo[oMsg.GetListId()].startIndex == oMsg.GetFirstVisibleIndex()))
      {
         _mbackListInfo.erase(oMsg.GetListId());
         status = true;
      }
      else if (_activelistinfo.startIndex != oMsg.GetFirstVisibleIndex())
      {
         //_activelistinfo.focusedIndex = -1;
         status = true;
      }

      if (status)
      {
         _activelistinfo.startIndex = oMsg.GetFirstVisibleIndex();
         updateFooterInfo();
      }
   }
   return false;
}


/**
 * Callback to handle Courier Message ListDateProviderResMsg Request
 * @param [in] : Reference of Courier::ListDateProviderResMsg
 * @return     :  bool true if message is consumed
 */
bool SxmUtilsListHandler::onCourierMessage(const ListDateProviderResMsg& oMsg)
{
   if (oMsg.GetListDateProvider() != NULL)
   {
      unsigned int listId = oMsg.GetListDateProvider()->listId();
      unsigned int listSize = oMsg.GetListDateProvider()->listSize();
      unsigned int listVirtualSize = oMsg.GetListDateProvider()->virtualListSize();
      ETG_TRACE_USR3(("ListDateProviderResMsg FocusIndex:%d, listId:%d, ListSize:%d", oMsg.GetListDateProvider()->getFocusIndex(), listId, listSize));
      if (listId == _activelistinfo.currentListId)
      {
         unsigned int totallistsize = (listVirtualSize != 0) ? listVirtualSize : listSize;
         if (_mbackListInfo.find(listId) != _mbackListInfo.end()) //Backward Transition.
         {
            _activelistinfo.listsize = totallistsize;
            _activelistinfo.startIndex = _mbackListInfo[listId].startIndex;
            POST_MSG((COURIER_MESSAGE_NEW(ListChangeMsg)(listId, ListChangeSet, _activelistinfo.startIndex)));
            updateFooterInfo();
         }
         else if ((_activelistinfo.listsize != totallistsize) || (_mbackListInfo.find(listId) == _mbackListInfo.end())) //dynamic list size change & forward transition
         {
            _activelistinfo.listsize = totallistsize;
            updateFooterInfo();
         }
      }
   }
   return false;
}


/**
* Callback to handle Courier Message Courier::StartupMsg Request
* @param [in] : StartupMsg
* @return     : bool true if message is consumed
*/
bool SxmUtilsListHandler::onCourierMessage(const Courier::StartupMsg& /*oMsg*/)
{
   FocusChangedUpdMsg::Subscribe(Courier::ComponentType::Model, Courier::ComponentType::View);
   return false;
}


/**
* Callback to handle Courier Message FocusChangedUpdMsg Request
* @param [in] : FocusChangedUpdMsg
* @return     : bool true if message is consumed
*/
bool SxmUtilsListHandler::onCourierMessage(const FocusChangedUpdMsg& oMsg)
{
   ListProviderEventInfo info;
   ETG_TRACE_USR4(("FocusChangedUpdMsg  Message is  posted: %s", oMsg.GetView()));
   _speedlockBviewName = oMsg.GetView();
   _isFocused = false;
   if (ListProviderEventInfo::GetItemIdentifierInfo(oMsg.GetWidget(), info))
   {
      uint32 listId = info.getListId();
      if (_activelistinfo.currentListId == listId)
      {
         _activelistinfo.focusedIndex = static_cast<uint32>(info.getHdlRow());// normally the index
         updateFooterInfo();
         _isFocused = true;
         return true;
      }
   }
   else
   {
      _activelistinfo.focusedIndex = -1;
      updateFooterInfo();
      std::string lSpeedlockBviewName = _speedlockBviewName.CStr();//coverity fix
      if (!lSpeedlockBviewName.empty())
      {
         _isFocused = true;
      }
   }

   if (_speedlockState)
   {
      hideFocusOnSpeedLock();
   }
   return false;
}


/**
 * Helper Function to set selectedindex for a list.
 * @param [in] : listid
 * @param [in] : selectedindex
 */
void SxmUtilsListHandler::vSetSelectedIndex(uint16 listid, uint16 selectedindex)
{
   ETG_TRACE_USR3(("vSetSelectedIndex listId:%d,StartIndex:%d", listid, selectedindex));
   _activelistinfo.startIndex = selectedindex;
}


/**
* Helper Function to update footer info for a list.
*/
void SxmUtilsListHandler::updateFooterInfo()
{
   unsigned int dispfocusedindex = (_activelistinfo.focusedIndex == -1) ? _activelistinfo.startIndex : _activelistinfo.focusedIndex;
   (*_mSxmUpdateFooter).mFooterVisible = (_activelistinfo.listsize == 0) ? false : true;
   (*_mSxmUpdateFooter).mTotalCount = SxmUtils::conCatData("i", 1, _activelistinfo.listsize).c_str();
   (*_mSxmUpdateFooter).mFocusedIndex = SxmUtils::conCatData("i", 1, (dispfocusedindex + 1)).c_str();
   _mSxmUpdateFooter.SendUpdate(true);
}


/**
* Callback to handle Courier Message SxmFooterListIdReqMsg Request
* @param [in] : SxmFooterListIdReqMsg
* @return     : bool true if message is consumed
*/
bool SxmUtilsListHandler::onCourierMessage(const SxmFooterListIdReqMsg& oMsg)
{
   _activelistinfo.currentListId = oMsg.GetListId();
   _activelistinfo.startIndex = 0;
   _activelistinfo.focusedIndex = -1;
   _activelistinfo.listsize = 0;
   updateFooterInfo();
   if (LIST_ID_SXM_AUDIO_MENU == oMsg.GetListId())
   {
      POST_MSG(COURIER_MESSAGE_NEW(SXMFooterIsUpdatedMsg)(LIST_ID_SXM_AUDIO_MENU));
   }
   return true;
}


/**
* Helper function to set list position.
* @param [in] : aListId
* @param [in] : aListChangeType
* @param [in] : aValue
* @return     : bool true if message is consumed
*/
bool SxmUtilsListHandler::SendListChangeMsg(::Courier::UInt32 const& aListId, ListChangeType
      const& aListChangeType, ::Courier::Int32 const& aValue)
{
   bool rc = false;
   ::Courier::Message* msg = COURIER_MESSAGE_NEW(::ListChangeMsg)(aListId, aListChangeType, aValue, ListChangeMsgSourceUnknown);
   COURIER_DEBUG_ASSERT(0 != msg);
   if (0 != msg)
   {
      rc = msg->Post();
      COURIER_DEBUG_ASSERT(rc);
      msg = 0;
   }
   return rc;
}


/**
* Helper function to return the current start list index
* @return     : the list index
*/
uint32 SxmUtilsListHandler::getCurrentStartIndex()
{
   ETG_TRACE_USR3(("getCurrentStartIndex StartIndex:%d", _activelistinfo.startIndex));
   return _activelistinfo.startIndex;
}


/**
* Helper function to clear footer info.
* @return     : the list index
*/
void SxmUtilsListHandler::clearFooterInfo(uint32 listId, bool clearFooter)
{
   if (clearFooter)
   {
      _mbackListInfo.clear();
   }
   else
   {
      _mbackListInfo.erase(listId);
   }
}


/**
* Helper Function to Hide the focus for B screens when speedlock is active
* @param [in] : listid
* @param [in] : selectedindex
*/
void SxmUtilsListHandler::hideFocusOnSpeedLock()
{
   unsigned int lspeedlockBHashid = _speedlockBviewName.GetHashId().HashValue();
   std::string lSpeedlockBviewName =  _speedlockBviewName.CStr();
   if ((lspeedlockBHashid == SXMCommonInterface::getActiveViewName(Enum_SURFACEID_MAIN_SURFACE_SXM)) && _isFocused)
   {
      switch (lspeedlockBHashid)
      {
         case Sxm_Scenes_SXM_RADIO__DIRECTTUNE:
         {
            POST_MSG(COURIER_MESSAGE_NEW(FocusReqMsg)(FOCUS_HIDE, FOCUS_TIMER_NO_ACTION, _speedlockBviewName, ""));
            break;
         }
         default:
            break;
      }
   }
   else if (("Global#Scenes#Speller#35_CHAR_LAYOUT" == lSpeedlockBviewName) || ("Global#Scenes#Speller#9_CHAR_LAYOUT" == lSpeedlockBviewName))
   {
      POST_MSG(COURIER_MESSAGE_NEW(FocusReqMsg)(FOCUS_HIDE, FOCUS_TIMER_NO_ACTION, _speedlockBviewName, ""));
   }
}


bool SxmUtilsListHandler::onCourierMessage(const SxmTouchReqMsg& oMsg)
{
   ETG_TRACE_USR3(("onCourierMessage Courier::TouchMsg"));
   if ((oMsg.GetTouchMsgState() == Courier::TouchMsgState::Up || oMsg.GetTouchMsgState() == Courier::TouchMsgState::Move) &&
         _activelistinfo.focusedIndex != -1)
   {
      _activelistinfo.focusedIndex = -1;
      updateFooterInfo();
   }
   return false;
}


bool SxmUtilsListHandler::onCourierMessage(const NavigationCarModeInfo& oMsg)
{
   ETG_TRACE_USR4(("SxmHall::onCourierMessage carMode :%d coachMode :%d", oMsg.GetCarMode(),  oMsg.GetCoachMode()));

   (*_carModeVisibilityStatus).mVisibleStatus = oMsg.GetCarMode();
   _carModeVisibilityStatus.MarkItemModified(ItemKey::CarModeVisibility::VisibleStatusItem);
   _carModeVisibilityStatus.SendUpdate(true);
   return true;
}


}  // namespace Core
}  // namespace App
