/**
 * @file        :  SxmUtils.cpp
 * @addtogroup  :  AppHmi_Sxm
 * @brief       :  Contains all utility methods used within AppHmi SXM
 * @copyright   :  (c) 2019-2019 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */
//#include "sys_std_if.h"
#include <sstream>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <iomanip>
#include <math.h>
#include <time.h>
#include "SxmUtils.h"
#include "asf/core/Types.h"
#include "../../../../../../../ai_osal_linux/components/system/system_types.h"
#include "Common/CommonUtils/CommonUnitsUtility.h"
#include "Common/CommonUtils/CommonClockUtility.h"
#include "AppUtils/HmiMacros.h"
#include "CanderaWidget/String/String.h"


namespace App {
namespace Core {


/**
 * Utility method to concatenate strings and variables and give a final output in a format to be printed on the screen
 * @param [in] typeFormate - Format of the paameter passed in the function
 * @param [in] listCount - count of the variables to be concatenated
 */
std::string SxmUtils::conCatData(std::string typeFormate, int listCount, ...)
{
   va_list vl;
   va_start(vl, listCount);
   std::string rString;
   char typeChar[listCount];
   strcpy(typeChar, typeFormate.c_str());
   for (int loopCount = 0; loopCount < listCount; ++loopCount)
   {
      switch (typeChar[loopCount])
      {
         case 'i':
         {
            char iStr[25];
            int iVal = va_arg(vl, int);
            snprintf(iStr, sizeof(iStr), "%d", iVal);
            rString = rString + iStr;
            break;
         }
         case 'd':
         {
            char iStr[25];
            double dVal = va_arg(vl, double);
            snprintf(iStr, sizeof(iStr), "%.2f", dVal);
            rString = rString + iStr;
            break;
         }
         case 's':
         {
            rString = rString + va_arg(vl, char*);
            break;
         }
         default:
            break;
      }
   }
   va_end(vl);
   return (rString);
}


/**
 * Utility method to calculate the percentage delta change in stock price.
 * @param [in] lMarketPrice - Market price of the stock.
 * @param [in] lDeltaValue - Delta value change of stock.
 * @param [in] lDirection - stocks price increase or decrease.
 * @return     String holding the percentage change in delta value of stock.
 */
std::string SxmUtils::ComputePerDeltaChangeForStocks(float lMarketPrice, float lDeltaValue, ::std::string lDirection)
{
   float lPercentageChange = 0.0;
   ::std::string lPer;
   lPer.clear();

   if (lDirection == "-")
   {
      if (0 < (lMarketPrice + lDeltaValue))
      {
         lPercentageChange = ((lDeltaValue / (lMarketPrice + lDeltaValue)) * 100);
         lPer = SxmUtils::conCatData("sssds", 5, " ", lDirection.c_str(), " ", lPercentageChange, " %");
      }
      else
      {
         lPer = "--";
      }
   }
   else if (lDirection == "+")
   {
      if (0 < (lMarketPrice - lDeltaValue))
      {
         lPercentageChange = ((lDeltaValue / (lMarketPrice - lDeltaValue)) * 100);
         lPer = SxmUtils::conCatData("sds", 3, lDirection.c_str(), lPercentageChange, " %");
      }
      else
      {
         lPer = "--";
      }
   }
   else if (lDirection == "")
   {
      lPer = "   0.00 %";
   }
   return lPer;
}


/**
 * Helper function to calculate direction using two set of coordinates.
 * @param [in] : destnLat   - target latitude.
 * @param [in] : destnLat  - target longitude.
 * @param [in] : curntlat   - current latitude.
 * @param [in] : curntlon  - current longitude.
 * @return : direction in degrees.
 */
double SxmUtils::calculateAirDistanceWithDirection(double destnLat, double destnLon, double curntlat, double curntlon)
{
   double deltaLatitude  = destnLat - curntlat;
   double deltaLongitude = destnLon - curntlon;
   deltaLongitude *= ::cos(degrees2radians((destnLat + curntlat) / 2.0));

   // range of direction in degrees: -180 to + 180
   double carDirection = radians2degrees(::atan2(deltaLongitude, deltaLatitude));

   // convert direction into North-relative angle:
   carDirection = ::fmod((carDirection + 360.0), 360.0);

   return carDirection;
}


/**
 * Helper function to convert degree2radians.
 * @param [in] : angle_degrees  - heading in degrees
 * @return : direction in radians.
 */
double SxmUtils::degrees2radians(double angle_degrees)
{
   return (M_PI / 180) * angle_degrees;
}


/**
 * Helper function to convert radians2degrees.
 * @param [in] : angle_degrees  - heading in radians
 * @return : direction in degrees.
 */
double SxmUtils::radians2degrees(double angle_degrees)
{
   return (180 / M_PI) * angle_degrees;
}


/**
 * Helper function to limits direction to 0-360.
 * @param [in] : degree   - direction
 * @return : direction in degrees.
 */
double SxmUtils::makeCircular(double degree)
{
   double fmod_degree = ::fmod(degree, 360.0);
   return ::fabs(fmod_degree >= 0.0 ? fmod_degree : fmod_degree + 360.0);
}


/**
 * Helper function to get lat or long in double format.
 * @return : returns lat long.
 */
double SxmUtils::convertlatlongTodouble(signed int latlong)
{
   const double DOUBLETOINT = 1000000.0;
   double doLatLong = static_cast <double>(latlong);
   return (doLatLong / DOUBLETOINT);
}


/**
 * Helper function to get lat or long in WGS84 format.
 * @param [in] : Lat/Lon
 * @return : returns lat/long in WGS84.
 */
int SxmUtils::convertlatlongToWGS84(int latlong)
{
   // Refer the type "T_s32_Latitude" in "di_fi\components\fi\pos_fi\pos_fi.pdf" for conversion factor
   const double fFactorNavPos = (4294967295.0 / (360.0 * 1000000.0));
   int latlon = static_cast <int>((latlong * fFactorNavPos));
   return latlon;
}


/**
 * Helper function to convert WGS84 to s32 format
 * @param [in] : Lat/Lon in WGS84 format
 * @return : returns lat/lon
 */
int SxmUtils::convertWGS84ToLatLong(int latlongwsg)
{
   // Refer the type "T_s32_Latitude" in "di_fi\components\fi\pos_fi\pos_fi.pdf" for conversion factor
   const double fFactorNavPos = ((360.0 * 1000000.0) / 4294967295.0);
   int latlon = static_cast <int>((latlongwsg * fFactorNavPos));
   return latlon;
}


/**
 * Function to check the North America boundary
 * @param [in] : Latitude
 * @param [in] : Longitude
 * @return : bool
 */
bool SxmUtils::bCheckLocationBoundary(double lat, double lon)
{
   bool bIsWithinLimit = true;
   //Limits from file "di_middleware_server\components\fc_sxm\smslib\sms\source\sms\source\_location_obj.h"
   const double MIN_LATITUDE = 0.0;
   const double MAX_LATITUDE = 90.0;
   const double MIN_LONGITUDE = -178.0;
   const double MAX_LONGITUDE = -30.0;
   if (lat < MIN_LATITUDE || lat > MAX_LATITUDE || lon < MIN_LONGITUDE || lon > MAX_LONGITUDE)
   {
      bIsWithinLimit = false;
   }
   return bIsWithinLimit;
}


std::string SxmUtils::convertToFeetInches(unsigned int inches)
{
   std::string lResultString;
   unsigned int lFeet;
   unsigned int lInches;
   lFeet = static_cast<unsigned int>(inches / 12);
   lInches = static_cast<unsigned int>(inches % 12);

   lResultString = conCatData("isis", 4, lFeet, " ft ", lInches, " in");
   return lResultString;
}


/**
 * Helper function to convert inches to meter in double format.
 * @return : returns meter.
 */
std::string SxmUtils::convertInchesToMeter(unsigned int inches)
{
   std::string lResultString;
   const double METERCONT = 0.0254;
   double doMts = METERCONT * inches;
   lResultString = SxmUtils::conCatData("ds", 2, doMts, " mts ");
   return lResultString;
}


/**
 * Helper function to convert millimeters to inches.
 * @return : returns meter.
 */
unsigned int SxmUtils::convertMillimeterToInches(unsigned int milliMeters)
{
   unsigned int result = 0;

   if (0 != milliMeters)
   {
      const double CENTIMETERS_PER_INCH = 2.54;
      double inches = ((static_cast<double>(milliMeters) / 10) / CENTIMETERS_PER_INCH);
      result = static_cast<unsigned int>(ceil(inches));
   }
   return result;
}


/**
 * Function to convert the angle to direction
 * @param [in] : angle
 * @param [in] : No. of directions (8/16 only)
 * @return : direction
 */
int SxmUtils::FormatDirectionInfo(double dActualAngle, unsigned int modValue)
{
   /*----
      8 directions: 360 / 8 = 45° for each direction sector
       N = -22.5 to 22.5, NE = 22.5 to 67.5, .....
   		   	                N (0°)
   		   	                |
   		   	                |
   		   	                |
   		      (270°) W -----X----- E (90°)
   		   	                |Reference
   		   	                |
   		   	                |
   		   	                S (180°)
   ----*/

   //modValue = 8; 360/45 = 8directions - NavigationService
   //modValue = 16; 360/22.5 = 16directions - Weather [SX-9845-0031_SXM_Tabular_Weather_Protocol_v1.1_2010-11-12.pdf]
   double dirAngle = (modValue == 8) ? 45 : 22.5; //22.5 or 45
   int heading = (dirAngle == 45) ? -1 : 16;//-1 = EN_HEADING_INDEX_UNKNOWN, 16 = Sxm_Max_Wind_Direction_Index
   unsigned int uHeadingArr[16] = {0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15}; //enHeadingType
   if (dActualAngle >= 0.0 && dActualAngle <= 360.0)
   {
      dActualAngle = round(dActualAngle);
      //double num = fmod(dActualAngle,360.0);
      unsigned int val = static_cast <unsigned int>(round(dActualAngle / dirAngle));
      heading = uHeadingArr[(val % modValue)];
   }
   return heading;
}


std::string SxmUtils::convertTimestampToTime(unsigned int timeStamp)
{
   time_t stamp = static_cast<time_t>(timeStamp);
   struct tm* pTimeValue = gmtime(&stamp);
   pTimeValue->tm_mon += 1;
   pTimeValue->tm_year += 1900;
   std::ostringstream sTime;
   sTime << pTimeValue->tm_mon << "/" << pTimeValue->tm_mday << "/" << pTimeValue->tm_year << " ";
   if (0 == (pTimeValue->tm_hour / 10))
   {
      sTime << 0;//Append 0 if single digit
   }
   sTime << pTimeValue->tm_hour << ":" ;
   if (0 == (pTimeValue->tm_min / 10))
   {
      sTime << 0;//Append 0 if single digit
   }
   sTime << pTimeValue->tm_min;

   return sTime.str();
}


std::string SxmUtils::representConvertedFeetInches(unsigned int inches)
{
   std::string lResultString;
   unsigned int lFeet;
   unsigned int lInches;
   lFeet = static_cast<unsigned int>(inches / 12);
   lInches = static_cast<unsigned int>(inches % 12);

   lResultString = conCatData("isis", 4, lFeet, "' ", lInches, "''");
   return lResultString;
}


/**
 *  Returns the date string in required format
 *
 *  @param[in] : Epoch time
 *  @param[in] : bool, if to return mm/dd/yyyy format or WeekDay, Month dd, h:mm PM format.
 *  @return    : Date string in required format.
 */
std::string SxmUtils::getConvertedDate(time_t lEpochTime, unsigned int lIsShort, short unsigned int format)
{
   struct tm* timeinfo;
   std::string strTime = "";
   if (lEpochTime == 0)
   {
      strTime = "--";
   }
   else
   {
      timeinfo = gmtime(&lEpochTime);
      if (lIsShort == EN_SHORT_DATE_MDY)                            //Date format according to system settings.
      {
         unsigned int day = timeinfo->tm_mday;
         unsigned int month = timeinfo->tm_mon + 1;                         //Month from 0 to 11
         unsigned int year = timeinfo->tm_year + 1900;                      //Year since 1900
         strTime.append(performDateFormatConversion(format, day, month, year));
      }
      else if (lIsShort == EN_LONG_DATE)                            //format required by WSA Details - WeekDay, Month dd, hh::mm PM
      {
         char buffer[25] = {0};
         strftime(buffer, 25, "%A, %B %d, ", timeinfo);                            //WeekDay, Month dd, format.
         strTime.append(buffer);
         //       strTime.append(clockUtil.performTimeFormatAndMeridiemCalc(timeinfo->tm_hour, timeinfo->tm_min, static_cast<Timeformat>(format)));
      }
   }
   return strTime;
}


/**
 * Helper Function to convert temperature from Fahrenheit to Centigrade
 */
signed short int SxmUtils::ConvertTemperatureFromFToC(signed short int TempinF)
{
#define TEMPCONVFORFTOCCONST1 32
#define TEMPCONVFORFTOCCONST2 1.8
   double lTempinC = 0;
   lTempinC = (TempinF - TEMPCONVFORFTOCCONST1) / TEMPCONVFORFTOCCONST2;
   return static_cast<signed short int>(round(lTempinC));
}


/**
 * Helper Function to replace special character in the translation string.
 *
 * @param[in] : TranslationStr - String from translation
 * @param[in] : StrForReplace - string to replace *
 * @param[in] : strToSearch - string to search in Translation ID (*, $)
 * @return    : string after replace
 */
std::string SxmUtils::replaceInTranslationTextID(std::string TranslationStr, std::string StrForReplace, std::string strToSearch)
{
   if (TranslationStr.find(strToSearch) != std::string::npos)
   {
      TranslationStr.replace(TranslationStr.find(strToSearch), strToSearch.size(), StrForReplace);
   }
   return TranslationStr;
}


/**
 * Helper Function to convert wind speed from mph to kph.
 *
 * param[in] : TranslationStr - String from translation
 */
unsigned int SxmUtils::convertWindSpeedFromMPPHtoKPH(unsigned int winSpeed)
{
   return static_cast<unsigned int>(round(winSpeed * 1.609344));
}


/**
 * Helper Function to have synchronized access to FeatStd::String via GetCString interface
 *
 * param[in] : textId - text id of the string
 * @return    : string after conversion
 */
std::string SxmUtils::safeGetCString(unsigned int textId)
{
   FeatStd::String safestring = Candera::String(textId);  //get a text via text id
   std::string formatedString;
   formatedString.clear();
   SECURE_FEATSTD_STRING_ACCESS_BEGIN(safestring);
   formatedString = safestring.GetCString();
   SECURE_FEATSTD_STRING_ACCESS_END()
   return formatedString;
}


/**
 * performDateFormatConversion - Handle to convert the display format of date with respect to dateformat
 * @param[in] year, month, day, dateformatIndex
 * @parm[out] dateText
 * @return string
 *
 */
std::string SxmUtils::performDateFormatConversion(unsigned int dateFormatIndex, unsigned int day, unsigned int month, unsigned int year)
{
   char itoa_txt[10] = "";
   snprintf(itoa_txt, sizeof(itoa_txt), "%d", day);
   std::string dayText = static_cast<std::string>(itoa_txt);
   snprintf(itoa_txt, sizeof(itoa_txt), "%d", month);
   std::string monthText = static_cast<std::string>(itoa_txt);
   snprintf(itoa_txt, sizeof(itoa_txt), "%d", year);
   std::string yearText = static_cast<std::string>(itoa_txt);

   return getdatetext(dateFormatIndex, dayText, monthText, yearText);
}


/**
 * getdatetext - Handle to convert the display format of date with respect to dateformat
 * @param[in] year, month, day, dateformatIndex
 * @parm[out] dateText
 * @return string
 *
 */
std::string SxmUtils::getdatetext(unsigned int dateFormatIndex, std::string dayText, std::string monthText, std::string yearText)
{
   std::string dateText = "";
   switch (dateFormatIndex)
   {
      case DD_MM_YYYY:
         dateText = (dayText + "-" + monthText + "-" + yearText);
         break;
      case MM_DD_YYYY:
         dateText = (monthText + "-" + dayText + "-" + yearText);
         break;
      case YYYY_MM_DD:
         dateText = (yearText + "-" + monthText + "-" + dayText);
         break;
      case DD_MM_YYYY_SLASH:
         dateText = (dayText + "/" + monthText + "/" + yearText);
         break;
      case MM_DD_YYYY_SLASH:
         dateText = (monthText + "/" + dayText + "/" + yearText);
         break;
      case DD_MM_YYYY_DOT:
         dateText = (dayText + "." + monthText + "." + yearText);
         break;
      default:
         break;
   }
   return dateText;
}


} // end of namespace Core
} // end of namespace App
