/********************************************************************
 * @file         : SxmHMIServiceStateHandler.h
 * @addtogroup   : AppHmi_Sxm
 * @brief        : SXM HMI Service State Handler
 * @copyright    : (c) 2019-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 ************************************************************************************/
#ifndef SXMHMISERVICESTATEHANDLER_H_
#define SXMHMISERVICESTATEHANDLER_H_

#include "App/Core/SxmUtils/SxmUtilsSingleton.h"
#include "ISxmHMIServiceStateHandler.h"
#include "AppHmi_SxmStateMachine.h"
#include "CgiExtensions/CourierMessageMapper.h"
#include "AppBase/ServiceAvailableIF.h"
#include  "CgiExtensions/DataBindingItem.hpp"
#include "App/Core/SxmUtils/SxmCommonTypes.h"
#include "Common/DataModel/LockoutRestrictionTypes.h"

namespace App {
namespace Core {
/**
 *
 * @class : SxmHMIServiceStateHandler
 */
class SxmHMIServiceStateHandler
   : public SxmUtilsSingleton<SxmHMIServiceStateHandler>
   , public StartupSync::PropertyRegistrationIF
   , public hmibase::ServiceAvailableIF
{
   public:
      /**
       * SxmHMIServiceStateHandler Class constructor
       */
      SxmHMIServiceStateHandler();
      /**
       * SxmHMIServiceStateHandler Class destructor
       */
      virtual ~SxmHMIServiceStateHandler();
      /**
       * Utility function that registers for notifications updates from SXM Audio MIDW layer
       */
      void registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/);
      /**
       * Utility function that deregisters for notifications updates from SXM Audio MIDW layer
       */
      void deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/);
      /**
       * Register call back function for Service Entry Exit updates
       */
      void registerServiceEntryExitNotification(ISxmHMIServiceStateHandler* _obsptr, enSXMServiceType enServiceType);
      /**
       *Register call back function for Application mode change updates
       */
      void registerApplicationModeChangeNotication(ISxmHMIServiceStateHandler* _obsptr, enSXMServiceType enServiceType);
      /**
       * DeRegister call back function for Application mode change updates
       */
      void deregisterApplicationModeChangeNotication(ISxmHMIServiceStateHandler* _obsptr, enSXMServiceType enServiceType);
      /**
       * DeRegister call back function for Service Entry exit change updates
       */
      void deregisterServiceEntryExitNotification(ISxmHMIServiceStateHandler* _obsptr, enSXMServiceType enServiceType);
      /**
      *Register call back function for Application state change updates
      */
      void registerApplicationStateNotication(ISxmHMIServiceStateHandler* _obsptr, enSXMServiceType enServiceType);
      /**
       * DeRegister call back function for Application state change updates
       */
      void deregisterApplicationStateNotication(ISxmHMIServiceStateHandler* _obsptr, enSXMServiceType enServiceType);
      /**
       * Helper Function to get Active View.
       */
      unsigned int getVisibleView(unsigned int uSurfaceId);
      /**
       * Helper Function to get Application state.
       */
      bool getApplicationState() const;
      /**
       * Helper Function to get Active state
       */
      unsigned int getActiveState()const;
      /**
       * Helper Function to get Active state of Data service
       */
      unsigned int getDataServiceState()const;
      /**
       * Helper Function to get GUI initial state
       */
      bool getGuiInitState()const;
      /**
       * Helper Function to get app mode.
       */
      unsigned int getActiveAppMode()const;
      /**
       * Helper Function to status view visibility.
       */
      bool isViewVisible(unsigned int viewid);
      /**
       * Helper Function to update map status bar.
       */
      void updateMapStatusBar(TChannelInformation chnInfo);
      /**
       * Helper Function to get TargetContextId.
       */
      uint32 getTargetContextId() const;
      /**
       * Helper Function to set TargetContextId.
       */
      void setTargetContextId(uint32 contextId);
      /**
       * Helper Function to get gadget visibility
       */
      bool getGadgetVisibility(enSXMServiceType serviceType);
      /**
       * Helper Function to get Speed Lock data from KDS
       */
      bool getLockoutKdsStatus(lockoutKDSConfig lockoutKDSConfig);
      /**
       * Helper function to set scene name.
       */
      void setSxmsceneName(const char* scenename);
      /**
       * Helper function to get scene name.
       */
      const char* getSxmSceneName();
   private:

      //Member Variable
      std::map <enSXMServiceType, ISxmHMIServiceStateHandler*> _entryExitStateObsList;
      std::map <enSXMServiceType, ISxmHMIServiceStateHandler*> _appmodeChangeObsList;
      std::map <enSXMServiceType, ISxmHMIServiceStateHandler*> _appStateObsList;
      unsigned short int _activeService;
      unsigned short int _serviceState;
      bool _isSxmInForeground;
      std::map<unsigned int, unsigned int> _visibleViewInfo;
      bool _screensaverStatus;
      unsigned short int _activeAppMode;
      bool _guiInitialized;
      std::string _statusBarChannelInfo;
      unsigned int _channelID;
      uint32 _targetContextId;
      const char* _sxmSceneName;
      bool _isTpopupActive;
      std::vector<enSXMServiceType> _visibleGadgetInfo;
      DataBindingItem<SxmApplicationStateDataBindingSource> _applicationState;
      //Member functions

      std::map<uint8_t, bool> _lockoutKdsConfigMap;
      /**
       * Helper function to update gadget visibility
       */
      void updateGadgetVisibility(enSXMServiceType ServiceType, unsigned short int status);
      /**
       * Helper function to read lock out from KDS
       */
      void readLockoutKDSData();
      /**
       * Callback to handle Courier Message SxmActiveSceneSelectReqMsg Request
       */
      bool onCourierMessage(const SXMActiveServiceMsg& oMsg);
      /**
       * Callback to handle Courier Message ApplicationStateUpdMsg Request
       */
      bool onCourierMessage(const ApplicationStateUpdMsg& oMsg);
      /**
       * Helper function to get current status of service.
       */
      bool onCourierMessage(const SxmActiveRenderedViewMsg& oMsg);
      /**
       * Callback to get GUI ready msg
       */
      bool onCourierMessage(const GuiStartupFinishedUpdMsg& msg);

      COURIER_MSG_MAP_BEGIN(TR_CLASS_APPHMI_SXM_COURIER_PAYLOAD_MODEL_COMP)
      ON_COURIER_MESSAGE(SXMActiveServiceMsg)
      ON_COURIER_MESSAGE(SxmActiveRenderedViewMsg)
      ON_COURIER_MESSAGE(ApplicationStateUpdMsg)
      COURIER_MSG_MAP_DELEGATE_DEFAULT_BEGIN()
      COURIER_MSG_MAP_DELEGATE_END()
};


}  // namespace Core
}  // namespace App

#undef HMI_DATA_SERVICE_NAMESPACE

#endif /* SXMHMISERVICESTATEHANDLER_H_ */
