/********************************************************************
 * @file         : SxmHMIServiceStateHandler.h
 * @addtogroup   : AppHmi_Sxm
 * @brief        : SXM HMI Service State Handler
 * @copyright    : (c) 2019-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 ************************************************************************************/

#include "hmi_trace_if.h"
#include "SxmHMIServiceStateHandler.h"
#include "App/Core/ClientHandler/SourceSwitch/SXMSourceSwitchInterface.h"
#include "CgiExtensions/DataBindingItem.hpp"
#include "HmiTranslation_TextIds.h"
#include "App/Core/LanguageDefines.h"
#include "App/Core/SxmUtils/SxmUtilsListHandler.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS       TR_CLASS_APPHMI_SXMCOMMON
#define ETG_I_TRACE_CHANNEL           TR_TTFIS_APPHMI_SXM
#define ETG_I_TTFIS_CMD_PREFIX        "APPHMI_SXM_"
#define ETG_I_FILE_PREFIX             App::Core::SxmHMIServiceStateHandler::
#include "trcGenProj/Header/SxmHMIServiceStateHandler.cpp.trc.h"
#endif

#define DEFAULT_CHANNEL_ID 65335

namespace App {
namespace Core {

/**
 * SxmHMIServiceStateHandler Class constructor
 */
SxmHMIServiceStateHandler::SxmHMIServiceStateHandler()
   : _activeService(0)
   , _serviceState(0)
   , _isSxmInForeground(false)
   , _screensaverStatus(false)
   , _activeAppMode(0)
   , _guiInitialized(false)
   , _statusBarChannelInfo("")
   , _channelID(DEFAULT_CHANNEL_ID)
   , _targetContextId(0)
   , _sxmSceneName("")
   , _isTpopupActive(false)
{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR4(("SxmHMIServiceStateHandler() Constructor"));
   _visibleViewInfo.clear();
   _entryExitStateObsList.clear();
   _appmodeChangeObsList.clear();
   _appStateObsList.clear();
   _visibleGadgetInfo.clear();
   _lockoutKdsConfigMap.clear();
   readLockoutKDSData();
}


/**
 * SxmHMIServiceStateHandler Class destructor
 */
SxmHMIServiceStateHandler::~SxmHMIServiceStateHandler()
{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR4(("SxmHMIServiceStateHandler() destructor"));
   _visibleViewInfo.clear();
   _activeService = 0;
   _serviceState = 0;
   _isSxmInForeground = false;
   _screensaverStatus = false;
   _guiInitialized = false;
   _activeAppMode = 0;
   _statusBarChannelInfo = "";
   _channelID = DEFAULT_CHANNEL_ID;
   _sxmSceneName = "";
   _isTpopupActive = false;
   _visibleGadgetInfo.clear();
}


/**
 * Utility function that registers for notifications updates from SXM Audio MIDW layer
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmHMIServiceStateHandler::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SxmHMIServiceStateHandler::registerProperties"));
}


/**
 * Utility function that deregisters for notifications updates from SXM Audio MIDW layer
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmHMIServiceStateHandler::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SxmHMIServiceStateHandler::deregisterProperties"));
}


/**
 * Register call back function to receive updates.
 * @param [in] : _obsptr - object reference of observer.
 * @param [in] : enSXMServiceType - data Service type
 */
void SxmHMIServiceStateHandler::registerServiceEntryExitNotification(ISxmHMIServiceStateHandler* _obsptr, enSXMServiceType enServiceType)
{
   ETG_TRACE_USR4(("SxmHMIServiceStateHandler::vRegisterServiceEntryExitNotification "));
   if (_entryExitStateObsList.find(enServiceType) != _entryExitStateObsList.end())
   {
      ETG_TRACE_ERR(("Service is already registered"));
   }
   else
   {
      _entryExitStateObsList.insert(std::make_pair(enServiceType, _obsptr));
   }
}


/**
 * Register call back function to receive updates.
 * @param [in] : _obsptr - object reference of observer.
 * @param [in] : enSXMServiceType - data Service type
 */
void SxmHMIServiceStateHandler::registerApplicationModeChangeNotication(ISxmHMIServiceStateHandler* _obsptr, enSXMServiceType enServiceType)
{
   ETG_TRACE_USR4(("SxmHMIServiceStateHandler::vRegisterApplicationModeChangeNotication "));
   if (_appmodeChangeObsList.find(enServiceType) != _appmodeChangeObsList.end())
   {
      ETG_TRACE_ERR(("Service is already registered"));
   }
   else
   {
      _appmodeChangeObsList.insert(std::make_pair(enServiceType, _obsptr));
   }
}


/**
 * De-Register call back function to receive updates.
 * @param [in] : _obsptr - object reference of observer.
 * @param [in] : enSXMServiceType - data Service type
 */
void SxmHMIServiceStateHandler::deregisterApplicationModeChangeNotication(ISxmHMIServiceStateHandler* /*_obsptr*/, enSXMServiceType enServiceType)
{
   if (_appmodeChangeObsList.find(enServiceType) != _appmodeChangeObsList.end())
   {
      _appmodeChangeObsList.erase(enServiceType);
   }
   else
   {
      ETG_TRACE_ERR(("Service is not registered"));
   }
}


/**
 * De-Register call back function to receive updates.
 * @param [in] : _obsptr - object reference of observer.
 * @param [in] : enSXMServiceType - data Service type
 */
void SxmHMIServiceStateHandler::deregisterServiceEntryExitNotification(ISxmHMIServiceStateHandler* /*_obsptr*/, enSXMServiceType enServiceType)
{
   if (_entryExitStateObsList.find(enServiceType) != _entryExitStateObsList.end())
   {
      _entryExitStateObsList.erase(enServiceType);
   }
   else
   {
      ETG_TRACE_ERR(("Service is not registered"));
   }
}


/**
 * Register call back function to receive AppState updates.
 * @param [in] : _obsptr - object reference of observer.
 * @param [in] : enSXMServiceType - data Service type
 */
void SxmHMIServiceStateHandler::registerApplicationStateNotication(ISxmHMIServiceStateHandler* _obsptr, enSXMServiceType enServiceType)
{
   ETG_TRACE_USR4(("SxmHMIServiceStateHandler::vRegisterApplicationModeChangeNotication "));
   if (_appStateObsList.find(enServiceType) != _appStateObsList.end())
   {
      ETG_TRACE_ERR(("Service is already registered"));
   }
   else
   {
      _appStateObsList.insert(std::make_pair(enServiceType, _obsptr));
   }
}


/**
 * De-Register call back function to receive AppState updates.
 * @param [in] : _obsptr - object reference of observer.
 * @param [in] : enSXMServiceType - data Service type
 */
void SxmHMIServiceStateHandler::deregisterApplicationStateNotication(ISxmHMIServiceStateHandler* /*_obsptr*/, enSXMServiceType enServiceType)
{
   if (_appStateObsList.find(enServiceType) != _appStateObsList.end())
   {
      _appStateObsList.erase(enServiceType);
   }
   else
   {
      ETG_TRACE_ERR(("Service is not registered"));
   }
}


/**
 * Helper Function to get Active View.
 */
unsigned int SxmHMIServiceStateHandler::getVisibleView(unsigned int uSurfaceId)
{
   ETG_TRACE_USR4(("SxmHMIServiceStateHandler::vGetVisibleView "));
   return _visibleViewInfo[uSurfaceId];
}


/**
 * Helper Function to get Application state.
 */
bool SxmHMIServiceStateHandler::getApplicationState() const
{
   ETG_TRACE_USR4(("SxmHMIServiceStateHandler::vGetAplicationState "));
   return _isSxmInForeground;
}


/**
 * Helper Function to get Active state
 */
unsigned int SxmHMIServiceStateHandler::getActiveState()const
{
   ETG_TRACE_USR4(("SxmHMIServiceStateHandler::vGetActiveState "));
   return _activeService;
}


/**
 * Helper Function to get Active state of Data service
 */
unsigned int SxmHMIServiceStateHandler::getDataServiceState()const
{
   ETG_TRACE_USR4(("SxmHMIServiceStateHandler::getDataServiceState"));
   return _serviceState;
}


/**
 * Helper Function to get GUI initial state
 */
bool SxmHMIServiceStateHandler::getGuiInitState()const
{
   return _guiInitialized;
}


/**
 * Helper Function to get app mode.
 */
unsigned int SxmHMIServiceStateHandler::getActiveAppMode()const
{
   return _activeAppMode;
}


/**
 * Helper Function to status view visibility.
 */
bool SxmHMIServiceStateHandler::isViewVisible(unsigned int viewid)
{
   bool status = false;
   if (_visibleViewInfo.find(Enum_SURFACEID_MAIN_SURFACE_SXM) != _visibleViewInfo.end())
   {
      if (viewid == _visibleViewInfo[Enum_SURFACEID_MAIN_SURFACE_SXM])
      {
         status = true;
      }
   }
   ETG_TRACE_USR4(("SxmHMIServiceStateHandler::isViewVisible status:%u", status));
   return status;
}


/**
 * Callback to handle Courier Message ActiveRenderedView Request
 * @param [in] : Reference of Courier::ActiveRenderedView
 * @return     : True-When Message is processed.
 */
bool SxmHMIServiceStateHandler::onCourierMessage(const SxmActiveRenderedViewMsg& msg)
{
   unsigned int lHashid = msg.GetSceneID();
   ETG_TRACE_USR4(("SxmActiveRenderedViewMsg hashID:%u", lHashid));
   if (Sxm_Scenes_SXM_TAB_MAIN != lHashid)
   {
      if (_visibleViewInfo.find(msg.GetSurfaceID()) != _visibleViewInfo.end())
      {
         _visibleViewInfo.erase(msg.GetSurfaceID());
         _visibleViewInfo.insert(std::pair<unsigned int, unsigned int>(msg.GetSurfaceID(), lHashid));
      }
      else
      {
         _visibleViewInfo.insert(std::pair<unsigned int, unsigned int>(msg.GetSurfaceID(), lHashid));
      }
   }
   setSxmsceneName(msg.GetSceneName().GetCString());
   return true;
   //Finally msg comes here. So return true
}


/**
 * Callback to handle Courier Message SXMActiveServiceMsg Request
 * @param [in] : Reference of Courier::SXMActiveServiceMsg
 * @return     : True-When Message is processed.
 */
bool SxmHMIServiceStateHandler::onCourierMessage(const SXMActiveServiceMsg& oMsg)
{
   ETG_TRACE_USR4(("SxmHMIServiceStateHandler SXMActiveServiceMsgState %d, %d", oMsg.GetServiceType(), oMsg.GetServiceEntryExitStatus()));
   unsigned short int lServiceEntryExitstate = oMsg.GetServiceEntryExitStatus();
   enSXMServiceType serviceType = static_cast<enSXMServiceType>(oMsg.GetServiceType());
   if (!_entryExitStateObsList.empty())
   {
      std::map <enSXMServiceType, ISxmHMIServiceStateHandler*>::const_iterator it = _entryExitStateObsList.begin();
      for (; it != _entryExitStateObsList.end(); ++it)
      {
         it->second->vNotifyServiceStatus(lServiceEntryExitstate, oMsg.GetServiceType());
      }
   }
   if (SXMServiceStateEntry == lServiceEntryExitstate)
   {
      _activeService = serviceType;
      _serviceState = serviceType;
      if (SXMAudio == _activeService || SXMTravelLink == _activeService)
      {
         SxmUtilsListHandler::instance()->clearFooterInfo(SXMINVALID_LIST_INDEX, true);
      }
      if ((SXMAudio == _activeService) && (_isTpopupActive))
      {
         POST_MSG((COURIER_MESSAGE_NEW(SXMDeActivatePopUpMsg)(Sxm_SxmPopups_SXM_RADIO__TPOP_AUDIO_WITH_STATIONNAME)));
      }
   }
   if (SXMServiceStateExit == lServiceEntryExitstate && !(SXMGadgetAudioLarge <= serviceType && SXMGadgetWeatherHourlyFilter >= serviceType))
   {
      _serviceState = SXMServiceExit;
   }
   if ((SXMGadgetAudioLarge <= serviceType) && (SXMGadgetWeatherHourlyFilter >= serviceType))
   {
      updateGadgetVisibility(serviceType, lServiceEntryExitstate);
   }

   return false;
}


/**
 * Callback to handle Courier Message ApplicationStateUpdMsg Request
 * @param [in] : Reference of Courier::ApplicationStateUpdMsg
 * @return     : True-When Message is processed.
 */
bool SxmHMIServiceStateHandler::onCourierMessage(const ApplicationStateUpdMsg& oMsg)
{
   unsigned short int appState = oMsg.GetState();
   ETG_TRACE_USR4(("SxmHMIServiceStateHandler ApplicationStateUpdMsg %d", appState));
   _isSxmInForeground = false;
   if (hmibase::IN_FOREGROUND == appState)
   {
      _isSxmInForeground = true;
   }

   if (appState == hmibase::IN_FOREGROUND)
   {
      (*_applicationState).mState = true;
      _applicationState.SendUpdate(true);
   }
   else if (appState == hmibase::IN_BACKGROUND)
   {
      (*_applicationState).mState = false;
      _applicationState.SendUpdate(true);
   }

   std::map <enSXMServiceType, ISxmHMIServiceStateHandler*>::const_iterator it;
   for (it = _appStateObsList.begin(); it != _appStateObsList.end(); ++it)
   {
      it->second->vNotifyAppState(appState);
   }

   return false;
}


/**
 * Helper Function to update map status bar.
 */
void SxmHMIServiceStateHandler::updateMapStatusBar(TChannelInformation chnInfo)
{
   ETG_TRACE_USR4(("SxmHMIServiceStateHandler updateMapStatusBar"));
}


/**
 * Helper function to set TargetContextId.
 * @param [in] : contextId - targetcontextid from vOnNewActiveContext
 */
void SxmHMIServiceStateHandler::setTargetContextId(uint32 contextId)
{
   _targetContextId = contextId;
}


/**
 * Helper function to get the TargetContextId.
 */
uint32 SxmHMIServiceStateHandler::getTargetContextId() const
{
   return _targetContextId ;
}


/**
 * Helper function to update gadget visibility
 */
void SxmHMIServiceStateHandler::updateGadgetVisibility(enSXMServiceType ServiceType, unsigned short int status)
{
   ETG_TRACE_USR4(("SxmHMIServiceStateHandler: updateGadgetVisibility :: ServiceType:%d status:%d size:%d", ServiceType, status, _visibleGadgetInfo.size()));
   if (!status)
   {
      _visibleGadgetInfo.push_back(ServiceType);
   }
   else
   {
      _visibleGadgetInfo.erase(std::remove(_visibleGadgetInfo.begin(), _visibleGadgetInfo.end(), ServiceType), _visibleGadgetInfo.end());
   }
}


/**
 * Helper Function to get gadget visibility
 */
bool SxmHMIServiceStateHandler::getGadgetVisibility(enSXMServiceType serviceType)
{
   ETG_TRACE_USR4(("SxmHMIServiceStateHandler: getGadgetVisibility :: ServiceType:%d size:%d", serviceType, _visibleGadgetInfo.size()));
   bool status = false;
   if (std::find(_visibleGadgetInfo.begin(), _visibleGadgetInfo.end(), serviceType) != _visibleGadgetInfo.end())
   {
      status = true;
   }
   else
   {
      status = false;
   }
   ETG_TRACE_USR4(("SxmHMIServiceStateHandler: getGadgetVisibility :: status:%d ", status));
   return status;
}


/**
 * Helper Function to get Speed Lock data from KDS
 */
bool SxmHMIServiceStateHandler::getLockoutKdsStatus(lockoutKDSConfig lockoutKDSConfig)
{
   bool lockoutConfig = false;
   std::map<uint8_t, bool>::iterator pos = _lockoutKdsConfigMap.find(lockoutKDSConfig);
   if (pos != _lockoutKdsConfigMap.end())
   {
      lockoutConfig = pos->second;
   }
   return lockoutConfig;
}


/**
 * Helper Function to get Speed Lock data from KDS
 */
void SxmHMIServiceStateHandler::readLockoutKDSData()
{
   /*   std::map<uint8_t, fptrGetKDSListLockoutConfig> lockoutKdsConfigMap;
   #define LOCKOUT_KDS_INFO(Category, KdsConfig) lockoutKdsConfigMap.insert(std::pair<uint8_t,fptrGetKDSListLockoutConfig>(Category,KdsConfig));
   #include "Common/DataModel/LockoutKDSConfig.dat"
     tU8 u8LockoutConfig;
     dp_tclKdsLockout oKdsLockout;
     for (std::map<uint8_t, fptrGetKDSListLockoutConfig>::iterator it = lockoutKdsConfigMap.begin(); it != lockoutKdsConfigMap.end(); ++it)
     {
        (oKdsLockout.*lockoutKdsConfigMap[it->first])(u8LockoutConfig);
        _lockoutKdsConfigMap.insert(std::pair<uint8_t, bool>(it->first, (static_cast<bool>(u8LockoutConfig))));
     } */
}


/**
 * Helper function to set scene name.
 */
void SxmHMIServiceStateHandler::setSxmsceneName(const char* scenename)
{
   _sxmSceneName = scenename;
}


/**
  * Helper function to get scene name.
  */
const char* SxmHMIServiceStateHandler::getSxmSceneName()
{
   return _sxmSceneName;
}


}  // namespace Core
}  // namespace App
