/**************************************************************************************
 * @file         : SxmAudioPreset.cpp
 * @addtogroup   : AppHmi_Sxm
 * @brief        : Functional class to handle business logic for Presets.
 * @copyright    : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 **************************************************************************************/
#include "hall_std_if.h"   // Including HALL framework code
#include "SxmAudioPreset.h"
#include "SxmAudioListHelper.h"
#include "SxmAudioDataBindingUtils.h"
#include "sxm_channelart_main_fi.h"
#include "App/Core/SxmCommon/SXMCommonInterface.h"
#include "App/Core/ClientHandler/Vehicle/SXMVehicleServiceInterface.h"
#include "HmiTranslation_TextIds.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS       TR_CLASS_APPHMI_SXMAUDIO
#define ETG_I_TRACE_CHANNEL           TR_TTFIS_APPHMI_SXM
#define ETG_I_TTFIS_CMD_PREFIX        "APPHMI_SXM_"
#define ETG_I_FILE_PREFIX             App::Core::SxmAudioPreset::
#include "trcGenProj/Header/SxmAudioPreset.cpp.trc.h"
#endif

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_hmi_06_if.h"

namespace App {
namespace Core {
using namespace ::sxm_audio_main_fi;
using namespace ::sxm_audio_main_fi_types;
using namespace ::sxm_channelart_main_fi;
using namespace ::sxm_channelart_main_fi_types;

/**
 * SxmAudioPreset Class constructor
 * @param [in] : pSxmAudioFiProxy - SxmAudioFiProxy class instance
 */
SxmAudioPreset::SxmAudioPreset(::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy > pSxmAudioFiProxy
                               , ISxmAudioChannelArtRequest* pSxmChannelartFiProxy)
   : _sxmAudioFiProxy(pSxmAudioFiProxy)
   , _sxmChannelart(pSxmChannelartFiProxy)
   , _activePresetIndex(0)
   , _cMAXPRESETSINONEBAND(6)
   , _isPresetListAvailable(false)
   , _noSystemMessageError(false)
   , _isPresetUpdateRegistered(false)
   , _isPresetSwipeOrScrollProgress(false)
{
   //ETG_I_REGISTER_FILE();
   ETG_TRACE_USR4(("SxmAudioPreset() Constructor"));

   _requestedPresetIndexs.clear();

   //Read initial values from datapool
   dp_tclAppHmi_SxmSxmAudioActivePresetBand dp_ActivePresetBand;
   _presetBand = dp_ActivePresetBand.tGetData();
}


/**
 * SxmAudioPreset Class destructor
 */
SxmAudioPreset::~SxmAudioPreset()
{
   ETG_TRACE_USR4(("~SxmAudioPreset() Destructor"));
   _sxmAudioFiProxy.reset();
   _requestedPresetIndexs.clear();
   _sxmChannelart = NULL;
   _activePresetIndex = 0;
   _presetBand = 0;
   _isPresetListAvailable = false;
   _noSystemMessageError = false;
   _isPresetSwipeOrScrollProgress = false;
}


/**
 * Utility function that registers for notifications updates from SXM Audio MIDW layer
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmAudioPreset::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SxmAudioPreset::registerProperties"));
   if ((proxy == _sxmAudioFiProxy) && (_sxmAudioFiProxy))
   {
      _sxmAudioFiProxy->sendAutoCompareUpReg(*this);
      _sxmAudioFiProxy->sendSystemMessageUpReg(*this);
   }
}


/**
 * Utility function that deregisters for notifications updates from SXM Audio MIDW layer
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmAudioPreset::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SxmAudioPreset::deregisterProperties"));
   if ((proxy == _sxmAudioFiProxy) && (_sxmAudioFiProxy))
   {
      _sxmAudioFiProxy->sendPresetChannelUpdateRelUpRegAll();
      _sxmAudioFiProxy->sendAutoCompareRelUpRegAll();
      _sxmAudioFiProxy->sendSystemMessageRelUpRegAll();
      _isPresetUpdateRegistered = false;
   }
}


/**
 * Error handling function for get PresetList Method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error Error type
 */
void SxmAudioPreset::onGetPresetListError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< GetPresetListError >& /*error*/)
{
   ETG_TRACE_USR4(("SxmAudioPreset::onGetPresetListError"));
}


/**
 * Result handling function for get PresetList Method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : result - preset list result
 */
void SxmAudioPreset::onGetPresetListResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< GetPresetListResult >& result)
{
   ETG_TRACE_USR4(("SxmAudioPreset::onGetPresetListResult size:%d", result->getPresetChannels().size()));
   if (_sxmChannelart)
   {
      ETG_TRACE_USR4(("SxmAudioPreset::onGetPresetListResult system status:%d, subscription channelart:%d",
                      SxmAudioDataBindingUtils::instance()->getSystemMessageStatus(), _sxmChannelart->getChannelArtSubscriptionState()));
   }
   if (T_e8_MethodStatus__XMTUN_MS_SUCCESS == result->getStatus())
   {
      _isPresetListAvailable = true;
      //Send preset list & Check if their is change in content.
      SxmAudioListHelper::instance()->setPresetList(result->getPresetChannels());
      //Request for Channel art.
      getPresetChannlLogo(true, 0, 0);
   }
}


/**
 * Error handling function for get PresetChannelUpdate Property
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error - Error type
 */
void SxmAudioPreset::onPresetChannelUpdateError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr<PresetChannelUpdateError >& /*error*/)
{
   ETG_TRACE_USR4(("SxmAudioPreset::onPresetChannelUpdateError"));
}


/**
 * Status handling function for get PresetChannelUpdate Property
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : status - preset ChannelUpdate status
 */
void SxmAudioPreset::onPresetChannelUpdateStatus(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr<PresetChannelUpdateStatus >& status)
{
   if (status && _isPresetListAvailable && status->getPresetBand() && status->getPresetIndex())
   {
      uint8 presetIndex = status->getPresetIndex() + ((status->getPresetBand() - 1) * MAXPRESETNOBANDONE);
      ETG_TRACE_USR4(("SxmAudioPreset::onPresetChannelUpdateStatus PresetBand:%d,PresetIndex:%d", status->getPresetBand(), presetIndex));
      if ((presetIndex < (MAXPRESETNOBANDTHREE + 1)) &&
            (SxmAudioListHelper::instance()->updatePresetItem(presetIndex, status->getChannelInfo())))// max presets is 18.
      {
         getPresetChannlLogo(false, presetIndex, status->getChannelInfo().getServiceID());
      }
   }
   else if (!_isPresetListAvailable)
   {
      getPresetList();
   }
}


/**
 * Error handling function for auto compare property;
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error Error type
 */
void SxmAudioPreset::onAutoCompareError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
                                        const boost::shared_ptr< AutoCompareError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioPreset::onAutoCompareError"));
}


/**
 * Status handling function for auto compare property
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : status - Property status info
 */
void SxmAudioPreset::onAutoCompareStatus(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< AutoCompareStatus >& status)
{
   ETG_TRACE_USR4(("SxmAudioPreset::onAutoCompareStatus CurrentTunerElemNr:%d,TunerElemNr:%d", status->getCurrentTunerElemNr(), status->getTunerElemNr()));
   if (status)
   {
      SxmAudioListHelper* pAudioUtilsInstance = SxmAudioListHelper::instance();
      uint8 presetIndex = status->getTunerElemNr();
      if (!SxmAudioDataBindingUtils::instance()->getSxmTuneScanStatus())
      {
         _activePresetIndex = presetIndex;
      }
      pAudioUtilsInstance->setActivePresetIndex(presetIndex);
      if (_sxmAudioFiProxy->hasSystemMessage() &&
            T_e8_SystemMessage__XMTUN_SYSTEM_MSG_SXM_INITIALIZING != _sxmAudioFiProxy->getSystemMessage().getSystemErrorType() &&
            SxmAudioDataBindingUtils::instance()->getSxmChannelInfoSyncStatus())
      {
         pAudioUtilsInstance->updatePresetList(true);
      }
      //To display preset info in TPop Audio.
      SxmAudioDataBindingUtils::instance()->setSxmChannelInfoPresetNumber(presetIndex);
      SXMVehicleServiceInterface::updateXMPresetStatus(presetIndex);
   }
}


/**
 * Error handling function for set auto compare parameters method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error - Error type
 */
void SxmAudioPreset::onSetAutoCompareParametersError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SetAutoCompareParametersError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioPreset::onSetAutoCompareParametersError"));
}


/**
 * Result handling function for set auto compare parameters method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : result - Method result data
 */
void SxmAudioPreset::onSetAutoCompareParametersResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SetAutoCompareParametersResult >& /*result*/)
{
   ETG_TRACE_USR4(("SxmAudioPreset::onSetAutoCompareParametersResult"));
}


/**
 * Error handling function for recall preset method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error - Error type
 */
void SxmAudioPreset::onRecallPresetError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< RecallPresetError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioPreset::onRecallPresetError"));
}


/**
 * Result handling function for recall preset method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : result - Method result data
 */
void SxmAudioPreset::onRecallPresetResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< RecallPresetResult >& result)
{
   ETG_TRACE_USR4(("SxmAudioPreset::onRecallPresetResult"));
   if (result->getStatus())
   {
      //To display Tpop Audio.

      SxmAudioDataBindingUtils::instance()->updatePresetPageIndicator(_presetBand);
      SxmAudioListHelper::instance()->switchPresetBand();
      POST_MSG((COURIER_MESSAGE_NEW(SxmUpdatePresetlistReqMsg)(result->getPresetElement())));
   }
}


/**
 * Error handling function for store preset method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error - Error type
 */
void SxmAudioPreset::onStorePresetError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
                                        const boost::shared_ptr< StorePresetError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioPreset::onStorePresetError"));
}


/**
 * Result handling function for store preset method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : result - Method result data
 */
void SxmAudioPreset::onStorePresetResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< StorePresetResult >& result)
{
   ETG_TRACE_USR4(("SxmAudioPreset::onStorePresetResult"));

   if (result->getStatus())
   {
      POST_MSG((COURIER_MESSAGE_NEW(PlayBeepReqMsg)(hmibase::BEEPTYPE_AVM_OPERATION)));
      SxmAudioListHelper::instance()->setPresetListElement(result->getPresetElement(), result->getPresetListElement());
      //Refresh list
      getPresetChannlLogo(false, result->getPresetElement(), result->getPresetListElement().getServiceID());
   }
}


/**
 * Error handling function for system messages property
 * @param [in] : proxy SXM Audio FI proxy class instance
 * @param [in] : error Error type
 */
void SxmAudioPreset::onSystemMessageError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SystemMessageError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioPreset::onSystemMessageError"));
}


/**
 * Status handling function for system messages property
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : status - Property status info
 */
void SxmAudioPreset::onSystemMessageStatus(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SystemMessageStatus >& status)
{
   const T_e8_SystemMessage&  systemErrorType = status->getSystemErrorType();
   _noSystemMessageError = (T_e8_SystemMessage__XMTUN_SYSTEM_MSG_NO_ERROR == systemErrorType);
   if (T_e8_SystemMessage__XMTUN_SYSTEM_MSG_SXM_INITIALIZING == systemErrorType)
   {
      _isPresetListAvailable = false;
      SxmAudioListHelper::instance()->setPresetList(::std::vector< T_PresetListEntry >());
      SxmAudioListHelper::instance()->updatePresetList(false);
      SxmAudioDataBindingUtils::instance()->setSxmPresetListAvailbility(false);
   }
}


/**
 * Helper Function to handle channel art result.
 */
void SxmAudioPreset::onGetChannelArtListResult(const ::std::vector< ChannelGraphicsType >& channelArtList)
{
   ETG_TRACE_USR4(("SxmAudioPreset::onGetChannelArtListResult size:%d", channelArtList.size()));
   SxmAudioListHelper::instance()->setPresetCGList(channelArtList);
   synchronizeAllChannelInfo();
}


void SxmAudioPreset::onGetChannelLogoResult(const uint16& sid, const uint8& /*ChannelGraphicsType*/, const std::vector<uint8>& u8ImageData,
      const std::vector<uint8>& /*albumImageData*/)
{
   //update only selected item;
   ETG_TRACE_USR4(("SxmAudioPreset::onGetChannelLogoResult sid:%d", sid));
   if (!_requestedPresetIndexs.empty())
   {
      SxmAudioListHelper::instance()->setPresetCGListElement(_requestedPresetIndexs[0], sid, u8ImageData);
      _requestedPresetIndexs.erase(_requestedPresetIndexs.begin());
      synchronizeAllChannelInfo();
   }
}


/**
 * Error handling function for GetChannelArtList method.
 */
void SxmAudioPreset::onGetChannelArtListError()
{
   ETG_TRACE_ERR(("SxmAudioPreset::onGetChannelArtListError"));
   SxmAudioListHelper::instance()->setPresetCGList(::std::vector< ChannelGraphicsType >());//empty list
   synchronizeAllChannelInfo();
}


/**
 * Error handling function for GetChannelArt method.
 */
void SxmAudioPreset::onGetChannelLogoError(const uint16& sid)
{
   ETG_TRACE_ERR(("SxmAudioPreset::onGetChannelLogoError"));
   if (!_requestedPresetIndexs.empty())
   {
      SxmAudioListHelper::instance()->setPresetCGListElement(_requestedPresetIndexs[0], sid, std::vector<uint8>());
      _requestedPresetIndexs.erase(_requestedPresetIndexs.begin());
      synchronizeAllChannelInfo();
   }
}


/**
 * Helper function to synchronize display of all channel information.
 */
void SxmAudioPreset::synchronizeAllChannelInfo()
{
   ETG_TRACE_USR4(("SxmAudioPreset::synchronizeAllChannelInfo"));
   SxmAudioDataBindingUtils* pInstance = SxmAudioDataBindingUtils::instance();
   if (!pInstance->getSxmChannelInfoSyncStatus() && pInstance->getSxmAllChannelInfoAvailbility())
   {
      //To handle dynamic updates in preset list.
      tSharedPtrDataProvider dataProvider = SxmAudioListHelper::instance()->getSxmPresetListDataProvider(true);
      pInstance->setSxmPresetListAvailbility(true);
      if (_noSystemMessageError)
      {
         pInstance->updateAllChannelInformation();
      }
      else
      {
         pInstance->updateAdvisoryMessage();
      }
      POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
      pInstance->enablePresetListButton(LEFTANDRIGHT, 1);//enable page up/Down buttons.
      SxmAudioListHelper::instance()->switchPresetBand();
      dp_tclAppHmi_SxmSxmAudioActivePresetBand dp_ActivePresetBand;
      SxmAudioDataBindingUtils::instance()->updatePresetPageIndicator(dp_ActivePresetBand.tGetData());
   }
   else if (pInstance->getSxmChannelInfoSyncStatus())
   {
      SxmAudioListHelper::instance()->updatePresetList(true);
   }
   else
   {
      pInstance->setSxmPresetListAvailbility(true);
   }

   if (_sxmAudioFiProxy && !_isPresetUpdateRegistered)
   {
      _sxmAudioFiProxy->sendPresetChannelUpdateUpReg(*this);
      _isPresetUpdateRegistered = true;
   }
}


/**
 * Callback to handle Courier Message SxmAudioChangePresetBankReqMsg Request
 * @param [in] : Reference of Courier::SxmAudioChangePresetBankReqMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioPreset::onCourierMessage(const SxmAudioChangePresetBankReqMsg& oMsg)
{
   _presetBand = _presetBand + oMsg.GetDirection();
   _presetBand = (_presetBand < MINBAND) ? MAXBAND : ((_presetBand > MAXBAND) ? MINBAND : _presetBand);
   dp_tclAppHmi_SxmSxmAudioActivePresetBand dp_ActivePresetBand;
   dp_ActivePresetBand.s32SetData(_presetBand);
   SxmAudioListHelper::instance()->switchPresetBand();
   SxmAudioDataBindingUtils::instance()->updatePresetPageIndicator(_presetBand);
   return true;
}


/**
 * Callback to handle Courier Message SXMAudioRecallorStoreReqMsg Request
 * @param [in] : Reference of Courier::SXMAudioRecallorStoreReqMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioPreset::onCourierMessage(const SXMAudioRecallorStoreReqMsg& oMsg)
{
   uint8 u8PresetElementNo = oMsg.GetPresetRowIndex() + 1;
   if (_sxmAudioFiProxy)
   {
      if (!oMsg.GetPresetAction())
      {
         //Store Preset
         _sxmAudioFiProxy->sendStorePresetStart(*this, u8PresetElementNo);
      }
      else
      {
         //Recall Preset
         _sxmAudioFiProxy->sendRecallPresetStart(*this, u8PresetElementNo);
      }
   }
   SxmAudioDataBindingUtils::instance()->setDefaultSxmAudioPlayPauseData(T_e8_IRPlaybackStatus__SXM_AUDIO_IR_PLAY_STATE);
   return true;
}


/**
 * Callback to handle Courier Message SXMTuneToChannelReqMsg Request
 * @param [in] : Reference of Courier::SXMTuneToChannelReqMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioPreset::onCourierMessage(const SXMTuneToChannelReqMsg& oMsg)
{
   uint8 u8Selectionmode = static_cast<T_e8_SelectChannelMode>(oMsg.GetSelectionMode());
   bool status = false;
   if (_sxmAudioFiProxy && (u8Selectionmode == Enum_SXM_TUN_SELECT_PRESET_UP ||
                            u8Selectionmode == Enum_SXM_TUN_SELECT_PRESET_DOWN))
   {
      uint8 presetSeekIndex = _activePresetIndex;

      if (u8Selectionmode == Enum_SXM_TUN_SELECT_PRESET_UP)
      {
         //if Seek position reaches last preset,then reset to first preset elem.
         (presetSeekIndex < MAXPRESETNOBANDTHREE) ? (++presetSeekIndex) : (presetSeekIndex = MINPRESETNOBANDONE);
      }
      else if (u8Selectionmode == Enum_SXM_TUN_SELECT_PRESET_DOWN)
      {
         //if Seek position reaches first preset,then reset to Last preset elem.
         (presetSeekIndex > MINPRESETNOBANDONE) ? (--presetSeekIndex) : (presetSeekIndex = MAXPRESETNOBANDTHREE);
      }

      _sxmAudioFiProxy->sendRecallPresetStart(*this, presetSeekIndex);
      _activePresetIndex = presetSeekIndex;
      _presetBand = (MAXPRESETNOBANDONE >= _activePresetIndex) ? MINBAND : ((MAXPRESETNOBANDTWO < _activePresetIndex) ? MAXBAND : MIDBAND) ;
      dp_tclAppHmi_SxmSxmAudioActivePresetBand dp_ActivePresetBand;
      dp_ActivePresetBand.s32SetData(_presetBand);
      status = true;
   }
   return status;
}


/**
 * Callback to handle Courier Message SxmAudioUpdatePresetListReqMsg Request
 * @param [in] : Reference of Courier::SxmAudioUpdatePresetListReqMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioPreset::onCourierMessage(const SxmAudioUpdatePresetListReqMsg& oMsg)
{
   SxmAudioListHelper::instance()->switchPresetBand();
   return true;
}


/**
 * Callback to handle Courier Message ListChangedUpdMsg Request
 * @param [in] : Reference of Courier::ListChangedUpdMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioPreset::onCourierMessage(const ListChangedUpdMsg& oMsg)
{
   if (LIST_ID_SXM_PRESET_BANK == oMsg.GetListId())
   {
      ETG_TRACE_USR4(("SxmAudioPreset::ListChangedUpdMsg GetMovementStatus %d, GetFirstVisibleIndex %d", oMsg.GetMovementStatus(), oMsg.GetFirstVisibleIndex()));
      if (((oMsg.GetMovementStatus() == ListMovementScrolling) || (oMsg.GetMovementStatus() == ListMovementSwiping) ||
            (oMsg.GetMovementStatus() == ListMovementPage) || (oMsg.GetMovementStatus() == ListMovementUnknown)) && !_isPresetSwipeOrScrollProgress)
      {
         _isPresetSwipeOrScrollProgress = true;
      }
      else if ((oMsg.GetMovementStatus() == ListMovementFinished) && _isPresetSwipeOrScrollProgress)
      {
         _isPresetSwipeOrScrollProgress = false;
         _presetBand = MINBAND;
         if ((oMsg.GetFirstVisibleIndex() >= MAXPRESETNOBANDONE) && (oMsg.GetFirstVisibleIndex() < MAXPRESETNOBANDTWO))
         {
            _presetBand = MIDBAND;
         }
         else if (oMsg.GetFirstVisibleIndex() >= MAXPRESETNOBANDTWO)
         {
            _presetBand = MAXBAND;
         }
         dp_tclAppHmi_SxmSxmAudioActivePresetBand dp_ActivePresetBand;
         dp_ActivePresetBand.s32SetData(_presetBand);
         SxmAudioDataBindingUtils::instance()->updatePresetPageIndicator(_presetBand);
      }
   }
   return false;
}


/**
 * Helper function to get preset list on startup.
 */
void SxmAudioPreset::getPresetList()
{
   //Set Autocompare parameters
   if (_sxmAudioFiProxy)
   {
      _sxmAudioFiProxy->sendSetAutoCompareParametersStart(*this, MINPRESETNOBANDONE, MAXPRESETNOBANDTHREE);
      //get Preset list for band1;
      _sxmAudioFiProxy->sendGetPresetListStart(*this, 255);//255:Get all presets.
   }
}


/**
 * Helper function for get Channel logos for PresetList
 */
void SxmAudioPreset::getPresetChannlLogo(bool allSid, uint8 presetIndex, uint16 serviceId)
{
   if (_sxmChannelart)
   {
      if (allSid)
      {
         ::std::vector< uint16 > listOfSID = SxmAudioListHelper::instance()->getListOfSid(SxmAudioListHelper::instance()->getPresetList(), 0,
                                             SxmAudioListHelper::instance()->getPresetList().size());
         ETG_TRACE_USR4(("SxmAudioPreset::getPresetChannlLogo size:%d", listOfSID.size()));
         _sxmChannelart->sendChannelArtListMethodRequest(this, listOfSID, 1);//only channel logo
      }
      else
      {
         _requestedPresetIndexs.push_back(presetIndex);
         ETG_TRACE_USR4(("SxmAudioPreset::getPresetChannlLogo serviceId:%d", serviceId));
         _sxmChannelart->sendChannelLogoMethodRequest(this, serviceId, 1); //only channel logo
      }
   }
}


/**
 * Restore Dp value for Audio Preset
 */
void SxmAudioPreset::restoreAudioPreset(void)
{
   ETG_TRACE_USR4(("SxmAudioPreset::restoreAudioPreset"));
   dp_tclAppHmi_SxmSxmAudioActivePresetBand dp_ActivePresetBand;
   dp_ActivePresetBand.s32SetData(MINBAND);
   _presetBand = MINBAND;

   SxmAudioDataBindingUtils* pAudioUtilsInstance = SxmAudioDataBindingUtils::instance();
   SxmAudioListHelper* pAudioListHelperInstance = SxmAudioListHelper::instance();

   pAudioListHelperInstance->setPresetList(::std::vector< T_PresetListEntry > ());
   pAudioListHelperInstance->setPresetCGList(::std::vector< ChannelGraphicsType > ());
   pAudioUtilsInstance->updatePresetPageIndicator(_presetBand);
   pAudioUtilsInstance->setSxmPresetListAvailbility(false);
   pAudioListHelperInstance->updatePresetList(false);
   pAudioListHelperInstance->switchPresetBand();
}


/**
 * Function to set default value on service availability from suspended
 */
void SxmAudioPreset::onServiceResumed(const ::boost::shared_ptr< asf::core::Proxy >& proxy)
{
   ETG_TRACE_USR4(("SxmAudioPreset::onServiceResumed"));
   if ((proxy == _sxmAudioFiProxy) && (_sxmAudioFiProxy))
   {
      getPresetList();
   }
}


}  // namespace Core
}  // namespace App
