/**************************************************************************************
 * @file         : SxmAudioListHelper.cpp
 * @addtogroup   : AppHmi_Sxm
 * @brief        : Utility class to handle all processing for List requests
 * @copyright    : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 **************************************************************************************/
#include "hall_std_if.h"   // Including HALL framework code
#include "SxmAudioListHelper.h"
#include "sxm_main_fi_types.h"
#include "sxm_audio_main_fi_types.h"
#include "CgiExtensions/ImageLoader.h"
#include "App/Core/SxmUtils/SxmUtils.h"
#include "AppHmi_SxmStateMachine.h"
#include "App/Core/SxmCommon/SXMCommonInterface.h"
#include "App/Core/ClientHandler/DataCollector/SXMDataCollectorInterface.h"
#include "App/Core/ClientHandler/ClockService/SXMClockServiceInterface.h"
#include "HmiTranslation_TextIds.h"
#include "App/Core/LanguageDefines.h"
#include "SxmAudioDataBindingUtils.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS       TR_CLASS_APPHMI_SXMAUDIO
#define ETG_I_TRACE_CHANNEL           TR_TTFIS_APPHMI_SXM
#define ETG_I_TTFIS_CMD_PREFIX        "APPHMI_SXM_"
#define ETG_I_FILE_PREFIX             App::Core::SxmAudioListHelper::
#include "trcGenProj/Header/SxmAudioListHelper.cpp.trc.h"
#endif

// Includes all header files for FI of client-component
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"
#include "dp_hmi_06_if.h"

#define DEFAULT_CATEGORY 255
#define DA_VARIANT 2

namespace App {
namespace Core {
using namespace ::sxm_audio_main_fi_types;
using namespace ::sxm_main_fi_types;

/**
 * SxmAudioListHelper Class constructor
 */
SxmAudioListHelper::SxmAudioListHelper()
   : _selectedChannelSID(0)
   , _selectedCatType(0)
   , _activeChannelSID(0)
   , _activeChannelID(0)
   , _activeCategoryID(0)
   , _activePresetIndex(0)
   , _isExternalDiagmode(true)
   , _isSystemInfoTraffic(false)
   , _isSystemInfoTravellink(false)
   , _isTuneScanContentAvail(false)
   , _radioID("")
   , _sxmSubStatus("")
   , _sxmSuspendDate("")
   , _isDAVariant(false)
{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR4(("SxmAudioListHelper() Constructor"));
   /*    dp_tclKdsAIVIVariantCoding variantCoding;
      tU8 variantType;
      variantCoding.u8GetVariantSelection(variantType);
      _isDAVariant = (variantType == DA_VARIANT) ? true : false;//1-Nav variant,2 - DA variant
      ETG_TRACE_USR4(("SxmAudioListHelper, Variant:%u", variantType)); */
   _channelListStatus = (ListDataInfo)
   {
      LIST_ID_SXM_AUDIO_CHANNELLIST, 0, 6
   };
   _categoryList.clear();
   _channelList.clear();
   _customizedChannelList.clear();
   _presetList.clear();
   _registeredFavoriteresult.clear();
   _alertInfoList.clear();
   _leagueList.clear();
   _teamList.clear();
   _sfFavoriteList.clear();
   _sfAlertList.clear();
   _marketListEntry.clear();
   _sportsFlashFavMonitorTeamsList.clear();
   _chanllistlogo.clear();
   _customChanllistlogo.clear();
   _sportsFlashChannelLogo.clear();
   _contentAlertlistlogo.clear();
   _diagnosisStmList.clear();
   _diagnosisMonitorList.clear();
   _diagnosisInfoList.clear();
   _presetBitmapImages.clear();
   initializeDiagMap();

   //  SXMCommonInterface::registerServiceEntryExitNotification(this, SXMAudio);
   SXMCommonInterface::registerApplicationStateNotication(this, SXMAudio);
}


/**
 * SxmAudioListHelper Class destructor
 */
SxmAudioListHelper::~SxmAudioListHelper()
{
   ETG_TRACE_USR4(("~SxmAudioListHelper() Destructor"));
   ETG_I_UNREGISTER_FILE();
   _channelListStatus = (ListDataInfo)
   {
      LIST_ID_SXM_AUDIO_CHANNELLIST, 0, 6
   };
   _selectedChannelSID = 0;
   _selectedCatType = 0;
   _activeChannelSID = 0;
   _activeChannelID = 0;
   _activeCategoryID = 0;
   _activePresetIndex = 0;
   _isExternalDiagmode = true;
   _isSystemInfoTraffic = false;
   _isSystemInfoTravellink = false;
   _isTuneScanContentAvail = false;
   _isDAVariant = false;
   _categoryList.clear();
   _channelList.clear();
   _customizedChannelList.clear();
   _presetList.clear();
   _leagueList.clear();
   _teamList.clear();
   _sfFavoriteList.clear();
   _sfAlertList.clear();
   _marketListEntry.clear();
   _sportsFlashFavMonitorTeamsList.clear();
   _registeredFavoriteresult.clear();
   _alertInfoList.clear();
   _chanllistlogo.clear();
   _customChanllistlogo.clear();
   _sportsFlashChannelLogo.clear();
   _contentAlertlistlogo.clear();
   _diagnosisStmList.clear();
   _diagnosisMonitorList.clear();
   _diagnosisInfoList.clear();
   _presetBitmapImages.clear();
   _radioID = "";
   _sxmSubStatus = "";
   _sxmSuspendDate = "";
   //   SXMCommonInterface::deregisterServiceEntryExitNotification(this, SXMAudio);
   SXMCommonInterface::deregisterApplicationStateNotication(this, SXMAudio);
}


/*******************************************************Channel list & Info ********************************************************/

/**
 * Helper function to set Active Channel Information
 */
void SxmAudioListHelper::setSxmChannelInfoData(const T_ChannelInformation& channelInfo)
{
   _activeChannelID = channelInfo.getChannelNumber();
   _activeChannelSID = channelInfo.getSID();
   _activeCategoryID = channelInfo.getCategoryType();
}


/**
 * Helper function to handle processing of categorie list.
 * @param [in] : categoryList - Category list.
 */
void SxmAudioListHelper::setCategoryList(const ::std::vector< T_CategoryListEntry >& categoryList)
{
   ETG_TRACE_USR4(("setCategoryList() size %d", categoryList.size()));
   _categoryList = categoryList;
}


/**
 * Helper function to get category list.
 * return - categorylist
 */
const ::std::vector< T_CategoryListEntry >& SxmAudioListHelper::getCategoryList(void) const
{
   return _categoryList;
}


/**
 * Helper function to handle processing of channel list.
 * @param [in] : channellist - channel list.
 */
void SxmAudioListHelper::setSxmChannelList(const ::std::vector<T_ChannelListEntry>& channellist)
{
   _channelList = channellist;
}


/**
 * Helper function to get channel list.
 * @return - Channel list
 */
const ::std::vector< T_ChannelListEntry >& SxmAudioListHelper::getChannelList() const
{
   ETG_TRACE_USR4(("GetChannelList() size %d", _channelList.size()));
   return _channelList;
}


/**
 * Helper function to get Sid for selected list item from channel list.
 * @param [In] : selectedChannelIndex - Row id of selected channel.
 * @return - Service id of selected channel.
 */
uint16 SxmAudioListHelper::getSelectedChannelSID(uint32 selectedChannelIndex)
{
   _selectedChannelSID = 0;
   if (selectedChannelIndex < _channelList.size())
   {
      _selectedChannelSID = _channelList[selectedChannelIndex].getServiceID();
   }
   return _selectedChannelSID;
}


/**
 * Helper function to get Sid for selected list item from channel list.
 * @param [In] : catID - Row id of selected channel.
 */
void SxmAudioListHelper::setSelectedCategoryType(uint16 catID)
{
   _selectedCatType = catID;
}


/**
 * Helper function to get Active Category type.
 * return - returns Active Category type
 */
uint16 SxmAudioListHelper::getSelectedCategoryType(void) const
{
   return _selectedCatType;
}


/**
 * Helper function to get Active Category type.
 * return - returns Active Category type
 */
uint16 SxmAudioListHelper::getActiveChannelSID(void) const
{
   return _activeChannelSID;
}


/**
 * Helper function to get Active Channel ID.
 * return - returns Active Channel ID
 */
uint16 SxmAudioListHelper::getActiveChannelCID(void) const
{
   return _activeChannelID;
}


/**
* Helper function to set tune scan content status
* @param [In] : status - availability of tune scan content.
*/
void SxmAudioListHelper::setTuneScanContentStatus(bool status)
{
   _isTuneScanContentAvail = status;
}


/**
 * Helper function to get Active Category type.
 * @param [In] :channellist : customize channel list received from middleware
 */
bool SxmAudioListHelper::setSxmCustomizedChannelList(const ::std::vector< T_ChannelListEntry >& channellist)
{
   bool status = false;
   _customizedChannelList.clear();
   uint16 size = channellist.size();
   for (uint16 chnIdx = 0; chnIdx < size; ++chnIdx)
   {
      uint16 uSid = channellist[chnIdx].getServiceID();
      if (uSid != 0 && uSid != 1)//do not copy radio id & Preview Channel information.
      {
         _customizedChannelList.push_back(channellist[chnIdx]);
         if (channellist[chnIdx].getIsSkipped())
         {
            status = true;
         }
      }
   }
   return status;
}


/**
 * Helper function to get customize channel list.
 * @return - customize channel list.
 */
const ::std::vector< T_ChannelListEntry >& SxmAudioListHelper::getCustomizedChannelList()
{
   return _customizedChannelList;
}


/**
 * Helper function to get Service id for selected customized channel list entry.
 * @param [In] :index :Index of selected customized channel list entry.
 * return - service id
 */
uint16 SxmAudioListHelper::getCustomizedListEntryServiceId(uint16 index)
{
   uint16 serviceId = 65535; //Invalid Channel id.
   if (index < _customizedChannelList.size())
   {
      serviceId = _customizedChannelList[index].getServiceID();
   }
   return serviceId;
}


/**
 * Helper function to get Channel id for selected customized channel list entry.
 * @param [In] :index :Index of selected customized channel list entry.
 * return - channel id.
 */
uint16 SxmAudioListHelper::getCustomizedListEntryChannelId(uint16 index)
{
   uint16 channelId = 65535;//Invalid Channel id.
   if (index < _customizedChannelList.size())
   {
      channelId = _customizedChannelList[index].getChannelID();
   }
   return channelId;
}


/**
 * Helper function to get Channel skipped status for selected customized channel list entry.
 * @param [In] :index :Index of selected customized channel list entry.
 * return - Channel skipped status.
 */
bool SxmAudioListHelper::getCustomizedListEntryStatus(uint16 index)
{
   bool status = false;
   if (index < _customizedChannelList.size())
   {
      status = _customizedChannelList[index].getIsSkipped();
   }
   return status;
}


/**
 * Helper function to set Channel Logos.
 * @param [in] : imageData - Image Data received from midw result.
 * @param [in] : listID - list type
 */
void SxmAudioListHelper::setChannellistlogo(const ::std::vector< ChannelGraphicsType >& imageData, uint32 listID)
{
   switch (listID)
   {
      case Sxm_Scenes_SXM_RADIO__CHANNELLIST_3:
      {
         _chanllistlogo = imageData;
         break;
      }
      case Sxm_Scenes_SXM_RADIO__CUSTOMIZE_CHANNELLIST:
      {
         _customChanllistlogo = imageData;
         break;
      }
      case Sxm_Scenes_SXM_RADIO__ALERT_ARTISTLIST:
      case Sxm_Scenes_SXM_RADIO__ALERT_SONGLIST:
      {
         _contentAlertlistlogo = imageData;
         break;
      }
      default:
      {
         break;
      }
   }
}


/**
 * GetSxmTraveLinkListDataProvider - Add the items required for SXMAudio Menu List and
 * sends the ListDataProvider to the FlexList Widget
 * @return - tSharedPtrDataProvider
 */
tSharedPtrDataProvider SxmAudioListHelper::getSxmAudioMenuListDataProvider()
{
   ETG_TRACE_USR4(("SxmAudioListHelper::getSxmAudioMenuListDataProvider"));
   const char* const DATA_CONTEXT_LIST_SXM_AUDIO_MENU           	= "SXMAudioListItemText";
   uint32 menuitems[Item_8] =
   {
      SXM__AUDIO_MENU_SETTINGS,
      SXM__AUDIO_MENU_CUSTOMIZECHANNELLIST,
      SXM__AUDIO_MENU_EDITFAVARTIST,
      SXM__AUDIO_MENU_EDITFAVSONG,
      SXM__AUDIO_MENU_EDITFAVTWN,
      SXM__AUDIO_MENU_EDITSFTEAMS,
      SXM__AUDIO_MENU_TUNESTART,
      SXM__AUDIO_DIAG_SIGNALQUALITY
   };

   ListDataProviderBuilder listBuilder(LIST_ID_SXM_AUDIO_MENU, DATA_CONTEXT_LIST_SXM_AUDIO_MENU);

   for (uint8 u8idx = 0; u8idx < Item_8; ++u8idx)
   {
      Courier::DataBindingUpdater<SxmMenuListDataBindingSource>* ptr;

      if (u8idx == Item_7)
      {
         ptr = listBuilder.AddItem(u8idx, 0, "SXMAudioMenu_SignalQuality").AddDataBindingUpdater<SxmMenuListDataBindingSource>();
      }
      else if (u8idx == Item_6)
      {
         ptr = listBuilder.AddItem(u8idx, 0, "SXMAudioMenu_Tune").AddDataBindingUpdater<SxmMenuListDataBindingSource>();
      }
      else if (u8idx == Item_2)
      {
         ptr = listBuilder.AddItem(u8idx, 0, "SXMAudioListItemArtist").AddDataBindingUpdater<SxmMenuListDataBindingSource>();
      }
      else if (u8idx == Item_3)
      {
         ptr = listBuilder.AddItem(u8idx, 0, "SXMAudioListItemSong").AddDataBindingUpdater<SxmMenuListDataBindingSource>();
      }
      else if (u8idx == Item_4)
      {
         ptr = listBuilder.AddItem(u8idx, 0, "SXMAudioListItemTWN").AddDataBindingUpdater<SxmMenuListDataBindingSource>();
      }
      else if (u8idx == Item_5)
      {
         ptr = listBuilder.AddItem(u8idx, 0, "SXMAudioListItemSF").AddDataBindingUpdater<SxmMenuListDataBindingSource>();
      }
      else if (u8idx == Item_1)
      {
         ptr = listBuilder.AddItem(u8idx, 0, "SXMAudioListItemCustomizeText").AddDataBindingUpdater<SxmMenuListDataBindingSource>();
      }
      else
      {
         ptr = listBuilder.AddItem(u8idx, 0, "SXMAudioListItemNotificationText").AddDataBindingUpdater<SxmMenuListDataBindingSource>();
      }

      if (ptr != NULL)
      {
         Courier::DataItemContainer<SxmMenuListDataBindingSource>& item = ptr->GetValueContainer();
         if (u8idx == Item_6)
         {
            dp_tclAppHmi_SxmSxmAudioTuneStartStatus dp_TuneStartStatus;
            (*item).mTuneScanChoice = !dp_TuneStartStatus.tGetData();
         }
         if (u8idx == Item_7)
         {
            (*item).mSignalQuality = Candera::String(_diagnosisInfoList[SXMINFO_SIGNAL_QUALITY]);
         }
         (*item).mMenuText = Candera::String(menuitems[u8idx]);
         (*item).mEnable = true;
      }
   }
   tSharedPtrDataProvider dataprovider = listBuilder.CreateDataProvider();
   SXMCommonInterface::dataproviderCacheSetting(dataprovider, false);//disable cache for dynamic lists
   return dataprovider;
}


/**
 * getSxmCategorieListDataProvider - Add the items required for SxmCategorie List and
 * sends the ListDataProvider to the FlexList Widget
 * @return - tSharedPtrDataProvider
 */
tSharedPtrDataProvider SxmAudioListHelper::getSxmCategorieListDataProvider()
{
   ETG_TRACE_USR4(("SxmAudioListHelper::getSxmCategorieListDataProvider"));

   std::string  str = "";
   //initializes the provider with the list id and the default data context
   ListDataProviderBuilder listBuilder(LIST_ID_SXM_CATEGORIES, "CategoriesListItem");
   ::std::vector< T_CategoryListEntry>::const_iterator  itrCategoryList(_categoryList.begin());
   uint16 categoryListSize = _categoryList.size();
   bool isNoCategory = (DEFAULT_CATEGORY == _activeCategoryID);
   ETG_TRACE_USR4(("SxmAudioListHelper::getSxmCategorieListDataProvider _activeCategoryID is %d", _activeCategoryID));
   for (uint16 catIndex = 0; catIndex < categoryListSize; ++catIndex)
   {
      bool status = (_categoryList[catIndex].getCategoryType() == _activeCategoryID);
      Courier::DataBindingUpdater<SxmCategoriesListDataBindingSource>* ptr
         = listBuilder.AddItem(catIndex).AddData(status).AddDataBindingUpdater<SxmCategoriesListDataBindingSource>();
      if (ptr != NULL)
      {
         Courier::DataItemContainer<SxmCategoriesListDataBindingSource>& item = ptr->GetValueContainer();
         str = _categoryList[catIndex].getCategoryName();
         str.append(" (");
         str.append(SxmUtils::conCatData("i", 1, (_categoryList[catIndex].getChannelList().size())));
         str.append(")");
         (*item).mCategoryName = str.c_str();
         if (!status && isNoCategory)
         {
            status = true;
            isNoCategory = false;
         }
         (*item).mCurrentCategory = status;
      }
   }

   tSharedPtrDataProvider dataprovider = listBuilder.CreateDataProvider();
   SXMCommonInterface::dataproviderCacheSetting(dataprovider, false);//disable cache for dynamic lists
   return dataprovider;
}


/**
 * getSxmChannelListDataProvider - Add the items required for SxmChannel List and
 * sends the ListDataProvider to the FlexList Widget
 * @return - tSharedPtrDataProvider
 */
tSharedPtrDataProvider SxmAudioListHelper::getSxmChannelListDataProvider()
{
   ETG_TRACE_USR4(("SxmAudioListHelper::getSxmChannelListDataProvider"));
   //initializes the provider with the list id and the default data context
   ListDataProviderBuilder listBuilder(_channelListStatus.listId, "ChannelListItem");
   uint16 channellistsize = _channelList.size();
   for (uint16 u16chnidx = _channelListStatus.startIndex; ((u16chnidx < (_channelListStatus.startIndex + _channelListStatus.windowSize)) && (u16chnidx < channellistsize)); ++u16chnidx)
   {
      uint16 sid = _channelList[u16chnidx].getServiceID();
      Courier::DataBindingUpdater<SxmAudioChannelListDataBindingSource>* ptr =
         listBuilder.AddItem(u16chnidx).AddDataBindingUpdater<SxmAudioChannelListDataBindingSource>();
      if (ptr != NULL)
      {
         Courier::DataItemContainer<SxmAudioChannelListDataBindingSource>& item = ptr->GetValueContainer();
         (*item).mChannelNumber = SxmUtils::conCatData("i", 1, _channelList[u16chnidx].getChannelID()).c_str();
         (*item).mArtist = _channelList[u16chnidx].getChannelArtistName().c_str();
         (*item).mSong = _channelList[u16chnidx].getChannelSongName().c_str();
         (*item).mCurrentPlay = (sid == _activeChannelSID);

         getChannelArt(sid, _channelList[u16chnidx].getIsAvailable(), _channelList[u16chnidx].getChannelGraphicAvail(),
                       (u16chnidx), _chanllistlogo, (*item).mSxmLogo);
      }
   }
   tSharedPtrDataProvider dataprovider = listBuilder.CreateDataProvider(_channelListStatus.startIndex, _channelList.size());
   SXMCommonInterface::dataproviderCacheSetting(dataprovider, false);//disable cache for dynamic lists
   return dataprovider;
}


/**
 * getSxmChannelListDataProvider - Add the items required for SxmCustomizeChannel List and
 * sends the ListDataProvider to the FlexList Widget
 * @return - tSharedPtrDataProvider
 */
tSharedPtrDataProvider SxmAudioListHelper::getSxmCustomizeChannelListDataProvider(uint16 startIndex, uint16 windowElemSize)
{
   ETG_TRACE_USR4(("SxmAudioListHelper::getSxmCustomizeChannelListDataProvider StartIndex = %d, WindowElemSize = %d", startIndex, windowElemSize));
   ListDataProviderBuilder listBuilder(LIST_ID_SXM_AUDIO_CUSTOMIZE_LIST, "CustomizeChannelListItem");
   uint16 channellistsize = _customizedChannelList.size();
   for (uint16 u16chnidx = startIndex; (u16chnidx < (startIndex + windowElemSize) && u16chnidx < channellistsize); ++u16chnidx)
   {
      Courier::DataBindingUpdater<SXMAudioCustomizeChannelListItemDataBindingSource>* ptr  = listBuilder.AddItem(u16chnidx).
            AddDataBindingUpdater<SXMAudioCustomizeChannelListItemDataBindingSource>();
      if (ptr != NULL)
      {
         Courier::DataItemContainer<SXMAudioCustomizeChannelListItemDataBindingSource>& item = ptr->GetValueContainer();
         (*item).mChannelNumber = SxmUtils::conCatData("i", 1, _customizedChannelList[u16chnidx].getChannelID()).c_str();
         (*item).mChannelName = _customizedChannelList[u16chnidx].getChannelName().c_str();
         (*item).mIndex = _customizedChannelList[u16chnidx].getIsSkipped();
         (*item).mEnable = true;

         getChannelArt(_customizedChannelList[u16chnidx].getServiceID(), _customizedChannelList[u16chnidx].getIsAvailable(),
                       _customizedChannelList[u16chnidx].getChannelGraphicAvail(), (u16chnidx), _customChanllistlogo, (*item).mSxmLogo);
      }
   }
   tSharedPtrDataProvider dataprovider = listBuilder.CreateDataProvider(startIndex, _customizedChannelList.size());
   SXMCommonInterface::dataproviderCacheSetting(dataprovider, false);//disable cache for dynamic lists
   return dataprovider;
}


/*******************************************************Preset********************************************************/

void SxmAudioListHelper::printPresetImagesize()
{
   uint64 size = 0;
   size_t presetImagesSize = _presetBitmapImages.size();
   for (uint8 presetIndex = 0; presetIndex < presetImagesSize; ++presetIndex)
   {
      if (!_presetBitmapImages[presetIndex].PointsToNull())
      {
         size = size + (_presetBitmapImages[presetIndex]->GetHeight() * _presetBitmapImages[presetIndex]->GetWidth() * 3);
         ETG_TRACE_USR4(("SxmAudioListHelper::printPresetImagesize size:%d", size));
      }
   }
   ETG_TRACE_USR4(("SxmAudioListHelper::printPresetImagesize total size:%d", size));
}


/**
 * Helper function to set Active Preset Index for Auto Compare.
 * @param [In] : presetIdx - active preset index.
 */
void SxmAudioListHelper::setActivePresetIndex(uint8 presetIdx)
{
   _activePresetIndex = 255;//Invalid
   if (presetIdx)
   {
      _activePresetIndex = presetIdx;
   }
}


/**
* Helper function to update preset list.
* return - preset list.
*/
const ::std::vector< T_PresetListEntry >& SxmAudioListHelper::getPresetList() const
{
   return _presetList;
}


/**
* Helper function to set preset list.
* @param [In] : presetList - Preset list received from middleware.
*/
void SxmAudioListHelper::setPresetList(const ::std::vector< T_PresetListEntry >& presetList)
{
   _presetList = presetList;
}


/**
* Helper function to set Channellogo for preset list.
* @param [In] : presetCGList - Preset list with Channel Logo.
*/
void SxmAudioListHelper::setPresetCGList(const ::std::vector< ChannelGraphicsType >& presetCGList)
{
   _presetBitmapImages.clear();//check for memory leak.
   size_t presetListSize = _presetList.size();
   for (uint8 presetIndex = 0; presetIndex < presetListSize; ++presetIndex)
   {
      bool status = false;
      uint16 serviceID = _presetList[presetIndex].getServiceID();
      size_t presetCGListSize = presetCGList.size();
      for (uint8 presetCGIndex = 0; presetCGIndex < presetCGListSize; ++presetCGIndex)
      {
         if (presetCGList[presetCGIndex].ServiceID == serviceID && !presetCGList[presetCGIndex].u8ImageData.empty())
         {
            Candera::Bitmap::SharedPointer bmp =
               ImageLoader::loadBitmapData(const_cast<uint8*>(& (presetCGList[presetCGIndex].u8ImageData)[0]), presetCGList[presetCGIndex].u32Size);
            if (bmp != NULL)
            {
               _presetBitmapImages[presetIndex] = ImageLoader::createImage(bmp);
            }
            status = true;
            break;
         }
      }
      if (!status)
      {
         _presetBitmapImages[presetIndex] = ImageLoader::getAssetBitmapImage("Sxm#Images#Landing#Layer_Default_channel_logoBmp");
      }
   }
   printPresetImagesize();
}


/**
 * Helper function to SetPresetListElement
 * @param [In] : presetIdx - PresetIndx Modified.
 * @param [In] : presetElement - Channel Info for preset entry
 */
void SxmAudioListHelper::setPresetListElement(uint8 presetIdx, const T_PresetListEntry& presetElement)
{
   uint8 updatedIdx =  presetIdx - 1;//Presetindex starts from 1.
   if (updatedIdx < _presetList.size())
   {
      ETG_TRACE_USR4(("SxmAudioListHelper::setPresetListElement sid:%d,presetIdx:%d", presetElement.getServiceID(), updatedIdx));
      _presetList[updatedIdx].setServiceID(presetElement.getServiceID());
      _presetList[updatedIdx].setChannelID(presetElement.getChannelID());
      _presetList[updatedIdx].setChannelNameAvail(presetElement.getChannelNameAvail());
      _presetList[updatedIdx].setChannelName(presetElement.getChannelName());
      _presetList[updatedIdx].setChannelActive(presetElement.getChannelActive());
      _presetList[updatedIdx].setChnGraphicAvail(presetElement.getChnGraphicAvail());
      _presetList[updatedIdx].setIsSmartFavorite(presetElement.getIsSmartFavorite());
      _presetList[updatedIdx].setIsLocked(presetElement.getIsLocked());
      _presetList[updatedIdx].setIsAvailable(presetElement.getIsAvailable());
      _presetList[updatedIdx].setIsMature(presetElement.getIsMature());
      _presetList[updatedIdx].setIsSkipped(presetElement.getIsSkipped());
   }
}


/**
 * Helper function to updatePresetItem
 * @param [In] : presetIdx - PresetIndx.
 * @param [In] : presetElement - Channel Info for preset entry
 * return value : bool - true - if data is updated
 */
bool SxmAudioListHelper::updatePresetItem(uint8 presetIdx, const ::sxm_audio_main_fi_types::T_PresetListEntry& presetElement)
{
   uint8 updatedIdx =  presetIdx - 1;//Presetindex starts from 1.
   bool isUpdated = false;
   ETG_TRACE_USR4(("SxmAudioListHelper::updatePresetItem sid:%d,presetIdx:%d", presetElement.getServiceID(), updatedIdx));

   if ((updatedIdx < _presetList.size()) && ((_presetList[updatedIdx] != presetElement) || (presetElement.getChnGraphicAvail() == T_e8_Graphics__GRAPHICS_NEEDS_REFRESH)))
   {
      ETG_TRACE_USR4(("SxmAudioListHelper::updatePresetItem data updated"));
      setPresetListElement(presetIdx, presetElement);
      isUpdated = true;
   }
   return isUpdated;
}


/**
 * Helper function to Preset Channel logo of a single channel
 * @param [In] : sid - service id of updated channel.
 * @param [In] : presetIdx - Updated PresetIndx.
 * @param [In] : imageData - channel logo data.
 */
void SxmAudioListHelper::setPresetCGListElement(uint8 presetIdx, uint16 sid, const std::vector<uint8>& imageData)
{
   uint8 updatedIdx =  presetIdx - 1;//Preset index starts from 1.
   ETG_TRACE_USR4(("SxmAudioListHelper::setPresetCGListElement sid:%d,presetIdx:%d", sid, updatedIdx));
   if (!imageData.empty())
   {
      Candera::Bitmap::SharedPointer bmp =  ImageLoader::loadBitmapData(const_cast<uint8*>(& (imageData)[0]), imageData.size());
      if (bmp != NULL)
      {
         _presetBitmapImages[updatedIdx] = ImageLoader::createImage(bmp);
      }
   }
   else
   {
      _presetBitmapImages[updatedIdx] = ImageLoader::getAssetBitmapImage("Sxm#Images#Landing#Layer_Default_channel_logoBmp");
   }
   printPresetImagesize();
}


/**
 * Helper function to get Preset list for list widget.
 * @return - tSharedPtrDataProvider for preset list.
 */
tSharedPtrDataProvider SxmAudioListHelper::getSxmPresetListDataProvider(bool btnstatus)
{
   ETG_TRACE_USR4(("SxmAudioListHelper::getSxmPresetListDataProvider"));
   //initializes the provider with the list id and the default data context
   ListDataProviderBuilder listBuilder(LIST_ID_SXM_PRESET_BANK);
   size_t presetSize = 6;
   uint8 index = 0;
   if (btnstatus)
   {
      presetSize = _presetList.size();
   }
   else
   {
      index = getPresetBandStartIndex();
   }
   if (presetSize != 0)
   {
      for (uint16 presetidx = 0; presetidx < presetSize; ++presetidx)
      {
         Courier::DataBindingUpdater<SxmPresetItemDataBindingSource>* ptr;
         ptr = listBuilder.AddItem(presetidx, 0, "Btn_Preset").AddDataBindingUpdater<SxmPresetItemDataBindingSource>();
         if (ptr != NULL)
         {
            Courier::DataItemContainer<SxmPresetItemDataBindingSource>& item = ptr->GetValueContainer();
            (*item).mChannelNo = "";
            (*item).mIsEnable = btnstatus;
            if (!btnstatus)
            {
               (*item).mPresetNo = SxmUtils::conCatData("i", 1, (index + 1)).c_str();
               ++index;
            }
            if (btnstatus && _presetList.size())
            {
               (*item).mPresetNo = SxmUtils::conCatData("i", 1, (presetidx + 1)).c_str();
               (*item).mIsSelected = (presetidx == (_activePresetIndex - 1));
               if (_presetList[presetidx].getChannelID() <= END_CHANNEL_ID)
               {
                  (*item).mChannelNo = SxmUtils::conCatData("i", 1, _presetList[presetidx].getChannelID()).c_str();
                  if (_presetList[presetidx].getIsAvailable() && _presetBitmapImages.find(presetidx) != _presetBitmapImages.end())
                  {
                     (*item).mSxmLogo = _presetBitmapImages[presetidx];
                  }
                  else
                  {
                     (*item).mSxmLogo = ImageLoader::getAssetBitmapImage("Sxm#Images#Landing#Layer_Default_channel_logoBmp");
                  }
               }
               else
               {
                  (*item).mSxmLogo = ImageLoader::getAssetBitmapImage("Sxm#Images#Landing#Layer_Default_channel_logoBmp");
               }
            }
         }
      }
   }
   tSharedPtrDataProvider dataprovider = listBuilder.CreateDataProvider();
   if (!dataprovider.PointsToNull())
   {
      //dataprovider->setCacheOnOff(false);
   }
   return dataprovider;
}


/**
 * Helper Function to update autocompare status.
 */
void SxmAudioListHelper::updatePresetList(bool isEnable)
{
   tSharedPtrDataProvider dataProvider = getSxmPresetListDataProvider(isEnable);
   POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
}


/**
 * Helper function to get preset band start index
 */
uint8 SxmAudioListHelper::getPresetBandStartIndex()
{
   //Read initial values from datapool
   dp_tclAppHmi_SxmSxmAudioActivePresetBand dp_ActivePresetBand;
   uint8 presetBand = dp_ActivePresetBand.tGetData();
   uint8 index = 0;
   if (MIDBAND == presetBand)
   {
      index = MAXPRESETNOBANDONE;
   }
   else if (MAXBAND == presetBand)
   {
      index = MAXPRESETNOBANDTWO;
   }
   ETG_TRACE_USR4(("SxmAudioListHelper::getPresetBandStartIndex _presetBand:%d index %d", presetBand, index));
   return index;
}


/**
 * Function to switch preset band
 */
void SxmAudioListHelper::switchPresetBand()
{
   uint8 index = getPresetBandStartIndex();
   bool status = SxmUtilsListHandler::instance()->SendListChangeMsg(LIST_ID_SXM_PRESET_BANK, ListChangeSet, index);
   ETG_TRACE_USR4(("SxmAudioListHelper::switchPresetBand msgsent:%d", status));
}


/*******************************************************TWN********************************************************/
/**
 * Helper Function to set TWN market list.
 * @param [in] : marketListEntry - TWN market list.
 */
void SxmAudioListHelper::setMarketList(const ::std::vector< T_MarketListEntry >& marketListEntry)
{
   _marketListEntry = marketListEntry;
}


/**
 * Helper function to get twn market info.
 * @param [in] : marketindex - market index.
 * @param [in] : marketname - market name.
 * @param [in] : isactive - is market active.
 */
bool SxmAudioListHelper::getSelectedMarketInfo(uint32 marketindex, std::string& marketname, bool& isactive)
{
   bool status = false;
   if (marketindex < _marketListEntry.size())
   {
      marketname = _marketListEntry[marketindex].getFullMarketName();
      isactive = _marketListEntry[marketindex].getBIsActiveMarket();
      status = true;
   }
   return status;
}


/**
 * Helper function to get TWN market list for list widget.
 * @return - tSharedPtrDataProvider for TWN market list.
 */
tSharedPtrDataProvider SxmAudioListHelper::getSxmTWNowListDataProvider()
{
   ETG_TRACE_USR4(("SxmAudioListHelper::getSxmTWNowListDataProvider"));
   ListDataProviderBuilder listBuilder(LIST_ID_SXM_AUDIO_TWN_ENTRY, "SXMAudioTWNowMarketList");
   uint16 marketsize = _marketListEntry.size();
   for (uint16 marketindex = 0; marketindex < marketsize; ++marketindex)
   {
      Courier::DataBindingUpdater<SXMAudioTWNDataBindingSource>* ptr
         = listBuilder.AddItem(marketindex).AddDataBindingUpdater<SXMAudioTWNDataBindingSource>();
      if (ptr != NULL)
      {
         Courier::DataItemContainer<SXMAudioTWNDataBindingSource>& item = ptr->GetValueContainer();
         (*item).mOnOrOff = !(_marketListEntry[marketindex].getBIsActiveMarket());
         (*item).mCity = _marketListEntry[marketindex].getFullMarketName().c_str();
      }
   }

   tSharedPtrDataProvider dataprovider = listBuilder.CreateDataProvider();
   SXMCommonInterface::dataproviderCacheSetting(dataprovider, false);//disable cache for dynamic lists
   return dataprovider;
}


/*******************************************************Content Alert********************************************************/
/**
 * Helper function to save Favorite List
 * @param [in] : mRegisteredFavoriteresult - Favorite List
 */
void SxmAudioListHelper::setFavoriteList(const ::std::vector< T_RegisteredFavorite >& mRegisteredFavoriteresult)
{
   _registeredFavoriteresult = mRegisteredFavoriteresult;
}


/**
 * Helper function to get alert list.
 * @return -  alert list.
 */
const ::std::vector< T_AlertInfo >& SxmAudioListHelper::getAlertInfoList() const
{
   return _alertInfoList;
}


/**
 * Helper function to get favorite list.
 * @return - favorite list.
 */
const ::std::vector< T_RegisteredFavorite >& SxmAudioListHelper::getRegisteredFavoriteList() const
{
   return _registeredFavoriteresult;
}


/**
 * Helper function to save Alert Info List.
 * @param [in] : alertInfoList - Alert Info List
 */
void SxmAudioListHelper::setAlertInfoList(const ::std::vector< T_AlertInfo >& alertInfoList)
{
   _alertInfoList.clear();
   for (::std::vector< T_AlertInfo >::const_iterator itr = alertInfoList.begin();
         itr != alertInfoList.end(); ++itr)
   {
      if (itr->getChannelStatus().getCHANNEL_STATUS_TUNABLE())
      {
         //display alerts which are tunable.
         _alertInfoList.push_back(*itr);
      }
   }
}


/**
 * Helper function to get favorite artist list for list widget.
 * @return - tSharedPtrDataProvider for favorite artist list.
 */
tSharedPtrDataProvider SxmAudioListHelper::getSxmFavoriteArtistSongListDataProvider(uint32 listId, const char* dataItemContext) const
{
   ETG_TRACE_USR4(("SxmAudioListHelper::GetSxmAudioFavtArtistListDataProvider listid:%d", listId));
   ListDataProviderBuilder listBuilder(listId, dataItemContext);
   uint8 favoritesize = _registeredFavoriteresult.size();
   for (uint8 favoriteindex = 0; favoriteindex < favoritesize; ++favoriteindex)
   {
      Courier::DataBindingUpdater<SxmAudioFavArtistSongTeamListDataBindingSource>* ptr
         = listBuilder.AddItem(favoriteindex).AddId(favoriteindex, 1).AddDataBindingUpdater<SxmAudioFavArtistSongTeamListDataBindingSource>();
      if (ptr != NULL)
      {
         Courier::DataItemContainer<SxmAudioFavArtistSongTeamListDataBindingSource>& item = ptr->GetValueContainer();
         (*item).mOnOrOff = !(_registeredFavoriteresult[favoriteindex].getIsEnabled());
         (*item).mFavName = _registeredFavoriteresult[favoriteindex].getContentName().c_str();
      }
   }
   tSharedPtrDataProvider dataprovider = listBuilder.CreateDataProvider();
   SXMCommonInterface::dataproviderCacheSetting(dataprovider, false);//disable cache for dynamic lists
   return dataprovider;
}


/**
 * Helper function to get replace favorite for list widget.
 * @return - tSharedPtrDataProvider for replace favorite list.
 */
tSharedPtrDataProvider SxmAudioListHelper::getSxmFavoriteReplaceListDataProvider()const
{
   ETG_TRACE_USR4(("SxmAudioListHelper::GetSxmAudioFavtListReplace"));
   ListDataProviderBuilder listBuilder(LIST_ID_SXM_AUDIO_FAVORITE_REPLACELIST, "AudioFavReplaceList");
   uint8 favoritesize = _registeredFavoriteresult.size();
   for (uint8 favoriteindex = 0; favoriteindex < favoritesize; ++favoriteindex)
   {
      listBuilder.AddItem(favoriteindex).AddData(_registeredFavoriteresult[favoriteindex].getContentName().c_str());
   }
   tSharedPtrDataProvider dataprovider = listBuilder.CreateDataProvider();
   SXMCommonInterface::dataproviderCacheSetting(dataprovider, false);
   return dataprovider;
}


/**
 * Helper function to get artist/Song alert list for list widget.
 * @return - tSharedPtrDataProvider for artist/Song alert list.
 */
tSharedPtrDataProvider SxmAudioListHelper::getSxmAlertArtistSongListDataProvider(uint32 listId, const char* dataItemContext)
{
   ETG_TRACE_USR4(("SxmAudioListHelper::getSxmAudioAlertArtistSongListDataProvider listId:%d", listId));
   ListDataProviderBuilder listBuilder(listId, dataItemContext);
   uint16 alertInfosize = _alertInfoList.size();
   for (uint16 alertindex = 0; alertindex < alertInfosize; ++alertindex)
   {
      uint16 sid = _alertInfoList[alertindex].getServiceID();
      Courier::DataBindingUpdater<SxmAudioChannelListDataBindingSource>* ptr =
         listBuilder.AddItem(alertindex).AddDataBindingUpdater<SxmAudioChannelListDataBindingSource>();
      if (ptr != NULL)
      {
         Courier::DataItemContainer<SxmAudioChannelListDataBindingSource>& item = ptr->GetValueContainer();
         (*item).mChannelNumber = SxmUtils::conCatData("i", 1, _alertInfoList[alertindex].getChannelID()).c_str();
         (*item).mArtist = _alertInfoList[alertindex].getArtistName().c_str();
         (*item).mSong = _alertInfoList[alertindex].getSongName().c_str();
         (*item).mCurrentPlay = (sid == getActiveChannelSID());
         getChannelArt(sid, true, _alertInfoList[alertindex].getChnGraphicsAvailable(), alertindex, _contentAlertlistlogo, (*item).mSxmLogo);
      }
   }
   tSharedPtrDataProvider dataprovider = listBuilder.CreateDataProvider();
   SXMCommonInterface::dataproviderCacheSetting(dataprovider, false);
   return dataprovider;
}


/*******************************************************Sports Flash********************************************************/
/**
 * Helper function to save Sports Flash logo data.
 * @param [in] : ImageData
 */
void SxmAudioListHelper::setLogoForSFAlertList(const ::std::vector<ChannelGraphicsType>& imageData)
{
   _sportsFlashChannelLogo = imageData;
}


/**
 * Helper function to save league list.
 * @param[in] : sportsLeagueList - List of leagues
 */
void SxmAudioListHelper::setSxmSportsFlashLeagueList(const ::std::vector< T_SportsFlash_League >& sportsLeagueList)
{
   _leagueList = sportsLeagueList;
}


/**
 * Helper function to get league list.
 * @return : List of leagues
 */
const ::std::vector< T_SportsFlash_League >& SxmAudioListHelper::getSxmSportsFlashLeagueList(void) const
{
   return _leagueList;
}


/**
 * Helper function to save sports Flash team list.
 * @param[in] : sportsTeamList - a list of sports flash teams.
 */
void SxmAudioListHelper::setSxmSportsFlashTeamList(const ::std::vector< T_SportsFlash_Team >& sportsTeamList)
{
   _teamList = sportsTeamList;
}


/*
 * Helper function to get sports Flash team list.
 * @return : a list of sports Flash team.
 */
const ::std::vector< T_SportsFlash_Team >& SxmAudioListHelper::getSxmSportsFlashTeamList(void) const
{
   return _teamList;
}


/**
 * Helper function to get Sport Name.
 * @param[in] : Enum for sports type
 * @return : Sport name
 */
std::string SxmAudioListHelper::getSportsName(uint32 sportType) const
{
   const char* sportlist[] =
   {
      "Football",
      "Hockey",
      "Basketball",
      "Baseball",
      "Soccer",
      "Auto Racing"
   };
   std::string lSportsType = "";
   if (sportType > T_e8_SportsFlash_Sport__SPORTS_UNKNOWN && sportType < T_e8_SportsFlash_Sport__SPORTS_OTHER)
   {
      lSportsType = sportlist[(sportType - 1)];//Sports index starts from 1.
   }
   return lSportsType;
}


/**
 * Helper function to setSxmSportsFlashFavList
 * @param[in] : sportsFavList - list of favorites
 */
void SxmAudioListHelper::setSxmSportsFlashFavoriteList(const ::std::vector< T_SportsFlash_FavoriteTeam >& sportsFavList)
{
   _sfFavoriteList = sportsFavList;
}


/**
 * Helper function to get Sports Flash favorite list.
 * @return : list of favorites.
 */
const ::std::vector< T_SportsFlash_FavoriteTeam >& SxmAudioListHelper::getSxmSportsFlashFavoriteList(void) const
{
   return _sfFavoriteList;
}


/**
 * Helper function to save sports flash alert list.
 * @param[in] : sportsAlertList - List of alerts
 */
void SxmAudioListHelper::setSxmSportsFlashAlertList(const ::std::vector< T_SportsFlash_FlashEvent >& sportsAlertList)
{
   _sfAlertList = sportsAlertList;
}


/**
 * Helper function to get sports flash alert list.
 * @return : list of sports flash alerts.
 */
const::std::vector< T_SportsFlash_FlashEvent >& SxmAudioListHelper::getSxmSportsFlashAlertList(void) const
{
   return _sfAlertList;
}


/**
 * Helper function to get Sports Flash league list for list widget.
 * @return   : tSharedPtrDataProvider - the list of league list.
 */
tSharedPtrDataProvider SxmAudioListHelper::getSxmSportsFlashLeagueListDataProvider(void)
{
   const char* const DATA_CONTEXT_LIST_SXM_LEAGUELIST = "btn_SxmSportsFlashLeagueList";
   //initializes the provider with the list id and the default data context
   ListDataProviderBuilder listBuilder(LIST_ID_SXM_SPORTSFLASH_LEAGUELIST, DATA_CONTEXT_LIST_SXM_LEAGUELIST);
   uint8 leaguesize = _leagueList.size();
   for (uint8 leagueindex = 0; leagueindex < leaguesize; ++leagueindex)
   {
      std::string  leagueName = SxmUtils::conCatData("sss", 3,
                                getSportsName(_leagueList[leagueindex].getSportType()).c_str(), " - ", _leagueList[leagueindex].getLeagueAbbrName().c_str());
      listBuilder.AddItem(leagueindex).AddData(leagueName.c_str());
   }
   return listBuilder.CreateDataProvider();
}


/**
 * Helper function to get Sports Flash Team list for list widget.
 * @return   : tSharedPtrDataProvider - the list of Team list for league.
 */
tSharedPtrDataProvider SxmAudioListHelper::getSxmSportsFlashTeamListDataProvider(void)
{
   ETG_TRACE_USR4(("SxmAudioListHelper::getSxmSportsFlashTeamListForleague"));
   const char* const DATA_CONTEXT_LIST_SXM_TEAMLIST_LEAGUE = "btn_SxmSportsFlashTeamListForLeague";
   //initializes the provider with the list id and the default data context
   ListDataProviderBuilder listBuilder(LIST_ID_SXM_SPORTSFLASH_TEAMLIST_LEAGUE, DATA_CONTEXT_LIST_SXM_TEAMLIST_LEAGUE);
   uint16 teamsize = _teamList.size();
   for (uint16 teamindex = 0; teamindex < teamsize; ++teamindex)
   {
      listBuilder.AddItem(teamindex,   // - identifies the row
                          0UL, // - unused
                          DATA_CONTEXT_LIST_SXM_TEAMLIST_LEAGUE)     // 0 - Name(Id) of the button
      .AddData(_teamList[teamindex].getTeamFullName().c_str())  // 1 - Text
      .AddData(!(_teamList[teamindex].getIsFavorite()));        // 2 - En
   }
   tSharedPtrDataProvider dataprovider = listBuilder.CreateDataProvider();
   SXMCommonInterface::dataproviderCacheSetting(dataprovider, false);
   return dataprovider;
}


/**
 * Helper function to get Replace Favorite list for list widget.
 * @return : tSharedPtrDataProvider - list of favorites.
 */
tSharedPtrDataProvider SxmAudioListHelper::getSxmSportsFlashReplaceListDataProvider(void)
{
   ETG_TRACE_USR4(("SxmAudioListHelper::getSxmSportsFlashFavListForReplace"));
   //initializes the provider with the list id and the default data context
   ListDataProviderBuilder listBuilder(LIST_ID_SXM_SPORTSFLASH_FAV_REPLACETEAM, "btn_SxmSportsFlashFavTeamList");
   uint16 teamsize = _sfFavoriteList.size();
   for (uint16 teamindex = 0; teamindex < teamsize; ++teamindex)
   {
      listBuilder.AddItem(teamindex).AddData(_sfFavoriteList[teamindex].getTeamFullName().c_str());
   }
   tSharedPtrDataProvider dataprovider = listBuilder.CreateDataProvider();
   SXMCommonInterface::dataproviderCacheSetting(dataprovider, false);
   return dataprovider;
}


/**
 * Helper function to get Sports Flash Favorite list for list widget.
 * @return : tSharedPtrDataProvider - list of favorites.
 */

tSharedPtrDataProvider SxmAudioListHelper::getSxmSportsFlashFavoriteListDataProvider(void) const
{
   ETG_TRACE_USR4(("SxmAudioListHelper::GetSxmAudioSportsfalshFavtTeamListDataProvider"));
   ListDataProviderBuilder listBuilder(LIST_ID_SXM_AUDIO_SPORTSFLASH_FAV_TEAMS, "Btn_Fav_Sports");
   dp_tclAppHmi_SxmSXMSportsFlashTeams dp_SportsFlash;
   HmiAppSxm::SXMSportsFlashFavTeams tSxmSportsFlash;
   tSxmSportsFlash = dp_SportsFlash.tGetData();
   uint8 favoritesize = _sfFavoriteList.size();
   for (uint8 favtindex = 0; favtindex < favoritesize ; ++favtindex)
   {
      Courier::DataBindingUpdater<SxmAudioFavArtistSongTeamListDataBindingSource>* ptr
         = listBuilder.AddItem(favtindex).AddId(favtindex, 1).AddDataBindingUpdater<SxmAudioFavArtistSongTeamListDataBindingSource>();
      if (ptr != NULL)
      {
         Courier::DataItemContainer<SxmAudioFavArtistSongTeamListDataBindingSource>& item = ptr->GetValueContainer();
         (*item).mFavName = _sfFavoriteList[favtindex].getTeamFullName().c_str();
         (*item).mOnOrOff = !tSxmSportsFlash.SFMonitorList[favtindex].Status;
      }
   }
   dp_SportsFlash.vSetData(tSxmSportsFlash); // To check : why datapool set.
   tSharedPtrDataProvider dataprovider = listBuilder.CreateDataProvider();
   SXMCommonInterface::dataproviderCacheSetting(dataprovider, false);
   return dataprovider;
}


/**
 * Helper function to get Replace Sports Flash Alert list for list widget.
 * @return : tSharedPtrDataProvider - list of alerts
 */
tSharedPtrDataProvider SxmAudioListHelper::getSxmSportsFlashAlertListDataProvider(void)
{
   ETG_TRACE_USR4(("SxmAudioListHelper::getSxmSportsFlashAlertList"));
   //initializes the provider with the list id and the default data context
   ListDataProviderBuilder listBuilder(LIST_ID_SXM_SPORTSFLASH_ALERT_LIST, "btn_SxmSportsFlashAlertList");
   uint16 alertsize = _sfAlertList.size();
   for (uint16 alertindex = 0; alertindex < alertsize; ++alertindex)
   {
      T_SportsFlash_Game gameInfo = _sfAlertList[alertindex].getGameInfo();
      std::string sportName = getSportsName(gameInfo.getAwayTeam().getSportType());
      std::string toptext = sportName + " - " + gameInfo.getAwayTeam().getLeagueAbbrName();
      std::string bottomtext = gameInfo.getAwayTeam().getTeamFullName() + " " + SxmUtils::safeGetCString(SXM__AUDIO_SF_VS) + " " +
                               gameInfo.getHomeTeam().getTeamFullName();
      std::string channelNo = SxmUtils::conCatData("i", 1, gameInfo.getChannelID());
      Courier::DataBindingUpdater<SxmAudioChannelListDataBindingSource>* ptr =
         listBuilder.AddItem(alertindex).AddDataBindingUpdater<SxmAudioChannelListDataBindingSource>();
      if (ptr != NULL)
      {
         Courier::DataItemContainer<SxmAudioChannelListDataBindingSource>& item = ptr->GetValueContainer();
         (*item).mChannelNumber = channelNo.c_str();
         (*item).mArtist = toptext.c_str();
         (*item).mSong = bottomtext.c_str();
         (*item).mCurrentPlay = (gameInfo.getServiceID() == getActiveChannelSID());
         getChannelArt(gameInfo.getServiceID(), true, gameInfo.getChnGraphicsAvailable(), alertindex, _sportsFlashChannelLogo, (*item).mSxmLogo);
      }
   }
   tSharedPtrDataProvider dataprovider = listBuilder.CreateDataProvider();
   SXMCommonInterface::dataproviderCacheSetting(dataprovider, false);
   return listBuilder.CreateDataProvider();
}


/*******************************************************Diagnosis********************************************************/

/**
 * Helper function to save radio id.
 * @param [in] : radioId - sxm module id.
 */
void SxmAudioListHelper::setRadioData(std::string radioId)
{
   _radioID = radioId;
}


/**
 * Helper function to save current antenna connection status.
 * @param [in] : antennastatus - antenna connection status.
 */
void SxmAudioListHelper::setDiagAntennaStatus(uint32 antennastatus)
{
   _diagnosisStmList[SXM_DIAG_ANTENNA_STATUS] = antennastatus;
}


/**
 * Helper function to save current signal quality.
 * @param [in] : signalQuality - current signal quality
 */
void SxmAudioListHelper::setDiagSignalQuality(uint32 signalQuality)
{
   _diagnosisStmList[SXM_DIAG_SIGNAL_QUALITY] = signalQuality;
   _diagnosisInfoList[SXMINFO_SIGNAL_QUALITY] = signalQuality;
}


/**
 * Helper function to save audio subscription information.
 * @param [in] : audioSubInfo - audio subscription information.
 */
void SxmAudioListHelper::setDiagAudioSubInfo(uint32 audioSubInfo)
{
   _diagnosisStmList[SXM_DIAG_AUDIO_SUBSTATUS] = audioSubInfo;
   _diagnosisInfoList[SXMINFO_AUDIO] = audioSubInfo;
}


/**
 * Helper function to save data service subscription.
 * @param [in] : dataServiceStatus - data service subscription information.
 */
void SxmAudioListHelper::setDiagDataServiceStatus(const ::std::vector< ::sxm_main_fi_types::T_SxmListElemDataSrvState >& dataServiceStatus)
{
   ETG_TRACE_USR4(("SxmAudioListHelper::setDiagDataServiceStatus"));
   bool _isTrafficSubscribeOn = false;
   uint32 serviceStatus[] =
   {
      SXM__AUDIO_DIAG_LOADING,//0 case will not happen
      SXM__AUDIO_DIAG_LOADING,
      SXM__AUDIO_DIAG_SUBSCRIBED,
      SXM__AUDIO_DIAG_AUDIONOTSUBSCRIBED,
      SXM__AUDIO_DIAG_DSNOTAVAILABLE,
      SXM__AUDIO_DIAG_LOADING,
      SXM__AUDIO_DIAG_LOADING
   };

   size_t listSize = dataServiceStatus.size();
   for (uint8 idx = 0; idx < listSize; ++idx)
   {
      T_e8_SxmDataServiceStatus dsStatus = dataServiceStatus[idx].getStatus().getStatus();
      T_e8_SxmDataServiceType dsType = dataServiceStatus[idx].getType();
      if (dsStatus <= T_e8_SxmDataServiceStatus__NO_SIGNAL)
      {
         ETG_TRACE_USR4(("SxmAudioListHelper::setDiagDataServiceStatus DStype:%d status:%d", dsType, dsStatus));

         uint32 serviceStatusMsg = serviceStatus[dsStatus];

         switch (dsType)
         {
            case T_e8_SxmDataServiceType__CHANNELART:
            {
               _diagnosisStmList[SXM_DIAG_CG_SUBSTATUS] = serviceStatusMsg;
               break;
            }
            case T_e8_SxmDataServiceType__TRAFFIC:
            {
               _diagnosisStmList[SXM_DIAG_TRAFFIC_SUBSTATUS] = serviceStatusMsg;
               _diagnosisInfoList[SXMINFO_TRAFFIC] =  serviceStatusMsg;
               if (serviceStatusMsg == SXM__AUDIO_DIAG_AUDIONOTSUBSCRIBED)
               {
                  _isTrafficSubscribeOn = true;
               }
               break;
            }
            case T_e8_SxmDataServiceType__MOVIES:
            {
               _diagnosisStmList[SXM_DIAG_MOVIES_SUBSTATUS] = serviceStatusMsg;
               break;
            }
            case T_e8_SxmDataServiceType__AGW:
            {
               _diagnosisStmList[SXM_DIAG_AGW_SUBSTATUS] = serviceStatusMsg;
               break;
            }
            case T_e8_SxmDataServiceType__TABWEATHER:
            {
               _diagnosisStmList[SXM_DIAG_TABWEATHER_SUBSTATUS] = serviceStatusMsg;
               break;
            }
            case T_e8_SxmDataServiceType__FUEL:
            {
               _diagnosisStmList[SXM_DIAG_FUEL_SUBSTATUS] = serviceStatusMsg;
               break;
            }
            case T_e8_SxmDataServiceType__STOCKS:
            {
               _diagnosisStmList[SXM_DIAG_STOCKS_SUBSTATUS] = serviceStatusMsg;
               break;
            }
            case T_e8_SxmDataServiceType__SPORTS:
            {
               _diagnosisStmList[SXM_DIAG_SPORTS_SUBSTATUS] = serviceStatusMsg;
               break;
            }
            case T_e8_SxmDataServiceType__PARKING:
            {
               _diagnosisStmList[SXM_DIAG_PARKING_SUBSTATUS] = serviceStatusMsg;
               break;
            }
            case T_e8_SxmDataServiceType__WSALERTS:
            {
               _diagnosisStmList[SXM_DIAG_WSALERTS_SUBSTATUS] = serviceStatusMsg;
               break;
            }
            default:
            {
               break;
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("SxmAudioListHelper::Invalid data service status"));
      }
   }
   setSxmTravelLink();
   setInfoTrafficStatus(_isTrafficSubscribeOn);
}


/**
 * Helper function to set Travel link subscription.
 */
void SxmAudioListHelper::setSxmTravelLink()
{
   uint32 travelLinkStatus = SXM__AUDIO_DIAG_LOADING;
   bool _isTravellinkSubscribeOn = false;
   if (_diagnosisStmList[SXM_DIAG_MOVIES_SUBSTATUS] == SXM__AUDIO_DIAG_DSSUBSCRIBED ||
         _diagnosisStmList[SXM_DIAG_AGW_SUBSTATUS] == SXM__AUDIO_DIAG_DSSUBSCRIBED ||
         _diagnosisStmList[SXM_DIAG_TABWEATHER_SUBSTATUS] == SXM__AUDIO_DIAG_DSSUBSCRIBED ||
         _diagnosisStmList[SXM_DIAG_FUEL_SUBSTATUS]  == SXM__AUDIO_DIAG_DSSUBSCRIBED ||
         _diagnosisStmList[SXM_DIAG_STOCKS_SUBSTATUS]  == SXM__AUDIO_DIAG_DSSUBSCRIBED ||
         _diagnosisStmList[SXM_DIAG_SPORTS_SUBSTATUS] == SXM__AUDIO_DIAG_DSSUBSCRIBED ||
         _diagnosisStmList[SXM_DIAG_PARKING_SUBSTATUS] == SXM__AUDIO_DIAG_DSSUBSCRIBED ||
         _diagnosisStmList[SXM_DIAG_WSALERTS_SUBSTATUS] == SXM__AUDIO_DIAG_DSSUBSCRIBED)
   {
      travelLinkStatus = SXM__AUDIO_DIAG_DSSUBSCRIBED;
   }
   else if (_diagnosisStmList[SXM_DIAG_MOVIES_SUBSTATUS] == SXM__AUDIO_DIAG_AUDIONOTSUBSCRIBED &&
            _diagnosisStmList[SXM_DIAG_AGW_SUBSTATUS] == SXM__AUDIO_DIAG_AUDIONOTSUBSCRIBED &&
            _diagnosisStmList[SXM_DIAG_TABWEATHER_SUBSTATUS] == SXM__AUDIO_DIAG_AUDIONOTSUBSCRIBED &&
            _diagnosisStmList[SXM_DIAG_FUEL_SUBSTATUS]  == SXM__AUDIO_DIAG_AUDIONOTSUBSCRIBED &&
            _diagnosisStmList[SXM_DIAG_STOCKS_SUBSTATUS]  == SXM__AUDIO_DIAG_AUDIONOTSUBSCRIBED &&
            _diagnosisStmList[SXM_DIAG_SPORTS_SUBSTATUS] == SXM__AUDIO_DIAG_AUDIONOTSUBSCRIBED &&
            _diagnosisStmList[SXM_DIAG_PARKING_SUBSTATUS] == SXM__AUDIO_DIAG_AUDIONOTSUBSCRIBED &&
            _diagnosisStmList[SXM_DIAG_WSALERTS_SUBSTATUS] == SXM__AUDIO_DIAG_AUDIONOTSUBSCRIBED)
   {
      travelLinkStatus = SXM__AUDIO_DIAG_AUDIONOTSUBSCRIBED;
      _isTravellinkSubscribeOn = true;
   }
   else if (_diagnosisStmList[SXM_DIAG_MOVIES_SUBSTATUS] == SXM__AUDIO_DIAG_DSNOTAVAILABLE &&
            _diagnosisStmList[SXM_DIAG_AGW_SUBSTATUS] == SXM__AUDIO_DIAG_DSNOTAVAILABLE &&
            _diagnosisStmList[SXM_DIAG_TABWEATHER_SUBSTATUS] == SXM__AUDIO_DIAG_DSNOTAVAILABLE &&
            _diagnosisStmList[SXM_DIAG_FUEL_SUBSTATUS]  == SXM__AUDIO_DIAG_DSNOTAVAILABLE &&
            _diagnosisStmList[SXM_DIAG_STOCKS_SUBSTATUS]  == SXM__AUDIO_DIAG_DSNOTAVAILABLE &&
            _diagnosisStmList[SXM_DIAG_SPORTS_SUBSTATUS] == SXM__AUDIO_DIAG_DSNOTAVAILABLE &&
            _diagnosisStmList[SXM_DIAG_PARKING_SUBSTATUS] == SXM__AUDIO_DIAG_DSNOTAVAILABLE &&
            _diagnosisStmList[SXM_DIAG_WSALERTS_SUBSTATUS] == SXM__AUDIO_DIAG_DSNOTAVAILABLE)
   {
      travelLinkStatus = SXM__AUDIO_DIAG_DSNOTAVAILABLE;
   }
   _diagnosisStmList[SXM_DIAG_TRAVELLINK_SUBSTATUS] = travelLinkStatus;
   _diagnosisInfoList[SXMINFO_TRAVEL_LINK] =  travelLinkStatus;

   setInfoTravellinkStatus(_isTravellinkSubscribeOn);
}


/**
 * Helper function to set Channel List Status
 */
void SxmAudioListHelper::setChannelListStatus(const ListDataInfo& listDataInfo)
{
   _channelListStatus = listDataInfo;
}


/**
 * Helper function to save DTM diagnosis data.
 * @param [in] : testModeData - DTM diagnosis data.
 */
void SxmAudioListHelper::setDiagTestModeData(const T_SxmDiagTestModeData& testModeData)
{
   const T_SxmDataSubStatus& sxmSubscriptionState = testModeData.getSxmSubscriptionState();

   ETG_TRACE_USR4(("SxmAudioListHelper::setDiagTestModeData substatus:%d", sxmSubscriptionState.getSubStatus()));
   std::string subStatus = "x";
   char iStr[25];
   snprintf(iStr, sizeof(iStr), "%02x", sxmSubscriptionState.getSubStatus());
   subStatus.append(iStr);
   //_diagnosisStmList[SXM_DIAG_MODULE_SUBSTATUS] =  subStatus;
   _diagnosisMonitorList[SXM_DIAG_SUBSTATUS] = subStatus;

   ETG_TRACE_USR4(("SxmAudioListHelper::setDiagTestModeData ReasonCode:%d", sxmSubscriptionState.getReasonCode()));
   std::string reasonCode = "x";
   char iStr2[25];
   snprintf(iStr2, sizeof(iStr2), "%02x", sxmSubscriptionState.getReasonCode());
   reasonCode.append(iStr2);
   _diagnosisMonitorList[SXM_DIAG_REASON_CODE] = reasonCode;
   _diagnosisMonitorList[SXM_DIAG_SUSPEND_DATE] = SxmUtils::getConvertedDate(sxmSubscriptionState.getSuspendedDate(), EN_SHORT_DATE_MDY,
         static_cast<short unsigned int>(SXMClockServiceInterface::GetCurrentDateFormat()));
   // _diagnosisStmList[SXM_DIAG_MODULE_SUSPENDDATE] =SxmUtils::getConvertedDate(sxmSubscriptionState.getSuspendedDate(), EN_SHORT_DATE_YMD);

   const T_SxmDetSignalQuality& sxmDetSignalQuality = testModeData.getDetSignalQuality();
   _diagnosisMonitorList[SXM_DIAG_SIGNAL_STRENGTH] = sxmDetSignalQuality.getSigQuality();
   _diagnosisMonitorList[SXM_DIAG_TUNER_STATUS] = sxmDetSignalQuality.getTunerStatus();
   _diagnosisMonitorList[SXM_DIAG_LOCKA]  = sxmDetSignalQuality.getENSALockStatus();
   _diagnosisMonitorList[SXM_DIAG_LOCKB]  = sxmDetSignalQuality.getENSBLockStatus();
   _diagnosisMonitorList[SXM_DIAG_BER]    = "S1: " + sxmDetSignalQuality.getBerS1() + " S2: " + sxmDetSignalQuality.getBerS2() +
         " T: " + sxmDetSignalQuality.getBerT();
   _diagnosisMonitorList[SXM_DIAG_CN_S1]  = "CN1A: " + sxmDetSignalQuality.getCtoNS1A() + " B: " + sxmDetSignalQuality.getCtoNS1B();
   _diagnosisMonitorList[SXM_DIAG_CN_S2]  = "CN2A: " + sxmDetSignalQuality.getCtoNS2A() + " B: " + sxmDetSignalQuality.getCtoNS2B();
   _diagnosisMonitorList[SXM_DIAG_RSERRS] = "W: " + sxmDetSignalQuality.getRSErrsWords() + " S: " + sxmDetSignalQuality.getRSErrSatSymb() +
         " T: " + sxmDetSignalQuality.getRSErrTerrSymb();
   _diagnosisMonitorList[SXM_DIAG_TUNER_FREQUENCY] = sxmDetSignalQuality.getTunerCarrFreqOfset();
   _diagnosisMonitorList[SXM_DIAG_RSSI] = sxmDetSignalQuality.getRSSI();

   const T_SxmDetOvrlSignalQuality& sxmDetOvrlSignalQuality = testModeData.getDetOvrlSignalQuality();
   _diagnosisMonitorList[SXM_DIAG_OVERLAY_STATUS] = sxmDetOvrlSignalQuality.getReceiverState();
   _diagnosisMonitorList[SXM_DIAG_OBER_S1] = "A: " + sxmDetOvrlSignalQuality.getOBerS1A() + " B: " + sxmDetOvrlSignalQuality.getOBerS1B();
   _diagnosisMonitorList[SXM_DIAG_OBER_S2] = "A: " + sxmDetOvrlSignalQuality.getOBerS2A() + " B: " + sxmDetOvrlSignalQuality.getOBerS2B();
   _diagnosisMonitorList[SXM_DIAG_OBER_T] = "A: " + sxmDetOvrlSignalQuality.getOBerTA() + " B: " + sxmDetOvrlSignalQuality.getOBerTB();
   _diagnosisMonitorList[SXM_DIAG_OTP0] = "A: " + sxmDetOvrlSignalQuality.getOBERTrbWrdErRt0A() + " B: " + sxmDetOvrlSignalQuality.getOBERTrbWrdErRt0B();
   _diagnosisMonitorList[SXM_DIAG_OTP1] = "A: " + sxmDetOvrlSignalQuality.getOBERTrbWrdErRt1A() + " B: " + sxmDetOvrlSignalQuality.getOBERTrbWrdErRt1B();
   _diagnosisMonitorList[SXM_DIAG_OTP2] = "A: " + sxmDetOvrlSignalQuality.getOBERTrbWrdErRt2A() + " B: " + sxmDetOvrlSignalQuality.getOBERTrbWrdErRt2B();

   const T_SxmVersionInfo& sxmVersionInfo = testModeData.getSxmVersionInfo();
   _diagnosisMonitorList[SXM_DIAG_MODULE_TYPE] = sxmVersionInfo.getModuleId();
   _diagnosisMonitorList[SXM_DIAG_MODULE_HW] = sxmVersionInfo.getHwRev();
   _diagnosisMonitorList[SXM_DIAG_MODULE_SW] = sxmVersionInfo.getSwRev();
   _diagnosisMonitorList[SXM_DIAG_SXI] = sxmVersionInfo.getSxiRev();
   _diagnosisMonitorList[SXM_DIAG_BB] = sxmVersionInfo.getBbRev();
   _diagnosisMonitorList[SXM_DIAG_HDEC] = sxmVersionInfo.getHDecRev();
   _diagnosisMonitorList[SXM_DIAG_RF] = sxmVersionInfo.getRFRev();
   _diagnosisMonitorList[SXM_DIAG_SPL] = sxmVersionInfo.getSPLRev();
   _diagnosisMonitorList[SXM_DIAG_CAPABILITY] = sxmVersionInfo.getCapability();
}


/**
 * Helper function to save module subscription information.
 * @param [in] : moduleSubInfo - subscription information.
 */
void SxmAudioListHelper::setModuleSubInfo(const ::sxm_audio_main_fi_types::T_SxmDataSubStatus& moduleSubInfo)
{
   ETG_TRACE_USR4(("SxmAudioListHelper::setModuleSubInfo"));
   std::string subStatus = "x";
   char iStr[25];
   snprintf(iStr, sizeof(iStr), "%02x", moduleSubInfo.getSubStatus());
   subStatus.append(iStr);
   _sxmSubStatus = subStatus ;
   _sxmSuspendDate = SxmUtils::getConvertedDate(moduleSubInfo.getSuspendedDate(), EN_SHORT_DATE_MDY,
                     static_cast<short unsigned int>(SXMClockServiceInterface::GetCurrentDateFormat()));
}


/**
 * Helper function to save sms database version.
 * @param [in] : smsDBVersion - smsdatabase version of dataservices.
 */
void SxmAudioListHelper::setDiagSmsDBVersion(const ::sxm_audio_main_fi_types::T_SxmSmsDbInfo& smsDBVersion)
{
   ETG_TRACE_USR4(("SxmAudioListHelper::setDiagSmsDBVersion"));
   _diagnosisMonitorList[SXM_DIAG_SMS_VERSION] = smsDBVersion.getSmsVersion();
   _diagnosisMonitorList[SXM_DIAG_CGS_DB] = smsDBVersion.getCgsDb();
}


/**
 * Helper function to save HMI version information.
 */
void SxmAudioListHelper::setDiagSxmVersion()
{
   ETG_TRACE_USR4(("SxmAudioListHelper::setDiagSxmVersion"));
   // initialize buffer and create object
   tChar tData[256] = {0};
   std::string huswversion ;
   dp_tclregVersionsCustomerVersion oData;
   //extract data and validate the pointer
   (tVoid)oData.u8GetData(tData, 256);
   huswversion = tData;
   _diagnosisMonitorList[SXM_DIAG_HU_SW_VERSION] = huswversion;
   std::string huhwversion = SXMDataCollectorInterface::vGetHardwareVersion();
   _diagnosisMonitorList[SXM_DIAG_HU_HW_VERSION] = huhwversion;
}


/**
 * Helper function to save diag status.
 * @param [in] : diagState - current diag state.
 */
void SxmAudioListHelper::setDiagStatus(bool diagState)
{
   _isExternalDiagmode = diagState;
}


/**
 * Helper function to save traffic status.
 * @param [in] : diagState - current traffic state.
 */
void SxmAudioListHelper::setInfoTrafficStatus(bool trafficstatus)
{
   _isSystemInfoTraffic = trafficstatus;
}


/**
 * Helper function to get traffic status.
 * return : _isSystemInfoTraffic - traffic status.
 */
bool SxmAudioListHelper::getInfoTrafficStatus()

{
   return _isSystemInfoTraffic;
}


/**
 * Helper function to save travellink status.
 * @param [in] : diagState - current travellink state.
 */
void SxmAudioListHelper::setInfoTravellinkStatus(bool travellinkstatus)
{
   _isSystemInfoTravellink = travellinkstatus;
}


/**
 * Helper function to save diag status.
 * return : _isSystemInfoTravellink - travellink status.
 */
bool SxmAudioListHelper::getInfoTravellinkStatus()
{
   return _isSystemInfoTravellink;
}


/**
 * Helper function to get DTM list elements for list widget.
 * @return - tSharedPtrDataProvider for DTM list elements.
 */
tSharedPtrDataProvider SxmAudioListHelper::getSxmDTMListDataProvider() const
{
   ETG_TRACE_USR4(("SxmAudioListHelper::getSxmDTMListDataProvider"));
   const char* const DATA_CONTEXT_LIST_SXM_DTM = "BtnList_SXM_Settings";
   const char* const DATA_CONTEXT_LIST_SXM_DTM_BUTTON = "BtnList_Options";
   //initializes the provider with the list id and the default data context
   ListDataProviderBuilder listBuilder(LIST_ID_SXM_DTM_MAIN, DATA_CONTEXT_LIST_SXM_DTM);

   listBuilder.AddItem(0, 0, DATA_CONTEXT_LIST_SXM_DTM).AddData("Clear All DB and User Data");
   listBuilder.AddItem(1, 0, DATA_CONTEXT_LIST_SXM_DTM).AddData("Clear CGS");
   listBuilder.AddItem(2, 0, DATA_CONTEXT_LIST_SXM_DTM).AddData("Reset SiriusXM Module");
   listBuilder.AddItem(3, 0, DATA_CONTEXT_LIST_SXM_DTM_BUTTON).AddData("External Diagnostic Mode").AddData(_isExternalDiagmode);
   return listBuilder.CreateDataProvider();
}


/**
 * Helper function to get Monitor Menu list elements for list widget.
 * @return - tSharedPtrDataProvider for Monitor Menu list elements.
 */
tSharedPtrDataProvider SxmAudioListHelper::getSxmMonitorMenuListDataProvider()
{
   ETG_TRACE_USR4(("SxmAudioListHelper::getSxmMonitorMenuListDataProvider"));
   //Look up table for STM Labels
   uint32 SxmStmDataLabel[Item_17] =
   {
      SXM__AUDIO_DIAG_RADIOID,
      SXM__AUDIO_DIAG_ANTENNASTATUS,
      SXM__AUDIO_DIAG_SIGNALQUALITY,
      SXM__AUDIO_DIAG_MODULESUBSCRIPTION,
      SXM__AUDIO_DIAG_MODULESUSPENDDATE,
      SXM__AUDIO_DIAG_AUDIOSUBSCRIPTION,
      SXM__AUDIO_DIAG_CHANNELARTSUBSCRIPTION,
      SXM__AUDIO_DIAG_TRAFFICSUBSCRIPTION,
      SXM__AUDIO_DIAG_TRAVELLINKSUBSCRIPTION,
      SXM__AUDIO_DIAG_MOVIESSUBSCRIPTION,
      SXM__AUDIO_DIAG_WEATHERMAPSUBSCRIPTION,
      SXM__AUDIO_DIAG_WSALERTSUBSCRIPTION,
      SXM__AUDIO_DIAG_CURRWEATHERSUBSCRIPTION,
      SXM__AUDIO_DIAG_FUELPRICESUBSCRIPTION,
      SXM__AUDIO_DIAG_PARKINGSUBSCRIPTION,
      SXM__AUDIO_DIAG_SPORTSSUBSCRIPTION,
      SXM__AUDIO_DIAG_STOCKSSUBSCRIPTION
   };

   //initializes the provider with the list id and the default data context
   const char* const DATA_CONTEXT_LIST_SXM_MONITOR_MENU = "Btn5RowSB_SXM";
   ListDataProviderBuilder listBuilder(LIST_ID_SXM_MONITOR, DATA_CONTEXT_LIST_SXM_MONITOR_MENU);
   uint8 idx = 0;
   for (idx = SXM_DIAG_RADIOID; idx <= SXM_DIAG_CG_SUBSTATUS; ++idx)
   {
      //if (!(_isDAVariant && (idx >= SXM_DIAG_TRAFFIC_SUBSTATUS)))
      //{
      if (SxmStmDataLabel[idx] == SXM__AUDIO_DIAG_RADIOID)
      {
         listBuilder.AddItem(idx, 0, DATA_CONTEXT_LIST_SXM_MONITOR_MENU).AddData(SxmUtils::safeGetCString(SxmStmDataLabel[idx]).append(":").c_str()).AddData(_radioID.c_str());
      }
      else if (SxmStmDataLabel[idx] == SXM__AUDIO_DIAG_MODULESUBSCRIPTION)
      {
         listBuilder.AddItem(idx, 0, DATA_CONTEXT_LIST_SXM_MONITOR_MENU).AddData(SxmUtils::safeGetCString(SxmStmDataLabel[idx]).append(":").c_str()).AddData(_sxmSubStatus.c_str());
      }
      else if (SxmStmDataLabel[idx] == SXM__AUDIO_DIAG_MODULESUSPENDDATE)
      {
         listBuilder.AddItem(idx, 0, DATA_CONTEXT_LIST_SXM_MONITOR_MENU).AddData(SxmUtils::safeGetCString(SxmStmDataLabel[idx]).append(":").c_str()).AddData(_sxmSuspendDate.c_str());
      }
      else
      {
         listBuilder.AddItem(idx, 0, DATA_CONTEXT_LIST_SXM_MONITOR_MENU).AddData(SxmUtils::safeGetCString(SxmStmDataLabel[idx]).append(":").c_str()).AddData(SxmUtils::safeGetCString(_diagnosisStmList[idx]).c_str());
      }
      //}
   }
   tSharedPtrDataProvider dataprovider = listBuilder.CreateDataProvider();
   SXMCommonInterface::dataproviderCacheSetting(dataprovider, false);//disable cache for dynamic lists
   return dataprovider;
}


/**
 * Helper function to get Testmode monitor list elements for list widget.
 * @return - tSharedPtrDataProvider for Testmode monitor list elements.
 */
tSharedPtrDataProvider SxmAudioListHelper::getSxmTestModeMonitorSelection()
{
   ETG_TRACE_USR4(("SxmAudioListHelper::getSxmTestModeMonitorSelection"));

   //Look up table for Dtm Labels
   const char* SxmDtmDataLabel[] =
   {
      "SIGNAL:",
      "Tuner Status:",
      "Lock A:",
      "Lock B:",
      "BER:",
      "C/N S1:",
      "C/N S2:",
      "RS:",
      "Tuner Freq:",
      "RSSI:",
      "Overlay Status:",
      "OBER S1:",
      "OBER S2:",
      "OBER T:",
      "TWER P0:",
      "TWER P1:",
      "TWER P2:",
      "Module Type:",
      "Module HW:",
      "Module SW:",
      "SXI:",
      "BB:",
      "HDEC:",
      "RF:",
      "SPL:",
      "Capability:",
      "Sub Status:",
      "Reason Code:",
      "Suspend Date:",
      "SMS Version:",
      "H/U SW Version:",
      "H/U HW Version:",
      "CGS DB:",
   };
   ListDataProviderBuilder listBuilder(LIST_ID_SXM_DTM_TESTMODE_MONITOR_SELECTION, "MonitorSelection");
   for (uint8 idx = SXM_DIAG_SIGNAL_STRENGTH; idx <= SXM_DIAG_CGS_DB; ++idx)
   {
      listBuilder.AddItem(idx).AddData(SxmDtmDataLabel[idx]).AddData(_diagnosisMonitorList[idx].c_str());
   }

   tSharedPtrDataProvider dataprovider = listBuilder.CreateDataProvider();
   SXMCommonInterface::dataproviderCacheSetting(dataprovider, false);//disable cache for dynamic lists
   return dataprovider;
}


/**
 * Helper function to get System Info list elements for list widget.
 * @return - tSharedPtrDataProvider System Info DTM list elements.
 */
tSharedPtrDataProvider SxmAudioListHelper::getSxmInfoListDataProvider()
{
   ETG_TRACE_USR4(("SxmAudioListHelper::getSxmInfoListDataProvider"));

   uint32 SxmInfoDataLabel[] =
   {
      SXM__AUDIO_DIAG_INFOSIGNALQUALITY,
      SXM__AUDIO_DIAG_SXMAUDIO,
      SXM__AUDIO_DIAG_SXMTRAFFIC,
      SXM__AUDIO_DIAG_SXMTRAVELLINK,
      SXM__AUDIO_DIAG_SXMRADIOID
   };

   ListDataProviderBuilder listBuilder(LIST_ID_SXM_AUDIO_SXM_INFOLIST, "SxmInfo");
   // bool lButtonstate = false;
   for (uint8 idx = SXMINFO_SIGNAL_QUALITY; idx <= SXMINFO_RADIO_ID; ++idx)
   {
      if (!(_isDAVariant && (idx == SXMINFO_TRAFFIC || idx == SXMINFO_TRAVEL_LINK)))
      {
         if (SxmInfoDataLabel[idx] == SXM__AUDIO_DIAG_SXMRADIOID)
         {
            listBuilder.AddItem(idx).AddData(SxmUtils::safeGetCString(SxmInfoDataLabel[idx]).c_str()).AddData(_radioID.c_str());
         }
         else
         {
            listBuilder.AddItem(idx).AddData(SxmUtils::safeGetCString(SxmInfoDataLabel[idx]).c_str()).AddData(SxmUtils::safeGetCString(_diagnosisInfoList[idx]).c_str());
         }
      }
   }

   tSharedPtrDataProvider dataprovider = listBuilder.CreateDataProvider();
   SXMCommonInterface::dataproviderCacheSetting(dataprovider, false);//disable cache for dynamic lists
   return dataprovider;
}


/**
 * Helper function to update Sxm setting list in DTM.
 * @param [in] : listId - list to be refreshed.
 * @param [in] : sceneId - applicable scene.
 */
void SxmAudioListHelper::refreshSxmAudioList(unsigned int listId, unsigned int sceneId) const
{
   ETG_TRACE_USR4(("SxmAudioListHelper::refreshSxmAudioList"));
   tSharedPtrDataProvider dataProvider = tSharedPtrDataProvider();
   if (SXMCommonInterface::isViewVisible(sceneId))
   {
      switch (listId)
      {
         case LIST_ID_SXM_DTM_MAIN:
         {
            dataProvider  =  SxmAudioListHelper::instance()->getSxmDTMListDataProvider();
            break;
         }
         case LIST_ID_SXM_MONITOR:
         {
            dataProvider = SxmAudioListHelper::instance()->getSxmMonitorMenuListDataProvider();
            break;
         }
         case LIST_ID_SXM_DTM_TESTMODE_MONITOR_SELECTION:
         {
            dataProvider = SxmAudioListHelper::instance()->getSxmTestModeMonitorSelection();
            break;
         }
         case LIST_ID_SXM_AUDIO_SXM_INFOLIST:
         {
            dataProvider = SxmAudioListHelper::instance()->getSxmInfoListDataProvider();
            break;
         }
         case LIST_ID_SXM_AUDIO_MENU:
         {
            dataProvider = SxmAudioListHelper::instance()->getSxmAudioMenuListDataProvider();
            break;
         }
         default:
         {
            break;
         }
      }
      POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
   }
}


/*******************************************************Common********************************************************/
/**
 *getSxmAudioAlertSettingListDataProvider -
 * sends the ListDataProvider to the FlexList Widget
 * @return - tSharedPtrDataProvider
 */
tSharedPtrDataProvider SxmAudioListHelper::getSxmAudioAlertSettingListDataProvider()const
{
   ETG_TRACE_USR4(("SxmAudioListHelper::GetSxmAudioAlertSongListDataProvider"));
   ListDataProviderBuilder listBuilder(LIST_ID_SXM_AUDIO_ALERTLIST, "AlertSettings");
   uint32 AlertList[Item_4] =
   {
      SXM__AUDIO_SETTINGS_ARTISTALERTS,
      SXM__AUDIO_SETTINGS_SONGALERTS,
      SXM__AUDIO_SETTINGS_SFALERTS,
      //SXM__AUDIO_SETTINGS_NONAUDIOALERTS
      SXM__AUDIO_SETTINGS_COCKPITALERTS
      //SXM__AUDIO_SETTINGS_CABIN_A_ALERTS,
      //SXM__AUDIO_SETTINGS_CABIN_B_ALERTS
   };
   dp_tclAppHmi_SxmSXMAudioAlert dp_AudioAlert;
   HmiAppSxm::AlertSetting tSXMAudioAlert = dp_AudioAlert.tGetData();
   for (tU8 u8Index = 0; u8Index < Item_4 ; ++u8Index)
   {
      Courier::DataBindingUpdater<SxmAudioAlertSettingDataBindingSource>* ptr
         = listBuilder.AddItem(u8Index).AddDataBindingUpdater<SxmAudioAlertSettingDataBindingSource>();
      if (ptr != NULL)
      {
         Courier::DataItemContainer<SxmAudioAlertSettingDataBindingSource>& item = ptr->GetValueContainer();
         ETG_TRACE_USR4(("SxmAudioListHelper::GetSxmAudioAlertSongListDataProvider AlertSetting value = %d", tSXMAudioAlert.OnOrOff[u8Index]));
         (*item).mOnOrOff = !(tSXMAudioAlert.OnOrOff[u8Index]);
         (*item).mAlertText = Candera::String(AlertList[u8Index]);
      }
   }
   return listBuilder.CreateDataProvider();
}


/**
 * Helper function to clear member variables
 */
void SxmAudioListHelper::clearListVectors(void)
{
   if (!_presetList.empty())
   {
      updatePresetList(false);
   }
}


/**
 * Helper function to get channel art.
 */
void SxmAudioListHelper::getChannelArt(uint16 dataSID, bool isChannelAvail, T_e8_Graphics graphicsAvailibility, uint32 cgIndex,
                                       const ::std::vector< ChannelGraphicsType >& cgList, Candera::MemoryManagement::SharedPointer<Candera::Image2D>& img)
{
   ETG_TRACE_USR4(("SxmAudioListHelper::channel logo list index:%d,size:%d", cgIndex, cgList.size()));
   if (cgIndex < cgList.size() && !cgList[cgIndex].u8ImageData.empty())
   {
      if (cgList[cgIndex].ServiceID == dataSID && isChannelAvail && graphicsAvailibility != T_e8_Graphics__GRAPHICS_NEEDS_DEFAULT)
      {
         Candera::Bitmap::SharedPointer bmp = ImageLoader::loadBitmapData(const_cast<uint8*>(& (cgList[cgIndex].u8ImageData)[0]), cgList[cgIndex].u32Size);
         if (bmp != NULL)
         {
            img = ImageLoader::createImage(bmp);
         }
      }
      else
      {
         ETG_TRACE_ERR(("SxmAudioListHelper::getChannelArt Service ID mismatch or listindex is not valid"));
         //img = ImageLoader::getAssetBitmapImage("Sxm#Images#Landing#Layer_Default_channel_logoBmp");
         img = ImageLoader::getAssetBitmapImage("Sxm#Images#Landing#Layer_Default_channel_logoBmp");
      }
   }
   else
   {
      ETG_TRACE_ERR(("SxmAudioListHandler::getChannelArt cgList size is less than index or no image data"));
      //img = ImageLoader::getAssetBitmapImage("Bitmaps##AIVIResources#SXM#SXM_ChannelArt");
      img = ImageLoader::getAssetBitmapImage("Sxm#Images#Landing#Layer_Default_channel_logoBmp");
      if (img != NULL)
      {
         ETG_TRACE_ERR(("ImageLoader::getAssetBitmapImage(Sxm#Images#Landing#Layer_Default_channel_logoBmp) data present"));
      }
      else
      {
         ETG_TRACE_ERR(("ImageLoader::getAssetBitmapImage(Sxm#Images#Landing#Layer_Default_channel_logoBmp) data not present"));
      }
   }
}


/**
 * Notifies Service entry/exit status
 * @param [in] : EntryExitStatus
 */
void SxmAudioListHelper::vNotifyServiceStatus(unsigned short int /*lEntryExitStatus*/, unsigned int /*serviceType*/)
{
}


/**
 * Notifies Application state. Based on state, data cleared
 * @param [in] : appState
 */
void SxmAudioListHelper::vNotifyAppState(unsigned short int appState)
{
   if (hmibase::IN_FOREGROUND != appState)
   {
      _channelList.clear();
      _categoryList.clear();
      _chanllistlogo.clear();
      _customizedChannelList.clear();
      _customChanllistlogo.clear();
   }
}


/**
 * Helper function to initialize diagnosis map elements.
 */
void SxmAudioListHelper::initializeDiagMap()
{
   for (uint8 idx = SXM_DIAG_AUDIO_SUBSTATUS; idx <= SXM_DIAG_STOCKS_SUBSTATUS; ++idx)
   {
      _diagnosisStmList[idx] = SXM__AUDIO_DIAG_LOADING;
   }
   _diagnosisInfoList[SXMINFO_TRAVEL_LINK] = SXM__AUDIO_DIAG_LOADING;
   _diagnosisInfoList[SXMINFO_TRAFFIC] = SXM__AUDIO_DIAG_LOADING;
   _diagnosisInfoList[SXMINFO_AUDIO] = SXM__AUDIO_DIAG_LOADING;
}


}  // namespace Core
}  // namespace App
