/**************************************************************************************
 * @file         : SxmAudioList.h
 * @addtogroup   : AppHmi_Sxm
 * @brief        : Utility class which contains all the necessary business logic required to handle all Audio Lists.
 * @copyright    : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 **************************************************************************************/
#ifndef _SxmAudioList_H_
#define _SxmAudioList_H_

#include "sxm_audio_main_fiProxy.h"
#include "Common/DataModel/ListRegistry.h"
#include "AppHmi_SxmMessages.h"
#include "SxmAudioTypes.h"
#include "ISxmAudioChannelArtResponse.h"
#include "ISxmAudioChannelArtRequest.h"

namespace App {
namespace Core {

/**
 * Utility class which contains all the necessary business logic required to handle all Audio Lists.
 * @class : SxmAudioList
 */
class SxmAudioList
   : public ::sxm_audio_main_fi::CategoryListStatusCallbackIF
   , public ::sxm_audio_main_fi::GetCategoryListCallbackIF
   , public ::sxm_audio_main_fi::GetChannelListCallbackIF
   , public ::sxm_audio_main_fi::SkipChannelCallbackIF
   , public ::sxm_audio_main_fi::SetCategoryMonitorCallbackIF
   , public ListImplementation
   , public ISxmAudioChannelArtResponse
{
   public:
      /**
       * SxmAudioList Class constructor
       */
      SxmAudioList(::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy > pSxmAudioFiProxy, ISxmAudioChannelArtRequest* pSxmChannelartFiProxy);
      /**
       * SxmAudioList Class destructor
       */
      virtual ~SxmAudioList();
      /**
       * Utility function that registers for notifications updates from SXM Audio MIDW layer
       */
      void registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/);
      /**
       * Utility function that deregisters for notifications updates from SXM Audio MIDW layer
       */
      void deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/);
      //****************************Middleware Callbacks******************************//
      /**
       * Error handling function for category list status property
       */
      void onCategoryListStatusError(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
                                     const boost::shared_ptr< ::sxm_audio_main_fi::CategoryListStatusError >& /*error*/);
      /**
       * Status handling function for category list status property
       */
      void onCategoryListStatusStatus(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
                                      const boost::shared_ptr< ::sxm_audio_main_fi::CategoryListStatusStatus >& /*status*/);
      /**
       * Error handling function for get category list method
       */
      void onGetCategoryListError(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
                                  const boost::shared_ptr< ::sxm_audio_main_fi::GetCategoryListError >& /*error*/);
      /**
       * Result handling function for get category list method
       */
      void onGetCategoryListResult(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
                                   const boost::shared_ptr< ::sxm_audio_main_fi::GetCategoryListResult >& result);
      /**
       * Error handling function for get channel list method
       */
      void onGetChannelListError(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
                                 const boost::shared_ptr< ::sxm_audio_main_fi::GetChannelListError >& /*error*/);
      /**
       * Result handling function for get channel list method
       */
      void onGetChannelListResult(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
                                  const boost::shared_ptr< ::sxm_audio_main_fi::GetChannelListResult >& /*result*/);
      /**
       * Error handling function for get channel list method
       */
      void onSkipChannelError(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& proxy,
                              const ::boost::shared_ptr< ::sxm_audio_main_fi::SkipChannelError >& error);
      /**
       * Result handling function for skipped channel
       */
      void onSkipChannelResult(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& proxy,
                               const ::boost::shared_ptr< ::sxm_audio_main_fi::SkipChannelResult >& result);
      /**
      * Error handling function for CategoryMonitor method
      */
      void onSetCategoryMonitorError(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& proxy,
                                     const ::boost::shared_ptr< ::sxm_audio_main_fi::SetCategoryMonitorError >& error);
      /**
       * Result handling function for CategoryMonitor method
       */
      void onSetCategoryMonitorResult(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& proxy,
                                      const ::boost::shared_ptr< ::sxm_audio_main_fi::SetCategoryMonitorResult >& result);
      /**
       * Result handling function for channel logo method
       */
      void onGetChannelLogoResult(const uint16& /*sid*/, const uint8& /*ChannelGraphicsType*/, const std::vector<uint8>& /*u8ImageData*/,
                                  const std::vector<uint8>& /*albumImageData*/) {}
      /**
       * Result handling function for channel art list method
       */
      void onGetChannelArtListResult(const ::std::vector< ChannelGraphicsType >& channelArtList);
      /**
       * Gets the ListDataProvider from the corresponding listdataprovider functions based on the List ID.
       */
      tSharedPtrDataProvider getListDataProvider(const ListDateProviderReqMsg& listDataInfo);
      /**
      * Error handling function for GetChannelArtList method.
      */
      virtual void onGetChannelArtListError();
      /**
       * Result handling function for GetChannelArt method.
       */
      virtual void onGetChannelLogoError(const uint16& /*sid*/) {}
      /**
       * Function to set default value on service availability from suspended
       */
      void onServiceResumed(const ::boost::shared_ptr< asf::core::Proxy >& /*proxy*/);
      /*
      * Helper Function to trigger handling for TTFis Command SxmAudio_GetChannelArt
      */
      void vGetChannelArt(uint8) const;

   private:
      //************************Courier Message callbacks**********//
      /**
      * Callback to handle Courier Message SxmAudioCategories Request
      */
      bool onCourierMessage(const SxmAudioCategoriesReqMsg& /*oMsg*/);
      /**
      * Callback to handle Courier Message SXMAudioChannelListReqMsg Request
      */
      bool onCourierMessage(const SXMAudioChannelListReqMsg& oMsg);
      /**
      * Callback to handle Courier Message SXMAudioCustomizeListReqMsg Request
      */
      bool onCourierMessage(const SXMAudioCustomizeListReqMsg& oMsg);
      /**
       * Callback to handle Courier Message SXMAudioCustomizeControlReqMsg Request
       */
      bool onCourierMessage(const SXMAudioCustomizeClearSelectReqMsg& oMsg);
      /**
       * Callback to handle Courier Message SXMAudioCustomizeControlReqMsg Request
       */
      bool onCourierMessage(const SXMAudioCustomizeControlReqMsg& oMsg);
      /*
       * Callback to handle Courier Message SxmDTMTestModeMonitorSelReqMsg
       */
      bool onCourierMessage(const SxmDTMTestModeMonitorSelReqMsg& oMsg) const;
      /**
       * Callback to handle Courier Message SXMAudioMonitorUpdatesMsg Request
       */
      bool onCourierMessage(const SXMAudioMonitorUpdatesMsg& /*oMsg*/);
      /**
      * Helper function to get current status of service.
      */
      bool onCourierMessage(const SxmActiveRenderedViewMsg& oMsg);
      /**
      * Helper function to get current status of list.
      */
      bool onCourierMessage(const ListStatusUpdMsg& oMsg);
      bool onCourierMessage(const SxmStringBoundaryResMsg& oMsg);
      bool onCourierMessage(const ListDateProviderReqMsg& oMsg);

      /*
      * Helper Function to send monitor update method start to Midw
      */
      void sendMonitorUpdateMsg(uint32 listId, uint8 enable, bool resetliststatus);
      /**
       * Courier Message Handler
       */
      COURIER_MSG_MAP_BEGIN(TR_CLASS_APPHMI_SXM_COURIER_PAYLOAD_MODEL_COMP)
      ON_COURIER_MESSAGE(ListDateProviderReqMsg)
      ON_COURIER_MESSAGE(SxmAudioCategoriesReqMsg)
      ON_COURIER_MESSAGE(SXMAudioChannelListReqMsg)
      ON_COURIER_MESSAGE(SXMAudioCustomizeListReqMsg)
      ON_COURIER_MESSAGE(SXMAudioCustomizeClearSelectReqMsg)
      ON_COURIER_MESSAGE(SXMAudioCustomizeControlReqMsg)
      ON_COURIER_MESSAGE(SXMAudioMonitorUpdatesMsg)
      ON_COURIER_MESSAGE(SxmActiveRenderedViewMsg)
      ON_COURIER_MESSAGE(ListStatusUpdMsg)
      ON_COURIER_MESSAGE(SxmStringBoundaryResMsg)
      COURIER_MSG_MAP_DELEGATE_START()
      COURIER_MSG_DELEGATE_TO_CLASS(ListImplementation)
      COURIER_MSG_MAP_DELEGATE_DEF_END()
      COURIER_MSG_MAP_DELEGATE_END()

      //Member Functions
      /**
       * Send Midw Customize channel list request.
       */
      void getCustomizeChannelList();
      void updateListDataInfo(const ListDataInfo& listDataInfo);
      void setChannelListMonitoring(bool status);
      void getChannelArtListRequest(uint32 listId);
      void refreshChannelList(const ::std::vector< ::sxm_audio_main_fi_types::T_CategoryListEntry >& categorylist);
      //Member Variable
      ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy > _sxmAudioFiProxy;
      ISxmAudioChannelArtRequest* _sxmChannelart;
      bool _custmClearAllorOne;
      bool _isSkip;
      bool _refreshChannelArt;
      bool _isCategoryMonitorStarted;
      bool _resetStartIndex;
      bool _isPreviousUpdateProcessed;
      bool _iswaitAnimation;
      uint16 _monitoredCategoryId;
      uint16 _customSelectedIndex;
      uint32 _tranisitonId;
      std::map<uint32, ListDataInfo>_currentListStatus;//Map to update list info
      const uint8 _ALLCATEGORIES;

   protected:
};


}  // namespace Core
}  // namespace App

#endif
