/**************************************************************************************
 * @file         : SxmAudioList.cpp
 * @addtogroup   : AppHmi_Sxm
 * @brief        : Utility class which contains all the necessary business logic required to handle all Audio Lists.
 * @copyright    : (c) 2019-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 **************************************************************************************/
#include "hall_std_if.h"   // Including HALL framework code
#include "SxmAudioList.h"
#include "SxmAudioListHelper.h"
#include "SxmAudioDataBindingUtils.h"
#include "App/Core/ClientHandler/DataCollector/SXMDataCollectorInterface.h"
#include "App/Core/SxmCommon/SXMCommonInterface.h"
#include "App/Core/SxmUtils/SxmUtils.h"
#include "HmiTranslation_TextIds.h"
#include "App/Core/LanguageDefines.h"
#include "App/Core/SxmUtils/SxmUtilsListHandler.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS       TR_CLASS_APPHMI_SXMAUDIO
#define ETG_I_TRACE_CHANNEL           TR_TTFIS_APPHMI_SXM
#define ETG_I_TTFIS_CMD_PREFIX        "APPHMI_SXM_"
#define ETG_I_FILE_PREFIX             App::Core::SxmAudioList::
#include "trcGenProj/Header/SxmAudioList.cpp.trc.h"
#endif

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_hmi_06_if.h"

namespace App {
namespace Core {
//using namespace ::App::Core;
using namespace ::sxm_audio_main_fi;
using namespace ::sxm_audio_main_fi_types;
using namespace Util;
/**
 * SxmAudioList Class constructor
 * @param [in] : pSxmAudioFiProxy - SxmAudioFiProxy class instance
 */
SxmAudioList::SxmAudioList(::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy > pSxmAudioFiProxy,
                           ISxmAudioChannelArtRequest* pSxmChannelartFiProxy)
   : _sxmAudioFiProxy(pSxmAudioFiProxy)
   , _sxmChannelart(pSxmChannelartFiProxy)
   , _custmClearAllorOne(false)
   , _isSkip(false)
   , _refreshChannelArt(false)
   , _isCategoryMonitorStarted(false)
   , _resetStartIndex(false)
   , _isPreviousUpdateProcessed(true)
   , _iswaitAnimation(false)
   , _monitoredCategoryId(0)
   , _customSelectedIndex(0)
   , _tranisitonId(0)
   , _ALLCATEGORIES(0)
{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR4(("SxmAudioList() Constructor"));
//#include "//Common/ListHandler/LockoutRestrictionData.h"
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_SXM_CATEGORIES, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_SXM_AUDIO_CHANNELLIST, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_SXM_AUDIO_MENU, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_SXM_AUDIO_CUSTOMIZE_LIST, this);
   //ListRegistry::s_getInstance().addListImplementation(LIST_ID_SXM_PRESET_BANK, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_SXM_DTM_MAIN, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_SXM_MONITOR, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_SXM_AUDIO_TWN_ENTRY, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_SXM_DTM_TESTMODE_MONITOR_SELECTION, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_SXM_AUDIO_SXM_INFOLIST, this);
   _currentListStatus.clear();
   ListDataInfo custmizelist = {LIST_ID_SXM_AUDIO_CUSTOMIZE_LIST, 0, 20};
   ListDataInfo channellist = {LIST_ID_SXM_AUDIO_CHANNELLIST, 0, 20};
   _currentListStatus[LIST_ID_SXM_AUDIO_CUSTOMIZE_LIST] = custmizelist;
   _currentListStatus[LIST_ID_SXM_AUDIO_CHANNELLIST] = channellist;
}


/**
 * SxmAudioList Class destructor
 */
SxmAudioList::~SxmAudioList()
{
   ETG_TRACE_USR4(("~SxmAudioList() Destructor"));
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_SXM_CATEGORIES);
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_SXM_AUDIO_CHANNELLIST);
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_SXM_AUDIO_MENU);
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_SXM_AUDIO_CUSTOMIZE_LIST);
   //ListRegistry::s_getInstance().removeListImplementation(LIST_ID_SXM_PRESET_BANK);
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_SXM_DTM_MAIN);
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_SXM_MONITOR);
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_SXM_AUDIO_TWN_ENTRY);
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_SXM_DTM_TESTMODE_MONITOR_SELECTION);
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_SXM_AUDIO_SXM_INFOLIST);
   ETG_I_UNREGISTER_FILE();
   _currentListStatus.clear();
   _sxmChannelart = NULL;
   _sxmAudioFiProxy.reset();
   _custmClearAllorOne = false;
   _isSkip = false;
   _refreshChannelArt = false;
   _isCategoryMonitorStarted = false;
   _resetStartIndex = false;
   _isPreviousUpdateProcessed = true;
   _iswaitAnimation = false;
   _monitoredCategoryId = 0;
   _customSelectedIndex = 0;
   _tranisitonId = 0;
}


/**
 * Utility function that registers for notifications updates from SXM Audio MIDW layer
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmAudioList::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SxmAudioList::registerProperties"));
   if ((proxy == _sxmAudioFiProxy) && (_sxmAudioFiProxy))
   {
      _sxmAudioFiProxy->sendCategoryListStatusUpReg(*this);
   }
}


/**
 * Utility function that deregisters for notifications updates from SXM Audio MIDW layer
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmAudioList::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SxmAudioList::deregisterProperties"));
   if ((proxy == _sxmAudioFiProxy) && (_sxmAudioFiProxy))
   {
      _sxmAudioFiProxy->sendCategoryListStatusRelUpRegAll();
   }
}


/**
 * Error handling function for category list status property
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error Error type
 */
void SxmAudioList::onCategoryListStatusError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< CategoryListStatusError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioList::onCategoryListStatusError"));
}


/**
 * Status handling function for category list status property
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : status - Property status info
 */
void SxmAudioList::onCategoryListStatusStatus(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< CategoryListStatusStatus >& /*status*/)
{
   ETG_TRACE_USR4(("SxmAudioList::onCategoryListStatusStatus"));
   _isPreviousUpdateProcessed = false;
   //While in Channel list request for data based on active category.
   if (_isCategoryMonitorStarted && _sxmAudioFiProxy)
   {
      _sxmAudioFiProxy->sendGetCategoryListStart(*this, _monitoredCategoryId);
      _isPreviousUpdateProcessed = true;
   }
}


/**
 * Error handling function for get category list method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error - Error type
 */
void SxmAudioList::onGetCategoryListError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< GetCategoryListError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioList::onGetCategoryListError"));
   _isPreviousUpdateProcessed = true;
}


/**
 * Result handling function for get category list method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : result - Method result data
 */
void SxmAudioList::onGetCategoryListResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< GetCategoryListResult >& result)
{
   ETG_TRACE_USR4(("SxmAudioList::onGetCategoryListResult size = %d, CatType = %d ", result->getCategoryList().size(), result->getCategoryType()));
   uint32 viewID = SXMCommonInterface::getActiveViewName(Enum_SURFACEID_MAIN_SURFACE_SXM);
   uint32 u32CatType = result->getCategoryType();

   ETG_TRACE_USR4(("SxmAudioList::onGetCategoryListResult viewID:%u, _tranisitonId = %u _isCategoryMonitorStarted = %d",
                   viewID, _tranisitonId, _isCategoryMonitorStarted));

   if ((Sxm_Scenes_SXM_RADIO__CATEGORYLIST == _tranisitonId) && (_ALLCATEGORIES == u32CatType))
   {
      //To handle Transition to category screen.
      SxmAudioListHelper::instance()->setCategoryList(result->getCategoryList());
      POST_MSG((COURIER_MESSAGE_NEW(SxmSceneTransitionReqMsg)(_tranisitonId)));
      _tranisitonId = 0; //Clear after transition.
   }
   else if (_isCategoryMonitorStarted && (Sxm_Scenes_SXM_RADIO__CATEGORYLIST == viewID))
   {
      //To handle dynamic updates while in Category list.
      SxmAudioListHelper::instance()->setCategoryList(result->getCategoryList());
      tSharedPtrDataProvider dataProvider = SxmAudioListHelper::instance()->getSxmCategorieListDataProvider();
      POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
      if (SxmAudioListHelper::instance()->getCategoryList().size() != result->getCategoryList().size())
      {
         //reset to top of list if list size is changed.
         SxmUtilsListHandler::instance()->SendListChangeMsg(LIST_ID_SXM_CATEGORIES, ListChangeSet, 0);
      }
   }
   else if ((_isCategoryMonitorStarted && (Sxm_Scenes_SXM_RADIO__CHANNELLIST_3 == viewID)) || (Sxm_Scenes_SXM_RADIO__CHANNELLIST_3 == _tranisitonId))
   {
      //To handle dynamic updates while in Channel list.
      refreshChannelList(result->getCategoryList());
   }
   else if (_isCategoryMonitorStarted && (Sxm_Scenes_SXM_RADIO__CUSTOMIZE_CHANNELLIST == viewID))
   {
      //To handle dynamic updates in Customize channel list.
      getCustomizeChannelList();
      //_refreshChannelArt = true;
   }
}


/**
 * Error handling function for get channel list method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error - Error type
 */
void SxmAudioList::onGetChannelListError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/, const boost::shared_ptr< GetChannelListError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioList::onGetChannelListError"));
}


/**
 * Result handling function for get channel list method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : result - Method result data
 */
void SxmAudioList::onGetChannelListResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/, const boost::shared_ptr< GetChannelListResult >& result)
{
   ETG_TRACE_USR4(("SxmAudioList::onGetChannelListResult"));
   if (T_e8_MethodStatus__XMTUN_MS_SUCCESS == result->getStatus())
   {
      //To "Enable/Disable" button.
      SxmAudioDataBindingUtils::instance()->setSxmAudioChannelSettingEnableorDisable(result->getChannelList().size());

      //On update if size of list is changed refetch channel art list and reset list index to top of list.
      const ::std::vector<T_ChannelListEntry >& lChannelList = result->getChannelList();
      if (result->getChannelList().size() != SxmAudioListHelper::instance()->getCustomizedChannelList().size() + 2) //2: Includes radio id  & preview channel.
      {
         _refreshChannelArt = true;
         _resetStartIndex = true;
         _currentListStatus[LIST_ID_SXM_AUDIO_CUSTOMIZE_LIST].startIndex = 0;
         _currentListStatus[LIST_ID_SXM_AUDIO_CUSTOMIZE_LIST].windowSize = 20;
      }
      bool status = SxmAudioListHelper::instance()->setSxmCustomizedChannelList(lChannelList);
      //To update "Enable All Channels/Disable All channels" button status
      SxmAudioDataBindingUtils::instance()->setSxmCustomizeChannelListData(status);
      //refresh channel art.
      if (_refreshChannelArt)
      {
         getChannelArtListRequest(LIST_ID_SXM_AUDIO_CUSTOMIZE_LIST);
         _refreshChannelArt = false;
      }
      else
      {
         tSharedPtrDataProvider dataProvider =
            SxmAudioListHelper::instance()->getSxmCustomizeChannelListDataProvider(_currentListStatus[LIST_ID_SXM_AUDIO_CUSTOMIZE_LIST].startIndex,
                  _currentListStatus[LIST_ID_SXM_AUDIO_CUSTOMIZE_LIST].windowSize);
         POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
      }
      if (_iswaitAnimation)
      {
         POST_MSG((COURIER_MESSAGE_NEW(SxmWaitAnimationReqMsg)(ANIMATION_STOP)));
         _iswaitAnimation = false;
      }
   }
}


/**
 * Error handling function for get channel list method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error - Error type
 */
void SxmAudioList::onSkipChannelError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/, const ::boost::shared_ptr<SkipChannelError >& /*error*/)
{
   ETG_TRACE_USR4(("SxmAudioList::onSkipChannelError"));
}


/**
 * Result handling function for skipped channel
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : result - Method result data
 */
void SxmAudioList::onSkipChannelResult(const ::boost::shared_ptr<Sxm_audio_main_fiProxy >& /*proxy*/, const ::boost::shared_ptr<SkipChannelResult >& result)
{
   ETG_TRACE_USR4(("SxmAudioList::onSkipChannelResult _iswaitAnimation:%d", _iswaitAnimation));
   if (result->getStatus())
   {
      _refreshChannelArt = false;
      if (SXMCommonInterface::isViewVisible(Sxm_Scenes_SXM_RADIO__CUSTOMIZE_CHANNELLIST))
      {
         getCustomizeChannelList();
      }
      else if (_iswaitAnimation)
      {
         POST_MSG((COURIER_MESSAGE_NEW(SxmWaitAnimationReqMsg)(ANIMATION_STOP)));
         _iswaitAnimation = false;
      }
   }
}


/**
 * Error handling function for get SetCategoryMonitor Method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : result - preset list result
 */
void SxmAudioList::onSetCategoryMonitorError(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi::SetCategoryMonitorError >& /*error*/)
{
   ETG_TRACE_USR4(("SxmAudioList::onSetCategoryMonitorError"));
}


/**
 * Result handling function for get SetCategoryMonitor Method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : result - preset list result
 */
void SxmAudioList::onSetCategoryMonitorResult(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi::SetCategoryMonitorResult >& result)
{
   ETG_TRACE_USR4(("SxmAudioList::onSetCategoryMonitorResult :status%d", result->getStatus()));
}


/**
 * Result handling function for get ChannelArtList Method
 * @param [in] : channelArtList - list of channel logos
 */
void SxmAudioList::onGetChannelArtListResult(const ::std::vector< ChannelGraphicsType >& channelArtList)
{
   ETG_TRACE_USR4(("SxmAudioList::onGetChannelArtListResult _tranisitonId:%d", _tranisitonId));
   SxmAudioListHelper* pAudioUtilsInstance = SxmAudioListHelper::instance();
   if (!_isCategoryMonitorStarted && (_tranisitonId == Sxm_Scenes_SXM_RADIO__CHANNELLIST_3 ||
                                      _tranisitonId == Sxm_Scenes_SXM_RADIO__CUSTOMIZE_CHANNELLIST))
   {
      //To handle transition for channel list and customize channel.
      pAudioUtilsInstance->setChannellistlogo(channelArtList, _tranisitonId);
      POST_MSG((COURIER_MESSAGE_NEW(SxmSceneTransitionReqMsg)(_tranisitonId)));
      _resetStartIndex = false;
   }
   else if (_isCategoryMonitorStarted && SXMCommonInterface::isViewVisible(Sxm_Scenes_SXM_RADIO__CHANNELLIST_3))
   {
      pAudioUtilsInstance->setChannellistlogo(channelArtList, Sxm_Scenes_SXM_RADIO__CHANNELLIST_3);
      //refresh list.
      pAudioUtilsInstance-> setChannelListStatus(_currentListStatus[LIST_ID_SXM_AUDIO_CHANNELLIST]);
      tSharedPtrDataProvider dataProvider =
         pAudioUtilsInstance->getSxmChannelListDataProvider();
      POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
      if (_resetStartIndex)
      {
         SxmUtilsListHandler::instance()->SendListChangeMsg(LIST_ID_SXM_AUDIO_CHANNELLIST, ListChangeSet, 0);
         _resetStartIndex = false;
      }
   }
   else if (_isCategoryMonitorStarted && SXMCommonInterface::isViewVisible(Sxm_Scenes_SXM_RADIO__CUSTOMIZE_CHANNELLIST))
   {
      pAudioUtilsInstance->setChannellistlogo(channelArtList, Sxm_Scenes_SXM_RADIO__CUSTOMIZE_CHANNELLIST);
      //refresh list.
      tSharedPtrDataProvider dataProvider =
         pAudioUtilsInstance->getSxmCustomizeChannelListDataProvider(_currentListStatus[LIST_ID_SXM_AUDIO_CUSTOMIZE_LIST].startIndex,
               _currentListStatus[LIST_ID_SXM_AUDIO_CUSTOMIZE_LIST].windowSize);
      POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
      if (_resetStartIndex)
      {
         //Reset start index if elements are removed from list.
         SxmUtilsListHandler::instance()->SendListChangeMsg(LIST_ID_SXM_AUDIO_CUSTOMIZE_LIST, ListChangeSet, 0);
         _resetStartIndex = false;
      }
   }

   if (_iswaitAnimation)
   {
      _iswaitAnimation = false;
      POST_MSG((COURIER_MESSAGE_NEW(SxmWaitAnimationReqMsg)(ANIMATION_STOP)));
   }
}


/**
* Error handling function for GetChannelArtList method.
*/
void SxmAudioList::onGetChannelArtListError()
{
   ETG_TRACE_ERR(("SxmAudioList::onGetChannelArtListError"));
}


/**
 * Callback to handle Courier Message SxmAudioCategoriesReqMsg Request
 * @param [in] : Reference of Courier::SxmAudioCategoriesReqMsg
 * @return     :  True-When Message is processed.
 */
bool SxmAudioList::onCourierMessage(const SxmAudioCategoriesReqMsg& /*oMsg*/)
{
   if (_sxmAudioFiProxy)
   {
      _isCategoryMonitorStarted = false;
      _tranisitonId = Sxm_Scenes_SXM_RADIO__CATEGORYLIST;
      _sxmAudioFiProxy->sendGetCategoryListStart(*this, 0);//get all categories.
   }
   return true;
}


/**
 * Callback to handle Courier Message SXMAudioChannelListReqMsg Request
 * @param [in] : Reference of Courier::SXMAudioChannelListReqMsg
 * @return     : status of message handled
 */
bool SxmAudioList::onCourierMessage(const SXMAudioChannelListReqMsg& oMsg)
{
   SxmAudioListHelper* pAudioUtilsInstance = SxmAudioListHelper::instance();
   uint32 rowid = oMsg.GetRowId();

   //Set category name is Header text.
   std::string sCatname = "";
   uint16 catID = 0;
   if (rowid == 0)
   {
      sCatname = SxmUtils::safeGetCString(SXM__AUDIO_MENU_CHANNELLIST);
   }
   else if (rowid < pAudioUtilsInstance->getCategoryList().size())
   {
      sCatname = pAudioUtilsInstance->getCategoryList()[rowid].getCategoryName();
      catID = pAudioUtilsInstance->getCategoryList()[rowid].getCategoryType();
   }
   SXMDataCollectorInterface::vSetSXMDynamicData(sCatname);

   POST_MSG((COURIER_MESSAGE_NEW(SxmStringBoundaryReqMsg)(0, "TextStyle##TextStyles#FrutigerNeueLTCom_M2", sCatname.c_str())));

   if (_sxmAudioFiProxy)
   {
      _isCategoryMonitorStarted = false;
      _tranisitonId = Sxm_Scenes_SXM_RADIO__CHANNELLIST_3;
      _monitoredCategoryId = catID;
      _sxmAudioFiProxy->sendGetCategoryListStart(*this, catID);
   }
   return true;
}


/**
 * Callback to handle Courier Message SXMAudioCustomizeListReqMsg Request
 * @param [in] : Reference of Courier::SXMAudioCustomizeListReqMsg
 * @return     :  True-When Message is processed.
 */
bool SxmAudioList::onCourierMessage(const SXMAudioCustomizeListReqMsg& /*oMsg*/)
{
   _iswaitAnimation = true;
   POST_MSG((COURIER_MESSAGE_NEW(SxmWaitAnimationReqMsg)(ANIMATION_START)));
   _refreshChannelArt = true;
   _isCategoryMonitorStarted = false;
   _tranisitonId = Sxm_Scenes_SXM_RADIO__CUSTOMIZE_CHANNELLIST;
   getCustomizeChannelList();
   return true;
}


/**
 * Callback to handle Courier Message SXMAudioCustomizeClearSelectReqMsg Request
 * @param [in] : Reference of Courier::SXMAudioCustomizeClearSelectReqMsg
 * @return     :  True-When Message is processed.
 */
bool SxmAudioList::onCourierMessage(const SXMAudioCustomizeClearSelectReqMsg& oMsg)
{
   POST_MSG((COURIER_MESSAGE_NEW(SxmWaitAnimationReqMsg)(ANIMATION_START)));
   _iswaitAnimation = true;
   if (_sxmAudioFiProxy)
   {
      _custmClearAllorOne = true;
      _isSkip = oMsg.GetValue();
      _sxmAudioFiProxy->sendSkipChannelStart(*this, static_cast<T_e8_ChannelSkipType>(oMsg.GetValue()), 0, 0);
   }
   return true;
}


/**
 * Callback to handle Courier Message SXMAudioCustomizeControlReqMsg Request
 * @param [in] : Reference of Courier::SXMAudioCustomizeControlReqMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioList::onCourierMessage(const SXMAudioCustomizeControlReqMsg& oMsg)
{
   _customSelectedIndex = oMsg.GetHdl();
   uint16 Serviceid = SxmAudioListHelper::instance()->getCustomizedListEntryServiceId(_customSelectedIndex);
   uint16 Channelid = SxmAudioListHelper::instance()->getCustomizedListEntryChannelId(_customSelectedIndex);

   if (_sxmAudioFiProxy)
   {
      _custmClearAllorOne = false;
      if (SxmAudioListHelper::instance()->getCustomizedListEntryStatus(_customSelectedIndex))
      {
         _sxmAudioFiProxy->sendSkipChannelStart(*this, T_e8_ChannelSkipType__UNSKIP_SELECTION, Serviceid, Channelid);
         _isSkip = false;
      }
      else
      {
         _sxmAudioFiProxy->sendSkipChannelStart(*this, T_e8_ChannelSkipType__SKIP_SELECTION, Serviceid, Channelid);
         _isSkip = true;
      }
   }
   return true;
}


/**
 * Callback to handle Courier Message SXMAudioMonitorUpdatesMsg Request
 * @param [in] : Reference of Courier::SXMAudioMonitorUpdatesMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioList::onCourierMessage(const SXMAudioMonitorUpdatesMsg& oMsg)
{
   sendMonitorUpdateMsg(oMsg.GetListId(), oMsg.GetEnableMonitoring(), true);
   return true;
}


/**
 * Helper Function to send monitor update method start to Midw
 * @param [in] : ListId, Enable - Enable/Disable monitor
 * @return     : True-When Message is processed.
 */

void SxmAudioList::sendMonitorUpdateMsg(uint32 listId, uint8 enable, bool resetliststatus)
{
   if (enable == _isCategoryMonitorStarted)
   {
      return;
   }

   T_b16_CHANNEL_EVENT channelEventInfo;
   _isCategoryMonitorStarted = enable;

   if (_isCategoryMonitorStarted)
   {
      switch (listId)
      {
         case LIST_ID_SXM_AUDIO_CHANNELLIST:
         {
            channelEventInfo.setCHANNEL_EVENT_ARTIST(true);
            channelEventInfo.setCHANNEL_EVENT_CHANNEL_ID(true);
            channelEventInfo.setCHANNEL_EVENT_SERVICE_ID(true);
            channelEventInfo.setCHANNEL_EVENT_TITLE(true);
            channelEventInfo.setCHANNEL_EVENT_REMOVED(true);
            break;
         }
         case LIST_ID_SXM_AUDIO_CUSTOMIZE_LIST:
         {
            channelEventInfo.setCHANNEL_EVENT_CHANNEL_ID(true);
            channelEventInfo.setCHANNEL_EVENT_SERVICE_ID(true);
            channelEventInfo.setCHANNEL_EVENT_REMOVED(true);
            _monitoredCategoryId = 0;//All channels.
            break;
         }
         case LIST_ID_SXM_CATEGORIES:
         {
            channelEventInfo.setCHANNEL_EVENT_CATEGORY(true);
            _monitoredCategoryId = 0;//All channels.
            break;
         }
         default:
         {
            break;
         }
      }
   }
   else if (!_isCategoryMonitorStarted && resetliststatus)
   {
      switch (listId)
      {
         case LIST_ID_SXM_AUDIO_CHANNELLIST:
         case LIST_ID_SXM_AUDIO_CUSTOMIZE_LIST:
         {
            _currentListStatus[listId].startIndex = 0;
            _currentListStatus[listId].windowSize = 20;
            break;
         }
         default:
         {
            break;
         }
      }
   }
   if (_sxmAudioFiProxy)
   {
      _sxmAudioFiProxy->sendSetCategoryMonitorStart(*this, _monitoredCategoryId, channelEventInfo, true, _isCategoryMonitorStarted);
   }
}


/**
 * Send Midw Customize channel list request.
 */
void SxmAudioList::getCustomizeChannelList()
{
   if (_sxmAudioFiProxy)
   {
      _sxmAudioFiProxy->sendGetChannelListStart(*this, 0, 0);// Index:0,Range:0 for complete list.
   }
}


/**
 * getListDataProvider : Gets the ListDataProvider from the corresponding listdataprovider functions
 * based on the List ID.
 * @param [in] : listId
 * @return     : tSharedPtrDataProvider class pointer
 */
tSharedPtrDataProvider SxmAudioList::getListDataProvider(const ListDateProviderReqMsg& msgListDataInfo)
{
   ListDataInfo listDataInfo;
   listDataInfo.listId = msgListDataInfo.GetListId();
   listDataInfo.startIndex = msgListDataInfo.GetStartIndex();
   listDataInfo.windowSize = msgListDataInfo.GetWindowElementSize();
   ETG_TRACE_USR4(("SxmAudioList::getListDataProvider() listid:%d,startindex:%d,windowsize:%d",
                   listDataInfo.listId, listDataInfo.startIndex, listDataInfo.windowSize));
   tSharedPtrDataProvider dataProvider = tSharedPtrDataProvider();
   SxmAudioListHelper* pAudioUtilsInstance = SxmAudioListHelper::instance();

   switch (listDataInfo.listId)
   {
      case LIST_ID_SXM_CATEGORIES:
      {
         dataProvider = pAudioUtilsInstance->getSxmCategorieListDataProvider();
         break;
      }
      case LIST_ID_SXM_AUDIO_MENU:
      {
         dataProvider =  pAudioUtilsInstance->getSxmAudioMenuListDataProvider();
         break;
      }
      case LIST_ID_SXM_AUDIO_CHANNELLIST:
      {
         if (_tranisitonId)
         {
            //On First entry return only first window elements and on request from widget
            //further contents are loaded.
            _currentListStatus[listDataInfo.listId].listId = LIST_ID_SXM_AUDIO_CHANNELLIST;
            _currentListStatus[listDataInfo.listId].startIndex = listDataInfo.startIndex;
            _currentListStatus[listDataInfo.listId].windowSize = 6;
            pAudioUtilsInstance->setChannelListStatus(_currentListStatus[listDataInfo.listId]);
            dataProvider = pAudioUtilsInstance->getSxmChannelListDataProvider();
            _tranisitonId = 0;//Clear transition id after transition.
         }
         else
         {
            if (listDataInfo.startIndex != _currentListStatus[listDataInfo.listId].startIndex ||
                  listDataInfo.windowSize != _currentListStatus[listDataInfo.listId].windowSize)
            {
               updateListDataInfo(listDataInfo);
               pAudioUtilsInstance->setChannelListStatus(listDataInfo);
               dataProvider = pAudioUtilsInstance->getSxmChannelListDataProvider();
            }
         }
         break;
      }
      case LIST_ID_SXM_AUDIO_CUSTOMIZE_LIST:
      {
         if (_tranisitonId)
         {
            updateListDataInfo(listDataInfo);
            dataProvider = pAudioUtilsInstance->getSxmCustomizeChannelListDataProvider(listDataInfo.startIndex, listDataInfo.windowSize);
            _tranisitonId = 0;//Clear transition id after transition.
         }
         else
         {
            if (listDataInfo.startIndex != _currentListStatus[listDataInfo.listId].startIndex ||
                  listDataInfo.windowSize != _currentListStatus[listDataInfo.listId].windowSize)
            {
               updateListDataInfo(listDataInfo);
               dataProvider = pAudioUtilsInstance->getSxmCustomizeChannelListDataProvider(listDataInfo.startIndex, listDataInfo.windowSize);
            }
         }
         break;
      }
      case LIST_ID_SXM_DTM_MAIN:
      {
         dataProvider =  pAudioUtilsInstance->getSxmDTMListDataProvider();
         break;
      }
      case LIST_ID_SXM_MONITOR:
      {
         dataProvider = pAudioUtilsInstance->getSxmMonitorMenuListDataProvider();
         break;
      }
      case LIST_ID_SXM_AUDIO_TWN_ENTRY:
      {
         dataProvider =  pAudioUtilsInstance->getSxmTWNowListDataProvider();
         break;
      }
      case LIST_ID_SXM_DTM_TESTMODE_MONITOR_SELECTION:
      {
         dataProvider = pAudioUtilsInstance->getSxmTestModeMonitorSelection();
         break;
      }
      case LIST_ID_SXM_AUDIO_SXM_INFOLIST:
      {
         dataProvider = pAudioUtilsInstance->getSxmInfoListDataProvider();
         break;
      }
      default:
      {
         ETG_TRACE_ERR(("getListDataProvider:ERROR:Invalid List"));
         break;
      }
   }
   return dataProvider;
}


/*
* Helper Function to store listinfo.
*/
void SxmAudioList::updateListDataInfo(const ListDataInfo& listDataInfo)
{
   ETG_TRACE_USR4(("updateListDataInfo ListId:%d,startIndex:%d,windowSize:%d", listDataInfo.listId, listDataInfo.startIndex, listDataInfo.windowSize));
   _currentListStatus[listDataInfo.listId] = listDataInfo;
}


/*
* Helper Function to trigger handling for TTFis Command SxmAudio_GetChannelArt
*/
void SxmAudioList::vGetChannelArt(uint8 /*testValue*/) const
{
   if (_sxmAudioFiProxy)
   {
      //_sxmAudioFiProxy->sendGetCategoryListStart(*this, 0);
   }
}


/**
 * Helper function to refresh channel list
 * @param [in] : categorylist - list of channels of a category
 */
void SxmAudioList::refreshChannelList(const ::std::vector< ::sxm_audio_main_fi_types::T_CategoryListEntry >& categorylist)
{
   SxmAudioListHelper* pAudioUtilsInstance = SxmAudioListHelper::instance();
   if (categorylist.size())
   {
      //Request for channel art only if number of entries is changed or on entry to list.
      if (pAudioUtilsInstance->getChannelList().size() != categorylist[0].getChannelList().size() ||
            _tranisitonId == Sxm_Scenes_SXM_RADIO__CHANNELLIST_3)
      {
         pAudioUtilsInstance->setSxmChannelList(categorylist[0].getChannelList());
         pAudioUtilsInstance->setSelectedCategoryType(categorylist[0].getCategoryType());
         _resetStartIndex = true;
         //Reset start index if elements are removed from list.
         _currentListStatus[LIST_ID_SXM_AUDIO_CHANNELLIST].startIndex = 0;
         _currentListStatus[LIST_ID_SXM_AUDIO_CHANNELLIST].windowSize = 20;
         getChannelArtListRequest(LIST_ID_SXM_AUDIO_CHANNELLIST);
      }
      else
      {
         pAudioUtilsInstance->setSxmChannelList(categorylist[0].getChannelList());
         pAudioUtilsInstance->setSelectedCategoryType(categorylist[0].getCategoryType());
         pAudioUtilsInstance-> setChannelListStatus(_currentListStatus[LIST_ID_SXM_AUDIO_CHANNELLIST]);
         tSharedPtrDataProvider dataProvider =
            pAudioUtilsInstance->getSxmChannelListDataProvider();
         POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
      }
   }
   else
   {
      //reset channel list and art when size is zero.
      pAudioUtilsInstance->setSxmChannelList(T_ChannelList());
      pAudioUtilsInstance->setChannellistlogo(::std::vector<ChannelGraphicsType >(), Sxm_Scenes_SXM_RADIO__CHANNELLIST_3);
      pAudioUtilsInstance-> setChannelListStatus(_currentListStatus[LIST_ID_SXM_AUDIO_CHANNELLIST]);
      tSharedPtrDataProvider dataProvider =
         pAudioUtilsInstance->getSxmChannelListDataProvider();
      POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
   }
}


/**
 * Callback to handle Courier Message ActiveRenderedView Request
 * @param [in] : Reference of Courier::ActiveRenderedView
 * @return     : True-When Message is processed.
 */
bool SxmAudioList::onCourierMessage(const SxmActiveRenderedViewMsg& msg)
{
   uint32 lHashid = msg.GetSceneID();
   uint32 SurfaceID = msg.GetSurfaceID();
   if (Enum_SURFACEID_MAIN_SURFACE_SXM == SurfaceID)
   {
      ETG_TRACE_USR4(("SxmAudioList::SxmActiveRenderedViewMsg"));
      if (Sxm_Scenes_SXM_RADIO__CHANNELLIST_3 == lHashid)
      {
         sendMonitorUpdateMsg(LIST_ID_SXM_AUDIO_CHANNELLIST, 1, false);
      }
      else if (Sxm_Scenes_SXM_RADIO__CATEGORYLIST == lHashid)
      {
         sendMonitorUpdateMsg(LIST_ID_SXM_CATEGORIES, 1, false);
      }
      else if (Sxm_Scenes_SXM_RADIO__CUSTOMIZE_CHANNELLIST == lHashid)
      {
         sendMonitorUpdateMsg(LIST_ID_SXM_AUDIO_CUSTOMIZE_LIST, 1, false);
      }
   }
   return false;
   //msg to go to SxmHMIServiceStateHandler, so return false
}


/**
* Helper function to get current status of list.
*/
bool SxmAudioList::onCourierMessage(const ListStatusUpdMsg& oMsg)
{
   ETG_TRACE_USR4(("SxmAudioList::ListStatusUpdMsg list state:%d" , oMsg.GetStatus()));
   bool status = false; // variable used only to avoid coverity issue.
   uint32 listId = oMsg.GetListId();
   if (LIST_ID_SXM_AUDIO_CHANNELLIST == listId)
   {
      sendMonitorUpdateMsg(LIST_ID_SXM_AUDIO_CHANNELLIST, !oMsg.GetStatus(), false);
   }
   else if (LIST_ID_SXM_AUDIO_CUSTOMIZE_LIST == listId)
   {
      sendMonitorUpdateMsg(LIST_ID_SXM_AUDIO_CUSTOMIZE_LIST, !oMsg.GetStatus(), false);
   }
   status = ListImplementation::OnMessage(oMsg);

   return false;
}


/**
 * Callback to handle Courier Message SxmStringBoundaryResMsg
 * @param [in] : Reference of Courier::SxmStringBoundaryResMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioList::onCourierMessage(const SxmStringBoundaryResMsg& oMsg)
{
#define MAX_HEADER_CENTER_ALIGNED 320
   ETG_TRACE_USR4(("SxmAudioList::SxmStringBoundaryResMsg width:%d,height:%d" , oMsg.GetWidth(), oMsg.GetHeight()));
   if (oMsg.GetWidth() <= MAX_HEADER_CENTER_ALIGNED)
   {
      SxmAudioDataBindingUtils::instance()->setHeaderTextIndex(false);
   }
   else
   {
      SxmAudioDataBindingUtils::instance()->setHeaderTextIndex(true);
   }
   return true;
}


/**
 * Function to set default value on service availability from suspended
 */
void SxmAudioList::onServiceResumed(const ::boost::shared_ptr< asf::core::Proxy >& /*proxy*/)
{
   ETG_TRACE_USR4(("SxmAudioList::onServiceResumed"));
}


/**
 * Helper Function to request channel art for list of Sids.
 * @param [in] : listOfSid - List of Service ids.
 * @param [in] : sceneId - Selected scene id.
 */
void SxmAudioList::getChannelArtListRequest(uint32 listId)
{
   if (_sxmChannelart)
   {
      ::std::vector< uint16 > listOfSid;

      if (LIST_ID_SXM_AUDIO_CUSTOMIZE_LIST == listId)
      {
         listOfSid = SxmAudioListHelper::instance()->getListOfSid(SxmAudioListHelper::instance()->getCustomizedChannelList(), 0,
                     SxmAudioListHelper::instance()->getCustomizedChannelList().size());
      }
      else if (LIST_ID_SXM_AUDIO_CHANNELLIST == listId)
      {
         listOfSid = SxmAudioListHelper::instance()->getListOfSid(SxmAudioListHelper::instance()->getChannelList(), 0,
                     SxmAudioListHelper::instance()->getChannelList().size());
      }
      _sxmChannelart->sendChannelArtListMethodRequest(this, listOfSid, 1); //Request for Channel logo only
   }
}


/**
* onCourierMessage - This message is received from the flexlist widget when it requires new data when the
*               list is displayed or scrolled
* @param[in] ListDateProviderReqMsg& oMsg
* @parm[out] bool
*/
bool SxmAudioList::onCourierMessage(const ListDateProviderReqMsg& oMsg)
{
   ETG_TRACE_USR3(("SxmAudioList : onCourierMessage ListDateProviderReqMsg"));
   return ListRegistry::s_getInstance().updateList(oMsg.GetListId(), oMsg.GetStartIndex(), oMsg.GetWindowElementSize());
   //TODO : Confirm if this should be included after list data is received
}


}  // namespace Core
}  // namespace App
