/**************************************************************************************
 * @file         : SxmAudioDataBindingUtils.h
 * @addtogroup   : AppHmi_Sxm
 * @brief        : Utility class to handle databinding updates.
 * @copyright    : (c) 2019-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 **************************************************************************************/
#ifndef _SXMAUDIODATABINDING_H_
#define _SXMAUDIODATABINDING_H_
#include "App/Core/SxmUtils/SxmUtilsSingleton.h"
#include "CgiExtensions/DataBindingItem.hpp"
#include "AppHmi_SxmStateMachine.h"
#include "App/Core/SxmUtils/SxmCommonTypes.h"

//forward declaration
namespace sxm_audio_main_fi_types {
class T_ChannelInformation;
}


namespace App {
namespace Core {

/**
 * Utility class to handle all processing for GUI requests
 * @class : SxmAudioDataBindingUtils
 */
class SxmAudioDataBindingUtils
   : public SxmUtilsSingleton<SxmAudioDataBindingUtils>
{
   public:
      /**
       * SxmAudioDataBindingUtils Class destructor
       */
      virtual ~SxmAudioDataBindingUtils();
      /**
       * Set binding source ProgressBar data
       */
      void setSxmAudioProgressBarData(uint8 playPercentage, uint8 fillPercentage);
      /**
       * Set AudioReplayContent
       */
      void setSxmAudioReplayContentData(uint32 timeToLive, uint32 recordDuration);
      /**
       * Update databinding of Audio ReplayContent
       */
      void updateAudioReplayContentData();
      /**
       * Set binding source SxmAudioPresetControl
       */
      void setSxmAudioPresetControlData(uint8 lButtonPressed);
      /**
       * Set binding source SxmAudioPlayPause
       */
      void setSxmAudioPlayPauseData(uint8 pressedButton);
      /**
       * Set default binding source SxmAudioPlayPause
       */
      void setDefaultSxmAudioPlayPauseData(uint8 status);
      /**
       * Set binding source SxmChannelInfoData
       */
      void setSxmChannelInfoData(const ::sxm_audio_main_fi_types::T_ChannelInformation& channelInfo);
      /**
       * Clear binding source SxmChannelInfoData
       */
      void clearSxmChannelInfoData(void);
      /**
       * Clear binding source IR data
       */
      void clearSXMIRData(void);
      /**
       * Clear binding source IR data
       */
      void clearChannelLogo(void);
      /**
       * Set binding source SxmChannelInfoData
       */
      void setSxmChannelInfoPresetNumber(uint8 presetNo);
      /**
       * Set binding source SetSxmChannelLogoData
       */
      void setSxmChannelLogoData(const ::std::vector< uint8 >& imageData, uint32 imageDataSize);
      /**
       * Set binding source setSxmCustomizeChannelListData
       */
      void setSxmCustomizeChannelListData(uint8 status);
      /**
       * Returns the channel number in the edit field of speller.
       */
      uint16 getEditFeildStringOfSpeller(void);
      /**
       * Clears the edit field of speller.
       */
      void clearSpellerEditField(void);
      /**
       * Updates message for Channel Not Available Popup.
       */
      void updateChannelNotAvailablePopup(uint16 lChannelNumber);
      /**
       * Updates message for Channel NotSubscribed Popup.
       */
      void updateChannelNotSubscribedPopup(uint16 lChannelNumber);
      /**
       * Sets binding source for scan button.
       */
      void updateForScanOn(bool bContentAvail, bool bScanActive);
      /**
       * Set binding source SetSxmAlbumartData
       */
      void setSxmAlbumartData(const ::std::vector< uint8 >& imageData, uint32 imageDataSize);
      /**
       * Helper function to enable or disable preset list button.
       */
      void enablePresetListButton(uint8 buttontype, bool status);
      /**
       * Helper function to set advisory text.
       */
      void setAdvisoryText(std::string advisoryMessage);
      /**
       * Helper function to set loading advisory.
       */
      void displayLoading();
      /**
       * Helper function to stop loading animation.
       */
      void stopLoading();
      /**
       * Helper function to set advisory text.
       */
      void setAdvisoryText(Candera::String advisoryMessage);
      /**
       * Helper function to set advisory text.
       */
      void setGadgetAdvisoryText(Candera::String advisoryMessage) const;
      /**
       * Helper function to set advisory text.
       */
      void setGadgetAdvisoryText(std::string advisoryMessage) ;
      /**
       * Helper function to update Preset page.
       */
      void updatePresetPageIndicator(uint8 presetPage);
      /**
       * Helper function to update channel Info
       */
      void updateAllChannelInformation();
      /**
       * Toggles ON/OFF for Fav Artist and Song
       */
      void updateOnOffButtonStatus(uint8 favtype, uint8 status);
      /**
       * Set binding source for setSxmAudioAlertOnorOff
       */
      void setSxmAudioAlertOnorOff(uint16 lHdl);
      /**
       * Set binding source SetSxmAlbumartData
       */
      void setSxmDefaultAlbumartData();
      /**
       * Set binding source SetSxmDefaultChannelLogoData
       */
      void setSxmDefaultChannelLogoData();
      /**
       * Set binding source to switch Preset mode.
       */
      void switchToPresetMode();
      /**
       * Set Binding Source for Save Favorite Result
       */
      void setConentAlertAdivisorypopup(std::string popupMsg, bool isTwoLine);
      /**
       * Set Binding Source for Delete Favorite Result
       */
      void setDeleteFavAdivisorypopup(std::string popupMsg);
      /**
       * Set Binding Source for skipped channel
       */
      void setAdvisoryForSkippedChannel(uint16 lChannelNumber);
      /**
       * Update testmode popup msg
       */
      void updateSxmSettingTestmodeUpdatePopup(const std::string& serviceName);
      /**
       * Sets the Team Name for the popup.
       */
      void setsTeamNameForPopup(std::string lTeamName);
      /**
       * Set binding source for setSxmAudioChannelSettingEnableorDisable
       */
      void setSxmAudioChannelSettingEnableorDisable(bool buttonState);
      /**
       *Set binding source for TWN Button
       */
      void setTWNButtonStatus(uint8 uindex, bool enable);
      /**
       * Set binding source to switch Sports Flash Alert On/Off.
       */
      void switchSportsFlashAlertOnOff(bool status);
      /**
       * Set binding source to switch Sports Flash Delete and alert button On/Off.
       */
      void switchSportsFlashDeleteAlertOnOff(bool bEnable);
      /**
       * Set binding source Sxm Audio Content alret info
       */
      void setSxmAudioContentAlertinfo(uint16 chanelnum, const std::string& contentname, unsigned int alertType, const std::string& alertimg, regionText regText);
      /**
       * Set binding source Sxm Audio Favorite button Enable disable
       */
      void setSxmAudioFavButtonEnable(bool enable);
      /**
       * Set binding source for SF Button
       */
      void setSFButtonStatus(uint8 uindex, bool isenable);
      /**
       * Set binding source for Tune Popup texts
       */
      void setSFTunePopup(uint16 channelID, const std::string& teamName1 , const std::string& teamName2, const std::string& leagueName, regionText regText);
      /**
       * Set binding source for Pop up status
       */
      void setPopUpStatus(bool visiblestate);
      /**
       * Helper function to set button mode.. Preset/ Control
       */
      void setTuneScanAvailableStatus(bool isAvailable);

      void setAudioActiveMode(uint8 mode);
      /**
       * Helper function to set advisory status of content alerts
       */
      void setAlertsAdvisory(bool advorlist, std::string message);
      /**
       * Helper function to set advisory status of TWN
       */
      void setTWNAdvisory(bool advorlist);
      /**
       * Helper function to set advisory status of SportsFlash
       */
      void setSFAdvisory(bool advorlist);
      /**
       * Helper function to Enable/disable SXM info Subscribe button
       */
      void setSxmInfoButtonEnable(bool) const;
      /**
       *Get Pop up status
       */
      bool getPopupStatus();
      /**
       *Set system message status
       */
      void setSystemMessageStatus(uint8 u8status);
      /**
       * Set TWN Pop up text.
       */
      void setTWNTpopText();
      /**
       *Get system message status
       */
      uint8 getSystemMessageStatus() const;
      /**
       * Enables position of delete button in direct tune
       */
      void enableDirecttuneDeletePos(void);
      /**
       * Enable/ disable menu items
       */
      void updateMenuItems(uint8 item, bool status);
      /**
       * Helper function to handle binding updates for SxmAudioDirectTuneTextInfo.
       */
      void setSxmAudioDirectTuneTextInfo(Courier::Request& request);
      /**
       * Helper function to handle binding updates for SxmAudioDirectTuneKeyPressInfo.
       */
      void setSxmAudioDirectTuneKeyPressInfo(Courier::Request& request);
      /**
       * Helper function to set availabilty of complete channel information.
       */
      void setSxmAllChannelInfoAvailbility(bool status);
      /**
       * Helper function to set availabilty of preset list.
       */
      void setSxmPresetListAvailbility(bool status);
      /**
       * Helper function to get current status of all channel information.
       */
      bool getSxmAllChannelInfoAvailbility();
      /**
       * Helper function to get current status of preset list.
       */
      bool getSxmPresetListAvailbility();
      /**
       * Helper function to get current status of preset list.
       */
      bool getSxmChannelInfoSyncStatus();
      /**
       * Helper function to update advisory message
       */
      void updateAdvisoryMessage();
      /**
       * Helper function to set TWN play status.
       */
      void setSxmTWNPlayStatus(bool status);
      /**
       * Helper function to get TWN play status.
       */
      bool getSxmTWNPlayStatus();
      /**
       * Helper function to set SF play status.
       */
      void setSxmSFPlayStatus(bool status);
      /**
       * Helper function to get SF play status.
       */
      bool getSxmSFPlayStatus();
      /**
       * Helper function to get SF play status.
       */
      bool getSxmTuneScanStatus();
      /**
       * Helper function to set Content Alert Status.
       */
      void setSxmContentAlertStatus(bool status);
      /**
       * Helper function to get Content Alert Status.
       */
      bool getSxmContentAlertStatus();
      /**
      * Helper function to set transition done status.
      */
      void setSxmTransitionDoneStatus(bool status);
      /**
       * Helper function to get transition done status.
       */
      bool getSxmTransitionDoneStatus();
      /**
       * Helper function to update Replay Mode status.
       */
      void updateReplayModeStatus();
      /**
       * Helper function to update Header Text index.
       */
      void setHeaderTextIndex(bool index);
      /**
       * Helper function to get Instant Replay status.
       */
      bool getSxmInstantReplayStatus();
      /**
       * Helper function to set Mute status.
       */
      void setMuteStatus(bool status);

   private:
      friend class  SxmUtilsSingleton<SxmAudioDataBindingUtils>;
      /**
       * SxmAudioDataBindingUtils Class constructor
       */
      SxmAudioDataBindingUtils();

      //Member Variables
      DataBindingItem<SxmAudioReplayContentDataBindingSource> _sxmAudioReplayContent;
      DataBindingItem<SxmAudioPlayPauseDataBindingSource> _sxmAudioPlayPause;
      DataBindingItem<SxmChannelInfoDataBindingSource> _sxmChannelInfo;
      DataBindingItem<SxmTpopChannelInfoDataBindingSource> _sxmTpopChannelInfo;
      DataBindingItem<SxmAudioModeDataBindingSource>_sxmAudiomode;
      DataBindingItem<SxmPopupInfoDataBindingSource> _sxmPopupInfo;
      DataBindingItem<SpellerDeleteBtnStateDataBindingSource> _sxmDeleteButtonStatus;
      DataBindingItem<SXMSFButtonDataBindingSource> _sxmSFButton;
      DataBindingItem<SxmPresetListDataBindingSource> _sxmPresetListData;
      DataBindingItem<SXTWNButtonDataBindingSource> _sxmTWNButton;
      DataBindingItem<SxmAudioDirectTuneTextInfoDataBindingSource> _sxmAudioDirectTuneTextInfo;
      DataBindingItem<SxmAudioDirectTuneKeyPressInfoDataBindingSource> _sxmAudioDirectTuneKeyPressInfo;
      DataBindingItem<SxmMenuItemDataBindingSource> _sxmMenuItem;
      DataBindingItem<SxmAudioSliderConfDataBindingSource> _sliderConf;
      DataBindingItem<SxmAudioMaintextAnimationDataBindingSource>_animationAudioLoading;
      DataBindingItem<SxmAudioTuneScanDataBindingSource> _sxmAudioTuneScan;
      DataBindingItem<AudioMuteInfoDataBindingSource>_audioMuteInfo;
      bool _isScanActive;
      bool _isAllChannelInfoAvail;
      bool _isPresetListAvail;
      bool _isTWNButtonEnabled;
      bool _isSFButtonEnabled;
      bool _isSFActive;
      bool _isTWNActive;
      bool  _istuneToContentAlert;
      bool _isTransitionDone;
      uint8 _activeMode;
      uint8 _systemErrorCode;
      const float DELETEPOSX;
      const float DELETEPOSY;
      Candera::String _sxmSceneName;
      std::string _sxmreplayMode;
      uint32 _timeToLive;
      uint32 _recordDuration;

   protected:
};


}  // namespace Core
}  // namespace App

#endif
