/**************************************************************************************
 * @file         : SxmAudioDataBindingUtils.cpp
 * @addtogroup   : AppHmi_Sxm
 * @brief        : Utility class to handle databinding updates.
 * @copyright    : (c) 2019-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 **************************************************************************************/
#include "hall_std_if.h"   // Including HALL framework code
#include "SxmAudioDataBindingUtils.h"
#include "sxm_audio_main_fi_types.h"
#include "CgiExtensions/ImageLoader.h"
#include "App/Core/SxmUtils/SxmUtils.h"
#include "HmiTranslation_TextIds.h"
#include "App/Core/SxmCommon/SXMCommonInterface.h"
#include "SxmAudioTypes.h"
#include "App/Core/LanguageDefines.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS       TR_CLASS_APPHMI_SXMAUDIO
#define ETG_I_TRACE_CHANNEL           TR_TTFIS_APPHMI_SXM
#define ETG_I_TTFIS_CMD_PREFIX        "APPHMI_SXM_"
#define ETG_I_FILE_PREFIX             App::Core::SxmAudioDataBindingUtils::
#include "trcGenProj/Header/SxmAudioDataBindingUtils.cpp.trc.h"
#endif

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_hmi_06_if.h"


namespace App {
namespace Core {
/**
 * SxmAudioDataBindingUtils Class constructor
 */
SxmAudioDataBindingUtils::SxmAudioDataBindingUtils()
   : _isScanActive(false)
   , _isAllChannelInfoAvail(false)
   , _isPresetListAvail(false)
   , _isTWNButtonEnabled(false)
   , _isSFButtonEnabled(false)
   , _isSFActive(false)
   , _isTWNActive(false)
   , _activeMode(PRESET)
   , _systemErrorCode(0)
   , DELETEPOSX(583.0)
   , DELETEPOSY(89.0)
   , _istuneToContentAlert(false)
   , _isTransitionDone(true)
   , _sxmSceneName("")
   , _timeToLive(0)
   , _recordDuration(0)
{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR4(("SxmAudioDataBindingUtils() Constructor"));
}


/**
 * SxmAudioDataBindingUtils Class destructor
 */
SxmAudioDataBindingUtils::~SxmAudioDataBindingUtils()
{
   ETG_TRACE_USR4(("~SxmAudioDataBindingUtils() Destructor"));
   ETG_I_UNREGISTER_FILE();
   _isScanActive = false;
   _isAllChannelInfoAvail = false;
   _isPresetListAvail = false;
   _isTWNButtonEnabled = false;
   _isSFButtonEnabled = false;
   _isSFActive = false;
   _isTWNActive = false;
   _activeMode = PRESET;
   _systemErrorCode = 0;
   _istuneToContentAlert = false;
   _isTransitionDone = true;
   _sxmSceneName = "";
}


/**
 * Set binding source ProgressBar data
 * @param [In] : u8PlayPercentage -Percentage of buffer at which the current play point is present
 * @param [In] : u8FillPercentage -Percentage of buffer at which the current play point is present
 */
void SxmAudioDataBindingUtils::setSxmAudioProgressBarData(uint8 playPercentage, uint8 fillPercentage)
{
   ETG_TRACE_USR4(("vSetSxmAudioProgressBarData() u8PlayPercentage:%d,u8FillPercentage:%d", playPercentage, fillPercentage));
   (*_sliderConf).mBufferValue = static_cast<Candera::Float>(fillPercentage);
   (*_sliderConf).mSliderValue = static_cast<Candera::Float>(playPercentage);
   if (!_isScanActive && !_systemErrorCode && _isAllChannelInfoAvail && _isPresetListAvail)
   {
      _sliderConf.SendUpdate(true);
   }
}


/**
 * Set SxmAudioReplayContent data
 * @param [In] : u32TimeToLive - Time from live point in secs
 * @param [In] : u32RecordDuration - Total buffer duration in secs
 */
void SxmAudioDataBindingUtils::setSxmAudioReplayContentData(uint32 timeToLive, uint32 recordDuration)
{
   _timeToLive = timeToLive;
   _recordDuration = recordDuration;
   updateAudioReplayContentData();
}


/**
 * Update databinding of Audio ReplayContent
 */
void SxmAudioDataBindingUtils::updateAudioReplayContentData()
{
#define MAX_TIME_BUFFER 60
   ETG_TRACE_USR4(("vSetSxmAudioReplayContent() u32TimeToLive:%d,u32RecordDuration:%d", _timeToLive, _recordDuration));
   std::ostringstream sTimeToLivemins;
   std::ostringstream sTimeToLivesecs;

   uint32 mins = (_timeToLive / MAX_TIME_BUFFER);
   uint32 secs = (_timeToLive % MAX_TIME_BUFFER);
   if (mins < 10)
   {
      sTimeToLivemins << "0";
   }
   (MAX_TIME_BUFFER == mins) ? (sTimeToLivemins << "60") : (sTimeToLivemins << (mins % MAX_TIME_BUFFER));
   if (secs < 10)
   {
      sTimeToLivesecs << "0";
   }
   sTimeToLivesecs << secs;
   std::string TimeToLivemins = sTimeToLivemins.str();
   std::string TimeToLivesecs = sTimeToLivesecs.str();

   (*_sxmAudioReplayContent).mPlayEnabled = true;
   (*_sxmAudioReplayContent).mPauseEnabled = true;
   if ((_timeToLive == 0) && (_recordDuration == 0))
   {
      (*_sxmAudioReplayContent).mSkipBackEnabled = true;
      (*_sxmAudioReplayContent).mRewindEnabled = true;
      (*_sxmAudioReplayContent).mFastForwardEnabled = false;
      (*_sxmAudioReplayContent).mForwardEnabled = false;
      _sxmreplayMode = (Candera::String(SXM__AUDIO_PLAYMODE_LIVE)).GetCString();
      (*_sxmAudioReplayContent).mActiveReplayMode = ("  " + _sxmreplayMode).c_str();
      (*_sxmAudioReplayContent).mTimeToLive = "00:00";
   }
   else if ((_timeToLive == 0) && (_recordDuration != 0))
   {
      (*_sxmAudioReplayContent).mSkipBackEnabled = true;
      (*_sxmAudioReplayContent).mRewindEnabled = true;
      (*_sxmAudioReplayContent).mFastForwardEnabled = false;
      (*_sxmAudioReplayContent).mForwardEnabled = false;
      _sxmreplayMode = (Candera::String(SXM__AUDIO_PLAYMODE_LIVE)).GetCString();
      (*_sxmAudioReplayContent).mActiveReplayMode = ("  " + _sxmreplayMode).c_str();
      (*_sxmAudioReplayContent).mTimeToLive = "00:00";
   }
   else if ((_timeToLive - _recordDuration) == 0 && _timeToLive != 0)
   {
      _sxmreplayMode = (Candera::String(SXM__AUDIO_PLAYMODE_END)).GetCString();
      (*_sxmAudioReplayContent).mActiveReplayMode = ("  " + _sxmreplayMode).c_str();
      (*_sxmAudioReplayContent).mTimeToLive = ("-" + TimeToLivemins + ":" + TimeToLivesecs).c_str();
      (*_sxmAudioReplayContent).mSkipBackEnabled = false;
      (*_sxmAudioReplayContent).mRewindEnabled = false;
      (*_sxmAudioReplayContent).mFastForwardEnabled = true;
      (*_sxmAudioReplayContent).mForwardEnabled = true;
   }
   else if ((_timeToLive - _recordDuration) != 0)
   {
      (*_sxmAudioReplayContent).mSkipBackEnabled = true;
      (*_sxmAudioReplayContent).mRewindEnabled = true;
      (*_sxmAudioReplayContent).mFastForwardEnabled = true;
      (*_sxmAudioReplayContent).mForwardEnabled = true;
      _sxmreplayMode = (Candera::String(SXM__AUDIO_PLAYMODE_REPLAY)).GetCString();
      (*_sxmAudioReplayContent).mActiveReplayMode = ("  " + _sxmreplayMode).c_str();
      (*_sxmAudioReplayContent).mTimeToLive = ("-" + TimeToLivemins + ":" + TimeToLivesecs).c_str();
   }
   (*_sxmAudioReplayContent).mReplayOrScan = false;
   (*_sxmAudioReplayContent).mReplayModeTextVisibility = true;
   if (!_isScanActive && !_systemErrorCode && _isAllChannelInfoAvail && _isPresetListAvail)
   {
      _sxmAudioReplayContent.SendUpdate(true);
   }
}


/**
 * Set binding source SxmAudioPresetControl data
 */
void SxmAudioDataBindingUtils::setSxmAudioPresetControlData(uint8 lButtonPressed)
{
   if (_isScanActive && PREV_MODE == lButtonPressed)
   {
      return; //ignore handling while scan is active.
   }
   ETG_TRACE_USR4(("SxmAudioDataBindingUtils::setSxmAudioPresetControlData %d", lButtonPressed));
   DataBindingItem<SxmAudioPresetControlDataBindingSource> sxmAudioPresetControl;
   switch (lButtonPressed)
   {
      case PRESET:
      {
         (*sxmAudioPresetControl).mSwitchPresetOrCtrlOrScan = CONTROL; // Control button visible.
         (*sxmAudioPresetControl).mSwitchPresetOrCtrlButton = true;
         (*sxmAudioPresetControl).mEnabilityPresetOrCtrlButton = true;
         (*sxmAudioPresetControl).mDefaultFocus = 6;
         (*sxmAudioPresetControl).mLapFocus = true;
         sxmAudioPresetControl.SendUpdate(true);
         break;
      }
      case CONTROL:
      {
         (*sxmAudioPresetControl).mSwitchPresetOrCtrlOrScan = PRESET; //Preset button visible.
         (*sxmAudioPresetControl).mSwitchPresetOrCtrlButton = false;
         (*sxmAudioPresetControl).mEnabilityPresetOrCtrlButton = true;
         (*sxmAudioPresetControl).mDefaultFocus = 1;
         (*sxmAudioPresetControl).mLapFocus = false;
         sxmAudioPresetControl.SendUpdate(true);
         break;
      }
      case SCAN:
      {
         (*sxmAudioPresetControl).mSwitchPresetOrCtrlOrScan = SCAN;
         (*sxmAudioPresetControl).mSwitchPresetOrCtrlButton = false;
         (*sxmAudioPresetControl).mEnabilityPresetOrCtrlButton = true;
         (*sxmAudioPresetControl).mDefaultFocus = 1;
         (*sxmAudioPresetControl).mLapFocus = false;
         sxmAudioPresetControl.SendUpdate(true);

         //To display scanning text.
         (*_sxmAudioReplayContent).mReplayModeTextVisibility = true;
         (*_sxmAudioReplayContent).mReplayOrScan = true;
         _sxmAudioReplayContent.MarkItemModified(ItemKey::SxmAudioReplayContent::ReplayOrScanItem);
         _sxmAudioReplayContent.SendUpdate(false);
         std::string scanningText = SxmUtils::safeGetCString(SXM__AUDIO_PLAYMODE_SCANNING);
         (*_sxmAudiomode).mScan1 = scanningText.c_str();
         (*_sxmAudiomode).mScan2 = SxmUtils::conCatData("ss", 2, scanningText.c_str(), ".").c_str();
         (*_sxmAudiomode).mScan3 = SxmUtils::conCatData("ss", 2, scanningText.c_str(), "..").c_str();
         (*_sxmAudiomode).mScan4 = SxmUtils::conCatData("ss", 2, scanningText.c_str(), "...").c_str();

         _sxmAudiomode.SendUpdate(true);
         break;
      }
      case PREV_MODE:
      {
         setSxmAudioPresetControlData(_activeMode);
         break;
      }
      default:
         break;
   }
}


/**
 * Set binding source to switch Preset mode.
 */
void SxmAudioDataBindingUtils::switchToPresetMode()
{
   if (_isScanActive)
   {
      DataBindingItem<SxmAudioPresetControlDataBindingSource> sxmAudioPresetControl;
      (*sxmAudioPresetControl).mSwitchPresetOrCtrlOrScan = CONTROL;
      (*sxmAudioPresetControl).mSwitchPresetOrCtrlButton = true;
      (*sxmAudioPresetControl).mEnabilityPresetOrCtrlButton = true;
      (*sxmAudioPresetControl).mDefaultFocus = 6;
      (*sxmAudioPresetControl).mLapFocus = true;
      sxmAudioPresetControl.SendUpdate(true);
   }
}


/**
 * Set binding source SxmAudioPresetControl data
 * @param [In] : u8PressedButton - pressed button from Instant Replay Controls
 */
void SxmAudioDataBindingUtils::setSxmAudioPlayPauseData(uint8 pressedButton)
{
   switch (pressedButton)
   {
      case SXM_AUDIO_REPLAY_PLAY:
      case SXM_AUDIO_REPLAY_PAUSE:
      {
         (*_sxmAudioPlayPause).mPlayOrPause = !pressedButton;
         break;
      }
      case SXM_AUDIO_REPLAY_FAST_FORWARD:
      case SXM_AUDIO_REPLAY_SKIP_BACKWARD:
      case SXM_AUDIO_REPLAY_FORWARD:
      case SXM_AUDIO_REPLAY_REWIND:
      {
         if (!(*_sxmAudioPlayPause).mPlayOrPause)
         {
            (*_sxmAudioPlayPause).mPlayOrPause = (!((*_sxmAudioPlayPause).mPlayOrPause));
         }
         break;
      }
      case SXM_AUDIO_REPLAY_JUMP_TO_LIVE:
      case SXM_AUDIO_REPLAY_JUMP_TO_BUFFEREND:
      {
         (*_sxmAudioPlayPause).mPlayOrPause = SXM_AUDIO_REPLAY_PAUSE;
         break;
      }
      default:
      {
         break;
      }
   }
   _sxmAudioPlayPause.SendUpdate(true);
}


/**
 * Set default binding source SxmAudioPlayPause
 */
void SxmAudioDataBindingUtils::setDefaultSxmAudioPlayPauseData(uint8 status)
{
   if ((*_sxmAudioPlayPause).mPlayOrPause == status)
   {
      ETG_TRACE_USR4(("SxmAudioPlayback::setDefaultSxmAudioPlayPauseData mPlayOrPause:%d ", (*_sxmAudioPlayPause).mPlayOrPause));
      if (::sxm_audio_main_fi_types::T_e8_IRPlaybackStatus__SXM_AUDIO_IR_PLAY_STATE == status)
      {
         (*_sxmAudioPlayPause).mPlayOrPause = SXM_AUDIO_REPLAY_PAUSE;
      }
      else if (::sxm_audio_main_fi_types::T_e8_IRPlaybackStatus__SXM_AUDIO_IR_PAUSE_STATE == status)
      {
         (*_sxmAudioPlayPause).mPlayOrPause = SXM_AUDIO_REPLAY_PLAY;
      }
      _sxmAudioPlayPause.SendUpdate(true);
   }
}


/**
 * Set binding source SxmAudioPresetControl data
 * @param [In] : channelInfo - Channel Information received from status message
 */
void SxmAudioDataBindingUtils::setSxmChannelInfoData(const ::sxm_audio_main_fi_types::T_ChannelInformation& channelInfo)
{
   std::ostringstream lChannelID;
   uint16 ServiceID = channelInfo.getSID();
   lChannelID << channelInfo.getChannelNumber();
   std::string strChannelID = lChannelID.str();
   std::string Channel = SxmUtils::safeGetCString(SXM__AUDIO_CHANNELINFO_CHID);
   if (ServiceID == 0) //for channel 0
   {
      //Disabling ChannelID
      (*_sxmChannelInfo).mChannelID = "";
      (*_sxmChannelInfo).mChannelName = "";
      (*_sxmChannelInfo).mArtistName =  channelInfo.getChannelName().c_str();//since channel name is not displayed in main screen
      clearSXMIRData();
   }
   else
   {
      (*_sxmChannelInfo).mChannelID = SxmUtils::replaceInTranslationTextID(Channel, strChannelID, "%1").c_str();
      (*_sxmChannelInfo).mArtistName = channelInfo.getArtistName().c_str();
      (*_sxmChannelInfo).mChannelName = channelInfo.getChannelName().c_str();
   }
   (*_sxmChannelInfo).mCategoryName = channelInfo.getCategoryName().c_str();
   (*_sxmChannelInfo).mSongTitle = channelInfo.getSongName().c_str();
   (*_sxmChannelInfo).mAdvisoryMessageVisibility = false;
   (*_sxmChannelInfo).mSxmInitialized = true;

   (*_sxmTpopChannelInfo).mChannelName = channelInfo.getChannelName().c_str();
   (*_sxmTpopChannelInfo).mChannelID = SxmUtils::replaceInTranslationTextID(Channel, strChannelID, "%1").c_str();
   _sxmTpopChannelInfo.SendUpdate(true);
   //Set menu button status.
   updateMenuItems(SXM_AUDIO_SAVEARTIST, channelInfo.getArtistAlertStatus());
   updateMenuItems(SXM_AUDIO_SAVESONG, channelInfo.getSongAlertStatus());
}


/**
 * Set binding source SxmChannelInfoData
 */
void SxmAudioDataBindingUtils::setSxmChannelInfoPresetNumber(uint8 presetNo)
{
   std::string sxmText = SxmUtils::safeGetCString(SXM__AUDIO_POPUP_STATIONNAME_SXM);
   if (!presetNo)
   {
      (*_sxmChannelInfo).mPresetNumber = sxmText.c_str();
      (*_sxmChannelInfo).mPresetNumberVisibility = false;
   }
   else
   {
      std::string preset = SxmUtils::safeGetCString(SXM__AUDIO_POPUP_STATIONNAME_PRESET);
      (*_sxmChannelInfo).mPresetNumber = SxmUtils::conCatData("ssssi", 5, sxmText.c_str(), " ", preset.c_str(), " ", presetNo).c_str();
      (*_sxmChannelInfo).mPresetNumberVisibility = true;
      (*_sxmChannelInfo).mPresetNumberForGadget = SxmUtils::conCatData("i", 1, presetNo).c_str();
   }
}


/**
 * Clear binding source SxmChannelInfoData
 */
void SxmAudioDataBindingUtils::clearSxmChannelInfoData(void)
{
   (*_sxmChannelInfo).mChannelID = "";
   (*_sxmChannelInfo).mCategoryName = "";
   (*_sxmChannelInfo).mArtistName = "";
   (*_sxmChannelInfo).mSongTitle = "";
   (*_sxmChannelInfo).mChannelName = "";
   (*_sxmChannelInfo).mSxmCGVisibility = false;
   (*_sxmChannelInfo).mSxmAlbumartVisibility = false;
   (*_sxmChannelInfo).mSxmCG = Candera::MemoryManagement::SharedPointer<Candera::Image2D>();
   (*_sxmChannelInfo).mSxmAlbumart = Candera::MemoryManagement::SharedPointer<Candera::Image2D>();
}


/**
 * Clear binding source IR data
 */
void SxmAudioDataBindingUtils::clearSXMIRData(void)
{
   ETG_TRACE_USR4(("SxmAudioDataBindingUtils::clearSXMIRData"));
   (*_sxmAudioReplayContent).mActiveReplayMode = "";
   (*_sxmAudioReplayContent).mFastForwardEnabled = false;
   (*_sxmAudioReplayContent).mForwardEnabled = false;
   (*_sxmAudioReplayContent).mPauseEnabled = false;
   (*_sxmAudioReplayContent).mPlayEnabled = false;
   (*_sxmAudioReplayContent).mReplayModeTextVisibility = false;
   (*_sxmAudioReplayContent).mReplayOrScan = false;
   (*_sxmAudioReplayContent).mRewindEnabled = false;
   (*_sxmAudioReplayContent).mSkipBackEnabled = false;
   (*_sxmAudioReplayContent).mTimeToLive = "";
   _sxmAudioReplayContent.SendUpdate(true);

   (*_sliderConf).mBufferValue = static_cast<Candera::Float>(0);
   (*_sliderConf).mSliderValue = static_cast<Candera::Float>(0);
   _sliderConf.SendUpdate(true);

   (*_sxmAudioPlayPause).mPlayOrPause = true;
   _sxmAudioPlayPause.SendUpdate(true);
}


/**
 * Clear binding source IR data
 */
void SxmAudioDataBindingUtils::clearChannelLogo(void)
{
   (*_sxmChannelInfo).mSxmCG = Candera::MemoryManagement::SharedPointer<Candera::Image2D>();
   (*_sxmChannelInfo).mSxmAlbumart = Candera::MemoryManagement::SharedPointer<Candera::Image2D>();
   _sxmChannelInfo.MarkItemModified(ItemKey::SxmChannelInfo::SxmCGItem);
   _sxmChannelInfo.MarkItemModified(ItemKey::SxmChannelInfo::SxmAlbumartItem);
   _sxmChannelInfo.SendUpdate(false);
}


/**
 * Set binding source SxmAudioPresetControl data
 * @param [In] : u8ImageData - Binary data received from status message.
 * @param [In] : u32ImageDataSize - size of the binary data.
 */
void SxmAudioDataBindingUtils::setSxmChannelLogoData(const ::std::vector< uint8 >& imageData, uint32 imageDataSize)
{
   ETG_TRACE_USR4(("SxmAudioDataBindingUtils::setSxmChannelLogoData"));

   if (imageData.empty())
   {
      ETG_TRACE_USR4(("imageData.empty()"));
      setSxmDefaultChannelLogoData();
   }
   else
   {
      Candera::Bitmap::SharedPointer bmp = ImageLoader::loadBitmapData(const_cast<uint8*>(& (imageData)[0]), imageDataSize);
      if (!bmp.PointsToNull())
      {
         ETG_TRACE_USR4(("!bmp.PointsToNull()"));
         (*_sxmChannelInfo).mSxmCG = ImageLoader::createImage(bmp);
         (*_sxmChannelInfo).mSxmCGVisibility = true;
      }
      else
      {
         setSxmDefaultChannelLogoData();
         ETG_TRACE_ERR(("SxmChannelArt is null"));
      }
   }
}


/**
 * Set binding source SetSxmAlbumartData
 * @param [In] : u8ImageData - Binary data received from status message.
 * @param [In] : u32ImageDataSize - size of the binary data.
 */
void SxmAudioDataBindingUtils::setSxmAlbumartData(const ::std::vector< uint8 >& imageData, uint32 imageDataSize)
{
   ETG_TRACE_USR4(("SxmAudioDataBindingUtils::setSxmAlbumartData"));
   if (imageData.empty())
   {
      setSxmDefaultAlbumartData();
   }
   else
   {
      Candera::Bitmap::SharedPointer bmp = ImageLoader::loadBitmapData(const_cast<uint8*>(& (imageData)[0]), imageDataSize);
      if (!bmp.PointsToNull())
      {
         (*_sxmChannelInfo).mSxmAlbumart = ImageLoader::createImage(bmp);
         (*_sxmChannelInfo).mSxmAlbumartVisibility = true;
      }
      else
      {
         setSxmDefaultAlbumartData();
         ETG_TRACE_ERR(("SxmAlbumart is null"));
      }
   }
}


/**
 * Set default album art
 */
void SxmAudioDataBindingUtils::setSxmDefaultAlbumartData()
{
   ETG_TRACE_USR4(("SxmAudioDataBindingUtils::setSxmDefaultAlbumartData"));
   //(*_sxmChannelInfo).mSxmAlbumart = ImageLoader::getAssetBitmapImage("Bitmaps##SXM#Temp#SxmDefaultAlbmArt");
   (*_sxmChannelInfo).mSxmAlbumart = ImageLoader::getAssetBitmapImage("Sxm#Images#Landing#Layer_Default_albumart_logoBmp");
   (*_sxmChannelInfo).mSxmAlbumartVisibility = true;
}


/**
 * Set default Channel Logo.
 */
void SxmAudioDataBindingUtils::setSxmDefaultChannelLogoData()
{
   ETG_TRACE_USR4(("SxmAudioDataBindingUtils::setSxmDefaultChannelLogoData"));
   //(*_sxmChannelInfo).mSxmCG = ImageLoader::getAssetBitmapImage("Bitmaps##AIVIResources#SXM#SXM_ChannelArt");
   (*_sxmChannelInfo).mSxmCG = ImageLoader::getAssetBitmapImage("Sxm#Images#Landing#Layer_Default_channel_logoBmp");
   (*_sxmChannelInfo).mSxmCGVisibility = true;
}


/**
 * Helper function to handle customized channel list
 */
void SxmAudioDataBindingUtils::setSxmCustomizeChannelListData(uint8 status)
{
   DataBindingItem<SXMAudioCustomizeChannelDataBindingSource> sxmAudioCustomizeSwitch;
   (*sxmAudioCustomizeSwitch).mSwitchClearOrSelect = status;
   sxmAudioCustomizeSwitch.SendUpdate(true);
}


/**
 * Helper function to handle binding updates for SxmAudioDirectTuneTextInfo.
 */
void SxmAudioDataBindingUtils::setSxmAudioDirectTuneTextInfo(Courier::Request& request)
{
   ETG_TRACE_USR4(("SxmAudioDataBindingUtils::setSxmAudioDirectTuneTextInfo"));
   _sxmAudioDirectTuneTextInfo.SetValue(request.ItemKey(), request.GetItemValue());
   if ((*_sxmAudioDirectTuneTextInfo).mEditorText.IsEmpty())
   {
      (*_sxmAudioDirectTuneTextInfo).mIsMaxCharReached = true;
      (*_sxmAudioDirectTuneTextInfo).mOkButtonEnability = false;
      (*_sxmDeleteButtonStatus).mIsEnabled = false;
   }
   else if ((*_sxmAudioDirectTuneTextInfo).mEditorText.GetCharCount() < 3)
   {
      (*_sxmAudioDirectTuneTextInfo).mIsMaxCharReached = true;
      (*_sxmAudioDirectTuneTextInfo).mOkButtonEnability = true;
      (*_sxmDeleteButtonStatus).mIsEnabled = true;
   }
   else
   {
      (*_sxmAudioDirectTuneTextInfo).mIsMaxCharReached = false;
      (*_sxmAudioDirectTuneTextInfo).mOkButtonEnability = true;
      (*_sxmDeleteButtonStatus).mIsEnabled = true;
   }
   _sxmAudioDirectTuneTextInfo.MarkItemModified(ItemKey::SxmAudioDirectTuneTextInfo::IsMaxCharReachedItem);
   _sxmAudioDirectTuneTextInfo.MarkItemModified(ItemKey::SxmAudioDirectTuneTextInfo::OkButtonEnabilityItem);
   _sxmAudioDirectTuneTextInfo.SendUpdate(false);

   _sxmDeleteButtonStatus.MarkItemModified(ItemKey::SpellerDeleteBtnState::IsEnabledItem);
   _sxmDeleteButtonStatus.SendUpdate(false);
}


/**
 * Helper function to handle binding updates for SxmAudioDirectTuneKeyPressInfo.
 */
void SxmAudioDataBindingUtils::setSxmAudioDirectTuneKeyPressInfo(Courier::Request& request)
{
   ETG_TRACE_USR4(("SxmAudioDataBindingUtils::setSxmAudioDirectTuneKeyPressInfo"));
   _sxmAudioDirectTuneKeyPressInfo.SetValue(request.ItemKey(), request.GetItemValue());
   _sxmAudioDirectTuneKeyPressInfo.SendUpdate(true);
}


/**
 * Returns the channel number in the edit field of speller.
 * @return     : The string in edit field.
 */
uint16 SxmAudioDataBindingUtils::getEditFeildStringOfSpeller(void)
{
   uint16 lChannelNumber;
   lChannelNumber = std::atoi((*_sxmAudioDirectTuneTextInfo).mEditorText.GetCString());
   ETG_TRACE_USR4((" getEditFeildStringOfSpeller:Channel Number=%d", lChannelNumber));
   return lChannelNumber;
}


/**
 * Clears the edit field of speller.
 */
void SxmAudioDataBindingUtils::clearSpellerEditField(void)
{
   ETG_TRACE_USR4(("ClearSpellerEditField"));

   (*_sxmAudioDirectTuneTextInfo).mEditorText = "";
   (*_sxmAudioDirectTuneTextInfo).mIsMaxCharReached = true;
   (*_sxmAudioDirectTuneTextInfo).mOkButtonEnability = false;
   _sxmAudioDirectTuneTextInfo.SendUpdate(true);

   (*_sxmAudioDirectTuneKeyPressInfo).mKeyPressed = "";
   _sxmAudioDirectTuneKeyPressInfo.SendUpdate(true);

   (*_sxmDeleteButtonStatus).mIsEnabled = false;
   _sxmDeleteButtonStatus.MarkItemModified(ItemKey::SpellerDeleteBtnState::IsEnabledItem);
   _sxmDeleteButtonStatus.SendUpdate(false);
}


/**
 * Enables position of delete button in direct tune
 */
void SxmAudioDataBindingUtils::enableDirecttuneDeletePos(void)
{
   // (*_sxmDeleteButtonStatus).mPosition.SetX(DELETEPOSX);
   // (*_sxmDeleteButtonStatus).mPosition.SetY(DELETEPOSY);
   // _sxmDeleteButtonStatus.MarkItemModified(ItemKey::SpellerDeleteBtnState::PositionItem);
   // _sxmDeleteButtonStatus.SendUpdate(false);
}


/**
 * Updates message for Channel Not Available Popup.
 * @param[in] : channel number
 */
void SxmAudioDataBindingUtils::updateChannelNotAvailablePopup(uint16 lChannelNumber)
{
   ETG_TRACE_USR4(("SxmAudioDataBindingUtils::updateChannelNotAvailablePopup: Channel NUmber %d", lChannelNumber));
   std::string ChannelNum = (lChannelNumber <= END_CHANNEL_ID) ? SxmUtils::conCatData("i", 1, lChannelNumber) : "";
   std::string ChannelNumAvailable = SxmUtils::safeGetCString(SXM__AUDIO_POPUP_CHANNELNOTAVAILABLE);
   ChannelNumAvailable = SxmUtils::replaceInTranslationTextID(ChannelNumAvailable, ChannelNum, "%1");
   DataBindingItem<SxmAudioChannelNotAvailablePopupDataBindingSource> sxmAudioChannelNotAvailablePopup;
   (*sxmAudioChannelNotAvailablePopup).mText = ChannelNumAvailable.c_str();
   sxmAudioChannelNotAvailablePopup.SendUpdate(true);
}


/**
 * Updates message for Channel Not Subscribed Popup.
 *
 * @param[in] : channel number
 */
void SxmAudioDataBindingUtils::updateChannelNotSubscribedPopup(uint16 lChannelNumber)
{
   ETG_TRACE_USR4(("SxmAudioDataBindingUtils::updateChannelNotSubscribedPopup: Channel NUmber %d", lChannelNumber));
   std::string ChannelNum = SxmUtils::conCatData("i", 1, lChannelNumber);
   std::string ChannelNumSubscription = SxmUtils::safeGetCString(SXM__AUDIO_POPUP_CHANNELNOTSUBSCRIBED);
   ChannelNumSubscription = SxmUtils::replaceInTranslationTextID(ChannelNumSubscription, ChannelNum, "%1");
   DataBindingItem<SxmAudioChannelNotSubscribedPopupDataBindingSource> sxmAudioChannelNotSubscribedPopup;
   (*sxmAudioChannelNotSubscribedPopup).mText = ChannelNumSubscription.c_str();
   sxmAudioChannelNotSubscribedPopup.SendUpdate(true);
}


/**
 * Sets binding source for scan button.
 */
void SxmAudioDataBindingUtils::updateForScanOn(bool bContentAvail, bool bScanActive)
{
   DataBindingItem<SxmAudioScanDataBindingSource> sxmAudioScan;
   (*sxmAudioScan).mScanContentAvail = bContentAvail;
   (*sxmAudioScan).mScanOn = bScanActive;
   _isScanActive = bScanActive;
   sxmAudioScan.SendUpdate(true);
}


/**
 * Helper function to handle Audio Alert Setting
 */
void SxmAudioDataBindingUtils::setSxmAudioAlertOnorOff(uint16 lHdl)
{
   ETG_TRACE_USR4(("SxmAudioAlert::vSetSxmAudioAlertList"));

   dp_tclAppHmi_SxmSXMAudioAlert dp_AudioAlert;
   HmiAppSxm::AlertSetting tSXMAudioAlert = dp_AudioAlert.tGetData();
   tSXMAudioAlert.OnOrOff[lHdl] = !(tSXMAudioAlert.OnOrOff[lHdl]);
   dp_AudioAlert.s32SetData(tSXMAudioAlert);
//   tSXMAudioAlert = dp_AudioAlert.tGetData();
//
//   DataBindingItem<SxmAudioAlertSettingDataBindingSource> sxmAudioAlertSwitch;
//   (*sxmAudioAlertSwitch).mOnOrOff = tSXMAudioAlert.OnOrOff[lHdl];
//   sxmAudioAlertSwitch.SendUpdate(true);
}


/**
 *toggle ON and OFF Button
 */
void SxmAudioDataBindingUtils::updateOnOffButtonStatus(uint8 favtype, uint8 status)
{
   if (FAV_ARTIST == favtype)
   {
      DataBindingItem<SxmAudioFavoriteArtistDataBindingSource> sxmAudioFavoriteArtist;
      (*sxmAudioFavoriteArtist).mSwitchArtistOnOrOffOrDelete = status;
      sxmAudioFavoriteArtist.SendUpdate(true);
   }
   else if (FAV_SONG == favtype)
   {
      DataBindingItem<SxmAudioFavoriteSongDataBindingSource> sxmAudioFavoriteSong;
      (*sxmAudioFavoriteSong).mSwitchSongtOnOrOffOrDelete = status;
      sxmAudioFavoriteSong.SendUpdate(true);
   }
}


/**
 * Helper function to handle enable/disable customized channel buttons
 */
void SxmAudioDataBindingUtils::setSxmAudioChannelSettingEnableorDisable(bool buttonState)
{
   DataBindingItem<SxmAudioCustomizeButtonDataBindingSource> sxmcustomizeButton;
   (*sxmcustomizeButton).mEnableOrDisable = buttonState;
   sxmcustomizeButton.SendUpdate(true);
}


/**
 * Helper function to enable or disable preset list button.
 * @param[in] : buttontype ( Presetlist left or right)
 * @param[in] : status ( enable or disable)
 */
void SxmAudioDataBindingUtils::enablePresetListButton(uint8 buttontype, bool status)
{
   switch (buttontype)
   {
      case LEFT:
      {
         (*_sxmPresetListData).mBtnListLeftEnable = status;
         _sxmPresetListData.MarkItemModified(ItemKey::SxmPresetList::BtnListLeftEnableItem);
         break;
      }
      case RIGHT:
      {
         (*_sxmPresetListData).mBtnListRightEnable = status;
         _sxmPresetListData.MarkItemModified(ItemKey::SxmPresetList::BtnListRightEnableItem);
         break;
      }
      default:
      {
         (*_sxmPresetListData).mBtnListLeftEnable = status;
         (*_sxmPresetListData).mBtnListRightEnable = status;
         _sxmPresetListData.MarkItemModified(ItemKey::SxmPresetList::BtnListRightEnableItem);
         _sxmPresetListData.MarkItemModified(ItemKey::SxmPresetList::BtnListLeftEnableItem);
         break;
      }
   }
   _sxmPresetListData.SendUpdate(false);
}


/**
 * Helper function to update Preset page.
 * @param[in] : u8PresetPage ( Preset page)
 */
void SxmAudioDataBindingUtils::updatePresetPageIndicator(uint8 presetPage)
{
   if (presetPage)
   {
      //PageIndicator range is 0 to 2,So subtract by one.
      (*_sxmPresetListData).mPageIndicator = presetPage - 1;
      _sxmPresetListData.MarkItemModified(ItemKey::SxmPresetList::PageIndicatorItem);
      _sxmPresetListData.SendUpdate(false);
   }
   else
   {
      ETG_TRACE_ERR(("SxmAudioDataBindingUtils::updatePresetPageIndicator Invalid preset page"));
   }
}


/**
 * Helper function to set advisory text.
 * @param[in] : AdvisoryMessage - advisory text.
 */
void SxmAudioDataBindingUtils::setAdvisoryText(std::string advisoryMessage)
{
   (*_sxmChannelInfo).mAdvisoryMessage = advisoryMessage.c_str();
   (*_sxmChannelInfo).mSxmInitialized = true;
   updateAdvisoryMessage();
}


/**
 * Helper function to set loading advisory.
 */
void SxmAudioDataBindingUtils::displayLoading()
{
   (*_sxmChannelInfo).mSxmInitialized = false;
   //(*_animationAudioLoading).mtextInformation = SxmUtils::safeGetCString(SXM__AUDIO_ADV_LOADING).c_str();
   (*_sxmChannelInfo).mAdvisoryMessage = Candera::String(SXM__AUDIO_ADV_LOADING);
   (*_animationAudioLoading).mshouldrun = true ;
   _animationAudioLoading.SendUpdate(true);
   setTWNButtonStatus(255, false); //255: Invalid index set to maintain last Twn mode.
   setSFButtonStatus(255, false);  //255: Invalid index set to maintain last SF mode.
   updateAdvisoryMessage();
}


/**
 *Helper function to stop loading animation.
 */
void SxmAudioDataBindingUtils::stopLoading()
{
   if ((*_animationAudioLoading).mshouldrun)
   {
      (*_animationAudioLoading).mtextInformation = "";
      (*_animationAudioLoading).mshouldrun = false ;
      _animationAudioLoading.SendUpdate(true);
   }
}


/**
 * Helper function to set advisory text.
 */
void SxmAudioDataBindingUtils::setAdvisoryText(Candera::String advisoryMessage)
{
   ETG_TRACE_USR4(("SxmAudioDataBindingUtils::setAdvisoryText"));
   (*_sxmChannelInfo).mSxmInitialized = true;
   (*_sxmChannelInfo).mAdvisoryMessage = advisoryMessage;
   updateAdvisoryMessage();
}


/**
 * Helper function to set advisory text.
 */
void SxmAudioDataBindingUtils::setGadgetAdvisoryText(Candera::String advisoryMessage)const
{
   DataBindingItem<SxmGadgetChannelInfoDataBindingSource> gadgetInfo;
   (*gadgetInfo).mAdvisoryMessage = advisoryMessage;
   (*gadgetInfo).mAdvisoryMessageVisibility = true;
   (*gadgetInfo).mSxmInitialized = true;
   if (advisoryMessage == Candera::String(SXM__AUDIO_GADGET_LOADING))        //if (AdvisoryMessage == "Loading")
   {
      (*gadgetInfo).mSxmInitialized = false;
   }
   gadgetInfo.SendUpdate(true);
}


/**
 * Helper function to set advisory text.
 */
void SxmAudioDataBindingUtils::setGadgetAdvisoryText(std::string advisoryMessage)
{
   DataBindingItem<SxmGadgetChannelInfoDataBindingSource> gadgetInfo;
   (*gadgetInfo).mAdvisoryMessage = advisoryMessage.c_str();
   (*gadgetInfo).mAdvisoryMessageVisibility = true;
   (*gadgetInfo).mSxmInitialized = true;
   gadgetInfo.SendUpdate(true);
}


/**
 * Helper function to update advisory message
 */
void SxmAudioDataBindingUtils::updateAdvisoryMessage()
{
   ETG_TRACE_USR4(("SxmAudioDataBindingUtils::updateAdvisoryMessage _isPresetListAvail:%d _isAllChannelInfoAvail:%d, (*_sxmChannelInfo).mSxmInitialized:%d", _isPresetListAvail, _isAllChannelInfoAvail, (*_sxmChannelInfo).mSxmInitialized));

   if ((_isPresetListAvail && _isAllChannelInfoAvail) || (!(*_sxmChannelInfo).mSxmInitialized))
   {
      (*_sxmChannelInfo).mAdvisoryMessageVisibility = true;
      (*_sxmChannelInfo).mSxmAlbumartVisibility = false;

      _sxmChannelInfo.MarkItemModified(ItemKey::SxmChannelInfo::AdvisoryMessageItem);
      _sxmChannelInfo.MarkItemModified(ItemKey::SxmChannelInfo::AdvisoryMessageVisibilityItem);
      _sxmChannelInfo.MarkItemModified(ItemKey::SxmChannelInfo::SxmAlbumartVisibilityItem);
      _sxmChannelInfo.MarkItemModified(ItemKey::SxmChannelInfo::SxmInitializedItem);
      _sxmChannelInfo.SendUpdate(false);

      //Clear Replay or Live Text
      (*_sxmAudioReplayContent).mReplayModeTextVisibility = false;
      _sxmAudioReplayContent.MarkItemModified(ItemKey::SxmAudioReplayContent::ReplayModeTextVisibilityItem);
      _sxmAudioReplayContent.SendUpdate(false);

      if ((*_sxmChannelInfo).mSxmInitialized)
      {
         stopLoading(); //need to check
      }
   }
}


/**
 * Helper function to update Replay Mode status.
 */
void SxmAudioDataBindingUtils::updateReplayModeStatus()
{
   if ((*_sxmAudioReplayContent).mReplayModeTextVisibility == false)
   {
      //to set Replay or Live mode
      (*_sxmAudioReplayContent).mReplayModeTextVisibility = true;
      _sxmAudioReplayContent.MarkItemModified(ItemKey::SxmAudioReplayContent::ReplayModeTextVisibilityItem);
      _sxmAudioReplayContent.SendUpdate(false);
   }
}


/**
 * Helper function to update channel Info
 */
void SxmAudioDataBindingUtils::updateAllChannelInformation()
{
   ETG_TRACE_USR4(("SxmAudioDataBindingUtils::updateAllChannelInformation"));
   if (_isPresetListAvail && _isAllChannelInfoAvail)
   {
      if (!_isScanActive)
      {
         _sxmAudioReplayContent.SendUpdate(true);
         _sliderConf.SendUpdate(true);
      }
      _sxmChannelInfo.SendUpdate(true);
      DataBindingItem<SxmGadgetChannelInfoDataBindingSource> gadgetInfo;
      (*gadgetInfo).mAdvisoryMessageVisibility = false;
      (*gadgetInfo).mAdvisoryMessage = "";
      (*gadgetInfo).mSxmInitialized = true;
      gadgetInfo.SendUpdate(true);
      setTWNButtonStatus(255, true); //255: Invalid index set to maintain last TWN mode.
      setSFButtonStatus(255, true); //255: Invalid index set to maintain last SF mode.
      stopLoading();
   }
}


/**
 * Helper Function to Set Pop up Data
 */
void SxmAudioDataBindingUtils::setConentAlertAdivisorypopup(std::string popupMsg, bool isTwoLine)
{
   DataBindingItem<SxmFavtUpdatePopupDataBindingSource> sxmFavUpdatePopup;
   (*sxmFavUpdatePopup).mText = popupMsg.c_str();
   (*sxmFavUpdatePopup).mPopupType = isTwoLine;
   sxmFavUpdatePopup.SendUpdate(true);
}


/**
 * Helper Function to Set Pop up Data
 */
void SxmAudioDataBindingUtils::setDeleteFavAdivisorypopup(std::string popupMsg)
{
   DataBindingItem<SxmFavtDeletePopupDataBindingSource> sxmFavDeletePopup;
   (*sxmFavDeletePopup).mText = popupMsg.c_str();
   sxmFavDeletePopup.SendUpdate(true);
}


/**
 * Set Binding Source for skipped channel
 */
void SxmAudioDataBindingUtils::setAdvisoryForSkippedChannel(uint16 lChannelNumber)
{
   DataBindingItem<SxmAudioInfoForPopupDataBindingSource> sxmAdvisoryPopupData;
   (*sxmAdvisoryPopupData).mChannelLocked = SxmUtils::replaceInTranslationTextID(SxmUtils::safeGetCString(SXM__AUDIO_POPUP_CHANNELSKIPPED),
         SxmUtils::conCatData("i", 1, lChannelNumber), "%1").c_str();
   sxmAdvisoryPopupData.SendUpdate(true);
}


/**
 * Helper function to update Preset page.
 * @param[in] : u8PresetPage ( Preset page)
 */
void SxmAudioDataBindingUtils::updateSxmSettingTestmodeUpdatePopup(const std::string& serviceName)
{
   DataBindingItem<SxmSettingTestmodeUpdatePopupDataBindingSource> sxmSettingTestmodeUpdatePopup;
   (*sxmSettingTestmodeUpdatePopup).mText = serviceName.c_str();
   sxmSettingTestmodeUpdatePopup.SendUpdate(true);
}


/**
 * Sets the Team Name for the popup.
 * @pram[in] : TeamName
 */
void SxmAudioDataBindingUtils::setsTeamNameForPopup(std::string lTeamName)
{
   ETG_TRACE_USR4(("SxmAudioDataBindingUtils::setsTeamNameForPopup %s", lTeamName.c_str()));
   DataBindingItem<SxmAudioSportsFlashTeamNamePopupDataBindingSource> sxmAudioSportsFlashTeamNamePopup;
   std::string popupMsg = SxmUtils::safeGetCString(SXM__AUDIO_SF_TEAMSELECTED);
   popupMsg = SxmUtils::replaceInTranslationTextID(popupMsg, lTeamName, "%1");
   (*sxmAudioSportsFlashTeamNamePopup).mText = popupMsg.c_str(); //new text id
   sxmAudioSportsFlashTeamNamePopup.SendUpdate(true);
}


/**
 *Set binding source for TWN Button
 */
void SxmAudioDataBindingUtils::setTWNButtonStatus(uint8 uindex, bool enable)
{
   ETG_TRACE_USR4(("SxmAudioDataBindingUtils::setTWNButtonStatus  index value %d", uindex));
   if (uindex != 255) // to update previous status after startup syncronisation
   {
      _isTWNButtonEnabled = enable;
      (*_sxmTWNButton).mTWNButton = uindex;
   }
   (*_sxmTWNButton).mBtnEnable = (_isTWNButtonEnabled && _isAllChannelInfoAvail && _isPresetListAvail);
   _sxmTWNButton.MarkItemModified(ItemKey::SXTWNButton::BtnEnableItem);
   _sxmTWNButton.MarkItemModified(ItemKey::SXTWNButton::TWNButtonItem);
   _sxmTWNButton.SendUpdate(false);
}


/**
 * Helper function to Switch Alert On/off
 */
void SxmAudioDataBindingUtils::switchSportsFlashAlertOnOff(bool status)
{
   DataBindingItem<SXMAudioSportsflashAlertDataBindingSource> sxmAudioSportsflashAlert;
   (*sxmAudioSportsflashAlert).mSwitchOnorOff = status;
   sxmAudioSportsflashAlert.SendUpdate(true);
}


/**
 * Helper function to Switch Alert and Delete On/off
 */
void SxmAudioDataBindingUtils::switchSportsFlashDeleteAlertOnOff(bool bEnable)
{
   DataBindingItem<SxmAudioSportsflashEnableDataBindingSource> sxmAudioSportsflashEnable;
   (*sxmAudioSportsflashEnable).mEnableOn = bEnable;
   sxmAudioSportsflashEnable.SendUpdate(true);
}


/**
 * Set binding source SxmAudiocontent alert pop up data
 * @param [In] : _mAlertinfo - Alert Information received from status message
 */
void SxmAudioDataBindingUtils::setSxmAudioContentAlertinfo(uint16 chanelnum, const std::string& contentname, unsigned int alertType,
      const std::string& alertimg, regionText regText)
{
   std::ostringstream strChannelId;
   strChannelId << chanelnum;
   std::string Channel = SxmUtils::replaceInTranslationTextID(SxmUtils::safeGetCString(SXM__AUDIO_POPUP_CHNUM), " ", ":");
   DataBindingItem<SxmAudioAlertFavtUpdatePopupDataBindingSource> sxmAudioAlertFavtUpdatePopup;
   //(*sxmAudioAlertFavtUpdatePopup).mTextType = alert_type.c_str();
   (*sxmAudioAlertFavtUpdatePopup).mTextDetailInfo = (Channel + strChannelId.str() + ": " + contentname).c_str();
   (*sxmAudioAlertFavtUpdatePopup).mAlertIcon = ImageLoader::getAssetBitmapImage(alertimg.c_str());
   std::string alertText = "";
   if (alertType == 0)
   {
      alertText = SxmUtils::replaceInTranslationTextID(SxmUtils::safeGetCString(SXM__AUDIO_POPUP_ALERTFORARTIST), regText.region1, "%1");
   }
   else if (alertType == 1)
   {
      alertText = SxmUtils::replaceInTranslationTextID(SxmUtils::safeGetCString(SXM__AUDIO_POPUP_ALERTFORSONG), regText.region1, "%1");
   }
   alertText = SxmUtils::replaceInTranslationTextID(alertText, regText.region2, "%2");
   (*sxmAudioAlertFavtUpdatePopup).mTextType = alertText.c_str();
   (*sxmAudioAlertFavtUpdatePopup).misRegionTextAllowed = !(regText.region1.empty() && regText.region2.empty());
   //If both are empty, one line will be displayed. If any one text is there, two lines (with region) to be displayed
   sxmAudioAlertFavtUpdatePopup.SendUpdate(true);
}


/**
 * Set binding source Sxm Audio Favorite button Enable disable
 */
void SxmAudioDataBindingUtils::setSxmAudioFavButtonEnable(bool enable)
{
   DataBindingItem<SxmAudioFavAlertButtonEnableDataBindingSource> sxmAudioFavAlertButtonEnable;
   (*sxmAudioFavAlertButtonEnable).mEnable = enable;
   sxmAudioFavAlertButtonEnable.SendUpdate(true);
}


/**
 * Helper function to update Header Text index.
 */
void SxmAudioDataBindingUtils::setHeaderTextIndex(bool index)
{
   DataBindingItem<SxmAudioSubScreenHeaderTextDataBindingSource> sxmHeaderText;
   (*sxmHeaderText).mIndex = index;
   sxmHeaderText.SendUpdate(true);
}


/**
 * Function to set binding source for Sports Flash Button
 */
void SxmAudioDataBindingUtils::setSFButtonStatus(uint8 uindex, bool isenable)
{
   if (uindex != 255) // to update previous status after startup syncronisation
   {
      _isSFButtonEnabled = isenable;
      (*_sxmSFButton).mSFButton = uindex;
   }
   (*_sxmSFButton).mBtnEnable = (_isSFButtonEnabled && _isAllChannelInfoAvail && _isPresetListAvail);
   _sxmSFButton.MarkItemModified(ItemKey::SXMSFButton::SFButtonItem);
   _sxmSFButton.MarkItemModified(ItemKey::SXMSFButton::BtnEnableItem);
   _sxmSFButton.SendUpdate(false);
}


/**
 * Function to set binding source for Tune Popup texts
 */
void SxmAudioDataBindingUtils::setSFTunePopup(uint16 channelID, const std::string& teamName1, const std::string& teamName2,
      const std::string& leagueName, regionText regText)
{
   std::ostringstream strChannelId;
   strChannelId << channelID;
   std::string Channel = SxmUtils::replaceInTranslationTextID(SxmUtils::safeGetCString(SXM__AUDIO_POPUP_CHNUM), " ", ":");
   DataBindingItem<SxmSFTunePopupDataBindingSource> sxmTunePopup;
   (*sxmTunePopup).mTeams = (Channel + strChannelId.str() + ": " + leagueName + ", " + teamName1 + " " +
                             SxmUtils::safeGetCString(SXM__AUDIO_SF_VS) + ". " + teamName2).c_str();
   std::string alertText = SxmUtils::replaceInTranslationTextID(SxmUtils::safeGetCString(SXM__AUDIO_POPUP_ALERTFORSPORTSFLASH), regText.region1, "%1");
   alertText = SxmUtils::replaceInTranslationTextID(alertText, regText.region2, "%2");
   (*sxmTunePopup).mSFTextType = alertText.c_str();
   (*sxmTunePopup).misRegionTextAllowed = !(regText.region1.empty() && regText.region2.empty());
   //If both are empty, one line will be displayed. If any one text is there, two lines (with region) to be displayed
   sxmTunePopup.SendUpdate(true);
}


/**
 * Set binding source for Pop up status
 */
void SxmAudioDataBindingUtils::setPopUpStatus(bool visiblestate)
{
   (*_sxmPopupInfo).mPopupVisible = visiblestate;
   _sxmPopupInfo.SendUpdate(true);
}


void SxmAudioDataBindingUtils::setTuneScanAvailableStatus(bool isAvailable)
{
   (*_sxmAudioTuneScan).mEnable = isAvailable;
   _sxmAudioTuneScan.SendUpdate(true);
}


/**
 * Get Pop up status
 */
bool SxmAudioDataBindingUtils::getPopupStatus()
{
   return (*_sxmPopupInfo).mPopupVisible;
}


/**
* Helper function to set button mode.. Preset/ Control
*/
void SxmAudioDataBindingUtils::setAudioActiveMode(uint8 mode)
{
   _activeMode = mode;
}


/**
 * Helper function to set advisory status of content alerts
 */
void SxmAudioDataBindingUtils::setAlertsAdvisory(bool advorlist, std::string message)
{
   ETG_TRACE_USR4(("SxmAudioDataBindingUtils::getAdvisoryTextForFavorites %d", advorlist));
   ETG_TRACE_USR4(("SxmAudioDataBindingUtils::getAdvisoryTextForFavorites %s", message.c_str()));
   DataBindingItem<SxmAudioContentAlertDataBindingSource> sxmAudioContentAlerts;
   DataBindingItem<SxmAudioLoadingInfoDataBindingSource> SxmAudioLoadingInfo;
   (*sxmAudioContentAlerts).mAdvisoryMessage = message.c_str();
   (*sxmAudioContentAlerts).mAdvisoryLoading = true;
   if (!advorlist && "Loading" == message)
   {
      (*sxmAudioContentAlerts).mAdvisoryLoading = false;
      std::string loadingText = SxmUtils::safeGetCString(SXM__AUDIO_ADV_LOADING);
      (*SxmAudioLoadingInfo).mLoad1 = loadingText.c_str();
      (*SxmAudioLoadingInfo).mLoad2 = SxmUtils::conCatData("ss", 2, loadingText.c_str(), ".").c_str();
      (*SxmAudioLoadingInfo).mLoad3 = SxmUtils::conCatData("ss", 2, loadingText.c_str(), "..").c_str();
      (*SxmAudioLoadingInfo).mLoad4 = SxmUtils::conCatData("ss", 2, loadingText.c_str(), "...").c_str();
      SxmAudioLoadingInfo.SendUpdate(true);
   }
   (*sxmAudioContentAlerts).mAdvisoryorList = advorlist;
   sxmAudioContentAlerts.SendUpdate(true);
}


/**
 * Helper function to set advisory status of TWN
 */
void SxmAudioDataBindingUtils::setTWNAdvisory(bool advorlist)
{
   DataBindingItem<SxmAudioLoadingInfoDataBindingSource>sxmAudioLoadingInfo;
   (*_sxmTWNButton).mAdvisoryorList = advorlist;
   std::string loadingText = SxmUtils::safeGetCString(SXM__AUDIO_ADV_LOADING);
   (*sxmAudioLoadingInfo).mLoad1 = loadingText.c_str();
   (*sxmAudioLoadingInfo).mLoad2 = SxmUtils::conCatData("ss", 2, loadingText.c_str(), ".").c_str();
   (*sxmAudioLoadingInfo).mLoad3 = SxmUtils::conCatData("ss", 2, loadingText.c_str(), "..").c_str();
   (*sxmAudioLoadingInfo).mLoad4 = SxmUtils::conCatData("ss", 2, loadingText.c_str(), "...").c_str();
   _sxmTWNButton.MarkItemModified(ItemKey::SXTWNButton::AdvisoryorListItem);
   _sxmTWNButton.SendUpdate(false);
   sxmAudioLoadingInfo.SendUpdate(true);
}


/**
 * Helper function to set advisory status of sportsFlash
 */
void SxmAudioDataBindingUtils::setSFAdvisory(bool advorlist)
{
   DataBindingItem<SxmAudioLoadingInfoDataBindingSource> SxmAudioLoadingInfo;
   (*_sxmSFButton).mAdvisoryorList = advorlist;
   std::string loadingText = SxmUtils::safeGetCString(SXM__AUDIO_ADV_LOADING);
   (*SxmAudioLoadingInfo).mLoad1 = loadingText.c_str();
   (*SxmAudioLoadingInfo).mLoad2 = SxmUtils::conCatData("ss", 2, loadingText.c_str(), ".").c_str();
   (*SxmAudioLoadingInfo).mLoad3 = SxmUtils::conCatData("ss", 2, loadingText.c_str(), "..").c_str();
   (*SxmAudioLoadingInfo).mLoad4 = SxmUtils::conCatData("ss", 2, loadingText.c_str(), "...").c_str();
   _sxmSFButton.MarkItemModified(ItemKey::SXMSFButton::AdvisoryorListItem);
   _sxmSFButton.SendUpdate(false);
   SxmAudioLoadingInfo.SendUpdate(true);
}


/**
 * Helper function to Enable/disable SXM info Subscribe button
 * Enable Subscribe button only when Audio and Travelink and Traffic is not subscribed.
 */
void SxmAudioDataBindingUtils::setSxmInfoButtonEnable(bool buttonstate) const
{
   DataBindingItem<SxmInfoButtonEnableDataBindingSource> lSxmInfoData;
   (*lSxmInfoData).mEnable = buttonstate;
   lSxmInfoData.SendUpdate(true);
}


/**
 * Set TWN Pop up text.
 */
void SxmAudioDataBindingUtils::setTWNTpopText()
{
   ETG_TRACE_USR4(("SxmAudioDataBindingUtils::vSetTWNTpopText"));
   DataBindingItem<SXMAudioTWNNoInfoDataBindingSource> sxmTWNNoInfo;
   std::string text = "";
   text.append(SxmUtils::safeGetCString(SXM__AUDIO_TWN_NOINFOL1)).append(" ").append(SxmUtils::safeGetCString(SXM__AUDIO_TWN_NOINFOL2));
   (*sxmTWNNoInfo).mtext = text.c_str();
   sxmTWNNoInfo.SendUpdate(true);
}


/**
 * Set system message status
 */
void SxmAudioDataBindingUtils::setSystemMessageStatus(uint8 u8status)
{
   _systemErrorCode = u8status;
}


/**
 * Get system message status
 */
uint8 SxmAudioDataBindingUtils::getSystemMessageStatus() const
{
   return _systemErrorCode;
}


/**
 * Enable/ disable menu items
 */
void SxmAudioDataBindingUtils::updateMenuItems(uint8 item, bool status)
{
   switch (item)
   {
      case SXM_AUDIO_SAVEARTIST:
      {
         (*_sxmMenuItem).mArtistEnable = status;
         _sxmMenuItem.MarkItemModified(ItemKey::SxmMenuItem::ArtistEnableItem);
         break;
      }
      case SXM_AUDIO_SAVESONG:
      {
         (*_sxmMenuItem).mSongEnable = status;
         _sxmMenuItem.MarkItemModified(ItemKey::SxmMenuItem::SongEnableItem);
         break;
      }
      default:
         break;
   }
   _sxmMenuItem.SendUpdate(false);
}


/**
 * Helper function to set availabilty of preset list.
 */
void SxmAudioDataBindingUtils::setSxmAllChannelInfoAvailbility(bool status)
{
   ETG_TRACE_USR4(("SxmAudioDataBindingUtils::setSxmAllChannelInfoAvailbility"));
   _isAllChannelInfoAvail = status;
}


/**
 * Helper function to set availabilty of preset list.
 */
void SxmAudioDataBindingUtils::setSxmPresetListAvailbility(bool status)
{
   ETG_TRACE_USR4(("SxmAudioDataBindingUtils::setSxmPresetListAvailbility"));
   _isPresetListAvail = status;
}


/**
 * Helper function to get current status of all channel information.
 */
bool SxmAudioDataBindingUtils::getSxmAllChannelInfoAvailbility()
{
   return _isAllChannelInfoAvail;
}


/**
 * Helper function to get current status of preset list.
 */
bool SxmAudioDataBindingUtils::getSxmPresetListAvailbility()
{
   return _isPresetListAvail;
}


/**
 * Helper function to get current status of preset list.
 */
bool SxmAudioDataBindingUtils::getSxmChannelInfoSyncStatus()
{
   return (_isAllChannelInfoAvail && _isPresetListAvail);
}


/**
 * Helper function to set TWN play status.
 */
void SxmAudioDataBindingUtils::setSxmTWNPlayStatus(bool status)
{
   ETG_TRACE_USR4(("SxmAudioDataBindingUtils::setSxmTWNPlayStatus %d", status));
   _isTWNActive = status;
}


/**
 * Helper function to get TWN play status.
 */
bool SxmAudioDataBindingUtils::getSxmTWNPlayStatus()
{
   return _isTWNActive;
}


/**
 * Helper function to set SF play status.
 */
void SxmAudioDataBindingUtils::setSxmSFPlayStatus(bool status)
{
   ETG_TRACE_USR4(("SxmAudioDataBindingUtils::setSxmSFPlayStatus %d", status));
   _isSFActive = status;
}


/**
 * Helper function to get SF play status.
 */
bool SxmAudioDataBindingUtils::getSxmSFPlayStatus()
{
   return _isSFActive;
}


/**
 * Helper function to get SF play status.
 */
bool SxmAudioDataBindingUtils::getSxmTuneScanStatus()
{
   return _isScanActive;
}


/**
 * Helper function to set Content Alert Status.
 */
void SxmAudioDataBindingUtils::setSxmContentAlertStatus(bool status)
{
   _istuneToContentAlert = status;
}


/**
 * Helper function to get Content Alert Status.
 */
bool SxmAudioDataBindingUtils::getSxmContentAlertStatus()
{
   return _istuneToContentAlert;
}


/**
 * Helper function to set transition done status.
 */
void SxmAudioDataBindingUtils::setSxmTransitionDoneStatus(bool status)
{
   _isTransitionDone = status;
}


/**
 * Helper function to get transition done status.
 */
bool SxmAudioDataBindingUtils::getSxmTransitionDoneStatus()
{
   return _isTransitionDone;
}


/**
 * Helper function to get Instant Replay status.
 */
bool SxmAudioDataBindingUtils::getSxmInstantReplayStatus()
{
   return (*_sxmAudioPlayPause).mPlayOrPause;
}


/**
 * Helper function to set Mute status.
 */
void SxmAudioDataBindingUtils::setMuteStatus(bool status)
{
   ETG_TRACE_USR4(("SxmAudioDataBindingUtils::setMuteStatus %d", status));
   if (status)
   {
      (*_audioMuteInfo).mMuteStatus = true;
   }
   else
   {
      (*_audioMuteInfo).mMuteStatus = false;
   }
   _audioMuteInfo.SendUpdate(true);
}


}  // namespace Core
}  // namespace App
