/**************************************************************************************
 * @file        : SXMVehicleService.cpp
 * @addtogroup  : AppHmi_Sxm
 * @brief       : SXM vehicle utility class required to interact with MIDW vehicle Fi.
 * @copyright   : (C) 2016 Robert Bosch Engineering and Business Solutions Limited.
 *                The reproduction, distribution and utilization of this file as
 *                well as the communication of its contents to others without express
 *                authorization is prohibited. Offenders will be held liable for the
 *                payment of damages. All rights reserved in the event of the grant
 *                of a patent, utility model or design.
 **************************************************************************************/
#include "hmi_trace_if.h"
#include "SXMVehicleService.h"
#include "App/Core/SxmUtils/SxmUtils.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS       TR_CLASS_APPHMI_SXM_HALL
#define ETG_I_TRACE_CHANNEL           TR_TTFIS_APPHMI_SXM
#define ETG_I_TTFIS_CMD_PREFIX        "APPHMI_SXM_"
#define ETG_I_FILE_PREFIX             App::Core::SXMVehicleService::
#include "trcGenProj/Header/SXMVehicleService.cpp.trc.h"
#endif

namespace App {
namespace Core {
/**
 * SXMVehicleService Class constructor
 */
SXMVehicleService::SXMVehicleService()
   : _mPtrvehicleproxy(VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy::createProxy("vehicleMainFiPort", *this))
{
   //ETG_I_REGISTER_FILE();
   ETG_TRACE_USR1(("SXMVehicleService() Constructor"));
   SXM_UTILS_REGISTER_PROPERTY_CALLBACKS(StartupSync::getInstance(), this, _mPtrvehicleproxy);
}


/**
 * SXMVehicleService Class destructor
 */
SXMVehicleService::~SXMVehicleService()
{
   ETG_TRACE_USR1(("~SXMVehicleService() Destructor"));
   _mPtrvehicleproxy.reset();
   ETG_I_UNREGISTER_FILE();
}


/**
 * Utility function that registers for notifications updates from vehicle service.
 * @param [in] : proxy - vehicle service class instance
 * @param [in] : stateChange - Current ASF state
 */
void SXMVehicleService::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& /*proxy*/, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR1(("SXMVehicleService registerProperties"));
}


/**
 * Utility function that deregisters for notifications updates from vehicle service.
 * @param [in] : proxy - vehicle service class instance
 * @param [in] : stateChange - Current ASF state
 */
void SXMVehicleService::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& /*proxy*/, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR1(("SXMVehicleService deregisterProperties"));
}


/**
 * Error handling function for sendXMInfo
 */
void SXMVehicleService::onSendXMInfoError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::SendXMInfoError >& /*error*/)
{
   ETG_TRACE_ERR(("SendXMInfoCallbackIF::onSendXMInfoError"));
}


/**
 * Result handling function for sendXMInfo
 */
void SXMVehicleService::onSendXMInfoResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::SendXMInfoResult >& /*result*/)
{
   ETG_TRACE_COMP(("SendXMInfoCallbackIF::onSendXMInfoResult"));
}


/**
 * Error handling function for sendXMPresetInfo
 */
void SXMVehicleService::onMediaStatusTunerPresetStatusError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::MediaStatusTunerPresetStatusError >& /*error*/)
{
   ETG_TRACE_ERR(("MediaStatusTunerPresetStatusCallbackIF::onMediaStatusTunerPresetStatusError"));
}


void SXMVehicleService::onMediaStatusTunerPresetStatusResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::MediaStatusTunerPresetStatusResult >& /*result*/)
{
   ETG_TRACE_COMP(("MediaStatusTunerPresetStatusCallbackIF::onMediaStatusTunerPresetStatusResult"));
}


/**
 * This function updates the SXM Advisory status to CAN
 * @param [in] : type - Advisory or channel or category
 * @param [in] : dataInfo - Information according to type
 */
void SXMVehicleService::sendXMInfoData(const int type, const std::string& dataInfo)
{
   //Concatenate Element type and terminating character
   if (!dataInfo.empty())
   {
      ETG_TRACE_USR4(("SXMVehicleService::sendXMInfoData  Type: %d and DataInfo: %s", type, dataInfo.c_str()));
      //Vector creation
      std::vector<uint8> sxmInfo;
      ::vehicle_main_fi_types::T_CAN_Mesg  sxmInfoCANMsg;
      //Push SXMtype into first position in the vector
      sxmInfo.push_back(static_cast<uint8>(type));
      //Setting TcanMsg initial parameter also with SXMtype
      sxmInfoCANMsg.setU8MediaType(type);
      size_t dataLength = dataInfo.size();
      //Push the SXM Data into our local vector from position index 1 as 0 is filled with sxm type
      for (uint i = 0; i < dataLength; ++i)
      {
         sxmInfo.push_back(dataInfo[i]);
         ETG_TRACE_USR4(("SXMVehicleService::sendXMInfoData DataInfo at %d: %d", i, dataInfo[i]));
      }
      //Updating the trailing character
      sxmInfo.push_back('\0');
      //Adding the local vector to TcanMsg second parameter
      sxmInfoCANMsg.setListData(sxmInfo);
      if (_mPtrvehicleproxy != NULL && (type == SXM_ADVISORY) || (type == SXM_CHANNEL_NUMBER) || (type == SXM_CATEGORY))
      {
         _mPtrvehicleproxy->sendSendXMInfoStart(*this, sxmInfoCANMsg);
      }
   }
   else
   {
      ETG_TRACE_USR4(("SXMVehicleService::sendXMInfoData the vector is empty"));
   }
}


/**
 * This function updates the SXM Preset status to CAN
 * @param [in] : presetId
 */
void SXMVehicleService::updateXMPresetStatus(const int presetId)
{
   ETG_TRACE_USR4(("SXMVehicleService::updateXMPresetStatus: presetId : %d", presetId));
   if (_mPtrvehicleproxy != NULL)
   {
      _mPtrvehicleproxy->sendMediaStatusTunerPresetStatusStart(*this, presetId);
   }
   else
   {
      ETG_TRACE_ERR(("SXMVehicleService::updateXMPresetStatus:_mPtrvehicleproxy is NULL"));
   }
}


}  // namespace Core
}  // namespace App
