/**************************************************************************************
 * @file        : DiagnosticsClient.cpp
 * @addtogroup  : AppHmi_Sxm
 * @brief       : SXM utility class required to interact with source run time requests.
 * @copyright   : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
 *                The reproduction, distribution and utilization of this file as
 *                well as the communication of its contents to others without express
 *                authorization is prohibited. Offenders will be held liable for the
 *                payment of damages. All rights reserved in the event of the grant
 *                of a patent, utility model or design.
 **************************************************************************************/
#include "hmi_trace_if.h"
#include "App/Core/SxmUtils/SxmUtils.h"
#include "App/Core/SxmCommon/SXMCommonInterface.h"
#include "DiagnosticsClient.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS       TR_CLASS_APPHMI_SXM_HALL
#define ETG_I_TRACE_CHANNEL           TR_TTFIS_APPHMI_SXM
#define ETG_I_TTFIS_CMD_PREFIX        "APPHMI_SXM_"
#define ETG_I_FILE_PREFIX             App::Core::DiagnosticsClient::
#include "trcGenProj/Header/DiagnosticsClient.cpp.trc.h"
#endif

static const uint16_t RUNTIME_U16_APP_SXM  = 0x008C;

using namespace ::asf::core;
using namespace Midw_Diagnostics_FI;

namespace App {
namespace Core {

/**
 * DiagnosticsClient Class constructor
 */
DiagnosticsClient::DiagnosticsClient()
   : _diagnosticProxy(Midw_Diagnostics_FIProxy::createProxy("diagnosticsFiPort", *this))
{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR4(("DiagnosticsClient() Constructor"));
}


/**
 * DiagnosticsClient Class destructor
 */
DiagnosticsClient::~DiagnosticsClient()
{
   ETG_TRACE_USR4(("~DiagnosticsClient() Destructor"));
   ETG_I_UNREGISTER_FILE();
   _diagnosticProxy.reset();
}


/**
 * Utility function that registers for notifications updates from PropertyRegistration Interface.
 * @param [in] : proxy - AudioSourceChange class instance
 * @param [in] : stateChange - Current ASF state
 */
void DiagnosticsClient::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& /*proxy*/,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR1(("DiagnosticsClient registerProperties"));
   StartupSync::getInstance().registerPropertyRegistrationIF(this, _diagnosticProxy->getPortName());
}


/**
 * Utility function that deregisters for notifications updates from PropertyRegistration Interface.
 * @param [in] : proxy - AudioSourceChange class instance
 * @param [in] : stateChange - Current ASF state
 */
void DiagnosticsClient::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& /*proxy*/,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR1(("DiagnosticsClient deregisterProperties"));
}


/**
 * Utility function that notifies on available updates from ServiceAvailable Interface.
 * @param [in] : proxy - AudioSourceChange class instance
 * @param [in] : stateChange - Current ASF state
 */
void DiagnosticsClient::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& /*proxy*/, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("DiagnosticsClient::onAvailable"));
}


/**
 * Utility function that notifies on unavailable updates from ServiceAvailable Interface.
 * @param [in] : proxy - AudioSourceChange class instance
 * @param [in] : stateChange - Current ASF state
 */
void DiagnosticsClient::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& /*proxy*/, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("DiagnosticsClient::onUnavailable"));
}


/**
 * Virtual function implemented to get update on Source Data Error.
 * @param [in] : proxy - the client side representation of the CCA Functional Interface 'DiagnosticsClientHandler'
 * @param [in] : error : The error message of 'SOurce Data Error'
 */
void DiagnosticsClient::onSetRunTimeSourceDataError(const ::boost::shared_ptr< Midw_Diagnostics_FI::Midw_Diagnostics_FIProxy >& /*proxy*/, const ::boost::shared_ptr< Midw_Diagnostics_FI::SetRunTimeSourceDataError >& /*error*/)
{
   ETG_TRACE_ERR(("DiagnosticsClient::onSetRunTimeSourceDataError()"));
}


/**
 * Virtual function implemented to get update on Runtime Source Data Result.
 * @param [in] : proxy - the client side representation of the CCA Functional Interface 'DiagnosticsClientHandler'
 * @param [in] : error : The error message of 'SOurce Data Error'
 */
void DiagnosticsClient::onSetRunTimeSourceDataResult(const ::boost::shared_ptr< Midw_Diagnostics_FI::Midw_Diagnostics_FIProxy >& /*proxy*/, const ::boost::shared_ptr< Midw_Diagnostics_FI::SetRunTimeSourceDataResult >& /*result*/)
{
   ETG_TRACE_USR4(("DiagnosticsClient::onSetRunTimeSourceDataResult()"));
}


/**
 * Utility function that updates SXM run time.
 * @param [in] : Seconds - duration of SXM played
 */
void DiagnosticsClient::updateSXMRuntimeInfo(const uint32_t& seconds)
{
   ETG_TRACE_USR4(("DiagnosticsClient::updateSXMRuntimeInfo : %d seconds", seconds));
   if (NULL != _diagnosticProxy)
   {
      _diagnosticProxy->sendSetRunTimeSourceDataStart(*this, RUNTIME_U16_APP_SXM, seconds);
   }
}


}  // namespace Core
}  // namespace App
